"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var rxjs_1 = require("rxjs");
var operators_1 = require("rxjs/operators");
/**
 * Determines the unwrapping behavior of asynchronous resolve values.
 *
 *   - When an Observable is returned from the resolveFn, wait until the Observable emits at least one item.
 *     If any other value will be converted to an Observable that emits such value.
 *   - The Observable item will not be unwrapped.
 *   - The Observable stream itself will be provided when the resolve is injected or bound elsewhere.
 *
 * #### Example:
 *
 * The `Transition` will wait for the `main.home` resolve observables to emit their first value.
 * Promises will be unwrapped and returned as observables before being provided to components.
 * ```js
 * var mainState = {
 *   name: 'main',
 *   resolve: mainResolves, // defined elsewhere
 *   resolvePolicy: { async: RXWAIT },
 * }
 * ```
 */
function RXWAIT(resolveFnValue) {
    if (!(resolveFnValue instanceof rxjs_1.Observable)) {
        resolveFnValue = rxjs_1.of(resolveFnValue);
    }
    var data$ = resolveFnValue.pipe(operators_1.shareReplay(1));
    return data$
        .pipe(operators_1.first())
        .toPromise()
        .then(function () {
        return data$;
    });
}
exports.RXWAIT = RXWAIT;
//# sourceMappingURL=rx-async-policy.js.map