/**
 * DevExtreme (ui/scheduler/ui.scheduler.table_creator.js)
 * Version: 19.2.3
 * Build date: Tue Oct 22 2019
 *
 * Copyright (c) 2012 - 2019 Developer Express Inc. ALL RIGHTS RESERVED
 * Read about DevExtreme licensing here: https://js.devexpress.com/Licensing/
 */
"use strict";
var _renderer = require("../../core/renderer");
var _renderer2 = _interopRequireDefault(_renderer);
var _dom_adapter = require("../../core/dom_adapter");
var _dom_adapter2 = _interopRequireDefault(_dom_adapter);
var _element_data = require("../../core/element_data");
var _element_data2 = _interopRequireDefault(_element_data);
var _type = require("../../core/utils/type");
var _type2 = _interopRequireDefault(_type);
var _dom = require("../../core/utils/dom");

function _interopRequireDefault(obj) {
    return obj && obj.__esModule ? obj : {
        "default": obj
    }
}
var ROW_SELECTOR = "tr";
var SchedulerTableCreator = {
    VERTICAL: "vertical",
    HORIZONTAL: "horizontal",
    insertAllDayRow: function(allDayElements, tableBody, index) {
        if (allDayElements[index]) {
            var row = allDayElements[index].find(ROW_SELECTOR);
            if (!row.length) {
                row = (0, _renderer2.default)(_dom_adapter2.default.createElement(ROW_SELECTOR));
                row.append(allDayElements[index].get(0))
            }
            tableBody.appendChild(row.get ? row.get(0) : row)
        }
    },
    makeTable: function(options) {
        var row, tableBody = _dom_adapter2.default.createElement("tbody"),
            templateCallbacks = [],
            rowCountInGroup = options.groupCount ? options.rowCount / options.groupCount : options.rowCount,
            allDayElementIndex = 0,
            allDayElements = options.allDayElements,
            groupIndex = options.groupIndex,
            rowCount = options.rowCount;
        (0, _renderer2.default)(options.container).append(tableBody);
        if (allDayElements) {
            this.insertAllDayRow(allDayElements, tableBody, 0);
            allDayElementIndex++
        }
        for (var i = 0; i < rowCount; i++) {
            row = _dom_adapter2.default.createElement(ROW_SELECTOR);
            tableBody.appendChild(row);
            var isLastRowInGroup = (i + 1) % rowCountInGroup === 0;
            if (options.rowClass) {
                row.className = options.rowClass
            }
            for (var j = 0; j < options.cellCount; j++) {
                var td = _dom_adapter2.default.createElement("td");
                row.appendChild(td);
                if (options.cellClass) {
                    if (_type2.default.isFunction(options.cellClass)) {
                        td.className = options.cellClass(i, j)
                    } else {
                        td.className = options.cellClass
                    }
                }
                var cellDataObject, dataKey, dataValue;
                if (options.getCellData) {
                    cellDataObject = options.getCellData(td, i, j, groupIndex);
                    dataKey = cellDataObject.key;
                    dataValue = cellDataObject.value;
                    dataKey && _element_data2.default.data(td, dataKey, dataValue)
                }
                if (options.cellTemplate && options.cellTemplate.render) {
                    var templateOptions = {
                        model: {
                            text: options.getCellText ? options.getCellText(i, j) : "",
                            date: options.getCellDate ? options.getCellDate(i) : void 0
                        },
                        container: (0, _dom.getPublicElement)((0, _renderer2.default)(td)),
                        index: i * options.cellCount + j
                    };
                    if (dataValue) {
                        if (dataValue.startDate) {
                            templateOptions.model.startDate = dataValue.startDate
                        }
                        if (dataValue.endDate) {
                            templateOptions.model.endDate = dataValue.endDate
                        }
                        if (dataValue.groups) {
                            templateOptions.model.groups = dataValue.groups
                        }
                        if (dataValue.allDay) {
                            templateOptions.model.allDay = dataValue.allDay
                        }
                    }
                    templateCallbacks.push(options.cellTemplate.render.bind(options.cellTemplate, templateOptions))
                } else {
                    if (options.getCellText) {
                        td.innerHTML = "<div>" + options.getCellText(i, j) + "</div>"
                    }
                }
            }
            if (allDayElements && isLastRowInGroup) {
                this.insertAllDayRow(allDayElements, tableBody, allDayElementIndex);
                allDayElementIndex++
            }
        }
        return templateCallbacks
    },
    makeGroupedTable: function(type, groups, cssClasses, cellCount, cellTemplate, rowCount, groupByDate) {
        var rows = [];
        if (type === this.VERTICAL) {
            rows = this._makeVerticalGroupedRows(groups, cssClasses, cellTemplate, rowCount)
        } else {
            rows = this._makeHorizontalGroupedRows(groups, cssClasses, cellCount, cellTemplate, groupByDate)
        }
        return rows
    },
    makeGroupedTableFromJSON: function(type, data, config) {
        var table, cellStorage = [],
            rowIndex = 0;
        config = config || {};
        var cellTag = config.cellTag || "td",
            childrenField = config.childrenField || "children",
            titleField = config.titleField || "title",
            groupTableClass = config.groupTableClass,
            groupRowClass = config.groupRowClass,
            groupCellClass = config.groupCellClass,
            groupCellCustomContent = config.groupCellCustomContent;

        function createTable() {
            table = _dom_adapter2.default.createElement("table");
            if (groupTableClass) {
                table.className = groupTableClass
            }
        }

        function getChildCount(item) {
            if (item[childrenField]) {
                return item[childrenField].length
            }
            return 0
        }

        function createCell(text, childCount, index, data) {
            var cell = {
                element: _dom_adapter2.default.createElement(cellTag),
                childCount: childCount
            };
            if (groupCellClass) {
                cell.element.className = groupCellClass
            }
            var cellText = _dom_adapter2.default.createTextNode(text);
            if ("function" === typeof groupCellCustomContent) {
                groupCellCustomContent(cell.element, cellText, index, data)
            } else {
                cell.element.appendChild(cellText)
            }
            return cell
        }

        function generateCells(data) {
            for (var i = 0; i < data.length; i++) {
                var childCount = getChildCount(data[i]),
                    cell = createCell(data[i][titleField], childCount, i, data[i]);
                if (!cellStorage[rowIndex]) {
                    cellStorage[rowIndex] = []
                }
                cellStorage[rowIndex].push(cell);
                if (childCount) {
                    generateCells(data[i][childrenField])
                } else {
                    rowIndex++
                }
            }
        }

        function putCellsToRows() {
            cellStorage.forEach(function(cells) {
                var row = _dom_adapter2.default.createElement(ROW_SELECTOR);
                if (groupRowClass) {
                    row.className = groupRowClass
                }
                var rowspans = [];
                for (var i = cells.length - 1; i >= 0; i--) {
                    var prev = cells[i + 1],
                        rowspan = cells[i].childCount;
                    if (prev && prev.childCount) {
                        rowspan *= prev.childCount
                    }
                    rowspans.push(rowspan)
                }
                rowspans.reverse();
                cells.forEach(function(cell, index) {
                    if (rowspans[index]) {
                        cell.element.setAttribute("rowSpan", rowspans[index])
                    }
                    row.appendChild(cell.element)
                });
                table.appendChild(row)
            })
        }
        createTable();
        generateCells(data);
        putCellsToRows();
        return table
    },
    _makeFlexGroupedRowCells: function(group, repeatCount, cssClasses, cellTemplate) {
        var repeatByDate = arguments.length > 4 && void 0 !== arguments[4] ? arguments[4] : 1;
        var cells = [],
            items = group.items,
            itemCount = items.length;
        for (var i = 0; i < repeatCount * repeatByDate; i++) {
            for (var j = 0; j < itemCount; j++) {
                var $container = (0, _renderer2.default)("<div>"),
                    cell = {};
                if (cellTemplate && cellTemplate.render) {
                    var templateOptions = {
                        model: items[j],
                        container: (0, _dom.getPublicElement)($container),
                        index: i * itemCount + j
                    };
                    if (group.data) {
                        templateOptions.model.data = group.data[j]
                    }
                    cell.template = cellTemplate.render.bind(cellTemplate, templateOptions)
                } else {
                    $container.text(items[j].text)
                }
                var cssClass = _type2.default.isFunction(cssClasses.groupHeaderClass) ? cssClasses.groupHeaderClass(j) : cssClasses.groupHeaderClass;
                cell.element = (0, _renderer2.default)("<div>").addClass(cssClass).append($container);
                cells.push(cell)
            }
        }
        return cells
    },
    _makeVerticalGroupedRows: function(groups, cssClasses, cellTemplate) {
        var cellTemplates = [],
            repeatCount = 1,
            cellsArray = [];
        var cellIterator = function(cell) {
            if (cell.template) {
                cellTemplates.push(cell.template)
            }
        };
        for (var i = 0; i < groups.length; i++) {
            if (i > 0) {
                repeatCount = groups[i - 1].items.length * repeatCount
            }
            var cells = this._makeFlexGroupedRowCells(groups[i], repeatCount, cssClasses, cellTemplate);
            cells.forEach(cellIterator);
            cellsArray.push(cells)
        }
        var rows = [],
            groupCount = cellsArray.length;
        for (var _i = 0; _i < groupCount; _i++) {
            rows.push((0, _renderer2.default)("<div>").addClass(cssClasses.groupHeaderRowClass))
        }
        for (var _i2 = groupCount - 1; _i2 >= 0; _i2--) {
            var currentColumnLength = cellsArray[_i2].length;
            for (var j = 0; j < currentColumnLength; j++) {
                rows[_i2].append(cellsArray[_i2][j].element)
            }
        }
        return {
            elements: (0, _renderer2.default)("<div>").addClass("dx-scheduler-group-flex-container").append(rows),
            cellTemplates: cellTemplates
        }
    },
    _makeHorizontalGroupedRows: function(groups, cssClasses, cellCount, cellTemplate, groupByDate) {
        var repeatCount = 1,
            groupCount = groups.length,
            rows = [],
            cellTemplates = [],
            repeatByDate = groupByDate ? cellCount : 1;
        var cellIterator = function(cell) {
            if (cell.template) {
                cellTemplates.push(cell.template)
            }
            return cell.element
        };
        for (var i = 0; i < groupCount; i++) {
            if (i > 0) {
                repeatCount = groups[i - 1].items.length * repeatCount
            }
            var cells = this._makeGroupedRowCells(groups[i], repeatCount, cssClasses, cellTemplate, repeatByDate);
            rows.push((0, _renderer2.default)("<tr>").addClass(cssClasses.groupRowClass).append(cells.map(cellIterator)))
        }
        var maxCellCount = rows[groupCount - 1].find("th").length;
        for (var j = 0; j < groupCount; j++) {
            var $cell = rows[j].find("th"),
                colspan = maxCellCount / $cell.length;
            if (!groupByDate) {
                colspan *= cellCount
            }
            if (colspan > 1 && 1 === repeatByDate || groupByDate && groupCount > 1) {
                $cell.attr("colSpan", colspan)
            }
        }
        return {
            elements: rows,
            cellTemplates: cellTemplates
        }
    },
    _makeGroupedRowCells: function(group, repeatCount, cssClasses, cellTemplate, repeatByDate) {
        repeatByDate = repeatByDate || 1;
        repeatCount *= repeatByDate;
        var cells = [],
            items = group.items,
            itemCount = items.length;
        for (var i = 0; i < repeatCount; i++) {
            for (var j = 0; j < itemCount; j++) {
                var $container = (0, _renderer2.default)("<div>"),
                    cell = {};
                if (cellTemplate && cellTemplate.render) {
                    var templateOptions = {
                        model: items[j],
                        container: (0, _dom.getPublicElement)($container),
                        index: i * itemCount + j
                    };
                    if (group.data) {
                        templateOptions.model.data = group.data[j]
                    }
                    cell.template = cellTemplate.render.bind(cellTemplate, templateOptions)
                } else {
                    $container.text(items[j].text);
                    $container = (0, _renderer2.default)("<div>").append($container)
                }
                $container.addClass(cssClasses.groupHeaderContentClass);
                var cssClass;
                if (_type2.default.isFunction(cssClasses.groupHeaderClass)) {
                    cssClass = cssClasses.groupHeaderClass(j)
                } else {
                    cssClass = cssClasses.groupHeaderClass
                }
                cell.element = (0, _renderer2.default)("<th>").addClass(cssClass).append($container);
                cells.push(cell)
            }
        }
        return cells
    }
};
module.exports = SchedulerTableCreator;
