/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,constantProperty,extraRequire,missingOverride,missingReturn,unusedPrivateMembers,uselessCode} checked by tsc
 */
import { Component, Input, ViewChild } from '@angular/core';
import { FormArray, FormBuilder, FormControl } from '@angular/forms';
import { NgbDropdown } from '@ng-bootstrap/ng-bootstrap';
import { DxDataGridComponent } from 'devextreme-angular';
import { isEqual, sortBy } from 'lodash-es';
import { first } from 'rxjs/operators';
import FilterConfig from './filter-config';
/**
 * @record
 */
function ICriterion() { }
if (false) {
    /** @type {?} */
    ICriterion.prototype.property;
    /** @type {?} */
    ICriterion.prototype.operator;
    /** @type {?} */
    ICriterion.prototype.value;
}
export class NwfFilterComponent {
    /**
     * @param {?} _fb
     */
    constructor(_fb) {
        this._fb = _fb;
        this.nwfFilters = [];
        this.config = FilterConfig;
        this.initialColumns = [];
    }
    /**
     * @return {?}
     */
    ngOnInit() {
        this.dataGrid.onContentReady.subscribe((/**
         * @param {?} data
         * @return {?}
         */
        (data) => {
            this.gridInstance = data.component.instance();
            // I store a reference on the instance so cells/cells-header.ts can find it without another painful binding. Probably useful later.
            this.gridInstance.nwfFilteringWidget = this;
            // getting new columns.
            /** @type {?} */
            const columnOnDxGrid = this.getAllColumnsOnDxGrid().map((/**
             * @param {?} col
             * @return {?}
             */
            (col) => col.dataField));
            // when intialColumns is set, we compare it with columnsOnDxGrid and
            // and only update intialColumns when its different than columnsOnDxGrid.
            // This happens when we reset the columns on the grid.
            if (this.initialColumns.length !== 0) {
                /** @type {?} */
                const areColumnsEqual = isEqual(columnOnDxGrid, this.initialColumns);
                // if new columns different than initialColumns, we reset the filter widget.
                if (!areColumnsEqual) {
                    this.initialColumns = columnOnDxGrid;
                    // setting the filter widget if columns are not same.
                    this.setDxGridFilterOnNwfFilterWidget();
                }
            }
            else {
                // setting the initialColumns.
                this.initialColumns = columnOnDxGrid;
                // setting the filter widget for the first time.
                this.setDxGridFilterOnNwfFilterWidget();
            }
        }));
    }
    /**
     * @return {?}
     */
    ngAfterViewInit() {
        /* I listen for the dropdown opening in order to initialize it lazily, only the first time we are opened. The subscription is cleaned up automatically. */
        this.ngbDropDown.openChange.pipe(first()).subscribe((/**
         * @param {?} isOpen
         * @return {?}
         */
        (isOpen) => {
            if (isOpen) {
                this.initPopover();
            }
        }));
    }
    /**
     * \@description adds a new default criterion on click of "Add Filter" button, or being called externally
     * @param {?=} schemaColumnObject
     * @return {?}
     */
    addCriterion(schemaColumnObject) {
        // we dont have filterFormGrp, if there is no filter applied, and we open the filter from a column header.
        if (!this.filterFormGrp) {
            /** @type {?} */
            const tempFormArray = [];
            // filterFormGrp consists of array of criterions.
            this.setFilterFormGrp(tempFormArray);
        }
        // we get reference of criterions array.
        /** @type {?} */
        const formArray = this.getCriterionsArray();
        /** @type {?} */
        let criterion;
        // if we send schemaColumnObject, we create a criterion and add that criterion to formArray.
        if (schemaColumnObject) {
            criterion = {
                property: schemaColumnObject.dataField,
                operator: this.getOperatorOptions(schemaColumnObject.dataType)[0].value,
                value: '',
            };
        }
        formArray.push(this.initCriterion(criterion));
        // I open the popover if its closed.
        if (!this.ngbDropDown.isOpen()) {
            this.ngbDropDown.open();
        }
    }
    /**
     * \@description deletes a criterion from current criterion, else deletes all the applied criterion. Invoked on clicking Trash icon.
     * @param {?} criterionIndex
     * @return {?}
     */
    deleteCriterion(criterionIndex) {
        // delete criterion from formArray.
        /** @type {?} */
        const formArray = this.getCriterionsArray();
        // when we have more than 1 criteria, this means we only delete the criterion at that index passed-in
        if (formArray.length > 1) {
            // deleting criterion from fromArray (formArray is array of criterion)
            formArray.removeAt(criterionIndex);
        }
        else {
            // this case applies, when we delete the only available criterion in the filter.
            // this means delete single/last criterion = filter reset.
            this.reset();
        }
    }
    /**
     * \@description deletes all the criterion.
     * @return {?}
     */
    reset() {
        // reset the formArray, so that form will come fresh on hitting filter button again.
        this.filterFormGrp = this._fb.group({
            criterion: new FormArray([]),
        });
        // clear existing filters and update it on gridInstance.
        this.apply();
        // We are done with this formGroup, so we set it to null.
        // On reopening this filter widget -  we set the formGroup again.
        this.filterFormGrp = null;
        // re-initialize the popover.
        this.initPopover();
    }
    /**
     * \@description applies the filter on dx grid instance on hitting Apply button.
     * @return {?}
     */
    apply() {
        // We touch all formGroups to ensure validation is invoked without manually touching them.
        this.validateAllFormFields(this.filterFormGrp.controls.criterion['controls']);
        // When we click on the submit button, we want to submit the form only if it is valid.
        // To validate all form fields, we need to iterate throughout all form controls and mark them as touched.
        if (this.filterFormGrp.valid) {
            // set new filters on the grid.
            this.setFiltersOnDxGridInstance();
            // closing the popover.
            this.ngbDropDown.close();
        }
    }
    /**
     * \@description cancel on hitting 'Cancel' button.
     * @return {?}
     */
    cancel() {
        // we want to reset the filter widget to the previous state, without submitting any pending changes.
        // we leverage 'createFormGroup' method to set filterWidget state.
        this.createFormGroup();
        // closing the popover.
        this.ngbDropDown.close();
        // We are done with this formGroup, so we set it to null.
        // On reopening this filter widget -  we set the formGroup again.
        this.filterFormGrp = null;
    }
    /**
     * @private
     * @return {?}
     */
    initPopover() {
        // when we initialize the popover, and we dont have any filters applied, we create a new filterForm and add an initial criteria to it.
        if (this.nwfFilters.length === 0 && !this.filterFormGrp) {
            /** @type {?} */
            const tempFormArray = [];
            // add default criterion.
            tempFormArray.push(this.initCriterion());
            // filterFormGrp consists of array of criterions.
            this.setFilterFormGrp(tempFormArray);
        }
    }
    /**
     * @private
     * @return {?}
     */
    setDxGridFilterOnNwfFilterWidget() {
        // get columns from gridInstance.
        /** @type {?} */
        const gridDxColumns = this.getAllColumnsOnDxGrid();
        // columns having allowFiltering as true are only allowed in the filter widget. they are sorted alphabettically by caption field.
        this.filterableColumns = sortBy(gridDxColumns.filter((/**
         * @param {?} dxColumn
         * @return {?}
         */
        (dxColumn) => (dxColumn.allowFiltering && dxColumn.showInColumnChooser))), [(/**
             * @param {?} dxColumn
             * @return {?}
             */
            function (dxColumn) { return dxColumn.caption; })]);
        /** @type {?} */
        const dxGridFilter = this.gridInstance.state().filterValue || [];
        // set value on filter widget.
        this.dxToNwfFilterConverter(dxGridFilter);
        // create the filter form group - its a group of filter criterion.
        this.createFormGroup();
    }
    /**
     * Creates the form that holds all the filter criterion as formControl inside it.
     * FormGroup holds the state of the filter widget.
     * @private
     * @return {?}
     */
    createFormGroup() {
        /** @type {?} */
        const tempFormArray = [];
        // if  nwfFilters, that means filter is preset,
        if (this.nwfFilters.length > 0) {
            // each nwf filter is represented as a criterion in filter widget.
            // so now, we loop over all nwfFilters and create criterion for each of them,
            // and push them into formArray.
            this.nwfFilters.forEach((/**
             * @param {?} c
             * @return {?}
             */
            (c) => {
                tempFormArray.push(this.initCriterion(c));
            }));
            // filterFormGrp consists of array of criterions.
            this.setFilterFormGrp(tempFormArray);
        }
    }
    /**
     * @private
     * @param {?} formArray
     * @return {?}
     */
    setFilterFormGrp(formArray) {
        // filterFormGrp consists of array of crtierions.
        this.filterFormGrp = this._fb.group({
            criterion: new FormArray(formArray),
        });
    }
    /**
     * Method to create a new filter criterion. Each filter criterion is form group of 3 form controls.
     * 1. property, 2. operator, 3. value.
     * @private
     * @param {?=} criterion new filter criterion
     * @return {?}
     */
    initCriterion(criterion = this.getCriterionObject()) {
        // creating a form group that contains formControl for filter-criterion-row component.
        return this._fb.group({
            property: new FormControl(criterion.property),
            operator: new FormControl(criterion.operator),
            value: new FormControl(criterion.value),
        });
    }
    /**
     * @private
     * @return {?}
     */
    getCriterionsArray() {
        return (/** @type {?} */ (this.filterFormGrp.get('criterion')));
    }
    /**
     * @private
     * @param {?} formControls
     * @return {?}
     */
    validateAllFormFields(formControls) {
        // We mark all formControl as 'touched' programatically,
        // This makes sure Angular validation is triggered for every formControl without
        // touching it manually.
        formControls.forEach((/**
         * @param {?} formGroup
         * @return {?}
         */
        (formGroup) => {
            for (const field in formGroup.controls) {
                if (formGroup.controls.hasOwnProperty(field)) {
                    /** @type {?} */
                    const control = formGroup.get(field);
                    if (control instanceof FormControl) {
                        control.markAsTouched({ onlySelf: true });
                    }
                }
            }
        }));
    }
    /**
     * @private
     * @return {?}
     */
    setFiltersOnDxGridInstance() {
        // set new filters on gridInstance.
        /** @type {?} */
        const dxFilters = [];
        // start updating the grid instance.
        this.dataGrid.instance.beginUpdate();
        // clearing the existing filters(if any).
        this.dataGrid.instance.option('filterValue', null);
        // this.dataGrid.instance.clearFilter();
        // we loop over filter form group and create an array containing filters in dx compatible format.
        this.filterFormGrp.get('criterion').value.forEach((/**
         * @param {?} criterion
         * @param {?} index
         * @param {?} array
         * @return {?}
         */
        (criterion, index, array) => {
            // tslint:disable-next-line: prefer-const
            let { property, operator, value } = criterion;
            /** @type {?} */
            const column = this.getPropertyById(property);
            switch (column.dataType.toLowerCase()) {
                case 'datetime':
                case 'date':
                    switch (operator) {
                        case 'between':
                            const [fromDateISOString, toDateISOString] = value.split('~');
                            value = [new Date(fromDateISOString), new Date(toDateISOString)];
                            break;
                        case '<=':
                        case '>=':
                            // we convert the value into JS date object, because in this case the value returned from datetime widget is an ISO string.
                            value = new Date(value);
                            break;
                        default:
                            break;
                    }
                    break;
                case 'enum-multiple':
                    /** @type {?} */
                    const possibleValues = value.split('|');
                    if (operator === '<>') {
                        operator = 'noneof';
                    }
                    else {
                        operator = 'anyof';
                    }
                    value = possibleValues;
                    break;
                default:
                    break;
            }
            //
            if (array.length === 1) {
                dxFilters.push(property);
                dxFilters.push(operator);
                dxFilters.push(value);
            }
            else {
                dxFilters.push([property, operator, value]);
                if (index < array.length - 1) {
                    dxFilters.push('and');
                }
            }
        }));
        // set filters on gridInstance.
        this.dataGrid.instance.option('filterValue', dxFilters);
        // update nwfFilters with new filters applied.
        // this ensures count shown on filter widget is updated, when new filters are applied.
        this.dxToNwfFilterConverter(dxFilters);
        // end update. this is now gonna fire loadOptions call with new filters.
        this.dataGrid.instance.endUpdate();
    }
    /**
     * @private
     * @return {?}
     */
    getCriterionObject() {
        /** @type {?} */
        const firstParam = this.filterableColumns[0];
        return {
            property: firstParam.dataField,
            operator: this.getOperatorOptions(firstParam.dataType)[0].value,
            value: '',
        };
    }
    /**
     * \@description fetch the property from the original params passed into the widget.
     * @private
     * @param {?} id
     * @return {?}
     */
    getPropertyById(id) {
        return this.filterableColumns.filter((/**
         * @param {?} property
         * @return {?}
         */
        (property) => property.dataField === id))[0];
    }
    /**
     * \@description method that return the options from filter config bundle based on the type passed, returns an object containing inputType and options.
     * @private
     * @param {?} paramType
     * @return {?}
     */
    getOperatorOptions(paramType) {
        return this.config[paramType].options;
    }
    /**
     * \@description: method to create the applied criteria from appliedFilters coming from filter widget consumer.
     * This allows to preset the filterWidget this it he VALUE of the filter widget.
     * @private
     * @param {?} dxGridFilters
     * @return {?}
     */
    dxToNwfFilterConverter(dxGridFilters) {
        if (dxGridFilters.length == 3 && this.getPropertyById(dxGridFilters[0])) {
            this.nwfFilters = [this.getFilterCriterionFromDxFilter(dxGridFilters)];
        }
        else {
            // multiple filters are connected by an 'and'. So we remove 'and' from dxGridFilter
            // to get actual filters, and then create filter crtierion from each filter in dxGridFilter.
            this.nwfFilters = dxGridFilters.filter((/**
             * @param {?} filter
             * @return {?}
             */
            (filter) => filter !== 'and')).map((/**
             * @param {?} filter
             * @return {?}
             */
            (filter) => {
                return this.getFilterCriterionFromDxFilter(filter);
            }));
        }
    }
    /**
     * @private
     * @param {?} filter
     * @return {?}
     */
    getFilterCriterionFromDxFilter(filter) {
        // tslint:disable-next-line: prefer-const
        let [filterName, operator, value] = filter;
        // get schema column object based on filterName(id).
        /** @type {?} */
        const schemaColumnObject = this.getPropertyById(filterName);
        // get filter operator options.
        /** @type {?} */
        const operatorOptions = this.getOperatorOptions(schemaColumnObject.dataType);
        // For 'anyof' operator, this means we are in multiselect list case.
        if (operator === 'anyof') {
            operator = '=';
            value = value.join('|');
        }
        if (operator === 'noneof') {
            operator = '<>';
            value = value.join('|');
        }
        // For 'between' operator we convert the date to ISOString and join the value using '~' sign.
        // This format is required by 'nwf-datetime-range-picker' component
        if (operator === 'between') {
            value = value.map((/**
             * @param {?} val
             * @return {?}
             */
            (val) => val.toISOString())).join('~');
        }
        // we return property, operator, value which is the format required by filterCriterionRow.
        return {
            property: schemaColumnObject.dataField,
            operator,
            value,
        };
    }
    /**
     * We find all columns on dx grid. Instead of using a method available on gridInstance directly, we use this way because
     * gridInstance.columnOption gives a complete column object (including default column settings),
     * whereas column object returned by gridInstance.option('columnms') only contains properties you explicitly set.
     * gridInstance.getVisibleColumns() method gives us the complete object but it only works for visible columns.
     * @private
     * @return {?}
     */
    getAllColumnsOnDxGrid() {
        /** @type {?} */
        const gridInstance = this.gridInstance;
        /** @type {?} */
        const columnsViaOptions = gridInstance && gridInstance.state().columns;
        if (columnsViaOptions) {
            return columnsViaOptions.map((/**
             * @param {?} __0
             * @return {?}
             */
            ({ dataField }) => gridInstance.columnOption(dataField)));
        }
        else {
            return;
        }
    }
}
NwfFilterComponent.decorators = [
    { type: Component, args: [{
                selector: 'nwf-grid-filter',
                template: "<div ngbDropdown class=\"d-inline-block\" autoClose=\"outside\">\n    <button type=\"button\" class=\"btn btn-sm btn-flat btn-icon is-caretless\" id=\"nwf-grid-filtering\" ngbDropdownToggle>\n        <nwf-icon iconClass=\"icon-action-filter\"></nwf-icon>\n        <span class=\"btn-icon-label\">{{'nwfjs:filterWidgetButtonLabel' | i18next}}</span>\n        <span class=\"badge badge-info ml-1\" *ngIf=\"nwfFilters.length > 0\">\n            {{nwfFilters.length}}\n        </span>\n    </button>\n    <div ngbDropdownMenu aria-labelledby=\"nwf-grid-filtering\" class=\"popover-body bg-white\">\n        <nwf-popover-title (onDismiss)=\"ngbDropDown.close()\"></nwf-popover-title>\n        <!-- begin filter popover -->\n        <div *ngIf=\"!filterableColumns\">\n            <div class=\"m-3\">{{'nwfjs:filterWidgetError' | i18next}}</div>\n        </div>\n        <form novalidate autocomplete=\"off\" *ngIf=\"filterableColumns && ngbDropDown.isOpen()\"\n            [formGroup]=\"filterFormGrp\" (ngSubmit)=\"apply()\">\n            <div formArrayName=\"criterion\" class=\"my-2\">\n                <nwf-grid-filter-criterion-row\n                    *ngFor=\"let criteria of filterFormGrp.get('criterion')['controls']; let i=index\"\n                    [dxColumns]=\"filterableColumns\" [criteria]=\"criteria\" (deleteCriterion)=\"deleteCriterion(i)\">\n                </nwf-grid-filter-criterion-row>\n            </div>\n\n            <button type=\"button\" *ngIf=\"filterFormGrp.get('criterion')['controls'].length < 4\"\n                class=\"btn btn-flat btn-icon\" (click)=\"addCriterion();$event.stopPropagation();\"\n                data-netapp-id=\"filter-widget-add-filter-btn\">\n                <nwf-icon iconClass=\"icon-action-add\"></nwf-icon>\n                <span class=\"btn-icon-label\">{{'nwfjs:filterWidgetAddFilter' | i18next}}</span>\n            </button>\n\n            <div class=\"d-flex mt-4\">\n                <button type=\"button\" data-netapp-id=\"filter-widget-remove-btn\" class=\"btn btn-link mr-auto\"\n                    (click)=\"reset()\" [disabled]=\"nwfFilters.length <= 0\"\n                    data-netapp-id=\"filter-widget-reset-btn\">{{'nwfjs:reset' | i18next}}\n                </button>\n\n                <button class=\"btn btn-primary\" type=\"submit\"\n                    data-netapp-id=\"filter-widget-apply-btn\">{{'nwfjs:filterWidgetApplyFilter' | i18next}}</button>\n            </div>\n        </form>\n        <!-- end filter popover -->\n    </div>\n</div>",
                styles: [".filter-badge-light{background:#1e4a93;color:#fff;font-weight:700}.btn-delete-criterion{height:35px}"]
            }] }
];
/** @nocollapse */
NwfFilterComponent.ctorParameters = () => [
    { type: FormBuilder }
];
NwfFilterComponent.propDecorators = {
    ngbDropDown: [{ type: ViewChild, args: [NgbDropdown, { static: false },] }],
    dataGrid: [{ type: Input }]
};
if (false) {
    /** @type {?} */
    NwfFilterComponent.prototype.ngbDropDown;
    /** @type {?} */
    NwfFilterComponent.prototype.dataGrid;
    /** @type {?} */
    NwfFilterComponent.prototype.nwfFilters;
    /** @type {?} */
    NwfFilterComponent.prototype.filterableColumns;
    /** @type {?} */
    NwfFilterComponent.prototype.filterFormGrp;
    /**
     * @type {?}
     * @private
     */
    NwfFilterComponent.prototype.config;
    /**
     * @type {?}
     * @private
     */
    NwfFilterComponent.prototype.gridInstance;
    /**
     * @type {?}
     * @private
     */
    NwfFilterComponent.prototype.initialColumns;
    /**
     * @type {?}
     * @private
     */
    NwfFilterComponent.prototype._fb;
}
//# sourceMappingURL=data:application/json;base64,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