"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var angular = require("angular");
var filterWidget_config_1 = require("./filterWidget-config");
/**
 * @description This is the model for the filtering widget.
 * This code contains two major arrays known as newCriteria and appliedCriteria.
 * appliedCriteria: array of criterion(filter) that finally is submitted(sent) to grid params as filter params.
 * newCriteria: array of criteria that we work with i.e.temporary array of criteria.
 * On Initial setup of filtering widget, appliedCriteria and newCriteria both are empty.
 * We add a new default criteria in newCriteria. Subsequent criteria addition and deletion happen on newCriteria array only.
 * Only on Apply(submit), newCriteria is copied into appliedCriteria.
 */
var FilterWidgetModel = /** @class */ (function () {
    function FilterWidgetModel(params) {
        // params are the list of items to be filtered(passed through directive).
        this.params = params;
        // this array contains all the filter criterion that are being edited/added/deleted. (stores the temporary state of filter criterion).
        this.newCriteria = [];
        // this array contains all the filter criterion that are being applied/set. (stores the saved/final state of filter criterion).
        this.appliedCriteria = [];
        // create the config so we can read them later;
        this.config = filterWidget_config_1.default();
    }
    /**
     * @description this method adds a new default criterion
     */
    FilterWidgetModel.prototype.addCriterion = function () {
        // initially every new criterion being added, defaults to the first possible value in params passed. Based on first param, we set the operator value.
        var criterion = {
            property: this.params[0],
            operator: this.getOperatorOptions(this.params[0].type).options[0],
            value: ''
        };
        // adding the new criterion to the new criteria array.
        this.newCriteria.push(criterion);
    };
    /**
     * @description delete a criterion.
     * @param id
     */
    FilterWidgetModel.prototype.deleteCriterion = function (criterion) {
        var indexOf = this.newCriteria.indexOf(criterion);
        this.newCriteria.splice(indexOf, 1); //splice the criterion out because we loop on the array and need the length to be equal to the count.
    };
    /**
     * @description method to delete the applied criteria (used in reset).
     */
    FilterWidgetModel.prototype.deleteAppliedCriteria = function () {
        //setting to empty to delete the applied criteria.
        this.appliedCriteria = [];
        //setting new criteria to empty, to handle the case when reset is clicked during interaction(edit phase).
        this.newCriteria = [];
    };
    /**
     * @description method to set the operator and value of a particular criterion when property dropdown value is changed.
     * @param id
     */
    FilterWidgetModel.prototype.updateCriterion = function (criterion) {
        // if the criterion exists, we reset the operator and value.
        if (criterion) {
            criterion.operator = this.getOperatorOptions(criterion.property.type).options[0];
            criterion.value = "";
        }
        else {
            console.log("ERROR : criterion not found");
        }
    };
    /**
     * @description method to generate new criterion.
     * This method is the start point for the entire filter widget logic.
     */
    FilterWidgetModel.prototype.generateNewCriteria = function () {
        var my = this;
        // check to see if an already applied criteria exists.
        if (this.appliedCriteria.length > 0) {
            // We create a copy of the applied criteria in the new criteria
            // Now, new criteria is the copy we play with, for all the interaction in the widget.
            // Applied criteria is touched/changed only when we Apply is clicked.
            this.newCriteria = angular.merge([], this.appliedCriteria);
            // iterating over new criteria and setting the property and operator from actual params being passed to have the same state.
            // As initially when we started filter widget, criteria and property were fetched from the actual params being passed in.
            this.newCriteria.forEach(function (criterion) {
                criterion.property = my.getProperty(criterion.property.id);
                criterion.operator = findItemByName(my.getOperatorOptions(criterion.property.type).options, criterion.operator.name);
            });
        }
        else {
            // There is no already applied criteria. Means filter is in reset state (first state).
            // set the new criterion to empty if it already exists.
            this.newCriteria = [];
            //Add the very first criterion to be displayed.
            this.addCriterion();
        }
    };
    /**
     * @description method to return the current criterion (edit mode).
     * @returns {Array}
     */
    FilterWidgetModel.prototype.getCriteria = function () {
        return this.newCriteria;
    };
    /**
     * @description method to cancel the applied criterion.
     */
    FilterWidgetModel.prototype.cancel = function () {
        //setting the new criterion to empty when cancel is clicked.
        // This means we don't want any interaction to persist.
        // So, we set the criterion to empty that contains the current state of all the criterions.
        this.newCriteria = [];
        // call generateCriteria that handles the state of the widget.
        this.generateNewCriteria();
    };
    /**
     * @description method to Apply(submit) the new criterion.
     */
    FilterWidgetModel.prototype.apply = function () {
        // We set whatever criterion exists in new criteria back to applied criterion to be saved
        this.appliedCriteria = angular.copy(this.newCriteria);
    };
    /**
     * @description method that fetch the property from the original params passed into the widget.
     * @param id
     * @returns {*}
     */
    FilterWidgetModel.prototype.getProperty = function (id) {
        return findItemById(this.params, id);
    };
    /**
     * @description method that return the options from filter config bundle based on the type passed, returns an object containing inputType and options.
     * @param paramType
     * @returns {Object}
     */
    FilterWidgetModel.prototype.getOperatorOptions = function (paramType) {
        return this.config[paramType];
    };
    /**
     * @description: method to create the applied criteria from appliedFilters coming from filter widget consumer.
     * This allows to preset the filterWidget this it he VALUE of the filter widget.
     * @param appliedFilters comma separted list of filter params. this should be value of gridParams.filter.
     */
    FilterWidgetModel.prototype.createAppliedCriteria = function (appliedFilters) {
        var _this = this;
        //delete existing criteria. 
        //this.deleteAppliedCriteria();
        //pre-setting filter criteria to filter widget, if appliedFilters exist.
        var criterionArray = appliedFilters.map(function (filter) {
            var _a = filter.split(','), filterName = _a[0], operator = _a[1], value = _a[2];
            //create criterion from appliedFilters.
            var criterionProperty = _this.getProperty(filterName);
            var criterion = {
                // id: this.counter++,
                property: criterionProperty,
                operator: findOperator(_this.getOperatorOptions(criterionProperty.type).options, operator),
                value: value
            };
            return criterion;
        });
        //setting the criteria to appliedCriteria.
        this.appliedCriteria = criterionArray;
        // call generateCriteria that handles the state of the widget.
        this.generateNewCriteria();
    };
    return FilterWidgetModel;
}());
exports.default = FilterWidgetModel;
/**
 * @description searches an item in array by its Id.
 * @param itemsArray
 * @param objid
 * @returns {*}
 */
function findItemById(itemsArray, objid) {
    var result = null;
    itemsArray.forEach(function (item) {
        if (item.id === objid) {
            result = item;
        }
    });
    return result;
}
/**
 * @description searches an item in the array by its name.
 * @param itemsArray
 * @param name
 * @returns {*}
 */
function findItemByName(itemsArray, name) {
    var result = null;
    itemsArray.forEach(function (item) {
        if (item.name === name) {
            result = item;
        }
    });
    return result;
}
/**
 * Function that finds the filter operator.
 * @param array
 * @param optionValue
 */
function findOperator(array, optionValue) {
    var result = null;
    array.forEach(function (item) {
        if (item.value === optionValue) {
            result = item;
        }
    });
    return result;
}
//# sourceMappingURL=filterWidget-model.js.map