import { Action, Store } from '@ngrx/store';
import { Observable } from 'rxjs';
import { CorrelationIdGenerator } from '../utils/correlation-id-generator';
import { EntityActionOptions } from '../actions/entity-action';
import { EntityCache } from '../reducers/entity-cache';
import { EntityDispatcherDefaultOptions } from './entity-dispatcher-default-options';
import { MergeStrategy } from '../actions/merge-strategy';
import { ChangeSet, ChangeSetItem } from '../actions/entity-cache-change-set';
import { EntityCacheQuerySet } from '../actions/entity-cache-action';
/**
 * Dispatches Entity Cache actions to the EntityCache reducer
 */
export declare class EntityCacheDispatcher {
    /** Generates correlation ids for query and save methods */
    private correlationIdGenerator;
    /**
     * Dispatcher options configure dispatcher behavior such as
     * whether add is optimistic or pessimistic by default.
     */
    private defaultDispatcherOptions;
    /** The store, scoped to the EntityCache */
    private store;
    /**
     * Actions scanned by the store after it processed them with reducers.
     * A replay observable of the most recent action reduced by the store.
     */
    reducedActions$: Observable<Action>;
    private raSubscription;
    constructor(
    /** Generates correlation ids for query and save methods */
    correlationIdGenerator: CorrelationIdGenerator, 
    /**
     * Dispatcher options configure dispatcher behavior such as
     * whether add is optimistic or pessimistic by default.
     */
    defaultDispatcherOptions: EntityDispatcherDefaultOptions, 
    /** Actions scanned by the store after it processed them with reducers. */
    scannedActions$: Observable<Action>, 
    /** The store, scoped to the EntityCache */
    store: Store<EntityCache>);
    /**
     * Dispatch an Action to the store.
     * @param action the Action
     * @returns the dispatched Action
     */
    dispatch(action: Action): Action;
    /**
     * Dispatch action to cancel the saveEntities request with matching correlation id.
     * @param correlationId The correlation id for the corresponding action
     * @param [reason] explains why canceled and by whom.
     * @param [entityNames] array of entity names so can turn off loading flag for their collections.
     * @param [tag] tag to identify the operation from the app perspective.
     */
    cancelSaveEntities(correlationId: any, reason?: string, entityNames?: string[], tag?: string): void;
    /** Clear the named entity collections in cache
     * @param [collections] Array of names of the collections to clear.
     * If empty array, does nothing. If null/undefined/no array, clear all collections.
     * @param [tag] tag to identify the operation from the app perspective.
     */
    clearCollections(collections?: string[], tag?: string): void;
    /**
     * Load multiple entity collections at the same time.
     * before any selectors$ observables emit.
     * @param collections The collections to load, typically the result of a query.
     * @param [tag] tag to identify the operation from the app perspective.
     * in the form of a map of entity collections.
     */
    loadCollections(collections: EntityCacheQuerySet, tag?: string): void;
    /**
     * Merges entities from a query result
     * that returned entities from multiple collections.
     * Corresponding entity cache reducer should add and update all collections
     * at the same time, before any selectors$ observables emit.
     * @param querySet The result of the query in the form of a map of entity collections.
     * These are the entity data to merge into the respective collections.
     * @param mergeStrategy How to merge a queried entity when it is already in the collection.
     * The default is MergeStrategy.PreserveChanges
     * @param [tag] tag to identify the operation from the app perspective.
     */
    mergeQuerySet(querySet: EntityCacheQuerySet, mergeStrategy?: MergeStrategy, tag?: string): void;
    /**
     * Create entity cache action for replacing the entire entity cache.
     * Dangerous because brute force but useful as when re-hydrating an EntityCache
     * from local browser storage when the application launches.
     * @param cache New state of the entity cache
     * @param [tag] tag to identify the operation from the app perspective.
     */
    setEntityCache(cache: EntityCache, tag?: string): void;
    /**
     * Dispatch action to save multiple entity changes to remote storage.
     * Relies on an Ngrx Effect such as EntityEffects.saveEntities$.
     * Important: only call if your server supports the SaveEntities protocol
     * through your EntityDataService.saveEntities method.
     * @param changes Either the entities to save, as an array of {ChangeSetItem}, or
     * a ChangeSet that holds such changes.
     * @param url The server url which receives the save request
     * @param [options] options such as tag, correlationId, isOptimistic, and mergeStrategy.
     * These values are defaulted if not supplied.
     * @returns A terminating Observable<ChangeSet> with data returned from the server
     * after server reports successful save OR the save error.
     * TODO: should return the matching entities from cache rather than the raw server data.
     */
    saveEntities(changes: ChangeSetItem[] | ChangeSet, url: string, options?: EntityActionOptions): Observable<ChangeSet>;
    /**
     * Return Observable of data from the server-success SaveEntities action with
     * the given Correlation Id, after that action was processed by the ngrx store.
     * or else put the server error on the Observable error channel.
     * @param crid The correlationId for both the save and response actions.
     */
    private getSaveEntitiesResponseData$;
}
