/**
 * # UI-Router for Angular (v2+)
 *
 * - [@uirouter/angular home page](https://ui-router.github.io/ng2)
 * - [tutorials](https://ui-router.github.io/tutorial/ng2/helloworld)
 * - [quick start repository](http://github.com/ui-router/quickstart-ng2)
 *
 * Getting started:
 *
 * - Use npm. Add a dependency on latest `@uirouter/angular`
 * - Import UI-Router classes directly from `"@uirouter/angular"`
 *
 * ```js
 * import {StateRegistry} from "@uirouter/angular";
 * ```
 *
 * - Create application states (as defined by [[Ng2StateDeclaration]]).
 *
 * ```js
 * export let state1: Ng2StateDeclaration = {
 *   name: 'state1',
 *   component: State1Component,
 *   url: '/one'
 * }
 *
 * export let state2: Ng2StateDeclaration = {
 *   name: 'state2',
 *   component: State2Component,
 *   url: '/two'
 * }
 * ```
 *
 * - Import a [[UIRouterModule.forChild]] module into your feature `NgModule`s.
 *
 * ```js
 * @ NgModule({
 *   imports: [
 *     SharedModule,
 *     UIRouterModule.forChild({ states: [state1, state2 ] })
 *   ],
 *   declarations: [
 *     State1Component,
 *     State2Component,
 *   ]
 * })
 * export class MyFeatureModule {}
 * ```
 *
 * - Import a [[UIRouterModule.forRoot]] module into your application root `NgModule`
 * - Either bootstrap a [[UIView]] component, or add a `<ui-view></ui-view>` viewport to your root component.
 *
 * ```js
 * @ NgModule({
 *   imports: [
 *     BrowserModule,
 *     UIRouterModule.forRoot({ states: [ homeState ] }),
 *     MyFeatureModule,
 *   ],
 *   declarations: [
 *     HomeComponent
 *   ]
 *   bootstrap: [ UIView ]
 * })
 * class RootAppModule {}
 *
 * browserPlatformDynamic.bootstrapModule(RootAppModule);
 * ```
 *
 * - Optionally specify a configuration class [[ChildModule.configClass]] for any module
 * to perform any router configuration during bootstrap or lazyload.
 * Pass the class to [[UIRouterModule.forRoot]] or [[UIRouterModule.forChild]].
 *
 * ```js
 * import {UIRouter} from "@uirouter/angular";
 *
 * @ Injectable()
 * export class MyUIRouterConfig {
 *   // Constructor is injectable
 *   constructor(uiRouter: UIRouter) {
 *     uiRouter.urlMatcherFactory.type('datetime', myDateTimeParamType);
 *   }
 * }
 * ```
 *
 * @preferred @module ng2
 */
/** */
import { Injector, PLATFORM_ID, APP_INITIALIZER } from '@angular/core';
import { isPlatformBrowser } from '@angular/common';
import { UIRouter, StateRegistry, StateService, TransitionService, UrlMatcherFactory, UrlRouter, ViewService, UrlService, UIRouterGlobals, services, Resolvable, NATIVE_INJECTOR_TOKEN, } from '@uirouter/core';
import { UIView } from './directives/uiView';
import { ng2ViewsBuilder, Ng2ViewConfig } from './statebuilders/views';
import { applyRootModuleConfig, applyModuleConfig } from './uiRouterConfig';
import { UIROUTER_ROOT_MODULE, UIROUTER_MODULE_TOKEN } from './uiRouterNgModule';
import { servicesPlugin } from '@uirouter/core';
import { ng2LazyLoadBuilder } from './statebuilders/lazyLoad';
import { UIRouterRx } from '@uirouter/rx';
import { LocationStrategy } from '@angular/common';
import { Ng2LocationServices } from './location/locationService';
import { Ng2LocationConfig } from './location/locationConfig';
/**
 * This is a factory function for a UIRouter instance
 *
 * Creates a UIRouter instance and configures it for Angular, then invokes router bootstrap.
 * This function is used as an Angular `useFactory` Provider.
 */
export function uiRouterFactory(locationStrategy, rootModules, modules, injector) {
    if (rootModules.length !== 1) {
        throw new Error("Exactly one UIRouterModule.forRoot() should be in the bootstrapped app module's imports: []");
    }
    // ----------------- Create router -----------------
    // Create a new ng2 UIRouter and configure it for ng2
    const router = new UIRouter();
    // Add RxJS plugin
    router.plugin(UIRouterRx);
    // Add $q-like and $injector-like service APIs
    router.plugin(servicesPlugin);
    // ----------------- Monkey Patches ----------------
    // Monkey patch the services.$injector to use the root ng2 Injector
    services.$injector.get = injector.get.bind(injector);
    // ----------------- Configure for ng2 -------------
    router.locationService = new Ng2LocationServices(router, locationStrategy, isPlatformBrowser(injector.get(PLATFORM_ID)));
    router.locationConfig = new Ng2LocationConfig(router, locationStrategy);
    // Apply ng2 ui-view handling code
    const viewConfigFactory = (path, config) => new Ng2ViewConfig(path, config);
    router.viewService._pluginapi._viewConfigFactory('ng2', viewConfigFactory);
    // Apply statebuilder decorator for ng2 NgModule registration
    const registry = router.stateRegistry;
    registry.decorator('views', ng2ViewsBuilder);
    registry.decorator('lazyLoad', ng2LazyLoadBuilder);
    // Prep the tree of NgModule by placing the root NgModule's Injector on the root state.
    const ng2InjectorResolvable = Resolvable.fromData(NATIVE_INJECTOR_TOKEN, injector);
    registry.root().resolvables.push(ng2InjectorResolvable);
    // Auto-flush the parameter type queue
    router.urlMatcherFactory.$get();
    // ----------------- Initialize router -------------
    rootModules.forEach(moduleConfig => applyRootModuleConfig(router, injector, moduleConfig));
    modules.forEach(moduleConfig => applyModuleConfig(router, injector, moduleConfig));
    return router;
}
// Start monitoring the URL when the app starts
export function appInitializer(router) {
    return () => {
        if (!router.urlRouter.interceptDeferred) {
            router.urlService.listen();
            router.urlService.sync();
        }
    };
}
export function parentUIViewInjectFactory(r) {
    return { fqn: null, context: r.root() };
}
export const _UIROUTER_INSTANCE_PROVIDERS = [
    {
        provide: UIRouter,
        useFactory: uiRouterFactory,
        deps: [LocationStrategy, UIROUTER_ROOT_MODULE, UIROUTER_MODULE_TOKEN, Injector],
    },
    { provide: UIView.PARENT_INJECT, useFactory: parentUIViewInjectFactory, deps: [StateRegistry] },
    { provide: APP_INITIALIZER, useFactory: appInitializer, deps: [UIRouter], multi: true },
];
export function fnStateService(r) {
    return r.stateService;
}
export function fnTransitionService(r) {
    return r.transitionService;
}
export function fnUrlMatcherFactory(r) {
    return r.urlMatcherFactory;
}
export function fnUrlRouter(r) {
    return r.urlRouter;
}
export function fnUrlService(r) {
    return r.urlService;
}
export function fnViewService(r) {
    return r.viewService;
}
export function fnStateRegistry(r) {
    return r.stateRegistry;
}
export function fnGlobals(r) {
    return r.globals;
}
export const _UIROUTER_SERVICE_PROVIDERS = [
    { provide: StateService, useFactory: fnStateService, deps: [UIRouter] },
    { provide: TransitionService, useFactory: fnTransitionService, deps: [UIRouter] },
    { provide: UrlMatcherFactory, useFactory: fnUrlMatcherFactory, deps: [UIRouter] },
    { provide: UrlRouter, useFactory: fnUrlRouter, deps: [UIRouter] },
    { provide: UrlService, useFactory: fnUrlService, deps: [UIRouter] },
    { provide: ViewService, useFactory: fnViewService, deps: [UIRouter] },
    { provide: StateRegistry, useFactory: fnStateRegistry, deps: [UIRouter] },
    { provide: UIRouterGlobals, useFactory: fnGlobals, deps: [UIRouter] },
];
/**
 * The UI-Router providers, for use in your application bootstrap
 *
 * @deprecated use [[UIRouterModule.forRoot]]
 */
export const UIROUTER_PROVIDERS = _UIROUTER_INSTANCE_PROVIDERS.concat(_UIROUTER_SERVICE_PROVIDERS);
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJvdmlkZXJzLmpzIiwic291cmNlUm9vdCI6Im5nOi8vQHVpcm91dGVyL2FuZ3VsYXIvIiwic291cmNlcyI6WyJwcm92aWRlcnMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FxRkc7QUFDSCxNQUFNO0FBQ04sT0FBTyxFQUFFLFFBQVEsRUFBWSxXQUFXLEVBQUUsZUFBZSxFQUFFLE1BQU0sZUFBZSxDQUFDO0FBQ2pGLE9BQU8sRUFBRSxpQkFBaUIsRUFBRSxNQUFNLGlCQUFpQixDQUFDO0FBQ3BELE9BQU8sRUFDTCxRQUFRLEVBRVIsYUFBYSxFQUNiLFlBQVksRUFDWixpQkFBaUIsRUFDakIsaUJBQWlCLEVBQ2pCLFNBQVMsRUFDVCxXQUFXLEVBQ1gsVUFBVSxFQUNWLGVBQWUsRUFDZixRQUFRLEVBQ1IsVUFBVSxFQUNWLHFCQUFxQixHQUN0QixNQUFNLGdCQUFnQixDQUFDO0FBQ3hCLE9BQU8sRUFBRSxNQUFNLEVBQXNCLE1BQU0scUJBQXFCLENBQUM7QUFDakUsT0FBTyxFQUFFLGVBQWUsRUFBRSxhQUFhLEVBQUUsTUFBTSx1QkFBdUIsQ0FBQztBQUV2RSxPQUFPLEVBQUUscUJBQXFCLEVBQUUsaUJBQWlCLEVBQUUsTUFBTSxrQkFBa0IsQ0FBQztBQUM1RSxPQUFPLEVBQTRCLG9CQUFvQixFQUFFLHFCQUFxQixFQUFFLE1BQU0sb0JBQW9CLENBQUM7QUFDM0csT0FBTyxFQUFFLGNBQWMsRUFBa0IsTUFBTSxnQkFBZ0IsQ0FBQztBQUNoRSxPQUFPLEVBQUUsa0JBQWtCLEVBQUUsTUFBTSwwQkFBMEIsQ0FBQztBQUM5RCxPQUFPLEVBQUUsVUFBVSxFQUFFLE1BQU0sY0FBYyxDQUFDO0FBQzFDLE9BQU8sRUFBRSxnQkFBZ0IsRUFBRSxNQUFNLGlCQUFpQixDQUFDO0FBQ25ELE9BQU8sRUFBRSxtQkFBbUIsRUFBRSxNQUFNLDRCQUE0QixDQUFDO0FBQ2pFLE9BQU8sRUFBRSxpQkFBaUIsRUFBRSxNQUFNLDJCQUEyQixDQUFDO0FBRTlEOzs7OztHQUtHO0FBQ0gsTUFBTSxVQUFVLGVBQWUsQ0FDN0IsZ0JBQWtDLEVBQ2xDLFdBQXlCLEVBQ3pCLE9BQXVCLEVBQ3ZCLFFBQWtCO0lBRWxCLElBQUksV0FBVyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7UUFDNUIsTUFBTSxJQUFJLEtBQUssQ0FBQyw2RkFBNkYsQ0FBQyxDQUFDO0tBQ2hIO0lBRUQsb0RBQW9EO0lBQ3BELHFEQUFxRDtJQUNyRCxNQUFNLE1BQU0sR0FBRyxJQUFJLFFBQVEsRUFBRSxDQUFDO0lBRTlCLGtCQUFrQjtJQUNsQixNQUFNLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBRTFCLDhDQUE4QztJQUM5QyxNQUFNLENBQUMsTUFBTSxDQUFpQixjQUFjLENBQUMsQ0FBQztJQUU5QyxvREFBb0Q7SUFDcEQsbUVBQW1FO0lBQ25FLFFBQVEsQ0FBQyxTQUFTLENBQUMsR0FBRyxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBRXJELG9EQUFvRDtJQUNwRCxNQUFNLENBQUMsZUFBZSxHQUFHLElBQUksbUJBQW1CLENBQzlDLE1BQU0sRUFDTixnQkFBZ0IsRUFDaEIsaUJBQWlCLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUM3QyxDQUFDO0lBQ0YsTUFBTSxDQUFDLGNBQWMsR0FBRyxJQUFJLGlCQUFpQixDQUFDLE1BQU0sRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO0lBRXhFLGtDQUFrQztJQUNsQyxNQUFNLGlCQUFpQixHQUFHLENBQUMsSUFBZ0IsRUFBRSxNQUEwQixFQUFFLEVBQUUsQ0FBQyxJQUFJLGFBQWEsQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDNUcsTUFBTSxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGlCQUFpQixDQUFDLENBQUM7SUFFM0UsNkRBQTZEO0lBQzdELE1BQU0sUUFBUSxHQUFHLE1BQU0sQ0FBQyxhQUFhLENBQUM7SUFDdEMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxPQUFPLEVBQUUsZUFBZSxDQUFDLENBQUM7SUFDN0MsUUFBUSxDQUFDLFNBQVMsQ0FBQyxVQUFVLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztJQUVuRCx1RkFBdUY7SUFDdkYsTUFBTSxxQkFBcUIsR0FBRyxVQUFVLENBQUMsUUFBUSxDQUFDLHFCQUFxQixFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ25GLFFBQVEsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFFeEQsc0NBQXNDO0lBQ3RDLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUVoQyxvREFBb0Q7SUFDcEQsV0FBVyxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLE1BQU0sRUFBRSxRQUFRLEVBQUUsWUFBWSxDQUFDLENBQUMsQ0FBQztJQUMzRixPQUFPLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsTUFBTSxFQUFFLFFBQVEsRUFBRSxZQUFZLENBQUMsQ0FBQyxDQUFDO0lBRW5GLE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUM7QUFFRCwrQ0FBK0M7QUFDL0MsTUFBTSxVQUFVLGNBQWMsQ0FBQyxNQUFnQjtJQUM3QyxPQUFPLEdBQUcsRUFBRTtRQUNWLElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLGlCQUFpQixFQUFFO1lBQ3ZDLE1BQU0sQ0FBQyxVQUFVLENBQUMsTUFBTSxFQUFFLENBQUM7WUFDM0IsTUFBTSxDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsQ0FBQztTQUMxQjtJQUNILENBQUMsQ0FBQztBQUNKLENBQUM7QUFFRCxNQUFNLFVBQVUseUJBQXlCLENBQUMsQ0FBZ0I7SUFDeEQsT0FBTyxFQUFFLEdBQUcsRUFBRSxJQUFJLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBd0IsQ0FBQztBQUNoRSxDQUFDO0FBRUQsTUFBTSxDQUFDLE1BQU0sNEJBQTRCLEdBQWU7SUFDdEQ7UUFDRSxPQUFPLEVBQUUsUUFBUTtRQUNqQixVQUFVLEVBQUUsZUFBZTtRQUMzQixJQUFJLEVBQUUsQ0FBQyxnQkFBZ0IsRUFBRSxvQkFBb0IsRUFBRSxxQkFBcUIsRUFBRSxRQUFRLENBQUM7S0FDaEY7SUFDRCxFQUFFLE9BQU8sRUFBRSxNQUFNLENBQUMsYUFBYSxFQUFFLFVBQVUsRUFBRSx5QkFBeUIsRUFBRSxJQUFJLEVBQUUsQ0FBQyxhQUFhLENBQUMsRUFBRTtJQUMvRixFQUFFLE9BQU8sRUFBRSxlQUFlLEVBQUUsVUFBVSxFQUFFLGNBQWMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxRQUFRLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxFQUFFO0NBQ3hGLENBQUM7QUFFRixNQUFNLFVBQVUsY0FBYyxDQUFDLENBQVc7SUFDeEMsT0FBTyxDQUFDLENBQUMsWUFBWSxDQUFDO0FBQ3hCLENBQUM7QUFDRCxNQUFNLFVBQVUsbUJBQW1CLENBQUMsQ0FBVztJQUM3QyxPQUFPLENBQUMsQ0FBQyxpQkFBaUIsQ0FBQztBQUM3QixDQUFDO0FBQ0QsTUFBTSxVQUFVLG1CQUFtQixDQUFDLENBQVc7SUFDN0MsT0FBTyxDQUFDLENBQUMsaUJBQWlCLENBQUM7QUFDN0IsQ0FBQztBQUNELE1BQU0sVUFBVSxXQUFXLENBQUMsQ0FBVztJQUNyQyxPQUFPLENBQUMsQ0FBQyxTQUFTLENBQUM7QUFDckIsQ0FBQztBQUNELE1BQU0sVUFBVSxZQUFZLENBQUMsQ0FBVztJQUN0QyxPQUFPLENBQUMsQ0FBQyxVQUFVLENBQUM7QUFDdEIsQ0FBQztBQUNELE1BQU0sVUFBVSxhQUFhLENBQUMsQ0FBVztJQUN2QyxPQUFPLENBQUMsQ0FBQyxXQUFXLENBQUM7QUFDdkIsQ0FBQztBQUNELE1BQU0sVUFBVSxlQUFlLENBQUMsQ0FBVztJQUN6QyxPQUFPLENBQUMsQ0FBQyxhQUFhLENBQUM7QUFDekIsQ0FBQztBQUNELE1BQU0sVUFBVSxTQUFTLENBQUMsQ0FBTTtJQUM5QixPQUFPLENBQUMsQ0FBQyxPQUFPLENBQUM7QUFDbkIsQ0FBQztBQUVELE1BQU0sQ0FBQyxNQUFNLDJCQUEyQixHQUFlO0lBQ3JELEVBQUUsT0FBTyxFQUFFLFlBQVksRUFBRSxVQUFVLEVBQUUsY0FBYyxFQUFFLElBQUksRUFBRSxDQUFDLFFBQVEsQ0FBQyxFQUFFO0lBQ3ZFLEVBQUUsT0FBTyxFQUFFLGlCQUFpQixFQUFFLFVBQVUsRUFBRSxtQkFBbUIsRUFBRSxJQUFJLEVBQUUsQ0FBQyxRQUFRLENBQUMsRUFBRTtJQUNqRixFQUFFLE9BQU8sRUFBRSxpQkFBaUIsRUFBRSxVQUFVLEVBQUUsbUJBQW1CLEVBQUUsSUFBSSxFQUFFLENBQUMsUUFBUSxDQUFDLEVBQUU7SUFDakYsRUFBRSxPQUFPLEVBQUUsU0FBUyxFQUFFLFVBQVUsRUFBRSxXQUFXLEVBQUUsSUFBSSxFQUFFLENBQUMsUUFBUSxDQUFDLEVBQUU7SUFDakUsRUFBRSxPQUFPLEVBQUUsVUFBVSxFQUFFLFVBQVUsRUFBRSxZQUFZLEVBQUUsSUFBSSxFQUFFLENBQUMsUUFBUSxDQUFDLEVBQUU7SUFDbkUsRUFBRSxPQUFPLEVBQUUsV0FBVyxFQUFFLFVBQVUsRUFBRSxhQUFhLEVBQUUsSUFBSSxFQUFFLENBQUMsUUFBUSxDQUFDLEVBQUU7SUFDckUsRUFBRSxPQUFPLEVBQUUsYUFBYSxFQUFFLFVBQVUsRUFBRSxlQUFlLEVBQUUsSUFBSSxFQUFFLENBQUMsUUFBUSxDQUFDLEVBQUU7SUFDekUsRUFBRSxPQUFPLEVBQUUsZUFBZSxFQUFFLFVBQVUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUMsUUFBUSxDQUFDLEVBQUU7Q0FDdEUsQ0FBQztBQUVGOzs7O0dBSUc7QUFDSCxNQUFNLENBQUMsTUFBTSxrQkFBa0IsR0FBZSw0QkFBNEIsQ0FBQyxNQUFNLENBQUMsMkJBQTJCLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIyBVSS1Sb3V0ZXIgZm9yIEFuZ3VsYXIgKHYyKylcbiAqXG4gKiAtIFtAdWlyb3V0ZXIvYW5ndWxhciBob21lIHBhZ2VdKGh0dHBzOi8vdWktcm91dGVyLmdpdGh1Yi5pby9uZzIpXG4gKiAtIFt0dXRvcmlhbHNdKGh0dHBzOi8vdWktcm91dGVyLmdpdGh1Yi5pby90dXRvcmlhbC9uZzIvaGVsbG93b3JsZClcbiAqIC0gW3F1aWNrIHN0YXJ0IHJlcG9zaXRvcnldKGh0dHA6Ly9naXRodWIuY29tL3VpLXJvdXRlci9xdWlja3N0YXJ0LW5nMilcbiAqXG4gKiBHZXR0aW5nIHN0YXJ0ZWQ6XG4gKlxuICogLSBVc2UgbnBtLiBBZGQgYSBkZXBlbmRlbmN5IG9uIGxhdGVzdCBgQHVpcm91dGVyL2FuZ3VsYXJgXG4gKiAtIEltcG9ydCBVSS1Sb3V0ZXIgY2xhc3NlcyBkaXJlY3RseSBmcm9tIGBcIkB1aXJvdXRlci9hbmd1bGFyXCJgXG4gKlxuICogYGBganNcbiAqIGltcG9ydCB7U3RhdGVSZWdpc3RyeX0gZnJvbSBcIkB1aXJvdXRlci9hbmd1bGFyXCI7XG4gKiBgYGBcbiAqXG4gKiAtIENyZWF0ZSBhcHBsaWNhdGlvbiBzdGF0ZXMgKGFzIGRlZmluZWQgYnkgW1tOZzJTdGF0ZURlY2xhcmF0aW9uXV0pLlxuICpcbiAqIGBgYGpzXG4gKiBleHBvcnQgbGV0IHN0YXRlMTogTmcyU3RhdGVEZWNsYXJhdGlvbiA9IHtcbiAqICAgbmFtZTogJ3N0YXRlMScsXG4gKiAgIGNvbXBvbmVudDogU3RhdGUxQ29tcG9uZW50LFxuICogICB1cmw6ICcvb25lJ1xuICogfVxuICpcbiAqIGV4cG9ydCBsZXQgc3RhdGUyOiBOZzJTdGF0ZURlY2xhcmF0aW9uID0ge1xuICogICBuYW1lOiAnc3RhdGUyJyxcbiAqICAgY29tcG9uZW50OiBTdGF0ZTJDb21wb25lbnQsXG4gKiAgIHVybDogJy90d28nXG4gKiB9XG4gKiBgYGBcbiAqXG4gKiAtIEltcG9ydCBhIFtbVUlSb3V0ZXJNb2R1bGUuZm9yQ2hpbGRdXSBtb2R1bGUgaW50byB5b3VyIGZlYXR1cmUgYE5nTW9kdWxlYHMuXG4gKlxuICogYGBganNcbiAqIEAgTmdNb2R1bGUoe1xuICogICBpbXBvcnRzOiBbXG4gKiAgICAgU2hhcmVkTW9kdWxlLFxuICogICAgIFVJUm91dGVyTW9kdWxlLmZvckNoaWxkKHsgc3RhdGVzOiBbc3RhdGUxLCBzdGF0ZTIgXSB9KVxuICogICBdLFxuICogICBkZWNsYXJhdGlvbnM6IFtcbiAqICAgICBTdGF0ZTFDb21wb25lbnQsXG4gKiAgICAgU3RhdGUyQ29tcG9uZW50LFxuICogICBdXG4gKiB9KVxuICogZXhwb3J0IGNsYXNzIE15RmVhdHVyZU1vZHVsZSB7fVxuICogYGBgXG4gKlxuICogLSBJbXBvcnQgYSBbW1VJUm91dGVyTW9kdWxlLmZvclJvb3RdXSBtb2R1bGUgaW50byB5b3VyIGFwcGxpY2F0aW9uIHJvb3QgYE5nTW9kdWxlYFxuICogLSBFaXRoZXIgYm9vdHN0cmFwIGEgW1tVSVZpZXddXSBjb21wb25lbnQsIG9yIGFkZCBhIGA8dWktdmlldz48L3VpLXZpZXc+YCB2aWV3cG9ydCB0byB5b3VyIHJvb3QgY29tcG9uZW50LlxuICpcbiAqIGBgYGpzXG4gKiBAIE5nTW9kdWxlKHtcbiAqICAgaW1wb3J0czogW1xuICogICAgIEJyb3dzZXJNb2R1bGUsXG4gKiAgICAgVUlSb3V0ZXJNb2R1bGUuZm9yUm9vdCh7IHN0YXRlczogWyBob21lU3RhdGUgXSB9KSxcbiAqICAgICBNeUZlYXR1cmVNb2R1bGUsXG4gKiAgIF0sXG4gKiAgIGRlY2xhcmF0aW9uczogW1xuICogICAgIEhvbWVDb21wb25lbnRcbiAqICAgXVxuICogICBib290c3RyYXA6IFsgVUlWaWV3IF1cbiAqIH0pXG4gKiBjbGFzcyBSb290QXBwTW9kdWxlIHt9XG4gKlxuICogYnJvd3NlclBsYXRmb3JtRHluYW1pYy5ib290c3RyYXBNb2R1bGUoUm9vdEFwcE1vZHVsZSk7XG4gKiBgYGBcbiAqXG4gKiAtIE9wdGlvbmFsbHkgc3BlY2lmeSBhIGNvbmZpZ3VyYXRpb24gY2xhc3MgW1tDaGlsZE1vZHVsZS5jb25maWdDbGFzc11dIGZvciBhbnkgbW9kdWxlXG4gKiB0byBwZXJmb3JtIGFueSByb3V0ZXIgY29uZmlndXJhdGlvbiBkdXJpbmcgYm9vdHN0cmFwIG9yIGxhenlsb2FkLlxuICogUGFzcyB0aGUgY2xhc3MgdG8gW1tVSVJvdXRlck1vZHVsZS5mb3JSb290XV0gb3IgW1tVSVJvdXRlck1vZHVsZS5mb3JDaGlsZF1dLlxuICpcbiAqIGBgYGpzXG4gKiBpbXBvcnQge1VJUm91dGVyfSBmcm9tIFwiQHVpcm91dGVyL2FuZ3VsYXJcIjtcbiAqXG4gKiBAIEluamVjdGFibGUoKVxuICogZXhwb3J0IGNsYXNzIE15VUlSb3V0ZXJDb25maWcge1xuICogICAvLyBDb25zdHJ1Y3RvciBpcyBpbmplY3RhYmxlXG4gKiAgIGNvbnN0cnVjdG9yKHVpUm91dGVyOiBVSVJvdXRlcikge1xuICogICAgIHVpUm91dGVyLnVybE1hdGNoZXJGYWN0b3J5LnR5cGUoJ2RhdGV0aW1lJywgbXlEYXRlVGltZVBhcmFtVHlwZSk7XG4gKiAgIH1cbiAqIH1cbiAqIGBgYFxuICpcbiAqIEBwcmVmZXJyZWQgQG1vZHVsZSBuZzJcbiAqL1xuLyoqICovXG5pbXBvcnQgeyBJbmplY3RvciwgUHJvdmlkZXIsIFBMQVRGT1JNX0lELCBBUFBfSU5JVElBTElaRVIgfSBmcm9tICdAYW5ndWxhci9jb3JlJztcbmltcG9ydCB7IGlzUGxhdGZvcm1Ccm93c2VyIH0gZnJvbSAnQGFuZ3VsYXIvY29tbW9uJztcbmltcG9ydCB7XG4gIFVJUm91dGVyLFxuICBQYXRoTm9kZSxcbiAgU3RhdGVSZWdpc3RyeSxcbiAgU3RhdGVTZXJ2aWNlLFxuICBUcmFuc2l0aW9uU2VydmljZSxcbiAgVXJsTWF0Y2hlckZhY3RvcnksXG4gIFVybFJvdXRlcixcbiAgVmlld1NlcnZpY2UsXG4gIFVybFNlcnZpY2UsXG4gIFVJUm91dGVyR2xvYmFscyxcbiAgc2VydmljZXMsXG4gIFJlc29sdmFibGUsXG4gIE5BVElWRV9JTkpFQ1RPUl9UT0tFTixcbn0gZnJvbSAnQHVpcm91dGVyL2NvcmUnO1xuaW1wb3J0IHsgVUlWaWV3LCBQYXJlbnRVSVZpZXdJbmplY3QgfSBmcm9tICcuL2RpcmVjdGl2ZXMvdWlWaWV3JztcbmltcG9ydCB7IG5nMlZpZXdzQnVpbGRlciwgTmcyVmlld0NvbmZpZyB9IGZyb20gJy4vc3RhdGVidWlsZGVycy92aWV3cyc7XG5pbXBvcnQgeyBOZzJWaWV3RGVjbGFyYXRpb24gfSBmcm9tICcuL2ludGVyZmFjZSc7XG5pbXBvcnQgeyBhcHBseVJvb3RNb2R1bGVDb25maWcsIGFwcGx5TW9kdWxlQ29uZmlnIH0gZnJvbSAnLi91aVJvdXRlckNvbmZpZyc7XG5pbXBvcnQgeyBSb290TW9kdWxlLCBTdGF0ZXNNb2R1bGUsIFVJUk9VVEVSX1JPT1RfTU9EVUxFLCBVSVJPVVRFUl9NT0RVTEVfVE9LRU4gfSBmcm9tICcuL3VpUm91dGVyTmdNb2R1bGUnO1xuaW1wb3J0IHsgc2VydmljZXNQbHVnaW4sIFNlcnZpY2VzUGx1Z2luIH0gZnJvbSAnQHVpcm91dGVyL2NvcmUnO1xuaW1wb3J0IHsgbmcyTGF6eUxvYWRCdWlsZGVyIH0gZnJvbSAnLi9zdGF0ZWJ1aWxkZXJzL2xhenlMb2FkJztcbmltcG9ydCB7IFVJUm91dGVyUnggfSBmcm9tICdAdWlyb3V0ZXIvcngnO1xuaW1wb3J0IHsgTG9jYXRpb25TdHJhdGVneSB9IGZyb20gJ0Bhbmd1bGFyL2NvbW1vbic7XG5pbXBvcnQgeyBOZzJMb2NhdGlvblNlcnZpY2VzIH0gZnJvbSAnLi9sb2NhdGlvbi9sb2NhdGlvblNlcnZpY2UnO1xuaW1wb3J0IHsgTmcyTG9jYXRpb25Db25maWcgfSBmcm9tICcuL2xvY2F0aW9uL2xvY2F0aW9uQ29uZmlnJztcblxuLyoqXG4gKiBUaGlzIGlzIGEgZmFjdG9yeSBmdW5jdGlvbiBmb3IgYSBVSVJvdXRlciBpbnN0YW5jZVxuICpcbiAqIENyZWF0ZXMgYSBVSVJvdXRlciBpbnN0YW5jZSBhbmQgY29uZmlndXJlcyBpdCBmb3IgQW5ndWxhciwgdGhlbiBpbnZva2VzIHJvdXRlciBib290c3RyYXAuXG4gKiBUaGlzIGZ1bmN0aW9uIGlzIHVzZWQgYXMgYW4gQW5ndWxhciBgdXNlRmFjdG9yeWAgUHJvdmlkZXIuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiB1aVJvdXRlckZhY3RvcnkoXG4gIGxvY2F0aW9uU3RyYXRlZ3k6IExvY2F0aW9uU3RyYXRlZ3ksXG4gIHJvb3RNb2R1bGVzOiBSb290TW9kdWxlW10sXG4gIG1vZHVsZXM6IFN0YXRlc01vZHVsZVtdLFxuICBpbmplY3RvcjogSW5qZWN0b3Jcbikge1xuICBpZiAocm9vdE1vZHVsZXMubGVuZ3RoICE9PSAxKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKFwiRXhhY3RseSBvbmUgVUlSb3V0ZXJNb2R1bGUuZm9yUm9vdCgpIHNob3VsZCBiZSBpbiB0aGUgYm9vdHN0cmFwcGVkIGFwcCBtb2R1bGUncyBpbXBvcnRzOiBbXVwiKTtcbiAgfVxuXG4gIC8vIC0tLS0tLS0tLS0tLS0tLS0tIENyZWF0ZSByb3V0ZXIgLS0tLS0tLS0tLS0tLS0tLS1cbiAgLy8gQ3JlYXRlIGEgbmV3IG5nMiBVSVJvdXRlciBhbmQgY29uZmlndXJlIGl0IGZvciBuZzJcbiAgY29uc3Qgcm91dGVyID0gbmV3IFVJUm91dGVyKCk7XG5cbiAgLy8gQWRkIFJ4SlMgcGx1Z2luXG4gIHJvdXRlci5wbHVnaW4oVUlSb3V0ZXJSeCk7XG5cbiAgLy8gQWRkICRxLWxpa2UgYW5kICRpbmplY3Rvci1saWtlIHNlcnZpY2UgQVBJc1xuICByb3V0ZXIucGx1Z2luPFNlcnZpY2VzUGx1Z2luPihzZXJ2aWNlc1BsdWdpbik7XG5cbiAgLy8gLS0tLS0tLS0tLS0tLS0tLS0gTW9ua2V5IFBhdGNoZXMgLS0tLS0tLS0tLS0tLS0tLVxuICAvLyBNb25rZXkgcGF0Y2ggdGhlIHNlcnZpY2VzLiRpbmplY3RvciB0byB1c2UgdGhlIHJvb3QgbmcyIEluamVjdG9yXG4gIHNlcnZpY2VzLiRpbmplY3Rvci5nZXQgPSBpbmplY3Rvci5nZXQuYmluZChpbmplY3Rvcik7XG5cbiAgLy8gLS0tLS0tLS0tLS0tLS0tLS0gQ29uZmlndXJlIGZvciBuZzIgLS0tLS0tLS0tLS0tLVxuICByb3V0ZXIubG9jYXRpb25TZXJ2aWNlID0gbmV3IE5nMkxvY2F0aW9uU2VydmljZXMoXG4gICAgcm91dGVyLFxuICAgIGxvY2F0aW9uU3RyYXRlZ3ksXG4gICAgaXNQbGF0Zm9ybUJyb3dzZXIoaW5qZWN0b3IuZ2V0KFBMQVRGT1JNX0lEKSlcbiAgKTtcbiAgcm91dGVyLmxvY2F0aW9uQ29uZmlnID0gbmV3IE5nMkxvY2F0aW9uQ29uZmlnKHJvdXRlciwgbG9jYXRpb25TdHJhdGVneSk7XG5cbiAgLy8gQXBwbHkgbmcyIHVpLXZpZXcgaGFuZGxpbmcgY29kZVxuICBjb25zdCB2aWV3Q29uZmlnRmFjdG9yeSA9IChwYXRoOiBQYXRoTm9kZVtdLCBjb25maWc6IE5nMlZpZXdEZWNsYXJhdGlvbikgPT4gbmV3IE5nMlZpZXdDb25maWcocGF0aCwgY29uZmlnKTtcbiAgcm91dGVyLnZpZXdTZXJ2aWNlLl9wbHVnaW5hcGkuX3ZpZXdDb25maWdGYWN0b3J5KCduZzInLCB2aWV3Q29uZmlnRmFjdG9yeSk7XG5cbiAgLy8gQXBwbHkgc3RhdGVidWlsZGVyIGRlY29yYXRvciBmb3IgbmcyIE5nTW9kdWxlIHJlZ2lzdHJhdGlvblxuICBjb25zdCByZWdpc3RyeSA9IHJvdXRlci5zdGF0ZVJlZ2lzdHJ5O1xuICByZWdpc3RyeS5kZWNvcmF0b3IoJ3ZpZXdzJywgbmcyVmlld3NCdWlsZGVyKTtcbiAgcmVnaXN0cnkuZGVjb3JhdG9yKCdsYXp5TG9hZCcsIG5nMkxhenlMb2FkQnVpbGRlcik7XG5cbiAgLy8gUHJlcCB0aGUgdHJlZSBvZiBOZ01vZHVsZSBieSBwbGFjaW5nIHRoZSByb290IE5nTW9kdWxlJ3MgSW5qZWN0b3Igb24gdGhlIHJvb3Qgc3RhdGUuXG4gIGNvbnN0IG5nMkluamVjdG9yUmVzb2x2YWJsZSA9IFJlc29sdmFibGUuZnJvbURhdGEoTkFUSVZFX0lOSkVDVE9SX1RPS0VOLCBpbmplY3Rvcik7XG4gIHJlZ2lzdHJ5LnJvb3QoKS5yZXNvbHZhYmxlcy5wdXNoKG5nMkluamVjdG9yUmVzb2x2YWJsZSk7XG5cbiAgLy8gQXV0by1mbHVzaCB0aGUgcGFyYW1ldGVyIHR5cGUgcXVldWVcbiAgcm91dGVyLnVybE1hdGNoZXJGYWN0b3J5LiRnZXQoKTtcblxuICAvLyAtLS0tLS0tLS0tLS0tLS0tLSBJbml0aWFsaXplIHJvdXRlciAtLS0tLS0tLS0tLS0tXG4gIHJvb3RNb2R1bGVzLmZvckVhY2gobW9kdWxlQ29uZmlnID0+IGFwcGx5Um9vdE1vZHVsZUNvbmZpZyhyb3V0ZXIsIGluamVjdG9yLCBtb2R1bGVDb25maWcpKTtcbiAgbW9kdWxlcy5mb3JFYWNoKG1vZHVsZUNvbmZpZyA9PiBhcHBseU1vZHVsZUNvbmZpZyhyb3V0ZXIsIGluamVjdG9yLCBtb2R1bGVDb25maWcpKTtcblxuICByZXR1cm4gcm91dGVyO1xufVxuXG4vLyBTdGFydCBtb25pdG9yaW5nIHRoZSBVUkwgd2hlbiB0aGUgYXBwIHN0YXJ0c1xuZXhwb3J0IGZ1bmN0aW9uIGFwcEluaXRpYWxpemVyKHJvdXRlcjogVUlSb3V0ZXIpIHtcbiAgcmV0dXJuICgpID0+IHtcbiAgICBpZiAoIXJvdXRlci51cmxSb3V0ZXIuaW50ZXJjZXB0RGVmZXJyZWQpIHtcbiAgICAgIHJvdXRlci51cmxTZXJ2aWNlLmxpc3RlbigpO1xuICAgICAgcm91dGVyLnVybFNlcnZpY2Uuc3luYygpO1xuICAgIH1cbiAgfTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHBhcmVudFVJVmlld0luamVjdEZhY3RvcnkocjogU3RhdGVSZWdpc3RyeSkge1xuICByZXR1cm4geyBmcW46IG51bGwsIGNvbnRleHQ6IHIucm9vdCgpIH0gYXMgUGFyZW50VUlWaWV3SW5qZWN0O1xufVxuXG5leHBvcnQgY29uc3QgX1VJUk9VVEVSX0lOU1RBTkNFX1BST1ZJREVSUzogUHJvdmlkZXJbXSA9IFtcbiAge1xuICAgIHByb3ZpZGU6IFVJUm91dGVyLFxuICAgIHVzZUZhY3Rvcnk6IHVpUm91dGVyRmFjdG9yeSxcbiAgICBkZXBzOiBbTG9jYXRpb25TdHJhdGVneSwgVUlST1VURVJfUk9PVF9NT0RVTEUsIFVJUk9VVEVSX01PRFVMRV9UT0tFTiwgSW5qZWN0b3JdLFxuICB9LFxuICB7IHByb3ZpZGU6IFVJVmlldy5QQVJFTlRfSU5KRUNULCB1c2VGYWN0b3J5OiBwYXJlbnRVSVZpZXdJbmplY3RGYWN0b3J5LCBkZXBzOiBbU3RhdGVSZWdpc3RyeV0gfSxcbiAgeyBwcm92aWRlOiBBUFBfSU5JVElBTElaRVIsIHVzZUZhY3Rvcnk6IGFwcEluaXRpYWxpemVyLCBkZXBzOiBbVUlSb3V0ZXJdLCBtdWx0aTogdHJ1ZSB9LFxuXTtcblxuZXhwb3J0IGZ1bmN0aW9uIGZuU3RhdGVTZXJ2aWNlKHI6IFVJUm91dGVyKSB7XG4gIHJldHVybiByLnN0YXRlU2VydmljZTtcbn1cbmV4cG9ydCBmdW5jdGlvbiBmblRyYW5zaXRpb25TZXJ2aWNlKHI6IFVJUm91dGVyKSB7XG4gIHJldHVybiByLnRyYW5zaXRpb25TZXJ2aWNlO1xufVxuZXhwb3J0IGZ1bmN0aW9uIGZuVXJsTWF0Y2hlckZhY3RvcnkocjogVUlSb3V0ZXIpIHtcbiAgcmV0dXJuIHIudXJsTWF0Y2hlckZhY3Rvcnk7XG59XG5leHBvcnQgZnVuY3Rpb24gZm5VcmxSb3V0ZXIocjogVUlSb3V0ZXIpIHtcbiAgcmV0dXJuIHIudXJsUm91dGVyO1xufVxuZXhwb3J0IGZ1bmN0aW9uIGZuVXJsU2VydmljZShyOiBVSVJvdXRlcikge1xuICByZXR1cm4gci51cmxTZXJ2aWNlO1xufVxuZXhwb3J0IGZ1bmN0aW9uIGZuVmlld1NlcnZpY2UocjogVUlSb3V0ZXIpIHtcbiAgcmV0dXJuIHIudmlld1NlcnZpY2U7XG59XG5leHBvcnQgZnVuY3Rpb24gZm5TdGF0ZVJlZ2lzdHJ5KHI6IFVJUm91dGVyKSB7XG4gIHJldHVybiByLnN0YXRlUmVnaXN0cnk7XG59XG5leHBvcnQgZnVuY3Rpb24gZm5HbG9iYWxzKHI6IGFueSkge1xuICByZXR1cm4gci5nbG9iYWxzO1xufVxuXG5leHBvcnQgY29uc3QgX1VJUk9VVEVSX1NFUlZJQ0VfUFJPVklERVJTOiBQcm92aWRlcltdID0gW1xuICB7IHByb3ZpZGU6IFN0YXRlU2VydmljZSwgdXNlRmFjdG9yeTogZm5TdGF0ZVNlcnZpY2UsIGRlcHM6IFtVSVJvdXRlcl0gfSxcbiAgeyBwcm92aWRlOiBUcmFuc2l0aW9uU2VydmljZSwgdXNlRmFjdG9yeTogZm5UcmFuc2l0aW9uU2VydmljZSwgZGVwczogW1VJUm91dGVyXSB9LFxuICB7IHByb3ZpZGU6IFVybE1hdGNoZXJGYWN0b3J5LCB1c2VGYWN0b3J5OiBmblVybE1hdGNoZXJGYWN0b3J5LCBkZXBzOiBbVUlSb3V0ZXJdIH0sXG4gIHsgcHJvdmlkZTogVXJsUm91dGVyLCB1c2VGYWN0b3J5OiBmblVybFJvdXRlciwgZGVwczogW1VJUm91dGVyXSB9LFxuICB7IHByb3ZpZGU6IFVybFNlcnZpY2UsIHVzZUZhY3Rvcnk6IGZuVXJsU2VydmljZSwgZGVwczogW1VJUm91dGVyXSB9LFxuICB7IHByb3ZpZGU6IFZpZXdTZXJ2aWNlLCB1c2VGYWN0b3J5OiBmblZpZXdTZXJ2aWNlLCBkZXBzOiBbVUlSb3V0ZXJdIH0sXG4gIHsgcHJvdmlkZTogU3RhdGVSZWdpc3RyeSwgdXNlRmFjdG9yeTogZm5TdGF0ZVJlZ2lzdHJ5LCBkZXBzOiBbVUlSb3V0ZXJdIH0sXG4gIHsgcHJvdmlkZTogVUlSb3V0ZXJHbG9iYWxzLCB1c2VGYWN0b3J5OiBmbkdsb2JhbHMsIGRlcHM6IFtVSVJvdXRlcl0gfSxcbl07XG5cbi8qKlxuICogVGhlIFVJLVJvdXRlciBwcm92aWRlcnMsIGZvciB1c2UgaW4geW91ciBhcHBsaWNhdGlvbiBib290c3RyYXBcbiAqXG4gKiBAZGVwcmVjYXRlZCB1c2UgW1tVSVJvdXRlck1vZHVsZS5mb3JSb290XV1cbiAqL1xuZXhwb3J0IGNvbnN0IFVJUk9VVEVSX1BST1ZJREVSUzogUHJvdmlkZXJbXSA9IF9VSVJPVVRFUl9JTlNUQU5DRV9QUk9WSURFUlMuY29uY2F0KF9VSVJPVVRFUl9TRVJWSUNFX1BST1ZJREVSUyk7XG4iXX0=