import { loadNgModule } from '../lazyLoad/lazyLoadNgModule';
/**
 * This is a [[StateBuilder.builder]] function for ngModule lazy loading in Angular.
 *
 * When the [[StateBuilder]] builds a [[State]] object from a raw [[StateDeclaration]], this builder
 * decorates the `lazyLoad` property for states that have a [[Ng2StateDeclaration.ngModule]] declaration.
 *
 * If the state has a [[Ng2StateDeclaration.ngModule]], it will create a `lazyLoad` function
 * that in turn calls `loadNgModule(loadNgModuleFn)`.
 *
 * #### Example:
 * A state that has a `ngModule`
 * ```js
 * var decl = {
 *   ngModule: () => import('./childModule.ts')
 * }
 * ```
 * would build a state with a `lazyLoad` function like:
 * ```js
 * import { loadNgModule } from "@uirouter/angular";
 * var decl = {
 *   lazyLoad: loadNgModule(() => import('./childModule.ts')
 * }
 * ```
 *
 * If the state has both a `ngModule:` *and* a `lazyLoad`, then the `lazyLoad` is run first.
 *
 * #### Example:
 * ```js
 * var decl = {
 *   lazyLoad: () => import('third-party-library'),
 *   ngModule: () => import('./childModule.ts')
 * }
 * ```
 * would build a state with a `lazyLoad` function like:
 * ```js
 * import { loadNgModule } from "@uirouter/angular";
 * var decl = {
 *   lazyLoad: () => import('third-party-library')
 *       .then(() => loadNgModule(() => import('./childModule.ts'))
 * }
 * ```
 *
 */
export function ng2LazyLoadBuilder(state, parent) {
    const loadNgModuleFn = state['loadChildren'];
    return loadNgModuleFn ? loadNgModule(loadNgModuleFn) : state.lazyLoad;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGF6eUxvYWQuanMiLCJzb3VyY2VSb290Ijoibmc6Ly9AdWlyb3V0ZXIvYW5ndWxhci8iLCJzb3VyY2VzIjpbInN0YXRlYnVpbGRlcnMvbGF6eUxvYWQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBSUEsT0FBTyxFQUFFLFlBQVksRUFBRSxNQUFNLDhCQUE4QixDQUFDO0FBRTVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0EwQ0c7QUFDSCxNQUFNLFVBQVUsa0JBQWtCLENBQUMsS0FBa0IsRUFBRSxNQUF1QjtJQUM1RSxNQUFNLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDN0MsT0FBTyxjQUFjLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQztBQUN4RSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqIEBtb2R1bGUgbmcyICovXG4vKiogKi9cbmltcG9ydCB7IExhenlMb2FkUmVzdWx0LCBUcmFuc2l0aW9uLCBTdGF0ZURlY2xhcmF0aW9uIH0gZnJvbSAnQHVpcm91dGVyL2NvcmUnOyAvLyBoYXMgb3IgaXMgdXNpbmdcbmltcG9ydCB7IEJ1aWxkZXJGdW5jdGlvbiwgU3RhdGVPYmplY3QgfSBmcm9tICdAdWlyb3V0ZXIvY29yZSc7XG5pbXBvcnQgeyBsb2FkTmdNb2R1bGUgfSBmcm9tICcuLi9sYXp5TG9hZC9sYXp5TG9hZE5nTW9kdWxlJztcblxuLyoqXG4gKiBUaGlzIGlzIGEgW1tTdGF0ZUJ1aWxkZXIuYnVpbGRlcl1dIGZ1bmN0aW9uIGZvciBuZ01vZHVsZSBsYXp5IGxvYWRpbmcgaW4gQW5ndWxhci5cbiAqXG4gKiBXaGVuIHRoZSBbW1N0YXRlQnVpbGRlcl1dIGJ1aWxkcyBhIFtbU3RhdGVdXSBvYmplY3QgZnJvbSBhIHJhdyBbW1N0YXRlRGVjbGFyYXRpb25dXSwgdGhpcyBidWlsZGVyXG4gKiBkZWNvcmF0ZXMgdGhlIGBsYXp5TG9hZGAgcHJvcGVydHkgZm9yIHN0YXRlcyB0aGF0IGhhdmUgYSBbW05nMlN0YXRlRGVjbGFyYXRpb24ubmdNb2R1bGVdXSBkZWNsYXJhdGlvbi5cbiAqXG4gKiBJZiB0aGUgc3RhdGUgaGFzIGEgW1tOZzJTdGF0ZURlY2xhcmF0aW9uLm5nTW9kdWxlXV0sIGl0IHdpbGwgY3JlYXRlIGEgYGxhenlMb2FkYCBmdW5jdGlvblxuICogdGhhdCBpbiB0dXJuIGNhbGxzIGBsb2FkTmdNb2R1bGUobG9hZE5nTW9kdWxlRm4pYC5cbiAqXG4gKiAjIyMjIEV4YW1wbGU6XG4gKiBBIHN0YXRlIHRoYXQgaGFzIGEgYG5nTW9kdWxlYFxuICogYGBganNcbiAqIHZhciBkZWNsID0ge1xuICogICBuZ01vZHVsZTogKCkgPT4gaW1wb3J0KCcuL2NoaWxkTW9kdWxlLnRzJylcbiAqIH1cbiAqIGBgYFxuICogd291bGQgYnVpbGQgYSBzdGF0ZSB3aXRoIGEgYGxhenlMb2FkYCBmdW5jdGlvbiBsaWtlOlxuICogYGBganNcbiAqIGltcG9ydCB7IGxvYWROZ01vZHVsZSB9IGZyb20gXCJAdWlyb3V0ZXIvYW5ndWxhclwiO1xuICogdmFyIGRlY2wgPSB7XG4gKiAgIGxhenlMb2FkOiBsb2FkTmdNb2R1bGUoKCkgPT4gaW1wb3J0KCcuL2NoaWxkTW9kdWxlLnRzJylcbiAqIH1cbiAqIGBgYFxuICpcbiAqIElmIHRoZSBzdGF0ZSBoYXMgYm90aCBhIGBuZ01vZHVsZTpgICphbmQqIGEgYGxhenlMb2FkYCwgdGhlbiB0aGUgYGxhenlMb2FkYCBpcyBydW4gZmlyc3QuXG4gKlxuICogIyMjIyBFeGFtcGxlOlxuICogYGBganNcbiAqIHZhciBkZWNsID0ge1xuICogICBsYXp5TG9hZDogKCkgPT4gaW1wb3J0KCd0aGlyZC1wYXJ0eS1saWJyYXJ5JyksXG4gKiAgIG5nTW9kdWxlOiAoKSA9PiBpbXBvcnQoJy4vY2hpbGRNb2R1bGUudHMnKVxuICogfVxuICogYGBgXG4gKiB3b3VsZCBidWlsZCBhIHN0YXRlIHdpdGggYSBgbGF6eUxvYWRgIGZ1bmN0aW9uIGxpa2U6XG4gKiBgYGBqc1xuICogaW1wb3J0IHsgbG9hZE5nTW9kdWxlIH0gZnJvbSBcIkB1aXJvdXRlci9hbmd1bGFyXCI7XG4gKiB2YXIgZGVjbCA9IHtcbiAqICAgbGF6eUxvYWQ6ICgpID0+IGltcG9ydCgndGhpcmQtcGFydHktbGlicmFyeScpXG4gKiAgICAgICAudGhlbigoKSA9PiBsb2FkTmdNb2R1bGUoKCkgPT4gaW1wb3J0KCcuL2NoaWxkTW9kdWxlLnRzJykpXG4gKiB9XG4gKiBgYGBcbiAqXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBuZzJMYXp5TG9hZEJ1aWxkZXIoc3RhdGU6IFN0YXRlT2JqZWN0LCBwYXJlbnQ6IEJ1aWxkZXJGdW5jdGlvbikge1xuICBjb25zdCBsb2FkTmdNb2R1bGVGbiA9IHN0YXRlWydsb2FkQ2hpbGRyZW4nXTtcbiAgcmV0dXJuIGxvYWROZ01vZHVsZUZuID8gbG9hZE5nTW9kdWxlKGxvYWROZ01vZHVsZUZuKSA6IHN0YXRlLmxhenlMb2FkO1xufVxuIl19