var parser = require('../lib/keyParser');

var path = require('path'),
    assert = require('assert'),
    inspect = require('util').inspect;

var group = path.basename(__filename, '.js') + '/';

[
  { source: [
      '-----BEGIN RSA PRIVATE KEY-----',
      'MIIBywIBAAJhANXy0sLFQFgcY6sfyqlrBBNAcow7MWjQS0flUEj8HmrdTX+denZ5',
      'sArVEBWIoIcCWt0DWGz/mZDS2XPqyy9IbEnhwtZ3bpHmw11J1k61odAwy5sE3P4S',
      'kS/Svl7Wxjcr5wIDAQABAmEAx50pG2QiZpMGQvEEixE32iZ1OXU44kwFEO1Y/N+O',
      'GbKkzf5F9igPIewsd4rkC1MbcVPYTN6a7WlFh2dP1kC5wdPAdVjBf1p7Ii4t4HZt',
      'OokidduJzCmC+TulDCNLFmlBAjEA/6SUxcxRXJV7GHBmu+FIbnR6/drHaBvMkNKZ',
      '/tGIq1KfrHIWrhGl2vYqot6hRgoXAjEA1j9VAyuDK2NhQG/WMmf8WV+OMeJQ7vpn',
      '3KmtPc/CsWfPNYeQagfwlj0SLGwz8J6xAjEA4qIRkllUN/P9MOlU6nMKN4HkF3EB',
      'HvamIC3Uf+dadifFjqeGG8NhegCKQ+GCsAj5AjAecs3AL5Tqu6r8n2hPrZudkAkD',
      'aeiarhfIsoyedkJzOA+oyeVqTEqS74b/rTkMf2ECMBrts0oFmVJYo77riKQBYbR7',
      '/8feisYSPHSg7hwaoXSwU+jU4g39w7rgvO+WkGuzyQ==',
      '-----END RSA PRIVATE KEY-----'
    ].join('\n'),
    expected: {
      fulltype: undefined,
      type: 'rsa',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: new Buffer([
        0x30, 0x82, 0x01, 0xcb, 0x02, 0x01, 0x00, 0x02, 0x61, 0x00, 0xd5, 0xf2,
        0xd2, 0xc2, 0xc5, 0x40, 0x58, 0x1c, 0x63, 0xab, 0x1f, 0xca, 0xa9, 0x6b,
        0x04, 0x13, 0x40, 0x72, 0x8c, 0x3b, 0x31, 0x68, 0xd0, 0x4b, 0x47, 0xe5,
        0x50, 0x48, 0xfc, 0x1e, 0x6a, 0xdd, 0x4d, 0x7f, 0x9d, 0x7a, 0x76, 0x79,
        0xb0, 0x0a, 0xd5, 0x10, 0x15, 0x88, 0xa0, 0x87, 0x02, 0x5a, 0xdd, 0x03,
        0x58, 0x6c, 0xff, 0x99, 0x90, 0xd2, 0xd9, 0x73, 0xea, 0xcb, 0x2f, 0x48,
        0x6c, 0x49, 0xe1, 0xc2, 0xd6, 0x77, 0x6e, 0x91, 0xe6, 0xc3, 0x5d, 0x49,
        0xd6, 0x4e, 0xb5, 0xa1, 0xd0, 0x30, 0xcb, 0x9b, 0x04, 0xdc, 0xfe, 0x12,
        0x91, 0x2f, 0xd2, 0xbe, 0x5e, 0xd6, 0xc6, 0x37, 0x2b, 0xe7, 0x02, 0x03,
        0x01, 0x00, 0x01, 0x02, 0x61, 0x00, 0xc7, 0x9d, 0x29, 0x1b, 0x64, 0x22,
        0x66, 0x93, 0x06, 0x42, 0xf1, 0x04, 0x8b, 0x11, 0x37, 0xda, 0x26, 0x75,
        0x39, 0x75, 0x38, 0xe2, 0x4c, 0x05, 0x10, 0xed, 0x58, 0xfc, 0xdf, 0x8e,
        0x19, 0xb2, 0xa4, 0xcd, 0xfe, 0x45, 0xf6, 0x28, 0x0f, 0x21, 0xec, 0x2c,
        0x77, 0x8a, 0xe4, 0x0b, 0x53, 0x1b, 0x71, 0x53, 0xd8, 0x4c, 0xde, 0x9a,
        0xed, 0x69, 0x45, 0x87, 0x67, 0x4f, 0xd6, 0x40, 0xb9, 0xc1, 0xd3, 0xc0,
        0x75, 0x58, 0xc1, 0x7f, 0x5a, 0x7b, 0x22, 0x2e, 0x2d, 0xe0, 0x76, 0x6d,
        0x3a, 0x89, 0x22, 0x75, 0xdb, 0x89, 0xcc, 0x29, 0x82, 0xf9, 0x3b, 0xa5,
        0x0c, 0x23, 0x4b, 0x16, 0x69, 0x41, 0x02, 0x31, 0x00, 0xff, 0xa4, 0x94,
        0xc5, 0xcc, 0x51, 0x5c, 0x95, 0x7b, 0x18, 0x70, 0x66, 0xbb, 0xe1, 0x48,
        0x6e, 0x74, 0x7a, 0xfd, 0xda, 0xc7, 0x68, 0x1b, 0xcc, 0x90, 0xd2, 0x99,
        0xfe, 0xd1, 0x88, 0xab, 0x52, 0x9f, 0xac, 0x72, 0x16, 0xae, 0x11, 0xa5,
        0xda, 0xf6, 0x2a, 0xa2, 0xde, 0xa1, 0x46, 0x0a, 0x17, 0x02, 0x31, 0x00,
        0xd6, 0x3f, 0x55, 0x03, 0x2b, 0x83, 0x2b, 0x63, 0x61, 0x40, 0x6f, 0xd6,
        0x32, 0x67, 0xfc, 0x59, 0x5f, 0x8e, 0x31, 0xe2, 0x50, 0xee, 0xfa, 0x67,
        0xdc, 0xa9, 0xad, 0x3d, 0xcf, 0xc2, 0xb1, 0x67, 0xcf, 0x35, 0x87, 0x90,
        0x6a, 0x07, 0xf0, 0x96, 0x3d, 0x12, 0x2c, 0x6c, 0x33, 0xf0, 0x9e, 0xb1,
        0x02, 0x31, 0x00, 0xe2, 0xa2, 0x11, 0x92, 0x59, 0x54, 0x37, 0xf3, 0xfd,
        0x30, 0xe9, 0x54, 0xea, 0x73, 0x0a, 0x37, 0x81, 0xe4, 0x17, 0x71, 0x01,
        0x1e, 0xf6, 0xa6, 0x20, 0x2d, 0xd4, 0x7f, 0xe7, 0x5a, 0x76, 0x27, 0xc5,
        0x8e, 0xa7, 0x86, 0x1b, 0xc3, 0x61, 0x7a, 0x00, 0x8a, 0x43, 0xe1, 0x82,
        0xb0, 0x08, 0xf9, 0x02, 0x30, 0x1e, 0x72, 0xcd, 0xc0, 0x2f, 0x94, 0xea,
        0xbb, 0xaa, 0xfc, 0x9f, 0x68, 0x4f, 0xad, 0x9b, 0x9d, 0x90, 0x09, 0x03,
        0x69, 0xe8, 0x9a, 0xae, 0x17, 0xc8, 0xb2, 0x8c, 0x9e, 0x76, 0x42, 0x73,
        0x38, 0x0f, 0xa8, 0xc9, 0xe5, 0x6a, 0x4c, 0x4a, 0x92, 0xef, 0x86, 0xff,
        0xad, 0x39, 0x0c, 0x7f, 0x61, 0x02, 0x30, 0x1a, 0xed, 0xb3, 0x4a, 0x05,
        0x99, 0x52, 0x58, 0xa3, 0xbe, 0xeb, 0x88, 0xa4, 0x01, 0x61, 0xb4, 0x7b,
        0xff, 0xc7, 0xde, 0x8a, 0xc6, 0x12, 0x3c, 0x74, 0xa0, 0xee, 0x1c, 0x1a,
        0xa1, 0x74, 0xb0, 0x53, 0xe8, 0xd4, 0xe2, 0x0d, 0xfd, 0xc3, 0xba, 0xe0,
        0xbc, 0xef, 0x96, 0x90, 0x6b, 0xb3, 0xc9,
      ]),
      privateOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x52,
        0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
        0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x42,
        0x79, 0x77, 0x49, 0x42, 0x41, 0x41, 0x4a, 0x68, 0x41, 0x4e, 0x58, 0x79,
        0x30, 0x73, 0x4c, 0x46, 0x51, 0x46, 0x67, 0x63, 0x59, 0x36, 0x73, 0x66,
        0x79, 0x71, 0x6c, 0x72, 0x42, 0x42, 0x4e, 0x41, 0x63, 0x6f, 0x77, 0x37,
        0x4d, 0x57, 0x6a, 0x51, 0x53, 0x30, 0x66, 0x6c, 0x55, 0x45, 0x6a, 0x38,
        0x48, 0x6d, 0x72, 0x64, 0x54, 0x58, 0x2b, 0x64, 0x65, 0x6e, 0x5a, 0x35,
        0x0a, 0x73, 0x41, 0x72, 0x56, 0x45, 0x42, 0x57, 0x49, 0x6f, 0x49, 0x63,
        0x43, 0x57, 0x74, 0x30, 0x44, 0x57, 0x47, 0x7a, 0x2f, 0x6d, 0x5a, 0x44,
        0x53, 0x32, 0x58, 0x50, 0x71, 0x79, 0x79, 0x39, 0x49, 0x62, 0x45, 0x6e,
        0x68, 0x77, 0x74, 0x5a, 0x33, 0x62, 0x70, 0x48, 0x6d, 0x77, 0x31, 0x31,
        0x4a, 0x31, 0x6b, 0x36, 0x31, 0x6f, 0x64, 0x41, 0x77, 0x79, 0x35, 0x73,
        0x45, 0x33, 0x50, 0x34, 0x53, 0x0a, 0x6b, 0x53, 0x2f, 0x53, 0x76, 0x6c,
        0x37, 0x57, 0x78, 0x6a, 0x63, 0x72, 0x35, 0x77, 0x49, 0x44, 0x41, 0x51,
        0x41, 0x42, 0x41, 0x6d, 0x45, 0x41, 0x78, 0x35, 0x30, 0x70, 0x47, 0x32,
        0x51, 0x69, 0x5a, 0x70, 0x4d, 0x47, 0x51, 0x76, 0x45, 0x45, 0x69, 0x78,
        0x45, 0x33, 0x32, 0x69, 0x5a, 0x31, 0x4f, 0x58, 0x55, 0x34, 0x34, 0x6b,
        0x77, 0x46, 0x45, 0x4f, 0x31, 0x59, 0x2f, 0x4e, 0x2b, 0x4f, 0x0a, 0x47,
        0x62, 0x4b, 0x6b, 0x7a, 0x66, 0x35, 0x46, 0x39, 0x69, 0x67, 0x50, 0x49,
        0x65, 0x77, 0x73, 0x64, 0x34, 0x72, 0x6b, 0x43, 0x31, 0x4d, 0x62, 0x63,
        0x56, 0x50, 0x59, 0x54, 0x4e, 0x36, 0x61, 0x37, 0x57, 0x6c, 0x46, 0x68,
        0x32, 0x64, 0x50, 0x31, 0x6b, 0x43, 0x35, 0x77, 0x64, 0x50, 0x41, 0x64,
        0x56, 0x6a, 0x42, 0x66, 0x31, 0x70, 0x37, 0x49, 0x69, 0x34, 0x74, 0x34,
        0x48, 0x5a, 0x74, 0x0a, 0x4f, 0x6f, 0x6b, 0x69, 0x64, 0x64, 0x75, 0x4a,
        0x7a, 0x43, 0x6d, 0x43, 0x2b, 0x54, 0x75, 0x6c, 0x44, 0x43, 0x4e, 0x4c,
        0x46, 0x6d, 0x6c, 0x42, 0x41, 0x6a, 0x45, 0x41, 0x2f, 0x36, 0x53, 0x55,
        0x78, 0x63, 0x78, 0x52, 0x58, 0x4a, 0x56, 0x37, 0x47, 0x48, 0x42, 0x6d,
        0x75, 0x2b, 0x46, 0x49, 0x62, 0x6e, 0x52, 0x36, 0x2f, 0x64, 0x72, 0x48,
        0x61, 0x42, 0x76, 0x4d, 0x6b, 0x4e, 0x4b, 0x5a, 0x0a, 0x2f, 0x74, 0x47,
        0x49, 0x71, 0x31, 0x4b, 0x66, 0x72, 0x48, 0x49, 0x57, 0x72, 0x68, 0x47,
        0x6c, 0x32, 0x76, 0x59, 0x71, 0x6f, 0x74, 0x36, 0x68, 0x52, 0x67, 0x6f,
        0x58, 0x41, 0x6a, 0x45, 0x41, 0x31, 0x6a, 0x39, 0x56, 0x41, 0x79, 0x75,
        0x44, 0x4b, 0x32, 0x4e, 0x68, 0x51, 0x47, 0x2f, 0x57, 0x4d, 0x6d, 0x66,
        0x38, 0x57, 0x56, 0x2b, 0x4f, 0x4d, 0x65, 0x4a, 0x51, 0x37, 0x76, 0x70,
        0x6e, 0x0a, 0x33, 0x4b, 0x6d, 0x74, 0x50, 0x63, 0x2f, 0x43, 0x73, 0x57,
        0x66, 0x50, 0x4e, 0x59, 0x65, 0x51, 0x61, 0x67, 0x66, 0x77, 0x6c, 0x6a,
        0x30, 0x53, 0x4c, 0x47, 0x77, 0x7a, 0x38, 0x4a, 0x36, 0x78, 0x41, 0x6a,
        0x45, 0x41, 0x34, 0x71, 0x49, 0x52, 0x6b, 0x6c, 0x6c, 0x55, 0x4e, 0x2f,
        0x50, 0x39, 0x4d, 0x4f, 0x6c, 0x55, 0x36, 0x6e, 0x4d, 0x4b, 0x4e, 0x34,
        0x48, 0x6b, 0x46, 0x33, 0x45, 0x42, 0x0a, 0x48, 0x76, 0x61, 0x6d, 0x49,
        0x43, 0x33, 0x55, 0x66, 0x2b, 0x64, 0x61, 0x64, 0x69, 0x66, 0x46, 0x6a,
        0x71, 0x65, 0x47, 0x47, 0x38, 0x4e, 0x68, 0x65, 0x67, 0x43, 0x4b, 0x51,
        0x2b, 0x47, 0x43, 0x73, 0x41, 0x6a, 0x35, 0x41, 0x6a, 0x41, 0x65, 0x63,
        0x73, 0x33, 0x41, 0x4c, 0x35, 0x54, 0x71, 0x75, 0x36, 0x72, 0x38, 0x6e,
        0x32, 0x68, 0x50, 0x72, 0x5a, 0x75, 0x64, 0x6b, 0x41, 0x6b, 0x44, 0x0a,
        0x61, 0x65, 0x69, 0x61, 0x72, 0x68, 0x66, 0x49, 0x73, 0x6f, 0x79, 0x65,
        0x64, 0x6b, 0x4a, 0x7a, 0x4f, 0x41, 0x2b, 0x6f, 0x79, 0x65, 0x56, 0x71,
        0x54, 0x45, 0x71, 0x53, 0x37, 0x34, 0x62, 0x2f, 0x72, 0x54, 0x6b, 0x4d,
        0x66, 0x32, 0x45, 0x43, 0x4d, 0x42, 0x72, 0x74, 0x73, 0x30, 0x6f, 0x46,
        0x6d, 0x56, 0x4a, 0x59, 0x6f, 0x37, 0x37, 0x72, 0x69, 0x4b, 0x51, 0x42,
        0x59, 0x62, 0x52, 0x37, 0x0a, 0x2f, 0x38, 0x66, 0x65, 0x69, 0x73, 0x59,
        0x53, 0x50, 0x48, 0x53, 0x67, 0x37, 0x68, 0x77, 0x61, 0x6f, 0x58, 0x53,
        0x77, 0x55, 0x2b, 0x6a, 0x55, 0x34, 0x67, 0x33, 0x39, 0x77, 0x37, 0x72,
        0x67, 0x76, 0x4f, 0x2b, 0x57, 0x6b, 0x47, 0x75, 0x7a, 0x79, 0x51, 0x3d,
        0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x52,
        0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
        0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d,
      ]),
      public: undefined,
      publicOrig: undefined
    },
    what: 'Unencrypted RSA private key (OpenSSH format)'
  },
  { source: [
      ' ',
      '-----BEGIN RSA PRIVATE KEY-----',
      'MIIBywIBAAJhANXy0sLFQFgcY6sfyqlrBBNAcow7MWjQS0flUEj8HmrdTX+denZ5',
      'sArVEBWIoIcCWt0DWGz/mZDS2XPqyy9IbEnhwtZ3bpHmw11J1k61odAwy5sE3P4S',
      'kS/Svl7Wxjcr5wIDAQABAmEAx50pG2QiZpMGQvEEixE32iZ1OXU44kwFEO1Y/N+O',
      'GbKkzf5F9igPIewsd4rkC1MbcVPYTN6a7WlFh2dP1kC5wdPAdVjBf1p7Ii4t4HZt',
      'OokidduJzCmC+TulDCNLFmlBAjEA/6SUxcxRXJV7GHBmu+FIbnR6/drHaBvMkNKZ',
      '/tGIq1KfrHIWrhGl2vYqot6hRgoXAjEA1j9VAyuDK2NhQG/WMmf8WV+OMeJQ7vpn',
      '3KmtPc/CsWfPNYeQagfwlj0SLGwz8J6xAjEA4qIRkllUN/P9MOlU6nMKN4HkF3EB',
      'HvamIC3Uf+dadifFjqeGG8NhegCKQ+GCsAj5AjAecs3AL5Tqu6r8n2hPrZudkAkD',
      'aeiarhfIsoyedkJzOA+oyeVqTEqS74b/rTkMf2ECMBrts0oFmVJYo77riKQBYbR7',
      '/8feisYSPHSg7hwaoXSwU+jU4g39w7rgvO+WkGuzyQ==',
      '-----END RSA PRIVATE KEY-----',
      ' '
    ].join('\n'),
    expected: {
      fulltype: undefined,
      type: 'rsa',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: new Buffer([
        0x30, 0x82, 0x01, 0xcb, 0x02, 0x01, 0x00, 0x02, 0x61, 0x00, 0xd5, 0xf2,
        0xd2, 0xc2, 0xc5, 0x40, 0x58, 0x1c, 0x63, 0xab, 0x1f, 0xca, 0xa9, 0x6b,
        0x04, 0x13, 0x40, 0x72, 0x8c, 0x3b, 0x31, 0x68, 0xd0, 0x4b, 0x47, 0xe5,
        0x50, 0x48, 0xfc, 0x1e, 0x6a, 0xdd, 0x4d, 0x7f, 0x9d, 0x7a, 0x76, 0x79,
        0xb0, 0x0a, 0xd5, 0x10, 0x15, 0x88, 0xa0, 0x87, 0x02, 0x5a, 0xdd, 0x03,
        0x58, 0x6c, 0xff, 0x99, 0x90, 0xd2, 0xd9, 0x73, 0xea, 0xcb, 0x2f, 0x48,
        0x6c, 0x49, 0xe1, 0xc2, 0xd6, 0x77, 0x6e, 0x91, 0xe6, 0xc3, 0x5d, 0x49,
        0xd6, 0x4e, 0xb5, 0xa1, 0xd0, 0x30, 0xcb, 0x9b, 0x04, 0xdc, 0xfe, 0x12,
        0x91, 0x2f, 0xd2, 0xbe, 0x5e, 0xd6, 0xc6, 0x37, 0x2b, 0xe7, 0x02, 0x03,
        0x01, 0x00, 0x01, 0x02, 0x61, 0x00, 0xc7, 0x9d, 0x29, 0x1b, 0x64, 0x22,
        0x66, 0x93, 0x06, 0x42, 0xf1, 0x04, 0x8b, 0x11, 0x37, 0xda, 0x26, 0x75,
        0x39, 0x75, 0x38, 0xe2, 0x4c, 0x05, 0x10, 0xed, 0x58, 0xfc, 0xdf, 0x8e,
        0x19, 0xb2, 0xa4, 0xcd, 0xfe, 0x45, 0xf6, 0x28, 0x0f, 0x21, 0xec, 0x2c,
        0x77, 0x8a, 0xe4, 0x0b, 0x53, 0x1b, 0x71, 0x53, 0xd8, 0x4c, 0xde, 0x9a,
        0xed, 0x69, 0x45, 0x87, 0x67, 0x4f, 0xd6, 0x40, 0xb9, 0xc1, 0xd3, 0xc0,
        0x75, 0x58, 0xc1, 0x7f, 0x5a, 0x7b, 0x22, 0x2e, 0x2d, 0xe0, 0x76, 0x6d,
        0x3a, 0x89, 0x22, 0x75, 0xdb, 0x89, 0xcc, 0x29, 0x82, 0xf9, 0x3b, 0xa5,
        0x0c, 0x23, 0x4b, 0x16, 0x69, 0x41, 0x02, 0x31, 0x00, 0xff, 0xa4, 0x94,
        0xc5, 0xcc, 0x51, 0x5c, 0x95, 0x7b, 0x18, 0x70, 0x66, 0xbb, 0xe1, 0x48,
        0x6e, 0x74, 0x7a, 0xfd, 0xda, 0xc7, 0x68, 0x1b, 0xcc, 0x90, 0xd2, 0x99,
        0xfe, 0xd1, 0x88, 0xab, 0x52, 0x9f, 0xac, 0x72, 0x16, 0xae, 0x11, 0xa5,
        0xda, 0xf6, 0x2a, 0xa2, 0xde, 0xa1, 0x46, 0x0a, 0x17, 0x02, 0x31, 0x00,
        0xd6, 0x3f, 0x55, 0x03, 0x2b, 0x83, 0x2b, 0x63, 0x61, 0x40, 0x6f, 0xd6,
        0x32, 0x67, 0xfc, 0x59, 0x5f, 0x8e, 0x31, 0xe2, 0x50, 0xee, 0xfa, 0x67,
        0xdc, 0xa9, 0xad, 0x3d, 0xcf, 0xc2, 0xb1, 0x67, 0xcf, 0x35, 0x87, 0x90,
        0x6a, 0x07, 0xf0, 0x96, 0x3d, 0x12, 0x2c, 0x6c, 0x33, 0xf0, 0x9e, 0xb1,
        0x02, 0x31, 0x00, 0xe2, 0xa2, 0x11, 0x92, 0x59, 0x54, 0x37, 0xf3, 0xfd,
        0x30, 0xe9, 0x54, 0xea, 0x73, 0x0a, 0x37, 0x81, 0xe4, 0x17, 0x71, 0x01,
        0x1e, 0xf6, 0xa6, 0x20, 0x2d, 0xd4, 0x7f, 0xe7, 0x5a, 0x76, 0x27, 0xc5,
        0x8e, 0xa7, 0x86, 0x1b, 0xc3, 0x61, 0x7a, 0x00, 0x8a, 0x43, 0xe1, 0x82,
        0xb0, 0x08, 0xf9, 0x02, 0x30, 0x1e, 0x72, 0xcd, 0xc0, 0x2f, 0x94, 0xea,
        0xbb, 0xaa, 0xfc, 0x9f, 0x68, 0x4f, 0xad, 0x9b, 0x9d, 0x90, 0x09, 0x03,
        0x69, 0xe8, 0x9a, 0xae, 0x17, 0xc8, 0xb2, 0x8c, 0x9e, 0x76, 0x42, 0x73,
        0x38, 0x0f, 0xa8, 0xc9, 0xe5, 0x6a, 0x4c, 0x4a, 0x92, 0xef, 0x86, 0xff,
        0xad, 0x39, 0x0c, 0x7f, 0x61, 0x02, 0x30, 0x1a, 0xed, 0xb3, 0x4a, 0x05,
        0x99, 0x52, 0x58, 0xa3, 0xbe, 0xeb, 0x88, 0xa4, 0x01, 0x61, 0xb4, 0x7b,
        0xff, 0xc7, 0xde, 0x8a, 0xc6, 0x12, 0x3c, 0x74, 0xa0, 0xee, 0x1c, 0x1a,
        0xa1, 0x74, 0xb0, 0x53, 0xe8, 0xd4, 0xe2, 0x0d, 0xfd, 0xc3, 0xba, 0xe0,
        0xbc, 0xef, 0x96, 0x90, 0x6b, 0xb3, 0xc9,
      ]),
      privateOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x52,
        0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
        0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x42,
        0x79, 0x77, 0x49, 0x42, 0x41, 0x41, 0x4a, 0x68, 0x41, 0x4e, 0x58, 0x79,
        0x30, 0x73, 0x4c, 0x46, 0x51, 0x46, 0x67, 0x63, 0x59, 0x36, 0x73, 0x66,
        0x79, 0x71, 0x6c, 0x72, 0x42, 0x42, 0x4e, 0x41, 0x63, 0x6f, 0x77, 0x37,
        0x4d, 0x57, 0x6a, 0x51, 0x53, 0x30, 0x66, 0x6c, 0x55, 0x45, 0x6a, 0x38,
        0x48, 0x6d, 0x72, 0x64, 0x54, 0x58, 0x2b, 0x64, 0x65, 0x6e, 0x5a, 0x35,
        0x0a, 0x73, 0x41, 0x72, 0x56, 0x45, 0x42, 0x57, 0x49, 0x6f, 0x49, 0x63,
        0x43, 0x57, 0x74, 0x30, 0x44, 0x57, 0x47, 0x7a, 0x2f, 0x6d, 0x5a, 0x44,
        0x53, 0x32, 0x58, 0x50, 0x71, 0x79, 0x79, 0x39, 0x49, 0x62, 0x45, 0x6e,
        0x68, 0x77, 0x74, 0x5a, 0x33, 0x62, 0x70, 0x48, 0x6d, 0x77, 0x31, 0x31,
        0x4a, 0x31, 0x6b, 0x36, 0x31, 0x6f, 0x64, 0x41, 0x77, 0x79, 0x35, 0x73,
        0x45, 0x33, 0x50, 0x34, 0x53, 0x0a, 0x6b, 0x53, 0x2f, 0x53, 0x76, 0x6c,
        0x37, 0x57, 0x78, 0x6a, 0x63, 0x72, 0x35, 0x77, 0x49, 0x44, 0x41, 0x51,
        0x41, 0x42, 0x41, 0x6d, 0x45, 0x41, 0x78, 0x35, 0x30, 0x70, 0x47, 0x32,
        0x51, 0x69, 0x5a, 0x70, 0x4d, 0x47, 0x51, 0x76, 0x45, 0x45, 0x69, 0x78,
        0x45, 0x33, 0x32, 0x69, 0x5a, 0x31, 0x4f, 0x58, 0x55, 0x34, 0x34, 0x6b,
        0x77, 0x46, 0x45, 0x4f, 0x31, 0x59, 0x2f, 0x4e, 0x2b, 0x4f, 0x0a, 0x47,
        0x62, 0x4b, 0x6b, 0x7a, 0x66, 0x35, 0x46, 0x39, 0x69, 0x67, 0x50, 0x49,
        0x65, 0x77, 0x73, 0x64, 0x34, 0x72, 0x6b, 0x43, 0x31, 0x4d, 0x62, 0x63,
        0x56, 0x50, 0x59, 0x54, 0x4e, 0x36, 0x61, 0x37, 0x57, 0x6c, 0x46, 0x68,
        0x32, 0x64, 0x50, 0x31, 0x6b, 0x43, 0x35, 0x77, 0x64, 0x50, 0x41, 0x64,
        0x56, 0x6a, 0x42, 0x66, 0x31, 0x70, 0x37, 0x49, 0x69, 0x34, 0x74, 0x34,
        0x48, 0x5a, 0x74, 0x0a, 0x4f, 0x6f, 0x6b, 0x69, 0x64, 0x64, 0x75, 0x4a,
        0x7a, 0x43, 0x6d, 0x43, 0x2b, 0x54, 0x75, 0x6c, 0x44, 0x43, 0x4e, 0x4c,
        0x46, 0x6d, 0x6c, 0x42, 0x41, 0x6a, 0x45, 0x41, 0x2f, 0x36, 0x53, 0x55,
        0x78, 0x63, 0x78, 0x52, 0x58, 0x4a, 0x56, 0x37, 0x47, 0x48, 0x42, 0x6d,
        0x75, 0x2b, 0x46, 0x49, 0x62, 0x6e, 0x52, 0x36, 0x2f, 0x64, 0x72, 0x48,
        0x61, 0x42, 0x76, 0x4d, 0x6b, 0x4e, 0x4b, 0x5a, 0x0a, 0x2f, 0x74, 0x47,
        0x49, 0x71, 0x31, 0x4b, 0x66, 0x72, 0x48, 0x49, 0x57, 0x72, 0x68, 0x47,
        0x6c, 0x32, 0x76, 0x59, 0x71, 0x6f, 0x74, 0x36, 0x68, 0x52, 0x67, 0x6f,
        0x58, 0x41, 0x6a, 0x45, 0x41, 0x31, 0x6a, 0x39, 0x56, 0x41, 0x79, 0x75,
        0x44, 0x4b, 0x32, 0x4e, 0x68, 0x51, 0x47, 0x2f, 0x57, 0x4d, 0x6d, 0x66,
        0x38, 0x57, 0x56, 0x2b, 0x4f, 0x4d, 0x65, 0x4a, 0x51, 0x37, 0x76, 0x70,
        0x6e, 0x0a, 0x33, 0x4b, 0x6d, 0x74, 0x50, 0x63, 0x2f, 0x43, 0x73, 0x57,
        0x66, 0x50, 0x4e, 0x59, 0x65, 0x51, 0x61, 0x67, 0x66, 0x77, 0x6c, 0x6a,
        0x30, 0x53, 0x4c, 0x47, 0x77, 0x7a, 0x38, 0x4a, 0x36, 0x78, 0x41, 0x6a,
        0x45, 0x41, 0x34, 0x71, 0x49, 0x52, 0x6b, 0x6c, 0x6c, 0x55, 0x4e, 0x2f,
        0x50, 0x39, 0x4d, 0x4f, 0x6c, 0x55, 0x36, 0x6e, 0x4d, 0x4b, 0x4e, 0x34,
        0x48, 0x6b, 0x46, 0x33, 0x45, 0x42, 0x0a, 0x48, 0x76, 0x61, 0x6d, 0x49,
        0x43, 0x33, 0x55, 0x66, 0x2b, 0x64, 0x61, 0x64, 0x69, 0x66, 0x46, 0x6a,
        0x71, 0x65, 0x47, 0x47, 0x38, 0x4e, 0x68, 0x65, 0x67, 0x43, 0x4b, 0x51,
        0x2b, 0x47, 0x43, 0x73, 0x41, 0x6a, 0x35, 0x41, 0x6a, 0x41, 0x65, 0x63,
        0x73, 0x33, 0x41, 0x4c, 0x35, 0x54, 0x71, 0x75, 0x36, 0x72, 0x38, 0x6e,
        0x32, 0x68, 0x50, 0x72, 0x5a, 0x75, 0x64, 0x6b, 0x41, 0x6b, 0x44, 0x0a,
        0x61, 0x65, 0x69, 0x61, 0x72, 0x68, 0x66, 0x49, 0x73, 0x6f, 0x79, 0x65,
        0x64, 0x6b, 0x4a, 0x7a, 0x4f, 0x41, 0x2b, 0x6f, 0x79, 0x65, 0x56, 0x71,
        0x54, 0x45, 0x71, 0x53, 0x37, 0x34, 0x62, 0x2f, 0x72, 0x54, 0x6b, 0x4d,
        0x66, 0x32, 0x45, 0x43, 0x4d, 0x42, 0x72, 0x74, 0x73, 0x30, 0x6f, 0x46,
        0x6d, 0x56, 0x4a, 0x59, 0x6f, 0x37, 0x37, 0x72, 0x69, 0x4b, 0x51, 0x42,
        0x59, 0x62, 0x52, 0x37, 0x0a, 0x2f, 0x38, 0x66, 0x65, 0x69, 0x73, 0x59,
        0x53, 0x50, 0x48, 0x53, 0x67, 0x37, 0x68, 0x77, 0x61, 0x6f, 0x58, 0x53,
        0x77, 0x55, 0x2b, 0x6a, 0x55, 0x34, 0x67, 0x33, 0x39, 0x77, 0x37, 0x72,
        0x67, 0x76, 0x4f, 0x2b, 0x57, 0x6b, 0x47, 0x75, 0x7a, 0x79, 0x51, 0x3d,
        0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x52,
        0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
        0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d,
      ]),
      public: undefined,
      publicOrig: undefined
    },
    what: 'Unencrypted RSA private key (OpenSSH format) with surrounding whitespace'
  },
  { source: [
      '-----BEGIN RSA PRIVATE KEY-----',
      'Proc-Type: 4,ENCRYPTED',
      'DEK-Info: AES-128-CBC,E9F7397F94E585BCE4C739D5C5226A2D',
      '',
      'mshPNeXq8XAh79N3F1vKUOCsWoTofhz/Kpt/ddYdgnGG9L5wGDoveJQtpsa5u+A4',
      'hCkAWA7KRCUKbOxZd8lG00cm7HILN7eHCKSoSkyA+X1jN7v1tSGPjiN2RSZFHCOI',
      '8jEF0fyTeg9GZzFaiyyD5p1niy7uCssEeo8AzbIBX4yLDR18mmBiMhn0FxVu/YFn',
      'rfkm+ciE4WhkqdCUOkKiqIux7RI6O9ttUuZlhwVvoKZqujMeNrFppOC9kF5hAi6Y',
      'oHvrweE8QmimEUIMAwdDrI1/jglFw2x2K5sRxPIIFJC1aHR8nXLTuO1g2/g6aqls',
      'jGkwlQU4cU5pnkxkLUVCSHi6A6nwbdjQTIIDofpcckQVFHZaBCJx9UF691NR05Dc',
      'GthFFfNGXWU6vbwKia/LDX8R98ay4Ci2zoDXxe7QBnvnWbs2m69HPh6opmo5gKL+',
      'y65Sf6MAi5udIJLVfgXQBgRyX3glS7/rH8YbrrNY9Iop1KcSWXAIC4zlCoO3tBJB',
      '9464YBD/R+qOvVMcBGxTjabudi00ougeTgsS7idDTmfc4HYIls0E7yWqzgspsws5',
      'w7K4crSgm+u3oypViUSre2qSvAHTC5gXth8LVCnDZuA=',
      '-----END RSA PRIVATE KEY-----'
    ].join('\n'),
    expected: {
      fulltype: undefined,
      type: 'rsa',
      extra: [ 'E9F7397F94E585BCE4C739D5C5226A2D' ],
      comment: undefined,
      encryption: 'aes-128-cbc',
      private: new Buffer([
        0x9a, 0xc8, 0x4f, 0x35, 0xe5, 0xea, 0xf1, 0x70, 0x21, 0xef, 0xd3, 0x77,
        0x17, 0x5b, 0xca, 0x50, 0xe0, 0xac, 0x5a, 0x84, 0xe8, 0x7e, 0x1c, 0xff,
        0x2a, 0x9b, 0x7f, 0x75, 0xd6, 0x1d, 0x82, 0x71, 0x86, 0xf4, 0xbe, 0x70,
        0x18, 0x3a, 0x2f, 0x78, 0x94, 0x2d, 0xa6, 0xc6, 0xb9, 0xbb, 0xe0, 0x38,
        0x84, 0x29, 0x00, 0x58, 0x0e, 0xca, 0x44, 0x25, 0x0a, 0x6c, 0xec, 0x59,
        0x77, 0xc9, 0x46, 0xd3, 0x47, 0x26, 0xec, 0x72, 0x0b, 0x37, 0xb7, 0x87,
        0x08, 0xa4, 0xa8, 0x4a, 0x4c, 0x80, 0xf9, 0x7d, 0x63, 0x37, 0xbb, 0xf5,
        0xb5, 0x21, 0x8f, 0x8e, 0x23, 0x76, 0x45, 0x26, 0x45, 0x1c, 0x23, 0x88,
        0xf2, 0x31, 0x05, 0xd1, 0xfc, 0x93, 0x7a, 0x0f, 0x46, 0x67, 0x31, 0x5a,
        0x8b, 0x2c, 0x83, 0xe6, 0x9d, 0x67, 0x8b, 0x2e, 0xee, 0x0a, 0xcb, 0x04,
        0x7a, 0x8f, 0x00, 0xcd, 0xb2, 0x01, 0x5f, 0x8c, 0x8b, 0x0d, 0x1d, 0x7c,
        0x9a, 0x60, 0x62, 0x32, 0x19, 0xf4, 0x17, 0x15, 0x6e, 0xfd, 0x81, 0x67,
        0xad, 0xf9, 0x26, 0xf9, 0xc8, 0x84, 0xe1, 0x68, 0x64, 0xa9, 0xd0, 0x94,
        0x3a, 0x42, 0xa2, 0xa8, 0x8b, 0xb1, 0xed, 0x12, 0x3a, 0x3b, 0xdb, 0x6d,
        0x52, 0xe6, 0x65, 0x87, 0x05, 0x6f, 0xa0, 0xa6, 0x6a, 0xba, 0x33, 0x1e,
        0x36, 0xb1, 0x69, 0xa4, 0xe0, 0xbd, 0x90, 0x5e, 0x61, 0x02, 0x2e, 0x98,
        0xa0, 0x7b, 0xeb, 0xc1, 0xe1, 0x3c, 0x42, 0x68, 0xa6, 0x11, 0x42, 0x0c,
        0x03, 0x07, 0x43, 0xac, 0x8d, 0x7f, 0x8e, 0x09, 0x45, 0xc3, 0x6c, 0x76,
        0x2b, 0x9b, 0x11, 0xc4, 0xf2, 0x08, 0x14, 0x90, 0xb5, 0x68, 0x74, 0x7c,
        0x9d, 0x72, 0xd3, 0xb8, 0xed, 0x60, 0xdb, 0xf8, 0x3a, 0x6a, 0xa9, 0x6c,
        0x8c, 0x69, 0x30, 0x95, 0x05, 0x38, 0x71, 0x4e, 0x69, 0x9e, 0x4c, 0x64,
        0x2d, 0x45, 0x42, 0x48, 0x78, 0xba, 0x03, 0xa9, 0xf0, 0x6d, 0xd8, 0xd0,
        0x4c, 0x82, 0x03, 0xa1, 0xfa, 0x5c, 0x72, 0x44, 0x15, 0x14, 0x76, 0x5a,
        0x04, 0x22, 0x71, 0xf5, 0x41, 0x7a, 0xf7, 0x53, 0x51, 0xd3, 0x90, 0xdc,
        0x1a, 0xd8, 0x45, 0x15, 0xf3, 0x46, 0x5d, 0x65, 0x3a, 0xbd, 0xbc, 0x0a,
        0x89, 0xaf, 0xcb, 0x0d, 0x7f, 0x11, 0xf7, 0xc6, 0xb2, 0xe0, 0x28, 0xb6,
        0xce, 0x80, 0xd7, 0xc5, 0xee, 0xd0, 0x06, 0x7b, 0xe7, 0x59, 0xbb, 0x36,
        0x9b, 0xaf, 0x47, 0x3e, 0x1e, 0xa8, 0xa6, 0x6a, 0x39, 0x80, 0xa2, 0xfe,
        0xcb, 0xae, 0x52, 0x7f, 0xa3, 0x00, 0x8b, 0x9b, 0x9d, 0x20, 0x92, 0xd5,
        0x7e, 0x05, 0xd0, 0x06, 0x04, 0x72, 0x5f, 0x78, 0x25, 0x4b, 0xbf, 0xeb,
        0x1f, 0xc6, 0x1b, 0xae, 0xb3, 0x58, 0xf4, 0x8a, 0x29, 0xd4, 0xa7, 0x12,
        0x59, 0x70, 0x08, 0x0b, 0x8c, 0xe5, 0x0a, 0x83, 0xb7, 0xb4, 0x12, 0x41,
        0xf7, 0x8e, 0xb8, 0x60, 0x10, 0xff, 0x47, 0xea, 0x8e, 0xbd, 0x53, 0x1c,
        0x04, 0x6c, 0x53, 0x8d, 0xa6, 0xee, 0x76, 0x2d, 0x34, 0xa2, 0xe8, 0x1e,
        0x4e, 0x0b, 0x12, 0xee, 0x27, 0x43, 0x4e, 0x67, 0xdc, 0xe0, 0x76, 0x08,
        0x96, 0xcd, 0x04, 0xef, 0x25, 0xaa, 0xce, 0x0b, 0x29, 0xb3, 0x0b, 0x39,
        0xc3, 0xb2, 0xb8, 0x72, 0xb4, 0xa0, 0x9b, 0xeb, 0xb7, 0xa3, 0x2a, 0x55,
        0x89, 0x44, 0xab, 0x7b, 0x6a, 0x92, 0xbc, 0x01, 0xd3, 0x0b, 0x98, 0x17,
        0xb6, 0x1f, 0x0b, 0x54, 0x29, 0xc3, 0x66, 0xe0,
      ]),
      privateOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x52,
        0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
        0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x50, 0x72, 0x6f, 0x63,
        0x2d, 0x54, 0x79, 0x70, 0x65, 0x3a, 0x20, 0x34, 0x2c, 0x45, 0x4e, 0x43,
        0x52, 0x59, 0x50, 0x54, 0x45, 0x44, 0x0a, 0x44, 0x45, 0x4b, 0x2d, 0x49,
        0x6e, 0x66, 0x6f, 0x3a, 0x20, 0x41, 0x45, 0x53, 0x2d, 0x31, 0x32, 0x38,
        0x2d, 0x43, 0x42, 0x43, 0x2c, 0x45, 0x39, 0x46, 0x37, 0x33, 0x39, 0x37,
        0x46, 0x39, 0x34, 0x45, 0x35, 0x38, 0x35, 0x42, 0x43, 0x45, 0x34, 0x43,
        0x37, 0x33, 0x39, 0x44, 0x35, 0x43, 0x35, 0x32, 0x32, 0x36, 0x41, 0x32,
        0x44, 0x0a, 0x0a, 0x6d, 0x73, 0x68, 0x50, 0x4e, 0x65, 0x58, 0x71, 0x38,
        0x58, 0x41, 0x68, 0x37, 0x39, 0x4e, 0x33, 0x46, 0x31, 0x76, 0x4b, 0x55,
        0x4f, 0x43, 0x73, 0x57, 0x6f, 0x54, 0x6f, 0x66, 0x68, 0x7a, 0x2f, 0x4b,
        0x70, 0x74, 0x2f, 0x64, 0x64, 0x59, 0x64, 0x67, 0x6e, 0x47, 0x47, 0x39,
        0x4c, 0x35, 0x77, 0x47, 0x44, 0x6f, 0x76, 0x65, 0x4a, 0x51, 0x74, 0x70,
        0x73, 0x61, 0x35, 0x75, 0x2b, 0x41, 0x34, 0x0a, 0x68, 0x43, 0x6b, 0x41,
        0x57, 0x41, 0x37, 0x4b, 0x52, 0x43, 0x55, 0x4b, 0x62, 0x4f, 0x78, 0x5a,
        0x64, 0x38, 0x6c, 0x47, 0x30, 0x30, 0x63, 0x6d, 0x37, 0x48, 0x49, 0x4c,
        0x4e, 0x37, 0x65, 0x48, 0x43, 0x4b, 0x53, 0x6f, 0x53, 0x6b, 0x79, 0x41,
        0x2b, 0x58, 0x31, 0x6a, 0x4e, 0x37, 0x76, 0x31, 0x74, 0x53, 0x47, 0x50,
        0x6a, 0x69, 0x4e, 0x32, 0x52, 0x53, 0x5a, 0x46, 0x48, 0x43, 0x4f, 0x49,
        0x0a, 0x38, 0x6a, 0x45, 0x46, 0x30, 0x66, 0x79, 0x54, 0x65, 0x67, 0x39,
        0x47, 0x5a, 0x7a, 0x46, 0x61, 0x69, 0x79, 0x79, 0x44, 0x35, 0x70, 0x31,
        0x6e, 0x69, 0x79, 0x37, 0x75, 0x43, 0x73, 0x73, 0x45, 0x65, 0x6f, 0x38,
        0x41, 0x7a, 0x62, 0x49, 0x42, 0x58, 0x34, 0x79, 0x4c, 0x44, 0x52, 0x31,
        0x38, 0x6d, 0x6d, 0x42, 0x69, 0x4d, 0x68, 0x6e, 0x30, 0x46, 0x78, 0x56,
        0x75, 0x2f, 0x59, 0x46, 0x6e, 0x0a, 0x72, 0x66, 0x6b, 0x6d, 0x2b, 0x63,
        0x69, 0x45, 0x34, 0x57, 0x68, 0x6b, 0x71, 0x64, 0x43, 0x55, 0x4f, 0x6b,
        0x4b, 0x69, 0x71, 0x49, 0x75, 0x78, 0x37, 0x52, 0x49, 0x36, 0x4f, 0x39,
        0x74, 0x74, 0x55, 0x75, 0x5a, 0x6c, 0x68, 0x77, 0x56, 0x76, 0x6f, 0x4b,
        0x5a, 0x71, 0x75, 0x6a, 0x4d, 0x65, 0x4e, 0x72, 0x46, 0x70, 0x70, 0x4f,
        0x43, 0x39, 0x6b, 0x46, 0x35, 0x68, 0x41, 0x69, 0x36, 0x59, 0x0a, 0x6f,
        0x48, 0x76, 0x72, 0x77, 0x65, 0x45, 0x38, 0x51, 0x6d, 0x69, 0x6d, 0x45,
        0x55, 0x49, 0x4d, 0x41, 0x77, 0x64, 0x44, 0x72, 0x49, 0x31, 0x2f, 0x6a,
        0x67, 0x6c, 0x46, 0x77, 0x32, 0x78, 0x32, 0x4b, 0x35, 0x73, 0x52, 0x78,
        0x50, 0x49, 0x49, 0x46, 0x4a, 0x43, 0x31, 0x61, 0x48, 0x52, 0x38, 0x6e,
        0x58, 0x4c, 0x54, 0x75, 0x4f, 0x31, 0x67, 0x32, 0x2f, 0x67, 0x36, 0x61,
        0x71, 0x6c, 0x73, 0x0a, 0x6a, 0x47, 0x6b, 0x77, 0x6c, 0x51, 0x55, 0x34,
        0x63, 0x55, 0x35, 0x70, 0x6e, 0x6b, 0x78, 0x6b, 0x4c, 0x55, 0x56, 0x43,
        0x53, 0x48, 0x69, 0x36, 0x41, 0x36, 0x6e, 0x77, 0x62, 0x64, 0x6a, 0x51,
        0x54, 0x49, 0x49, 0x44, 0x6f, 0x66, 0x70, 0x63, 0x63, 0x6b, 0x51, 0x56,
        0x46, 0x48, 0x5a, 0x61, 0x42, 0x43, 0x4a, 0x78, 0x39, 0x55, 0x46, 0x36,
        0x39, 0x31, 0x4e, 0x52, 0x30, 0x35, 0x44, 0x63, 0x0a, 0x47, 0x74, 0x68,
        0x46, 0x46, 0x66, 0x4e, 0x47, 0x58, 0x57, 0x55, 0x36, 0x76, 0x62, 0x77,
        0x4b, 0x69, 0x61, 0x2f, 0x4c, 0x44, 0x58, 0x38, 0x52, 0x39, 0x38, 0x61,
        0x79, 0x34, 0x43, 0x69, 0x32, 0x7a, 0x6f, 0x44, 0x58, 0x78, 0x65, 0x37,
        0x51, 0x42, 0x6e, 0x76, 0x6e, 0x57, 0x62, 0x73, 0x32, 0x6d, 0x36, 0x39,
        0x48, 0x50, 0x68, 0x36, 0x6f, 0x70, 0x6d, 0x6f, 0x35, 0x67, 0x4b, 0x4c,
        0x2b, 0x0a, 0x79, 0x36, 0x35, 0x53, 0x66, 0x36, 0x4d, 0x41, 0x69, 0x35,
        0x75, 0x64, 0x49, 0x4a, 0x4c, 0x56, 0x66, 0x67, 0x58, 0x51, 0x42, 0x67,
        0x52, 0x79, 0x58, 0x33, 0x67, 0x6c, 0x53, 0x37, 0x2f, 0x72, 0x48, 0x38,
        0x59, 0x62, 0x72, 0x72, 0x4e, 0x59, 0x39, 0x49, 0x6f, 0x70, 0x31, 0x4b,
        0x63, 0x53, 0x57, 0x58, 0x41, 0x49, 0x43, 0x34, 0x7a, 0x6c, 0x43, 0x6f,
        0x4f, 0x33, 0x74, 0x42, 0x4a, 0x42, 0x0a, 0x39, 0x34, 0x36, 0x34, 0x59,
        0x42, 0x44, 0x2f, 0x52, 0x2b, 0x71, 0x4f, 0x76, 0x56, 0x4d, 0x63, 0x42,
        0x47, 0x78, 0x54, 0x6a, 0x61, 0x62, 0x75, 0x64, 0x69, 0x30, 0x30, 0x6f,
        0x75, 0x67, 0x65, 0x54, 0x67, 0x73, 0x53, 0x37, 0x69, 0x64, 0x44, 0x54,
        0x6d, 0x66, 0x63, 0x34, 0x48, 0x59, 0x49, 0x6c, 0x73, 0x30, 0x45, 0x37,
        0x79, 0x57, 0x71, 0x7a, 0x67, 0x73, 0x70, 0x73, 0x77, 0x73, 0x35, 0x0a,
        0x77, 0x37, 0x4b, 0x34, 0x63, 0x72, 0x53, 0x67, 0x6d, 0x2b, 0x75, 0x33,
        0x6f, 0x79, 0x70, 0x56, 0x69, 0x55, 0x53, 0x72, 0x65, 0x32, 0x71, 0x53,
        0x76, 0x41, 0x48, 0x54, 0x43, 0x35, 0x67, 0x58, 0x74, 0x68, 0x38, 0x4c,
        0x56, 0x43, 0x6e, 0x44, 0x5a, 0x75, 0x41, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d,
        0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x52, 0x53, 0x41, 0x20, 0x50, 0x52,
        0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d,
        0x2d, 0x2d,
      ]),
      public: undefined,
      publicOrig: undefined
    },
    what: 'Encrypted RSA private key (OpenSSH format)'
  },
  { source: 'ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRLlnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0fxoLSF3lxQFJ+32/rS0=',
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x07, 0x73, 0x73, 0x68, 0x2d, 0x72, 0x73, 0x61, 0x00,
        0x00, 0x00, 0x03, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x61, 0x00, 0xe5,
        0xe9, 0xdc, 0x4b, 0xfa, 0x18, 0x31, 0xc0, 0x63, 0x61, 0x66, 0xba, 0xb5,
        0xf2, 0xf0, 0x2b, 0x6c, 0xd0, 0x74, 0x52, 0xa3, 0x71, 0xbd, 0xc9, 0x5f,
        0xfa, 0x87, 0x22, 0x44, 0xb9, 0x67, 0x47, 0x93, 0xba, 0xaf, 0xf7, 0x33,
        0x02, 0x1c, 0x92, 0x3b, 0xe7, 0x8e, 0x73, 0x94, 0x85, 0xef, 0x07, 0xb1,
        0x67, 0x35, 0xa8, 0xfc, 0x23, 0x90, 0xdf, 0xbb, 0x44, 0x8b, 0xda, 0xde,
        0x4e, 0xe7, 0x18, 0x14, 0xd4, 0x10, 0x58, 0x5b, 0xf1, 0xe0, 0x4f, 0x02,
        0x44, 0x53, 0xfe, 0x50, 0xf4, 0x7a, 0x85, 0x05, 0xf4, 0x7f, 0x1a, 0x0b,
        0x48, 0x5d, 0xe5, 0xc5, 0x01, 0x49, 0xfb, 0x7d, 0xbf, 0xad, 0x2d,
      ]),
      publicOrig: new Buffer([
        0x73, 0x73, 0x68, 0x2d, 0x72, 0x73, 0x61, 0x20, 0x41, 0x41, 0x41, 0x41,
        0x42, 0x33, 0x4e, 0x7a, 0x61, 0x43, 0x31, 0x79, 0x63, 0x32, 0x45, 0x41,
        0x41, 0x41, 0x41, 0x44, 0x41, 0x51, 0x41, 0x42, 0x41, 0x41, 0x41, 0x41,
        0x59, 0x51, 0x44, 0x6c, 0x36, 0x64, 0x78, 0x4c, 0x2b, 0x68, 0x67, 0x78,
        0x77, 0x47, 0x4e, 0x68, 0x5a, 0x72, 0x71, 0x31, 0x38, 0x76, 0x41, 0x72,
        0x62, 0x4e, 0x42, 0x30, 0x55, 0x71, 0x4e, 0x78, 0x76, 0x63, 0x6c, 0x66,
        0x2b, 0x6f, 0x63, 0x69, 0x52, 0x4c, 0x6c, 0x6e, 0x52, 0x35, 0x4f, 0x36,
        0x72, 0x2f, 0x63, 0x7a, 0x41, 0x68, 0x79, 0x53, 0x4f, 0x2b, 0x65, 0x4f,
        0x63, 0x35, 0x53, 0x46, 0x37, 0x77, 0x65, 0x78, 0x5a, 0x7a, 0x57, 0x6f,
        0x2f, 0x43, 0x4f, 0x51, 0x33, 0x37, 0x74, 0x45, 0x69, 0x39, 0x72, 0x65,
        0x54, 0x75, 0x63, 0x59, 0x46, 0x4e, 0x51, 0x51, 0x57, 0x46, 0x76, 0x78,
        0x34, 0x45, 0x38, 0x43, 0x52, 0x46, 0x50, 0x2b, 0x55, 0x50, 0x52, 0x36,
        0x68, 0x51, 0x58, 0x30, 0x66, 0x78, 0x6f, 0x4c, 0x53, 0x46, 0x33, 0x6c,
        0x78, 0x51, 0x46, 0x4a, 0x2b, 0x33, 0x32, 0x2f, 0x72, 0x53, 0x30, 0x3d,
      ])
    },
    what: 'RSA public key (OpenSSH format)'
  },
  { source: 'ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAAYQD3txsEf0HAKElAFUvIXzsM98gfPlIbG4/GlqbYYBulkHu6z0laOdoT14Zx2M+3q+9RjhTZjHxyMfePdcgNK9z98V6tOz5bIQhtMS8tl1Tnw5qZByGqpqOKf665ev62LaM= testing-ssh2-from-node.js',
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: 'testing-ssh2-from-node.js',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x07, 0x73, 0x73, 0x68, 0x2d, 0x72, 0x73, 0x61, 0x00,
        0x00, 0x00, 0x03, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x61, 0x00, 0xf7,
        0xb7, 0x1b, 0x04, 0x7f, 0x41, 0xc0, 0x28, 0x49, 0x40, 0x15, 0x4b, 0xc8,
        0x5f, 0x3b, 0x0c, 0xf7, 0xc8, 0x1f, 0x3e, 0x52, 0x1b, 0x1b, 0x8f, 0xc6,
        0x96, 0xa6, 0xd8, 0x60, 0x1b, 0xa5, 0x90, 0x7b, 0xba, 0xcf, 0x49, 0x5a,
        0x39, 0xda, 0x13, 0xd7, 0x86, 0x71, 0xd8, 0xcf, 0xb7, 0xab, 0xef, 0x51,
        0x8e, 0x14, 0xd9, 0x8c, 0x7c, 0x72, 0x31, 0xf7, 0x8f, 0x75, 0xc8, 0x0d,
        0x2b, 0xdc, 0xfd, 0xf1, 0x5e, 0xad, 0x3b, 0x3e, 0x5b, 0x21, 0x08, 0x6d,
        0x31, 0x2f, 0x2d, 0x97, 0x54, 0xe7, 0xc3, 0x9a, 0x99, 0x07, 0x21, 0xaa,
        0xa6, 0xa3, 0x8a, 0x7f, 0xae, 0xb9, 0x7a, 0xfe, 0xb6, 0x2d, 0xa3,
      ]),
      publicOrig: new Buffer([
        0x73, 0x73, 0x68, 0x2d, 0x72, 0x73, 0x61, 0x20, 0x41, 0x41, 0x41, 0x41,
        0x42, 0x33, 0x4e, 0x7a, 0x61, 0x43, 0x31, 0x79, 0x63, 0x32, 0x45, 0x41,
        0x41, 0x41, 0x41, 0x44, 0x41, 0x51, 0x41, 0x42, 0x41, 0x41, 0x41, 0x41,
        0x59, 0x51, 0x44, 0x33, 0x74, 0x78, 0x73, 0x45, 0x66, 0x30, 0x48, 0x41,
        0x4b, 0x45, 0x6c, 0x41, 0x46, 0x55, 0x76, 0x49, 0x58, 0x7a, 0x73, 0x4d,
        0x39, 0x38, 0x67, 0x66, 0x50, 0x6c, 0x49, 0x62, 0x47, 0x34, 0x2f, 0x47,
        0x6c, 0x71, 0x62, 0x59, 0x59, 0x42, 0x75, 0x6c, 0x6b, 0x48, 0x75, 0x36,
        0x7a, 0x30, 0x6c, 0x61, 0x4f, 0x64, 0x6f, 0x54, 0x31, 0x34, 0x5a, 0x78,
        0x32, 0x4d, 0x2b, 0x33, 0x71, 0x2b, 0x39, 0x52, 0x6a, 0x68, 0x54, 0x5a,
        0x6a, 0x48, 0x78, 0x79, 0x4d, 0x66, 0x65, 0x50, 0x64, 0x63, 0x67, 0x4e,
        0x4b, 0x39, 0x7a, 0x39, 0x38, 0x56, 0x36, 0x74, 0x4f, 0x7a, 0x35, 0x62,
        0x49, 0x51, 0x68, 0x74, 0x4d, 0x53, 0x38, 0x74, 0x6c, 0x31, 0x54, 0x6e,
        0x77, 0x35, 0x71, 0x5a, 0x42, 0x79, 0x47, 0x71, 0x70, 0x71, 0x4f, 0x4b,
        0x66, 0x36, 0x36, 0x35, 0x65, 0x76, 0x36, 0x32, 0x4c, 0x61, 0x4d, 0x3d,
        0x20, 0x74, 0x65, 0x73, 0x74, 0x69, 0x6e, 0x67, 0x2d, 0x73, 0x73, 0x68,
        0x32, 0x2d, 0x66, 0x72, 0x6f, 0x6d, 0x2d, 0x6e, 0x6f, 0x64, 0x65, 0x2e,
        0x6a, 0x73,
      ])
    },
    what: 'RSA public key with comment (OpenSSH format)'
  },
  { source: [
      '-----BEGIN DSA PRIVATE KEY-----',
      'MIIBuwIBAAKBgQD7v1kR31D2NhZIzOjJL1hPzvw79K3eWZqQEgqKmeB+P9MhOx51',
      'MOZrvs3hZoYqmsNxT/Y29EwQ+o+SXpTYjJhw/s2vR0AeJBj32l8weD804+T+S8yr',
      'MJBdgsc5AbV2XYKnsNzl65kAEOLPCobUNysKqYnJ8naYCYL3jjwGhW36iwIVANqh',
      'gZgvnVpVXyYx1GlceA5/8mI1AoGBAOyjuQhGoOW4hILR02WrqwCEPqhRyMp87dMU',
      '9z0PQKKrACmiQpEUawvtR4aMB7Xzy+f6MRLwdonQISzswopa7Gwl3CYLiOdKFFIQ',
      'zDdymwGovTjN2fQm8v/UHdafZBPImGBDzf+iykNBhZc7UP1rciMzKOnT3BTF/al8',
      'hhBy0bjsAoGAYrykryPYWUGArKS6NO9Ijtwc7SbbNXZCuBcl6hc1Hdr1UEEKQcsW',
      '5fajF5Ut9ZLwcggsWSKZtZk9NA7xybmObuDEU7oiFFNRAbBkam/v/3a3bwvSHGKg',
      'q15cvC313zg1ii9NXylvBIoIS8EWfekl1LMvHC7NdJ41wPYhhmPrUk4CFHJjavJH',
      'kb6tDtemyiAq3N9aQj/D',
      '-----END DSA PRIVATE KEY-----'
    ].join('\n'),
    expected: {
      fulltype: undefined,
      type: 'dss',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: new Buffer([
        0x30, 0x82, 0x01, 0xbb, 0x02, 0x01, 0x00, 0x02, 0x81, 0x81, 0x00, 0xfb,
        0xbf, 0x59, 0x11, 0xdf, 0x50, 0xf6, 0x36, 0x16, 0x48, 0xcc, 0xe8, 0xc9,
        0x2f, 0x58, 0x4f, 0xce, 0xfc, 0x3b, 0xf4, 0xad, 0xde, 0x59, 0x9a, 0x90,
        0x12, 0x0a, 0x8a, 0x99, 0xe0, 0x7e, 0x3f, 0xd3, 0x21, 0x3b, 0x1e, 0x75,
        0x30, 0xe6, 0x6b, 0xbe, 0xcd, 0xe1, 0x66, 0x86, 0x2a, 0x9a, 0xc3, 0x71,
        0x4f, 0xf6, 0x36, 0xf4, 0x4c, 0x10, 0xfa, 0x8f, 0x92, 0x5e, 0x94, 0xd8,
        0x8c, 0x98, 0x70, 0xfe, 0xcd, 0xaf, 0x47, 0x40, 0x1e, 0x24, 0x18, 0xf7,
        0xda, 0x5f, 0x30, 0x78, 0x3f, 0x34, 0xe3, 0xe4, 0xfe, 0x4b, 0xcc, 0xab,
        0x30, 0x90, 0x5d, 0x82, 0xc7, 0x39, 0x01, 0xb5, 0x76, 0x5d, 0x82, 0xa7,
        0xb0, 0xdc, 0xe5, 0xeb, 0x99, 0x00, 0x10, 0xe2, 0xcf, 0x0a, 0x86, 0xd4,
        0x37, 0x2b, 0x0a, 0xa9, 0x89, 0xc9, 0xf2, 0x76, 0x98, 0x09, 0x82, 0xf7,
        0x8e, 0x3c, 0x06, 0x85, 0x6d, 0xfa, 0x8b, 0x02, 0x15, 0x00, 0xda, 0xa1,
        0x81, 0x98, 0x2f, 0x9d, 0x5a, 0x55, 0x5f, 0x26, 0x31, 0xd4, 0x69, 0x5c,
        0x78, 0x0e, 0x7f, 0xf2, 0x62, 0x35, 0x02, 0x81, 0x81, 0x00, 0xec, 0xa3,
        0xb9, 0x08, 0x46, 0xa0, 0xe5, 0xb8, 0x84, 0x82, 0xd1, 0xd3, 0x65, 0xab,
        0xab, 0x00, 0x84, 0x3e, 0xa8, 0x51, 0xc8, 0xca, 0x7c, 0xed, 0xd3, 0x14,
        0xf7, 0x3d, 0x0f, 0x40, 0xa2, 0xab, 0x00, 0x29, 0xa2, 0x42, 0x91, 0x14,
        0x6b, 0x0b, 0xed, 0x47, 0x86, 0x8c, 0x07, 0xb5, 0xf3, 0xcb, 0xe7, 0xfa,
        0x31, 0x12, 0xf0, 0x76, 0x89, 0xd0, 0x21, 0x2c, 0xec, 0xc2, 0x8a, 0x5a,
        0xec, 0x6c, 0x25, 0xdc, 0x26, 0x0b, 0x88, 0xe7, 0x4a, 0x14, 0x52, 0x10,
        0xcc, 0x37, 0x72, 0x9b, 0x01, 0xa8, 0xbd, 0x38, 0xcd, 0xd9, 0xf4, 0x26,
        0xf2, 0xff, 0xd4, 0x1d, 0xd6, 0x9f, 0x64, 0x13, 0xc8, 0x98, 0x60, 0x43,
        0xcd, 0xff, 0xa2, 0xca, 0x43, 0x41, 0x85, 0x97, 0x3b, 0x50, 0xfd, 0x6b,
        0x72, 0x23, 0x33, 0x28, 0xe9, 0xd3, 0xdc, 0x14, 0xc5, 0xfd, 0xa9, 0x7c,
        0x86, 0x10, 0x72, 0xd1, 0xb8, 0xec, 0x02, 0x81, 0x80, 0x62, 0xbc, 0xa4,
        0xaf, 0x23, 0xd8, 0x59, 0x41, 0x80, 0xac, 0xa4, 0xba, 0x34, 0xef, 0x48,
        0x8e, 0xdc, 0x1c, 0xed, 0x26, 0xdb, 0x35, 0x76, 0x42, 0xb8, 0x17, 0x25,
        0xea, 0x17, 0x35, 0x1d, 0xda, 0xf5, 0x50, 0x41, 0x0a, 0x41, 0xcb, 0x16,
        0xe5, 0xf6, 0xa3, 0x17, 0x95, 0x2d, 0xf5, 0x92, 0xf0, 0x72, 0x08, 0x2c,
        0x59, 0x22, 0x99, 0xb5, 0x99, 0x3d, 0x34, 0x0e, 0xf1, 0xc9, 0xb9, 0x8e,
        0x6e, 0xe0, 0xc4, 0x53, 0xba, 0x22, 0x14, 0x53, 0x51, 0x01, 0xb0, 0x64,
        0x6a, 0x6f, 0xef, 0xff, 0x76, 0xb7, 0x6f, 0x0b, 0xd2, 0x1c, 0x62, 0xa0,
        0xab, 0x5e, 0x5c, 0xbc, 0x2d, 0xf5, 0xdf, 0x38, 0x35, 0x8a, 0x2f, 0x4d,
        0x5f, 0x29, 0x6f, 0x04, 0x8a, 0x08, 0x4b, 0xc1, 0x16, 0x7d, 0xe9, 0x25,
        0xd4, 0xb3, 0x2f, 0x1c, 0x2e, 0xcd, 0x74, 0x9e, 0x35, 0xc0, 0xf6, 0x21,
        0x86, 0x63, 0xeb, 0x52, 0x4e, 0x02, 0x14, 0x72, 0x63, 0x6a, 0xf2, 0x47,
        0x91, 0xbe, 0xad, 0x0e, 0xd7, 0xa6, 0xca, 0x20, 0x2a, 0xdc, 0xdf, 0x5a,
        0x42, 0x3f, 0xc3,
      ]),
      privateOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x44,
        0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
        0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x42,
        0x75, 0x77, 0x49, 0x42, 0x41, 0x41, 0x4b, 0x42, 0x67, 0x51, 0x44, 0x37,
        0x76, 0x31, 0x6b, 0x52, 0x33, 0x31, 0x44, 0x32, 0x4e, 0x68, 0x5a, 0x49,
        0x7a, 0x4f, 0x6a, 0x4a, 0x4c, 0x31, 0x68, 0x50, 0x7a, 0x76, 0x77, 0x37,
        0x39, 0x4b, 0x33, 0x65, 0x57, 0x5a, 0x71, 0x51, 0x45, 0x67, 0x71, 0x4b,
        0x6d, 0x65, 0x42, 0x2b, 0x50, 0x39, 0x4d, 0x68, 0x4f, 0x78, 0x35, 0x31,
        0x0a, 0x4d, 0x4f, 0x5a, 0x72, 0x76, 0x73, 0x33, 0x68, 0x5a, 0x6f, 0x59,
        0x71, 0x6d, 0x73, 0x4e, 0x78, 0x54, 0x2f, 0x59, 0x32, 0x39, 0x45, 0x77,
        0x51, 0x2b, 0x6f, 0x2b, 0x53, 0x58, 0x70, 0x54, 0x59, 0x6a, 0x4a, 0x68,
        0x77, 0x2f, 0x73, 0x32, 0x76, 0x52, 0x30, 0x41, 0x65, 0x4a, 0x42, 0x6a,
        0x33, 0x32, 0x6c, 0x38, 0x77, 0x65, 0x44, 0x38, 0x30, 0x34, 0x2b, 0x54,
        0x2b, 0x53, 0x38, 0x79, 0x72, 0x0a, 0x4d, 0x4a, 0x42, 0x64, 0x67, 0x73,
        0x63, 0x35, 0x41, 0x62, 0x56, 0x32, 0x58, 0x59, 0x4b, 0x6e, 0x73, 0x4e,
        0x7a, 0x6c, 0x36, 0x35, 0x6b, 0x41, 0x45, 0x4f, 0x4c, 0x50, 0x43, 0x6f,
        0x62, 0x55, 0x4e, 0x79, 0x73, 0x4b, 0x71, 0x59, 0x6e, 0x4a, 0x38, 0x6e,
        0x61, 0x59, 0x43, 0x59, 0x4c, 0x33, 0x6a, 0x6a, 0x77, 0x47, 0x68, 0x57,
        0x33, 0x36, 0x69, 0x77, 0x49, 0x56, 0x41, 0x4e, 0x71, 0x68, 0x0a, 0x67,
        0x5a, 0x67, 0x76, 0x6e, 0x56, 0x70, 0x56, 0x58, 0x79, 0x59, 0x78, 0x31,
        0x47, 0x6c, 0x63, 0x65, 0x41, 0x35, 0x2f, 0x38, 0x6d, 0x49, 0x31, 0x41,
        0x6f, 0x47, 0x42, 0x41, 0x4f, 0x79, 0x6a, 0x75, 0x51, 0x68, 0x47, 0x6f,
        0x4f, 0x57, 0x34, 0x68, 0x49, 0x4c, 0x52, 0x30, 0x32, 0x57, 0x72, 0x71,
        0x77, 0x43, 0x45, 0x50, 0x71, 0x68, 0x52, 0x79, 0x4d, 0x70, 0x38, 0x37,
        0x64, 0x4d, 0x55, 0x0a, 0x39, 0x7a, 0x30, 0x50, 0x51, 0x4b, 0x4b, 0x72,
        0x41, 0x43, 0x6d, 0x69, 0x51, 0x70, 0x45, 0x55, 0x61, 0x77, 0x76, 0x74,
        0x52, 0x34, 0x61, 0x4d, 0x42, 0x37, 0x58, 0x7a, 0x79, 0x2b, 0x66, 0x36,
        0x4d, 0x52, 0x4c, 0x77, 0x64, 0x6f, 0x6e, 0x51, 0x49, 0x53, 0x7a, 0x73,
        0x77, 0x6f, 0x70, 0x61, 0x37, 0x47, 0x77, 0x6c, 0x33, 0x43, 0x59, 0x4c,
        0x69, 0x4f, 0x64, 0x4b, 0x46, 0x46, 0x49, 0x51, 0x0a, 0x7a, 0x44, 0x64,
        0x79, 0x6d, 0x77, 0x47, 0x6f, 0x76, 0x54, 0x6a, 0x4e, 0x32, 0x66, 0x51,
        0x6d, 0x38, 0x76, 0x2f, 0x55, 0x48, 0x64, 0x61, 0x66, 0x5a, 0x42, 0x50,
        0x49, 0x6d, 0x47, 0x42, 0x44, 0x7a, 0x66, 0x2b, 0x69, 0x79, 0x6b, 0x4e,
        0x42, 0x68, 0x5a, 0x63, 0x37, 0x55, 0x50, 0x31, 0x72, 0x63, 0x69, 0x4d,
        0x7a, 0x4b, 0x4f, 0x6e, 0x54, 0x33, 0x42, 0x54, 0x46, 0x2f, 0x61, 0x6c,
        0x38, 0x0a, 0x68, 0x68, 0x42, 0x79, 0x30, 0x62, 0x6a, 0x73, 0x41, 0x6f,
        0x47, 0x41, 0x59, 0x72, 0x79, 0x6b, 0x72, 0x79, 0x50, 0x59, 0x57, 0x55,
        0x47, 0x41, 0x72, 0x4b, 0x53, 0x36, 0x4e, 0x4f, 0x39, 0x49, 0x6a, 0x74,
        0x77, 0x63, 0x37, 0x53, 0x62, 0x62, 0x4e, 0x58, 0x5a, 0x43, 0x75, 0x42,
        0x63, 0x6c, 0x36, 0x68, 0x63, 0x31, 0x48, 0x64, 0x72, 0x31, 0x55, 0x45,
        0x45, 0x4b, 0x51, 0x63, 0x73, 0x57, 0x0a, 0x35, 0x66, 0x61, 0x6a, 0x46,
        0x35, 0x55, 0x74, 0x39, 0x5a, 0x4c, 0x77, 0x63, 0x67, 0x67, 0x73, 0x57,
        0x53, 0x4b, 0x5a, 0x74, 0x5a, 0x6b, 0x39, 0x4e, 0x41, 0x37, 0x78, 0x79,
        0x62, 0x6d, 0x4f, 0x62, 0x75, 0x44, 0x45, 0x55, 0x37, 0x6f, 0x69, 0x46,
        0x46, 0x4e, 0x52, 0x41, 0x62, 0x42, 0x6b, 0x61, 0x6d, 0x2f, 0x76, 0x2f,
        0x33, 0x61, 0x33, 0x62, 0x77, 0x76, 0x53, 0x48, 0x47, 0x4b, 0x67, 0x0a,
        0x71, 0x31, 0x35, 0x63, 0x76, 0x43, 0x33, 0x31, 0x33, 0x7a, 0x67, 0x31,
        0x69, 0x69, 0x39, 0x4e, 0x58, 0x79, 0x6c, 0x76, 0x42, 0x49, 0x6f, 0x49,
        0x53, 0x38, 0x45, 0x57, 0x66, 0x65, 0x6b, 0x6c, 0x31, 0x4c, 0x4d, 0x76,
        0x48, 0x43, 0x37, 0x4e, 0x64, 0x4a, 0x34, 0x31, 0x77, 0x50, 0x59, 0x68,
        0x68, 0x6d, 0x50, 0x72, 0x55, 0x6b, 0x34, 0x43, 0x46, 0x48, 0x4a, 0x6a,
        0x61, 0x76, 0x4a, 0x48, 0x0a, 0x6b, 0x62, 0x36, 0x74, 0x44, 0x74, 0x65,
        0x6d, 0x79, 0x69, 0x41, 0x71, 0x33, 0x4e, 0x39, 0x61, 0x51, 0x6a, 0x2f,
        0x44, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x44,
        0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
        0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d,
      ]),
      public: undefined,
      publicOrig: undefined
    },
    what: 'Unencrypted DSA private key (OpenSSH format)'
  },
  { source: [
      '-----BEGIN DSA PRIVATE KEY-----',
      'Proc-Type: 4,ENCRYPTED',
      'DEK-Info: AES-128-CBC,D6218FEC397F57918B06DC3F9EC33487',
      '',
      'OjtbW9WChM2baMDRXW/IYhmqAmlaulb8pQd0rDcFYGrAS/5PnA96R3X6f9tdooSv',
      'uPrQ37oCI7FfDrRmVy6pn6QOLxLZHk7LViBAA7q8VytRHUyezauicTkhc6/nfxQU',
      'tYN5dNajSl/aFuv/IitaYtuG2ORvXLLJhR3sYe5fs2ai14qtULdWxtrZ4cACXPSz',
      'Ij29/lUfeBIYxNFFDF5hgsyO+jHLHO8bJCgXLCEb1o/Im68kQT3LjmvLbdhVM2+V',
      '5c+YVDW/yXKpaSDNqctJFsETP3SsI5k9PVijSeND4U27W9HAoukJmG8jNBXmMOKW',
      'MtOEp0Yw5+fu0OPTACMcK7UoiN90RTFLNcDT9kKjeQwqk37sZBNyOpnQ57Mu6YWX',
      'BS5nnT4bK/WpyWUgGJ+MVzcp6pErBUy3SxlV2LN/LIZq+YzME55sygtwNSuVhpBH',
      '+9UwrUIlaT8Ru//OdRKY5Vp8AqomjlAzCkBxnmC9XhVB0r4zqsqgVyzSfsX+oGLe',
      'dbixpbeMXe3WoAowx6LmqojuP46MIL80/9BH6Xr51UqJj156bYXashJtgpYaAQIn',
      '8YB9sXf4S027sgHxK0Xanw==',
      '-----END DSA PRIVATE KEY-----'
    ].join('\n'),
    expected: {
      fulltype: undefined,
      type: 'dss',
      extra: [ 'D6218FEC397F57918B06DC3F9EC33487' ],
      comment: undefined,
      encryption: 'aes-128-cbc',
      private: new Buffer([
        0x3a, 0x3b, 0x5b, 0x5b, 0xd5, 0x82, 0x84, 0xcd, 0x9b, 0x68, 0xc0, 0xd1,
        0x5d, 0x6f, 0xc8, 0x62, 0x19, 0xaa, 0x02, 0x69, 0x5a, 0xba, 0x56, 0xfc,
        0xa5, 0x07, 0x74, 0xac, 0x37, 0x05, 0x60, 0x6a, 0xc0, 0x4b, 0xfe, 0x4f,
        0x9c, 0x0f, 0x7a, 0x47, 0x75, 0xfa, 0x7f, 0xdb, 0x5d, 0xa2, 0x84, 0xaf,
        0xb8, 0xfa, 0xd0, 0xdf, 0xba, 0x02, 0x23, 0xb1, 0x5f, 0x0e, 0xb4, 0x66,
        0x57, 0x2e, 0xa9, 0x9f, 0xa4, 0x0e, 0x2f, 0x12, 0xd9, 0x1e, 0x4e, 0xcb,
        0x56, 0x20, 0x40, 0x03, 0xba, 0xbc, 0x57, 0x2b, 0x51, 0x1d, 0x4c, 0x9e,
        0xcd, 0xab, 0xa2, 0x71, 0x39, 0x21, 0x73, 0xaf, 0xe7, 0x7f, 0x14, 0x14,
        0xb5, 0x83, 0x79, 0x74, 0xd6, 0xa3, 0x4a, 0x5f, 0xda, 0x16, 0xeb, 0xff,
        0x22, 0x2b, 0x5a, 0x62, 0xdb, 0x86, 0xd8, 0xe4, 0x6f, 0x5c, 0xb2, 0xc9,
        0x85, 0x1d, 0xec, 0x61, 0xee, 0x5f, 0xb3, 0x66, 0xa2, 0xd7, 0x8a, 0xad,
        0x50, 0xb7, 0x56, 0xc6, 0xda, 0xd9, 0xe1, 0xc0, 0x02, 0x5c, 0xf4, 0xb3,
        0x22, 0x3d, 0xbd, 0xfe, 0x55, 0x1f, 0x78, 0x12, 0x18, 0xc4, 0xd1, 0x45,
        0x0c, 0x5e, 0x61, 0x82, 0xcc, 0x8e, 0xfa, 0x31, 0xcb, 0x1c, 0xef, 0x1b,
        0x24, 0x28, 0x17, 0x2c, 0x21, 0x1b, 0xd6, 0x8f, 0xc8, 0x9b, 0xaf, 0x24,
        0x41, 0x3d, 0xcb, 0x8e, 0x6b, 0xcb, 0x6d, 0xd8, 0x55, 0x33, 0x6f, 0x95,
        0xe5, 0xcf, 0x98, 0x54, 0x35, 0xbf, 0xc9, 0x72, 0xa9, 0x69, 0x20, 0xcd,
        0xa9, 0xcb, 0x49, 0x16, 0xc1, 0x13, 0x3f, 0x74, 0xac, 0x23, 0x99, 0x3d,
        0x3d, 0x58, 0xa3, 0x49, 0xe3, 0x43, 0xe1, 0x4d, 0xbb, 0x5b, 0xd1, 0xc0,
        0xa2, 0xe9, 0x09, 0x98, 0x6f, 0x23, 0x34, 0x15, 0xe6, 0x30, 0xe2, 0x96,
        0x32, 0xd3, 0x84, 0xa7, 0x46, 0x30, 0xe7, 0xe7, 0xee, 0xd0, 0xe3, 0xd3,
        0x00, 0x23, 0x1c, 0x2b, 0xb5, 0x28, 0x88, 0xdf, 0x74, 0x45, 0x31, 0x4b,
        0x35, 0xc0, 0xd3, 0xf6, 0x42, 0xa3, 0x79, 0x0c, 0x2a, 0x93, 0x7e, 0xec,
        0x64, 0x13, 0x72, 0x3a, 0x99, 0xd0, 0xe7, 0xb3, 0x2e, 0xe9, 0x85, 0x97,
        0x05, 0x2e, 0x67, 0x9d, 0x3e, 0x1b, 0x2b, 0xf5, 0xa9, 0xc9, 0x65, 0x20,
        0x18, 0x9f, 0x8c, 0x57, 0x37, 0x29, 0xea, 0x91, 0x2b, 0x05, 0x4c, 0xb7,
        0x4b, 0x19, 0x55, 0xd8, 0xb3, 0x7f, 0x2c, 0x86, 0x6a, 0xf9, 0x8c, 0xcc,
        0x13, 0x9e, 0x6c, 0xca, 0x0b, 0x70, 0x35, 0x2b, 0x95, 0x86, 0x90, 0x47,
        0xfb, 0xd5, 0x30, 0xad, 0x42, 0x25, 0x69, 0x3f, 0x11, 0xbb, 0xff, 0xce,
        0x75, 0x12, 0x98, 0xe5, 0x5a, 0x7c, 0x02, 0xaa, 0x26, 0x8e, 0x50, 0x33,
        0x0a, 0x40, 0x71, 0x9e, 0x60, 0xbd, 0x5e, 0x15, 0x41, 0xd2, 0xbe, 0x33,
        0xaa, 0xca, 0xa0, 0x57, 0x2c, 0xd2, 0x7e, 0xc5, 0xfe, 0xa0, 0x62, 0xde,
        0x75, 0xb8, 0xb1, 0xa5, 0xb7, 0x8c, 0x5d, 0xed, 0xd6, 0xa0, 0x0a, 0x30,
        0xc7, 0xa2, 0xe6, 0xaa, 0x88, 0xee, 0x3f, 0x8e, 0x8c, 0x20, 0xbf, 0x34,
        0xff, 0xd0, 0x47, 0xe9, 0x7a, 0xf9, 0xd5, 0x4a, 0x89, 0x8f, 0x5e, 0x7a,
        0x6d, 0x85, 0xda, 0xb2, 0x12, 0x6d, 0x82, 0x96, 0x1a, 0x01, 0x02, 0x27,
        0xf1, 0x80, 0x7d, 0xb1, 0x77, 0xf8, 0x4b, 0x4d, 0xbb, 0xb2, 0x01, 0xf1,
        0x2b, 0x45, 0xda, 0x9f,
      ]),
      privateOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x44,
        0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
        0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x50, 0x72, 0x6f, 0x63,
        0x2d, 0x54, 0x79, 0x70, 0x65, 0x3a, 0x20, 0x34, 0x2c, 0x45, 0x4e, 0x43,
        0x52, 0x59, 0x50, 0x54, 0x45, 0x44, 0x0a, 0x44, 0x45, 0x4b, 0x2d, 0x49,
        0x6e, 0x66, 0x6f, 0x3a, 0x20, 0x41, 0x45, 0x53, 0x2d, 0x31, 0x32, 0x38,
        0x2d, 0x43, 0x42, 0x43, 0x2c, 0x44, 0x36, 0x32, 0x31, 0x38, 0x46, 0x45,
        0x43, 0x33, 0x39, 0x37, 0x46, 0x35, 0x37, 0x39, 0x31, 0x38, 0x42, 0x30,
        0x36, 0x44, 0x43, 0x33, 0x46, 0x39, 0x45, 0x43, 0x33, 0x33, 0x34, 0x38,
        0x37, 0x0a, 0x0a, 0x4f, 0x6a, 0x74, 0x62, 0x57, 0x39, 0x57, 0x43, 0x68,
        0x4d, 0x32, 0x62, 0x61, 0x4d, 0x44, 0x52, 0x58, 0x57, 0x2f, 0x49, 0x59,
        0x68, 0x6d, 0x71, 0x41, 0x6d, 0x6c, 0x61, 0x75, 0x6c, 0x62, 0x38, 0x70,
        0x51, 0x64, 0x30, 0x72, 0x44, 0x63, 0x46, 0x59, 0x47, 0x72, 0x41, 0x53,
        0x2f, 0x35, 0x50, 0x6e, 0x41, 0x39, 0x36, 0x52, 0x33, 0x58, 0x36, 0x66,
        0x39, 0x74, 0x64, 0x6f, 0x6f, 0x53, 0x76, 0x0a, 0x75, 0x50, 0x72, 0x51,
        0x33, 0x37, 0x6f, 0x43, 0x49, 0x37, 0x46, 0x66, 0x44, 0x72, 0x52, 0x6d,
        0x56, 0x79, 0x36, 0x70, 0x6e, 0x36, 0x51, 0x4f, 0x4c, 0x78, 0x4c, 0x5a,
        0x48, 0x6b, 0x37, 0x4c, 0x56, 0x69, 0x42, 0x41, 0x41, 0x37, 0x71, 0x38,
        0x56, 0x79, 0x74, 0x52, 0x48, 0x55, 0x79, 0x65, 0x7a, 0x61, 0x75, 0x69,
        0x63, 0x54, 0x6b, 0x68, 0x63, 0x36, 0x2f, 0x6e, 0x66, 0x78, 0x51, 0x55,
        0x0a, 0x74, 0x59, 0x4e, 0x35, 0x64, 0x4e, 0x61, 0x6a, 0x53, 0x6c, 0x2f,
        0x61, 0x46, 0x75, 0x76, 0x2f, 0x49, 0x69, 0x74, 0x61, 0x59, 0x74, 0x75,
        0x47, 0x32, 0x4f, 0x52, 0x76, 0x58, 0x4c, 0x4c, 0x4a, 0x68, 0x52, 0x33,
        0x73, 0x59, 0x65, 0x35, 0x66, 0x73, 0x32, 0x61, 0x69, 0x31, 0x34, 0x71,
        0x74, 0x55, 0x4c, 0x64, 0x57, 0x78, 0x74, 0x72, 0x5a, 0x34, 0x63, 0x41,
        0x43, 0x58, 0x50, 0x53, 0x7a, 0x0a, 0x49, 0x6a, 0x32, 0x39, 0x2f, 0x6c,
        0x55, 0x66, 0x65, 0x42, 0x49, 0x59, 0x78, 0x4e, 0x46, 0x46, 0x44, 0x46,
        0x35, 0x68, 0x67, 0x73, 0x79, 0x4f, 0x2b, 0x6a, 0x48, 0x4c, 0x48, 0x4f,
        0x38, 0x62, 0x4a, 0x43, 0x67, 0x58, 0x4c, 0x43, 0x45, 0x62, 0x31, 0x6f,
        0x2f, 0x49, 0x6d, 0x36, 0x38, 0x6b, 0x51, 0x54, 0x33, 0x4c, 0x6a, 0x6d,
        0x76, 0x4c, 0x62, 0x64, 0x68, 0x56, 0x4d, 0x32, 0x2b, 0x56, 0x0a, 0x35,
        0x63, 0x2b, 0x59, 0x56, 0x44, 0x57, 0x2f, 0x79, 0x58, 0x4b, 0x70, 0x61,
        0x53, 0x44, 0x4e, 0x71, 0x63, 0x74, 0x4a, 0x46, 0x73, 0x45, 0x54, 0x50,
        0x33, 0x53, 0x73, 0x49, 0x35, 0x6b, 0x39, 0x50, 0x56, 0x69, 0x6a, 0x53,
        0x65, 0x4e, 0x44, 0x34, 0x55, 0x32, 0x37, 0x57, 0x39, 0x48, 0x41, 0x6f,
        0x75, 0x6b, 0x4a, 0x6d, 0x47, 0x38, 0x6a, 0x4e, 0x42, 0x58, 0x6d, 0x4d,
        0x4f, 0x4b, 0x57, 0x0a, 0x4d, 0x74, 0x4f, 0x45, 0x70, 0x30, 0x59, 0x77,
        0x35, 0x2b, 0x66, 0x75, 0x30, 0x4f, 0x50, 0x54, 0x41, 0x43, 0x4d, 0x63,
        0x4b, 0x37, 0x55, 0x6f, 0x69, 0x4e, 0x39, 0x30, 0x52, 0x54, 0x46, 0x4c,
        0x4e, 0x63, 0x44, 0x54, 0x39, 0x6b, 0x4b, 0x6a, 0x65, 0x51, 0x77, 0x71,
        0x6b, 0x33, 0x37, 0x73, 0x5a, 0x42, 0x4e, 0x79, 0x4f, 0x70, 0x6e, 0x51,
        0x35, 0x37, 0x4d, 0x75, 0x36, 0x59, 0x57, 0x58, 0x0a, 0x42, 0x53, 0x35,
        0x6e, 0x6e, 0x54, 0x34, 0x62, 0x4b, 0x2f, 0x57, 0x70, 0x79, 0x57, 0x55,
        0x67, 0x47, 0x4a, 0x2b, 0x4d, 0x56, 0x7a, 0x63, 0x70, 0x36, 0x70, 0x45,
        0x72, 0x42, 0x55, 0x79, 0x33, 0x53, 0x78, 0x6c, 0x56, 0x32, 0x4c, 0x4e,
        0x2f, 0x4c, 0x49, 0x5a, 0x71, 0x2b, 0x59, 0x7a, 0x4d, 0x45, 0x35, 0x35,
        0x73, 0x79, 0x67, 0x74, 0x77, 0x4e, 0x53, 0x75, 0x56, 0x68, 0x70, 0x42,
        0x48, 0x0a, 0x2b, 0x39, 0x55, 0x77, 0x72, 0x55, 0x49, 0x6c, 0x61, 0x54,
        0x38, 0x52, 0x75, 0x2f, 0x2f, 0x4f, 0x64, 0x52, 0x4b, 0x59, 0x35, 0x56,
        0x70, 0x38, 0x41, 0x71, 0x6f, 0x6d, 0x6a, 0x6c, 0x41, 0x7a, 0x43, 0x6b,
        0x42, 0x78, 0x6e, 0x6d, 0x43, 0x39, 0x58, 0x68, 0x56, 0x42, 0x30, 0x72,
        0x34, 0x7a, 0x71, 0x73, 0x71, 0x67, 0x56, 0x79, 0x7a, 0x53, 0x66, 0x73,
        0x58, 0x2b, 0x6f, 0x47, 0x4c, 0x65, 0x0a, 0x64, 0x62, 0x69, 0x78, 0x70,
        0x62, 0x65, 0x4d, 0x58, 0x65, 0x33, 0x57, 0x6f, 0x41, 0x6f, 0x77, 0x78,
        0x36, 0x4c, 0x6d, 0x71, 0x6f, 0x6a, 0x75, 0x50, 0x34, 0x36, 0x4d, 0x49,
        0x4c, 0x38, 0x30, 0x2f, 0x39, 0x42, 0x48, 0x36, 0x58, 0x72, 0x35, 0x31,
        0x55, 0x71, 0x4a, 0x6a, 0x31, 0x35, 0x36, 0x62, 0x59, 0x58, 0x61, 0x73,
        0x68, 0x4a, 0x74, 0x67, 0x70, 0x59, 0x61, 0x41, 0x51, 0x49, 0x6e, 0x0a,
        0x38, 0x59, 0x42, 0x39, 0x73, 0x58, 0x66, 0x34, 0x53, 0x30, 0x32, 0x37,
        0x73, 0x67, 0x48, 0x78, 0x4b, 0x30, 0x58, 0x61, 0x6e, 0x77, 0x3d, 0x3d,
        0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x44, 0x53,
        0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45,
        0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d,
      ]),
      public: undefined,
      publicOrig: undefined
    },
    what: 'Encrypted DSA private key (OpenSSH format)'
  },
  { source: 'ssh-dss AAAAB3NzaC1kc3MAAACBAL1FzdhtSYN22YgbInypxa//RyKiZAvWxNzDKNF0cYvabH6ESjdwdEcW96H7pHaKb69hoCHKZzfWmdpj93CcezGnorWvUaO4nAvbR+l/gH+AEYzHmJsECvM/nWnmPltwbvrPi/NmXCNf5aeqQlPbL44vZZjFRcCEwOomGY2FWUKJAAAAFQDBD5Y6VM8slL2I/HhqTmnoVFuGmQAAAIAUkwKajnUklN4CX/ZWEhIOZWCRa6ZKCosICBc/yC4RGJldqsXaM0PCmdy0dmcv0XuOZ4Nb92aiZnWdGRXTN5KJ405WVOQLBB3LpebA6qeyDGiVt/iPOxgJGZy0LIWiqWuzAAf199cQ2BkWmGTC6AAJi1depXC+KVAP+HwXOLjSUAAAAIBeoyKKEiVDQOpB0KAr5RLH3/+5BP+mR+52gTAbxZbxQPYyH7mqq2/PI1A0bjnvnLe45a/iVumoIjP/atMtVJ6w103Ex5JGMwVB7uZBWeyJ4m3MqQdPf1/+r+UCoOaL7stK3rWeTCydb82z34ejX0dZobAPGj2F7SyZy7jM8THgcw==',
    expected: {
      fulltype: 'ssh-dss',
      type: 'dss',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x07, 0x73, 0x73, 0x68, 0x2d, 0x64, 0x73, 0x73, 0x00,
        0x00, 0x00, 0x81, 0x00, 0xbd, 0x45, 0xcd, 0xd8, 0x6d, 0x49, 0x83, 0x76,
        0xd9, 0x88, 0x1b, 0x22, 0x7c, 0xa9, 0xc5, 0xaf, 0xff, 0x47, 0x22, 0xa2,
        0x64, 0x0b, 0xd6, 0xc4, 0xdc, 0xc3, 0x28, 0xd1, 0x74, 0x71, 0x8b, 0xda,
        0x6c, 0x7e, 0x84, 0x4a, 0x37, 0x70, 0x74, 0x47, 0x16, 0xf7, 0xa1, 0xfb,
        0xa4, 0x76, 0x8a, 0x6f, 0xaf, 0x61, 0xa0, 0x21, 0xca, 0x67, 0x37, 0xd6,
        0x99, 0xda, 0x63, 0xf7, 0x70, 0x9c, 0x7b, 0x31, 0xa7, 0xa2, 0xb5, 0xaf,
        0x51, 0xa3, 0xb8, 0x9c, 0x0b, 0xdb, 0x47, 0xe9, 0x7f, 0x80, 0x7f, 0x80,
        0x11, 0x8c, 0xc7, 0x98, 0x9b, 0x04, 0x0a, 0xf3, 0x3f, 0x9d, 0x69, 0xe6,
        0x3e, 0x5b, 0x70, 0x6e, 0xfa, 0xcf, 0x8b, 0xf3, 0x66, 0x5c, 0x23, 0x5f,
        0xe5, 0xa7, 0xaa, 0x42, 0x53, 0xdb, 0x2f, 0x8e, 0x2f, 0x65, 0x98, 0xc5,
        0x45, 0xc0, 0x84, 0xc0, 0xea, 0x26, 0x19, 0x8d, 0x85, 0x59, 0x42, 0x89,
        0x00, 0x00, 0x00, 0x15, 0x00, 0xc1, 0x0f, 0x96, 0x3a, 0x54, 0xcf, 0x2c,
        0x94, 0xbd, 0x88, 0xfc, 0x78, 0x6a, 0x4e, 0x69, 0xe8, 0x54, 0x5b, 0x86,
        0x99, 0x00, 0x00, 0x00, 0x80, 0x14, 0x93, 0x02, 0x9a, 0x8e, 0x75, 0x24,
        0x94, 0xde, 0x02, 0x5f, 0xf6, 0x56, 0x12, 0x12, 0x0e, 0x65, 0x60, 0x91,
        0x6b, 0xa6, 0x4a, 0x0a, 0x8b, 0x08, 0x08, 0x17, 0x3f, 0xc8, 0x2e, 0x11,
        0x18, 0x99, 0x5d, 0xaa, 0xc5, 0xda, 0x33, 0x43, 0xc2, 0x99, 0xdc, 0xb4,
        0x76, 0x67, 0x2f, 0xd1, 0x7b, 0x8e, 0x67, 0x83, 0x5b, 0xf7, 0x66, 0xa2,
        0x66, 0x75, 0x9d, 0x19, 0x15, 0xd3, 0x37, 0x92, 0x89, 0xe3, 0x4e, 0x56,
        0x54, 0xe4, 0x0b, 0x04, 0x1d, 0xcb, 0xa5, 0xe6, 0xc0, 0xea, 0xa7, 0xb2,
        0x0c, 0x68, 0x95, 0xb7, 0xf8, 0x8f, 0x3b, 0x18, 0x09, 0x19, 0x9c, 0xb4,
        0x2c, 0x85, 0xa2, 0xa9, 0x6b, 0xb3, 0x00, 0x07, 0xf5, 0xf7, 0xd7, 0x10,
        0xd8, 0x19, 0x16, 0x98, 0x64, 0xc2, 0xe8, 0x00, 0x09, 0x8b, 0x57, 0x5e,
        0xa5, 0x70, 0xbe, 0x29, 0x50, 0x0f, 0xf8, 0x7c, 0x17, 0x38, 0xb8, 0xd2,
        0x50, 0x00, 0x00, 0x00, 0x80, 0x5e, 0xa3, 0x22, 0x8a, 0x12, 0x25, 0x43,
        0x40, 0xea, 0x41, 0xd0, 0xa0, 0x2b, 0xe5, 0x12, 0xc7, 0xdf, 0xff, 0xb9,
        0x04, 0xff, 0xa6, 0x47, 0xee, 0x76, 0x81, 0x30, 0x1b, 0xc5, 0x96, 0xf1,
        0x40, 0xf6, 0x32, 0x1f, 0xb9, 0xaa, 0xab, 0x6f, 0xcf, 0x23, 0x50, 0x34,
        0x6e, 0x39, 0xef, 0x9c, 0xb7, 0xb8, 0xe5, 0xaf, 0xe2, 0x56, 0xe9, 0xa8,
        0x22, 0x33, 0xff, 0x6a, 0xd3, 0x2d, 0x54, 0x9e, 0xb0, 0xd7, 0x4d, 0xc4,
        0xc7, 0x92, 0x46, 0x33, 0x05, 0x41, 0xee, 0xe6, 0x41, 0x59, 0xec, 0x89,
        0xe2, 0x6d, 0xcc, 0xa9, 0x07, 0x4f, 0x7f, 0x5f, 0xfe, 0xaf, 0xe5, 0x02,
        0xa0, 0xe6, 0x8b, 0xee, 0xcb, 0x4a, 0xde, 0xb5, 0x9e, 0x4c, 0x2c, 0x9d,
        0x6f, 0xcd, 0xb3, 0xdf, 0x87, 0xa3, 0x5f, 0x47, 0x59, 0xa1, 0xb0, 0x0f,
        0x1a, 0x3d, 0x85, 0xed, 0x2c, 0x99, 0xcb, 0xb8, 0xcc, 0xf1, 0x31, 0xe0,
        0x73,
      ]),
      publicOrig: new Buffer([
        0x73, 0x73, 0x68, 0x2d, 0x64, 0x73, 0x73, 0x20, 0x41, 0x41, 0x41, 0x41,
        0x42, 0x33, 0x4e, 0x7a, 0x61, 0x43, 0x31, 0x6b, 0x63, 0x33, 0x4d, 0x41,
        0x41, 0x41, 0x43, 0x42, 0x41, 0x4c, 0x31, 0x46, 0x7a, 0x64, 0x68, 0x74,
        0x53, 0x59, 0x4e, 0x32, 0x32, 0x59, 0x67, 0x62, 0x49, 0x6e, 0x79, 0x70,
        0x78, 0x61, 0x2f, 0x2f, 0x52, 0x79, 0x4b, 0x69, 0x5a, 0x41, 0x76, 0x57,
        0x78, 0x4e, 0x7a, 0x44, 0x4b, 0x4e, 0x46, 0x30, 0x63, 0x59, 0x76, 0x61,
        0x62, 0x48, 0x36, 0x45, 0x53, 0x6a, 0x64, 0x77, 0x64, 0x45, 0x63, 0x57,
        0x39, 0x36, 0x48, 0x37, 0x70, 0x48, 0x61, 0x4b, 0x62, 0x36, 0x39, 0x68,
        0x6f, 0x43, 0x48, 0x4b, 0x5a, 0x7a, 0x66, 0x57, 0x6d, 0x64, 0x70, 0x6a,
        0x39, 0x33, 0x43, 0x63, 0x65, 0x7a, 0x47, 0x6e, 0x6f, 0x72, 0x57, 0x76,
        0x55, 0x61, 0x4f, 0x34, 0x6e, 0x41, 0x76, 0x62, 0x52, 0x2b, 0x6c, 0x2f,
        0x67, 0x48, 0x2b, 0x41, 0x45, 0x59, 0x7a, 0x48, 0x6d, 0x4a, 0x73, 0x45,
        0x43, 0x76, 0x4d, 0x2f, 0x6e, 0x57, 0x6e, 0x6d, 0x50, 0x6c, 0x74, 0x77,
        0x62, 0x76, 0x72, 0x50, 0x69, 0x2f, 0x4e, 0x6d, 0x58, 0x43, 0x4e, 0x66,
        0x35, 0x61, 0x65, 0x71, 0x51, 0x6c, 0x50, 0x62, 0x4c, 0x34, 0x34, 0x76,
        0x5a, 0x5a, 0x6a, 0x46, 0x52, 0x63, 0x43, 0x45, 0x77, 0x4f, 0x6f, 0x6d,
        0x47, 0x59, 0x32, 0x46, 0x57, 0x55, 0x4b, 0x4a, 0x41, 0x41, 0x41, 0x41,
        0x46, 0x51, 0x44, 0x42, 0x44, 0x35, 0x59, 0x36, 0x56, 0x4d, 0x38, 0x73,
        0x6c, 0x4c, 0x32, 0x49, 0x2f, 0x48, 0x68, 0x71, 0x54, 0x6d, 0x6e, 0x6f,
        0x56, 0x46, 0x75, 0x47, 0x6d, 0x51, 0x41, 0x41, 0x41, 0x49, 0x41, 0x55,
        0x6b, 0x77, 0x4b, 0x61, 0x6a, 0x6e, 0x55, 0x6b, 0x6c, 0x4e, 0x34, 0x43,
        0x58, 0x2f, 0x5a, 0x57, 0x45, 0x68, 0x49, 0x4f, 0x5a, 0x57, 0x43, 0x52,
        0x61, 0x36, 0x5a, 0x4b, 0x43, 0x6f, 0x73, 0x49, 0x43, 0x42, 0x63, 0x2f,
        0x79, 0x43, 0x34, 0x52, 0x47, 0x4a, 0x6c, 0x64, 0x71, 0x73, 0x58, 0x61,
        0x4d, 0x30, 0x50, 0x43, 0x6d, 0x64, 0x79, 0x30, 0x64, 0x6d, 0x63, 0x76,
        0x30, 0x58, 0x75, 0x4f, 0x5a, 0x34, 0x4e, 0x62, 0x39, 0x32, 0x61, 0x69,
        0x5a, 0x6e, 0x57, 0x64, 0x47, 0x52, 0x58, 0x54, 0x4e, 0x35, 0x4b, 0x4a,
        0x34, 0x30, 0x35, 0x57, 0x56, 0x4f, 0x51, 0x4c, 0x42, 0x42, 0x33, 0x4c,
        0x70, 0x65, 0x62, 0x41, 0x36, 0x71, 0x65, 0x79, 0x44, 0x47, 0x69, 0x56,
        0x74, 0x2f, 0x69, 0x50, 0x4f, 0x78, 0x67, 0x4a, 0x47, 0x5a, 0x79, 0x30,
        0x4c, 0x49, 0x57, 0x69, 0x71, 0x57, 0x75, 0x7a, 0x41, 0x41, 0x66, 0x31,
        0x39, 0x39, 0x63, 0x51, 0x32, 0x42, 0x6b, 0x57, 0x6d, 0x47, 0x54, 0x43,
        0x36, 0x41, 0x41, 0x4a, 0x69, 0x31, 0x64, 0x65, 0x70, 0x58, 0x43, 0x2b,
        0x4b, 0x56, 0x41, 0x50, 0x2b, 0x48, 0x77, 0x58, 0x4f, 0x4c, 0x6a, 0x53,
        0x55, 0x41, 0x41, 0x41, 0x41, 0x49, 0x42, 0x65, 0x6f, 0x79, 0x4b, 0x4b,
        0x45, 0x69, 0x56, 0x44, 0x51, 0x4f, 0x70, 0x42, 0x30, 0x4b, 0x41, 0x72,
        0x35, 0x52, 0x4c, 0x48, 0x33, 0x2f, 0x2b, 0x35, 0x42, 0x50, 0x2b, 0x6d,
        0x52, 0x2b, 0x35, 0x32, 0x67, 0x54, 0x41, 0x62, 0x78, 0x5a, 0x62, 0x78,
        0x51, 0x50, 0x59, 0x79, 0x48, 0x37, 0x6d, 0x71, 0x71, 0x32, 0x2f, 0x50,
        0x49, 0x31, 0x41, 0x30, 0x62, 0x6a, 0x6e, 0x76, 0x6e, 0x4c, 0x65, 0x34,
        0x35, 0x61, 0x2f, 0x69, 0x56, 0x75, 0x6d, 0x6f, 0x49, 0x6a, 0x50, 0x2f,
        0x61, 0x74, 0x4d, 0x74, 0x56, 0x4a, 0x36, 0x77, 0x31, 0x30, 0x33, 0x45,
        0x78, 0x35, 0x4a, 0x47, 0x4d, 0x77, 0x56, 0x42, 0x37, 0x75, 0x5a, 0x42,
        0x57, 0x65, 0x79, 0x4a, 0x34, 0x6d, 0x33, 0x4d, 0x71, 0x51, 0x64, 0x50,
        0x66, 0x31, 0x2f, 0x2b, 0x72, 0x2b, 0x55, 0x43, 0x6f, 0x4f, 0x61, 0x4c,
        0x37, 0x73, 0x74, 0x4b, 0x33, 0x72, 0x57, 0x65, 0x54, 0x43, 0x79, 0x64,
        0x62, 0x38, 0x32, 0x7a, 0x33, 0x34, 0x65, 0x6a, 0x58, 0x30, 0x64, 0x5a,
        0x6f, 0x62, 0x41, 0x50, 0x47, 0x6a, 0x32, 0x46, 0x37, 0x53, 0x79, 0x5a,
        0x79, 0x37, 0x6a, 0x4d, 0x38, 0x54, 0x48, 0x67, 0x63, 0x77, 0x3d, 0x3d,
      ])
    },
    what: 'DSA public key (OpenSSH format)'
  },
  { source: 'ssh-dss AAAAB3NzaC1kc3MAAACBAILCaN5QbaErJBa0nNoCY2QvvgcSd7WMgIgQ2qPAPH3Sav+9SWZB0SlHP8QYmV/ntBufaRw9ZdIop6esZBoXmNXDxjcfwpJAzrT3qOMfHXeNcc7w/6dHp+2DlkWD3yzK07SheIZnGNtF1fpfrUvkHFA8UX96awWRLnYuSvWvAK9dAAAAFQCrfeadO0tCCwgnq97Skk0Ng/xV6wAAAIBcVYaMkjXNgfvcUS4pwMabY8rFNGbYo9Jgv7IU2LGFEzD6yFbp2t1sQw031EYlad4n3BMTrZdE6K2/SMi3f3yRr0Z6S6nVSbo7/kIT8/tGJAj9/8RHdCihrFUMaAJNzdk5dmbyaCzexLmLXlKfUVW9A1WOsASjeJBwpFrJ8deuVAAAAIAu/zdW1qxdUVd0b7sjWFFZpp3l6HgOXlZpaOYchIoybs7n2dkYwmvn9glNcLCvRqiZpm2oEZIg1dy8RaH5b7AqY2huwmv+hLZ1XkUFyS2Ae2AKux5rslmiKmvglbDY0Rng1Wj3r3/N0KcKke1g1ohEUaQfXnz7VVx2YoVxhT1dTQ== testing ssh2 from node.js',
    expected: {
      fulltype: 'ssh-dss',
      type: 'dss',
      extra: undefined,
      comment: 'testing ssh2 from node.js',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x07, 0x73, 0x73, 0x68, 0x2d, 0x64, 0x73, 0x73, 0x00,
        0x00, 0x00, 0x81, 0x00, 0x82, 0xc2, 0x68, 0xde, 0x50, 0x6d, 0xa1, 0x2b,
        0x24, 0x16, 0xb4, 0x9c, 0xda, 0x02, 0x63, 0x64, 0x2f, 0xbe, 0x07, 0x12,
        0x77, 0xb5, 0x8c, 0x80, 0x88, 0x10, 0xda, 0xa3, 0xc0, 0x3c, 0x7d, 0xd2,
        0x6a, 0xff, 0xbd, 0x49, 0x66, 0x41, 0xd1, 0x29, 0x47, 0x3f, 0xc4, 0x18,
        0x99, 0x5f, 0xe7, 0xb4, 0x1b, 0x9f, 0x69, 0x1c, 0x3d, 0x65, 0xd2, 0x28,
        0xa7, 0xa7, 0xac, 0x64, 0x1a, 0x17, 0x98, 0xd5, 0xc3, 0xc6, 0x37, 0x1f,
        0xc2, 0x92, 0x40, 0xce, 0xb4, 0xf7, 0xa8, 0xe3, 0x1f, 0x1d, 0x77, 0x8d,
        0x71, 0xce, 0xf0, 0xff, 0xa7, 0x47, 0xa7, 0xed, 0x83, 0x96, 0x45, 0x83,
        0xdf, 0x2c, 0xca, 0xd3, 0xb4, 0xa1, 0x78, 0x86, 0x67, 0x18, 0xdb, 0x45,
        0xd5, 0xfa, 0x5f, 0xad, 0x4b, 0xe4, 0x1c, 0x50, 0x3c, 0x51, 0x7f, 0x7a,
        0x6b, 0x05, 0x91, 0x2e, 0x76, 0x2e, 0x4a, 0xf5, 0xaf, 0x00, 0xaf, 0x5d,
        0x00, 0x00, 0x00, 0x15, 0x00, 0xab, 0x7d, 0xe6, 0x9d, 0x3b, 0x4b, 0x42,
        0x0b, 0x08, 0x27, 0xab, 0xde, 0xd2, 0x92, 0x4d, 0x0d, 0x83, 0xfc, 0x55,
        0xeb, 0x00, 0x00, 0x00, 0x80, 0x5c, 0x55, 0x86, 0x8c, 0x92, 0x35, 0xcd,
        0x81, 0xfb, 0xdc, 0x51, 0x2e, 0x29, 0xc0, 0xc6, 0x9b, 0x63, 0xca, 0xc5,
        0x34, 0x66, 0xd8, 0xa3, 0xd2, 0x60, 0xbf, 0xb2, 0x14, 0xd8, 0xb1, 0x85,
        0x13, 0x30, 0xfa, 0xc8, 0x56, 0xe9, 0xda, 0xdd, 0x6c, 0x43, 0x0d, 0x37,
        0xd4, 0x46, 0x25, 0x69, 0xde, 0x27, 0xdc, 0x13, 0x13, 0xad, 0x97, 0x44,
        0xe8, 0xad, 0xbf, 0x48, 0xc8, 0xb7, 0x7f, 0x7c, 0x91, 0xaf, 0x46, 0x7a,
        0x4b, 0xa9, 0xd5, 0x49, 0xba, 0x3b, 0xfe, 0x42, 0x13, 0xf3, 0xfb, 0x46,
        0x24, 0x08, 0xfd, 0xff, 0xc4, 0x47, 0x74, 0x28, 0xa1, 0xac, 0x55, 0x0c,
        0x68, 0x02, 0x4d, 0xcd, 0xd9, 0x39, 0x76, 0x66, 0xf2, 0x68, 0x2c, 0xde,
        0xc4, 0xb9, 0x8b, 0x5e, 0x52, 0x9f, 0x51, 0x55, 0xbd, 0x03, 0x55, 0x8e,
        0xb0, 0x04, 0xa3, 0x78, 0x90, 0x70, 0xa4, 0x5a, 0xc9, 0xf1, 0xd7, 0xae,
        0x54, 0x00, 0x00, 0x00, 0x80, 0x2e, 0xff, 0x37, 0x56, 0xd6, 0xac, 0x5d,
        0x51, 0x57, 0x74, 0x6f, 0xbb, 0x23, 0x58, 0x51, 0x59, 0xa6, 0x9d, 0xe5,
        0xe8, 0x78, 0x0e, 0x5e, 0x56, 0x69, 0x68, 0xe6, 0x1c, 0x84, 0x8a, 0x32,
        0x6e, 0xce, 0xe7, 0xd9, 0xd9, 0x18, 0xc2, 0x6b, 0xe7, 0xf6, 0x09, 0x4d,
        0x70, 0xb0, 0xaf, 0x46, 0xa8, 0x99, 0xa6, 0x6d, 0xa8, 0x11, 0x92, 0x20,
        0xd5, 0xdc, 0xbc, 0x45, 0xa1, 0xf9, 0x6f, 0xb0, 0x2a, 0x63, 0x68, 0x6e,
        0xc2, 0x6b, 0xfe, 0x84, 0xb6, 0x75, 0x5e, 0x45, 0x05, 0xc9, 0x2d, 0x80,
        0x7b, 0x60, 0x0a, 0xbb, 0x1e, 0x6b, 0xb2, 0x59, 0xa2, 0x2a, 0x6b, 0xe0,
        0x95, 0xb0, 0xd8, 0xd1, 0x19, 0xe0, 0xd5, 0x68, 0xf7, 0xaf, 0x7f, 0xcd,
        0xd0, 0xa7, 0x0a, 0x91, 0xed, 0x60, 0xd6, 0x88, 0x44, 0x51, 0xa4, 0x1f,
        0x5e, 0x7c, 0xfb, 0x55, 0x5c, 0x76, 0x62, 0x85, 0x71, 0x85, 0x3d, 0x5d,
        0x4d,
      ]),
      publicOrig: new Buffer([
        0x73, 0x73, 0x68, 0x2d, 0x64, 0x73, 0x73, 0x20, 0x41, 0x41, 0x41, 0x41,
        0x42, 0x33, 0x4e, 0x7a, 0x61, 0x43, 0x31, 0x6b, 0x63, 0x33, 0x4d, 0x41,
        0x41, 0x41, 0x43, 0x42, 0x41, 0x49, 0x4c, 0x43, 0x61, 0x4e, 0x35, 0x51,
        0x62, 0x61, 0x45, 0x72, 0x4a, 0x42, 0x61, 0x30, 0x6e, 0x4e, 0x6f, 0x43,
        0x59, 0x32, 0x51, 0x76, 0x76, 0x67, 0x63, 0x53, 0x64, 0x37, 0x57, 0x4d,
        0x67, 0x49, 0x67, 0x51, 0x32, 0x71, 0x50, 0x41, 0x50, 0x48, 0x33, 0x53,
        0x61, 0x76, 0x2b, 0x39, 0x53, 0x57, 0x5a, 0x42, 0x30, 0x53, 0x6c, 0x48,
        0x50, 0x38, 0x51, 0x59, 0x6d, 0x56, 0x2f, 0x6e, 0x74, 0x42, 0x75, 0x66,
        0x61, 0x52, 0x77, 0x39, 0x5a, 0x64, 0x49, 0x6f, 0x70, 0x36, 0x65, 0x73,
        0x5a, 0x42, 0x6f, 0x58, 0x6d, 0x4e, 0x58, 0x44, 0x78, 0x6a, 0x63, 0x66,
        0x77, 0x70, 0x4a, 0x41, 0x7a, 0x72, 0x54, 0x33, 0x71, 0x4f, 0x4d, 0x66,
        0x48, 0x58, 0x65, 0x4e, 0x63, 0x63, 0x37, 0x77, 0x2f, 0x36, 0x64, 0x48,
        0x70, 0x2b, 0x32, 0x44, 0x6c, 0x6b, 0x57, 0x44, 0x33, 0x79, 0x7a, 0x4b,
        0x30, 0x37, 0x53, 0x68, 0x65, 0x49, 0x5a, 0x6e, 0x47, 0x4e, 0x74, 0x46,
        0x31, 0x66, 0x70, 0x66, 0x72, 0x55, 0x76, 0x6b, 0x48, 0x46, 0x41, 0x38,
        0x55, 0x58, 0x39, 0x36, 0x61, 0x77, 0x57, 0x52, 0x4c, 0x6e, 0x59, 0x75,
        0x53, 0x76, 0x57, 0x76, 0x41, 0x4b, 0x39, 0x64, 0x41, 0x41, 0x41, 0x41,
        0x46, 0x51, 0x43, 0x72, 0x66, 0x65, 0x61, 0x64, 0x4f, 0x30, 0x74, 0x43,
        0x43, 0x77, 0x67, 0x6e, 0x71, 0x39, 0x37, 0x53, 0x6b, 0x6b, 0x30, 0x4e,
        0x67, 0x2f, 0x78, 0x56, 0x36, 0x77, 0x41, 0x41, 0x41, 0x49, 0x42, 0x63,
        0x56, 0x59, 0x61, 0x4d, 0x6b, 0x6a, 0x58, 0x4e, 0x67, 0x66, 0x76, 0x63,
        0x55, 0x53, 0x34, 0x70, 0x77, 0x4d, 0x61, 0x62, 0x59, 0x38, 0x72, 0x46,
        0x4e, 0x47, 0x62, 0x59, 0x6f, 0x39, 0x4a, 0x67, 0x76, 0x37, 0x49, 0x55,
        0x32, 0x4c, 0x47, 0x46, 0x45, 0x7a, 0x44, 0x36, 0x79, 0x46, 0x62, 0x70,
        0x32, 0x74, 0x31, 0x73, 0x51, 0x77, 0x30, 0x33, 0x31, 0x45, 0x59, 0x6c,
        0x61, 0x64, 0x34, 0x6e, 0x33, 0x42, 0x4d, 0x54, 0x72, 0x5a, 0x64, 0x45,
        0x36, 0x4b, 0x32, 0x2f, 0x53, 0x4d, 0x69, 0x33, 0x66, 0x33, 0x79, 0x52,
        0x72, 0x30, 0x5a, 0x36, 0x53, 0x36, 0x6e, 0x56, 0x53, 0x62, 0x6f, 0x37,
        0x2f, 0x6b, 0x49, 0x54, 0x38, 0x2f, 0x74, 0x47, 0x4a, 0x41, 0x6a, 0x39,
        0x2f, 0x38, 0x52, 0x48, 0x64, 0x43, 0x69, 0x68, 0x72, 0x46, 0x55, 0x4d,
        0x61, 0x41, 0x4a, 0x4e, 0x7a, 0x64, 0x6b, 0x35, 0x64, 0x6d, 0x62, 0x79,
        0x61, 0x43, 0x7a, 0x65, 0x78, 0x4c, 0x6d, 0x4c, 0x58, 0x6c, 0x4b, 0x66,
        0x55, 0x56, 0x57, 0x39, 0x41, 0x31, 0x57, 0x4f, 0x73, 0x41, 0x53, 0x6a,
        0x65, 0x4a, 0x42, 0x77, 0x70, 0x46, 0x72, 0x4a, 0x38, 0x64, 0x65, 0x75,
        0x56, 0x41, 0x41, 0x41, 0x41, 0x49, 0x41, 0x75, 0x2f, 0x7a, 0x64, 0x57,
        0x31, 0x71, 0x78, 0x64, 0x55, 0x56, 0x64, 0x30, 0x62, 0x37, 0x73, 0x6a,
        0x57, 0x46, 0x46, 0x5a, 0x70, 0x70, 0x33, 0x6c, 0x36, 0x48, 0x67, 0x4f,
        0x58, 0x6c, 0x5a, 0x70, 0x61, 0x4f, 0x59, 0x63, 0x68, 0x49, 0x6f, 0x79,
        0x62, 0x73, 0x37, 0x6e, 0x32, 0x64, 0x6b, 0x59, 0x77, 0x6d, 0x76, 0x6e,
        0x39, 0x67, 0x6c, 0x4e, 0x63, 0x4c, 0x43, 0x76, 0x52, 0x71, 0x69, 0x5a,
        0x70, 0x6d, 0x32, 0x6f, 0x45, 0x5a, 0x49, 0x67, 0x31, 0x64, 0x79, 0x38,
        0x52, 0x61, 0x48, 0x35, 0x62, 0x37, 0x41, 0x71, 0x59, 0x32, 0x68, 0x75,
        0x77, 0x6d, 0x76, 0x2b, 0x68, 0x4c, 0x5a, 0x31, 0x58, 0x6b, 0x55, 0x46,
        0x79, 0x53, 0x32, 0x41, 0x65, 0x32, 0x41, 0x4b, 0x75, 0x78, 0x35, 0x72,
        0x73, 0x6c, 0x6d, 0x69, 0x4b, 0x6d, 0x76, 0x67, 0x6c, 0x62, 0x44, 0x59,
        0x30, 0x52, 0x6e, 0x67, 0x31, 0x57, 0x6a, 0x33, 0x72, 0x33, 0x2f, 0x4e,
        0x30, 0x4b, 0x63, 0x4b, 0x6b, 0x65, 0x31, 0x67, 0x31, 0x6f, 0x68, 0x45,
        0x55, 0x61, 0x51, 0x66, 0x58, 0x6e, 0x7a, 0x37, 0x56, 0x56, 0x78, 0x32,
        0x59, 0x6f, 0x56, 0x78, 0x68, 0x54, 0x31, 0x64, 0x54, 0x51, 0x3d, 0x3d,
        0x20, 0x74, 0x65, 0x73, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x73, 0x73, 0x68,
        0x32, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x6e, 0x6f, 0x64, 0x65, 0x2e,
        0x6a, 0x73,
      ])
    },
    what: 'DSA public key (OpenSSH format) with comment'
  },
  { source: 'ssh-dss AAAAB3NzaC1kc3MAAACBAILCaN5QbaErJBa0nNoCY2QvvgcSd7WMgIgQ2qPAPH3Sav+9SWZB0SlHP8QYmV/ntBufaRw9ZdIop6esZBoXmNXDxjcfwpJAzrT3qOMfHXeNcc7w/6dHp+2DlkWD3yzK07SheIZnGNtF1fpfrUvkHFA8UX96awWRLnYuSvWvAK9dAAAAFQCrfeadO0tCCwgnq97Skk0Ng/xV6wAAAIBcVYaMkjXNgfvcUS4pwMabY8rFNGbYo9Jgv7IU2LGFEzD6yFbp2t1sQw031EYlad4n3BMTrZdE6K2/SMi3f3yRr0Z6S6nVSbo7/kIT8/tGJAj9/8RHdCihrFUMaAJNzdk5dmbyaCzexLmLXlKfUVW9A1WOsASjeJBwpFrJ8deuVAAAAIAu/zdW1qxdUVd0b7sjWFFZpp3l6HgOXlZpaOYchIoybs7n2dkYwmvn9glNcLCvRqiZpm2oEZIg1dy8RaH5b7AqY2huwmv+hLZ1XkUFyS2Ae2AKux5rslmiKmvglbDY0Rng1Wj3r3/N0KcKke1g1ohEUaQfXnz7VVx2YoVxhT1dTQ== ssh2test',
    expected: {
      fulltype: 'ssh-dss',
      type: 'dss',
      extra: undefined,
      comment: 'ssh2test',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x07, 0x73, 0x73, 0x68, 0x2d, 0x64, 0x73, 0x73, 0x00,
        0x00, 0x00, 0x81, 0x00, 0x82, 0xc2, 0x68, 0xde, 0x50, 0x6d, 0xa1, 0x2b,
        0x24, 0x16, 0xb4, 0x9c, 0xda, 0x02, 0x63, 0x64, 0x2f, 0xbe, 0x07, 0x12,
        0x77, 0xb5, 0x8c, 0x80, 0x88, 0x10, 0xda, 0xa3, 0xc0, 0x3c, 0x7d, 0xd2,
        0x6a, 0xff, 0xbd, 0x49, 0x66, 0x41, 0xd1, 0x29, 0x47, 0x3f, 0xc4, 0x18,
        0x99, 0x5f, 0xe7, 0xb4, 0x1b, 0x9f, 0x69, 0x1c, 0x3d, 0x65, 0xd2, 0x28,
        0xa7, 0xa7, 0xac, 0x64, 0x1a, 0x17, 0x98, 0xd5, 0xc3, 0xc6, 0x37, 0x1f,
        0xc2, 0x92, 0x40, 0xce, 0xb4, 0xf7, 0xa8, 0xe3, 0x1f, 0x1d, 0x77, 0x8d,
        0x71, 0xce, 0xf0, 0xff, 0xa7, 0x47, 0xa7, 0xed, 0x83, 0x96, 0x45, 0x83,
        0xdf, 0x2c, 0xca, 0xd3, 0xb4, 0xa1, 0x78, 0x86, 0x67, 0x18, 0xdb, 0x45,
        0xd5, 0xfa, 0x5f, 0xad, 0x4b, 0xe4, 0x1c, 0x50, 0x3c, 0x51, 0x7f, 0x7a,
        0x6b, 0x05, 0x91, 0x2e, 0x76, 0x2e, 0x4a, 0xf5, 0xaf, 0x00, 0xaf, 0x5d,
        0x00, 0x00, 0x00, 0x15, 0x00, 0xab, 0x7d, 0xe6, 0x9d, 0x3b, 0x4b, 0x42,
        0x0b, 0x08, 0x27, 0xab, 0xde, 0xd2, 0x92, 0x4d, 0x0d, 0x83, 0xfc, 0x55,
        0xeb, 0x00, 0x00, 0x00, 0x80, 0x5c, 0x55, 0x86, 0x8c, 0x92, 0x35, 0xcd,
        0x81, 0xfb, 0xdc, 0x51, 0x2e, 0x29, 0xc0, 0xc6, 0x9b, 0x63, 0xca, 0xc5,
        0x34, 0x66, 0xd8, 0xa3, 0xd2, 0x60, 0xbf, 0xb2, 0x14, 0xd8, 0xb1, 0x85,
        0x13, 0x30, 0xfa, 0xc8, 0x56, 0xe9, 0xda, 0xdd, 0x6c, 0x43, 0x0d, 0x37,
        0xd4, 0x46, 0x25, 0x69, 0xde, 0x27, 0xdc, 0x13, 0x13, 0xad, 0x97, 0x44,
        0xe8, 0xad, 0xbf, 0x48, 0xc8, 0xb7, 0x7f, 0x7c, 0x91, 0xaf, 0x46, 0x7a,
        0x4b, 0xa9, 0xd5, 0x49, 0xba, 0x3b, 0xfe, 0x42, 0x13, 0xf3, 0xfb, 0x46,
        0x24, 0x08, 0xfd, 0xff, 0xc4, 0x47, 0x74, 0x28, 0xa1, 0xac, 0x55, 0x0c,
        0x68, 0x02, 0x4d, 0xcd, 0xd9, 0x39, 0x76, 0x66, 0xf2, 0x68, 0x2c, 0xde,
        0xc4, 0xb9, 0x8b, 0x5e, 0x52, 0x9f, 0x51, 0x55, 0xbd, 0x03, 0x55, 0x8e,
        0xb0, 0x04, 0xa3, 0x78, 0x90, 0x70, 0xa4, 0x5a, 0xc9, 0xf1, 0xd7, 0xae,
        0x54, 0x00, 0x00, 0x00, 0x80, 0x2e, 0xff, 0x37, 0x56, 0xd6, 0xac, 0x5d,
        0x51, 0x57, 0x74, 0x6f, 0xbb, 0x23, 0x58, 0x51, 0x59, 0xa6, 0x9d, 0xe5,
        0xe8, 0x78, 0x0e, 0x5e, 0x56, 0x69, 0x68, 0xe6, 0x1c, 0x84, 0x8a, 0x32,
        0x6e, 0xce, 0xe7, 0xd9, 0xd9, 0x18, 0xc2, 0x6b, 0xe7, 0xf6, 0x09, 0x4d,
        0x70, 0xb0, 0xaf, 0x46, 0xa8, 0x99, 0xa6, 0x6d, 0xa8, 0x11, 0x92, 0x20,
        0xd5, 0xdc, 0xbc, 0x45, 0xa1, 0xf9, 0x6f, 0xb0, 0x2a, 0x63, 0x68, 0x6e,
        0xc2, 0x6b, 0xfe, 0x84, 0xb6, 0x75, 0x5e, 0x45, 0x05, 0xc9, 0x2d, 0x80,
        0x7b, 0x60, 0x0a, 0xbb, 0x1e, 0x6b, 0xb2, 0x59, 0xa2, 0x2a, 0x6b, 0xe0,
        0x95, 0xb0, 0xd8, 0xd1, 0x19, 0xe0, 0xd5, 0x68, 0xf7, 0xaf, 0x7f, 0xcd,
        0xd0, 0xa7, 0x0a, 0x91, 0xed, 0x60, 0xd6, 0x88, 0x44, 0x51, 0xa4, 0x1f,
        0x5e, 0x7c, 0xfb, 0x55, 0x5c, 0x76, 0x62, 0x85, 0x71, 0x85, 0x3d, 0x5d,
        0x4d,
      ]),
      publicOrig: new Buffer([
        0x73, 0x73, 0x68, 0x2d, 0x64, 0x73, 0x73, 0x20, 0x41, 0x41, 0x41, 0x41,
        0x42, 0x33, 0x4e, 0x7a, 0x61, 0x43, 0x31, 0x6b, 0x63, 0x33, 0x4d, 0x41,
        0x41, 0x41, 0x43, 0x42, 0x41, 0x49, 0x4c, 0x43, 0x61, 0x4e, 0x35, 0x51,
        0x62, 0x61, 0x45, 0x72, 0x4a, 0x42, 0x61, 0x30, 0x6e, 0x4e, 0x6f, 0x43,
        0x59, 0x32, 0x51, 0x76, 0x76, 0x67, 0x63, 0x53, 0x64, 0x37, 0x57, 0x4d,
        0x67, 0x49, 0x67, 0x51, 0x32, 0x71, 0x50, 0x41, 0x50, 0x48, 0x33, 0x53,
        0x61, 0x76, 0x2b, 0x39, 0x53, 0x57, 0x5a, 0x42, 0x30, 0x53, 0x6c, 0x48,
        0x50, 0x38, 0x51, 0x59, 0x6d, 0x56, 0x2f, 0x6e, 0x74, 0x42, 0x75, 0x66,
        0x61, 0x52, 0x77, 0x39, 0x5a, 0x64, 0x49, 0x6f, 0x70, 0x36, 0x65, 0x73,
        0x5a, 0x42, 0x6f, 0x58, 0x6d, 0x4e, 0x58, 0x44, 0x78, 0x6a, 0x63, 0x66,
        0x77, 0x70, 0x4a, 0x41, 0x7a, 0x72, 0x54, 0x33, 0x71, 0x4f, 0x4d, 0x66,
        0x48, 0x58, 0x65, 0x4e, 0x63, 0x63, 0x37, 0x77, 0x2f, 0x36, 0x64, 0x48,
        0x70, 0x2b, 0x32, 0x44, 0x6c, 0x6b, 0x57, 0x44, 0x33, 0x79, 0x7a, 0x4b,
        0x30, 0x37, 0x53, 0x68, 0x65, 0x49, 0x5a, 0x6e, 0x47, 0x4e, 0x74, 0x46,
        0x31, 0x66, 0x70, 0x66, 0x72, 0x55, 0x76, 0x6b, 0x48, 0x46, 0x41, 0x38,
        0x55, 0x58, 0x39, 0x36, 0x61, 0x77, 0x57, 0x52, 0x4c, 0x6e, 0x59, 0x75,
        0x53, 0x76, 0x57, 0x76, 0x41, 0x4b, 0x39, 0x64, 0x41, 0x41, 0x41, 0x41,
        0x46, 0x51, 0x43, 0x72, 0x66, 0x65, 0x61, 0x64, 0x4f, 0x30, 0x74, 0x43,
        0x43, 0x77, 0x67, 0x6e, 0x71, 0x39, 0x37, 0x53, 0x6b, 0x6b, 0x30, 0x4e,
        0x67, 0x2f, 0x78, 0x56, 0x36, 0x77, 0x41, 0x41, 0x41, 0x49, 0x42, 0x63,
        0x56, 0x59, 0x61, 0x4d, 0x6b, 0x6a, 0x58, 0x4e, 0x67, 0x66, 0x76, 0x63,
        0x55, 0x53, 0x34, 0x70, 0x77, 0x4d, 0x61, 0x62, 0x59, 0x38, 0x72, 0x46,
        0x4e, 0x47, 0x62, 0x59, 0x6f, 0x39, 0x4a, 0x67, 0x76, 0x37, 0x49, 0x55,
        0x32, 0x4c, 0x47, 0x46, 0x45, 0x7a, 0x44, 0x36, 0x79, 0x46, 0x62, 0x70,
        0x32, 0x74, 0x31, 0x73, 0x51, 0x77, 0x30, 0x33, 0x31, 0x45, 0x59, 0x6c,
        0x61, 0x64, 0x34, 0x6e, 0x33, 0x42, 0x4d, 0x54, 0x72, 0x5a, 0x64, 0x45,
        0x36, 0x4b, 0x32, 0x2f, 0x53, 0x4d, 0x69, 0x33, 0x66, 0x33, 0x79, 0x52,
        0x72, 0x30, 0x5a, 0x36, 0x53, 0x36, 0x6e, 0x56, 0x53, 0x62, 0x6f, 0x37,
        0x2f, 0x6b, 0x49, 0x54, 0x38, 0x2f, 0x74, 0x47, 0x4a, 0x41, 0x6a, 0x39,
        0x2f, 0x38, 0x52, 0x48, 0x64, 0x43, 0x69, 0x68, 0x72, 0x46, 0x55, 0x4d,
        0x61, 0x41, 0x4a, 0x4e, 0x7a, 0x64, 0x6b, 0x35, 0x64, 0x6d, 0x62, 0x79,
        0x61, 0x43, 0x7a, 0x65, 0x78, 0x4c, 0x6d, 0x4c, 0x58, 0x6c, 0x4b, 0x66,
        0x55, 0x56, 0x57, 0x39, 0x41, 0x31, 0x57, 0x4f, 0x73, 0x41, 0x53, 0x6a,
        0x65, 0x4a, 0x42, 0x77, 0x70, 0x46, 0x72, 0x4a, 0x38, 0x64, 0x65, 0x75,
        0x56, 0x41, 0x41, 0x41, 0x41, 0x49, 0x41, 0x75, 0x2f, 0x7a, 0x64, 0x57,
        0x31, 0x71, 0x78, 0x64, 0x55, 0x56, 0x64, 0x30, 0x62, 0x37, 0x73, 0x6a,
        0x57, 0x46, 0x46, 0x5a, 0x70, 0x70, 0x33, 0x6c, 0x36, 0x48, 0x67, 0x4f,
        0x58, 0x6c, 0x5a, 0x70, 0x61, 0x4f, 0x59, 0x63, 0x68, 0x49, 0x6f, 0x79,
        0x62, 0x73, 0x37, 0x6e, 0x32, 0x64, 0x6b, 0x59, 0x77, 0x6d, 0x76, 0x6e,
        0x39, 0x67, 0x6c, 0x4e, 0x63, 0x4c, 0x43, 0x76, 0x52, 0x71, 0x69, 0x5a,
        0x70, 0x6d, 0x32, 0x6f, 0x45, 0x5a, 0x49, 0x67, 0x31, 0x64, 0x79, 0x38,
        0x52, 0x61, 0x48, 0x35, 0x62, 0x37, 0x41, 0x71, 0x59, 0x32, 0x68, 0x75,
        0x77, 0x6d, 0x76, 0x2b, 0x68, 0x4c, 0x5a, 0x31, 0x58, 0x6b, 0x55, 0x46,
        0x79, 0x53, 0x32, 0x41, 0x65, 0x32, 0x41, 0x4b, 0x75, 0x78, 0x35, 0x72,
        0x73, 0x6c, 0x6d, 0x69, 0x4b, 0x6d, 0x76, 0x67, 0x6c, 0x62, 0x44, 0x59,
        0x30, 0x52, 0x6e, 0x67, 0x31, 0x57, 0x6a, 0x33, 0x72, 0x33, 0x2f, 0x4e,
        0x30, 0x4b, 0x63, 0x4b, 0x6b, 0x65, 0x31, 0x67, 0x31, 0x6f, 0x68, 0x45,
        0x55, 0x61, 0x51, 0x66, 0x58, 0x6e, 0x7a, 0x37, 0x56, 0x56, 0x78, 0x32,
        0x59, 0x6f, 0x56, 0x78, 0x68, 0x54, 0x31, 0x64, 0x54, 0x51, 0x3d, 0x3d,
        0x20, 0x73, 0x73, 0x68, 0x32, 0x74, 0x65, 0x73, 0x74,
      ])
    },
    what: 'DSA public key (OpenSSH format) with comment (no spaces)'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'Comment: "testing ssh2 from node.js"',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: 'testing ssh2 from node.js',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x61, 0x00,
        0xe5, 0xe9, 0xdc, 0x4b, 0xfa, 0x18, 0x31, 0xc0, 0x63, 0x61, 0x66, 0xba,
        0xb5, 0xf2, 0xf0, 0x2b, 0x6c, 0xd0, 0x74, 0x52, 0xa3, 0x71, 0xbd, 0xc9,
        0x5f, 0xfa, 0x87, 0x22, 0x44, 0xb9, 0x67, 0x47, 0x93, 0xba, 0xaf, 0xf7,
        0x33, 0x02, 0x1c, 0x92, 0x3b, 0xe7, 0x8e, 0x73, 0x94, 0x85, 0xef, 0x07,
        0xb1, 0x67, 0x35, 0xa8, 0xfc, 0x23, 0x90, 0xdf, 0xbb, 0x44, 0x8b, 0xda,
        0xde, 0x4e, 0xe7, 0x18, 0x14, 0xd4, 0x10, 0x58, 0x5b, 0xf1, 0xe0, 0x4f,
        0x02, 0x44, 0x53, 0xfe, 0x50, 0xf4, 0x7a, 0x85, 0x05, 0xf4, 0x7f, 0x1a,
        0x0b, 0x48, 0x5d, 0xe5, 0xc5, 0x01, 0x49, 0xfb, 0x7d, 0xbf, 0xad, 0x2d,
      ]),
      publicOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x53,
        0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b,
        0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x43, 0x6f, 0x6d, 0x6d,
        0x65, 0x6e, 0x74, 0x3a, 0x20, 0x22, 0x74, 0x65, 0x73, 0x74, 0x69, 0x6e,
        0x67, 0x20, 0x73, 0x73, 0x68, 0x32, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20,
        0x6e, 0x6f, 0x64, 0x65, 0x2e, 0x6a, 0x73, 0x22, 0x0a, 0x41, 0x41, 0x41,
        0x41, 0x42, 0x33, 0x4e, 0x7a, 0x61, 0x43, 0x31, 0x79, 0x63, 0x32, 0x45,
        0x41, 0x41, 0x41, 0x41, 0x44, 0x41, 0x51, 0x41, 0x42, 0x41, 0x41, 0x41,
        0x41, 0x59, 0x51, 0x44, 0x6c, 0x36, 0x64, 0x78, 0x4c, 0x2b, 0x68, 0x67,
        0x78, 0x77, 0x47, 0x4e, 0x68, 0x5a, 0x72, 0x71, 0x31, 0x38, 0x76, 0x41,
        0x72, 0x62, 0x4e, 0x42, 0x30, 0x55, 0x71, 0x4e, 0x78, 0x76, 0x63, 0x6c,
        0x66, 0x2b, 0x6f, 0x63, 0x69, 0x52, 0x4c, 0x0a, 0x6c, 0x6e, 0x52, 0x35,
        0x4f, 0x36, 0x72, 0x2f, 0x63, 0x7a, 0x41, 0x68, 0x79, 0x53, 0x4f, 0x2b,
        0x65, 0x4f, 0x63, 0x35, 0x53, 0x46, 0x37, 0x77, 0x65, 0x78, 0x5a, 0x7a,
        0x57, 0x6f, 0x2f, 0x43, 0x4f, 0x51, 0x33, 0x37, 0x74, 0x45, 0x69, 0x39,
        0x72, 0x65, 0x54, 0x75, 0x63, 0x59, 0x46, 0x4e, 0x51, 0x51, 0x57, 0x46,
        0x76, 0x78, 0x34, 0x45, 0x38, 0x43, 0x52, 0x46, 0x50, 0x2b, 0x55, 0x50,
        0x52, 0x36, 0x68, 0x51, 0x58, 0x30, 0x0a, 0x66, 0x78, 0x6f, 0x4c, 0x53,
        0x46, 0x33, 0x6c, 0x78, 0x51, 0x46, 0x4a, 0x2b, 0x33, 0x32, 0x2f, 0x72,
        0x53, 0x30, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x45, 0x4e, 0x44,
        0x20, 0x53, 0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43,
        0x20, 0x4b, 0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d,
      ])
    },
    what: 'RSA public key (RFC4716 format) with quoted comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'Comment: testing ssh2 from node.js',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: 'testing ssh2 from node.js',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x61, 0x00,
        0xe5, 0xe9, 0xdc, 0x4b, 0xfa, 0x18, 0x31, 0xc0, 0x63, 0x61, 0x66, 0xba,
        0xb5, 0xf2, 0xf0, 0x2b, 0x6c, 0xd0, 0x74, 0x52, 0xa3, 0x71, 0xbd, 0xc9,
        0x5f, 0xfa, 0x87, 0x22, 0x44, 0xb9, 0x67, 0x47, 0x93, 0xba, 0xaf, 0xf7,
        0x33, 0x02, 0x1c, 0x92, 0x3b, 0xe7, 0x8e, 0x73, 0x94, 0x85, 0xef, 0x07,
        0xb1, 0x67, 0x35, 0xa8, 0xfc, 0x23, 0x90, 0xdf, 0xbb, 0x44, 0x8b, 0xda,
        0xde, 0x4e, 0xe7, 0x18, 0x14, 0xd4, 0x10, 0x58, 0x5b, 0xf1, 0xe0, 0x4f,
        0x02, 0x44, 0x53, 0xfe, 0x50, 0xf4, 0x7a, 0x85, 0x05, 0xf4, 0x7f, 0x1a,
        0x0b, 0x48, 0x5d, 0xe5, 0xc5, 0x01, 0x49, 0xfb, 0x7d, 0xbf, 0xad, 0x2d,
      ]),
      publicOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x53,
        0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b,
        0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x43, 0x6f, 0x6d, 0x6d,
        0x65, 0x6e, 0x74, 0x3a, 0x20, 0x74, 0x65, 0x73, 0x74, 0x69, 0x6e, 0x67,
        0x20, 0x73, 0x73, 0x68, 0x32, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x6e,
        0x6f, 0x64, 0x65, 0x2e, 0x6a, 0x73, 0x0a, 0x41, 0x41, 0x41, 0x41, 0x42,
        0x33, 0x4e, 0x7a, 0x61, 0x43, 0x31, 0x79, 0x63, 0x32, 0x45, 0x41, 0x41,
        0x41, 0x41, 0x44, 0x41, 0x51, 0x41, 0x42, 0x41, 0x41, 0x41, 0x41, 0x59,
        0x51, 0x44, 0x6c, 0x36, 0x64, 0x78, 0x4c, 0x2b, 0x68, 0x67, 0x78, 0x77,
        0x47, 0x4e, 0x68, 0x5a, 0x72, 0x71, 0x31, 0x38, 0x76, 0x41, 0x72, 0x62,
        0x4e, 0x42, 0x30, 0x55, 0x71, 0x4e, 0x78, 0x76, 0x63, 0x6c, 0x66, 0x2b,
        0x6f, 0x63, 0x69, 0x52, 0x4c, 0x0a, 0x6c, 0x6e, 0x52, 0x35, 0x4f, 0x36,
        0x72, 0x2f, 0x63, 0x7a, 0x41, 0x68, 0x79, 0x53, 0x4f, 0x2b, 0x65, 0x4f,
        0x63, 0x35, 0x53, 0x46, 0x37, 0x77, 0x65, 0x78, 0x5a, 0x7a, 0x57, 0x6f,
        0x2f, 0x43, 0x4f, 0x51, 0x33, 0x37, 0x74, 0x45, 0x69, 0x39, 0x72, 0x65,
        0x54, 0x75, 0x63, 0x59, 0x46, 0x4e, 0x51, 0x51, 0x57, 0x46, 0x76, 0x78,
        0x34, 0x45, 0x38, 0x43, 0x52, 0x46, 0x50, 0x2b, 0x55, 0x50, 0x52, 0x36,
        0x68, 0x51, 0x58, 0x30, 0x0a, 0x66, 0x78, 0x6f, 0x4c, 0x53, 0x46, 0x33,
        0x6c, 0x78, 0x51, 0x46, 0x4a, 0x2b, 0x33, 0x32, 0x2f, 0x72, 0x53, 0x30,
        0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x45, 0x4e, 0x44, 0x20, 0x53,
        0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b,
        0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d,
      ])
    },
    what: 'RSA public key (RFC4716 format) with unquoted comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'Comment: "testing ssh2 \\',
      'from node.js"',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: 'testing ssh2 from node.js',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x61, 0x00,
        0xe5, 0xe9, 0xdc, 0x4b, 0xfa, 0x18, 0x31, 0xc0, 0x63, 0x61, 0x66, 0xba,
        0xb5, 0xf2, 0xf0, 0x2b, 0x6c, 0xd0, 0x74, 0x52, 0xa3, 0x71, 0xbd, 0xc9,
        0x5f, 0xfa, 0x87, 0x22, 0x44, 0xb9, 0x67, 0x47, 0x93, 0xba, 0xaf, 0xf7,
        0x33, 0x02, 0x1c, 0x92, 0x3b, 0xe7, 0x8e, 0x73, 0x94, 0x85, 0xef, 0x07,
        0xb1, 0x67, 0x35, 0xa8, 0xfc, 0x23, 0x90, 0xdf, 0xbb, 0x44, 0x8b, 0xda,
        0xde, 0x4e, 0xe7, 0x18, 0x14, 0xd4, 0x10, 0x58, 0x5b, 0xf1, 0xe0, 0x4f,
        0x02, 0x44, 0x53, 0xfe, 0x50, 0xf4, 0x7a, 0x85, 0x05, 0xf4, 0x7f, 0x1a,
        0x0b, 0x48, 0x5d, 0xe5, 0xc5, 0x01, 0x49, 0xfb, 0x7d, 0xbf, 0xad, 0x2d,
      ]),
      publicOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x53,
        0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b,
        0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x43, 0x6f, 0x6d, 0x6d,
        0x65, 0x6e, 0x74, 0x3a, 0x20, 0x22, 0x74, 0x65, 0x73, 0x74, 0x69, 0x6e,
        0x67, 0x20, 0x73, 0x73, 0x68, 0x32, 0x20, 0x5c, 0x0a, 0x66, 0x72, 0x6f,
        0x6d, 0x20, 0x6e, 0x6f, 0x64, 0x65, 0x2e, 0x6a, 0x73, 0x22, 0x0a, 0x41,
        0x41, 0x41, 0x41, 0x42, 0x33, 0x4e, 0x7a, 0x61, 0x43, 0x31, 0x79, 0x63,
        0x32, 0x45, 0x41, 0x41, 0x41, 0x41, 0x44, 0x41, 0x51, 0x41, 0x42, 0x41,
        0x41, 0x41, 0x41, 0x59, 0x51, 0x44, 0x6c, 0x36, 0x64, 0x78, 0x4c, 0x2b,
        0x68, 0x67, 0x78, 0x77, 0x47, 0x4e, 0x68, 0x5a, 0x72, 0x71, 0x31, 0x38,
        0x76, 0x41, 0x72, 0x62, 0x4e, 0x42, 0x30, 0x55, 0x71, 0x4e, 0x78, 0x76,
        0x63, 0x6c, 0x66, 0x2b, 0x6f, 0x63, 0x69, 0x52, 0x4c, 0x0a, 0x6c, 0x6e,
        0x52, 0x35, 0x4f, 0x36, 0x72, 0x2f, 0x63, 0x7a, 0x41, 0x68, 0x79, 0x53,
        0x4f, 0x2b, 0x65, 0x4f, 0x63, 0x35, 0x53, 0x46, 0x37, 0x77, 0x65, 0x78,
        0x5a, 0x7a, 0x57, 0x6f, 0x2f, 0x43, 0x4f, 0x51, 0x33, 0x37, 0x74, 0x45,
        0x69, 0x39, 0x72, 0x65, 0x54, 0x75, 0x63, 0x59, 0x46, 0x4e, 0x51, 0x51,
        0x57, 0x46, 0x76, 0x78, 0x34, 0x45, 0x38, 0x43, 0x52, 0x46, 0x50, 0x2b,
        0x55, 0x50, 0x52, 0x36, 0x68, 0x51, 0x58, 0x30, 0x0a, 0x66, 0x78, 0x6f,
        0x4c, 0x53, 0x46, 0x33, 0x6c, 0x78, 0x51, 0x46, 0x4a, 0x2b, 0x33, 0x32,
        0x2f, 0x72, 0x53, 0x30, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x45,
        0x4e, 0x44, 0x20, 0x53, 0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c,
        0x49, 0x43, 0x20, 0x4b, 0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d,
      ])
    },
    what: 'RSA public key (RFC4716 format) with multi-line quoted comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'Comment: testing ssh2 \\',
      'from node.js',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: 'testing ssh2 from node.js',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x61, 0x00,
        0xe5, 0xe9, 0xdc, 0x4b, 0xfa, 0x18, 0x31, 0xc0, 0x63, 0x61, 0x66, 0xba,
        0xb5, 0xf2, 0xf0, 0x2b, 0x6c, 0xd0, 0x74, 0x52, 0xa3, 0x71, 0xbd, 0xc9,
        0x5f, 0xfa, 0x87, 0x22, 0x44, 0xb9, 0x67, 0x47, 0x93, 0xba, 0xaf, 0xf7,
        0x33, 0x02, 0x1c, 0x92, 0x3b, 0xe7, 0x8e, 0x73, 0x94, 0x85, 0xef, 0x07,
        0xb1, 0x67, 0x35, 0xa8, 0xfc, 0x23, 0x90, 0xdf, 0xbb, 0x44, 0x8b, 0xda,
        0xde, 0x4e, 0xe7, 0x18, 0x14, 0xd4, 0x10, 0x58, 0x5b, 0xf1, 0xe0, 0x4f,
        0x02, 0x44, 0x53, 0xfe, 0x50, 0xf4, 0x7a, 0x85, 0x05, 0xf4, 0x7f, 0x1a,
        0x0b, 0x48, 0x5d, 0xe5, 0xc5, 0x01, 0x49, 0xfb, 0x7d, 0xbf, 0xad, 0x2d,
      ]),
      publicOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x53,
        0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b,
        0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x43, 0x6f, 0x6d, 0x6d,
        0x65, 0x6e, 0x74, 0x3a, 0x20, 0x74, 0x65, 0x73, 0x74, 0x69, 0x6e, 0x67,
        0x20, 0x73, 0x73, 0x68, 0x32, 0x20, 0x5c, 0x0a, 0x66, 0x72, 0x6f, 0x6d,
        0x20, 0x6e, 0x6f, 0x64, 0x65, 0x2e, 0x6a, 0x73, 0x0a, 0x41, 0x41, 0x41,
        0x41, 0x42, 0x33, 0x4e, 0x7a, 0x61, 0x43, 0x31, 0x79, 0x63, 0x32, 0x45,
        0x41, 0x41, 0x41, 0x41, 0x44, 0x41, 0x51, 0x41, 0x42, 0x41, 0x41, 0x41,
        0x41, 0x59, 0x51, 0x44, 0x6c, 0x36, 0x64, 0x78, 0x4c, 0x2b, 0x68, 0x67,
        0x78, 0x77, 0x47, 0x4e, 0x68, 0x5a, 0x72, 0x71, 0x31, 0x38, 0x76, 0x41,
        0x72, 0x62, 0x4e, 0x42, 0x30, 0x55, 0x71, 0x4e, 0x78, 0x76, 0x63, 0x6c,
        0x66, 0x2b, 0x6f, 0x63, 0x69, 0x52, 0x4c, 0x0a, 0x6c, 0x6e, 0x52, 0x35,
        0x4f, 0x36, 0x72, 0x2f, 0x63, 0x7a, 0x41, 0x68, 0x79, 0x53, 0x4f, 0x2b,
        0x65, 0x4f, 0x63, 0x35, 0x53, 0x46, 0x37, 0x77, 0x65, 0x78, 0x5a, 0x7a,
        0x57, 0x6f, 0x2f, 0x43, 0x4f, 0x51, 0x33, 0x37, 0x74, 0x45, 0x69, 0x39,
        0x72, 0x65, 0x54, 0x75, 0x63, 0x59, 0x46, 0x4e, 0x51, 0x51, 0x57, 0x46,
        0x76, 0x78, 0x34, 0x45, 0x38, 0x43, 0x52, 0x46, 0x50, 0x2b, 0x55, 0x50,
        0x52, 0x36, 0x68, 0x51, 0x58, 0x30, 0x0a, 0x66, 0x78, 0x6f, 0x4c, 0x53,
        0x46, 0x33, 0x6c, 0x78, 0x51, 0x46, 0x4a, 0x2b, 0x33, 0x32, 0x2f, 0x72,
        0x53, 0x30, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x45, 0x4e, 0x44,
        0x20, 0x53, 0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43,
        0x20, 0x4b, 0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d,
      ])
    },
    what: 'RSA public key (RFC4716 format) with multi-line unquoted comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'Comment: ""',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: '',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x61, 0x00,
        0xe5, 0xe9, 0xdc, 0x4b, 0xfa, 0x18, 0x31, 0xc0, 0x63, 0x61, 0x66, 0xba,
        0xb5, 0xf2, 0xf0, 0x2b, 0x6c, 0xd0, 0x74, 0x52, 0xa3, 0x71, 0xbd, 0xc9,
        0x5f, 0xfa, 0x87, 0x22, 0x44, 0xb9, 0x67, 0x47, 0x93, 0xba, 0xaf, 0xf7,
        0x33, 0x02, 0x1c, 0x92, 0x3b, 0xe7, 0x8e, 0x73, 0x94, 0x85, 0xef, 0x07,
        0xb1, 0x67, 0x35, 0xa8, 0xfc, 0x23, 0x90, 0xdf, 0xbb, 0x44, 0x8b, 0xda,
        0xde, 0x4e, 0xe7, 0x18, 0x14, 0xd4, 0x10, 0x58, 0x5b, 0xf1, 0xe0, 0x4f,
        0x02, 0x44, 0x53, 0xfe, 0x50, 0xf4, 0x7a, 0x85, 0x05, 0xf4, 0x7f, 0x1a,
        0x0b, 0x48, 0x5d, 0xe5, 0xc5, 0x01, 0x49, 0xfb, 0x7d, 0xbf, 0xad, 0x2d,
      ]),
      publicOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x53,
        0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b,
        0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x43, 0x6f, 0x6d, 0x6d,
        0x65, 0x6e, 0x74, 0x3a, 0x20, 0x22, 0x22, 0x0a, 0x41, 0x41, 0x41, 0x41,
        0x42, 0x33, 0x4e, 0x7a, 0x61, 0x43, 0x31, 0x79, 0x63, 0x32, 0x45, 0x41,
        0x41, 0x41, 0x41, 0x44, 0x41, 0x51, 0x41, 0x42, 0x41, 0x41, 0x41, 0x41,
        0x59, 0x51, 0x44, 0x6c, 0x36, 0x64, 0x78, 0x4c, 0x2b, 0x68, 0x67, 0x78,
        0x77, 0x47, 0x4e, 0x68, 0x5a, 0x72, 0x71, 0x31, 0x38, 0x76, 0x41, 0x72,
        0x62, 0x4e, 0x42, 0x30, 0x55, 0x71, 0x4e, 0x78, 0x76, 0x63, 0x6c, 0x66,
        0x2b, 0x6f, 0x63, 0x69, 0x52, 0x4c, 0x0a, 0x6c, 0x6e, 0x52, 0x35, 0x4f,
        0x36, 0x72, 0x2f, 0x63, 0x7a, 0x41, 0x68, 0x79, 0x53, 0x4f, 0x2b, 0x65,
        0x4f, 0x63, 0x35, 0x53, 0x46, 0x37, 0x77, 0x65, 0x78, 0x5a, 0x7a, 0x57,
        0x6f, 0x2f, 0x43, 0x4f, 0x51, 0x33, 0x37, 0x74, 0x45, 0x69, 0x39, 0x72,
        0x65, 0x54, 0x75, 0x63, 0x59, 0x46, 0x4e, 0x51, 0x51, 0x57, 0x46, 0x76,
        0x78, 0x34, 0x45, 0x38, 0x43, 0x52, 0x46, 0x50, 0x2b, 0x55, 0x50, 0x52,
        0x36, 0x68, 0x51, 0x58, 0x30, 0x0a, 0x66, 0x78, 0x6f, 0x4c, 0x53, 0x46,
        0x33, 0x6c, 0x78, 0x51, 0x46, 0x4a, 0x2b, 0x33, 0x32, 0x2f, 0x72, 0x53,
        0x30, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x45, 0x4e, 0x44, 0x20,
        0x53, 0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20,
        0x4b, 0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d,
      ])
    },
    what: 'RSA public key (RFC4716 format) with empty quoted comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'Comment: ',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: '',
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x61, 0x00,
        0xe5, 0xe9, 0xdc, 0x4b, 0xfa, 0x18, 0x31, 0xc0, 0x63, 0x61, 0x66, 0xba,
        0xb5, 0xf2, 0xf0, 0x2b, 0x6c, 0xd0, 0x74, 0x52, 0xa3, 0x71, 0xbd, 0xc9,
        0x5f, 0xfa, 0x87, 0x22, 0x44, 0xb9, 0x67, 0x47, 0x93, 0xba, 0xaf, 0xf7,
        0x33, 0x02, 0x1c, 0x92, 0x3b, 0xe7, 0x8e, 0x73, 0x94, 0x85, 0xef, 0x07,
        0xb1, 0x67, 0x35, 0xa8, 0xfc, 0x23, 0x90, 0xdf, 0xbb, 0x44, 0x8b, 0xda,
        0xde, 0x4e, 0xe7, 0x18, 0x14, 0xd4, 0x10, 0x58, 0x5b, 0xf1, 0xe0, 0x4f,
        0x02, 0x44, 0x53, 0xfe, 0x50, 0xf4, 0x7a, 0x85, 0x05, 0xf4, 0x7f, 0x1a,
        0x0b, 0x48, 0x5d, 0xe5, 0xc5, 0x01, 0x49, 0xfb, 0x7d, 0xbf, 0xad, 0x2d,
      ]),
      publicOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x53,
        0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b,
        0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x43, 0x6f, 0x6d, 0x6d,
        0x65, 0x6e, 0x74, 0x3a, 0x20, 0x0a, 0x41, 0x41, 0x41, 0x41, 0x42, 0x33,
        0x4e, 0x7a, 0x61, 0x43, 0x31, 0x79, 0x63, 0x32, 0x45, 0x41, 0x41, 0x41,
        0x41, 0x44, 0x41, 0x51, 0x41, 0x42, 0x41, 0x41, 0x41, 0x41, 0x59, 0x51,
        0x44, 0x6c, 0x36, 0x64, 0x78, 0x4c, 0x2b, 0x68, 0x67, 0x78, 0x77, 0x47,
        0x4e, 0x68, 0x5a, 0x72, 0x71, 0x31, 0x38, 0x76, 0x41, 0x72, 0x62, 0x4e,
        0x42, 0x30, 0x55, 0x71, 0x4e, 0x78, 0x76, 0x63, 0x6c, 0x66, 0x2b, 0x6f,
        0x63, 0x69, 0x52, 0x4c, 0x0a, 0x6c, 0x6e, 0x52, 0x35, 0x4f, 0x36, 0x72,
        0x2f, 0x63, 0x7a, 0x41, 0x68, 0x79, 0x53, 0x4f, 0x2b, 0x65, 0x4f, 0x63,
        0x35, 0x53, 0x46, 0x37, 0x77, 0x65, 0x78, 0x5a, 0x7a, 0x57, 0x6f, 0x2f,
        0x43, 0x4f, 0x51, 0x33, 0x37, 0x74, 0x45, 0x69, 0x39, 0x72, 0x65, 0x54,
        0x75, 0x63, 0x59, 0x46, 0x4e, 0x51, 0x51, 0x57, 0x46, 0x76, 0x78, 0x34,
        0x45, 0x38, 0x43, 0x52, 0x46, 0x50, 0x2b, 0x55, 0x50, 0x52, 0x36, 0x68,
        0x51, 0x58, 0x30, 0x0a, 0x66, 0x78, 0x6f, 0x4c, 0x53, 0x46, 0x33, 0x6c,
        0x78, 0x51, 0x46, 0x4a, 0x2b, 0x33, 0x32, 0x2f, 0x72, 0x53, 0x30, 0x3d,
        0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x45, 0x4e, 0x44, 0x20, 0x53, 0x53,
        0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b, 0x45,
        0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d,
      ])
    },
    what: 'RSA public key (RFC4716 format) with empty unquoted comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x61, 0x00,
        0xe5, 0xe9, 0xdc, 0x4b, 0xfa, 0x18, 0x31, 0xc0, 0x63, 0x61, 0x66, 0xba,
        0xb5, 0xf2, 0xf0, 0x2b, 0x6c, 0xd0, 0x74, 0x52, 0xa3, 0x71, 0xbd, 0xc9,
        0x5f, 0xfa, 0x87, 0x22, 0x44, 0xb9, 0x67, 0x47, 0x93, 0xba, 0xaf, 0xf7,
        0x33, 0x02, 0x1c, 0x92, 0x3b, 0xe7, 0x8e, 0x73, 0x94, 0x85, 0xef, 0x07,
        0xb1, 0x67, 0x35, 0xa8, 0xfc, 0x23, 0x90, 0xdf, 0xbb, 0x44, 0x8b, 0xda,
        0xde, 0x4e, 0xe7, 0x18, 0x14, 0xd4, 0x10, 0x58, 0x5b, 0xf1, 0xe0, 0x4f,
        0x02, 0x44, 0x53, 0xfe, 0x50, 0xf4, 0x7a, 0x85, 0x05, 0xf4, 0x7f, 0x1a,
        0x0b, 0x48, 0x5d, 0xe5, 0xc5, 0x01, 0x49, 0xfb, 0x7d, 0xbf, 0xad, 0x2d,
      ]),
      publicOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x53,
        0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b,
        0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x41, 0x41, 0x41, 0x41,
        0x42, 0x33, 0x4e, 0x7a, 0x61, 0x43, 0x31, 0x79, 0x63, 0x32, 0x45, 0x41,
        0x41, 0x41, 0x41, 0x44, 0x41, 0x51, 0x41, 0x42, 0x41, 0x41, 0x41, 0x41,
        0x59, 0x51, 0x44, 0x6c, 0x36, 0x64, 0x78, 0x4c, 0x2b, 0x68, 0x67, 0x78,
        0x77, 0x47, 0x4e, 0x68, 0x5a, 0x72, 0x71, 0x31, 0x38, 0x76, 0x41, 0x72,
        0x62, 0x4e, 0x42, 0x30, 0x55, 0x71, 0x4e, 0x78, 0x76, 0x63, 0x6c, 0x66,
        0x2b, 0x6f, 0x63, 0x69, 0x52, 0x4c, 0x0a, 0x6c, 0x6e, 0x52, 0x35, 0x4f,
        0x36, 0x72, 0x2f, 0x63, 0x7a, 0x41, 0x68, 0x79, 0x53, 0x4f, 0x2b, 0x65,
        0x4f, 0x63, 0x35, 0x53, 0x46, 0x37, 0x77, 0x65, 0x78, 0x5a, 0x7a, 0x57,
        0x6f, 0x2f, 0x43, 0x4f, 0x51, 0x33, 0x37, 0x74, 0x45, 0x69, 0x39, 0x72,
        0x65, 0x54, 0x75, 0x63, 0x59, 0x46, 0x4e, 0x51, 0x51, 0x57, 0x46, 0x76,
        0x78, 0x34, 0x45, 0x38, 0x43, 0x52, 0x46, 0x50, 0x2b, 0x55, 0x50, 0x52,
        0x36, 0x68, 0x51, 0x58, 0x30, 0x0a, 0x66, 0x78, 0x6f, 0x4c, 0x53, 0x46,
        0x33, 0x6c, 0x78, 0x51, 0x46, 0x4a, 0x2b, 0x33, 0x32, 0x2f, 0x72, 0x53,
        0x30, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x45, 0x4e, 0x44, 0x20,
        0x53, 0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20,
        0x4b, 0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d,
      ])
    },
    what: 'RSA public key (RFC4716 format) with no comment'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      '',
      '',
      'AAAAB3NzaC1yc2EAAAADAQABAAAAYQDl6dxL+hgxwGNhZrq18vArbNB0UqNxvclf+ociRL',
      'lnR5O6r/czAhySO+eOc5SF7wexZzWo/COQ37tEi9reTucYFNQQWFvx4E8CRFP+UPR6hQX0',
      'fxoLSF3lxQFJ+32/rS0=',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x61, 0x00,
        0xe5, 0xe9, 0xdc, 0x4b, 0xfa, 0x18, 0x31, 0xc0, 0x63, 0x61, 0x66, 0xba,
        0xb5, 0xf2, 0xf0, 0x2b, 0x6c, 0xd0, 0x74, 0x52, 0xa3, 0x71, 0xbd, 0xc9,
        0x5f, 0xfa, 0x87, 0x22, 0x44, 0xb9, 0x67, 0x47, 0x93, 0xba, 0xaf, 0xf7,
        0x33, 0x02, 0x1c, 0x92, 0x3b, 0xe7, 0x8e, 0x73, 0x94, 0x85, 0xef, 0x07,
        0xb1, 0x67, 0x35, 0xa8, 0xfc, 0x23, 0x90, 0xdf, 0xbb, 0x44, 0x8b, 0xda,
        0xde, 0x4e, 0xe7, 0x18, 0x14, 0xd4, 0x10, 0x58, 0x5b, 0xf1, 0xe0, 0x4f,
        0x02, 0x44, 0x53, 0xfe, 0x50, 0xf4, 0x7a, 0x85, 0x05, 0xf4, 0x7f, 0x1a,
        0x0b, 0x48, 0x5d, 0xe5, 0xc5, 0x01, 0x49, 0xfb, 0x7d, 0xbf, 0xad, 0x2d,
      ]),
      publicOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x53,
        0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b,
        0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x0a, 0x0a, 0x41, 0x41,
        0x41, 0x41, 0x42, 0x33, 0x4e, 0x7a, 0x61, 0x43, 0x31, 0x79, 0x63, 0x32,
        0x45, 0x41, 0x41, 0x41, 0x41, 0x44, 0x41, 0x51, 0x41, 0x42, 0x41, 0x41,
        0x41, 0x41, 0x59, 0x51, 0x44, 0x6c, 0x36, 0x64, 0x78, 0x4c, 0x2b, 0x68,
        0x67, 0x78, 0x77, 0x47, 0x4e, 0x68, 0x5a, 0x72, 0x71, 0x31, 0x38, 0x76,
        0x41, 0x72, 0x62, 0x4e, 0x42, 0x30, 0x55, 0x71, 0x4e, 0x78, 0x76, 0x63,
        0x6c, 0x66, 0x2b, 0x6f, 0x63, 0x69, 0x52, 0x4c, 0x0a, 0x6c, 0x6e, 0x52,
        0x35, 0x4f, 0x36, 0x72, 0x2f, 0x63, 0x7a, 0x41, 0x68, 0x79, 0x53, 0x4f,
        0x2b, 0x65, 0x4f, 0x63, 0x35, 0x53, 0x46, 0x37, 0x77, 0x65, 0x78, 0x5a,
        0x7a, 0x57, 0x6f, 0x2f, 0x43, 0x4f, 0x51, 0x33, 0x37, 0x74, 0x45, 0x69,
        0x39, 0x72, 0x65, 0x54, 0x75, 0x63, 0x59, 0x46, 0x4e, 0x51, 0x51, 0x57,
        0x46, 0x76, 0x78, 0x34, 0x45, 0x38, 0x43, 0x52, 0x46, 0x50, 0x2b, 0x55,
        0x50, 0x52, 0x36, 0x68, 0x51, 0x58, 0x30, 0x0a, 0x66, 0x78, 0x6f, 0x4c,
        0x53, 0x46, 0x33, 0x6c, 0x78, 0x51, 0x46, 0x4a, 0x2b, 0x33, 0x32, 0x2f,
        0x72, 0x53, 0x30, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x45, 0x4e,
        0x44, 0x20, 0x53, 0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49,
        0x43, 0x20, 0x4b, 0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d,
      ])
    },
    what: 'RSA public key (RFC4716 format) with blank lines'
  },
  { source: [
      '---- BEGIN SSH2 PUBLIC KEY ----',
      'AAAAB3NzaC1kc3MAAACBAI4CR+tgz/kZnbmKJjccZNWKqcifcxLy/pTMnRLh5SrzQ44R8T',
      'QSbQTNnJHVpc6ucW369+TtQRpRmr4iU65ttZkZqCJ1jawXAnxcpIrVq/xBfsXfCIr26LaG',
      'qNm//vYBAPs5j648Fhg6AHyM/LewPSnek3fE8gIB9gtCsPei5L7lAAAAFQDJ3k6TdJK/eo',
      'LgUqg2bUB94Mjg9wAAAIAp9Q3SyooklLGMPdG2kj0vMWF+cVtChcjGjemhPeHXyjD5/in8',
      'gWYFifVqoYKBJhAaPP5HpPtA3BEGgIkLVGRzCk3tnwXbutGbwt7PzutbJ9LKiiR7z1HLqn',
      'E+r0//6Nhqz/ZNQPWZRbcG0l1EIQNgjuZkSzJLLO15TSPZeVhZcQAAAIAUAYobyKQ3Rl4q',
      'XAcOP8uOt5q7YHMy43JmZJnTCQ7X95TuNv5kFj7a5RkDU9HjgzSg5aabVsJAD6wDy5/rm+',
      'kJA9uXGOvHfiToh+tRjDp2SrWDERfPCRDl8oTJr36p1zqFXERwTJsNJF32QClszhzk7NYv',
      'G6kcaQC8Hhm9yccRoA==',
      '---- END SSH2 PUBLIC KEY ----'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-dss',
      type: 'dss',
      extra: undefined,
      comment: undefined,
      encryption: undefined,
      private: undefined,
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x81, 0x00, 0x8e, 0x02, 0x47, 0xeb, 0x60, 0xcf, 0xf9,
        0x19, 0x9d, 0xb9, 0x8a, 0x26, 0x37, 0x1c, 0x64, 0xd5, 0x8a, 0xa9, 0xc8,
        0x9f, 0x73, 0x12, 0xf2, 0xfe, 0x94, 0xcc, 0x9d, 0x12, 0xe1, 0xe5, 0x2a,
        0xf3, 0x43, 0x8e, 0x11, 0xf1, 0x34, 0x12, 0x6d, 0x04, 0xcd, 0x9c, 0x91,
        0xd5, 0xa5, 0xce, 0xae, 0x71, 0x6d, 0xfa, 0xf7, 0xe4, 0xed, 0x41, 0x1a,
        0x51, 0x9a, 0xbe, 0x22, 0x53, 0xae, 0x6d, 0xb5, 0x99, 0x19, 0xa8, 0x22,
        0x75, 0x8d, 0xac, 0x17, 0x02, 0x7c, 0x5c, 0xa4, 0x8a, 0xd5, 0xab, 0xfc,
        0x41, 0x7e, 0xc5, 0xdf, 0x08, 0x8a, 0xf6, 0xe8, 0xb6, 0x86, 0xa8, 0xd9,
        0xbf, 0xfe, 0xf6, 0x01, 0x00, 0xfb, 0x39, 0x8f, 0xae, 0x3c, 0x16, 0x18,
        0x3a, 0x00, 0x7c, 0x8c, 0xfc, 0xb7, 0xb0, 0x3d, 0x29, 0xde, 0x93, 0x77,
        0xc4, 0xf2, 0x02, 0x01, 0xf6, 0x0b, 0x42, 0xb0, 0xf7, 0xa2, 0xe4, 0xbe,
        0xe5, 0x00, 0x00, 0x00, 0x15, 0x00, 0xc9, 0xde, 0x4e, 0x93, 0x74, 0x92,
        0xbf, 0x7a, 0x82, 0xe0, 0x52, 0xa8, 0x36, 0x6d, 0x40, 0x7d, 0xe0, 0xc8,
        0xe0, 0xf7, 0x00, 0x00, 0x00, 0x80, 0x29, 0xf5, 0x0d, 0xd2, 0xca, 0x8a,
        0x24, 0x94, 0xb1, 0x8c, 0x3d, 0xd1, 0xb6, 0x92, 0x3d, 0x2f, 0x31, 0x61,
        0x7e, 0x71, 0x5b, 0x42, 0x85, 0xc8, 0xc6, 0x8d, 0xe9, 0xa1, 0x3d, 0xe1,
        0xd7, 0xca, 0x30, 0xf9, 0xfe, 0x29, 0xfc, 0x81, 0x66, 0x05, 0x89, 0xf5,
        0x6a, 0xa1, 0x82, 0x81, 0x26, 0x10, 0x1a, 0x3c, 0xfe, 0x47, 0xa4, 0xfb,
        0x40, 0xdc, 0x11, 0x06, 0x80, 0x89, 0x0b, 0x54, 0x64, 0x73, 0x0a, 0x4d,
        0xed, 0x9f, 0x05, 0xdb, 0xba, 0xd1, 0x9b, 0xc2, 0xde, 0xcf, 0xce, 0xeb,
        0x5b, 0x27, 0xd2, 0xca, 0x8a, 0x24, 0x7b, 0xcf, 0x51, 0xcb, 0xaa, 0x71,
        0x3e, 0xaf, 0x4f, 0xff, 0xe8, 0xd8, 0x6a, 0xcf, 0xf6, 0x4d, 0x40, 0xf5,
        0x99, 0x45, 0xb7, 0x06, 0xd2, 0x5d, 0x44, 0x21, 0x03, 0x60, 0x8e, 0xe6,
        0x64, 0x4b, 0x32, 0x4b, 0x2c, 0xed, 0x79, 0x4d, 0x23, 0xd9, 0x79, 0x58,
        0x59, 0x71, 0x00, 0x00, 0x00, 0x80, 0x14, 0x01, 0x8a, 0x1b, 0xc8, 0xa4,
        0x37, 0x46, 0x5e, 0x2a, 0x5c, 0x07, 0x0e, 0x3f, 0xcb, 0x8e, 0xb7, 0x9a,
        0xbb, 0x60, 0x73, 0x32, 0xe3, 0x72, 0x66, 0x64, 0x99, 0xd3, 0x09, 0x0e,
        0xd7, 0xf7, 0x94, 0xee, 0x36, 0xfe, 0x64, 0x16, 0x3e, 0xda, 0xe5, 0x19,
        0x03, 0x53, 0xd1, 0xe3, 0x83, 0x34, 0xa0, 0xe5, 0xa6, 0x9b, 0x56, 0xc2,
        0x40, 0x0f, 0xac, 0x03, 0xcb, 0x9f, 0xeb, 0x9b, 0xe9, 0x09, 0x03, 0xdb,
        0x97, 0x18, 0xeb, 0xc7, 0x7e, 0x24, 0xe8, 0x87, 0xeb, 0x51, 0x8c, 0x3a,
        0x76, 0x4a, 0xb5, 0x83, 0x11, 0x17, 0xcf, 0x09, 0x10, 0xe5, 0xf2, 0x84,
        0xc9, 0xaf, 0x7e, 0xa9, 0xd7, 0x3a, 0x85, 0x5c, 0x44, 0x70, 0x4c, 0x9b,
        0x0d, 0x24, 0x5d, 0xf6, 0x40, 0x29, 0x6c, 0xce, 0x1c, 0xe4, 0xec, 0xd6,
        0x2f, 0x1b, 0xa9, 0x1c, 0x69, 0x00, 0xbc, 0x1e, 0x19, 0xbd, 0xc9, 0xc7,
        0x11, 0xa0,
      ]),
      publicOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x20, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x53,
        0x53, 0x48, 0x32, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b,
        0x45, 0x59, 0x20, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x41, 0x41, 0x41, 0x41,
        0x42, 0x33, 0x4e, 0x7a, 0x61, 0x43, 0x31, 0x6b, 0x63, 0x33, 0x4d, 0x41,
        0x41, 0x41, 0x43, 0x42, 0x41, 0x49, 0x34, 0x43, 0x52, 0x2b, 0x74, 0x67,
        0x7a, 0x2f, 0x6b, 0x5a, 0x6e, 0x62, 0x6d, 0x4b, 0x4a, 0x6a, 0x63, 0x63,
        0x5a, 0x4e, 0x57, 0x4b, 0x71, 0x63, 0x69, 0x66, 0x63, 0x78, 0x4c, 0x79,
        0x2f, 0x70, 0x54, 0x4d, 0x6e, 0x52, 0x4c, 0x68, 0x35, 0x53, 0x72, 0x7a,
        0x51, 0x34, 0x34, 0x52, 0x38, 0x54, 0x0a, 0x51, 0x53, 0x62, 0x51, 0x54,
        0x4e, 0x6e, 0x4a, 0x48, 0x56, 0x70, 0x63, 0x36, 0x75, 0x63, 0x57, 0x33,
        0x36, 0x39, 0x2b, 0x54, 0x74, 0x51, 0x52, 0x70, 0x52, 0x6d, 0x72, 0x34,
        0x69, 0x55, 0x36, 0x35, 0x74, 0x74, 0x5a, 0x6b, 0x5a, 0x71, 0x43, 0x4a,
        0x31, 0x6a, 0x61, 0x77, 0x58, 0x41, 0x6e, 0x78, 0x63, 0x70, 0x49, 0x72,
        0x56, 0x71, 0x2f, 0x78, 0x42, 0x66, 0x73, 0x58, 0x66, 0x43, 0x49, 0x72,
        0x32, 0x36, 0x4c, 0x61, 0x47, 0x0a, 0x71, 0x4e, 0x6d, 0x2f, 0x2f, 0x76,
        0x59, 0x42, 0x41, 0x50, 0x73, 0x35, 0x6a, 0x36, 0x34, 0x38, 0x46, 0x68,
        0x67, 0x36, 0x41, 0x48, 0x79, 0x4d, 0x2f, 0x4c, 0x65, 0x77, 0x50, 0x53,
        0x6e, 0x65, 0x6b, 0x33, 0x66, 0x45, 0x38, 0x67, 0x49, 0x42, 0x39, 0x67,
        0x74, 0x43, 0x73, 0x50, 0x65, 0x69, 0x35, 0x4c, 0x37, 0x6c, 0x41, 0x41,
        0x41, 0x41, 0x46, 0x51, 0x44, 0x4a, 0x33, 0x6b, 0x36, 0x54, 0x64, 0x4a,
        0x4b, 0x2f, 0x65, 0x6f, 0x0a, 0x4c, 0x67, 0x55, 0x71, 0x67, 0x32, 0x62,
        0x55, 0x42, 0x39, 0x34, 0x4d, 0x6a, 0x67, 0x39, 0x77, 0x41, 0x41, 0x41,
        0x49, 0x41, 0x70, 0x39, 0x51, 0x33, 0x53, 0x79, 0x6f, 0x6f, 0x6b, 0x6c,
        0x4c, 0x47, 0x4d, 0x50, 0x64, 0x47, 0x32, 0x6b, 0x6a, 0x30, 0x76, 0x4d,
        0x57, 0x46, 0x2b, 0x63, 0x56, 0x74, 0x43, 0x68, 0x63, 0x6a, 0x47, 0x6a,
        0x65, 0x6d, 0x68, 0x50, 0x65, 0x48, 0x58, 0x79, 0x6a, 0x44, 0x35, 0x2f,
        0x69, 0x6e, 0x38, 0x0a, 0x67, 0x57, 0x59, 0x46, 0x69, 0x66, 0x56, 0x71,
        0x6f, 0x59, 0x4b, 0x42, 0x4a, 0x68, 0x41, 0x61, 0x50, 0x50, 0x35, 0x48,
        0x70, 0x50, 0x74, 0x41, 0x33, 0x42, 0x45, 0x47, 0x67, 0x49, 0x6b, 0x4c,
        0x56, 0x47, 0x52, 0x7a, 0x43, 0x6b, 0x33, 0x74, 0x6e, 0x77, 0x58, 0x62,
        0x75, 0x74, 0x47, 0x62, 0x77, 0x74, 0x37, 0x50, 0x7a, 0x75, 0x74, 0x62,
        0x4a, 0x39, 0x4c, 0x4b, 0x69, 0x69, 0x52, 0x37, 0x7a, 0x31, 0x48, 0x4c,
        0x71, 0x6e, 0x0a, 0x45, 0x2b, 0x72, 0x30, 0x2f, 0x2f, 0x36, 0x4e, 0x68,
        0x71, 0x7a, 0x2f, 0x5a, 0x4e, 0x51, 0x50, 0x57, 0x5a, 0x52, 0x62, 0x63,
        0x47, 0x30, 0x6c, 0x31, 0x45, 0x49, 0x51, 0x4e, 0x67, 0x6a, 0x75, 0x5a,
        0x6b, 0x53, 0x7a, 0x4a, 0x4c, 0x4c, 0x4f, 0x31, 0x35, 0x54, 0x53, 0x50,
        0x5a, 0x65, 0x56, 0x68, 0x5a, 0x63, 0x51, 0x41, 0x41, 0x41, 0x49, 0x41,
        0x55, 0x41, 0x59, 0x6f, 0x62, 0x79, 0x4b, 0x51, 0x33, 0x52, 0x6c, 0x34,
        0x71, 0x0a, 0x58, 0x41, 0x63, 0x4f, 0x50, 0x38, 0x75, 0x4f, 0x74, 0x35,
        0x71, 0x37, 0x59, 0x48, 0x4d, 0x79, 0x34, 0x33, 0x4a, 0x6d, 0x5a, 0x4a,
        0x6e, 0x54, 0x43, 0x51, 0x37, 0x58, 0x39, 0x35, 0x54, 0x75, 0x4e, 0x76,
        0x35, 0x6b, 0x46, 0x6a, 0x37, 0x61, 0x35, 0x52, 0x6b, 0x44, 0x55, 0x39,
        0x48, 0x6a, 0x67, 0x7a, 0x53, 0x67, 0x35, 0x61, 0x61, 0x62, 0x56, 0x73,
        0x4a, 0x41, 0x44, 0x36, 0x77, 0x44, 0x79, 0x35, 0x2f, 0x72, 0x6d, 0x2b,
        0x0a, 0x6b, 0x4a, 0x41, 0x39, 0x75, 0x58, 0x47, 0x4f, 0x76, 0x48, 0x66,
        0x69, 0x54, 0x6f, 0x68, 0x2b, 0x74, 0x52, 0x6a, 0x44, 0x70, 0x32, 0x53,
        0x72, 0x57, 0x44, 0x45, 0x52, 0x66, 0x50, 0x43, 0x52, 0x44, 0x6c, 0x38,
        0x6f, 0x54, 0x4a, 0x72, 0x33, 0x36, 0x70, 0x31, 0x7a, 0x71, 0x46, 0x58,
        0x45, 0x52, 0x77, 0x54, 0x4a, 0x73, 0x4e, 0x4a, 0x46, 0x33, 0x32, 0x51,
        0x43, 0x6c, 0x73, 0x7a, 0x68, 0x7a, 0x6b, 0x37, 0x4e, 0x59, 0x76, 0x0a,
        0x47, 0x36, 0x6b, 0x63, 0x61, 0x51, 0x43, 0x38, 0x48, 0x68, 0x6d, 0x39,
        0x79, 0x63, 0x63, 0x52, 0x6f, 0x41, 0x3d, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d,
        0x2d, 0x20, 0x45, 0x4e, 0x44, 0x20, 0x53, 0x53, 0x48, 0x32, 0x20, 0x50,
        0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b, 0x45, 0x59, 0x20, 0x2d, 0x2d,
        0x2d, 0x2d,
      ])
    },
    what: 'DSA public key (RFC4716 format) with no comment'
  },
  { source: [
      'PuTTY-User-Key-File-2: ssh-rsa',
      'Encryption: none',
      'Comment: rsa-key-20141119',
      'Public-Lines: 4',
      'AAAAB3NzaC1yc2EAAAABJQAAAIBrBWETAVAyJmuNG53jwTNDlbIcH5lrEvcx6lx5',
      'bM6EKg0XmOIH96VqUjS7eRRTTD9lpBA8hYhkrOjOx93/JWB/pcVN8/B3DYHshT9O',
      'BW1DCkrNwut2pbJ2oZOBirhhAr+xqWFr3551FqbzaCIXpOKubr4EcIwCipBl6PxL',
      'USfHgw==',
      'Private-Lines: 8',
      'AAAAgFPhnxy71xKM0NZhwPDH3BJgkoS+0jFUDdsDy/B34CJmJe/mh6VhPHXtZ5nb',
      'cMuqduD2Nj1GEPT2Oe4t8y/DWXfFkO0YNEQRw5Z3WhXheBH9Li26KmHf8TdK838u',
      'ec+4Vji/vFYaMU6wVL73joWcAT5Dnh38ZpFc98W264N5DZaNAAAAQQDJYPKtCs/l',
      '46KJmN3lUANdI4QIuWQ+Zllz7p94FfdotnkvqG++Bp1wOqJSCih6UViwLfvpNZtG',
      'MCtk46WNhc0zAAAAQQCIDI3hZSz/THhai8b3nonUD65IQp5bO6p+kYZtaBn1d+c+',
      'eT7UCXwU5bW271Zasw8hq9Cdlb91fGGR41ZMfvxxAAAAQFSBjCa/fzeICVkPFBaK',
      'QUmXjQ3IcPTOr90mSAiPnAAppSwTj5SYSfE9rSVb+EhQ0hk2VKWIfocNHBD1MAN9',
      'zb4=',
      'Private-MAC: 0bc5cc619df85b79dfd3ea25f0e59230cf671cd2'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: 'rsa-key-20141119',
      encryption: undefined,
      private: new Buffer([
        0x30, 0x82, 0x02, 0x59, 0x02, 0x01, 0x00, 0x02, 0x81, 0x80, 0x6b, 0x05,
        0x61, 0x13, 0x01, 0x50, 0x32, 0x26, 0x6b, 0x8d, 0x1b, 0x9d, 0xe3, 0xc1,
        0x33, 0x43, 0x95, 0xb2, 0x1c, 0x1f, 0x99, 0x6b, 0x12, 0xf7, 0x31, 0xea,
        0x5c, 0x79, 0x6c, 0xce, 0x84, 0x2a, 0x0d, 0x17, 0x98, 0xe2, 0x07, 0xf7,
        0xa5, 0x6a, 0x52, 0x34, 0xbb, 0x79, 0x14, 0x53, 0x4c, 0x3f, 0x65, 0xa4,
        0x10, 0x3c, 0x85, 0x88, 0x64, 0xac, 0xe8, 0xce, 0xc7, 0xdd, 0xff, 0x25,
        0x60, 0x7f, 0xa5, 0xc5, 0x4d, 0xf3, 0xf0, 0x77, 0x0d, 0x81, 0xec, 0x85,
        0x3f, 0x4e, 0x05, 0x6d, 0x43, 0x0a, 0x4a, 0xcd, 0xc2, 0xeb, 0x76, 0xa5,
        0xb2, 0x76, 0xa1, 0x93, 0x81, 0x8a, 0xb8, 0x61, 0x02, 0xbf, 0xb1, 0xa9,
        0x61, 0x6b, 0xdf, 0x9e, 0x75, 0x16, 0xa6, 0xf3, 0x68, 0x22, 0x17, 0xa4,
        0xe2, 0xae, 0x6e, 0xbe, 0x04, 0x70, 0x8c, 0x02, 0x8a, 0x90, 0x65, 0xe8,
        0xfc, 0x4b, 0x51, 0x27, 0xc7, 0x83, 0x02, 0x01, 0x25, 0x02, 0x81, 0x80,
        0x53, 0xe1, 0x9f, 0x1c, 0xbb, 0xd7, 0x12, 0x8c, 0xd0, 0xd6, 0x61, 0xc0,
        0xf0, 0xc7, 0xdc, 0x12, 0x60, 0x92, 0x84, 0xbe, 0xd2, 0x31, 0x54, 0x0d,
        0xdb, 0x03, 0xcb, 0xf0, 0x77, 0xe0, 0x22, 0x66, 0x25, 0xef, 0xe6, 0x87,
        0xa5, 0x61, 0x3c, 0x75, 0xed, 0x67, 0x99, 0xdb, 0x70, 0xcb, 0xaa, 0x76,
        0xe0, 0xf6, 0x36, 0x3d, 0x46, 0x10, 0xf4, 0xf6, 0x39, 0xee, 0x2d, 0xf3,
        0x2f, 0xc3, 0x59, 0x77, 0xc5, 0x90, 0xed, 0x18, 0x34, 0x44, 0x11, 0xc3,
        0x96, 0x77, 0x5a, 0x15, 0xe1, 0x78, 0x11, 0xfd, 0x2e, 0x2d, 0xba, 0x2a,
        0x61, 0xdf, 0xf1, 0x37, 0x4a, 0xf3, 0x7f, 0x2e, 0x79, 0xcf, 0xb8, 0x56,
        0x38, 0xbf, 0xbc, 0x56, 0x1a, 0x31, 0x4e, 0xb0, 0x54, 0xbe, 0xf7, 0x8e,
        0x85, 0x9c, 0x01, 0x3e, 0x43, 0x9e, 0x1d, 0xfc, 0x66, 0x91, 0x5c, 0xf7,
        0xc5, 0xb6, 0xeb, 0x83, 0x79, 0x0d, 0x96, 0x8d, 0x02, 0x41, 0x00, 0xc9,
        0x60, 0xf2, 0xad, 0x0a, 0xcf, 0xe5, 0xe3, 0xa2, 0x89, 0x98, 0xdd, 0xe5,
        0x50, 0x03, 0x5d, 0x23, 0x84, 0x08, 0xb9, 0x64, 0x3e, 0x66, 0x59, 0x73,
        0xee, 0x9f, 0x78, 0x15, 0xf7, 0x68, 0xb6, 0x79, 0x2f, 0xa8, 0x6f, 0xbe,
        0x06, 0x9d, 0x70, 0x3a, 0xa2, 0x52, 0x0a, 0x28, 0x7a, 0x51, 0x58, 0xb0,
        0x2d, 0xfb, 0xe9, 0x35, 0x9b, 0x46, 0x30, 0x2b, 0x64, 0xe3, 0xa5, 0x8d,
        0x85, 0xcd, 0x33, 0x02, 0x41, 0x00, 0x88, 0x0c, 0x8d, 0xe1, 0x65, 0x2c,
        0xff, 0x4c, 0x78, 0x5a, 0x8b, 0xc6, 0xf7, 0x9e, 0x89, 0xd4, 0x0f, 0xae,
        0x48, 0x42, 0x9e, 0x5b, 0x3b, 0xaa, 0x7e, 0x91, 0x86, 0x6d, 0x68, 0x19,
        0xf5, 0x77, 0xe7, 0x3e, 0x79, 0x3e, 0xd4, 0x09, 0x7c, 0x14, 0xe5, 0xb5,
        0xb6, 0xef, 0x56, 0x5a, 0xb3, 0x0f, 0x21, 0xab, 0xd0, 0x9d, 0x95, 0xbf,
        0x75, 0x7c, 0x61, 0x91, 0xe3, 0x56, 0x4c, 0x7e, 0xfc, 0x71, 0x02, 0x41,
        0x00, 0xb9, 0x0c, 0xfa, 0xac, 0xd9, 0x80, 0xc5, 0x69, 0x64, 0xed, 0x24,
        0xb0, 0x3a, 0x80, 0xe0, 0x7f, 0x19, 0xb7, 0x99, 0x50, 0x69, 0xf4, 0x26,
        0xb3, 0x10, 0x96, 0x15, 0xff, 0xa5, 0x7b, 0x90, 0xa7, 0xad, 0xa1, 0x6a,
        0x58, 0xd8, 0x21, 0xc1, 0x1b, 0x05, 0x72, 0x90, 0x93, 0xb6, 0x7e, 0x3c,
        0xe9, 0xb6, 0xa6, 0xcb, 0xcf, 0x61, 0xb1, 0x47, 0x6a, 0x88, 0xbd, 0x92,
        0xeb, 0x28, 0x1a, 0x16, 0x81, 0x02, 0x40, 0x50, 0xe4, 0xde, 0xbd, 0x5e,
        0xc0, 0xcf, 0x26, 0x8c, 0xc0, 0x37, 0x6f, 0x62, 0xcc, 0xf8, 0x01, 0x8c,
        0xc8, 0x7d, 0xfe, 0x18, 0xf7, 0xf9, 0xf6, 0xac, 0x1f, 0x2d, 0x55, 0xcf,
        0x32, 0x07, 0x93, 0x66, 0xe6, 0xe0, 0x4e, 0xde, 0xf0, 0xe1, 0xfe, 0x96,
        0x6c, 0x0b, 0xe8, 0x41, 0x2f, 0x02, 0xaf, 0x0d, 0x1a, 0x0d, 0x56, 0xc7,
        0xbd, 0xf2, 0xd4, 0x55, 0xb0, 0xb0, 0xaf, 0xda, 0x75, 0x04, 0xcd, 0x02,
        0x40, 0x54, 0x81, 0x8c, 0x26, 0xbf, 0x7f, 0x37, 0x88, 0x09, 0x59, 0x0f,
        0x14, 0x16, 0x8a, 0x41, 0x49, 0x97, 0x8d, 0x0d, 0xc8, 0x70, 0xf4, 0xce,
        0xaf, 0xdd, 0x26, 0x48, 0x08, 0x8f, 0x9c, 0x00, 0x29, 0xa5, 0x2c, 0x13,
        0x8f, 0x94, 0x98, 0x49, 0xf1, 0x3d, 0xad, 0x25, 0x5b, 0xf8, 0x48, 0x50,
        0xd2, 0x19, 0x36, 0x54, 0xa5, 0x88, 0x7e, 0x87, 0x0d, 0x1c, 0x10, 0xf5,
        0x30, 0x03, 0x7d, 0xcd, 0xbe,
      ]),
      privateOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x52,
        0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
        0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x43,
        0x57, 0x51, 0x49, 0x42, 0x41, 0x41, 0x4b, 0x42, 0x67, 0x47, 0x73, 0x46,
        0x59, 0x52, 0x4d, 0x42, 0x55, 0x44, 0x49, 0x6d, 0x61, 0x34, 0x30, 0x62,
        0x6e, 0x65, 0x50, 0x42, 0x4d, 0x30, 0x4f, 0x56, 0x73, 0x68, 0x77, 0x66,
        0x6d, 0x57, 0x73, 0x53, 0x39, 0x7a, 0x48, 0x71, 0x58, 0x48, 0x6c, 0x73,
        0x7a, 0x6f, 0x51, 0x71, 0x44, 0x52, 0x65, 0x59, 0x34, 0x67, 0x66, 0x33,
        0x0a, 0x70, 0x57, 0x70, 0x53, 0x4e, 0x4c, 0x74, 0x35, 0x46, 0x46, 0x4e,
        0x4d, 0x50, 0x32, 0x57, 0x6b, 0x45, 0x44, 0x79, 0x46, 0x69, 0x47, 0x53,
        0x73, 0x36, 0x4d, 0x37, 0x48, 0x33, 0x66, 0x38, 0x6c, 0x59, 0x48, 0x2b,
        0x6c, 0x78, 0x55, 0x33, 0x7a, 0x38, 0x48, 0x63, 0x4e, 0x67, 0x65, 0x79,
        0x46, 0x50, 0x30, 0x34, 0x46, 0x62, 0x55, 0x4d, 0x4b, 0x53, 0x73, 0x33,
        0x43, 0x36, 0x33, 0x61, 0x6c, 0x0a, 0x73, 0x6e, 0x61, 0x68, 0x6b, 0x34,
        0x47, 0x4b, 0x75, 0x47, 0x45, 0x43, 0x76, 0x37, 0x47, 0x70, 0x59, 0x57,
        0x76, 0x66, 0x6e, 0x6e, 0x55, 0x57, 0x70, 0x76, 0x4e, 0x6f, 0x49, 0x68,
        0x65, 0x6b, 0x34, 0x71, 0x35, 0x75, 0x76, 0x67, 0x52, 0x77, 0x6a, 0x41,
        0x4b, 0x4b, 0x6b, 0x47, 0x58, 0x6f, 0x2f, 0x45, 0x74, 0x52, 0x4a, 0x38,
        0x65, 0x44, 0x41, 0x67, 0x45, 0x6c, 0x41, 0x6f, 0x47, 0x41, 0x0a, 0x55,
        0x2b, 0x47, 0x66, 0x48, 0x4c, 0x76, 0x58, 0x45, 0x6f, 0x7a, 0x51, 0x31,
        0x6d, 0x48, 0x41, 0x38, 0x4d, 0x66, 0x63, 0x45, 0x6d, 0x43, 0x53, 0x68,
        0x4c, 0x37, 0x53, 0x4d, 0x56, 0x51, 0x4e, 0x32, 0x77, 0x50, 0x4c, 0x38,
        0x48, 0x66, 0x67, 0x49, 0x6d, 0x59, 0x6c, 0x37, 0x2b, 0x61, 0x48, 0x70,
        0x57, 0x45, 0x38, 0x64, 0x65, 0x31, 0x6e, 0x6d, 0x64, 0x74, 0x77, 0x79,
        0x36, 0x70, 0x32, 0x0a, 0x34, 0x50, 0x59, 0x32, 0x50, 0x55, 0x59, 0x51,
        0x39, 0x50, 0x59, 0x35, 0x37, 0x69, 0x33, 0x7a, 0x4c, 0x38, 0x4e, 0x5a,
        0x64, 0x38, 0x57, 0x51, 0x37, 0x52, 0x67, 0x30, 0x52, 0x42, 0x48, 0x44,
        0x6c, 0x6e, 0x64, 0x61, 0x46, 0x65, 0x46, 0x34, 0x45, 0x66, 0x30, 0x75,
        0x4c, 0x62, 0x6f, 0x71, 0x59, 0x64, 0x2f, 0x78, 0x4e, 0x30, 0x72, 0x7a,
        0x66, 0x79, 0x35, 0x35, 0x7a, 0x37, 0x68, 0x57, 0x0a, 0x4f, 0x4c, 0x2b,
        0x38, 0x56, 0x68, 0x6f, 0x78, 0x54, 0x72, 0x42, 0x55, 0x76, 0x76, 0x65,
        0x4f, 0x68, 0x5a, 0x77, 0x42, 0x50, 0x6b, 0x4f, 0x65, 0x48, 0x66, 0x78,
        0x6d, 0x6b, 0x56, 0x7a, 0x33, 0x78, 0x62, 0x62, 0x72, 0x67, 0x33, 0x6b,
        0x4e, 0x6c, 0x6f, 0x30, 0x43, 0x51, 0x51, 0x44, 0x4a, 0x59, 0x50, 0x4b,
        0x74, 0x43, 0x73, 0x2f, 0x6c, 0x34, 0x36, 0x4b, 0x4a, 0x6d, 0x4e, 0x33,
        0x6c, 0x0a, 0x55, 0x41, 0x4e, 0x64, 0x49, 0x34, 0x51, 0x49, 0x75, 0x57,
        0x51, 0x2b, 0x5a, 0x6c, 0x6c, 0x7a, 0x37, 0x70, 0x39, 0x34, 0x46, 0x66,
        0x64, 0x6f, 0x74, 0x6e, 0x6b, 0x76, 0x71, 0x47, 0x2b, 0x2b, 0x42, 0x70,
        0x31, 0x77, 0x4f, 0x71, 0x4a, 0x53, 0x43, 0x69, 0x68, 0x36, 0x55, 0x56,
        0x69, 0x77, 0x4c, 0x66, 0x76, 0x70, 0x4e, 0x5a, 0x74, 0x47, 0x4d, 0x43,
        0x74, 0x6b, 0x34, 0x36, 0x57, 0x4e, 0x0a, 0x68, 0x63, 0x30, 0x7a, 0x41,
        0x6b, 0x45, 0x41, 0x69, 0x41, 0x79, 0x4e, 0x34, 0x57, 0x55, 0x73, 0x2f,
        0x30, 0x78, 0x34, 0x57, 0x6f, 0x76, 0x47, 0x39, 0x35, 0x36, 0x4a, 0x31,
        0x41, 0x2b, 0x75, 0x53, 0x45, 0x4b, 0x65, 0x57, 0x7a, 0x75, 0x71, 0x66,
        0x70, 0x47, 0x47, 0x62, 0x57, 0x67, 0x5a, 0x39, 0x58, 0x66, 0x6e, 0x50,
        0x6e, 0x6b, 0x2b, 0x31, 0x41, 0x6c, 0x38, 0x46, 0x4f, 0x57, 0x31, 0x0a,
        0x74, 0x75, 0x39, 0x57, 0x57, 0x72, 0x4d, 0x50, 0x49, 0x61, 0x76, 0x51,
        0x6e, 0x5a, 0x57, 0x2f, 0x64, 0x58, 0x78, 0x68, 0x6b, 0x65, 0x4e, 0x57,
        0x54, 0x48, 0x37, 0x38, 0x63, 0x51, 0x4a, 0x42, 0x41, 0x4c, 0x6b, 0x4d,
        0x2b, 0x71, 0x7a, 0x5a, 0x67, 0x4d, 0x56, 0x70, 0x5a, 0x4f, 0x30, 0x6b,
        0x73, 0x44, 0x71, 0x41, 0x34, 0x48, 0x38, 0x5a, 0x74, 0x35, 0x6c, 0x51,
        0x61, 0x66, 0x51, 0x6d, 0x0a, 0x73, 0x78, 0x43, 0x57, 0x46, 0x66, 0x2b,
        0x6c, 0x65, 0x35, 0x43, 0x6e, 0x72, 0x61, 0x46, 0x71, 0x57, 0x4e, 0x67,
        0x68, 0x77, 0x52, 0x73, 0x46, 0x63, 0x70, 0x43, 0x54, 0x74, 0x6e, 0x34,
        0x38, 0x36, 0x62, 0x61, 0x6d, 0x79, 0x38, 0x39, 0x68, 0x73, 0x55, 0x64,
        0x71, 0x69, 0x4c, 0x32, 0x53, 0x36, 0x79, 0x67, 0x61, 0x46, 0x6f, 0x45,
        0x43, 0x51, 0x46, 0x44, 0x6b, 0x33, 0x72, 0x31, 0x65, 0x0a, 0x77, 0x4d,
        0x38, 0x6d, 0x6a, 0x4d, 0x41, 0x33, 0x62, 0x32, 0x4c, 0x4d, 0x2b, 0x41,
        0x47, 0x4d, 0x79, 0x48, 0x33, 0x2b, 0x47, 0x50, 0x66, 0x35, 0x39, 0x71,
        0x77, 0x66, 0x4c, 0x56, 0x58, 0x50, 0x4d, 0x67, 0x65, 0x54, 0x5a, 0x75,
        0x62, 0x67, 0x54, 0x74, 0x37, 0x77, 0x34, 0x66, 0x36, 0x57, 0x62, 0x41,
        0x76, 0x6f, 0x51, 0x53, 0x38, 0x43, 0x72, 0x77, 0x30, 0x61, 0x44, 0x56,
        0x62, 0x48, 0x0a, 0x76, 0x66, 0x4c, 0x55, 0x56, 0x62, 0x43, 0x77, 0x72,
        0x39, 0x70, 0x31, 0x42, 0x4d, 0x30, 0x43, 0x51, 0x46, 0x53, 0x42, 0x6a,
        0x43, 0x61, 0x2f, 0x66, 0x7a, 0x65, 0x49, 0x43, 0x56, 0x6b, 0x50, 0x46,
        0x42, 0x61, 0x4b, 0x51, 0x55, 0x6d, 0x58, 0x6a, 0x51, 0x33, 0x49, 0x63,
        0x50, 0x54, 0x4f, 0x72, 0x39, 0x30, 0x6d, 0x53, 0x41, 0x69, 0x50, 0x6e,
        0x41, 0x41, 0x70, 0x70, 0x53, 0x77, 0x54, 0x0a, 0x6a, 0x35, 0x53, 0x59,
        0x53, 0x66, 0x45, 0x39, 0x72, 0x53, 0x56, 0x62, 0x2b, 0x45, 0x68, 0x51,
        0x30, 0x68, 0x6b, 0x32, 0x56, 0x4b, 0x57, 0x49, 0x66, 0x6f, 0x63, 0x4e,
        0x48, 0x42, 0x44, 0x31, 0x4d, 0x41, 0x4e, 0x39, 0x7a, 0x62, 0x34, 0x3d,
        0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x52, 0x53,
        0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45,
        0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d,
      ]),
      public: new Buffer([
        0x00, 0x00, 0x00, 0x07, 0x73, 0x73, 0x68, 0x2d, 0x72, 0x73, 0x61, 0x00,
        0x00, 0x00, 0x01, 0x25, 0x00, 0x00, 0x00, 0x80, 0x6b, 0x05, 0x61, 0x13,
        0x01, 0x50, 0x32, 0x26, 0x6b, 0x8d, 0x1b, 0x9d, 0xe3, 0xc1, 0x33, 0x43,
        0x95, 0xb2, 0x1c, 0x1f, 0x99, 0x6b, 0x12, 0xf7, 0x31, 0xea, 0x5c, 0x79,
        0x6c, 0xce, 0x84, 0x2a, 0x0d, 0x17, 0x98, 0xe2, 0x07, 0xf7, 0xa5, 0x6a,
        0x52, 0x34, 0xbb, 0x79, 0x14, 0x53, 0x4c, 0x3f, 0x65, 0xa4, 0x10, 0x3c,
        0x85, 0x88, 0x64, 0xac, 0xe8, 0xce, 0xc7, 0xdd, 0xff, 0x25, 0x60, 0x7f,
        0xa5, 0xc5, 0x4d, 0xf3, 0xf0, 0x77, 0x0d, 0x81, 0xec, 0x85, 0x3f, 0x4e,
        0x05, 0x6d, 0x43, 0x0a, 0x4a, 0xcd, 0xc2, 0xeb, 0x76, 0xa5, 0xb2, 0x76,
        0xa1, 0x93, 0x81, 0x8a, 0xb8, 0x61, 0x02, 0xbf, 0xb1, 0xa9, 0x61, 0x6b,
        0xdf, 0x9e, 0x75, 0x16, 0xa6, 0xf3, 0x68, 0x22, 0x17, 0xa4, 0xe2, 0xae,
        0x6e, 0xbe, 0x04, 0x70, 0x8c, 0x02, 0x8a, 0x90, 0x65, 0xe8, 0xfc, 0x4b,
        0x51, 0x27, 0xc7, 0x83,
      ]),
      publicOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x50,
        0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d,
        0x2d, 0x2d, 0x0a, 0x4d, 0x49, 0x47, 0x63, 0x4d, 0x41, 0x30, 0x47, 0x43,
        0x53, 0x71, 0x47, 0x53, 0x49, 0x62, 0x33, 0x44, 0x51, 0x45, 0x42, 0x41,
        0x51, 0x55, 0x41, 0x41, 0x34, 0x47, 0x4b, 0x41, 0x44, 0x43, 0x42, 0x68,
        0x67, 0x4b, 0x42, 0x67, 0x47, 0x73, 0x46, 0x59, 0x52, 0x4d, 0x42, 0x55,
        0x44, 0x49, 0x6d, 0x61, 0x34, 0x30, 0x62, 0x6e, 0x65, 0x50, 0x42, 0x4d,
        0x30, 0x4f, 0x56, 0x73, 0x68, 0x77, 0x66, 0x0a, 0x6d, 0x57, 0x73, 0x53,
        0x39, 0x7a, 0x48, 0x71, 0x58, 0x48, 0x6c, 0x73, 0x7a, 0x6f, 0x51, 0x71,
        0x44, 0x52, 0x65, 0x59, 0x34, 0x67, 0x66, 0x33, 0x70, 0x57, 0x70, 0x53,
        0x4e, 0x4c, 0x74, 0x35, 0x46, 0x46, 0x4e, 0x4d, 0x50, 0x32, 0x57, 0x6b,
        0x45, 0x44, 0x79, 0x46, 0x69, 0x47, 0x53, 0x73, 0x36, 0x4d, 0x37, 0x48,
        0x33, 0x66, 0x38, 0x6c, 0x59, 0x48, 0x2b, 0x6c, 0x78, 0x55, 0x33, 0x7a,
        0x0a, 0x38, 0x48, 0x63, 0x4e, 0x67, 0x65, 0x79, 0x46, 0x50, 0x30, 0x34,
        0x46, 0x62, 0x55, 0x4d, 0x4b, 0x53, 0x73, 0x33, 0x43, 0x36, 0x33, 0x61,
        0x6c, 0x73, 0x6e, 0x61, 0x68, 0x6b, 0x34, 0x47, 0x4b, 0x75, 0x47, 0x45,
        0x43, 0x76, 0x37, 0x47, 0x70, 0x59, 0x57, 0x76, 0x66, 0x6e, 0x6e, 0x55,
        0x57, 0x70, 0x76, 0x4e, 0x6f, 0x49, 0x68, 0x65, 0x6b, 0x34, 0x71, 0x35,
        0x75, 0x76, 0x67, 0x52, 0x77, 0x0a, 0x6a, 0x41, 0x4b, 0x4b, 0x6b, 0x47,
        0x58, 0x6f, 0x2f, 0x45, 0x74, 0x52, 0x4a, 0x38, 0x65, 0x44, 0x41, 0x67,
        0x45, 0x6c, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20,
        0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d,
        0x2d, 0x2d, 0x2d,
      ]),
      ppk: true,
      privateMAC: '0bc5cc619df85b79dfd3ea25f0e59230cf671cd2',
      _converted: true,
      _macresult: true
    },
    what: 'Unencrypted RSA key (PPK format)'
  },
  { source: [
      'PuTTY-User-Key-File-2: ssh-rsa',
      'Encryption: aes256-cbc',
      'Comment: rsa-key-20141119',
      'Public-Lines: 4',
      'AAAAB3NzaC1yc2EAAAABJQAAAIBrBWETAVAyJmuNG53jwTNDlbIcH5lrEvcx6lx5',
      'bM6EKg0XmOIH96VqUjS7eRRTTD9lpBA8hYhkrOjOx93/JWB/pcVN8/B3DYHshT9O',
      'BW1DCkrNwut2pbJ2oZOBirhhAr+xqWFr3551FqbzaCIXpOKubr4EcIwCipBl6PxL',
      'USfHgw==',
      'Private-Lines: 8',
      '8O3NrBePR4+4RHHys8wrRKCmgx3Gsdz1cKoRJJDgnnrQxuAxBTVUlVTC2vzSOXrP',
      'jlKdRP9DbtrL5YF8g9HkMPpzzTdgpiEAGikpIc+L0sJhN+S9VvMoXRRKqyuB7o1C',
      'xZhAeRaZ68izdUUbFd7ajUwBNpGoFppOznGXyf/3/Ao9FfoTKReZzeBd/e2/JFhc',
      'nsYkSbtWfKQBVXF1Fhr10UwRWSMaVJSDkcSuk8ghICoKBBCgRBnZFap0SR77oIJh',
      'DKgmNFktoKzEqh111vYPhQyEEyGNxpD0aEPaGUJEjPEd3C5a46n7mIiqrNX7QJoo',
      'xxZtkueGdXWaoe5mBf1tFc+nCA1l72nUlghJZooQhnO9NPpieu6NNZ8X+tFQ1Rq/',
      'xvOZHzpDOOeOgWdV7oAmRDbDjYPh0H67z2OKCFaP0Z9kgmnwqV2IJvTDrexj1VwY',
      '6kFaPldnK+ohXl37oVIlWA==',
      'Private-MAC: 9d09a15a122e48955682ba969d33c75ba8e4be2c'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-rsa',
      type: 'rsa',
      extra: undefined,
      comment: 'rsa-key-20141119',
      encryption: 'aes256-cbc',
      private: new Buffer([
        0xf0, 0xed, 0xcd, 0xac, 0x17, 0x8f, 0x47, 0x8f, 0xb8, 0x44, 0x71, 0xf2,
        0xb3, 0xcc, 0x2b, 0x44, 0xa0, 0xa6, 0x83, 0x1d, 0xc6, 0xb1, 0xdc, 0xf5,
        0x70, 0xaa, 0x11, 0x24, 0x90, 0xe0, 0x9e, 0x7a, 0xd0, 0xc6, 0xe0, 0x31,
        0x05, 0x35, 0x54, 0x95, 0x54, 0xc2, 0xda, 0xfc, 0xd2, 0x39, 0x7a, 0xcf,
        0x8e, 0x52, 0x9d, 0x44, 0xff, 0x43, 0x6e, 0xda, 0xcb, 0xe5, 0x81, 0x7c,
        0x83, 0xd1, 0xe4, 0x30, 0xfa, 0x73, 0xcd, 0x37, 0x60, 0xa6, 0x21, 0x00,
        0x1a, 0x29, 0x29, 0x21, 0xcf, 0x8b, 0xd2, 0xc2, 0x61, 0x37, 0xe4, 0xbd,
        0x56, 0xf3, 0x28, 0x5d, 0x14, 0x4a, 0xab, 0x2b, 0x81, 0xee, 0x8d, 0x42,
        0xc5, 0x98, 0x40, 0x79, 0x16, 0x99, 0xeb, 0xc8, 0xb3, 0x75, 0x45, 0x1b,
        0x15, 0xde, 0xda, 0x8d, 0x4c, 0x01, 0x36, 0x91, 0xa8, 0x16, 0x9a, 0x4e,
        0xce, 0x71, 0x97, 0xc9, 0xff, 0xf7, 0xfc, 0x0a, 0x3d, 0x15, 0xfa, 0x13,
        0x29, 0x17, 0x99, 0xcd, 0xe0, 0x5d, 0xfd, 0xed, 0xbf, 0x24, 0x58, 0x5c,
        0x9e, 0xc6, 0x24, 0x49, 0xbb, 0x56, 0x7c, 0xa4, 0x01, 0x55, 0x71, 0x75,
        0x16, 0x1a, 0xf5, 0xd1, 0x4c, 0x11, 0x59, 0x23, 0x1a, 0x54, 0x94, 0x83,
        0x91, 0xc4, 0xae, 0x93, 0xc8, 0x21, 0x20, 0x2a, 0x0a, 0x04, 0x10, 0xa0,
        0x44, 0x19, 0xd9, 0x15, 0xaa, 0x74, 0x49, 0x1e, 0xfb, 0xa0, 0x82, 0x61,
        0x0c, 0xa8, 0x26, 0x34, 0x59, 0x2d, 0xa0, 0xac, 0xc4, 0xaa, 0x1d, 0x75,
        0xd6, 0xf6, 0x0f, 0x85, 0x0c, 0x84, 0x13, 0x21, 0x8d, 0xc6, 0x90, 0xf4,
        0x68, 0x43, 0xda, 0x19, 0x42, 0x44, 0x8c, 0xf1, 0x1d, 0xdc, 0x2e, 0x5a,
        0xe3, 0xa9, 0xfb, 0x98, 0x88, 0xaa, 0xac, 0xd5, 0xfb, 0x40, 0x9a, 0x28,
        0xc7, 0x16, 0x6d, 0x92, 0xe7, 0x86, 0x75, 0x75, 0x9a, 0xa1, 0xee, 0x66,
        0x05, 0xfd, 0x6d, 0x15, 0xcf, 0xa7, 0x08, 0x0d, 0x65, 0xef, 0x69, 0xd4,
        0x96, 0x08, 0x49, 0x66, 0x8a, 0x10, 0x86, 0x73, 0xbd, 0x34, 0xfa, 0x62,
        0x7a, 0xee, 0x8d, 0x35, 0x9f, 0x17, 0xfa, 0xd1, 0x50, 0xd5, 0x1a, 0xbf,
        0xc6, 0xf3, 0x99, 0x1f, 0x3a, 0x43, 0x38, 0xe7, 0x8e, 0x81, 0x67, 0x55,
        0xee, 0x80, 0x26, 0x44, 0x36, 0xc3, 0x8d, 0x83, 0xe1, 0xd0, 0x7e, 0xbb,
        0xcf, 0x63, 0x8a, 0x08, 0x56, 0x8f, 0xd1, 0x9f, 0x64, 0x82, 0x69, 0xf0,
        0xa9, 0x5d, 0x88, 0x26, 0xf4, 0xc3, 0xad, 0xec, 0x63, 0xd5, 0x5c, 0x18,
        0xea, 0x41, 0x5a, 0x3e, 0x57, 0x67, 0x2b, 0xea, 0x21, 0x5e, 0x5d, 0xfb,
        0xa1, 0x52, 0x25, 0x58,
      ]),
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x07, 0x73, 0x73, 0x68, 0x2d, 0x72, 0x73, 0x61, 0x00,
        0x00, 0x00, 0x01, 0x25, 0x00, 0x00, 0x00, 0x80, 0x6b, 0x05, 0x61, 0x13,
        0x01, 0x50, 0x32, 0x26, 0x6b, 0x8d, 0x1b, 0x9d, 0xe3, 0xc1, 0x33, 0x43,
        0x95, 0xb2, 0x1c, 0x1f, 0x99, 0x6b, 0x12, 0xf7, 0x31, 0xea, 0x5c, 0x79,
        0x6c, 0xce, 0x84, 0x2a, 0x0d, 0x17, 0x98, 0xe2, 0x07, 0xf7, 0xa5, 0x6a,
        0x52, 0x34, 0xbb, 0x79, 0x14, 0x53, 0x4c, 0x3f, 0x65, 0xa4, 0x10, 0x3c,
        0x85, 0x88, 0x64, 0xac, 0xe8, 0xce, 0xc7, 0xdd, 0xff, 0x25, 0x60, 0x7f,
        0xa5, 0xc5, 0x4d, 0xf3, 0xf0, 0x77, 0x0d, 0x81, 0xec, 0x85, 0x3f, 0x4e,
        0x05, 0x6d, 0x43, 0x0a, 0x4a, 0xcd, 0xc2, 0xeb, 0x76, 0xa5, 0xb2, 0x76,
        0xa1, 0x93, 0x81, 0x8a, 0xb8, 0x61, 0x02, 0xbf, 0xb1, 0xa9, 0x61, 0x6b,
        0xdf, 0x9e, 0x75, 0x16, 0xa6, 0xf3, 0x68, 0x22, 0x17, 0xa4, 0xe2, 0xae,
        0x6e, 0xbe, 0x04, 0x70, 0x8c, 0x02, 0x8a, 0x90, 0x65, 0xe8, 0xfc, 0x4b,
        0x51, 0x27, 0xc7, 0x83,
      ]),
      publicOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x50,
        0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d,
        0x2d, 0x2d, 0x0a, 0x4d, 0x49, 0x47, 0x63, 0x4d, 0x41, 0x30, 0x47, 0x43,
        0x53, 0x71, 0x47, 0x53, 0x49, 0x62, 0x33, 0x44, 0x51, 0x45, 0x42, 0x41,
        0x51, 0x55, 0x41, 0x41, 0x34, 0x47, 0x4b, 0x41, 0x44, 0x43, 0x42, 0x68,
        0x67, 0x4b, 0x42, 0x67, 0x47, 0x73, 0x46, 0x59, 0x52, 0x4d, 0x42, 0x55,
        0x44, 0x49, 0x6d, 0x61, 0x34, 0x30, 0x62, 0x6e, 0x65, 0x50, 0x42, 0x4d,
        0x30, 0x4f, 0x56, 0x73, 0x68, 0x77, 0x66, 0x0a, 0x6d, 0x57, 0x73, 0x53,
        0x39, 0x7a, 0x48, 0x71, 0x58, 0x48, 0x6c, 0x73, 0x7a, 0x6f, 0x51, 0x71,
        0x44, 0x52, 0x65, 0x59, 0x34, 0x67, 0x66, 0x33, 0x70, 0x57, 0x70, 0x53,
        0x4e, 0x4c, 0x74, 0x35, 0x46, 0x46, 0x4e, 0x4d, 0x50, 0x32, 0x57, 0x6b,
        0x45, 0x44, 0x79, 0x46, 0x69, 0x47, 0x53, 0x73, 0x36, 0x4d, 0x37, 0x48,
        0x33, 0x66, 0x38, 0x6c, 0x59, 0x48, 0x2b, 0x6c, 0x78, 0x55, 0x33, 0x7a,
        0x0a, 0x38, 0x48, 0x63, 0x4e, 0x67, 0x65, 0x79, 0x46, 0x50, 0x30, 0x34,
        0x46, 0x62, 0x55, 0x4d, 0x4b, 0x53, 0x73, 0x33, 0x43, 0x36, 0x33, 0x61,
        0x6c, 0x73, 0x6e, 0x61, 0x68, 0x6b, 0x34, 0x47, 0x4b, 0x75, 0x47, 0x45,
        0x43, 0x76, 0x37, 0x47, 0x70, 0x59, 0x57, 0x76, 0x66, 0x6e, 0x6e, 0x55,
        0x57, 0x70, 0x76, 0x4e, 0x6f, 0x49, 0x68, 0x65, 0x6b, 0x34, 0x71, 0x35,
        0x75, 0x76, 0x67, 0x52, 0x77, 0x0a, 0x6a, 0x41, 0x4b, 0x4b, 0x6b, 0x47,
        0x58, 0x6f, 0x2f, 0x45, 0x74, 0x52, 0x4a, 0x38, 0x65, 0x44, 0x41, 0x67,
        0x45, 0x6c, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20,
        0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d,
        0x2d, 0x2d, 0x2d,
      ]),
      ppk: true,
      privateMAC: '9d09a15a122e48955682ba969d33c75ba8e4be2c'
    },
    what: 'Encrypted RSA key (PPK format)'
  },
  { source: [
      'PuTTY-User-Key-File-2: ssh-dss',
      'Encryption: none',
      'Comment: dsa-key-20141202',
      'Public-Lines: 10',
      'AAAAB3NzaC1kc3MAAACBAJn2I8YefRo3BsEeinQt8KQ4cEyArAs7Y/W733oRSYOI',
      'zWF1Ju124ysKrmg2okv+05CYcjV3Yp4AzQeomYAlgmB/7xCEnWaEnxCwAxmrrJMm',
      'PrkwNjHOIi7yM5QOE90IM/Q+IJA4EPBfSb+Xr8fYhrp53KNHVSnc2KkOqpo2FsIj',
      'AAAAFQC4NlP50GqyUqq2B82Vh/w5j3TzwQAAAIAeSGom9LLNdzcwCHnGfxKNnEz3',
      '55KITADTxiIpBvnQW+eDHwQvIw6V2Oc73bKCu5ZirZmIMW5w6KjQVwkuQBoF9Koq',
      '/2u6VeevtL9pD6TBzSLMVw5pV3PmE4/C/eLiaUxZLIHdbzpqPkAvAUBrXKkj0ijz',
      'cNzCp1fuF8H0pvR8yQAAAIAmvV+kqWhUgDYwNNz1qDaoS8XdsOponutZ/0stRQ66',
      'mKAy8kNVNNQ6oUx1XFl1WUt4iyFY/2Rz2fZhLz5/TbZRK5ygo666WgnxB/Ud4GAx',
      '/BPQTghOJJOL00vJk+8jVCGNDc942V6nFXznDMXwqxhRCW6dm+2lTh7ntrli8mCk',
      '5g==',
      'Private-Lines: 1',
      'AAAAFAiWSgDAM5Mq9r5dj3uPGnYvFbPb',
      'Private-MAC: c07e1d3ae080613e7cdb63ab06bde2e805bc420b'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-dss',
      type: 'dss',
      extra: undefined,
      comment: 'dsa-key-20141202',
      encryption: undefined,
      private: new Buffer([
        0x30, 0x82, 0x01, 0xba, 0x02, 0x01, 0x00, 0x02, 0x81, 0x81, 0x00, 0x99,
        0xf6, 0x23, 0xc6, 0x1e, 0x7d, 0x1a, 0x37, 0x06, 0xc1, 0x1e, 0x8a, 0x74,
        0x2d, 0xf0, 0xa4, 0x38, 0x70, 0x4c, 0x80, 0xac, 0x0b, 0x3b, 0x63, 0xf5,
        0xbb, 0xdf, 0x7a, 0x11, 0x49, 0x83, 0x88, 0xcd, 0x61, 0x75, 0x26, 0xed,
        0x76, 0xe3, 0x2b, 0x0a, 0xae, 0x68, 0x36, 0xa2, 0x4b, 0xfe, 0xd3, 0x90,
        0x98, 0x72, 0x35, 0x77, 0x62, 0x9e, 0x00, 0xcd, 0x07, 0xa8, 0x99, 0x80,
        0x25, 0x82, 0x60, 0x7f, 0xef, 0x10, 0x84, 0x9d, 0x66, 0x84, 0x9f, 0x10,
        0xb0, 0x03, 0x19, 0xab, 0xac, 0x93, 0x26, 0x3e, 0xb9, 0x30, 0x36, 0x31,
        0xce, 0x22, 0x2e, 0xf2, 0x33, 0x94, 0x0e, 0x13, 0xdd, 0x08, 0x33, 0xf4,
        0x3e, 0x20, 0x90, 0x38, 0x10, 0xf0, 0x5f, 0x49, 0xbf, 0x97, 0xaf, 0xc7,
        0xd8, 0x86, 0xba, 0x79, 0xdc, 0xa3, 0x47, 0x55, 0x29, 0xdc, 0xd8, 0xa9,
        0x0e, 0xaa, 0x9a, 0x36, 0x16, 0xc2, 0x23, 0x02, 0x15, 0x00, 0xb8, 0x36,
        0x53, 0xf9, 0xd0, 0x6a, 0xb2, 0x52, 0xaa, 0xb6, 0x07, 0xcd, 0x95, 0x87,
        0xfc, 0x39, 0x8f, 0x74, 0xf3, 0xc1, 0x02, 0x81, 0x80, 0x1e, 0x48, 0x6a,
        0x26, 0xf4, 0xb2, 0xcd, 0x77, 0x37, 0x30, 0x08, 0x79, 0xc6, 0x7f, 0x12,
        0x8d, 0x9c, 0x4c, 0xf7, 0xe7, 0x92, 0x88, 0x4c, 0x00, 0xd3, 0xc6, 0x22,
        0x29, 0x06, 0xf9, 0xd0, 0x5b, 0xe7, 0x83, 0x1f, 0x04, 0x2f, 0x23, 0x0e,
        0x95, 0xd8, 0xe7, 0x3b, 0xdd, 0xb2, 0x82, 0xbb, 0x96, 0x62, 0xad, 0x99,
        0x88, 0x31, 0x6e, 0x70, 0xe8, 0xa8, 0xd0, 0x57, 0x09, 0x2e, 0x40, 0x1a,
        0x05, 0xf4, 0xaa, 0x2a, 0xff, 0x6b, 0xba, 0x55, 0xe7, 0xaf, 0xb4, 0xbf,
        0x69, 0x0f, 0xa4, 0xc1, 0xcd, 0x22, 0xcc, 0x57, 0x0e, 0x69, 0x57, 0x73,
        0xe6, 0x13, 0x8f, 0xc2, 0xfd, 0xe2, 0xe2, 0x69, 0x4c, 0x59, 0x2c, 0x81,
        0xdd, 0x6f, 0x3a, 0x6a, 0x3e, 0x40, 0x2f, 0x01, 0x40, 0x6b, 0x5c, 0xa9,
        0x23, 0xd2, 0x28, 0xf3, 0x70, 0xdc, 0xc2, 0xa7, 0x57, 0xee, 0x17, 0xc1,
        0xf4, 0xa6, 0xf4, 0x7c, 0xc9, 0x02, 0x81, 0x80, 0x26, 0xbd, 0x5f, 0xa4,
        0xa9, 0x68, 0x54, 0x80, 0x36, 0x30, 0x34, 0xdc, 0xf5, 0xa8, 0x36, 0xa8,
        0x4b, 0xc5, 0xdd, 0xb0, 0xea, 0x68, 0x9e, 0xeb, 0x59, 0xff, 0x4b, 0x2d,
        0x45, 0x0e, 0xba, 0x98, 0xa0, 0x32, 0xf2, 0x43, 0x55, 0x34, 0xd4, 0x3a,
        0xa1, 0x4c, 0x75, 0x5c, 0x59, 0x75, 0x59, 0x4b, 0x78, 0x8b, 0x21, 0x58,
        0xff, 0x64, 0x73, 0xd9, 0xf6, 0x61, 0x2f, 0x3e, 0x7f, 0x4d, 0xb6, 0x51,
        0x2b, 0x9c, 0xa0, 0xa3, 0xae, 0xba, 0x5a, 0x09, 0xf1, 0x07, 0xf5, 0x1d,
        0xe0, 0x60, 0x31, 0xfc, 0x13, 0xd0, 0x4e, 0x08, 0x4e, 0x24, 0x93, 0x8b,
        0xd3, 0x4b, 0xc9, 0x93, 0xef, 0x23, 0x54, 0x21, 0x8d, 0x0d, 0xcf, 0x78,
        0xd9, 0x5e, 0xa7, 0x15, 0x7c, 0xe7, 0x0c, 0xc5, 0xf0, 0xab, 0x18, 0x51,
        0x09, 0x6e, 0x9d, 0x9b, 0xed, 0xa5, 0x4e, 0x1e, 0xe7, 0xb6, 0xb9, 0x62,
        0xf2, 0x60, 0xa4, 0xe6, 0x02, 0x14, 0x08, 0x96, 0x4a, 0x00, 0xc0, 0x33,
        0x93, 0x2a, 0xf6, 0xbe, 0x5d, 0x8f, 0x7b, 0x8f, 0x1a, 0x76, 0x2f, 0x15,
        0xb3, 0xdb,
      ]),
      privateOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x44,
        0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
        0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x42,
        0x75, 0x67, 0x49, 0x42, 0x41, 0x41, 0x4b, 0x42, 0x67, 0x51, 0x43, 0x5a,
        0x39, 0x69, 0x50, 0x47, 0x48, 0x6e, 0x30, 0x61, 0x4e, 0x77, 0x62, 0x42,
        0x48, 0x6f, 0x70, 0x30, 0x4c, 0x66, 0x43, 0x6b, 0x4f, 0x48, 0x42, 0x4d,
        0x67, 0x4b, 0x77, 0x4c, 0x4f, 0x32, 0x50, 0x31, 0x75, 0x39, 0x39, 0x36,
        0x45, 0x55, 0x6d, 0x44, 0x69, 0x4d, 0x31, 0x68, 0x64, 0x53, 0x62, 0x74,
        0x0a, 0x64, 0x75, 0x4d, 0x72, 0x43, 0x71, 0x35, 0x6f, 0x4e, 0x71, 0x4a,
        0x4c, 0x2f, 0x74, 0x4f, 0x51, 0x6d, 0x48, 0x49, 0x31, 0x64, 0x32, 0x4b,
        0x65, 0x41, 0x4d, 0x30, 0x48, 0x71, 0x4a, 0x6d, 0x41, 0x4a, 0x59, 0x4a,
        0x67, 0x66, 0x2b, 0x38, 0x51, 0x68, 0x4a, 0x31, 0x6d, 0x68, 0x4a, 0x38,
        0x51, 0x73, 0x41, 0x4d, 0x5a, 0x71, 0x36, 0x79, 0x54, 0x4a, 0x6a, 0x36,
        0x35, 0x4d, 0x44, 0x59, 0x78, 0x0a, 0x7a, 0x69, 0x49, 0x75, 0x38, 0x6a,
        0x4f, 0x55, 0x44, 0x68, 0x50, 0x64, 0x43, 0x44, 0x50, 0x30, 0x50, 0x69,
        0x43, 0x51, 0x4f, 0x42, 0x44, 0x77, 0x58, 0x30, 0x6d, 0x2f, 0x6c, 0x36,
        0x2f, 0x48, 0x32, 0x49, 0x61, 0x36, 0x65, 0x64, 0x79, 0x6a, 0x52, 0x31,
        0x55, 0x70, 0x33, 0x4e, 0x69, 0x70, 0x44, 0x71, 0x71, 0x61, 0x4e, 0x68,
        0x62, 0x43, 0x49, 0x77, 0x49, 0x56, 0x41, 0x4c, 0x67, 0x32, 0x0a, 0x55,
        0x2f, 0x6e, 0x51, 0x61, 0x72, 0x4a, 0x53, 0x71, 0x72, 0x59, 0x48, 0x7a,
        0x5a, 0x57, 0x48, 0x2f, 0x44, 0x6d, 0x50, 0x64, 0x50, 0x50, 0x42, 0x41,
        0x6f, 0x47, 0x41, 0x48, 0x6b, 0x68, 0x71, 0x4a, 0x76, 0x53, 0x79, 0x7a,
        0x58, 0x63, 0x33, 0x4d, 0x41, 0x68, 0x35, 0x78, 0x6e, 0x38, 0x53, 0x6a,
        0x5a, 0x78, 0x4d, 0x39, 0x2b, 0x65, 0x53, 0x69, 0x45, 0x77, 0x41, 0x30,
        0x38, 0x59, 0x69, 0x0a, 0x4b, 0x51, 0x62, 0x35, 0x30, 0x46, 0x76, 0x6e,
        0x67, 0x78, 0x38, 0x45, 0x4c, 0x79, 0x4d, 0x4f, 0x6c, 0x64, 0x6a, 0x6e,
        0x4f, 0x39, 0x32, 0x79, 0x67, 0x72, 0x75, 0x57, 0x59, 0x71, 0x32, 0x5a,
        0x69, 0x44, 0x46, 0x75, 0x63, 0x4f, 0x69, 0x6f, 0x30, 0x46, 0x63, 0x4a,
        0x4c, 0x6b, 0x41, 0x61, 0x42, 0x66, 0x53, 0x71, 0x4b, 0x76, 0x39, 0x72,
        0x75, 0x6c, 0x58, 0x6e, 0x72, 0x37, 0x53, 0x2f, 0x0a, 0x61, 0x51, 0x2b,
        0x6b, 0x77, 0x63, 0x30, 0x69, 0x7a, 0x46, 0x63, 0x4f, 0x61, 0x56, 0x64,
        0x7a, 0x35, 0x68, 0x4f, 0x50, 0x77, 0x76, 0x33, 0x69, 0x34, 0x6d, 0x6c,
        0x4d, 0x57, 0x53, 0x79, 0x42, 0x33, 0x57, 0x38, 0x36, 0x61, 0x6a, 0x35,
        0x41, 0x4c, 0x77, 0x46, 0x41, 0x61, 0x31, 0x79, 0x70, 0x49, 0x39, 0x49,
        0x6f, 0x38, 0x33, 0x44, 0x63, 0x77, 0x71, 0x64, 0x58, 0x37, 0x68, 0x66,
        0x42, 0x0a, 0x39, 0x4b, 0x62, 0x30, 0x66, 0x4d, 0x6b, 0x43, 0x67, 0x59,
        0x41, 0x6d, 0x76, 0x56, 0x2b, 0x6b, 0x71, 0x57, 0x68, 0x55, 0x67, 0x44,
        0x59, 0x77, 0x4e, 0x4e, 0x7a, 0x31, 0x71, 0x44, 0x61, 0x6f, 0x53, 0x38,
        0x58, 0x64, 0x73, 0x4f, 0x70, 0x6f, 0x6e, 0x75, 0x74, 0x5a, 0x2f, 0x30,
        0x73, 0x74, 0x52, 0x51, 0x36, 0x36, 0x6d, 0x4b, 0x41, 0x79, 0x38, 0x6b,
        0x4e, 0x56, 0x4e, 0x4e, 0x51, 0x36, 0x0a, 0x6f, 0x55, 0x78, 0x31, 0x58,
        0x46, 0x6c, 0x31, 0x57, 0x55, 0x74, 0x34, 0x69, 0x79, 0x46, 0x59, 0x2f,
        0x32, 0x52, 0x7a, 0x32, 0x66, 0x5a, 0x68, 0x4c, 0x7a, 0x35, 0x2f, 0x54,
        0x62, 0x5a, 0x52, 0x4b, 0x35, 0x79, 0x67, 0x6f, 0x36, 0x36, 0x36, 0x57,
        0x67, 0x6e, 0x78, 0x42, 0x2f, 0x55, 0x64, 0x34, 0x47, 0x41, 0x78, 0x2f,
        0x42, 0x50, 0x51, 0x54, 0x67, 0x68, 0x4f, 0x4a, 0x4a, 0x4f, 0x4c, 0x0a,
        0x30, 0x30, 0x76, 0x4a, 0x6b, 0x2b, 0x38, 0x6a, 0x56, 0x43, 0x47, 0x4e,
        0x44, 0x63, 0x39, 0x34, 0x32, 0x56, 0x36, 0x6e, 0x46, 0x58, 0x7a, 0x6e,
        0x44, 0x4d, 0x58, 0x77, 0x71, 0x78, 0x68, 0x52, 0x43, 0x57, 0x36, 0x64,
        0x6d, 0x2b, 0x32, 0x6c, 0x54, 0x68, 0x37, 0x6e, 0x74, 0x72, 0x6c, 0x69,
        0x38, 0x6d, 0x43, 0x6b, 0x35, 0x67, 0x49, 0x55, 0x43, 0x4a, 0x5a, 0x4b,
        0x41, 0x4d, 0x41, 0x7a, 0x0a, 0x6b, 0x79, 0x72, 0x32, 0x76, 0x6c, 0x32,
        0x50, 0x65, 0x34, 0x38, 0x61, 0x64, 0x69, 0x38, 0x56, 0x73, 0x39, 0x73,
        0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x44,
        0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b,
        0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d,
      ]),
      public: new Buffer([
        0x00, 0x00, 0x00, 0x07, 0x73, 0x73, 0x68, 0x2d, 0x64, 0x73, 0x73, 0x00,
        0x00, 0x00, 0x81, 0x00, 0x99, 0xf6, 0x23, 0xc6, 0x1e, 0x7d, 0x1a, 0x37,
        0x06, 0xc1, 0x1e, 0x8a, 0x74, 0x2d, 0xf0, 0xa4, 0x38, 0x70, 0x4c, 0x80,
        0xac, 0x0b, 0x3b, 0x63, 0xf5, 0xbb, 0xdf, 0x7a, 0x11, 0x49, 0x83, 0x88,
        0xcd, 0x61, 0x75, 0x26, 0xed, 0x76, 0xe3, 0x2b, 0x0a, 0xae, 0x68, 0x36,
        0xa2, 0x4b, 0xfe, 0xd3, 0x90, 0x98, 0x72, 0x35, 0x77, 0x62, 0x9e, 0x00,
        0xcd, 0x07, 0xa8, 0x99, 0x80, 0x25, 0x82, 0x60, 0x7f, 0xef, 0x10, 0x84,
        0x9d, 0x66, 0x84, 0x9f, 0x10, 0xb0, 0x03, 0x19, 0xab, 0xac, 0x93, 0x26,
        0x3e, 0xb9, 0x30, 0x36, 0x31, 0xce, 0x22, 0x2e, 0xf2, 0x33, 0x94, 0x0e,
        0x13, 0xdd, 0x08, 0x33, 0xf4, 0x3e, 0x20, 0x90, 0x38, 0x10, 0xf0, 0x5f,
        0x49, 0xbf, 0x97, 0xaf, 0xc7, 0xd8, 0x86, 0xba, 0x79, 0xdc, 0xa3, 0x47,
        0x55, 0x29, 0xdc, 0xd8, 0xa9, 0x0e, 0xaa, 0x9a, 0x36, 0x16, 0xc2, 0x23,
        0x00, 0x00, 0x00, 0x15, 0x00, 0xb8, 0x36, 0x53, 0xf9, 0xd0, 0x6a, 0xb2,
        0x52, 0xaa, 0xb6, 0x07, 0xcd, 0x95, 0x87, 0xfc, 0x39, 0x8f, 0x74, 0xf3,
        0xc1, 0x00, 0x00, 0x00, 0x80, 0x1e, 0x48, 0x6a, 0x26, 0xf4, 0xb2, 0xcd,
        0x77, 0x37, 0x30, 0x08, 0x79, 0xc6, 0x7f, 0x12, 0x8d, 0x9c, 0x4c, 0xf7,
        0xe7, 0x92, 0x88, 0x4c, 0x00, 0xd3, 0xc6, 0x22, 0x29, 0x06, 0xf9, 0xd0,
        0x5b, 0xe7, 0x83, 0x1f, 0x04, 0x2f, 0x23, 0x0e, 0x95, 0xd8, 0xe7, 0x3b,
        0xdd, 0xb2, 0x82, 0xbb, 0x96, 0x62, 0xad, 0x99, 0x88, 0x31, 0x6e, 0x70,
        0xe8, 0xa8, 0xd0, 0x57, 0x09, 0x2e, 0x40, 0x1a, 0x05, 0xf4, 0xaa, 0x2a,
        0xff, 0x6b, 0xba, 0x55, 0xe7, 0xaf, 0xb4, 0xbf, 0x69, 0x0f, 0xa4, 0xc1,
        0xcd, 0x22, 0xcc, 0x57, 0x0e, 0x69, 0x57, 0x73, 0xe6, 0x13, 0x8f, 0xc2,
        0xfd, 0xe2, 0xe2, 0x69, 0x4c, 0x59, 0x2c, 0x81, 0xdd, 0x6f, 0x3a, 0x6a,
        0x3e, 0x40, 0x2f, 0x01, 0x40, 0x6b, 0x5c, 0xa9, 0x23, 0xd2, 0x28, 0xf3,
        0x70, 0xdc, 0xc2, 0xa7, 0x57, 0xee, 0x17, 0xc1, 0xf4, 0xa6, 0xf4, 0x7c,
        0xc9, 0x00, 0x00, 0x00, 0x80, 0x26, 0xbd, 0x5f, 0xa4, 0xa9, 0x68, 0x54,
        0x80, 0x36, 0x30, 0x34, 0xdc, 0xf5, 0xa8, 0x36, 0xa8, 0x4b, 0xc5, 0xdd,
        0xb0, 0xea, 0x68, 0x9e, 0xeb, 0x59, 0xff, 0x4b, 0x2d, 0x45, 0x0e, 0xba,
        0x98, 0xa0, 0x32, 0xf2, 0x43, 0x55, 0x34, 0xd4, 0x3a, 0xa1, 0x4c, 0x75,
        0x5c, 0x59, 0x75, 0x59, 0x4b, 0x78, 0x8b, 0x21, 0x58, 0xff, 0x64, 0x73,
        0xd9, 0xf6, 0x61, 0x2f, 0x3e, 0x7f, 0x4d, 0xb6, 0x51, 0x2b, 0x9c, 0xa0,
        0xa3, 0xae, 0xba, 0x5a, 0x09, 0xf1, 0x07, 0xf5, 0x1d, 0xe0, 0x60, 0x31,
        0xfc, 0x13, 0xd0, 0x4e, 0x08, 0x4e, 0x24, 0x93, 0x8b, 0xd3, 0x4b, 0xc9,
        0x93, 0xef, 0x23, 0x54, 0x21, 0x8d, 0x0d, 0xcf, 0x78, 0xd9, 0x5e, 0xa7,
        0x15, 0x7c, 0xe7, 0x0c, 0xc5, 0xf0, 0xab, 0x18, 0x51, 0x09, 0x6e, 0x9d,
        0x9b, 0xed, 0xa5, 0x4e, 0x1e, 0xe7, 0xb6, 0xb9, 0x62, 0xf2, 0x60, 0xa4,
        0xe6,
      ]),
      publicOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x50,
        0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d,
        0x2d, 0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x42, 0x74, 0x6a, 0x43, 0x43, 0x41,
        0x53, 0x73, 0x47, 0x42, 0x79, 0x71, 0x47, 0x53, 0x4d, 0x34, 0x34, 0x42,
        0x41, 0x45, 0x77, 0x67, 0x67, 0x45, 0x65, 0x41, 0x6f, 0x47, 0x42, 0x41,
        0x4a, 0x6e, 0x32, 0x49, 0x38, 0x59, 0x65, 0x66, 0x52, 0x6f, 0x33, 0x42,
        0x73, 0x45, 0x65, 0x69, 0x6e, 0x51, 0x74, 0x38, 0x4b, 0x51, 0x34, 0x63,
        0x45, 0x79, 0x41, 0x72, 0x41, 0x73, 0x37, 0x0a, 0x59, 0x2f, 0x57, 0x37,
        0x33, 0x33, 0x6f, 0x52, 0x53, 0x59, 0x4f, 0x49, 0x7a, 0x57, 0x46, 0x31,
        0x4a, 0x75, 0x31, 0x32, 0x34, 0x79, 0x73, 0x4b, 0x72, 0x6d, 0x67, 0x32,
        0x6f, 0x6b, 0x76, 0x2b, 0x30, 0x35, 0x43, 0x59, 0x63, 0x6a, 0x56, 0x33,
        0x59, 0x70, 0x34, 0x41, 0x7a, 0x51, 0x65, 0x6f, 0x6d, 0x59, 0x41, 0x6c,
        0x67, 0x6d, 0x42, 0x2f, 0x37, 0x78, 0x43, 0x45, 0x6e, 0x57, 0x61, 0x45,
        0x0a, 0x6e, 0x78, 0x43, 0x77, 0x41, 0x78, 0x6d, 0x72, 0x72, 0x4a, 0x4d,
        0x6d, 0x50, 0x72, 0x6b, 0x77, 0x4e, 0x6a, 0x48, 0x4f, 0x49, 0x69, 0x37,
        0x79, 0x4d, 0x35, 0x51, 0x4f, 0x45, 0x39, 0x30, 0x49, 0x4d, 0x2f, 0x51,
        0x2b, 0x49, 0x4a, 0x41, 0x34, 0x45, 0x50, 0x42, 0x66, 0x53, 0x62, 0x2b,
        0x58, 0x72, 0x38, 0x66, 0x59, 0x68, 0x72, 0x70, 0x35, 0x33, 0x4b, 0x4e,
        0x48, 0x56, 0x53, 0x6e, 0x63, 0x0a, 0x32, 0x4b, 0x6b, 0x4f, 0x71, 0x70,
        0x6f, 0x32, 0x46, 0x73, 0x49, 0x6a, 0x41, 0x68, 0x55, 0x41, 0x75, 0x44,
        0x5a, 0x54, 0x2b, 0x64, 0x42, 0x71, 0x73, 0x6c, 0x4b, 0x71, 0x74, 0x67,
        0x66, 0x4e, 0x6c, 0x59, 0x66, 0x38, 0x4f, 0x59, 0x39, 0x30, 0x38, 0x38,
        0x45, 0x43, 0x67, 0x59, 0x41, 0x65, 0x53, 0x47, 0x6f, 0x6d, 0x39, 0x4c,
        0x4c, 0x4e, 0x64, 0x7a, 0x63, 0x77, 0x43, 0x48, 0x6e, 0x47, 0x0a, 0x66,
        0x78, 0x4b, 0x4e, 0x6e, 0x45, 0x7a, 0x33, 0x35, 0x35, 0x4b, 0x49, 0x54,
        0x41, 0x44, 0x54, 0x78, 0x69, 0x49, 0x70, 0x42, 0x76, 0x6e, 0x51, 0x57,
        0x2b, 0x65, 0x44, 0x48, 0x77, 0x51, 0x76, 0x49, 0x77, 0x36, 0x56, 0x32,
        0x4f, 0x63, 0x37, 0x33, 0x62, 0x4b, 0x43, 0x75, 0x35, 0x5a, 0x69, 0x72,
        0x5a, 0x6d, 0x49, 0x4d, 0x57, 0x35, 0x77, 0x36, 0x4b, 0x6a, 0x51, 0x56,
        0x77, 0x6b, 0x75, 0x0a, 0x51, 0x42, 0x6f, 0x46, 0x39, 0x4b, 0x6f, 0x71,
        0x2f, 0x32, 0x75, 0x36, 0x56, 0x65, 0x65, 0x76, 0x74, 0x4c, 0x39, 0x70,
        0x44, 0x36, 0x54, 0x42, 0x7a, 0x53, 0x4c, 0x4d, 0x56, 0x77, 0x35, 0x70,
        0x56, 0x33, 0x50, 0x6d, 0x45, 0x34, 0x2f, 0x43, 0x2f, 0x65, 0x4c, 0x69,
        0x61, 0x55, 0x78, 0x5a, 0x4c, 0x49, 0x48, 0x64, 0x62, 0x7a, 0x70, 0x71,
        0x50, 0x6b, 0x41, 0x76, 0x41, 0x55, 0x42, 0x72, 0x0a, 0x58, 0x4b, 0x6b,
        0x6a, 0x30, 0x69, 0x6a, 0x7a, 0x63, 0x4e, 0x7a, 0x43, 0x70, 0x31, 0x66,
        0x75, 0x46, 0x38, 0x48, 0x30, 0x70, 0x76, 0x52, 0x38, 0x79, 0x51, 0x4f,
        0x42, 0x68, 0x41, 0x41, 0x43, 0x67, 0x59, 0x41, 0x6d, 0x76, 0x56, 0x2b,
        0x6b, 0x71, 0x57, 0x68, 0x55, 0x67, 0x44, 0x59, 0x77, 0x4e, 0x4e, 0x7a,
        0x31, 0x71, 0x44, 0x61, 0x6f, 0x53, 0x38, 0x58, 0x64, 0x73, 0x4f, 0x70,
        0x6f, 0x0a, 0x6e, 0x75, 0x74, 0x5a, 0x2f, 0x30, 0x73, 0x74, 0x52, 0x51,
        0x36, 0x36, 0x6d, 0x4b, 0x41, 0x79, 0x38, 0x6b, 0x4e, 0x56, 0x4e, 0x4e,
        0x51, 0x36, 0x6f, 0x55, 0x78, 0x31, 0x58, 0x46, 0x6c, 0x31, 0x57, 0x55,
        0x74, 0x34, 0x69, 0x79, 0x46, 0x59, 0x2f, 0x32, 0x52, 0x7a, 0x32, 0x66,
        0x5a, 0x68, 0x4c, 0x7a, 0x35, 0x2f, 0x54, 0x62, 0x5a, 0x52, 0x4b, 0x35,
        0x79, 0x67, 0x6f, 0x36, 0x36, 0x36, 0x0a, 0x57, 0x67, 0x6e, 0x78, 0x42,
        0x2f, 0x55, 0x64, 0x34, 0x47, 0x41, 0x78, 0x2f, 0x42, 0x50, 0x51, 0x54,
        0x67, 0x68, 0x4f, 0x4a, 0x4a, 0x4f, 0x4c, 0x30, 0x30, 0x76, 0x4a, 0x6b,
        0x2b, 0x38, 0x6a, 0x56, 0x43, 0x47, 0x4e, 0x44, 0x63, 0x39, 0x34, 0x32,
        0x56, 0x36, 0x6e, 0x46, 0x58, 0x7a, 0x6e, 0x44, 0x4d, 0x58, 0x77, 0x71,
        0x78, 0x68, 0x52, 0x43, 0x57, 0x36, 0x64, 0x6d, 0x2b, 0x32, 0x6c, 0x0a,
        0x54, 0x68, 0x37, 0x6e, 0x74, 0x72, 0x6c, 0x69, 0x38, 0x6d, 0x43, 0x6b,
        0x35, 0x67, 0x3d, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e,
        0x44, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b, 0x45, 0x59,
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d,
      ]),
      ppk: true,
      privateMAC: 'c07e1d3ae080613e7cdb63ab06bde2e805bc420b',
      _converted: true,
      _macresult: true
    },
    what: 'Unencrypted DSA key (PPK format)'
  },
  { source: [
      'PuTTY-User-Key-File-2: ssh-dss',
      'Encryption: aes256-cbc',
      'Comment: dsa-key-20141202',
      'Public-Lines: 10',
      'AAAAB3NzaC1kc3MAAACBAJn2I8YefRo3BsEeinQt8KQ4cEyArAs7Y/W733oRSYOI',
      'zWF1Ju124ysKrmg2okv+05CYcjV3Yp4AzQeomYAlgmB/7xCEnWaEnxCwAxmrrJMm',
      'PrkwNjHOIi7yM5QOE90IM/Q+IJA4EPBfSb+Xr8fYhrp53KNHVSnc2KkOqpo2FsIj',
      'AAAAFQC4NlP50GqyUqq2B82Vh/w5j3TzwQAAAIAeSGom9LLNdzcwCHnGfxKNnEz3',
      '55KITADTxiIpBvnQW+eDHwQvIw6V2Oc73bKCu5ZirZmIMW5w6KjQVwkuQBoF9Koq',
      '/2u6VeevtL9pD6TBzSLMVw5pV3PmE4/C/eLiaUxZLIHdbzpqPkAvAUBrXKkj0ijz',
      'cNzCp1fuF8H0pvR8yQAAAIAmvV+kqWhUgDYwNNz1qDaoS8XdsOponutZ/0stRQ66',
      'mKAy8kNVNNQ6oUx1XFl1WUt4iyFY/2Rz2fZhLz5/TbZRK5ygo666WgnxB/Ud4GAx',
      '/BPQTghOJJOL00vJk+8jVCGNDc942V6nFXznDMXwqxhRCW6dm+2lTh7ntrli8mCk',
      '5g==',
      'Private-Lines: 1',
      'BytvbK+jNyMjiVxCO5lcE4YbW7q293oC+LZjkZ8Ajlw=',
      'Private-MAC: c3da536ea28851fc32d5d1ff01498c8fcebc1170'
    ].join('\n'),
    expected: {
      fulltype: 'ssh-dss',
      type: 'dss',
      extra: undefined,
      comment: 'dsa-key-20141202',
      encryption: 'aes256-cbc',
      private: new Buffer([
        0x07, 0x2b, 0x6f, 0x6c, 0xaf, 0xa3, 0x37, 0x23, 0x23, 0x89, 0x5c, 0x42,
        0x3b, 0x99, 0x5c, 0x13, 0x86, 0x1b, 0x5b, 0xba, 0xb6, 0xf7, 0x7a, 0x02,
        0xf8, 0xb6, 0x63, 0x91, 0x9f, 0x00, 0x8e, 0x5c,
      ]),
      privateOrig: undefined,
      public: new Buffer([
        0x00, 0x00, 0x00, 0x07, 0x73, 0x73, 0x68, 0x2d, 0x64, 0x73, 0x73, 0x00,
        0x00, 0x00, 0x81, 0x00, 0x99, 0xf6, 0x23, 0xc6, 0x1e, 0x7d, 0x1a, 0x37,
        0x06, 0xc1, 0x1e, 0x8a, 0x74, 0x2d, 0xf0, 0xa4, 0x38, 0x70, 0x4c, 0x80,
        0xac, 0x0b, 0x3b, 0x63, 0xf5, 0xbb, 0xdf, 0x7a, 0x11, 0x49, 0x83, 0x88,
        0xcd, 0x61, 0x75, 0x26, 0xed, 0x76, 0xe3, 0x2b, 0x0a, 0xae, 0x68, 0x36,
        0xa2, 0x4b, 0xfe, 0xd3, 0x90, 0x98, 0x72, 0x35, 0x77, 0x62, 0x9e, 0x00,
        0xcd, 0x07, 0xa8, 0x99, 0x80, 0x25, 0x82, 0x60, 0x7f, 0xef, 0x10, 0x84,
        0x9d, 0x66, 0x84, 0x9f, 0x10, 0xb0, 0x03, 0x19, 0xab, 0xac, 0x93, 0x26,
        0x3e, 0xb9, 0x30, 0x36, 0x31, 0xce, 0x22, 0x2e, 0xf2, 0x33, 0x94, 0x0e,
        0x13, 0xdd, 0x08, 0x33, 0xf4, 0x3e, 0x20, 0x90, 0x38, 0x10, 0xf0, 0x5f,
        0x49, 0xbf, 0x97, 0xaf, 0xc7, 0xd8, 0x86, 0xba, 0x79, 0xdc, 0xa3, 0x47,
        0x55, 0x29, 0xdc, 0xd8, 0xa9, 0x0e, 0xaa, 0x9a, 0x36, 0x16, 0xc2, 0x23,
        0x00, 0x00, 0x00, 0x15, 0x00, 0xb8, 0x36, 0x53, 0xf9, 0xd0, 0x6a, 0xb2,
        0x52, 0xaa, 0xb6, 0x07, 0xcd, 0x95, 0x87, 0xfc, 0x39, 0x8f, 0x74, 0xf3,
        0xc1, 0x00, 0x00, 0x00, 0x80, 0x1e, 0x48, 0x6a, 0x26, 0xf4, 0xb2, 0xcd,
        0x77, 0x37, 0x30, 0x08, 0x79, 0xc6, 0x7f, 0x12, 0x8d, 0x9c, 0x4c, 0xf7,
        0xe7, 0x92, 0x88, 0x4c, 0x00, 0xd3, 0xc6, 0x22, 0x29, 0x06, 0xf9, 0xd0,
        0x5b, 0xe7, 0x83, 0x1f, 0x04, 0x2f, 0x23, 0x0e, 0x95, 0xd8, 0xe7, 0x3b,
        0xdd, 0xb2, 0x82, 0xbb, 0x96, 0x62, 0xad, 0x99, 0x88, 0x31, 0x6e, 0x70,
        0xe8, 0xa8, 0xd0, 0x57, 0x09, 0x2e, 0x40, 0x1a, 0x05, 0xf4, 0xaa, 0x2a,
        0xff, 0x6b, 0xba, 0x55, 0xe7, 0xaf, 0xb4, 0xbf, 0x69, 0x0f, 0xa4, 0xc1,
        0xcd, 0x22, 0xcc, 0x57, 0x0e, 0x69, 0x57, 0x73, 0xe6, 0x13, 0x8f, 0xc2,
        0xfd, 0xe2, 0xe2, 0x69, 0x4c, 0x59, 0x2c, 0x81, 0xdd, 0x6f, 0x3a, 0x6a,
        0x3e, 0x40, 0x2f, 0x01, 0x40, 0x6b, 0x5c, 0xa9, 0x23, 0xd2, 0x28, 0xf3,
        0x70, 0xdc, 0xc2, 0xa7, 0x57, 0xee, 0x17, 0xc1, 0xf4, 0xa6, 0xf4, 0x7c,
        0xc9, 0x00, 0x00, 0x00, 0x80, 0x26, 0xbd, 0x5f, 0xa4, 0xa9, 0x68, 0x54,
        0x80, 0x36, 0x30, 0x34, 0xdc, 0xf5, 0xa8, 0x36, 0xa8, 0x4b, 0xc5, 0xdd,
        0xb0, 0xea, 0x68, 0x9e, 0xeb, 0x59, 0xff, 0x4b, 0x2d, 0x45, 0x0e, 0xba,
        0x98, 0xa0, 0x32, 0xf2, 0x43, 0x55, 0x34, 0xd4, 0x3a, 0xa1, 0x4c, 0x75,
        0x5c, 0x59, 0x75, 0x59, 0x4b, 0x78, 0x8b, 0x21, 0x58, 0xff, 0x64, 0x73,
        0xd9, 0xf6, 0x61, 0x2f, 0x3e, 0x7f, 0x4d, 0xb6, 0x51, 0x2b, 0x9c, 0xa0,
        0xa3, 0xae, 0xba, 0x5a, 0x09, 0xf1, 0x07, 0xf5, 0x1d, 0xe0, 0x60, 0x31,
        0xfc, 0x13, 0xd0, 0x4e, 0x08, 0x4e, 0x24, 0x93, 0x8b, 0xd3, 0x4b, 0xc9,
        0x93, 0xef, 0x23, 0x54, 0x21, 0x8d, 0x0d, 0xcf, 0x78, 0xd9, 0x5e, 0xa7,
        0x15, 0x7c, 0xe7, 0x0c, 0xc5, 0xf0, 0xab, 0x18, 0x51, 0x09, 0x6e, 0x9d,
        0x9b, 0xed, 0xa5, 0x4e, 0x1e, 0xe7, 0xb6, 0xb9, 0x62, 0xf2, 0x60, 0xa4,
        0xe6,
      ]),
      publicOrig: new Buffer([
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x50,
        0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d,
        0x2d, 0x2d, 0x0a, 0x4d, 0x49, 0x49, 0x42, 0x74, 0x6a, 0x43, 0x43, 0x41,
        0x53, 0x73, 0x47, 0x42, 0x79, 0x71, 0x47, 0x53, 0x4d, 0x34, 0x34, 0x42,
        0x41, 0x45, 0x77, 0x67, 0x67, 0x45, 0x65, 0x41, 0x6f, 0x47, 0x42, 0x41,
        0x4a, 0x6e, 0x32, 0x49, 0x38, 0x59, 0x65, 0x66, 0x52, 0x6f, 0x33, 0x42,
        0x73, 0x45, 0x65, 0x69, 0x6e, 0x51, 0x74, 0x38, 0x4b, 0x51, 0x34, 0x63,
        0x45, 0x79, 0x41, 0x72, 0x41, 0x73, 0x37, 0x0a, 0x59, 0x2f, 0x57, 0x37,
        0x33, 0x33, 0x6f, 0x52, 0x53, 0x59, 0x4f, 0x49, 0x7a, 0x57, 0x46, 0x31,
        0x4a, 0x75, 0x31, 0x32, 0x34, 0x79, 0x73, 0x4b, 0x72, 0x6d, 0x67, 0x32,
        0x6f, 0x6b, 0x76, 0x2b, 0x30, 0x35, 0x43, 0x59, 0x63, 0x6a, 0x56, 0x33,
        0x59, 0x70, 0x34, 0x41, 0x7a, 0x51, 0x65, 0x6f, 0x6d, 0x59, 0x41, 0x6c,
        0x67, 0x6d, 0x42, 0x2f, 0x37, 0x78, 0x43, 0x45, 0x6e, 0x57, 0x61, 0x45,
        0x0a, 0x6e, 0x78, 0x43, 0x77, 0x41, 0x78, 0x6d, 0x72, 0x72, 0x4a, 0x4d,
        0x6d, 0x50, 0x72, 0x6b, 0x77, 0x4e, 0x6a, 0x48, 0x4f, 0x49, 0x69, 0x37,
        0x79, 0x4d, 0x35, 0x51, 0x4f, 0x45, 0x39, 0x30, 0x49, 0x4d, 0x2f, 0x51,
        0x2b, 0x49, 0x4a, 0x41, 0x34, 0x45, 0x50, 0x42, 0x66, 0x53, 0x62, 0x2b,
        0x58, 0x72, 0x38, 0x66, 0x59, 0x68, 0x72, 0x70, 0x35, 0x33, 0x4b, 0x4e,
        0x48, 0x56, 0x53, 0x6e, 0x63, 0x0a, 0x32, 0x4b, 0x6b, 0x4f, 0x71, 0x70,
        0x6f, 0x32, 0x46, 0x73, 0x49, 0x6a, 0x41, 0x68, 0x55, 0x41, 0x75, 0x44,
        0x5a, 0x54, 0x2b, 0x64, 0x42, 0x71, 0x73, 0x6c, 0x4b, 0x71, 0x74, 0x67,
        0x66, 0x4e, 0x6c, 0x59, 0x66, 0x38, 0x4f, 0x59, 0x39, 0x30, 0x38, 0x38,
        0x45, 0x43, 0x67, 0x59, 0x41, 0x65, 0x53, 0x47, 0x6f, 0x6d, 0x39, 0x4c,
        0x4c, 0x4e, 0x64, 0x7a, 0x63, 0x77, 0x43, 0x48, 0x6e, 0x47, 0x0a, 0x66,
        0x78, 0x4b, 0x4e, 0x6e, 0x45, 0x7a, 0x33, 0x35, 0x35, 0x4b, 0x49, 0x54,
        0x41, 0x44, 0x54, 0x78, 0x69, 0x49, 0x70, 0x42, 0x76, 0x6e, 0x51, 0x57,
        0x2b, 0x65, 0x44, 0x48, 0x77, 0x51, 0x76, 0x49, 0x77, 0x36, 0x56, 0x32,
        0x4f, 0x63, 0x37, 0x33, 0x62, 0x4b, 0x43, 0x75, 0x35, 0x5a, 0x69, 0x72,
        0x5a, 0x6d, 0x49, 0x4d, 0x57, 0x35, 0x77, 0x36, 0x4b, 0x6a, 0x51, 0x56,
        0x77, 0x6b, 0x75, 0x0a, 0x51, 0x42, 0x6f, 0x46, 0x39, 0x4b, 0x6f, 0x71,
        0x2f, 0x32, 0x75, 0x36, 0x56, 0x65, 0x65, 0x76, 0x74, 0x4c, 0x39, 0x70,
        0x44, 0x36, 0x54, 0x42, 0x7a, 0x53, 0x4c, 0x4d, 0x56, 0x77, 0x35, 0x70,
        0x56, 0x33, 0x50, 0x6d, 0x45, 0x34, 0x2f, 0x43, 0x2f, 0x65, 0x4c, 0x69,
        0x61, 0x55, 0x78, 0x5a, 0x4c, 0x49, 0x48, 0x64, 0x62, 0x7a, 0x70, 0x71,
        0x50, 0x6b, 0x41, 0x76, 0x41, 0x55, 0x42, 0x72, 0x0a, 0x58, 0x4b, 0x6b,
        0x6a, 0x30, 0x69, 0x6a, 0x7a, 0x63, 0x4e, 0x7a, 0x43, 0x70, 0x31, 0x66,
        0x75, 0x46, 0x38, 0x48, 0x30, 0x70, 0x76, 0x52, 0x38, 0x79, 0x51, 0x4f,
        0x42, 0x68, 0x41, 0x41, 0x43, 0x67, 0x59, 0x41, 0x6d, 0x76, 0x56, 0x2b,
        0x6b, 0x71, 0x57, 0x68, 0x55, 0x67, 0x44, 0x59, 0x77, 0x4e, 0x4e, 0x7a,
        0x31, 0x71, 0x44, 0x61, 0x6f, 0x53, 0x38, 0x58, 0x64, 0x73, 0x4f, 0x70,
        0x6f, 0x0a, 0x6e, 0x75, 0x74, 0x5a, 0x2f, 0x30, 0x73, 0x74, 0x52, 0x51,
        0x36, 0x36, 0x6d, 0x4b, 0x41, 0x79, 0x38, 0x6b, 0x4e, 0x56, 0x4e, 0x4e,
        0x51, 0x36, 0x6f, 0x55, 0x78, 0x31, 0x58, 0x46, 0x6c, 0x31, 0x57, 0x55,
        0x74, 0x34, 0x69, 0x79, 0x46, 0x59, 0x2f, 0x32, 0x52, 0x7a, 0x32, 0x66,
        0x5a, 0x68, 0x4c, 0x7a, 0x35, 0x2f, 0x54, 0x62, 0x5a, 0x52, 0x4b, 0x35,
        0x79, 0x67, 0x6f, 0x36, 0x36, 0x36, 0x0a, 0x57, 0x67, 0x6e, 0x78, 0x42,
        0x2f, 0x55, 0x64, 0x34, 0x47, 0x41, 0x78, 0x2f, 0x42, 0x50, 0x51, 0x54,
        0x67, 0x68, 0x4f, 0x4a, 0x4a, 0x4f, 0x4c, 0x30, 0x30, 0x76, 0x4a, 0x6b,
        0x2b, 0x38, 0x6a, 0x56, 0x43, 0x47, 0x4e, 0x44, 0x63, 0x39, 0x34, 0x32,
        0x56, 0x36, 0x6e, 0x46, 0x58, 0x7a, 0x6e, 0x44, 0x4d, 0x58, 0x77, 0x71,
        0x78, 0x68, 0x52, 0x43, 0x57, 0x36, 0x64, 0x6d, 0x2b, 0x32, 0x6c, 0x0a,
        0x54, 0x68, 0x37, 0x6e, 0x74, 0x72, 0x6c, 0x69, 0x38, 0x6d, 0x43, 0x6b,
        0x35, 0x67, 0x3d, 0x3d, 0x0a, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e,
        0x44, 0x20, 0x50, 0x55, 0x42, 0x4c, 0x49, 0x43, 0x20, 0x4b, 0x45, 0x59,
        0x2d, 0x2d, 0x2d, 0x2d, 0x2d,
      ]),
      ppk: true,
      privateMAC: 'c3da536ea28851fc32d5d1ff01498c8fcebc1170'
    },
    what: 'Encrypted DSA key (PPK format)'
  },
].forEach(function(v) {
  require('buffer').INSPECT_MAX_BYTES = Infinity;
  var result;
  try {
    result = parser(v.source);
  } catch (ex) {
    console.log('Exception encountered for: ' + v.what + '\n' + ex.stack);
    process.exit(1);
  }
  /*if (result.private) {
    for (var i = 0; i < result.private.length; ++i) {
      if (i % 12 === 0)
        process.stdout.write('\n');
      process.stdout.write('0x' + (result.private[i] < 16 ? '0' : '') + result.private[i].toString(16) + ', ');
    }
  }
  console.log('');*/
  assert.deepEqual(result,
                   v.expected,
                   '[' + group + v.what + ']: parser output mismatch.'
                   + '\nSaw: ' + inspect(result)
                   + '\nExpected: ' + inspect(v.expected));
});
