"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const source_map_1 = require("source-map");
// Matches <%= expr %>. This does not support structural JavaScript (for/if/...).
const kInterpolateRe = /<%=([\s\S]+?)%>/g;
// Matches <%# text %>. It's a comment and will be entirely ignored.
const kCommentRe = /<%#([\s\S]+?)%>/g;
// Used to match template delimiters.
// <%- expr %>: HTML escape the value.
// <% ... %>: Structural template code.
const kEscapeRe = /<%-([\s\S]+?)%>/g;
const kEvaluateRe = /<%([\s\S]+?)%>/g;
/** Used to map characters to HTML entities. */
const kHtmlEscapes = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '&quot;',
    "'": '&#39;',
    '`': '&#96;',
};
// Used to match HTML entities and HTML characters.
const reUnescapedHtml = new RegExp(`[${Object.keys(kHtmlEscapes).join('')}]`, 'g');
function _positionFor(content, offset) {
    let line = 1;
    let column = 0;
    for (let i = 0; i < offset - 1; i++) {
        if (content[i] == '\n') {
            line++;
            column = 0;
        }
        else {
            column++;
        }
    }
    return {
        line,
        column,
    };
}
/**
 * Given a source text (and a fileName), returns a TemplateAst.
 */
function templateParser(sourceText, fileName) {
    const children = [];
    // Compile the regexp to match each delimiter.
    const reExpressions = [kEscapeRe, kCommentRe, kInterpolateRe, kEvaluateRe];
    const reDelimiters = RegExp(reExpressions.map(x => x.source).join('|') + '|$', 'g');
    const parsed = sourceText.split(reDelimiters);
    let offset = 0;
    // Optimization that uses the fact that the end of a node is always the beginning of the next
    // node, so we keep the positioning of the nodes in memory.
    let start = _positionFor(sourceText, offset);
    let end;
    const increment = reExpressions.length + 1;
    for (let i = 0; i < parsed.length; i += increment) {
        const [content, escape, comment, interpolate, evaluate] = parsed.slice(i, i + increment);
        if (content) {
            end = _positionFor(sourceText, offset + content.length);
            offset += content.length;
            children.push({ kind: 'content', content, start, end });
            start = end;
        }
        if (escape) {
            end = _positionFor(sourceText, offset + escape.length + 5);
            offset += escape.length + 5;
            children.push({ kind: 'escape', expression: escape, start, end });
            start = end;
        }
        if (comment) {
            end = _positionFor(sourceText, offset + comment.length + 5);
            offset += comment.length + 5;
            children.push({ kind: 'comment', text: comment, start, end });
            start = end;
        }
        if (interpolate) {
            end = _positionFor(sourceText, offset + interpolate.length + 5);
            offset += interpolate.length + 5;
            children.push({
                kind: 'interpolate',
                expression: interpolate,
                start,
                end,
            });
            start = end;
        }
        if (evaluate) {
            end = _positionFor(sourceText, offset + evaluate.length + 5);
            offset += evaluate.length + 5;
            children.push({ kind: 'evaluate', expression: evaluate, start, end });
            start = end;
        }
    }
    return {
        fileName,
        content: sourceText,
        children,
    };
}
exports.templateParser = templateParser;
/**
 * Fastest implementation of the templating algorithm. It only add strings and does not bother
 * with source maps.
 */
function templateFast(ast, options) {
    const module = options && options.module ? 'module.exports.default =' : '';
    const reHtmlEscape = reUnescapedHtml.source.replace(/[']/g, '\\\\\\\'');
    return `
    return ${module} function(obj) {
      obj || (obj = {});
      let __t;
      let __p = '';
      const __escapes = ${JSON.stringify(kHtmlEscapes)};
      const __escapesre = new RegExp('${reHtmlEscape}', 'g');

      const __e = function(s) {
        return s ? s.replace(__escapesre, function(key) { return __escapes[key]; }) : '';
      };
      with (obj) {
        ${ast.children.map(node => {
        switch (node.kind) {
            case 'content':
                return `__p += ${JSON.stringify(node.content)};`;
            case 'interpolate':
                return `__p += ((__t = (${node.expression})) == null) ? '' : __t;`;
            case 'escape':
                return `__p += __e(${node.expression});`;
            case 'evaluate':
                return node.expression;
        }
    }).join('\n')}
      }

      return __p;
    };
  `;
}
/**
 * Templating algorithm with source map support. The map is outputted as //# sourceMapUrl=...
 */
function templateWithSourceMap(ast, options) {
    const sourceUrl = ast.fileName;
    const module = options && options.module ? 'module.exports.default =' : '';
    const reHtmlEscape = reUnescapedHtml.source.replace(/[']/g, '\\\\\\\'');
    const preamble = (new source_map_1.SourceNode(1, 0, sourceUrl, ''))
        .add(new source_map_1.SourceNode(1, 0, sourceUrl, [
        `return ${module} function(obj) {\n`,
        '  obj || (obj = {});\n',
        '  let __t;\n',
        '  let __p = "";\n',
        `  const __escapes = ${JSON.stringify(kHtmlEscapes)};\n`,
        `  const __escapesre = new RegExp('${reHtmlEscape}', 'g');\n`,
        `\n`,
        `  const __e = function(s) { `,
        `    return s ? s.replace(__escapesre, function(key) { return __escapes[key]; }) : '';`,
        `  };\n`,
        `  with (obj) {\n`,
    ]));
    const end = ast.children.length
        ? ast.children[ast.children.length - 1].end
        : { line: 0, column: 0 };
    const nodes = ast.children.reduce((chunk, node) => {
        let code = '';
        switch (node.kind) {
            case 'content':
                code = [
                    new source_map_1.SourceNode(node.start.line, node.start.column, sourceUrl, '__p = __p'),
                    ...node.content.split('\n').map((line, i, arr) => {
                        return new source_map_1.SourceNode(node.start.line + i, i == 0 ? node.start.column : 0, sourceUrl, '\n    + '
                            + JSON.stringify(line + (i == arr.length - 1 ? '' : '\n')));
                    }),
                    new source_map_1.SourceNode(node.end.line, node.end.column, sourceUrl, ';\n'),
                ];
                break;
            case 'interpolate':
                code = [
                    new source_map_1.SourceNode(node.start.line, node.start.column, sourceUrl, '__p += ((__t = '),
                    ...node.expression.split('\n').map((line, i, arr) => {
                        return new source_map_1.SourceNode(node.start.line + i, i == 0 ? node.start.column : 0, sourceUrl, line + ((i == arr.length - 1) ? '' : '\n'));
                    }),
                    new source_map_1.SourceNode(node.end.line, node.end.column, sourceUrl, ') == null ? "" : __t);\n'),
                ];
                break;
            case 'escape':
                code = [
                    new source_map_1.SourceNode(node.start.line, node.start.column, sourceUrl, '__p += __e('),
                    ...node.expression.split('\n').map((line, i, arr) => {
                        return new source_map_1.SourceNode(node.start.line + i, i == 0 ? node.start.column : 0, sourceUrl, line + ((i == arr.length - 1) ? '' : '\n'));
                    }),
                    new source_map_1.SourceNode(node.end.line, node.end.column, sourceUrl, ');\n'),
                ];
                break;
            case 'evaluate':
                code = [
                    ...node.expression.split('\n').map((line, i, arr) => {
                        return new source_map_1.SourceNode(node.start.line + i, i == 0 ? node.start.column : 0, sourceUrl, line + ((i == arr.length - 1) ? '' : '\n'));
                    }),
                    new source_map_1.SourceNode(node.end.line, node.end.column, sourceUrl, '\n'),
                ];
                break;
        }
        return chunk.add(new source_map_1.SourceNode(node.start.line, node.start.column, sourceUrl, code));
    }, preamble)
        .add(new source_map_1.SourceNode(end.line, end.column, sourceUrl, [
        '  };\n',
        '\n',
        '  return __p;\n',
        '}\n',
    ]));
    const code = nodes.toStringWithSourceMap({
        file: sourceUrl,
        sourceRoot: options && options.sourceRoot || '.',
    });
    // Set the source content in the source map, otherwise the sourceUrl is not enough
    // to find the content.
    code.map.setSourceContent(sourceUrl, ast.content);
    return code.code
        + '\n//# sourceMappingURL=data:application/json;base64,'
        + Buffer.from(code.map.toString()).toString('base64');
}
/**
 * An equivalent of EJS templates, which is based on John Resig's `tmpl` implementation
 * (http://ejohn.org/blog/javascript-micro-templating/) and Laura Doktorova's doT.js
 * (https://github.com/olado/doT).
 *
 * This version differs from lodash by removing support from ES6 quasi-literals, and making the
 * code slightly simpler to follow. It also does not depend on any third party, which is nice.
 *
 * Finally, it supports SourceMap, if you ever need to debug, which is super nice.
 *
 * @param content The template content.
 * @param options Optional Options. See TemplateOptions for more description.
 * @return {(input: T) => string} A function that accept an input object and returns the content
 *         of the template with the input applied.
 */
function template(content, options) {
    const sourceUrl = options && options.sourceURL || 'ejs';
    const ast = templateParser(content, sourceUrl);
    let source;
    // If there's no need for source map support, we revert back to the fast implementation.
    if (options && options.sourceMap) {
        source = templateWithSourceMap(ast, options);
    }
    else {
        source = templateFast(ast, options);
    }
    // We pass a dummy module in case the module option is passed. If `module: true` is passed, we
    // need to only use the source, not the function itself. Otherwise expect a module object to be
    // passed, and we use that one.
    const fn = Function('module', source);
    const module = options && options.module
        ? (options.module === true ? { exports: {} } : options.module)
        : null;
    const result = fn(module);
    // Provide the compiled function's source by its `toString` method or
    // the `source` property as a convenience for inlining compiled templates.
    result.source = source;
    return result;
}
exports.template = template;
//# sourceMappingURL=data:application/json;base64,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