"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("../path");
const buffer_1 = require("./buffer");
const memory_1 = require("./memory");
const sync_1 = require("./sync");
class TestHost extends memory_1.SimpleMemoryHost {
    constructor(map = {}) {
        super();
        this._records = [];
        for (const filePath of Object.getOwnPropertyNames(map)) {
            this._write(path_1.normalize(filePath), buffer_1.stringToFileBuffer(map[filePath]));
        }
    }
    get records() {
        return [...this._records];
    }
    clearRecords() {
        this._records = [];
    }
    get files() {
        const sync = this.sync;
        function _visit(p) {
            return sync.list(p)
                .map(fragment => path_1.join(p, fragment))
                .reduce((files, path) => {
                if (sync.isDirectory(path)) {
                    return files.concat(_visit(path));
                }
                else {
                    return files.concat(path);
                }
            }, []);
        }
        return _visit(path_1.normalize('/'));
    }
    get sync() {
        if (!this._sync) {
            this._sync = new sync_1.SyncDelegateHost(this);
        }
        return this._sync;
    }
    clone() {
        const newHost = new TestHost();
        newHost._cache = new Map(this._cache);
        return newHost;
    }
    // Override parents functions to keep a record of all operators that were done.
    _write(path, content) {
        this._records.push({ kind: 'write', path });
        return super._write(path, content);
    }
    _read(path) {
        this._records.push({ kind: 'read', path });
        return super._read(path);
    }
    _delete(path) {
        this._records.push({ kind: 'delete', path });
        return super._delete(path);
    }
    _rename(from, to) {
        this._records.push({ kind: 'rename', from, to });
        return super._rename(from, to);
    }
    _list(path) {
        this._records.push({ kind: 'list', path });
        return super._list(path);
    }
    _exists(path) {
        this._records.push({ kind: 'exists', path });
        return super._exists(path);
    }
    _isDirectory(path) {
        this._records.push({ kind: 'isDirectory', path });
        return super._isDirectory(path);
    }
    _isFile(path) {
        this._records.push({ kind: 'isFile', path });
        return super._isFile(path);
    }
    _stat(path) {
        this._records.push({ kind: 'stat', path });
        return super._stat(path);
    }
    _watch(path, options) {
        this._records.push({ kind: 'watch', path });
        return super._watch(path, options);
    }
    $write(path, content) {
        return super._write(path_1.normalize(path), buffer_1.stringToFileBuffer(content));
    }
    $read(path) {
        return buffer_1.fileBufferToString(super._read(path_1.normalize(path)));
    }
    $list(path) {
        return super._list(path_1.normalize(path));
    }
    $exists(path) {
        return super._exists(path_1.normalize(path));
    }
    $isDirectory(path) {
        return super._isDirectory(path_1.normalize(path));
    }
    $isFile(path) {
        return super._isFile(path_1.normalize(path));
    }
}
exports.TestHost = TestHost;
//# sourceMappingURL=data:application/json;base64,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