"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const exception_1 = require("../exception");
class InvalidPathException extends exception_1.BaseException {
    constructor(path) { super(`Path ${JSON.stringify(path)} is invalid.`); }
}
exports.InvalidPathException = InvalidPathException;
class PathMustBeAbsoluteException extends exception_1.BaseException {
    constructor(path) { super(`Path ${JSON.stringify(path)} must be absolute.`); }
}
exports.PathMustBeAbsoluteException = PathMustBeAbsoluteException;
class PathCannotBeFragmentException extends exception_1.BaseException {
    constructor(path) { super(`Path ${JSON.stringify(path)} cannot be made a fragment.`); }
}
exports.PathCannotBeFragmentException = PathCannotBeFragmentException;
/**
 * The Separator for normalized path.
 * @type {Path}
 */
exports.NormalizedSep = '/';
/**
 * The root of a normalized path.
 * @type {Path}
 */
exports.NormalizedRoot = exports.NormalizedSep;
/**
 * Split a path into multiple path fragments. Each fragments except the last one will end with
 * a path separator.
 * @param {Path} path The path to split.
 * @returns {Path[]} An array of path fragments.
 */
function split(path) {
    const fragments = path.split(exports.NormalizedSep).map(x => fragment(x));
    if (fragments[fragments.length - 1].length === 0) {
        fragments.pop();
    }
    return fragments;
}
exports.split = split;
/**
 *
 */
function extname(path) {
    const base = basename(path);
    const i = base.lastIndexOf('.');
    if (i < 1) {
        return '';
    }
    else {
        return base.substr(i);
    }
}
exports.extname = extname;
/**
 * Return the basename of the path, as a Path. See path.basename
 */
function basename(path) {
    const i = path.lastIndexOf(exports.NormalizedSep);
    if (i == -1) {
        return fragment(path);
    }
    else {
        return fragment(path.substr(path.lastIndexOf(exports.NormalizedSep) + 1));
    }
}
exports.basename = basename;
/**
 * Return the dirname of the path, as a Path. See path.dirname
 */
function dirname(path) {
    const index = path.lastIndexOf(exports.NormalizedSep);
    if (index === -1) {
        return '';
    }
    const endIndex = index === 0 ? 1 : index; // case of file under root: '/file'
    return normalize(path.substr(0, endIndex));
}
exports.dirname = dirname;
/**
 * Join multiple paths together, and normalize the result. Accepts strings that will be
 * normalized as well (but the original must be a path).
 */
function join(p1, ...others) {
    if (others.length > 0) {
        return normalize((p1 ? p1 + exports.NormalizedSep : '') + others.join(exports.NormalizedSep));
    }
    else {
        return p1;
    }
}
exports.join = join;
/**
 * Returns true if a path is absolute.
 */
function isAbsolute(p) {
    return p.startsWith(exports.NormalizedSep);
}
exports.isAbsolute = isAbsolute;
/**
 * Returns a path such that `join(from, relative(from, to)) == to`.
 * Both paths must be absolute, otherwise it does not make much sense.
 */
function relative(from, to) {
    if (!isAbsolute(from)) {
        throw new PathMustBeAbsoluteException(from);
    }
    if (!isAbsolute(to)) {
        throw new PathMustBeAbsoluteException(to);
    }
    let p;
    if (from == to) {
        p = '';
    }
    else {
        const splitFrom = from.split(exports.NormalizedSep);
        const splitTo = to.split(exports.NormalizedSep);
        while (splitFrom.length > 0 && splitTo.length > 0 && splitFrom[0] == splitTo[0]) {
            splitFrom.shift();
            splitTo.shift();
        }
        if (splitFrom.length == 0) {
            p = splitTo.join(exports.NormalizedSep);
        }
        else {
            p = splitFrom.map(_ => '..').concat(splitTo).join(exports.NormalizedSep);
        }
    }
    return normalize(p);
}
exports.relative = relative;
/**
 * Returns a Path that is the resolution of p2, from p1. If p2 is absolute, it will return p2,
 * otherwise will join both p1 and p2.
 */
function resolve(p1, p2) {
    if (isAbsolute(p2)) {
        return p2;
    }
    else {
        return join(p1, p2);
    }
}
exports.resolve = resolve;
function fragment(path) {
    if (path.indexOf(exports.NormalizedSep) != -1) {
        throw new PathCannotBeFragmentException(path);
    }
    return path;
}
exports.fragment = fragment;
/**
 * Normalize a string into a Path. This is the only mean to get a Path type from a string that
 * represents a system path. Normalization includes:
 *   - Windows backslashes `\\` are replaced with `/`.
 *   - Windows drivers are replaced with `/X/`, where X is the drive letter.
 *   - Absolute paths starts with `/`.
 *   - Multiple `/` are replaced by a single one.
 *   - Path segments `.` are removed.
 *   - Path segments `..` are resolved.
 *   - If a path is absolute, having a `..` at the start is invalid (and will throw).
 */
function normalize(path) {
    if (path == '' || path == '.') {
        return '';
    }
    else if (path == exports.NormalizedRoot) {
        return exports.NormalizedRoot;
    }
    // Match absolute windows path.
    const original = path;
    if (path.match(/^[A-Z]:[\/\\]/i)) {
        path = '\\' + path[0] + '\\' + path.substr(3);
    }
    // We convert Windows paths as well here.
    const p = path.split(/[\/\\]/g);
    let relative = false;
    let i = 1;
    // Special case the first one.
    if (p[0] != '') {
        p.unshift('.');
        relative = true;
    }
    while (i < p.length) {
        if (p[i] == '.') {
            p.splice(i, 1);
        }
        else if (p[i] == '..') {
            if (i < 2 && !relative) {
                throw new InvalidPathException(original);
            }
            else if (i >= 2 && p[i - 1] != '..') {
                p.splice(i - 1, 2);
                i--;
            }
            else {
                i++;
            }
        }
        else if (p[i] == '') {
            p.splice(i, 1);
        }
        else {
            i++;
        }
    }
    if (p.length == 1) {
        return p[0] == '' ? exports.NormalizedSep : '';
    }
    else {
        if (p[0] == '.') {
            p.shift();
        }
        return p.join(exports.NormalizedSep);
    }
}
exports.normalize = normalize;
exports.path = (strings, ...values) => {
    return normalize(String.raw(strings, ...values));
};
function asWindowsPath(path) {
    const drive = path.match(/^\/(\w)\/(.*)$/);
    if (drive) {
        return `${drive[1]}:\\${drive[2].replace(/\//g, '\\')}`;
    }
    return path.replace(/\//g, '\\');
}
exports.asWindowsPath = asWindowsPath;
function asPosixPath(path) {
    return path;
}
exports.asPosixPath = asPosixPath;
function getSystemPath(path) {
    if (process.platform.startsWith('win32')) {
        return asWindowsPath(path);
    }
    else {
        return asPosixPath(path);
    }
}
exports.getSystemPath = getSystemPath;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGF0aC5qcyIsInNvdXJjZVJvb3QiOiIuLyIsInNvdXJjZXMiOlsicGFja2FnZXMvYW5ndWxhcl9kZXZraXQvY29yZS9zcmMvdmlydHVhbC1mcy9wYXRoLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUE7Ozs7OztHQU1HO0FBQ0gsNENBQTZDO0FBSTdDLDBCQUFrQyxTQUFRLHlCQUFhO0lBQ3JELFlBQVksSUFBWSxJQUFJLEtBQUssQ0FBQyxRQUFRLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQztDQUNqRjtBQUZELG9EQUVDO0FBQ0QsaUNBQXlDLFNBQVEseUJBQWE7SUFDNUQsWUFBWSxJQUFZLElBQUksS0FBSyxDQUFDLFFBQVEsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDLENBQUM7Q0FDdkY7QUFGRCxrRUFFQztBQUNELG1DQUEyQyxTQUFRLHlCQUFhO0lBQzlELFlBQVksSUFBWSxJQUFJLEtBQUssQ0FBQyxRQUFRLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLDZCQUE2QixDQUFDLENBQUMsQ0FBQyxDQUFDO0NBQ2hHO0FBRkQsc0VBRUM7QUFrQkQ7OztHQUdHO0FBQ1UsUUFBQSxhQUFhLEdBQUcsR0FBVyxDQUFDO0FBR3pDOzs7R0FHRztBQUNVLFFBQUEsY0FBYyxHQUFHLHFCQUFxQixDQUFDO0FBR3BEOzs7OztHQUtHO0FBQ0gsZUFBc0IsSUFBVTtJQUM5QixNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLHFCQUFhLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNsRSxFQUFFLENBQUMsQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNqRCxTQUFTLENBQUMsR0FBRyxFQUFFLENBQUM7SUFDbEIsQ0FBQztJQUVELE1BQU0sQ0FBQyxTQUFTLENBQUM7QUFDbkIsQ0FBQztBQVBELHNCQU9DO0FBRUQ7O0dBRUc7QUFDSCxpQkFBd0IsSUFBVTtJQUNoQyxNQUFNLElBQUksR0FBRyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDNUIsTUFBTSxDQUFDLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUNoQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNWLE1BQU0sQ0FBQyxFQUFFLENBQUM7SUFDWixDQUFDO0lBQUMsSUFBSSxDQUFDLENBQUM7UUFDTixNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN4QixDQUFDO0FBQ0gsQ0FBQztBQVJELDBCQVFDO0FBR0Q7O0dBRUc7QUFDSCxrQkFBeUIsSUFBVTtJQUNqQyxNQUFNLENBQUMsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLHFCQUFhLENBQUMsQ0FBQztJQUMxQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ1osTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN4QixDQUFDO0lBQUMsSUFBSSxDQUFDLENBQUM7UUFDTixNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxxQkFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNwRSxDQUFDO0FBQ0gsQ0FBQztBQVBELDRCQU9DO0FBR0Q7O0dBRUc7QUFDSCxpQkFBd0IsSUFBVTtJQUNoQyxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLHFCQUFhLENBQUMsQ0FBQztJQUM5QyxFQUFFLENBQUMsQ0FBQyxLQUFLLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2pCLE1BQU0sQ0FBQyxFQUFVLENBQUM7SUFDcEIsQ0FBQztJQUVELE1BQU0sUUFBUSxHQUFHLEtBQUssS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsbUNBQW1DO0lBRTdFLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsUUFBUSxDQUFDLENBQUMsQ0FBQztBQUM3QyxDQUFDO0FBVEQsMEJBU0M7QUFHRDs7O0dBR0c7QUFDSCxjQUFxQixFQUFRLEVBQUUsR0FBRyxNQUFnQjtJQUNoRCxFQUFFLENBQUMsQ0FBQyxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdEIsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxHQUFHLHFCQUFhLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMscUJBQWEsQ0FBQyxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUFDLElBQUksQ0FBQyxDQUFDO1FBQ04sTUFBTSxDQUFDLEVBQUUsQ0FBQztJQUNaLENBQUM7QUFDSCxDQUFDO0FBTkQsb0JBTUM7QUFHRDs7R0FFRztBQUNILG9CQUEyQixDQUFPO0lBQ2hDLE1BQU0sQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLHFCQUFhLENBQUMsQ0FBQztBQUNyQyxDQUFDO0FBRkQsZ0NBRUM7QUFHRDs7O0dBR0c7QUFDSCxrQkFBeUIsSUFBVSxFQUFFLEVBQVE7SUFDM0MsRUFBRSxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3RCLE1BQU0sSUFBSSwyQkFBMkIsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBQ0QsRUFBRSxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3BCLE1BQU0sSUFBSSwyQkFBMkIsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQsSUFBSSxDQUFTLENBQUM7SUFFZCxFQUFFLENBQUMsQ0FBQyxJQUFJLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztRQUNmLENBQUMsR0FBRyxFQUFFLENBQUM7SUFDVCxDQUFDO0lBQUMsSUFBSSxDQUFDLENBQUM7UUFDTixNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLHFCQUFhLENBQUMsQ0FBQztRQUM1QyxNQUFNLE9BQU8sR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDLHFCQUFhLENBQUMsQ0FBQztRQUV4QyxPQUFPLFNBQVMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxJQUFJLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztZQUNoRixTQUFTLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDbEIsT0FBTyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQ2xCLENBQUM7UUFFRCxFQUFFLENBQUMsQ0FBQyxTQUFTLENBQUMsTUFBTSxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDMUIsQ0FBQyxHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMscUJBQWEsQ0FBQyxDQUFDO1FBQ2xDLENBQUM7UUFBQyxJQUFJLENBQUMsQ0FBQztZQUNOLENBQUMsR0FBRyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksQ0FBQyxxQkFBYSxDQUFDLENBQUM7UUFDbkUsQ0FBQztJQUNILENBQUM7SUFFRCxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBQ3RCLENBQUM7QUE3QkQsNEJBNkJDO0FBR0Q7OztHQUdHO0FBQ0gsaUJBQXdCLEVBQVEsRUFBRSxFQUFRO0lBQ3hDLEVBQUUsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDbkIsTUFBTSxDQUFDLEVBQUUsQ0FBQztJQUNaLENBQUM7SUFBQyxJQUFJLENBQUMsQ0FBQztRQUNOLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7QUFDSCxDQUFDO0FBTkQsMEJBTUM7QUFHRCxrQkFBeUIsSUFBWTtJQUNuQyxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLHFCQUFhLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdEMsTUFBTSxJQUFJLDZCQUE2QixDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRCxNQUFNLENBQUMsSUFBb0IsQ0FBQztBQUM5QixDQUFDO0FBTkQsNEJBTUM7QUFHRDs7Ozs7Ozs7OztHQVVHO0FBQ0gsbUJBQTBCLElBQVk7SUFDcEMsRUFBRSxDQUFDLENBQUMsSUFBSSxJQUFJLEVBQUUsSUFBSSxJQUFJLElBQUksR0FBRyxDQUFDLENBQUMsQ0FBQztRQUM5QixNQUFNLENBQUMsRUFBVSxDQUFDO0lBQ3BCLENBQUM7SUFBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsSUFBSSxJQUFJLHNCQUFjLENBQUMsQ0FBQyxDQUFDO1FBQ2xDLE1BQU0sQ0FBQyxzQkFBYyxDQUFDO0lBQ3hCLENBQUM7SUFFRCwrQkFBK0I7SUFDL0IsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDO0lBQ3RCLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDakMsSUFBSSxHQUFHLElBQUksR0FBRyxJQUFJLENBQUMsQ0FBQyxDQUFDLEdBQUcsSUFBSSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVELHlDQUF5QztJQUN6QyxNQUFNLENBQUMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQ2hDLElBQUksUUFBUSxHQUFHLEtBQUssQ0FBQztJQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7SUFFViw4QkFBOEI7SUFDOUIsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDZixDQUFDLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2YsUUFBUSxHQUFHLElBQUksQ0FBQztJQUNsQixDQUFDO0lBRUQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBQ3BCLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQ2hCLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ2pCLENBQUM7UUFBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDeEIsRUFBRSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7Z0JBQ3ZCLE1BQU0sSUFBSSxvQkFBb0IsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUMzQyxDQUFDO1lBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDO2dCQUN0QyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7Z0JBQ25CLENBQUMsRUFBRSxDQUFDO1lBQ04sQ0FBQztZQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNOLENBQUMsRUFBRSxDQUFDO1lBQ04sQ0FBQztRQUNILENBQUM7UUFBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDdEIsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDakIsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sQ0FBQyxFQUFFLENBQUM7UUFDTixDQUFDO0lBQ0gsQ0FBQztJQUVELEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNsQixNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMscUJBQWEsQ0FBQyxDQUFDLENBQUMsRUFBVSxDQUFDO0lBQ2pELENBQUM7SUFBQyxJQUFJLENBQUMsQ0FBQztRQUNOLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxDQUFDO1lBQ2hCLENBQUMsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUNaLENBQUM7UUFFRCxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxxQkFBYSxDQUFTLENBQUM7SUFDdkMsQ0FBQztBQUNILENBQUM7QUFwREQsOEJBb0RDO0FBR1ksUUFBQSxJQUFJLEdBQXNCLENBQUMsT0FBTyxFQUFFLEdBQUcsTUFBTSxFQUFFLEVBQUU7SUFDNUQsTUFBTSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLE9BQU8sRUFBRSxHQUFHLE1BQU0sQ0FBQyxDQUFDLENBQUM7QUFDbkQsQ0FBQyxDQUFDO0FBV0YsdUJBQThCLElBQVU7SUFDdEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQzNDLEVBQUUsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDVixNQUFNLENBQUMsR0FBRyxLQUFLLENBQUMsQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLEVBQWlCLENBQUM7SUFDekUsQ0FBQztJQUVELE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxJQUFJLENBQWdCLENBQUM7QUFDbEQsQ0FBQztBQVBELHNDQU9DO0FBRUQscUJBQTRCLElBQVU7SUFDcEMsTUFBTSxDQUFDLElBQTJCLENBQUM7QUFDckMsQ0FBQztBQUZELGtDQUVDO0FBRUQsdUJBQThCLElBQVU7SUFDdEMsRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3pDLE1BQU0sQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDN0IsQ0FBQztJQUFDLElBQUksQ0FBQyxDQUFDO1FBQ04sTUFBTSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUMzQixDQUFDO0FBQ0gsQ0FBQztBQU5ELHNDQU1DIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBJbmMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuaW1wb3J0IHsgQmFzZUV4Y2VwdGlvbiB9IGZyb20gJy4uL2V4Y2VwdGlvbic7XG5pbXBvcnQgeyBUZW1wbGF0ZVRhZyB9IGZyb20gJy4uL3V0aWxzL2xpdGVyYWxzJztcblxuXG5leHBvcnQgY2xhc3MgSW52YWxpZFBhdGhFeGNlcHRpb24gZXh0ZW5kcyBCYXNlRXhjZXB0aW9uIHtcbiAgY29uc3RydWN0b3IocGF0aDogc3RyaW5nKSB7IHN1cGVyKGBQYXRoICR7SlNPTi5zdHJpbmdpZnkocGF0aCl9IGlzIGludmFsaWQuYCk7IH1cbn1cbmV4cG9ydCBjbGFzcyBQYXRoTXVzdEJlQWJzb2x1dGVFeGNlcHRpb24gZXh0ZW5kcyBCYXNlRXhjZXB0aW9uIHtcbiAgY29uc3RydWN0b3IocGF0aDogc3RyaW5nKSB7IHN1cGVyKGBQYXRoICR7SlNPTi5zdHJpbmdpZnkocGF0aCl9IG11c3QgYmUgYWJzb2x1dGUuYCk7IH1cbn1cbmV4cG9ydCBjbGFzcyBQYXRoQ2Fubm90QmVGcmFnbWVudEV4Y2VwdGlvbiBleHRlbmRzIEJhc2VFeGNlcHRpb24ge1xuICBjb25zdHJ1Y3RvcihwYXRoOiBzdHJpbmcpIHsgc3VwZXIoYFBhdGggJHtKU09OLnN0cmluZ2lmeShwYXRoKX0gY2Fubm90IGJlIG1hZGUgYSBmcmFnbWVudC5gKTsgfVxufVxuXG5cbi8qKlxuICogQSBQYXRoIHJlY29nbml6ZWQgYnkgbW9zdCBtZXRob2RzIGluIHRoZSBEZXZLaXQuXG4gKi9cbmV4cG9ydCB0eXBlIFBhdGggPSBzdHJpbmcgJiB7XG4gIF9fUFJJVkFURV9ERVZLSVRfUEFUSDogdm9pZDtcbn07XG5cbi8qKlxuICogQSBQYXRoIGZyYWdtZW50IChmaWxlIG9yIGRpcmVjdG9yeSBuYW1lKSByZWNvZ25pemVkIGJ5IG1vc3QgbWV0aG9kcyBpbiB0aGUgRGV2S2l0LlxuICovXG5leHBvcnQgdHlwZSBQYXRoRnJhZ21lbnQgPSBQYXRoICYge1xuICBfX1BSSVZBVEVfREVWS0lUX1BBVEhfRlJBR01FTlQ6IHZvaWQ7XG59O1xuXG5cbi8qKlxuICogVGhlIFNlcGFyYXRvciBmb3Igbm9ybWFsaXplZCBwYXRoLlxuICogQHR5cGUge1BhdGh9XG4gKi9cbmV4cG9ydCBjb25zdCBOb3JtYWxpemVkU2VwID0gJy8nIGFzIFBhdGg7XG5cblxuLyoqXG4gKiBUaGUgcm9vdCBvZiBhIG5vcm1hbGl6ZWQgcGF0aC5cbiAqIEB0eXBlIHtQYXRofVxuICovXG5leHBvcnQgY29uc3QgTm9ybWFsaXplZFJvb3QgPSBOb3JtYWxpemVkU2VwIGFzIFBhdGg7XG5cblxuLyoqXG4gKiBTcGxpdCBhIHBhdGggaW50byBtdWx0aXBsZSBwYXRoIGZyYWdtZW50cy4gRWFjaCBmcmFnbWVudHMgZXhjZXB0IHRoZSBsYXN0IG9uZSB3aWxsIGVuZCB3aXRoXG4gKiBhIHBhdGggc2VwYXJhdG9yLlxuICogQHBhcmFtIHtQYXRofSBwYXRoIFRoZSBwYXRoIHRvIHNwbGl0LlxuICogQHJldHVybnMge1BhdGhbXX0gQW4gYXJyYXkgb2YgcGF0aCBmcmFnbWVudHMuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBzcGxpdChwYXRoOiBQYXRoKTogUGF0aEZyYWdtZW50W10ge1xuICBjb25zdCBmcmFnbWVudHMgPSBwYXRoLnNwbGl0KE5vcm1hbGl6ZWRTZXApLm1hcCh4ID0+IGZyYWdtZW50KHgpKTtcbiAgaWYgKGZyYWdtZW50c1tmcmFnbWVudHMubGVuZ3RoIC0gMV0ubGVuZ3RoID09PSAwKSB7XG4gICAgZnJhZ21lbnRzLnBvcCgpO1xuICB9XG5cbiAgcmV0dXJuIGZyYWdtZW50cztcbn1cblxuLyoqXG4gKlxuICovXG5leHBvcnQgZnVuY3Rpb24gZXh0bmFtZShwYXRoOiBQYXRoKTogc3RyaW5nIHtcbiAgY29uc3QgYmFzZSA9IGJhc2VuYW1lKHBhdGgpO1xuICBjb25zdCBpID0gYmFzZS5sYXN0SW5kZXhPZignLicpO1xuICBpZiAoaSA8IDEpIHtcbiAgICByZXR1cm4gJyc7XG4gIH0gZWxzZSB7XG4gICAgcmV0dXJuIGJhc2Uuc3Vic3RyKGkpO1xuICB9XG59XG5cblxuLyoqXG4gKiBSZXR1cm4gdGhlIGJhc2VuYW1lIG9mIHRoZSBwYXRoLCBhcyBhIFBhdGguIFNlZSBwYXRoLmJhc2VuYW1lXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBiYXNlbmFtZShwYXRoOiBQYXRoKTogUGF0aEZyYWdtZW50IHtcbiAgY29uc3QgaSA9IHBhdGgubGFzdEluZGV4T2YoTm9ybWFsaXplZFNlcCk7XG4gIGlmIChpID09IC0xKSB7XG4gICAgcmV0dXJuIGZyYWdtZW50KHBhdGgpO1xuICB9IGVsc2Uge1xuICAgIHJldHVybiBmcmFnbWVudChwYXRoLnN1YnN0cihwYXRoLmxhc3RJbmRleE9mKE5vcm1hbGl6ZWRTZXApICsgMSkpO1xuICB9XG59XG5cblxuLyoqXG4gKiBSZXR1cm4gdGhlIGRpcm5hbWUgb2YgdGhlIHBhdGgsIGFzIGEgUGF0aC4gU2VlIHBhdGguZGlybmFtZVxuICovXG5leHBvcnQgZnVuY3Rpb24gZGlybmFtZShwYXRoOiBQYXRoKTogUGF0aCB7XG4gIGNvbnN0IGluZGV4ID0gcGF0aC5sYXN0SW5kZXhPZihOb3JtYWxpemVkU2VwKTtcbiAgaWYgKGluZGV4ID09PSAtMSkge1xuICAgIHJldHVybiAnJyBhcyBQYXRoO1xuICB9XG5cbiAgY29uc3QgZW5kSW5kZXggPSBpbmRleCA9PT0gMCA/IDEgOiBpbmRleDsgLy8gY2FzZSBvZiBmaWxlIHVuZGVyIHJvb3Q6ICcvZmlsZSdcblxuICByZXR1cm4gbm9ybWFsaXplKHBhdGguc3Vic3RyKDAsIGVuZEluZGV4KSk7XG59XG5cblxuLyoqXG4gKiBKb2luIG11bHRpcGxlIHBhdGhzIHRvZ2V0aGVyLCBhbmQgbm9ybWFsaXplIHRoZSByZXN1bHQuIEFjY2VwdHMgc3RyaW5ncyB0aGF0IHdpbGwgYmVcbiAqIG5vcm1hbGl6ZWQgYXMgd2VsbCAoYnV0IHRoZSBvcmlnaW5hbCBtdXN0IGJlIGEgcGF0aCkuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBqb2luKHAxOiBQYXRoLCAuLi5vdGhlcnM6IHN0cmluZ1tdKTogUGF0aCB7XG4gIGlmIChvdGhlcnMubGVuZ3RoID4gMCkge1xuICAgIHJldHVybiBub3JtYWxpemUoKHAxID8gcDEgKyBOb3JtYWxpemVkU2VwIDogJycpICsgb3RoZXJzLmpvaW4oTm9ybWFsaXplZFNlcCkpO1xuICB9IGVsc2Uge1xuICAgIHJldHVybiBwMTtcbiAgfVxufVxuXG5cbi8qKlxuICogUmV0dXJucyB0cnVlIGlmIGEgcGF0aCBpcyBhYnNvbHV0ZS5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGlzQWJzb2x1dGUocDogUGF0aCkge1xuICByZXR1cm4gcC5zdGFydHNXaXRoKE5vcm1hbGl6ZWRTZXApO1xufVxuXG5cbi8qKlxuICogUmV0dXJucyBhIHBhdGggc3VjaCB0aGF0IGBqb2luKGZyb20sIHJlbGF0aXZlKGZyb20sIHRvKSkgPT0gdG9gLlxuICogQm90aCBwYXRocyBtdXN0IGJlIGFic29sdXRlLCBvdGhlcndpc2UgaXQgZG9lcyBub3QgbWFrZSBtdWNoIHNlbnNlLlxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVsYXRpdmUoZnJvbTogUGF0aCwgdG86IFBhdGgpOiBQYXRoIHtcbiAgaWYgKCFpc0Fic29sdXRlKGZyb20pKSB7XG4gICAgdGhyb3cgbmV3IFBhdGhNdXN0QmVBYnNvbHV0ZUV4Y2VwdGlvbihmcm9tKTtcbiAgfVxuICBpZiAoIWlzQWJzb2x1dGUodG8pKSB7XG4gICAgdGhyb3cgbmV3IFBhdGhNdXN0QmVBYnNvbHV0ZUV4Y2VwdGlvbih0byk7XG4gIH1cblxuICBsZXQgcDogc3RyaW5nO1xuXG4gIGlmIChmcm9tID09IHRvKSB7XG4gICAgcCA9ICcnO1xuICB9IGVsc2Uge1xuICAgIGNvbnN0IHNwbGl0RnJvbSA9IGZyb20uc3BsaXQoTm9ybWFsaXplZFNlcCk7XG4gICAgY29uc3Qgc3BsaXRUbyA9IHRvLnNwbGl0KE5vcm1hbGl6ZWRTZXApO1xuXG4gICAgd2hpbGUgKHNwbGl0RnJvbS5sZW5ndGggPiAwICYmIHNwbGl0VG8ubGVuZ3RoID4gMCAmJiBzcGxpdEZyb21bMF0gPT0gc3BsaXRUb1swXSkge1xuICAgICAgc3BsaXRGcm9tLnNoaWZ0KCk7XG4gICAgICBzcGxpdFRvLnNoaWZ0KCk7XG4gICAgfVxuXG4gICAgaWYgKHNwbGl0RnJvbS5sZW5ndGggPT0gMCkge1xuICAgICAgcCA9IHNwbGl0VG8uam9pbihOb3JtYWxpemVkU2VwKTtcbiAgICB9IGVsc2Uge1xuICAgICAgcCA9IHNwbGl0RnJvbS5tYXAoXyA9PiAnLi4nKS5jb25jYXQoc3BsaXRUbykuam9pbihOb3JtYWxpemVkU2VwKTtcbiAgICB9XG4gIH1cblxuICByZXR1cm4gbm9ybWFsaXplKHApO1xufVxuXG5cbi8qKlxuICogUmV0dXJucyBhIFBhdGggdGhhdCBpcyB0aGUgcmVzb2x1dGlvbiBvZiBwMiwgZnJvbSBwMS4gSWYgcDIgaXMgYWJzb2x1dGUsIGl0IHdpbGwgcmV0dXJuIHAyLFxuICogb3RoZXJ3aXNlIHdpbGwgam9pbiBib3RoIHAxIGFuZCBwMi5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHJlc29sdmUocDE6IFBhdGgsIHAyOiBQYXRoKSB7XG4gIGlmIChpc0Fic29sdXRlKHAyKSkge1xuICAgIHJldHVybiBwMjtcbiAgfSBlbHNlIHtcbiAgICByZXR1cm4gam9pbihwMSwgcDIpO1xuICB9XG59XG5cblxuZXhwb3J0IGZ1bmN0aW9uIGZyYWdtZW50KHBhdGg6IHN0cmluZyk6IFBhdGhGcmFnbWVudCB7XG4gIGlmIChwYXRoLmluZGV4T2YoTm9ybWFsaXplZFNlcCkgIT0gLTEpIHtcbiAgICB0aHJvdyBuZXcgUGF0aENhbm5vdEJlRnJhZ21lbnRFeGNlcHRpb24ocGF0aCk7XG4gIH1cblxuICByZXR1cm4gcGF0aCBhcyBQYXRoRnJhZ21lbnQ7XG59XG5cblxuLyoqXG4gKiBOb3JtYWxpemUgYSBzdHJpbmcgaW50byBhIFBhdGguIFRoaXMgaXMgdGhlIG9ubHkgbWVhbiB0byBnZXQgYSBQYXRoIHR5cGUgZnJvbSBhIHN0cmluZyB0aGF0XG4gKiByZXByZXNlbnRzIGEgc3lzdGVtIHBhdGguIE5vcm1hbGl6YXRpb24gaW5jbHVkZXM6XG4gKiAgIC0gV2luZG93cyBiYWNrc2xhc2hlcyBgXFxcXGAgYXJlIHJlcGxhY2VkIHdpdGggYC9gLlxuICogICAtIFdpbmRvd3MgZHJpdmVycyBhcmUgcmVwbGFjZWQgd2l0aCBgL1gvYCwgd2hlcmUgWCBpcyB0aGUgZHJpdmUgbGV0dGVyLlxuICogICAtIEFic29sdXRlIHBhdGhzIHN0YXJ0cyB3aXRoIGAvYC5cbiAqICAgLSBNdWx0aXBsZSBgL2AgYXJlIHJlcGxhY2VkIGJ5IGEgc2luZ2xlIG9uZS5cbiAqICAgLSBQYXRoIHNlZ21lbnRzIGAuYCBhcmUgcmVtb3ZlZC5cbiAqICAgLSBQYXRoIHNlZ21lbnRzIGAuLmAgYXJlIHJlc29sdmVkLlxuICogICAtIElmIGEgcGF0aCBpcyBhYnNvbHV0ZSwgaGF2aW5nIGEgYC4uYCBhdCB0aGUgc3RhcnQgaXMgaW52YWxpZCAoYW5kIHdpbGwgdGhyb3cpLlxuICovXG5leHBvcnQgZnVuY3Rpb24gbm9ybWFsaXplKHBhdGg6IHN0cmluZyk6IFBhdGgge1xuICBpZiAocGF0aCA9PSAnJyB8fCBwYXRoID09ICcuJykge1xuICAgIHJldHVybiAnJyBhcyBQYXRoO1xuICB9IGVsc2UgaWYgKHBhdGggPT0gTm9ybWFsaXplZFJvb3QpIHtcbiAgICByZXR1cm4gTm9ybWFsaXplZFJvb3Q7XG4gIH1cblxuICAvLyBNYXRjaCBhYnNvbHV0ZSB3aW5kb3dzIHBhdGguXG4gIGNvbnN0IG9yaWdpbmFsID0gcGF0aDtcbiAgaWYgKHBhdGgubWF0Y2goL15bQS1aXTpbXFwvXFxcXF0vaSkpIHtcbiAgICBwYXRoID0gJ1xcXFwnICsgcGF0aFswXSArICdcXFxcJyArIHBhdGguc3Vic3RyKDMpO1xuICB9XG5cbiAgLy8gV2UgY29udmVydCBXaW5kb3dzIHBhdGhzIGFzIHdlbGwgaGVyZS5cbiAgY29uc3QgcCA9IHBhdGguc3BsaXQoL1tcXC9cXFxcXS9nKTtcbiAgbGV0IHJlbGF0aXZlID0gZmFsc2U7XG4gIGxldCBpID0gMTtcblxuICAvLyBTcGVjaWFsIGNhc2UgdGhlIGZpcnN0IG9uZS5cbiAgaWYgKHBbMF0gIT0gJycpIHtcbiAgICBwLnVuc2hpZnQoJy4nKTtcbiAgICByZWxhdGl2ZSA9IHRydWU7XG4gIH1cblxuICB3aGlsZSAoaSA8IHAubGVuZ3RoKSB7XG4gICAgaWYgKHBbaV0gPT0gJy4nKSB7XG4gICAgICBwLnNwbGljZShpLCAxKTtcbiAgICB9IGVsc2UgaWYgKHBbaV0gPT0gJy4uJykge1xuICAgICAgaWYgKGkgPCAyICYmICFyZWxhdGl2ZSkge1xuICAgICAgICB0aHJvdyBuZXcgSW52YWxpZFBhdGhFeGNlcHRpb24ob3JpZ2luYWwpO1xuICAgICAgfSBlbHNlIGlmIChpID49IDIgJiYgcFtpIC0gMV0gIT0gJy4uJykge1xuICAgICAgICBwLnNwbGljZShpIC0gMSwgMik7XG4gICAgICAgIGktLTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGkrKztcbiAgICAgIH1cbiAgICB9IGVsc2UgaWYgKHBbaV0gPT0gJycpIHtcbiAgICAgIHAuc3BsaWNlKGksIDEpO1xuICAgIH0gZWxzZSB7XG4gICAgICBpKys7XG4gICAgfVxuICB9XG5cbiAgaWYgKHAubGVuZ3RoID09IDEpIHtcbiAgICByZXR1cm4gcFswXSA9PSAnJyA/IE5vcm1hbGl6ZWRTZXAgOiAnJyBhcyBQYXRoO1xuICB9IGVsc2Uge1xuICAgIGlmIChwWzBdID09ICcuJykge1xuICAgICAgcC5zaGlmdCgpO1xuICAgIH1cblxuICAgIHJldHVybiBwLmpvaW4oTm9ybWFsaXplZFNlcCkgYXMgUGF0aDtcbiAgfVxufVxuXG5cbmV4cG9ydCBjb25zdCBwYXRoOiBUZW1wbGF0ZVRhZzxQYXRoPiA9IChzdHJpbmdzLCAuLi52YWx1ZXMpID0+IHtcbiAgcmV0dXJuIG5vcm1hbGl6ZShTdHJpbmcucmF3KHN0cmluZ3MsIC4uLnZhbHVlcykpO1xufTtcblxuXG4vLyBQbGF0Zm9ybS1zcGVjaWZpYyBwYXRocy5cbmV4cG9ydCB0eXBlIFdpbmRvd3NQYXRoID0gc3RyaW5nICYge1xuICBfX1BSSVZBVEVfREVWS0lUX1dJTkRPV1NfUEFUSDogdm9pZDtcbn07XG5leHBvcnQgdHlwZSBQb3NpeFBhdGggPSBzdHJpbmcgJiB7XG4gIF9fUFJJVkFURV9ERVZLSVRfUE9TSVhfUEFUSDogdm9pZDtcbn07XG5cbmV4cG9ydCBmdW5jdGlvbiBhc1dpbmRvd3NQYXRoKHBhdGg6IFBhdGgpOiBXaW5kb3dzUGF0aCB7XG4gIGNvbnN0IGRyaXZlID0gcGF0aC5tYXRjaCgvXlxcLyhcXHcpXFwvKC4qKSQvKTtcbiAgaWYgKGRyaXZlKSB7XG4gICAgcmV0dXJuIGAke2RyaXZlWzFdfTpcXFxcJHtkcml2ZVsyXS5yZXBsYWNlKC9cXC8vZywgJ1xcXFwnKX1gIGFzIFdpbmRvd3NQYXRoO1xuICB9XG5cbiAgcmV0dXJuIHBhdGgucmVwbGFjZSgvXFwvL2csICdcXFxcJykgYXMgV2luZG93c1BhdGg7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBhc1Bvc2l4UGF0aChwYXRoOiBQYXRoKTogUG9zaXhQYXRoIHtcbiAgcmV0dXJuIHBhdGggYXMgc3RyaW5nIGFzIFBvc2l4UGF0aDtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGdldFN5c3RlbVBhdGgocGF0aDogUGF0aCk6IHN0cmluZyB7XG4gIGlmIChwcm9jZXNzLnBsYXRmb3JtLnN0YXJ0c1dpdGgoJ3dpbjMyJykpIHtcbiAgICByZXR1cm4gYXNXaW5kb3dzUGF0aChwYXRoKTtcbiAgfSBlbHNlIHtcbiAgICByZXR1cm4gYXNQb3NpeFBhdGgocGF0aCk7XG4gIH1cbn1cbiJdfQ==