"use strict";
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const exception_1 = require("../exception");
const json_1 = require("../json");
const virtual_fs_1 = require("../virtual-fs");
class ProjectNotFoundException extends exception_1.BaseException {
    constructor(name) {
        super(`Project '${name}' could not be found in workspace.`);
    }
}
exports.ProjectNotFoundException = ProjectNotFoundException;
class WorkspaceToolNotFoundException extends exception_1.BaseException {
    constructor(name) {
        super(`Tool ${name} could not be found in workspace.`);
    }
}
exports.WorkspaceToolNotFoundException = WorkspaceToolNotFoundException;
class ProjectToolNotFoundException extends exception_1.BaseException {
    constructor(name) {
        super(`Tool ${name} could not be found in project.`);
    }
}
exports.ProjectToolNotFoundException = ProjectToolNotFoundException;
class WorkspaceNotYetLoadedException extends exception_1.BaseException {
    constructor() { super(`Workspace needs to be loaded before it is used.`); }
}
exports.WorkspaceNotYetLoadedException = WorkspaceNotYetLoadedException;
class AmbiguousProjectPathException extends exception_1.BaseException {
    constructor(path, projects) {
        super(`Current active project is ambiguous (${projects.join(',')}) using path: '${path}'`);
        this.path = path;
        this.projects = projects;
    }
}
exports.AmbiguousProjectPathException = AmbiguousProjectPathException;
class Workspace {
    constructor(_root, _host) {
        this._root = _root;
        this._host = _host;
        this._workspaceSchemaPath = virtual_fs_1.join(virtual_fs_1.normalize(__dirname), 'workspace-schema.json');
        this._registry = new json_1.schema.CoreSchemaRegistry();
    }
    loadWorkspaceFromJson(json) {
        return this._loadWorkspaceSchema().pipe(operators_1.concatMap((workspaceSchema) => this.validateAgainstSchema(json, workspaceSchema)), operators_1.tap((validatedWorkspace) => this._workspace = validatedWorkspace), operators_1.map(() => this));
    }
    loadWorkspaceFromHost(workspacePath) {
        return this._loadWorkspaceSchema().pipe(operators_1.concatMap(() => this._loadJsonFile(virtual_fs_1.join(this._root, workspacePath))), operators_1.concatMap(json => this.loadWorkspaceFromJson(json)));
    }
    _loadWorkspaceSchema() {
        if (this._workspaceSchema) {
            return rxjs_1.of(this._workspaceSchema);
        }
        else {
            return this._loadJsonFile(this._workspaceSchemaPath).pipe(operators_1.tap((workspaceSchema) => this._workspaceSchema = workspaceSchema));
        }
    }
    _assertLoaded() {
        if (!this._workspace) {
            throw new WorkspaceNotYetLoadedException();
        }
    }
    get root() {
        return this._root;
    }
    get host() {
        return this._host;
    }
    get version() {
        this._assertLoaded();
        return this._workspace.version;
    }
    get newProjectRoot() {
        this._assertLoaded();
        return this._workspace.newProjectRoot;
    }
    listProjectNames() {
        return Object.keys(this._workspace.projects);
    }
    getProject(projectName) {
        this._assertLoaded();
        const workspaceProject = this._workspace.projects[projectName];
        if (!workspaceProject) {
            throw new ProjectNotFoundException(projectName);
        }
        return Object.assign({}, workspaceProject, { 
            // Return only the project properties, and remove the tools.
            cli: {}, schematics: {}, architect: {} });
    }
    getDefaultProjectName() {
        this._assertLoaded();
        if (this._workspace.defaultProject) {
            // If there is a default project name, return it.
            return this._workspace.defaultProject;
        }
        else if (this.listProjectNames().length === 1) {
            // If there is only one project, return that one.
            return this.listProjectNames()[0];
        }
        // Otherwise return null.
        return null;
    }
    getProjectByPath(path) {
        this._assertLoaded();
        const projectNames = this.listProjectNames();
        if (projectNames.length === 1) {
            return projectNames[0];
        }
        const isInside = (base, potential) => {
            const absoluteBase = virtual_fs_1.resolve(this.root, base);
            const absolutePotential = virtual_fs_1.resolve(this.root, potential);
            const relativePotential = virtual_fs_1.relative(absoluteBase, absolutePotential);
            if (!relativePotential.startsWith('..') && !virtual_fs_1.isAbsolute(relativePotential)) {
                return true;
            }
            return false;
        };
        const projects = this.listProjectNames()
            .map(name => [this.getProject(name).root, name])
            .filter(tuple => isInside(tuple[0], path))
            .sort((a, b) => b[0].length - a[0].length);
        if (projects.length === 0) {
            return null;
        }
        else if (projects.length > 1) {
            const found = new Set();
            const sameRoots = projects.filter(v => {
                if (!found.has(v[0])) {
                    found.add(v[0]);
                    return false;
                }
                return true;
            });
            if (sameRoots.length > 0) {
                throw new AmbiguousProjectPathException(path, sameRoots.map(v => v[1]));
            }
        }
        return projects[0][1];
    }
    getCli() {
        return this._getTool('cli');
    }
    getSchematics() {
        return this._getTool('schematics');
    }
    getArchitect() {
        return this._getTool('architect');
    }
    getProjectCli(projectName) {
        return this._getProjectTool(projectName, 'cli');
    }
    getProjectSchematics(projectName) {
        return this._getProjectTool(projectName, 'schematics');
    }
    getProjectArchitect(projectName) {
        return this._getProjectTool(projectName, 'architect');
    }
    _getTool(toolName) {
        this._assertLoaded();
        const workspaceTool = this._workspace[toolName];
        if (!workspaceTool) {
            throw new WorkspaceToolNotFoundException(toolName);
        }
        return workspaceTool;
    }
    _getProjectTool(projectName, toolName) {
        this._assertLoaded();
        const workspaceProject = this._workspace.projects[projectName];
        if (!workspaceProject) {
            throw new ProjectNotFoundException(projectName);
        }
        const projectTool = workspaceProject[toolName];
        if (!projectTool) {
            throw new ProjectToolNotFoundException(toolName);
        }
        return projectTool;
    }
    // TODO: add transforms to resolve paths.
    validateAgainstSchema(contentJson, schemaJson) {
        // JSON validation modifies the content, so we validate a copy of it instead.
        const contentJsonCopy = JSON.parse(JSON.stringify(contentJson));
        return this._registry.compile(schemaJson).pipe(operators_1.concatMap(validator => validator(contentJsonCopy)), operators_1.concatMap(validatorResult => {
            if (validatorResult.success) {
                return rxjs_1.of(contentJsonCopy);
            }
            else {
                return rxjs_1.throwError(new json_1.schema.SchemaValidationException(validatorResult.errors));
            }
        }));
    }
    _loadJsonFile(path) {
        return this._host.read(virtual_fs_1.normalize(path)).pipe(operators_1.map(buffer => virtual_fs_1.virtualFs.fileBufferToString(buffer)), operators_1.map(str => json_1.parseJson(str, json_1.JsonParseMode.Loose)));
    }
}
exports.Workspace = Workspace;
//# sourceMappingURL=data:application/json;base64,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