/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Identifiers, createTokenForExternalReference } from '../identifiers';
import { LifecycleHooks } from '../lifecycle_reflector';
import * as o from '../output/output_ast';
import { convertValueToOutputAst } from '../output/value_util';
import { ProviderAstType } from '../template_parser/template_ast';
export function providerDef(ctx, providerAst) {
    var flags = 0 /* None */;
    if (!providerAst.eager) {
        flags |= 4096 /* LazyProvider */;
    }
    if (providerAst.providerType === ProviderAstType.PrivateService) {
        flags |= 8192 /* PrivateProvider */;
    }
    if (providerAst.isModule) {
        flags |= 1073741824 /* TypeModuleProvider */;
    }
    providerAst.lifecycleHooks.forEach(function (lifecycleHook) {
        // for regular providers, we only support ngOnDestroy
        if (lifecycleHook === LifecycleHooks.OnDestroy ||
            providerAst.providerType === ProviderAstType.Directive ||
            providerAst.providerType === ProviderAstType.Component) {
            flags |= lifecycleHookToNodeFlag(lifecycleHook);
        }
    });
    var _a = providerAst.multiProvider ?
        multiProviderDef(ctx, flags, providerAst.providers) :
        singleProviderDef(ctx, flags, providerAst.providerType, providerAst.providers[0]), providerExpr = _a.providerExpr, providerFlags = _a.flags, depsExpr = _a.depsExpr;
    return {
        providerExpr: providerExpr,
        flags: providerFlags, depsExpr: depsExpr,
        tokenExpr: tokenExpr(ctx, providerAst.token),
    };
}
function multiProviderDef(ctx, flags, providers) {
    var allDepDefs = [];
    var allParams = [];
    var exprs = providers.map(function (provider, providerIndex) {
        var expr;
        if (provider.useClass) {
            var depExprs = convertDeps(providerIndex, provider.deps || provider.useClass.diDeps);
            expr = ctx.importExpr(provider.useClass.reference).instantiate(depExprs);
        }
        else if (provider.useFactory) {
            var depExprs = convertDeps(providerIndex, provider.deps || provider.useFactory.diDeps);
            expr = ctx.importExpr(provider.useFactory.reference).callFn(depExprs);
        }
        else if (provider.useExisting) {
            var depExprs = convertDeps(providerIndex, [{ token: provider.useExisting }]);
            expr = depExprs[0];
        }
        else {
            expr = convertValueToOutputAst(ctx, provider.useValue);
        }
        return expr;
    });
    var providerExpr = o.fn(allParams, [new o.ReturnStatement(o.literalArr(exprs))], o.INFERRED_TYPE);
    return {
        providerExpr: providerExpr,
        flags: flags | 1024 /* TypeFactoryProvider */,
        depsExpr: o.literalArr(allDepDefs)
    };
    function convertDeps(providerIndex, deps) {
        return deps.map(function (dep, depIndex) {
            var paramName = "p" + providerIndex + "_" + depIndex;
            allParams.push(new o.FnParam(paramName, o.DYNAMIC_TYPE));
            allDepDefs.push(depDef(ctx, dep));
            return o.variable(paramName);
        });
    }
}
function singleProviderDef(ctx, flags, providerType, providerMeta) {
    var providerExpr;
    var deps;
    if (providerType === ProviderAstType.Directive || providerType === ProviderAstType.Component) {
        providerExpr = ctx.importExpr(providerMeta.useClass.reference);
        flags |= 16384 /* TypeDirective */;
        deps = providerMeta.deps || providerMeta.useClass.diDeps;
    }
    else {
        if (providerMeta.useClass) {
            providerExpr = ctx.importExpr(providerMeta.useClass.reference);
            flags |= 512 /* TypeClassProvider */;
            deps = providerMeta.deps || providerMeta.useClass.diDeps;
        }
        else if (providerMeta.useFactory) {
            providerExpr = ctx.importExpr(providerMeta.useFactory.reference);
            flags |= 1024 /* TypeFactoryProvider */;
            deps = providerMeta.deps || providerMeta.useFactory.diDeps;
        }
        else if (providerMeta.useExisting) {
            providerExpr = o.NULL_EXPR;
            flags |= 2048 /* TypeUseExistingProvider */;
            deps = [{ token: providerMeta.useExisting }];
        }
        else {
            providerExpr = convertValueToOutputAst(ctx, providerMeta.useValue);
            flags |= 256 /* TypeValueProvider */;
            deps = [];
        }
    }
    var depsExpr = o.literalArr(deps.map(function (dep) { return depDef(ctx, dep); }));
    return { providerExpr: providerExpr, flags: flags, depsExpr: depsExpr };
}
function tokenExpr(ctx, tokenMeta) {
    return tokenMeta.identifier ? ctx.importExpr(tokenMeta.identifier.reference) :
        o.literal(tokenMeta.value);
}
export function depDef(ctx, dep) {
    // Note: the following fields have already been normalized out by provider_analyzer:
    // - isAttribute, isHost
    var expr = dep.isValue ? convertValueToOutputAst(ctx, dep.value) : tokenExpr(ctx, dep.token);
    var flags = 0 /* None */;
    if (dep.isSkipSelf) {
        flags |= 1 /* SkipSelf */;
    }
    if (dep.isOptional) {
        flags |= 2 /* Optional */;
    }
    if (dep.isSelf) {
        flags |= 4 /* Self */;
    }
    if (dep.isValue) {
        flags |= 8 /* Value */;
    }
    return flags === 0 /* None */ ? expr : o.literalArr([o.literal(flags), expr]);
}
export function lifecycleHookToNodeFlag(lifecycleHook) {
    var nodeFlag = 0 /* None */;
    switch (lifecycleHook) {
        case LifecycleHooks.AfterContentChecked:
            nodeFlag = 2097152 /* AfterContentChecked */;
            break;
        case LifecycleHooks.AfterContentInit:
            nodeFlag = 1048576 /* AfterContentInit */;
            break;
        case LifecycleHooks.AfterViewChecked:
            nodeFlag = 8388608 /* AfterViewChecked */;
            break;
        case LifecycleHooks.AfterViewInit:
            nodeFlag = 4194304 /* AfterViewInit */;
            break;
        case LifecycleHooks.DoCheck:
            nodeFlag = 262144 /* DoCheck */;
            break;
        case LifecycleHooks.OnChanges:
            nodeFlag = 524288 /* OnChanges */;
            break;
        case LifecycleHooks.OnDestroy:
            nodeFlag = 131072 /* OnDestroy */;
            break;
        case LifecycleHooks.OnInit:
            nodeFlag = 65536 /* OnInit */;
            break;
    }
    return nodeFlag;
}
export function componentFactoryResolverProviderDef(reflector, ctx, flags, entryComponents) {
    var entryComponentFactories = entryComponents.map(function (entryComponent) { return ctx.importExpr(entryComponent.componentFactory); });
    var token = createTokenForExternalReference(reflector, Identifiers.ComponentFactoryResolver);
    var classMeta = {
        diDeps: [
            { isValue: true, value: o.literalArr(entryComponentFactories) },
            { token: token, isSkipSelf: true, isOptional: true },
            { token: createTokenForExternalReference(reflector, Identifiers.NgModuleRef) },
        ],
        lifecycleHooks: [],
        reference: reflector.resolveExternalReference(Identifiers.CodegenComponentFactoryResolver)
    };
    var _a = singleProviderDef(ctx, flags, ProviderAstType.PrivateService, {
        token: token,
        multi: false,
        useClass: classMeta,
    }), providerExpr = _a.providerExpr, providerFlags = _a.flags, depsExpr = _a.depsExpr;
    return { providerExpr: providerExpr, flags: providerFlags, depsExpr: depsExpr, tokenExpr: tokenExpr(ctx, token) };
}
//# sourceMappingURL=data:application/json;base64,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