"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const path_1 = require("path");
const ts = require("typescript");
const ast_helpers_1 = require("./ast_helpers");
const insert_import_1 = require("./insert_import");
const interfaces_1 = require("./interfaces");
const make_transform_1 = require("./make_transform");
function replaceServerBootstrap(shouldTransform, getEntryModule, getTypeChecker) {
    const standardTransform = function (sourceFile) {
        const ops = [];
        const entryModule = getEntryModule();
        if (!shouldTransform(sourceFile.fileName) || !entryModule) {
            return ops;
        }
        // Find all identifiers.
        const entryModuleIdentifiers = ast_helpers_1.collectDeepNodes(sourceFile, ts.SyntaxKind.Identifier)
            .filter(identifier => identifier.text === entryModule.className);
        if (entryModuleIdentifiers.length === 0) {
            return [];
        }
        const relativeEntryModulePath = path_1.relative(path_1.dirname(sourceFile.fileName), entryModule.path);
        const normalizedEntryModulePath = `./${relativeEntryModulePath}`.replace(/\\/g, '/');
        const factoryClassName = entryModule.className + 'NgFactory';
        const factoryModulePath = normalizedEntryModulePath + '.ngfactory';
        // Find the bootstrap calls.
        entryModuleIdentifiers.forEach(entryModuleIdentifier => {
            if (!entryModuleIdentifier.parent) {
                return;
            }
            if (entryModuleIdentifier.parent.kind !== ts.SyntaxKind.CallExpression &&
                entryModuleIdentifier.parent.kind !== ts.SyntaxKind.PropertyAssignment) {
                return;
            }
            if (entryModuleIdentifier.parent.kind === ts.SyntaxKind.CallExpression) {
                // Figure out if it's a `platformDynamicServer().bootstrapModule(AppModule)` call.
                const callExpr = entryModuleIdentifier.parent;
                if (callExpr.expression.kind === ts.SyntaxKind.PropertyAccessExpression) {
                    const propAccessExpr = callExpr.expression;
                    if (!(propAccessExpr.name.text === 'bootstrapModule'
                        && propAccessExpr.expression.kind === ts.SyntaxKind.CallExpression)) {
                        return;
                    }
                    const bootstrapModuleIdentifier = propAccessExpr.name;
                    const innerCallExpr = propAccessExpr.expression;
                    if (!(innerCallExpr.expression.kind === ts.SyntaxKind.Identifier
                        && innerCallExpr.expression.text === 'platformDynamicServer')) {
                        return;
                    }
                    const platformDynamicServerIdentifier = innerCallExpr.expression;
                    const idPlatformServer = ts.createUniqueName('__NgCli_bootstrap_');
                    const idNgFactory = ts.createUniqueName('__NgCli_bootstrap_');
                    // Add the transform operations.
                    ops.push(
                    // Replace the entry module import.
                    ...insert_import_1.insertStarImport(sourceFile, idNgFactory, factoryModulePath), new interfaces_1.ReplaceNodeOperation(sourceFile, entryModuleIdentifier, ts.createPropertyAccess(idNgFactory, ts.createIdentifier(factoryClassName))), 
                    // Replace the platformBrowserDynamic import.
                    ...insert_import_1.insertStarImport(sourceFile, idPlatformServer, '@angular/platform-server'), new interfaces_1.ReplaceNodeOperation(sourceFile, platformDynamicServerIdentifier, ts.createPropertyAccess(idPlatformServer, 'platformServer')), new interfaces_1.ReplaceNodeOperation(sourceFile, bootstrapModuleIdentifier, ts.createIdentifier('bootstrapModuleFactory')));
                }
                else if (callExpr.expression.kind === ts.SyntaxKind.Identifier) {
                    // Figure out if it is renderModule
                    const identifierExpr = callExpr.expression;
                    if (identifierExpr.text !== 'renderModule') {
                        return;
                    }
                    const renderModuleIdentifier = identifierExpr;
                    const idPlatformServer = ts.createUniqueName('__NgCli_bootstrap_');
                    const idNgFactory = ts.createUniqueName('__NgCli_bootstrap_');
                    ops.push(
                    // Replace the entry module import.
                    ...insert_import_1.insertStarImport(sourceFile, idNgFactory, factoryModulePath), new interfaces_1.ReplaceNodeOperation(sourceFile, entryModuleIdentifier, ts.createPropertyAccess(idNgFactory, ts.createIdentifier(factoryClassName))), 
                    // Replace the renderModule import.
                    ...insert_import_1.insertStarImport(sourceFile, idPlatformServer, '@angular/platform-server'), new interfaces_1.ReplaceNodeOperation(sourceFile, renderModuleIdentifier, ts.createPropertyAccess(idPlatformServer, 'renderModuleFactory')));
                }
            }
            else if (entryModuleIdentifier.parent.kind === ts.SyntaxKind.PropertyAssignment) {
                // This is for things that accept a module as a property in a config object
                // .ie the express engine
                const idNgFactory = ts.createUniqueName('__NgCli_bootstrap_');
                ops.push(...insert_import_1.insertStarImport(sourceFile, idNgFactory, factoryModulePath), new interfaces_1.ReplaceNodeOperation(sourceFile, entryModuleIdentifier, ts.createPropertyAccess(idNgFactory, ts.createIdentifier(factoryClassName))));
            }
        });
        return ops;
    };
    return make_transform_1.makeTransform(standardTransform, getTypeChecker);
}
exports.replaceServerBootstrap = replaceServerBootstrap;
//# sourceMappingURL=data:application/json;base64,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