"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const schematics_1 = require("@angular-devkit/schematics");
const tasks_1 = require("@angular-devkit/schematics/tasks");
const ts = require("typescript");
const ast_utils_1 = require("../utility/ast-utils");
const change_1 = require("../utility/change");
const config_1 = require("../utility/config");
const dependencies_1 = require("../utility/dependencies");
const ng_ast_utils_1 = require("../utility/ng-ast-utils");
function getWorkspacePath(host) {
    const possibleFiles = ['/angular.json', '/.angular.json'];
    return possibleFiles.filter(path => host.exists(path))[0];
}
function getClientProject(host, options) {
    const workspace = config_1.getWorkspace(host);
    const clientProject = workspace.projects[options.clientProject];
    if (!clientProject) {
        throw new schematics_1.SchematicsException(`Client app ${options.clientProject} not found.`);
    }
    return clientProject;
}
function getClientArchitect(host, options) {
    const clientArchitect = getClientProject(host, options).architect;
    if (!clientArchitect) {
        throw new Error('Client project architect not found.');
    }
    return clientArchitect;
}
function updateConfigFile(options, tsConfigDirectory) {
    return (host) => {
        const workspace = config_1.getWorkspace(host);
        if (!workspace.projects[options.clientProject]) {
            throw new schematics_1.SchematicsException(`Client app ${options.clientProject} not found.`);
        }
        const clientProject = workspace.projects[options.clientProject];
        if (!clientProject.architect) {
            throw new Error('Client project architect not found.');
        }
        const builderOptions = {
            outputPath: `dist/${options.clientProject}-server`,
            main: `${clientProject.root}src/main.server.ts`,
            tsConfig: core_1.join(tsConfigDirectory, `${options.tsconfigFileName}.json`),
        };
        const serverTarget = {
            builder: '@angular-devkit/build-angular:server',
            options: builderOptions,
        };
        clientProject.architect.server = serverTarget;
        const workspacePath = getWorkspacePath(host);
        host.overwrite(workspacePath, JSON.stringify(workspace, null, 2));
        return host;
    };
}
function findBrowserModuleImport(host, modulePath) {
    const moduleBuffer = host.read(modulePath);
    if (!moduleBuffer) {
        throw new schematics_1.SchematicsException(`Module file (${modulePath}) not found`);
    }
    const moduleFileText = moduleBuffer.toString('utf-8');
    const source = ts.createSourceFile(modulePath, moduleFileText, ts.ScriptTarget.Latest, true);
    const decoratorMetadata = ast_utils_1.getDecoratorMetadata(source, 'NgModule', '@angular/core')[0];
    const browserModuleNode = ast_utils_1.findNode(decoratorMetadata, ts.SyntaxKind.Identifier, 'BrowserModule');
    if (browserModuleNode === null) {
        throw new schematics_1.SchematicsException(`Cannot find BrowserModule import in ${modulePath}`);
    }
    return browserModuleNode;
}
function wrapBootstrapCall(options) {
    return (host) => {
        const clientArchitect = getClientArchitect(host, options);
        const mainPath = core_1.normalize('/' + clientArchitect.build.options.main);
        let bootstrapCall = ng_ast_utils_1.findBootstrapModuleCall(host, mainPath);
        if (bootstrapCall === null) {
            throw new schematics_1.SchematicsException('Bootstrap module not found.');
        }
        let bootstrapCallExpression = null;
        let currentCall = bootstrapCall;
        while (bootstrapCallExpression === null && currentCall.parent) {
            currentCall = currentCall.parent;
            if (currentCall.kind === ts.SyntaxKind.ExpressionStatement) {
                bootstrapCallExpression = currentCall;
            }
        }
        bootstrapCall = currentCall;
        const recorder = host.beginUpdate(mainPath);
        const beforeText = `document.addEventListener('DOMContentLoaded', () => {\n  `;
        const afterText = `\n});`;
        recorder.insertLeft(bootstrapCall.getStart(), beforeText);
        recorder.insertRight(bootstrapCall.getEnd(), afterText);
        host.commitUpdate(recorder);
    };
}
function addServerTransition(options) {
    return (host) => {
        const clientProject = getClientProject(host, options);
        const clientArchitect = getClientArchitect(host, options);
        const mainPath = core_1.normalize('/' + clientArchitect.build.options.main);
        const bootstrapModuleRelativePath = ng_ast_utils_1.findBootstrapModulePath(host, mainPath);
        const bootstrapModulePath = core_1.normalize(`/${clientProject.root}/src/${bootstrapModuleRelativePath}.ts`);
        const browserModuleImport = findBrowserModuleImport(host, bootstrapModulePath);
        const appId = options.appId;
        const transitionCall = `.withServerTransition({ appId: '${appId}' })`;
        const position = browserModuleImport.pos + browserModuleImport.getFullText().length;
        const transitionCallChange = new change_1.InsertChange(bootstrapModulePath, position, transitionCall);
        const transitionCallRecorder = host.beginUpdate(bootstrapModulePath);
        transitionCallRecorder.insertLeft(transitionCallChange.pos, transitionCallChange.toAdd);
        host.commitUpdate(transitionCallRecorder);
    };
}
function addDependencies() {
    return (host) => {
        const coreDep = dependencies_1.getPackageJsonDependency(host, '@angular/core');
        if (coreDep === null) {
            throw new schematics_1.SchematicsException('Could not find version.');
        }
        const platformServerDep = Object.assign({}, coreDep, { name: '@angular/platform-server' });
        dependencies_1.addPackageJsonDependency(host, platformServerDep);
        return host;
    };
}
function getTsConfigOutDir(host, architect) {
    const tsConfigPath = architect.build.options.tsConfig;
    const tsConfigBuffer = host.read(tsConfigPath);
    if (!tsConfigBuffer) {
        throw new schematics_1.SchematicsException(`Could not read ${tsConfigPath}`);
    }
    const tsConfigContent = tsConfigBuffer.toString();
    const tsConfig = core_1.parseJson(tsConfigContent);
    if (tsConfig === null || typeof tsConfig !== 'object' || Array.isArray(tsConfig) ||
        tsConfig.compilerOptions === null || typeof tsConfig.compilerOptions !== 'object' ||
        Array.isArray(tsConfig.compilerOptions)) {
        throw new schematics_1.SchematicsException(`Invalid tsconfig - ${tsConfigPath}`);
    }
    const outDir = tsConfig.compilerOptions.outDir;
    return outDir;
}
function default_1(options) {
    return (host, context) => {
        const clientProject = getClientProject(host, options);
        if (clientProject.projectType !== 'application') {
            throw new schematics_1.SchematicsException(`Universal requires a project type of "application".`);
        }
        const clientArchitect = getClientArchitect(host, options);
        const outDir = getTsConfigOutDir(host, clientArchitect);
        const tsConfigExtends = core_1.basename(clientArchitect.build.options.tsConfig);
        const rootInSrc = clientProject.root === '';
        const tsConfigDirectory = core_1.join(core_1.normalize(clientProject.root), rootInSrc ? 'src' : '');
        if (!options.skipInstall) {
            context.addTask(new tasks_1.NodePackageInstallTask());
        }
        const templateSource = schematics_1.apply(schematics_1.url('./files/src'), [
            schematics_1.template(Object.assign({}, core_1.strings, options, { stripTsExtension: (s) => { return s.replace(/\.ts$/, ''); } })),
            schematics_1.move(core_1.join(core_1.normalize(clientProject.root), 'src')),
        ]);
        const rootSource = schematics_1.apply(schematics_1.url('./files/root'), [
            schematics_1.template(Object.assign({}, core_1.strings, options, { stripTsExtension: (s) => { return s.replace(/\.ts$/, ''); }, outDir,
                tsConfigExtends,
                rootInSrc })),
            schematics_1.move(tsConfigDirectory),
        ]);
        return schematics_1.chain([
            schematics_1.mergeWith(templateSource),
            schematics_1.mergeWith(rootSource),
            addDependencies(),
            updateConfigFile(options, tsConfigDirectory),
            wrapBootstrapCall(options),
            addServerTransition(options),
        ]);
    };
}
exports.default = default_1;
//# sourceMappingURL=data:application/json;base64,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