"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const schematics_1 = require("@angular-devkit/schematics");
function getWorkspacePath(host) {
    const possibleFiles = ['/angular.json', '/.angular.json'];
    const path = possibleFiles.filter(path => host.exists(path))[0];
    return path;
}
exports.getWorkspacePath = getWorkspacePath;
function getWorkspace(host) {
    const path = getWorkspacePath(host);
    const configBuffer = host.read(path);
    if (configBuffer === null) {
        throw new schematics_1.SchematicsException(`Could not find (${path})`);
    }
    const content = configBuffer.toString();
    return core_1.parseJson(content, core_1.JsonParseMode.Loose);
}
exports.getWorkspace = getWorkspace;
function addProjectToWorkspace(workspace, name, project) {
    return (host, context) => {
        if (workspace.projects[name]) {
            throw new Error(`Project '${name}' already exists in workspace.`);
        }
        // Add project to workspace.
        workspace.projects[name] = project;
        if (!workspace.defaultProject && Object.keys(workspace.projects).length === 1) {
            // Make the new project the default one.
            workspace.defaultProject = name;
        }
        host.overwrite(getWorkspacePath(host), JSON.stringify(workspace, null, 2));
    };
}
exports.addProjectToWorkspace = addProjectToWorkspace;
exports.configPath = '/.angular-cli.json';
function getConfig(host) {
    const configBuffer = host.read(exports.configPath);
    if (configBuffer === null) {
        throw new schematics_1.SchematicsException('Could not find .angular-cli.json');
    }
    const config = core_1.parseJson(configBuffer.toString(), core_1.JsonParseMode.Loose);
    return config;
}
exports.getConfig = getConfig;
function getAppFromConfig(config, appIndexOrName) {
    if (!config.apps) {
        return null;
    }
    if (parseInt(appIndexOrName) >= 0) {
        return config.apps[parseInt(appIndexOrName)];
    }
    return config.apps.filter((app) => app.name === appIndexOrName)[0];
}
exports.getAppFromConfig = getAppFromConfig;
//# sourceMappingURL=data:application/json;base64,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