"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = require("fs");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const url = require("url");
const RegistryClient = require('npm-registry-client');
const rc = require('rc');
const npmPackageJsonCache = new Map();
const npmConfigOptionCache = new Map();
const npmConfig = rc('npm', {}, {});
function getNpmConfigOption(option, scope, tryWithoutScope) {
    if (scope && tryWithoutScope) {
        return rxjs_1.concat(getNpmConfigOption(option, scope), getNpmConfigOption(option)).pipe(operators_1.filter(result => !!result), operators_1.defaultIfEmpty(), operators_1.first());
    }
    const fullOption = `${scope ? scope + ':' : ''}${option}`;
    let value = npmConfigOptionCache.get(fullOption);
    if (value) {
        return value;
    }
    const subject = new rxjs_1.ReplaySubject(1);
    const optionValue = npmConfig && npmConfig[fullOption];
    if (optionValue == undefined || optionValue == null) {
        subject.next();
    }
    else {
        subject.next(optionValue);
    }
    subject.complete();
    value = subject.asObservable();
    npmConfigOptionCache.set(fullOption, value);
    return value;
}
function getNpmClientSslOptions(strictSsl, cafile) {
    const sslOptions = {};
    if (strictSsl === 'false') {
        sslOptions.strict = false;
    }
    else if (strictSsl === 'true') {
        sslOptions.strict = true;
    }
    if (cafile) {
        sslOptions.ca = fs_1.readFileSync(cafile);
    }
    return sslOptions;
}
/**
 * Get the NPM repository's package.json for a package. This is p
 * @param {string} packageName The package name to fetch.
 * @param {string} registryUrl The NPM Registry URL to use.
 * @param {LoggerApi} logger A logger instance to log debug information.
 * @returns An observable that will put the pacakge.json content.
 * @private
 */
function getNpmPackageJson(packageName, registryUrl, logger) {
    const scope = packageName.startsWith('@') ? packageName.split('/')[0] : undefined;
    return (registryUrl ? rxjs_1.of(registryUrl) : getNpmConfigOption('registry', scope, true)).pipe(operators_1.map(partialUrl => {
        if (!partialUrl) {
            partialUrl = 'https://registry.npmjs.org/';
        }
        const partial = url.parse(partialUrl);
        let fullUrl = new url.URL(`http://${partial.host}/${packageName.replace(/\//g, '%2F')}`);
        try {
            const registry = new url.URL(partialUrl);
            registry.pathname = (registry.pathname || '')
                .replace(/\/?$/, '/' + packageName.replace(/\//g, '%2F'));
            fullUrl = new url.URL(url.format(registry));
        }
        catch (_a) { }
        logger.debug(`Getting package.json from '${packageName}' (url: ${JSON.stringify(fullUrl)})...`);
        return fullUrl;
    }), operators_1.concatMap(fullUrl => {
        let maybeRequest = npmPackageJsonCache.get(fullUrl.toString());
        if (maybeRequest) {
            return maybeRequest;
        }
        const registryKey = `//${fullUrl.host}/`;
        return rxjs_1.concat(getNpmConfigOption('proxy'), getNpmConfigOption('https-proxy'), getNpmConfigOption('strict-ssl'), getNpmConfigOption('cafile'), getNpmConfigOption('_auth'), getNpmConfigOption('_authToken', registryKey), getNpmConfigOption('username', registryKey, true), getNpmConfigOption('password', registryKey, true), getNpmConfigOption('alwaysAuth', registryKey, true)).pipe(operators_1.toArray(), operators_1.concatMap(options => {
            const [http, https, strictSsl, cafile, token, authToken, username, password, alwaysAuth,] = options;
            const subject = new rxjs_1.ReplaySubject(1);
            const sslOptions = getNpmClientSslOptions(strictSsl, cafile);
            const auth = {};
            if (alwaysAuth !== undefined) {
                auth.alwaysAuth = alwaysAuth === 'false' ? false : !!alwaysAuth;
            }
            if (authToken) {
                auth.token = authToken;
            }
            else if (token) {
                auth.token = token;
            }
            else if (username) {
                auth.username = username;
                auth.password = password;
            }
            const client = new RegistryClient({
                proxy: { http, https },
                ssl: sslOptions,
            });
            client.log.level = 'silent';
            const params = {
                timeout: 30000,
                auth,
            };
            client.get(fullUrl.toString(), params, (error, data) => {
                if (error) {
                    subject.error(error);
                }
                subject.next(data);
                subject.complete();
            });
            maybeRequest = subject.asObservable();
            npmPackageJsonCache.set(fullUrl.toString(), maybeRequest);
            return maybeRequest;
        }));
    }));
}
exports.getNpmPackageJson = getNpmPackageJson;
//# sourceMappingURL=data:application/json;base64,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