import { Inject, Injectable, InjectionToken, NgModule, Pipe } from '@angular/core';
import { addResource, addResourceBundle, addResources, changeLanguage, dir, exists, format, getFixedT, getResource, getResourceBundle, hasResourceBundle, init, language, languages, loadNamespaces, on, options, reloadResources, removeResourceBundle, setDefaultNamespace, t, use } from 'i18next/index';
import * as i18next from 'i18next/index';
import { BehaviorSubject, Subject } from 'rxjs';
import { DOCUMENT, Title } from '@angular/platform-browser';

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
const I18NEXT_SCOPE = new InjectionToken('I18NEXT_SCOPE');
const I18NEXT_NAMESPACE = new InjectionToken('I18NEXT_NAMESPACE');
const I18NEXT_SERVICE = new InjectionToken('I18NEXT_SERVICE');
const I18NEXT_NAMESPACE_RESOLVER = new InjectionToken('I18NEXT_NAMESPACE_RESOLVER');
const I18NEXT_ERROR_HANDLING_STRATEGY = new InjectionToken('I18NEXT_ERROR_HANDLING_STRATEGY');

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class I18NextPipe {
    /**
     * @param {?} translateI18Next
     * @param {?} ns
     * @param {?} scope
     */
    constructor(translateI18Next, ns, scope) {
        this.translateI18Next = translateI18Next;
        this.ns = ns;
        this.scope = scope;
    }
    /**
     * @param {?} key
     * @param {?=} options
     * @return {?}
     */
    transform(key, options$$1) {
        options$$1 = this.prepareOptions(options$$1);
        let /** @type {?} */ i18nOpts = this.translateI18Next.options;
        if (options$$1.prependScope === undefined || options$$1.prependScope === true) {
            if (this.scope) {
                key = this.prependScope(key, this.scope, i18nOpts.keySeparator, i18nOpts.nsSeparator);
            }
        }
        if (options$$1.prependNamespace === undefined || options$$1.prependNamespace === true) {
            if (this.ns) {
                key = this.prependNamespace(key, this.ns, i18nOpts.nsSeparator);
            }
        }
        let /** @type {?} */ result = this.translateI18Next.t(key, options$$1);
        if (options$$1.format) {
            if (result) {
                result = this.translateI18Next
                    .format(result, options$$1.format, this.translateI18Next.language);
            }
        }
        return result;
    }
    /**
     * @param {?} key
     * @param {?} scope
     * @param {?} keySeparator
     * @param {?} nsSeparator
     * @return {?}
     */
    prependScope(key, scope, keySeparator, nsSeparator) {
        if (typeof (key) === 'string') {
            key = [key];
        }
        if (typeof (scope) === 'string') {
            scope = [scope];
        }
        let /** @type {?} */ keysWithScope = [];
        for (let /** @type {?} */ i = 0; i < key.length; i++) {
            const /** @type {?} */ k = key[i];
            if (!this.keyContainsNsSeparator(k, nsSeparator)) {
                // Do not set scope, if key contains a namespace
                keysWithScope.push(...scope.map(sc => this.joinStrings(keySeparator, sc, k)));
            }
            keysWithScope.push(k);
        }
        return keysWithScope;
    }
    /**
     * @param {?} key
     * @param {?} ns
     * @param {?} nsSeparator
     * @return {?}
     */
    prependNamespace(key, ns, nsSeparator) {
        if (typeof (key) === 'string') {
            key = [key];
        }
        if (typeof (ns) === 'string') {
            ns = [ns];
        }
        let /** @type {?} */ keysWithNamespace = [];
        for (let /** @type {?} */ i = 0; i < key.length; i++) {
            const /** @type {?} */ k = key[i];
            if (!this.keyContainsNsSeparator(k, nsSeparator)) {
                // Do not set namespace, if key contains a namespace
                keysWithNamespace.push(...ns.map(n => this.joinStrings(nsSeparator, n, k)));
            }
            keysWithNamespace.push(k);
        }
        return keysWithNamespace;
    }
    /**
     * @param {?} separator
     * @param {...?} str
     * @return {?}
     */
    joinStrings(separator, ...str) {
        return [...str].join(separator);
    }
    /**
     * @param {?} key
     * @param {?} nsSeparator
     * @return {?}
     */
    keyContainsNsSeparator(key, nsSeparator) {
        return key.indexOf(nsSeparator) !== -1;
    }
    /**
     * @param {?} options
     * @return {?}
     */
    prepareOptions(options$$1) {
        options$$1 = options$$1 || {};
        if (options$$1.context != null)
            options$$1.context = options$$1.context.toString();
        return options$$1;
    }
}
I18NextPipe.decorators = [
    { type: Injectable },
    { type: Pipe, args: [{
                name: 'i18next'
            },] },
];
/** @nocollapse */
I18NextPipe.ctorParameters = () => [
    { type: undefined, decorators: [{ type: Inject, args: [I18NEXT_SERVICE,] }] },
    { type: undefined, decorators: [{ type: Inject, args: [I18NEXT_NAMESPACE,] }] },
    { type: undefined, decorators: [{ type: Inject, args: [I18NEXT_SCOPE,] }] }
];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class I18NextCapPipe extends I18NextPipe {
    /**
     * @param {?} translateI18Next
     * @param {?} ns
     * @param {?} scope
     */
    constructor(translateI18Next, ns, scope) {
        super(translateI18Next, ns, scope);
    }
    /**
     * @param {?} key
     * @param {?=} options
     * @return {?}
     */
    transform(key, options$$1) {
        options$$1 = options$$1 || {};
        options$$1.format = 'cap';
        return super.transform(key, options$$1);
    }
}
I18NextCapPipe.decorators = [
    { type: Injectable },
    { type: Pipe, args: [{
                name: 'i18nextCap'
            },] },
];
/** @nocollapse */
I18NextCapPipe.ctorParameters = () => [
    { type: undefined, decorators: [{ type: Inject, args: [I18NEXT_SERVICE,] }] },
    { type: undefined, decorators: [{ type: Inject, args: [I18NEXT_NAMESPACE,] }] },
    { type: undefined, decorators: [{ type: Inject, args: [I18NEXT_SCOPE,] }] }
];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * @record
 */

class NativeErrorHandlingStrategy {
    /**
     * @param {?} resolve
     * @param {?} reject
     * @return {?}
     */
    handle(resolve, reject) {
        return (err, t$$1) => {
            let /** @type {?} */ result = {
                err: err,
                t: t$$1
            };
            resolve(result);
        };
    }
}
class StrictErrorHandlingStrategy {
    /**
     * @param {?} resolve
     * @param {?} reject
     * @return {?}
     */
    handle(resolve, reject) {
        return (err, t$$1) => {
            let /** @type {?} */ result = {
                err: err,
                t: t$$1
            };
            if (!err) {
                resolve(result);
                return;
            }
            reject(err);
        };
    }
}

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class I18NextFormatPipe {
    /**
     * @param {?} translateI18Next
     */
    constructor(translateI18Next) {
        this.translateI18Next = translateI18Next;
    }
    /**
     * @param {?} value
     * @param {?} options
     * @return {?}
     */
    transform(value, options$$1) {
        let /** @type {?} */ opts = typeof (options$$1) === 'string' ? { format: options$$1 } : options$$1;
        return this.translateI18Next.format(value, opts.format, opts.lng);
    }
}
I18NextFormatPipe.decorators = [
    { type: Injectable },
    { type: Pipe, args: [{
                name: 'i18nextFormat'
            },] },
];
/** @nocollapse */
I18NextFormatPipe.ctorParameters = () => [
    { type: undefined, decorators: [{ type: Inject, args: [I18NEXT_SERVICE,] }] }
];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class I18NextEvents {
    constructor() {
        this.initialized = new BehaviorSubject(false);
        this.loaded = new BehaviorSubject(false);
        this.failedLoading = new Subject();
        this.missingKey = new Subject();
        this.added = new Subject();
        this.removed = new Subject();
        this.languageChanged = new BehaviorSubject(null);
    }
}

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class I18NextService {
    /**
     * @param {?} errorHandlingStrategy
     */
    constructor(errorHandlingStrategy) {
        this.errorHandlingStrategy = errorHandlingStrategy;
        this.events = new I18NextEvents();
        this.language = '';
        this.languages = [];
    }
    /**
     * @return {?}
     */
    get options() {
        return options;
    }
    /**
     * @param {?} plugin
     * @return {?}
     */
    use(plugin) {
        use.call(i18next, plugin);
        return this;
    }
    /**
     * @param {?=} options
     * @return {?}
     */
    init(options$$1) {
        options$$1 = options$$1 || {};
        this.subscribeEvents();
        return this.i18nextPromise =
            new Promise((resolve, reject) => {
                init.call(i18next, Object.assign({}, options$$1), this.errorHandlingStrategy.handle(resolve, reject));
            });
    }
    /**
     * @param {?} key
     * @param {?=} options
     * @return {?}
     */
    t(key, options$$1) {
        options$$1 = options$$1 || {};
        return t.call(i18next, /** @type {?} */ (key), options$$1);
    }
    /**
     * @param {?} value
     * @param {?} format
     * @param {?} lng
     * @return {?}
     */
    format(value, format$$1, lng) {
        return format.call(i18next, value, format$$1, lng);
    }
    /**
     * @param {?} key
     * @param {?} options
     * @return {?}
     */
    exists(key, options$$1) {
        return exists.call(i18next, key, options$$1);
    }
    /**
     * @param {?} lng
     * @param {?} ns
     * @return {?}
     */
    getFixedT(lng, ns) {
        return getFixedT.call(i18next, lng, ns);
    }
    /**
     * @param {?} ns
     * @return {?}
     */
    setDefaultNamespace(ns) {
        setDefaultNamespace.call(i18next, ns);
    }
    /**
     * @param {?=} lng
     * @return {?}
     */
    dir(lng = undefined) {
        return dir.call(i18next, lng);
    }
    /**
     * @param {?} lng
     * @return {?}
     */
    changeLanguage(lng) {
        return new Promise((resolve, reject) => {
            changeLanguage.call(i18next, lng, this.errorHandlingStrategy.handle(resolve, reject));
        });
    }
    /**
     * @param {?} namespaces
     * @return {?}
     */
    loadNamespaces(namespaces) {
        return new Promise((resolve, reject) => {
            loadNamespaces.call(i18next, namespaces, this.errorHandlingStrategy.handle(resolve, reject));
        });
    }
    /**
     * @param {?} lngs
     * @param {?} callback
     * @return {?}
     */
    loadLanguages(lngs, callback) {
        return new Promise((resolve, reject) => {
            loadNamespaces.call(i18next, lngs, this.errorHandlingStrategy.handle(resolve, reject));
        });
    }
    /**
     * @param {...?} params
     * @return {?}
     */
    reloadResources(...params) {
        reloadResources.apply(i18next, params);
    }
    /**
     * @param {?} lng
     * @param {?} ns
     * @param {?} key
     * @param {?} options
     * @return {?}
     */
    getResource(lng, ns, key, options$$1) {
        return getResource.call(i18next, lng, ns, key, options$$1);
    }
    /**
     * @param {?} lng
     * @param {?} ns
     * @param {?} key
     * @param {?} value
     * @param {?} options
     * @return {?}
     */
    addResource(lng, ns, key, value, options$$1) {
        addResource.call(i18next, lng, ns, key, value, options$$1);
    }
    /**
     * @param {?} lng
     * @param {?} ns
     * @param {?} resources
     * @return {?}
     */
    addResources(lng, ns, resources) {
        addResources.call(i18next, lng, ns, resources);
    }
    /**
     * @param {?} lng
     * @param {?} ns
     * @param {?} resources
     * @param {?} deep
     * @param {?} overwrite
     * @return {?}
     */
    addResourceBundle(lng, ns, resources, deep, overwrite) {
        addResourceBundle.call(i18next, lng, ns, resources, deep, overwrite);
    }
    /**
     * @param {?} lng
     * @param {?} ns
     * @return {?}
     */
    hasResourceBundle(lng, ns) {
        return hasResourceBundle.call(i18next, lng, ns);
    }
    /**
     * @param {?} lng
     * @param {?} ns
     * @return {?}
     */
    getResourceBundle(lng, ns) {
        return getResourceBundle.call(i18next, lng, ns);
    }
    /**
     * @param {?} lng
     * @param {?} ns
     * @return {?}
     */
    removeResourceBundle(lng, ns) {
        removeResourceBundle.call(i18next, lng, ns);
    }
    /**
     * @return {?}
     */
    subscribeEvents() {
        on.call(i18next, 'initialized', options$$1 => {
            this.language = language;
            this.languages = languages;
            this.events.initialized.next(options$$1);
        });
        on.call(i18next, 'loaded', loaded => this.events.loaded.next(loaded));
        on.call(i18next, 'failedLoading', (lng, ns, msg) => this.events.failedLoading.next({ lng, ns, msg }));
        on.call(i18next, 'languageChanged', lng => {
            this.language = language;
            this.languages = languages;
            this.events.languageChanged.next(lng);
        });
        on.call(i18next, 'missingKey', (lngs, namespace, key, res) => this.events.missingKey.next({ lngs, namespace, key, res }));
        on.call(i18next, 'added', (lng, ns) => this.events.added.next({ lng, ns }));
        on.call(i18next, 'removed', (lng, ns) => this.events.removed.next({ lng, ns }));
    }
}
I18NextService.decorators = [
    { type: Injectable },
];
/** @nocollapse */
I18NextService.ctorParameters = () => [
    { type: undefined, decorators: [{ type: Inject, args: [I18NEXT_ERROR_HANDLING_STRATEGY,] }] }
];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class I18NextTitle extends Title {
    /**
     * @param {?} i18nextPipe
     * @param {?} doc
     */
    constructor(i18nextPipe, doc) {
        super(doc);
        this.i18nextPipe = i18nextPipe;
    }
    /**
     * @param {?} value
     * @return {?}
     */
    setTitle(value) {
        return super.setTitle(this.translate(value));
    }
    /**
     * @param {?} text
     * @return {?}
     */
    translate(text) {
        return this.i18nextPipe.transform(text, { format: 'cap' });
    }
}
I18NextTitle.decorators = [
    { type: Injectable },
];
/** @nocollapse */
I18NextTitle.ctorParameters = () => [
    { type: I18NextPipe },
    { type: undefined, decorators: [{ type: Inject, args: [DOCUMENT,] }] }
];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * @param {?} activatedRouteSnapshot
 * @param {?} routerStateSnapshot
 * @return {?}
 */
function resolver(activatedRouteSnapshot, routerStateSnapshot) {
    let /** @type {?} */ namespaces = [];
    namespaces = activatedRouteSnapshot.data && activatedRouteSnapshot.data.i18nextNamespaces || namespaces;
    return this.loadNamespaces(namespaces.filter(n => n));
}
/**
 * @param {?} i18next
 * @return {?}
 */
function i18nextNamespaceResolverFactory(i18next$$1) {
    return resolver.bind(i18next$$1);
}
class I18NextModule {
    /**
     * @param {?=} params
     * @return {?}
     */
    static forRoot(params = {}) {
        return {
            ngModule: I18NextModule,
            providers: [
                {
                    provide: I18NEXT_SERVICE,
                    useClass: I18NextService
                },
                {
                    provide: I18NEXT_ERROR_HANDLING_STRATEGY,
                    useClass: params.errorHandlingStrategy || NativeErrorHandlingStrategy
                },
                I18NextService,
                I18NextPipe,
                I18NextCapPipe,
                I18NextFormatPipe,
                I18NextTitle,
                {
                    provide: I18NEXT_NAMESPACE_RESOLVER,
                    useFactory: i18nextNamespaceResolverFactory,
                    deps: [I18NEXT_SERVICE]
                }
            ]
        };
    }
    /**
     * @param {?=} customFormat
     * @return {?}
     */
    static interpolationFormat(customFormat = null) {
        /**
         * @param {?} value
         * @param {?} format
         * @param {?} lng
         * @return {?}
         */
        function formatDelegate(value, format$$1, lng) {
            let /** @type {?} */ formatedValue = defaultInterpolationFormat(value, format$$1, lng);
            if (customFormat === null)
                return formatedValue;
            return customFormat(formatedValue, format$$1, lng);
        }
        return formatDelegate;
    }
}
I18NextModule.decorators = [
    { type: NgModule, args: [{
                providers: [
                    {
                        provide: I18NEXT_NAMESPACE,
                        useValue: ''
                    },
                    {
                        provide: I18NEXT_SCOPE,
                        useValue: ''
                    },
                    I18NextPipe,
                    I18NextCapPipe,
                    I18NextFormatPipe,
                    I18NextTitle
                ],
                declarations: [
                    I18NextPipe,
                    I18NextCapPipe,
                    I18NextFormatPipe
                ],
                exports: [
                    I18NextPipe,
                    I18NextCapPipe,
                    I18NextFormatPipe
                ]
            },] },
];
/**
 * @param {?} value
 * @param {?} format
 * @param {?=} lng
 * @return {?}
 */
function defaultInterpolationFormat(value, format$$1, lng = undefined) {
    if (!value)
        return value;
    switch (format$$1) {
        case 'upper':
        case 'uppercase':
            return value.toUpperCase();
        case 'lower':
        case 'lowercase':
            return value.toLowerCase();
        case 'cap':
        case 'capitalize':
            return value.charAt(0).toUpperCase() + value.slice(1);
        case null:
        case undefined:
        case 'none':
        default:
            return value;
    }
}

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * Generated bundle index. Do not edit.
 */

export { resolver, i18nextNamespaceResolverFactory, I18NextModule, defaultInterpolationFormat, I18NEXT_SCOPE, I18NEXT_NAMESPACE, I18NEXT_SERVICE, I18NEXT_NAMESPACE_RESOLVER, I18NEXT_ERROR_HANDLING_STRATEGY, I18NextPipe, I18NextCapPipe, I18NextFormatPipe, I18NextService, I18NextTitle, NativeErrorHandlingStrategy, StrictErrorHandlingStrategy };
//# sourceMappingURL=angular-i18next.js.map
