/**
 * DevExtreme (viz/chart_components/scroll_bar.js)
 * Version: 18.1.6
 * Build date: Mon Sep 03 2018
 *
 * Copyright (c) 2012 - 2018 Developer Express Inc. ALL RIGHTS RESERVED
 * Read about DevExtreme licensing here: https://js.devexpress.com/Licensing/
 */
"use strict";
var $ = require("../../core/renderer"),
    domAdapter = require("../../core/dom_adapter"),
    eventsEngine = require("../../events/core/events_engine"),
    eventUtils = require("../../events/utils"),
    extend = require("../../core/utils/extend").extend,
    MIN_SCROLL_BAR_SIZE = 2,
    translator2DModule = require("../translators/translator2d"),
    pointerEvents = require("../../events/pointer"),
    isDefined = require("../../core/utils/type").isDefined,
    _min = Math.min,
    _max = Math.max;
var ScrollBar = function(renderer, group) {
    this._translator = new translator2DModule.Translator2D({}, {}, {});
    this._scroll = renderer.rect().append(group);
    this._addEvents()
};

function _getXCoord(canvas, pos, offset, width) {
    var x = 0;
    if ("right" === pos) {
        x = canvas.width - canvas.right + offset
    } else {
        if ("left" === pos) {
            x = canvas.left - offset - width
        }
    }
    return x
}

function _getYCoord(canvas, pos, offset, width) {
    var y = 0;
    if ("top" === pos) {
        y = canvas.top - offset
    } else {
        if ("bottom" === pos) {
            y = canvas.height - canvas.bottom + width + offset
        }
    }
    return y
}
ScrollBar.prototype = {
    _addEvents: function() {
        var document = domAdapter.getDocument(),
            that = this,
            $scroll = $(that._scroll.element),
            startPosX = 0,
            startPosY = 0,
            scrollChangeHandler = function(e) {
                var dX = (startPosX - e.pageX) * that._scale,
                    dY = (startPosY - e.pageY) * that._scale;
                eventUtils.fireEvent({
                    type: "dxc-scroll-move",
                    originalEvent: e,
                    target: $scroll.get(0),
                    pointers: [{
                        pageX: startPosX + dX,
                        pageY: startPosY + dY
                    }]
                })
            };
        eventsEngine.on($scroll, pointerEvents.down, function(e) {
            startPosX = e.pageX;
            startPosY = e.pageY;
            eventUtils.fireEvent({
                type: "dxc-scroll-start",
                originalEvent: e,
                target: $scroll.get(0),
                pointers: [{
                    pageX: startPosX,
                    pageY: startPosY
                }]
            });
            eventsEngine.on(document, pointerEvents.move, scrollChangeHandler)
        });
        eventsEngine.on(document, pointerEvents.up, function() {
            eventsEngine.off(document, pointerEvents.move, scrollChangeHandler)
        })
    },
    update: function(options) {
        var that = this,
            position = options.position,
            isVertical = options.rotated,
            defaultPosition = isVertical ? "right" : "top",
            secondaryPosition = isVertical ? "left" : "bottom";
        if (position !== defaultPosition && position !== secondaryPosition) {
            position = defaultPosition
        }
        that._scroll.attr({
            rotate: !options.rotated ? -90 : 0,
            rotateX: 0,
            rotateY: 0,
            fill: options.color,
            width: options.width,
            opacity: options.opacity
        });
        that._layoutOptions = {
            width: options.width,
            offset: options.offset,
            vertical: isVertical,
            position: position
        };
        return that
    },
    init: function(range, stick) {
        var that = this;
        that._translateWithOffset = "discrete" === range.axisType && !stick && 1 || 0;
        that._translator.update(extend({}, range, {
            minVisible: null,
            maxVisible: null,
            visibleCategories: null
        }), that._canvas, {
            isHorizontal: !that._layoutOptions.vertical,
            stick: stick
        });
        return that
    },
    getOptions: function() {
        return this._layoutOptions
    },
    setPane: function(panes) {
        var pane, position = this._layoutOptions.position;
        if ("left" === position || "top" === position) {
            pane = panes[0]
        } else {
            pane = panes[panes.length - 1]
        }
        this.pane = pane.name;
        return this
    },
    updateSize: function(canvas) {
        this._canvas = extend({}, canvas);
        var options = this._layoutOptions,
            pos = options.position,
            offset = options.offset,
            width = options.width;
        this._scroll.attr({
            translateX: _getXCoord(canvas, pos, offset, width),
            translateY: _getYCoord(canvas, pos, offset, width)
        })
    },
    getMultipleAxesSpacing: function() {
        return 0
    },
    estimateMargins: function() {
        return this.getMargins()
    },
    getMargins: function() {
        var options = this._layoutOptions,
            margins = {
                left: 0,
                top: 0,
                right: 0,
                bottom: 0
            };
        margins[options.position] = options.width + options.offset;
        return margins
    },
    draw: function() {},
    shift: function() {},
    hideTitle: function() {},
    hideOuterElements: function() {},
    setPosition: function(min, max) {
        var that = this,
            translator = that._translator,
            minPoint = isDefined(min) ? translator.translate(min, -that._translateWithOffset) : translator.translate("canvas_position_start"),
            maxPoint = isDefined(max) ? translator.translate(max, that._translateWithOffset) : translator.translate("canvas_position_end");
        that._offset = _min(minPoint, maxPoint);
        that._scale = translator.getScale(min, max);
        that._applyPosition(_min(minPoint, maxPoint), _max(minPoint, maxPoint))
    },
    transform: function(translate, scale) {
        var translator = this._translator,
            x = translator.getCanvasVisibleArea().min,
            dx = x - (x * scale - translate),
            lx = this._offset + dx / (this._scale * scale);
        this._applyPosition(lx, lx + translator.canvasLength / (this._scale * scale))
    },
    dispose: function() {
        this._scroll.dispose();
        this._scroll = this._translator = null
    },
    _applyPosition: function(x1, x2) {
        var height, that = this,
            visibleArea = that._translator.getCanvasVisibleArea();
        x1 = _max(x1, visibleArea.min);
        x1 = _min(x1, visibleArea.max);
        x2 = _min(x2, visibleArea.max);
        x2 = _max(x2, visibleArea.min);
        height = Math.abs(x2 - x1);
        that._scroll.attr({
            y: x1,
            height: height < MIN_SCROLL_BAR_SIZE ? MIN_SCROLL_BAR_SIZE : height
        })
    }
};
exports.ScrollBar = ScrollBar;
