/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __values = (this && this.__values) || function (o) {
    var m = typeof Symbol === "function" && o[Symbol.iterator], i = 0;
    if (m) return m.call(o);
    return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
};
var __spread = (this && this.__spread) || function () {
    for (var ar = [], i = 0; i < arguments.length; i++) ar = ar.concat(__read(arguments[i]));
    return ar;
};
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("tsickle/src/googmodule", ["require", "exports", "path", "tsickle/src/transformer_util", "tsickle/src/typescript"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    var path = require("path");
    var transformer_util_1 = require("tsickle/src/transformer_util");
    var ts = require("tsickle/src/typescript");
    /**
     * Returns true if node is a property access of `child` on the identifier `parent`.
     */
    function isPropertyAccess(node, parent, child) {
        if (!ts.isPropertyAccessExpression(node))
            return false;
        return ts.isIdentifier(node.expression) && node.expression.escapedText === parent &&
            node.name.escapedText === child;
    }
    /** Returns true if expr is "module.exports = ...;". */
    function isModuleExportsAssignment(expr) {
        if (!ts.isBinaryExpression(expr.expression))
            return false;
        if (expr.expression.operatorToken.kind !== ts.SyntaxKind.EqualsToken)
            return false;
        return isPropertyAccess(expr.expression.left, 'module', 'exports');
    }
    /** Returns true if expr is "exports = ...;". */
    function isExportsAssignment(expr) {
        if (!ts.isBinaryExpression(expr.expression))
            return false;
        if (expr.expression.operatorToken.kind !== ts.SyntaxKind.EqualsToken)
            return false;
        return ts.isIdentifier(expr.expression.left) && expr.expression.left.text === 'exports';
    }
    /** isUseStrict returns true if node is a "use strict"; statement. */
    function isUseStrict(node) {
        if (node.kind !== ts.SyntaxKind.ExpressionStatement)
            return false;
        var exprStmt = node;
        var expr = exprStmt.expression;
        if (expr.kind !== ts.SyntaxKind.StringLiteral)
            return false;
        var literal = expr;
        return literal.text === 'use strict';
    }
    /**
     * TypeScript inserts the following code to mark ES moduels in CommonJS:
     *   Object.defineProperty(exports, "__esModule", { value: true });
     * This matches that code snippet.
     */
    function isEsModuleProperty(stmt) {
        // We're matching the explicit source text generated by the TS compiler.
        // Object.defineProperty(exports, "__esModule", { value: true });
        var expr = stmt.expression;
        if (!ts.isCallExpression(expr))
            return false;
        if (!isPropertyAccess(expr.expression, 'Object', 'defineProperty'))
            return false;
        if (expr.arguments.length !== 3)
            return false;
        var _a = __read(expr.arguments, 3), exp = _a[0], esM = _a[1], val = _a[2];
        if (!ts.isIdentifier(exp) || exp.escapedText !== 'exports')
            return false;
        if (!ts.isStringLiteral(esM) || esM.text !== '__esModule')
            return false;
        if (!ts.isObjectLiteralExpression(val) || val.properties.length !== 1)
            return false;
        var prop = val.properties[0];
        if (!ts.isPropertyAssignment(prop))
            return false;
        var ident = prop.name;
        if (!ident || !ts.isIdentifier(ident) || ident.text !== 'value')
            return false;
        return prop.initializer.kind === ts.SyntaxKind.TrueKeyword;
    }
    /**
     * Returns the string argument if call is of the form
     *   require('foo')
     */
    function extractRequire(call) {
        // Verify that the call is a call to require(...).
        if (call.expression.kind !== ts.SyntaxKind.Identifier)
            return null;
        var ident = call.expression;
        if (ident.escapedText !== 'require')
            return null;
        // Verify the call takes a single string argument and grab it.
        if (call.arguments.length !== 1)
            return null;
        var arg = call.arguments[0];
        if (arg.kind !== ts.SyntaxKind.StringLiteral)
            return null;
        return arg.text;
    }
    /** Creates a call expression corresponding to `goog.${methodName}(${literal})`. */
    function createGoogCall(methodName, literal) {
        return ts.createCall(ts.createPropertyAccess(ts.createIdentifier('goog'), methodName), undefined, [literal]);
    }
    /**
     * Extracts the namespace part of a goog: import URL, or returns null if the given import URL is not
     * a goog: import.
     *
     * For example, for `import 'goog:foo.Bar';`, returns `foo.Bar`.
     */
    function extractGoogNamespaceImport(tsImport) {
        if (tsImport.match(/^goog:/))
            return tsImport.substring('goog:'.length);
        return null;
    }
    exports.extractGoogNamespaceImport = extractGoogNamespaceImport;
    // Matches common extensions of TypeScript input filenames
    var TS_EXTENSIONS = /(\.ts|\.d\.ts|\.js|\.jsx|\.tsx)$/;
    /**
     * Convert from implicit `import {} from 'pkg'` to `import {} from 'pkg/index'.
     * TypeScript supports the shorthand, but not all ES6 module loaders do.
     * Workaround for https://github.com/Microsoft/TypeScript/issues/12597
     */
    function resolveIndexShorthand(_a, pathOfImportingFile, imported) {
        // The strategy taken here is to use ts.resolveModuleName() to resolve the import to
        // a specific path, and then if that path is different than the path that was
        // asked for, we assume it was an index import and construct a new import statement.
        //
        // We need to be careful about paths -- `pathOfImportingFile` is the absolute path to the
        // source file, while `imported` may be a path relative to that.
        var options = _a.options, host = _a.host;
        var resolved = ts.resolveModuleName(imported, pathOfImportingFile, options, host);
        if (!resolved || !resolved.resolvedModule)
            return imported;
        var requestedModule = imported.replace(TS_EXTENSIONS, '');
        var resolvedModule = resolved.resolvedModule.resolvedFileName.replace(TS_EXTENSIONS, '');
        // Check if the resolution went into node_modules.
        // Note that the ResolvedModule returned by resolveModuleName() has an
        // attribute isExternalLibraryImport that is documented with
        // "True if resolvedFileName comes from node_modules", but actually it is just
        // true if the absolute path includes node_modules, and is always true when
        // tsickle itself is under a directory named node_modules.
        var relativeResolved = path.relative(options.rootDir || '', resolvedModule);
        if (relativeResolved.indexOf('node_modules') !== -1) {
            // Imports into node_modules resolve through package.json and must be
            // specially handled by the loader anyway.  Return the input.
            return imported;
        }
        // Check if the resolution chose a different file than what was asked for.
        // Compare filenames, but don't use path.basename() because we want a file
        // name of '' if the requested path ends in a slash.
        var requestedFileName = requestedModule.substr(requestedModule.lastIndexOf('/'));
        var resolvedFileName = resolvedModule.substr(resolvedModule.lastIndexOf('/'));
        if (requestedFileName === resolvedFileName) {
            // It ended up at the same file as it started with, so we don't need
            // a rewrite anyway.
            return imported;
        }
        // If we get here, it seems that the import resolved to somewhere else.
        // Construct a new import path.  Note that it must be relative to the original
        // filename, and also that path.relative() randomly uses the cwd if any of
        // its arguments aren't themselves absolute.
        return './' +
            path.relative(path.dirname(pathOfImportingFile), resolvedModule).replace(path.sep, '/');
    }
    exports.resolveIndexShorthand = resolveIndexShorthand;
    /**
     * importPathToGoogNamespace converts a TS/ES module './import/path' into a goog.module compatible
     * namespace, handling regular imports and `goog:` namespace imports.
     */
    function importPathToGoogNamespace(host, file, tsImport) {
        var modName;
        var nsImport = extractGoogNamespaceImport(tsImport);
        if (nsImport !== null) {
            // This is a namespace import, of the form "goog:foo.bar".
            // Fix it to just "foo.bar".
            modName = nsImport;
        }
        else {
            if (host.convertIndexImportShorthand) {
                tsImport = resolveIndexShorthand(host, file.fileName, tsImport);
            }
            modName = host.pathToModuleName(file.fileName, tsImport);
        }
        return transformer_util_1.createSingleQuoteStringLiteral(modName);
    }
    /**
     * Replace "module.exports = ..." with just "exports = ...". Returns null if `expr` is not an
     * exports assignment.
     */
    function rewriteModuleExportsAssignment(expr) {
        if (!ts.isBinaryExpression(expr.expression))
            return null;
        if (expr.expression.operatorToken.kind !== ts.SyntaxKind.EqualsToken)
            return null;
        if (!isPropertyAccess(expr.expression.left, 'module', 'exports'))
            return null;
        return ts.setOriginalNode(ts.setTextRange(ts.createStatement(ts.createAssignment(ts.createIdentifier('exports'), expr.expression.right)), expr), expr);
    }
    /**
     * commonJsToGoogmoduleTransformer returns a transformer factory that converts TypeScript's CommonJS
     * module emit to Closure Compiler compatible goog.module and goog.require statements.
     */
    function commonJsToGoogmoduleTransformer(host, modulesManifest, typeChecker, diagnostics) {
        return function (context) {
            // TS' CommonJS processing uses onSubstituteNode to, at the very end of processing, substitute
            // "modulename.default" for default imports. We intercept the substitution here, check if it's a
            // .default access, then check if the original node (and thus original import) was from a goog:
            // module, and if so, replace with just the module name.
            var previousOnSubstituteNode = context.onSubstituteNode;
            context.enableSubstitution(ts.SyntaxKind.PropertyAccessExpression);
            context.onSubstituteNode = function (hint, node) {
                node = previousOnSubstituteNode(hint, node);
                if (!ts.isPropertyAccessExpression(node))
                    return node;
                if (node.name.text !== 'default')
                    return node;
                if (!ts.isIdentifier(node.expression))
                    return node;
                // Find the import declaration this node comes from.
                // This may be the original node, if the identifier was transformed from it.
                var orig = ts.getOriginalNode(node.expression);
                var importExportDecl;
                if (ts.isImportDeclaration(orig) || ts.isExportDeclaration(orig)) {
                    importExportDecl = orig;
                }
                else {
                    // Alternatively, we can try to find the declaration of the symbol. This only works for
                    // user-written .default accesses, the generated ones do not have a symbol associated as
                    // they are only produced in the CommonJS transformation, after type checking.
                    var sym = typeChecker.getSymbolAtLocation(node.expression);
                    if (!sym)
                        return node;
                    var decls = sym.getDeclarations();
                    if (!decls || !decls.length)
                        return node;
                    var decl = decls[0];
                    if (decl.parent && decl.parent.parent && ts.isImportDeclaration(decl.parent.parent)) {
                        importExportDecl = decl.parent.parent;
                    }
                    else {
                        return node;
                    }
                }
                // If the import declaration's URL is a "goog:..." style namespace, then all ".default"
                // accesses on it should be replaced with the symbol itself.
                // This allows referring to the module-level export of a "goog.module" or "goog.provide" as if
                // it was an ES6 default export.
                if (extractGoogNamespaceImport(importExportDecl.moduleSpecifier.text)) {
                    // Substitute "foo.default" with just "foo".
                    return node.expression;
                }
                return node;
            };
            return function (sf) {
                var e_1, _a, e_2, _b;
                // In TS2.9, transformers can receive Bundle objects, which this code cannot handle (given
                // that a bundle by definition cannot be a goog.module()). The cast through any is necessary
                // to remain compatible with earlier TS versions.
                // tslint:disable-next-line:no-any
                if (sf.kind !== ts.SyntaxKind.SourceFile)
                    return sf;
                var moduleVarCounter = 1;
                /**
                 * Creates a new unique variable to assign side effect imports into. This allows us to re-use
                 * the variable later on for other imports of the same namespace.
                 */
                function nextModuleVar() {
                    return "tsickle_module_" + moduleVarCounter++ + "_";
                }
                /**
                 * Maps goog.require namespaces to the variable name they are assigned into. E.g.:
                 *     var $varName = goog.require('$namespace'));
                 */
                var namespaceToModuleVarName = new Map();
                /**
                 * maybeCreateGoogRequire returns a `goog.require()` call for the given CommonJS `require`
                 * call. Returns null if `call` is not a CommonJS require.
                 */
                function maybeCreateGoogRequire(original, call, newIdent) {
                    var importedUrl = extractRequire(call);
                    if (!importedUrl)
                        return null;
                    var imp = importPathToGoogNamespace(host, sf, importedUrl);
                    modulesManifest.addReferencedModule(sf.fileName, imp.text);
                    var ident = namespaceToModuleVarName.get(imp.text);
                    var initializer;
                    if (!ident) {
                        namespaceToModuleVarName.set(imp.text, newIdent);
                        initializer = createGoogCall('require', imp);
                    }
                    else {
                        initializer = ident;
                    }
                    var varDecl = ts.createVariableDeclaration(newIdent, /* type */ undefined, initializer);
                    var newStmt = ts.createVariableStatement(
                    /* modifiers */ undefined, ts.createVariableDeclarationList([varDecl]));
                    return ts.setOriginalNode(ts.setTextRange(newStmt, original), original);
                }
                /**
                 * maybeRewriteRequireTslib rewrites a require('tslib') calls to goog.require('tslib'). It
                 * returns the input statement untouched if it does not match.
                 */
                function maybeRewriteRequireTslib(stmt) {
                    if (!ts.isExpressionStatement(stmt))
                        return stmt;
                    if (!ts.isCallExpression(stmt.expression))
                        return stmt;
                    var callExpr = stmt.expression;
                    if (!ts.isIdentifier(callExpr.expression) || callExpr.expression.text !== 'require') {
                        return stmt;
                    }
                    if (callExpr.arguments.length !== 1)
                        return stmt;
                    var arg = callExpr.arguments[0];
                    if (!ts.isStringLiteral(arg) || arg.text !== 'tslib')
                        return stmt;
                    return ts.setOriginalNode(ts.setTextRange(ts.createStatement(createGoogCall('require', arg)), stmt), stmt);
                }
                /**
                 * visitTopLevelStatement implements the main CommonJS to goog.module conversion. It visits a
                 * SourceFile level statement and adds a (possibly) transformed representation of it into
                 * statements. It adds at least one node per statement to statements.
                 *
                 * visitTopLevelStatement:
                 * - converts require() calls to goog.require() calls, with or w/o var assignment
                 * - removes "use strict"; and "Object.defineProperty(__esModule)" statements
                 * - converts module.exports assignments to just exports assignments
                 * - splits __exportStar() calls into require and export (this needs two statements)
                 * - makes sure to only import each namespace exactly once, and use variables later on
                 */
                function visitTopLevelStatement(statements, sf, node) {
                    // Handle each particular case by adding node to statements, then return.
                    // For unhandled cases, break to jump to the default handling below.
                    switch (node.kind) {
                        case ts.SyntaxKind.ExpressionStatement: {
                            var exprStmt = node;
                            // Check for "use strict" and certain Object.defineProperty and skip it if necessary.
                            if (isUseStrict(exprStmt) || isEsModuleProperty(exprStmt)) {
                                stmts.push(transformer_util_1.createNotEmittedStatementWithComments(sf, exprStmt));
                                return;
                            }
                            // Check for:
                            //   module.exports = ...;
                            var modExports = rewriteModuleExportsAssignment(exprStmt);
                            if (modExports) {
                                stmts.push(modExports);
                                return;
                            }
                            // Check for:
                            //   "require('foo');" (a require for its side effects)
                            var expr = exprStmt.expression;
                            if (!ts.isCallExpression(expr))
                                break;
                            var callExpr = expr;
                            // Handle export * in ES5 mode (in ES6 mode, export * is dereferenced already).
                            // export * creates either a pure top-level '__export(require(...))' or the imported
                            // version, 'tslib.__exportStar(require(...))'. The imported version is only substituted
                            // later on though, so appears as a plain "__exportStar" on the top level here.
                            var isExportStar = (ts.isIdentifier(expr.expression) && expr.expression.text === '__exportStar') ||
                                (ts.isIdentifier(expr.expression) && expr.expression.text === '__export');
                            if (isExportStar)
                                callExpr = expr.arguments[0];
                            var ident = ts.createIdentifier(nextModuleVar());
                            var require_1 = maybeCreateGoogRequire(exprStmt, callExpr, ident);
                            if (!require_1)
                                break;
                            statements.push(require_1);
                            // If this is an export star, split it up into the import (created by the maybe call
                            // above), and the export operation. This avoids a Closure complaint about non-top-level
                            // requires.
                            if (isExportStar) {
                                var args = [ident];
                                if (expr.arguments.length > 1)
                                    args.push(expr.arguments[1]);
                                statements.push(ts.createStatement(ts.createCall(expr.expression, undefined, args)));
                            }
                            return;
                        }
                        case ts.SyntaxKind.VariableStatement: {
                            // It's possibly of the form "var x = require(...);".
                            var varStmt = node;
                            // Verify it's a single decl (and not "var x = ..., y = ...;").
                            if (varStmt.declarationList.declarations.length !== 1)
                                break;
                            var decl = varStmt.declarationList.declarations[0];
                            // Grab the variable name (avoiding things like destructuring binds).
                            if (decl.name.kind !== ts.SyntaxKind.Identifier)
                                break;
                            if (!decl.initializer || !ts.isCallExpression(decl.initializer)) {
                                break;
                            }
                            var require_2 = maybeCreateGoogRequire(varStmt, decl.initializer, decl.name);
                            if (!require_2)
                                break;
                            statements.push(require_2);
                            return;
                        }
                        default:
                            break;
                    }
                    statements.push(node);
                }
                var moduleName = host.pathToModuleName('', sf.fileName);
                // Register the namespace this file provides.
                modulesManifest.addModule(sf.fileName, moduleName);
                // In JS transpilation mode, keep all CommonJS code, and only rewrite `require('tslib')` to
                // a goog.require().
                if (host.isJsTranspilation) {
                    var stmts_1 = [];
                    try {
                        for (var _c = __values(sf.statements), _d = _c.next(); !_d.done; _d = _c.next()) {
                            var stmt = _d.value;
                            stmts_1.push(maybeRewriteRequireTslib(stmt));
                        }
                    }
                    catch (e_1_1) { e_1 = { error: e_1_1 }; }
                    finally {
                        try {
                            if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
                        }
                        finally { if (e_1) throw e_1.error; }
                    }
                    return ts.updateSourceFileNode(sf, stmts_1);
                }
                // Convert each top level statement to goog.module.
                var stmts = [];
                try {
                    for (var _e = __values(sf.statements), _f = _e.next(); !_f.done; _f = _e.next()) {
                        var stmt = _f.value;
                        visitTopLevelStatement(stmts, sf, stmt);
                    }
                }
                catch (e_2_1) { e_2 = { error: e_2_1 }; }
                finally {
                    try {
                        if (_f && !_f.done && (_b = _e.return)) _b.call(_e);
                    }
                    finally { if (e_2) throw e_2.error; }
                }
                // Additional statements that will be prepended (goog.module call etc).
                var headerStmts = [];
                // Emit: goog.module('moduleName');
                var googModule = ts.createStatement(createGoogCall('module', transformer_util_1.createSingleQuoteStringLiteral(moduleName)));
                headerStmts.push(googModule);
                // Allow code to use `module.id` to discover its module URL, e.g. to resolve a template URL
                // against. Uses 'var', as this code is inserted in ES6 and ES5 modes. The following pattern
                // ensures closure doesn't throw an error in advanced optimizations mode.
                // var module = module || {id: 'path/to/module.ts'};
                var moduleId = host.fileNameToModuleId(sf.fileName);
                var moduleVarInitializer = ts.createBinary(ts.createIdentifier('module'), ts.SyntaxKind.BarBarToken, ts.createObjectLiteral([ts.createPropertyAssignment('id', transformer_util_1.createSingleQuoteStringLiteral(moduleId))]));
                var modAssign = ts.createVariableStatement(
                /* modifiers */ undefined, ts.createVariableDeclarationList([ts.createVariableDeclaration('module', /* type */ undefined, moduleVarInitializer)]));
                headerStmts.push(modAssign);
                if (!host.es5Mode) {
                    // The module=module assignment suppresses an unused variable warning which may trigger
                    // depending on the project's compilation flags.
                    headerStmts.push(ts.createStatement(ts.createAssignment(ts.createIdentifier('module'), ts.createIdentifier('module'))));
                    // The `exports = {}` serves as a default export to disable Closure Compiler's error
                    // checking
                    // for mutable exports. That's OK because TS compiler makes sure that consuming code always
                    // accesses exports through the module object, so mutable exports work.
                    // It is only inserted in ES6 because we strip `.default` accesses in ES5 mode, which breaks
                    // when assigning an `exports = {}` object and then later accessing it.
                    // However Closure bails if code later on assigns into exports directly, as we do if we have
                    // an "exports = " block, so skip emit if that's the case.
                    if (!sf.statements.find(function (s) { return ts.isExpressionStatement(s) &&
                        (isModuleExportsAssignment(s) || isExportsAssignment(s)); })) {
                        headerStmts.push(ts.createStatement(ts.createAssignment(ts.createIdentifier('exports'), ts.createObjectLiteral())));
                    }
                }
                // Insert goog.module() etc after any leading comments in the source file. The comments have
                // been converted to NotEmittedStatements by transformer_util, which this depends on.
                var insertionIdx = stmts.findIndex(function (s) { return s.kind !== ts.SyntaxKind.NotEmittedStatement; });
                if (insertionIdx === -1) {
                    stmts.push.apply(stmts, __spread(headerStmts));
                }
                else {
                    stmts.splice.apply(stmts, __spread([insertionIdx, 0], headerStmts));
                }
                return ts.updateSourceFileNode(sf, ts.setTextRange(ts.createNodeArray(stmts), sf.statements));
            };
        };
    }
    exports.commonJsToGoogmoduleTransformer = commonJsToGoogmoduleTransformer;
});
//# sourceMappingURL=data:application/json;base64,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