"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("../path");
const buffer_1 = require("./buffer");
const memory_1 = require("./memory");
const sync_1 = require("./sync");
class TestHost extends memory_1.SimpleMemoryHost {
    constructor(map = {}) {
        super();
        this._records = [];
        for (const filePath of Object.getOwnPropertyNames(map)) {
            this._write(path_1.normalize(filePath), buffer_1.stringToFileBuffer(map[filePath]));
        }
    }
    get records() {
        return [...this._records];
    }
    clearRecords() {
        this._records = [];
    }
    get files() {
        const sync = this.sync;
        function _visit(p) {
            return sync.list(p)
                .map(fragment => path_1.join(p, fragment))
                .reduce((files, path) => {
                if (sync.isDirectory(path)) {
                    return files.concat(_visit(path));
                }
                else {
                    return files.concat(path);
                }
            }, []);
        }
        return _visit(path_1.normalize('/'));
    }
    get sync() {
        if (!this._sync) {
            this._sync = new sync_1.SyncDelegateHost(this);
        }
        return this._sync;
    }
    clone() {
        const newHost = new TestHost();
        newHost._cache = new Map(this._cache);
        return newHost;
    }
    // Override parents functions to keep a record of all operators that were done.
    _write(path, content) {
        this._records.push({ kind: 'write', path });
        return super._write(path, content);
    }
    _read(path) {
        this._records.push({ kind: 'read', path });
        return super._read(path);
    }
    _delete(path) {
        this._records.push({ kind: 'delete', path });
        return super._delete(path);
    }
    _rename(from, to) {
        this._records.push({ kind: 'rename', from, to });
        return super._rename(from, to);
    }
    _list(path) {
        this._records.push({ kind: 'list', path });
        return super._list(path);
    }
    _exists(path) {
        this._records.push({ kind: 'exists', path });
        return super._exists(path);
    }
    _isDirectory(path) {
        this._records.push({ kind: 'isDirectory', path });
        return super._isDirectory(path);
    }
    _isFile(path) {
        this._records.push({ kind: 'isFile', path });
        return super._isFile(path);
    }
    _stat(path) {
        this._records.push({ kind: 'stat', path });
        return super._stat(path);
    }
    _watch(path, options) {
        this._records.push({ kind: 'watch', path });
        return super._watch(path, options);
    }
    $write(path, content) {
        return super._write(path_1.normalize(path), buffer_1.stringToFileBuffer(content));
    }
    $read(path) {
        return buffer_1.fileBufferToString(super._read(path_1.normalize(path)));
    }
    $list(path) {
        return super._list(path_1.normalize(path));
    }
    $exists(path) {
        return super._exists(path_1.normalize(path));
    }
    $isDirectory(path) {
        return super._isDirectory(path_1.normalize(path));
    }
    $isFile(path) {
        return super._isFile(path_1.normalize(path));
    }
}
exports.TestHost = TestHost;
//# sourceMappingURL=data:application/json;base64,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