"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ts = require("typescript");
function validateDiagnostics(diagnostics, strict) {
    // Print error diagnostics.
    const checkDiagnostics = (diagnostics) => {
        if (diagnostics && diagnostics.length > 0) {
            let errors = '';
            errors = errors + '\n' + ts.formatDiagnostics(diagnostics, {
                getCurrentDirectory: () => ts.sys.getCurrentDirectory(),
                getNewLine: () => ts.sys.newLine,
                getCanonicalFileName: (f) => f,
            });
            return errors;
        }
    };
    const hasError = diagnostics.some(diag => diag.category === ts.DiagnosticCategory.Error);
    if (hasError) {
        // Throw only if we're in strict mode, otherwise return original content.
        if (strict) {
            throw new Error(`
        TS failed with the following error messages:

        ${checkDiagnostics(diagnostics)}
      `);
        }
        else {
            return false;
        }
    }
    return true;
}
function transformJavascript(options) {
    const { content, getTransforms, emitSourceMap, inputFilePath, outputFilePath, strict, } = options;
    // Bail if there's no transform to do.
    if (getTransforms.length === 0) {
        return {
            content: null,
            sourceMap: null,
            emitSkipped: true,
        };
    }
    const allowFastPath = options.typeCheck === false && !emitSourceMap;
    const outputs = new Map();
    const tempFilename = 'bo-default-file.js';
    const tempSourceFile = ts.createSourceFile(tempFilename, content, ts.ScriptTarget.Latest, allowFastPath);
    const parseDiagnostics = tempSourceFile.parseDiagnostics;
    const tsOptions = {
        // We target latest so that there is no downleveling.
        target: ts.ScriptTarget.Latest,
        isolatedModules: true,
        suppressOutputPathCheck: true,
        allowNonTsExtensions: true,
        noLib: true,
        noResolve: true,
        sourceMap: emitSourceMap,
        inlineSources: emitSourceMap,
        inlineSourceMap: false,
    };
    if (allowFastPath && parseDiagnostics) {
        if (!validateDiagnostics(parseDiagnostics, strict)) {
            return {
                content: null,
                sourceMap: null,
                emitSkipped: true,
            };
        }
        const transforms = getTransforms.map((getTf) => getTf(undefined));
        const result = ts.transform(tempSourceFile, transforms, tsOptions);
        if (result.transformed.length === 0 || result.transformed[0] === tempSourceFile) {
            return {
                content: null,
                sourceMap: null,
                emitSkipped: true,
            };
        }
        const printer = ts.createPrinter(undefined, {
            onEmitNode: result.emitNodeWithNotification,
            substituteNode: result.substituteNode,
        });
        const output = printer.printFile(result.transformed[0]);
        result.dispose();
        return {
            content: output,
            sourceMap: null,
            emitSkipped: false,
        };
    }
    const host = {
        getSourceFile: (fileName) => {
            if (fileName !== tempFilename) {
                throw new Error(`File ${fileName} does not have a sourceFile.`);
            }
            return tempSourceFile;
        },
        getDefaultLibFileName: () => 'lib.d.ts',
        getCurrentDirectory: () => '',
        getDirectories: () => [],
        getCanonicalFileName: (fileName) => fileName,
        useCaseSensitiveFileNames: () => true,
        getNewLine: () => '\n',
        fileExists: (fileName) => fileName === tempFilename,
        readFile: (_fileName) => '',
        writeFile: (fileName, text) => outputs.set(fileName, text),
    };
    const program = ts.createProgram([tempFilename], tsOptions, host);
    const diagnostics = program.getSyntacticDiagnostics(tempSourceFile);
    if (!validateDiagnostics(diagnostics, strict)) {
        return {
            content: null,
            sourceMap: null,
            emitSkipped: true,
        };
    }
    // We need the checker inside transforms.
    const transforms = getTransforms.map((getTf) => getTf(program));
    program.emit(undefined, undefined, undefined, undefined, { before: transforms, after: [] });
    let transformedContent = outputs.get(tempFilename);
    if (!transformedContent) {
        return {
            content: null,
            sourceMap: null,
            emitSkipped: true,
        };
    }
    let sourceMap = null;
    const tsSourceMap = outputs.get(`${tempFilename}.map`);
    if (emitSourceMap && tsSourceMap) {
        const urlRegExp = /^\/\/# sourceMappingURL=[^\r\n]*/gm;
        sourceMap = JSON.parse(tsSourceMap);
        // Fix sourcemaps file references.
        if (outputFilePath) {
            sourceMap.file = outputFilePath;
            transformedContent = transformedContent.replace(urlRegExp, `//# sourceMappingURL=${sourceMap.file}.map\n`);
            if (inputFilePath) {
                sourceMap.sources = [inputFilePath];
            }
            else {
                sourceMap.sources = [''];
            }
        }
        else {
            // TODO: figure out if we should inline sources here.
            transformedContent = transformedContent.replace(urlRegExp, '');
            sourceMap.file = '';
            sourceMap.sources = [''];
        }
    }
    return {
        content: transformedContent,
        sourceMap,
        emitSkipped: false,
    };
}
exports.transformJavascript = transformJavascript;
//# sourceMappingURL=data:application/json;base64,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