"use strict";
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
// tslint:disable:no-global-tslint-disable no-any
const core_1 = require("@angular-devkit/core");
const yargsParser = require("yargs-parser");
const command_1 = require("../models/command");
const project_1 = require("../utilities/project");
// Based off https://en.wikipedia.org/wiki/Levenshtein_distance
// No optimization, really.
function levenshtein(a, b) {
    /* base case: empty strings */
    if (a.length == 0) {
        return b.length;
    }
    if (b.length == 0) {
        return a.length;
    }
    // Test if last characters of the strings match.
    const cost = a[a.length - 1] == b[b.length - 1] ? 0 : 1;
    /* return minimum of delete char from s, delete char from t, and delete char from both */
    return Math.min(levenshtein(a.slice(0, -1), b) + 1, levenshtein(a, b.slice(0, -1)) + 1, levenshtein(a.slice(0, -1), b.slice(0, -1)) + cost);
}
/**
 * Run a command.
 * @param commandMap Map of available commands.
 * @param args Raw unparsed arguments.
 * @param logger The logger to use.
 * @param context Execution context.
 */
function runCommand(commandMap, args, logger, context) {
    return __awaiter(this, void 0, void 0, function* () {
        // if not args supplied, just run the help command.
        if (!args || args.length === 0) {
            args = ['help'];
        }
        const rawOptions = yargsParser(args, { alias: { help: ['h'] }, boolean: ['help'] });
        let commandName = rawOptions._[0];
        // remove the command name
        rawOptions._ = rawOptions._.slice(1);
        const executionScope = project_1.insideProject()
            ? command_1.CommandScope.inProject
            : command_1.CommandScope.outsideProject;
        let Cmd;
        Cmd = findCommand(commandMap, commandName);
        if (!Cmd && !commandName && (rawOptions.v || rawOptions.version)) {
            commandName = 'version';
            Cmd = findCommand(commandMap, commandName);
        }
        if (!Cmd && rawOptions.help) {
            commandName = 'help';
            Cmd = findCommand(commandMap, commandName);
        }
        if (!Cmd) {
            const commandsDistance = {};
            const allCommands = listAllCommandNames(commandMap).sort((a, b) => {
                if (!(a in commandsDistance)) {
                    commandsDistance[a] = levenshtein(a, commandName);
                }
                if (!(b in commandsDistance)) {
                    commandsDistance[b] = levenshtein(b, commandName);
                }
                return commandsDistance[a] - commandsDistance[b];
            });
            logger.error(core_1.tags.stripIndent `
        The specified command ("${commandName}") is invalid. For a list of available options,
        run "ng help".

        Did you mean "${allCommands[0]}"?
    `);
            return 1;
        }
        const command = new Cmd(context, logger);
        args = yield command.initializeRaw(args);
        let options = parseOptions(args, command.options, command.arguments, command.argStrategy);
        yield command.initialize(options);
        options = parseOptions(args, command.options, command.arguments, command.argStrategy);
        if (commandName === 'help') {
            options.commandMap = commandMap;
        }
        if (options.help) {
            command.printHelp(options);
            return;
        }
        else {
            if (Cmd.scope !== undefined && Cmd.scope !== command_1.CommandScope.everywhere) {
                if (Cmd.scope !== executionScope) {
                    let errorMessage;
                    if (Cmd.scope === command_1.CommandScope.inProject) {
                        errorMessage = `This command can only be run inside of a CLI project.`;
                    }
                    else {
                        errorMessage = `This command can not be run inside of a CLI project.`;
                    }
                    logger.fatal(errorMessage);
                    return 1;
                }
                if (Cmd.scope === command_1.CommandScope.inProject) {
                    if (!context.project.configFile) {
                        logger.fatal('Invalid project: missing workspace file.');
                        return 1;
                    }
                    if (['.angular-cli.json', 'angular-cli.json'].includes(context.project.configFile)) {
                        // --------------------------------------------------------------------------------
                        // If changing this message, please update the same message in
                        // `packages/@angular/cli/bin/ng-update-message.js`
                        const message = core_1.tags.stripIndent `
            The Angular CLI configuration format has been changed, and your existing configuration
            can be updated automatically by running the following command:

              ng update @angular/cli
          `;
                        logger.warn(message);
                        return 1;
                    }
                }
            }
            delete options.h;
            delete options.help;
            const isValid = yield command.validate(options);
            if (!isValid) {
                logger.fatal(`Validation error. Invalid command`);
                return 1;
            }
            return command.run(options);
        }
    });
}
exports.runCommand = runCommand;
function parseOptions(args, cmdOpts, commandArguments, argStrategy) {
    const parser = yargsParser;
    const aliases = cmdOpts.concat()
        .filter(o => o.aliases && o.aliases.length > 0)
        .reduce((aliases, opt) => {
        aliases[opt.name] = (opt.aliases || [])
            .filter(a => a.length === 1);
        return aliases;
    }, {});
    const booleans = cmdOpts
        .filter(o => o.type && o.type === Boolean)
        .map(o => o.name);
    const defaults = cmdOpts
        .filter(o => o.default !== undefined || booleans.indexOf(o.name) !== -1)
        .reduce((defaults, opt) => {
        defaults[opt.name] = opt.default;
        return defaults;
    }, {});
    const strings = cmdOpts
        .filter(o => o.type === String)
        .map(o => o.name);
    const numbers = cmdOpts
        .filter(o => o.type === Number)
        .map(o => o.name);
    aliases.help = ['h'];
    booleans.push('help');
    const yargsOptions = {
        alias: aliases,
        boolean: booleans,
        default: defaults,
        string: strings,
        number: numbers,
    };
    const parsedOptions = parser(args, yargsOptions);
    // Remove aliases.
    cmdOpts
        .filter(o => o.aliases && o.aliases.length > 0)
        .map(o => o.aliases)
        .reduce((allAliases, aliases) => {
        return allAliases.concat([...aliases]);
    }, [])
        .forEach((alias) => {
        delete parsedOptions[alias];
    });
    // Remove undefined booleans
    booleans
        .filter(b => parsedOptions[b] === undefined)
        .map(b => core_1.strings.camelize(b))
        .forEach(b => delete parsedOptions[b]);
    // remove options with dashes.
    Object.keys(parsedOptions)
        .filter(key => key.indexOf('-') !== -1)
        .forEach(key => delete parsedOptions[key]);
    // remove the command name
    parsedOptions._ = parsedOptions._.slice(1);
    switch (argStrategy) {
        case command_1.ArgumentStrategy.MapToOptions:
            parsedOptions._.forEach((value, index) => {
                const arg = commandArguments[index];
                if (arg) {
                    parsedOptions[arg] = value;
                }
            });
            delete parsedOptions._;
            break;
    }
    return parsedOptions;
}
exports.parseOptions = parseOptions;
// Find a command.
function findCommand(map, name) {
    let Cmd = map[name];
    if (!Cmd) {
        // find command via aliases
        Cmd = Object.keys(map)
            .filter(key => {
            if (!map[key].aliases) {
                return false;
            }
            const foundAlias = map[key].aliases
                .filter((alias) => alias === name);
            return foundAlias.length > 0;
        })
            .map((key) => {
            return map[key];
        })[0];
    }
    if (!Cmd) {
        return null;
    }
    return Cmd;
}
function listAllCommandNames(map) {
    return Object.keys(map).concat(Object.keys(map)
        .reduce((acc, key) => {
        if (!map[key].aliases) {
            return acc;
        }
        return acc.concat(map[key].aliases);
    }, []));
}
//# sourceMappingURL=data:application/json;base64,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