/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { CssToken, CssTokenType } from './css_lexer';
export var BlockType;
(function (BlockType) {
    BlockType[BlockType["Import"] = 0] = "Import";
    BlockType[BlockType["Charset"] = 1] = "Charset";
    BlockType[BlockType["Namespace"] = 2] = "Namespace";
    BlockType[BlockType["Supports"] = 3] = "Supports";
    BlockType[BlockType["Keyframes"] = 4] = "Keyframes";
    BlockType[BlockType["MediaQuery"] = 5] = "MediaQuery";
    BlockType[BlockType["Selector"] = 6] = "Selector";
    BlockType[BlockType["FontFace"] = 7] = "FontFace";
    BlockType[BlockType["Page"] = 8] = "Page";
    BlockType[BlockType["Document"] = 9] = "Document";
    BlockType[BlockType["Viewport"] = 10] = "Viewport";
    BlockType[BlockType["Unsupported"] = 11] = "Unsupported";
})(BlockType || (BlockType = {}));
export class CssAst {
    constructor(location) {
        this.location = location;
    }
    get start() { return this.location.start; }
    get end() { return this.location.end; }
}
export class CssStyleValueAst extends CssAst {
    constructor(location, tokens, strValue) {
        super(location);
        this.tokens = tokens;
        this.strValue = strValue;
    }
    visit(visitor, context) { return visitor.visitCssValue(this); }
}
export class CssRuleAst extends CssAst {
    constructor(location) { super(location); }
}
export class CssBlockRuleAst extends CssRuleAst {
    constructor(location, type, block, name = null) {
        super(location);
        this.location = location;
        this.type = type;
        this.block = block;
        this.name = name;
    }
    visit(visitor, context) {
        return visitor.visitCssBlock(this.block, context);
    }
}
export class CssKeyframeRuleAst extends CssBlockRuleAst {
    constructor(location, name, block) {
        super(location, BlockType.Keyframes, block, name);
    }
    visit(visitor, context) {
        return visitor.visitCssKeyframeRule(this, context);
    }
}
export class CssKeyframeDefinitionAst extends CssBlockRuleAst {
    constructor(location, steps, block) {
        super(location, BlockType.Keyframes, block, mergeTokens(steps, ','));
        this.steps = steps;
    }
    visit(visitor, context) {
        return visitor.visitCssKeyframeDefinition(this, context);
    }
}
export class CssBlockDefinitionRuleAst extends CssBlockRuleAst {
    constructor(location, strValue, type, query, block) {
        super(location, type, block);
        this.strValue = strValue;
        this.query = query;
        const firstCssToken = query.tokens[0];
        this.name = new CssToken(firstCssToken.index, firstCssToken.column, firstCssToken.line, CssTokenType.Identifier, this.strValue);
    }
    visit(visitor, context) {
        return visitor.visitCssBlock(this.block, context);
    }
}
export class CssMediaQueryRuleAst extends CssBlockDefinitionRuleAst {
    constructor(location, strValue, query, block) {
        super(location, strValue, BlockType.MediaQuery, query, block);
    }
    visit(visitor, context) {
        return visitor.visitCssMediaQueryRule(this, context);
    }
}
export class CssAtRulePredicateAst extends CssAst {
    constructor(location, strValue, tokens) {
        super(location);
        this.strValue = strValue;
        this.tokens = tokens;
    }
    visit(visitor, context) {
        return visitor.visitCssAtRulePredicate(this, context);
    }
}
export class CssInlineRuleAst extends CssRuleAst {
    constructor(location, type, value) {
        super(location);
        this.type = type;
        this.value = value;
    }
    visit(visitor, context) {
        return visitor.visitCssInlineRule(this, context);
    }
}
export class CssSelectorRuleAst extends CssBlockRuleAst {
    constructor(location, selectors, block) {
        super(location, BlockType.Selector, block);
        this.selectors = selectors;
        this.strValue = selectors.map(selector => selector.strValue).join(',');
    }
    visit(visitor, context) {
        return visitor.visitCssSelectorRule(this, context);
    }
}
export class CssDefinitionAst extends CssAst {
    constructor(location, property, value) {
        super(location);
        this.property = property;
        this.value = value;
    }
    visit(visitor, context) {
        return visitor.visitCssDefinition(this, context);
    }
}
export class CssSelectorPartAst extends CssAst {
    constructor(location) { super(location); }
}
export class CssSelectorAst extends CssSelectorPartAst {
    constructor(location, selectorParts) {
        super(location);
        this.selectorParts = selectorParts;
        this.strValue = selectorParts.map(part => part.strValue).join('');
    }
    visit(visitor, context) {
        return visitor.visitCssSelector(this, context);
    }
}
export class CssSimpleSelectorAst extends CssSelectorPartAst {
    constructor(location, tokens, strValue, pseudoSelectors, operator) {
        super(location);
        this.tokens = tokens;
        this.strValue = strValue;
        this.pseudoSelectors = pseudoSelectors;
        this.operator = operator;
    }
    visit(visitor, context) {
        return visitor.visitCssSimpleSelector(this, context);
    }
}
export class CssPseudoSelectorAst extends CssSelectorPartAst {
    constructor(location, strValue, name, tokens, innerSelectors) {
        super(location);
        this.strValue = strValue;
        this.name = name;
        this.tokens = tokens;
        this.innerSelectors = innerSelectors;
    }
    visit(visitor, context) {
        return visitor.visitCssPseudoSelector(this, context);
    }
}
export class CssBlockAst extends CssAst {
    constructor(location, entries) {
        super(location);
        this.entries = entries;
    }
    visit(visitor, context) { return visitor.visitCssBlock(this, context); }
}
/*
 a style block is different from a standard block because it contains
 css prop:value definitions. A regular block can contain a list of Ast entries.
 */
export class CssStylesBlockAst extends CssBlockAst {
    constructor(location, definitions) {
        super(location, definitions);
        this.definitions = definitions;
    }
    visit(visitor, context) {
        return visitor.visitCssStylesBlock(this, context);
    }
}
export class CssStyleSheetAst extends CssAst {
    constructor(location, rules) {
        super(location);
        this.rules = rules;
    }
    visit(visitor, context) {
        return visitor.visitCssStyleSheet(this, context);
    }
}
export class CssUnknownRuleAst extends CssRuleAst {
    constructor(location, ruleName, tokens) {
        super(location);
        this.ruleName = ruleName;
        this.tokens = tokens;
    }
    visit(visitor, context) {
        return visitor.visitCssUnknownRule(this, context);
    }
}
export class CssUnknownTokenListAst extends CssRuleAst {
    constructor(location, name, tokens) {
        super(location);
        this.name = name;
        this.tokens = tokens;
    }
    visit(visitor, context) {
        return visitor.visitCssUnknownTokenList(this, context);
    }
}
export function mergeTokens(tokens, separator = '') {
    const mainToken = tokens[0];
    let str = mainToken.strValue;
    for (let i = 1; i < tokens.length; i++) {
        str += separator + tokens[i].strValue;
    }
    return new CssToken(mainToken.index, mainToken.column, mainToken.line, mainToken.type, str);
}
//# sourceMappingURL=data:application/json;base64,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