/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { resolveRendererType2 } from '../view/util';
/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ngComponentDef = defineComponent({
 *     ...
 *   });
 * }
 * ```
 */
export function defineComponent(componentDefinition) {
    var type = componentDefinition.type;
    var pipeTypes = componentDefinition.pipes;
    var directiveTypes = componentDefinition.directives;
    var declaredInputs = {};
    var def = {
        type: type,
        diPublic: null,
        factory: componentDefinition.factory,
        template: componentDefinition.template || null,
        hostBindings: componentDefinition.hostBindings || null,
        contentQueries: componentDefinition.contentQueries || null,
        contentQueriesRefresh: componentDefinition.contentQueriesRefresh || null,
        attributes: componentDefinition.attributes || null,
        inputs: invertObject(componentDefinition.inputs, declaredInputs),
        declaredInputs: declaredInputs,
        outputs: invertObject(componentDefinition.outputs),
        rendererType: resolveRendererType2(componentDefinition.rendererType) || null,
        exportAs: componentDefinition.exportAs || null,
        onInit: type.prototype.ngOnInit || null,
        doCheck: type.prototype.ngDoCheck || null,
        afterContentInit: type.prototype.ngAfterContentInit || null,
        afterContentChecked: type.prototype.ngAfterContentChecked || null,
        afterViewInit: type.prototype.ngAfterViewInit || null,
        afterViewChecked: type.prototype.ngAfterViewChecked || null,
        onDestroy: type.prototype.ngOnDestroy || null,
        onPush: componentDefinition.changeDetection === ChangeDetectionStrategy.OnPush,
        directiveDefs: directiveTypes ?
            function () { return (typeof directiveTypes === 'function' ? directiveTypes() : directiveTypes)
                .map(extractDirectiveDef); } :
            null,
        pipeDefs: pipeTypes ?
            function () { return (typeof pipeTypes === 'function' ? pipeTypes() : pipeTypes).map(extractPipeDef); } :
            null,
        selectors: componentDefinition.selectors,
        viewQuery: componentDefinition.viewQuery || null,
        features: componentDefinition.features || null,
    };
    var feature = componentDefinition.features;
    feature && feature.forEach(function (fn) { return fn(def); });
    return def;
}
export function extractDirectiveDef(type) {
    var def = type.ngComponentDef || type.ngDirectiveDef;
    if (ngDevMode && !def) {
        throw new Error("'" + type.name + "' is neither 'ComponentType' or 'DirectiveType'.");
    }
    return def;
}
export function extractPipeDef(type) {
    var def = type.ngPipeDef;
    if (ngDevMode && !def) {
        throw new Error("'" + type.name + "' is not a 'PipeType'.");
    }
    return def;
}
export function defineNgModule(def) {
    var res = {
        type: def.type,
        bootstrap: def.bootstrap || [],
        declarations: def.declarations || [],
        imports: def.imports || [],
        exports: def.exports || [],
        transitiveCompileScopes: null,
    };
    return res;
}
var EMPTY = {};
/**
 * Inverts an inputs or outputs lookup such that the keys, which were the
 * minified keys, are part of the values, and the values are parsed so that
 * the publicName of the property is the new key
 *
 * e.g. for
 *
 * ```
 * class Comp {
 *   @Input()
 *   propName1: string;
 *
 *   @Input('publicName')
 *   propName2: number;
 * }
 * ```
 *
 * will be serialized as
 *
 * ```
 * {
 *   a0: 'propName1',
 *   b1: ['publicName', 'propName2'],
 * }
 * ```
 *
 * becomes
 *
 * ```
 * {
 *  'propName1': 'a0',
 *  'publicName': 'b1'
 * }
 * ```
 *
 * Optionally the function can take `secondary` which will result in:
 *
 * ```
 * {
 *  'propName1': 'a0',
 *  'propName2': 'b1'
 * }
 * ```
 *

 */
function invertObject(obj, secondary) {
    if (obj == null)
        return EMPTY;
    var newLookup = {};
    for (var minifiedKey in obj) {
        if (obj.hasOwnProperty(minifiedKey)) {
            var publicName = obj[minifiedKey];
            var declaredName = publicName;
            if (Array.isArray(publicName)) {
                declaredName = publicName[1];
                publicName = publicName[0];
            }
            newLookup[publicName] = minifiedKey;
            if (secondary) {
                (secondary[declaredName] = minifiedKey);
            }
        }
    }
    return newLookup;
}
/**
 * Create a directive definition object.
 *
 * # Example
 * ```
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ngDirectiveDef = defineDirective({
 *     ...
 *   });
 * }
 * ```
 */
export var defineDirective = defineComponent;
/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ngPipeDef = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @param pipeDef Pipe definition generated by the compiler
 */
export function definePipe(pipeDef) {
    return {
        name: pipeDef.name,
        factory: pipeDef.factory,
        pure: pipeDef.pure !== false,
        onDestroy: pipeDef.type.prototype.ngOnDestroy || null
    };
}

//# sourceMappingURL=data:application/json;base64,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