/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,missingOverride,missingReturn,unusedPrivateMembers,uselessCode} checked by tsc
 */
import { Component, Input, ViewChild } from '@angular/core';
import { NgbPopover } from '@ng-bootstrap/ng-bootstrap';
import { I18NextPipe } from 'angular-i18next';
import { DxDataGridComponent } from 'devextreme-angular';
import { filter, find, findIndex, forEach, isEqual, merge, remove, sortBy } from 'lodash-es';
export class ColumnChooserComponent {
    /**
     * @param {?} i18NextPipe
     */
    constructor(i18NextPipe) {
        this.i18NextPipe = i18NextPipe;
        this._availableColumns = [];
        this._columnsToDisplay = [];
        this.isInvalid = false;
        this.value = [];
    }
    /**
     * @return {?}
     */
    ngOnInit() {
        this.dataGrid.onContentReady.subscribe((/**
         * @param {?} data
         * @return {?}
         */
        (data) => {
            this.gridInstance = data.component.instance();
            /** @type {?} */
            const newColumns = this.gridInstance.state().columns;
            /** @type {?} */
            const areColumnsEqual = isEqual(newColumns, this.value);
            if (!areColumnsEqual) {
                // Update columns and refresh columns to display only if columns have changed
                this.value = newColumns;
                this.setColumnsToDisplay();
            }
        }));
    }
    /**
     * @return {?}
     */
    setColumnsToDisplay() {
        /** @type {?} */
        const dxGridColumns = this.getAllColumnsOnDxGrid();
        // Remove all columns that are not shown in the column chooser
        /** @type {?} */
        const shownColumns = filter(dxGridColumns, (/**
         * @param {?} col
         * @return {?}
         */
        (col) => col.showInColumnChooser));
        // updating dxGridColumns with the value of column chooser.
        shownColumns.forEach((/**
         * @param {?} origCol
         * @return {?}
         */
        (origCol) => {
            // find if a column in dxGridColumns also exist in value. Columns should have same dataField.
            /** @type {?} */
            const columnInValue = find(this.value, (/**
             * @param {?} valueCol
             * @return {?}
             */
            (valueCol) => origCol.dataField === valueCol.dataField));
            origCol = merge(origCol, columnInValue);
        }));
        // we set columnsToDisplay for visible columns and sort them in order of visibleIndex.
        this._columnsToDisplay = sortBy(filter(shownColumns, (/**
         * @param {?} col
         * @return {?}
         */
        (col) => col.visible)), [(/**
             * @param {?} o
             * @return {?}
             */
            function (o) { return o.visibleIndex; })]);
        // available columns are all columns with caption but are not visible.
        this._availableColumns = filter(shownColumns, (/**
         * @param {?} col
         * @return {?}
         */
        (col) => !col.visible));
    }
    /**
     * @return {?}
     */
    togglePopover() {
        this.ngbPopover.toggle();
    }
    /**
     * @return {?}
     */
    closePopover() {
        this.ngbPopover.close();
    }
    /**
     * @param {?} selection
     * @return {?}
     */
    toggleSelection(selection) {
        /** @type {?} */
        const isSel = selection.isSelected || null;
        if (!isSel) {
            this.selectOption(selection);
        }
        else {
            this.deselectColumn(selection);
        }
    }
    /**
     * @param {?} selIn
     * @return {?}
     */
    deselectColumn(selIn) {
        delete selIn.isSelected;
    }
    /**
     * @param {?} selIn
     * @return {?}
     */
    selectOption(selIn) {
        selIn.isSelected = true;
    }
    /**
     * @return {?}
     */
    deSelectAvailableColumns() {
        // Deselect means I am taking a selected column and moving it to available the [ < ]
        /** @type {?} */
        const columnsSelectedToHide = remove(this._columnsToDisplay, { isSelected: true });
        [].splice.apply(this._availableColumns, [this._availableColumns.length, 0].concat(columnsSelectedToHide));
        this.removeIsSelectedFromAllColumns(this._availableColumns);
    }
    /**
     * @param {?} availableColumns
     * @return {?}
     */
    removeIsSelectedFromAllColumns(availableColumns) {
        forEach(availableColumns, (/**
         * @param {?} column
         * @return {?}
         */
        (column) => {
            this.deselectColumn(column);
        }));
    }
    /**
     * @return {?}
     */
    selectAvailableColumns() {
        // Find my self in the schema availableColumns so I Can be put back where I belong.
        /** @type {?} */
        const columnsSelectedToShow = remove(this._availableColumns, { isSelected: true });
        [].splice.apply(this._columnsToDisplay, [this._columnsToDisplay.length, 0].concat(columnsSelectedToShow));
        if (this._columnsToDisplay.length > 0) {
            this.isInvalid = false;
        }
        this.removeIsSelectedFromAllColumns(this._columnsToDisplay);
    }
    /**
     * @param {?} fromIndex
     * @param {?} toIndex
     * @return {?}
     */
    moveSelection(fromIndex, toIndex) {
        /** @type {?} */
        const sels = this._columnsToDisplay;
        /** @type {?} */
        const element = sels[fromIndex];
        if (fromIndex === 0 && toIndex === -1) {
            return;
        }
        sels.splice(fromIndex, 1);
        sels.splice(toIndex, 0, element);
    }
    /**
     * @return {?}
     */
    moveColumnsUp() {
        /** @type {?} */
        const selectedAvailableColumns = filter(this._columnsToDisplay, { isSelected: true });
        forEach(selectedAvailableColumns, (/**
         * @param {?} optionToMove
         * @return {?}
         */
        (optionToMove) => {
            /** @type {?} */
            const idx = findIndex(this._columnsToDisplay, { dataField: optionToMove.dataField });
            if (idx > 0) {
                // toIndex will be one greater than currentIndex.
                /** @type {?} */
                let toIndex = idx - 1;
                /** @type {?} */
                let doUpdate = false;
                // toIndex has to be greater than -1.
                while (toIndex > -1) {
                    /** @type {?} */
                    const toElement = this._columnsToDisplay[toIndex];
                    // find element at toIndex and see if its fixed or not (i.e) cannot be reordered/hidden and it should not be selected.
                    // all selected elements are handled on their own.
                    if (toElement.allowReordering && !toElement.isSelected) {
                        toIndex = doUpdate ? toIndex + 1 : toIndex;
                        this.moveSelection(idx, toIndex);
                        break;
                    }
                    else {
                        // else, we decrement the toIndex and move-on.
                        toIndex = toIndex - 1;
                        doUpdate = true;
                    }
                }
            }
        }));
    }
    /**
     * @return {?}
     */
    moveColumnsDown() {
        /** @type {?} */
        const selectedAvailableColumns = filter(this._columnsToDisplay, { isSelected: true }).reverse();
        forEach(selectedAvailableColumns, (/**
         * @param {?} optionToMove
         * @return {?}
         */
        (optionToMove) => {
            // find index of selected column in columnsToDisplay.
            /** @type {?} */
            const idx = findIndex(this._columnsToDisplay, { dataField: optionToMove.dataField });
            if (idx < this._columnsToDisplay.length - 1) {
                // toIndex will be one greater than currentIndex.
                // we also have a flag that tells us if we need to update toIndex or not.
                /** @type {?} */
                let toIndex = idx + 1;
                /** @type {?} */
                let doUpdate = false;
                // we loop over till we reach the last index of columnsToDisplay as thats the max we can go down.
                while (toIndex < this._columnsToDisplay.length) {
                    // find element at toIndex and see if its fixed or not (i.e) cannot be reordered/hidden.
                    /** @type {?} */
                    const toElement = this._columnsToDisplay[toIndex];
                    // if next column can be reordered, we jump to next to next spot.
                    if (toElement.allowReordering) {
                        // we move the selection.
                        toIndex = doUpdate ? toIndex - 1 : toIndex;
                        this.moveSelection(idx, toIndex);
                        break;
                    }
                    else {
                        // else, we increment the toIndex and move on.
                        toIndex = toIndex + 1;
                        doUpdate = true;
                    }
                }
            }
        }));
    }
    /**
     * @return {?}
     */
    cancel() {
        this.setColumnsToDisplay();
        this.removeIsSelectedFromAllColumns(this._columnsToDisplay);
        this.removeIsSelectedFromAllColumns(this._availableColumns);
        this.closePopover();
    }
    /**
     * @return {?}
     */
    apply() {
        if (this._columnsToDisplay.length == 0) {
            // show error msg
            this.isInvalid = true;
            // dont close the popover
            return;
        }
        this.removeIsSelectedFromAllColumns(this._columnsToDisplay);
        this.removeIsSelectedFromAllColumns(this._availableColumns);
        this.updateIndexAndVisibilityOfColumns();
        this.closePopover();
    }
    /**
     * Method to update the index and visibility of grid columns and set it on the grid.
     * @return {?}
     */
    updateIndexAndVisibilityOfColumns() {
        /** @type {?} */
        const gridInstance = this.gridInstance;
        this._columnsToDisplay.forEach((/**
         * @param {?} col
         * @param {?} index
         * @return {?}
         */
        (col, index) => {
            col.visibleIndex = index;
        }));
        gridInstance.beginUpdate();
        // _columnsToDisplay are columns on right side of column chooser, columns that are shown.
        // we loop over all _columnsToDisplay and set visible to true, and update visibleIndex.
        this._columnsToDisplay.forEach((/**
         * @param {?} col
         * @param {?} visibleIndex
         * @return {?}
         */
        (col, visibleIndex) => {
            gridInstance.columnOption(col.dataField, {
                visible: true,
                visibleIndex,
            });
        }));
        // _availableColumns are columns on left side of column chooser, columns that are hidden.
        // we loop over all availableColumns and set visible to false.
        this._availableColumns.forEach((/**
         * @param {?} __0
         * @return {?}
         */
        ({ dataField }) => {
            gridInstance.columnOption(dataField, {
                visible: false,
                visibleIndex: undefined,
            });
        }));
        gridInstance.endUpdate();
    }
    /**
     * We find all columns on dx grid. Instead of using a method available on gridInstance directly, we use this way because
     * gridInstance.columnOption gives a complete column object (including default column settings),
     * whereas column object returned by gridInstance.option('columnms') only contains properties you explicitly set.
     * gridInstance.getVisibleColumns() method gives us the complete object but it only works for visible columns.
     * @private
     * @return {?}
     */
    getAllColumnsOnDxGrid() {
        /** @type {?} */
        const gridInstance = this.gridInstance;
        /** @type {?} */
        const columnsViaOptions = gridInstance && gridInstance.state().columns;
        if (columnsViaOptions) {
            return columnsViaOptions.map((/**
             * @param {?} __0
             * @return {?}
             */
            ({ dataField }) => gridInstance.columnOption(dataField)));
        }
        else {
            return;
        }
    }
}
ColumnChooserComponent.decorators = [
    { type: Component, args: [{
                selector: 'nwf-grid-column-chooser',
                template: "<button [ngbPopover]=\"popContent\" [autoClose]=\"'outside'\" placement=\"bottom-right\" title=\"{{'nwfjs:columnChooserTitle' | i18next}}\" (click)=\"togglePopover()\"\n    triggers=\"none\" type=\"button\" class=\"btn btn-flat btn-icon text-dark\" data-netapp-id=\"nwf-column-chooser-btn\">\n    <nwf-icon iconClass=\"icon-action-settings\"></nwf-icon>\n</button>\n<ng-template #popContent>\n    <figure class=\"nwf-column-chooser\">\n        <div class=\"l-nwf-column-chooser-options\">\n            <h4>{{'nwfjs:columnChooserAvailableColumn' | i18next}}</h4>\n            <div class=\"nwf-column-chooser-options-no-msg\" *ngIf=\"!_availableColumns.length\">{{'nwfjs:columnChooserAllColumnSelected' | i18next}}</div>\n            <ul class=\"nwf-column-chooser-selections\" data-netapp-id=\"nwf-column-chooser-selections\" [ngClass]=\"{ 'is-empty': !_availableColumns.length }\">\n                <li class=\"nwf-column-chooser-option\" *ngFor=\"let selection of _availableColumns\" (click)=\"toggleSelection(selection)\" [ngClass]=\"{ 'is-selected': selection.isSelected }\">\n                    <span class=\"nwf-column-chooser-option-label\">{{selection.caption}}</span>\n                </li>\n            </ul>\n        </div>\n        <div class=\"l-nwf-column-chooser-controls\">\n            <button type=\"button\" class=\"btn btn-default\" (click)=\"deSelectAvailableColumns()\" data-netapp-id=\"nwf-column-chooser-select-option-btn\">\n                <nwf-icon iconClass=\"icon-misc-chevronleft\"></nwf-icon>\n            </button>\n            <span class=\"nwf-column-chooser-control-divider\"></span>\n            <button type=\"button\" class=\"btn btn-default\" (click)=\"selectAvailableColumns()\" data-netapp-id=\"nwf-column-chooser-select-option-btn\">\n                <nwf-icon iconClass=\"icon-misc-chevronright\"></nwf-icon>\n            </button>\n        </div>\n        <div class=\"l-nwf-column-chooser-selections\">\n            <h4>{{'nwfjs:columnChooserColumnToDisplay' | i18next}}</h4>\n            <ul class=\"nwf-column-chooser-selections\" data-netapp-id=\"nwf-column-chooser-selections\" [ngClass]=\"{'is-invalid': isInvalid}\"\n                [ngStyle]=\"{'overflow': isInvalid ? 'hidden' : ''}\">\n                <li class=\"nwf-column-chooser-option\" *ngFor=\"let selection of _columnsToDisplay;\" (click)=\"toggleSelection(selection)\" [ngClass]=\"{ 'is-selected': selection.isSelected, 'is-disabled': !(selection.allowHiding && selection.allowReordering) }\">\n                    <span class=\"nwf-column-chooser-option-label\">{{selection.caption}}</span>\n                </li>\n            </ul>\n            <div *ngIf=\"isInvalid\" class=\"invalid-feedback\">{{'nwfjs:isRequiredAtLeastOne' | i18next}}</div>\n        </div>\n\n        <div class=\"l-nwf-column-chooser-controls\">\n            <button type=\"button\" class=\"btn btn-default\" (click)=\"moveColumnsUp()\" data-netapp-id=\"nwf-column-chooser-select-option-btn\">\n                <nwf-icon iconClass=\"icon-misc-chevronup\"></nwf-icon>\n            </button>\n            <span class=\"nwf-column-chooser-control-divider\"></span>\n            <button class=\"btn btn-default\" (click)=\"moveColumnsDown()\" data-netapp-id=\"nwf-column-chooser-select-option-btn\">\n                <nwf-icon iconClass=\"icon-misc-chevrondown\"></nwf-icon>\n            </button>\n        </div>\n    </figure>\n    <footer class=\"d-flex mt-2\">\n        <div class=\"d-flex flex-fill justify-content-end\">\n            <button type=\"button\" class=\"btn btn-secondary mr-2\" (click)=\"cancel()\" data-netapp-id=\"nwf-column-chooser-close-btn\">{{'nwfjs:cancel' | i18next}}</button>\n            <button type=\"submit\" class=\"btn btn-primary\" (click)=\"apply()\" data-netapp-id=\"nwf-column-chooser-apply-btn\">{{'nwfjs:columnChooserApplyColumns' | i18next}}</button>\n        </div>\n    </footer>\n</ng-template>",
                styles: [".dropdown-toggle::after{display:none}.nwf-column-chooser{display:flex;width:550px}.l-nwf-column-chooser-options{width:250px}.nwf-column-chooser-options-no-msg{position:absolute;padding-left:10px;top:45%}.nwf-column-chooser-option-label{flex:1}.nwf-column-chooser-option{padding:6px;border-bottom:1px solid #9c9c9c;display:flex;flex:1;cursor:pointer}.nwf-column-chooser-option.is-selected{background-color:#88c5ff}.nwf-column-chooser-option.is-disabled{pointer-events:none;cursor:pointer;background-color:#f2f2f2}.l-nwf-column-chooser-controls{display:flex;flex-direction:column;align-items:center;justify-content:center;padding:0 10px}.l-nwf-column-chooser-controls .btn{display:inline-block}.nwf-column-chooser-control-divider{height:10px}.l-nwf-column-chooser-selections{width:250px;display:flex;flex-direction:column}.nwf-column-chooser-selections{margin:0;padding:0;list-style:none;overflow-y:scroll;border:1px solid #c2c2c2;flex:1;max-height:328px;height:328px}.nwf-column-chooser-selections.is-empty{overflow-y:hidden}"]
            }] }
];
/** @nocollapse */
ColumnChooserComponent.ctorParameters = () => [
    { type: I18NextPipe }
];
ColumnChooserComponent.propDecorators = {
    dataGrid: [{ type: Input }],
    ngbPopover: [{ type: ViewChild, args: [NgbPopover,] }]
};
if (false) {
    /** @type {?} */
    ColumnChooserComponent.prototype.dataGrid;
    /** @type {?} */
    ColumnChooserComponent.prototype._availableColumns;
    /** @type {?} */
    ColumnChooserComponent.prototype._columnsToDisplay;
    /** @type {?} */
    ColumnChooserComponent.prototype.isInvalid;
    /**
     * @type {?}
     * @private
     */
    ColumnChooserComponent.prototype.ngbPopover;
    /**
     * @type {?}
     * @private
     */
    ColumnChooserComponent.prototype.gridInstance;
    /**
     * @type {?}
     * @private
     */
    ColumnChooserComponent.prototype.value;
    /**
     * @type {?}
     * @private
     */
    ColumnChooserComponent.prototype.i18NextPipe;
}
//# sourceMappingURL=data:application/json;base64,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