/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,missingOverride,missingReturn,unusedPrivateMembers,uselessCode} checked by tsc
 */
import { Component, Input, ViewChild } from '@angular/core';
import { FormArray, FormBuilder, FormControl } from '@angular/forms';
import { NgbPopover } from '@ng-bootstrap/ng-bootstrap';
import { DxDataGridComponent } from 'devextreme-angular';
import { isEqual } from 'lodash-es';
import FilterConfig from './filter-config';
export class NwfFilterComponent {
    /**
     * @param {?} _fb
     */
    constructor(_fb) {
        this._fb = _fb;
        this.nwfFilters = [];
        this.config = FilterConfig;
        this.initialColumns = [];
    }
    /**
     * @return {?}
     */
    ngOnInit() {
        this.dataGrid.onContentReady.subscribe((/**
         * @param {?} data
         * @return {?}
         */
        (data) => {
            this.gridInstance = data.component.instance();
            // getting new columns.
            /** @type {?} */
            const columnOnDxGrid = this.getAllColumnsOnDxGrid().map((/**
             * @param {?} col
             * @return {?}
             */
            (col) => col.dataField));
            // when intialColumns is set, we compare it with columnsOnDxGrid and
            // and only update intialColumns when its different than columnsOnDxGrid.
            // This happens when we reset the columns on the grid.
            if (this.initialColumns.length !== 0) {
                /** @type {?} */
                const areColumnsEqual = isEqual(columnOnDxGrid, this.initialColumns);
                // if new columns different than initialColumns, we reset the filter widget.
                if (!areColumnsEqual) {
                    this.initialColumns = columnOnDxGrid;
                    // setting the filter widget if columns are not same.
                    this.setDxGridFilterOnNwfFilterWidget();
                }
            }
            else {
                // setting the initialColumns.
                this.initialColumns = columnOnDxGrid;
                // setting the filter widget for the first time.
                this.setDxGridFilterOnNwfFilterWidget();
            }
        }));
    }
    /**
     * @return {?}
     */
    setDxGridFilterOnNwfFilterWidget() {
        // get columns from gridInstance.
        /** @type {?} */
        const gridDxColumns = this.getAllColumnsOnDxGrid();
        // columns having allowFiltering as true are only allowed in the filter widget.
        this.filterableColumns = gridDxColumns.filter((/**
         * @param {?} dxColumn
         * @return {?}
         */
        (dxColumn) => (dxColumn.allowFiltering && dxColumn.showInColumnChooser)));
        /** @type {?} */
        const dxGridFilter = this.gridInstance.state().filterValue || [];
        // set value on filter widget.
        this.dxToNwfFilterConverter(dxGridFilter);
        // create the filter form group - its a group of filter criterion.
        this.createFormGroup();
    }
    /**
     * @return {?}
     */
    togglePopover() {
        this._ngbPopover.toggle();
    }
    /**
     * Creates the form that holds all the filter criterion as formControl inside it.
     * FormGroup holds the state of the filter widget.
     * @return {?}
     */
    createFormGroup() {
        /** @type {?} */
        const tempFormArray = [];
        // if  nwfFilters, that means filter is preset,
        if (this.nwfFilters.length > 0) {
            // each nwf filter is represented as a criterion in filter widget.
            // so now, we loop over all nwfFilters and create criterion for each of them,
            // and push them into formArray.
            this.nwfFilters.forEach((/**
             * @param {?} c
             * @return {?}
             */
            (c) => {
                tempFormArray.push(this.initCriterion(c));
            }));
        }
        else {
            // add default criterion.
            tempFormArray.push(this.initCriterion());
        }
        // filterFormGrp consists of array of crtierions.
        this.filterFormGrp = this._fb.group({
            criterion: new FormArray(tempFormArray),
        });
    }
    /**
     * \@description adds a new default criterion on click of "Add Filter" button
     * @return {?}
     */
    addCriterion() {
        /** @type {?} */
        const formArray = this.getFormArray();
        // push new formControl(new criterion) to formArray.
        formArray.push(this.initCriterion());
    }
    /**
     * Method to create a new filter criterion. Each filter criterion is form group of 3 form controls.
     * 1. property, 2. operator, 3. value.
     * @param {?=} criterion new filter criterion
     * @return {?}
     */
    initCriterion(criterion = this.getCriterionObject()) {
        // creating a form group that contains formControl for filter-criterion-row component.
        return this._fb.group({
            property: new FormControl(criterion.property.dataField),
            operator: new FormControl(criterion.operator.value),
            value: new FormControl(criterion.value),
        });
    }
    /**
     * \@description deletes a criterion from current criterion, else deletes all the applied criterion. Invoked on clicking Trash icon.
     * @param {?} criterionIndex
     * @return {?}
     */
    deleteCriterion(criterionIndex) {
        // delete criterion from formArray.
        /** @type {?} */
        const formArray = this.getFormArray();
        // when we have more than 1 criteria, this means we only delete the criterion at that index passed-in
        if (formArray.length > 1) {
            // deleting criterion from fromArray (formArray is array of criterion)
            formArray.removeAt(criterionIndex);
        }
        else {
            // this case applies, when we delete the only available criterion in the filter.
            // this means delete single/last criterion = filter reset.
            this.reset();
        }
    }
    /**
     * \@description deletes all the criterion.
     * @return {?}
     */
    reset() {
        // reset the formArray, so that form will come fresh on hitting filter button again.
        this.filterFormGrp = this._fb.group({
            criterion: new FormArray([]),
        });
        // clear existing filters and update it on gridInstance.
        this.apply();
        /** @type {?} */
        const formArray = this.getFormArray();
        // Adds the default criterion to filter, which will shown up when filter is opened again.
        formArray.push(this.initCriterion());
    }
    /**
     * @return {?}
     */
    getFormArray() {
        return (/** @type {?} */ (this.filterFormGrp.get('criterion')));
    }
    /**
     * \@description applies the filter on dx grid instance on hitting Apply button.
     * @return {?}
     */
    apply() {
        // We touch all formGroups to ensure validation is invoked without manually touching them.
        this.validateAllFormFields(this.filterFormGrp.controls.criterion['controls']);
        // When we click on the submit button, we want to submit the form only if it is valid.
        // To validate all form fields, we need to iterate throughout all form controls and mark them as touched.
        if (this.filterFormGrp.valid) {
            // set new filters on the grid.
            this.setFiltersOnDxGridInstance();
            // closing the popover.
            this._ngbPopover.close();
        }
    }
    /**
     * @param {?} formControls
     * @return {?}
     */
    validateAllFormFields(formControls) {
        // We mark all formControl as 'touched' programatically,
        // This makes sure Angular validation is triggered for every formControl without
        // touching it manually.
        formControls.forEach((/**
         * @param {?} formGroup
         * @return {?}
         */
        (formGroup) => {
            for (const field in formGroup.controls) {
                if (formGroup.controls.hasOwnProperty(field)) {
                    /** @type {?} */
                    const control = formGroup.get(field);
                    if (control instanceof FormControl) {
                        control.markAsTouched({ onlySelf: true });
                    }
                }
            }
        }));
    }
    /**
     * @return {?}
     */
    setFiltersOnDxGridInstance() {
        // set new filters on gridInstance.
        /** @type {?} */
        const dxFilters = [];
        // start updating the grid instance.
        this.dataGrid.instance.beginUpdate();
        // clearing the existing filters(if any).
        this.dataGrid.instance.option('filterValue', null);
        // this.dataGrid.instance.clearFilter();
        // we loop over filter form group and create an array containing filters in dx compatible format.
        this.filterFormGrp.get('criterion').value.forEach((/**
         * @param {?} criterion
         * @param {?} index
         * @param {?} array
         * @return {?}
         */
        (criterion, index, array) => {
            let { property, operator, value } = criterion;
            /** @type {?} */
            const column = this.getPropertyById(property);
            switch (column.dataType.toLowerCase()) {
                case 'datetime':
                case 'date':
                    switch (operator) {
                        case 'between':
                            const [fromDateISOString, toDateISOString] = value.split('~');
                            value = [new Date(fromDateISOString), new Date(toDateISOString)];
                            break;
                        case '<=':
                        case '>=':
                            // we convert the value into JS date object, because in this case the value returned from datetime widget is an ISO string.
                            value = new Date(value);
                            break;
                        default:
                            break;
                    }
                    break;
                case 'enum-multiple':
                    /** @type {?} */
                    const possibleValues = value.split('|');
                    if (operator === '<>') {
                        operator = 'noneof';
                    }
                    else {
                        operator = 'anyof';
                    }
                    value = possibleValues;
                    break;
                default:
                    break;
            }
            //
            if (array.length === 1) {
                dxFilters.push(property);
                dxFilters.push(operator);
                dxFilters.push(value);
            }
            else {
                dxFilters.push([property, operator, value]);
                if (index < array.length - 1) {
                    dxFilters.push('and');
                }
            }
        }));
        // set filters on gridInstance.
        this.dataGrid.instance.option('filterValue', dxFilters);
        // update nwfFilters with new filters applied.
        // this ensures count shown on filter widget is updated, when new filters are applied.
        this.dxToNwfFilterConverter(dxFilters);
        // end update. this is now gonna fire loadOptions call with new filters.
        this.dataGrid.instance.endUpdate();
    }
    /**
     * \@description cancel on hitting 'Cancel' button.
     * @return {?}
     */
    cancel() {
        // we want to reset the filter widget to the previous state, without submitting any pending changes.
        // we leverage 'createFormGroup' method to set filterWidget state.
        this.createFormGroup();
        // closing the popover.
        this._ngbPopover.close();
    }
    /**
     * @return {?}
     */
    getCriterionObject() {
        /** @type {?} */
        const firstParam = this.filterableColumns[0];
        return {
            property: firstParam,
            operator: this.getOperatorOptions(firstParam.dataType)[0],
            value: '',
        };
    }
    /**
     * \@description fetch the property from the original params passed into the widget.
     * @param {?} id
     * @return {?}
     */
    getPropertyById(id) {
        return this.filterableColumns.filter((/**
         * @param {?} property
         * @return {?}
         */
        (property) => property.dataField === id))[0];
    }
    /**
     * \@description method that return the options from filter config bundle based on the type passed, returns an object containing inputType and options.
     * @param {?} paramType
     * @return {?}
     */
    getOperatorOptions(paramType) {
        return this.config[paramType].options;
    }
    /**
     * \@description: method to create the applied criteria from appliedFilters coming from filter widget consumer.
     * This allows to preset the filterWidget this it he VALUE of the filter widget.
     * @param {?} dxGridFilters
     * @return {?}
     */
    dxToNwfFilterConverter(dxGridFilters) {
        if (dxGridFilters.length == 3 && this.getPropertyById(dxGridFilters[0])) {
            this.nwfFilters = [this.getFilterCriterionFromDxFilter(dxGridFilters)];
        }
        else {
            // multiple filters are connected by an 'and'. So we remove 'and' from dxGridFilter
            // to get actual filters, and then create filter crtierion from each filter in dxGridFilter.
            this.nwfFilters = dxGridFilters.filter((/**
             * @param {?} filter
             * @return {?}
             */
            (filter) => filter !== 'and')).map((/**
             * @param {?} filter
             * @return {?}
             */
            (filter) => {
                return this.getFilterCriterionFromDxFilter(filter);
            }));
        }
    }
    /**
     * @param {?} filter
     * @return {?}
     */
    getFilterCriterionFromDxFilter(filter) {
        let [filterName, operator, value] = filter;
        // get schema column object based on filterName(id).
        /** @type {?} */
        const schemaColumnObject = this.getPropertyById(filterName);
        // get filter operator options.
        /** @type {?} */
        const operatorOptions = this.getOperatorOptions(schemaColumnObject.dataType);
        // For 'anyof' operator, this means we are in multiselect list case.
        if (operator === 'anyof') {
            operator = '=';
            value = value.join('|');
        }
        if (operator === 'noneof') {
            operator = '<>';
            value = value.join('|');
        }
        // For 'between' operator we convert the date to ISOString and join the value using '~' sign.
        // This format is required by 'nwf-datetime-range-picker' component
        if (operator === 'between') {
            value = value.map((/**
             * @param {?} val
             * @return {?}
             */
            (val) => val.toISOString())).join('~');
        }
        // we return property, operator, value which is the format required by filterCriterionRow.
        return {
            property: schemaColumnObject,
            operator: operatorOptions.filter((/**
             * @param {?} operatorOption
             * @return {?}
             */
            (operatorOption) => operatorOption.value === operator))[0],
            value,
        };
    }
    /**
     * We find all columns on dx grid. Instead of using a method available on gridInstance directly, we use this way because
     * gridInstance.columnOption gives a complete column object (including default column settings),
     * whereas column object returned by gridInstance.option('columnms') only contains properties you explicitly set.
     * gridInstance.getVisibleColumns() method gives us the complete object but it only works for visible columns.
     * @private
     * @return {?}
     */
    getAllColumnsOnDxGrid() {
        /** @type {?} */
        const gridInstance = this.gridInstance;
        /** @type {?} */
        const columnsViaOptions = gridInstance && gridInstance.state().columns;
        if (columnsViaOptions) {
            return columnsViaOptions.map((/**
             * @param {?} __0
             * @return {?}
             */
            ({ dataField }) => gridInstance.columnOption(dataField)));
        }
        else {
            return;
        }
    }
}
NwfFilterComponent.decorators = [
    { type: Component, args: [{
                selector: 'nwf-grid-filter',
                template: "<ng-template #filterWidgetTemplate>\n    <div *ngIf=\"!filterableColumns\">\n        <div class=\"m-3\">{{'nwfjs:filterWidgetError' | i18next}}</div>\n    </div>\n    <form novalidate autocomplete=\"off\" *ngIf=\"filterableColumns\" [formGroup]=\"filterFormGrp\" (ngSubmit)=\"apply()\">\n        <div formArrayName=\"criterion\">\n            <nwf-grid-filter-criterion-row *ngFor=\"let criteria of filterFormGrp.get('criterion')['controls']; let i=index\" [dxColumns]=\"filterableColumns\"\n                [criteria]=\"criteria\" (deleteCriterion)=\"deleteCriterion(i)\">\n            </nwf-grid-filter-criterion-row>\n        </div>\n\n        <button type=\"button\" *ngIf=\"filterFormGrp.get('criterion')['controls'].length < 4\" class=\"my-3 btn btn-default\" (click)=\"addCriterion();$event.stopPropagation();\"\n            data-netapp-id=\"filter-widget-add-filter-btn\">\n            <nwf-icon iconClass=\"icon-action-add\"></nwf-icon>\n            <span class=\"nwf-icon-label\">{{'nwfjs:filterWidgetAddFilter' | i18next}}</span>\n        </button>\n\n        <div class=\"d-flex\">\n            <button type=\"button\" data-netapp-id=\"filter-widget-remove-btn\" class=\"btn btn-secondary mr-auto\" (click)=\"reset()\" [disabled]=\"nwfFilters.length <= 0\"\n                data-netapp-id=\"filter-widget-reset-btn\">{{'nwfjs:reset' | i18next}}\n            </button>\n            <button type=\"button\" class=\"btn btn-secondary mr-2\" (click)=\"cancel()\" data-netapp-id=\"filter-widget-close-btn\">{{'nwfjs:cancel' | i18next}}</button>\n            <button class=\"btn btn-primary\" type=\"submit\" data-netapp-id=\"filter-widget-apply-btn\">{{'nwfjs:filterWidgetApplyFilter' | i18next}}</button>\n        </div>\n    </form>\n</ng-template>\n<button [ngbPopover]=\"filterWidgetTemplate\" [autoClose]=\"'outside'\" triggers=\"manual\" placement=\"bottom-left\" container=\"body\"\n    (click)=\"togglePopover()\" type=\"button\" class=\"btn btn-flat btn-icon\" data-netapp-id=\"filter-widget-btn\">\n    <nwf-icon iconClass=\"icon-action-filter\"></nwf-icon>\n    <span class=\"badge badge-light\" *ngIf=\"nwfFilters.length > 0\">\n        {{nwfFilters.length}}\n    </span>\n</button>",
                styles: [".filter-badge-light{background:#1e4a93;color:#fff;font-weight:700}.btn-delete-criterion{height:35px}"]
            }] }
];
/** @nocollapse */
NwfFilterComponent.ctorParameters = () => [
    { type: FormBuilder }
];
NwfFilterComponent.propDecorators = {
    _ngbPopover: [{ type: ViewChild, args: [NgbPopover,] }],
    dataGrid: [{ type: Input }]
};
if (false) {
    /** @type {?} */
    NwfFilterComponent.prototype._ngbPopover;
    /** @type {?} */
    NwfFilterComponent.prototype.dataGrid;
    /** @type {?} */
    NwfFilterComponent.prototype.nwfFilters;
    /** @type {?} */
    NwfFilterComponent.prototype.filterableColumns;
    /** @type {?} */
    NwfFilterComponent.prototype.filterFormGrp;
    /**
     * @type {?}
     * @private
     */
    NwfFilterComponent.prototype.config;
    /**
     * @type {?}
     * @private
     */
    NwfFilterComponent.prototype.gridInstance;
    /**
     * @type {?}
     * @private
     */
    NwfFilterComponent.prototype.initialColumns;
    /**
     * @type {?}
     * @private
     */
    NwfFilterComponent.prototype._fb;
}
//# sourceMappingURL=data:application/json;base64,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