/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,uselessCode} checked by tsc
 */
import * as tslib_1 from "tslib";
import { NgbCalendarIslamicCivil } from './ngb-calendar-islamic-civil';
import { NgbDate } from '../ngb-date';
import { Injectable } from '@angular/core';
/** *
 * Umalqura calendar is one type of Hijri calendars used in islamic countries.
 * This Calendar is used by Saudi Arabia for administrative purpose.
 * Unlike tabular calendars, the algorithm involves astronomical calculation, but it's still deterministic.
 * http://cldr.unicode.org/development/development-process/design-proposals/islamic-calendar-types
  @type {?} */
var GREGORIAN_FIRST_DATE = new Date(1882, 10, 12);
/** @type {?} */
var GREGORIAN_LAST_DATE = new Date(2174, 10, 25);
/** @type {?} */
var HIJRI_BEGIN = 1300;
/** @type {?} */
var HIJRI_END = 1600;
/** @type {?} */
var ONE_DAY = 1000 * 60 * 60 * 24;
/** @type {?} */
var MONTH_LENGTH = [
    '101010101010', '110101010100', '111011001001', '011011010100', '011011101010',
    '001101101100', '101010101101', '010101010101', '011010101001', '011110010010',
    '101110101001', '010111010100', '101011011010', '010101011100', '110100101101',
    '011010010101', '011101001010', '101101010100', '101101101010', '010110101101',
    '010010101110', '101001001111', '010100010111', '011010001011', '011010100101',
    '101011010101', '001011010110', '100101011011', '010010011101', '101001001101',
    '110100100110', '110110010101', '010110101100', '100110110110', '001010111010',
    '101001011011', '010100101011', '101010010101', '011011001010', '101011101001',
    '001011110100', '100101110110', '001010110110', '100101010110', '101011001010',
    '101110100100', '101111010010', '010111011001', '001011011100', '100101101101',
    '010101001101', '101010100101', '101101010010', '101110100101', '010110110100',
    '100110110110', '010101010111', '001010010111', '010101001011', '011010100011',
    '011101010010', '101101100101', '010101101010', '101010101011', '010100101011',
    '110010010101', '110101001010', '110110100101', '010111001010', '101011010110',
    '100101010111', '010010101011', '100101001011', '101010100101', '101101010010',
    '101101101010', '010101110101', '001001110110', '100010110111', '010001011011',
    '010101010101', '010110101001', '010110110100', '100111011010', '010011011101',
    '001001101110', '100100110110', '101010101010', '110101010100', '110110110010',
    '010111010101', '001011011010', '100101011011', '010010101011', '101001010101',
    '101101001001', '101101100100', '101101110001', '010110110100', '101010110101',
    '101001010101', '110100100101', '111010010010', '111011001001', '011011010100',
    '101011101001', '100101101011', '010010101011', '101010010011', '110101001001',
    '110110100100', '110110110010', '101010111001', '010010111010', '101001011011',
    '010100101011', '101010010101', '101100101010', '101101010101', '010101011100',
    '010010111101', '001000111101', '100100011101', '101010010101', '101101001010',
    '101101011010', '010101101101', '001010110110', '100100111011', '010010011011',
    '011001010101', '011010101001', '011101010100', '101101101010', '010101101100',
    '101010101101', '010101010101', '101100101001', '101110010010', '101110101001',
    '010111010100', '101011011010', '010101011010', '101010101011', '010110010101',
    '011101001001', '011101100100', '101110101010', '010110110101', '001010110110',
    '101001010110', '111001001101', '101100100101', '101101010010', '101101101010',
    '010110101101', '001010101110', '100100101111', '010010010111', '011001001011',
    '011010100101', '011010101100', '101011010110', '010101011101', '010010011101',
    '101001001101', '110100010110', '110110010101', '010110101010', '010110110101',
    '001011011010', '100101011011', '010010101101', '010110010101', '011011001010',
    '011011100100', '101011101010', '010011110101', '001010110110', '100101010110',
    '101010101010', '101101010100', '101111010010', '010111011001', '001011101010',
    '100101101101', '010010101101', '101010010101', '101101001010', '101110100101',
    '010110110010', '100110110101', '010011010110', '101010010111', '010101000111',
    '011010010011', '011101001001', '101101010101', '010101101010', '101001101011',
    '010100101011', '101010001011', '110101000110', '110110100011', '010111001010',
    '101011010110', '010011011011', '001001101011', '100101001011', '101010100101',
    '101101010010', '101101101001', '010101110101', '000101110110', '100010110111',
    '001001011011', '010100101011', '010101100101', '010110110100', '100111011010',
    '010011101101', '000101101101', '100010110110', '101010100110', '110101010010',
    '110110101001', '010111010100', '101011011010', '100101011011', '010010101011',
    '011001010011', '011100101001', '011101100010', '101110101001', '010110110010',
    '101010110101', '010101010101', '101100100101', '110110010010', '111011001001',
    '011011010010', '101011101001', '010101101011', '010010101011', '101001010101',
    '110100101001', '110101010100', '110110101010', '100110110101', '010010111010',
    '101000111011', '010010011011', '101001001101', '101010101010', '101011010101',
    '001011011010', '100101011101', '010001011110', '101000101110', '110010011010',
    '110101010101', '011010110010', '011010111001', '010010111010', '101001011101',
    '010100101101', '101010010101', '101101010010', '101110101000', '101110110100',
    '010110111001', '001011011010', '100101011010', '101101001010', '110110100100',
    '111011010001', '011011101000', '101101101010', '010101101101', '010100110101',
    '011010010101', '110101001010', '110110101000', '110111010100', '011011011010',
    '010101011011', '001010011101', '011000101011', '101100010101', '101101001010',
    '101110010101', '010110101010', '101010101110', '100100101110', '110010001111',
    '010100100111', '011010010101', '011010101010', '101011010110', '010101011101',
    '001010011101'
];
/**
 * @param {?} date1
 * @param {?} date2
 * @return {?}
 */
function getDaysDiff(date1, date2) {
    /** @type {?} */
    var diff = Math.abs(date1.getTime() - date2.getTime());
    return Math.round(diff / ONE_DAY);
}
var NgbCalendarIslamicUmalqura = /** @class */ (function (_super) {
    tslib_1.__extends(NgbCalendarIslamicUmalqura, _super);
    function NgbCalendarIslamicUmalqura() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    /**
    * Returns the equivalent islamic(Umalqura) date value for a give input Gregorian date.
    * `gdate` is s JS Date to be converted to Hijri.
    */
    /**
     * Returns the equivalent islamic(Umalqura) date value for a give input Gregorian date.
     * `gdate` is s JS Date to be converted to Hijri.
     * @param {?} gDate
     * @return {?}
     */
    NgbCalendarIslamicUmalqura.prototype.fromGregorian = /**
     * Returns the equivalent islamic(Umalqura) date value for a give input Gregorian date.
     * `gdate` is s JS Date to be converted to Hijri.
     * @param {?} gDate
     * @return {?}
     */
    function (gDate) {
        /** @type {?} */
        var hDay = 1;
        /** @type {?} */
        var hMonth = 0;
        /** @type {?} */
        var hYear = 1300;
        /** @type {?} */
        var daysDiff = getDaysDiff(gDate, GREGORIAN_FIRST_DATE);
        if (gDate.getTime() - GREGORIAN_FIRST_DATE.getTime() >= 0 && gDate.getTime() - GREGORIAN_LAST_DATE.getTime() <= 0) {
            /** @type {?} */
            var year = 1300;
            for (var i = 0; i < MONTH_LENGTH.length; i++, year++) {
                for (var j = 0; j < 12; j++) {
                    /** @type {?} */
                    var numOfDays = +MONTH_LENGTH[i][j] + 29;
                    if (daysDiff <= numOfDays) {
                        hDay = daysDiff + 1;
                        if (hDay > numOfDays) {
                            hDay = 1;
                            j++;
                        }
                        if (j > 11) {
                            j = 0;
                            year++;
                        }
                        hMonth = j;
                        hYear = year;
                        return new NgbDate(hYear, hMonth + 1, hDay);
                    }
                    daysDiff = daysDiff - numOfDays;
                }
            }
        }
        else {
            return _super.prototype.fromGregorian.call(this, gDate);
        }
    };
    /**
    * Converts the current Hijri date to Gregorian.
    */
    /**
     * Converts the current Hijri date to Gregorian.
     * @param {?} hDate
     * @return {?}
     */
    NgbCalendarIslamicUmalqura.prototype.toGregorian = /**
     * Converts the current Hijri date to Gregorian.
     * @param {?} hDate
     * @return {?}
     */
    function (hDate) {
        /** @type {?} */
        var hYear = hDate.year;
        /** @type {?} */
        var hMonth = hDate.month - 1;
        /** @type {?} */
        var hDay = hDate.day;
        /** @type {?} */
        var gDate = new Date(GREGORIAN_FIRST_DATE);
        /** @type {?} */
        var dayDiff = hDay - 1;
        if (hYear >= HIJRI_BEGIN && hYear <= HIJRI_END) {
            for (var y = 0; y < hYear - HIJRI_BEGIN; y++) {
                for (var m = 0; m < 12; m++) {
                    dayDiff += +MONTH_LENGTH[y][m] + 29;
                }
            }
            for (var m = 0; m < hMonth; m++) {
                dayDiff += +MONTH_LENGTH[hYear - HIJRI_BEGIN][m] + 29;
            }
            gDate.setDate(GREGORIAN_FIRST_DATE.getDate() + dayDiff);
        }
        else {
            gDate = _super.prototype.toGregorian.call(this, hDate);
        }
        return gDate;
    };
    /**
    * Returns the number of days in a specific Hijri hMonth.
    * `hMonth` is 1 for Muharram, 2 for Safar, etc.
    * `hYear` is any Hijri hYear.
    */
    /**
     * Returns the number of days in a specific Hijri hMonth.
     * `hMonth` is 1 for Muharram, 2 for Safar, etc.
     * `hYear` is any Hijri hYear.
     * @param {?} hMonth
     * @param {?} hYear
     * @return {?}
     */
    NgbCalendarIslamicUmalqura.prototype.getDaysPerMonth = /**
     * Returns the number of days in a specific Hijri hMonth.
     * `hMonth` is 1 for Muharram, 2 for Safar, etc.
     * `hYear` is any Hijri hYear.
     * @param {?} hMonth
     * @param {?} hYear
     * @return {?}
     */
    function (hMonth, hYear) {
        if (hYear >= HIJRI_BEGIN && hYear <= HIJRI_END) {
            /** @type {?} */
            var pos = hYear - HIJRI_BEGIN;
            return +MONTH_LENGTH[pos][hMonth - 1] + 29;
        }
        return _super.prototype.getDaysPerMonth.call(this, hMonth, hYear);
    };
    NgbCalendarIslamicUmalqura.decorators = [
        { type: Injectable },
    ];
    return NgbCalendarIslamicUmalqura;
}(NgbCalendarIslamicCivil));
export { NgbCalendarIslamicUmalqura };

//# sourceMappingURL=data:application/json;base64,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