"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const node_1 = require("@angular-devkit/core/node");
const path_1 = require("path");
const ts = require("typescript");
const dev = Math.floor(Math.random() * 10000);
class VirtualStats {
    constructor(_path) {
        this._path = _path;
        this._ctime = new Date();
        this._mtime = new Date();
        this._atime = new Date();
        this._btime = new Date();
        this._dev = dev;
        this._ino = Math.floor(Math.random() * 100000);
        this._mode = parseInt('777', 8); // RWX for everyone.
        this._uid = Number(process.env['UID']) || 0;
        this._gid = Number(process.env['GID']) || 0;
    }
    isFile() { return false; }
    isDirectory() { return false; }
    isBlockDevice() { return false; }
    isCharacterDevice() { return false; }
    isSymbolicLink() { return false; }
    isFIFO() { return false; }
    isSocket() { return false; }
    get dev() { return this._dev; }
    get ino() { return this._ino; }
    get mode() { return this._mode; }
    get nlink() { return 1; } // Default to 1 hard link.
    get uid() { return this._uid; }
    get gid() { return this._gid; }
    get rdev() { return 0; }
    get size() { return 0; }
    get blksize() { return 512; }
    get blocks() { return Math.ceil(this.size / this.blksize); }
    get atime() { return this._atime; }
    get atimeMs() { return this._atime.getTime(); }
    get mtime() { return this._mtime; }
    get mtimeMs() { return this._mtime.getTime(); }
    get ctime() { return this._ctime; }
    get ctimeMs() { return this._ctime.getTime(); }
    get birthtime() { return this._btime; }
    get birthtimeMs() { return this._btime.getTime(); }
}
exports.VirtualStats = VirtualStats;
class VirtualDirStats extends VirtualStats {
    constructor(_fileName) {
        super(_fileName);
    }
    isDirectory() { return true; }
    get size() { return 1024; }
}
exports.VirtualDirStats = VirtualDirStats;
class VirtualFileStats extends VirtualStats {
    constructor(_fileName) {
        super(_fileName);
    }
    static createFromString(_fileName, _content) {
        const stats = new VirtualFileStats(_fileName);
        stats.content = _content;
        return stats;
    }
    static createFromBuffer(_fileName, _buffer) {
        const stats = new VirtualFileStats(_fileName);
        stats.bufferContent = _buffer;
        return stats;
    }
    get content() {
        if (!this._content && this.bufferContent) {
            this._content = core_1.virtualFs.fileBufferToString(this.bufferContent);
        }
        return this._content || '';
    }
    set content(v) {
        this._content = v;
        this._bufferContent = null;
        this.resetMetadata();
    }
    get bufferContent() {
        if (!this._bufferContent && this._content) {
            this._bufferContent = core_1.virtualFs.stringToFileBuffer(this._content);
        }
        return this._bufferContent || core_1.virtualFs.stringToFileBuffer('');
    }
    set bufferContent(buf) {
        this._bufferContent = buf;
        this._content = null;
        this.resetMetadata();
    }
    setSourceFile(sourceFile) {
        this._sourceFile = sourceFile;
    }
    getSourceFile(languageVersion, setParentNodes) {
        if (!this._sourceFile) {
            this._sourceFile = ts.createSourceFile(workaroundResolve(this._path), this.content, languageVersion, setParentNodes);
        }
        return this._sourceFile;
    }
    resetMetadata() {
        this._mtime = new Date();
        this._sourceFile = null;
    }
    isFile() { return true; }
    get size() { return this.content.length; }
}
exports.VirtualFileStats = VirtualFileStats;
class WebpackCompilerHost {
    constructor(_options, basePath, _host = new node_1.NodeJsSyncHost()) {
        this._options = _options;
        this._host = _host;
        this._files = Object.create(null);
        this._directories = Object.create(null);
        this._changedFiles = Object.create(null);
        this._changedDirs = Object.create(null);
        this._cache = false;
        this._syncHost = new core_1.virtualFs.SyncDelegateHost(_host);
        this._setParentNodes = true;
        this._basePath = this._normalizePath(basePath);
    }
    _normalizePath(path) {
        return core_1.normalize(path);
    }
    denormalizePath(path) {
        return core_1.getSystemPath(core_1.normalize(path));
    }
    resolve(path) {
        const p = this._normalizePath(path);
        if (p[0] == '.') {
            return this._normalizePath(path_1.join(this.getCurrentDirectory(), p));
        }
        else if (p[0] == '/' || p.match(/^\w:\//)) {
            return p;
        }
        else {
            return this._normalizePath(path_1.join(this._basePath, p));
        }
    }
    _cacheFile(fileName, stats) {
        this._files[fileName] = stats;
        let p = path_1.dirname(fileName);
        while (p && !this._directories[p]) {
            this._directories[p] = new VirtualDirStats(p);
            this._changedDirs[p] = true;
            p = path_1.dirname(p);
        }
        this._changedFiles[fileName] = true;
    }
    get dirty() {
        return Object.keys(this._changedFiles).length > 0;
    }
    enableCaching() {
        this._cache = true;
    }
    resetChangedFileTracker() {
        this._changedFiles = Object.create(null);
        this._changedDirs = Object.create(null);
    }
    getChangedFilePaths() {
        return Object.keys(this._changedFiles);
    }
    getNgFactoryPaths() {
        return Object.keys(this._files)
            .filter(fileName => fileName.endsWith('.ngfactory.js') || fileName.endsWith('.ngstyle.js'))
            // These paths are used by the virtual file system decorator so we must denormalize them.
            .map(path => this.denormalizePath(path));
    }
    invalidate(fileName) {
        const fullPath = this.resolve(fileName);
        if (fullPath in this._files) {
            this._files[fullPath] = null;
        }
        else {
            for (const file in this._files) {
                if (file.startsWith(fullPath + '/')) {
                    this._files[file] = null;
                }
            }
        }
        if (this.fileExists(fullPath)) {
            this._changedFiles[fullPath] = true;
        }
    }
    fileExists(fileName, delegate = true) {
        const p = this.resolve(fileName);
        return this._files[p] != null
            || (delegate && this._syncHost.exists(core_1.normalize(p)));
    }
    readFile(fileName) {
        const stats = this.findVirtualFile(fileName);
        return stats && stats.content;
    }
    readFileBuffer(fileName) {
        const stats = this.findVirtualFile(fileName);
        if (stats) {
            const buffer = Buffer.from(stats.bufferContent);
            return buffer;
        }
    }
    findVirtualFile(fileName) {
        const p = this.resolve(fileName);
        const stats = this._files[p];
        if (stats) {
            return stats;
        }
        try {
            const fileBuffer = this._syncHost.read(p);
            if (fileBuffer) {
                const stats = VirtualFileStats.createFromBuffer(p, fileBuffer);
                if (this._cache) {
                    this._cacheFile(p, stats);
                }
                return stats;
            }
        }
        catch (_a) {
            return undefined;
        }
    }
    stat(path) {
        const p = this.resolve(path);
        const stats = this._files[p] || this._directories[p];
        if (!stats) {
            return this._syncHost.stat(p);
        }
        return stats;
    }
    directoryExists(directoryName, delegate = true) {
        const p = this.resolve(directoryName);
        return (this._directories[p] != null)
            || (delegate && this._syncHost.exists(p) && this._syncHost.isDirectory(p));
    }
    getFiles(path) {
        const p = this.resolve(path);
        const subfiles = Object.keys(this._files)
            .filter(fileName => path_1.dirname(fileName) == p)
            .map(p => path_1.basename(p));
        let delegated;
        try {
            delegated = this._syncHost.list(p).filter((x) => {
                try {
                    return this._syncHost.isFile(core_1.join(p, x));
                }
                catch (_a) {
                    return false;
                }
            });
        }
        catch (_a) {
            delegated = [];
        }
        return delegated.concat(subfiles);
    }
    getDirectories(path) {
        const p = this.resolve(path);
        const subdirs = Object.keys(this._directories)
            .filter(fileName => path_1.dirname(fileName) == p)
            .map(path => path_1.basename(path));
        let delegated;
        try {
            delegated = this._syncHost.list(p).filter((x) => {
                try {
                    return this._syncHost.isDirectory(core_1.join(p, x));
                }
                catch (_a) {
                    return false;
                }
            });
        }
        catch (_a) {
            delegated = [];
        }
        return delegated.concat(subdirs);
    }
    getSourceFile(fileName, languageVersion, _onError) {
        fileName = this.resolve(fileName);
        let stats = this._files[fileName];
        if (!stats) {
            const content = this.readFile(fileName);
            if (!this._cache && content) {
                return ts.createSourceFile(workaroundResolve(fileName), content, languageVersion, this._setParentNodes);
            }
            else {
                stats = this._files[fileName];
                if (!stats) {
                    // If cache is turned on and the file exists, the readFile call will have populated stats.
                    // Empty stats at this point mean the file doesn't exist at and so we should return
                    // undefined.
                    return undefined;
                }
            }
        }
        return stats && stats.getSourceFile(languageVersion, this._setParentNodes);
    }
    get getCancellationToken() {
        // return this._delegate.getCancellationToken;
        // TODO: consider implementing a cancellation token.
        return undefined;
    }
    getDefaultLibFileName(options) {
        return ts.createCompilerHost(options, false).getDefaultLibFileName(options);
    }
    // This is due to typescript CompilerHost interface being weird on writeFile. This shuts down
    // typings in WebStorm.
    get writeFile() {
        return (fileName, data, _writeByteOrderMark, _onError, _sourceFiles) => {
            const p = this.resolve(fileName);
            const stats = VirtualFileStats.createFromString(p, data);
            this._cacheFile(p, stats);
        };
    }
    getCurrentDirectory() {
        return this._basePath !== null ? this._basePath : '/';
    }
    getCanonicalFileName(fileName) {
        return this.resolve(fileName);
    }
    useCaseSensitiveFileNames() {
        return !process.platform.startsWith('win32');
    }
    getNewLine() {
        return '\n';
    }
    setResourceLoader(resourceLoader) {
        this._resourceLoader = resourceLoader;
    }
    readResource(fileName) {
        if (this._resourceLoader) {
            // These paths are meant to be used by the loader so we must denormalize them.
            const denormalizedFileName = this.denormalizePath(core_1.normalize(fileName));
            return this._resourceLoader.get(denormalizedFileName);
        }
        else {
            return this.readFile(fileName);
        }
    }
}
exports.WebpackCompilerHost = WebpackCompilerHost;
// `TsCompilerAotCompilerTypeCheckHostAdapter` in @angular/compiler-cli seems to resolve module
// names directly via `resolveModuleName`, which prevents full Path usage.
// To work around this we must provide the same path format as TS internally uses in
// the SourceFile paths.
function workaroundResolve(path) {
    return core_1.getSystemPath(core_1.normalize(path)).replace(/\\/g, '/');
}
exports.workaroundResolve = workaroundResolve;
//# sourceMappingURL=data:application/json;base64,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