"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const ts = require("typescript");
const compiler_host_1 = require("../compiler_host");
// Find all nodes from the AST in the subtree of node of SyntaxKind kind.
function collectDeepNodes(node, kind) {
    const nodes = [];
    const helper = (child) => {
        if (child.kind === kind) {
            nodes.push(child);
        }
        ts.forEachChild(child, helper);
    };
    ts.forEachChild(node, helper);
    return nodes;
}
exports.collectDeepNodes = collectDeepNodes;
function getFirstNode(sourceFile) {
    if (sourceFile.statements.length > 0) {
        return sourceFile.statements[0];
    }
    return sourceFile.getChildAt(0);
}
exports.getFirstNode = getFirstNode;
function getLastNode(sourceFile) {
    if (sourceFile.statements.length > 0) {
        return sourceFile.statements[sourceFile.statements.length - 1] || null;
    }
    return null;
}
exports.getLastNode = getLastNode;
// Test transform helpers.
const basePath = '/project/src/';
const fileName = basePath + 'test-file.ts';
function createTypescriptContext(content) {
    // Set compiler options.
    const compilerOptions = {
        noEmitOnError: false,
        allowJs: true,
        newLine: ts.NewLineKind.LineFeed,
        target: ts.ScriptTarget.ESNext,
        skipLibCheck: true,
        sourceMap: false,
        importHelpers: true,
    };
    // Create compiler host.
    const compilerHost = new compiler_host_1.WebpackCompilerHost(compilerOptions, basePath);
    // Add a dummy file to host content.
    compilerHost.writeFile(fileName, content, false);
    // Create the TypeScript program.
    const program = ts.createProgram([fileName], compilerOptions, compilerHost);
    return { compilerHost, program };
}
exports.createTypescriptContext = createTypescriptContext;
function transformTypescript(content, transformers, program, compilerHost) {
    // Use given context or create a new one.
    if (content !== undefined) {
        const typescriptContext = createTypescriptContext(content);
        program = typescriptContext.program;
        compilerHost = typescriptContext.compilerHost;
    }
    else if (!program || !compilerHost) {
        throw new Error('transformTypescript needs either `content` or a `program` and `compilerHost');
    }
    // Emit.
    const { emitSkipped, diagnostics } = program.emit(undefined, undefined, undefined, undefined, { before: transformers });
    // Log diagnostics if emit wasn't successfull.
    if (emitSkipped) {
        console.log(diagnostics);
        return null;
    }
    // Return the transpiled js.
    return compilerHost.readFile(fileName.replace(/\.ts$/, '.js'));
}
exports.transformTypescript = transformTypescript;
//# sourceMappingURL=data:application/json;base64,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