package com.bycast.db;

import java.sql.Array;
import java.sql.Blob;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.NClob;
import java.sql.PreparedStatement;
import java.sql.SQLClientInfoException;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.SQLXML;
import java.sql.Savepoint;
import java.sql.Statement;
import java.sql.Struct;
import java.util.Map;
import java.util.Properties;

/**
 * The JDBCConnection class provides a wrapper around the <code>Connection</code> object with
 * additional information used for database connection pooling.
 *
 * This class is adapted from
 * http://java.sun.com/developer/onlineTraining/Programming/JDCBook/conpool.html.
 * License info : http://developers.sun.com/license/berkeley_license.html
 * -------------------------------------------------------------------------------
 * Copyright (c) 2008, 2010 Oracle and/or its affiliates. All rights reserved. Use 
 * is subject to license terms. Redistribution and use in source and binary 
 * forms, with or without modification, are permitted provided that the following 
 * conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 *  * Neither the name of Oracle Corporation nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software 
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * You acknowledge that this software is not designed, licensed or intended for use 
 * in the design, construction, operation or maintenance of any nuclear facility. 
 * ---------------------------------------------------------------------------------- 
 *
 * @version 20/01/2006
 * @author  cng
 * @since   4.0
 */
public abstract class JDBCConnection implements Connection 
{
    /**
     * The pool to which the connection belongs in.
     */
    private JDBCConnectionPool pool;

    /**
     * The <code>Connection</code> object that is being wrapped around.
     */
    private Connection conn;

    /**
     * Flag indicating whether the connection has been leased from the pool
     * and is being used by other processes.
     */
    private boolean inuse;

    /**
     * The time at which the connection was leased from the pool
     */
    private long timestamp;

    /**
     * The time at which the connection started idling in the pool
     */
    private long idleTimestamp;

    /**
     * Constructor.
     *
     * cng
     * @param conn  database connection
     * @param pool  connection pool which this JDBCConnection is a part of
     * @since       4.0
     */
    public JDBCConnection(Connection conn, JDBCConnectionPool pool) 
    {
        this.conn=conn;
        this.pool=pool;
        this.inuse=false;
        this.timestamp=0;
        this.idleTimestamp = System.currentTimeMillis();
    }

    /**
     * Flags this instance as being leased from the pool and is inuse.
     * 
     * cng
     * @return <code>false</code> if this connection is already in use; otherwise <code>true</code>
     * @since   4.0
     */
    public boolean lease()
    {
        boolean isLeased = false;
        if(!inuse && validate())
        {
            // ensure that the connection is by default leased with transactions off
            try
            {
                setAutoCommit(true);
            }
            catch(SQLException e)
            {
                return false;
            }

            inuse=true;
            timestamp=System.currentTimeMillis();
            isLeased = true;
        }
        return isLeased;
    }

    /**
     * Validate if this instance's connection is still connected to the database.
     *
     * cng
     * @return <code>true</code> if connection is connected to the database; 
     *           <code>false</code> otherwise
     * @since   4.0
     */
    public boolean validate() 
    {
        try
        {
            conn.getMetaData();
        }
        catch(Exception e) 
        {
            return false;
        }
        return true;
    }

    /**
     * Check if this instance is being used.
     *
     * cng
     * @return <code>true</code> if in use; otherwise <code>false</code>
     * @since   4.0
     */
    public boolean inUse() 
    {
        return inuse;
    }

    /**
     * Retrieve the timestamp of when this instance was last leased.
     * 
     * cng
     * @return the timestamp (in milliseconds)
     * @since   4.0
     */
    public long getLastUse() 
    {
        return timestamp;
    }

    /**
     * Retrieve the timestamp at which the connection started idling in the pool.
     * 
     * cng
     * @return the timestamp (in milliseconds)
     * @since   4.0
     */
    public long getIdle() 
    {
        return idleTimestamp;
    }


    /** 
     * Return this to the database pool.
     *
     * cng
     * @throws SQLException if a database error occurs
     * @since   4.0
     */
    public void close() throws SQLException 
    {
        pool.returnConnection(this);
    }

    /**
     * Releases this from its lease.
     * Also resets the timestamp for when the connection started to idle in the pool.
     *
     * cng
     * @since   4.0
     */
    protected void expireLease() 
    {
        inuse = false;
        idleTimestamp = System.currentTimeMillis();
    }

    /**
     * Retrieve this instance's database connection. 
     * <b>Note:</b> This method should never be called. This method is made public instead of 
     * protected purely for for automated testing.
     * 
     * cng
     * @return the associated <code>Connection</code> object
     * @since   4.0
     */
    public Connection getConnection() 
    {
        return conn;
    }

    /**
     * Creates a <code>PreparedStatement</code> object for sending parameterized SQL statements to the database.
     *
     * cng
     * @param sql   a SQL statement that may contain one or more '?' IN parameter placeholders
     * @return      a new default <code>PreparedStatement</code> object containing the pre-compiled SQL statement
     * @throws SQLException if a database acess error occurs
     * @since   4.0
     */
    public PreparedStatement prepareStatement(String sql) throws SQLException 
    {
        return conn.prepareStatement(sql);
    }

    /**
     * Creates a <code>Statement</code> object for sending SQL statements to the database.
     *
     * cng
     * @return  a new default <code>Statement</code> object
     * @throws SQLException if a database error occurs
     * @since   4.0
     */
    public Statement createStatement() throws SQLException 
    {
        return conn.createStatement();
    }

    /**
     * Retrieves whether this instance's <code>Connection</code> object has been closed.
     * 
     * cng
     * @return <code>true</code> if closed; else <code>false</code>
     * @throws SQLException if a database access error occurs
     * @since   4.0
     */
    public boolean isClosed() throws SQLException 
    {
        return conn.isClosed();
    }

    /**
     * Retrieves a <code>DatabaseMetaData</code> object that contains metadata about the database 
     * to which this <code>Connection</code> object represents a connection.
     * 
     * cng
     * @return  a <code>DatabaseMetaData</code> object for this <code>Connection</code> object
     * @throws if a database access error occurs
     * @since   4.0
     */
    public DatabaseMetaData getMetaData() throws SQLException 
    {
        return conn.getMetaData();
    }

    /**
     * Puts  this instance's connection in read-only mode as a hint to the driver to enable database optimizations.
     * 
     * cng
     * @param readOnly  <code>true</code> enables read-only mode; <code>false</code> disables it 
     * @throws SQLException if a database access error occurs or this method is called during a transaction
     * @since   4.0
     */
    public void setReadOnly(boolean readOnly) throws SQLException 
    {
        conn.setReadOnly(readOnly);
    }

    /**
     * Retrieves whether this instance's <code>Connection</code> object is in read-only mode.
     * 
     * cng
     * @return <code>true</code> if this instance's <code>Connection</code> object is read-only; <code>false</code> otherwise 
     * @throws SQLException if a database access error occurs
     * @since   4.0
     */
    public boolean isReadOnly() throws SQLException 
    {
        return conn.isReadOnly();
    }

    /**
     * Attemps to change the transaction isolation level for this instance's <code>Connection</code> object to the one given.
     * 
     * cng
     * @param level the new isolation level to change to; MySQL does not support <code>Connection.TRANSACTION_NONE</code>
     * @throws SQLException if a database access error occurs or level was trying to be set to <code>Connection.TRANSACTION_NONE</code>
     * @since   4.0
     */ 
    public void setTransactionIsolation(int level) throws SQLException
    {
        conn.setTransactionIsolation(level);
    }

    /**
     * Retrieves this instance's <code>Connection</code> object's current transaction isolation level.
     * 
     * cng
     * @return  the current transaction isolation level
     * @throws SQLException if a database error occurs
     * @since   4.0
     */ 
    public int getTransactionIsolation() throws SQLException 
    {
        return conn.getTransactionIsolation();
    }

    /**
     * Sets this instance's connection's auto-commit mode to the given state
     * 
     * cng
     * @param autoCommit    <code>true</code> to enable auto-commit mode; <code>false</code> to disable it
     * @throws SQLException if a database access error occurs
     * @since   4.0
     */
    public void setAutoCommit(boolean autoCommit) throws SQLException 
    {
        conn.setAutoCommit(autoCommit);
    }

    /**
     * Retrieves the current auto-commit mode for this instance's <code>Connection</code> object
     * 
     * cng
     * @return <code>true</code> if auto-commit mode is enabled; otherwise <code>false</code>
     * @throws SQLException if a database access error occurs
     * @since   4.0
     */
    public boolean getAutoCommit() throws SQLException 
    {
        return conn.getAutoCommit();
    }

    /**
     * Makes all changes made since the previous commit/rollback permanent and 
     * releases any database locks currently held by this instance's <code>Connection</code> object.
     *
     * cng
     * @throws SQLException if a database access error occurs
     * @since   4.0
     */
    public void commit() throws SQLException 
    {
        conn.commit();
    }

    /**
     * Undoes all changes made in the current transaction and releases any database locks currently 
     * held by this instance's <code>Connection</code> object
     * 
     * cng
     * @throws SQLException if a database access error occurs
     * @since   4.0
     */
    public void rollback() throws SQLException 
    {
        conn.rollback();
    }

    /**
     * Creates an unnamed savepoint in the current transaction and returns the new Savepoint object that represents it.
     *
     * cng
     * @return  the new <code>Savepoint</code> object
     * @throws SQLException     if a database access error occurs or this <code>Connection</code> 
     *                          object is currently in auto-commit mode
     * @since   4.0
     */
    public Savepoint setSavepoint() throws SQLException
    {
        return conn.setSavepoint();
    }

    /**
     * Creates an unnamed savepoint in the current transaction and returns the new Savepoint object that represents it.
     *
     * cng
     * @param name   a <code>String></code> containing the name of the savepoint
     * @return       the new <code>Savepoint</code> object
     * @throws SQLException   if a database access error occurs or this <code>Connection</code> object is currently 
     *                           in auto-commit mode
     * @since   4.0
     */
    public Savepoint setSavepoint(String name) throws SQLException
    {
        return conn.setSavepoint(name);
    }

    /** 
     * Removes the given Savepoint object from the current transaction.
     *
     * cng
     * @param savepoint  the <code>Savepoint</code> object to be removed
     * @throws SQLExcetion    if a database access error occur or the given <code>Savepoint</code> object is not a 
     *                           valid savepoint in the current transaction
     * @since   4.0
     */
    public void releaseSavepoint(Savepoint savepoint) throws SQLException
    {
        conn.releaseSavepoint(savepoint);
    }

    /**
     * Undoes all changes made after the given Savepoint object was set.
     *
     * cng
     * @param savepoint  the <code>Savepoint</code> object to roll back to
     * @throws SQLException   if a database access error occurs , the <code>Savepoint</code> object is no longer valid, 
     *                           or this <code>Connection</code> object is currently in auto-commit mode
     * @since   4.0
     */
    public void rollback(Savepoint savepoint) throws SQLException
    {
        conn.rollback(savepoint);
    }

// Uncommon methods

    /**
     * Converts the given SQL statement into the system's native SQL grammar.
     * 
     * cng
     * @param sql   the sql string to be converted
     * @return the converted string in the system's native SQL grammar
     * @throws SQLException if a database access error occurs
     * @since   4.0
     */
    public String nativeSQL(String sql) throws SQLException 
    {
        return conn.nativeSQL(sql);
    }

    /**
     * Creates a <code>Statement</code> object that will generate <code>ResultSet</code> 
     * objects with the given type and concurrency.
     *
     * cng
     * @param resultSetType         a result set type; one of <code>ResultSet.TYPE_FORWARD_ONLY</code>,
     *                              <code>ResultSet.TYPE_SCROLL_INSENSITIVE</code>, or <code>ResultSet.TYPE_SCROLL_SENSITIVE</code>
     * @param resultSetConcurrency  a concurrency type; one of <code>ResultSet.CONCUR_READ_ONLY</code> or
     *                              <code>ResultSet.CONCUR_UPDATABLE</code>
     * @return  a new <code>Statement</code> object that will generate <code>ResultSet</code> objects with the given type 
     *          and concurrency
     * @throws SQLException  if a database access error occurs or the given parameters are not <code>ResultSet</code> 
     *                          constants indicating type and concurrency
     * @since   4.0
     */
    public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException 
    {
        return conn.createStatement(resultSetType, resultSetConcurrency);
    }

    /**
     * Creates a <code>Statement</code> object that will generate <code>ResultSet</code> 
     * objects with the given type, concurrency and holdability.
     *
     * cng
     * @param resultSetType         a result set type; one of <code>ResultSet.TYPE_FORWARD_ONLY</code>,
     *                              <code>ResultSet.TYPE_SCROLL_INSENSITIVE</code>, or <code>ResultSet.TYPE_SCROLL_SENSITIVE</code>
     * @param resultSetConcurrency  a concurrency type; one of <code>ResultSet.CONCUR_READ_ONLY</code> or
     *                              <code>ResultSet.CONCUR_UPDATABLE</code>
     * @param resultSetHoldability  one of the following ResultSet constants: <code>ResultSet.HOLD_CURSORS_OVER_COMMIT</code> 
     *                              or <code>ResultSet.CLOSE_CURSORS_AT_COMMIT</code> 
     * @return  a new <code>Statement</code> object that will generate <code>ResultSet</code> objects with the given type, 
     *          concurrency, and holdability
     * @throws SQLException  if a database access error occurs or the given parameters are not <code>ResultSet</code> 
     *                          constants indicating type, concurrency, and holdability
     * @since   4.0
     */
    public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException 
    {
        return conn.createStatement(resultSetType, resultSetConcurrency, resultSetHoldability);
    }


    /**
     * Creates a default <code>PreparedStatement</code> object capable of returning the 
     * auto-generated keys designated by the given array.
     *
     * cng
     * @param sql           a SQL statement that may contain one or more '?' IN parameter placeholders
     * @param columnNames   an array of column names indicating the columns that should be returned 
     *                      from the inserted row or rows
     * @return              a new default <code>PreparedStatement</code> object containing 
     *                      the pre-compiled SQL statement
     * @throws SQLException if a database acess error occurs
     * @since   4.0
     */
    public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException 
    {
         return conn.prepareStatement(sql, columnNames);
    }

    /**
     * Creates a default <code>PreparedStatement</code> object capable of returning 
     * the auto-generated keys designated by the given array.
     *
     * cng
     * @param sql           a SQL statement that may contain one or more '?' IN parameter placeholders
     * @param columnIndexes an array of column indexes indicating the columns that should be returned from 
     *                      the inserted row or rows 
     * @return              a new default <code>PreparedStatement</code> object containing 
     *                      the pre-compiled SQL statement, that is capable of returning the auto-generated
     *                      keys designated by the given array of column indexes
     * @throws SQLException if a database access error occurs
     * @since   4.0
     */
    public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException 
    {
         return conn.prepareStatement(sql, columnIndexes);
    }

    /**
     * Creates a default <code>PreparedStatement</code> that has the capability to 
     * retrieve auto-generated keys.
     *
     * cng
     * @param sql           a SQL statement taht may contain one or more '?' IN parameter placeholders
     * @param autoGeneratedKeys a flag indicating whether auto-generated keys should be returned; one of
     *                          <code>Statement.RETURN_GENERATED_KEYS</code> or
     *                          <code>Statement.NO_GENERATED_KEYS</code> 
     *                      the inserted row or rows 
     * @return              a new default <code>PreparedStatement</code> object containing 
     *                      the pre-compiled SQL statement, that will have the capability of returning 
     *                      auto-generated keys
     * @throws SQLException  if a database access error occurs or the given parameter is not a 
     *                          <code>Statement</code> constant indicating whether auto-generated keys
     *                          should be returned
     * @since   4.0
     */
    public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException 
    {
         return conn.prepareStatement(sql, autoGeneratedKeys);
    }

    /**
     * Creates a default <code>PreparedStatement</code> object that will generate <code>ResultSet</code> 
     * objects with the given type and concurrency. 
     *
     * cng
     * @param sql           a <code>String</code> object that is the SQL statement to be sent to the database; may contain one or 
     *                      more ? IN parameters
     * @param resultSetType a result set type; one of <code>ResultSet.TYPE_FORWARD_ONLY</code>,
     *                      <code>ResultSet.TYPE_SCROLL_INSENSITIVE</code>, or <code>ResultSet.TYPE_SCROLL_SENSITIVE</code>
     * @param resultSetConcurrency  a concurrency type; one of <code>ResultSet.CONCUR_READ_ONLY</code> 
     *                              or <code>ResultSet.CONCUR_UPDATABLE</code>
     * @return              a new <code>PreparedStatement</code> object containing 
     *                      the pre-compiled SQL statement that will produce <code>ResultSet</code> objects with the given type 
     *                      and concurrency
     * @throws SQLException  if a database access error occurs or the given parameter is not a 
     *                          <code>ResultSet</code> objects with the given type and concurrency
     * @since   4.0
     */
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency) throws SQLException 
    {
         return conn.prepareStatement(sql, resultSetType, resultSetConcurrency);
    }

    /**
     * Creates a default <code>PreparedStatement</code> object that will generate <code>ResultSet</code>
     * objects with the given type and concurrency and holdability
     * Exists for conforming to <code>java.sql.Connection</code> interface. Returns <code>null</code>.
     *
     * cng
     * @param sql           a <code>String</code> object that is the SQL statement to be sent to the database; may contain one or 
     *                      more ? IN parameters
     * @param resultSetType a result set type; one of <code>ResultSet.TYPE_FORWARD_ONLY</code>,
     *                      <code>ResultSet.TYPE_SCROLL_INSENSITIVE</code>, or <code>ResultSet.TYPE_SCROLL_SENSITIVE</code>
     * @param resultSetConcurrency  a concurrency type; one of <code>ResultSet.CONCUR_READ_ONLY</code> 
     *                              or <code>ResultSet.CONCUR_UPDATABLE</code>
     * @param resultSetHoldability  one of the folloing <code>ResultSet</code> constants:
     *                              <code>ResultSet.HOLD_CURSORS_OVER_COMMIT</code> or <code>ResultSet.CLOSE_CURSORS_AT_COMMIT</code>
     * @return              a new <code>PreparedStatement</code> object containing 
     *                      the pre-compiled SQL statement that will produce <code>ResultSet</code> objects with the given type,
     *                      concurrency and holdability
     * @throws SQLException  if a database access error occurs or the given parameter is not a 
     *                          <code>ResultSet</code> objects with the given type, concurrency and holdability
     * @since   4.0
     */
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException 
    {
         return conn.prepareStatement(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
    }

    /**
     * Creates a <code>CallableStatement</code> object for calling database stored procedures.
     *
     * cng
     * @param sql   an SQL statement that may contain one or more '?' paraemter placeholders. 
     *              Typically this statement is a JDBC function call escape string.
     * @return      a new default <code>CallableStatement</code> object containing pre-compiled SQL statement
     * @throws SQLException if a database access error occurs
     * @since   4.0
     */
    public CallableStatement prepareCall(String sql) throws SQLException 
    {
        return conn.prepareCall(sql);
    }

    /**
     * Creates a <code>CallableStatement</code> object that will generate <code>ResultSet</code> objects 
     * with the given type and concurrency.
     *
     * cng
     * @param sql   an SQL statement that may contain one or more '?' paraemter placeholders. 
     *              Typically this statemetn is a JDBC function call escape string.
     * @param resultSetType a result set type; one of <code>ResultSet.TYPE_FORWARD_ONLY</code>,
     *                      <code>ResultSet.TYPE_SCROLL_INSENSITIVE</code>, or <code>ResultSet.TYPE_SCROLL_SENSITIVE</code>
     * @param resultSetConcurrency  a concurrency type; one of <code>ResultSet.CONCUR_READ_ONLY</code> or
     *                              <code>ResultSet.CONCUR_UPDATABLE</code>
     * @return      a new default <code>CallableStatement</code> object containing pre-compiled SQL statement
     * @throws SQLException  if a database access error occurs or the given parameters are not <code>ResultSet</code> 
     *                          constants indicating type and concurrency
     * @since   4.0
     */
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency) throws SQLException 
    {
         return conn.prepareCall(sql, resultSetType, resultSetConcurrency);
    }

    /**
     * Creates a <code>CallableStatement</code> object that will generate <code>ResultSet</code> objects
     * with the given type and concurrency.
     * 
     * cng
     * @param sql   an SQL statement that may contain one or more '?' paraemter placeholders. 
     *              Typically this statemetn is a JDBC function call escape string.
     * @param resultSetType         a result set type; one of <code>ResultSet.TYPE_FORWARD_ONLY</code>,
     *                              <code>ResultSet.TYPE_SCROLL_INSENSITIVE</code>, or <code>ResultSet.TYPE_SCROLL_SENSITIVE</code>
     * @param resultSetConcurrency  a concurrency type; one of <code>ResultSet.CONCUR_READ_ONLY</code> or
     *                              <code>ResultSet.CONCUR_UPDATABLE</code>
     * @param resultSetHoldability  one of the following ResultSet constants: <code>ResultSet.HOLD_CURSORS_OVER_COMMIT</code> 
     *                              or <code>ResultSet.CLOSE_CURSORS_AT_COMMIT</code> 
     * @return      a new default <code>CallableStatement</code> object containing pre-compiled SQL statement
     * @throws SQLException if a database access error occurs
     * @since   4.0
     */
    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) 
            throws SQLException 
    {
        return conn.prepareCall(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
    }

    /**
     * Retrieves the first warning reported by calls on this instance's <code>Connection</code> object.
     * 
     * cng
     * @throws SQLException if a database error occurs
     * @since   4.0
     */ 
    public SQLWarning getWarnings() throws SQLException 
    {
        return conn.getWarnings();
    }

    /**
     * Clears all warnings reported for this instance's <code>Connection</code> object.
     * 
     * cng
     * @throws SQLException if a database error occurs
     * @since   4.0
     */ 
    public void clearWarnings() throws SQLException
    {
        conn.clearWarnings();
    }

    /**
     * Retrieves the current holdability of <code>ResultSet</code> objects created using 
     * this <code>Connection</code> object.
     *
     * cng
     * @return               the holdability, one of <code>ResultSet.HOLD_CURSORS_OVER_COMMIT</code> 
     *                       or <code>ResultSet.CLOSE_CURSORS_AT_COMMIT</code> 
     * @throws SQLException  if a database access error occurs
     * @since   4.0
     */
    public int getHoldability() throws SQLException
    {
        return conn.getHoldability();
    }

    /**
     * Changes the holdability of <code>ResultSet</code> objects created using this
     * <code>Connection</code> object to the given holdability. 
     *
     * cng
     * @param holdability   a <code>ResultSet</code> holdability constant; one of <code>ResultSet.HOLD_CURSORS_OVER_COMMIT</code> 
     *                      or <code>ResultSet.CLOSE_CURSORS_AT_COMMIT</code> 
     * @throws SQLException if a database access error occurs or the given parameter is not a <code>ResultSet</code> 
     *                      constant indicating holdability, or the given holdability is not supported
     * @since   4.0
     */
    public void setHoldability(int holdability) throws SQLException
    {
        conn.setHoldability(holdability);
    }

    /**
     * Sets the given catalog name in order to select a subspace of this instance's <code>Connection</code> object's database in which to work.
     * 
     * cng
     * @param catalog   the name of a catalog (subspace in this instance's <code>Connection</code> object's database) 
     *                  in which to work 
     * @throws SQLException if a database access error occurs
     * @since   4.0
     */
    public void setCatalog(String catalog) throws SQLException 
    {
        conn.setCatalog(catalog);
    }

    /**
     * Retrieves this instance's <code>Connection</code> object's current catalog name.
     * 
     * cng
     * @return the current catalog name or <code>null</code> if there is none 
     * @throws SQLException if a database error occurs
     * @since   4.0
     */ 
    public String getCatalog() throws SQLException 
    {
        return conn.getCatalog();
    }

    /**
     * Retrieves the <code>Map</code> object associated with this <code>Connection</code> object.
     * 
     * cng
     * @return the <code>java.util.Map</code> object associated with this instance's <code>Connection</code> object
     * @throws SQLException  if a database access error occurs
     * @since   4.0
     */ 
    public Map<String,Class<?>> getTypeMap() throws SQLException 
    {
        return conn.getTypeMap();
    }

    /**
     * Installs the given <code>TypeMap</code> object as the type map for this instance's
     * <code>Connection</code> object.
     * 
     * cng
     * @param map   the <code>java.util.Map</code> object to install as the replacement for this instance's <code>Connection</code>
     *              object's default type map
     * @throws SQLException if a database error occurs or the given parameter is not a <code>java.util.Map</code> object
     * @since   4.0
     */ 
    public void setTypeMap(Map<String,Class<?>> map) throws SQLException
    {
        conn.setTypeMap(map);
    }

    /**
     * Factory method for creating Struct objects.
     * 
     * cng
     * @param typeName the SQL type name of the SQL structured type that this <code>Struct</code> object maps to. The typeName is the name of the user-defined type that has been defined for this database. It is the value returned by <code>Struct.getSQLTypeName</code>
     * @param attributes the attributes that populate the returned object
     * @return a <code>Struct</code> object that maps to the given SQL type and is populated with the given attributes
     * @throws SQLException if a database error occurs, the <code>typeName</code> is null or this method is called on a closed connection
     * @since 4.2.0
     */
    public Struct createStruct(String typeName, Object[] attributes) throws SQLException
    {
        return conn.createStruct(typeName,attributes);
    }

    /**
     * Factory method for creating Array objects.
     * 
     * cng
     * @param typeName the SQL name of the type the elements of the array map to. The typeName is a database-specific name which may be the name of a built-in type, a user-defined type or a standard SQL type supported by this database. This is the value returned by Array.getBaseTypeName
     * @param elements the elements that populate the returned object 
     * @return an Array object whose elements map to the specified SQL type
     * @throws SQLException if a database error occurs, the JDBC type is not appropriate for the typeName and the conversion is not supported, the typeName is null or this method is called on a closed connection 
     * @since 4.2.0
     */
    public Array createArrayOf(String typeName, Object[] elements) throws SQLException
    {
        return conn.createArrayOf(typeName,elements);
    }

    /**
     * Returns a list containing the name and current value of each client info property supported by the driver.
     * 
     * cng
     * @return a <code>Properties</code> object that contains the name and current value of each of the client info properties supported by the driver
     * @throws SQLException if the database server returns an error when fetching the client info values from the database or this method is called on a closed connection
     * @since 4.2.0
     */
    public Properties getClientInfo() throws SQLException
    {
        return conn.getClientInfo();
    }

    /**
     * Returns the value of hte client info property specified by name.
     * 
     * cng
     * @param name the name of the client infor property to retrieve
     * @return the value of the client info property specified
     * @throws SQLException if the database server returns an error when fetching the client info values from the database or this method is called on a closed connection
     * @since 4.2.0
     */
    public String getClientInfo(String name) throws SQLException
    {
        return conn.getClientInfo(name);
    }

    /**
     * Sets the value of hte connection's client info properties.
     * 
     * cng
     * @param properties the list of client info properties to set
     * @throws SQLClientInfoException if the database server returns an error when setting the client info values on the database server or this method is called on a closed connection
     * @since 4.2.0
     */
    public void setClientInfo(Properties properties) throws SQLClientInfoException
    {
        conn.setClientInfo(properties);
    }

    /**
     * Sets the value of the client info property specified by name to the value specified by the value.
     * 
     * cng
     * @param name the name of the client infor property to retrieve
     * @param value the new value for the client info property
     * @throws SQLClientInfoException if the database server returns an error when setting the client info values on the database server or this method is called on a closed connection
     * @since 4.2.0
     */
    public void setClientInfo(String name, String value) throws SQLClientInfoException
    {
        conn.setClientInfo(name,value);
    }

    /** 
     * Returns true if the connection has not been closed and is still valid.
     *
     * cng
     * @param timeout the time in seconds to wait for the database operation used to validate the connection to complete. If the timeout period expires before the operation completes, this method returns false. A value of 0 indicates a timeout isnot applied to the database operation.
     * @return true if the connection is valid; false otherwise
     * @throws SQLException if the value supplied for timeout is less than 0
     * @since 4.2.0
     */
    public boolean isValid(int timeout) throws SQLException
    {
        boolean valid = true;
        if (timeout < 0)
        {
            throw new SQLException("Negative timeout supplied");
        }
        try
        {
            valid = conn.isValid(timeout);
        } catch(Exception e)
        {
            valid = false;
        }
        return valid;
    }
    
    /**
     * Constructs an object that implements the <code>SQLXML</code> interface. 
     * 
     * cng
     * @return an object that implements the <code>SQLXML</code> interface.
     * @throws SQLException if an object that implements the <code>SQLXML</code> interface can not be constructed, this method is called on a closed connection on a database access error occurs.
     * @since 4.2.0
     */
    public SQLXML createSQLXML() throws SQLException
    {
        return conn.createSQLXML();
    }

    /**
     * Constructs an object that implements the <code>NClob</code> interface.
     * 
     * cng
     * @return an object that implments the <code>NClob</code> interface
     * @throws SQLException if an object that implments the <code>NClob</code> interface can not be constructed, this method is call on a closed connection or a database access error occurs.
     * @since 4.2.0
     */
    public NClob createNClob() throws SQLException
    {
        return conn.createNClob();
    }

    /**
     * Constructs an object that implements the <code>Clob</code> interface.
     * 
     * cng
     * @return an object that implments the <code>Clob</code> interface
     * @throws SQLException if an object that implments the <code>Clob</code> interface can not be constructed, this method is call on a closed connection or a database access error occurs.
     * @since 4.2.0
     */
    public Clob createClob() throws SQLException
    {
        return conn.createClob();
    }

    /**
     * Constructs an object that implements the <code>Blob</code> interface.
     * 
     * cng
     * @return an object that implments the <code>Blob</code> interface
     * @throws SQLException if an object that implments the <code>Blob</code> interface can not be constructed, this method is call on a closed connection or a database access error occurs.
     * @since 4.2.0
     */
    public Blob createBlob() throws SQLException
    {
        return conn.createBlob();
    }

    /**
     * Returns true if this either implements the interface argument or is directly or indirectly a wrapper for an object that does.
     *
     * cng
     * @param iface a Class defining an interface
     * @return <code>true</code> if this implements the interface or directly or indirectly wraps an object that does
     * @throws SQLException if an error occurs while determining whether this is a wrapper for an object with the given interface
     * @since 4.2.0
     */
    public boolean isWrapperFor(Class<?> iface) throws SQLException
    {
        return conn.isWrapperFor(iface);
    }
     
    /**
     * Returns an object that implements the given interface to allow access to non-standard methods, or standard methods not exposed by the proxy.
     *
     * cng
     * @param iface a Class defining an interface that the result must implement
     * @return an object that implements the interface. May be a proxy for the actual implementing object.
     * @throws SQLException if no object found that implements the interface
     * @since 4.2.0
     */
    public <T> T unwrap(Class<T> iface) throws SQLException
    {
        return conn.unwrap(iface);
    }
}
