/*******************************************************************************

    fpga_mm.c
    Copyright 2018-2020 NetApp, Inc. All rights reserved.

    Memory-mapped FPGA loadable kernel driver module.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

*******************************************************************************/

#include <linux/module.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/uaccess.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/io.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/eventfd.h>
#include <linux/jiffies.h>
#include <linux/kthread.h>
#include <linux/delay.h>
#include <asm/nmi.h>
/*#include <linux/netapp/eseries-fpga.h>*/

#include "fpga.h"

#define FPGA_VERSION   "2.0"

#define FPGA_MM_NAME   FPGA_NAME "_mm"

/**
 * Use DBG for standard debug message queue messages (always on).
 * Use EXT_DBG for extended debug messages that are turned on by compile time switch.
 * EXTENDED_DEBUG turns on EXT_DBG messages.
 */
#define DBG(msg, args...) printk(FPGA_MM_NAME ": %s: " msg , __FUNCTION__, ## args)

#undef EXTENDED_DEBUG
#ifdef EXTENDED_DEBUG
    #define EXT_DBG(msg, args...) printk(FPGA_MM_NAME ": %s: " msg , __FUNCTION__, ## args)
#else
    #define EXT_DBG(msg, args...)
#endif

/*
 * This needs to be less than the shortest timeout value
 * among all the platforms supported by this driver.
 */
#define WDT_INTERVAL_MSEC      500

/*
 * The FPGA specification allows for a configurable IRQ number but the actual FPGA implementation
 * does not implement the feature. The only supported value is 6.
 */
#define SERIRQ_NUM 6

/**
 * The FPGA device driver data structure
 */
struct fpgaDev
{
    uintptr_t          phys_regs_base;    /* Physical kernel address of platform "registers" */
    unsigned long      virt_regs_base;    /* Virtual kernel address of platform "registers" */
    unsigned long      regs_size;         /* Size of FPGA "register" space */
    dev_t              device_number;     /* The character device MAJOR/MINOR device number */
    char               device_name[30];   /* Device name */
    unsigned long      watchdogTimeout;   /* Watchdog time out value in jiffies */

    unsigned int       irq;
    struct eventfd_ctx* trigger;          /* EventFD for communicating interrupt to user space */
    atomic_t           irq_disabled;      /* Flag to indicate if IRQ line is disabled */
};

/* Pointer to class structure */
static  struct class*      fpgaMmClass = 0;
/* cdev major number */
static  int                fpgaMmMajor = 0;

/* Counter for number of interrupts */
static unsigned long    interruptCount = 0;

/* Variable to hold hardware ID as external code may need it. */
static int32_t          hardwareId = 0;

/* Struct for our FPGA device */
static struct fpgaDev fpgaMmDev;

/**
 * Character driver supported file operations:
 *     ioctl
 *     open
 *     close/release
 */
static long fpgaMmIoctl(struct file*, unsigned int, unsigned long);
static int  fpgaMmOpen(struct inode*, struct file*);
static int  fpgaMmRelease(struct inode*, struct file*);
static int  fpgaMmap(struct file*, struct vm_area_struct*);

static struct file_operations fpgaFileOps =
{
    .owner            = THIS_MODULE,
    .unlocked_ioctl   = fpgaMmIoctl,
    .open             = fpgaMmOpen,
    .release          = fpgaMmRelease,
    .mmap             = fpgaMmap,
};

/* IO ports that contain FPGA address info on NetApp systems. */
#define IOPORT_FPGA_ADDR_BYTE3  0x701
#define IOPORT_FPGA_ADDR_BYTE2  0x700

/**
 * Macros to read/write platform registers.
 */
#define IS_PLATFORM_REGS_DEFINED()       (fpgaMmDev.virt_regs_base != 0)
#define WRITE_REG(regoff, value)    (*((char*)(fpgaMmDev.virt_regs_base + regoff)) = ((value) & 0xFF))
#define READ_REG(regoff)            (*((char*)(fpgaMmDev.virt_regs_base + regoff)))

#define REG_OFFSET_WATCHDOG     0x20
#define     RESET_WDT           0x01    /* Bit 0 */
#define     ENABLE_WDT          0x80    /* Bit 7 */

#define REG_OFFSET_REBOOT       0x80
#define     REBOOT_PROC         0x5A

#define REG_OFFSET_SERIRQ       0xE0
#define     ENABLE_SERIRQ       0x80    /* Bit 7 */

/**
 * The FPGA ISR.
 *
 * This ISR is executed on SERIRQ interrupts from the FPGA device.
 * It will wake up any processes that were waiting on the interrupt
 * to occur.
 *
 */
static irqreturn_t
fpgaIsr(int irq, void* dev_id)
{
    struct fpgaDev* fdev = (struct fpgaDev*)dev_id;

    interruptCount++;
    EXT_DBG("fpga interruptCount = %ld\n", interruptCount);

    /* If we don't have any regs_base defined, exit. */
    if (!IS_PLATFORM_REGS_DEFINED())
        return IRQ_NONE;

    /*
     * Disable the interrupt line for the entire system. It will be enabled
     * later either by user space code via FPGA_IOCTL_ACK_INT ioctl system call
     * or when watchdog timeout is handled in the below if block.
     */
    disable_irq_nosync(irq);

    /* Write to eventfd. Clearing of interrupt and unmasking
     * will be done by the sleeping thread.
     */
    eventfd_signal(fdev->trigger, 1);
    atomic_set(&fdev->irq_disabled, 1);

    return IRQ_HANDLED;
}

/**
 * Function to pacify the watchdog for a certain period of time.
 *
 */
static int
pacifyWatchdog (void* unused)
{
    uint8_t tmp;
    if (fpgaMmDev.watchdogTimeout == 0)
    {
        /*
         * This should never happen as the IOCTL should always set before
         * calling this function.
         */
        DBG("Pacify request with watchdog disabled. Returning...");
        return -1;
    }

    while (time_before(jiffies, fpgaMmDev.watchdogTimeout))
    {
        tmp = READ_REG(REG_OFFSET_WATCHDOG);
        WRITE_REG( REG_OFFSET_WATCHDOG, (tmp | RESET_WDT) );
        EXT_DBG("Pacifying watchdog. watchdogTimeout = %ld jiffies = %ld\n",
                fpgaMmDev.watchdogTimeout, jiffies);

        msleep(WDT_INTERVAL_MSEC);
    }

    DBG("Watchdog pacify request done. watchdogTimeout = %ld jiffies = %ld\n",
            fpgaMmDev.watchdogTimeout, jiffies);

    return 0;
}

/**
 * FPGA processor reset function
 *
 * This function is passed to the kernel to call during kernel
 * initiated processor resets so that it resets using the FPGA
 * instead of how the kernel wishes to reset it
 */
static void processorReset(void* arg)
{
    (void)arg;

    EXT_DBG( "resetting processor\n");
    WRITE_REG(REG_OFFSET_REBOOT, REBOOT_PROC);
}

/**
 * FPGA perform the shadow register write sequence
 *
 */
static
int
fpgaMmShadowWrite
    (
    fpga_reg* regInfo
    )
{
    int iteration;
    uint8_t regValue;
    uint8_t newValue;

    /* Try shadow writes up to 3 times */
    for (iteration = 0; iteration < 3; ++iteration)
    {
        regValue = READ_REG(regInfo->offset) & ~regInfo->mask;
        regValue |= (regInfo->data & regInfo->mask);
        EXT_DBG("Writing register offset x%x, mask x%x, Value x%x\n",
                regInfo->offset, regInfo->mask, regValue);

        /* First write to shadow register, then the active register */
        WRITE_REG(regInfo->shadow, regValue);
        WRITE_REG(regInfo->offset, regValue);

        /* Read it back and stop if they match */
        newValue = READ_REG(regInfo->offset);
        if (newValue == regValue)
            return 1;
    }
    DBG("Failure writing shadow register sequence: regValue=x%x newValue=x%x\n",
         regValue, newValue);
    return 0;
}

/**
 * MM FPGA ioctl function.
 *
 * This function operates on the following ioctl commands:
 *
 *     FPGA_IOCTL_START_WDT:        not implemented (-1)
 *     FPGA_IOCTL_DISABLE_WDT:      not implemented (0)
 *     FPGA_IOCTL_PET_WDT:          bumps timeout value (0)
 *     FPGA_IOCTL_READ_REG:         reads FPGA offset (0)
 *     FPGA_IOCTL_WRITE_REG:        writes FPGA offset (0)
 *     FPGA_IOCTL_WRITE_SHADOW:     writes FPGA offset and its shadow reg (0)
 *     FPGA_IOCTL_WRITE_PROT_REG:   writes FPGA offset (0)
 *     FPGA_IOCTL_REG_PROT_WRITE:   not implemented (0)
 *     FPGA_IOCTL_READ_NVSRAM       Returns a 0 to the user (0)
 *     FPGA_IOCTL_WRITE_NVSRAM      Writes a byte to thebitbucket (0)
 *     FPGA_IOCTL_REGISTER_ISR_INT  registers interrupt service routine.
 *     FPGA_IOCTL_REGISTER_ISR_MSI  registers interrupt service routine.
 *     FPGA_IOCTL_ACK_INT           enable irq line.
 *
 * Other commands will return error (-1).
 *
 */
static long fpgaMmIoctl(struct file* filp, unsigned int command, unsigned long data)
{
    unsigned long regData;
    fpga_reg* regP = 0;
    fpga_reg regInfo;
    unsigned long copyRet = 0;
    unsigned long newTimeout;
    int32_t __user *hwid_ptr;
    struct task_struct *wdTask;

    /* Not used */
    (void)filp;

    EXT_DBG("Doing ioctl %x on %lx:  fdev=%p\n", command, data, &fpgaMmDev);

    if ((command != FPGA_IOCTL_ACK_INT) &&
        (command != FPGA_IOCTL_DISABLE_WDT) &&
        (command != FPGA_IOCTL_GET_HW_ID) &&
        (command != FPGA_IOCTL_SET_HW_ID))
    {
        regP = ((fpga_reg*)((unsigned long *)data));
        if(regP == NULL)
        {
            DBG("No register info received\n");
            return -1;
        }

        /* The copy_from/to_user functions return the number of uncopied bytes,
         * so a return of zero is a successul status
         */
        copyRet = copy_from_user(&regInfo, regP, sizeof(regInfo));

        if (copyRet)
            return -1;
    }

    switch (command)
    {
        case FPGA_IOCTL_START_WDT:
            /* Not yet implemented (not yet needed) */
            /* enable the watchdog timer */
            DBG("Invalid request: FPGA_IOCTL_START_WDT\n");
            return -1;

        case FPGA_IOCTL_DISABLE_WDT:
            WRITE_REG(REG_OFFSET_WATCHDOG, 0);
            fpgaMmDev.watchdogTimeout = 0;
            DBG("watchdog timer disabled\n");
            return 0;

        case FPGA_IOCTL_PET_WDT:
            newTimeout = jiffies + msecs_to_jiffies(regInfo.data);
            if (time_before(fpgaMmDev.watchdogTimeout, newTimeout))
                fpgaMmDev.watchdogTimeout = newTimeout;
            DBG("current jiffy = %ld watchdogTimeout is %ld\n", jiffies, fpgaMmDev.watchdogTimeout);

            /* We spawn off a kthread that will pacify the watchdog hardware for
             * the requested amount of time.
             */
            wdTask = kthread_run( pacifyWatchdog, NULL, "kthread_wdt");
            if (!wdTask)
            {
                DBG("Failed to create pacifying thread!\n");
                return -1;
            }

            return 0;

        case FPGA_IOCTL_READ_REG:
            EXT_DBG("Doing fpga ioctl %x (READ_REG) on %lx\n", command, data);
            EXT_DBG("Reading register offset %x\n", regInfo.offset);
            regInfo.data = READ_REG(regInfo.offset) & 0xFF;
            copyRet = copy_to_user(regP, &regInfo, sizeof(regInfo));
            break;

        case FPGA_IOCTL_WRITE_REG:
            EXT_DBG("Doing fpga ioctl %x (WRITE_REG) on %lx\n", command, data);
            regData = READ_REG(regInfo.offset) & ~regInfo.mask;
            regData |= (regInfo.data & regInfo.mask);
            EXT_DBG("Writing register offset %x, mask %x, data %lx\n",
                     regInfo.offset, regInfo.mask, regData);
            WRITE_REG(regInfo.offset, regData);
            break;

        case FPGA_IOCTL_WRITE_SHADOW:
            EXT_DBG("Doing fpga ioctl %x (WRITE_SHADOW) on %lx\n", command, data);
            if (!fpgaMmShadowWrite(&regInfo))
            {
                return -1;
            }
            break;

        case FPGA_IOCTL_WRITE_PROT_REG:
            EXT_DBG("Doing fpga ioctl %x (WRITE_PROT_REG) on %lx\n", command, data);
            /* Just do the same thing as FPGA_IOCTL_WRITE_REG for now */
            regData = READ_REG(regInfo.offset) & ~regInfo.mask;
            regData |= (regInfo.data & regInfo.mask);
            EXT_DBG("Writing register offset %x, mask %x, data %lx\n",
                     regInfo.offset, regInfo.mask, regData);
            WRITE_REG(regInfo.offset, regData);
            break;

        case FPGA_IOCTL_REG_PROT_WRITE:
            /* Not implemented, just print DBG info so we can see it happen */
            DBG("Stub Function: PROT WRITE sequence requested, not implemented.\n");
            break;

        case FPGA_IOCTL_READ_NVSRAM:
            /* MM FPGA doesn't have NVSRAM, return data of 0 for now */
            DBG("Doing fpga ioctl %x (READ_NVSRAM) on %lx\n", command, data);

            DBG("Reading NVSRAM offset %x\n", regInfo.offset);
            regInfo.data = 0;
            copyRet = copy_to_user(regP, &regInfo, sizeof(regInfo));
            break;

        case FPGA_IOCTL_WRITE_NVSRAM:
            /* MM FPGA doesn't have NVSRAM, just log and continue for now */
            DBG("Doing fpga ioctl %x (WRITE_NVSRAM) on %lx\n", command, data);

            DBG("Writing NVSRAM offset %x, mask %x, data %x\n", regInfo.offset, regInfo.mask, regInfo.data);
            break;

        /* Either IOCTL can be used to configure the FPGA interrupt. */
        case FPGA_IOCTL_REGISTER_ISR_INT:
        case FPGA_IOCTL_REGISTER_ISR_MSI:
            if (fpgaMmDev.trigger)
            {
                DBG("EventFD already registered\n");
                return -1;
            }

            if (fpgaMmDev.irq == 0)
            {
                /* Program IRQ number into FPGA, enable SERIRQ bus and allocate the interrupt line. */
                fpgaMmDev.irq = SERIRQ_NUM;
                WRITE_REG(REG_OFFSET_SERIRQ, fpgaMmDev.irq | ENABLE_SERIRQ);
                if (request_irq(fpgaMmDev.irq, fpgaIsr, 0, FPGA_MM_NAME, &fpgaMmDev) != 0)
                {
                    DBG(FPGA_MM_NAME " IRQ registration failed\n");

                    /* Clear the SERIRQ enable bit in the FPGA. */
                    WRITE_REG(REG_OFFSET_SERIRQ, 0);

                    fpgaMmDev.irq = 0;
                    return -1;
                }
            }

            EXT_DBG("Doing fpga ioctl %x (REGISTER_ISR_INT) on %lx\n", command, data);

            fpgaMmDev.trigger = eventfd_ctx_fdget(regInfo.data);
            if (IS_ERR(fpgaMmDev.trigger))
            {
                DBG("Error in fpga ioctl %x (REGISTER_ISR_INT) on %lx\n", command, data);
                fpgaMmDev.trigger = 0;
                return -1;
            }

            DBG("Enabled INT trigger: %p\n", fpgaMmDev.trigger);
            break;

        case FPGA_IOCTL_ACK_INT:
            EXT_DBG("Doing fpga ioctl %x (ACK_INT) on %lx\n", command, data);
            if (fpgaMmDev.irq == 0)
            {
                DBG("IRQ not registered yet.\n");
                return -1;
            }

            /* Unbalanced enable for IRQ is not allowed.
             * Enable the IRQ only if was disabled earlier.
             */
            if (atomic_dec_and_test(&fpgaMmDev.irq_disabled))
                enable_irq(fpgaMmDev.irq);
            break;

        case FPGA_IOCTL_SET_HW_ID:
            hardwareId = (int32_t) data;
            copyRet = 0;
            break;

        case FPGA_IOCTL_GET_HW_ID:
            hwid_ptr = (int32_t *)data;
            if (put_user(hardwareId, hwid_ptr))
                return -EFAULT;

            copyRet = 0;
            break;

        default:
            return -ENOTTY;
    }

    return copyRet ? -1: 0;
}


/**
 * FPGA open routine.
 *
 * Nothing to do since there's only a single instance of the MM FPGA
 *
 */
static int fpgaMmOpen(struct inode* inode, struct file* filp)
{
    EXT_DBG("Opening fpga node\n");

    return 0;
}

/**
 * FPGA close routine.
 *
 * No special operations are required for this close routine.
 *
 */
static int fpgaMmRelease(struct inode* inode, struct file* filp)
{
    EXT_DBG("Closing fpga node\n");

    /*
     * Nothing was allocated or mapped during open, so there's really nothing
     * to do here. No resources need to be freed or anything like that.
     */
    return 0;
}

/**
 * FPGA mmap routine.
 *
 * Maps the FPGA region to user space
 *
 */
static int fpgaMmap(struct file* filp, struct vm_area_struct* vma)
{
    unsigned long size = vma->vm_end - vma->vm_start;
    unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
    (void)filp;   /* not used */

    DBG("Attempting to map FPGA registers, size 0x%lx\n", size);

    /* Make sure not trying to get more memory than permitted */
    if (size > fpgaMmDev.regs_size)
    {
        DBG("Invalid size being mapped (%ld)\n", size);
        return -EINVAL;
    }

    /* Make sure starting point is at beginning of FPGA */
    if (offset != 0)
    {
        DBG("Offset must be zero (%ld)\n", offset);
        return -EINVAL;
    }

    /* Make sure user specified shared page */
    if (!(vma->vm_flags & VM_SHARED))
    {
        DBG("Invalid vm_flags (%lx)\n", vma->vm_flags);
        return -EINVAL;
    }

    /* Prevent memory from being swapped out */
    vma->vm_flags |= VM_LOCKED;

    /* Now map the region */
    if (remap_pfn_range(vma, vma->vm_start,
                        PFN_DOWN(fpgaMmDev.phys_regs_base),
                        size, PAGE_SHARED))
    {
        DBG("Unable to map region\n");
        return -EAGAIN;
    }

    return 0;
}

static int fpgaMmClassCreate(void)
{
    fpgaMmMajor = register_chrdev(0, FPGA_MM_NAME, &fpgaFileOps);
    if (fpgaMmMajor < 0)
    {
        DBG("Failed to allocate major device numbers\n");
        return fpgaMmMajor;
    }
    DBG("Character device number assigned to major(%d)\n", fpgaMmMajor);

    /* Only create the class if not already created or if last attempt failed */
    fpgaMmClass = class_create(THIS_MODULE, FPGA_MM_NAME);
    if(IS_ERR(fpgaMmClass))
    {
        unregister_chrdev(fpgaMmMajor, FPGA_MM_NAME);
        DBG(FPGA_MM_NAME " class registration failed\n");
        return PTR_ERR(fpgaMmClass);
    }
    DBG(FPGA_MM_NAME " class registration completed\n");

    return 0;
}

static void fpgaMmClassDestroy(void)
{
    class_destroy(fpgaMmClass);
    unregister_chrdev(fpgaMmMajor, FPGA_MM_NAME);
}

/*
 * Catches NMIs that would be generated if the FPGA
 * address IO ports are not present.
 */
static int fpga_nmi_catcher(unsigned int cmd, struct pt_regs *regs)
{
    return NMI_HANDLED;
}

/*
 * This function will check to see if there is an fpga on the
 * board, and initialize the driver appropriately if so. It returns
 * 0 if a physical device was initialized, and -1 if none was found
 * indicating that the driver should be set up with a "fake" FPGA
 * of allocated memory (useful for new boards).
 */
static int fpgaMmRealDevInit(const char* name)
{
    unsigned char addrBits31_24;
    unsigned char addrBits23_16;

    DBG(FPGA_MM_NAME "fpgaMmRealDevInit 1\n");
    if (request_region(IOPORT_FPGA_ADDR_BYTE2, 2, name))
    {
        /*
         * IO port access cause NMIs where there is not HW to access,
         * so catch NMIs here assuming any are from the attempted reads.
         */
        register_nmi_handler(NMI_LOCAL, fpga_nmi_catcher, 0, "fpga");
        addrBits31_24 = inb(IOPORT_FPGA_ADDR_BYTE3);
        addrBits23_16 = inb(IOPORT_FPGA_ADDR_BYTE2);
        unregister_nmi_handler(NMI_LOCAL, "fpga");
        release_region(IOPORT_FPGA_ADDR_BYTE2, 2);

        /* Sanity check for Falcon.  As other platforms are added
         * we may want to enhance this sanity check somehow.
         */
        if (addrBits31_24==0xFC && addrBits23_16==0x80)
        {
            fpgaMmDev.phys_regs_base = 0xFC800000;
            fpgaMmDev.regs_size = 0xD000;
            fpgaMmDev.virt_regs_base = (unsigned long)ioremap_nocache(fpgaMmDev.phys_regs_base,
                                                                      fpgaMmDev.regs_size);
            if (fpgaMmDev.virt_regs_base != 0)
            {
                return 0;
            }

            /* If ioremap failed we fall through and return error */
        }
    }

    return -1;
}

static int fpgaMmDevCreate(void)
{
    DBG(FPGA_MM_NAME "fpgaMmDevCreate called\n");

    if (IS_PLATFORM_REGS_DEFINED())
    {
        DBG(FPGA_MM_NAME "MM FPGA already exists\n");
        return 0;
    }

    /* Only allow a single device - fpga_mm0 */
    strncpy(fpgaMmDev.device_name, FPGA_MM_NAME "0", sizeof(fpgaMmDev.device_name));

    if (fpgaMmRealDevInit(fpgaMmDev.device_name) != 0)
    {
        DBG(FPGA_MM_NAME "MM FPGA failed to init device.  Using fake FPGA.\n");
        /* If failed, set up a fake "register space" to keep
         * attempted accesses from blowing up.
         */
        fpgaMmDev.regs_size = 0x1000;
        fpgaMmDev.virt_regs_base = (unsigned long)kmalloc(fpgaMmDev.regs_size, GFP_KERNEL);
        if (fpgaMmDev.virt_regs_base == 0)
        {
            DBG("Could not allocate memory for the virt base\n");
            return -ENODEV;
        }

        /* Initialize registers */
        memset((void*)fpgaMmDev.virt_regs_base, 0, fpgaMmDev.regs_size);
    }

    /* Create the device */
    fpgaMmDev.device_number = MKDEV(fpgaMmMajor, 0);
    DBG(FPGA_MM_NAME "device number: %d\n", fpgaMmDev.device_number);

    if (IS_ERR(device_create(fpgaMmClass, NULL, fpgaMmDev.device_number, NULL, fpgaMmDev.device_name)))
    {
        DBG(FPGA_MM_NAME " class device creation failed\n");
        return -ENODEV;
    }

    /* Register the reboot function for this FPGA device */
    /* eseries_register_reboot_function(processorReset, 0); */

    /* Now the driver is initialized and registered and ready for action. */
    return 0;
}

static int fpgaMmDevDestroy(void)
{
    DBG("Removing MM FPGA\n");

    if (IS_PLATFORM_REGS_DEFINED())
    {
        if (fpgaMmDev.trigger)
        {
            /* Notify userspace the eventfd is going away. */
            eventfd_signal(fpgaMmDev.trigger, 1);

            /* Release the reference to the internal eventfd context. */
            eventfd_ctx_put(fpgaMmDev.trigger);
        }

        if (fpgaMmDev.irq > 0)
        {
            /* Clear the SERIRQ enable bit in the FPGA. */
            WRITE_REG(REG_OFFSET_SERIRQ, 0);

            /* Detach the kernel from the FPGA interrupt. */
            free_irq(fpgaMmDev.irq, &fpgaMmDev);

            fpgaMmDev.irq = 0;
        }

        /* eseries_deregister_reboot_function(processorReset);*/

        device_destroy(fpgaMmClass, fpgaMmDev.device_number);
        kfree((void*)fpgaMmDev.virt_regs_base);
        memset(&fpgaMmDev, 0, sizeof(struct fpgaDev));
        DBG("device destroyed\n");
    }

    return 0;
}

/**
 * Standard Linux module registration
 */

static __init int fpgaMmInit(void)
{
    int rc = 0;
    DBG("Initializing FPGA driver\n");
    rc = fpgaMmClassCreate();
    if (rc)
    {
        DBG("Could not perform fpga class create\n");
        return rc;
    }

    rc = fpgaMmDevCreate();
    if (rc)
    {
        DBG("Could not perform fpga dev create\n");
    }
    return rc;
}

static __exit void fpgaMmExit(void)
{
    fpgaMmDevDestroy();
    fpgaMmClassDestroy();
}

module_init(fpgaMmInit);
module_exit(fpgaMmExit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("NetApp FPGA MM Driver " FPGA_VERSION);
MODULE_AUTHOR("NetApp, Inc");

