/*******************************************************************************

    fpga.c
    Copyright 2014-2020 NetApp, Inc. All rights reserved.

    PCI FPGA loadable kernel driver module.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

*******************************************************************************/

#include <linux/module.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/uaccess.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/io.h>
#include <linux/sched.h>
#include <linux/eventfd.h>
#include <linux/jiffies.h>
#include <linux/kthread.h>
#include <linux/delay.h>
#include <linux/sysrq.h>

/*#include <linux/netapp/eseries-fpga.h>*/

#include "fpga.h"

#define FPGA_VERSION   "2.0"

#define FPGA_PCI_NAME  FPGA_NAME "_pci"

/**
 * Use DBG for standard debug message queue messages (always on).
 * Use EXT_DBG for extended debug messages that are turned on by compile time switch.
 * EXTENDED_DEBUG turns on EXT_DBG messages.
 */
#define DBG(msg, args...) printk(FPGA_PCI_NAME ": %s: " msg , __FUNCTION__, ## args)

#undef EXTENDED_DEBUG
#ifdef EXTENDED_DEBUG
    #define EXT_DBG(msg, args...) printk(FPGA_PCI_NAME ": %s: " msg , __FUNCTION__, ## args)
#else
    #define EXT_DBG(msg, args...)
#endif

#ifndef PCI_VENDOR_ID_NETAPP
#define PCI_VENDOR_ID_NETAPP 0x1275
#endif

/**
 * FPGA register offsets needed by the driver
 */
#define WDT_PROTECT            0x09
#define WD_TIMER_ENABLE        0x0A
#define WDT_KEY                0x0B
#define WDT_CONTROL            0x0C
#define WDT_RESET_INFO         0x24
#define ENCODED_WRITE          0x30
#define INTERRUPT_MASK_INT_B   0x91
#define PROTECTED_WRITE1       0x31
#define BOARD_RESET_UNLOCK     0x5A
#define BOARD_RESET            0x10

#define FPGA_REGS_SIZE 0x100
#define FPGA_MAX_DEVS  5

#define WDT_FAULT              0x10

/*
 * This needs to be the shortest timeout value
 * among all the platforms supported by this driver.
 */
#define WDT_INTERVAL_MSEC      400

/*
 * Reset the watchdog interrupt count to 1 if there hasn't been another
 * watchdog interrupt for this long.
 */
#define RESET_WDOG_INT_COUNT_AFTER_X_SECS   600

/*
 * Panic if the watchdog interrupt count gets to this number.
 */
#define PANIC_AFTER_X_WDOG_INTS             10

/**
 * The FPGA device driver data structure
 */
struct fpgaDev
{
    unsigned long long phys_regs_base;    /* Physical address of the platform registers (BAR0) */
    unsigned long      virt_regs_base;    /* Virtual kernel address of platform registers */
    unsigned long      regs_size;         /* Size of PCI space BAR0 */
    dev_t              device_number;     /* The character device MAJOR/MINOR device number */
    uint32_t           irq_registered;    /* Flag to indicate if ISR is registered */
    uint32_t           msi_enabled;       /* Flag to indicate if MSI is used */
    uint32_t           numProtWrite;      /* Number of bytes to write for protected */
                                          /* write sequence */
    unsigned char*     protWriteSeq;      /* Sequence of bytes to write for protected */
                                          /* write sequence */
    unsigned long long phys_nvsram_base;  /* Physical address of the NVSRAM (BAR1) */
    unsigned long      virt_nvsram_base;  /* Virtual kernel address of NVSRAM */
    unsigned long      nvsram_size;       /* Size of PCI space BAR1 */
    struct pci_dev*    pdev;              /* Point to kernel pci structure */
    struct eventfd_ctx* trigger;          /* EventFD for communicating interrupt to user space */
    char               device_name[30];   /* Device name */
    atomic_t           irq_disabled;      /* Flag to indicate if IRQ line is disabled */
    unsigned long      watchdogTimeout;   /* Watchdog time out value in jiffies */
};

/* Pointer to class structure */
static  struct class*      fpgaPciClass = 0;
/* cdev major number */
static  int                fpgaPciMajor = 0;

/* Pointer to the list of FPGA PCI devices */
static struct fpgaDev*  fpgaPciList[FPGA_MAX_DEVS];

/* Counter for number of interrupts */
static unsigned long    interruptCount = 0;

/* Variable to hold hardware ID as external code may need it. */
static int32_t          hardwareId = 0;

/* Timestamp of last watchdog ISR */
static unsigned long    jiffiesAtLastWatchdog = 0;

/**
 * Character driver supported file operations:
 *     ioctl
 *     open
 *     close/release
 */
static long fpgaIoctl(struct file*, unsigned int, unsigned long);
static int  fpgaOpen(struct inode*, struct file*);
static int  fpgaRelease(struct inode*, struct file*);
static int  fpgaMmap(struct file*, struct vm_area_struct*);

static struct file_operations fpgaFileOps =
{
    .owner            = THIS_MODULE,
    .unlocked_ioctl   = fpgaIoctl,
    .open             = fpgaOpen,
    .release          = fpgaRelease,
    .mmap             = fpgaMmap,
};

/**
 * Macros to read/write platform registers.
 */
#define IS_PLATFORM_REGS_DEFINED(dev)    (dev->virt_regs_base != 0)
#define WRITE_REG(dev, regoff, value)    (*((char*)(dev->virt_regs_base + (regoff))) = ((value) & 0xFF))
#define READ_REG(dev, regoff)            (*((char*)(dev->virt_regs_base + (regoff))))
#define WRITE_NVSRAM(dev, offset, value) (*((char*)(dev->virt_nvsram_base + (offset))) = ((value) & 0xFF))
#define READ_NVSRAM(dev, offset)         (*((char*)(dev->virt_nvsram_base + (offset))))


/**
 * Macros to manage watchdog timer
 */

#define FPGA_WATCHDOG_DISABLE(dev)          \
    do {                                    \
        WRITE_REG(dev, WDT_PROTECT, 0);     \
        WRITE_REG(dev, WD_TIMER_ENABLE, 0); \
        WRITE_REG(dev, WDT_PROTECT, 1);     \
    } while(0)

#define FPGA_WATCHDOG_CLR(dev)  \
       WRITE_REG(dev, WDT_CONTROL, READ_REG(dev, WDT_KEY))

#define FPGA_WATCHDOG_FAULT(dev) \
        ((READ_REG(dev, WDT_RESET_INFO) & WDT_FAULT) == WDT_FAULT)

/**
 * PCI driver list of supported FPGA devices.
 */
static const struct pci_device_id fpgaPciTbl[] =
{
    {
      .vendor    = PCI_VENDOR_ID_LSI_LOGIC,
      .device    = 0x7110,
      .subvendor = PCI_ANY_ID,
      .subdevice = PCI_ANY_ID,
      .class     = PCI_CLASS_SYSTEM_OTHER << 8,
    },
    {
      .vendor    = PCI_VENDOR_ID_NETAPP,
      .device    = 0x7110,
      .subvendor = PCI_ANY_ID,
      .subdevice = PCI_ANY_ID,
      .class     = PCI_CLASS_SYSTEM_OTHER << 8,
    },
    { 0, },
};

/**
 * The FPGA ISR.
 *
 * This ISR is executed on PCI interrupts from the FPGA device.
 * It will wake up any processes that were waiting on the interrupt
 * to occur.
 *
 */
static irqreturn_t
fpgaIsr(int irq, void *dev_id)
{
    unsigned long secsSinceLastWatchdog;
    struct fpgaDev* fdev;

    /* Increment interrupt counter */
    interruptCount++;

    EXT_DBG("interruptCount = %ld\n", interruptCount);

    /* The dev_id pointer passed to us is the fpgaDev for this device. */
    fdev = (struct fpgaDev*) dev_id;

    /* If we don't have any regs_base defined, exit. */
    if (!IS_PLATFORM_REGS_DEFINED(fdev))
        return IRQ_NONE;

    DBG("fpgaIsr fired, interruptCount = %ld\n", interruptCount);
    FPGA_WATCHDOG_CLR(fdev);

    if (interruptCount > 1)
    {
        secsSinceLastWatchdog = jiffies_to_msecs(jiffies - jiffiesAtLastWatchdog)/1000;
        if (secsSinceLastWatchdog >= RESET_WDOG_INT_COUNT_AFTER_X_SECS)
        {
            DBG("fpgaIsr hasn't fired for %ld seconds; resetting watchdog count\n",
                secsSinceLastWatchdog);
            interruptCount = 1;
        }
        else
        {
            DBG("fpgaIsr fired %ld seconds ago; NOT resetting watchdog count\n",
                secsSinceLastWatchdog);
        }
    }

    DBG("fpgaIsr logging memory usage\n");
    handle_sysrq('m');
    FPGA_WATCHDOG_CLR(fdev);

    DBG("fpgaIsr logging CPU backtrace\n");
    handle_sysrq('l');
    FPGA_WATCHDOG_CLR(fdev);

    DBG("fpgaIsr logging blocked tasks\n");
    handle_sysrq('w');
    FPGA_WATCHDOG_CLR(fdev);

    DBG("fpgaIsr logging all ftrace buffers\n");
    handle_sysrq('z');
    FPGA_WATCHDOG_CLR(fdev);

    jiffiesAtLastWatchdog = jiffies;

    if (interruptCount < PANIC_AFTER_X_WDOG_INTS)
    {
        DBG("fpgaIsr serviced watchdog\n");
        return IRQ_HANDLED;
    }

    panic("fpgaIsr triggering kernel panic due to watchdog interrupts");
    /* NOTREACHED */
    return IRQ_HANDLED;
}

/**
 * Function to pacify the watchdog for a certain period of time.
 *
 */
static int
pacifyWatchdog
    (
    void* dev_id
    )
{
    struct fpgaDev* fdev = (struct fpgaDev*) dev_id;

    if (fdev->watchdogTimeout == 0)
    {
        /*
         * This should never happen as the IOCTL should always set before
         * calling this function.
         */
        DBG("Pacify request with watchdog disabled. Returning...");
        return -1;
    }

    while (time_before(jiffies, fdev->watchdogTimeout))
    {
        FPGA_WATCHDOG_CLR(fdev);
        EXT_DBG("Pacifying watchdog. watchdogTimeout = %ld jiffies = %ld\n",
                fdev->watchdogTimeout, jiffies);

        msleep(WDT_INTERVAL_MSEC);
    }

    DBG("Watchdog pacify request done. watchdogTimeout = %ld jiffies = %ld\n",
            fdev->watchdogTimeout, jiffies);

    return 0;
}

/**
 * FPGA perform the protected write sequence for data
 *
 */
int
fpgaWritePwData
    (
    struct fpgaDev *dev,
    fpga_reg* regInfo
    )
{
    uint32_t i;
    int iteration;
    unsigned long regValue;
    unsigned long newValue;
    unsigned long flags;

    /* Try protected writes up to 3 times */
    for (iteration = 0; iteration < 3; iteration++)
    {
        /* Protect the protected write sequence by disabling interrupts */
        local_irq_save(flags);
        regValue = READ_REG(dev,regInfo->offset) & ~regInfo->mask;
        regValue |= (regInfo->data & regInfo->mask);
        EXT_DBG("Writing register offset %x, mask %x, Value %lx\n",
                 regInfo->offset, regInfo->mask, regValue);

        for ( i = 0; i < dev->numProtWrite; i++)
        {
            WRITE_REG(dev, ENCODED_WRITE, dev->protWriteSeq[i]);
        }

        WRITE_REG(dev, regInfo->offset, regValue);

        /* Read it back, restore interrupts, and stop if they match */
        newValue = READ_REG(dev, regInfo->offset);
        local_irq_restore(flags);

        if (newValue == regValue)
            return 1;
    }
    return 0;
}

/**
 * FPGA processor reset function
 *
 * This function is passed to the kernel to call during kernel
 * initiated processor resets so that it resets using the FPGA
 * instead of how the kernel wishes to reset it
 */
static void processorReset(void* arg)
{
    struct fpgaDev* fdev = (struct fpgaDev*)arg;
    char val;

    EXT_DBG( "resetting processor\n");
    val = READ_REG(fdev, PROTECTED_WRITE1);
    WRITE_REG(fdev, ENCODED_WRITE, BOARD_RESET_UNLOCK);
    WRITE_REG(fdev, PROTECTED_WRITE1, val | BOARD_RESET);
}


/**
 * FPGA ioctl function.
 *
 * This function operates on the following ioctl commands:
 *
 *     FPGA_IOCTL_START_WDT:      enables the watchdog timer
 *     FPGA_IOCTL_DISABLE_WDT:    disables the watchdog timer
 *     FPGA_IOCTL_PET_WDT:        pacifies the watchdog timer
 *     FPGA_IOCTL_READ_REG:       reads an FPGA register specified by
 *                                the offset
 *     FPGA_IOCTL_WRITE_REG:      writes an FPGA register specified
 *                                by the offset
 *     FPGA_IOCTL_WRITE_SHADOW:   writes to shadow/active registers
 *     FPGA_IOCTL_WRITE_PROT_REG: writes an FPGA protected
 *                                register specified by the offset
 *     FPGA_IOCTL_REG_PROT_WRITE: Registers the protected write
 *                                sequence.  Done during SO init.
 *     FPGA_IOCTL_READ_NVSRAM     reads the NVSRAM byte
 *     FPGA_IOCTL_WRITE_NVSRAM    writes the NVSRAM byte
 *     FPGA_IOCTL_REGISTER_ISR    registers interrupt service routine.
 *     FPGA_IOCTL_ACK_INT         enable irq line.
 *
 * Other commands will return error.
 *
 */
static long fpgaIoctl(struct file* filp, unsigned int command, unsigned long data)
{
    unsigned long regData;
    struct fpgaDev* fdev;
    fpga_reg* regP = 0;
    fpga_reg regInfo;
    fpga_prot_write pwData;
    unsigned long copyRet = 0;
    unsigned long newTimeout;
    int32_t __user *hwid_ptr;
    struct task_struct *wdTask;

    fdev = (struct fpgaDev*)filp->private_data;

    if (!IS_PLATFORM_REGS_DEFINED(fdev))
        return -1;

    EXT_DBG("Doing fpga ioctl %x on %lx:  fdev=%p\n", command, data, fdev);

    if ((command != FPGA_IOCTL_ACK_INT) &&
        (command != FPGA_IOCTL_DISABLE_WDT) &&
        (command != FPGA_IOCTL_GET_HW_ID) &&
        (command != FPGA_IOCTL_SET_HW_ID))
    {
        regP = ((fpga_reg*)((unsigned long *)data));
        if(regP == NULL)
        {
            DBG("No register info received\n");
            return -1;
        }

        /* The copy_from/to_user functions return the number of uncopied bytes,
         * so a return of zero is a successul statusfpgaMmap
         */
        if (command == FPGA_IOCTL_REG_PROT_WRITE)
            copyRet = copy_from_user(&pwData, regP, sizeof(pwData));
        else
            copyRet = copy_from_user(&regInfo, regP, sizeof(regInfo));

        if (copyRet)
            return -1;
    }

    switch (command)
    {
        case FPGA_IOCTL_START_WDT:
            //to be implemented
            //enable the watchdog timer
            DBG("Invalid request: FPGA_IOCTL_START_WDT\n");
            return -1;

        case FPGA_IOCTL_DISABLE_WDT:
            FPGA_WATCHDOG_DISABLE(fdev);
            fdev->watchdogTimeout = 0;
            DBG("watchdog timer disabled\n");
            return 0;

        case FPGA_IOCTL_PET_WDT:
            newTimeout = jiffies + msecs_to_jiffies(regInfo.data);
            if (time_before(fdev->watchdogTimeout, newTimeout))
                fdev->watchdogTimeout = newTimeout;
            FPGA_WATCHDOG_CLR(fdev);
            DBG("current jiffy = %ld watchdogTimeout is %ld\n", jiffies, fdev->watchdogTimeout);

            /* We spawn off a kthread that will pacify the watchdog hardware for
             * the requested amount of time.
             */
            wdTask = kthread_run( pacifyWatchdog, (void*) fdev, "kthread_wdt");
            if (!wdTask)
            {
                DBG("Failed to create pacifying thread!\n");
                return -1;
            }

            return 0;

        case FPGA_IOCTL_READ_REG:
            EXT_DBG("Doing fpga ioctl %x (READ_REG) on %lx\n", command, data);
            EXT_DBG("Reading register offset %x\n", regInfo.offset);
            regInfo.data = READ_REG(fdev,regInfo.offset) & 0xFF;
            copyRet = copy_to_user(regP, &regInfo, sizeof(regInfo));
            break;

        case FPGA_IOCTL_WRITE_REG:
            EXT_DBG("Doing fpga ioctl %x (WRITE_REG) on %lx\n", command, data);
            regData = READ_REG(fdev,regInfo.offset) & ~regInfo.mask;
            regData |= (regInfo.data & regInfo.mask);
            EXT_DBG("Writing register offset %x, mask %x, data %lx\n",
                     regInfo.offset, regInfo.mask, regData);
            WRITE_REG(fdev, regInfo.offset, regData);
            break;

        case FPGA_IOCTL_WRITE_SHADOW:
            DBG("Invalid request: FPGA_IOCTL_WRITE_SHADOW\n");
            return -1;

        case FPGA_IOCTL_WRITE_PROT_REG:
            EXT_DBG("Doing fpga pw ioctl %x (WRITE_REG) on %lx\n", command, data);
            if (!fpgaWritePwData(fdev, &regInfo))
            {
                DBG("Could not write out protected write data\n");
                return -1;
            }
            break;

        case FPGA_IOCTL_REG_PROT_WRITE:
            /* If it is already initialized, then return an error */
            if (fdev->protWriteSeq)
                return -1;
            fdev->numProtWrite = pwData.size;
            fdev->protWriteSeq =
                  (unsigned char*)kmalloc(sizeof(unsigned char) * fdev->numProtWrite, GFP_KERNEL);
            if (fdev->protWriteSeq == 0)
            {
                DBG("Could not allocate memory for protected write sequence\n");
                return -1;
            }
            copyRet = copy_from_user(fdev->protWriteSeq, (void*)pwData.addr, fdev->numProtWrite);
            break;

        case FPGA_IOCTL_READ_NVSRAM:
            {
                uint32_t offset = regInfo.offset;

                EXT_DBG("Doing fpga ioctl %x (READ_NVSRAM) on %lx\n", command, data);

                /* Check for special offset */
                if (offset == NVSRAM_DUAL_BIOS_OFFSET)
                {
                    offset = fdev->nvsram_size - 1;
                }
                else if (offset >= fdev->nvsram_size)
                {
                    EXT_DBG("Reading NVSRAM offset %x is outside the range of the NVSRAM size %lx\n",
                        offset, fdev->nvsram_size);
                    return -1;
                }
                EXT_DBG("Reading NVSRAM offset %x\n", offset);
                regInfo.data = READ_NVSRAM(fdev, offset) & 0xFF;
                copyRet = copy_to_user(regP, &regInfo, sizeof(regInfo));
            }
            break;

        case FPGA_IOCTL_WRITE_NVSRAM:
            {
                uint32_t offset = regInfo.offset;

                EXT_DBG("Doing fpga ioctl %x (WRITE_NVSRAM) on %lx\n", command, data);

                /* Check for special offset */
                if (offset == NVSRAM_DUAL_BIOS_OFFSET)
                {
                    offset = fdev->nvsram_size - 1;
                }
                else if (offset >= fdev->nvsram_size)
                {
                    EXT_DBG("Writing NVSRAM offset %x is outside the range of the NVSRAM size %lx\n",
                            offset, fdev->nvsram_size);
                    return -1;
                }
                regData = READ_NVSRAM(fdev, offset) & ~regInfo.mask;
                regData |= (regInfo.data & regInfo.mask);
                EXT_DBG("Writing NVSRAM offset %x, mask %x, data %lx\n", offset,
                    regInfo.mask, regData);
                WRITE_NVSRAM(fdev, offset, regData);
            }
            break;

        case FPGA_IOCTL_REGISTER_ISR_INT:
            if (fdev->irq_registered || fdev->trigger)
            {
                DBG("IRQ/EventFD already registered\n");
                return -1;
            }

            EXT_DBG("Doing fpga ioctl %x (REGISTER_ISR_INT) on %lx\n", command, data);

            /**
             * Register an ISR for this PCI interrupt.
             * The flag IRQF_SHARED is used to allow other devices to use this interrupt also.
             * A pointer to the fpgaDev is passed to the ISR so we can easily access the
             * fpga specific device data (such as the BARs) without having to follow
             * a bunch of pci_dev pointers.
             */
            if (request_irq(fdev->pdev->irq, fpgaIsr, IRQF_SHARED, FPGA_PCI_NAME, (void*)fdev))
            {
                DBG("IRQ registration failed\n");
                return -1;
            }
            DBG("Registered IRQ %x for fpgaIsr\n", fdev->pdev->irq);

            fdev->trigger = eventfd_ctx_fdget(regInfo.data);
            if (IS_ERR(fdev->trigger))
            {
                DBG("Error in fpga ioctl %x (REGISTER_ISR_INT) on %lx\n", command, data);
                free_irq(fdev->pdev->irq, fdev->pdev);
                fdev->trigger = 0;
                return -1;
            }

            EXT_DBG("Enabled INT trigger: %p\n", fdev->trigger);
            fdev->irq_registered = 1;
            break;

        case FPGA_IOCTL_REGISTER_ISR_MSI:
            if (fdev->irq_registered || fdev->trigger)
            {
                DBG("IRQ/EventFD already registered\n");
                return -1;
            }

            EXT_DBG("Doing fpga ioctl %x (REGISTER_ISR_MSI) on %lx\n", command, data);

            /**
             * Enable MSI.
             */
            if (!fdev->msi_enabled)
            {
                if (pci_enable_msi(fdev->pdev))
                {
                    DBG("Enable MSI failed\n");
                    return -1;
                }
                else
                {
                    fdev->msi_enabled = 1;
                }
            }

            /**
             * Register an ISR for this PCI interrupt.
             * MSI should not be shared with other devices.
             * A pointer to the fpgaDev is passed to the ISR so we can easily access the
             * fpga specific device data (such as the BARs) without having to follow
             * a bunch of pci_dev pointers.
             */
            if (request_irq(fdev->pdev->irq, fpgaIsr, 0, FPGA_PCI_NAME, (void*)fdev))
            {
                DBG("IRQ registration failed\n");
                pci_disable_msi(fdev->pdev);
                fdev->msi_enabled = 0;
                return -1;
            }
            DBG("Registered IRQ %x for fpgaIsr\n", fdev->pdev->irq);

            fdev->trigger = eventfd_ctx_fdget(regInfo.data);
            if (IS_ERR(fdev->trigger))
            {
                DBG("Error in fpga ioctl %x (REGISTER_ISR_MSI) on %lx\n", command, data);
                free_irq(fdev->pdev->irq, fdev->pdev);
                pci_disable_msi(fdev->pdev);
                fdev->msi_enabled = 0;
                fdev->trigger = 0;
                return -1;
            }

            EXT_DBG("Enabled MSI trigger: %p\n", fdev->trigger);
            fdev->irq_registered = 1;
            break;

        case FPGA_IOCTL_ACK_INT:
            EXT_DBG("Doing fpga ioctl %x (ACK_INT) on %lx\n", command, data);
            if (!fdev->irq_registered)
            {
                DBG("IRQ not registered yet.\n");
                return -1;
            }

            /* Unbalanced enable for IRQ is not allowed.
             * Enable the IRQ only if was disabled earlier.
             */
            if (atomic_dec_and_test(&fdev->irq_disabled))
                enable_irq(fdev->pdev->irq);
            break;

        case FPGA_IOCTL_SET_HW_ID:
            hardwareId = (int32_t) data;
            copyRet = 0;
            break;

        case FPGA_IOCTL_GET_HW_ID:
            hwid_ptr = (int32_t *)data;
            if (put_user(hardwareId, hwid_ptr))
                return -EFAULT;

            copyRet = 0;
            break;

        default:
            return -ENOTTY;
    }

    return copyRet? -1: 0;
}

/**
 * FPGA open routine.
 *
 * This function figures out the instances of this driver and stores
 * the private data structures into the file pointer for this device
 * to be used by the other fileOps.
 *
 */
static int fpgaOpen(struct inode* inode, struct file* filp)
{

    EXT_DBG("Opening fpga node\n");

    filp->private_data = fpgaPciList[iminor(inode)];

    return 0;
}

/**
 * FPGA close routine.
 *
 * No special operations are required for this close routine.
 *
 */
static int fpgaRelease(struct inode* inode, struct file* filp)
{
    EXT_DBG("Closing fpga node\n");

    /*
     * Nothing was allocated or mapped during open, so there's really nothing
     * to do here. No resources need to be freed or anything like that.
     */
    return 0;
}

/**
 * FPGA mmap routine.
 *
 * Maps the FPGA region to user space
 *
 */
static int fpgaMmap(struct file* filp, struct vm_area_struct* vma)
{
    struct fpgaDev* fdev = (struct fpgaDev*)filp->private_data;
    unsigned long size = vma->vm_end - vma->vm_start;
    unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
    unsigned long long physaddr;

    EXT_DBG("Attempting to map FPGA memory, size 0x%lx\n", size);

    /* Make sure starting point is at beginning of FPGA */
    if (offset != 0)
    {
        DBG("Offset must be zero (%ld)\n", offset);
        return -EINVAL;
    }

    /* Make sure user specified shared page */
    if (!(vma->vm_flags & VM_SHARED))
    {
        DBG("Invalid vm_flags (%lx)\n", vma->vm_flags);
        return -EINVAL;
    }

    /* Prevent memory from being swapped out */
    vma->vm_flags |= VM_LOCKED;

    /* Size <= 0x1000 maps registers; larger size maps nvsram */
    if (size <= fdev->regs_size)
        physaddr = fdev->phys_regs_base;
    else if (size <= fdev->nvsram_size)
        physaddr = fdev->phys_nvsram_base;
    else {
        DBG("Invalid size (0x%lx)\n", size);
        return -EINVAL;
    }

    /* Now map the region */
    if (remap_pfn_range(vma, vma->vm_start,
                        PFN_DOWN(physaddr),
                        size, PAGE_SHARED))
    {
        DBG("Unable to map region\n");
        return -EAGAIN;
    }

    return 0;
}

/**
 * FPGA cleanup routine.
 *
 * This routine cleans up all of the resources if an error happened during
 * initialization or a device is removed.
 *
 *  Input parameters:
 *     pdev - pci dev structure pointer
 *     disablePci - flag when non zero indicating to disable the PCI device
 *     releasePciRegion - Remove the PCI regions
 *     destroyDevice - destroy the device
 *
 */
static void
fpgaRemoveDev
    (
    struct pci_dev *pdev,
    int disablePci,
    int releasePciRegion,
    int destroyDevice
    )
{
    struct fpgaDev *fdev = (struct fpgaDev*) pci_get_drvdata(pdev);

    DBG("Removing PCI driver for function %d\n", PCI_FUNC(pdev->devfn));

    /* Deregister function - it can be ran for all functions */
    /* since the kernel handles multiple calls to this deregister function */
    /* eseries_deregister_reboot_function(processorReset); */

    if (fdev)
    {
        /* Mask all interrupt sources */
        if (fdev->virt_regs_base)
        {
            WRITE_REG(fdev, INTERRUPT_MASK_INT_B, 0xff);
        }

        if (destroyDevice)
        {
            device_destroy(fpgaPciClass, fdev->device_number);
        }

        /* Unmap the IO resources if mapped currently */
        if (fdev->virt_regs_base)
        {
            DBG("Unmapping IO resources\n");
            iounmap((void*)(fdev->virt_regs_base));
        }

        /* Unmap the NVSRAM resources if mapped currently */
        if (fdev->virt_nvsram_base)
        {
            DBG("Unmapping NVSRAM resources\n");
            iounmap((void*)(fdev->virt_nvsram_base));
        }

        /*
         * Free the PCI interrupt, should be done before the fpgaDev is free'd
         * and before the PCI IO resources are unmapped (in case an interrupt is
         * in progress).
         */
        if (fdev->irq_registered)
        {
            free_irq(pdev->irq, fdev);
        }

        if (fdev->msi_enabled)
        {
            pci_disable_msi(pdev);
            fdev->msi_enabled = 0;
        }

        pci_set_drvdata(pdev, NULL);

        DBG("Freeing fpgaDev\n");
        kfree(fdev);
    }

    /* Release the PCI regions if flag set */
    if (releasePciRegion)
    {
        pci_release_regions(pdev);
    }

    /* Disable the PCI device if flag is set */
    if (disablePci)
    {
        DBG("Disabling PCI device\n");
        pci_disable_device(pdev);
    }
}


/**
 * This function initializes an FPGA PCI device.
 *
 * The function is called on each PCI device in the system that matches the vendor ID,
 * device ID, and class code listed in fpgaPciTbl. However, only FPGA_MAX_DEVS
 * fpgaDev devices are supported by this driver. Additionally, multiple functions
 * of the FPGA PCI device might match the criteria, but may need to be ignored by this
 * routine (since those devices may not be intended for this driver). For the cases
 * where we've already initialized FPGA_MAX_DEVS devices or found a PCI function
 * that is not the one we're looking for, the routine will return without initializing
 * a driver for the device.
 *
 * This function is called in response to the modprobe command
 * to load a kernel module or during initialization.
 *
 * The general steps performed by this initialization are as follows:
 * 1) Enable the PCI device
 * 2) Retrieve the BARs and store them in the FPGA driver's private data structure
 * 3) Connect an ISR to the device
 * 4) Initialize the character driver
 * 5) Create the /dev/fpga0 device
 * 6) Unmask the FPGA's subsystem interrupt
 *
 */
static int fpgaPciInitOne(struct pci_dev *pdev, const struct pci_device_id *ent)
{
    struct fpgaDev *fdev;
    int rc = 0;

    /* Map all fpga functions which class is 0x88000.  In vEOS, the scripts will
     * cause function 0 to deregister so that EOS can claim it.
     * Linux EOS will use function 0 via this driver for all interrupt handling which
     * will include the watchdog timer.
     */
    if ((pdev->class & 0xffff0) != 0x88000)
    {
        DBG("Skipping unhandled function %x\n", PCI_FUNC(pdev->devfn));
        return -ENODEV;
    }

    /* Make sure function number is within allowable range */
    if (PCI_FUNC(pdev->devfn) >= FPGA_MAX_DEVS)
    {
        DBG("FPGA function exceeds maximum allowable limit %x\n", PCI_FUNC(pdev->devfn));
        return -ENODEV;
    }

    /* Preset to 0 "just in case" */
    fpgaPciList[PCI_FUNC(pdev->devfn)] = 0;

    /* Create a new fpgaDev, fail if we can't allocate the memory. */
    fdev = kmalloc(sizeof(struct fpgaDev), GFP_KERNEL);
    if (!fdev)
    {
        DBG("Could not allocate memory for the fpgaDev\n");
        return -ENODEV;
    }
    memset(fdev, 0, sizeof(struct fpgaDev));

    /*
     * Name will be fpga_pciX (where X is PCI B:D:F function number).
     * Some use cases require a specific function, for those cases,
     * ensure that the appropriate /dev/fpgaY points to the correct
     * /dev/fpga_pciX.
     */
    snprintf(fdev->device_name, sizeof(fdev->device_name), "%s%d", FPGA_PCI_NAME, PCI_FUNC(pdev->devfn));

    /* Called to enable the device, which wakes up the device and prepares it for use */
    rc = pci_enable_device (pdev);
    if (rc)
    {
        DBG("Failed to enable pci device (%d)\n", rc);
        fpgaRemoveDev(pdev, 0, 0, 0);
        return rc;
    }

    /* Check to see if this PCI device has BAR0 mapped space */
    if (!(pci_resource_flags(pdev, 0) & IORESOURCE_MEM))
    {
        DBG("Cannot find PCI device base address 0, exiting\n");
        fpgaRemoveDev(pdev, 1, 0, 0);
        return -ENODEV;
    }

    /* Check to see if this PCI device has BAR1 mapped space */
    if (!(pci_resource_flags(pdev, 1) & IORESOURCE_MEM))
    {
        DBG("Cannot find PCI device base address 1, exiting\n");
        fpgaRemoveDev(pdev, 1, 0, 0);
        return -ENODEV;
    }

    /* Reserve the memory resources of this PCI devices for the driver FPGA_PCI_NAME */
    rc = pci_request_regions(pdev, FPGA_PCI_NAME);
    if (rc)
    {
        DBG("Unable to map BARs \n");
        fpgaRemoveDev(pdev, 1, 0, 0);
        return -ENODEV;
    }

    /* Get the platform register base (BAR0) in both physical and virtual address form. */
    fdev->phys_regs_base = pci_resource_start(pdev, 0);
    fdev->regs_size = pci_resource_len(pdev, 0);
    fdev->virt_regs_base = (unsigned long)ioremap_nocache(fdev->phys_regs_base, fdev->regs_size);
    if ( (fdev->virt_regs_base == 0) || (fdev->regs_size < FPGA_REGS_SIZE) )
    {
        DBG("Failed to map virt_regs_base\n");
        fpgaRemoveDev(pdev, 1, 1, 0);
        return -ENODEV;
    }

    DBG("virt_regs_base=%lx for BAR0(0x%llx) size (0x%lx)\n",
            fdev->virt_regs_base, fdev->phys_regs_base, fdev->regs_size);

    /* Get the platform register base (BAR1) in both physical and virtual address form. */
    fdev->phys_nvsram_base = pci_resource_start(pdev, 1);
    fdev->nvsram_size = pci_resource_len(pdev, 1);
    fdev->virt_nvsram_base = (unsigned long)ioremap_nocache(fdev->phys_nvsram_base,
                                                            fdev->nvsram_size);
    if (fdev->virt_nvsram_base == 0)
    {
        DBG("Failed to map virt_nvsram_base\n");
        fpgaRemoveDev(pdev, 1, 1, 0);
        return -ENODEV;
    }

    DBG("virt_nvsram_base=%lx for BAR1(0x%llx) size (0x%lx)\n",
            fdev->virt_nvsram_base, fdev->phys_nvsram_base, fdev->nvsram_size);

    /*
     * PCI driver init is basically complete now, but we must store a pointer to our fpgaDev
     * somewhere so that it can be accessed through the pci_dev structure. This data typically
     * goes into the driver_data pointer of the dev field in the pci_dev.
     */
    pci_set_drvdata(pdev, fdev);

    /* Save to pdev pointer for later access */
    fdev->pdev = pdev;

    /* Create the device */
    fdev->device_number = MKDEV(fpgaPciMajor, PCI_FUNC(pdev->devfn));
    if (IS_ERR(device_create(fpgaPciClass, NULL, fdev->device_number, NULL, fdev->device_name)))
    {
        DBG(FPGA_PCI_NAME " class device creation failed\n");
        fpgaRemoveDev(pdev, 1, 1, 0);
        return -ENODEV;
    }
    fpgaPciList[PCI_FUNC(pdev->devfn)] = fdev;

    /* Only register for the first function */
    /* if (PCI_FUNC(pdev->devfn) == 0)
     *     eseries_register_reboot_function(processorReset, fdev);
     */

    /**
     * Enable MSI.
     */
    if (pci_enable_msi(fdev->pdev))
    {
        DBG("Enable MSI failed\n");
    }
    else
    {
        fdev->msi_enabled = 1;
    }

    /**
     * Register an ISR for this PCI interrupt.
     * MSI should not be shared with other devices.
     * A pointer to the fpgaDev is passed to the ISR so we can easily access the
     * fpga specific device data (such as the BARs) without having to follow
     * a bunch of pci_dev pointers.
     */
    if (request_irq(fdev->pdev->irq, fpgaIsr, 0, FPGA_PCI_NAME, (void*)fdev))
    {
        DBG("IRQ registration failed\n");
        pci_disable_msi(fdev->pdev);
        fdev->msi_enabled = 0;
    }
    DBG("Registered IRQ %x for fpgaIsr\n", fdev->pdev->irq);

    fdev->irq_registered = 1;

    /* Enable bus mastering (gets disabled by pci_disable_device if we
     * are unloaded; does not get enabled by pci_enable_device)
     */
    pci_set_master(pdev);

    /* 3.2.56 Interrupt Mask B (0x91)  R/W
     * A write to this address will set an interrupt mask for the interrupt
     * sources listed. A read from this address will return the current state of
     * the interrupt mask. Writing a ‘1’ to a bit will prevent an external
     * interrupt from being passed on to the processor. Writing a ‘0’ will allow
     * the interrupt to be passed on to the Processor. Subsystem Mask bit masks
     * any interrupt pending indicated by bit 0 of Interrupt Source INT_B Register
     * (0x71) RO.
     * 
     * Write to Function 0 BAR1 + 0000 0091h will mask Subsystem status lines
     * contributing to interrupt generated by Function 0.
     *
     * Write to Function 1 BAR1 + 0000 0091h will mask Subsystem status lines
     * contributing to interrupt generated by Function 1.
     *
     * Reset State: 0xFF
     * This register resets to default value with Power-on Reset and Board Reset.
     *
     * Table 59: Interrupt Mask B Register (0x91)
     *
     * BIT R/W Name Description
     * 0 R/W SUBSYS_MASK A '1' will mask this bit.
     * 1 R/W I2C_MASK A '1' will mask this bit.
     * 2 R/W WDT_MASK A '1' will mask this bit.
     * 3 R/W ALT_MSG_INT_MASK A '1' will mask this
     * bit.
     * 4 R/W COB_UART_INT_MASK A '1' will mask this bit.
     * 5 R -- UNUSED, always returns 0
     * 6 R -- UNUSED, always returns 0
     * 7 R -- UNUSED, always returns 0
     */

    /* If this is function 0, unmask the watchdog interrupt */
    if (PCI_FUNC(pdev->devfn) == 0)
    {
        WRITE_REG(fdev, INTERRUPT_MASK_INT_B, 0xfb);
    }

    /* Now the driver is initialized and registered and ready for action. */
    return 0;
}

/**
 * This function removes an FPGA PCI device from the system.
 *
 * The function is called on each driver that was previous successfully loaded.
 * It is called in response to the "modprobe -r <module_name>"
 * command.
 *
 * This function frees up any resources the driver has been consuming.
 * It also disables the PCI devices when it is finished.
 *
 * This function is called for each fpga device that was successfully loaded
 * and unloads each instance individually.
 *
 */
static void fpgaPciRemoveOne(struct pci_dev *pdev)
{
    fpgaRemoveDev(pdev, 1, 1, 1);
}

static int fpgaPciClassCreate(void)
{
    fpgaPciMajor = register_chrdev(0, FPGA_PCI_NAME, &fpgaFileOps);
    if (fpgaPciMajor < 0)
    {
        DBG("Failed to allocate major device numbers\n");
        return fpgaPciMajor;
    }
    DBG("Character device number assigned to major(%d)\n", fpgaPciMajor);

    /* Only create the class if not already created or if last attempt failed */
    fpgaPciClass = class_create(THIS_MODULE, FPGA_PCI_NAME);
    if(IS_ERR(fpgaPciClass))
    {
        unregister_chrdev(fpgaPciMajor, FPGA_PCI_NAME);
        DBG(FPGA_PCI_NAME " class registration failed\n");
        return PTR_ERR(fpgaPciClass);
    }
    DBG(FPGA_PCI_NAME " class registration completed\n");

    return 0;
}

static void fpgaPciClassDestroy(void)
{
    class_destroy(fpgaPciClass);
    unregister_chrdev(fpgaPciMajor, FPGA_PCI_NAME);
}

#if 0
/* This might be useful if we want to probe for device */
static int fpgaPciPresent(void)
{
    return pci_dev_present(fpgaPciTbl));
}
#endif

/**
 * Standard PCI driver functions
 */
static struct pci_driver fpgaPciDriver =
{
    .name       = FPGA_PCI_NAME,
    .probe      = fpgaPciInitOne,
    .remove     = fpgaPciRemoveOne,
    .id_table   = fpgaPciTbl,
};

/**
 * Standard Linux module registration
 */

static __init int fpgaPciInit(void)
{
    int rc = 0;
    DBG("Initializing FPGA driver\n");
    rc = fpgaPciClassCreate();
    if (rc)
    {
        DBG("Could not perform fpga class create\n");
        return rc;
    }
    return pci_register_driver(&fpgaPciDriver);
}

static __exit void fpgaPciExit(void)
{
    pci_unregister_driver(&fpgaPciDriver);
    fpgaPciClassDestroy();
}

module_init(fpgaPciInit);
module_exit(fpgaPciExit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("NetApp FPGA Driver " FPGA_VERSION);
MODULE_AUTHOR("NetApp, Inc");

MODULE_DEVICE_TABLE(pci, fpgaPciTbl);

