/** 
  * TableDnD plug-in for JQuery, allows you to drag and drop table rows 
  * You can set up various options to control how the system will work 
  * Copyright <c> Denis Howlett <denish@isocra.com> 
  * Licensed like jQuery, see http://docs.jquery.com/License. 
  * 
  * Configuration options: 
  *  
  * onDragStyle 
  *     This is the style that is assigned to the row during drag. There are 
  *     limitations to the styles that can be associated with a row (such as 
  *     you can't assign a border as well you can, but it won't be displayed).
  *     (So instead consider using onDragClass.) The CSS style to apply is 
  *     specified as  a map (as used in the jQuery css(...) function). 
  * onDropStyle 
  *     This is the style that is assigned to the row when it is dropped. As 
  *     for onDragStyle, there are limitations to what you can do. Also this 
  *     replaces the original style, so again consider using onDragClass which 
  *     is simply added and then removed on drop. 
  * onDragClass 
  *     This class is added for the duration of the drag and then removed when 
  *     the row is dropped. It is more flexible than using onDragStyle since 
  *     it can be inherited by the row cells and other content. The default 
  *     is class is tDnD_whileDrag. So to use the default, simply customise 
  *     this CSS class in your stylesheet. 
  * onDrop 
  *     Pass a function that will be called when the row is dropped. The 
  *     function takes 2 parameters: the table and the row that was dropped. 
  *     You can work out the new order of the rows by using table.rows. 
  * onDragStart 
  *     Pass a function that will be called when the user starts dragging. The 
  *     function takes 2 parameters: the table and the row which the user has 
  *     started to drag. 
  * onAllowDrop 
  *     Pass a function that will be called as a row is over another row. If 
  *     the function returns true, allow dropping on that row, otherwise not. 
  *     The function takes 2 parameters: the dragged row and the row under the 
  *     cursor. It returns a boolean: true allows the drop, false doesn't allow it. 
  * scrollAmount 
  *     This is the number of pixels to scroll if the user moves the mouse cursor
  *     to the top or bottom of the window. The page should automatically scroll 
  *     up or down as appropriate (tested in IE6, IE7, Safari, FF2, FF3 beta 
  * dragHandle 
  *     This is the name of a class that you assign to one or more cells in each 
  *     row that is draggable. If you specify this class, then you are responsible
  *     for setting cursor: move in the CSS and only these cells will have the 
  *     drag behaviour. If you do not specify a dragHandle, then you get the old 
  *     behaviour where the whole row is draggable. 
  *
  *
  * This class is customized to satisfy our specifc requirements for the 
  * drap-n-drop feature in the ILM Editor.
  *  
 **/

var tableDnDObjects = new Hashtable();

function setupTableMove(groups)
{
    for ( var i = 0 ; i < groups.size(); i++)
    {
        var groupID = groups.getGroupIDByIndex(i);
        var groupType = groups.getGroupTypeByIndex(i);
        var groupAlignment = groups.getGroupAlignmentByIndex(i);

        if ( groupType == "BDLE" && groupAlignment != "vertical")
        {
            var table = document.getElementById(groupID);
            var tableDnD = new TableDnD();
            tableDnD.init(table);
            tableDnDObjects.put(groupID, tableDnD);
        }
    }

    // set up timeout for CFMN page
    if ( document.onmousemove == null || document.onmousemove == "")
    {
        document.onmousemove = function(ev){
            top.startTimer();
        }
    }
}


/** Keep hold of the current table being dragged */
var currenttable = null;

/** Capture the onmousemove so that we can see if a row from the current
 *  table if any is being dragged.
 * @param ev the event (for Firefox and Safari, otherwise we use window.event for IE)
 */
document.onmousemove = function(ev){
    // this for timing out feature, put this here due to this page also have onmousemove event
    top.startTimer();

    if (currenttable && currenttable.dragObject) {
        //alert("dragObject:" + currenttable.dragObject.id);
        ev   = ev || window.event;
        var mousePos = currenttable.mouseCoords(ev);
        var y = mousePos.y - currenttable.mouseOffset.y;
        if (y != currenttable.oldY) {
            // work out if we're going up or down...
            var movingDown = y > currenttable.oldY;
            // update the old value
            currenttable.oldY = y;
            // update the style to show we're dragging
            // currenttable.dragObject.style.backgroundColor = "#eee";
            // If we're over a row then move the dragged row to there so that the user sees the
            // effect dynamically
            var currentRow = currenttable.findDropTargetRow(y);
            if (currentRow  && currentRow.rowIndex != 0) {
                if (movingDown && currenttable.dragObject != currentRow) {
                    currenttable.dragObject.parentNode.insertBefore(currenttable.dragObject, currentRow.nextSibling);
                } else if (! movingDown && currenttable.dragObject != currentRow) {
                    currenttable.dragObject.parentNode.insertBefore(currenttable.dragObject, currentRow);
                }
            }
        }

        return false;
    }
}


// Similarly for the mouseup
document.onmouseup   = function(ev){
    if (currenttable && currenttable.dragObject) {
        var droppedRow = currenttable.dragObject;
        // If we have a dragObject, then we need to release it,
        // The row will already have been moved to the right place so we just reset stuff
        //droppedRow.style.backgroundColor = 'transparent';
        // And then call the onDrop method in case anyone wants to do any post processing
        currenttable.onDrop(currenttable.table, droppedRow, currenttable.initialRowIndex);
        currenttable.initialRowIndex = null;
        currenttable.dragObject   = null;
        currenttable = null; // let go of the table too
    }
}


/** get the source element from an event in a way that works for IE and Firefox and Safari
 * @param evt the source event for Firefox (but not IE--IE uses window.event) */
function getEventSource(evt) {
    if (window.event) {
        evt = window.event; // For IE
        return evt.srcElement;
    } else {
        return evt.target; // For Firefox
    }
}

/**
 * Encapsulate table Drag and Drop in a class. We'll have this as a Singleton
 * so we don't get scoping problems.
 */
function TableDnD() {
    /** Keep hold of the current drag object if any */
    this.dragObject = null;
    /** The current mouse offset */
    this.mouseOffset = null;
    /** The current table */
    this.table = null;
    /** Remember the old value of Y so that we don't do too much processing */
    this.oldY = 0;

    /** variable keep tracks the initial row position */
    this.initialRowIndex = null;

    /** Initialize the drag and drop by capturing mouse move events */
    this.init = function(table) {
        if (table != null)
        {
            this.table = table;
            var rows = table.tBodies[0].rows; //getElementsByTagName("tr")
            for (var i=1; i<rows.length; i++) {
                // added to ignore rows that I've added the NoDnD attribute to (Category and Header rows)
                var nodrag = rows[i].getAttribute("NoDrag")
                if (nodrag == null || nodrag == "undefined") { //There is no NoDnD attribute on rows I want to drag
                    this.makeDraggable(rows[i]);
                }
            }
        }
    }

    /** This function is called when you drop a row, so redefine it in your code
        to do whatever you want, for example use Ajax to update the server */
    this.onDrop = function(table, droppedRow, initialRowIndex)
    {
        // compute the row changes and update the configChangesList
        var MOVE_TYPE =     1;
        var rowIdSplit = droppedRow.id.split("_");
        var tableId = rowIdSplit[0];

        var pagingInfo = pagingDataList.get(tableId);
        var numberOfRowsPerPage = pagingInfo[0];
        var startRecordNum = pagingInfo[1];
        var pageIndex = Math.ceil(startRecordNum / numberOfRowsPerPage);

        var initialIndex = parseInt(rowIdSplit[1]);
        if ( isNaN(initialIndex) )
        {
            initialIndex = -1;
        }
        var finalIndex = (pageIndex-1) * numberOfRowsPerPage + droppedRow.rowIndex;
        // this variable is used to keep track if the user has made any movement
        var rowMovement = droppedRow.rowIndex - initialRowIndex;

        //alert ( "initial row index: " + initialIndex + " final index:" + finalIndex + " row changes: " + rowMovement);
        if (rowMovement != 0)
        {
            SetConfigurationChanged();
            configChangesList.addChangedRow(droppedRow.id, MOVE_TYPE, initialIndex, finalIndex, null);
            //alert("initialIndex: " + initialIndex + " finalIndex: " + finalIndex);
            showSubmitButton(configChangesList);
        }

    }


	/** Get the position of an element by going up the DOM tree and adding up all the offsets */
    this.getPosition = function(e){
        var left = 0;
        var top  = 0;
        /** Safari fix */
        if (e.offsetHeight == 0) {
            /** Safari 2 doesn't correctly grab the offsetTop of a table row
                this is detailed here:
                http://jacob.peargrove.com/blog/2006/technical/table-row-offsettop-bug-in-safari/
                the solution is likewise noted there, grab the offset of a table cell in the row - the firstChild.
                note that firefox will return a text node as a first child, so designing a more thorough
                solution may need to take that into account, for now this seems to work in firefox, safari, ie */
            e = e.firstChild; // a table cell
        }

        while (e.offsetParent){
            left += e.offsetLeft;
            top  += e.offsetTop;
            e     = e.offsetParent;
        }

        left += e.offsetLeft;
        top  += e.offsetTop;

        return {x:left, y:top};
    }

	/** Get the mouse coordinates from the event (allowing for browser differences) */
    this.mouseCoords = function(ev){
        if(ev.pageX || ev.pageY){
            return {x:ev.pageX, y:ev.pageY};
        }
        return {
            x:ev.clientX + document.body.scrollLeft - document.body.clientLeft,
            y:ev.clientY + document.body.scrollTop  - document.body.clientTop
        };
    }

	/** Given a target element and a mouse event, get the mouse offset from that element.
		To do this we need the element's position and the mouse position */
    this.getMouseOffset = function(target, ev){
        ev = ev || window.event;

        var docPos    = this.getPosition(target);
        var mousePos  = this.mouseCoords(ev);
        return {x:mousePos.x - docPos.x, y:mousePos.y - docPos.y};
    }

    /** Take an item and add an onmousedown method so that we can make it draggable */
    this.makeDraggable = function(item){
        if(!item) return;
        var self = this; // Keep the context of the TableDnd inside the function

        var moveButtonId = item.id + "_MoveRow";
        var moveButton = document.getElementById(moveButtonId);
        if(moveButton != null)
        {
            moveButton.onmousedown = function(ev){
                //alert("HERE __ MOUSE DOWN");
                // Need to check to see if we are an input or not, if we are an input, then
                // return true to allow normal processing
                //alert(" move button id: " + moveButton.id);
                var target = getEventSource(ev);
                if (target.tagName == 'INPUT' || target.tagName == 'SELECT') return true;
                currenttable = self;
                //self.dragObject  = this;
                self.dragObject  = item;
                self.initialRowIndex = item.rowIndex;
                //alert("dragObject:" + self.dragObject.id + " self.initialRowIndex:" + self.initialRowIndex );
                self.mouseOffset = self.getMouseOffset(this, ev);
                return false;
            }
        }
        //item.style.cursor = "move";
    }

    /** We're only worried about the y position really, because we can only move rows up and down */
    this.findDropTargetRow = function(y) {
        var rows = this.table.tBodies[0].rows;
        for (var i=0; i<rows.length; i++) {
            var row = rows[i];
            // added to ignore rows that I've added the NoDnD attribute to (Header rows)
            var nodrop = row.getAttribute("NoDrop");
            if (nodrop == null || nodrop == "undefined") {  //There is no NoDnD attribute on rows I want to drag
                var rowY    = this.getPosition(row).y;
                var rowHeight = parseInt(row.offsetHeight)/2;
                if (row.offsetHeight == 0) {
                    rowY = this.getPosition(row.firstChild).y;
                    rowHeight = parseInt(row.firstChild.offsetHeight)/2;
                }
                // Because we always have to insert before, we need to offset the height a bit
                if ((y > rowY - rowHeight) && (y < (rowY + rowHeight))) {
                        // that's the row we're over
                    return row;
                }
            }
        }
        return null;
    }
}
