/*******************************************************************************

    fpga_sds.c
    Copyright 2018-2020 NetApp, Inc. All rights reserved.

    Software-defined pseudo-FPGA loadable kernel driver module.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

*******************************************************************************/

#include <linux/module.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/uaccess.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/io.h>
#include <linux/sched.h>
#include <linux/eventfd.h>
#include <linux/jiffies.h>

#include "fpga.h"

#define FPGA_VERSION   "2.0"

#define FPGA_SDS_NAME   FPGA_NAME "_sds"

/**
 * Use DBG for standard debug message queue messages (always on).
 * Use EXT_DBG for extended debug messages that are turned on by compile time switch.
 * EXTENDED_DEBUG turns on EXT_DBG messages.
 */
#define DBG(msg, args...) printk(FPGA_SDS_NAME ": %s: " msg , __FUNCTION__, ## args)

#undef EXTENDED_DEBUG
#ifdef EXTENDED_DEBUG
    #define EXT_DBG(msg, args...) printk(FPGA_SDS_NAME ": %s: " msg , __FUNCTION__, ## args)
#else
    #define EXT_DBG(msg, args...)
#endif


/**
 * The FPGA device driver data structure
 */
struct fpgaDev
{
    unsigned long      virt_regs_base;    /* Virtual kernel address of platform "registers" */
    unsigned long      regs_size;         /* Size of FPGA "register" space */
    dev_t              device_number;     /* The character device MAJOR/MINOR device number */
    char               device_name[30];   /* Device name */
    unsigned long      watchdogTimeout;   /* Watchdog time out value in jiffies */
};

/* Pointer to class structure */
static  struct class*      fpgaMmClass = 0;
/* cdev major number */
static  int                fpgaMmMajor = 0;

/* Variable to hold hardware ID as external code may need it. */
static int32_t          hardwareId = 0;

/* Struct for our FPGA device */
static struct fpgaDev fpgaMmDev;

/**
 * Character driver supported file operations:
 *     ioctl
 *     open
 *     close/release
 */
static long fpgaMmIoctl(struct file*, unsigned int, unsigned long);
static int  fpgaMmOpen(struct inode*, struct file*);
static int  fpgaMmRelease(struct inode*, struct file*);
static int  fpgaMmap(struct file*, struct vm_area_struct*);

static struct file_operations fpgaFileOps =
{
    .owner            = THIS_MODULE,
    .unlocked_ioctl   = fpgaMmIoctl,
    .open             = fpgaMmOpen,
    .release          = fpgaMmRelease,
    .mmap             = fpgaMmap,
};

/**
 * Macros to read/write platform registers.
 */
#define IS_PLATFORM_REGS_DEFINED()       (fpgaMmDev.virt_regs_base != 0)
#define WRITE_REG(regoff, value)    (*((char*)(fpgaMmDev.virt_regs_base + regoff)) = (value & 0xFF))
#define READ_REG(regoff)            (*((char*)(fpgaMmDev.virt_regs_base + regoff)))

/**
 * MM FPGA ioctl function.
 *
 * This function operates on the following ioctl commands:
 *
 *     FPGA_IOCTL_START_WDT:        not implemented (-ENOTTY)
 *     FPGA_IOCTL_DISABLE_WDT:      not implemented (-ENOTTY)
 *     FPGA_IOCTL_PET_WDT:          not implemented (-ENOTTY)
 *     FPGA_IOCTL_READ_REG:         reads FPGA offset (0)
 *     FPGA_IOCTL_WRITE_REG:        writes FPGA offset (0)
 *     FPGA_IOCTL_WRITE_SHADOW:     not implemented (-ENOTTY)
 *     FPGA_IOCTL_WRITE_PROT_REG:   writes FPGA offset (0)
 *     FPGA_IOCTL_REG_PROT_WRITE:   not implemented (0)
 *     FPGA_IOCTL_READ_NVSRAM       Returns a 0 to the user (0)
 *     FPGA_IOCTL_WRITE_NVSRAM      Writes a byte to thebitbucket (0)
 *     FPGA_IOCTL_REGISTER_ISR_INT  not implemented (-1)
 *     FPGA_IOCTL_REGISTER_ISR_MSI  not implemented (-1)
 *     FPGA_IOCTL_ACK_INT           not implemented (-1)
 *
 * Other commands will return error (-1).
 *
 */
static long fpgaMmIoctl(struct file* filp, unsigned int command, unsigned long data)
{
    unsigned long regData;
    fpga_reg* regP = 0;
    fpga_reg regInfo;
    unsigned long copyRet = 0;
    int32_t __user *hwid_ptr;

    /* Not used */
    (void)filp;

    EXT_DBG("Doing ioctl %x on %lx:  fdev=%p\n", command, data, &fpgaMmDev);

    if ((command != FPGA_IOCTL_ACK_INT) &&
        (command != FPGA_IOCTL_DISABLE_WDT) &&
        (command != FPGA_IOCTL_GET_HW_ID) &&
        (command != FPGA_IOCTL_SET_HW_ID))
    {
        regP = ((fpga_reg*)((unsigned long *)data));
        if(regP == NULL)
        {
            DBG("No register info received\n");
            return -1;
        }

        /* The copy_from/to_user functions return the number of uncopied bytes,
         * so a return of zero is a successul status
         */
        copyRet = copy_from_user(&regInfo, regP, sizeof(regInfo));

        if (copyRet)
            return -1;
    }

    switch (command)
    {
        case FPGA_IOCTL_START_WDT:
        case FPGA_IOCTL_DISABLE_WDT:
        case FPGA_IOCTL_PET_WDT:
            /* These IOCTLs are not implemented as the
             * FPGA does not currently have a role in
             * watchdog timers for SDS platforms.
             */
            DBG("Invalid Watchdog-FPGA IOCTL request: 0x%x\n", command);
            return -ENOTTY;

        case FPGA_IOCTL_READ_REG:
            EXT_DBG("Doing fpga ioctl %x (READ_REG) on %lx\n", command, data);
            EXT_DBG("Reading register offset %x\n", regInfo.offset);
            regInfo.data = READ_REG(regInfo.offset) & 0xFF;
            copyRet = copy_to_user(regP, &regInfo, sizeof(regInfo));
            break;

        case FPGA_IOCTL_WRITE_REG:
            EXT_DBG("Doing fpga ioctl %x (WRITE_REG) on %lx\n", command, data);
            regData = READ_REG(regInfo.offset) & ~regInfo.mask;
            regData |= (regInfo.data & regInfo.mask);
            EXT_DBG("Writing register offset %x, mask %x, data %lx\n",
                     regInfo.offset, regInfo.mask, regData);
            WRITE_REG(regInfo.offset, regData);
            break;

        case FPGA_IOCTL_WRITE_SHADOW:
            DBG("Invalid request: FPGA_IOCTL_WRITE_SHADOW\n");
            return -1;

        case FPGA_IOCTL_WRITE_PROT_REG:
            EXT_DBG("Doing fpga ioctl %x (WRITE_PROT_REG) on %lx\n", command, data);
            /* Just do the same thing as FPGA_IOCTL_WRITE_REG for now */
            regData = READ_REG(regInfo.offset) & ~regInfo.mask;
            regData |= (regInfo.data & regInfo.mask);
            EXT_DBG("Writing register offset %x, mask %x, data %lx\n",
                     regInfo.offset, regInfo.mask, regData);
            WRITE_REG(regInfo.offset, regData);
            break;

        case FPGA_IOCTL_REG_PROT_WRITE:
            /* Not implemented, just print DBG info so we can see it happen */
            DBG("Writing PROT WRITE sequence\n");
            break;

        case FPGA_IOCTL_READ_NVSRAM:
            /* MM FPGA doesn't have NVSRAM, return data of 0 for now */
            DBG("Doing fpga ioctl %x (READ_NVSRAM) on %lx\n", command, data);

            DBG("Reading NVSRAM offset %x\n", regInfo.offset);
            regInfo.data = 0;
            copyRet = copy_to_user(regP, &regInfo, sizeof(regInfo));
            break;

        case FPGA_IOCTL_WRITE_NVSRAM:
            /* MM FPGA doesn't have NVSRAM, just log and continue for now */
            DBG("Doing fpga ioctl %x (WRITE_NVSRAM) on %lx\n", command, data);

            DBG("Writing NVSRAM offset %x, mask %x, data %x\n", regInfo.offset, regInfo.mask, regInfo.data);
            break;

        case FPGA_IOCTL_REGISTER_ISR_INT:
            EXT_DBG("Doing fpga ioctl %x (REGISTER_ISR_INT) on %lx\n", command, data);

            /* Implement IRQ later, just fail for now */
            DBG("IRQ registration failed\n");
            return -1;

        case FPGA_IOCTL_REGISTER_ISR_MSI:
            EXT_DBG("Doing fpga ioctl %x (REGISTER_ISR_MSI) on %lx\n", command, data);

            /* Implement MSI later, just fail for now */
            DBG("MSI registration failed\n");
            return -1;

        case FPGA_IOCTL_ACK_INT:
            EXT_DBG("Doing fpga ioctl %x (ACK_INT) on %lx\n", command, data);

            /* Can't ACK until we implement an IRQ/MSI */
            DBG("IRQ ACK not implemented yet.\n");
            return -1;

        case FPGA_IOCTL_SET_HW_ID:
            hardwareId = (int32_t) data;
            copyRet = 0;
            break;

        case FPGA_IOCTL_GET_HW_ID:
            hwid_ptr = (int32_t *)data;
            if (put_user(hardwareId, hwid_ptr))
                return -EFAULT;

            copyRet = 0;
            break;

        default:
            return -ENOTTY;
    }

    return copyRet ? -1: 0;
}


/**
 * FPGA open routine.
 *
 * Nothing to do since there's only a single instance of the MM FPGA
 *
 */
static int fpgaMmOpen(struct inode* inode, struct file* filp)
{
    EXT_DBG("Opening fpga node\n");

    return 0;
}

/**
 * FPGA close routine.
 *
 * No special operations are required for this close routine.
 *
 */
static int fpgaMmRelease(struct inode* inode, struct file* filp)
{
    EXT_DBG("Closing fpga node\n");

    /*
     * Nothing was allocated or mapped during open, so there's really nothing
     * to do here. No resources need to be freed or anything like that.
     */
    return 0;
}

/**
 * FPGA mmap routine.
 *
 * Maps the FPGA region to user space
 *
 */
static int fpgaMmap(struct file* filp, struct vm_area_struct* vma)
{
    unsigned long size = vma->vm_end - vma->vm_start;
    unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
    (void)filp;   /* Not used */

    DBG("Attempting to map FPGA registers, size 0x%lx\n", size);

    /* Make sure not trying to get more memory than permitted */
    if (size > fpgaMmDev.regs_size)
    {
        DBG("Invalid size being mapped (%ld)\n", size);
        return -EINVAL;
    }

    /* Make sure starting point is at beginning of FPGA */
    if (offset != 0)
    {
        DBG("Offset must be zero (%ld)\n", offset);
        return -EINVAL;
    }

    /* Make sure user specified shared page */
    if (!(vma->vm_flags & VM_SHARED))
    {
        DBG("Invalid vm_flags (%lx)\n", vma->vm_flags);
        return -EINVAL;
    }


    /* Prevent memory from being swapped out */
    vma->vm_flags |= VM_LOCKED;

    /* Now map the region */
    if (remap_pfn_range(vma, vma->vm_start,
                        PFN_DOWN(virt_to_phys((void*)fpgaMmDev.virt_regs_base)),
                        size, PAGE_SHARED))
    {
        DBG("Unable to map region\n");
        return -EAGAIN;
    }

    return 0;
}

static int fpgaMmClassCreate(void)
{
    fpgaMmMajor = register_chrdev(0, FPGA_SDS_NAME, &fpgaFileOps);
    if (fpgaMmMajor < 0)
    {
        DBG("Failed to allocate major device numbers\n");
        return fpgaMmMajor;
    }
    DBG("Character device number assigned to major(%d)\n", fpgaMmMajor);

    /* Only create the class if not already created or if last attempt failed */
    fpgaMmClass = class_create(THIS_MODULE, FPGA_SDS_NAME);
    if(IS_ERR(fpgaMmClass))
    {
        unregister_chrdev(fpgaMmMajor, FPGA_SDS_NAME);
        DBG(FPGA_SDS_NAME " class registration failed\n");
        return PTR_ERR(fpgaMmClass);
    }
    DBG(FPGA_SDS_NAME " class registration completed\n");

    return 0;
}

static void fpgaMmClassDestroy(void)
{
    class_destroy(fpgaMmClass);
    unregister_chrdev(fpgaMmMajor, FPGA_SDS_NAME);
}

static int fpgaMmDevCreate(void)
{
    unsigned long kmalloc_area;
    unsigned long virt_addr;

    DBG(FPGA_SDS_NAME "fpgaMmDevCreate called\n");

    if (IS_PLATFORM_REGS_DEFINED())
    {
        DBG(FPGA_SDS_NAME "MM FPGA already exists\n");
        return 0;
    }

    /* Set up "register space" (just regular memory for now) */
    fpgaMmDev.regs_size = 0x1000;
    fpgaMmDev.virt_regs_base = (unsigned long)kmalloc(fpgaMmDev.regs_size, GFP_KERNEL);
    if (fpgaMmDev.virt_regs_base == 0)
    {
        DBG("Could not allocate memory for the virt base\n");
        return -ENODEV;
    }

    /* Get the start of the kmalloc memory address aligned to PAGE_SIZE
       and then iterate through the pages for the kmalloc'ed memory
       and reserve them so that they will not be swapped out
     */
    kmalloc_area = (fpgaMmDev.virt_regs_base + PAGE_SIZE - 1) & PAGE_MASK;
    for (virt_addr = kmalloc_area; virt_addr < kmalloc_area + fpgaMmDev.regs_size;
         virt_addr += PAGE_SIZE)
    {
        SetPageReserved(virt_to_page(virt_addr));
    }

    /* Initialize registers */
    memset((void*)fpgaMmDev.virt_regs_base, 0, fpgaMmDev.regs_size);

    /* Only allow a single device - fpga_sds0 */
    strncpy(fpgaMmDev.device_name, FPGA_SDS_NAME "0", sizeof(fpgaMmDev.device_name));

    /* Create the device */
    fpgaMmDev.device_number = MKDEV(fpgaMmMajor, 0);
    DBG(FPGA_SDS_NAME "device number: %d\n", fpgaMmDev.device_number);

    if (IS_ERR(device_create(fpgaMmClass, NULL, fpgaMmDev.device_number, NULL, fpgaMmDev.device_name)))
    {
        DBG(FPGA_SDS_NAME " class device creation failed\n");
        return -ENODEV;
    }

    /* Now the driver is initialized and registered and ready for action. */
    return 0;
}

static int fpgaMmDevDestroy(void)
{
    unsigned long kmalloc_area;
    unsigned long virt_addr;

    DBG("Removing MM FPGA\n");

    if (IS_PLATFORM_REGS_DEFINED())
    {
        device_destroy(fpgaMmClass, fpgaMmDev.device_number);

        /* Get the start of the kmalloc memory address aligned to PAGE_SIZE
           and then iterate through the pages for the kmalloc'ed memory
           and reserve them so that they will be remapable
         */
        kmalloc_area = (fpgaMmDev.virt_regs_base + PAGE_SIZE - 1) & PAGE_MASK;
        for (virt_addr = kmalloc_area; virt_addr < kmalloc_area + fpgaMmDev.regs_size;
             virt_addr += PAGE_SIZE)
        {
            ClearPageReserved(virt_to_page(virt_addr));
        }
        kfree((void*)fpgaMmDev.virt_regs_base);
        memset(&fpgaMmDev, 0, sizeof(struct fpgaDev));
        DBG("device destroyed\n");
    }

    return 0;
}

/**
 * Standard Linux module registration
 */

static __init int fpgaMmInit(void)
{
    int rc = 0;
    DBG("Initializing FPGA driver\n");
    rc = fpgaMmClassCreate();
    if (rc)
    {
        DBG("Could not perform fpga class create\n");
        return rc;
    }

    rc = fpgaMmDevCreate();
    if (rc)
    {
        DBG("Could not perform fpga dev create\n");
    }
    return rc;
}

static __exit void fpgaMmExit(void)
{
    fpgaMmDevDestroy();
    fpgaMmClassDestroy();
}

module_init(fpgaMmInit);
module_exit(fpgaMmExit);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("NetApp FPGA SDS Driver " FPGA_VERSION);
MODULE_AUTHOR("NetApp, Inc");

