#!/usr/bin/env python
# -*- coding: utf-8 -*-

# --------------------------------------------
import logging
class NullHandler(logging.Handler):
    def emit(self, record):
        pass
logger = logging.getLogger('pyagentx.agent')
logger.addHandler(NullHandler())
# --------------------------------------------

from threading import Event
import time
import queue

import pyagentx
from pyagentx.updater import Updater
from pyagentx.network import Network


class AgentError(Exception):
    pass


class Agent(object):

    def __init__(self):
        self._updater_list = []
        self._sethandlers = {}
        self._threads = []
        self._queue = queue.Queue(maxsize=20)
        self.network = None
        self.die = Event()

    def register(self, oid, class_, freq=10):
        if Updater not in class_.__bases__:
            raise AgentError('Class given isn\'t an updater')
        # cleanup and test oid
        try:
            oid = oid.strip(' .')
            [int(i) for i in oid.split('.')]
        except ValueError:
            raise AgentError('OID isn\'t valid')
        self._updater_list.append({'oid':oid, 'class':class_, 'freq':freq})

    def register_set(self, oid, class_):
        if pyagentx.SetHandler not in class_.__bases__:
            raise AgentError('Class given isn\'t a SetHandler')
        # cleanup and test oid
        try:
            oid = oid.strip(' .')
            [int(i) for i in oid.split('.')]
        except ValueError:
            raise AgentError('OID isn\'t valid')
        self._sethandlers[oid] = class_()

    def setup(self):
        # Override this
        pass

    def start(self):
        # Start Updaters
        for u in self._updater_list:
            name = u['class'].__name__
            logger.info('Starting updater %s [%s]' % (name, u['oid']))
            t = u['class'](name=name)
            t.agent_setup(self._queue, u['oid'], u['freq'])
            t.daemon = True
            t.start()
            self._threads.append(t)
        # Start Network
        oid_list = [u['oid'] for u in self._updater_list]
        self.network = Network(self._queue, oid_list, self._sethandlers)
        self.network.start_network(self.die)

    def stop(self):
        for t in self._threads:
            logger.info("Stopping updater {}".format(t.name))
            t.stop.set()
        for t in self._threads:
            logger.info("Waiting to join thread {}".format(t.name))
            t.join()
        del self._threads[:]
