/*******************************************************************************

    fpga.h
    Copyright 2014-2020 NetApp, Inc. All rights reserved.

    Provides definitions needed to access the FPGA loadable kernel
    module by application code.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

*******************************************************************************/


#ifndef __INCfpga
#define __INCfpga

#include <linux/types.h>
#include <linux/ioctl.h>

/**
 * \brief Structure used for register accesses.
 */
typedef struct _fpga_reg
{
    uint32_t offset;    /* register offset */
    uint32_t mask;      /* 32 bits even though only 8 bits are needed for alignment */
    uint32_t data;      /* 32 bits even though only 8 bits are needed for alignment */
    uint32_t shadow;    /* shadow register offset */
} fpga_reg;

/**
 * \brief Structure used to pass over the protected write sequence
 */
typedef struct _fpga_prot_write
{
    uint32_t size;      /* Number of bytes of data in the protected write sequence */
    uint32_t reserved;  /* Padding to align data */
    uint64_t addr;      /* Pointer to user structure which contains the protected write sequence */
} fpga_prot_write;

/**
 * \brief IOCTL codes
 *
 * - magic number of 'f' was selected for this driver
 * - ioctl index numbers are chosen sequentially starting with 1
 *
 */
#define FPGA_IOCTL_MAGIC_NUMBER 'f'

#define FPGA_IOCTL_START_WDT                _IOW(FPGA_IOCTL_MAGIC_NUMBER, 1, fpga_reg)
#define FPGA_IOCTL_DISABLE_WDT              _IO(FPGA_IOCTL_MAGIC_NUMBER, 2)
#define FPGA_IOCTL_PET_WDT                  _IOW(FPGA_IOCTL_MAGIC_NUMBER, 3, fpga_reg)
#define FPGA_IOCTL_READ_REG                 _IOR(FPGA_IOCTL_MAGIC_NUMBER, 4, fpga_reg)
#define FPGA_IOCTL_WRITE_REG                _IOW(FPGA_IOCTL_MAGIC_NUMBER, 5, fpga_reg)
#define FPGA_IOCTL_WRITE_PROT_REG           _IOW(FPGA_IOCTL_MAGIC_NUMBER, 6, fpga_reg)
#define FPGA_IOCTL_REG_PROT_WRITE           _IOW(FPGA_IOCTL_MAGIC_NUMBER, 7, fpga_prot_write)
#define FPGA_IOCTL_READ_NVSRAM              _IOW(FPGA_IOCTL_MAGIC_NUMBER, 8, fpga_reg)
#define FPGA_IOCTL_WRITE_NVSRAM             _IOW(FPGA_IOCTL_MAGIC_NUMBER, 9, fpga_reg)
#define FPGA_IOCTL_REGISTER_ISR_INT         _IOW(FPGA_IOCTL_MAGIC_NUMBER, 10, fpga_reg)
#define FPGA_IOCTL_REGISTER_ISR_MSI         _IOW(FPGA_IOCTL_MAGIC_NUMBER, 11, fpga_reg)
#define FPGA_IOCTL_ACK_INT                  _IO(FPGA_IOCTL_MAGIC_NUMBER, 12)
#define FPGA_IOCTL_SET_HW_ID                _IOW(FPGA_IOCTL_MAGIC_NUMBER, 14, int32_t)
#define FPGA_IOCTL_GET_HW_ID                _IOR(FPGA_IOCTL_MAGIC_NUMBER, 15, int32_t *)
#define FPGA_IOCTL_WRITE_SHADOW             _IOW(FPGA_IOCTL_MAGIC_NUMBER, 16, fpga_reg)

#define FPGA_NAME          "fpga"

#define FPGA_DEV_PATH      "/dev/" FPGA_NAME "0"
#define FPGA_RAID_PATH     "/dev/" FPGA_NAME "1"

#define NVSRAM_DUAL_BIOS_OFFSET ((uint32_t)(~0))  /* All F's to designate a special address */

#endif /*__INCfpga*/
