// To change feature flags, edit:
//  pkg/services/featuremgmt/registry.go
// Then run tests in:
//  pkg/services/featuremgmt/toggles_gen_test.go
// twice to generate and validate the feature flag files
//
// Alternatively, use `make gen-feature-toggles`

package featuremgmt

import (
	"embed"
	"encoding/json"

	featuretoggle "github.com/grafana/grafana/pkg/apis/featuretoggle/v0alpha1"
)

var (
	// Register each toggle here
	standardFeatureFlags = []FeatureFlag{
		{
			Name:              "disableEnvelopeEncryption",
			Description:       "Disable envelope encryption (emergency only)",
			Stage:             FeatureStageGeneralAvailability,
			Owner:             grafanaAsCodeSquad,
			HideFromAdminPage: true,
			AllowSelfServe:    false,
			Expression:        "false",
		},
		{
			Name:         "live-service-web-worker",
			Description:  "This will use a webworker thread to processes events rather than the main thread",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
		},
		{
			Name:         "queryOverLive",
			Description:  "Use Grafana Live WebSocket to execute backend queries",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
		},
		{
			Name:              "panelTitleSearch",
			Description:       "Search for dashboards using panel title",
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromAdminPage: true,
		},
		{
			Name:           "publicDashboards",
			Description:    "[Deprecated] Public dashboards are now enabled by default; to disable them, use the configuration setting. This feature toggle will be removed in the next major version.",
			Stage:          FeatureStageGeneralAvailability,
			Owner:          grafanaSharingSquad,
			Expression:     "true", // enabled by default
			AllowSelfServe: true,
		},
		{
			Name:              "publicDashboardsEmailSharing",
			Description:       "Enables public dashboard sharing to be restricted to only allowed emails",
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaSharingSquad,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:         "publicDashboardsScene",
			Description:  "Enables public dashboard rendering using scenes",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaSharingSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:        "lokiExperimentalStreaming",
			Description: "Support new streaming approach for loki (prototype, needs special loki build)",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaObservabilityLogsSquad,
		},
		{
			Name:           "featureHighlights",
			Description:    "Highlight Grafana Enterprise features",
			Stage:          FeatureStageGeneralAvailability,
			Owner:          grafanaAsCodeSquad,
			AllowSelfServe: true,
			Expression:     "false",
		},
		{
			Name:        "storage",
			Description: "Configurable storage for dashboards, datasources, and resources",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaSearchAndStorageSquad,
		},
		{
			Name:           "correlations",
			Description:    "Correlations page",
			Stage:          FeatureStageGeneralAvailability,
			Owner:          grafanaExploreSquad,
			Expression:     "true", // enabled by default
			AllowSelfServe: true,
		},
		{
			Name:         "autoMigrateOldPanels",
			Description:  "Migrate old angular panels to supported versions (graph, table-old, worldmap, etc)",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "autoMigrateGraphPanel",
			Description:  "Migrate old graph panel to supported time series panel - broken out from autoMigrateOldPanels to enable granular tracking",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "autoMigrateTablePanel",
			Description:  "Migrate old table panel to supported table panel - broken out from autoMigrateOldPanels to enable granular tracking",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "autoMigratePiechartPanel",
			Description:  "Migrate old piechart panel to supported piechart panel - broken out from autoMigrateOldPanels to enable granular tracking",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "autoMigrateWorldmapPanel",
			Description:  "Migrate old worldmap panel to supported geomap panel - broken out from autoMigrateOldPanels to enable granular tracking",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "autoMigrateStatPanel",
			Description:  "Migrate old stat panel to supported stat panel - broken out from autoMigrateOldPanels to enable granular tracking",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "autoMigrateXYChartPanel",
			Description:  "Migrate old XYChart panel to new XYChart2 model",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Expression:   "true", // enabled by default
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:              "disableAngular",
			Description:       "Dynamic flag to disable angular at runtime. The preferred method is to set `angular_support_enabled` to `false` in the [security] settings, which allows you to change the state at runtime.",
			Stage:             FeatureStagePublicPreview,
			FrontendOnly:      true,
			Owner:             grafanaDatavizSquad,
			HideFromAdminPage: true,
		},
		{
			Name:              "canvasPanelNesting",
			Description:       "Allow elements nesting",
			Stage:             FeatureStageExperimental,
			FrontendOnly:      true,
			Owner:             grafanaDatavizSquad,
			HideFromAdminPage: true,
		},
		{
			Name:              "vizActions",
			Description:       "Allow actions in visualizations",
			Stage:             FeatureStageExperimental,
			FrontendOnly:      true,
			Owner:             grafanaDatavizSquad,
			HideFromAdminPage: true,
		},
		{
			Name:            "disableSecretsCompatibility",
			Description:     "Disable duplicated secret storage in legacy tables",
			Stage:           FeatureStageExperimental,
			RequiresRestart: true,
			Owner:           hostedGrafanaTeam,
		},
		{
			Name:        "logRequestsInstrumentedAsUnknown",
			Description: "Logs the path for requests that are instrumented as unknown",
			Stage:       FeatureStageExperimental,
			Owner:       hostedGrafanaTeam,
		},
		{
			// Some plugins rely on topnav feature flag being enabled, so we cannot remove this until we
			// can afford the breaking change, or we've detemined no one else is relying on it
			Name:        "topnav",
			Description: "Enables topnav support in external plugins. The new Grafana navigation cannot be disabled.",
			Stage:       FeatureStageDeprecated,
			Expression:  "true", // enabled by default
			Owner:       grafanaFrontendPlatformSquad,
		},
		{
			Name:              "grpcServer",
			Description:       "Run the GRPC server",
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromAdminPage: true,
		},
		{
			Name:           "cloudWatchCrossAccountQuerying",
			Description:    "Enables cross-account querying in CloudWatch datasources",
			Stage:          FeatureStageGeneralAvailability,
			Expression:     "true", // enabled by default
			Owner:          awsDatasourcesSquad,
			AllowSelfServe: true,
		},
		{
			Name:        "showDashboardValidationWarnings",
			Description: "Show warnings when dashboards do not validate against the schema",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaDashboardsSquad,
		},
		{
			Name:        "mysqlAnsiQuotes",
			Description: "Use double quotes to escape keyword in a MySQL query",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaSearchAndStorageSquad,
		},
		{
			Name:        "mysqlParseTime",
			Description: "Ensure the parseTime flag is set for MySQL driver",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaSearchAndStorageSquad,
		},
		{
			Name:              "accessControlOnCall",
			Description:       "Access control primitives for OnCall",
			Stage:             FeatureStageGeneralAvailability,
			Owner:             identityAccessTeam,
			HideFromAdminPage: true,
			Expression:        "true", // enabled by default
		},
		{
			Name:        "nestedFolders",
			Description: "Enable folder nesting",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaSearchAndStorageSquad,
			Expression:  "true", // enabled by default
		},
		{
			Name:        "alertingBacktesting",
			Description: "Rule backtesting API for alerting",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAlertingSquad,
		},
		{
			Name:         "editPanelCSVDragAndDrop",
			Description:  "Enables drag and drop for CSV and Excel files",
			FrontendOnly: true,
			Stage:        FeatureStageExperimental,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:              "alertingNoNormalState",
			Description:       "Stop maintaining state of alerts that are not firing",
			Stage:             FeatureStagePublicPreview,
			RequiresRestart:   false,
			Owner:             grafanaAlertingSquad,
			HideFromAdminPage: true,
		},
		{
			Name:           "logsContextDatasourceUi",
			Description:    "Allow datasource to provide custom UI for context view",
			Stage:          FeatureStageGeneralAvailability,
			FrontendOnly:   true,
			Owner:          grafanaObservabilityLogsSquad,
			Expression:     "true", // turned on by default
			AllowSelfServe: true,
		},
		{
			Name:           "lokiQuerySplitting",
			Description:    "Split large interval queries into subqueries with smaller time intervals",
			Stage:          FeatureStageGeneralAvailability,
			FrontendOnly:   true,
			Owner:          grafanaObservabilityLogsSquad,
			Expression:     "true", // turned on by default
			AllowSelfServe: true,
		},
		{
			Name:         "lokiQuerySplittingConfig",
			Description:  "Give users the option to configure split durations for Loki queries",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:        "individualCookiePreferences",
			Description: "Support overriding cookie preferences per user",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaBackendGroup,
		},
		{
			Name:           "prometheusMetricEncyclopedia",
			Description:    "Adds the metrics explorer component to the Prometheus query builder as an option in metric select",
			Expression:     "true",
			Stage:          FeatureStageGeneralAvailability,
			FrontendOnly:   true,
			Owner:          grafanaObservabilityMetricsSquad,
			AllowSelfServe: true,
		},
		{
			Name:           "influxdbBackendMigration",
			Description:    "Query InfluxDB InfluxQL without the proxy",
			Stage:          FeatureStageGeneralAvailability,
			FrontendOnly:   true,
			Owner:          grafanaObservabilityMetricsSquad,
			Expression:     "true", // enabled by default
			AllowSelfServe: false,
		},
		{
			Name:        "influxqlStreamingParser",
			Description: "Enable streaming JSON parser for InfluxDB datasource InfluxQL query language",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaObservabilityMetricsSquad,
		},
		{
			Name:         "influxdbRunQueriesInParallel",
			Description:  "Enables running InfluxDB Influxql queries in parallel",
			Stage:        FeatureStagePrivatePreview,
			FrontendOnly: false,
			Owner:        grafanaObservabilityMetricsSquad,
		},
		{
			Name:         "prometheusRunQueriesInParallel",
			Description:  "Enables running Prometheus queries in parallel",
			Stage:        FeatureStagePrivatePreview,
			FrontendOnly: false,
			Owner:        grafanaObservabilityMetricsSquad,
		},
		{
			Name:           "lokiMetricDataplane",
			Description:    "Changes metric responses from Loki to be compliant with the dataplane specification.",
			Stage:          FeatureStageGeneralAvailability,
			Expression:     "true",
			Owner:          grafanaObservabilityLogsSquad,
			AllowSelfServe: true,
		},
		{
			Name:        "lokiLogsDataplane",
			Description: "Changes logs responses from Loki to be compliant with the dataplane specification.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaObservabilityLogsSquad,
		},
		{
			Name:           "dataplaneFrontendFallback",
			Description:    "Support dataplane contract field name change for transformations and field name matchers where the name is different",
			Stage:          FeatureStageGeneralAvailability,
			FrontendOnly:   true,
			Expression:     "true",
			Owner:          grafanaObservabilityMetricsSquad,
			AllowSelfServe: true,
		},
		{
			Name:        "disableSSEDataplane",
			Description: "Disables dataplane specific processing in server side expressions.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaObservabilityMetricsSquad,
		},
		{
			Name:        "alertStateHistoryLokiSecondary",
			Description: "Enable Grafana to write alert state history to an external Loki instance in addition to Grafana annotations.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAlertingSquad,
		},
		{
			Name:        "alertStateHistoryLokiPrimary",
			Description: "Enable a remote Loki instance as the primary source for state history reads.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAlertingSquad,
		},
		{
			Name:        "alertStateHistoryLokiOnly",
			Description: "Disable Grafana alerts from emitting annotations when a remote Loki instance is available.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAlertingSquad,
		},
		{
			Name:        "unifiedRequestLog",
			Description: "Writes error logs to the request logger",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaBackendGroup,
		},
		{
			Name:              "renderAuthJWT",
			Description:       "Uses JWT-based auth for rendering instead of relying on remote cache",
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaAsCodeSquad,
			HideFromAdminPage: true,
		},
		{
			Name:              "refactorVariablesTimeRange",
			Description:       "Refactor time range variables flow to reduce number of API calls made when query variables are chained",
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaDashboardsSquad,
			HideFromAdminPage: true, // Non-feature, used to test out a bug fix that impacts the performance of template variables.
		},
		{
			Name:         "faroDatasourceSelector",
			Description:  "Enable the data source selector within the Frontend Apps section of the Frontend Observability",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        appO11ySquad,
		},
		{
			Name:         "enableDatagridEditing",
			Description:  "Enables the edit functionality in the datagrid panel",
			FrontendOnly: true,
			Stage:        FeatureStagePublicPreview,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "extraThemes",
			Description:  "Enables extra themes",
			FrontendOnly: true,
			Stage:        FeatureStageExperimental,
			Owner:        grafanaFrontendPlatformSquad,
		},
		{
			Name:         "lokiPredefinedOperations",
			Description:  "Adds predefined query operations to Loki query editor",
			FrontendOnly: true,
			Stage:        FeatureStageExperimental,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:        "pluginsFrontendSandbox",
			Description: "Enables the plugins frontend sandbox",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaPluginsPlatformSquad,
		},
		{
			Name:         "frontendSandboxMonitorOnly",
			Description:  "Enables monitor only in the plugin frontend sandbox (if enabled)",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaPluginsPlatformSquad,
		},
		{
			Name:         "pluginsDetailsRightPanel",
			Description:  "Enables right panel for the plugins details page",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaPluginsPlatformSquad,
		},
		{
			Name:              "sqlDatasourceDatabaseSelection",
			Description:       "Enables previous SQL data source dataset dropdown behavior",
			FrontendOnly:      true,
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaDatavizSquad,
			HideFromAdminPage: true,
		},
		{
			Name:           "recordedQueriesMulti",
			Description:    "Enables writing multiple items from a single query within Recorded Queries",
			Stage:          FeatureStageGeneralAvailability,
			Expression:     "true",
			Owner:          grafanaObservabilityMetricsSquad,
			AllowSelfServe: false,
		},
		{
			Name:         "vizAndWidgetSplit",
			Description:  "Split panels between visualizations and widgets",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
		},
		{
			Name:         "logsExploreTableVisualisation",
			Description:  "A table visualisation for logs in Explore",
			Stage:        FeatureStageGeneralAvailability,
			Expression:   "true", // enabled by default,
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:        "awsDatasourcesTempCredentials",
			Description: "Support temporary security credentials in AWS plugins for Grafana Cloud customers",
			Stage:       FeatureStageExperimental,
			Owner:       awsDatasourcesSquad,
		},
		{
			Name:           "transformationsRedesign",
			Description:    "Enables the transformations redesign",
			Stage:          FeatureStageGeneralAvailability,
			FrontendOnly:   true,
			Expression:     "true", // enabled by default
			Owner:          grafanaObservabilityMetricsSquad,
			AllowSelfServe: true,
		},
		{
			Name:         "mlExpressions",
			Description:  "Enable support for Machine Learning in server-side expressions",
			Stage:        FeatureStageExperimental,
			FrontendOnly: false,
			Owner:        grafanaAlertingSquad,
		},
		{
			Name:         "traceQLStreaming",
			Description:  "Enables response streaming of TraceQL queries of the Tempo data source",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaObservabilityTracesAndProfilingSquad,
			Expression:   "false",
		},
		{
			Name:         "metricsSummary",
			Description:  "Enables metrics summary queries in the Tempo data source",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaObservabilityTracesAndProfilingSquad,
		},
		{
			Name:            "datasourceAPIServers",
			Description:     "Expose some datasources as apiservers.",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true, // changes the API routing
		},
		{
			Name:            "grafanaAPIServerWithExperimentalAPIs",
			Description:     "Register experimental APIs with the k8s API server, including all datasources",
			Stage:           FeatureStageExperimental,
			RequiresRestart: true,
			RequiresDevMode: true,
			Owner:           grafanaAppPlatformSquad,
		},
		{
			Name:            "grafanaAPIServerEnsureKubectlAccess",
			Description:     "Start an additional https handler and write kubectl options",
			Stage:           FeatureStageExperimental,
			RequiresDevMode: true,
			RequiresRestart: true,
			Owner:           grafanaAppPlatformSquad,
		},
		{
			Name:            "featureToggleAdminPage",
			Description:     "Enable admin page for managing feature toggles from the Grafana front-end. Grafana Cloud only.",
			Stage:           FeatureStageExperimental,
			FrontendOnly:    false,
			Owner:           grafanaOperatorExperienceSquad,
			RequiresRestart: true,
			HideFromDocs:    true,
		},
		{
			Name:        "awsAsyncQueryCaching",
			Description: "Enable caching for async queries for Redshift and Athena. Requires that the datasource has caching and async query support enabled",
			Stage:       FeatureStageGeneralAvailability,
			Expression:  "true", // enabled by default
			Owner:       awsDatasourcesSquad,
		},
		{
			Name:        "permissionsFilterRemoveSubquery",
			Description: "Alternative permission filter implementation that does not use subqueries for fetching the dashboard folder",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaBackendGroup,
		},
		{
			Name:           "prometheusConfigOverhaulAuth",
			Description:    "Update the Prometheus configuration page with the new auth component",
			Owner:          grafanaObservabilityMetricsSquad,
			Stage:          FeatureStageGeneralAvailability,
			Expression:     "true", // on by default
			AllowSelfServe: false,
		},
		{
			Name:            "configurableSchedulerTick",
			Description:     "Enable changing the scheduler base interval via configuration option unified_alerting.scheduler_tick_interval",
			Stage:           FeatureStageExperimental,
			FrontendOnly:    false,
			Owner:           grafanaAlertingSquad,
			RequiresRestart: true,
			HideFromDocs:    true,
		},
		{
			Name:            "alertingNoDataErrorExecution",
			Description:     "Changes how Alerting state manager handles execution of NoData/Error",
			Stage:           FeatureStageGeneralAvailability,
			FrontendOnly:    false,
			Owner:           grafanaAlertingSquad,
			RequiresRestart: true,
			Expression:      "true", // enabled by default
		},
		{
			Name:         "angularDeprecationUI",
			Description:  "Display Angular warnings in dashboards and panels",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaPluginsPlatformSquad,
			Expression:   "true", // Enabled by default
		},
		{
			Name:         "dashgpt",
			Description:  "Enable AI powered features in dashboards",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:         "aiGeneratedDashboardChanges",
			Description:  "Enable AI powered features for dashboards to auto-summary changes when saving",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
		},
		{
			Name:            "reportingRetries",
			Description:     "Enables rendering retries for the reporting feature",
			Stage:           FeatureStagePublicPreview,
			FrontendOnly:    false,
			Owner:           grafanaSharingSquad,
			RequiresRestart: true,
		},
		{
			Name:        "sseGroupByDatasource",
			Description: "Send query to the same datasource in a single request when using server side expressions. The `cloudWatchBatchQueries` feature toggle should be enabled if this used with CloudWatch.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaObservabilityMetricsSquad,
		},
		{
			Name:            "libraryPanelRBAC",
			Description:     "Enables RBAC support for library panels",
			Stage:           FeatureStageExperimental,
			FrontendOnly:    false,
			Owner:           grafanaDashboardsSquad,
			RequiresRestart: true,
		},
		{
			Name:         "lokiRunQueriesInParallel",
			Description:  "Enables running Loki queries in parallel",
			Stage:        FeatureStagePrivatePreview,
			FrontendOnly: false,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:         "wargamesTesting",
			Description:  "Placeholder feature flag for internal testing",
			Stage:        FeatureStageExperimental,
			FrontendOnly: false,
			Owner:        hostedGrafanaTeam,
		},
		{
			Name:              "alertingInsights",
			Description:       "Show the new alerting insights landing page",
			FrontendOnly:      true,
			Stage:             FeatureStageGeneralAvailability,
			Owner:             grafanaAlertingSquad,
			Expression:        "true", // enabled by default
			AllowSelfServe:    false,
			HideFromAdminPage: true, // This is moving away from being a feature toggle.
		},
		{
			Name:        "externalCorePlugins",
			Description: "Allow core plugins to be loaded as external",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaPluginsPlatformSquad,
		},
		{
			Name:         "pluginsAPIMetrics",
			Description:  "Sends metrics of public grafana packages usage by plugins",
			FrontendOnly: true,
			Stage:        FeatureStageExperimental,
			Owner:        grafanaPluginsPlatformSquad,
		},
		{
			Name:              "externalServiceAccounts",
			Description:       "Automatic service account and token setup for plugins",
			HideFromAdminPage: true,
			Stage:             FeatureStagePublicPreview,
			Owner:             identityAccessTeam,
		},
		{
			Name:         "panelMonitoring",
			Description:  "Enables panel monitoring through logs and measurements",
			Stage:        FeatureStageGeneralAvailability,
			Expression:   "true", // enabled by default
			Owner:        grafanaDatavizSquad,
			FrontendOnly: true,
		},
		{
			Name:              "enableNativeHTTPHistogram",
			Description:       "Enables native HTTP Histograms",
			Stage:             FeatureStageExperimental,
			FrontendOnly:      false,
			Owner:             grafanaBackendServicesSquad,
			HideFromAdminPage: true,
			AllowSelfServe:    false,
			RequiresRestart:   true,
		},
		{
			Name:              "disableClassicHTTPHistogram",
			Description:       "Disables classic HTTP Histogram (use with enableNativeHTTPHistogram)",
			Stage:             FeatureStageExperimental,
			FrontendOnly:      false,
			Owner:             grafanaBackendServicesSquad,
			HideFromAdminPage: true,
			AllowSelfServe:    false,
			RequiresRestart:   true,
		},
		{
			Name:         "formatString",
			Description:  "Enable format string transformer",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:         "transformationsVariableSupport",
			Description:  "Allows using variables in transformations",
			FrontendOnly: true,
			Stage:        FeatureStageGeneralAvailability,
			Owner:        grafanaDatavizSquad,
			Expression:   "true", // Enabled by default
		},
		{
			Name:            "kubernetesPlaylists",
			Description:     "Use the kubernetes API in the frontend for playlists, and route /api/playlist requests to k8s",
			Stage:           FeatureStageGeneralAvailability,
			Owner:           grafanaAppPlatformSquad,
			Expression:      "true",
			RequiresRestart: true, // changes the API routing
		},
		{
			Name:            "kubernetesSnapshots",
			Description:     "Routes snapshot requests from /api to the /apis endpoint",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true, // changes the API routing
		},
		{
			Name:         "kubernetesDashboards",
			Description:  "Use the kubernetes API in the frontend for dashboards",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaAppPlatformSquad,
			FrontendOnly: true,
		},
		{
			Name:        "kubernetesFolders",
			Description: "Use the kubernetes API in the frontend for folders, and route /api/folders requests to k8s",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaSearchAndStorageSquad,
		},
		{
			Name:        "grafanaAPIServerTestingWithExperimentalAPIs",
			Description: "Facilitate integration testing of experimental APIs",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaSearchAndStorageSquad,
		},
		{
			Name:            "datasourceQueryTypes",
			Description:     "Show query type endpoints in datasource API servers (currently hardcoded for testdata, expressions, and prometheus)",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true, // changes the API routing
		},
		{
			Name:            "queryService",
			Description:     "Register /apis/query.grafana.app/ -- will eventually replace /api/ds/query",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true, // Adds a route at startup
		},
		{
			Name:            "queryServiceRewrite",
			Description:     "Rewrite requests targeting /ds/query to the query service",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true, // changes the API routing
		},
		{
			Name:         "queryServiceFromUI",
			Description:  "Routes requests to the new query service",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaAppPlatformSquad,
			FrontendOnly: true, // and can change at startup
		},
		{
			Name:        "cloudWatchBatchQueries",
			Description: "Runs CloudWatch metrics queries as separate batches",
			Stage:       FeatureStagePublicPreview,
			Owner:       awsDatasourcesSquad,
		},
		{
			Name:            "recoveryThreshold",
			Description:     "Enables feature recovery threshold (aka hysteresis) for threshold server-side expression",
			Stage:           FeatureStageGeneralAvailability,
			FrontendOnly:    false,
			Owner:           grafanaAlertingSquad,
			RequiresRestart: true,
			Expression:      "true",
		},
		{
			Name:         "lokiStructuredMetadata",
			Description:  "Enables the loki data source to request structured metadata from the Loki server",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: false,
			Owner:        grafanaObservabilityLogsSquad,
			Expression:   "true",
		},
		{
			Name:           "teamHttpHeaders",
			Description:    "Enables LBAC for datasources to apply LogQL filtering of logs to the client requests for users in teams",
			Stage:          FeatureStagePublicPreview,
			FrontendOnly:   false,
			AllowSelfServe: true,
			Owner:          identityAccessTeam,
		},
		{
			Name:         "cachingOptimizeSerializationMemoryUsage",
			Description:  "If enabled, the caching backend gradually serializes query responses for the cache, comparing against the configured `[caching]max_value_mb` value as it goes. This can can help prevent Grafana from running out of memory while attempting to cache very large query responses.",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaOperatorExperienceSquad,
			FrontendOnly: false,
		},
		{
			Name:            "panelTitleSearchInV1",
			Description:     "Enable searching for dashboards using panel title in search v1",
			RequiresDevMode: true,
			Stage:           FeatureStageExperimental,
			Owner:           grafanaSearchAndStorageSquad,
		},
		{
			Name:            "managedPluginsInstall",
			Description:     "Install managed plugins directly from plugins catalog",
			Stage:           FeatureStageGeneralAvailability,
			RequiresDevMode: false,
			Owner:           grafanaPluginsPlatformSquad,
			Expression:      "true", // enabled by default
		},
		{
			Name:         "prometheusPromQAIL",
			Description:  "Prometheus and AI/ML to assist users in creating a query",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaObservabilityMetricsSquad,
		},
		{
			Name:         "prometheusCodeModeMetricNamesSearch",
			Description:  "Enables search for metric names in Code Mode, to improve performance when working with an enormous number of metric names",
			FrontendOnly: true,
			Stage:        FeatureStageExperimental,
			Owner:        grafanaObservabilityMetricsSquad,
		},
		{
			Name:         "addFieldFromCalculationStatFunctions",
			Description:  "Add cumulative and window functions to the add field from calculation transformation",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:        "alertmanagerRemoteSecondary",
			Description: "Enable Grafana to sync configuration and state with a remote Alertmanager.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAlertingSquad,
		},
		{
			Name:        "alertmanagerRemotePrimary",
			Description: "Enable Grafana to have a remote Alertmanager instance as the primary Alertmanager.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAlertingSquad,
		},
		{
			Name:        "alertmanagerRemoteOnly",
			Description: "Disable the internal Alertmanager and only use the external one defined.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaAlertingSquad,
		},
		{
			Name:            "annotationPermissionUpdate",
			Description:     "Change the way annotation permissions work by scoping them to folders and dashboards.",
			Stage:           FeatureStageGeneralAvailability,
			RequiresDevMode: false,
			Expression:      "true", // enabled by default
			Owner:           identityAccessTeam,
		},
		{
			Name:         "extractFieldsNameDeduplication",
			Description:  "Make sure extracted field names are unique in the dataframe",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "dashboardSceneForViewers",
			Description:  "Enables dashboard rendering using Scenes for viewer roles",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:         "dashboardSceneSolo",
			Description:  "Enables rendering dashboards using scenes for solo panels",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:         "dashboardScene",
			Description:  "Enables dashboard rendering using scenes for all roles",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:         "panelFilterVariable",
			Description:  "Enables use of the `systemPanelFilterVar` variable to filter panels in a dashboard",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
			HideFromDocs: true,
		},
		{
			Name:         "pdfTables",
			Description:  "Enables generating table data as PDF in reporting",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: false,
			Owner:        grafanaSharingSquad,
		},
		{
			Name:           "ssoSettingsApi",
			Description:    "Enables the SSO settings API and the OAuth configuration UIs in Grafana",
			Stage:          FeatureStageGeneralAvailability,
			Expression:     "true",
			AllowSelfServe: true,
			FrontendOnly:   false,
			Owner:          identityAccessTeam,
		},
		{
			Name:         "canvasPanelPanZoom",
			Description:  "Allow pan and zoom in canvas panel",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "logsInfiniteScrolling",
			Description:  "Enables infinite scrolling for the Logs panel in Explore and Dashboards",
			Stage:        FeatureStageGeneralAvailability,
			Expression:   "true",
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:         "exploreMetrics",
			Description:  "Enables the new Explore Metrics core app",
			Stage:        FeatureStageGeneralAvailability,
			Expression:   "true", // enabled by default
			FrontendOnly: true,
			Owner:        grafanaDashboardsSquad,
		},
		{
			Name:         "alertingSimplifiedRouting",
			Description:  "Enables users to easily configure alert notifications by specifying a contact point directly when editing or creating an alert rule",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: false,
			Owner:        grafanaAlertingSquad,
			Expression:   "true", // enabled by default
		},
		{
			Name:         "logRowsPopoverMenu",
			Description:  "Enable filtering menu displayed when text of a log line is selected",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Expression:   "true",
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:         "pluginsSkipHostEnvVars",
			Description:  "Disables passing host environment variable to plugin processes",
			Stage:        FeatureStageExperimental,
			FrontendOnly: false,
			Owner:        grafanaPluginsPlatformSquad,
		},
		{
			Name:         "tableSharedCrosshair",
			Description:  "Enables shared crosshair in table panel",
			FrontendOnly: true,
			Stage:        FeatureStageExperimental,
			Owner:        grafanaDatavizSquad,
		},
		{
			Name:         "regressionTransformation",
			Description:  "Enables regression analysis transformation",
			Stage:        FeatureStagePublicPreview,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
		},
		{
			// this is mainly used as a way to quickly disable query hints as a safeguard for our infrastructure
			Name:           "lokiQueryHints",
			Description:    "Enables query hints for Loki",
			Stage:          FeatureStageGeneralAvailability,
			FrontendOnly:   true,
			Expression:     "true",
			Owner:          grafanaObservabilityLogsSquad,
			AllowSelfServe: false,
		},
		{
			Name:              "kubernetesFeatureToggles",
			Description:       "Use the kubernetes API for feature toggle management in the frontend",
			Stage:             FeatureStageExperimental,
			FrontendOnly:      true,
			Owner:             grafanaOperatorExperienceSquad,
			AllowSelfServe:    false,
			HideFromAdminPage: true,
		},
		{
			Name:              "cloudRBACRoles",
			Description:       "Enabled grafana cloud specific RBAC roles",
			Stage:             FeatureStagePublicPreview,
			Owner:             identityAccessTeam,
			HideFromDocs:      true,
			AllowSelfServe:    true,
			HideFromAdminPage: true,
			RequiresRestart:   true,
		},
		{
			Name:           "alertingQueryOptimization",
			Description:    "Optimizes eligible queries in order to reduce load on datasources",
			Stage:          FeatureStageGeneralAvailability,
			Owner:          grafanaAlertingSquad,
			AllowSelfServe: false,
			Expression:     "false",
		},
		{
			Name:         "newFolderPicker",
			Description:  "Enables the nested folder picker without having nested folders enabled",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaFrontendPlatformSquad,
			FrontendOnly: true,
		},
		{
			Name:              "jitterAlertRulesWithinGroups",
			Description:       "Distributes alert rule evaluations more evenly over time, including spreading out rules within the same group",
			FrontendOnly:      false,
			Stage:             FeatureStagePublicPreview,
			Owner:             grafanaAlertingSquad,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: false,
			RequiresRestart:   true,
		},
		{
			Name:        "onPremToCloudMigrations",
			Description: "Enable the Grafana Migration Assistant, which helps you easily migrate on-prem dashboards, folders, and data source configurations to your Grafana Cloud stack.",
			Stage:       FeatureStagePublicPreview,
			Owner:       grafanaOperatorExperienceSquad,
			Expression:  "false",
		},
		{
			Name:        "onPremToCloudMigrationsAlerts",
			Description: "Enables the migration of alerts and its child resources to your Grafana Cloud stack. Requires `onPremToCloudMigrations` to be enabled in conjunction.",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaOperatorExperienceSquad,
		},
		{
			Name:         "alertingSaveStatePeriodic",
			Description:  "Writes the state periodically to the database, asynchronous to rule evaluation",
			Stage:        FeatureStagePrivatePreview,
			FrontendOnly: false,
			Owner:        grafanaAlertingSquad,
		},
		{
			Name:        "promQLScope",
			Description: "In-development feature that will allow injection of labels into prometheus queries.",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaObservabilityMetricsSquad,
			Expression:  "true",
		},
		{
			Name:         "sqlExpressions",
			Description:  "Enables using SQL and DuckDB functions as Expressions.",
			Stage:        FeatureStageExperimental,
			FrontendOnly: false,
			Owner:        grafanaAppPlatformSquad,
		},
		{
			Name:         "nodeGraphDotLayout",
			Description:  "Changed the layout algorithm for the node graph",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaObservabilityTracesAndProfilingSquad,
		},
		{
			Name:         "groupToNestedTableTransformation",
			Description:  "Enables the group to nested table transformation",
			Stage:        FeatureStageGeneralAvailability,
			FrontendOnly: true,
			Owner:        grafanaDatavizSquad,
			Expression:   "true", // enabled by default,
		},
		{
			Name:        "newPDFRendering",
			Description: "New implementation for the dashboard-to-PDF rendering",
			Stage:       FeatureStagePublicPreview,
			Owner:       grafanaSharingSquad,
		},
		{
			Name:           "tlsMemcached",
			Description:    "Use TLS-enabled memcached in the enterprise caching feature",
			Stage:          FeatureStageGeneralAvailability,
			Owner:          grafanaOperatorExperienceSquad,
			Expression:     "true",
			AllowSelfServe: false, // the non-tls implementation is slated for removal
		},
		{
			Name:            "kubernetesAggregator",
			Description:     "Enable grafana's embedded kube-aggregator",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true,
		},
		{
			Name:            "expressionParser",
			Description:     "Enable new expression parser",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true,
		},
		{
			Name:              "groupByVariable",
			Description:       "Enable groupBy variable support in scenes dashboards",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaDashboardsSquad,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "authAPIAccessTokenAuth",
			Description:       "Enables the use of Auth API access tokens for authentication",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "scopeFilters",
			Description:       "Enables the use of scope filters in Grafana",
			FrontendOnly:      false,
			Stage:             FeatureStageExperimental,
			Owner:             grafanaDashboardsSquad,
			RequiresRestart:   false,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:           "ssoSettingsSAML",
			Description:    "Use the new SSO Settings API to configure the SAML connector",
			Stage:          FeatureStagePublicPreview,
			Owner:          identityAccessTeam,
			AllowSelfServe: true,
		},
		{
			Name:              "oauthRequireSubClaim",
			Description:       "Require that sub claims is present in oauth tokens.",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "newDashboardWithFiltersAndGroupBy",
			Description:       "Enables filters and group by variables on all new dashboards. Variables are added only if default data source supports filtering.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaDashboardsSquad,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:           "cloudWatchNewLabelParsing",
			Description:    "Updates CloudWatch label parsing to be more accurate",
			Stage:          FeatureStageGeneralAvailability,
			Expression:     "true", // enabled by default
			Owner:          awsDatasourcesSquad,
			FrontendOnly:   false,
			AllowSelfServe: false,
		},
		{
			Name:        "accessActionSets",
			Description: "Introduces action sets for resource permissions. Also ensures that all folder editors and admins can create subfolders without needing any additional permissions.",
			Stage:       FeatureStagePublicPreview,
			Owner:       identityAccessTeam,
		},
		{
			Name:            "disableNumericMetricsSortingInExpressions",
			Description:     "In server-side expressions, disable the sorting of numeric-kind metrics by their metric name or labels.",
			Stage:           FeatureStageExperimental,
			FrontendOnly:    false,
			Owner:           grafanaObservabilityMetricsSquad,
			RequiresRestart: true,
		},
		{
			Name:              "grafanaManagedRecordingRules",
			Description:       "Enables Grafana-managed recording rules.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:           "queryLibrary",
			Description:    "Enables Query Library feature in Explore",
			Stage:          FeatureStageExperimental,
			Owner:          grafanaExploreSquad,
			FrontendOnly:   false,
			AllowSelfServe: false,
		},
		{
			Name:         "logsExploreTableDefaultVisualization",
			Description:  "Sets the logs table as default visualisation in logs explore",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaObservabilityLogsSquad,
			FrontendOnly: true,
		},
		{
			Name:         "newDashboardSharingComponent",
			Description:  "Enables the new sharing drawer design",
			Stage:        FeatureStageGeneralAvailability,
			Owner:        grafanaSharingSquad,
			FrontendOnly: true,
			Expression:   "false", // disabled by default
		},
		{
			Name:         "alertingListViewV2",
			Description:  "Enables the new alert list view design",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaAlertingSquad,
			FrontendOnly: true,
		},
		{
			Name:         "notificationBanner",
			Description:  "Enables the notification banner UI and API",
			Stage:        FeatureStageGeneralAvailability,
			Owner:        grafanaFrontendPlatformSquad,
			FrontendOnly: false,
			Expression:   "true",
		},
		{
			Name:              "dashboardRestore",
			Description:       "Enables deleted dashboard restore feature (backend only)",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromAdminPage: true,
		},
		{
			Name:         "datasourceProxyDisableRBAC",
			Description:  "Disables applying a plugin route's ReqAction field to authorization",
			Stage:        FeatureStageGeneralAvailability,
			Owner:        identityAccessTeam,
			HideFromDocs: true,
			Expression:   "false",
		},
		{
			Name:              "alertingDisableSendAlertsExternal",
			Description:       "Disables the ability to send alerts to an external Alertmanager datasource.",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaAlertingSquad,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "preserveDashboardStateWhenNavigating",
			Description:       "Enables possibility to preserve dashboard variables and time range when navigating between dashboards",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaDashboardsSquad,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:         "alertingCentralAlertHistory",
			Description:  "Enables the new central alert history.",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaAlertingSquad,
			FrontendOnly: true,
		},
		{
			Name:        "pluginProxyPreserveTrailingSlash",
			Description: "Preserve plugin proxy trailing slash.",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaPluginsPlatformSquad,
			Expression:  "false", // disabled by default
		},
		{
			Name:        "azureMonitorPrometheusExemplars",
			Description: "Allows configuration of Azure Monitor as a data source that can provide Prometheus exemplars",
			Stage:       FeatureStagePublicPreview,
			Owner:       grafanaPartnerPluginsSquad,
		},
		{
			Name:        "pinNavItems",
			Description: "Enables pinning of nav items",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaFrontendPlatformSquad,
			Expression:  "true", // enabled by default
		},
		{
			Name:              "authZGRPCServer",
			Description:       "Enables the gRPC server for authorization",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromAdminPage: true,
			HideFromDocs:      true,
		},
		{
			Name:        "openSearchBackendFlowEnabled",
			Description: "Enables the backend query flow for Open Search datasource plugin",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       awsDatasourcesSquad,
			Expression:  "true",
		},
		{
			Name:            "ssoSettingsLDAP",
			Description:     "Use the new SSO Settings API to configure LDAP",
			Stage:           FeatureStagePublicPreview,
			Owner:           identityAccessTeam,
			AllowSelfServe:  true,
			RequiresRestart: true,
		},
		{
			Name:        "failWrongDSUID",
			Description: "Throws an error if a datasource has an invalid UIDs",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaPluginsPlatformSquad,
		},
		{
			Name:              "zanzana",
			Description:       "Use openFGA as authorization engine.",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:              "passScopeToDashboardApi",
			Description:       "Enables the passing of scopes to dashboards fetching in Grafana",
			FrontendOnly:      false,
			Stage:             FeatureStageExperimental,
			Owner:             grafanaDashboardsSquad,
			RequiresRestart:   false,
			AllowSelfServe:    false,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:            "alertingApiServer",
			Description:     "Register Alerting APIs with the K8s API server",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAlertingSquad,
			RequiresRestart: true,
		},
		{
			Name:        "dashboardRestoreUI",
			Description: "Enables the frontend to be able to restore a recently deleted dashboard",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaFrontendPlatformSquad,
			Expression:  "false", // enabled by default
		},
		{
			Name:        "cloudWatchRoundUpEndTime",
			Description: "Round up end time for metric queries to the next minute to avoid missing data",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       awsDatasourcesSquad,
			Expression:  "true",
		},
		{
			Name:         "cloudwatchMetricInsightsCrossAccount",
			Description:  "Enables cross account observability for Cloudwatch Metric Insights query builder",
			Stage:        FeatureStagePublicPreview,
			Owner:        awsDatasourcesSquad,
			FrontendOnly: true,
		},
		{
			Name:        "prometheusAzureOverrideAudience",
			Description: "Deprecated. Allow override default AAD audience for Azure Prometheus endpoint. Enabled by default. This feature should no longer be used and will be removed in the future.",
			Stage:       FeatureStageDeprecated,
			Owner:       grafanaPartnerPluginsSquad,
			Expression:  "true", // Enabled by default for now
		}, {
			Name:         "alertingFilterV2",
			Description:  "Enable the new alerting search experience",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaAlertingSquad,
			HideFromDocs: true,
		},
		{
			Name:            "dataplaneAggregator",
			Description:     "Enable grafana dataplane aggregator",
			Stage:           FeatureStageExperimental,
			Owner:           grafanaAppPlatformSquad,
			RequiresRestart: true,
		},
		{
			Name:        "newFiltersUI",
			Description: "Enables new combobox style UI for the Ad hoc filters variable in scenes architecture",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaDashboardsSquad,
		},
		{
			Name:        "lokiSendDashboardPanelNames",
			Description: "Send dashboard and panel names to Loki when querying",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaObservabilityLogsSquad,
		},
		{
			Name:         "alertingPrometheusRulesPrimary",
			Description:  "Uses Prometheus rules as the primary source of truth for ruler-enabled data sources",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaAlertingSquad,
			FrontendOnly: true,
		},
		{
			Name:         "singleTopNav",
			Description:  "Unifies the top search bar and breadcrumb bar into one",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaFrontendPlatformSquad,
		},
		{
			Name:         "exploreLogsShardSplitting",
			Description:  "Used in Explore Logs to split queries into multiple queries based on the number of shards",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:         "exploreLogsAggregatedMetrics",
			Description:  "Used in Explore Logs to query by aggregated metrics",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:         "exploreLogsLimitedTimeRange",
			Description:  "Used in Explore Logs to limit the time range",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        grafanaObservabilityLogsSquad,
		},
		{
			Name:         "homeSetupGuide",
			Description:  "Used in Home for users who want to return to the onboarding flow or quickly find popular config pages",
			Stage:        FeatureStageExperimental,
			FrontendOnly: true,
			Owner:        growthAndOnboarding,
		},
		{
			Name:              "appPlatformAccessTokens",
			Description:       "Enables the use of access tokens for the App Platform",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:        "appSidecar",
			Description: "Enable the app sidecar feature that allows rendering 2 apps at the same time",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaExploreSquad,
		},
		{
			Name:         "groupAttributeSync",
			Description:  "Enable the groupsync extension for managing Group Attribute Sync feature",
			Stage:        FeatureStageExperimental,
			Owner:        identityAccessTeam,
			HideFromDocs: true,
		},
		{
			Name:         "alertingQueryAndExpressionsStepMode",
			Description:  "Enables step mode for alerting queries and expressions",
			Stage:        FeatureStageExperimental,
			Owner:        grafanaAlertingSquad,
			FrontendOnly: true,
		},
		{
			Name:              "improvedExternalSessionHandling",
			Description:       "Enable improved support for external sessions in Grafana",
			Stage:             FeatureStageExperimental,
			Owner:             identityAccessTeam,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:        "useSessionStorageForRedirection",
			Description: "Use session storage for handling the redirection after login",
			Stage:       FeatureStagePublicPreview,
			Owner:       identityAccessTeam,
		},
		{
			Name:        "rolePickerDrawer",
			Description: "Enables the new role picker drawer design",
			Stage:       FeatureStageExperimental,
			Owner:       identityAccessTeam,
		},
		{
			Name:              "unifiedStorageSearch",
			Description:       "Enable unified storage search",
			Stage:             FeatureStageExperimental,
			Owner:             grafanaSearchAndStorageSquad,
			HideFromDocs:      true,
			HideFromAdminPage: true,
		},
		{
			Name:        "pluginsSriChecks",
			Description: "Enables SRI checks for plugin assets",
			Stage:       FeatureStageExperimental,
			Owner:       grafanaPluginsPlatformSquad,
		},
		{
			Name:        "azureMonitorEnableUserAuth",
			Description: "Enables user auth for Azure Monitor datasource only",
			Stage:       FeatureStageGeneralAvailability,
			Owner:       grafanaPartnerPluginsSquad,
			Expression:  "true", // Enabled by default for now
		},
	}
)

//go:embed toggles_gen.json
var f embed.FS

// Get the cached feature list (exposed as a k8s resource)
func GetEmbeddedFeatureList() (featuretoggle.FeatureList, error) {
	features := featuretoggle.FeatureList{}
	body, err := f.ReadFile("toggles_gen.json")
	if err == nil {
		err = json.Unmarshal(body, &features)
	}
	return features, err
}
