
/*
 * Copyright (C) 2024 NetApp
 */

#include <ngx_shared_metrics.h>

#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>

#define METRICS_SHM "/nginx-shared-connection-metrics"
#define PAGE_BYTES 4096


volatile ngx_conn_metrics_t* ngx_conn_metrics = NULL;

int ngx_init_shared_metrics()
{
    // Open a shared memory descriptor
    int shm_fd = shm_open(METRICS_SHM, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);
    if (shm_fd < 0) {
        ngx_log_error(NGX_LOG_ERR, ngx_cycle->log, ngx_errno, "Failed to open shm for shared conn metrics");
        ngx_conn_metrics = NULL;
        return 0;
    }

    struct stat st;
    if (fstat(shm_fd, &st) < 0) {
        ngx_log_error(NGX_LOG_ERR, ngx_cycle->log, ngx_errno, "Failed to fstat shared conn metrics shm");
        ngx_conn_metrics = NULL;
        return 0;
    }

    const off_t map_size = (sizeof(ngx_conn_metrics_t) / PAGE_BYTES + 1) * PAGE_BYTES;

    // If the shared memory region hasn't already been assigned bytes then do so now
    int initRequired = 0;
    if (st.st_size < map_size) {
        // Perform the allocation
        if (ftruncate(shm_fd, map_size) < 0) {
            ngx_log_error(NGX_LOG_ERR, ngx_cycle->log, ngx_errno, "Failed to allocate memory for shared conn metrics");
            ngx_conn_metrics = NULL;
            return 0;
        }
        initRequired = 1;
    }

    // mmap the file descriptor into memory
    ngx_conn_metrics = (ngx_conn_metrics_t*)mmap(NULL, map_size, PROT_READ | PROT_WRITE, MAP_SHARED, shm_fd, 0);
    if (ngx_conn_metrics == MAP_FAILED || ngx_conn_metrics == NULL) {
        ngx_log_error(NGX_LOG_ERR, ngx_cycle->log, ngx_errno, "Failed to init shm for connection metric tracking");
        ngx_conn_metrics = NULL;
        return 0;
    }

    if (initRequired) {
        ngx_conn_metrics->ngx_zero_req_conn = 0;
        ngx_conn_metrics->ngx_one_req_conn = 0;
        ngx_conn_metrics->ngx_reused_req_conn = 0;
        ngx_conn_metrics->ngx_completed_conn = 0;
        ngx_conn_metrics->ngx_completed_req = 0;
        ngx_conn_metrics->ngx_initiated_http_conn = 0;
        ngx_conn_metrics->ngx_total_idle_upstream_conn = 0;
        ngx_conn_metrics->ngx_new_upstream_conn = 0;
        ngx_conn_metrics->ngx_reused_upstream_conn = 0;
    }
    return 1;
}

void ngx_unlink_shared_metrics()
{
    shm_unlink(METRICS_SHM);
}

int ngx_fetch_sub_shared_metrics(int metricType, ngx_int_t delta)
{
    if (ngx_conn_metrics == NULL || ngx_conn_metrics == MAP_FAILED) {
        return -1;
    }

    ngx_uint_t val = 0;
    switch (metricType) {
        case NGX_ZERO_REQ_CONN:
            val = ngx_atomic_fetch_sub(&ngx_conn_metrics->ngx_zero_req_conn, delta);
            break;
        case NGX_ONE_REQ_CONN:
            val = ngx_atomic_fetch_sub(&ngx_conn_metrics->ngx_one_req_conn, delta);
            break;
        case NGX_REUSED_CONN:
            val = ngx_atomic_fetch_sub(&ngx_conn_metrics->ngx_reused_req_conn, delta);
            break;
        case NGX_COMPLETED_CONN:
            val = ngx_atomic_fetch_sub(&ngx_conn_metrics->ngx_completed_conn, delta);
            break;
        case NGX_COMPLETED_REQ:
            val = ngx_atomic_fetch_sub(&ngx_conn_metrics->ngx_completed_req, delta);
            break;
        case NGX_INIT_HTTP_CONN:
            val = ngx_atomic_fetch_sub(&ngx_conn_metrics->ngx_initiated_http_conn, delta);
            break;
        case NGX_TOTAL_IDLE_UPSTREAM_CONN:
            val = ngx_atomic_fetch_sub(&ngx_conn_metrics->ngx_total_idle_upstream_conn, delta);
            break;
        case NGX_NEW_UPSTREAM_CONN:
            val = ngx_atomic_fetch_sub(&ngx_conn_metrics->ngx_new_upstream_conn, delta);
            break;
        case NGX_REUSED_UPSTREAM_CONN:
            val = ngx_atomic_fetch_sub(&ngx_conn_metrics->ngx_reused_upstream_conn, delta);
            break;
        default:
            break;
    }
    return val;
}

int ngx_fetch_add_shared_metrics(int metricType, ngx_int_t delta)
{
    if (ngx_conn_metrics == NULL || ngx_conn_metrics == MAP_FAILED) {
        return -1;
    }

    ngx_uint_t val = 0;
    switch (metricType) {
        case NGX_ZERO_REQ_CONN:
            val = ngx_atomic_fetch_add(&ngx_conn_metrics->ngx_zero_req_conn, delta);
            break;
        case NGX_ONE_REQ_CONN:
            val = ngx_atomic_fetch_add(&ngx_conn_metrics->ngx_one_req_conn, delta);
            break;
        case NGX_REUSED_CONN:
            val = ngx_atomic_fetch_add(&ngx_conn_metrics->ngx_reused_req_conn, delta);
            break;
        case NGX_COMPLETED_CONN:
            val = ngx_atomic_fetch_add(&ngx_conn_metrics->ngx_completed_conn, delta);
            break;
        case NGX_COMPLETED_REQ:
            val = ngx_atomic_fetch_add(&ngx_conn_metrics->ngx_completed_req, delta);
            break;
        case NGX_INIT_HTTP_CONN:
            val = ngx_atomic_fetch_add(&ngx_conn_metrics->ngx_initiated_http_conn, delta);
            break;
        case NGX_TOTAL_IDLE_UPSTREAM_CONN:
            val = ngx_atomic_fetch_add(&ngx_conn_metrics->ngx_total_idle_upstream_conn, delta);
            break;
        case NGX_NEW_UPSTREAM_CONN:
            val = ngx_atomic_fetch_add(&ngx_conn_metrics->ngx_new_upstream_conn, delta);
            break;
        case NGX_REUSED_UPSTREAM_CONN:
            val = ngx_atomic_fetch_add(&ngx_conn_metrics->ngx_reused_upstream_conn, delta);
            break;
        default:
            break;
    }
    return val;
}

ngx_atomic_uint_t ngx_fetch_shared_metric(int metricType)
{
    if (ngx_conn_metrics == NULL || ngx_conn_metrics == MAP_FAILED) {
        return 0;
    }

    ngx_atomic_uint_t val = 0;
    switch (metricType) {
        case NGX_ZERO_REQ_CONN:
            val = ngx_conn_metrics->ngx_zero_req_conn;
            break;
        case NGX_ONE_REQ_CONN:
            val = ngx_conn_metrics->ngx_one_req_conn;
            break;
        case NGX_REUSED_CONN:
            val = ngx_conn_metrics->ngx_reused_req_conn;
            break;
        case NGX_COMPLETED_CONN:
            val = ngx_conn_metrics->ngx_completed_conn;
            break;
        case NGX_COMPLETED_REQ:
            val = ngx_conn_metrics->ngx_completed_req;
            break;
        case NGX_INIT_HTTP_CONN:
            val = ngx_conn_metrics->ngx_initiated_http_conn;
            break;
        case NGX_TOTAL_IDLE_UPSTREAM_CONN:
            sg_atomic_t signedVal = ngx_conn_metrics->ngx_total_idle_upstream_conn;

            // If a negative value is observed, then out-of-order / in-progress atomic operations are likely to blame.
            // The issue should correct itself, but report 0 in the meantime and issue a debug log.
            if (signedVal < 0) {
                ngx_log_debug1(NGX_LOG_DEBUG_CORE, ngx_cycle->log, 0, "NGX_TOTAL_IDLE_UPSTREAM_CONN is reporting a negative value: %d", signedVal);
                val = 0;
            }
            else {
                val = signedVal;
            }
            break;
        case NGX_NEW_UPSTREAM_CONN:
            val = ngx_conn_metrics->ngx_new_upstream_conn;
            break;
        case NGX_REUSED_UPSTREAM_CONN:
            val = ngx_conn_metrics->ngx_reused_upstream_conn;
            break;
        default:
            break;
    }

    return val;

}
