# Third-party, see oss-summary/src/tools/ftp-posting-tool/README.md
# Copyright (c) 2018 NetApp, Inc.
# Copyright DataStax, Inc.

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

# http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

require 'datastax/cassandra'

module CassandraExtensions
  module Cassandra
    module Promise
      module Signal
        # @param timeout [nil, Numeric] a maximum number of seconds to block
        #   current thread for while waiting for this future to resolve. Will
        #   wait indefinitely if passed `nil`.
        #
        # @raise [ArgumentError] raised when a negative timeout is given
        # @raise [Errors::TimeoutError] raised when wait time exceeds the timeout
        # @raise [Exception] raises when the future has been resolved with an
        #   error. The original exception will be raised.
        #
        # @return [Object] the value that the future has been resolved with
        def get(timeout = nil)
          timeout &&= Float(timeout)

          if timeout
            raise ::ArgumentError.new("timeout cannot be negative, #{timeout.inspect} given") if timeout < 0
            start    = ::Time.now
            now      = start
            deadline = start + timeout
          end

          if @state == :pending
            synchronize do
              if @state == :pending
                @waiting += 1
                while @state == :pending
                  if deadline
                    break if now >= deadline
                    @cond.wait(deadline - now)
                    now = ::Time.now
                  else
                    @cond.wait
                  end
                end
                @waiting -= 1
              end
            end

            if @state == :pending
              total_wait = deadline - start
              raise Errors::TimeoutError.new("Future did not complete within #{timeout.inspect} seconds. " \
                                             "Wait time: #{total_wait.inspect}")
            end
          end

          raise(@error, @error.message, @error.backtrace) if @state == :broken

          @value
        end
        alias_method :join, :get
      end
    end
  end
end

class DataStax::Cassandra::Promise::Signal
  prepend(CassandraExtensions::Cassandra::Promise::Signal)
end
