/* QLogic (R)NIC Driver/Library
 * Copyright (c) 2010-2017  Cavium, Inc.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#ifndef _QED_DBG_HSI_H
#define _QED_DBG_HSI_H
#include <linux/types.h>
#include <linux/bitops.h>
#include <linux/bug.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/list.h>
#include <linux/slab.h>
#include "qed_hsi.h"
/****************************************/
/* Debug Tools HSI constants and macros */
/****************************************/

/* HW block IDs (includes physical blocks and debug bus blocks) */
enum block_id {
	BLOCK_GRC,
	BLOCK_MISCS,
	BLOCK_MISC,
	BLOCK_DBU,
	BLOCK_PGLUE_B,
	BLOCK_CNIG,
	BLOCK_CPMU,
	BLOCK_NCSI,
	BLOCK_OPTE,
	BLOCK_BMB,
	BLOCK_PCIE,
	BLOCK_MCP,
	BLOCK_MCP2,
	BLOCK_PSWHST,
	BLOCK_PSWHST2,
	BLOCK_PSWRD,
	BLOCK_PSWRD2,
	BLOCK_PSWWR,
	BLOCK_PSWWR2,
	BLOCK_PSWRQ,
	BLOCK_PSWRQ2,
	BLOCK_PGLCS,
	BLOCK_DMAE,
	BLOCK_PTU,
	BLOCK_TCM,
	BLOCK_MCM,
	BLOCK_UCM,
	BLOCK_XCM,
	BLOCK_YCM,
	BLOCK_PCM,
	BLOCK_QM,
	BLOCK_TM,
	BLOCK_DORQ,
	BLOCK_BRB,
	BLOCK_SRC,
	BLOCK_PRS,
	BLOCK_TSDM,
	BLOCK_MSDM,
	BLOCK_USDM,
	BLOCK_XSDM,
	BLOCK_YSDM,
	BLOCK_PSDM,
	BLOCK_TSEM,
	BLOCK_MSEM,
	BLOCK_USEM,
	BLOCK_XSEM,
	BLOCK_YSEM,
	BLOCK_PSEM,
	BLOCK_RSS,
	BLOCK_TMLD,
	BLOCK_MULD,
	BLOCK_YULD,
	BLOCK_XYLD,
	BLOCK_PRM,
	BLOCK_PBF_PB1,
	BLOCK_PBF_PB2,
	BLOCK_RPB,
	BLOCK_BTB,
	BLOCK_PBF,
	BLOCK_RDIF,
	BLOCK_TDIF,
	BLOCK_CDU,
	BLOCK_CCFC,
	BLOCK_TCFC,
	BLOCK_IGU,
	BLOCK_CAU,
	BLOCK_UMAC,
	BLOCK_XMAC,
	BLOCK_MSTAT,
	BLOCK_DBG,
	BLOCK_NIG,
	BLOCK_WOL,
	BLOCK_BMBN,
	BLOCK_IPC,
	BLOCK_NWM,
	BLOCK_NWS,
	BLOCK_MS,
	BLOCK_PHY_PCIE,
	BLOCK_LED,
	BLOCK_AVS_WRAP,
	BLOCK_PXPREQBUS,
	BLOCK_BAR0_MAP,
	BLOCK_MCP_FIO,
	BLOCK_LAST_INIT,
	BLOCK_PRS_FC,
	BLOCK_PBF_FC,
	BLOCK_NIG_LB_FC,
	BLOCK_NIG_LB_FC_PLLH,
	BLOCK_NIG_TX_FC_PLLH,
	BLOCK_NIG_TX_FC,
	BLOCK_NIG_RX_FC_PLLH,
	BLOCK_NIG_RX_FC,
	MAX_BLOCK_ID
};

/* binary debug buffer types */
enum bin_dbg_buffer_type {
	BIN_BUF_DBG_MODE_TREE,	/* HSI_COMMENT: init modes tree */
	BIN_BUF_DBG_DUMP_REG,	/* HSI_COMMENT: GRC Dump registers */
	BIN_BUF_DBG_DUMP_MEM,	/* HSI_COMMENT: GRC Dump memories */
	BIN_BUF_DBG_IDLE_CHK_REGS,	/* HSI_COMMENT: Idle Check registers */
	BIN_BUF_DBG_IDLE_CHK_IMMS,	/* HSI_COMMENT: Idle Check immediates */
	BIN_BUF_DBG_IDLE_CHK_RULES,	/* HSI_COMMENT: Idle Check rules */
	BIN_BUF_DBG_IDLE_CHK_PARSING_DATA,	/* HSI_COMMENT: Idle Check parsing data */
	BIN_BUF_DBG_ATTN_BLOCKS,	/* HSI_COMMENT: Attention blocks */
	BIN_BUF_DBG_ATTN_REGS,	/* HSI_COMMENT: Attention registers */
	BIN_BUF_DBG_ATTN_INDEXES,	/* HSI_COMMENT: Attention indexes */
	BIN_BUF_DBG_ATTN_NAME_OFFSETS,	/* HSI_COMMENT: Attention name offsets */
	BIN_BUF_DBG_BLOCKS,	/* HSI_COMMENT: Blocks debug data */
	BIN_BUF_DBG_BLOCKS_CHIP_DATA,	/* HSI_COMMENT: Blocks debug chip data */
	BIN_BUF_DBG_BUS_LINES,	/* HSI_COMMENT: Blocks debug bus lines */
	BIN_BUF_DBG_BLOCKS_USER_DATA,	/* HSI_COMMENT: Blocks debug user data */
	BIN_BUF_DBG_BLOCKS_CHIP_USER_DATA,	/* HSI_COMMENT: Blocks debug chip user data */
	BIN_BUF_DBG_BUS_LINE_NAME_OFFSETS,	/* HSI_COMMENT: Debug Bus line name offsets */
	BIN_BUF_DBG_RESET_REGS,	/* HSI_COMMENT: Reset registers */
	BIN_BUF_DBG_PARSING_STRINGS,	/* HSI_COMMENT: Debug Tools parsing strings */
	MAX_BIN_DBG_BUFFER_TYPE
};

/* Attention bit mapping */
struct dbg_attn_bit_mapping {
	u16 data;
#define DBG_ATTN_BIT_MAPPING_VAL_MASK			0x7FFF	/* HSI_COMMENT: The index of an attention in the blocks attentions list (if is_unused_bit_cnt=0), or a number of consecutive unused attention bits (if is_unused_bit_cnt=1) */
#define DBG_ATTN_BIT_MAPPING_VAL_SHIFT			0
#define DBG_ATTN_BIT_MAPPING_IS_UNUSED_BIT_CNT_MASK	0x1	/* HSI_COMMENT: if set, the val field indicates the number of consecutive unused attention bits */
#define DBG_ATTN_BIT_MAPPING_IS_UNUSED_BIT_CNT_SHIFT	15
};

/* Attention block per-type data */
struct dbg_attn_block_type_data {
	u16 names_offset;	/* HSI_COMMENT: Offset of this block attention names in the debug attention name offsets array */
	u16 reserved1;
	u8 num_regs;		/* HSI_COMMENT: Number of attention registers in this block */
	u8 reserved2;
	u16 regs_offset;	/* HSI_COMMENT: Offset of this blocks attention registers in the attention registers array (in dbg_attn_reg units) */
};

/* Block attentions */
struct dbg_attn_block {
	struct dbg_attn_block_type_data per_type_data[2];	/* HSI_COMMENT: attention block per-type data. Count must match the number of elements in dbg_attn_type. */
};

/* Attention register result */
struct dbg_attn_reg_result {
	u32 data;
#define DBG_ATTN_REG_RESULT_STS_ADDRESS_MASK		0xFFFFFF	/* HSI_COMMENT: STS attention register GRC address (in dwords) */
#define DBG_ATTN_REG_RESULT_STS_ADDRESS_SHIFT		0
#define DBG_ATTN_REG_RESULT_NUM_REG_ATTN_MASK		0xFF	/* HSI_COMMENT: Number of attention indexes in this register */
#define DBG_ATTN_REG_RESULT_NUM_REG_ATTN_SHIFT		24
	u16 block_attn_offset;	/* HSI_COMMENT: The offset of this registers attentions within the blocks attentions list (a value in the range 0..number of block attentions-1) */
	u16 reserved;
	u32 sts_val;		/* HSI_COMMENT: Value read from the STS attention register */
	u32 mask_val;		/* HSI_COMMENT: Value read from the MASK attention register */
};

/* Attention block result */
struct dbg_attn_block_result {
	u8 block_id;		/* HSI_COMMENT: Registers block ID */
	u8 data;
#define DBG_ATTN_BLOCK_RESULT_ATTN_TYPE_MASK		0x3	/* HSI_COMMENT: Value from dbg_attn_type enum */
#define DBG_ATTN_BLOCK_RESULT_ATTN_TYPE_SHIFT		0
#define DBG_ATTN_BLOCK_RESULT_NUM_REGS_MASK		0x3F	/* HSI_COMMENT: Number of registers in the block in which at least one attention bit is set */
#define DBG_ATTN_BLOCK_RESULT_NUM_REGS_SHIFT		2
	u16 names_offset;	/* HSI_COMMENT: Offset of this registers block attention names in the attention name offsets array */
	struct dbg_attn_reg_result reg_results[15];	/* HSI_COMMENT: result data for each register in the block in which at least one attention bit is set */
};

/* mode header */
struct dbg_mode_hdr {
	u16 data;
#define DBG_MODE_HDR_EVAL_MODE_MASK		0x1	/* HSI_COMMENT: indicates if a mode expression should be evaluated (0/1) */
#define DBG_MODE_HDR_EVAL_MODE_SHIFT		0
#define DBG_MODE_HDR_MODES_BUF_OFFSET_MASK	0x7FFF	/* HSI_COMMENT: offset (in bytes) in modes expression buffer. valid only if eval_mode is set. */
#define DBG_MODE_HDR_MODES_BUF_OFFSET_SHIFT	1
};

/* Attention register */
struct dbg_attn_reg {
	struct dbg_mode_hdr mode;	/* HSI_COMMENT: Mode header */
	u16 block_attn_offset;	/* HSI_COMMENT: The offset of this registers attentions within the blocks attentions list (a value in the range 0..number of block attentions-1) */
	u32 data;
#define DBG_ATTN_REG_STS_ADDRESS_MASK		0xFFFFFF	/* HSI_COMMENT: STS attention register GRC address (in dwords) */
#define DBG_ATTN_REG_STS_ADDRESS_SHIFT		0
#define DBG_ATTN_REG_NUM_REG_ATTN_MASK		0xFF	/* HSI_COMMENT: Number of attention in this register */
#define DBG_ATTN_REG_NUM_REG_ATTN_SHIFT		24
	u32 sts_clr_address;	/* HSI_COMMENT: STS_CLR attention register GRC address (in dwords) */
	u32 mask_address;	/* HSI_COMMENT: MASK attention register GRC address (in dwords) */
};

/* attention types */
enum dbg_attn_type {
	ATTN_TYPE_INTERRUPT,
	ATTN_TYPE_PARITY,
	MAX_DBG_ATTN_TYPE
};

/* Block debug data */
struct dbg_block {
	u8 name[15];		/* HSI_COMMENT: Block name */
	u8 associated_storm_letter;	/* HSI_COMMENT: The letter (char) of the associated Storm, or 0 if no associated Storm. */
};

/* Chip-specific block debug data */
struct dbg_block_chip {
	u8 flags;
#define DBG_BLOCK_CHIP_IS_REMOVED_MASK			0x1	/* HSI_COMMENT: Indicates if the block is removed in this chip (0/1). */
#define DBG_BLOCK_CHIP_IS_REMOVED_SHIFT			0
#define DBG_BLOCK_CHIP_HAS_RESET_REG_MASK		0x1	/* HSI_COMMENT: Indicates if this block has a reset register (0/1). */
#define DBG_BLOCK_CHIP_HAS_RESET_REG_SHIFT		1
#define DBG_BLOCK_CHIP_UNRESET_BEFORE_DUMP_MASK		0x1	/* HSI_COMMENT: Indicates if this block should be taken out of reset before GRC Dump (0/1). Valid only if has_reset_reg is set. */
#define DBG_BLOCK_CHIP_UNRESET_BEFORE_DUMP_SHIFT	2
#define DBG_BLOCK_CHIP_HAS_DBG_BUS_MASK			0x1	/* HSI_COMMENT: Indicates if this block has a debug bus (0/1). */
#define DBG_BLOCK_CHIP_HAS_DBG_BUS_SHIFT		3
#define DBG_BLOCK_CHIP_HAS_LATENCY_EVENTS_MASK		0x1	/* HSI_COMMENT: Indicates if this block has a latency events debug line (0/1). Valid only if has_dbg_bus is set. */
#define DBG_BLOCK_CHIP_HAS_LATENCY_EVENTS_SHIFT		4
#define DBG_BLOCK_CHIP_RESERVED0_MASK			0x7
#define DBG_BLOCK_CHIP_RESERVED0_SHIFT			5
	u8 dbg_client_id;	/* HSI_COMMENT: The DBG block client ID of this block/chip. Valid only if has_dbg_bus is set. */
	u8 reset_reg_id;	/* HSI_COMMENT: The ID of the reset register of this block/chip in the dbg_reset_reg array. */
	u8 reset_reg_bit_offset;	/* HSI_COMMENT: The bit offset of this block/chip in the reset register. Valid only if has_reset_reg is set. */
	struct dbg_mode_hdr dbg_bus_mode;	/* HSI_COMMENT: Mode header */
	u16 reserved1;
	u8 reserved2;
	u8 num_of_dbg_bus_lines;	/* HSI_COMMENT: Number of Debug Bus lines in this block/chip (excluding signature and latency events). Valid only if has_dbg_bus is set. */
	u16 dbg_bus_lines_offset;	/* HSI_COMMENT: Offset of this block/chip Debug Bus lines in the Debug Bus lines array. Valid only if has_dbg_bus is set. */
	u32 dbg_select_reg_addr;	/* HSI_COMMENT: GRC address of the Debug Bus dbg_select register (in dwords). Valid only if has_dbg_bus is set. */
	u32 dbg_dword_enable_reg_addr;	/* HSI_COMMENT: GRC address of the Debug Bus dbg_dword_enable register (in dwords). Valid only if has_dbg_bus is set. */
	u32 dbg_shift_reg_addr;	/* HSI_COMMENT: GRC address of the Debug Bus dbg_shift register (in dwords). Valid only if has_dbg_bus is set. */
	u32 dbg_force_valid_reg_addr;	/* HSI_COMMENT: GRC address of the Debug Bus dbg_force_valid register (in dwords). Valid only if has_dbg_bus is set. */
	u32 dbg_force_frame_reg_addr;	/* HSI_COMMENT: GRC address of the Debug Bus dbg_force_frame register (in dwords). Valid only if has_dbg_bus is set. */
};

/* Chip-specific block user debug data */
struct dbg_block_chip_user {
	u8 num_of_dbg_bus_lines;	/* HSI_COMMENT: Number of debug bus lines in this block (excluding signature and latency events). */
	u8 has_latency_events;	/* HSI_COMMENT: Indicates if this block has a latency events debug line (0/1). */
	u16 names_offset;	/* HSI_COMMENT: Offset of this blocks lines in the debug bus line name offsets array. */
};

/* Block user debug data */
struct dbg_block_user {
	u8 name[16];		/* HSI_COMMENT: Block name */
};

/* Block Debug line data */
struct dbg_bus_line {
	u8 data;
#define DBG_BUS_LINE_NUM_OF_GROUPS_MASK		0xF	/* HSI_COMMENT: Number of groups in the line (0-3) */
#define DBG_BUS_LINE_NUM_OF_GROUPS_SHIFT	0
#define DBG_BUS_LINE_IS_256B_MASK		0x1	/* HSI_COMMENT: Indicates if this is a 128b line (0) or a 256b line (1). */
#define DBG_BUS_LINE_IS_256B_SHIFT		4
#define DBG_BUS_LINE_RESERVED_MASK		0x7
#define DBG_BUS_LINE_RESERVED_SHIFT		5
	u8 group_sizes;		/* HSI_COMMENT: Four 2-bit values, indicating the size of each group minus 1 (i.e. value=0 means size=1, value=1 means size=2, etc), starting from lsb. The sizes are in dwords (if is_256b=0) or in qwords (if is_256b=1). */
};

/* condition header for registers dump */
struct dbg_dump_cond_hdr {
	struct dbg_mode_hdr mode;	/* HSI_COMMENT: Mode header */
	u8 block_id;		/* HSI_COMMENT: block ID */
	u8 data_size;		/* HSI_COMMENT: size in dwords of the data following this header */
};

/* memory data for registers dump */
struct dbg_dump_mem {
	u32 dword0;
#define DBG_DUMP_MEM_ADDRESS_MASK		0xFFFFFF	/* HSI_COMMENT: register address (in dwords) */
#define DBG_DUMP_MEM_ADDRESS_SHIFT		0
#define DBG_DUMP_MEM_MEM_GROUP_ID_MASK		0xFF	/* HSI_COMMENT: memory group ID */
#define DBG_DUMP_MEM_MEM_GROUP_ID_SHIFT		24
	u32 dword1;
#define DBG_DUMP_MEM_LENGTH_MASK		0xFFFFFF	/* HSI_COMMENT: register size (in dwords) */
#define DBG_DUMP_MEM_LENGTH_SHIFT		0
#define DBG_DUMP_MEM_WIDE_BUS_MASK		0x1	/* HSI_COMMENT: indicates if the register is wide-bus */
#define DBG_DUMP_MEM_WIDE_BUS_SHIFT		24
#define DBG_DUMP_MEM_RESERVED_MASK		0x7F
#define DBG_DUMP_MEM_RESERVED_SHIFT		25
};

/* register data for registers dump */
struct dbg_dump_reg {
	u32 data;
#define DBG_DUMP_REG_ADDRESS_MASK	0x7FFFFF	/* HSI_COMMENT: register address (in dwords) */
#define DBG_DUMP_REG_ADDRESS_SHIFT	0
#define DBG_DUMP_REG_WIDE_BUS_MASK	0x1	/* HSI_COMMENT: indicates if the register is wide-bus */
#define DBG_DUMP_REG_WIDE_BUS_SHIFT	23
#define DBG_DUMP_REG_LENGTH_MASK	0xFF	/* HSI_COMMENT: register size (in dwords) */
#define DBG_DUMP_REG_LENGTH_SHIFT	24
};

/* split header for registers dump */
struct dbg_dump_split_hdr {
	u32 hdr;
#define DBG_DUMP_SPLIT_HDR_DATA_SIZE_MASK		0xFFFFFF	/* HSI_COMMENT: size in dwords of the data following this header */
#define DBG_DUMP_SPLIT_HDR_DATA_SIZE_SHIFT		0
#define DBG_DUMP_SPLIT_HDR_SPLIT_TYPE_ID_MASK		0xFF	/* HSI_COMMENT: split type ID */
#define DBG_DUMP_SPLIT_HDR_SPLIT_TYPE_ID_SHIFT		24
};

/* condition header for idle check */
struct dbg_idle_chk_cond_hdr {
	struct dbg_mode_hdr mode;	/* HSI_COMMENT: Mode header */
	u16 data_size;		/* HSI_COMMENT: size in dwords of the data following this header */
};

/* Idle Check condition register */
struct dbg_idle_chk_cond_reg {
	u32 data;
#define DBG_IDLE_CHK_COND_REG_ADDRESS_MASK	0x7FFFFF	/* HSI_COMMENT: Register GRC address (in dwords) */
#define DBG_IDLE_CHK_COND_REG_ADDRESS_SHIFT	0
#define DBG_IDLE_CHK_COND_REG_WIDE_BUS_MASK	0x1	/* HSI_COMMENT: indicates if the register is wide-bus */
#define DBG_IDLE_CHK_COND_REG_WIDE_BUS_SHIFT	23
#define DBG_IDLE_CHK_COND_REG_BLOCK_ID_MASK	0xFF	/* HSI_COMMENT: value from block_id enum */
#define DBG_IDLE_CHK_COND_REG_BLOCK_ID_SHIFT	24
	u16 num_entries;	/* HSI_COMMENT: number of registers entries to check */
	u8 entry_size;		/* HSI_COMMENT: size of registers entry (in dwords) */
	u8 start_entry;		/* HSI_COMMENT: index of the first entry to check */
};

/* Idle Check info register */
struct dbg_idle_chk_info_reg {
	u32 data;
#define DBG_IDLE_CHK_INFO_REG_ADDRESS_MASK	0x7FFFFF	/* HSI_COMMENT: Register GRC address (in dwords) */
#define DBG_IDLE_CHK_INFO_REG_ADDRESS_SHIFT	0
#define DBG_IDLE_CHK_INFO_REG_WIDE_BUS_MASK	0x1	/* HSI_COMMENT: indicates if the register is wide-bus */
#define DBG_IDLE_CHK_INFO_REG_WIDE_BUS_SHIFT	23
#define DBG_IDLE_CHK_INFO_REG_BLOCK_ID_MASK	0xFF	/* HSI_COMMENT: value from block_id enum */
#define DBG_IDLE_CHK_INFO_REG_BLOCK_ID_SHIFT	24
	u16 size;		/* HSI_COMMENT: register size in dwords */
	struct dbg_mode_hdr mode;	/* HSI_COMMENT: Mode header */
};

/* Idle Check register */
union dbg_idle_chk_reg {
	struct dbg_idle_chk_cond_reg cond_reg;	/* HSI_COMMENT: condition register */
	struct dbg_idle_chk_info_reg info_reg;	/* HSI_COMMENT: info register */
};

/* Idle Check result header */
struct dbg_idle_chk_result_hdr {
	u16 rule_id;		/* HSI_COMMENT: Failing rule index */
	u16 mem_entry_id;	/* HSI_COMMENT: Failing memory entry index */
	u8 num_dumped_cond_regs;	/* HSI_COMMENT: number of dumped condition registers */
	u8 num_dumped_info_regs;	/* HSI_COMMENT: number of dumped condition registers */
	u8 severity;		/* HSI_COMMENT: from dbg_idle_chk_severity_types enum */
	u8 reserved;
};

/* Idle Check result register header */
struct dbg_idle_chk_result_reg_hdr {
	u8 data;
#define DBG_IDLE_CHK_RESULT_REG_HDR_IS_MEM_MASK		0x1	/* HSI_COMMENT: indicates if this register is a memory */
#define DBG_IDLE_CHK_RESULT_REG_HDR_IS_MEM_SHIFT	0
#define DBG_IDLE_CHK_RESULT_REG_HDR_REG_ID_MASK		0x7F	/* HSI_COMMENT: register index within the failing rule */
#define DBG_IDLE_CHK_RESULT_REG_HDR_REG_ID_SHIFT	1
	u8 start_entry;		/* HSI_COMMENT: index of the first checked entry */
	u16 size;		/* HSI_COMMENT: register size in dwords */
};

/* Idle Check rule */
struct dbg_idle_chk_rule {
	u16 rule_id;		/* HSI_COMMENT: Idle Check rule ID */
	u8 severity;		/* HSI_COMMENT: value from dbg_idle_chk_severity_types enum */
	u8 cond_id;		/* HSI_COMMENT: Condition ID */
	u8 num_cond_regs;	/* HSI_COMMENT: number of condition registers */
	u8 num_info_regs;	/* HSI_COMMENT: number of info registers */
	u8 num_imms;		/* HSI_COMMENT: number of immediates in the condition */
	u8 reserved1;
	u16 reg_offset;		/* HSI_COMMENT: offset of this rules registers in the idle check register array (in dbg_idle_chk_reg units) */
	u16 imm_offset;		/* HSI_COMMENT: offset of this rules immediate values in the immediate values array (in dwords) */
};

/* Idle Check rule parsing data */
struct dbg_idle_chk_rule_parsing_data {
	u32 data;
#define DBG_IDLE_CHK_RULE_PARSING_DATA_HAS_FW_MSG_MASK		0x1	/* HSI_COMMENT: indicates if this register has a FW message */
#define DBG_IDLE_CHK_RULE_PARSING_DATA_HAS_FW_MSG_SHIFT		0
#define DBG_IDLE_CHK_RULE_PARSING_DATA_STR_OFFSET_MASK		0x7FFFFFFF	/* HSI_COMMENT: Offset of this rules strings in the debug strings array (in bytes) */
#define DBG_IDLE_CHK_RULE_PARSING_DATA_STR_OFFSET_SHIFT		1
};

/* idle check severity types */
enum dbg_idle_chk_severity_types {
	IDLE_CHK_SEVERITY_ERROR,	/* HSI_COMMENT: idle check failure should cause an error */
	IDLE_CHK_SEVERITY_ERROR_NO_TRAFFIC,	/* HSI_COMMENT: idle check failure should cause an error only if theres no traffic */
	IDLE_CHK_SEVERITY_WARNING,	/* HSI_COMMENT: idle check failure should cause a warning */
	MAX_DBG_IDLE_CHK_SEVERITY_TYPES
};

/* Reset register */
struct dbg_reset_reg {
	u32 data;
#define DBG_RESET_REG_ADDR_MASK			0xFFFFFF	/* HSI_COMMENT: GRC address (in dwords) */
#define DBG_RESET_REG_ADDR_SHIFT		0
#define DBG_RESET_REG_IS_REMOVED_MASK		0x1	/* HSI_COMMENT: indicates if this register is removed (0/1). */
#define DBG_RESET_REG_IS_REMOVED_SHIFT		24
#define DBG_RESET_REG_RESERVED_MASK		0x7F
#define DBG_RESET_REG_RESERVED_SHIFT		25
};

/* Debug Bus block data */
struct dbg_bus_block_data {
	u8 enable_mask;		/* HSI_COMMENT: 4 bit value, bit i set -> dword/qword i is enabled in block. */
	u8 right_shift;		/* HSI_COMMENT: Number of dwords/qwords to cyclically  right the blocks output (0-3). */
	u8 force_valid_mask;	/* HSI_COMMENT: 4 bit value, bit i set -> dword/qword i is forced valid in block. */
	u8 force_frame_mask;	/* HSI_COMMENT: 4 bit value, bit i set -> dword/qword i frame bit is forced in block. */
	u8 dword_mask;		/* HSI_COMMENT: bit i set -> dword i contains this blocks data (after shifting). */
	u8 line_num;		/* HSI_COMMENT: Debug line number to select */
	u8 hw_id;		/* HSI_COMMENT: HW ID associated with the block */
	u8 flags;
#define DBG_BUS_BLOCK_DATA_IS_256B_LINE_MASK		0x1	/* HSI_COMMENT: 0/1. If 1, the debug line is 256b, otherwise its 128b. */
#define DBG_BUS_BLOCK_DATA_IS_256B_LINE_SHIFT		0
#define DBG_BUS_BLOCK_DATA_RESERVED_MASK		0x7F
#define DBG_BUS_BLOCK_DATA_RESERVED_SHIFT		1
};

/* Debug Bus constraint operation types */
enum dbg_bus_constraint_ops {
	DBG_BUS_CONSTRAINT_OP_EQ,	/* HSI_COMMENT: equal */
	DBG_BUS_CONSTRAINT_OP_NE,	/* HSI_COMMENT: not equal */
	DBG_BUS_CONSTRAINT_OP_LT,	/* HSI_COMMENT: less than */
	DBG_BUS_CONSTRAINT_OP_LTC,	/* HSI_COMMENT: less than (cyclic) */
	DBG_BUS_CONSTRAINT_OP_LE,	/* HSI_COMMENT: less than or equal */
	DBG_BUS_CONSTRAINT_OP_LEC,	/* HSI_COMMENT: less than or equal (cyclic) */
	DBG_BUS_CONSTRAINT_OP_GT,	/* HSI_COMMENT: greater than */
	DBG_BUS_CONSTRAINT_OP_GTC,	/* HSI_COMMENT: greater than (cyclic) */
	DBG_BUS_CONSTRAINT_OP_GE,	/* HSI_COMMENT: greater than or equal */
	DBG_BUS_CONSTRAINT_OP_GEC,	/* HSI_COMMENT: greater than or equal (cyclic) */
	MAX_DBG_BUS_CONSTRAINT_OPS
};

/* Debug Bus trigger state data */
struct dbg_bus_trigger_state_data {
	u8 msg_len;		/* HSI_COMMENT: Message length (in cycles) to be used for message-based trigger constraints. If set to 0, message length is based only on frame bit received from HW. */
	u8 constraint_dword_mask;	/* HSI_COMMENT: A bit for each dword in the debug bus cycle, indicating if this dword appears in a trigger constraint (1) or not (0) */
	u8 storm_id;		/* HSI_COMMENT: Storm ID to trigger on. Valid only when triggering on Storm data. (use enum dbg_storms) */
	u8 reserved;
};

/* Debug Bus memory address */
struct dbg_bus_mem_addr {
	u32 lo;
	u32 hi;
};

/* Debug Bus PCI buffer data */
struct dbg_bus_pci_buf_data {
	struct dbg_bus_mem_addr phys_addr;	/* HSI_COMMENT: PCI buffer physical address */
	struct dbg_bus_mem_addr virt_addr;	/* HSI_COMMENT: PCI buffer virtual address */
	u32 size;		/* HSI_COMMENT: PCI buffer size in bytes */
};

/* Debug Bus Storm EID range filter params */
struct dbg_bus_storm_eid_range_params {
	u8 min;			/* HSI_COMMENT: Minimal event ID to filter on */
	u8 max;			/* HSI_COMMENT: Maximal event ID to filter on */
};

/* Debug Bus Storm EID mask filter params */
struct dbg_bus_storm_eid_mask_params {
	u8 val;			/* HSI_COMMENT: Event ID value */
	u8 mask;		/* HSI_COMMENT: Event ID mask. 1s in the mask = dont care bits. */
};

/* Debug Bus Storm EID filter params */
union dbg_bus_storm_eid_params {
	struct dbg_bus_storm_eid_range_params range;	/* HSI_COMMENT: EID range filter params */
	struct dbg_bus_storm_eid_mask_params mask;	/* HSI_COMMENT: EID mask filter params */
};

/* Debug Bus Storm data */
struct dbg_bus_storm_data {
	u8 enabled;		/* HSI_COMMENT: indicates if the Storm is enabled for recording */
	u8 mode;		/* HSI_COMMENT: Storm debug mode, valid only if the Storm is enabled (use enum dbg_bus_storm_modes) */
	u8 hw_id;		/* HSI_COMMENT: HW ID associated with the Storm */
	u8 eid_filter_en;	/* HSI_COMMENT: Indicates if EID filtering is performed (0/1) */
	u8 eid_range_not_mask;	/* HSI_COMMENT: 1 = EID range filter, 0 = EID mask filter. Valid only if eid_filter_en is set,  */
	u8 cid_filter_en;	/* HSI_COMMENT: Indicates if CID filtering is performed (0/1) */
	union dbg_bus_storm_eid_params eid_filter_params;	/* HSI_COMMENT: EID filter params to filter on. Valid only if eid_filter_en is set. */
	u32 cid;		/* HSI_COMMENT: CID to filter on. Valid only if cid_filter_en is set. */
};

/* Debug Bus data */
struct dbg_bus_data {
	u32 app_version;	/* HSI_COMMENT: The tools version number of the application */
	u8 state;		/* HSI_COMMENT: The current debug bus state (use enum dbg_bus_states) */
	u8 mode_256b_en;	/* HSI_COMMENT: Indicates if the 256 bit mode is enabled */
	u8 num_enabled_blocks;	/* HSI_COMMENT: Number of blocks enabled for recording */
	u8 num_enabled_storms;	/* HSI_COMMENT: Number of Storms enabled for recording */
	u8 target;		/* HSI_COMMENT: Output target (use enum dbg_bus_targets) */
	u8 one_shot_en;		/* HSI_COMMENT: Indicates if one-shot mode is enabled (0/1) */
	u8 grc_input_en;	/* HSI_COMMENT: Indicates if GRC recording is enabled (0/1) */
	u8 timestamp_input_en;	/* HSI_COMMENT: Indicates if timestamp recording is enabled (0/1) */
	u8 filter_en;		/* HSI_COMMENT: Indicates if the recording filter is enabled (0/1) */
	u8 adding_filter;	/* HSI_COMMENT: If true, the next added constraint belong to the filter. Otherwise, it belongs to the last added trigger state. Valid only if either filter or triggers are enabled. */
	u8 filter_pre_trigger;	/* HSI_COMMENT: Indicates if the recording filter should be applied before the trigger. Valid only if both filter and trigger are enabled (0/1) */
	u8 filter_post_trigger;	/* HSI_COMMENT: Indicates if the recording filter should be applied after the trigger. Valid only if both filter and trigger are enabled (0/1) */
	u8 trigger_en;		/* HSI_COMMENT: Indicates if the recording trigger is enabled (0/1) */
	u8 filter_constraint_dword_mask;	/* HSI_COMMENT: A bit for each dword in the debug bus cycle, indicating if this dword appears in a filter constraint (1) or not (0) */
	u8 next_trigger_state;	/* HSI_COMMENT: ID of next trigger state to be added */
	u8 next_constraint_id;	/* HSI_COMMENT: ID of next filter/trigger constraint to be added */
	struct dbg_bus_trigger_state_data trigger_states[3];	/* HSI_COMMENT: trigger states data */
	u8 filter_msg_len;	/* HSI_COMMENT: Message length (in cycles) to be used for message-based filter constraints. If set to 0, message length is based only on frame bit received from HW. */
	u8 rcv_from_other_engine;	/* HSI_COMMENT: Indicates if the other engine sends it NW recording to this engine (0/1) */
	u8 blocks_dword_mask;	/* HSI_COMMENT: A bit for each dword in the debug bus cycle, indicating if this dword is recorded (1) or not (0) */
	u8 blocks_dword_overlap;	/* HSI_COMMENT: Indicates if there are dwords in the debug bus cycle which are recorded by more than one block (0/1) */
	u32 hw_id_mask;		/* HSI_COMMENT: The HW IDs of the recorded HW blocks, where bits i*3..i*3+2 contain the HW ID of dword/qword i */
	struct dbg_bus_pci_buf_data pci_buf;	/* HSI_COMMENT: Debug Bus PCI buffer data. Valid only when the target is DBG_BUS_TARGET_ID_PCI. */
	struct dbg_bus_block_data blocks[132];	/* HSI_COMMENT: Debug Bus data for each block */
	struct dbg_bus_storm_data storms[6];	/* HSI_COMMENT: Debug Bus data for each block */
};

/* Debug bus states */
enum dbg_bus_states {
	DBG_BUS_STATE_IDLE,	/* HSI_COMMENT: debug bus idle state (not recording) */
	DBG_BUS_STATE_READY,	/* HSI_COMMENT: debug bus is ready for configuration and recording */
	DBG_BUS_STATE_RECORDING,	/* HSI_COMMENT: debug bus is currently recording */
	DBG_BUS_STATE_STOPPED,	/* HSI_COMMENT: debug bus recording has stopped */
	MAX_DBG_BUS_STATES
};

/* Debug Bus Storm modes */
enum dbg_bus_storm_modes {
	DBG_BUS_STORM_MODE_PRINTF,	/* HSI_COMMENT: store data (fast debug) */
	DBG_BUS_STORM_MODE_PRAM_ADDR,	/* HSI_COMMENT: pram address (fast debug) */
	DBG_BUS_STORM_MODE_DRA_RW,	/* HSI_COMMENT: DRA read/write data (fast debug) */
	DBG_BUS_STORM_MODE_DRA_W,	/* HSI_COMMENT: DRA write data (fast debug) */
	DBG_BUS_STORM_MODE_LD_ST_ADDR,	/* HSI_COMMENT: load/store address (fast debug) */
	DBG_BUS_STORM_MODE_DRA_FSM,	/* HSI_COMMENT: DRA state machines (fast debug) */
	DBG_BUS_STORM_MODE_FAST_DBGMUX,	/* HSI_COMMENT: fast DBGMUX (fast debug - E5 only) */
	DBG_BUS_STORM_MODE_RH,	/* HSI_COMMENT: recording handlers (fast debug) */
	DBG_BUS_STORM_MODE_RH_WITH_STORE,	/* HSI_COMMENT: recording handlers with store messages (fast debug) */
	DBG_BUS_STORM_MODE_FOC,	/* HSI_COMMENT: FOC: FIN + DRA Rd (slow debug - E4 only) */
	DBG_BUS_STORM_MODE_EXT_STORE,	/* HSI_COMMENT: FOC: External Store (slow debug - E4 only) */
	MAX_DBG_BUS_STORM_MODES
};

/* Debug bus target IDs */
enum dbg_bus_targets {
	DBG_BUS_TARGET_ID_INT_BUF,	/* HSI_COMMENT: records debug bus to DBG block internal buffer */
	DBG_BUS_TARGET_ID_NIG,	/* HSI_COMMENT: records debug bus to the NW */
	DBG_BUS_TARGET_ID_PCI,	/* HSI_COMMENT: records debug bus to a PCI buffer */
	MAX_DBG_BUS_TARGETS
};

/* GRC Dump data */
struct dbg_grc_data {
	u8 params_initialized;	/* HSI_COMMENT: Indicates if the GRC parameters were initialized */
	u8 reserved1;
	u16 reserved2;
	u32 param_val[48];	/* HSI_COMMENT: Value of each GRC parameter. Array size must match the enum dbg_grc_params. */
};

/* Debug GRC params */
enum dbg_grc_params {
	DBG_GRC_PARAM_DUMP_TSTORM,	/* HSI_COMMENT: dump Tstorm memories (0/1) */
	DBG_GRC_PARAM_DUMP_MSTORM,	/* HSI_COMMENT: dump Mstorm memories (0/1) */
	DBG_GRC_PARAM_DUMP_USTORM,	/* HSI_COMMENT: dump Ustorm memories (0/1) */
	DBG_GRC_PARAM_DUMP_XSTORM,	/* HSI_COMMENT: dump Xstorm memories (0/1) */
	DBG_GRC_PARAM_DUMP_YSTORM,	/* HSI_COMMENT: dump Ystorm memories (0/1) */
	DBG_GRC_PARAM_DUMP_PSTORM,	/* HSI_COMMENT: dump Pstorm memories (0/1) */
	DBG_GRC_PARAM_DUMP_REGS,	/* HSI_COMMENT: dump non-memory registers (0/1) */
	DBG_GRC_PARAM_DUMP_RAM,	/* HSI_COMMENT: dump Storm internal RAMs (0/1) */
	DBG_GRC_PARAM_DUMP_PBUF,	/* HSI_COMMENT: dump Storm passive buffer (0/1) */
	DBG_GRC_PARAM_DUMP_IOR,	/* HSI_COMMENT: dump Storm IORs (0/1) */
	DBG_GRC_PARAM_DUMP_VFC,	/* HSI_COMMENT: dump VFC memories (0/1) */
	DBG_GRC_PARAM_DUMP_CM_CTX,	/* HSI_COMMENT: dump CM contexts (0/1) */
	DBG_GRC_PARAM_DUMP_PXP,	/* HSI_COMMENT: dump PXP memories (0/1) */
	DBG_GRC_PARAM_DUMP_RSS,	/* HSI_COMMENT: dump RSS memories (0/1) */
	DBG_GRC_PARAM_DUMP_CAU,	/* HSI_COMMENT: dump CAU memories (0/1) */
	DBG_GRC_PARAM_DUMP_QM,	/* HSI_COMMENT: dump QM memories (0/1) */
	DBG_GRC_PARAM_DUMP_MCP,	/* HSI_COMMENT: dump MCP memories (0/1) */
	DBG_GRC_PARAM_DUMP_DORQ,	/* HSI_COMMENT: dump DORQ memories (0/1) */
	DBG_GRC_PARAM_DUMP_CFC,	/* HSI_COMMENT: dump CFC memories (0/1) */
	DBG_GRC_PARAM_DUMP_IGU,	/* HSI_COMMENT: dump IGU memories (0/1) */
	DBG_GRC_PARAM_DUMP_BRB,	/* HSI_COMMENT: dump BRB memories (0/1) */
	DBG_GRC_PARAM_DUMP_BTB,	/* HSI_COMMENT: dump BTB memories (0/1) */
	DBG_GRC_PARAM_DUMP_BMB,	/* HSI_COMMENT: dump BMB memories (0/1) */
	DBG_GRC_PARAM_RESERVD1,	/* HSI_COMMENT: reserved */
	DBG_GRC_PARAM_DUMP_MULD,	/* HSI_COMMENT: dump MULD memories (0/1) */
	DBG_GRC_PARAM_DUMP_PRS,	/* HSI_COMMENT: dump PRS memories (0/1) */
	DBG_GRC_PARAM_DUMP_DMAE,	/* HSI_COMMENT: dump PRS memories (0/1) */
	DBG_GRC_PARAM_DUMP_TM,	/* HSI_COMMENT: dump TM (timers) memories (0/1) */
	DBG_GRC_PARAM_DUMP_SDM,	/* HSI_COMMENT: dump SDM memories (0/1) */
	DBG_GRC_PARAM_DUMP_DIF,	/* HSI_COMMENT: dump DIF memories (0/1) */
	DBG_GRC_PARAM_DUMP_STATIC,	/* HSI_COMMENT: dump static debug data (0/1) */
	DBG_GRC_PARAM_UNSTALL,	/* HSI_COMMENT: un-stall Storms after dump (0/1) */
	DBG_GRC_PARAM_DUMP_SEM,	/* HSI_COMMENT: dump SEM memories (0/1) */
	DBG_GRC_PARAM_MCP_TRACE_META_SIZE,	/* HSI_COMMENT: MCP Trace meta data size in bytes */
	DBG_GRC_PARAM_EXCLUDE_ALL,	/* HSI_COMMENT: preset: exclude all memories from dump (1 only) */
	DBG_GRC_PARAM_CRASH,	/* HSI_COMMENT: preset: include memories for crash dump (1 only) */
	DBG_GRC_PARAM_PARITY_SAFE,	/* HSI_COMMENT: perform dump only if MFW is responding (0/1) */
	DBG_GRC_PARAM_DUMP_CM,	/* HSI_COMMENT: dump CM memories (0/1) */
	DBG_GRC_PARAM_DUMP_PHY,	/* HSI_COMMENT: dump PHY memories (0/1) */
	DBG_GRC_PARAM_NO_MCP,	/* HSI_COMMENT: dont perform MCP commands (0/1) */
	DBG_GRC_PARAM_NO_FW_VER,	/* HSI_COMMENT: dont read FW/MFW version (0/1) */
	DBG_GRC_PARAM_RESERVED3,	/* HSI_COMMENT: reserved */
	DBG_GRC_PARAM_DUMP_MCP_HW_DUMP,	/* HSI_COMMENT: dump MCP HW Dump (0/1) */
	DBG_GRC_PARAM_DUMP_ILT_CDUC,	/* HSI_COMMENT: dump ILT CDUC client (0/1) */
	DBG_GRC_PARAM_DUMP_ILT_CDUT,	/* HSI_COMMENT: dump ILT CDUT client (0/1) */
	DBG_GRC_PARAM_DUMP_CAU_EXT,	/* HSI_COMMENT: dump CAU extended memories (0/1) */
	MAX_DBG_GRC_PARAMS
};

/* Debug status codes */
enum dbg_status {
	DBG_STATUS_OK,
	DBG_STATUS_APP_VERSION_NOT_SET,
	DBG_STATUS_UNSUPPORTED_APP_VERSION,
	DBG_STATUS_DBG_BLOCK_NOT_RESET,
	DBG_STATUS_INVALID_ARGS,
	DBG_STATUS_OUTPUT_ALREADY_SET,
	DBG_STATUS_INVALID_PCI_BUF_SIZE,
	DBG_STATUS_PCI_BUF_ALLOC_FAILED,
	DBG_STATUS_PCI_BUF_NOT_ALLOCATED,
	DBG_STATUS_INVALID_FILTER_TRIGGER_DWORDS,
	DBG_STATUS_NO_MATCHING_FRAMING_MODE,
	DBG_STATUS_VFC_READ_ERROR,
	DBG_STATUS_STORM_ALREADY_ENABLED,
	DBG_STATUS_STORM_NOT_ENABLED,
	DBG_STATUS_BLOCK_ALREADY_ENABLED,
	DBG_STATUS_BLOCK_NOT_ENABLED,
	DBG_STATUS_NO_INPUT_ENABLED,
	DBG_STATUS_NO_FILTER_TRIGGER_256B,
	DBG_STATUS_FILTER_ALREADY_ENABLED,
	DBG_STATUS_TRIGGER_ALREADY_ENABLED,
	DBG_STATUS_TRIGGER_NOT_ENABLED,
	DBG_STATUS_CANT_ADD_CONSTRAINT,
	DBG_STATUS_TOO_MANY_TRIGGER_STATES,
	DBG_STATUS_TOO_MANY_CONSTRAINTS,
	DBG_STATUS_RECORDING_NOT_STARTED,
	DBG_STATUS_DATA_DIDNT_TRIGGER,
	DBG_STATUS_NO_DATA_RECORDED,
	DBG_STATUS_DUMP_BUF_TOO_SMALL,
	DBG_STATUS_DUMP_NOT_CHUNK_ALIGNED,
	DBG_STATUS_UNKNOWN_CHIP,
	DBG_STATUS_VIRT_MEM_ALLOC_FAILED,
	DBG_STATUS_BLOCK_IN_RESET,
	DBG_STATUS_INVALID_TRACE_SIGNATURE,
	DBG_STATUS_INVALID_NVRAM_BUNDLE,
	DBG_STATUS_NVRAM_GET_IMAGE_FAILED,
	DBG_STATUS_NON_ALIGNED_NVRAM_IMAGE,
	DBG_STATUS_NVRAM_READ_FAILED,
	DBG_STATUS_IDLE_CHK_PARSE_FAILED,
	DBG_STATUS_MCP_TRACE_BAD_DATA,
	DBG_STATUS_MCP_TRACE_NO_META,
	DBG_STATUS_MCP_COULD_NOT_HALT,
	DBG_STATUS_MCP_COULD_NOT_RESUME,
	DBG_STATUS_RESERVED0,
	DBG_STATUS_SEMI_FIFO_NOT_EMPTY,
	DBG_STATUS_IGU_FIFO_BAD_DATA,
	DBG_STATUS_MCP_COULD_NOT_MASK_PRTY,
	DBG_STATUS_FW_ASSERTS_PARSE_FAILED,
	DBG_STATUS_REG_FIFO_BAD_DATA,
	DBG_STATUS_PROTECTION_OVERRIDE_BAD_DATA,
	DBG_STATUS_DBG_ARRAY_NOT_SET,
	DBG_STATUS_RESERVED1,
	DBG_STATUS_NON_MATCHING_LINES,
	DBG_STATUS_INSUFFICIENT_HW_IDS,
	DBG_STATUS_DBG_BUS_IN_USE,
	DBG_STATUS_INVALID_STORM_DBG_MODE,
	DBG_STATUS_OTHER_ENGINE_BB_ONLY,
	DBG_STATUS_FILTER_SINGLE_HW_ID,
	DBG_STATUS_TRIGGER_SINGLE_HW_ID,
	DBG_STATUS_MISSING_TRIGGER_STATE_STORM,
	DBG_STATUS_MDUMP2_INVALID_LOG_DATA,
	DBG_STATUS_MDUMP2_INVALID_LOG_SIZE,
	DBG_STATUS_MDUMP2_INVALID_SIGNATURE,
	DBG_STATUS_MDUMP2_INVALID_LOG_HDR,
	DBG_STATUS_MDUMP2_ERROR_READING_LANE_REGS,
	DBG_STATUS_MDUMP2_FAILED_TO_REQUEST_OFFSIZE,
	DBG_STATUS_MDUMP2_FAILED_VALIDATION_OF_DATA_CRC,
	DBG_STATUS_MDUMP2_ERROR_ALLOCATING_BUF,
	DBG_STATUS_MDUMP2_ERROR_EXTRACTING_NUM_PORTS,
	DBG_STATUS_MDUMP2_ERROR_EXTRACTING_MFW_STATUS,
	DBG_STATUS_MDUMP2_ERROR_DISPLAYING_LINKDUMP,
	DBG_STATUS_MDUMP2_ERROR_READING_PHY_CFG,
	DBG_STATUS_MDUMP2_ERROR_READING_PLL_MODE,
	MAX_DBG_STATUS
};

/* Debug Storms IDs */
enum dbg_storms {
	DBG_TSTORM_ID,
	DBG_MSTORM_ID,
	DBG_USTORM_ID,
	DBG_XSTORM_ID,
	DBG_YSTORM_ID,
	DBG_PSTORM_ID,
	MAX_DBG_STORMS
};

/* Idle Check data */
struct idle_chk_data {
	u32 buf_size;		/* HSI_COMMENT: Idle check buffer size in dwords */
	u8 buf_size_set;	/* HSI_COMMENT: Indicates if the idle check buffer size was set (0/1) */
	u8 reserved1;
	u16 reserved2;
};

/* Pretend parameters */
struct pretend_params {
	u8 split_type;		/* HSI_COMMENT: Pretend split type (from enum init_split_types) */
	u8 reserved;
	u16 split_id;		/* HSI_COMMENT: Preted split ID (within the pretend split type) */
};

/* Debug Tools data (per HW function) */
struct dbg_tools_data {
	struct dbg_grc_data grc;	/* HSI_COMMENT: GRC Dump data */
	struct dbg_bus_data bus;	/* HSI_COMMENT: Debug Bus data */
	struct idle_chk_data idle_chk;	/* HSI_COMMENT: Idle Check data */
	u8 mode_enable[40];	/* HSI_COMMENT: Indicates if a mode is enabled (0/1) */
	u8 block_in_reset[132];	/* HSI_COMMENT: Indicates if a block is in reset state (0/1) */
	u8 chip_id;		/* HSI_COMMENT: Chip ID (from enum chip_ids) */
	u8 hw_type;		/* HSI_COMMENT: HW Type */
	u8 num_ports;		/* HSI_COMMENT: Number of ports in the chip */
	u8 num_pfs_per_port;	/* HSI_COMMENT: Number of PFs in each port */
	u8 num_vfs;		/* HSI_COMMENT: Number of VFs in the chip */
	u8 initialized;		/* HSI_COMMENT: Indicates if the data was initialized */
	u8 use_dmae;		/* HSI_COMMENT: Indicates if DMAE should be used */
	u8 reserved;
	struct pretend_params pretend;	/* HSI_COMMENT: Current pretend parameters */
	u32 num_regs_read;	/* HSI_COMMENT: Numbers of registers that were read since last log */
};

/* ILT Clients */
enum ilt_clients {
	ILT_CLI_CDUC,
	ILT_CLI_CDUT,
	ILT_CLI_QM,
	ILT_CLI_TM,
	ILT_CLI_SRC,
	ILT_CLI_TSDM,
	ILT_CLI_RGFS,
	ILT_CLI_TGFS,
	MAX_ILT_CLIENTS
};

/**************************** Public Functions *******************************/

/**
 * @brief qed_dbg_set_bin_ptr - Sets a pointer to the binary data with debug
 * arrays.
 *
 * @param p_hwfn -	    HW device data
 * @param bin_ptr - a pointer to the binary data with debug arrays.
 */
enum dbg_status qed_dbg_set_bin_ptr(struct qed_hwfn *p_hwfn,
				    const u8 * const bin_ptr);

/**
 * @brief qed_dbg_set_app_ver - Sets the version of the calling app.
 *
 * The application should call this function with the TOOLS_VERSION
 * it compiles with. Must be called before all other debug functions.
 *
 * @return error if one of the following holds:
 *	- the specified app version is not supported
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_set_app_ver(u32 ver);

/**
 * @brief qed_dbg_get_fw_func_ver - Returns the FW func version.
 *
 * @return the FW func version.
 */
u32 qed_dbg_get_fw_func_ver(void);

/**
 * @brief qed_dbg_get_chip_id - Returns the FW func version.
 *
 * @param p_hwfn - HW device data
 *
 * @return the chip ID.
 */
enum chip_ids qed_dbg_get_chip_id(struct qed_hwfn *p_hwfn);

/**
 * @brief qed_read_regs - Reads registers into a buffer (using GRC).
 *
 * @param p_hwfn -	HW device data
 * @param p_ptt -	Ptt window used for writing the registers.
 * @param buf -	Destination buffer.
 * @param addr-	Source GRC address in dwords.
 * @param len -	Number of registers to read.
 */
void qed_read_regs(struct qed_hwfn *p_hwfn,
		   struct qed_ptt *p_ptt, u32 * buf, u32 addr, u32 len);

/**
 * @brief qed_read_fw_info - Reads FW info from the chip.
 *
 * The FW info contains FW-related information, such as the FW version,
 * FW image (main/L2B/kuku), FW timestamp, etc.
 * The FW info is read from the internal RAM of the first Storm that is not in
 * reset.
 *
 * @param p_hwfn -	    HW device data
 * @param p_ptt -	    Ptt window used for writing the registers.
 * @param fw_info - Out: a pointer to write the FW info into.
 *
 * @return true if the FW info was read successfuly from one of the Storms,
 * or false if all Storms are in reset.
 */
bool qed_read_fw_info(struct qed_hwfn *p_hwfn,
		      struct qed_ptt *p_ptt, struct fw_info *fw_info);

/**
 * @brief qed_dbg_bus_reset - Resets the Debug block.
 *
 * After reset:
 * - The last recording is erased.
 * - Recording is directed to the internal buffer.
 * - Wrap-around recording is selected.
 * - All HW blocks are disabled.
 * - All Storms are disabled and all SEM filters are cleared.
 *
 * @param p_hwfn -		 HW device data
 * @param p_ptt -		 Ptt window used for writing the registers.
 * @param one_shot_en -  Enable/Disable one-shot recording. If disabled,
 *			 wrap-around recording is used instead.
 * @param hw_dwords -    Obsolete (value is ignored).
 * @param unify_inputs - Obsolete (value is ignored).
 * @param grc_input_en - Enable/Disable recording GRC input. If enabled, the
 *			 GRC input is recorded to the lsb dword of a cycle.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_reset(struct qed_hwfn *p_hwfn,
				  struct qed_ptt *p_ptt,
				  bool one_shot_en,
				  u8 hw_dwords,
				  bool unify_inputs, bool grc_input_en);

/**
 * @brief qed_dbg_bus_set_pci_output - Directs debug output to a PCI buffer.
 *
 * @param p_hwfn -		HW device data
 * @param p_ptt -		Ptt window used for writing the registers.
 * @param buf_size_kb - Size of PCI buffer to allocate (in KB). Must be aligned
 *			to PCI request size.
 *
 * @return error if one of the following holds:
 *	- the Debug block wasn't reset since last recording
 *	- the version wasn't set
 *	- the output was already set
 *	- the PCI buffer size is not aligned to PCI packet size
 *	- the PCI buffer allocation failed
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_set_pci_output(struct qed_hwfn *p_hwfn,
					   struct qed_ptt *p_ptt,
					   u16 buf_size_kb);

/**
 * @brief qed_dbg_bus_set_nw_output - Directs debug output to the network.
 *
 * @param p_hwfn -			HW device data
 * @param p_ptt -			Ptt window used for writing the registers.
 * @param port_id -		Port ID to transmit the debug data on
 * @param dest_addr_lo32 -	Destination MAC address (for Eth header)
 * @param dest_addr_hi16
 * @param data_limit_size_kb -  Data limit size in KB (valid only for one-shot)
 *				If set to 0, data limit won't be configured.
 * @param send_to_other_engine -If true:
 *				1) The NW output will be sent to the DBG block
 *				   of the other engine.
 *				2) port_id argument is ignored.
 *				3) rcv_from_other_engine should be set to false
 *				   The other engine DBG block should call this
 *				   function with rcv_from_other_engine set to
 *				   true.
 * @param rcv_from_other_engine-If true:
 *				1) the DBG block receives the NW output sent
 *				   from the other engine DBG block, and sends
 *				   it to a NW port in the current engine
 *				   (according to port_id).
 *				2) The src/dest addresses and eth_type
 *				   arguments are ignored.
 *				3) send_to_other_engine should be set to false.
 *				   The other engine DBG block should call this
 *				   function with send_to_other_engine set to
 *				   true.
 *
 * @return error if one of the following holds:
 *	- the Debug block wasn't reset since last recording
 *	- the version wasn't set
 *	- the output was already set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_set_nw_output(struct qed_hwfn *p_hwfn,
					  struct qed_ptt *p_ptt,
					  u8 port_id,
					  u32 dest_addr_lo32,
					  u16 dest_addr_hi16,
					  u16
					  data_limit_size_kb,
					  bool
					  send_to_other_engine,
					  bool rcv_from_other_engine);

/**
 * @brief qed_dbg_bus_enable_block - Enables recording of the specified block
 *
 * Each recording cycle contains 4 dwords (for 128-bit debug lines) or 4 qwords
 * (for 256-bit debug lines).
 *
 * @param p_hwfn -		HW device data
 * @param block -	block to be enabled.
 * @param line_num -	debug line number to select.
 * @param cycle_en -	4-bit, if bit i is set, dword/qword i is enabled.
 * @param right_shift -	number of dwords/qwords to cyclically  right the
 *			debug line.
 * @param force_valid - 4-bit, if bit i is set, dword/qword i is forced valid.
 * @param force_frame - 4-bit, if bit i is set, the frame bit of dword/qword i
 *			is forced.
 *
 * @return error if one of the following holds:
 *	- the Debug block wasn't reset since last recording
 *	- the version wasn't set
 *	- block is not valid
 *	- block was already enabled
 *	- cycle_en, force_valid or force_frame are wider than 4 bits
 *	- dword/qword 0 is enabled, but GRC or timestamp were also enabled.
 *	- Too many inputs were enabled.
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_enable_block(struct qed_hwfn *p_hwfn,
					 enum block_id block,
					 u8 line_num,
					 u8 cycle_en,
					 u8 right_shift,
					 u8 force_valid, u8 force_frame);

/**
 * @brief qed_dbg_bus_enable_storm - Enables recording of the specified Storm
 *
 * @param p_hwfn -		HW device data
 * @param storm -	Storm to be enabled.
 * @param storm_mode-	Storm mode
 *
 * @return error if one of the following holds:
 *	- the Debug block wasn't reset since last recording
 *	- the version wasn't set
 *	- the specified storm or mode is invalid
 *	- Storm was already enabled
 *	- only HW data can be recorded
 *	- Too many inputs were enabled.
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_enable_storm(struct qed_hwfn *p_hwfn,
					 enum dbg_storms storm,
					 enum dbg_bus_storm_modes storm_mode);

/**
 * @brief qed_dbg_bus_enable_timestamp - Enables timestamp recording.
 *
 * When enabled, the timestamp input is always recorded to the lsb dword of
 * a cycle, with HW ID 0.
 *
 * @param p_hwfn -	     HW device data
 * @param p_ptt -	     Ptt window used for writing the registers.
 * @param valid_en - 3-bit value. The Timestamp will be recorded in a cycle if
 *		     bit i is set and unit i+1 is valid.
 * @param frame_en - 3-bit value. The Timestamp will be recorded in a cycle if
 *		     bit i is set and unit i+1 has frame bit set.
 * @param tick_len - timestamp tick length in cycles, minus 1. A value of 0
 *		     means one cycle.
 *
 * @return error if one of the following holds:
 *	- the Debug block wasn't reset since last recording
 *	- the version wasn't set
 *	- valid_en or frame_en are wider than 4 bits
 *	- Both timestamp and GRC are enabled.
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_enable_timestamp(struct qed_hwfn *p_hwfn,
					     struct qed_ptt *p_ptt,
					     u8 valid_en,
					     u8 frame_en, u32 tick_len);

/**
 * @brief qed_dbg_bus_add_eid_range_sem_filter- Add Event ID range SEM filter
 *
 * @param p_hwfn -     HW device data
 * @param storm -   Storm to be filtered.
 * @param min_eid - minimal Event ID to filter on.
 * @param max_eid - maximal Event ID to filter on.
 *
 * @return error if one of the following holds:
 *	- the specified Storm is invalid
 *	- the specified Storm wasn't enabled
 *	- the EID range is not valid
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_add_eid_range_sem_filter(struct qed_hwfn *p_hwfn,
						     enum dbg_storms storm,
						     u8 min_eid, u8 max_eid);

/**
 * @brief qed_dbg_bus_add_eid_mask_sem_filter - Add Event ID mask SEM filter
 *
 * @param p_hwfn -      HW device data
 * @param storm -    Storm to be filtered.
 * @param eid_val -  Event ID value.
 * @param eid_mask - Event ID mask. 0's in the mask = don't care bits.
 *
 * @return error if one of the following holds:
 *	- the specified Storm is invalid
 *	- the specified Storm wasn't enabled
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_add_eid_mask_sem_filter(struct qed_hwfn *p_hwfn,
						    enum dbg_storms storm,
						    u8 eid_val, u8 eid_mask);

/**
 * @brief qed_dbg_bus_add_cid_sem_filter - Adds a CID SEM filter.
 *
 * @param p_hwfn -   HW device data
 * @param storm	- Storm to be filtered.
 * @param cid -   CID to filter on.
 *
 * @return error if one of the following holds:
 *	- the specified Storm is invalid
 *	- the specified Storm wasn't enabled
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_add_cid_sem_filter(struct qed_hwfn *p_hwfn,
					       enum dbg_storms storm, u32 cid);

/**
 * @brief qed_dbg_bus_enable_filter - Enables the recording filter.
 *
 * A filter contains up to 4 constraints. The data is "filtered in" when the
 * added constraints hold.
 *
 * @param p_hwfn -		  HW device data
 * @param p_ptt -		  Ptt window used for writing the registers.
 * @param block -	  Obsolete (value is ignored).
 * @param msg_len -	  Message length (in cycles) to be used for message-based
 *                        filter constraints. If set to 0, message length is based
 *                        only on frame bit received from HW.
 *
 * @return error if one of the following holds:
 *	- the Debug block wasn't reset since last recording
 *	- the version wasn't set
 *	- the filter was already enabled
 *	- E4 256 bit recording mode
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_enable_filter(struct qed_hwfn *p_hwfn,
					  struct qed_ptt *p_ptt,
					  enum block_id block, u8 msg_len);

/**
 * @brief qed_dbg_bus_enable_trigger - Enables the recording trigger.
 *
 * A trigger contains up to 3 states, where each state contains up to
 * 4 constraints. After the constraints of a state hold for a specified number
 * of times, the DBG block moves to the next state. If there's no next state,
 * the DBG block triggers.
 *
 * @param p_hwfn -			HW device data
 * @param p_ptt -			Ptt window used for writing the registers.
 * @param rec_pre_trigger -	if true, recording starts before the trigger.
 *				if false, recording starts at the trigger.
 * @param pre_chunks -		max number of chunks to record before the
 *				trigger (1-47). If set to 0, recording starts
 *				from time 0. Ignored if rec_pre_trigger is
 *				false.
 * @param rec_post_trigger -	if true, recording ends after the trigger.
 *				if false, recording ends at the trigger.
 * @param post_cycles -		max number of cycles to record after the
 *				trigger (0x1-0xffffffff). If set to 0,
 *				recording ends only when stopped by the user.
 *				Ignored if rec_post_trigger is false.
 * @param filter_pre_trigger -	if true, data is filtered before the trigger.
 *				Ignored if the filter wasn't enabled.
 * @param filter_post_trigger -	if true, data is filtered after the trigger.
 *				Ignored if the filter wasn't enabled.
 *
 * @return error if one of the following holds:
 *	- the Debug block wasn't reset since last recording
 *	- the version wasn't set
 *	- the trigger was already enabled
 *	- more than 4 dwords are recorded per-cycle (forbids triggers)
 *	- pre_chunks is not in the range 0-47.
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_enable_trigger(struct qed_hwfn *p_hwfn,
					   struct qed_ptt *p_ptt,
					   bool rec_pre_trigger,
					   u8 pre_chunks,
					   bool
					   rec_post_trigger,
					   u32 post_cycles,
					   bool
					   filter_pre_trigger,
					   bool filter_post_trigger);

/**
 * @brief qed_dbg_bus_add_trigger_state - Adds a trigger state.
 *
 * Up to 3 trigger states can be added, where each state contains up to
 * 4 constraints. After the constraints of a state hold for the specified
 * number of times, the DBG block moves to the next state. If there's no next
 * state, the DBG block triggers.
 *
 * @param p_hwfn -		  HW device data
 * @param p_ptt -		  Ptt window used for writing the registers.
 * @param block -	  If recording a Storm, specify the matching SEMI block
 *			  (e.g. BLOCK_XSEM), Otherwise, specify MAX_BLOCK_ID.
 * @param msg_len	- Message length (in cycles) to be used for message-based
 *			  trigger constraints. If set to 0, message length is based
 *			  only on frame bit received from HW.
 * @param count_to_next	- The number of times the constraints of the state
 *			  should hold before moving to the next state. Must be
 *			  non-zero.
 *
 * @return error if one of the following holds:
 *	- The trigger wasn't enabled.
 *	- more than 3 trigger states were added
 *	- block is not a SEMI block and not MAX_BLOCK_ID
 *	- count_to_next is 0
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_add_trigger_state(struct qed_hwfn *p_hwfn,
					      struct qed_ptt *p_ptt,
					      enum block_id block,
					      u8 const_msg_len,
					      u16 count_to_next);

/**
 * @brief qed_dbg_bus_add_constraint - Adds a filter/trigger constraint.
 *
 * The constraint is added to a filter or trigger state, which ever was added
 * last. The filter/trigger happens if both of the following hold:
 * 1. All mandatory constraints are true.
 * 2. At least one optional (non-mandatory) constraints is true.
 *
 * @param p_hwfn -			  HW device data
 * @param p_ptt -			  Ptt window used for writing the registers.
 * @param op -			  constraint operation
 * @param data -		  32-bit data to compare with the recorded
 *				  data.
 * @param data_mask -		  32-bit mask for data comparison. If mask bit
 *				  i is 1, data bit i is compared, otherwise
 *				  it's ignored.
 *				  For eq/ne operations: any mask can be used.
 *				  For other operations: the mask must be
 *				  non-zero, and the 1's in the mask must be
 *				  continuous.
 * @param compare_frame -	  indicates if the frame bit should be
 *				  compared. Must be false for all operations
 *				  other than eq/ne.
 * @param frame_bit -		  frame bit to compare with the recorded data
 *				  (0/1). ignored if compare_frame is false.
 * @param cycle_offset -	  offset in cycles from the beginning of the
 *				  message, where cycle = 4 dwords.
 * @param dword_offset_in_cycle	- offset in dwords from the beginning of the
 *				  cycle (0-3).
 * @param is_mandatory -	  indicates if this constraint is mandatory
 *				  (true) or optional (false). The data is
 *				  filtered-in if all mandatory constraints hold
 *				  AND at least one optional constraint (if
 *				  added) holds.
 *
 * @return error if one of the following holds:
 *	- a filter or trigger state weren't enabled
 *	- all 4 filter constraints were added already
 *	- the op string is invalid
 *	- the data mask is invalid.
 *	- frame bit is not 0/1.
 *	- cycle_offset and dword_offset are not in the range 0-3.
 *	- compare_frame is true and operation is not eq/ne.
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_add_constraint(struct qed_hwfn *p_hwfn,
					   struct qed_ptt *p_ptt,
					   enum dbg_bus_constraint_ops
					   constraint_op,
					   u32 data,
					   u32
					   data_mask,
					   bool
					   compare_frame,
					   u8
					   frame_bit,
					   u8
					   cycle_offset,
					   u8
					   dword_offset_in_cycle,
					   bool is_mandatory);

/**
 * @brief qed_dbg_bus_start - Starts the recording.
 *
 * @param p_hwfn - HW device data
 * @param p_ptt - Ptt window used for writing the registers.
 *
 * @return error if one of the following holds:
 *	- the Debug block wasn't reset since last recording
 *	- the version wasn't set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_start(struct qed_hwfn *p_hwfn,
				  struct qed_ptt *p_ptt);

/**
 * @brief qed_dbg_bus_stop - Stops the recording and flushes the internal
 * buffer.
 *
 * @param p_hwfn - HW device data
 * @param p_ptt - Ptt window used for writing the registers.
 *
 * @return error if a recording is not in progress, ok otherwise.
 */
enum dbg_status qed_dbg_bus_stop(struct qed_hwfn *p_hwfn,
				 struct qed_ptt *p_ptt);

/**
 * @brief qed_dbg_bus_get_dump_buf_size - Returns the required buffer size
 * for Debug Bus recording.
 *
 * @param p_hwfn -      HW device data
 * @param p_ptt -	     Ptt window used for writing the registers.
 * @param buf_size - OUT: the required size (in dwords) of the buffer for
 *		     dumping the recorded Debug Bus data. If recording to the
 *		     internal buffer, the size of the internal buffer is
 *		     returned. If recording to PCI, the size of the PCI buffer
 *		     is returned. Otherwise, 0 is returned.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_get_dump_buf_size(struct qed_hwfn *p_hwfn,
					      struct qed_ptt *p_ptt,
					      u32 * buf_size);

/**
 * @brief qed_dbg_bus_dump - Dumps the recorded Debug Bus data into the
 * specified buffer.
 *
 * The dumped data starts with a header. If recording to NW, only a header is
 * dumped. The dumped size is assigned to num_dumped_dwords.
 *
 * @param p_hwfn -			HW device data
 * @param p_ptt -			Ptt window used for writing the registers.
 * @param dump_buf -		Pointer to copy the recorded data into.
 * @param buf_size_in_dwords -	Size of the specified buffer in dwords.
 * @param num_dumped_dwords -	OUT: number of dumped dwords.
 *
 * @return error if one of the following holds:
 *	- a recording wasn't started/stopped
 *	- the specified dump buffer is too small
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_bus_dump(struct qed_hwfn *p_hwfn,
				 struct qed_ptt *p_ptt,
				 u32 * dump_buf,
				 u32 buf_size_in_dwords,
				 u32 * num_dumped_dwords);

/**
 * @brief qed_dbg_grc_config - Sets the value of a GRC parameter.
 *
 * @param p_hwfn -		HW device data
 * @param grc_param -	GRC parameter
 * @param val -		Value to set.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 *	- grc_param is invalid
 *	- val is outside the allowed boundaries
 */
enum dbg_status qed_dbg_grc_config(struct qed_hwfn *p_hwfn,
				   enum dbg_grc_params grc_param, u32 val);

/**
 * @brief qed_dbg_grc_set_params_default - Reverts all GRC parameters to their
 * default value.
 *
 * @param p_hwfn - HW device data
 */
void qed_dbg_grc_set_params_default(struct qed_hwfn *p_hwfn);

/**
 * @brief qed_dbg_grc_get_dump_buf_size - Returns the required buffer size
 * for GRC Dump.
 *
 * @param p_hwfn -      HW device data
 * @param p_ptt -	     Ptt window used for writing the registers.
 * @param buf_size - OUT: required buffer size (in dwords) for GRC Dump data.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_grc_get_dump_buf_size(struct qed_hwfn *p_hwfn,
					      struct qed_ptt *p_ptt,
					      u32 * buf_size);

/**
 * @brief qed_dbg_grc_dump - Dumps GRC data into the specified buffer.
 *
 * @param p_hwfn -			HW device data
 * @param p_ptt -			Ptt window used for writing the registers.
 * @param dump_buf -		Pointer to write the collected GRC data into.
 * @param buf_size_in_dwords -	Size of the specified buffer in dwords.
 * @param num_dumped_dwords -	OUT: number of dumped dwords.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 *	- the specified dump buffer is too small
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_grc_dump(struct qed_hwfn *p_hwfn,
				 struct qed_ptt *p_ptt,
				 u32 * dump_buf,
				 u32 buf_size_in_dwords,
				 u32 * num_dumped_dwords);

/**
 * @brief qed_dbg_idle_chk_get_dump_buf_size - Returns the required buffer
 * size for idle check results.
 *
 * @param p_hwfn -      HW device data
 * @param p_ptt -      Ptt window used for writing the registers.
 * @param buf_size - OUT: required buffer size (in dwords) for idle check data.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_idle_chk_get_dump_buf_size(struct qed_hwfn *p_hwfn,
						   struct qed_ptt *p_ptt,
						   u32 * buf_size);

/**
 * @brief qed_dbg_idle_chk_dump - Performs idle check and writes the results
 * into the specified buffer.
 *
 * @param p_hwfn -			HW device data
 * @param p_ptt -			Ptt window used for writing the registers.
 * @param dump_buf -		Pointer to write the idle check data into.
 * @param buf_size_in_dwords -	Size of the specified buffer in dwords.
 * @param num_dumped_dwords -	OUT: number of dumped dwords.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 *	- the specified buffer is too small
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_idle_chk_dump(struct qed_hwfn *p_hwfn,
				      struct qed_ptt *p_ptt,
				      u32 * dump_buf,
				      u32 buf_size_in_dwords,
				      u32 * num_dumped_dwords);

/**
 * @brief qed_dbg_mcp_trace_get_dump_buf_size - Returns the required buffer
 * size for mcp trace results.
 *
 * @param p_hwfn -	     HW device data
 * @param p_ptt -	     Ptt window used for writing the registers.
 * @param buf_size - OUT: required buffer size (in dwords) for mcp trace data.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 *	- the trace data in MCP scratchpad contain an invalid signature
 *	- the bundle ID in NVRAM is invalid
 *	- the trace meta data cannot be found (in NVRAM or image file)
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_mcp_trace_get_dump_buf_size(struct qed_hwfn *p_hwfn,
						    struct qed_ptt *p_ptt,
						    u32 * buf_size);

/**
 * @brief qed_dbg_mcp_trace_dump - Performs mcp trace and writes the results
 * into the specified buffer.
 *
 * @param p_hwfn -			HW device data
 * @param p_ptt -			Ptt window used for writing the registers.
 * @param dump_buf -		Pointer to write the mcp trace data into.
 * @param buf_size_in_dwords -	Size of the specified buffer in dwords.
 * @param num_dumped_dwords -	OUT: number of dumped dwords.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 *	- the specified buffer is too small
 *	- the trace data in MCP scratchpad contain an invalid signature
 *	- the bundle ID in NVRAM is invalid
 *	- the trace meta data cannot be found (in NVRAM or image file)
 *	- the trace meta data cannot be read (from NVRAM or image file)
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_mcp_trace_dump(struct qed_hwfn *p_hwfn,
				       struct qed_ptt *p_ptt,
				       u32 * dump_buf,
				       u32 buf_size_in_dwords,
				       u32 * num_dumped_dwords);

/**
 * @brief qed_dbg_reg_fifo_get_dump_buf_size - Returns the required buffer
 * size for grc trace fifo results.
 *
 * @param p_hwfn -      HW device data
 * @param p_ptt -      Ptt window used for writing the registers.
 * @param buf_size - OUT: required buffer size (in dwords) for reg fifo data.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_reg_fifo_get_dump_buf_size(struct qed_hwfn *p_hwfn,
						   struct qed_ptt *p_ptt,
						   u32 * buf_size);

/**
 * @brief qed_dbg_reg_fifo_dump - Reads the reg fifo and writes the results
 * into the specified buffer.
 *
 * @param p_hwfn -			HW device data
 * @param p_ptt -			Ptt window used for writing the registers.
 * @param dump_buf -		Pointer to write the reg fifo data into.
 * @param buf_size_in_dwords -	Size of the specified buffer in dwords.
 * @param num_dumped_dwords -	OUT: number of dumped dwords.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 *	- the specified buffer is too small
 *	- DMAE transaction failed
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_reg_fifo_dump(struct qed_hwfn *p_hwfn,
				      struct qed_ptt *p_ptt,
				      u32 * dump_buf,
				      u32 buf_size_in_dwords,
				      u32 * num_dumped_dwords);

/**
 * @brief qed_dbg_igu_fifo_get_dump_buf_size - Returns the required buffer
 * size for the IGU fifo results.
 *
 * @param p_hwfn -      HW device data
 * @param p_ptt -      Ptt window used for writing the registers.
 * @param buf_size - OUT: required buffer size (in dwords) for IGU fifo data.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_igu_fifo_get_dump_buf_size(struct qed_hwfn *p_hwfn,
						   struct qed_ptt *p_ptt,
						   u32 * buf_size);

/**
 * @brief qed_dbg_igu_fifo_dump - Reads the IGU fifo and writes the results
 * into the specified buffer.
 *
 * @param p_hwfn -			HW device data
 * @param p_ptt -			Ptt window used for writing the registers.
 * @param dump_buf -		Pointer to write the IGU fifo data into.
 * @param buf_size_in_dwords -	Size of the specified buffer in dwords.
 * @param num_dumped_dwords -	OUT: number of dumped dwords.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 *	- the specified buffer is too small
 *	- DMAE transaction failed
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_igu_fifo_dump(struct qed_hwfn *p_hwfn,
				      struct qed_ptt *p_ptt,
				      u32 * dump_buf,
				      u32 buf_size_in_dwords,
				      u32 * num_dumped_dwords);

/**
 * @brief qed_dbg_protection_override_get_dump_buf_size - Return the required
 * buffer size for protection override window results.
 *
 * @param p_hwfn -      HW device data
 * @param p_ptt -      Ptt window used for writing the registers.
 * @param buf_size - OUT: required buffer size (in dwords) for protection
 *		     override data.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_protection_override_get_dump_buf_size(struct qed_hwfn
							      *p_hwfn,
							      struct qed_ptt
							      *p_ptt,
							      u32 * buf_size);
/**
 * @brief qed_dbg_protection_override_dump - Reads protection override window
 * entries and writes the results into the specified buffer.
 *
 * @param p_hwfn -			HW device data
 * @param p_ptt -			Ptt window used for writing the registers.
 * @param dump_buf -		Pointer to write the protection override data
 *				into.
 * @param buf_size_in_dwords -	Size of the specified buffer in dwords.
 * @param num_dumped_dwords -	OUT: number of dumped dwords.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 *	- the specified buffer is too small
 *	- DMAE transaction failed
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_protection_override_dump(struct qed_hwfn *p_hwfn,
						 struct qed_ptt *p_ptt,
						 u32 *
						 dump_buf,
						 u32
						 buf_size_in_dwords,
						 u32 * num_dumped_dwords);

/**
 * @brief qed_dbg_fw_asserts_get_dump_buf_size - Returns the required buffer
 * size for FW Asserts results.
 *
 * @param p_hwfn -	    HW device data
 * @param p_ptt -	    Ptt window used for writing the registers.
 * @param buf_size - OUT: required buffer size (in dwords) for FW Asserts data.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_fw_asserts_get_dump_buf_size(struct qed_hwfn *p_hwfn,
						     struct qed_ptt *p_ptt,
						     u32 * buf_size);

/**
 * @brief qed_dbg_fw_asserts_dump - Reads the FW Asserts and writes the
 * results into the specified buffer.
 *
 * @param p_hwfn -			HW device data
 * @param p_ptt -			Ptt window used for writing the registers.
 * @param dump_buf -		Pointer to write the FW Asserts data into.
 * @param buf_size_in_dwords -	Size of the specified buffer in dwords.
 * @param num_dumped_dwords -	OUT: number of dumped dwords.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 *	- the specified buffer is too small
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_fw_asserts_dump(struct qed_hwfn *p_hwfn,
					struct qed_ptt *p_ptt,
					u32 * dump_buf,
					u32 buf_size_in_dwords,
					u32 * num_dumped_dwords);

/**
 * @brief qed_dbg_ilt_get_dump_buf_size - Returns the required buffer size
 * for ILT Dump.
 *
 * @param p_hwfn -      HW device data
 * @param p_ptt -	     Ptt window used for writing the registers.
 * @param buf_size - OUT: required buffer size (in dwords) for ILT Dump data.
 *                        Limitation: the ILT dump size must not exceed
 *                        buf_size bit-width capability.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_ilt_get_dump_buf_size(struct qed_hwfn *p_hwfn,
					      struct qed_ptt *p_ptt,
					      u32 * buf_size);

/**
 * @brief qed_dbg_ilt_dump - Dumps ILT data into the specified buffer.
 *
 * @param p_hwfn -			HW device data
 * @param p_ptt -			Ptt window used for writing the registers.
 * @param dump_buf -		Pointer to write the collected ILT data into.
 * @param buf_size_in_dwords -	Size of the specified buffer in dwords.
 * @param num_dumped_dwords -	OUT: number of dumped dwords.
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 *	- the specified dump buffer is too small
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_ilt_dump(struct qed_hwfn *p_hwfn,
				 struct qed_ptt *p_ptt,
				 u32 * dump_buf,
				 u32 buf_size_in_dwords,
				 u32 * num_dumped_dwords);

/**
 * @brief qed_dbg_read_attn - Reads the attention registers of the specified
 * block and type, and writes the results into the specified buffer.
 *
 * @param p_hwfn -		HW device data
 * @param p_ptt -		Ptt window used for writing the registers.
 * @param block -	Block ID.
 * @param attn_type -	Attention type.
 * @param clear_status -	Indicates if the attention status should be cleared.
 * @param results -	OUT: Pointer to write the read results into
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_read_attn(struct qed_hwfn *p_hwfn,
				  struct qed_ptt *p_ptt,
				  enum block_id block,
				  enum dbg_attn_type attn_type,
				  bool clear_status,
				  struct dbg_attn_block_result *results);

/**
 * @brief qed_dbg_print_attn - Prints attention registers values in the
 * specified results struct.
 *
 * @param p_hwfn -     HW device data
 * @param results - Pointer to the attention read results
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_print_attn(struct qed_hwfn *p_hwfn,
				   struct dbg_attn_block_result *results);

/**
 * @brief qed_is_block_in_reset - Returns true if the specified block is in
 * reset, false otherwise.
 *
 * @param p_hwfn   - HW device data
 * @param p_ptt   - Ptt window used for writing the registers.
 * @param block - Block ID.
 *
 * @return true if the specified block is in reset, false otherwise.
 */
bool qed_is_block_in_reset(struct qed_hwfn *p_hwfn,
			   struct qed_ptt *p_ptt, enum block_id block);

/**
 * @brief qed_mask_parities_no_mcp - Mask/unmask all parities. Requires full
 * register access previleges. To be used then MCP is not available.
 *
 * @param p_hwfn   - HW device data
 * @param p_ptt   - Ptt window used for writing the registers.
 * @param mask  - indicates if parities should be masked (true) or unmasked
 *		 (false).
 */
void qed_mask_parities_no_mcp(struct qed_hwfn *p_hwfn,
			      struct qed_ptt *p_ptt, bool mask);

/**
 * @brief qed_grc_dump_addr_range - Dumps the GRC registers in the specified*                                    address range.
 *
 * @param p_hwfn        - HW device data
 * @param p_ptt        - Ptt window used for writing the registers.
 * @param dump_buf   - to dump the address range onto.
 * @param dump       - Whether to actually dump (1) or just estimate size (0).
 * @param addr       - (Dword) Address to dump from.
 * @param len        - Length (in Dwords) to dump.
 * @param wide_bus   - Register Wide-bus indeication
 * @param split_type - Register Split type (per-port, per-pf)
 * @param split_id   - Split-ID of register
 *
 * @return size of the (or estimeded to be) dumped size in Dwords.
 */
u32 qed_grc_dump_addr_range(struct qed_hwfn *p_hwfn,
			    struct qed_ptt *p_ptt,
			    u32 * dump_buf,
			    bool dump,
			    u32 addr,
			    u32 len,
			    bool wide_bus,
			    enum init_split_types split_type, u8 split_id);

/**
 * @brief qed_dbg_dev_init - Initializes debug data for the specified device.
 *
 * @param p_hwfn - HW device data.
 *
 * @return status of operation (success/fail)
 */
enum dbg_status qed_dbg_dev_init(struct qed_hwfn *p_hwfn);

/**
 * @brief qed_is_reg_read_valid - Returns true if theres no issue with PCI,
 * false otherwise.
 *
 * @param p_hwfn   - HW device data
 * @param p_ptt   - Ptt window used for writing the registers.
 *
 *
 * @Returns true if theres no issue with PCI, false otherwise.
 */
bool qed_is_reg_read_valid(struct qed_hwfn *p_hwfn, struct qed_ptt *p_ptt);

#ifndef _USER_DBG_FW_FUNCS_H
#define _USER_DBG_FW_FUNCS_H
/******************************* Data Types **********************************/

/* MCP Trace  structure, generated by a perl script during MFW build.
 * Therefore, this struct is duplicated in the perl script.
 */
struct mcp_trace_format {
	u32 data;
#define MCP_TRACE_FORMAT_MODULE_MASK		0x0000ffff
#define MCP_TRACE_FORMAT_MODULE_OFFSET		0
#define MCP_TRACE_FORMAT_LEVEL_MASK		0x00030000
#define MCP_TRACE_FORMAT_LEVEL_OFFSET		16
#define MCP_TRACE_FORMAT_P1_SIZE_MASK		0x000c0000
#define MCP_TRACE_FORMAT_P1_SIZE_OFFSET		18
#define MCP_TRACE_FORMAT_P2_SIZE_MASK		0x00300000
#define MCP_TRACE_FORMAT_P2_SIZE_OFFSET		20
#define MCP_TRACE_FORMAT_P3_SIZE_MASK		0x00c00000
#define MCP_TRACE_FORMAT_P3_SIZE_OFFSET		22
#define MCP_TRACE_FORMAT_LEN_MASK		0xff000000
#define MCP_TRACE_FORMAT_LEN_OFFSET		24

	char *format_str;
};

/* MCP Trace Meta data structure, generated by a perl script during MFW build.
 * Therefore, this struct is duplicated in the perl script.
 */
struct mcp_trace_meta {
	u32 modules_num;
	char **modules;
	u32 formats_num;
	struct mcp_trace_format *formats;
	bool is_allocated;
};

/* Debug Tools user data */
struct dbg_tools_user_data {
	struct mcp_trace_meta mcp_trace_meta;
	const u32 *mcp_trace_user_meta_buf;
};

/******************************** Constants **********************************/

#define MAX_NAME_LEN	16

/***************************** Public Functions *******************************/

/**
 * @brief qed_dbg_user_set_bin_ptr - Sets a pointer to the binary data with
 * debug arrays.
 *
 * @param p_hwfn -	    HW device data
 * @param bin_ptr - a pointer to the binary data with debug arrays.
 */
enum dbg_status qed_dbg_user_set_bin_ptr(struct qed_hwfn *p_hwfn,
					 const u8 * const bin_ptr);

/**
 * @brief qed_dbg_alloc_user_data - Allocates user debug data.
 *
 * @param p_hwfn -		 HW device data
 * @param user_data_ptr - OUT: a pointer to the allocated memory.
 */
enum dbg_status qed_dbg_alloc_user_data(struct qed_hwfn *p_hwfn,
					void **user_data_ptr);

/**
 * @brief qed_dbg_get_storm_id - Returns an ID for the specified storm name.
 *
 * @param storm_name - Storm name.
 *
 * @return an ID for the specified storm name, or NUM_OF_STORMS if not found.
 */
enum dbg_storms qed_dbg_get_storm_id(const char *storm_name);

/**
 * @brief qed_dbg_get_block_id - Returns an ID for the specified block name.
 *
 * @param p_hwfn -	       HW device data
 * @param block_name - Block name.
 *
 * @return an ID for the specified block name, or NUM_OF_BLOCKS if not found.
 */
enum block_id qed_dbg_get_block_id(struct qed_hwfn *p_hwfn,
				   const char *block_name);

/**
 * @brief qed_dbg_get_storm_mode_id - Returns an ID for the specified Storm
 * mode name.
 *
 * @param storm_mode_name - Storm mode name.
 *
 * @return an ID for the specified Storm mode name, or MAX_DBG_BUS_STORM_MODES
 * if not found.
 */
enum dbg_bus_storm_modes qed_dbg_get_storm_mode_id(const char *storm_mode_name);

/**
 * @brief qed_dbg_get_constraint_op_id - Returns an ID for the specified
 * constraint operation name.
 *
 * @param op_name - operation name.
 *
 * @return an ID for the specified constraint operation name, or
 * MAX_DBG_BUS_CONSTRAINT_OPS if not found.
 */
enum dbg_bus_constraint_ops qed_dbg_get_constraint_op_id(const char *op_name);

/**
 * @brief qed_dbg_get_status_str - Returns a string for the specified status.
 *
 * @param status - a debug status code.
 *
 * @return a string for the specified status
 */
const char *qed_dbg_get_status_str(enum dbg_status status);

/**
 * @brief qed_dbg_get_grc_param_id - Returns an ID for the specified GRC
 * param name.
 *
 * @param param_name - GRC param name.
 *
 * @return an ID for the specified GRC param name, or NUM_OF_GRC_PARAMS if not
 * found.
 */
enum dbg_grc_params qed_dbg_get_grc_param_id(const char *param_name);

/**
 * @brief qed_dbg_get_dbg_bus_line - Returns an ID for the specified Debug Bus
 * line.
 *
 * @param p_hwfn -	     HW device data
 * @param block_id - block ID
 * @param chip_id -  chip ID
 * @param line -     a string containing a debug line name that belongs to the
 *		     specified block/chip, or an 8-bit debug line number.
 *
 * @return an ID for the specified Debug Bus line name, or -1 if not found.
 */
int qed_dbg_get_dbg_bus_line(struct qed_hwfn *p_hwfn,
			     enum block_id block_id,
			     enum chip_ids chip_id, const char *line);

/**
 * @brief qed_get_idle_chk_results_buf_size - Returns the required buffer
 * size for idle check results (in bytes).
 *
 * @param p_hwfn -		      HW device data
 * @param dump_buf -	      idle check dump buffer.
 * @param num_dumped_dwords - number of dwords that were dumped.
 * @param results_buf_size -  OUT: required buffer size (in bytes) for the
 *			      parsed results.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_get_idle_chk_results_buf_size(struct qed_hwfn *p_hwfn,
						  u32 *
						  dump_buf,
						  u32
						  num_dumped_dwords,
						  u32 * results_buf_size);

/**
 * @brief qed_print_idle_chk_results - Prints idle check results
 *
 * @param p_hwfn -			HW device data
 * @param dump_buf -		idle check dump buffer.
 * @param num_dumped_dwords -	number of dwords that were dumped.
 * @param results_buf -		buffer for printing the idle check results.
 * @param results_buf_size -size of results_buf.
 * @param num_errors -		OUT: number of errors found in idle check.
 * @param num_warnings -	OUT: number of warnings found in idle check.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_print_idle_chk_results(struct qed_hwfn *p_hwfn,
					   u32 * dump_buf,
					   u32
					   num_dumped_dwords,
					   char *results_buf,
					   u32
					   results_buf_size,
					   u32 * num_errors,
					   u32 * num_warnings);

/**
 * @brief qed_dbg_mcp_trace_set_meta_data - Sets the MCP Trace meta data.
 *
 * Needed in case the MCP Trace dump doesn't contain the meta data (e.g. due to
 * no NVRAM access).
 *
 * @param p_hwfn -	     HW device data
 * @param meta_buf - pointer to MCP Trace meta data
 */
void qed_dbg_mcp_trace_set_meta_data(struct qed_hwfn *p_hwfn,
				     const u32 * meta_buf);

/**
 * @brief qed_get_mcp_trace_results_buf_size - Returns the required buffer
 * size for MCP Trace results (in bytes).
 *
 * @param p_hwfn -		      HW device data
 * @param dump_buf -	      MCP Trace dump buffer.
 * @param num_dumped_dwords - number of dwords that were dumped.
 * @param results_buf_size -  OUT: required buffer size (in bytes) for the
 *			      parsed results.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_get_mcp_trace_results_buf_size(struct qed_hwfn *p_hwfn,
						   u32 *
						   dump_buf,
						   u32
						   num_dumped_dwords,
						   u32 * results_buf_size);

/**
 * @brief qed_print_mcp_trace_results - Prints MCP Trace results
 *
 * @param p_hwfn -		      HW device data
 * @param dump_buf -	      mcp trace dump buffer, starting from the header.
 * @param num_dumped_dwords - number of dwords that were dumped.
 * @param results_buf -	      buffer for printing the mcp trace results.
 * @param results_buf_size -  size of results_buf.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_print_mcp_trace_results(struct qed_hwfn *p_hwfn,
					    u32 * dump_buf,
					    u32
					    num_dumped_dwords,
					    char *results_buf,
					    u32 results_buf_size);

/**
 * @brief qed_print_mcp_trace_results_cont - Prints MCP Trace results, and
 * keeps the MCP trace meta data allocated, to support continuous MCP Trace
 * parsing. After the continuous parsing ends, mcp_trace_free_meta_data should
 * be called to free the meta data.
 *
 * @param p_hwfn -		      HW device data
 * @param dump_buf -	      mcp trace dump buffer, starting from the header.
 * @param num_dumped_dwords - number of dwords that were dumped.
 * @param results_buf -	      buffer for printing the mcp trace results.
 * @param results_buf_size -  size of results_buf.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_print_mcp_trace_results_cont(struct qed_hwfn *p_hwfn,
						 u32 *
						 dump_buf,
						 u32
						 num_dumped_dwords,
						 char *results_buf,
						 u32 results_buf_size);

/**
 * @brief print_mcp_trace_line - Prints MCP Trace results for a single line
 *
 * @param p_hwfn -		      HW device data
 * @param dump_buf -	      mcp trace dump buffer, starting from the header.
 * @param num_dumped_bytes -  number of bytes that were dumped.
 * @param results_buf -	      buffer for printing the mcp trace results.
 * @param results_buf_size -  size of results_buf.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_print_mcp_trace_line(struct qed_hwfn *p_hwfn,
					 u8 * dump_buf,
					 u32
					 num_dumped_bytes,
					 char *results_buf,
					 u32 results_buf_size);

/**
 * @brief mcp_trace_free_meta_data - Frees the MCP Trace meta data.
 * Should be called after continuous MCP Trace parsing.
 *
 * @param p_hwfn - HW device data
 */
void qed_mcp_trace_free_meta_data(struct qed_hwfn *p_hwfn);

/**
 * @brief qed_get_reg_fifo_results_buf_size - Returns the required buffer
 * size for reg_fifo results (in bytes).
 *
 * @param p_hwfn -		      HW device data
 * @param dump_buf -	      reg fifo dump buffer.
 * @param num_dumped_dwords - number of dwords that were dumped.
 * @param results_buf_size -  OUT: required buffer size (in bytes) for the
 *			      parsed results.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_get_reg_fifo_results_buf_size(struct qed_hwfn *p_hwfn,
						  u32 *
						  dump_buf,
						  u32
						  num_dumped_dwords,
						  u32 * results_buf_size);

/**
 * @brief qed_print_reg_fifo_results - Prints reg fifo results
 *
 * @param p_hwfn -			HW device data
 * @param dump_buf -		reg fifo dump buffer, starting from the header.
 * @param num_dumped_dwords -	number of dwords that were dumped.
 * @param results_buf -		buffer for printing the reg fifo results.
 * @param results_buf_size -  size of results_buf.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_print_reg_fifo_results(struct qed_hwfn *p_hwfn,
					   u32 * dump_buf,
					   u32
					   num_dumped_dwords,
					   char *results_buf,
					   u32 results_buf_size);

/**
 * @brief qed_get_igu_fifo_results_buf_size - Returns the required buffer size
 * for igu_fifo results (in bytes).
 *
 * @param p_hwfn -		      HW device data
 * @param dump_buf -	      IGU fifo dump buffer.
 * @param num_dumped_dwords - number of dwords that were dumped.
 * @param results_buf_size -  OUT: required buffer size (in bytes) for the
 *			      parsed results.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_get_igu_fifo_results_buf_size(struct qed_hwfn *p_hwfn,
						  u32 *
						  dump_buf,
						  u32
						  num_dumped_dwords,
						  u32 * results_buf_size);

/**
 * @brief qed_print_igu_fifo_results - Prints IGU fifo results
 *
 * @param p_hwfn -		     HW device data
 * @param dump_buf -	     IGU fifo dump buffer, starting from the header.
 * @param num_dumped_dwords - number of dwords that were dumped.
 * @param results_buf -	     buffer for printing the IGU fifo results.
 * @param results_buf_size -  size of results_buf.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_print_igu_fifo_results(struct qed_hwfn *p_hwfn,
					   u32 * dump_buf,
					   u32
					   num_dumped_dwords,
					   char *results_buf,
					   u32 results_buf_size);

/**
 * @brief qed_get_protection_override_results_buf_size - Returns the required
 * buffer size for protection override results (in bytes).
 *
 * @param p_hwfn -		      HW device data
 * @param dump_buf -	      protection override dump buffer.
 * @param num_dumped_dwords - number of dwords that were dumped.
 * @param results_buf_size -  OUT: required buffer size (in bytes) for the parsed results.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_get_protection_override_results_buf_size(struct qed_hwfn
							     *p_hwfn,
							     u32 * dump_buf,
							     u32
							     num_dumped_dwords,
							     u32 *
							     results_buf_size);

/*
 * @brief qed_print_protection_override_results - Prints protection override
 * results.
 *
 * @param p_hwfn -               HW device data
 * @param dump_buf -          protection override dump buffer, starting from
 *			      the header.
 * @param num_dumped_dwords - number of dwords that were dumped.
 * @param results_buf -       buffer for printing the reg fifo results.
 * @param results_buf_size -  size of results_buf.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_print_protection_override_results(struct qed_hwfn *p_hwfn,
						      u32 *
						      dump_buf,
						      u32
						      num_dumped_dwords,
						      char *results_buf,
						      u32 results_buf_size);

/**
 * @brief qed_get_fw_asserts_results_buf_size - Returns the required buffer
 * size for FW Asserts results (in bytes).
 *
 * @param p_hwfn -		      HW device data
 * @param dump_buf -	      FW Asserts dump buffer.
 * @param num_dumped_dwords - number of dwords that were dumped.
 * @param results_buf_size -  OUT: required buffer size (in bytes) for the
 *			      parsed results.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_get_fw_asserts_results_buf_size(struct qed_hwfn *p_hwfn,
						    u32 *
						    dump_buf,
						    u32
						    num_dumped_dwords,
						    u32 * results_buf_size);

/**
 * @brief qed_print_fw_asserts_results - Prints FW Asserts results
 *
 * @param p_hwfn -		     HW device data
 * @param dump_buf -	     FW Asserts dump buffer, starting from the header.
 * @param num_dumped_dwords - number of dwords that were dumped.
 * @param results_buf -	     buffer for printing the FW Asserts results.
 * @param results_buf_size -  size of results_buf.
 *
 * @return error if the parsing fails, ok otherwise.
 */
enum dbg_status qed_print_fw_asserts_results(struct qed_hwfn *p_hwfn,
					     u32 * dump_buf,
					     u32
					     num_dumped_dwords,
					     char *results_buf,
					     u32 results_buf_size);

/**
 * @brief qed_dbg_parse_attn - Parses and prints attention registers values in
 * the specified results struct.
 *
 * @param p_hwfn -	    HW device data
 * @param results - Pointer to the attention read results
 *
 * @return error if one of the following holds:
 *	- the version wasn't set
 * Otherwise, returns ok.
 */
enum dbg_status qed_dbg_parse_attn(struct qed_hwfn *p_hwfn,
				   struct dbg_attn_block_result *results);

#endif

#ifndef __DBG_VALUES_H__
#define __DBG_VALUES_H__

/* modes tree buffer */
static const u8 dbg_modes_tree_buf[] = {
	0x02, 0x02, 0x04, 0x02, 0x00, 0x11, 0x0f, 0x04, 0x02, 0x02, 0x04, 0x02,
	0x00, 0x11, 0x0e, 0x04, 0x02, 0x00, 0x04, 0x00, 0x00, 0x01, 0x06, 0x13,
	0x02, 0x00, 0x00, 0x01, 0x06, 0x13, 0x04, 0x02, 0x05, 0x00, 0x00, 0x01,
	0x06, 0x13, 0x02, 0x02, 0x00, 0x06, 0x10, 0x00, 0x04, 0x02, 0x02, 0x00,
	0x06, 0x0f, 0x00, 0x04, 0x02, 0x00, 0x04, 0x00, 0x01, 0x07, 0x09, 0x02,
	0x02, 0x00, 0x06, 0x04, 0x00, 0x05, 0x02, 0x02, 0x00, 0x06, 0x0e, 0x00,
	0x04, 0x02, 0x02, 0x00, 0x06, 0x05, 0x00, 0x04, 0x02, 0x02, 0x05, 0x00,
	0x10, 0x00, 0x04, 0x02, 0x02, 0x00, 0x06, 0x04, 0x04, 0x02, 0x02, 0x00,
	0x06, 0x10, 0x04, 0x02, 0x02, 0x00, 0x06, 0x0f, 0x04, 0x02, 0x00, 0x04,
	0x00, 0x00, 0x06, 0x02, 0x00, 0x01, 0x07, 0x09, 0x04, 0x02, 0x00, 0x05,
	0x00, 0x00, 0x06, 0x02, 0x02, 0x00, 0x06, 0x0e, 0x04, 0x02, 0x02, 0x04,
	0x00, 0x11, 0x04, 0x02, 0x02, 0x05, 0x0c, 0x00, 0x04, 0x02, 0x02, 0x05,
	0x10, 0x00, 0x04, 0x02, 0x05, 0x02, 0x00, 0x0b, 0x0e, 0x02, 0x05, 0x02,
	0x00, 0x0b, 0x0f, 0x02, 0x05, 0x02, 0x00, 0x0b, 0x10, 0x02, 0x00, 0x04,
	0x00, 0x09, 0x02, 0x05, 0x02, 0x0b, 0x10, 0x02, 0x01, 0x0b, 0x0d, 0x04,
	0x02, 0x02, 0x04, 0x11, 0x04, 0x02, 0x02, 0x04, 0x0f, 0x04, 0x02, 0x04,
	0x00, 0x00, 0x06, 0x01, 0x06, 0x01, 0x07, 0x09, 0x02, 0x00, 0x00, 0x06,
	0x04, 0x02, 0x05, 0x00, 0x00, 0x06, 0x02, 0x00, 0x00, 0x06, 0x05, 0x02,
	0x00, 0x06, 0x00, 0x04, 0x02, 0x05, 0x02, 0x0b, 0x0e, 0x02, 0x05, 0x02,
	0x0b, 0x0f, 0x02, 0x00, 0x0b, 0x04, 0x02, 0x00, 0x09, 0x04, 0x02, 0x00,
	0x0d, 0x04, 0x02, 0x04, 0x00, 0x09, 0x02, 0x04, 0x00, 0x05, 0x02, 0x05,
	0x00, 0x04, 0x02, 0x05, 0x0f, 0x02, 0x04, 0x06, 0x02, 0x0c, 0x04, 0x02,
	0x0d, 0x04, 0x02, 0x0e, 0x04, 0x02, 0x0f, 0x04, 0x02, 0x0b, 0x04, 0x00,
	0x07, 0x00, 0x13, 0x03, 0x0a, 0x08, 0x12,
};

/* Data size: 283 bytes */

/* Array of registers to be dumped */
static const u32 dump_reg[] = {
	0x0000094f,		/* split NONE */
	0x06000000,		/* block grc */
	0x02014000,		/* grc.override_window_mem_self_init_start .. grc.override_window_mem_self_init_done (2 regs) */
	0x0a014010,		/* grc.rsv_attn_access_data_0 .. grc.trace_fifo_valid_data (10 regs) */
	0x1201401c,		/* grc.trace_fifo_enable .. grc.dbg_force_frame (18 regs) */
	0x0201403a,		/* grc.dbgsyn_status .. grc.dbgsyn_almost_full_thr (2 regs) */
	0x02014060,		/* grc.INT_STS_0 .. grc.INT_MASK_0 (2 regs) */
	0x04014100,		/* grc.timeout_val .. grc.number_valid_override_window (4 regs) */
	0x0a010000,		/* block miscs */
	0x02002410,		/* miscs.reset_config .. miscs.reset_config_por (2 regs) */
	0x0500241c,		/* miscs.clk_100g_mode .. miscs.NVM_WR_EN (5 regs) */
	0x0100245b,		/* miscs.memctrl_status (1 regs) */
	0x02002460,		/* miscs.INT_STS_0 .. miscs.INT_MASK_0 (2 regs) */
	0x83002500,		/* miscs.gpio0_driver .. miscs.gpio_event_en (131 regs) */
	0x120025af,		/* miscs.LINK_HOLDOFF_STATUS .. miscs.func_hide_pin (18 regs) */
	0x150025c2,		/* miscs.four_port_shared_mdio_en .. miscs.unprepared_fw (21 regs) */
	0x100025d8,		/* miscs.VAUX_PRESENT .. miscs.perst_deassert_cnt (16 regs) */
	0x010025eb,		/* miscs.hot_reset_en (1 regs) */
	0x020025ed,		/* miscs.eco_reserved .. miscs.mcp_rom_tm (2 regs) */
	0x09020000,		/* block misc */
	0x01002010,		/* misc.reset_config (1 regs) */
	0x02002060,		/* misc.INT_STS .. misc.INT_MASK (2 regs) */
	0xff002100,		/* misc.aeu_general_attn_0 .. misc.aeu_after_invert_9_mcp (255 regs) */
	0x0c0021ff,		/* misc.aeu_sys_kill_occurred .. misc.aeu_general_mask (12 regs) */
	0x0100220c,		/* misc.aeu_mask_attn_igu_msb (1 regs) */
	0x0200220e,		/* misc.aeu_vpd_latch_status .. misc.attn_num_st (2 regs) */
	0x01002300,		/* misc.port_mode (1 regs) */
	0x16002303,		/* misc.opte_mode .. misc.sw_timer_reload_val_8 (22 regs) */
	0x0f00231a,		/* misc.sw_timer_event .. misc.eco_reserved (15 regs) */
	0x1d040000,		/* block pglue_b */
	0x010aa001,		/* pglue_b.init_done_inb_int_mem (1 regs) */
	0x010aa003,		/* pglue_b.init_done_ptt_gtt (1 regs) */
	0x010aa005,		/* pglue_b.init_done_zone_a (1 regs) */
	0x020aa060,		/* pglue_b.INT_STS .. pglue_b.INT_MASK (2 regs) */
	0x050aa100,		/* pglue_b.dbg_select .. pglue_b.dbg_force_frame (5 regs) */
	0x170aa118,		/* pglue_b.pgl_eco_reserved .. pglue_b.DISABLE_HIGHER_BW (23 regs) */
	0x040aa132,		/* pglue_b.memctrl_status .. pglue_b.tc_per_vq (4 regs) */
	0x020aa148,		/* pglue_b.pgl_control0 .. pglue_b.cssnoop_almost_full_thr (2 regs) */
	0x020aa158,		/* pglue_b.pgl_txr_cdts .. pglue_b.pgl_txw_cdts (2 regs) */
	0x050aa800,		/* pglue_b.cfg_space_a_address .. pglue_b.cfg_space_a_request (5 regs) */
	0x010aa806,		/* pglue_b.cfg_space_b_request (1 regs) */
	0x090aa808,		/* pglue_b.flr_request_vf_31_0 .. pglue_b.flr_request_pf_31_0 (9 regs) */
	0x020aa81a,		/* pglue_b.DISABLE_FLR_SRIOV_DISABLED .. pglue_b.sr_iov_disabled_request (2 regs) */
	0x090aa81d,		/* pglue_b.shadow_bme_vf_31_0 .. pglue_b.shadow_bme_pf_31_0 (9 regs) */
	0x0a0aa82f,		/* pglue_b.shadow_ats_enable_vf_31_0 .. pglue_b.shadow_vf_enable_pf_31_0 (10 regs) */
	0x0c0aa83a,		/* pglue_b.shadow_ido_bits .. pglue_b.was_error_pf_31_0 (12 regs) */
	0x0b0aa84f,		/* pglue_b.rx_err_details .. pglue_b.tx_err_wr_details_icpl (11 regs) */
	0x400aa85e,		/* pglue_b.internal_vfid_enable_31_0_value .. pglue_b.psdm_inb_int_b_vf_1 (64 regs) */
	0x290aa8c6,		/* pglue_b.tsdm_zone_a_size_pf .. pglue_b.vf_grc_space_violation_details (41 regs) */
	0x090aa8f0,		/* pglue_b.ido_enable_master_rw .. pglue_b.cpu_mbist_memctrl_1_cntrl_cmd (9 regs) */
	0x030aa8fb,		/* pglue_b.disable_tcpl_translation_size_check .. pglue_b.cpu_mbist_memctrl_3_cntrl_cmd (3 regs) */
	0x010aa900,		/* pglue_b.pgl_tgtwr_mlength (1 regs) */
	0x070aa905,		/* pglue_b.pgl_exp_rom_addr .. pglue_b.pgl_tags_limit (7 regs) */
	0x0e0aa951,		/* pglue_b.master_zlr_err_add_31_0 .. pglue_b.psdm_queue_zone_size (14 regs) */
	0x050aa960,		/* pglue_b.sdm_channel_enable .. pglue_b.MASTER_ATTENTION_SETTING (5 regs) */
	0x170aab80,		/* pglue_b.write_fifo_occupancy_level .. pglue_b.pcie_ltr_state (23 regs) */
	0x070aab9c,		/* pglue_b.mctp_tc .. pglue_b.expansion_rom_attn (7 regs) */
	0x010aaba4,		/* pglue_b.mps_attn (1 regs) */
	0x060aaba6,		/* pglue_b.vpd_request_pf_31_0 .. pglue_b.sticky_master_error_en (6 regs) */
	0x010a0000,		/* block pcie */
	0x04015080,		/* pcie.eco_reserved .. pcie.pcie_debug_bits (4 regs) */
	0x030c0000,		/* block mcp2 */
	0x01014880,		/* mcp2.eco_reserved (1 regs) */
	0x03014900,		/* mcp2.dbg_select .. mcp2.dbg_shift (3 regs) */
	0x02014910,		/* mcp2.dbg_force_valid .. mcp2.dbg_force_frame (2 regs) */
	0x060d0000,		/* block pswhst */
	0x020a8000,		/* pswhst.zone_perm_table_init .. pswhst.zone_perm_table_init_done (2 regs) */
	0x1c0a8010,		/* pswhst.discard_internal_writes .. pswhst.is_in_drain_mode (28 regs) */
	0x040a802d,		/* pswhst.timeout_data .. pswhst.source_usdm_credits (4 regs) */
	0x0d0a8032,		/* pswhst.host_strict_priority .. pswhst.psdm_swap_mode (13 regs) */
	0x050a8040,		/* pswhst.dbg_select .. pswhst.dbg_force_frame (5 regs) */
	0x020a8060,		/* pswhst.INT_STS .. pswhst.INT_MASK (2 regs) */
	0x020e0000,		/* block pswhst2 */
	0x0b0a7810,		/* pswhst2.header_fifo_status .. pswhst2.dbg_force_frame (11 regs) */
	0x020a7860,		/* pswhst2.INT_STS .. pswhst2.INT_MASK (2 regs) */
	0x030f0000,		/* block pswrd */
	0x050a7010,		/* pswrd.dbg_select .. pswrd.dbg_force_frame (5 regs) */
	0x030a7028,		/* pswrd.eco_reserved .. pswrd.fifo_full_sticky (3 regs) */
	0x020a7060,		/* pswrd.INT_STS .. pswrd.INT_MASK (2 regs) */
	0x08100000,		/* block pswrd2 */
	0x020a7400,		/* pswrd2.start_init .. pswrd2.init_done (2 regs) */
	0x0c0a7410,		/* pswrd2.first_sr_nodes .. pswrd2.cpl_err_details2 (12 regs) */
	0x050a741d,		/* pswrd2.arb_delay .. pswrd2.eco_reserved (5 regs) */
	0x060a7430,		/* pswrd2.pbf_swap_mode .. pswrd2.ptu_swap_mode (6 regs) */
	0x1e0a7438,		/* pswrd2.almost_full_0 .. pswrd2.max_fill_level_pbf (30 regs) */
	0x020a7460,		/* pswrd2.INT_STS .. pswrd2.INT_MASK (2 regs) */
	0x050a7500,		/* pswrd2.dbg_select .. pswrd2.dbg_force_frame (5 regs) */
	0x140a7518,		/* pswrd2.disable_inputs .. pswrd2.prm_additional_requests (20 regs) */
	0x03110000,		/* block pswwr */
	0x160a6810,		/* pswwr.usdm_full_th .. pswwr.dbg_force_frame (22 regs) */
	0x010a6832,		/* pswwr.eco_reserved (1 regs) */
	0x020a6860,		/* pswwr.INT_STS .. pswwr.INT_MASK (2 regs) */
	0x03120000,		/* block pswwr2 */
	0x030a6c10,		/* pswwr2.cdu_full_th2 .. pswwr2.pglue_eop_err_details (3 regs) */
	0x060a6c14,		/* pswwr2.prm_curr_fill_level .. pswwr2.eco_reserved (6 regs) */
	0x020a6c60,		/* pswwr2.INT_STS .. pswwr2.INT_MASK (2 regs) */
	0x03130000,		/* block pswrq */
	0x050a0008,		/* pswrq.dbg_select .. pswrq.dbg_force_frame (5 regs) */
	0x010a0018,		/* pswrq.eco_reserved (1 regs) */
	0x020a0060,		/* pswrq.INT_STS .. pswrq.INT_MASK (2 regs) */
	0x08140000,		/* block pswrq2 */
	0x03090000,		/* pswrq2.rbc_done .. pswrq2.reset_stt (3 regs) */
	0x0609001e,		/* pswrq2.endianity_00 .. pswrq2.m2p_endian_m (6 regs) */
	0x05090040,		/* pswrq2.dbg_select .. pswrq2.dbg_force_frame (5 regs) */
	0x02090060,		/* pswrq2.INT_STS .. pswrq2.INT_MASK (2 regs) */
	0xfc090100,		/* pswrq2.wr_mbs0 .. pswrq2.atc_vq_enable (252 regs) */
	0x250901fd,		/* pswrq2.atc_internal_ats_enable_all .. pswrq2.sr_cnt_window_value (37 regs) */
	0x2a090224,		/* pswrq2.sr_cnt_start_mode .. pswrq2.l2p_err_details2 (42 regs) */
	0xc609024f,		/* pswrq2.sr_num_cfg .. pswrq2.l2p_validate_vfid (198 regs) */
	0x01150000,		/* block pglcs */
	0x02000740,		/* pglcs.INT_STS .. pglcs.INT_MASK (2 regs) */
	0x05160000,		/* block dmae */
	0x01003000,		/* dmae.init (1 regs) */
	0x22003010,		/* dmae.pci_ifen .. dmae.go_c31 (34 regs) */
	0x02003060,		/* dmae.INT_STS .. dmae.INT_MASK (2 regs) */
	0x14003100,		/* dmae.pxp_req_init_crd .. dmae.fsm_st (20 regs) */
	0x06003143,		/* dmae.memctrl_status .. dmae.dbg_force_frame (6 regs) */
	0x06170000,		/* block ptu */
	0x02158000,		/* ptu.atc_init_array .. ptu.atc_init_done (2 regs) */
	0x0e158010,		/* ptu.LOG_TRANSPEND_REUSE_MISS_TID .. ptu.inv_err_ctr (14 regs) */
	0x14158023,		/* ptu.inv_halt_on_reuse_cnt_err .. ptu.ptu_b0_disable (20 regs) */
	0x05158040,		/* ptu.dbg_select .. ptu.dbg_force_frame (5 regs) */
	0x02158060,		/* ptu.INT_STS .. ptu.INT_MASK (2 regs) */
	0x79158100,		/* ptu.atc_num_sets .. ptu.atc_ireq_fifo_tm (121 regs) */
	0x10180000,		/* block tcm */
	0x01460000,		/* tcm.init (1 regs) */
	0x05460010,		/* tcm.dbg_select .. tcm.dbg_force_frame (5 regs) */
	0x02460060,		/* tcm.INT_STS_0 .. tcm.INT_MASK_0 (2 regs) */
	0x02460064,		/* tcm.INT_STS_1 .. tcm.INT_MASK_1 (2 regs) */
	0x02460068,		/* tcm.INT_STS_2 .. tcm.INT_MASK_2 (2 regs) */
	0x5a460100,		/* tcm.ifen .. tcm.err_evnt_id (90 regs) */
	0x21460181,		/* tcm.storm_weight .. tcm.ia_trans_part_fill_lvl (33 regs) */
	0x284601c1,		/* tcm.xx_msg_up_bnd .. tcm.unlock_miss (40 regs) */
	0x39460201,		/* tcm.prcs_agg_con_curr_st .. tcm.sm_task_ctx_size (57 regs) */
	0x22460245,		/* tcm.agg_con_cf0_q .. tcm.agg_task_rule5_q (34 regs) */
	0x0d460281,		/* tcm.in_prcs_tbl_crd_agg .. tcm.xx_byp_task_state_evnt_id_flg (13 regs) */
	0x054602a1,		/* tcm.ccfc_init_crd .. tcm.fic_init_crd (5 regs) */
	0x114602a9,		/* tcm.dir_byp_msg_cnt .. tcm.tm_msg_cntr (17 regs) */
	0x0a4602bb,		/* tcm.is_qm_p_fill_lvl .. tcm.is_prs_fill_lvl (10 regs) */
	0x074602d1,		/* tcm.fic_msg_cntr .. tcm.tcfc_cntr (7 regs) */
	0x084602e1,		/* tcm.eco_reserved .. tcm.is_foc_tsdm_nxt_inf_unit (8 regs) */
	0x10190000,		/* block mcm */
	0x01480000,		/* mcm.init (1 regs) */
	0x05480010,		/* mcm.dbg_select .. mcm.dbg_force_frame (5 regs) */
	0x02480060,		/* mcm.INT_STS_0 .. mcm.INT_MASK_0 (2 regs) */
	0x02480064,		/* mcm.INT_STS_1 .. mcm.INT_MASK_1 (2 regs) */
	0x02480068,		/* mcm.INT_STS_2 .. mcm.INT_MASK_2 (2 regs) */
	0x5a480100,		/* mcm.ifen .. mcm.err_evnt_id (90 regs) */
	0x23480181,		/* mcm.storm_weight .. mcm.ia_trans_part_fill_lvl (35 regs) */
	0x284801c1,		/* mcm.xx_msg_up_bnd .. mcm.unlock_miss (40 regs) */
	0x39480201,		/* mcm.prcs_agg_con_curr_st .. mcm.sm_task_ctx_size (57 regs) */
	0x12480245,		/* mcm.agg_con_cf0_q .. mcm.agg_task_rule6_q (18 regs) */
	0x0b480281,		/* mcm.in_prcs_tbl_crd_agg .. mcm.xx_byp_task_state_evnt_id_flg (11 regs) */
	0x064802a1,		/* mcm.ccfc_init_crd .. mcm.fic_init_crd (6 regs) */
	0x124802a9,		/* mcm.dir_byp_msg_cnt .. mcm.qm_s_msg_cntr (18 regs) */
	0x0a4802bc,		/* mcm.is_qm_p_fill_lvl .. mcm.is_pbf_fill_lvl (10 regs) */
	0x074802d1,		/* mcm.fic_msg_cntr .. mcm.tcfc_cntr (7 regs) */
	0x094802e1,		/* mcm.eco_reserved .. mcm.is_foc_tmld_nxt_inf_unit (9 regs) */
	0x101a0000,		/* block ucm */
	0x014a0000,		/* ucm.init (1 regs) */
	0x064a0013,		/* ucm.memctrl_status .. ucm.dbg_force_frame (6 regs) */
	0x024a0060,		/* ucm.INT_STS_0 .. ucm.INT_MASK_0 (2 regs) */
	0x024a0064,		/* ucm.INT_STS_1 .. ucm.INT_MASK_1 (2 regs) */
	0x024a0068,		/* ucm.INT_STS_2 .. ucm.INT_MASK_2 (2 regs) */
	0x5a4a0100,		/* ucm.ifen .. ucm.err_evnt_id (90 regs) */
	0x294a0181,		/* ucm.storm_weight .. ucm.ia_trans_part_fill_lvl (41 regs) */
	0x284a01c1,		/* ucm.xx_msg_up_bnd .. ucm.unlock_miss (40 regs) */
	0x394a0201,		/* ucm.prcs_agg_con_curr_st .. ucm.sm_task_ctx_size (57 regs) */
	0x1c4a0245,		/* ucm.agg_con_cf0_q .. ucm.agg_task_rule6_q (28 regs) */
	0x0d4a0281,		/* ucm.in_prcs_tbl_crd_agg .. ucm.xx_byp_task_state_evnt_id_flg (13 regs) */
	0x054a02a1,		/* ucm.ccfc_init_crd .. ucm.fic_init_crd (5 regs) */
	0x194a02a9,		/* ucm.dir_byp_msg_cnt .. ucm.tm_msg_cntr (25 regs) */
	0x0d4a02c3,		/* ucm.is_qm_p_fill_lvl .. ucm.is_pbf_fill_lvl (13 regs) */
	0x094a02d1,		/* ucm.fic_msg_cntr .. ucm.tcfc_cntr (9 regs) */
	0x0b4a02e1,		/* ucm.eco_reserved .. ucm.is_foc_yuld_nxt_inf_unit (11 regs) */
	0x111b0000,		/* block xcm */
	0x01400000,		/* xcm.init (1 regs) */
	0x01400002,		/* xcm.qm_act_st_cnt_init_done (1 regs) */
	0x05400010,		/* xcm.dbg_select .. xcm.dbg_force_frame (5 regs) */
	0x02400060,		/* xcm.INT_STS_0 .. xcm.INT_MASK_0 (2 regs) */
	0x02400064,		/* xcm.INT_STS_1 .. xcm.INT_MASK_1 (2 regs) */
	0x02400068,		/* xcm.INT_STS_2 .. xcm.INT_MASK_2 (2 regs) */
	0x32400100,		/* xcm.ifen .. xcm.err_evnt_id (50 regs) */
	0x25400181,		/* xcm.storm_weight .. xcm.ia_trans_part_fill_lvl (37 regs) */
	0x254001c1,		/* xcm.xx_msg_up_bnd .. xcm.unlock_miss (37 regs) */
	0x1e400201,		/* xcm.prcs_agg_con_curr_st .. xcm.sm_con_ctx_size (30 regs) */
	0x32400242,		/* xcm.agg_con_cf0_q .. xcm.agg_con_rule25_q (50 regs) */
	0x09400281,		/* xcm.in_prcs_tbl_crd_agg .. xcm.xx_byp_con_state_evnt_id_flg (9 regs) */
	0x054002a1,		/* xcm.ccfc_init_crd .. xcm.fic_init_crd (5 regs) */
	0x164002a9,		/* xcm.dir_byp_msg_cnt .. xcm.tm_msg_cntr (22 regs) */
	0x0d4002c0,		/* xcm.is_qm_p_fill_lvl .. xcm.is_pbf_fill_lvl (13 regs) */
	0x0b4002d1,		/* xcm.qm_act_st_fifo_fill_lvl .. xcm.ccfc_cntr (11 regs) */
	0x0b4002e1,		/* xcm.eco_reserved .. xcm.is_foc_ysdm_nxt_inf_unit (11 regs) */
	0x101c0000,		/* block ycm */
	0x01420000,		/* ycm.init (1 regs) */
	0x05420010,		/* ycm.dbg_select .. ycm.dbg_force_frame (5 regs) */
	0x02420060,		/* ycm.INT_STS_0 .. ycm.INT_MASK_0 (2 regs) */
	0x02420064,		/* ycm.INT_STS_1 .. ycm.INT_MASK_1 (2 regs) */
	0x02420068,		/* ycm.INT_STS_2 .. ycm.INT_MASK_2 (2 regs) */
	0x5a420100,		/* ycm.ifen .. ycm.err_evnt_id (90 regs) */
	0x21420181,		/* ycm.storm_weight .. ycm.ia_trans_part_fill_lvl (33 regs) */
	0x284201c1,		/* ycm.xx_msg_up_bnd .. ycm.unlock_miss (40 regs) */
	0x39420201,		/* ycm.prcs_agg_con_curr_st .. ycm.sm_task_ctx_size (57 regs) */
	0x11420245,		/* ycm.agg_con_cf0_q .. ycm.agg_task_rule6_q (17 regs) */
	0x0b420281,		/* ycm.in_prcs_tbl_crd_agg .. ycm.xx_byp_task_state_evnt_id_flg (11 regs) */
	0x064202a1,		/* ycm.ccfc_init_crd .. ycm.fic_init_crd (6 regs) */
	0x104202a9,		/* ycm.dir_byp_msg_cnt .. ycm.qm_s_msg_cntr (16 regs) */
	0x094202ba,		/* ycm.is_qm_p_fill_lvl .. ycm.is_pbf_fill_lvl (9 regs) */
	0x084202d1,		/* ycm.fic_msg_cntr .. ycm.tcfc_cntr (8 regs) */
	0x084202e1,		/* ycm.eco_reserved .. ycm.is_foc_xyld_nxt_inf_unit (8 regs) */
	0x101d0000,		/* block pcm */
	0x01440000,		/* pcm.init (1 regs) */
	0x05440010,		/* pcm.dbg_select .. pcm.dbg_force_frame (5 regs) */
	0x02440060,		/* pcm.INT_STS_0 .. pcm.INT_MASK_0 (2 regs) */
	0x02440064,		/* pcm.INT_STS_1 .. pcm.INT_MASK_1 (2 regs) */
	0x02440068,		/* pcm.INT_STS_2 .. pcm.INT_MASK_2 (2 regs) */
	0x02440100,		/* pcm.ifen .. pcm.err_evnt_id (2 regs) */
	0x13440181,		/* pcm.storm_weight .. pcm.ia_trans_part_fill_lvl (19 regs) */
	0x184401c1,		/* pcm.xx_msg_up_bnd .. pcm.unlock_miss (24 regs) */
	0x0f440201,		/* pcm.prcs_sm_con_curr_st .. pcm.sm_con_ctx_size (15 regs) */
	0x07440281,		/* pcm.in_prcs_tbl_crd_agg .. pcm.xx_byp_con_state_evnt_id_flg (7 regs) */
	0x024402a1,		/* pcm.ccfc_init_crd .. pcm.fic_init_crd (2 regs) */
	0x084402a9,		/* pcm.dir_byp_msg_cnt .. pcm.pbf_msg_cntr (8 regs) */
	0x034402b2,		/* pcm.is_storm_fill_lvl .. pcm.is_pbf_fill_lvl (3 regs) */
	0x024402d1,		/* pcm.fic_msg_cntr .. pcm.ccfc_cntr (2 regs) */
	0x044402e1,		/* pcm.eco_reserved .. pcm.is_foc_psdm_nxt_inf_unit (4 regs) */
	0x02440510,		/* pcm.ctx_rbc_accs .. pcm.sm_con_ctx (2 regs) */
	0x1e1e0000,		/* block qm */
	0x020bc060,		/* qm.INT_STS .. qm.INT_MASK (2 regs) */
	0x0d0bc100,		/* qm.wrc_drop_cnt_0 .. qm.cm_push_int_en (13 regs) */
	0x380bc110,		/* qm.MaxPqSizeTxSel_0 .. qm.MaxPqSizeTxSel_55 (56 regs) */
	0x040bc200,		/* qm.OutLdReqSizeConnTx .. qm.OutLdReqCrdConnOther (4 regs) */
	0x0e0bc410,		/* qm.QstatusTx_0 .. qm.QstatusTx_13 (14 regs) */
	0x020bc430,		/* qm.QstatusOther_0 .. qm.QstatusOther_1 (2 regs) */
	0x280bc448,		/* qm.CtxRegCcfc_0 .. qm.CtxRegCcfc_39 (40 regs) */
	0x280bc488,		/* qm.CtxRegTcfc_0 .. qm.CtxRegTcfc_39 (40 regs) */
	0x280bc4c8,		/* qm.ActCtrInitValCcfc_0 .. qm.ActCtrInitValCcfc_39 (40 regs) */
	0x280bc508,		/* qm.ActCtrInitValTcfc_0 .. qm.ActCtrInitValTcfc_39 (40 regs) */
	0x040bc548,		/* qm.PciReqQId .. qm.QmPageSize (4 regs) */
	0x050bc54d,		/* qm.PciReqPadToCacheLine .. qm.OvfErrorOther (5 regs) */
	0x010bc580,		/* qm.VoqCrdLineFull (1 regs) */
	0x010bc5c0,		/* qm.TaskLineCrdCost (1 regs) */
	0x010bc600,		/* qm.VoqCrdByteFull (1 regs) */
	0x220bc640,		/* qm.TaskByteCrdCost_0 .. qm.WrrOtherPqGrp_7 (34 regs) */
	0x040bc67a,		/* qm.WrrOtherGrpWeight_0 .. qm.WrrOtherGrpWeight_3 (4 regs) */
	0x170bc682,		/* qm.WrrTxGrpWeight_0 .. qm.CmIntEn (23 regs) */
	0x010bc780,		/* qm.VoqByteCrdEnable (1 regs) */
	0x040bc900,		/* qm.MHQTxNumSel .. qm.QOtherLevelMHVal (4 regs) */
	0x390bcb00,		/* qm.Soft_Reset .. qm.PqTx2Pf_55 (57 regs) */
	0x080bcb81,		/* qm.PqOther2Pf_0 .. qm.PqOther2Pf_7 (8 regs) */
	0x090bcb99,		/* qm.arb_tx_en .. qm.dbg_force_frame (9 regs) */
	0x020bcba6,		/* qm.eco_reserved .. qm.TxPqMap_MaskAccess (2 regs) */
	0x0f0bcbaf,		/* qm.Xsdm_Fifo_Full_Thr .. qm.RlGlblPeriodSel_7 (15 regs) */
	0x090bd300,		/* qm.RlGlblEnable .. qm.RlPfPeriodTimer (9 regs) */
	0x060bd380,		/* qm.RlPfEnable .. qm.Err_Mask_RlPfCrd (6 regs) */
	0x130bd700,		/* qm.WfqPfEnable .. qm.Voq_Arb_Grp0_Weight_7 (19 regs) */
	0x080bd72b,		/* qm.Voq_Arb_Grp1_Weight_0 .. qm.Voq_Arb_Grp1_Weight_7 (8 regs) */
	0x200bd74b,		/* qm.Voq_Arb_Timeout .. qm.cam_bist_status (32 regs) */
	0x14200000,		/* block dorq */
	0x01040000,		/* dorq.INIT (1 regs) */
	0x01040010,		/* dorq.ifen (1 regs) */
	0x02040060,		/* dorq.INT_STS .. dorq.INT_MASK (2 regs) */
	0x27040118,		/* dorq.dems_target_1 .. dorq.cm_ac_upd (39 regs) */
	0x3d040180,		/* dorq.dpm_l2_succ_cflg_cmd .. dorq.dpm_xcm_event_id_7 (61 regs) */
	0x0d0401be,		/* dorq.xcm_agg_type .. dorq.qm_byp_agg_ctx_size_7 (13 regs) */
	0x07040200,		/* dorq.xcm_ccfc_regn .. dorq.dpm_xcm_db_abrt_th (7 regs) */
	0x0a040208,		/* dorq.dpm_ent_abrt_th .. dorq.dq_full_cycles (10 regs) */
	0x08040229,		/* dorq.grh_nxt_header .. dorq.tag4_ovrd_mode (8 regs) */
	0x01040233,		/* dorq.rroce_dst_udp_port (1 regs) */
	0x0b040240,		/* dorq.l2_edpm_num_bd_thr .. dorq.l2_edpm_pkt_hdr_size (11 regs) */
	0x04040260,		/* dorq.xcm_msg_init_crd .. dorq.pbf_cmd_init_crd (4 regs) */
	0x03040277,		/* dorq.db_drop_reason_mask .. dorq.auto_freeze_st (3 regs) */
	0x0204027b,		/* dorq.auto_drop_en .. dorq.auto_drop_st (2 regs) */
	0x0c04027e,		/* dorq.pxp_trans_size .. dorq.db_drop_details (12 regs) */
	0x0704028b,		/* dorq.db_drop_reason .. dorq.dpm_abort_details_reason (7 regs) */
	0x11040293,		/* dorq.dpm_abort_reason .. dorq.cfc_bypass_cnt (17 regs) */
	0x048402a4,		/* dorq.mini_cache_entry .. dorq.cfc_lcres_err_detail (4 regs, WB) */
	0x030402a8,		/* dorq.cfc_ld_req_cnt .. dorq.eco_reserved (3 regs) */
	0x060402b3,		/* dorq.memctrl_status .. dorq.dbg_force_frame (6 regs) */
	0x33210000,		/* block brb */
	0x030d0001,		/* brb.hw_init_en .. brb.start_en (3 regs) */
	0x020d0030,		/* brb.INT_STS_0 .. brb.INT_MASK_0 (2 regs) */
	0x020d0036,		/* brb.INT_STS_1 .. brb.INT_MASK_1 (2 regs) */
	0x020d003c,		/* brb.INT_STS_2 .. brb.INT_MASK_2 (2 regs) */
	0x020d0042,		/* brb.INT_STS_3 .. brb.INT_MASK_3 (2 regs) */
	0x020d0048,		/* brb.INT_STS_4 .. brb.INT_MASK_4 (2 regs) */
	0x020d004e,		/* brb.INT_STS_5 .. brb.INT_MASK_5 (2 regs) */
	0x020d0054,		/* brb.INT_STS_6 .. brb.INT_MASK_6 (2 regs) */
	0x020d005a,		/* brb.INT_STS_7 .. brb.INT_MASK_7 (2 regs) */
	0x020d0061,		/* brb.INT_STS_8 .. brb.INT_MASK_8 (2 regs) */
	0x020d0067,		/* brb.INT_STS_9 .. brb.INT_MASK_9 (2 regs) */
	0x020d006d,		/* brb.INT_STS_10 .. brb.INT_MASK_10 (2 regs) */
	0x020d0073,		/* brb.INT_STS_11 .. brb.INT_MASK_11 (2 regs) */
	0x020d0200,		/* brb.big_ram_address .. brb.header_size (2 regs) */
	0x020d0210,		/* brb.max_releases .. brb.stop_on_len_err (2 regs) */
	0x120d0240,		/* brb.tc_guarantied_0 .. brb.tc_guarantied_17 (18 regs) */
	0x100d025e,		/* brb.main_tc_guarantied_hyst_0 .. brb.main_tc_guarantied_hyst_15 (16 regs) */
	0x120d0276,		/* brb.lb_tc_guarantied_hyst_0 .. brb.lb_tc_guarantied_hyst_17 (18 regs) */
	0x100d0294,		/* brb.main_tc_pause_xoff_threshold_0 .. brb.main_tc_pause_xoff_threshold_15 (16 regs) */
	0x120d02ac,		/* brb.lb_tc_pause_xoff_threshold_0 .. brb.lb_tc_pause_xoff_threshold_17 (18 regs) */
	0x100d02ca,		/* brb.main_tc_pause_xon_threshold_0 .. brb.main_tc_pause_xon_threshold_15 (16 regs) */
	0x120d02e2,		/* brb.lb_tc_pause_xon_threshold_0 .. brb.lb_tc_pause_xon_threshold_17 (18 regs) */
	0x100d0300,		/* brb.main_tc_full_xoff_threshold_0 .. brb.main_tc_full_xoff_threshold_15 (16 regs) */
	0x120d0318,		/* brb.lb_tc_full_xoff_threshold_0 .. brb.lb_tc_full_xoff_threshold_17 (18 regs) */
	0x100d0336,		/* brb.main_tc_full_xon_threshold_0 .. brb.main_tc_full_xon_threshold_15 (16 regs) */
	0x120d034e,		/* brb.lb_tc_full_xon_threshold_0 .. brb.lb_tc_full_xon_threshold_17 (18 regs) */
	0x080d036c,		/* brb.lossless_threshold .. brb.rc_pkt_priority (8 regs) */
	0x110d0382,		/* brb.rc_sop_priority .. brb.pm_tc_latency_sensitive_1 (17 regs) */
	0x080d03b1,		/* brb.dbgsyn_almost_full_thr .. brb.dbg_force_frame (8 regs) */
	0x060d03ca,		/* brb.inp_if_enable .. brb.wc_empty_3 (6 regs) */
	0x040d03dc,		/* brb.wc_full_0 .. brb.wc_full_3 (4 regs) */
	0x070d03ec,		/* brb.wc_bandwidth_if_full .. brb.rc_pkt_empty_4 (7 regs) */
	0x050d03fd,		/* brb.rc_pkt_full_0 .. brb.rc_pkt_full_4 (5 regs) */
	0x050d040c,		/* brb.rc_pkt_status_0 .. brb.rc_pkt_status_4 (5 regs) */
	0x0b0d041b,		/* brb.rc_sop_empty .. brb.empty_if_1 (11 regs) */
	0x050d042a,		/* brb.rc_sop_inp_sync_fifo_push_status .. brb.rc_inp_sync_fifo_push_status_3 (5 regs) */
	0x050d043a,		/* brb.rc_out_sync_fifo_push_status_0 .. brb.rc_out_sync_fifo_push_status_4 (5 regs) */
	0x020d0449,		/* brb.rc_eop_inp_sync_fifo_push_status_0 .. brb.rc_eop_inp_sync_fifo_push_status_1 (2 regs) */
	0x020d0458,		/* brb.rc_eop_out_sync_fifo_push_status_0 .. brb.rc_eop_out_sync_fifo_push_status_1 (2 regs) */
	0x040d0467,		/* brb.pkt_avail_sync_fifo_push_status .. brb.rc_pkt_state (4 regs) */
	0x020d046e,		/* brb.mac_free_shared_hr_0 .. brb.mac_free_shared_hr_1 (2 regs) */
	0x090d0474,		/* brb.mac0_tc_occupancy_0 .. brb.mac0_tc_occupancy_8 (9 regs) */
	0x090d0484,		/* brb.mac1_tc_occupancy_0 .. brb.mac1_tc_occupancy_8 (9 regs) */
	0x020d04b4,		/* brb.available_mac_size_0 .. brb.available_mac_size_1 (2 regs) */
	0x020d04ba,		/* brb.main_tc_pause_0 .. brb.main_tc_pause_1 (2 regs) */
	0x020d04c0,		/* brb.lb_tc_pause_0 .. brb.lb_tc_pause_1 (2 regs) */
	0x020d04c6,		/* brb.main_tc_full_0 .. brb.main_tc_full_1 (2 regs) */
	0x020d04cc,		/* brb.lb_tc_full_0 .. brb.lb_tc_full_1 (2 regs) */
	0x080d04d2,		/* brb.main0_tc_lossless_cnt_0 .. brb.main0_tc_lossless_cnt_7 (8 regs) */
	0x080d04e2,		/* brb.main1_tc_lossless_cnt_0 .. brb.main1_tc_lossless_cnt_7 (8 regs) */
	0x020d0512,		/* brb.main_tc_lossless_int_0 .. brb.main_tc_lossless_int_1 (2 regs) */
	0x26230000,		/* block prs */
	0x0107c000,		/* prs.soft_rst (1 regs) */
	0x0307c002,		/* prs.mac_vlan_cache_init_done .. prs.cam_scrub_miss_en (3 regs) */
	0x0207c010,		/* prs.INT_STS_0 .. prs.INT_MASK_0 (2 regs) */
	0x1a07c040,		/* prs.packet_region_0 .. prs.search_resp_initiator_type (26 regs) */
	0x0607c05e,		/* prs.task_id_segment .. prs.roce_con_type (6 regs) */
	0x0107c065,		/* prs.roce_opcode_req_res (1 regs) */
	0x0107c067,		/* prs.cfc_load_mini_cache_en (1 regs) */
	0x0107c080,		/* prs.eco_reserved (1 regs) */
	0x0107c104,		/* prs.search_tcp_first_frag (1 regs) */
	0x0107c10a,		/* prs.roce_spcl_qp_val (1 regs) */
	0x0907c114,		/* prs.tenant_id_default_val_enable .. prs.tenant_id_default_val_ttag (9 regs) */
	0x0407c11e,		/* prs.tenant_id_mask_eth_nge .. prs.tenant_id_default_val_ip_nge (4 regs) */
	0x0407c140,		/* prs.ports_arb_scheme .. prs.max_packet_size (4 regs) */
	0x0b07c1c1,		/* prs.llc_type_threshold .. prs.icmpv4_protocol (11 regs) */
	0x0107c1cd,		/* prs.gre_protocol (1 regs) */
	0x0e07c1d1,		/* prs.fcoe_type .. prs.tag_len_5 (14 regs) */
	0x0407c1ef,		/* prs.dst_mac_global_0 .. prs.dst_mac_global_mask_1 (4 regs) */
	0x0107c21a,		/* prs.nge_eth_type (1 regs) */
	0x0107c21c,		/* prs.rroce_port (1 regs) */
	0x1007c240,		/* prs.l2_irreg_cases .. prs.def_l2_con_type (16 regs) */
	0x0607c254,		/* prs.light_l2_ethertype_0 .. prs.light_l2_ethertype_5 (6 regs) */
	0x1307c25c,		/* prs.dst_mac_select .. prs.output_format_7_1 (19 regs) */
	0x0407c270,		/* prs.mac_vlan_flex_upper .. prs.mac_vlan_flex_bitmask_1 (4 regs) */
	0x0107c275,		/* prs.sack_blk_override (1 regs) */
	0x0807c277,		/* prs.rdma_syn_seed_0 .. prs.rdma_syn_seed_7 (8 regs) */
	0x0207c2c0,		/* prs.num_of_cfc_flush_messages .. prs.num_of_transparent_flush_messages (2 regs) */
	0x0887c2cc,		/* prs.fifo_empty_flags .. prs.fifo_full_flags (8 regs, WB) */
	0x0107c2d4,		/* prs.prs_pkt_ct (1 regs) */
	0x0107c2d7,		/* prs.stop_parsing_status (1 regs) */
	0x0287c2d8,		/* prs.mini_cache_entry (2 regs, WB) */
	0x0407c2da,		/* prs.mini_cache_failed_response .. prs.dbg_shift (4 regs) */
	0x0207c2e8,		/* prs.dbg_force_valid .. prs.dbg_force_frame (2 regs) */
	0x0307c2ec,		/* prs.fc_dbg_select .. prs.fc_dbg_shift (3 regs) */
	0x0887c2f0,		/* prs.fc_dbg_out_data (8 regs, WB) */
	0x0407c2f8,		/* prs.fc_dbg_force_valid .. prs.fc_dbg_out_frame (4 regs) */
	0x0d07c3c0,		/* prs.tcm_initial_credit .. prs.sop_req_ct (13 regs) */
	0x0f07c460,		/* prs.gft_hash_key_0 .. prs.gft_tunnel_vlan_select (15 regs) */
	0x0407c471,		/* prs.gft_connection_type .. prs.gft_cam_scrub_miss_en (4 regs) */
	0x0a240000,		/* block tsdm */
	0x053ec001,		/* tsdm.enable_in1 .. tsdm.disable_engine (5 regs) */
	0x023ec010,		/* tsdm.INT_STS .. tsdm.INT_MASK (2 regs) */
	0x023ec100,		/* tsdm.timer_tick .. tsdm.timers_tick_enable (2 regs) */
	0x093ec103,		/* tsdm.grc_privilege_level .. tsdm.eco_reserved (9 regs) */
	0x053ec140,		/* tsdm.init_credit_pxp .. tsdm.init_credit_cm (5 regs) */
	0x0c3ec180,		/* tsdm.num_of_dma_cmd .. tsdm.num_of_dpm_req (12 regs) */
	0x033ec1c0,		/* tsdm.brb_almost_full .. tsdm.dorq_almost_full (3 regs) */
	0x203ec300,		/* tsdm.queue_full .. tsdm.prm_fifo_full (32 regs) */
	0x1a3ec340,		/* tsdm.int_cmpl_pend_empty .. tsdm.prm_fifo_empty (26 regs) */
	0x053ec38a,		/* tsdm.dbg_select .. tsdm.dbg_force_frame (5 regs) */
	0x0a250000,		/* block msdm */
	0x053f0001,		/* msdm.enable_in1 .. msdm.disable_engine (5 regs) */
	0x023f0010,		/* msdm.INT_STS .. msdm.INT_MASK (2 regs) */
	0x023f0100,		/* msdm.timer_tick .. msdm.timers_tick_enable (2 regs) */
	0x093f0103,		/* msdm.grc_privilege_level .. msdm.eco_reserved (9 regs) */
	0x053f0140,		/* msdm.init_credit_pxp .. msdm.init_credit_cm (5 regs) */
	0x0c3f0180,		/* msdm.num_of_dma_cmd .. msdm.num_of_dpm_req (12 regs) */
	0x033f01c0,		/* msdm.brb_almost_full .. msdm.dorq_almost_full (3 regs) */
	0x203f0300,		/* msdm.queue_full .. msdm.prm_fifo_full (32 regs) */
	0x1a3f0340,		/* msdm.int_cmpl_pend_empty .. msdm.prm_fifo_empty (26 regs) */
	0x053f038a,		/* msdm.dbg_select .. msdm.dbg_force_frame (5 regs) */
	0x0a260000,		/* block usdm */
	0x053f4001,		/* usdm.enable_in1 .. usdm.disable_engine (5 regs) */
	0x023f4010,		/* usdm.INT_STS .. usdm.INT_MASK (2 regs) */
	0x023f4100,		/* usdm.timer_tick .. usdm.timers_tick_enable (2 regs) */
	0x093f4103,		/* usdm.grc_privilege_level .. usdm.eco_reserved (9 regs) */
	0x053f4140,		/* usdm.init_credit_pxp .. usdm.init_credit_cm (5 regs) */
	0x0c3f4180,		/* usdm.num_of_dma_cmd .. usdm.num_of_dpm_req (12 regs) */
	0x033f41c0,		/* usdm.brb_almost_full .. usdm.dorq_almost_full (3 regs) */
	0x203f4300,		/* usdm.queue_full .. usdm.prm_fifo_full (32 regs) */
	0x1a3f4340,		/* usdm.int_cmpl_pend_empty .. usdm.prm_fifo_empty (26 regs) */
	0x053f438a,		/* usdm.dbg_select .. usdm.dbg_force_frame (5 regs) */
	0x0b270000,		/* block xsdm */
	0x053e0001,		/* xsdm.enable_in1 .. xsdm.disable_engine (5 regs) */
	0x023e0010,		/* xsdm.INT_STS .. xsdm.INT_MASK (2 regs) */
	0x023e0100,		/* xsdm.timer_tick .. xsdm.timers_tick_enable (2 regs) */
	0x093e0103,		/* xsdm.grc_privilege_level .. xsdm.eco_reserved (9 regs) */
	0x053e0140,		/* xsdm.init_credit_pxp .. xsdm.init_credit_cm (5 regs) */
	0x013e0148,		/* xsdm.init_credit_cm_rmt (1 regs) */
	0x0c3e0180,		/* xsdm.num_of_dma_cmd .. xsdm.num_of_dpm_req (12 regs) */
	0x033e01c0,		/* xsdm.brb_almost_full .. xsdm.dorq_almost_full (3 regs) */
	0x203e0300,		/* xsdm.queue_full .. xsdm.prm_fifo_full (32 regs) */
	0x1a3e0340,		/* xsdm.int_cmpl_pend_empty .. xsdm.prm_fifo_empty (26 regs) */
	0x053e038a,		/* xsdm.dbg_select .. xsdm.dbg_force_frame (5 regs) */
	0x0a280000,		/* block ysdm */
	0x053e4001,		/* ysdm.enable_in1 .. ysdm.disable_engine (5 regs) */
	0x023e4010,		/* ysdm.INT_STS .. ysdm.INT_MASK (2 regs) */
	0x023e4100,		/* ysdm.timer_tick .. ysdm.timers_tick_enable (2 regs) */
	0x093e4103,		/* ysdm.grc_privilege_level .. ysdm.eco_reserved (9 regs) */
	0x053e4140,		/* ysdm.init_credit_pxp .. ysdm.init_credit_cm (5 regs) */
	0x0c3e4180,		/* ysdm.num_of_dma_cmd .. ysdm.num_of_dpm_req (12 regs) */
	0x033e41c0,		/* ysdm.brb_almost_full .. ysdm.dorq_almost_full (3 regs) */
	0x203e4300,		/* ysdm.queue_full .. ysdm.prm_fifo_full (32 regs) */
	0x1a3e4340,		/* ysdm.int_cmpl_pend_empty .. ysdm.prm_fifo_empty (26 regs) */
	0x053e438a,		/* ysdm.dbg_select .. ysdm.dbg_force_frame (5 regs) */
	0x0a290000,		/* block psdm */
	0x053e8001,		/* psdm.enable_in1 .. psdm.disable_engine (5 regs) */
	0x023e8010,		/* psdm.INT_STS .. psdm.INT_MASK (2 regs) */
	0x023e8100,		/* psdm.timer_tick .. psdm.timers_tick_enable (2 regs) */
	0x093e8103,		/* psdm.grc_privilege_level .. psdm.eco_reserved (9 regs) */
	0x053e8140,		/* psdm.init_credit_pxp .. psdm.init_credit_cm (5 regs) */
	0x0c3e8180,		/* psdm.num_of_dma_cmd .. psdm.num_of_dpm_req (12 regs) */
	0x033e81c0,		/* psdm.brb_almost_full .. psdm.dorq_almost_full (3 regs) */
	0x203e8300,		/* psdm.queue_full .. psdm.prm_fifo_full (32 regs) */
	0x1a3e8340,		/* psdm.int_cmpl_pend_empty .. psdm.prm_fifo_empty (26 regs) */
	0x053e838a,		/* psdm.dbg_select .. psdm.dbg_force_frame (5 regs) */
	0x282a0000,		/* block tsem */
	0x045c0001,		/* tsem.enable_in .. tsem.pas_disable (4 regs) */
	0x025c0010,		/* tsem.INT_STS_0 .. tsem.INT_MASK_0 (2 regs) */
	0x025c0014,		/* tsem.INT_STS_1 .. tsem.INT_MASK_1 (2 regs) */
	0x015c0100,		/* tsem.arb_cycle_size (1 regs) */
	0x085c0110,		/* tsem.pf_err_vector .. tsem.eco_reserved (8 regs) */
	0x015c0180,		/* tsem.fic_min_msg (1 regs) */
	0x025c0188,		/* tsem.fic_empty_ct_mode .. tsem.fic_empty_ct_cnt (2 regs) */
	0x025c01a0,		/* tsem.foc_credit (2 regs) */
	0x015c01b0,		/* tsem.full_foc_dra_strt_en (1 regs) */
	0x10dc01c0,		/* tsem.fin_command (16 regs, WB) */
	0x015c0240,		/* tsem.invld_pas_wr_en (1 regs) */
	0x035c0260,		/* tsem.arbiter_request .. tsem.arbiter_slot (3 regs) */
	0x035c02c0,		/* tsem.num_of_threads .. tsem.thread_rdy (3 regs) */
	0x015c02c5,		/* tsem.threads_list (1 regs) */
	0x015c0380,		/* tsem.order_pop_en (1 regs) */
	0x015c0382,		/* tsem.order_wake_en (1 regs) */
	0x015c0384,		/* tsem.pf_num_order_base (1 regs) */
	0x045c0400,		/* tsem.dbg_alm_full .. tsem.sync_ram_wr_alm_full (4 regs) */
	0x025c0440,		/* tsem.dra_empty .. tsem.ext_pas_empty (2 regs) */
	0x015c0448,		/* tsem.fic_empty (1 regs) */
	0x0b5c0450,		/* tsem.slow_dbg_empty .. tsem.ord_id_fifo_empty (11 regs) */
	0x025c0480,		/* tsem.ext_pas_full .. tsem.ext_store_if_full (2 regs) */
	0x015c0488,		/* tsem.fic_full (1 regs) */
	0x0f5c0490,		/* tsem.pas_if_full .. tsem.ord_id_fifo_full (15 regs) */
	0x035c04c0,		/* tsem.thread_inter_cnt .. tsem.thread_orun_num (3 regs) */
	0x0a5c0500,		/* tsem.slow_dbg_active .. tsem.dbg_msg_src (10 regs) */
	0x055c054a,		/* tsem.dbg_select .. tsem.dbg_force_frame (5 regs) */
	0x015d0001,		/* tsem.fast_memory.ram_ext_disable (1 regs) */
	0x025d0010,		/* tsem.fast_memory.INT_STS .. tsem.fast_memory.INT_MASK (2 regs) */
	0x055d0120,		/* tsem.fast_memory.gpre0 .. tsem.fast_memory.stall_2 (5 regs) */
	0x045d0128,		/* tsem.fast_memory.storm_stack_size .. tsem.fast_memory.pram_prty_addr_high (4 regs) */
	0x025d012e,		/* tsem.fast_memory.port_id_width .. tsem.fast_memory.port_id_offset (2 regs) */
	0x055d0131,		/* tsem.fast_memory.state_machine .. tsem.fast_memory.storm_pc (5 regs) */
	0x0e5d018a,		/* tsem.fast_memory.rt_clk_enable .. tsem.fast_memory.cam_init_in_process (14 regs) */
	0x0d5d01d0,		/* tsem.fast_memory.debug_active .. tsem.fast_memory.dbg_store_addr_value (13 regs) */
	0x045d0210,		/* tsem.fast_memory.sync_dra_rd_alm_full .. tsem.fast_memory.dbg_alm_full (4 regs) */
	0x035d0250,		/* tsem.fast_memory.full .. tsem.fast_memory.alm_full (3 regs) */
	0x065d0290,		/* tsem.fast_memory.active_filter_enable .. tsem.fast_memory.idle_inactive_cycles (6 regs) */
	0x015d02d3,		/* tsem.fast_memory.vfc_status (1 regs) */
	0x045d0310,		/* tsem.fast_memory.cam_bist_en .. tsem.fast_memory.cam_bist_status (4 regs) */
	0x282b0000,		/* block msem */
	0x04600001,		/* msem.enable_in .. msem.pas_disable (4 regs) */
	0x02600010,		/* msem.INT_STS_0 .. msem.INT_MASK_0 (2 regs) */
	0x02600014,		/* msem.INT_STS_1 .. msem.INT_MASK_1 (2 regs) */
	0x01600100,		/* msem.arb_cycle_size (1 regs) */
	0x08600110,		/* msem.pf_err_vector .. msem.eco_reserved (8 regs) */
	0x01600180,		/* msem.fic_min_msg (1 regs) */
	0x02600188,		/* msem.fic_empty_ct_mode .. msem.fic_empty_ct_cnt (2 regs) */
	0x066001a0,		/* msem.foc_credit (6 regs) */
	0x016001b0,		/* msem.full_foc_dra_strt_en (1 regs) */
	0x10e001c0,		/* msem.fin_command (16 regs, WB) */
	0x01600240,		/* msem.invld_pas_wr_en (1 regs) */
	0x03600260,		/* msem.arbiter_request .. msem.arbiter_slot (3 regs) */
	0x036002c0,		/* msem.num_of_threads .. msem.thread_rdy (3 regs) */
	0x016002c5,		/* msem.threads_list (1 regs) */
	0x01600380,		/* msem.order_pop_en (1 regs) */
	0x01600382,		/* msem.order_wake_en (1 regs) */
	0x01600384,		/* msem.pf_num_order_base (1 regs) */
	0x04600400,		/* msem.dbg_alm_full .. msem.sync_ram_wr_alm_full (4 regs) */
	0x02600440,		/* msem.dra_empty .. msem.ext_pas_empty (2 regs) */
	0x01600448,		/* msem.fic_empty (1 regs) */
	0x0b600450,		/* msem.slow_dbg_empty .. msem.ord_id_fifo_empty (11 regs) */
	0x02600480,		/* msem.ext_pas_full .. msem.ext_store_if_full (2 regs) */
	0x01600488,		/* msem.fic_full (1 regs) */
	0x0f600490,		/* msem.pas_if_full .. msem.ord_id_fifo_full (15 regs) */
	0x036004c0,		/* msem.thread_inter_cnt .. msem.thread_orun_num (3 regs) */
	0x0a600500,		/* msem.slow_dbg_active .. msem.dbg_msg_src (10 regs) */
	0x0560054a,		/* msem.dbg_select .. msem.dbg_force_frame (5 regs) */
	0x01610001,		/* msem.fast_memory.ram_ext_disable (1 regs) */
	0x02610010,		/* msem.fast_memory.INT_STS .. msem.fast_memory.INT_MASK (2 regs) */
	0x05610120,		/* msem.fast_memory.gpre0 .. msem.fast_memory.stall_2 (5 regs) */
	0x04610128,		/* msem.fast_memory.storm_stack_size .. msem.fast_memory.pram_prty_addr_high (4 regs) */
	0x0261012e,		/* msem.fast_memory.port_id_width .. msem.fast_memory.port_id_offset (2 regs) */
	0x05610131,		/* msem.fast_memory.state_machine .. msem.fast_memory.storm_pc (5 regs) */
	0x0e61018a,		/* msem.fast_memory.rt_clk_enable .. msem.fast_memory.cam_init_in_process (14 regs) */
	0x0d6101d0,		/* msem.fast_memory.debug_active .. msem.fast_memory.dbg_store_addr_value (13 regs) */
	0x04610210,		/* msem.fast_memory.sync_dra_rd_alm_full .. msem.fast_memory.dbg_alm_full (4 regs) */
	0x03610250,		/* msem.fast_memory.full .. msem.fast_memory.alm_full (3 regs) */
	0x06610290,		/* msem.fast_memory.active_filter_enable .. msem.fast_memory.idle_inactive_cycles (6 regs) */
	0x016102d3,		/* msem.fast_memory.vfc_status (1 regs) */
	0x04610310,		/* msem.fast_memory.cam_bist_en .. msem.fast_memory.cam_bist_status (4 regs) */
	0x282c0000,		/* block usem */
	0x04640001,		/* usem.enable_in .. usem.pas_disable (4 regs) */
	0x02640010,		/* usem.INT_STS_0 .. usem.INT_MASK_0 (2 regs) */
	0x02640014,		/* usem.INT_STS_1 .. usem.INT_MASK_1 (2 regs) */
	0x01640100,		/* usem.arb_cycle_size (1 regs) */
	0x08640110,		/* usem.pf_err_vector .. usem.eco_reserved (8 regs) */
	0x01640180,		/* usem.fic_min_msg (1 regs) */
	0x02640188,		/* usem.fic_empty_ct_mode .. usem.fic_empty_ct_cnt (2 regs) */
	0x056401a0,		/* usem.foc_credit (5 regs) */
	0x016401b0,		/* usem.full_foc_dra_strt_en (1 regs) */
	0x10e401c0,		/* usem.fin_command (16 regs, WB) */
	0x01640240,		/* usem.invld_pas_wr_en (1 regs) */
	0x03640260,		/* usem.arbiter_request .. usem.arbiter_slot (3 regs) */
	0x036402c0,		/* usem.num_of_threads .. usem.thread_rdy (3 regs) */
	0x016402c5,		/* usem.threads_list (1 regs) */
	0x01640380,		/* usem.order_pop_en (1 regs) */
	0x01640382,		/* usem.order_wake_en (1 regs) */
	0x01640384,		/* usem.pf_num_order_base (1 regs) */
	0x04640400,		/* usem.dbg_alm_full .. usem.sync_ram_wr_alm_full (4 regs) */
	0x02640440,		/* usem.dra_empty .. usem.ext_pas_empty (2 regs) */
	0x01640448,		/* usem.fic_empty (1 regs) */
	0x0b640450,		/* usem.slow_dbg_empty .. usem.ord_id_fifo_empty (11 regs) */
	0x02640480,		/* usem.ext_pas_full .. usem.ext_store_if_full (2 regs) */
	0x01640488,		/* usem.fic_full (1 regs) */
	0x0f640490,		/* usem.pas_if_full .. usem.ord_id_fifo_full (15 regs) */
	0x036404c0,		/* usem.thread_inter_cnt .. usem.thread_orun_num (3 regs) */
	0x0a640500,		/* usem.slow_dbg_active .. usem.dbg_msg_src (10 regs) */
	0x0564054a,		/* usem.dbg_select .. usem.dbg_force_frame (5 regs) */
	0x01650001,		/* usem.fast_memory.ram_ext_disable (1 regs) */
	0x02650010,		/* usem.fast_memory.INT_STS .. usem.fast_memory.INT_MASK (2 regs) */
	0x05650120,		/* usem.fast_memory.gpre0 .. usem.fast_memory.stall_2 (5 regs) */
	0x04650128,		/* usem.fast_memory.storm_stack_size .. usem.fast_memory.pram_prty_addr_high (4 regs) */
	0x0265012e,		/* usem.fast_memory.port_id_width .. usem.fast_memory.port_id_offset (2 regs) */
	0x05650131,		/* usem.fast_memory.state_machine .. usem.fast_memory.storm_pc (5 regs) */
	0x0e65018a,		/* usem.fast_memory.rt_clk_enable .. usem.fast_memory.cam_init_in_process (14 regs) */
	0x0d6501d0,		/* usem.fast_memory.debug_active .. usem.fast_memory.dbg_store_addr_value (13 regs) */
	0x04650210,		/* usem.fast_memory.sync_dra_rd_alm_full .. usem.fast_memory.dbg_alm_full (4 regs) */
	0x03650250,		/* usem.fast_memory.full .. usem.fast_memory.alm_full (3 regs) */
	0x06650290,		/* usem.fast_memory.active_filter_enable .. usem.fast_memory.idle_inactive_cycles (6 regs) */
	0x016502d3,		/* usem.fast_memory.vfc_status (1 regs) */
	0x04650310,		/* usem.fast_memory.cam_bist_en .. usem.fast_memory.cam_bist_status (4 regs) */
	0x252d0000,		/* block xsem */
	0x04500001,		/* xsem.enable_in .. xsem.pas_disable (4 regs) */
	0x02500010,		/* xsem.INT_STS_0 .. xsem.INT_MASK_0 (2 regs) */
	0x02500014,		/* xsem.INT_STS_1 .. xsem.INT_MASK_1 (2 regs) */
	0x01500100,		/* xsem.arb_cycle_size (1 regs) */
	0x08500110,		/* xsem.pf_err_vector .. xsem.eco_reserved (8 regs) */
	0x02500188,		/* xsem.fic_empty_ct_mode .. xsem.fic_empty_ct_cnt (2 regs) */
	0x025001a0,		/* xsem.foc_credit (2 regs) */
	0x015001b0,		/* xsem.full_foc_dra_strt_en (1 regs) */
	0x10d001c0,		/* xsem.fin_command (16 regs, WB) */
	0x01500240,		/* xsem.invld_pas_wr_en (1 regs) */
	0x03500260,		/* xsem.arbiter_request .. xsem.arbiter_slot (3 regs) */
	0x035002c0,		/* xsem.num_of_threads .. xsem.thread_rdy (3 regs) */
	0x015002c5,		/* xsem.threads_list (1 regs) */
	0x01500380,		/* xsem.order_pop_en (1 regs) */
	0x01500382,		/* xsem.order_wake_en (1 regs) */
	0x01500384,		/* xsem.pf_num_order_base (1 regs) */
	0x04500400,		/* xsem.dbg_alm_full .. xsem.sync_ram_wr_alm_full (4 regs) */
	0x02500440,		/* xsem.dra_empty .. xsem.ext_pas_empty (2 regs) */
	0x0b500450,		/* xsem.slow_dbg_empty .. xsem.ord_id_fifo_empty (11 regs) */
	0x02500480,		/* xsem.ext_pas_full .. xsem.ext_store_if_full (2 regs) */
	0x0f500490,		/* xsem.pas_if_full .. xsem.ord_id_fifo_full (15 regs) */
	0x035004c0,		/* xsem.thread_inter_cnt .. xsem.thread_orun_num (3 regs) */
	0x0a500500,		/* xsem.slow_dbg_active .. xsem.dbg_msg_src (10 regs) */
	0x0550054a,		/* xsem.dbg_select .. xsem.dbg_force_frame (5 regs) */
	0x01510001,		/* xsem.fast_memory.ram_ext_disable (1 regs) */
	0x02510010,		/* xsem.fast_memory.INT_STS .. xsem.fast_memory.INT_MASK (2 regs) */
	0x05510120,		/* xsem.fast_memory.gpre0 .. xsem.fast_memory.stall_2 (5 regs) */
	0x04510128,		/* xsem.fast_memory.storm_stack_size .. xsem.fast_memory.pram_prty_addr_high (4 regs) */
	0x0251012e,		/* xsem.fast_memory.port_id_width .. xsem.fast_memory.port_id_offset (2 regs) */
	0x05510131,		/* xsem.fast_memory.state_machine .. xsem.fast_memory.storm_pc (5 regs) */
	0x0e51018a,		/* xsem.fast_memory.rt_clk_enable .. xsem.fast_memory.cam_init_in_process (14 regs) */
	0x0d5101d0,		/* xsem.fast_memory.debug_active .. xsem.fast_memory.dbg_store_addr_value (13 regs) */
	0x04510210,		/* xsem.fast_memory.sync_dra_rd_alm_full .. xsem.fast_memory.dbg_alm_full (4 regs) */
	0x03510250,		/* xsem.fast_memory.full .. xsem.fast_memory.alm_full (3 regs) */
	0x06510290,		/* xsem.fast_memory.active_filter_enable .. xsem.fast_memory.idle_inactive_cycles (6 regs) */
	0x015102d3,		/* xsem.fast_memory.vfc_status (1 regs) */
	0x04510310,		/* xsem.fast_memory.cam_bist_en .. xsem.fast_memory.cam_bist_status (4 regs) */
	0x252e0000,		/* block ysem */
	0x04540001,		/* ysem.enable_in .. ysem.pas_disable (4 regs) */
	0x02540010,		/* ysem.INT_STS_0 .. ysem.INT_MASK_0 (2 regs) */
	0x02540014,		/* ysem.INT_STS_1 .. ysem.INT_MASK_1 (2 regs) */
	0x01540100,		/* ysem.arb_cycle_size (1 regs) */
	0x08540110,		/* ysem.pf_err_vector .. ysem.eco_reserved (8 regs) */
	0x02540188,		/* ysem.fic_empty_ct_mode .. ysem.fic_empty_ct_cnt (2 regs) */
	0x065401a0,		/* ysem.foc_credit (6 regs) */
	0x015401b0,		/* ysem.full_foc_dra_strt_en (1 regs) */
	0x10d401c0,		/* ysem.fin_command (16 regs, WB) */
	0x01540240,		/* ysem.invld_pas_wr_en (1 regs) */
	0x03540260,		/* ysem.arbiter_request .. ysem.arbiter_slot (3 regs) */
	0x035402c0,		/* ysem.num_of_threads .. ysem.thread_rdy (3 regs) */
	0x015402c5,		/* ysem.threads_list (1 regs) */
	0x01540380,		/* ysem.order_pop_en (1 regs) */
	0x01540382,		/* ysem.order_wake_en (1 regs) */
	0x01540384,		/* ysem.pf_num_order_base (1 regs) */
	0x04540400,		/* ysem.dbg_alm_full .. ysem.sync_ram_wr_alm_full (4 regs) */
	0x02540440,		/* ysem.dra_empty .. ysem.ext_pas_empty (2 regs) */
	0x0b540450,		/* ysem.slow_dbg_empty .. ysem.ord_id_fifo_empty (11 regs) */
	0x02540480,		/* ysem.ext_pas_full .. ysem.ext_store_if_full (2 regs) */
	0x0f540490,		/* ysem.pas_if_full .. ysem.ord_id_fifo_full (15 regs) */
	0x035404c0,		/* ysem.thread_inter_cnt .. ysem.thread_orun_num (3 regs) */
	0x0a540500,		/* ysem.slow_dbg_active .. ysem.dbg_msg_src (10 regs) */
	0x0554054a,		/* ysem.dbg_select .. ysem.dbg_force_frame (5 regs) */
	0x01550001,		/* ysem.fast_memory.ram_ext_disable (1 regs) */
	0x02550010,		/* ysem.fast_memory.INT_STS .. ysem.fast_memory.INT_MASK (2 regs) */
	0x05550120,		/* ysem.fast_memory.gpre0 .. ysem.fast_memory.stall_2 (5 regs) */
	0x04550128,		/* ysem.fast_memory.storm_stack_size .. ysem.fast_memory.pram_prty_addr_high (4 regs) */
	0x0255012e,		/* ysem.fast_memory.port_id_width .. ysem.fast_memory.port_id_offset (2 regs) */
	0x05550131,		/* ysem.fast_memory.state_machine .. ysem.fast_memory.storm_pc (5 regs) */
	0x0e55018a,		/* ysem.fast_memory.rt_clk_enable .. ysem.fast_memory.cam_init_in_process (14 regs) */
	0x0d5501d0,		/* ysem.fast_memory.debug_active .. ysem.fast_memory.dbg_store_addr_value (13 regs) */
	0x04550210,		/* ysem.fast_memory.sync_dra_rd_alm_full .. ysem.fast_memory.dbg_alm_full (4 regs) */
	0x03550250,		/* ysem.fast_memory.full .. ysem.fast_memory.alm_full (3 regs) */
	0x06550290,		/* ysem.fast_memory.active_filter_enable .. ysem.fast_memory.idle_inactive_cycles (6 regs) */
	0x015502d3,		/* ysem.fast_memory.vfc_status (1 regs) */
	0x04550310,		/* ysem.fast_memory.cam_bist_en .. ysem.fast_memory.cam_bist_status (4 regs) */
	0x282f0000,		/* block psem */
	0x04580001,		/* psem.enable_in .. psem.pas_disable (4 regs) */
	0x02580010,		/* psem.INT_STS_0 .. psem.INT_MASK_0 (2 regs) */
	0x02580014,		/* psem.INT_STS_1 .. psem.INT_MASK_1 (2 regs) */
	0x01580100,		/* psem.arb_cycle_size (1 regs) */
	0x08580110,		/* psem.pf_err_vector .. psem.eco_reserved (8 regs) */
	0x01580180,		/* psem.fic_min_msg (1 regs) */
	0x02580188,		/* psem.fic_empty_ct_mode .. psem.fic_empty_ct_cnt (2 regs) */
	0x025801a0,		/* psem.foc_credit (2 regs) */
	0x015801b0,		/* psem.full_foc_dra_strt_en (1 regs) */
	0x10d801c0,		/* psem.fin_command (16 regs, WB) */
	0x01580240,		/* psem.invld_pas_wr_en (1 regs) */
	0x03580260,		/* psem.arbiter_request .. psem.arbiter_slot (3 regs) */
	0x035802c0,		/* psem.num_of_threads .. psem.thread_rdy (3 regs) */
	0x015802c5,		/* psem.threads_list (1 regs) */
	0x01580380,		/* psem.order_pop_en (1 regs) */
	0x01580382,		/* psem.order_wake_en (1 regs) */
	0x01580384,		/* psem.pf_num_order_base (1 regs) */
	0x04580400,		/* psem.dbg_alm_full .. psem.sync_ram_wr_alm_full (4 regs) */
	0x02580440,		/* psem.dra_empty .. psem.ext_pas_empty (2 regs) */
	0x01580448,		/* psem.fic_empty (1 regs) */
	0x0b580450,		/* psem.slow_dbg_empty .. psem.ord_id_fifo_empty (11 regs) */
	0x02580480,		/* psem.ext_pas_full .. psem.ext_store_if_full (2 regs) */
	0x01580488,		/* psem.fic_full (1 regs) */
	0x0f580490,		/* psem.pas_if_full .. psem.ord_id_fifo_full (15 regs) */
	0x035804c0,		/* psem.thread_inter_cnt .. psem.thread_orun_num (3 regs) */
	0x0a580500,		/* psem.slow_dbg_active .. psem.dbg_msg_src (10 regs) */
	0x0558054a,		/* psem.dbg_select .. psem.dbg_force_frame (5 regs) */
	0x01590001,		/* psem.fast_memory.ram_ext_disable (1 regs) */
	0x02590010,		/* psem.fast_memory.INT_STS .. psem.fast_memory.INT_MASK (2 regs) */
	0x05590120,		/* psem.fast_memory.gpre0 .. psem.fast_memory.stall_2 (5 regs) */
	0x04590128,		/* psem.fast_memory.storm_stack_size .. psem.fast_memory.pram_prty_addr_high (4 regs) */
	0x0259012e,		/* psem.fast_memory.port_id_width .. psem.fast_memory.port_id_offset (2 regs) */
	0x05590131,		/* psem.fast_memory.state_machine .. psem.fast_memory.storm_pc (5 regs) */
	0x0e59018a,		/* psem.fast_memory.rt_clk_enable .. psem.fast_memory.cam_init_in_process (14 regs) */
	0x0d5901d0,		/* psem.fast_memory.debug_active .. psem.fast_memory.dbg_store_addr_value (13 regs) */
	0x04590210,		/* psem.fast_memory.sync_dra_rd_alm_full .. psem.fast_memory.dbg_alm_full (4 regs) */
	0x03590250,		/* psem.fast_memory.full .. psem.fast_memory.alm_full (3 regs) */
	0x06590290,		/* psem.fast_memory.active_filter_enable .. psem.fast_memory.idle_inactive_cycles (6 regs) */
	0x015902d3,		/* psem.fast_memory.vfc_status (1 regs) */
	0x04590310,		/* psem.fast_memory.cam_bist_en .. psem.fast_memory.cam_bist_status (4 regs) */
	0x04300000,		/* block rss */
	0x0308e201,		/* rss.rss_init_en .. rss.if_enable (3 regs) */
	0x0208e260,		/* rss.INT_STS .. rss.INT_MASK (2 regs) */
	0x0208e300,		/* rss.key_rss_ext5 .. rss.tmld_credit (2 regs) */
	0x0b08e30d,		/* rss.rbc_status .. rss.dbg_force_frame (11 regs) */
	0x05310000,		/* block tmld */
	0x2f134000,		/* tmld.scbd_strict_prio .. tmld.cm_hdr_127_96 (47 regs) */
	0x05134030,		/* tmld.stat_fic_msg .. tmld.len_err_log_2 (5 regs) */
	0x01134036,		/* tmld.len_err_log_v (1 regs) */
	0x02134060,		/* tmld.INT_STS .. tmld.INT_MASK (2 regs) */
	0x05134580,		/* tmld.dbg_select .. tmld.dbg_force_frame (5 regs) */
	0x05320000,		/* block muld */
	0x38138000,		/* muld.scbd_strict_prio .. muld.cm_hdr_127_96 (56 regs) */
	0x05138039,		/* muld.stat_fic_msg .. muld.len_err_log_2 (5 regs) */
	0x0113803f,		/* muld.len_err_log_v (1 regs) */
	0x02138060,		/* muld.INT_STS .. muld.INT_MASK (2 regs) */
	0x05138580,		/* muld.dbg_select .. muld.dbg_force_frame (5 regs) */
	0x05330000,		/* block yuld */
	0x2a132000,		/* yuld.scbd_strict_prio .. yuld.cm_hdr_127_96 (42 regs) */
	0x0513202b,		/* yuld.stat_fic_msg .. yuld.len_err_log_2 (5 regs) */
	0x01132031,		/* yuld.len_err_log_v (1 regs) */
	0x02132060,		/* yuld.INT_STS .. yuld.INT_MASK (2 regs) */
	0x05132580,		/* yuld.dbg_select .. yuld.dbg_force_frame (5 regs) */
	0x06340000,		/* block xyld */
	0x31130000,		/* xyld.scbd_strict_prio .. xyld.cm_hdr_127_96 (49 regs) */
	0x04130032,		/* xyld.seg_msg_log .. xyld.seg_msg_log_len_arr_95_64 (4 regs) */
	0x06130037,		/* xyld.seg_msg_log_v .. xyld.len_err_log_2 (6 regs) */
	0x0113003e,		/* xyld.len_err_log_v (1 regs) */
	0x02130060,		/* xyld.INT_STS .. xyld.INT_MASK (2 regs) */
	0x05130580,		/* xyld.dbg_select .. xyld.dbg_force_frame (5 regs) */
	0x06350000,		/* block prm */
	0x0608c000,		/* prm.disable_prm .. prm.disable_outputs (6 regs) */
	0x0208c010,		/* prm.INT_STS .. prm.INT_MASK (2 regs) */
	0x0508c108,		/* prm.pad_data .. prm.init_credit_rdif_pth (5 regs) */
	0x0508c140,		/* prm.rpb_db_full_thr .. prm.pxp_resp_full_thr (5 regs) */
	0x0208c180,		/* prm.num_of_mstorm_cmd .. prm.num_of_ustorm_cmd (2 regs) */
	0x0508c1aa,		/* prm.dbg_select .. prm.dbg_force_frame (5 regs) */
	0x05360000,		/* block pbf_pb1 */
	0x02368010,		/* pbf_pb1.INT_STS .. pbf_pb1.INT_MASK (2 regs) */
	0x0d368100,		/* pbf_pb1.control .. pbf_pb1.crc_mask_3_3 (13 regs) */
	0x09368140,		/* pbf_pb1.db_empty .. pbf_pb1.tq_th_empty (9 regs) */
	0x06368180,		/* pbf_pb1.errored_crc .. pbf_pb1.eco_reserved (6 regs) */
	0x053681ca,		/* pbf_pb1.dbg_select .. pbf_pb1.dbg_force_frame (5 regs) */
	0x05370000,		/* block pbf_pb2 */
	0x02369010,		/* pbf_pb2.INT_STS .. pbf_pb2.INT_MASK (2 regs) */
	0x0d369100,		/* pbf_pb2.control .. pbf_pb2.crc_mask_3_3 (13 regs) */
	0x09369140,		/* pbf_pb2.db_empty .. pbf_pb2.tq_th_empty (9 regs) */
	0x06369180,		/* pbf_pb2.errored_crc .. pbf_pb2.eco_reserved (6 regs) */
	0x053691ca,		/* pbf_pb2.dbg_select .. pbf_pb2.dbg_force_frame (5 regs) */
	0x05380000,		/* block rpb */
	0x0208f010,		/* rpb.INT_STS .. rpb.INT_MASK (2 regs) */
	0x0d08f100,		/* rpb.control .. rpb.crc_mask_3_3 (13 regs) */
	0x0908f140,		/* rpb.db_empty .. rpb.tq_th_empty (9 regs) */
	0x0608f180,		/* rpb.errored_crc .. rpb.eco_reserved (6 regs) */
	0x0508f1ca,		/* rpb.dbg_select .. rpb.dbg_force_frame (5 regs) */
	0x17390000,		/* block btb */
	0x0336c001,		/* btb.hw_init_en .. btb.start_en (3 regs) */
	0x0236c030,		/* btb.INT_STS_0 .. btb.INT_MASK_0 (2 regs) */
	0x0236c036,		/* btb.INT_STS_1 .. btb.INT_MASK_1 (2 regs) */
	0x0236c03c,		/* btb.INT_STS_2 .. btb.INT_MASK_2 (2 regs) */
	0x0236c042,		/* btb.INT_STS_3 .. btb.INT_MASK_3 (2 regs) */
	0x0236c048,		/* btb.INT_STS_4 .. btb.INT_MASK_4 (2 regs) */
	0x0236c04e,		/* btb.INT_STS_5 .. btb.INT_MASK_5 (2 regs) */
	0x0236c054,		/* btb.INT_STS_6 .. btb.INT_MASK_6 (2 regs) */
	0x0236c061,		/* btb.INT_STS_8 .. btb.INT_MASK_8 (2 regs) */
	0x0236c067,		/* btb.INT_STS_9 .. btb.INT_MASK_9 (2 regs) */
	0x0236c06d,		/* btb.INT_STS_10 .. btb.INT_MASK_10 (2 regs) */
	0x0236c073,		/* btb.INT_STS_11 .. btb.INT_MASK_11 (2 regs) */
	0x0236c200,		/* btb.big_ram_address .. btb.header_size (2 regs) */
	0x0536c210,		/* btb.max_releases .. btb.rc_pkt_priority (5 regs) */
	0x1436c223,		/* btb.rc_sop_priority .. btb.dbg_force_frame (20 regs) */
	0x0636c242,		/* btb.inp_if_enable .. btb.wc_empty_0 (6 regs) */
	0x0136c257,		/* btb.wc_full_0 (1 regs) */
	0x0636c267,		/* btb.wc_bandwidth_if_full .. btb.rc_pkt_empty_3 (6 regs) */
	0x0436c278,		/* btb.rc_pkt_full_0 .. btb.rc_pkt_full_3 (4 regs) */
	0x0436c287,		/* btb.rc_pkt_status_0 .. btb.rc_pkt_status_3 (4 regs) */
	0x0936c296,		/* btb.rc_sop_empty .. btb.wc_sync_fifo_push_status_0 (9 regs) */
	0x0236c2ad,		/* btb.rls_sync_fifo_push_status .. btb.rc_pkt_state (2 regs) */
	0x0136c2b2,		/* btb.clocks_ratio (1 regs) */
	0x283a0000,		/* block pbf */
	0x01360000,		/* pbf.init (1 regs) */
	0x01360010,		/* pbf.if_enable_reg (1 regs) */
	0x05360018,		/* pbf.dbg_select .. pbf.dbg_force_frame (5 regs) */
	0x0336002a,		/* pbf.fc_dbg_select .. pbf.fc_dbg_shift (3 regs) */
	0x08b60030,		/* pbf.fc_dbg_out_data (8 regs, WB) */
	0x04360038,		/* pbf.fc_dbg_force_valid .. pbf.fc_dbg_out_frame (4 regs) */
	0x01360043,		/* pbf.memctrl_status (1 regs) */
	0x02360060,		/* pbf.INT_STS .. pbf.INT_MASK (2 regs) */
	0x0c360100,		/* pbf.pxp_req_if_init_crd .. pbf.ycm_if_init_crd (12 regs) */
	0x08360110,		/* pbf.pb1_db_almost_full_thrsh .. pbf.mrku_almost_full_thrsh (8 regs) */
	0x0c360120,		/* pbf.tag_ethertype_0 .. pbf.tag_len_5 (12 regs) */
	0x0836013e,		/* pbf.llc_type_threshold .. pbf.gre_protocol (8 regs) */
	0x01360148,		/* pbf.nge_eth_type (1 regs) */
	0x01360161,		/* pbf.regular_inband_tag_order (1 regs) */
	0x07360163,		/* pbf.dst_mac_global_0 .. pbf.udp_dst_port_cfg_2 (7 regs) */
	0x02360175,		/* pbf.l2_edpm_thrsh .. pbf.cpmu_thrsh (2 regs) */
	0x0e360180,		/* pbf.ip_id_mask_0 .. pbf.tcm_snd_nxt_reg_offset (14 regs) */
	0x02360190,		/* pbf.pci_vq_id .. pbf.drop_pkt_upon_err (2 regs) */
	0x07360196,		/* pbf.per_voq_stat_mask .. pbf.num_pkts_sent_with_drop_to_btb (7 regs) */
	0x0c3601a8,		/* pbf.ycmd_qs_num_lines_voq0 .. pbf.num_blocks_allocated_cons_voq0 (12 regs) */
	0x0c3601b8,		/* pbf.ycmd_qs_num_lines_voq1 .. pbf.num_blocks_allocated_cons_voq1 (12 regs) */
	0x0c3601c8,		/* pbf.ycmd_qs_num_lines_voq2 .. pbf.num_blocks_allocated_cons_voq2 (12 regs) */
	0x0c3601d8,		/* pbf.ycmd_qs_num_lines_voq3 .. pbf.num_blocks_allocated_cons_voq3 (12 regs) */
	0x0c3601e8,		/* pbf.ycmd_qs_num_lines_voq4 .. pbf.num_blocks_allocated_cons_voq4 (12 regs) */
	0x0c3601f8,		/* pbf.ycmd_qs_num_lines_voq5 .. pbf.num_blocks_allocated_cons_voq5 (12 regs) */
	0x0c360208,		/* pbf.ycmd_qs_num_lines_voq6 .. pbf.num_blocks_allocated_cons_voq6 (12 regs) */
	0x0c360218,		/* pbf.ycmd_qs_num_lines_voq7 .. pbf.num_blocks_allocated_cons_voq7 (12 regs) */
	0x0c360228,		/* pbf.ycmd_qs_num_lines_voq8 .. pbf.num_blocks_allocated_cons_voq8 (12 regs) */
	0x0c360238,		/* pbf.ycmd_qs_num_lines_voq9 .. pbf.num_blocks_allocated_cons_voq9 (12 regs) */
	0x0c360248,		/* pbf.ycmd_qs_num_lines_voq10 .. pbf.num_blocks_allocated_cons_voq10 (12 regs) */
	0x0c360258,		/* pbf.ycmd_qs_num_lines_voq11 .. pbf.num_blocks_allocated_cons_voq11 (12 regs) */
	0x0c360268,		/* pbf.ycmd_qs_num_lines_voq12 .. pbf.num_blocks_allocated_cons_voq12 (12 regs) */
	0x0c360278,		/* pbf.ycmd_qs_num_lines_voq13 .. pbf.num_blocks_allocated_cons_voq13 (12 regs) */
	0x0c360288,		/* pbf.ycmd_qs_num_lines_voq14 .. pbf.num_blocks_allocated_cons_voq14 (12 regs) */
	0x0c360298,		/* pbf.ycmd_qs_num_lines_voq15 .. pbf.num_blocks_allocated_cons_voq15 (12 regs) */
	0x0c3602a8,		/* pbf.ycmd_qs_num_lines_voq16 .. pbf.num_blocks_allocated_cons_voq16 (12 regs) */
	0x0c3602b8,		/* pbf.ycmd_qs_num_lines_voq17 .. pbf.num_blocks_allocated_cons_voq17 (12 regs) */
	0x0c3602c8,		/* pbf.ycmd_qs_num_lines_voq18 .. pbf.num_blocks_allocated_cons_voq18 (12 regs) */
	0x0c3602d8,		/* pbf.ycmd_qs_num_lines_voq19 .. pbf.num_blocks_allocated_cons_voq19 (12 regs) */
	0x013603a8,		/* pbf.eco_reserved (1 regs) */
	0x053b0000,		/* block rdif */
	0x040c0010,		/* rdif.stop_on_error .. rdif.min_eob2wf_l1_rd_del (4 regs) */
	0x010c0015,		/* rdif.dirty_l1 (1 regs) */
	0x1c0c001c,		/* rdif.debug_command_fifo_empty .. rdif.stat_num_err_interval_0 (28 regs) */
	0x020c0060,		/* rdif.INT_STS .. rdif.INT_MASK (2 regs) */
	0x050c0140,		/* rdif.dbg_select .. rdif.dbg_force_frame (5 regs) */
	0x043c0000,		/* block tdif */
	0x060c4010,		/* tdif.stop_on_error .. tdif.dirty_l1 (6 regs) */
	0x200c401c,		/* tdif.debug_command_fifo_empty .. tdif.stat_num_err_interval_3 (32 regs) */
	0x020c4060,		/* tdif.INT_STS .. tdif.INT_MASK (2 regs) */
	0x050c4140,		/* tdif.dbg_select .. tdif.dbg_force_frame (5 regs) */
	0x0a3d0000,		/* block cdu */
	0x01160010,		/* cdu.control0 (1 regs) */
	0x01160070,		/* cdu.INT_STS (1 regs) */
	0x01160073,		/* cdu.INT_MASK (1 regs) */
	0x04160100,		/* cdu.ccfc_ctx_valid0 .. cdu.tcfc_ctx_valid1 (4 regs) */
	0x02160140,		/* cdu.ldbuf_af_thresh .. cdu.wbbuf_af_thresh (2 regs) */
	0x04160180,		/* cdu.ccfc_pxp .. cdu.wb_vqid (4 regs) */
	0x061601c0,		/* cdu.debug .. cdu.dbg_force_frame (6 regs) */
	0x011601d2,		/* cdu.eco_reserved (1 regs) */
	0x06160200,		/* cdu.ccfc_cvld_error_data .. cdu.tcfc_wb_l1_num_error_data (6 regs) */
	0x03160240,		/* cdu.cid_addr_params .. cdu.segment1_params (3 regs) */
	0x103e0000,		/* block ccfc */
	0x050b8000,		/* ccfc.init_reg .. ccfc.tidram_init_done (5 regs) */
	0x020b8060,		/* ccfc.INT_STS_0 .. ccfc.INT_MASK_0 (2 regs) */
	0x0c0b8100,		/* ccfc.lc_blocked .. ccfc.cdu_write_backs (12 regs) */
	0x050b8140,		/* ccfc.dbg_select .. ccfc.dbg_force_frame (5 regs) */
	0x090b8152,		/* ccfc.eco_reserved .. ccfc.arbiters_reg (9 regs) */
	0x060b8170,		/* ccfc.debug0 .. ccfc.cdu_pcie_err_mask (6 regs) */
	0x020b8177,		/* ccfc.sreq_full_sticky .. ccfc.prsresp_full_sticky (2 regs) */
	0x0b0b8180,		/* ccfc.num_lcids_empty .. ccfc.max_inside (11 regs) */
	0x020b81c4,		/* ccfc.LoadRetry_Types .. ccfc.MiniCache_Control (2 regs) */
	0x010b81c7,		/* ccfc.control0 (1 regs) */
	0x040b81e0,		/* ccfc.prsresp_credit .. ccfc.cduwb_credit (4 regs) */
	0x040b8200,		/* ccfc.ll_policy_cfg .. ccfc.empty_size (4 regs) */
	0x1f0b8240,		/* ccfc.lc_client_0_lcid_threshold .. ccfc.wave_sm_2_one_count (31 regs) */
	0x120b8280,		/* ccfc.cache_string_type .. ccfc.include_vlan_in_hash (18 regs) */
	0x080b82c0,		/* ccfc.cid_cam_bist_en .. ccfc.string_cam_bist_status (8 regs) */
	0x020bb400,		/* ccfc.vpf1_lstate_sel .. ccfc.vpf2_lstate_sel (2 regs) */
	0x0e400000,		/* block igu */
	0x01060000,		/* igu.reset_memories (1 regs) */
	0x01060010,		/* igu.block_configuration (1 regs) */
	0x01060014,		/* igu.pxp_requester_initial_credit (1 regs) */
	0x02060060,		/* igu.INT_STS .. igu.INT_MASK (2 regs) */
	0x08060103,		/* igu.pxp_request_counter .. igu.cons_upd_counter (8 regs) */
	0x01060202,		/* igu.message_fields (1 regs) */
	0x01060213,		/* igu.statistic_en (1 regs) */
	0x03060218,		/* igu.cam_parity_scrubbing .. igu.eco_reserved (3 regs) */
	0x10060300,		/* igu.vf_with_more_16sb_0 .. igu.vf_with_more_16sb_15 (16 regs) */
	0x03060480,		/* igu.global_rate_limiter_vari0 .. igu.global_rate_tick_rate_counter (3 regs) */
	0x01060485,		/* igu.clk25_counter_sensitivity (1 regs) */
	0x06060487,		/* igu.group_rl_en_0 .. igu.group_rl_pending_1 (6 regs) */
	0x08060540,		/* igu.attention_signal_p0_status .. igu.Interrupt_status (8 regs) */
	0x1706054c,		/* igu.error_handling_data_valid .. igu.dbg_force_frame (23 regs) */
	0x10410000,		/* block cau */
	0x01070035,		/* cau.INT_STS (1 regs) */
	0x01070038,		/* cau.INT_MASK (1 regs) */
	0x05070100,		/* cau.num_pi_per_sb .. cau.reset_memories (5 regs) */
	0x02070140,		/* cau.in_arb_priority .. cau.in_arb_timeout (2 regs) */
	0x05070180,		/* cau.cqe_size .. cau.agg_release_timer (5 regs) */
	0x040701c0,		/* cau.tick_size .. cau.stop_scan (4 regs) */
	0x02070220,		/* cau.wdata_fifo_afull_thr .. cau.cqe_fifo_afull_thr (2 regs) */
	0x02070260,		/* cau.igu_req_credit_status .. cau.igu_cmd_credit_status (2 regs) */
	0x060702a0,		/* cau.stat_ctrl_sb_select .. cau.stat_ctrl_timer_cmd_type (6 regs) */
	0x110702e0,		/* cau.stat_counter_sb_gen .. cau.stat_counter_cqe_partial_cache (17 regs) */
	0x06070320,		/* cau.debug_fifo_status .. cau.error_cleanup_cmd_reg (6 regs) */
	0x05070327,		/* cau.agg_units_0to15_state .. cau.eco_reserved (5 regs) */
	0x05070360,		/* cau.debug_record_mask_min_sb .. cau.debug_record_mask_cmd_type (5 regs) */
	0x03070380,		/* cau.req_counter .. cau.wdone_counter (3 regs) */
	0x050703aa,		/* cau.dbg_select .. cau.dbg_force_frame (5 regs) */
	0x050703c0,		/* cau.main_fsm_status .. cau.igu_cqe_agg_fsm_status (5 regs) */
	0x09450000,		/* block dbg */
	0x18004001,		/* dbg.client_enable .. dbg.full_mode (24 regs) */
	0x02004060,		/* dbg.INT_STS .. dbg.INT_MASK (2 regs) */
	0x02004100,		/* dbg.intr_buffer_rd_ptr .. dbg.intr_buffer_wr_ptr (2 regs) */
	0x04804102,		/* dbg.ext_buffer_rd_ptr .. dbg.ext_buffer_wr_ptr (4 regs, WB) */
	0x13004106,		/* dbg.wrap_on_int_buffer .. dbg.pci_logic_addr (19 regs) */
	0xff004150,		/* dbg.pattern_recognition_disable .. dbg.trigger_state_set_cnstr_cyclic_15 (255 regs) */
	0x6100424f,		/* dbg.trigger_state_set_cnstr_cyclic_16 .. dbg.dbg_driver_trigger (97 regs) */
	0x210042c4,		/* dbg.hw_id_num .. dbg.filter_status_match_cnstr (33 regs) */
	0x020042e8,		/* dbg.memctrl_status .. dbg.num_of_empty_lines_in_int_buffer (2 regs) */
	0x26460000,		/* block nig */
	0x02140010,		/* nig.INT_STS_0 .. nig.INT_MASK_0 (2 regs) */
	0x02140014,		/* nig.INT_STS_1 .. nig.INT_MASK_1 (2 regs) */
	0x02140018,		/* nig.INT_STS_2 .. nig.INT_MASK_2 (2 regs) */
	0x0214001c,		/* nig.INT_STS_3 .. nig.INT_MASK_3 (2 regs) */
	0x02140020,		/* nig.INT_STS_4 .. nig.INT_MASK_4 (2 regs) */
	0x02140024,		/* nig.INT_STS_5 .. nig.INT_MASK_5 (2 regs) */
	0xb2140200,		/* nig.close_gate_disable .. nig.tx_lb_vport_drop_159 (178 regs) */
	0x04140301,		/* nig.lb_sopq_empty .. nig.tx_sopq_full (4 regs) */
	0x04140380,		/* nig.dorq_in_en .. nig.ppp_out_en (4 regs) */
	0x01140401,		/* nig.initial_header_size (1 regs) */
	0x01140403,		/* nig.llh_arp_type (1 regs) */
	0x0214041b,		/* nig.roce_type .. nig.gre_eth_type (2 regs) */
	0x01140421,		/* nig.gre_protocol (1 regs) */
	0x0114044e,		/* nig.rx_llh_svol_mcp_fwd_allpf (1 regs) */
	0x01140454,		/* nig.rx_llh_svol_brb_dntfwd_allpf (1 regs) */
	0x0114047c,		/* nig.rx_llh_brb_gate_dntfwd_clsfailed (1 regs) */
	0x01140598,		/* nig.lb_llh_brb_gate_dntfwd_clsfailed (1 regs) */
	0x041406e0,		/* nig.llh_eng_cls_type .. nig.llh_eng_cls_crc8_init_val (4 regs) */
	0x029406e4,		/* nig.llh_eng_cls_eng_id_tbl (2 regs, WB) */
	0x011406e6,		/* nig.llh_eng_cls_roce_qp_sel (1 regs) */
	0x04140715,		/* nig.ppp_address .. nig.ppp_trig (4 regs) */
	0x0114074a,		/* nig.stat_rx_storm_packet_sent (1 regs) */
	0x0114082b,		/* nig.bmb_fifo_alm_full_thr (1 regs) */
	0x05140830,		/* nig.dorq_fifo_alm_full_thr .. nig.debug_port (5 regs) */
	0x04140836,		/* nig.debug_pkt_wait_size .. nig.debug_fifo_full (4 regs) */
	0x10140871,		/* nig.rx_fc_dbg_select_pllh .. nig.eco_reserved (16 regs) */
	0x02142216,		/* nig.pm_timer_select .. nig.ts_for_semi_select (2 regs) */
	0x0214221c,		/* nig.ts_output_enable_pda .. nig.ts_output_enable_hv (2 regs) */
	0x0214222a,		/* nig.tsgen_free_cnt_value_lsb .. nig.tsgen_free_cnt_value_msb (2 regs) */
	0x0214222e,		/* nig.tsgen_freecnt_lsb .. nig.tsgen_freecnt_msb (2 regs) */
	0x02142234,		/* nig.tsgen_pps_high_time .. nig.tsgen_pps_low_time (2 regs) */
	0x03142238,		/* nig.tsgen_tsio_oeb .. nig.edpm_fifo_full_thresh (3 regs) */
	0x011422c3,		/* nig.rroce_port (1 regs) */
	0x011422cd,		/* nig.nge_eth_type (1 regs) */
	0x021422d0,		/* nig.bth_hdr_flow_ctrl_opcode_1 .. nig.bth_hdr_flow_ctrl_opcode_2 (2 regs) */
	0x011422d7,		/* nig.dbgmux_ovflw_ind_en (1 regs) */
	0x011422df,		/* nig.tx_parity_error_timer (1 regs) */
	0x021422e3,		/* nig.tx_inhibit_bmb_arb_en .. nig.lb_inhibit_bmb_arb_en (2 regs) */
	0x02050005,		/* mode bb, block cnig */
	0x0108608e,		/* cnig.eco_reserved (1 regs) */
	0x020860ba,		/* cnig.INT_STS .. cnig.INT_MASK (2 regs) */
	0x02210000,		/* block brb */
	0x020d0220,		/* brb.shared_hr_area (2 regs) */
	0x020d0230,		/* brb.total_mac_size (2 regs) */
	0x012a0000,		/* block tsem */
	0x04dc0108,		/* tsem.vf_err_vector (4 regs, WB) */
	0x012b0000,		/* block msem */
	0x04e00108,		/* msem.vf_err_vector (4 regs, WB) */
	0x012c0000,		/* block usem */
	0x04e40108,		/* usem.vf_err_vector (4 regs, WB) */
	0x012d0000,		/* block xsem */
	0x04d00108,		/* xsem.vf_err_vector (4 regs, WB) */
	0x012e0000,		/* block ysem */
	0x04d40108,		/* ysem.vf_err_vector (4 regs, WB) */
	0x012f0000,		/* block psem */
	0x04d80108,		/* psem.vf_err_vector (4 regs, WB) */
	0x04010023,		/* mode !bb, block miscs */
	0x0100245c,		/* miscs.bsc_smbio_enable_glitch_filter (1 regs) */
	0x060024f0,		/* miscs.pcie_link_up_state .. miscs.main_pll_status (6 regs) */
	0x05002594,		/* miscs.bsc_sda_sel .. miscs.pcie_phy_rst_n_status (5 regs) */
	0x010025ae,		/* miscs.core_rst_n_status (1 regs) */
	0x07040000,		/* block pglue_b */
	0x030aa14a,		/* pglue_b.txw_h_syncfifo_almostfull_th .. pglue_b.txr_h_syncfifo_almostfull_th (3 regs) */
	0x010aabac,		/* pglue_b.cfg_no_l1_on_int (1 regs) */
	0x020aabaf,		/* pglue_b.mctp_max_length .. pglue_b.mctp_reqid (2 regs) */
	0x090aabb2,		/* pglue_b.pbus_num .. pglue_b.mrrs_attn (9 regs) */
	0x090aabbc,		/* pglue_b.txw_b2b_disable .. pglue_b.pgl_pm_dstate_47_32 (9 regs) */
	0x0f0aabd7,		/* pglue_b.check_tc_on_err .. pglue_b.mctp_venderid_chk_disable (15 regs) */
	0x0b0aabea,		/* pglue_b.ext_tag_en_pf_31_0 .. pglue_b.rxd_syncfifo_pop_status (11 regs) */
	0x03050000,		/* block cnig */
	0x08086080,		/* cnig.nig_port0_conf .. cnig.INT_MASK (8 regs) */
	0x0108608a,		/* cnig.nwm_lpi_defualt_value (1 regs) */
	0x06086094,		/* cnig.eco_reserved .. cnig.dbg_force_frame (6 regs) */
	0x040a0000,		/* block pcie */
	0x6a015084,		/* pcie.soft_reset_control .. pcie.msix_synch_sticky (106 regs) */
	0x020151e8,		/* pcie.INT_STS .. pcie.INT_MASK (2 regs) */
	0x050151fa,		/* pcie.dbg_select .. pcie.dbg_force_frame (5 regs) */
	0x02015200,		/* pcie.reset_status_2 .. pcie.reset_status_3 (2 regs) */
	0x10150000,		/* block pglcs */
	0x06000744,		/* pglcs.rasdp_error_mode_en_off .. pglcs.dbg_force_frame (6 regs) */
	0x07000852,		/* pglcs.pgl_cs.VC_BASE .. pglcs.pgl_cs.RESOURCE_STATUS_REG_VC0 (7 regs) */
	0x07000866,		/* pglcs.pgl_cs.SPCIE_CAP_HEADER_REG .. pglcs.pgl_cs.SPCIE_CAP_OFF_18H_REG (7 regs) */
	0x020008a1,		/* pglcs.pgl_cs.LTR_CAP_HDR_REG .. pglcs.pgl_cs.LTR_LATENCY_REG (2 regs) */
	0x030008f1,		/* pglcs.pgl_cs.PTM_EXT_CAP_HDR_OFF .. pglcs.pgl_cs.PTM_CONTROL_OFF (3 regs) */
	0x090009c0,		/* pglcs.pgl_cs.ACK_LATENCY_TIMER_OFF .. pglcs.pgl_cs.FILTER_MASK_2_OFF (9 regs) */
	0x0b0009ca,		/* pglcs.pgl_cs.PL_DEBUG0_OFF .. pglcs.pgl_cs.VC0_CPL_RX_Q_CTRL_OFF (11 regs) */
	0x03000a03,		/* pglcs.pgl_cs.GEN2_CTRL_OFF .. pglcs.pgl_cs.PHY_CONTROL_OFF (3 regs) */
	0x01000a24,		/* pglcs.pgl_cs.GEN3_RELATED_OFF (1 regs) */
	0x01000a28,		/* pglcs.pgl_cs.PF_HIDE_CONTROL (1 regs) */
	0x02000a2a,		/* pglcs.pgl_cs.GEN3_EQ_CONTROL_OFF .. pglcs.pgl_cs.GEN3_EQ_FB_MODE_DIR_CHANGE_OFF (2 regs) */
	0x06000a2d,		/* pglcs.pgl_cs.ORDER_RULE_CTRL_OFF .. pglcs.pgl_cs.TRGT_CPL_LUT_DELETE_ENTRY_OFF (6 regs) */
	0x01000a3f,		/* pglcs.pgl_cs.PL_LAST_OFF (1 regs) */
	0x01000acc,		/* pglcs.pgl_cs.PL_LTR_LATENCY_OFF (1 regs) */
	0x01000ad0,		/* pglcs.pgl_cs.AUX_CLK_FREQ_OFF (1 regs) */
	0x0e000e01,		/* pglcs.discard_poisoned_mctp_tgtwr .. pglcs.tx_syncfifo_pop_status (14 regs) */
	0x01170000,		/* block ptu */
	0x01158037,		/* ptu.atc_otb_overrun_fix_chicken_bit (1 regs) */
	0x061e0000,		/* block qm */
	0x080bc148,		/* qm.MaxPqSizeTxSel_56 .. qm.MaxPqSizeTxSel_63 (8 regs) */
	0x020bc41e,		/* qm.QstatusTx_14 .. qm.QstatusTx_15 (2 regs) */
	0x020bc432,		/* qm.QstatusOther_2 .. qm.QstatusOther_3 (2 regs) */
	0x080bc662,		/* qm.WrrOtherPqGrp_8 .. qm.WrrOtherPqGrp_15 (8 regs) */
	0x080bcb39,		/* qm.PqTx2Pf_56 .. qm.PqTx2Pf_63 (8 regs) */
	0x080bcb89,		/* qm.PqOther2Pf_8 .. qm.PqOther2Pf_15 (8 regs) */
	0x01200000,		/* block dorq */
	0x0204024b,		/* dorq.l2_edpm_tunnel_nge_ip_en .. dorq.l2_edpm_tunnel_nge_eth_en (2 regs) */
	0x1a210000,		/* block brb */
	0x040d0220,		/* brb.shared_hr_area (4 regs) */
	0x040d0230,		/* brb.total_mac_size (4 regs) */
	0x020d0252,		/* brb.tc_guarantied_18 .. brb.tc_guarantied_19 (2 regs) */
	0x020d0288,		/* brb.lb_tc_guarantied_hyst_18 .. brb.lb_tc_guarantied_hyst_19 (2 regs) */
	0x020d02be,		/* brb.lb_tc_pause_xoff_threshold_18 .. brb.lb_tc_pause_xoff_threshold_19 (2 regs) */
	0x020d02f4,		/* brb.lb_tc_pause_xon_threshold_18 .. brb.lb_tc_pause_xon_threshold_19 (2 regs) */
	0x020d032a,		/* brb.lb_tc_full_xoff_threshold_18 .. brb.lb_tc_full_xoff_threshold_19 (2 regs) */
	0x020d0360,		/* brb.lb_tc_full_xon_threshold_18 .. brb.lb_tc_full_xon_threshold_19 (2 regs) */
	0x020d0374,		/* brb.wc_no_dead_cycles_en .. brb.wc_highest_pri_en (2 regs) */
	0x020d0393,		/* brb.pm_tc_latency_sensitive_2 .. brb.pm_tc_latency_sensitive_3 (2 regs) */
	0x040d03d0,		/* brb.wc_empty_4 .. brb.wc_empty_7 (4 regs) */
	0x040d03e0,		/* brb.wc_full_4 .. brb.wc_full_7 (4 regs) */
	0x020d0426,		/* brb.empty_if_2 .. brb.empty_if_3 (2 regs) */
	0x020d044b,		/* brb.rc_eop_inp_sync_fifo_push_status_2 .. brb.rc_eop_inp_sync_fifo_push_status_3 (2 regs) */
	0x020d045a,		/* brb.rc_eop_out_sync_fifo_push_status_2 .. brb.rc_eop_out_sync_fifo_push_status_3 (2 regs) */
	0x020d0470,		/* brb.mac_free_shared_hr_2 .. brb.mac_free_shared_hr_3 (2 regs) */
	0x050d0494,		/* brb.mac2_tc_occupancy_0 .. brb.mac2_tc_occupancy_4 (5 regs) */
	0x050d04a4,		/* brb.mac3_tc_occupancy_0 .. brb.mac3_tc_occupancy_4 (5 regs) */
	0x020d04b6,		/* brb.available_mac_size_2 .. brb.available_mac_size_3 (2 regs) */
	0x020d04bc,		/* brb.main_tc_pause_2 .. brb.main_tc_pause_3 (2 regs) */
	0x020d04c2,		/* brb.lb_tc_pause_2 .. brb.lb_tc_pause_3 (2 regs) */
	0x020d04c8,		/* brb.main_tc_full_2 .. brb.main_tc_full_3 (2 regs) */
	0x020d04ce,		/* brb.lb_tc_full_2 .. brb.lb_tc_full_3 (2 regs) */
	0x040d04f2,		/* brb.main2_tc_lossless_cnt_0 .. brb.main2_tc_lossless_cnt_3 (4 regs) */
	0x040d0502,		/* brb.main3_tc_lossless_cnt_0 .. brb.main3_tc_lossless_cnt_3 (4 regs) */
	0x020d0514,		/* brb.main_tc_lossless_int_2 .. brb.main_tc_lossless_int_3 (2 regs) */
	0x02240000,		/* block tsdm */
	0x023ec320,		/* tsdm.rmt_xcm_fifo_full .. tsdm.rmt_ycm_fifo_full (2 regs) */
	0x023ec35a,		/* tsdm.rmt_xcm_fifo_empty .. tsdm.rmt_ycm_fifo_empty (2 regs) */
	0x02250000,		/* block msdm */
	0x023f0320,		/* msdm.rmt_xcm_fifo_full .. msdm.rmt_ycm_fifo_full (2 regs) */
	0x023f035a,		/* msdm.rmt_xcm_fifo_empty .. msdm.rmt_ycm_fifo_empty (2 regs) */
	0x02260000,		/* block usdm */
	0x023f4320,		/* usdm.rmt_xcm_fifo_full .. usdm.rmt_ycm_fifo_full (2 regs) */
	0x023f435a,		/* usdm.rmt_xcm_fifo_empty .. usdm.rmt_ycm_fifo_empty (2 regs) */
	0x02270000,		/* block xsdm */
	0x023e0320,		/* xsdm.rmt_xcm_fifo_full .. xsdm.rmt_ycm_fifo_full (2 regs) */
	0x023e035a,		/* xsdm.rmt_xcm_fifo_empty .. xsdm.rmt_ycm_fifo_empty (2 regs) */
	0x02280000,		/* block ysdm */
	0x023e4320,		/* ysdm.rmt_xcm_fifo_full .. ysdm.rmt_ycm_fifo_full (2 regs) */
	0x023e435a,		/* ysdm.rmt_xcm_fifo_empty .. ysdm.rmt_ycm_fifo_empty (2 regs) */
	0x02290000,		/* block psdm */
	0x023e8320,		/* psdm.rmt_xcm_fifo_full .. psdm.rmt_ycm_fifo_full (2 regs) */
	0x023e835a,		/* psdm.rmt_xcm_fifo_empty .. psdm.rmt_ycm_fifo_empty (2 regs) */
	0x0f2a0000,		/* block tsem */
	0x08dc0108,		/* tsem.vf_err_vector (8 regs, WB) */
	0x015d0087,		/* tsem.fast_memory.reserved_21C (1 regs) */
	0x015d008a,		/* tsem.fast_memory.reserved_228 (1 regs) */
	0x015d008d,		/* tsem.fast_memory.reserved_234 (1 regs) */
	0x015d008f,		/* tsem.fast_memory.reserved_23C (1 regs) */
	0x015d0091,		/* tsem.fast_memory.reserved_244 (1 regs) */
	0x015d0094,		/* tsem.fast_memory.reserved_250 (1 regs) */
	0x015d0096,		/* tsem.fast_memory.reserved_258 (1 regs) */
	0x015d0098,		/* tsem.fast_memory.reserved_260 (1 regs) */
	0x015d009a,		/* tsem.fast_memory.reserved_268 (1 regs) */
	0x015d009c,		/* tsem.fast_memory.reserved_270 (1 regs) */
	0x015d009e,		/* tsem.fast_memory.reserved_278 (1 regs) */
	0x015d00a0,		/* tsem.fast_memory.reserved_280 (1 regs) */
	0x0f5d00a2,		/* tsem.fast_memory.reserved_288 .. tsem.fast_memory.reserved_2C0 (15 regs) */
	0x025d00b3,		/* tsem.fast_memory.reserved_2CC .. tsem.fast_memory.reserved_2D0 (2 regs) */
	0x0f2b0000,		/* block msem */
	0x08e00108,		/* msem.vf_err_vector (8 regs, WB) */
	0x01610087,		/* msem.fast_memory.reserved_21C (1 regs) */
	0x0161008a,		/* msem.fast_memory.reserved_228 (1 regs) */
	0x0161008d,		/* msem.fast_memory.reserved_234 (1 regs) */
	0x0161008f,		/* msem.fast_memory.reserved_23C (1 regs) */
	0x01610091,		/* msem.fast_memory.reserved_244 (1 regs) */
	0x01610094,		/* msem.fast_memory.reserved_250 (1 regs) */
	0x01610096,		/* msem.fast_memory.reserved_258 (1 regs) */
	0x01610098,		/* msem.fast_memory.reserved_260 (1 regs) */
	0x0161009a,		/* msem.fast_memory.reserved_268 (1 regs) */
	0x0161009c,		/* msem.fast_memory.reserved_270 (1 regs) */
	0x0161009e,		/* msem.fast_memory.reserved_278 (1 regs) */
	0x016100a0,		/* msem.fast_memory.reserved_280 (1 regs) */
	0x0f6100a2,		/* msem.fast_memory.reserved_288 .. msem.fast_memory.reserved_2C0 (15 regs) */
	0x026100b3,		/* msem.fast_memory.reserved_2CC .. msem.fast_memory.reserved_2D0 (2 regs) */
	0x0f2c0000,		/* block usem */
	0x08e40108,		/* usem.vf_err_vector (8 regs, WB) */
	0x01650087,		/* usem.fast_memory.reserved_21C (1 regs) */
	0x0165008a,		/* usem.fast_memory.reserved_228 (1 regs) */
	0x0165008d,		/* usem.fast_memory.reserved_234 (1 regs) */
	0x0165008f,		/* usem.fast_memory.reserved_23C (1 regs) */
	0x01650091,		/* usem.fast_memory.reserved_244 (1 regs) */
	0x01650094,		/* usem.fast_memory.reserved_250 (1 regs) */
	0x01650096,		/* usem.fast_memory.reserved_258 (1 regs) */
	0x01650098,		/* usem.fast_memory.reserved_260 (1 regs) */
	0x0165009a,		/* usem.fast_memory.reserved_268 (1 regs) */
	0x0165009c,		/* usem.fast_memory.reserved_270 (1 regs) */
	0x0165009e,		/* usem.fast_memory.reserved_278 (1 regs) */
	0x016500a0,		/* usem.fast_memory.reserved_280 (1 regs) */
	0x0f6500a2,		/* usem.fast_memory.reserved_288 .. usem.fast_memory.reserved_2C0 (15 regs) */
	0x026500b3,		/* usem.fast_memory.reserved_2CC .. usem.fast_memory.reserved_2D0 (2 regs) */
	0x0f2d0000,		/* block xsem */
	0x08d00108,		/* xsem.vf_err_vector (8 regs, WB) */
	0x01510087,		/* xsem.fast_memory.reserved_21C (1 regs) */
	0x0151008a,		/* xsem.fast_memory.reserved_228 (1 regs) */
	0x0151008d,		/* xsem.fast_memory.reserved_234 (1 regs) */
	0x0151008f,		/* xsem.fast_memory.reserved_23C (1 regs) */
	0x01510091,		/* xsem.fast_memory.reserved_244 (1 regs) */
	0x01510094,		/* xsem.fast_memory.reserved_250 (1 regs) */
	0x01510096,		/* xsem.fast_memory.reserved_258 (1 regs) */
	0x01510098,		/* xsem.fast_memory.reserved_260 (1 regs) */
	0x0151009a,		/* xsem.fast_memory.reserved_268 (1 regs) */
	0x0151009c,		/* xsem.fast_memory.reserved_270 (1 regs) */
	0x0151009e,		/* xsem.fast_memory.reserved_278 (1 regs) */
	0x015100a0,		/* xsem.fast_memory.reserved_280 (1 regs) */
	0x0f5100a2,		/* xsem.fast_memory.reserved_288 .. xsem.fast_memory.reserved_2C0 (15 regs) */
	0x025100b3,		/* xsem.fast_memory.reserved_2CC .. xsem.fast_memory.reserved_2D0 (2 regs) */
	0x0f2e0000,		/* block ysem */
	0x08d40108,		/* ysem.vf_err_vector (8 regs, WB) */
	0x01550087,		/* ysem.fast_memory.reserved_21C (1 regs) */
	0x0155008a,		/* ysem.fast_memory.reserved_228 (1 regs) */
	0x0155008d,		/* ysem.fast_memory.reserved_234 (1 regs) */
	0x0155008f,		/* ysem.fast_memory.reserved_23C (1 regs) */
	0x01550091,		/* ysem.fast_memory.reserved_244 (1 regs) */
	0x01550094,		/* ysem.fast_memory.reserved_250 (1 regs) */
	0x01550096,		/* ysem.fast_memory.reserved_258 (1 regs) */
	0x01550098,		/* ysem.fast_memory.reserved_260 (1 regs) */
	0x0155009a,		/* ysem.fast_memory.reserved_268 (1 regs) */
	0x0155009c,		/* ysem.fast_memory.reserved_270 (1 regs) */
	0x0155009e,		/* ysem.fast_memory.reserved_278 (1 regs) */
	0x015500a0,		/* ysem.fast_memory.reserved_280 (1 regs) */
	0x0f5500a2,		/* ysem.fast_memory.reserved_288 .. ysem.fast_memory.reserved_2C0 (15 regs) */
	0x025500b3,		/* ysem.fast_memory.reserved_2CC .. ysem.fast_memory.reserved_2D0 (2 regs) */
	0x0f2f0000,		/* block psem */
	0x08d80108,		/* psem.vf_err_vector (8 regs, WB) */
	0x01590087,		/* psem.fast_memory.reserved_21C (1 regs) */
	0x0159008a,		/* psem.fast_memory.reserved_228 (1 regs) */
	0x0159008d,		/* psem.fast_memory.reserved_234 (1 regs) */
	0x0159008f,		/* psem.fast_memory.reserved_23C (1 regs) */
	0x01590091,		/* psem.fast_memory.reserved_244 (1 regs) */
	0x01590094,		/* psem.fast_memory.reserved_250 (1 regs) */
	0x01590096,		/* psem.fast_memory.reserved_258 (1 regs) */
	0x01590098,		/* psem.fast_memory.reserved_260 (1 regs) */
	0x0159009a,		/* psem.fast_memory.reserved_268 (1 regs) */
	0x0159009c,		/* psem.fast_memory.reserved_270 (1 regs) */
	0x0159009e,		/* psem.fast_memory.reserved_278 (1 regs) */
	0x015900a0,		/* psem.fast_memory.reserved_280 (1 regs) */
	0x0f5900a2,		/* psem.fast_memory.reserved_288 .. psem.fast_memory.reserved_2C0 (15 regs) */
	0x025900b3,		/* psem.fast_memory.reserved_2CC .. psem.fast_memory.reserved_2D0 (2 regs) */
	0x01350000,		/* block prm */
	0x0108d800,		/* prm.nop_without_completion_fix_disable (1 regs) */
	0x04390000,		/* block btb */
	0x0236c215,		/* btb.wc_no_dead_cycles_en .. btb.wc_highest_pri_en (2 regs) */
	0x0436c26d,		/* btb.rc_pkt_empty_4 .. btb.rc_pkt_empty_7 (4 regs) */
	0x0436c27c,		/* btb.rc_pkt_full_4 .. btb.rc_pkt_full_7 (4 regs) */
	0x0436c28b,		/* btb.rc_pkt_status_4 .. btb.rc_pkt_status_7 (4 regs) */
	0x013c0000,		/* block tdif */
	0x040c403c,		/* tdif.stat_num_err_interval_4 .. tdif.stat_num_err_interval_7 (4 regs) */
	0x01400000,		/* block igu */
	0x05060310,		/* igu.vf_with_more_16sb_16 .. igu.vf_with_more_16sb_20 (5 regs) */
	0x01450000,		/* block dbg */
	0x01004119,		/* dbg.ifmux_select (1 regs) */
	0x0a460000,		/* block nig */
	0x02140028,		/* nig.INT_STS_6 .. nig.INT_MASK_6 (2 regs) */
	0x0214002c,		/* nig.INT_STS_7 .. nig.INT_MASK_7 (2 regs) */
	0x02140030,		/* nig.INT_STS_8 .. nig.INT_MASK_8 (2 regs) */
	0x02140034,		/* nig.INT_STS_9 .. nig.INT_MASK_9 (2 regs) */
	0x301402b2,		/* nig.tx_lb_vport_drop_160 .. nig.tx_lb_vport_drop_207 (48 regs) */
	0x07142400,		/* nig.tx_tdm_0_enable .. nig.tsgen_pps_out_sel_mask_3 (7 regs) */
	0x01142409,		/* nig.tsgen_tsio_in_val (1 regs) */
	0x01142413,		/* nig.ts_for_pxp_select (1 regs) */
	0x02942414,		/* nig.ptm_time_latch (2 regs, WB) */
	0x0114241a,		/* nig.mpa_mul_pdu_crc_calc_en (1 regs) */
	0x03480000,		/* block bmbn */
	0x02184010,		/* bmbn.INT_STS_0 .. bmbn.INT_MASK_0 (2 regs) */
	0x05184050,		/* bmbn.dbg_select .. bmbn.dbg_force_frame (5 regs) */
	0x0318407e,		/* bmbn.tag_len_0 .. bmbn.eco_reserved (3 regs) */
	0x024e0000,		/* block led */
	0x061ae006,		/* led.mac_led_swap .. led.eco_reserved (6 regs) */
	0x021ae060,		/* led.INT_STS_0 .. led.INT_MASK_0 (2 regs) */
	0x024f0000,		/* block avs_wrap */
	0x041ad000,		/* avs_wrap.avs_control .. avs_wrap.INT_MASK (4 regs) */
	0x091ad00a,		/* avs_wrap.eco_reserved .. avs_wrap.efuse_data_word_23 (9 regs) */
	0x01500000,		/* block pxpreqbus */
	0x01015880,		/* pxpreqbus.eco_reserved (1 regs) */
	0x08010081,		/* mode !k2, block miscs */
	0x03002458,		/* miscs.memctrl_wr_rd_n .. miscs.memctrl_address (3 regs) */
	0x02002464,		/* miscs.INT_STS_1 .. miscs.INT_MASK_1 (2 regs) */
	0x010025c1,		/* miscs.nig_dbg_vector (1 regs) */
	0x030025e8,		/* miscs.pcie_rst_prepared_assert_cnt .. miscs.pcie_rst_deassert_cnt (3 regs) */
	0x010025ec,		/* miscs.pcie_rst_n (1 regs) */
	0x030025ef,		/* miscs.avs_otp_sram_ctrl .. miscs.avs_otp_ctrl_vmgmt (3 regs) */
	0x060025f4,		/* miscs.avs_pvtmon_daccode .. miscs.opte_almfull_thr (6 regs) */
	0x040025fb,		/* miscs.avs_clock_observe .. miscs.avs_tp_out (4 regs) */
	0x01020000,		/* block misc */
	0x02002301,		/* misc.xmac_phy_port_mode .. misc.xmac_core_port_mode (2 regs) */
	0x01040000,		/* block pglue_b */
	0x030aa12f,		/* pglue_b.memctrl_wr_rd_n .. pglue_b.memctrl_address (3 regs) */
	0x04050000,		/* block cnig */
	0x02086080,		/* cnig.nw_port_mode .. cnig.nw_serdes_swap (2 regs) */
	0x25086095,		/* cnig.mac_led_swap .. cnig.cnig_dbg_ifmux_phy_lasi_b (37 regs) */
	0x010860be,		/* cnig.cnig_dbg_nigtx_fifo_afull_thresh_large (1 regs) */
	0x0a0860c8,		/* cnig.pmeg_sign_ext .. cnig.pmfc_crc_tx_corrupt_on_error (10 regs) */
	0x0c150000,		/* block pglcs */
	0x02000904,		/* pglcs.pgl_cs.pm_data_a .. pglcs.pgl_cs.pm_data_b (2 regs) */
	0x0200090e,		/* pglcs.pgl_cs.reg_id_val2 .. pglcs.pgl_cs.reg_id_val3 (2 regs) */
	0x01000932,		/* pglcs.pgl_cs.reg_msix_pba_off_bir (1 regs) */
	0x01000934,		/* pglcs.pgl_cs.reg_pcie_capability (1 regs) */
	0x01000937,		/* pglcs.pgl_cs.reg_link_capability (1 regs) */
	0x0300093a,		/* pglcs.pgl_cs.reg_pcie_link_capability_2 .. pglcs.pgl_cs.reg_pcie_link_capability_rc (3 regs) */
	0x04000940,		/* pglcs.pgl_cs.reg_dev_ser_num_cap_id .. pglcs.pgl_cs.reg_adv_err_cap (4 regs) */
	0x01000955,		/* pglcs.pgl_cs.reg_vsec_hdr (1 regs) */
	0x0100097c,		/* pglcs.pgl_cs.reg_PTM_CAP (1 regs) */
	0x01000983,		/* pglcs.pgl_cs.reg_VF_SUPP_PAGE_SIZE (1 regs) */
	0x02000985,		/* pglcs.pgl_cs.reg_VF_MSIX_TBL_BIR_OFF .. pglcs.pgl_cs.reg_VF_msix_pba_off_bit (2 regs) */
	0x0100098a,		/* pglcs.pgl_cs.reg_vf_nsp (1 regs) */
	0x01160000,		/* block dmae */
	0x03003140,		/* dmae.memctrl_wr_rd_n .. dmae.memctrl_address (3 regs) */
	0x011a0000,		/* block ucm */
	0x034a0010,		/* ucm.memctrl_wr_rd_n .. ucm.memctrl_address (3 regs) */
	0x01200000,		/* block dorq */
	0x030402b0,		/* dorq.memctrl_wr_rd_n .. dorq.memctrl_address (3 regs) */
	0x01210000,		/* block brb */
	0x040d0700,		/* brb.memctrl_wr_rd_n .. brb.memctrl_status (4 regs) */
	0x01230000,		/* block prs */
	0x0707c3e0,		/* prs.cam_bist_en .. prs.cam_bist_dbg_compare_en (7 regs) */
	0x012a0000,		/* block tsem */
	0x045d0330,		/* tsem.fast_memory.memctrl_wr_rd_n .. tsem.fast_memory.memctrl_status (4 regs) */
	0x012b0000,		/* block msem */
	0x04610330,		/* msem.fast_memory.memctrl_wr_rd_n .. msem.fast_memory.memctrl_status (4 regs) */
	0x012c0000,		/* block usem */
	0x04650330,		/* usem.fast_memory.memctrl_wr_rd_n .. usem.fast_memory.memctrl_status (4 regs) */
	0x012d0000,		/* block xsem */
	0x04510330,		/* xsem.fast_memory.memctrl_wr_rd_n .. xsem.fast_memory.memctrl_status (4 regs) */
	0x012e0000,		/* block ysem */
	0x04550330,		/* ysem.fast_memory.memctrl_wr_rd_n .. ysem.fast_memory.memctrl_status (4 regs) */
	0x012f0000,		/* block psem */
	0x04590330,		/* psem.fast_memory.memctrl_wr_rd_n .. psem.fast_memory.memctrl_status (4 regs) */
	0x01300000,		/* block rss */
	0x0408e322,		/* rss.memctrl_wr_rd_n .. rss.memctrl_status (4 regs) */
	0x013a0000,		/* block pbf */
	0x03360040,		/* pbf.memctrl_wr_rd_n .. pbf.memctrl_address (3 regs) */
	0x013d0000,		/* block cdu */
	0x041601d3,		/* cdu.memctrl_wr_rd_n .. cdu.memctrl_status (4 regs) */
	0x01400000,		/* block igu */
	0x07060018,		/* igu.cam_bist_en .. igu.cam_bist_dbg_compare_en (7 regs) */
	0x02450000,		/* block dbg */
	0x0a0042b0,		/* dbg.cpu_mbist_memctrl_0_cntrl_cmd .. dbg.cpu_mbist_memctrl_9_cntrl_cmd (10 regs) */
	0x030042e5,		/* dbg.memctrl_wr_rd_n .. dbg.memctrl_address (3 regs) */
	0x02460000,		/* block nig */
	0x04142218,		/* nig.user_one_step_type .. nig.ts_shift (4 regs) */
	0x0114221e,		/* nig.user_one_step_32 (1 regs) */
	0x0203006f,		/* mode !(emul_reduced|fpga), block dbu */
	0x02002802,		/* dbu.config .. dbu.timing (2 regs) */
	0x02002805,		/* dbu.txdata .. dbu.vfid_cfg (2 regs) */
	0x0b060000,		/* block cpmu */
	0x0200c088,		/* cpmu.obff_mode_config .. cpmu.obff_mode_control (2 regs) */
	0x1300c08b,		/* cpmu.obff_mem_timer_short_threshold .. cpmu.sw_force_l1 (19 regs) */
	0x0500c09f,		/* cpmu.ltr_mode_config .. cpmu.sw_force_ltr (5 regs) */
	0x0e00c0a5,		/* cpmu.clk_en_config .. cpmu.sw_force_main_clk_slowdown (14 regs) */
	0x0300c0b4,		/* cpmu.storm_clk_slowdown_entry_en .. cpmu.sw_force_storm_clk_slowdown (3 regs) */
	0x0300c0b8,		/* cpmu.nw_clk_slowdown_entry_en .. cpmu.sw_force_nw_clk_slowdown (3 regs) */
	0x0300c0bc,		/* cpmu.pci_clk_slowdown_entry_en .. cpmu.sw_force_pci_clk_slowdown (3 regs) */
	0x0900c0c0,		/* cpmu.pxp_vq_empty_status_e0_0 .. cpmu.cpmu_output_sig_status (9 regs) */
	0x1000c0cf,		/* cpmu.obff_stall_mem_stat_ro .. cpmu.pcs_duration_stat_ro (16 regs) */
	0x1500c0e5,		/* cpmu.obff_stall_mem_stat .. cpmu.INT_MASK_0 (21 regs) */
	0x0400c0fc,		/* cpmu.sdm_sq_counter_e0_p0 .. cpmu.sdm_sq_counter_e1_p1 (4 regs) */
	0x180b0000,		/* block mcp */
	0x0e380020,		/* mcp.mcp_control .. mcp.mcp_doorbell (14 regs) */
	0x09380031,		/* mcp.mcp_vfid .. mcp.gp_event_vec (9 regs) */
	0x03381400,		/* mcp.cpu_mode .. mcp.cpu_event_mask (3 regs) */
	0x03381407,		/* mcp.cpu_program_counter .. mcp.cpu_data_access (3 regs) */
	0x0438140b,		/* mcp.cpu_interrupt_vector .. mcp.cpu_debug_vect_peek (4 regs) */
	0x01381412,		/* mcp.cpu_last_branch_addr (1 regs) */
	0x0538152a,		/* mcp.mdio_auto_poll .. mcp.mdio_auto_status (5 regs) */
	0x0b381640,		/* mcp.ucint_warp_mode .. mcp.ucint_avs_address (11 regs) */
	0x02381680,		/* mcp.imc_command .. mcp.imc_slave_control (2 regs) */
	0x07381685,		/* mcp.imc_timing0 .. mcp.imc_datareg3 (7 regs) */
	0x01381840,		/* mcp.m2p_m2p_status (1 regs) */
	0x09381842,		/* mcp.m2p_m2p_vdm_length .. mcp.m2p_m2p_path_id (9 regs) */
	0x17381880,		/* mcp.p2m_p2m_status .. mcp.p2m_p2m_other_hdr_fields (23 regs) */
	0x023818c0,		/* mcp.cache_pim_nvram_base .. mcp.cache_paging_enable (2 regs) */
	0x1b3818c3,		/* mcp.cache_cache_ctrl_status_0 .. mcp.cache_cache_error_status (27 regs) */
	0x0d381900,		/* mcp.nvm_command .. mcp.nvm_reconfig (13 regs) */
	0x16381a00,		/* mcp.erngn_exp_rom_ctrl .. mcp.erngn_img_loader2_cfg (22 regs) */
	0x14382000,		/* mcp.smbus_config .. mcp.smbus_slave_data_read (20 regs) */
	0x01382020,		/* mcp.smbus_arp_state (1 regs) */
	0x08382024,		/* mcp.smbus_udid0_3 .. mcp.smbus_udid1_0 (8 regs) */
	0x013820ff,		/* mcp.smbus_smb_reg_end (1 regs) */
	0x01382109,		/* mcp.frm_bmb_fifo_status (1 regs) */
	0x0538210b,		/* mcp.frm_bmb_fifo_rd_data .. mcp.frm_bmb_fifo_sop_dscr3 (5 regs) */
	0x013821ff,		/* mcp.bmb_reg_end (1 regs) */
	0x042f0000,		/* block psem */
	0x0e592800,		/* psem.fast_memory.vfc_config.mask_lsb_0_low .. psem.fast_memory.vfc_config.indications2 (14 regs) */
	0x1359280f,		/* psem.fast_memory.vfc_config.memories_rst .. psem.fast_memory.vfc_config.cpu_mbist_memctrl_1_cntrl_cmd (19 regs) */
	0x12592824,		/* psem.fast_memory.vfc_config.debug_data .. psem.fast_memory.vfc_config.mask_lsb_7_high (18 regs) */
	0x0c59283e,		/* psem.fast_memory.vfc_config.offset_alu_vector_0 .. psem.fast_memory.vfc_config.cam_bist_skip_error_cnt (12 regs) */
	0x01060069,		/* mode (!bb)&(!(emul_reduced|fpga)), block cpmu */
	0x0200c100,		/* cpmu.sdm_sq_counter_e0_p2 .. cpmu.sdm_sq_counter_e0_p3 (2 regs) */
	0x020700d9,		/* mode !(!asic), block ncsi */
	0xa2010080,		/* ncsi.config .. ncsi.dbg_force_frame (162 regs) */
	0x03010132,		/* ncsi.eco_reserved .. ncsi.INT_MASK_0 (3 regs) */
	0x25090000,		/* block bmb */
	0x03150001,		/* bmb.hw_init_en .. bmb.start_en (3 regs) */
	0x02150030,		/* bmb.INT_STS_0 .. bmb.INT_MASK_0 (2 regs) */
	0x02150036,		/* bmb.INT_STS_1 .. bmb.INT_MASK_1 (2 regs) */
	0x0215003c,		/* bmb.INT_STS_2 .. bmb.INT_MASK_2 (2 regs) */
	0x02150042,		/* bmb.INT_STS_3 .. bmb.INT_MASK_3 (2 regs) */
	0x02150048,		/* bmb.INT_STS_4 .. bmb.INT_MASK_4 (2 regs) */
	0x0215004e,		/* bmb.INT_STS_5 .. bmb.INT_MASK_5 (2 regs) */
	0x02150054,		/* bmb.INT_STS_6 .. bmb.INT_MASK_6 (2 regs) */
	0x0215005a,		/* bmb.INT_STS_7 .. bmb.INT_MASK_7 (2 regs) */
	0x02150061,		/* bmb.INT_STS_8 .. bmb.INT_MASK_8 (2 regs) */
	0x02150067,		/* bmb.INT_STS_9 .. bmb.INT_MASK_9 (2 regs) */
	0x0215006d,		/* bmb.INT_STS_10 .. bmb.INT_MASK_10 (2 regs) */
	0x02150073,		/* bmb.INT_STS_11 .. bmb.INT_MASK_11 (2 regs) */
	0x02150200,		/* bmb.big_ram_address .. bmb.header_size (2 regs) */
	0x0a150210,		/* bmb.max_releases .. bmb.tc_guarantied_5 (10 regs) */
	0x0615021e,		/* bmb.tc_guarantied_hyst_0 .. bmb.tc_guarantied_hyst_5 (6 regs) */
	0x06150228,		/* bmb.tc_pause_xoff_threshold_0 .. bmb.tc_pause_xoff_threshold_5 (6 regs) */
	0x06150232,		/* bmb.tc_pause_xon_threshold_0 .. bmb.tc_pause_xon_threshold_5 (6 regs) */
	0x0615023c,		/* bmb.tc_full_xoff_threshold_0 .. bmb.tc_full_xoff_threshold_5 (6 regs) */
	0x06150246,		/* bmb.tc_full_xon_threshold_0 .. bmb.tc_full_xon_threshold_5 (6 regs) */
	0x02150250,		/* bmb.no_dead_cycles_en .. bmb.rc_pkt_priority (2 regs) */
	0x0d150260,		/* bmb.rc_sop_priority .. bmb.pm_tc_latency_sensitive_0 (13 regs) */
	0x0615028c,		/* bmb.pm_cos_threshold_0 .. bmb.pm_cos_threshold_5 (6 regs) */
	0x0815029c,		/* bmb.dbgsyn_almost_full_thr .. bmb.dbg_force_frame (8 regs) */
	0x0c1502b2,		/* bmb.inp_if_enable .. bmb.wc_empty_9 (12 regs) */
	0x0a1502c4,		/* bmb.wc_full_0 .. bmb.wc_full_9 (10 regs) */
	0x0c1502d4,		/* bmb.wc_bandwidth_if_full .. bmb.rc_pkt_empty_9 (12 regs) */
	0x0a1502e5,		/* bmb.rc_pkt_full_0 .. bmb.rc_pkt_full_9 (10 regs) */
	0x0a1502f4,		/* bmb.rc_pkt_status_0 .. bmb.rc_pkt_status_9 (10 regs) */
	0x43150303,		/* bmb.rc_sop_empty .. bmb.rc_out_sync_fifo_push_status_9 (67 regs) */
	0x02150353,		/* bmb.wc_sync_fifo_push_status_8 .. bmb.wc_sync_fifo_push_status_9 (2 regs) */
	0x03150362,		/* bmb.pkt_avail_sync_fifo_push_status .. bmb.rc_pkt_state_1 (3 regs) */
	0x01150367,		/* bmb.mac_free_shared_hr_0 (1 regs) */
	0x0615036d,		/* bmb.tc_occupancy_0 .. bmb.tc_occupancy_5 (6 regs) */
	0x0115037d,		/* bmb.available_mac_size_0 (1 regs) */
	0x01150383,		/* bmb.tc_pause_0 (1 regs) */
	0x01150389,		/* bmb.tc_full_0 (1 regs) */
	0x06420000,		/* block umac */
	0x02014401,		/* umac.ipg_hd_bkp_cntl .. umac.command_config (2 regs) */
	0x01014405,		/* umac.frm_length (1 regs) */
	0x01014411,		/* umac.mac_mode (1 regs) */
	0x01014417,		/* umac.tx_ipg_length (1 regs) */
	0x010144c0,		/* umac.mac_pfc_type (1 regs) */
	0x010144cc,		/* umac.pause_control (1 regs) */
	0x06440000,		/* block mstat */
	0x36085080,		/* mstat.TX_STAT_GTXPOK_lo .. mstat.DOT3STATSLATECOLLISIONS_hi (54 regs) */
	0x3c085100,		/* mstat.RX_STAT_GR64_lo .. mstat.RxLLFCMSGCNT_hi (60 regs) */
	0x36885280,		/* mstat.TX_STAT_GTXPOK_wb .. mstat.DOT3STATSLATECOLLISIONS_wb (54 regs, WB) */
	0x3c885300,		/* mstat.RX_STAT_GR64_wb .. mstat.RxLLFCMSGCNT_wb (60 regs, WB) */
	0x010853fc,		/* mstat.INT_STS (1 regs) */
	0x010853ff,		/* mstat.INT_MASK (1 regs) */
	0x010900d3,		/* mode (!bb)&(!(!asic)), block bmb */
	0x02150252,		/* bmb.wc_no_dead_cycles_en .. bmb.wc_highest_pri_en (2 regs) */
	0x04420000,		/* block umac */
	0x07014423,		/* umac.tx_ipg_length1 .. umac.dbg_force_frame (7 regs) */
	0x02014460,		/* umac.INT_STS .. umac.INT_MASK (2 regs) */
	0x070144c1,		/* umac.pause_opcode .. umac.mac_pause_sa_1 (7 regs) */
	0x030144cd,		/* umac.rsv_err_mask .. umac.probe_data (3 regs) */
	0x04470000,		/* block wol */
	0x02180010,		/* wol.INT_STS_0 .. wol.INT_MASK_0 (2 regs) */
	0x05180050,		/* wol.dbg_select .. wol.dbg_force_frame (5 regs) */
	0x06182067,		/* wol.tag_len_0 .. wol.tag_len_5 (6 regs) */
	0x0118206e,		/* wol.eco_reserved (1 regs) */
	0x03490000,		/* block ipc */
	0x26008080,		/* ipc.pll_main_divr .. ipc.sgmii_rstb_mdioregs (38 regs) */
	0x120080a7,		/* ipc.freq_main .. ipc.INT_MASK_0 (18 regs) */
	0x230080bb,		/* ipc.jtag_compliance .. ipc.eco_reserved (35 regs) */
	0xff4b0000,		/* block nws */
	0x171c0000,		/* nws.common_control .. nws.eco_reserved (23 regs) */
	0x0b1c004a,		/* nws.dbg_select .. nws.dbg_fw_trigger_enable (11 regs) */
	0x021c0060,		/* nws.INT_STS_0 .. nws.INT_MASK_0 (2 regs) */
	0x021c0064,		/* nws.INT_STS_1 .. nws.INT_MASK_1 (2 regs) */
	0x021c0068,		/* nws.INT_STS_2 .. nws.INT_MASK_2 (2 regs) */
	0x021c006c,		/* nws.INT_STS_3 .. nws.INT_MASK_3 (2 regs) */
	0x031c8000,		/* nws.nws_cmu.phy0_top_ReservedRegister0 .. nws.nws_cmu.phy0_top_ReservedRegister2 (3 regs) */
	0x031c8004,		/* nws.nws_cmu.phy0_top_ReservedRegister3 .. nws.nws_cmu.phy0_top_ReservedRegister5 (3 regs) */
	0x011c8009,		/* nws.nws_cmu.phy0_top_ReservedRegister6 (1 regs) */
	0x041c8030,		/* nws.nws_cmu.phy0_top_ReservedRegister7 .. nws.nws_cmu.phy0_top_ReservedRegister10 (4 regs) */
	0x021c8038,		/* nws.nws_cmu.phy0_top_afe_atest_ctrl0 .. nws.nws_cmu.phy0_top_afe_atest_ctrl1 (2 regs) */
	0x011c8040,		/* nws.nws_cmu.phy0_top_ReservedRegister11 (1 regs) */
	0x011c8050,		/* nws.nws_cmu.phy0_top_ReservedRegister12 (1 regs) */
	0x041c8054,		/* nws.nws_cmu.phy0_top_ReservedRegister13 .. nws.nws_cmu.phy0_top_ReservedRegister16 (4 regs) */
	0x0b1c80f0,		/* nws.nws_cmu.phy0_top_ReservedRegister17 .. nws.nws_cmu.phy0_top_clock_cm_lc0_clk_cmudiv_ctrl1 (11 regs) */
	0x0a1c8100,		/* nws.nws_cmu.phy0_top_ReservedRegister26 .. nws.nws_cmu.phy0_top_clock_cm_r0_clk_pll3div_ctrl1 (10 regs) */
	0x061c8110,		/* nws.nws_cmu.phy0_top_clock_ln0_clk_tx .. nws.nws_cmu.phy0_top_ReservedRegister35 (6 regs) */
	0x061c8118,		/* nws.nws_cmu.phy0_top_clock_ln1_clk_tx .. nws.nws_cmu.phy0_top_ReservedRegister39 (6 regs) */
	0x061c8120,		/* nws.nws_cmu.phy0_top_clock_ln2_clk_tx .. nws.nws_cmu.phy0_top_ReservedRegister43 (6 regs) */
	0x061c8128,		/* nws.nws_cmu.phy0_top_clock_ln3_clk_tx .. nws.nws_cmu.phy0_top_ReservedRegister47 (6 regs) */
	0x021c8130,		/* nws.nws_cmu.phy0_top_ReservedRegister48 .. nws.nws_cmu.phy0_top_ReservedRegister49 (2 regs) */
	0x011c8170,		/* nws.nws_cmu.phy0_top_ReservedRegister50 (1 regs) */
	0x031c8180,		/* nws.nws_cmu.phy0_top_err_ctrl0 .. nws.nws_cmu.phy0_top_err_ctrl2 (3 regs) */
	0x061c8187,		/* nws.nws_cmu.phy0_top_regbus_err_info_ctrl .. nws.nws_cmu.phy0_top_regbus_err_info_status4 (6 regs) */
	0x051c81a0,		/* nws.nws_cmu.phy0_top_tbus_addr_7_0 .. nws.nws_cmu.phy0_top_ReservedRegister53 (5 regs) */
	0x021c81b0,		/* nws.nws_cmu.phy0_top_tbus_data_7_0 .. nws.nws_cmu.phy0_top_tbus_data_11_8 (2 regs) */
	0x021c81c0,		/* nws.nws_cmu.phy0_top_sim_ctrl .. nws.nws_cmu.phy0_top_fw_ctrl (2 regs) */
	0x011c8200,		/* nws.nws_cmu.phy0_mb_cmd (1 regs) */
	0x081c8203,		/* nws.nws_cmu.phy0_mb_cmd_data0 .. nws.nws_cmu.phy0_mb_cmd_data7 (8 regs) */
	0x011c8210,		/* nws.nws_cmu.phy0_mb_rsp (1 regs) */
	0x101c8213,		/* nws.nws_cmu.phy0_mb_rsp_data0 .. nws.nws_cmu.phy0_mb_rsp_data15 (16 regs) */
	0x221c8300,		/* nws.nws_cmu.phy0_ovr_cmu_lc_ReservedRegister54 .. nws.nws_cmu.phy0_ovr_cmu_lc_ReservedRegister87 (34 regs) */
	0x0a1c8380,		/* nws.nws_cmu.phy0_ovr_cmu_r_ReservedRegister88 .. nws.nws_cmu.phy0_ovr_cmu_r_ReservedRegister97 (10 regs) */
	0x931c8400,		/* nws.nws_cmu.phy0_ovr_ln0_ReservedRegister98 .. nws.nws_cmu.phy0_ovr_ln0_ReservedRegister244 (147 regs) */
	0x931c8500,		/* nws.nws_cmu.phy0_ovr_ln1_ReservedRegister245 .. nws.nws_cmu.phy0_ovr_ln1_ReservedRegister391 (147 regs) */
	0x931c8600,		/* nws.nws_cmu.phy0_ovr_ln2_ReservedRegister392 .. nws.nws_cmu.phy0_ovr_ln2_ReservedRegister538 (147 regs) */
	0x931c8700,		/* nws.nws_cmu.phy0_ovr_ln3_ReservedRegister539 .. nws.nws_cmu.phy0_ovr_ln3_ReservedRegister685 (147 regs) */
	0x021c8800,		/* nws.nws_cmu.cmu_lc0_top_ReservedRegister686 .. nws.nws_cmu.cmu_lc0_top_ReservedRegister687 (2 regs) */
	0x011c8803,		/* nws.nws_cmu.cmu_lc0_top_ReservedRegister688 (1 regs) */
	0x061c8805,		/* nws.nws_cmu.cmu_lc0_top_ReservedRegister689 .. nws.nws_cmu.cmu_lc0_top_ReservedRegister694 (6 regs) */
	0x011c880c,		/* nws.nws_cmu.cmu_lc0_top_ReservedRegister695 (1 regs) */
	0x021c881a,		/* nws.nws_cmu.cmu_lc0_top_ReservedRegister696 .. nws.nws_cmu.cmu_lc0_top_ReservedRegister697 (2 regs) */
	0x081c881f,		/* nws.nws_cmu.cmu_lc0_top_ReservedRegister698 .. nws.nws_cmu.cmu_lc0_top_ReservedRegister704 (8 regs) */
	0x011c8828,		/* nws.nws_cmu.cmu_lc0_top_afe_tstclk_ctrl0 (1 regs) */
	0x051c8830,		/* nws.nws_cmu.cmu_lc0_top_ReservedRegister705 .. nws.nws_cmu.cmu_lc0_top_ReservedRegister709 (5 regs) */
	0x031c8850,		/* nws.nws_cmu.cmu_lc0_top_ReservedRegister710 .. nws.nws_cmu.cmu_lc0_top_phy_if_status (3 regs) */
	0x021c8858,		/* nws.nws_cmu.cmu_lc0_top_ReservedRegister712 .. nws.nws_cmu.cmu_lc0_top_ReservedRegister713 (2 regs) */
	0x031c8880,		/* nws.nws_cmu.cmu_lc0_top_err_ctrl1 .. nws.nws_cmu.cmu_lc0_top_err_ctrl3 (3 regs) */
	0x031c888a,		/* nws.nws_cmu.cmu_lc0_top_ReservedRegister714 .. nws.nws_cmu.cmu_lc0_top_ReservedRegister716 (3 regs) */
	0x041c8900,		/* nws.nws_cmu.cmu_lc0_pll_ReservedRegister717 .. nws.nws_cmu.cmu_lc0_pll_afe_reg_ctrl1 (4 regs) */
	0x021c8905,		/* nws.nws_cmu.cmu_lc0_pll_ReservedRegister720 .. nws.nws_cmu.cmu_lc0_pll_ReservedRegister721 (2 regs) */
	0x031c8908,		/* nws.nws_cmu.cmu_lc0_pll_ReservedRegister722 .. nws.nws_cmu.cmu_lc0_pll_ReservedRegister724 (3 regs) */
	0x021c8910,		/* nws.nws_cmu.cmu_lc0_pll_ReservedRegister725 .. nws.nws_cmu.cmu_lc0_pll_ReservedRegister726 (2 regs) */
	0x041c8913,		/* nws.nws_cmu.cmu_lc0_pll_ReservedRegister727 .. nws.nws_cmu.cmu_lc0_pll_ReservedRegister730 (4 regs) */
	0x051c8918,		/* nws.nws_cmu.cmu_lc0_pll_ReservedRegister731 .. nws.nws_cmu.cmu_lc0_pll_ReservedRegister735 (5 regs) */
	0x1b1c8920,		/* nws.nws_cmu.cmu_lc0_pll_ReservedRegister736 .. nws.nws_cmu.cmu_lc0_pll_ReservedRegister762 (27 regs) */
	0x011c893c,		/* nws.nws_cmu.cmu_lc0_pll_ReservedRegister763 (1 regs) */
	0x031c8944,		/* nws.nws_cmu.cmu_lc0_pll_ReservedRegister764 .. nws.nws_cmu.cmu_lc0_pll_lockdet_status (3 regs) */
	0x091c8949,		/* nws.nws_cmu.cmu_lc0_pll_ReservedRegister766 .. nws.nws_cmu.cmu_lc0_pll_ReservedRegister774 (9 regs) */
	0x021c8954,		/* nws.nws_cmu.cmu_lc0_pll_ReservedRegister775 .. nws.nws_cmu.cmu_lc0_pll_ReservedRegister776 (2 regs) */
	0x021c8958,		/* nws.nws_cmu.cmu_lc0_pll_ReservedRegister777 .. nws.nws_cmu.cmu_lc0_pll_ReservedRegister778 (2 regs) */
	0x041c8a00,		/* nws.nws_cmu.cmu_lc0_gcfsm2_ReservedRegister779 .. nws.nws_cmu.cmu_lc0_gcfsm2_ReservedRegister782 (4 regs) */
	0x041c8a10,		/* nws.nws_cmu.cmu_lc0_gcfsm2_ReservedRegister783 .. nws.nws_cmu.cmu_lc0_gcfsm2_ReservedRegister786 (4 regs) */
	0x071c8a20,		/* nws.nws_cmu.cmu_lc0_gcfsm2_ReservedRegister787 .. nws.nws_cmu.cmu_lc0_gcfsm2_ReservedRegister793 (7 regs) */
	0x031c8a30,		/* nws.nws_cmu.cmu_lc0_gcfsm2_ReservedRegister794 .. nws.nws_cmu.cmu_lc0_gcfsm2_ReservedRegister796 (3 regs) */
	0x091c8a40,		/* nws.nws_cmu.cmu_lc0_gcfsm2_ReservedRegister797 .. nws.nws_cmu.cmu_lc0_gcfsm2_ReservedRegister805 (9 regs) */
	0x021c8a50,		/* nws.nws_cmu.cmu_lc0_gcfsm2_ReservedRegister806 .. nws.nws_cmu.cmu_lc0_gcfsm2_ReservedRegister807 (2 regs) */
	0x061c8b00,		/* nws.nws_cmu.cmu_lc0_feature_ReservedRegister808 .. nws.nws_cmu.cmu_lc0_feature_ReservedRegister813 (6 regs) */
	0x011c8b08,		/* nws.nws_cmu.cmu_lc0_feature_ReservedRegister814 (1 regs) */
	0x081c8b10,		/* nws.nws_cmu.cmu_lc0_feature_ReservedRegister815 .. nws.nws_cmu.cmu_lc0_feature_ReservedRegister822 (8 regs) */
	0x021c8c00,		/* nws.nws_cmu.cmu_r0_top_ReservedRegister823 .. nws.nws_cmu.cmu_r0_top_ReservedRegister824 (2 regs) */
	0x011c8c03,		/* nws.nws_cmu.cmu_r0_top_ReservedRegister825 (1 regs) */
	0x041c8c10,		/* nws.nws_cmu.cmu_r0_top_ReservedRegister826 .. nws.nws_cmu.cmu_r0_top_ReservedRegister829 (4 regs) */
	0x041c8c20,		/* nws.nws_cmu.cmu_r0_top_ReservedRegister830 .. nws.nws_cmu.cmu_r0_top_ReservedRegister833 (4 regs) */
	0x011c8c30,		/* nws.nws_cmu.cmu_r0_top_ReservedRegister834 (1 regs) */
	0x031c8c50,		/* nws.nws_cmu.cmu_r0_top_ReservedRegister835 .. nws.nws_cmu.cmu_r0_top_phy_if_status (3 regs) */
	0x021c8c58,		/* nws.nws_cmu.cmu_r0_top_ReservedRegister837 .. nws.nws_cmu.cmu_r0_top_ReservedRegister838 (2 regs) */
	0x031c8c80,		/* nws.nws_cmu.cmu_r0_top_err_ctrl1 .. nws.nws_cmu.cmu_r0_top_err_ctrl3 (3 regs) */
	0x031c8c8a,		/* nws.nws_cmu.cmu_r0_top_ReservedRegister839 .. nws.nws_cmu.cmu_r0_top_ReservedRegister841 (3 regs) */
	0x051c8d00,		/* nws.nws_cmu.cmu_r0_rpll_afe_pd_ctrl0 .. nws.nws_cmu.cmu_r0_rpll_afe_vco_ctrl0 (5 regs) */
	0x011c8d06,		/* nws.nws_cmu.cmu_r0_rpll_afe_clkdiv_ctrl0 (1 regs) */
	0x061c8d08,		/* nws.nws_cmu.cmu_r0_rpll_ReservedRegister843 .. nws.nws_cmu.cmu_r0_rpll_afe_int_ctrl0 (6 regs) */
	0x041c8d0f,		/* nws.nws_cmu.cmu_r0_rpll_afe_int_ctrl1 .. nws.nws_cmu.cmu_r0_rpll_afe_fracn_ctrl0 (4 regs) */
	0x021c8d16,		/* nws.nws_cmu.cmu_r0_rpll_afe_misc_ctrl0 .. nws.nws_cmu.cmu_r0_rpll_afe_misc_ctrl1 (2 regs) */
	0x131c8d20,		/* nws.nws_cmu.cmu_r0_rpll_ReservedRegister844 .. nws.nws_cmu.cmu_r0_rpll_ReservedRegister862 (19 regs) */
	0x071c8d34,		/* nws.nws_cmu.cmu_r0_rpll_ReservedRegister863 .. nws.nws_cmu.cmu_r0_rpll_ReservedRegister869 (7 regs) */
	0x031c8d44,		/* nws.nws_cmu.cmu_r0_rpll_ReservedRegister870 .. nws.nws_cmu.cmu_r0_rpll_lockdet_status (3 regs) */
	0x091c8d49,		/* nws.nws_cmu.cmu_r0_rpll_ssc_gen_ctrl0 .. nws.nws_cmu.cmu_r0_rpll_fracn_ctrl2 (9 regs) */
	0x021c8d54,		/* nws.nws_cmu.cmu_r0_rpll_fracn_ctrl3 .. nws.nws_cmu.cmu_r0_rpll_fracn_ctrl4 (2 regs) */
	0x021c8d58,		/* nws.nws_cmu.cmu_r0_rpll_ReservedRegister872 .. nws.nws_cmu.cmu_r0_rpll_ReservedRegister873 (2 regs) */
	0x041c8e00,		/* nws.nws_cmu.cmu_r0_gcfsm2_ReservedRegister874 .. nws.nws_cmu.cmu_r0_gcfsm2_ReservedRegister877 (4 regs) */
	0x041c8e10,		/* nws.nws_cmu.cmu_r0_gcfsm2_ReservedRegister878 .. nws.nws_cmu.cmu_r0_gcfsm2_ReservedRegister881 (4 regs) */
	0x071c8e20,		/* nws.nws_cmu.cmu_r0_gcfsm2_ReservedRegister882 .. nws.nws_cmu.cmu_r0_gcfsm2_ReservedRegister888 (7 regs) */
	0x031c8e30,		/* nws.nws_cmu.cmu_r0_gcfsm2_ReservedRegister889 .. nws.nws_cmu.cmu_r0_gcfsm2_ReservedRegister891 (3 regs) */
	0x091c8e40,		/* nws.nws_cmu.cmu_r0_gcfsm2_ReservedRegister892 .. nws.nws_cmu.cmu_r0_gcfsm2_ReservedRegister900 (9 regs) */
	0x021c8e50,		/* nws.nws_cmu.cmu_r0_gcfsm2_ReservedRegister901 .. nws.nws_cmu.cmu_r0_gcfsm2_ReservedRegister902 (2 regs) */
	0x011c8f00,		/* nws.nws_cmu.cmu_r0_feature_ReservedRegister903 (1 regs) */
	0x021c8f04,		/* nws.nws_cmu.cmu_r0_feature_ReservedRegister904 .. nws.nws_cmu.cmu_r0_feature_ReservedRegister905 (2 regs) */
	0x011c8f08,		/* nws.nws_cmu.cmu_r0_feature_ReservedRegister906 (1 regs) */
	0x081c8f10,		/* nws.nws_cmu.cmu_r0_feature_ReservedRegister907 .. nws.nws_cmu.cmu_r0_feature_ReservedRegister914 (8 regs) */
	0x071c9800,		/* nws.nws_cmu.ln0_top_afe_loopback_ctrl .. nws.nws_cmu.ln0_top_ReservedRegister920 (7 regs) */
	0x011c9810,		/* nws.nws_cmu.ln0_top_ReservedRegister921 (1 regs) */
	0x031c9812,		/* nws.nws_cmu.ln0_top_ReservedRegister922 .. nws.nws_cmu.ln0_top_ReservedRegister924 (3 regs) */
	0x011c9816,		/* nws.nws_cmu.ln0_top_ReservedRegister925 (1 regs) */
	0x011c9819,		/* nws.nws_cmu.ln0_top_ReservedRegister926 (1 regs) */
	0x021c981b,		/* nws.nws_cmu.ln0_top_ReservedRegister927 .. nws.nws_cmu.ln0_top_ReservedRegister928 (2 regs) */
	0x011c981e,		/* nws.nws_cmu.ln0_top_ReservedRegister929 (1 regs) */
	0x011c9822,		/* nws.nws_cmu.ln0_top_dpl_txdp_ctrl1 (1 regs) */
	0x041c9824,		/* nws.nws_cmu.ln0_top_dpl_rxdp_ctrl1 .. nws.nws_cmu.ln0_top_phy_if_status (4 regs) */
	0x021c9830,		/* nws.nws_cmu.ln0_top_ReservedRegister932 .. nws.nws_cmu.ln0_top_ReservedRegister933 (2 regs) */
	0x091c9838,		/* nws.nws_cmu.ln0_top_ln_stat_ctrl0 .. nws.nws_cmu.ln0_top_ReservedRegister940 (9 regs) */
	0x021c9842,		/* nws.nws_cmu.ln0_top_ReservedRegister941 .. nws.nws_cmu.ln0_top_ReservedRegister942 (2 regs) */
	0x051c9848,		/* nws.nws_cmu.ln0_top_ReservedRegister943 .. nws.nws_cmu.ln0_top_ReservedRegister947 (5 regs) */
	0x031c9850,		/* nws.nws_cmu.ln0_top_err_ctrl1 .. nws.nws_cmu.ln0_top_err_ctrl3 (3 regs) */
	0x021c9890,		/* nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister948 .. nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister949 (2 regs) */
	0x021c98a1,		/* nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister950 .. nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister951 (2 regs) */
	0x061c98a6,		/* nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister952 .. nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister957 (6 regs) */
	0x011c98ad,		/* nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister958 (1 regs) */
	0x031c98b0,		/* nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister959 .. nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister961 (3 regs) */
	0x011c98b4,		/* nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister962 (1 regs) */
	0x021c98b6,		/* nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister963 .. nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister964 (2 regs) */
	0x091c98b9,		/* nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister965 .. nws.nws_cmu.ln0_cdr_rxclk_dlpf_status4 (9 regs) */
	0x031c98c4,		/* nws.nws_cmu.ln0_cdr_rxclk_dlpf_status5 .. nws.nws_cmu.ln0_cdr_rxclk_integral_status1 (3 regs) */
	0x081c98c8,		/* nws.nws_cmu.ln0_cdr_rxclk_integral_status2 .. nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister977 (8 regs) */
	0x041c98e0,		/* nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister978 .. nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister981 (4 regs) */
	0x091c98e8,		/* nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister982 .. nws.nws_cmu.ln0_cdr_rxclk_ReservedRegister990 (9 regs) */
	0x021c9900,		/* nws.nws_cmu.ln0_cdr_refclk_ReservedRegister991 .. nws.nws_cmu.ln0_cdr_refclk_ReservedRegister992 (2 regs) */
	0x011c9904,		/* nws.nws_cmu.ln0_cdr_refclk_ReservedRegister993 (1 regs) */
	0x011c9906,		/* nws.nws_cmu.ln0_cdr_refclk_ReservedRegister994 (1 regs) */
	0x031c990a,		/* nws.nws_cmu.ln0_cdr_refclk_ReservedRegister995 .. nws.nws_cmu.ln0_cdr_refclk_ReservedRegister997 (3 regs) */
	0x021c9910,		/* nws.nws_cmu.ln0_cdr_refclk_ReservedRegister998 .. nws.nws_cmu.ln0_cdr_refclk_ReservedRegister999 (2 regs) */
	0x041c9918,		/* nws.nws_cmu.ln0_cdr_refclk_ReservedRegister1000 .. nws.nws_cmu.ln0_cdr_refclk_ReservedRegister1003 (4 regs) */
	0x061c9920,		/* nws.nws_cmu.ln0_cdr_refclk_ReservedRegister1004 .. nws.nws_cmu.ln0_cdr_refclk_ReservedRegister1009 (6 regs) */
	0x011c9930,		/* nws.nws_cmu.ln0_cdr_refclk_ReservedRegister1010 (1 regs) */
	0x0f1c9980,		/* nws.nws_cmu.ln0_aneg_ReservedRegister1011 .. nws.nws_cmu.ln0_aneg_ReservedRegister1022 (15 regs) */
	0x011c9990,		/* nws.nws_cmu.ln0_aneg_status0 (1 regs) */
	0x021c9994,		/* nws.nws_cmu.ln0_aneg_status_dbg0 .. nws.nws_cmu.ln0_aneg_status_dbg1 (2 regs) */
	0x251c9998,		/* nws.nws_cmu.ln0_aneg_base_page0 .. nws.nws_cmu.ln0_aneg_resolution_eee (37 regs) */
	0x021c99be,		/* nws.nws_cmu.ln0_aneg_link_status0 .. nws.nws_cmu.ln0_aneg_link_status1 (2 regs) */
	0x031c99c1,		/* nws.nws_cmu.ln0_eee_ReservedRegister1039 .. nws.nws_cmu.ln0_eee_ReservedRegister1041 (3 regs) */
	0x041c99c5,		/* nws.nws_cmu.ln0_eee_ReservedRegister1042 .. nws.nws_cmu.ln0_eee_ReservedRegister1045 (4 regs) */
	0x011c9a00,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1046 (1 regs) */
	0x021c9a02,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1047 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1048 (2 regs) */
	0x011c9a05,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1049 (1 regs) */
	0x011c9a07,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1050 (1 regs) */
	0x041c9a09,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1051 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1054 (4 regs) */
	0x041c9a0e,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1055 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1058 (4 regs) */
	0x0d1c9a20,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1059 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1071 (13 regs) */
	0x011c9a2e,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1072 (1 regs) */
	0x0b1c9a30,		/* nws.nws_cmu.ln0_leq_refclk_agclos_ctrl0 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1082 (11 regs) */
	0x021c9a3d,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1083 .. nws.nws_cmu.ln0_leq_refclk_ple_att_ctrl1 (2 regs) */
	0x071c9a40,		/* nws.nws_cmu.ln0_leq_refclk_eq_hfg_sql_ctrl0 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1089 (7 regs) */
	0x191c9a50,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1090 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1114 (25 regs) */
	0x071c9a70,		/* nws.nws_cmu.ln0_leq_refclk_gn_apg_ctrl0 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1120 (7 regs) */
	0x091c9a80,		/* nws.nws_cmu.ln0_leq_refclk_eq_lfg_ctrl0 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1126 (9 regs) */
	0x071c9a90,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1127 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1133 (7 regs) */
	0x061c9a98,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1134 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1138 (6 regs) */
	0x041c9aa0,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1139 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1142 (4 regs) */
	0x041c9aa6,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1143 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1146 (4 regs) */
	0x021c9aab,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1147 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1148 (2 regs) */
	0x021c9aae,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1149 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1150 (2 regs) */
	0x021c9ab8,		/* nws.nws_cmu.ln0_leq_refclk_ReservedRegister1151 .. nws.nws_cmu.ln0_leq_refclk_ReservedRegister1152 (2 regs) */
	0x061c9b00,		/* nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1153 .. nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1158 (6 regs) */
	0x021c9b08,		/* nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1159 .. nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1160 (2 regs) */
	0x021c9b0c,		/* nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1161 .. nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1162 (2 regs) */
	0x021c9b10,		/* nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1163 .. nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1164 (2 regs) */
	0x021c9b13,		/* nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1165 .. nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1166 (2 regs) */
	0x021c9b16,		/* nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1167 .. nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1168 (2 regs) */
	0x021c9b20,		/* nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1169 .. nws.nws_cmu.ln0_leq_rxclk_ReservedRegister1170 (2 regs) */
	0x071c9b80,		/* nws.nws_cmu.ln0_drv_refclk_afe_pd_ctrl0 .. nws.nws_cmu.ln0_drv_refclk_ReservedRegister1175 (7 regs) */
	0x061c9b88,		/* nws.nws_cmu.ln0_drv_refclk_ReservedRegister1176 .. nws.nws_cmu.ln0_drv_refclk_ReservedRegister1181 (6 regs) */
	0x0a1c9b90,		/* nws.nws_cmu.ln0_drv_refclk_txeq_ctrl0 .. nws.nws_cmu.ln0_drv_refclk_ReservedRegister1186 (10 regs) */
	0x011c9b9b,		/* nws.nws_cmu.ln0_drv_refclk_ReservedRegister1187 (1 regs) */
	0x051c9c00,		/* nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1188 .. nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1192 (5 regs) */
	0x011c9c06,		/* nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1193 (1 regs) */
	0x011c9c0a,		/* nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1194 (1 regs) */
	0x011c9c0c,		/* nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1195 (1 regs) */
	0x011c9c0e,		/* nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1196 (1 regs) */
	0x011c9c10,		/* nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1197 (1 regs) */
	0x011c9c12,		/* nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1198 (1 regs) */
	0x011c9c14,		/* nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1199 (1 regs) */
	0x011c9c16,		/* nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1200 (1 regs) */
	0x021c9c18,		/* nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1201 .. nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1202 (2 regs) */
	0x011c9c1b,		/* nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1203 (1 regs) */
	0x091c9c20,		/* nws.nws_cmu.ln0_dfe_refclk_fsm_ctrl0 .. nws.nws_cmu.ln0_dfe_refclk_fsm_status0 (9 regs) */
	0x191c9c2a,		/* nws.nws_cmu.ln0_dfe_refclk_tap_ctrl0 .. nws.nws_cmu.ln0_dfe_refclk_tap_val_status7 (25 regs) */
	0x161c9c50,		/* nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1211 .. nws.nws_cmu.ln0_dfe_refclk_ReservedRegister1232 (22 regs) */
	0x031c9c80,		/* nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1233 .. nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1235 (3 regs) */
	0x081c9c86,		/* nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1236 .. nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1243 (8 regs) */
	0x031c9c90,		/* nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1244 .. nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1246 (3 regs) */
	0x091c9c96,		/* nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1247 .. nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1255 (9 regs) */
	0x161c9ca4,		/* nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1256 .. nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1277 (22 regs) */
	0x121c9cc0,		/* nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1278 .. nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1295 (18 regs) */
	0x031c9cd6,		/* nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1296 .. nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1298 (3 regs) */
	0x0c1c9ce0,		/* nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1299 .. nws.nws_cmu.ln0_dfe_rxclk_ReservedRegister1310 (12 regs) */
	0x061c9d00,		/* nws.nws_cmu.ln0_los_refclk_afe_cal_ctrl .. nws.nws_cmu.ln0_los_refclk_run_length_status0 (6 regs) */
	0x071c9d10,		/* nws.nws_cmu.ln0_los_refclk_filter_ctrl0 .. nws.nws_cmu.ln0_los_refclk_filter_ctrl6 (7 regs) */
	0x051c9d20,		/* nws.nws_cmu.ln0_los_refclk_ReservedRegister1313 .. nws.nws_cmu.ln0_los_refclk_ReservedRegister1317 (5 regs) */
	0x041c9d30,		/* nws.nws_cmu.ln0_los_refclk_override_ctrl0 .. nws.nws_cmu.ln0_los_refclk_ReservedRegister1320 (4 regs) */
	0x041c9d40,		/* nws.nws_cmu.ln0_los_refclk_ReservedRegister1321 .. nws.nws_cmu.ln0_los_refclk_ReservedRegister1324 (4 regs) */
	0x011c9d46,		/* nws.nws_cmu.ln0_los_refclk_ReservedRegister1325 (1 regs) */
	0x011c9d51,		/* nws.nws_cmu.ln0_los_refclk_ReservedRegister1326 (1 regs) */
	0x011c9d59,		/* nws.nws_cmu.ln0_los_refclk_ReservedRegister1327 (1 regs) */
	0x011c9d60,		/* nws.nws_cmu.ln0_los_refclk_ReservedRegister1328 (1 regs) */
	0x021c9d70,		/* nws.nws_cmu.ln0_los_refclk_ReservedRegister1329 .. nws.nws_cmu.ln0_los_refclk_status0 (2 regs) */
	0x041c9d80,		/* nws.nws_cmu.ln0_gcfsm2_ReservedRegister1330 .. nws.nws_cmu.ln0_gcfsm2_ReservedRegister1333 (4 regs) */
	0x041c9d90,		/* nws.nws_cmu.ln0_gcfsm2_ReservedRegister1334 .. nws.nws_cmu.ln0_gcfsm2_ReservedRegister1337 (4 regs) */
	0x071c9da0,		/* nws.nws_cmu.ln0_gcfsm2_ReservedRegister1338 .. nws.nws_cmu.ln0_gcfsm2_ReservedRegister1344 (7 regs) */
	0x031c9db0,		/* nws.nws_cmu.ln0_gcfsm2_ReservedRegister1345 .. nws.nws_cmu.ln0_gcfsm2_ReservedRegister1347 (3 regs) */
	0x091c9dc0,		/* nws.nws_cmu.ln0_gcfsm2_ReservedRegister1348 .. nws.nws_cmu.ln0_gcfsm2_ReservedRegister1356 (9 regs) */
	0x021c9dd0,		/* nws.nws_cmu.ln0_gcfsm2_ReservedRegister1357 .. nws.nws_cmu.ln0_gcfsm2_ReservedRegister1358 (2 regs) */
	0x051c9e00,		/* nws.nws_cmu.ln0_bist_tx_ctrl .. nws.nws_cmu.ln0_bist_tx_ReservedRegister1362 (5 regs) */
	0x081c9e06,		/* nws.nws_cmu.ln0_bist_tx_ber_ctrl0 .. nws.nws_cmu.ln0_bist_tx_ber_ctrl7 (8 regs) */
	0x011c9e20,		/* nws.nws_cmu.ln0_bist_tx_udp_shift_amount (1 regs) */
	0x191c9e24,		/* nws.nws_cmu.ln0_bist_tx_udp_7_0 .. nws.nws_cmu.ln0_bist_tx_udp_199_192 (25 regs) */
	0x011c9e80,		/* nws.nws_cmu.ln0_bist_rx_ctrl (1 regs) */
	0x011c9e84,		/* nws.nws_cmu.ln0_bist_rx_status (1 regs) */
	0x031c9e88,		/* nws.nws_cmu.ln0_bist_rx_ber_status0 .. nws.nws_cmu.ln0_bist_rx_ber_status2 (3 regs) */
	0x031c9e8c,		/* nws.nws_cmu.ln0_bist_rx_ber_status4 .. nws.nws_cmu.ln0_bist_rx_ber_status6 (3 regs) */
	0x041c9e94,		/* nws.nws_cmu.ln0_bist_rx_lock_ctrl0 .. nws.nws_cmu.ln0_bist_rx_lock_ctrl3 (4 regs) */
	0x051c9ea0,		/* nws.nws_cmu.ln0_bist_rx_loss_lock_ctrl0 .. nws.nws_cmu.ln0_bist_rx_loss_lock_ctrl4 (5 regs) */
	0x011c9eb0,		/* nws.nws_cmu.ln0_bist_rx_udp_shift_amount (1 regs) */
	0x191c9eb4,		/* nws.nws_cmu.ln0_bist_rx_udp_7_0 .. nws.nws_cmu.ln0_bist_rx_udp_199_192 (25 regs) */
	0x021c9f00,		/* nws.nws_cmu.ln0_feature_rxterm_cfg0 .. nws.nws_cmu.ln0_feature_rxclkdiv_cfg0 (2 regs) */
	0x061c9f04,		/* nws.nws_cmu.ln0_feature_ReservedRegister1363 .. nws.nws_cmu.ln0_feature_ReservedRegister1368 (6 regs) */
	0x071c9f10,		/* nws.nws_cmu.ln0_feature_ReservedRegister1369 .. nws.nws_cmu.ln0_feature_ReservedRegister1375 (7 regs) */
	0x0a1c9f20,		/* nws.nws_cmu.ln0_feature_ReservedRegister1376 .. nws.nws_cmu.ln0_feature_ReservedRegister1378 (10 regs) */
	0x071c9f30,		/* nws.nws_cmu.ln0_feature_dfe_cfg .. nws.nws_cmu.ln0_feature_dfe_adapt_tap5_cfg (7 regs) */
	0x101c9f38,		/* nws.nws_cmu.ln0_feature_adapt_cont_cfg0 .. nws.nws_cmu.ln0_feature_ReservedRegister1390 (16 regs) */
	0x011c9f50,		/* nws.nws_cmu.ln0_feature_test_cfg0 (1 regs) */
	0x081c9f58,		/* nws.nws_cmu.ln0_feature_ReservedRegister1391 .. nws.nws_cmu.ln0_feature_ReservedRegister1398 (8 regs) */
	0x061c9f80,		/* nws.nws_cmu.ln0_lt_tx_fsm_ctrl0 .. nws.nws_cmu.ln0_lt_tx_fsm_ctrl5 (6 regs) */
	0x011c9f90,		/* nws.nws_cmu.ln0_lt_tx_fsm_status (1 regs) */
	0x031c9f93,		/* nws.nws_cmu.ln0_lt_tx_prbs_ctrl0 .. nws.nws_cmu.ln0_lt_tx_prbs_ctrl2 (3 regs) */
	0x011c9fa0,		/* nws.nws_cmu.ln0_lt_tx_coefficient_update_ctrl (1 regs) */
	0x011c9fa2,		/* nws.nws_cmu.ln0_lt_tx_status_report_ctrl (1 regs) */
	0x021c9fb0,		/* nws.nws_cmu.ln0_lt_tx_fsm_state_status0 .. nws.nws_cmu.ln0_lt_tx_fsm_state_status1 (2 regs) */
	0x011c9fc0,		/* nws.nws_cmu.ln0_lt_rx_ctrl0 (1 regs) */
	0x021c9fc2,		/* nws.nws_cmu.ln0_lt_rx_prbs_ctrl0 .. nws.nws_cmu.ln0_lt_rx_prbs_ctrl1 (2 regs) */
	0x031c9fc5,		/* nws.nws_cmu.ln0_lt_rx_prbs_status0 .. nws.nws_cmu.ln0_lt_rx_prbs_status2 (3 regs) */
	0x011c9fd0,		/* nws.nws_cmu.ln0_lt_rx_frame_ctrl (1 regs) */
	0x051c9fd3,		/* nws.nws_cmu.ln0_lt_rx_frame_status0 .. nws.nws_cmu.ln0_lt_rx_frame_status4 (5 regs) */
	0x011c9fe0,		/* nws.nws_cmu.ln0_lt_rx_coefficient_update_status (1 regs) */
	0x011c9fe2,		/* nws.nws_cmu.ln0_lt_rx_report_status (1 regs) */
	0x071ca000,		/* nws.nws_cmu.ln1_top_afe_loopback_ctrl .. nws.nws_cmu.ln1_top_ReservedRegister1404 (7 regs) */
	0x011ca010,		/* nws.nws_cmu.ln1_top_ReservedRegister1405 (1 regs) */
	0x031ca012,		/* nws.nws_cmu.ln1_top_ReservedRegister1406 .. nws.nws_cmu.ln1_top_ReservedRegister1408 (3 regs) */
	0x011ca016,		/* nws.nws_cmu.ln1_top_ReservedRegister1409 (1 regs) */
	0x011ca019,		/* nws.nws_cmu.ln1_top_ReservedRegister1410 (1 regs) */
	0x021ca01b,		/* nws.nws_cmu.ln1_top_ReservedRegister1411 .. nws.nws_cmu.ln1_top_ReservedRegister1412 (2 regs) */
	0x011ca01e,		/* nws.nws_cmu.ln1_top_ReservedRegister1413 (1 regs) */
	0x011ca022,		/* nws.nws_cmu.ln1_top_dpl_txdp_ctrl1 (1 regs) */
	0x041ca024,		/* nws.nws_cmu.ln1_top_dpl_rxdp_ctrl1 .. nws.nws_cmu.ln1_top_phy_if_status (4 regs) */
	0x021ca030,		/* nws.nws_cmu.ln1_top_ReservedRegister1416 .. nws.nws_cmu.ln1_top_ReservedRegister1417 (2 regs) */
	0x091ca038,		/* nws.nws_cmu.ln1_top_ln_stat_ctrl0 .. nws.nws_cmu.ln1_top_ReservedRegister1424 (9 regs) */
	0xff4b0000,		/* block nws */
	0x021ca042,		/* nws.nws_cmu.ln1_top_ReservedRegister1425 .. nws.nws_cmu.ln1_top_ReservedRegister1426 (2 regs) */
	0x051ca048,		/* nws.nws_cmu.ln1_top_ReservedRegister1427 .. nws.nws_cmu.ln1_top_ReservedRegister1431 (5 regs) */
	0x031ca050,		/* nws.nws_cmu.ln1_top_err_ctrl1 .. nws.nws_cmu.ln1_top_err_ctrl3 (3 regs) */
	0x021ca090,		/* nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1432 .. nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1433 (2 regs) */
	0x021ca0a1,		/* nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1434 .. nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1435 (2 regs) */
	0x061ca0a6,		/* nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1436 .. nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1441 (6 regs) */
	0x011ca0ad,		/* nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1442 (1 regs) */
	0x031ca0b0,		/* nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1443 .. nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1445 (3 regs) */
	0x011ca0b4,		/* nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1446 (1 regs) */
	0x021ca0b6,		/* nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1447 .. nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1448 (2 regs) */
	0x091ca0b9,		/* nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1449 .. nws.nws_cmu.ln1_cdr_rxclk_dlpf_status4 (9 regs) */
	0x031ca0c4,		/* nws.nws_cmu.ln1_cdr_rxclk_dlpf_status5 .. nws.nws_cmu.ln1_cdr_rxclk_integral_status1 (3 regs) */
	0x081ca0c8,		/* nws.nws_cmu.ln1_cdr_rxclk_integral_status2 .. nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1461 (8 regs) */
	0x041ca0e0,		/* nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1462 .. nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1465 (4 regs) */
	0x091ca0e8,		/* nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1466 .. nws.nws_cmu.ln1_cdr_rxclk_ReservedRegister1474 (9 regs) */
	0x021ca100,		/* nws.nws_cmu.ln1_cdr_refclk_ReservedRegister1475 .. nws.nws_cmu.ln1_cdr_refclk_ReservedRegister1476 (2 regs) */
	0x011ca104,		/* nws.nws_cmu.ln1_cdr_refclk_ReservedRegister1477 (1 regs) */
	0x011ca106,		/* nws.nws_cmu.ln1_cdr_refclk_ReservedRegister1478 (1 regs) */
	0x031ca10a,		/* nws.nws_cmu.ln1_cdr_refclk_ReservedRegister1479 .. nws.nws_cmu.ln1_cdr_refclk_ReservedRegister1481 (3 regs) */
	0x021ca110,		/* nws.nws_cmu.ln1_cdr_refclk_ReservedRegister1482 .. nws.nws_cmu.ln1_cdr_refclk_ReservedRegister1483 (2 regs) */
	0x041ca118,		/* nws.nws_cmu.ln1_cdr_refclk_ReservedRegister1484 .. nws.nws_cmu.ln1_cdr_refclk_ReservedRegister1487 (4 regs) */
	0x061ca120,		/* nws.nws_cmu.ln1_cdr_refclk_ReservedRegister1488 .. nws.nws_cmu.ln1_cdr_refclk_ReservedRegister1493 (6 regs) */
	0x011ca130,		/* nws.nws_cmu.ln1_cdr_refclk_ReservedRegister1494 (1 regs) */
	0x0f1ca180,		/* nws.nws_cmu.ln1_aneg_ReservedRegister1495 .. nws.nws_cmu.ln1_aneg_ReservedRegister1506 (15 regs) */
	0x011ca190,		/* nws.nws_cmu.ln1_aneg_status0 (1 regs) */
	0x021ca194,		/* nws.nws_cmu.ln1_aneg_status_dbg0 .. nws.nws_cmu.ln1_aneg_status_dbg1 (2 regs) */
	0x251ca198,		/* nws.nws_cmu.ln1_aneg_base_page0 .. nws.nws_cmu.ln1_aneg_resolution_eee (37 regs) */
	0x021ca1be,		/* nws.nws_cmu.ln1_aneg_link_status0 .. nws.nws_cmu.ln1_aneg_link_status1 (2 regs) */
	0x031ca1c1,		/* nws.nws_cmu.ln1_eee_ReservedRegister1523 .. nws.nws_cmu.ln1_eee_ReservedRegister1525 (3 regs) */
	0x041ca1c5,		/* nws.nws_cmu.ln1_eee_ReservedRegister1526 .. nws.nws_cmu.ln1_eee_ReservedRegister1529 (4 regs) */
	0x011ca200,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1530 (1 regs) */
	0x021ca202,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1531 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1532 (2 regs) */
	0x011ca205,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1533 (1 regs) */
	0x011ca207,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1534 (1 regs) */
	0x041ca209,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1535 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1538 (4 regs) */
	0x041ca20e,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1539 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1542 (4 regs) */
	0x0d1ca220,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1543 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1555 (13 regs) */
	0x011ca22e,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1556 (1 regs) */
	0x0b1ca230,		/* nws.nws_cmu.ln1_leq_refclk_agclos_ctrl0 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1566 (11 regs) */
	0x021ca23d,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1567 .. nws.nws_cmu.ln1_leq_refclk_ple_att_ctrl1 (2 regs) */
	0x071ca240,		/* nws.nws_cmu.ln1_leq_refclk_eq_hfg_sql_ctrl0 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1573 (7 regs) */
	0x191ca250,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1574 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1598 (25 regs) */
	0x071ca270,		/* nws.nws_cmu.ln1_leq_refclk_gn_apg_ctrl0 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1604 (7 regs) */
	0x091ca280,		/* nws.nws_cmu.ln1_leq_refclk_eq_lfg_ctrl0 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1610 (9 regs) */
	0x071ca290,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1611 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1617 (7 regs) */
	0x061ca298,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1618 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1622 (6 regs) */
	0x041ca2a0,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1623 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1626 (4 regs) */
	0x041ca2a6,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1627 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1630 (4 regs) */
	0x021ca2ab,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1631 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1632 (2 regs) */
	0x021ca2ae,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1633 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1634 (2 regs) */
	0x021ca2b8,		/* nws.nws_cmu.ln1_leq_refclk_ReservedRegister1635 .. nws.nws_cmu.ln1_leq_refclk_ReservedRegister1636 (2 regs) */
	0x061ca300,		/* nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1637 .. nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1642 (6 regs) */
	0x021ca308,		/* nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1643 .. nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1644 (2 regs) */
	0x021ca30c,		/* nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1645 .. nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1646 (2 regs) */
	0x021ca310,		/* nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1647 .. nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1648 (2 regs) */
	0x021ca313,		/* nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1649 .. nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1650 (2 regs) */
	0x021ca316,		/* nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1651 .. nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1652 (2 regs) */
	0x021ca320,		/* nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1653 .. nws.nws_cmu.ln1_leq_rxclk_ReservedRegister1654 (2 regs) */
	0x071ca380,		/* nws.nws_cmu.ln1_drv_refclk_afe_pd_ctrl0 .. nws.nws_cmu.ln1_drv_refclk_ReservedRegister1659 (7 regs) */
	0x061ca388,		/* nws.nws_cmu.ln1_drv_refclk_ReservedRegister1660 .. nws.nws_cmu.ln1_drv_refclk_ReservedRegister1665 (6 regs) */
	0x0a1ca390,		/* nws.nws_cmu.ln1_drv_refclk_txeq_ctrl0 .. nws.nws_cmu.ln1_drv_refclk_ReservedRegister1670 (10 regs) */
	0x011ca39b,		/* nws.nws_cmu.ln1_drv_refclk_ReservedRegister1671 (1 regs) */
	0x051ca400,		/* nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1672 .. nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1676 (5 regs) */
	0x011ca406,		/* nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1677 (1 regs) */
	0x011ca40a,		/* nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1678 (1 regs) */
	0x011ca40c,		/* nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1679 (1 regs) */
	0x011ca40e,		/* nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1680 (1 regs) */
	0x011ca410,		/* nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1681 (1 regs) */
	0x011ca412,		/* nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1682 (1 regs) */
	0x011ca414,		/* nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1683 (1 regs) */
	0x011ca416,		/* nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1684 (1 regs) */
	0x021ca418,		/* nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1685 .. nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1686 (2 regs) */
	0x011ca41b,		/* nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1687 (1 regs) */
	0x091ca420,		/* nws.nws_cmu.ln1_dfe_refclk_fsm_ctrl0 .. nws.nws_cmu.ln1_dfe_refclk_fsm_status0 (9 regs) */
	0x191ca42a,		/* nws.nws_cmu.ln1_dfe_refclk_tap_ctrl0 .. nws.nws_cmu.ln1_dfe_refclk_tap_val_status7 (25 regs) */
	0x161ca450,		/* nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1695 .. nws.nws_cmu.ln1_dfe_refclk_ReservedRegister1716 (22 regs) */
	0x031ca480,		/* nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1717 .. nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1719 (3 regs) */
	0x081ca486,		/* nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1720 .. nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1727 (8 regs) */
	0x031ca490,		/* nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1728 .. nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1730 (3 regs) */
	0x091ca496,		/* nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1731 .. nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1739 (9 regs) */
	0x161ca4a4,		/* nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1740 .. nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1761 (22 regs) */
	0x121ca4c0,		/* nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1762 .. nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1779 (18 regs) */
	0x031ca4d6,		/* nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1780 .. nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1782 (3 regs) */
	0x0c1ca4e0,		/* nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1783 .. nws.nws_cmu.ln1_dfe_rxclk_ReservedRegister1794 (12 regs) */
	0x061ca500,		/* nws.nws_cmu.ln1_los_refclk_afe_cal_ctrl .. nws.nws_cmu.ln1_los_refclk_run_length_status0 (6 regs) */
	0x071ca510,		/* nws.nws_cmu.ln1_los_refclk_filter_ctrl0 .. nws.nws_cmu.ln1_los_refclk_filter_ctrl6 (7 regs) */
	0x051ca520,		/* nws.nws_cmu.ln1_los_refclk_ReservedRegister1797 .. nws.nws_cmu.ln1_los_refclk_ReservedRegister1801 (5 regs) */
	0x041ca530,		/* nws.nws_cmu.ln1_los_refclk_override_ctrl0 .. nws.nws_cmu.ln1_los_refclk_ReservedRegister1804 (4 regs) */
	0x041ca540,		/* nws.nws_cmu.ln1_los_refclk_ReservedRegister1805 .. nws.nws_cmu.ln1_los_refclk_ReservedRegister1808 (4 regs) */
	0x011ca546,		/* nws.nws_cmu.ln1_los_refclk_ReservedRegister1809 (1 regs) */
	0x011ca551,		/* nws.nws_cmu.ln1_los_refclk_ReservedRegister1810 (1 regs) */
	0x011ca559,		/* nws.nws_cmu.ln1_los_refclk_ReservedRegister1811 (1 regs) */
	0x011ca560,		/* nws.nws_cmu.ln1_los_refclk_ReservedRegister1812 (1 regs) */
	0x021ca570,		/* nws.nws_cmu.ln1_los_refclk_ReservedRegister1813 .. nws.nws_cmu.ln1_los_refclk_status0 (2 regs) */
	0x041ca580,		/* nws.nws_cmu.ln1_gcfsm2_ReservedRegister1814 .. nws.nws_cmu.ln1_gcfsm2_ReservedRegister1817 (4 regs) */
	0x041ca590,		/* nws.nws_cmu.ln1_gcfsm2_ReservedRegister1818 .. nws.nws_cmu.ln1_gcfsm2_ReservedRegister1821 (4 regs) */
	0x071ca5a0,		/* nws.nws_cmu.ln1_gcfsm2_ReservedRegister1822 .. nws.nws_cmu.ln1_gcfsm2_ReservedRegister1828 (7 regs) */
	0x031ca5b0,		/* nws.nws_cmu.ln1_gcfsm2_ReservedRegister1829 .. nws.nws_cmu.ln1_gcfsm2_ReservedRegister1831 (3 regs) */
	0x091ca5c0,		/* nws.nws_cmu.ln1_gcfsm2_ReservedRegister1832 .. nws.nws_cmu.ln1_gcfsm2_ReservedRegister1840 (9 regs) */
	0x021ca5d0,		/* nws.nws_cmu.ln1_gcfsm2_ReservedRegister1841 .. nws.nws_cmu.ln1_gcfsm2_ReservedRegister1842 (2 regs) */
	0x051ca600,		/* nws.nws_cmu.ln1_bist_tx_ctrl .. nws.nws_cmu.ln1_bist_tx_ReservedRegister1846 (5 regs) */
	0x081ca606,		/* nws.nws_cmu.ln1_bist_tx_ber_ctrl0 .. nws.nws_cmu.ln1_bist_tx_ber_ctrl7 (8 regs) */
	0x011ca620,		/* nws.nws_cmu.ln1_bist_tx_udp_shift_amount (1 regs) */
	0x191ca624,		/* nws.nws_cmu.ln1_bist_tx_udp_7_0 .. nws.nws_cmu.ln1_bist_tx_udp_199_192 (25 regs) */
	0x011ca680,		/* nws.nws_cmu.ln1_bist_rx_ctrl (1 regs) */
	0x011ca684,		/* nws.nws_cmu.ln1_bist_rx_status (1 regs) */
	0x031ca688,		/* nws.nws_cmu.ln1_bist_rx_ber_status0 .. nws.nws_cmu.ln1_bist_rx_ber_status2 (3 regs) */
	0x031ca68c,		/* nws.nws_cmu.ln1_bist_rx_ber_status4 .. nws.nws_cmu.ln1_bist_rx_ber_status6 (3 regs) */
	0x041ca694,		/* nws.nws_cmu.ln1_bist_rx_lock_ctrl0 .. nws.nws_cmu.ln1_bist_rx_lock_ctrl3 (4 regs) */
	0x051ca6a0,		/* nws.nws_cmu.ln1_bist_rx_loss_lock_ctrl0 .. nws.nws_cmu.ln1_bist_rx_loss_lock_ctrl4 (5 regs) */
	0x011ca6b0,		/* nws.nws_cmu.ln1_bist_rx_udp_shift_amount (1 regs) */
	0x191ca6b4,		/* nws.nws_cmu.ln1_bist_rx_udp_7_0 .. nws.nws_cmu.ln1_bist_rx_udp_199_192 (25 regs) */
	0x021ca700,		/* nws.nws_cmu.ln1_feature_rxterm_cfg0 .. nws.nws_cmu.ln1_feature_rxclkdiv_cfg0 (2 regs) */
	0x061ca704,		/* nws.nws_cmu.ln1_feature_ReservedRegister1847 .. nws.nws_cmu.ln1_feature_ReservedRegister1852 (6 regs) */
	0x071ca710,		/* nws.nws_cmu.ln1_feature_ReservedRegister1853 .. nws.nws_cmu.ln1_feature_ReservedRegister1859 (7 regs) */
	0x0a1ca720,		/* nws.nws_cmu.ln1_feature_ReservedRegister1860 .. nws.nws_cmu.ln1_feature_ReservedRegister1862 (10 regs) */
	0x071ca730,		/* nws.nws_cmu.ln1_feature_dfe_cfg .. nws.nws_cmu.ln1_feature_dfe_adapt_tap5_cfg (7 regs) */
	0x101ca738,		/* nws.nws_cmu.ln1_feature_adapt_cont_cfg0 .. nws.nws_cmu.ln1_feature_ReservedRegister1874 (16 regs) */
	0x011ca750,		/* nws.nws_cmu.ln1_feature_test_cfg0 (1 regs) */
	0x081ca758,		/* nws.nws_cmu.ln1_feature_ReservedRegister1875 .. nws.nws_cmu.ln1_feature_ReservedRegister1882 (8 regs) */
	0x061ca780,		/* nws.nws_cmu.ln1_lt_tx_fsm_ctrl0 .. nws.nws_cmu.ln1_lt_tx_fsm_ctrl5 (6 regs) */
	0x011ca790,		/* nws.nws_cmu.ln1_lt_tx_fsm_status (1 regs) */
	0x031ca793,		/* nws.nws_cmu.ln1_lt_tx_prbs_ctrl0 .. nws.nws_cmu.ln1_lt_tx_prbs_ctrl2 (3 regs) */
	0x011ca7a0,		/* nws.nws_cmu.ln1_lt_tx_coefficient_update_ctrl (1 regs) */
	0x011ca7a2,		/* nws.nws_cmu.ln1_lt_tx_status_report_ctrl (1 regs) */
	0x021ca7b0,		/* nws.nws_cmu.ln1_lt_tx_fsm_state_status0 .. nws.nws_cmu.ln1_lt_tx_fsm_state_status1 (2 regs) */
	0x011ca7c0,		/* nws.nws_cmu.ln1_lt_rx_ctrl0 (1 regs) */
	0x021ca7c2,		/* nws.nws_cmu.ln1_lt_rx_prbs_ctrl0 .. nws.nws_cmu.ln1_lt_rx_prbs_ctrl1 (2 regs) */
	0x031ca7c5,		/* nws.nws_cmu.ln1_lt_rx_prbs_status0 .. nws.nws_cmu.ln1_lt_rx_prbs_status2 (3 regs) */
	0x011ca7d0,		/* nws.nws_cmu.ln1_lt_rx_frame_ctrl (1 regs) */
	0x051ca7d3,		/* nws.nws_cmu.ln1_lt_rx_frame_status0 .. nws.nws_cmu.ln1_lt_rx_frame_status4 (5 regs) */
	0x011ca7e0,		/* nws.nws_cmu.ln1_lt_rx_coefficient_update_status (1 regs) */
	0x011ca7e2,		/* nws.nws_cmu.ln1_lt_rx_report_status (1 regs) */
	0x071ca800,		/* nws.nws_cmu.ln2_top_afe_loopback_ctrl .. nws.nws_cmu.ln2_top_ReservedRegister1888 (7 regs) */
	0x011ca810,		/* nws.nws_cmu.ln2_top_ReservedRegister1889 (1 regs) */
	0x031ca812,		/* nws.nws_cmu.ln2_top_ReservedRegister1890 .. nws.nws_cmu.ln2_top_ReservedRegister1892 (3 regs) */
	0x011ca816,		/* nws.nws_cmu.ln2_top_ReservedRegister1893 (1 regs) */
	0x011ca819,		/* nws.nws_cmu.ln2_top_ReservedRegister1894 (1 regs) */
	0x021ca81b,		/* nws.nws_cmu.ln2_top_ReservedRegister1895 .. nws.nws_cmu.ln2_top_ReservedRegister1896 (2 regs) */
	0x011ca81e,		/* nws.nws_cmu.ln2_top_ReservedRegister1897 (1 regs) */
	0x011ca822,		/* nws.nws_cmu.ln2_top_dpl_txdp_ctrl1 (1 regs) */
	0x041ca824,		/* nws.nws_cmu.ln2_top_dpl_rxdp_ctrl1 .. nws.nws_cmu.ln2_top_phy_if_status (4 regs) */
	0x021ca830,		/* nws.nws_cmu.ln2_top_ReservedRegister1900 .. nws.nws_cmu.ln2_top_ReservedRegister1901 (2 regs) */
	0x091ca838,		/* nws.nws_cmu.ln2_top_ln_stat_ctrl0 .. nws.nws_cmu.ln2_top_ReservedRegister1908 (9 regs) */
	0x021ca842,		/* nws.nws_cmu.ln2_top_ReservedRegister1909 .. nws.nws_cmu.ln2_top_ReservedRegister1910 (2 regs) */
	0x051ca848,		/* nws.nws_cmu.ln2_top_ReservedRegister1911 .. nws.nws_cmu.ln2_top_ReservedRegister1915 (5 regs) */
	0x031ca850,		/* nws.nws_cmu.ln2_top_err_ctrl1 .. nws.nws_cmu.ln2_top_err_ctrl3 (3 regs) */
	0x021ca890,		/* nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1916 .. nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1917 (2 regs) */
	0x021ca8a1,		/* nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1918 .. nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1919 (2 regs) */
	0x061ca8a6,		/* nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1920 .. nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1925 (6 regs) */
	0x011ca8ad,		/* nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1926 (1 regs) */
	0x031ca8b0,		/* nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1927 .. nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1929 (3 regs) */
	0x011ca8b4,		/* nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1930 (1 regs) */
	0x021ca8b6,		/* nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1931 .. nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1932 (2 regs) */
	0x091ca8b9,		/* nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1933 .. nws.nws_cmu.ln2_cdr_rxclk_dlpf_status4 (9 regs) */
	0x031ca8c4,		/* nws.nws_cmu.ln2_cdr_rxclk_dlpf_status5 .. nws.nws_cmu.ln2_cdr_rxclk_integral_status1 (3 regs) */
	0x081ca8c8,		/* nws.nws_cmu.ln2_cdr_rxclk_integral_status2 .. nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1945 (8 regs) */
	0x041ca8e0,		/* nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1946 .. nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1949 (4 regs) */
	0x091ca8e8,		/* nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1950 .. nws.nws_cmu.ln2_cdr_rxclk_ReservedRegister1958 (9 regs) */
	0x021ca900,		/* nws.nws_cmu.ln2_cdr_refclk_ReservedRegister1959 .. nws.nws_cmu.ln2_cdr_refclk_ReservedRegister1960 (2 regs) */
	0x011ca904,		/* nws.nws_cmu.ln2_cdr_refclk_ReservedRegister1961 (1 regs) */
	0x011ca906,		/* nws.nws_cmu.ln2_cdr_refclk_ReservedRegister1962 (1 regs) */
	0x031ca90a,		/* nws.nws_cmu.ln2_cdr_refclk_ReservedRegister1963 .. nws.nws_cmu.ln2_cdr_refclk_ReservedRegister1965 (3 regs) */
	0x021ca910,		/* nws.nws_cmu.ln2_cdr_refclk_ReservedRegister1966 .. nws.nws_cmu.ln2_cdr_refclk_ReservedRegister1967 (2 regs) */
	0x041ca918,		/* nws.nws_cmu.ln2_cdr_refclk_ReservedRegister1968 .. nws.nws_cmu.ln2_cdr_refclk_ReservedRegister1971 (4 regs) */
	0x061ca920,		/* nws.nws_cmu.ln2_cdr_refclk_ReservedRegister1972 .. nws.nws_cmu.ln2_cdr_refclk_ReservedRegister1977 (6 regs) */
	0x011ca930,		/* nws.nws_cmu.ln2_cdr_refclk_ReservedRegister1978 (1 regs) */
	0x0f1ca980,		/* nws.nws_cmu.ln2_aneg_ReservedRegister1979 .. nws.nws_cmu.ln2_aneg_ReservedRegister1990 (15 regs) */
	0x011ca990,		/* nws.nws_cmu.ln2_aneg_status0 (1 regs) */
	0x021ca994,		/* nws.nws_cmu.ln2_aneg_status_dbg0 .. nws.nws_cmu.ln2_aneg_status_dbg1 (2 regs) */
	0x251ca998,		/* nws.nws_cmu.ln2_aneg_base_page0 .. nws.nws_cmu.ln2_aneg_resolution_eee (37 regs) */
	0x021ca9be,		/* nws.nws_cmu.ln2_aneg_link_status0 .. nws.nws_cmu.ln2_aneg_link_status1 (2 regs) */
	0x031ca9c1,		/* nws.nws_cmu.ln2_eee_ReservedRegister2007 .. nws.nws_cmu.ln2_eee_ReservedRegister2009 (3 regs) */
	0x041ca9c5,		/* nws.nws_cmu.ln2_eee_ReservedRegister2010 .. nws.nws_cmu.ln2_eee_ReservedRegister2013 (4 regs) */
	0x011caa00,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2014 (1 regs) */
	0x021caa02,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2015 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2016 (2 regs) */
	0x011caa05,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2017 (1 regs) */
	0x011caa07,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2018 (1 regs) */
	0x041caa09,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2019 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2022 (4 regs) */
	0x041caa0e,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2023 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2026 (4 regs) */
	0x0d1caa20,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2027 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2039 (13 regs) */
	0x011caa2e,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2040 (1 regs) */
	0x0b1caa30,		/* nws.nws_cmu.ln2_leq_refclk_agclos_ctrl0 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2050 (11 regs) */
	0x021caa3d,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2051 .. nws.nws_cmu.ln2_leq_refclk_ple_att_ctrl1 (2 regs) */
	0x071caa40,		/* nws.nws_cmu.ln2_leq_refclk_eq_hfg_sql_ctrl0 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2057 (7 regs) */
	0x191caa50,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2058 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2082 (25 regs) */
	0x071caa70,		/* nws.nws_cmu.ln2_leq_refclk_gn_apg_ctrl0 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2088 (7 regs) */
	0x091caa80,		/* nws.nws_cmu.ln2_leq_refclk_eq_lfg_ctrl0 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2094 (9 regs) */
	0x071caa90,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2095 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2101 (7 regs) */
	0x061caa98,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2102 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2106 (6 regs) */
	0x041caaa0,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2107 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2110 (4 regs) */
	0x041caaa6,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2111 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2114 (4 regs) */
	0x021caaab,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2115 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2116 (2 regs) */
	0x021caaae,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2117 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2118 (2 regs) */
	0x021caab8,		/* nws.nws_cmu.ln2_leq_refclk_ReservedRegister2119 .. nws.nws_cmu.ln2_leq_refclk_ReservedRegister2120 (2 regs) */
	0x061cab00,		/* nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2121 .. nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2126 (6 regs) */
	0x021cab08,		/* nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2127 .. nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2128 (2 regs) */
	0x021cab0c,		/* nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2129 .. nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2130 (2 regs) */
	0x021cab10,		/* nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2131 .. nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2132 (2 regs) */
	0x021cab13,		/* nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2133 .. nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2134 (2 regs) */
	0x021cab16,		/* nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2135 .. nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2136 (2 regs) */
	0x021cab20,		/* nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2137 .. nws.nws_cmu.ln2_leq_rxclk_ReservedRegister2138 (2 regs) */
	0x071cab80,		/* nws.nws_cmu.ln2_drv_refclk_afe_pd_ctrl0 .. nws.nws_cmu.ln2_drv_refclk_ReservedRegister2143 (7 regs) */
	0x061cab88,		/* nws.nws_cmu.ln2_drv_refclk_ReservedRegister2144 .. nws.nws_cmu.ln2_drv_refclk_ReservedRegister2149 (6 regs) */
	0x0a1cab90,		/* nws.nws_cmu.ln2_drv_refclk_txeq_ctrl0 .. nws.nws_cmu.ln2_drv_refclk_ReservedRegister2154 (10 regs) */
	0x011cab9b,		/* nws.nws_cmu.ln2_drv_refclk_ReservedRegister2155 (1 regs) */
	0x051cac00,		/* nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2156 .. nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2160 (5 regs) */
	0x011cac06,		/* nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2161 (1 regs) */
	0x011cac0a,		/* nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2162 (1 regs) */
	0x011cac0c,		/* nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2163 (1 regs) */
	0x011cac0e,		/* nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2164 (1 regs) */
	0x011cac10,		/* nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2165 (1 regs) */
	0x011cac12,		/* nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2166 (1 regs) */
	0x011cac14,		/* nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2167 (1 regs) */
	0x011cac16,		/* nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2168 (1 regs) */
	0x021cac18,		/* nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2169 .. nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2170 (2 regs) */
	0x011cac1b,		/* nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2171 (1 regs) */
	0x091cac20,		/* nws.nws_cmu.ln2_dfe_refclk_fsm_ctrl0 .. nws.nws_cmu.ln2_dfe_refclk_fsm_status0 (9 regs) */
	0x191cac2a,		/* nws.nws_cmu.ln2_dfe_refclk_tap_ctrl0 .. nws.nws_cmu.ln2_dfe_refclk_tap_val_status7 (25 regs) */
	0x161cac50,		/* nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2179 .. nws.nws_cmu.ln2_dfe_refclk_ReservedRegister2200 (22 regs) */
	0x031cac80,		/* nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2201 .. nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2203 (3 regs) */
	0x081cac86,		/* nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2204 .. nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2211 (8 regs) */
	0x031cac90,		/* nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2212 .. nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2214 (3 regs) */
	0x091cac96,		/* nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2215 .. nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2223 (9 regs) */
	0x161caca4,		/* nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2224 .. nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2245 (22 regs) */
	0x121cacc0,		/* nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2246 .. nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2263 (18 regs) */
	0x031cacd6,		/* nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2264 .. nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2266 (3 regs) */
	0x0c1cace0,		/* nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2267 .. nws.nws_cmu.ln2_dfe_rxclk_ReservedRegister2278 (12 regs) */
	0x061cad00,		/* nws.nws_cmu.ln2_los_refclk_afe_cal_ctrl .. nws.nws_cmu.ln2_los_refclk_run_length_status0 (6 regs) */
	0x071cad10,		/* nws.nws_cmu.ln2_los_refclk_filter_ctrl0 .. nws.nws_cmu.ln2_los_refclk_filter_ctrl6 (7 regs) */
	0x051cad20,		/* nws.nws_cmu.ln2_los_refclk_ReservedRegister2281 .. nws.nws_cmu.ln2_los_refclk_ReservedRegister2285 (5 regs) */
	0x041cad30,		/* nws.nws_cmu.ln2_los_refclk_override_ctrl0 .. nws.nws_cmu.ln2_los_refclk_ReservedRegister2288 (4 regs) */
	0x041cad40,		/* nws.nws_cmu.ln2_los_refclk_ReservedRegister2289 .. nws.nws_cmu.ln2_los_refclk_ReservedRegister2292 (4 regs) */
	0x011cad46,		/* nws.nws_cmu.ln2_los_refclk_ReservedRegister2293 (1 regs) */
	0x011cad51,		/* nws.nws_cmu.ln2_los_refclk_ReservedRegister2294 (1 regs) */
	0x011cad59,		/* nws.nws_cmu.ln2_los_refclk_ReservedRegister2295 (1 regs) */
	0x011cad60,		/* nws.nws_cmu.ln2_los_refclk_ReservedRegister2296 (1 regs) */
	0x021cad70,		/* nws.nws_cmu.ln2_los_refclk_ReservedRegister2297 .. nws.nws_cmu.ln2_los_refclk_status0 (2 regs) */
	0x041cad80,		/* nws.nws_cmu.ln2_gcfsm2_ReservedRegister2298 .. nws.nws_cmu.ln2_gcfsm2_ReservedRegister2301 (4 regs) */
	0x041cad90,		/* nws.nws_cmu.ln2_gcfsm2_ReservedRegister2302 .. nws.nws_cmu.ln2_gcfsm2_ReservedRegister2305 (4 regs) */
	0x071cada0,		/* nws.nws_cmu.ln2_gcfsm2_ReservedRegister2306 .. nws.nws_cmu.ln2_gcfsm2_ReservedRegister2312 (7 regs) */
	0x031cadb0,		/* nws.nws_cmu.ln2_gcfsm2_ReservedRegister2313 .. nws.nws_cmu.ln2_gcfsm2_ReservedRegister2315 (3 regs) */
	0x091cadc0,		/* nws.nws_cmu.ln2_gcfsm2_ReservedRegister2316 .. nws.nws_cmu.ln2_gcfsm2_ReservedRegister2324 (9 regs) */
	0x021cadd0,		/* nws.nws_cmu.ln2_gcfsm2_ReservedRegister2325 .. nws.nws_cmu.ln2_gcfsm2_ReservedRegister2326 (2 regs) */
	0x051cae00,		/* nws.nws_cmu.ln2_bist_tx_ctrl .. nws.nws_cmu.ln2_bist_tx_ReservedRegister2330 (5 regs) */
	0x081cae06,		/* nws.nws_cmu.ln2_bist_tx_ber_ctrl0 .. nws.nws_cmu.ln2_bist_tx_ber_ctrl7 (8 regs) */
	0x011cae20,		/* nws.nws_cmu.ln2_bist_tx_udp_shift_amount (1 regs) */
	0x191cae24,		/* nws.nws_cmu.ln2_bist_tx_udp_7_0 .. nws.nws_cmu.ln2_bist_tx_udp_199_192 (25 regs) */
	0x011cae80,		/* nws.nws_cmu.ln2_bist_rx_ctrl (1 regs) */
	0x011cae84,		/* nws.nws_cmu.ln2_bist_rx_status (1 regs) */
	0x031cae88,		/* nws.nws_cmu.ln2_bist_rx_ber_status0 .. nws.nws_cmu.ln2_bist_rx_ber_status2 (3 regs) */
	0x031cae8c,		/* nws.nws_cmu.ln2_bist_rx_ber_status4 .. nws.nws_cmu.ln2_bist_rx_ber_status6 (3 regs) */
	0x041cae94,		/* nws.nws_cmu.ln2_bist_rx_lock_ctrl0 .. nws.nws_cmu.ln2_bist_rx_lock_ctrl3 (4 regs) */
	0x051caea0,		/* nws.nws_cmu.ln2_bist_rx_loss_lock_ctrl0 .. nws.nws_cmu.ln2_bist_rx_loss_lock_ctrl4 (5 regs) */
	0x011caeb0,		/* nws.nws_cmu.ln2_bist_rx_udp_shift_amount (1 regs) */
	0xa64b0000,		/* block nws */
	0x191caeb4,		/* nws.nws_cmu.ln2_bist_rx_udp_7_0 .. nws.nws_cmu.ln2_bist_rx_udp_199_192 (25 regs) */
	0x021caf00,		/* nws.nws_cmu.ln2_feature_rxterm_cfg0 .. nws.nws_cmu.ln2_feature_rxclkdiv_cfg0 (2 regs) */
	0x061caf04,		/* nws.nws_cmu.ln2_feature_ReservedRegister2331 .. nws.nws_cmu.ln2_feature_ReservedRegister2336 (6 regs) */
	0x071caf10,		/* nws.nws_cmu.ln2_feature_ReservedRegister2337 .. nws.nws_cmu.ln2_feature_ReservedRegister2343 (7 regs) */
	0x0a1caf20,		/* nws.nws_cmu.ln2_feature_ReservedRegister2344 .. nws.nws_cmu.ln2_feature_ReservedRegister2346 (10 regs) */
	0x071caf30,		/* nws.nws_cmu.ln2_feature_dfe_cfg .. nws.nws_cmu.ln2_feature_dfe_adapt_tap5_cfg (7 regs) */
	0x101caf38,		/* nws.nws_cmu.ln2_feature_adapt_cont_cfg0 .. nws.nws_cmu.ln2_feature_ReservedRegister2358 (16 regs) */
	0x011caf50,		/* nws.nws_cmu.ln2_feature_test_cfg0 (1 regs) */
	0x081caf58,		/* nws.nws_cmu.ln2_feature_ReservedRegister2359 .. nws.nws_cmu.ln2_feature_ReservedRegister2366 (8 regs) */
	0x061caf80,		/* nws.nws_cmu.ln2_lt_tx_fsm_ctrl0 .. nws.nws_cmu.ln2_lt_tx_fsm_ctrl5 (6 regs) */
	0x011caf90,		/* nws.nws_cmu.ln2_lt_tx_fsm_status (1 regs) */
	0x031caf93,		/* nws.nws_cmu.ln2_lt_tx_prbs_ctrl0 .. nws.nws_cmu.ln2_lt_tx_prbs_ctrl2 (3 regs) */
	0x011cafa0,		/* nws.nws_cmu.ln2_lt_tx_coefficient_update_ctrl (1 regs) */
	0x011cafa2,		/* nws.nws_cmu.ln2_lt_tx_status_report_ctrl (1 regs) */
	0x021cafb0,		/* nws.nws_cmu.ln2_lt_tx_fsm_state_status0 .. nws.nws_cmu.ln2_lt_tx_fsm_state_status1 (2 regs) */
	0x011cafc0,		/* nws.nws_cmu.ln2_lt_rx_ctrl0 (1 regs) */
	0x021cafc2,		/* nws.nws_cmu.ln2_lt_rx_prbs_ctrl0 .. nws.nws_cmu.ln2_lt_rx_prbs_ctrl1 (2 regs) */
	0x031cafc5,		/* nws.nws_cmu.ln2_lt_rx_prbs_status0 .. nws.nws_cmu.ln2_lt_rx_prbs_status2 (3 regs) */
	0x011cafd0,		/* nws.nws_cmu.ln2_lt_rx_frame_ctrl (1 regs) */
	0x051cafd3,		/* nws.nws_cmu.ln2_lt_rx_frame_status0 .. nws.nws_cmu.ln2_lt_rx_frame_status4 (5 regs) */
	0x011cafe0,		/* nws.nws_cmu.ln2_lt_rx_coefficient_update_status (1 regs) */
	0x011cafe2,		/* nws.nws_cmu.ln2_lt_rx_report_status (1 regs) */
	0x071cb000,		/* nws.nws_cmu.ln3_top_afe_loopback_ctrl .. nws.nws_cmu.ln3_top_ReservedRegister2372 (7 regs) */
	0x011cb010,		/* nws.nws_cmu.ln3_top_ReservedRegister2373 (1 regs) */
	0x031cb012,		/* nws.nws_cmu.ln3_top_ReservedRegister2374 .. nws.nws_cmu.ln3_top_ReservedRegister2376 (3 regs) */
	0x011cb016,		/* nws.nws_cmu.ln3_top_ReservedRegister2377 (1 regs) */
	0x011cb019,		/* nws.nws_cmu.ln3_top_ReservedRegister2378 (1 regs) */
	0x021cb01b,		/* nws.nws_cmu.ln3_top_ReservedRegister2379 .. nws.nws_cmu.ln3_top_ReservedRegister2380 (2 regs) */
	0x011cb01e,		/* nws.nws_cmu.ln3_top_ReservedRegister2381 (1 regs) */
	0x011cb022,		/* nws.nws_cmu.ln3_top_dpl_txdp_ctrl1 (1 regs) */
	0x041cb024,		/* nws.nws_cmu.ln3_top_dpl_rxdp_ctrl1 .. nws.nws_cmu.ln3_top_phy_if_status (4 regs) */
	0x021cb030,		/* nws.nws_cmu.ln3_top_ReservedRegister2384 .. nws.nws_cmu.ln3_top_ReservedRegister2385 (2 regs) */
	0x091cb038,		/* nws.nws_cmu.ln3_top_ln_stat_ctrl0 .. nws.nws_cmu.ln3_top_ReservedRegister2392 (9 regs) */
	0x021cb042,		/* nws.nws_cmu.ln3_top_ReservedRegister2393 .. nws.nws_cmu.ln3_top_ReservedRegister2394 (2 regs) */
	0x051cb048,		/* nws.nws_cmu.ln3_top_ReservedRegister2395 .. nws.nws_cmu.ln3_top_ReservedRegister2399 (5 regs) */
	0x031cb050,		/* nws.nws_cmu.ln3_top_err_ctrl1 .. nws.nws_cmu.ln3_top_err_ctrl3 (3 regs) */
	0x021cb090,		/* nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2400 .. nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2401 (2 regs) */
	0x021cb0a1,		/* nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2402 .. nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2403 (2 regs) */
	0x061cb0a6,		/* nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2404 .. nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2409 (6 regs) */
	0x011cb0ad,		/* nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2410 (1 regs) */
	0x031cb0b0,		/* nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2411 .. nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2413 (3 regs) */
	0x011cb0b4,		/* nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2414 (1 regs) */
	0x021cb0b6,		/* nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2415 .. nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2416 (2 regs) */
	0x091cb0b9,		/* nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2417 .. nws.nws_cmu.ln3_cdr_rxclk_dlpf_status4 (9 regs) */
	0x031cb0c4,		/* nws.nws_cmu.ln3_cdr_rxclk_dlpf_status5 .. nws.nws_cmu.ln3_cdr_rxclk_integral_status1 (3 regs) */
	0x081cb0c8,		/* nws.nws_cmu.ln3_cdr_rxclk_integral_status2 .. nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2429 (8 regs) */
	0x041cb0e0,		/* nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2430 .. nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2433 (4 regs) */
	0x091cb0e8,		/* nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2434 .. nws.nws_cmu.ln3_cdr_rxclk_ReservedRegister2442 (9 regs) */
	0x021cb100,		/* nws.nws_cmu.ln3_cdr_refclk_ReservedRegister2443 .. nws.nws_cmu.ln3_cdr_refclk_ReservedRegister2444 (2 regs) */
	0x011cb104,		/* nws.nws_cmu.ln3_cdr_refclk_ReservedRegister2445 (1 regs) */
	0x011cb106,		/* nws.nws_cmu.ln3_cdr_refclk_ReservedRegister2446 (1 regs) */
	0x031cb10a,		/* nws.nws_cmu.ln3_cdr_refclk_ReservedRegister2447 .. nws.nws_cmu.ln3_cdr_refclk_ReservedRegister2449 (3 regs) */
	0x021cb110,		/* nws.nws_cmu.ln3_cdr_refclk_ReservedRegister2450 .. nws.nws_cmu.ln3_cdr_refclk_ReservedRegister2451 (2 regs) */
	0x041cb118,		/* nws.nws_cmu.ln3_cdr_refclk_ReservedRegister2452 .. nws.nws_cmu.ln3_cdr_refclk_ReservedRegister2455 (4 regs) */
	0x061cb120,		/* nws.nws_cmu.ln3_cdr_refclk_ReservedRegister2456 .. nws.nws_cmu.ln3_cdr_refclk_ReservedRegister2461 (6 regs) */
	0x011cb130,		/* nws.nws_cmu.ln3_cdr_refclk_ReservedRegister2462 (1 regs) */
	0x0f1cb180,		/* nws.nws_cmu.ln3_aneg_ReservedRegister2463 .. nws.nws_cmu.ln3_aneg_ReservedRegister2474 (15 regs) */
	0x011cb190,		/* nws.nws_cmu.ln3_aneg_status0 (1 regs) */
	0x021cb194,		/* nws.nws_cmu.ln3_aneg_status_dbg0 .. nws.nws_cmu.ln3_aneg_status_dbg1 (2 regs) */
	0x251cb198,		/* nws.nws_cmu.ln3_aneg_base_page0 .. nws.nws_cmu.ln3_aneg_resolution_eee (37 regs) */
	0x021cb1be,		/* nws.nws_cmu.ln3_aneg_link_status0 .. nws.nws_cmu.ln3_aneg_link_status1 (2 regs) */
	0x031cb1c1,		/* nws.nws_cmu.ln3_eee_ReservedRegister2491 .. nws.nws_cmu.ln3_eee_ReservedRegister2493 (3 regs) */
	0x041cb1c5,		/* nws.nws_cmu.ln3_eee_ReservedRegister2494 .. nws.nws_cmu.ln3_eee_ReservedRegister2497 (4 regs) */
	0x011cb200,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2498 (1 regs) */
	0x021cb202,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2499 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2500 (2 regs) */
	0x011cb205,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2501 (1 regs) */
	0x011cb207,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2502 (1 regs) */
	0x041cb209,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2503 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2506 (4 regs) */
	0x041cb20e,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2507 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2510 (4 regs) */
	0x0d1cb220,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2511 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2523 (13 regs) */
	0x011cb22e,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2524 (1 regs) */
	0x0b1cb230,		/* nws.nws_cmu.ln3_leq_refclk_agclos_ctrl0 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2534 (11 regs) */
	0x021cb23d,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2535 .. nws.nws_cmu.ln3_leq_refclk_ple_att_ctrl1 (2 regs) */
	0x071cb240,		/* nws.nws_cmu.ln3_leq_refclk_eq_hfg_sql_ctrl0 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2541 (7 regs) */
	0x191cb250,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2542 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2566 (25 regs) */
	0x071cb270,		/* nws.nws_cmu.ln3_leq_refclk_gn_apg_ctrl0 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2572 (7 regs) */
	0x091cb280,		/* nws.nws_cmu.ln3_leq_refclk_eq_lfg_ctrl0 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2578 (9 regs) */
	0x071cb290,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2579 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2585 (7 regs) */
	0x061cb298,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2586 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2590 (6 regs) */
	0x041cb2a0,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2591 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2594 (4 regs) */
	0x041cb2a6,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2595 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2598 (4 regs) */
	0x021cb2ab,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2599 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2600 (2 regs) */
	0x021cb2ae,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2601 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2602 (2 regs) */
	0x021cb2b8,		/* nws.nws_cmu.ln3_leq_refclk_ReservedRegister2603 .. nws.nws_cmu.ln3_leq_refclk_ReservedRegister2604 (2 regs) */
	0x061cb300,		/* nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2605 .. nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2610 (6 regs) */
	0x021cb308,		/* nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2611 .. nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2612 (2 regs) */
	0x021cb30c,		/* nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2613 .. nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2614 (2 regs) */
	0x021cb310,		/* nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2615 .. nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2616 (2 regs) */
	0x021cb313,		/* nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2617 .. nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2618 (2 regs) */
	0x021cb316,		/* nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2619 .. nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2620 (2 regs) */
	0x021cb320,		/* nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2621 .. nws.nws_cmu.ln3_leq_rxclk_ReservedRegister2622 (2 regs) */
	0x071cb380,		/* nws.nws_cmu.ln3_drv_refclk_afe_pd_ctrl0 .. nws.nws_cmu.ln3_drv_refclk_ReservedRegister2627 (7 regs) */
	0x061cb388,		/* nws.nws_cmu.ln3_drv_refclk_ReservedRegister2628 .. nws.nws_cmu.ln3_drv_refclk_ReservedRegister2633 (6 regs) */
	0x0a1cb390,		/* nws.nws_cmu.ln3_drv_refclk_txeq_ctrl0 .. nws.nws_cmu.ln3_drv_refclk_ReservedRegister2638 (10 regs) */
	0x011cb39b,		/* nws.nws_cmu.ln3_drv_refclk_ReservedRegister2639 (1 regs) */
	0x051cb400,		/* nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2640 .. nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2644 (5 regs) */
	0x011cb406,		/* nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2645 (1 regs) */
	0x011cb40a,		/* nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2646 (1 regs) */
	0x011cb40c,		/* nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2647 (1 regs) */
	0x011cb40e,		/* nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2648 (1 regs) */
	0x011cb410,		/* nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2649 (1 regs) */
	0x011cb412,		/* nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2650 (1 regs) */
	0x011cb414,		/* nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2651 (1 regs) */
	0x011cb416,		/* nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2652 (1 regs) */
	0x021cb418,		/* nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2653 .. nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2654 (2 regs) */
	0x011cb41b,		/* nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2655 (1 regs) */
	0x091cb420,		/* nws.nws_cmu.ln3_dfe_refclk_fsm_ctrl0 .. nws.nws_cmu.ln3_dfe_refclk_fsm_status0 (9 regs) */
	0x191cb42a,		/* nws.nws_cmu.ln3_dfe_refclk_tap_ctrl0 .. nws.nws_cmu.ln3_dfe_refclk_tap_val_status7 (25 regs) */
	0x161cb450,		/* nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2663 .. nws.nws_cmu.ln3_dfe_refclk_ReservedRegister2684 (22 regs) */
	0x031cb480,		/* nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2685 .. nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2687 (3 regs) */
	0x081cb486,		/* nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2688 .. nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2695 (8 regs) */
	0x031cb490,		/* nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2696 .. nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2698 (3 regs) */
	0x091cb496,		/* nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2699 .. nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2707 (9 regs) */
	0x161cb4a4,		/* nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2708 .. nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2729 (22 regs) */
	0x121cb4c0,		/* nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2730 .. nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2747 (18 regs) */
	0x031cb4d6,		/* nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2748 .. nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2750 (3 regs) */
	0x0c1cb4e0,		/* nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2751 .. nws.nws_cmu.ln3_dfe_rxclk_ReservedRegister2762 (12 regs) */
	0x061cb500,		/* nws.nws_cmu.ln3_los_refclk_afe_cal_ctrl .. nws.nws_cmu.ln3_los_refclk_run_length_status0 (6 regs) */
	0x071cb510,		/* nws.nws_cmu.ln3_los_refclk_filter_ctrl0 .. nws.nws_cmu.ln3_los_refclk_filter_ctrl6 (7 regs) */
	0x051cb520,		/* nws.nws_cmu.ln3_los_refclk_ReservedRegister2765 .. nws.nws_cmu.ln3_los_refclk_ReservedRegister2769 (5 regs) */
	0x041cb530,		/* nws.nws_cmu.ln3_los_refclk_override_ctrl0 .. nws.nws_cmu.ln3_los_refclk_ReservedRegister2772 (4 regs) */
	0x041cb540,		/* nws.nws_cmu.ln3_los_refclk_ReservedRegister2773 .. nws.nws_cmu.ln3_los_refclk_ReservedRegister2776 (4 regs) */
	0x011cb546,		/* nws.nws_cmu.ln3_los_refclk_ReservedRegister2777 (1 regs) */
	0x011cb551,		/* nws.nws_cmu.ln3_los_refclk_ReservedRegister2778 (1 regs) */
	0x011cb559,		/* nws.nws_cmu.ln3_los_refclk_ReservedRegister2779 (1 regs) */
	0x011cb560,		/* nws.nws_cmu.ln3_los_refclk_ReservedRegister2780 (1 regs) */
	0x021cb570,		/* nws.nws_cmu.ln3_los_refclk_ReservedRegister2781 .. nws.nws_cmu.ln3_los_refclk_status0 (2 regs) */
	0x041cb580,		/* nws.nws_cmu.ln3_gcfsm2_ReservedRegister2782 .. nws.nws_cmu.ln3_gcfsm2_ReservedRegister2785 (4 regs) */
	0x041cb590,		/* nws.nws_cmu.ln3_gcfsm2_ReservedRegister2786 .. nws.nws_cmu.ln3_gcfsm2_ReservedRegister2789 (4 regs) */
	0x071cb5a0,		/* nws.nws_cmu.ln3_gcfsm2_ReservedRegister2790 .. nws.nws_cmu.ln3_gcfsm2_ReservedRegister2796 (7 regs) */
	0x031cb5b0,		/* nws.nws_cmu.ln3_gcfsm2_ReservedRegister2797 .. nws.nws_cmu.ln3_gcfsm2_ReservedRegister2799 (3 regs) */
	0x091cb5c0,		/* nws.nws_cmu.ln3_gcfsm2_ReservedRegister2800 .. nws.nws_cmu.ln3_gcfsm2_ReservedRegister2808 (9 regs) */
	0x021cb5d0,		/* nws.nws_cmu.ln3_gcfsm2_ReservedRegister2809 .. nws.nws_cmu.ln3_gcfsm2_ReservedRegister2810 (2 regs) */
	0x051cb600,		/* nws.nws_cmu.ln3_bist_tx_ctrl .. nws.nws_cmu.ln3_bist_tx_ReservedRegister2814 (5 regs) */
	0x081cb606,		/* nws.nws_cmu.ln3_bist_tx_ber_ctrl0 .. nws.nws_cmu.ln3_bist_tx_ber_ctrl7 (8 regs) */
	0x011cb620,		/* nws.nws_cmu.ln3_bist_tx_udp_shift_amount (1 regs) */
	0x191cb624,		/* nws.nws_cmu.ln3_bist_tx_udp_7_0 .. nws.nws_cmu.ln3_bist_tx_udp_199_192 (25 regs) */
	0x011cb680,		/* nws.nws_cmu.ln3_bist_rx_ctrl (1 regs) */
	0x011cb684,		/* nws.nws_cmu.ln3_bist_rx_status (1 regs) */
	0x031cb688,		/* nws.nws_cmu.ln3_bist_rx_ber_status0 .. nws.nws_cmu.ln3_bist_rx_ber_status2 (3 regs) */
	0x031cb68c,		/* nws.nws_cmu.ln3_bist_rx_ber_status4 .. nws.nws_cmu.ln3_bist_rx_ber_status6 (3 regs) */
	0x041cb694,		/* nws.nws_cmu.ln3_bist_rx_lock_ctrl0 .. nws.nws_cmu.ln3_bist_rx_lock_ctrl3 (4 regs) */
	0x051cb6a0,		/* nws.nws_cmu.ln3_bist_rx_loss_lock_ctrl0 .. nws.nws_cmu.ln3_bist_rx_loss_lock_ctrl4 (5 regs) */
	0x011cb6b0,		/* nws.nws_cmu.ln3_bist_rx_udp_shift_amount (1 regs) */
	0x191cb6b4,		/* nws.nws_cmu.ln3_bist_rx_udp_7_0 .. nws.nws_cmu.ln3_bist_rx_udp_199_192 (25 regs) */
	0x021cb700,		/* nws.nws_cmu.ln3_feature_rxterm_cfg0 .. nws.nws_cmu.ln3_feature_rxclkdiv_cfg0 (2 regs) */
	0x061cb704,		/* nws.nws_cmu.ln3_feature_ReservedRegister2815 .. nws.nws_cmu.ln3_feature_ReservedRegister2820 (6 regs) */
	0x071cb710,		/* nws.nws_cmu.ln3_feature_ReservedRegister2821 .. nws.nws_cmu.ln3_feature_ReservedRegister2827 (7 regs) */
	0x0a1cb720,		/* nws.nws_cmu.ln3_feature_ReservedRegister2828 .. nws.nws_cmu.ln3_feature_ReservedRegister2830 (10 regs) */
	0x071cb730,		/* nws.nws_cmu.ln3_feature_dfe_cfg .. nws.nws_cmu.ln3_feature_dfe_adapt_tap5_cfg (7 regs) */
	0x101cb738,		/* nws.nws_cmu.ln3_feature_adapt_cont_cfg0 .. nws.nws_cmu.ln3_feature_ReservedRegister2842 (16 regs) */
	0x011cb750,		/* nws.nws_cmu.ln3_feature_test_cfg0 (1 regs) */
	0x081cb758,		/* nws.nws_cmu.ln3_feature_ReservedRegister2843 .. nws.nws_cmu.ln3_feature_ReservedRegister2850 (8 regs) */
	0x061cb780,		/* nws.nws_cmu.ln3_lt_tx_fsm_ctrl0 .. nws.nws_cmu.ln3_lt_tx_fsm_ctrl5 (6 regs) */
	0x011cb790,		/* nws.nws_cmu.ln3_lt_tx_fsm_status (1 regs) */
	0x031cb793,		/* nws.nws_cmu.ln3_lt_tx_prbs_ctrl0 .. nws.nws_cmu.ln3_lt_tx_prbs_ctrl2 (3 regs) */
	0x011cb7a0,		/* nws.nws_cmu.ln3_lt_tx_coefficient_update_ctrl (1 regs) */
	0x011cb7a2,		/* nws.nws_cmu.ln3_lt_tx_status_report_ctrl (1 regs) */
	0x021cb7b0,		/* nws.nws_cmu.ln3_lt_tx_fsm_state_status0 .. nws.nws_cmu.ln3_lt_tx_fsm_state_status1 (2 regs) */
	0x011cb7c0,		/* nws.nws_cmu.ln3_lt_rx_ctrl0 (1 regs) */
	0x021cb7c2,		/* nws.nws_cmu.ln3_lt_rx_prbs_ctrl0 .. nws.nws_cmu.ln3_lt_rx_prbs_ctrl1 (2 regs) */
	0x031cb7c5,		/* nws.nws_cmu.ln3_lt_rx_prbs_status0 .. nws.nws_cmu.ln3_lt_rx_prbs_status2 (3 regs) */
	0x011cb7d0,		/* nws.nws_cmu.ln3_lt_rx_frame_ctrl (1 regs) */
	0x051cb7d3,		/* nws.nws_cmu.ln3_lt_rx_frame_status0 .. nws.nws_cmu.ln3_lt_rx_frame_status4 (5 regs) */
	0x011cb7e0,		/* nws.nws_cmu.ln3_lt_rx_coefficient_update_status (1 regs) */
	0x011cb7e2,		/* nws.nws_cmu.ln3_lt_rx_report_status (1 regs) */
	0x2f4c0000,		/* block ms */
	0x061a8000,		/* ms.common_control .. ms.eco_reserved (6 regs) */
	0x021a8060,		/* ms.INT_STS .. ms.INT_MASK (2 regs) */
	0x0b1a808a,		/* ms.dbg_select .. ms.dbg_fw_trigger_enable (11 regs) */
	0x241a9000,		/* ms.ms_cmu.ahb_cmu_csr_0_x0 .. ms.ms_cmu.ahb_cmu_csr_0_x35 (36 regs) */
	0x4f1a9038,		/* ms.ms_cmu.ahb_cmu_csr_0_x56 .. ms.ms_cmu.ahb_cmu_csr_0_x134 (79 regs) */
	0x011a9090,		/* ms.ms_cmu.ahb_cmu_csr_0_x144 (1 regs) */
	0x041a9092,		/* ms.ms_cmu.ahb_cmu_csr_0_x146 .. ms.ms_cmu.ahb_cmu_csr_0_x149 (4 regs) */
	0x021a9099,		/* ms.ms_cmu.ahb_cmu_csr_0_x153 .. ms.ms_cmu.ahb_cmu_csr_0_x154 (2 regs) */
	0x2a1a90a1,		/* ms.ms_cmu.ahb_cmu_csr_0_x161 .. ms.ms_cmu.ahb_cmu_csr_0_x202 (42 regs) */
	0x011a90d2,		/* ms.ms_cmu.ahb_cmu_csr_0_x210 (1 regs) */
	0x061a9400,		/* ms.ms_cmu.ahb_lane_csr_1_x0 .. ms.ms_cmu.ahb_lane_csr_1_x5 (6 regs) */
	0x401a9407,		/* ms.ms_cmu.ahb_lane_csr_1_x7 .. ms.ms_cmu.ahb_lane_csr_1_x70 (64 regs) */
	0x221a9448,		/* ms.ms_cmu.ahb_lane_csr_1_x72 .. ms.ms_cmu.ahb_lane_csr_1_x105 (34 regs) */
	0x0e1a9473,		/* ms.ms_cmu.ahb_lane_csr_1_x115 .. ms.ms_cmu.ahb_lane_csr_1_x128 (14 regs) */
	0x1b1a9482,		/* ms.ms_cmu.ahb_lane_csr_1_x130 .. ms.ms_cmu.ahb_lane_csr_1_x156 (27 regs) */
	0x021a949e,		/* ms.ms_cmu.ahb_lane_csr_1_x158 .. ms.ms_cmu.ahb_lane_csr_1_x159 (2 regs) */
	0x011a94a1,		/* ms.ms_cmu.ahb_lane_csr_1_x161 (1 regs) */
	0x011a94a7,		/* ms.ms_cmu.ahb_lane_csr_1_x167 (1 regs) */
	0x551a94c9,		/* ms.ms_cmu.ahb_lane_csr_1_x201 .. ms.ms_cmu.ahb_lane_csr_1_x285 (85 regs) */
	0x081a952d,		/* ms.ms_cmu.ahb_lane_csr_1_x301 .. ms.ms_cmu.ahb_lane_csr_1_x308 (8 regs) */
	0x011a9536,		/* ms.ms_cmu.ahb_lane_csr_1_x310 (1 regs) */
	0x121a9539,		/* ms.ms_cmu.ahb_lane_csr_1_ReservedReg53 .. ms.ms_cmu.ahb_lane_csr_1_x330 (18 regs) */
	0x021a9a00,		/* ms.ms_cmu.ahb_comlane_csr_5_x0 .. ms.ms_cmu.ahb_comlane_csr_5_x1 (2 regs) */
	0x051a9a04,		/* ms.ms_cmu.ahb_comlane_csr_5_x4 .. ms.ms_cmu.ahb_comlane_csr_5_x8 (5 regs) */
	0x011a9a1f,		/* ms.ms_cmu.ahb_comlane_csr_5_x31 (1 regs) */
	0x031a9a2a,		/* ms.ms_cmu.ahb_comlane_csr_5_ReservedReg17 .. ms.ms_cmu.ahb_comlane_csr_5_x44 (3 regs) */
	0x851a9a31,		/* ms.ms_cmu.ahb_comlane_csr_5_x49 .. ms.ms_cmu.ahb_comlane_csr_5_x181 (133 regs) */
	0x341a9ad2,		/* ms.ms_cmu.ahb_comlane_csr_5_x210 .. ms.ms_cmu.ahb_comlane_csr_5_x261 (52 regs) */
	0x0d1a9b07,		/* ms.ms_cmu.ahb_comlane_csr_5_x263 .. ms.ms_cmu.ahb_comlane_csr_5_x275 (13 regs) */
	0x011a9b19,		/* ms.ms_cmu.ahb_comlane_csr_5_x281 (1 regs) */
	0x021a9b28,		/* ms.ms_cmu.ahb_comlane_csr_5_x296 .. ms.ms_cmu.ahb_comlane_csr_5_x297 (2 regs) */
	0x011a9b2d,		/* ms.ms_cmu.ahb_comlane_csr_5_x301 (1 regs) */
	0x061a9b2f,		/* ms.ms_cmu.ahb_comlane_csr_5_ReservedReg23 .. ms.ms_cmu.ahb_comlane_csr_5_x308 (6 regs) */
	0x061a9b36,		/* ms.ms_cmu.ahb_comlane_csr_5_x310 .. ms.ms_cmu.ahb_comlane_csr_5_x315 (6 regs) */
	0x061a9b3d,		/* ms.ms_cmu.ahb_comlane_csr_5_ReservedReg24 .. ms.ms_cmu.ahb_comlane_csr_5_x322 (6 regs) */
	0x471a9b44,		/* ms.ms_cmu.ahb_comlane_csr_5_x324 .. ms.ms_cmu.ahb_comlane_csr_5_x394 (71 regs) */
	0x031a9b91,		/* ms.ms_cmu.ahb_comlane_csr_5_x401 .. ms.ms_cmu.ahb_comlane_csr_5_x403 (3 regs) */
	0x041a9b96,		/* ms.ms_cmu.ahb_comlane_csr_5_x406 .. ms.ms_cmu.ahb_comlane_csr_5_x409 (4 regs) */
	0x011a9b9c,		/* ms.ms_cmu.ahb_comlane_csr_5_ReservedReg40 (1 regs) */
	0x361a9bbe,		/* ms.ms_cmu.ahb_comlane_csr_5_x446 .. ms.ms_cmu.ahb_comlane_csr_5_x499 (54 regs) */
	0x241a9c00,		/* ms.ms_cmu.ahb_cmu1_csr_6_x0 .. ms.ms_cmu.ahb_cmu1_csr_6_x35 (36 regs) */
	0x4f1a9c38,		/* ms.ms_cmu.ahb_cmu1_csr_6_x56 .. ms.ms_cmu.ahb_cmu1_csr_6_x134 (79 regs) */
	0x011a9c90,		/* ms.ms_cmu.ahb_cmu1_csr_6_x144 (1 regs) */
	0x041a9c92,		/* ms.ms_cmu.ahb_cmu1_csr_6_x146 .. ms.ms_cmu.ahb_cmu1_csr_6_x149 (4 regs) */
	0x021a9c99,		/* ms.ms_cmu.ahb_cmu1_csr_6_x153 .. ms.ms_cmu.ahb_cmu1_csr_6_x154 (2 regs) */
	0x2a1a9ca1,		/* ms.ms_cmu.ahb_cmu1_csr_6_x161 .. ms.ms_cmu.ahb_cmu1_csr_6_x202 (42 regs) */
	0x011a9cd2,		/* ms.ms_cmu.ahb_cmu1_csr_6_x210 (1 regs) */
	0x924d0000,		/* block phy_pcie */
	0x23188000,		/* phy_pcie.phy0.ahb_cmu_csr_0_x0 .. phy_pcie.phy0.ahb_cmu_csr_0_x34 (35 regs) */
	0x2f188038,		/* phy_pcie.phy0.ahb_cmu_csr_0_x56 .. phy_pcie.phy0.ahb_cmu_csr_0_x102 (47 regs) */
	0x2818806c,		/* phy_pcie.phy0.ahb_cmu_csr_0_x108 .. phy_pcie.phy0.ahb_cmu_csr_0_x147 (40 regs) */
	0x06188095,		/* phy_pcie.phy0.ahb_cmu_csr_0_x149 .. phy_pcie.phy0.ahb_cmu_csr_0_x154 (6 regs) */
	0x1c1880a1,		/* phy_pcie.phy0.ahb_cmu_csr_0_x161 .. phy_pcie.phy0.ahb_cmu_csr_0_x188 (28 regs) */
	0x141880bf,		/* phy_pcie.phy0.ahb_cmu_csr_0_x191 .. phy_pcie.phy0.ahb_cmu_csr_0_x210 (20 regs) */
	0x06188200,		/* phy_pcie.phy0.ahb_lane_csr_1_x0 .. phy_pcie.phy0.ahb_lane_csr_1_x5 (6 regs) */
	0x38188207,		/* phy_pcie.phy0.ahb_lane_csr_1_x7 .. phy_pcie.phy0.ahb_lane_csr_1_x62 (56 regs) */
	0x21188241,		/* phy_pcie.phy0.ahb_lane_csr_1_x65 .. phy_pcie.phy0.ahb_lane_csr_1_x97 (33 regs) */
	0x11188263,		/* phy_pcie.phy0.ahb_lane_csr_1_x99 .. phy_pcie.phy0.ahb_lane_csr_1_x115 (17 regs) */
	0x0a188277,		/* phy_pcie.phy0.ahb_lane_csr_1_x119 .. phy_pcie.phy0.ahb_lane_csr_1_x128 (10 regs) */
	0x29188282,		/* phy_pcie.phy0.ahb_lane_csr_1_x130 .. phy_pcie.phy0.ahb_lane_csr_1_x170 (41 regs) */
	0x081882c9,		/* phy_pcie.phy0.ahb_lane_csr_1_x201 .. phy_pcie.phy0.ahb_lane_csr_1_x208 (8 regs) */
	0x051882d5,		/* phy_pcie.phy0.ahb_lane_csr_1_x213 .. phy_pcie.phy0.ahb_lane_csr_1_x217 (5 regs) */
	0x421882dc,		/* phy_pcie.phy0.ahb_lane_csr_1_x220 .. phy_pcie.phy0.ahb_lane_csr_1_x285 (66 regs) */
	0x1b18832d,		/* phy_pcie.phy0.ahb_lane_csr_1_x301 .. phy_pcie.phy0.ahb_lane_csr_1_x327 (27 regs) */
	0x0118834a,		/* phy_pcie.phy0.ahb_lane_csr_1_x330 (1 regs) */
	0x06188400,		/* phy_pcie.phy0.ahb_lane_csr_2_x0 .. phy_pcie.phy0.ahb_lane_csr_2_x5 (6 regs) */
	0x38188407,		/* phy_pcie.phy0.ahb_lane_csr_2_x7 .. phy_pcie.phy0.ahb_lane_csr_2_x62 (56 regs) */
	0x21188441,		/* phy_pcie.phy0.ahb_lane_csr_2_x65 .. phy_pcie.phy0.ahb_lane_csr_2_x97 (33 regs) */
	0x11188463,		/* phy_pcie.phy0.ahb_lane_csr_2_x99 .. phy_pcie.phy0.ahb_lane_csr_2_x115 (17 regs) */
	0x0a188477,		/* phy_pcie.phy0.ahb_lane_csr_2_x119 .. phy_pcie.phy0.ahb_lane_csr_2_x128 (10 regs) */
	0x29188482,		/* phy_pcie.phy0.ahb_lane_csr_2_x130 .. phy_pcie.phy0.ahb_lane_csr_2_x170 (41 regs) */
	0x081884c9,		/* phy_pcie.phy0.ahb_lane_csr_2_x201 .. phy_pcie.phy0.ahb_lane_csr_2_x208 (8 regs) */
	0x051884d5,		/* phy_pcie.phy0.ahb_lane_csr_2_x213 .. phy_pcie.phy0.ahb_lane_csr_2_x217 (5 regs) */
	0x421884dc,		/* phy_pcie.phy0.ahb_lane_csr_2_x220 .. phy_pcie.phy0.ahb_lane_csr_2_x285 (66 regs) */
	0x1b18852d,		/* phy_pcie.phy0.ahb_lane_csr_2_x301 .. phy_pcie.phy0.ahb_lane_csr_2_x327 (27 regs) */
	0x0118854a,		/* phy_pcie.phy0.ahb_lane_csr_2_x330 (1 regs) */
	0x06188600,		/* phy_pcie.phy0.ahb_lane_csr_3_x0 .. phy_pcie.phy0.ahb_lane_csr_3_x5 (6 regs) */
	0x38188607,		/* phy_pcie.phy0.ahb_lane_csr_3_x7 .. phy_pcie.phy0.ahb_lane_csr_3_x62 (56 regs) */
	0x21188641,		/* phy_pcie.phy0.ahb_lane_csr_3_x65 .. phy_pcie.phy0.ahb_lane_csr_3_x97 (33 regs) */
	0x11188663,		/* phy_pcie.phy0.ahb_lane_csr_3_x99 .. phy_pcie.phy0.ahb_lane_csr_3_x115 (17 regs) */
	0x0a188677,		/* phy_pcie.phy0.ahb_lane_csr_3_x119 .. phy_pcie.phy0.ahb_lane_csr_3_x128 (10 regs) */
	0x29188682,		/* phy_pcie.phy0.ahb_lane_csr_3_x130 .. phy_pcie.phy0.ahb_lane_csr_3_x170 (41 regs) */
	0x081886c9,		/* phy_pcie.phy0.ahb_lane_csr_3_x201 .. phy_pcie.phy0.ahb_lane_csr_3_x208 (8 regs) */
	0x051886d5,		/* phy_pcie.phy0.ahb_lane_csr_3_x213 .. phy_pcie.phy0.ahb_lane_csr_3_x217 (5 regs) */
	0x421886dc,		/* phy_pcie.phy0.ahb_lane_csr_3_x220 .. phy_pcie.phy0.ahb_lane_csr_3_x285 (66 regs) */
	0x1b18872d,		/* phy_pcie.phy0.ahb_lane_csr_3_x301 .. phy_pcie.phy0.ahb_lane_csr_3_x327 (27 regs) */
	0x0118874a,		/* phy_pcie.phy0.ahb_lane_csr_3_x330 (1 regs) */
	0x06188800,		/* phy_pcie.phy0.ahb_lane_csr_4_x0 .. phy_pcie.phy0.ahb_lane_csr_4_x5 (6 regs) */
	0x38188807,		/* phy_pcie.phy0.ahb_lane_csr_4_x7 .. phy_pcie.phy0.ahb_lane_csr_4_x62 (56 regs) */
	0x21188841,		/* phy_pcie.phy0.ahb_lane_csr_4_x65 .. phy_pcie.phy0.ahb_lane_csr_4_x97 (33 regs) */
	0x11188863,		/* phy_pcie.phy0.ahb_lane_csr_4_x99 .. phy_pcie.phy0.ahb_lane_csr_4_x115 (17 regs) */
	0x0a188877,		/* phy_pcie.phy0.ahb_lane_csr_4_x119 .. phy_pcie.phy0.ahb_lane_csr_4_x128 (10 regs) */
	0x29188882,		/* phy_pcie.phy0.ahb_lane_csr_4_x130 .. phy_pcie.phy0.ahb_lane_csr_4_x170 (41 regs) */
	0x081888c9,		/* phy_pcie.phy0.ahb_lane_csr_4_x201 .. phy_pcie.phy0.ahb_lane_csr_4_x208 (8 regs) */
	0x051888d5,		/* phy_pcie.phy0.ahb_lane_csr_4_x213 .. phy_pcie.phy0.ahb_lane_csr_4_x217 (5 regs) */
	0x421888dc,		/* phy_pcie.phy0.ahb_lane_csr_4_x220 .. phy_pcie.phy0.ahb_lane_csr_4_x285 (66 regs) */
	0x1b18892d,		/* phy_pcie.phy0.ahb_lane_csr_4_x301 .. phy_pcie.phy0.ahb_lane_csr_4_x327 (27 regs) */
	0x0118894a,		/* phy_pcie.phy0.ahb_lane_csr_4_x330 (1 regs) */
	0x05188a00,		/* phy_pcie.phy0.ahb_comlane_csr_5_x0 .. phy_pcie.phy0.ahb_comlane_csr_5_x4 (5 regs) */
	0x15188a06,		/* phy_pcie.phy0.ahb_comlane_csr_5_x6 .. phy_pcie.phy0.ahb_comlane_csr_5_x26 (21 regs) */
	0x07188a1d,		/* phy_pcie.phy0.ahb_comlane_csr_5_x29 .. phy_pcie.phy0.ahb_comlane_csr_5_x35 (7 regs) */
	0x04188a26,		/* phy_pcie.phy0.ahb_comlane_csr_5_x38 .. phy_pcie.phy0.ahb_comlane_csr_5_x41 (4 regs) */
	0x78188a2b,		/* phy_pcie.phy0.ahb_comlane_csr_5_x43 .. phy_pcie.phy0.ahb_comlane_csr_5_x162 (120 regs) */
	0x5e188aa8,		/* phy_pcie.phy0.ahb_comlane_csr_5_x168 .. phy_pcie.phy0.ahb_comlane_csr_5_x261 (94 regs) */
	0x23188b07,		/* phy_pcie.phy0.ahb_comlane_csr_5_x263 .. phy_pcie.phy0.ahb_comlane_csr_5_x297 (35 regs) */
	0x01188b2d,		/* phy_pcie.phy0.ahb_comlane_csr_5_x301 (1 regs) */
	0x05188b30,		/* phy_pcie.phy0.ahb_comlane_csr_5_x304 .. phy_pcie.phy0.ahb_comlane_csr_5_x308 (5 regs) */
	0x06188b36,		/* phy_pcie.phy0.ahb_comlane_csr_5_x310 .. phy_pcie.phy0.ahb_comlane_csr_5_x315 (6 regs) */
	0x05188b3e,		/* phy_pcie.phy0.ahb_comlane_csr_5_x318 .. phy_pcie.phy0.ahb_comlane_csr_5_x322 (5 regs) */
	0x0a188b44,		/* phy_pcie.phy0.ahb_comlane_csr_5_x324 .. phy_pcie.phy0.ahb_comlane_csr_5_x333 (10 regs) */
	0x02188b52,		/* phy_pcie.phy0.ahb_comlane_csr_5_x338 .. phy_pcie.phy0.ahb_comlane_csr_5_x339 (2 regs) */
	0x02188b55,		/* phy_pcie.phy0.ahb_comlane_csr_5_x341 .. phy_pcie.phy0.ahb_comlane_csr_5_x342 (2 regs) */
	0x01188b58,		/* phy_pcie.phy0.ahb_comlane_csr_5_x344 (1 regs) */
	0x0a188b5a,		/* phy_pcie.phy0.ahb_comlane_csr_5_x346 .. phy_pcie.phy0.ahb_comlane_csr_5_x355 (10 regs) */
	0x01188b66,		/* phy_pcie.phy0.ahb_comlane_csr_5_x358 (1 regs) */
	0x0c188b6a,		/* phy_pcie.phy0.ahb_comlane_csr_5_x362 .. phy_pcie.phy0.ahb_comlane_csr_5_x373 (12 regs) */
	0x11188b78,		/* phy_pcie.phy0.ahb_comlane_csr_5_x376 .. phy_pcie.phy0.ahb_comlane_csr_5_x392 (17 regs) */
	0x01188b8a,		/* phy_pcie.phy0.ahb_comlane_csr_5_x394 (1 regs) */
	0x0b188b91,		/* phy_pcie.phy0.ahb_comlane_csr_5_x401 .. phy_pcie.phy0.ahb_comlane_csr_5_x411 (11 regs) */
	0x57188b9d,		/* phy_pcie.phy0.ahb_comlane_csr_5_x413 .. phy_pcie.phy0.ahb_comlane_csr_5_x499 (87 regs) */
	0x23189000,		/* phy_pcie.phy1.ahb_cmu_csr_0_x0 .. phy_pcie.phy1.ahb_cmu_csr_0_x34 (35 regs) */
	0x2f189038,		/* phy_pcie.phy1.ahb_cmu_csr_0_x56 .. phy_pcie.phy1.ahb_cmu_csr_0_x102 (47 regs) */
	0x2818906c,		/* phy_pcie.phy1.ahb_cmu_csr_0_x108 .. phy_pcie.phy1.ahb_cmu_csr_0_x147 (40 regs) */
	0x06189095,		/* phy_pcie.phy1.ahb_cmu_csr_0_x149 .. phy_pcie.phy1.ahb_cmu_csr_0_x154 (6 regs) */
	0x1c1890a1,		/* phy_pcie.phy1.ahb_cmu_csr_0_x161 .. phy_pcie.phy1.ahb_cmu_csr_0_x188 (28 regs) */
	0x141890bf,		/* phy_pcie.phy1.ahb_cmu_csr_0_x191 .. phy_pcie.phy1.ahb_cmu_csr_0_x210 (20 regs) */
	0x06189200,		/* phy_pcie.phy1.ahb_lane_csr_1_x0 .. phy_pcie.phy1.ahb_lane_csr_1_x5 (6 regs) */
	0x38189207,		/* phy_pcie.phy1.ahb_lane_csr_1_x7 .. phy_pcie.phy1.ahb_lane_csr_1_x62 (56 regs) */
	0x21189241,		/* phy_pcie.phy1.ahb_lane_csr_1_x65 .. phy_pcie.phy1.ahb_lane_csr_1_x97 (33 regs) */
	0x11189263,		/* phy_pcie.phy1.ahb_lane_csr_1_x99 .. phy_pcie.phy1.ahb_lane_csr_1_x115 (17 regs) */
	0x0a189277,		/* phy_pcie.phy1.ahb_lane_csr_1_x119 .. phy_pcie.phy1.ahb_lane_csr_1_x128 (10 regs) */
	0x29189282,		/* phy_pcie.phy1.ahb_lane_csr_1_x130 .. phy_pcie.phy1.ahb_lane_csr_1_x170 (41 regs) */
	0x081892c9,		/* phy_pcie.phy1.ahb_lane_csr_1_x201 .. phy_pcie.phy1.ahb_lane_csr_1_x208 (8 regs) */
	0x051892d5,		/* phy_pcie.phy1.ahb_lane_csr_1_x213 .. phy_pcie.phy1.ahb_lane_csr_1_x217 (5 regs) */
	0x421892dc,		/* phy_pcie.phy1.ahb_lane_csr_1_x220 .. phy_pcie.phy1.ahb_lane_csr_1_x285 (66 regs) */
	0x1b18932d,		/* phy_pcie.phy1.ahb_lane_csr_1_x301 .. phy_pcie.phy1.ahb_lane_csr_1_x327 (27 regs) */
	0x0118934a,		/* phy_pcie.phy1.ahb_lane_csr_1_x330 (1 regs) */
	0x06189400,		/* phy_pcie.phy1.ahb_lane_csr_2_x0 .. phy_pcie.phy1.ahb_lane_csr_2_x5 (6 regs) */
	0x38189407,		/* phy_pcie.phy1.ahb_lane_csr_2_x7 .. phy_pcie.phy1.ahb_lane_csr_2_x62 (56 regs) */
	0x21189441,		/* phy_pcie.phy1.ahb_lane_csr_2_x65 .. phy_pcie.phy1.ahb_lane_csr_2_x97 (33 regs) */
	0x11189463,		/* phy_pcie.phy1.ahb_lane_csr_2_x99 .. phy_pcie.phy1.ahb_lane_csr_2_x115 (17 regs) */
	0x0a189477,		/* phy_pcie.phy1.ahb_lane_csr_2_x119 .. phy_pcie.phy1.ahb_lane_csr_2_x128 (10 regs) */
	0x29189482,		/* phy_pcie.phy1.ahb_lane_csr_2_x130 .. phy_pcie.phy1.ahb_lane_csr_2_x170 (41 regs) */
	0x081894c9,		/* phy_pcie.phy1.ahb_lane_csr_2_x201 .. phy_pcie.phy1.ahb_lane_csr_2_x208 (8 regs) */
	0x051894d5,		/* phy_pcie.phy1.ahb_lane_csr_2_x213 .. phy_pcie.phy1.ahb_lane_csr_2_x217 (5 regs) */
	0x421894dc,		/* phy_pcie.phy1.ahb_lane_csr_2_x220 .. phy_pcie.phy1.ahb_lane_csr_2_x285 (66 regs) */
	0x1b18952d,		/* phy_pcie.phy1.ahb_lane_csr_2_x301 .. phy_pcie.phy1.ahb_lane_csr_2_x327 (27 regs) */
	0x0118954a,		/* phy_pcie.phy1.ahb_lane_csr_2_x330 (1 regs) */
	0x06189600,		/* phy_pcie.phy1.ahb_lane_csr_3_x0 .. phy_pcie.phy1.ahb_lane_csr_3_x5 (6 regs) */
	0x38189607,		/* phy_pcie.phy1.ahb_lane_csr_3_x7 .. phy_pcie.phy1.ahb_lane_csr_3_x62 (56 regs) */
	0x21189641,		/* phy_pcie.phy1.ahb_lane_csr_3_x65 .. phy_pcie.phy1.ahb_lane_csr_3_x97 (33 regs) */
	0x11189663,		/* phy_pcie.phy1.ahb_lane_csr_3_x99 .. phy_pcie.phy1.ahb_lane_csr_3_x115 (17 regs) */
	0x0a189677,		/* phy_pcie.phy1.ahb_lane_csr_3_x119 .. phy_pcie.phy1.ahb_lane_csr_3_x128 (10 regs) */
	0x29189682,		/* phy_pcie.phy1.ahb_lane_csr_3_x130 .. phy_pcie.phy1.ahb_lane_csr_3_x170 (41 regs) */
	0x081896c9,		/* phy_pcie.phy1.ahb_lane_csr_3_x201 .. phy_pcie.phy1.ahb_lane_csr_3_x208 (8 regs) */
	0x051896d5,		/* phy_pcie.phy1.ahb_lane_csr_3_x213 .. phy_pcie.phy1.ahb_lane_csr_3_x217 (5 regs) */
	0x421896dc,		/* phy_pcie.phy1.ahb_lane_csr_3_x220 .. phy_pcie.phy1.ahb_lane_csr_3_x285 (66 regs) */
	0x1b18972d,		/* phy_pcie.phy1.ahb_lane_csr_3_x301 .. phy_pcie.phy1.ahb_lane_csr_3_x327 (27 regs) */
	0x0118974a,		/* phy_pcie.phy1.ahb_lane_csr_3_x330 (1 regs) */
	0x06189800,		/* phy_pcie.phy1.ahb_lane_csr_4_x0 .. phy_pcie.phy1.ahb_lane_csr_4_x5 (6 regs) */
	0x38189807,		/* phy_pcie.phy1.ahb_lane_csr_4_x7 .. phy_pcie.phy1.ahb_lane_csr_4_x62 (56 regs) */
	0x21189841,		/* phy_pcie.phy1.ahb_lane_csr_4_x65 .. phy_pcie.phy1.ahb_lane_csr_4_x97 (33 regs) */
	0x11189863,		/* phy_pcie.phy1.ahb_lane_csr_4_x99 .. phy_pcie.phy1.ahb_lane_csr_4_x115 (17 regs) */
	0x0a189877,		/* phy_pcie.phy1.ahb_lane_csr_4_x119 .. phy_pcie.phy1.ahb_lane_csr_4_x128 (10 regs) */
	0x29189882,		/* phy_pcie.phy1.ahb_lane_csr_4_x130 .. phy_pcie.phy1.ahb_lane_csr_4_x170 (41 regs) */
	0x081898c9,		/* phy_pcie.phy1.ahb_lane_csr_4_x201 .. phy_pcie.phy1.ahb_lane_csr_4_x208 (8 regs) */
	0x051898d5,		/* phy_pcie.phy1.ahb_lane_csr_4_x213 .. phy_pcie.phy1.ahb_lane_csr_4_x217 (5 regs) */
	0x421898dc,		/* phy_pcie.phy1.ahb_lane_csr_4_x220 .. phy_pcie.phy1.ahb_lane_csr_4_x285 (66 regs) */
	0x1b18992d,		/* phy_pcie.phy1.ahb_lane_csr_4_x301 .. phy_pcie.phy1.ahb_lane_csr_4_x327 (27 regs) */
	0x0118994a,		/* phy_pcie.phy1.ahb_lane_csr_4_x330 (1 regs) */
	0x05189a00,		/* phy_pcie.phy1.ahb_comlane_csr_5_x0 .. phy_pcie.phy1.ahb_comlane_csr_5_x4 (5 regs) */
	0x15189a06,		/* phy_pcie.phy1.ahb_comlane_csr_5_x6 .. phy_pcie.phy1.ahb_comlane_csr_5_x26 (21 regs) */
	0x07189a1d,		/* phy_pcie.phy1.ahb_comlane_csr_5_x29 .. phy_pcie.phy1.ahb_comlane_csr_5_x35 (7 regs) */
	0x04189a26,		/* phy_pcie.phy1.ahb_comlane_csr_5_x38 .. phy_pcie.phy1.ahb_comlane_csr_5_x41 (4 regs) */
	0x78189a2b,		/* phy_pcie.phy1.ahb_comlane_csr_5_x43 .. phy_pcie.phy1.ahb_comlane_csr_5_x162 (120 regs) */
	0x5e189aa8,		/* phy_pcie.phy1.ahb_comlane_csr_5_x168 .. phy_pcie.phy1.ahb_comlane_csr_5_x261 (94 regs) */
	0x23189b07,		/* phy_pcie.phy1.ahb_comlane_csr_5_x263 .. phy_pcie.phy1.ahb_comlane_csr_5_x297 (35 regs) */
	0x01189b2d,		/* phy_pcie.phy1.ahb_comlane_csr_5_x301 (1 regs) */
	0x05189b30,		/* phy_pcie.phy1.ahb_comlane_csr_5_x304 .. phy_pcie.phy1.ahb_comlane_csr_5_x308 (5 regs) */
	0x06189b36,		/* phy_pcie.phy1.ahb_comlane_csr_5_x310 .. phy_pcie.phy1.ahb_comlane_csr_5_x315 (6 regs) */
	0x05189b3e,		/* phy_pcie.phy1.ahb_comlane_csr_5_x318 .. phy_pcie.phy1.ahb_comlane_csr_5_x322 (5 regs) */
	0x0a189b44,		/* phy_pcie.phy1.ahb_comlane_csr_5_x324 .. phy_pcie.phy1.ahb_comlane_csr_5_x333 (10 regs) */
	0x02189b52,		/* phy_pcie.phy1.ahb_comlane_csr_5_x338 .. phy_pcie.phy1.ahb_comlane_csr_5_x339 (2 regs) */
	0x02189b55,		/* phy_pcie.phy1.ahb_comlane_csr_5_x341 .. phy_pcie.phy1.ahb_comlane_csr_5_x342 (2 regs) */
	0x01189b58,		/* phy_pcie.phy1.ahb_comlane_csr_5_x344 (1 regs) */
	0x0a189b5a,		/* phy_pcie.phy1.ahb_comlane_csr_5_x346 .. phy_pcie.phy1.ahb_comlane_csr_5_x355 (10 regs) */
	0x01189b66,		/* phy_pcie.phy1.ahb_comlane_csr_5_x358 (1 regs) */
	0x0c189b6a,		/* phy_pcie.phy1.ahb_comlane_csr_5_x362 .. phy_pcie.phy1.ahb_comlane_csr_5_x373 (12 regs) */
	0x11189b78,		/* phy_pcie.phy1.ahb_comlane_csr_5_x376 .. phy_pcie.phy1.ahb_comlane_csr_5_x392 (17 regs) */
	0x01189b8a,		/* phy_pcie.phy1.ahb_comlane_csr_5_x394 (1 regs) */
	0x0b189b91,		/* phy_pcie.phy1.ahb_comlane_csr_5_x401 .. phy_pcie.phy1.ahb_comlane_csr_5_x411 (11 regs) */
	0x57189b9d,		/* phy_pcie.phy1.ahb_comlane_csr_5_x413 .. phy_pcie.phy1.ahb_comlane_csr_5_x499 (87 regs) */
	0x0c18a000,		/* phy_pcie.eco_reserved .. phy_pcie.dbg_status (12 regs) */
	0x0518a7fa,		/* phy_pcie.dbg_select .. phy_pcie.dbg_force_frame (5 regs) */
	0x0a1f0151,		/* mode !fpga, block tm */
	0x100b0000,		/* tm.memory_self_init_start .. tm.ac_command_fifo_init (16 regs) */
	0x040b0018,		/* tm.pxp_interface_enable .. tm.client_in_interface_enable (4 regs) */
	0x050b001e,		/* tm.pxp_request_credit .. tm.load_request_credit (5 regs) */
	0x020b0060,		/* tm.INT_STS_0 .. tm.INT_MASK_0 (2 regs) */
	0x020b0064,		/* tm.INT_STS_1 .. tm.INT_MASK_1 (2 regs) */
	0x0b0b0100,		/* tm.pxp_read_data_fifo_a_f_thr .. tm.ac_command_fifo_a_f_thr (11 regs) */
	0x2d0b0112,		/* tm.tick_timer_val .. tm.task_timer_threshold_2 (45 regs) */
	0x030b0143,		/* tm.during_scan_conn .. tm.during_scan (3 regs) */
	0x1f0b0180,		/* tm.completed_scans .. tm.pxp_read_data_error (31 regs) */
	0x2f0b01c0,		/* tm.current_time .. tm.dbg_force_frame (47 regs) */
	0x08220000,		/* block src */
	0x0108e010,		/* src.CTRL (1 regs) */
	0x0108e076,		/* src.INT_STS (1 regs) */
	0x0108e079,		/* src.INT_MASK (1 regs) */
	0x0b08e100,		/* src.KeySearch_0 .. src.KeySearch_vlan (11 regs) */
	0x0708e120,		/* src.IF_Stat_PF_Config .. src.IF_Stat_No_Read_Counter (7 regs) */
	0x0108e180,		/* src.PXP_CTRL (1 regs) */
	0x0508e1c0,		/* src.dbg_select .. src.dbg_force_frame (5 regs) */
	0x0208e1d2,		/* src.eco_reserved .. src.soft_rst (2 regs) */
	0x042a0000,		/* block tsem */
	0x0e5d2800,		/* tsem.fast_memory.vfc_config.mask_lsb_0_low .. tsem.fast_memory.vfc_config.indications2 (14 regs) */
	0x135d280f,		/* tsem.fast_memory.vfc_config.memories_rst .. tsem.fast_memory.vfc_config.cpu_mbist_memctrl_1_cntrl_cmd (19 regs) */
	0x125d2824,		/* tsem.fast_memory.vfc_config.debug_data .. tsem.fast_memory.vfc_config.mask_lsb_7_high (18 regs) */
	0x0c5d283e,		/* tsem.fast_memory.vfc_config.offset_alu_vector_0 .. tsem.fast_memory.vfc_config.cam_bist_skip_error_cnt (12 regs) */
	0x103f0000,		/* block tcfc */
	0x050b4000,		/* tcfc.init_reg .. tcfc.tidram_init_done (5 regs) */
	0x020b4060,		/* tcfc.INT_STS_0 .. tcfc.INT_MASK_0 (2 regs) */
	0x0c0b4100,		/* tcfc.lc_blocked .. tcfc.cdu_write_backs (12 regs) */
	0x050b4140,		/* tcfc.dbg_select .. tcfc.dbg_force_frame (5 regs) */
	0x090b4152,		/* tcfc.eco_reserved .. tcfc.arbiters_reg (9 regs) */
	0x060b4170,		/* tcfc.debug0 .. tcfc.cdu_pcie_err_mask (6 regs) */
	0x020b4177,		/* tcfc.sreq_full_sticky .. tcfc.prsresp_full_sticky (2 regs) */
	0x0b0b4180,		/* tcfc.num_lcids_empty .. tcfc.max_inside (11 regs) */
	0x020b41c4,		/* tcfc.LoadRetry_Types .. tcfc.MiniCache_Control (2 regs) */
	0x010b41c7,		/* tcfc.control0 (1 regs) */
	0x040b41e0,		/* tcfc.prsresp_credit .. tcfc.cduwb_credit (4 regs) */
	0x040b4200,		/* tcfc.ll_policy_cfg .. tcfc.empty_size (4 regs) */
	0x1f0b4240,		/* tcfc.lc_client_0_lcid_threshold .. tcfc.wave_sm_2_one_count (31 regs) */
	0x120b4280,		/* tcfc.cache_string_type .. tcfc.include_vlan_in_hash (18 regs) */
	0x080b42c0,		/* tcfc.cid_cam_bist_en .. tcfc.string_cam_bist_status (8 regs) */
	0x020b7400,		/* tcfc.vpf1_lstate_sel .. tcfc.vpf2_lstate_sel (2 regs) */
	0x084200eb,		/* mode (!k2)&(!(!asic)), block umac */
	0x02014403,		/* umac.mac_0 .. umac.mac_1 (2 regs) */
	0x01014406,		/* umac.pause_quant (1 regs) */
	0x01014410,		/* umac.sfd_offset (1 regs) */
	0x04014412,		/* umac.tag_0 .. umac.tx_preamble (4 regs) */
	0x05014418,		/* umac.pfc_xoff_timer .. umac.umac_eee_ref_count (5 regs) */
	0x0501441e,		/* umac.umac_rx_pkt_drop_status .. umac.umac_rev_id (5 regs) */
	0x070144c1,		/* umac.mac_pfc_opcode .. umac.tx_ts_data (7 regs) */
	0x050144cd,		/* umac.flush_control .. umac.mac_pfc_refresh_ctrl (5 regs) */
	0x05490000,		/* block ipc */
	0xa3008080,		/* ipc.mdio_voltage_sel .. ipc.swreg_sync_clk_en (163 regs) */
	0x03008126,		/* ipc.nw_serdes_mdio_comm .. ipc.nw_serdes_mdio_mode (3 regs) */
	0x0100812c,		/* ipc.freq_nw (1 regs) */
	0x0800813a,		/* ipc.otp_config_0 .. ipc.otp_config_7 (8 regs) */
	0x0500814b,		/* ipc.lcpll_refclk_sel .. ipc.cpu_otp_rd_syndrome (5 regs) */
	0x0449017d,		/* mode bb&(!(!asic)), block ipc */
	0x03008123,		/* ipc.mdio_comm .. ipc.mdio_mode (3 regs) */
	0x0200812a,		/* ipc.freq_main .. ipc.freq_storm (2 regs) */
	0x0d00812d,		/* ipc.free_running_cntr_0 .. ipc.hw_straps (13 regs) */
	0x03008142,		/* ipc.jtag_compliance .. ipc.INT_MASK_0 (3 regs) */
	0x644a0021,		/* mode (!bb)&(!(!(asic|emul_mac))), block nwm */
	0x02200001,		/* nwm.INT_STS .. nwm.INT_MASK (2 regs) */
	0x3b200005,		/* nwm.mac0_peer_delay .. nwm.dbg_force_frame (59 regs) */
	0x06200100,		/* nwm.mac0.REVISION .. nwm.mac0.FRM_LENGTH (6 regs) */
	0x08200107,		/* nwm.mac0.RX_FIFO_SECTIONS .. nwm.mac0.MDIO_DATA (8 regs) */
	0x0e200110,		/* nwm.mac0.STATUS .. nwm.mac0.RX_PAUSE_STATUS (14 regs) */
	0x0220011f,		/* nwm.mac0.TS_TIMESTAMP .. nwm.mac0.XIF_MODE (2 regs) */
	0x03200138,		/* nwm.mac0.STATN_CONFIG .. nwm.mac0.STATN_CLEARVALUE_HI (3 regs) */
	0x36200140,		/* nwm.mac0.etherStatsOctets .. nwm.mac0.aInRangeLengthError_h (54 regs) */
	0x04200180,		/* nwm.mac0.TXetherStatsOctets .. nwm.mac0.TXOctetsOK_h (4 regs) */
	0x24200186,		/* nwm.mac0.TXaPauseMacCtrlFrames .. nwm.mac0.TXetherStatsPkts1519toTX_MTU_h (36 regs) */
	0x022001b0,		/* nwm.mac0.TXaMACControlFrames .. nwm.mac0.TXaMACControlFrames_h (2 regs) */
	0x262001e0,		/* nwm.mac0.aCBFCPAUSEFramesReceived_0 .. nwm.mac1.FRM_LENGTH (38 regs) */
	0x08200207,		/* nwm.mac1.RX_FIFO_SECTIONS .. nwm.mac1.MDIO_DATA (8 regs) */
	0x0e200210,		/* nwm.mac1.STATUS .. nwm.mac1.RX_PAUSE_STATUS (14 regs) */
	0x0220021f,		/* nwm.mac1.TS_TIMESTAMP .. nwm.mac1.XIF_MODE (2 regs) */
	0x03200238,		/* nwm.mac1.STATN_CONFIG .. nwm.mac1.STATN_CLEARVALUE_HI (3 regs) */
	0x36200240,		/* nwm.mac1.etherStatsOctets .. nwm.mac1.aInRangeLengthError_h (54 regs) */
	0x04200280,		/* nwm.mac1.TXetherStatsOctets .. nwm.mac1.TXOctetsOK_h (4 regs) */
	0x24200286,		/* nwm.mac1.TXaPauseMacCtrlFrames .. nwm.mac1.TXetherStatsPkts1519toTX_MTU_h (36 regs) */
	0x022002b0,		/* nwm.mac1.TXaMACControlFrames .. nwm.mac1.TXaMACControlFrames_h (2 regs) */
	0x262002e0,		/* nwm.mac1.aCBFCPAUSEFramesReceived_0 .. nwm.mac2.FRM_LENGTH (38 regs) */
	0x08200307,		/* nwm.mac2.RX_FIFO_SECTIONS .. nwm.mac2.MDIO_DATA (8 regs) */
	0x0e200310,		/* nwm.mac2.STATUS .. nwm.mac2.RX_PAUSE_STATUS (14 regs) */
	0x0220031f,		/* nwm.mac2.TS_TIMESTAMP .. nwm.mac2.XIF_MODE (2 regs) */
	0x03200338,		/* nwm.mac2.STATN_CONFIG .. nwm.mac2.STATN_CLEARVALUE_HI (3 regs) */
	0x36200340,		/* nwm.mac2.etherStatsOctets .. nwm.mac2.aInRangeLengthError_h (54 regs) */
	0x04200380,		/* nwm.mac2.TXetherStatsOctets .. nwm.mac2.TXOctetsOK_h (4 regs) */
	0x24200386,		/* nwm.mac2.TXaPauseMacCtrlFrames .. nwm.mac2.TXetherStatsPkts1519toTX_MTU_h (36 regs) */
	0x022003b0,		/* nwm.mac2.TXaMACControlFrames .. nwm.mac2.TXaMACControlFrames_h (2 regs) */
	0x262003e0,		/* nwm.mac2.aCBFCPAUSEFramesReceived_0 .. nwm.mac3.FRM_LENGTH (38 regs) */
	0x08200407,		/* nwm.mac3.RX_FIFO_SECTIONS .. nwm.mac3.MDIO_DATA (8 regs) */
	0x0e200410,		/* nwm.mac3.STATUS .. nwm.mac3.RX_PAUSE_STATUS (14 regs) */
	0x0220041f,		/* nwm.mac3.TS_TIMESTAMP .. nwm.mac3.XIF_MODE (2 regs) */
	0x03200438,		/* nwm.mac3.STATN_CONFIG .. nwm.mac3.STATN_CLEARVALUE_HI (3 regs) */
	0x36200440,		/* nwm.mac3.etherStatsOctets .. nwm.mac3.aInRangeLengthError_h (54 regs) */
	0x04200480,		/* nwm.mac3.TXetherStatsOctets .. nwm.mac3.TXOctetsOK_h (4 regs) */
	0x24200486,		/* nwm.mac3.TXaPauseMacCtrlFrames .. nwm.mac3.TXetherStatsPkts1519toTX_MTU_h (36 regs) */
	0x022004b0,		/* nwm.mac3.TXaMACControlFrames .. nwm.mac3.TXaMACControlFrames_h (2 regs) */
	0x272004e0,		/* nwm.mac3.aCBFCPAUSEFramesReceived_0 .. nwm.pcs_reg91_0.RS_FEC_LANEMAP (39 regs) */
	0x0820050a,		/* nwm.pcs_reg91_0.RS_FEC_SYMBLERR0_LO .. nwm.pcs_reg91_0.RS_FEC_SYMBLERR3_HI (8 regs) */
	0x08200580,		/* nwm.pcs_reg91_0.RS_FEC_VENDOR_CONTROL .. nwm.pcs_reg91_0.RS_FEC_VENDOR_TX_TESTTRIGGER (8 regs) */
	0x07200600,		/* nwm.pcs_reg91_1.RS_FEC_CONTROL .. nwm.pcs_reg91_1.RS_FEC_LANEMAP (7 regs) */
	0x0820060a,		/* nwm.pcs_reg91_1.RS_FEC_SYMBLERR0_LO .. nwm.pcs_reg91_1.RS_FEC_SYMBLERR3_HI (8 regs) */
	0x08200680,		/* nwm.pcs_reg91_1.RS_FEC_VENDOR_CONTROL .. nwm.pcs_reg91_1.RS_FEC_VENDOR_TX_TESTTRIGGER (8 regs) */
	0x07200700,		/* nwm.pcs_reg91_2.RS_FEC_CONTROL .. nwm.pcs_reg91_2.RS_FEC_LANEMAP (7 regs) */
	0x0820070a,		/* nwm.pcs_reg91_2.RS_FEC_SYMBLERR0_LO .. nwm.pcs_reg91_2.RS_FEC_SYMBLERR3_HI (8 regs) */
	0x08200780,		/* nwm.pcs_reg91_2.RS_FEC_VENDOR_CONTROL .. nwm.pcs_reg91_2.RS_FEC_VENDOR_TX_TESTTRIGGER (8 regs) */
	0x07200800,		/* nwm.pcs_reg91_3.RS_FEC_CONTROL .. nwm.pcs_reg91_3.RS_FEC_LANEMAP (7 regs) */
	0x0820080a,		/* nwm.pcs_reg91_3.RS_FEC_SYMBLERR0_LO .. nwm.pcs_reg91_3.RS_FEC_SYMBLERR3_HI (8 regs) */
	0x08200880,		/* nwm.pcs_reg91_3.RS_FEC_VENDOR_CONTROL .. nwm.pcs_reg91_3.RS_FEC_VENDOR_TX_TESTTRIGGER (8 regs) */
	0x09200900,		/* nwm.pcs_ls0.CONTROL .. nwm.pcs_ls0.LP_NP_RX (9 regs) */
	0x06200910,		/* nwm.pcs_ls0.SCRATCH .. nwm.pcs_ls0.DECODE_ERRORS (6 regs) */
	0x09200920,		/* nwm.pcs_ls1.CONTROL .. nwm.pcs_ls1.LP_NP_RX (9 regs) */
	0x06200930,		/* nwm.pcs_ls1.SCRATCH .. nwm.pcs_ls1.DECODE_ERRORS (6 regs) */
	0x09200940,		/* nwm.pcs_ls2.CONTROL .. nwm.pcs_ls2.LP_NP_RX (9 regs) */
	0x06200950,		/* nwm.pcs_ls2.SCRATCH .. nwm.pcs_ls2.DECODE_ERRORS (6 regs) */
	0x09200960,		/* nwm.pcs_ls3.CONTROL .. nwm.pcs_ls3.LP_NP_RX (9 regs) */
	0x06200970,		/* nwm.pcs_ls3.SCRATCH .. nwm.pcs_ls3.DECODE_ERRORS (6 regs) */
	0x09210000,		/* nwm.pcs_hs0.CONTROL1 .. nwm.pcs_hs0.STATUS2 (9 regs) */
	0x0221000e,		/* nwm.pcs_hs0.PKG_ID0 .. nwm.pcs_hs0.PKG_ID1 (2 regs) */
	0x01210014,		/* nwm.pcs_hs0.EEE_CTRL_CAPABILITY (1 regs) */
	0x01210016,		/* nwm.pcs_hs0.WAKE_ERR_COUNTER (1 regs) */
	0x0e210020,		/* nwm.pcs_hs0.BASER_STATUS1 .. nwm.pcs_hs0.ERR_BLK_HIGH_ORDER_CNT (14 regs) */
	0x01210032,		/* nwm.pcs_hs0.MULTILANE_ALIGN_STAT1 (1 regs) */
	0x01210034,		/* nwm.pcs_hs0.MULTILANE_ALIGN_STAT3 (1 regs) */
	0x042100c8,		/* nwm.pcs_hs0.BIP_ERR_CNT_LANE0 .. nwm.pcs_hs0.BIP_ERR_CNT_LANE3 (4 regs) */
	0x04210190,		/* nwm.pcs_hs0.LANE0_MAPPING .. nwm.pcs_hs0.LANE3_MAPPING (4 regs) */
	0x05218000,		/* nwm.pcs_hs0.VENDOR_SCRATCH .. nwm.pcs_hs0.VENDOR_RXLAUI_CONFIG (5 regs) */
	0x09218008,		/* nwm.pcs_hs0.VENDOR_VL0_0 .. nwm.pcs_hs0.VENDOR_PCS_MODE (9 regs) */
	0x09220000,		/* nwm.pcs_hs1.CONTROL1 .. nwm.pcs_hs1.STATUS2 (9 regs) */
	0x0222000e,		/* nwm.pcs_hs1.PKG_ID0 .. nwm.pcs_hs1.PKG_ID1 (2 regs) */
	0x01220014,		/* nwm.pcs_hs1.EEE_CTRL_CAPABILITY (1 regs) */
	0x01220016,		/* nwm.pcs_hs1.WAKE_ERR_COUNTER (1 regs) */
	0x0e220020,		/* nwm.pcs_hs1.BASER_STATUS1 .. nwm.pcs_hs1.ERR_BLK_HIGH_ORDER_CNT (14 regs) */
	0x01220032,		/* nwm.pcs_hs1.MULTILANE_ALIGN_STAT1 (1 regs) */
	0x01220034,		/* nwm.pcs_hs1.MULTILANE_ALIGN_STAT3 (1 regs) */
	0x042200c8,		/* nwm.pcs_hs1.BIP_ERR_CNT_LANE0 .. nwm.pcs_hs1.BIP_ERR_CNT_LANE3 (4 regs) */
	0x04220190,		/* nwm.pcs_hs1.LANE0_MAPPING .. nwm.pcs_hs1.LANE3_MAPPING (4 regs) */
	0x05228000,		/* nwm.pcs_hs1.VENDOR_SCRATCH .. nwm.pcs_hs1.VENDOR_RXLAUI_CONFIG (5 regs) */
	0x09228008,		/* nwm.pcs_hs1.VENDOR_VL0_0 .. nwm.pcs_hs1.VENDOR_PCS_MODE (9 regs) */
	0x09230000,		/* nwm.pcs_hs2.CONTROL1 .. nwm.pcs_hs2.STATUS2 (9 regs) */
	0x0223000e,		/* nwm.pcs_hs2.PKG_ID0 .. nwm.pcs_hs2.PKG_ID1 (2 regs) */
	0x01230014,		/* nwm.pcs_hs2.EEE_CTRL_CAPABILITY (1 regs) */
	0x01230016,		/* nwm.pcs_hs2.WAKE_ERR_COUNTER (1 regs) */
	0x0e230020,		/* nwm.pcs_hs2.BASER_STATUS1 .. nwm.pcs_hs2.ERR_BLK_HIGH_ORDER_CNT (14 regs) */
	0x01230032,		/* nwm.pcs_hs2.MULTILANE_ALIGN_STAT1 (1 regs) */
	0x01230034,		/* nwm.pcs_hs2.MULTILANE_ALIGN_STAT3 (1 regs) */
	0x042300c8,		/* nwm.pcs_hs2.BIP_ERR_CNT_LANE0 .. nwm.pcs_hs2.BIP_ERR_CNT_LANE3 (4 regs) */
	0x04238000,		/* nwm.pcs_hs2.VENDOR_SCRATCH .. nwm.pcs_hs2.VENDOR_TXLANE_THRESH (4 regs) */
	0x09238008,		/* nwm.pcs_hs2.VENDOR_VL0_0 .. nwm.pcs_hs2.VENDOR_PCS_MODE (9 regs) */
	0x09240000,		/* nwm.pcs_hs3.CONTROL1 .. nwm.pcs_hs3.STATUS2 (9 regs) */
	0x0224000e,		/* nwm.pcs_hs3.PKG_ID0 .. nwm.pcs_hs3.PKG_ID1 (2 regs) */
	0x01240014,		/* nwm.pcs_hs3.EEE_CTRL_CAPABILITY (1 regs) */
	0x01240016,		/* nwm.pcs_hs3.WAKE_ERR_COUNTER (1 regs) */
	0x0e240020,		/* nwm.pcs_hs3.BASER_STATUS1 .. nwm.pcs_hs3.ERR_BLK_HIGH_ORDER_CNT (14 regs) */
	0x01240032,		/* nwm.pcs_hs3.MULTILANE_ALIGN_STAT1 (1 regs) */
	0x01240034,		/* nwm.pcs_hs3.MULTILANE_ALIGN_STAT3 (1 regs) */
	0x042400c8,		/* nwm.pcs_hs3.BIP_ERR_CNT_LANE0 .. nwm.pcs_hs3.BIP_ERR_CNT_LANE3 (4 regs) */
	0x04248000,		/* nwm.pcs_hs3.VENDOR_SCRATCH .. nwm.pcs_hs3.VENDOR_TXLANE_THRESH (4 regs) */
	0x09248008,		/* nwm.pcs_hs3.VENDOR_VL0_0 .. nwm.pcs_hs3.VENDOR_PCS_MODE (9 regs) */
	0x01000077,		/* split PORT */
	0x02200000,		/* block dorq */
	0x0104013f,		/* dorq.wake_misc_en (1 regs) */
	0x08040221,		/* dorq.tag1_ethertype .. dorq.tag4_size (8 regs) */
	0x0f230000,		/* block prs */
	0x0107c11d,		/* prs.t_tag_tagnum (1 regs) */
	0x5607c144,		/* prs.ets_packet_additional_network_size .. prs.wfq_port_arb_current_credit (86 regs) */
	0x0107c1c0,		/* prs.prop_hdr_size (1 regs) */
	0x0107c1cc,		/* prs.encapsulation_type_en (1 regs) */
	0x0107c1ce,		/* prs.vxlan_port (1 regs) */
	0x1007c1df,		/* prs.first_hdr_hdrs_after_basic .. prs.inner_hdr_must_have_hdrs (16 regs) */
	0x2007c1fa,		/* prs.src_mac_0_0 .. prs.src_mac_15_1 (32 regs) */
	0x0107c21b,		/* prs.nge_port (1 regs) */
	0x0207c21d,		/* prs.rroce_enable .. prs.nge_comp_ver (2 regs) */
	0x0107c250,		/* prs.no_match_pfid (1 regs) */
	0x0307c284,		/* prs.classify_failed_pkt_len_stat_add_crc .. prs.classify_failed_pkt_len_stat_tags_not_counted_first (3 regs) */
	0x1307c288,		/* prs.nig_classify_failed .. prs.ignore_udp_zero_checksum (19 regs) */
	0x0907c2c2,		/* prs.num_of_packets_0 .. prs.num_of_packets_8 (9 regs) */
	0x0207c2d5,		/* prs.queue_pkt_avail_status .. prs.storm_bkprs_status (2 regs) */
	0x0107c3cd,		/* prs.eop_req_ct (1 regs) */
	0x063a0000,		/* block pbf */
	0x1236012c,		/* pbf.first_hdr_hdrs_after_basic .. pbf.inner_hdr_must_have_hdrs (18 regs) */
	0x02360146,		/* pbf.vxlan_port .. pbf.nge_port (2 regs) */
	0x01360149,		/* pbf.nge_comp_ver (1 regs) */
	0x01360160,		/* pbf.prop_hdr_size (1 regs) */
	0x01360162,		/* pbf.t_tag_tagnum (1 regs) */
	0x05360170,		/* pbf.btb_shared_area_size .. pbf.num_strict_priority_slots (5 regs) */
	0x31460000,		/* block nig */
	0x05140384,		/* nig.mac_in_en .. nig.flowctrl_out_en (5 regs) */
	0x01140390,		/* nig.tx_macfifo_alm_full_thr (1 regs) */
	0x02140393,		/* nig.tx_macfifo_alm_full .. nig.tx_macfifo_empty (2 regs) */
	0x011403c0,		/* nig.tx_macfifo_full (1 regs) */
	0x01140402,		/* nig.rx_pkt_has_fcs (1 regs) */
	0x03140404,		/* nig.llc_jumbo_type .. nig.first_hdr_hdrs_after_basic (3 regs) */
	0x07140408,		/* nig.first_hdr_hdrs_after_tag_0 .. nig.inner_hdr_hdrs_after_basic (7 regs) */
	0x08140410,		/* nig.inner_hdr_hdrs_after_tag_0 .. nig.vxlan_ctrl (8 regs) */
	0x2c140422,		/* nig.llh_dest_mac_0_0 .. nig.rx_llh_svol_mcp_mask (44 regs) */
	0x04140450,		/* nig.rx_llh_ncsi_brb_dntfwd_mask .. nig.rx_llh_svol_brb_dntfwd_mask (4 regs) */
	0x26140456,		/* nig.l2filt_ethertype0 .. nig.rx_llh_brb_gate_dntfwd (38 regs) */
	0x0514047e,		/* nig.storm_ethertype0 .. nig.rx_llh_storm_mask (5 regs) */
	0x061404c2,		/* nig.rx_llh_dfifo_empty .. nig.rx_llh_rfifo_full (6 regs) */
	0x01140500,		/* nig.storm_status (1 regs) */
	0x33140540,		/* nig.lb_min_cyc_threshold .. nig.lb_arb_num_strict_arb_slots (51 regs) */
	0x02940574,		/* nig.lb_arb_priority_client (2 regs, WB) */
	0x22140576,		/* nig.lb_arb_burst_mode .. nig.lb_llh_brb_gate_dntfwd (34 regs) */
	0x0f1405d9,		/* nig.lb_btb_fifo_alm_full_thr .. nig.lb_llh_rfifo_alm_full (15 regs) */
	0x01140600,		/* nig.lb_llh_rfifo_full (1 regs) */
	0x29140640,		/* nig.rx_ptp_en .. nig.outer_tag_value_mask (41 regs) */
	0x2d1406e8,		/* nig.flowctrl_mode .. nig.rx_flowctrl_status_clear (45 regs) */
	0x08140719,		/* nig.stat_rx_brb_packet_priority_0 .. nig.stat_rx_brb_packet_priority_7 (8 regs) */
	0x22140728,		/* nig.stat_rx_brb_octet_priority_0 .. nig.stat_rx_brb_discard_priority_7 (34 regs) */
	0x0214074b,		/* nig.stat_rx_storm_packet_discard .. nig.stat_rx_storm_packet_truncate (2 regs) */
	0x46140750,		/* nig.stat_lb_brb_packet_priority_0 .. nig.stat_tx_octet_tc_7 (70 regs) */
	0x20940796,		/* nig.tx_xoff_cyc_tc_0 .. nig.lb_xoff_cyc_tc_7 (32 regs, WB) */
	0x1a1407b6,		/* nig.stat_rx_bmb_octet .. nig.tx_arb_num_strict_arb_slots (26 regs) */
	0x029407d0,		/* nig.tx_arb_priority_client (2 regs, WB) */
	0x3d1407d2,		/* nig.tx_arb_burst_mode .. nig.tx_gnt_fifo_full (61 regs) */
	0x02140825,		/* nig.mng_tc .. nig.tx_mng_tc_en (2 regs) */
	0x03140828,		/* nig.tx_mng_timestamp_pkt .. nig.bmb_pkt_len (3 regs) */
	0x0414082c,		/* nig.tx_bmb_fifo_empty .. nig.lb_bmb_fifo_full (4 regs) */
	0x01140835,		/* nig.debug_pkt_len (1 regs) */
	0x05140850,		/* nig.dbg_select .. nig.dbg_force_frame (5 regs) */
	0x0f140862,		/* nig.rx_fc_dbg_select .. nig.lb_fc_dbg_force_frame (15 regs) */
	0x01140881,		/* nig.eco_reserved_perport (1 regs) */
	0x0214220a,		/* nig.tx_ptp_one_stp_en .. nig.rx_ptp_one_stp_en (2 regs) */
	0x01142214,		/* nig.add_freecnt_offset (1 regs) */
	0x0214222c,		/* nig.tsgen_offset_value_lsb .. nig.tsgen_offset_value_msb (2 regs) */
	0x04142230,		/* nig.tsgen_sync_time_lsb .. nig.tsgen_sw_pps_en (4 regs) */
	0x02142236,		/* nig.tsgen_rst_drift_cntr .. nig.tsgen_drift_cntr_conf (2 regs) */
	0x0214223c,		/* nig.roce_duplicate_to_host .. nig.default_engine_id_sel (2 regs) */
	0x43142280,		/* nig.dscp_to_tc_map .. nig.rroce_zero_udp_ignore (67 regs) */
	0x011422c8,		/* nig.add_eth_crc (1 regs) */
	0x031422ca,		/* nig.nge_ip_enable .. nig.nge_comp_ver (3 regs) */
	0x021422ce,		/* nig.nge_port .. nig.llh_lb_tc_remap (2 regs) */
	0x051422d2,		/* nig.rx_llh_ncsi_mcp_mask_2 .. nig.tx_llh_ncsi_ntwk_mask_2 (5 regs) */
	0x071422d8,		/* nig.timer_counter .. nig.tx_parity_error_close_egress (7 regs) */
	0x031422e0,		/* nig.tx_arb_client_0_map .. nig.lb_arb_client_0_map (3 regs) */
	0x04460023,		/* mode !bb, block nig */
	0x02142407,		/* nig.tsgen_tsio_in_sel_mask .. nig.tsgen_tsio_in_sel_pol (2 regs) */
	0x0494240a,		/* nig.tsgen_tsio_in_latched_value .. nig.tsgen_tsio_out_next_toggle_time (4 regs, WB) */
	0x0514240e,		/* nig.tsgen_pps_start_time_0 .. nig.ptp_update_sw_osts_pkt_time (5 regs) */
	0x04142416,		/* nig.llh_dest_mac_6_0 .. nig.llh_dest_mac_7_1 (4 regs) */
	0x02480000,		/* block bmbn */
	0x06184078,		/* bmbn.mng_outer_tag0_0 .. bmbn.mng_inner_vlan_tag1 (6 regs) */
	0x01184081,		/* bmbn.eco_reserved_perport (1 regs) */
	0x014e0000,		/* block led */
	0x061ae000,		/* led.control .. led.mac_led_speed (6 regs) */
	0x01050081,		/* mode !k2, block cnig */
	0x0608608f,		/* cnig.led_control .. cnig.mac_led_speed (6 regs) */
	0x14460000,		/* block nig */
	0x02140391,		/* nig.rx_macfifo_empty .. nig.rx_macfifo_full (2 regs) */
	0x01140400,		/* nig.hdr_skip_size (1 regs) */
	0x01140407,		/* nig.first_hdr_hdrs_after_llc (1 regs) */
	0x0114040f,		/* nig.inner_hdr_hdrs_after_llc (1 regs) */
	0x03140418,		/* nig.ipv4_type .. nig.fcoe_type (3 regs) */
	0x0414041d,		/* nig.tcp_protocol .. nig.icmpv6_protocol (4 regs) */
	0x1614080f,		/* nig.mng_outer_tag0_0 .. nig.mng_prop_hdr1_7 (22 regs) */
	0x01140827,		/* nig.tx_host_mng_enable (1 regs) */
	0x02142002,		/* nig.mf_global_en .. nig.upon_mgmt (2 regs) */
	0x04142070,		/* nig.wake_buffer_clear .. nig.wake_details (4 regs) */
	0x02142208,		/* nig.tx_up_ts_en .. nig.rx_up_ts_en (2 regs) */
	0x0814220c,		/* nig.tx_up_ts_addr_0 .. nig.rx_enable_up_rules (8 regs) */
	0x01142215,		/* nig.up_ts_insert_en (1 regs) */
	0x0114221f,		/* nig.llh_up_buf_seqid (1 regs) */
	0x04942220,		/* nig.llh_up_buf_timestamp .. nig.llh_up_buf_src_addr (4 regs, WB) */
	0x01142224,		/* nig.tx_llh_up_buf_seqid (1 regs) */
	0x04942226,		/* nig.tx_llh_up_buf_timestamp .. nig.llh_up_buf_dst_addr (4 regs, WB) */
	0x0114223b,		/* nig.mld_msg_type (1 regs) */
	0x041422c4,		/* nig.acpi_tag_remove .. nig.rm_eth_crc (4 regs) */
	0x011422c9,		/* nig.corrupt_eth_crc (1 regs) */
	0x0306006f,		/* mode !(emul_reduced|fpga), block cpmu */
	0x0700c080,		/* cpmu.lpi_mode_config .. cpmu.sw_force_lpi (7 regs) */
	0x0600c0c9,		/* cpmu.lpi_tx_req_stat_ro .. cpmu.lpi_duration_stat_ro (6 regs) */
	0x0600c0df,		/* cpmu.lpi_tx_req_stat .. cpmu.lpi_duration_stat (6 regs) */
	0x054700d3,		/* mode (!bb)&(!(!asic)), block wol */
	0x02182000,		/* wol.acpi_tag_rm .. wol.upon_mgmt (2 regs) */
	0x08982058,		/* wol.wake_buffer (8 regs, WB) */
	0x06182060,		/* wol.wake_buffer_clear .. wol.acpi_pat_sel (6 regs) */
	0x0118206d,		/* wol.wake_mem_rd_offset (1 regs) */
	0x0418206f,		/* wol.eco_reserved_perport .. wol.hdr_fifo_error (4 regs) */
	0x020000ce,		/* split PF */
	0x01010000,		/* block miscs */
	0x010025d7,		/* miscs.unprepared_dr (1 regs) */
	0x05040000,		/* block pglue_b */
	0x040aa136,		/* pglue_b.pseudo_vf_master_enable .. pglue_b.vf_base (4 regs) */
	0x030aa85b,		/* pglue_b.internal_pfid_enable_master .. pglue_b.internal_pfid_enable_target_read (3 regs) */
	0x010aa950,		/* pglue_b.pf_trusted (1 regs) */
	0x010aa965,		/* pglue_b.mask_block_discard_attn_pf (1 regs) */
	0x040aab97,		/* pglue_b.config_reg_78 .. pglue_b.vf_bar1_size (4 regs) */
	0x03140000,		/* block pswrq2 */
	0x1b090003,		/* pswrq2.cdut_p_size .. pswrq2.dbg_last_ilt (27 regs) */
	0x0b090024,		/* pswrq2.tm_number_of_pf_blocks .. pswrq2.vf_last_ilt (11 regs) */
	0x010901fc,		/* pswrq2.atc_internal_ats_enable (1 regs) */
	0x01170000,		/* block ptu */
	0x0515801e,		/* ptu.inv_tid .. ptu.inv_halt_on_err (5 regs) */
	0x01180000,		/* block tcm */
	0x04460241,		/* tcm.con_phy_q0 .. tcm.task_phy_q1 (4 regs) */
	0x01190000,		/* block mcm */
	0x04480241,		/* mcm.con_phy_q0 .. mcm.task_phy_q1 (4 regs) */
	0x011a0000,		/* block ucm */
	0x044a0241,		/* ucm.con_phy_q0 .. ucm.task_phy_q1 (4 regs) */
	0x011b0000,		/* block xcm */
	0x01400241,		/* xcm.con_phy_q3 (1 regs) */
	0x011c0000,		/* block ycm */
	0x04420241,		/* ycm.con_phy_q0 .. ycm.task_phy_q1 (4 regs) */
	0x041e0000,		/* block qm */
	0x030bc10d,		/* qm.MaxPqSize_0 .. qm.MaxPqSize_2 (3 regs) */
	0x010bc54c,		/* qm.PciReqTph (1 regs) */
	0x020bcba8,		/* qm.pci_rd_err .. qm.pf_en (2 regs) */
	0x020bcbab,		/* qm.usg_cnt_pf_tx .. qm.usg_cnt_pf_other (2 regs) */
	0x06200000,		/* block dorq */
	0x18040100,		/* dorq.pf_min_addr_reg1 .. dorq.vf_min_val_dpi (24 regs) */
	0x03040140,		/* dorq.pf_net_port_id .. dorq.pf_db_enable (3 regs) */
	0x02040144,		/* dorq.pf_dpm_enable .. dorq.vf_dpm_enable (2 regs) */
	0x02040231,		/* dorq.pf_pcp .. dorq.pf_ext_vid (2 regs) */
	0x01040270,		/* dorq.pf_usage_cnt (1 regs) */
	0x03040272,		/* dorq.pf_usage_cnt_lim .. dorq.pf_ovfl_sticky (3 regs) */
	0x11230000,		/* block prs */
	0x0407c05a,		/* prs.task_id_max_initiator_pf .. prs.task_id_max_target_vf (4 regs) */
	0x0107c064,		/* prs.roce_separate_rx_tx_cid_flg (1 regs) */
	0x0107c066,		/* prs.load_l2_filter (1 regs) */
	0x0207c068,		/* prs.target_initiator_select .. prs.fcoe_search_with_exchange_context (2 regs) */
	0x0407c100,		/* prs.search_tcp .. prs.search_roce (4 regs) */
	0x0507c105,		/* prs.tcp_search_key_mask .. prs.roce_build_cid_wo_search (5 regs) */
	0x0907c10b,		/* prs.roce_dest_qp_max_vf .. prs.search_tenant_id (9 regs) */
	0x0207c1cf,		/* prs.roce_icid_base_pf .. prs.roce_icid_base_vf (2 regs) */
	0x0507c1f5,		/* prs.first_hdr_dst_ip_0 .. prs.first_hdr_dst_ip_4 (5 regs) */
	0x0307c251,		/* prs.override_pfid_if_no_match .. prs.no_match_lcid (3 regs) */
	0x0207c25a,		/* prs.light_l2_ethertype_en .. prs.use_light_l2 (2 regs) */
	0x0107c26f,		/* prs.mac_vlan_cache_use_tenant_id (1 regs) */
	0x0107c274,		/* prs.sort_sack (1 regs) */
	0x0107c276,		/* prs.rdma_syn_mask (1 regs) */
	0x0507c27f,		/* prs.rdma_syn_cookie_en .. prs.pkt_len_stat_tags_not_counted_first (5 regs) */
	0x0107c287,		/* prs.msg_info (1 regs) */
	0x0207c46f,		/* prs.search_gft .. prs.search_non_ip_as_gft (2 regs) */
	0x013d0000,		/* block cdu */
	0x0a160243,		/* cdu.pf_seg0_type_offset .. cdu.vf_fl_seg_type_offset (10 regs) */
	0x063e0000,		/* block ccfc */
	0x010b8176,		/* ccfc.robustwb_pf (1 regs) */
	0x010b81c0,		/* ccfc.weak_enable_pf (1 regs) */
	0x010b81c2,		/* ccfc.strong_enable_pf (1 regs) */
	0x010b81c6,		/* ccfc.pf_minicache_enable (1 regs) */
	0x010bb403,		/* ccfc.pf_lstate_cnt1 (1 regs) */
	0x010bb405,		/* ccfc.pf_lstate_cnt2 (1 regs) */
	0x08400000,		/* block igu */
	0x02060100,		/* igu.statistic_num_pf_msg_sent (2 regs) */
	0x0106010b,		/* igu.statistic_num_of_inta_asserted (1 regs) */
	0x01060200,		/* igu.pf_configuration (1 regs) */
	0x0a060208,		/* igu.attn_msg_addr_l .. igu.command_reg_32msb_data (10 regs) */
	0x05060328,		/* igu.int_before_mask_sts_pf (5 regs) */
	0x05060338,		/* igu.int_mask_sts_pf (5 regs) */
	0x05060348,		/* igu.pba_sts_pf (5 regs) */
	0x01060486,		/* igu.attn_tph (1 regs) */
	0x01410000,		/* block cau */
	0x01070106,		/* cau.cleanup_command_done (1 regs) */
	0x09460000,		/* block nig */
	0x01140300,		/* nig.tx_lb_pf_drop_perpf (1 regs) */
	0x0114044f,		/* nig.rx_llh_svol_mcp_fwd_perpf (1 regs) */
	0x01140455,		/* nig.rx_llh_svol_brb_dntfwd_perpf (1 regs) */
	0x0114047d,		/* nig.rx_llh_brb_gate_dntfwd_perpf (1 regs) */
	0x01140599,		/* nig.lb_llh_brb_gate_dntfwd_perpf (1 regs) */
	0x011406e7,		/* nig.llh_eng_cls_eng_id_perpf (1 regs) */
	0x07142051,		/* nig.tcp_syn_enable .. nig.tcp_syn_ipv4_dst_port (7 regs) */
	0x08942058,		/* nig.tcp_syn_ipv6_src_addr .. nig.tcp_syn_ipv6_dst_addr (8 regs, WB) */
	0x02142060,		/* nig.tcp_syn_ipv4_src_addr .. nig.tcp_syn_ipv4_dst_addr (2 regs) */
	0x02040023,		/* mode !bb, block pglue_b */
	0x020aabad,		/* pglue_b.vf_bar0_size .. pglue_b.pf_rom_size (2 regs) */
	0x040aabe6,		/* pglue_b.pgl_addr_e8_f0 .. pglue_b.pgl_addr_f4_f0 (4 regs) */
	0x1c150000,		/* block pglcs */
	0x0e000800,		/* pglcs.pgl_cs.DEVICE_ID_VENDOR_ID_REG .. pglcs.pgl_cs.PCI_CAP_PTR_REG (14 regs) */
	0x0300080f,		/* pglcs.pgl_cs.MAX_LATENCY_MIN_GRANT_INTERRUPT_PIN_INTERRUPT_LINE_REG .. pglcs.pgl_cs.CON_STATUS_REG (3 regs) */
	0x06000814,		/* pglcs.pgl_cs.PCI_MSI_CAP_ID_NEXT_CTRL_REG .. pglcs.pgl_cs.MSI_CAP_OFF_14H_REG (6 regs) */
	0x0500081c,		/* pglcs.pgl_cs.PCIE_CAP_ID_PCIE_NEXT_CAP_PTR_PCIE_CAP_REG .. pglcs.pgl_cs.LINK_CONTROL_LINK_STATUS_REG (5 regs) */
	0x04000825,		/* pglcs.pgl_cs.DEVICE_CAPABILITIES2_REG .. pglcs.pgl_cs.LINK_CONTROL2_LINK_STATUS2_REG (4 regs) */
	0x0300082c,		/* pglcs.pgl_cs.PCI_MSIX_CAP_ID_NEXT_CTRL_REG .. pglcs.pgl_cs.MSIX_PBA_OFFSET_REG (3 regs) */
	0x02000834,		/* pglcs.pgl_cs.VPD_BASE .. pglcs.pgl_cs.DATA_REG (2 regs) */
	0x0b000840,		/* pglcs.pgl_cs.AER_EXT_CAP_HDR_OFF .. pglcs.pgl_cs.HDR_LOG_3_OFF (11 regs) */
	0x0400084e,		/* pglcs.pgl_cs.TLP_PREFIX_LOG_1_OFF .. pglcs.pgl_cs.TLP_PREFIX_LOG_4_OFF (4 regs) */
	0x0300085a,		/* pglcs.pgl_cs.SN_BASE .. pglcs.pgl_cs.SER_NUM_REG_DW_2 (3 regs) */
	0x0600085e,		/* pglcs.pgl_cs.PB_BASE .. pglcs.pgl_cs.CAP_REG (6 regs) */
	0x1400086e,		/* pglcs.pgl_cs.SRIOV_BASE_REG .. pglcs.pgl_cs.TPH_ST_TABLE_REG_0 (20 regs) */
	0x060008a3,		/* pglcs.pgl_cs.RAS_DES_CAP_HEADER_REG .. pglcs.pgl_cs.TIME_BASED_ANALYSIS_DATA_REG (6 regs) */
	0x180008af,		/* pglcs.pgl_cs.EINJ_ENABLE_REG .. pglcs.pgl_cs.EINJ6_TLP_REG (24 regs) */
	0x020008cb,		/* pglcs.pgl_cs.SD_CONTROL1_REG .. pglcs.pgl_cs.SD_CONTROL2_REG (2 regs) */
	0x060008cf,		/* pglcs.pgl_cs.SD_STATUS_L1LANE_REG .. pglcs.pgl_cs.SD_STATUS_L3_REG (6 regs) */
	0x030008d7,		/* pglcs.pgl_cs.SD_EQ_CONTROL1_REG .. pglcs.pgl_cs.SD_EQ_CONTROL3_REG (3 regs) */
	0x030008db,		/* pglcs.pgl_cs.SD_EQ_STATUS1_REG .. pglcs.pgl_cs.SD_EQ_STATUS3_REG (3 regs) */
	0x0e0008e3,		/* pglcs.pgl_cs.RASDP_EXT_CAP_HDR_OFF .. pglcs.pgl_cs.RASDP_RAM_ADDR_UNCORR_ERROR_OFF (14 regs) */
	0x180008f4,		/* pglcs.pgl_cs.PTM_REQ_CAP_HDR_OFF .. pglcs.pgl_cs.RESBAR_CTRL_REG_0_REG (24 regs) */
	0x01000d05,		/* pglcs.pgl_cs_shadow.BAR1_MASK_REG (1 regs) */
	0x01000d0c,		/* pglcs.pgl_cs_shadow.EXP_ROM_BAR_MASK_REG (1 regs) */
	0x01000d71,		/* pglcs.pgl_cs_shadow.SHADOW_SRIOV_INITIAL_VFS (1 regs) */
	0x01000d73,		/* pglcs.pgl_cs_shadow.SHADOW_SRIOV_VF_OFFSET_POSITION (1 regs) */
	0x01000d78,		/* pglcs.pgl_cs_shadow.SRIOV_BAR1_MASK_REG (1 regs) */
	0x01000d7a,		/* pglcs.pgl_cs_shadow.SRIOV_BAR3_MASK_REG (1 regs) */
	0x01000d7c,		/* pglcs.pgl_cs_shadow.SRIOV_BAR5_MASK_REG (1 regs) */
	0x01000e00,		/* pglcs.first_vf (1 regs) */
	0x02040081,		/* mode !k2, block pglue_b */
	0x010aa839,		/* pglue_b.shadow_ats_stu (1 regs) */
	0x040aa901,		/* pglue_b.pgl_addr_88_f0 .. pglue_b.pgl_addr_94_f0 (4 regs) */
	0x3e150000,		/* block pglcs */
	0x0e000800,		/* pglcs.pgl_cs.device_vendor_id .. pglcs.pgl_cs.cap_pointer (14 regs) */
	0x0100080f,		/* pglcs.pgl_cs.lat_min_grant_int_pin_int_line (1 regs) */
	0x08000812,		/* pglcs.pgl_cs.pm_cap .. pglcs.pgl_cs.msi_data (8 regs) */
	0x12000828,		/* pglcs.pgl_cs.msix_cap .. pglcs.pgl_cs.slot_status_control_2 (18 regs) */
	0x0e000840,		/* pglcs.pgl_cs.adv_err_cap .. pglcs.pgl_cs.root_err_id (14 regs) */
	0x0300084f,		/* pglcs.pgl_cs.device_ser_num_cap .. pglcs.pgl_cs.upper_ser_num (3 regs) */
	0x0b000854,		/* pglcs.pgl_cs.pwr_bdgt_cap .. pglcs.pgl_cs.vc_rsrc_status (11 regs) */
	0x07000860,		/* pglcs.pgl_cs.vendor_cap .. pglcs.pgl_cs.vendor_specific_reg5 (7 regs) */
	0x1300086c,		/* pglcs.pgl_cs.LTR_cap .. pglcs.pgl_cs.VF_BAR5 (19 regs) */
	0x03000880,		/* pglcs.pgl_cs.PTM_extended_cap .. pglcs.pgl_cs.ptm_ctrl_reg (3 regs) */
	0x02000884,		/* pglcs.pgl_cs.ATS_cap .. pglcs.pgl_cs.ATS_control (2 regs) */
	0x03000888,		/* pglcs.pgl_cs.RBAR_ext_cap .. pglcs.pgl_cs.RBAR_CTRL (3 regs) */
	0x0300088c,		/* pglcs.pgl_cs.TPH_extended_cap .. pglcs.pgl_cs.tph_req_control (3 regs) */
	0x04000890,		/* pglcs.pgl_cs.PML1sub_capID .. pglcs.pgl_cs.PML1_sub_control2 (4 regs) */
	0x0b0008c0,		/* pglcs.pgl_cs.Secondary_PCIE_Extended_Cap .. pglcs.pgl_cs.Lane14_15_equalization_ctrl (11 regs) */
	0x02000902,		/* pglcs.pgl_cs.config_2 .. pglcs.pgl_cs.config_3 (2 regs) */
	0x01000906,		/* pglcs.pgl_cs.pci_extended_bar_siz (1 regs) */
	0x0400090a,		/* pglcs.pgl_cs.reg_vpd_intf .. pglcs.pgl_cs.reg_id_val1 (4 regs) */
	0x02000910,		/* pglcs.pgl_cs.reg_id_val4 .. pglcs.pgl_cs.reg_id_val5 (2 regs) */
	0x04000913,		/* pglcs.pgl_cs.reg_id_val6 .. pglcs.pgl_cs.reg_msi_addr_l (4 regs) */
	0x03000919,		/* pglcs.pgl_cs.reg_msi_mask .. pglcs.pgl_cs.reg_pm_data_c (3 regs) */
	0x02000930,		/* pglcs.pgl_cs.reg_msix_control .. pglcs.pgl_cs.reg_msix_tbl_off_bir (2 regs) */
	0x02000935,		/* pglcs.pgl_cs.reg_device_capability .. pglcs.pgl_cs.reg_device_control (2 regs) */
	0x02000938,		/* pglcs.pgl_cs.reg_bar2_config .. pglcs.pgl_cs.reg_pcie_device_capability_2 (2 regs) */
	0x0300093d,		/* pglcs.pgl_cs.reg_bar3_config .. pglcs.pgl_cs.reg_root_control (3 regs) */
	0x0a000944,		/* pglcs.pgl_cs.reg_pwr_bdgt_data_0 .. pglcs.pgl_cs.reg_pwr_bdgt_data_8 (10 regs) */
	0x02000950,		/* pglcs.pgl_cs.reg_l1sub_cap .. pglcs.pgl_cs.reg_l1sub_ext_cap (2 regs) */
	0x01000954,		/* pglcs.pgl_cs.reg_pwr_bdgt_capability (1 regs) */
	0x04000956,		/* pglcs.pgl_cs.reg_rc_user_mem_lo1 .. pglcs.pgl_cs.reg_rc_user_mem_hi2 (4 regs) */
	0x0100097b,		/* pglcs.pgl_cs.reg_PCIER_MC_WINDOW_SIZE_REQ (1 regs) */
	0x0600097d,		/* pglcs.pgl_cs.reg_TPH_CAP .. pglcs.pgl_cs.reg_VF_BAR_REG (6 regs) */
	0x01000984,		/* pglcs.pgl_cs.reg_VF_CAP_EN (1 regs) */
	0x03000987,		/* pglcs.pgl_cs.reg_vf_msix_control .. pglcs.pgl_cs.reg_pf_initvf (3 regs) */
	0x0200098c,		/* pglcs.pgl_cs.reg_ats_inld_queue_depth .. pglcs.pgl_cs.reg_VFTPH_CAP (2 regs) */
	0x0e000a00,		/* pglcs.pgl_cs.tl_control_0 .. pglcs.pgl_cs.user_control_8 (14 regs) */
	0x0c000a0f,		/* pglcs.pgl_cs.tl_control_6 .. pglcs.pgl_cs.tl_rst_ctrl (12 regs) */
	0x0a000a1c,		/* pglcs.pgl_cs.tl_obff_ctrl .. pglcs.pgl_cs.tl_func14to15_stat (10 regs) */
	0x29000a40,		/* pglcs.pgl_cs.tl_status_0 .. pglcs.pgl_cs.tl_rst_debug (41 regs) */
	0x01000a81,		/* pglcs.pgl_cs.tl_iov_vfctl_0 (1 regs) */
	0x02000a84,		/* pglcs.pgl_cs.tl_fcimm_np_limit .. pglcs.pgl_cs.tl_fcimm_p_limit (2 regs) */
	0x0a000a87,		/* pglcs.pgl_cs.reg_capena_fn0_mask .. pglcs.pgl_cs.ptm_mstr_prop_dly (10 regs) */
	0x0a000a94,		/* pglcs.pgl_cs.PCIER_TL_STAT_TX_CTL .. pglcs.pgl_cs.PCIER_TL_STAT_RX_CTR_HI (10 regs) */
	0x05000b00,		/* pglcs.pgl_cs.PCIER_DBG_FIFO_CTLSTAT .. pglcs.pgl_cs.PCIER_TLPL_DBG_FIFO_CTL (5 regs) */
	0x1a000b06,		/* pglcs.pgl_cs.PCIER_DBG_FIFO_RD_9 .. pglcs.pgl_cs.PCIER_TLDA1_RDFIFO_0 (26 regs) */
	0x0f000c00,		/* pglcs.pgl_cs.pdl_control_0 .. pglcs.pgl_cs.pdl_control_14 (15 regs) */
	0x08000c10,		/* pglcs.pgl_cs.DLATTN_VEC .. pglcs.pgl_cs.dl_spare0 (8 regs) */
	0x09000c40,		/* pglcs.pgl_cs.mdio_addr .. pglcs.pgl_cs.ate_tlp_ctl (9 regs) */
	0x03000c4c,		/* pglcs.pgl_cs.serdes_pmi_addr .. pglcs.pgl_cs.serdes_pmi_rdata (3 regs) */
	0x14000d00,		/* pglcs.pgl_cs.dl_dbg_0 .. pglcs.pgl_cs.dl_dbg_19 (20 regs) */
	0x0a000e00,		/* pglcs.pgl_cs.reg_phy_ctl_0 .. pglcs.pgl_cs.phy_err_attn_mask (10 regs) */
	0x08000e0c,		/* pglcs.pgl_cs.reg_phy_ctl_8 .. pglcs.pgl_cs.reg_phy_ctl_15 (8 regs) */
	0x04000e15,		/* pglcs.pgl_cs.reg_phy_ctl_16 .. pglcs.pgl_cs.pl_gen3_ena_frmerr (4 regs) */
	0x05000e40,		/* pglcs.pgl_cs.pl_lpbk_master_ctl0 .. pglcs.pgl_cs.pl_lpbk_master_tx_setting (5 regs) */
	0x01000e4c,		/* pglcs.pgl_cs.pl_sw_ltssm_ctl (1 regs) */
	0x10000e50,		/* pglcs.pgl_cs.pcie_statis_ctl .. pglcs.pgl_cs.pcie_rxtlperr_statis (16 regs) */
	0x06000e68,		/* pglcs.pgl_cs.ltssm_statis_ctl .. pglcs.pgl_cs.ltssm_statis_cnt (6 regs) */
	0x09000f00,		/* pglcs.pgl_cs.Received_MCP_Errors_1512 .. pglcs.pgl_cs.rx_fts_limit (9 regs) */
	0x06000f34,		/* pglcs.pgl_cs.fts_hist .. pglcs.pgl_cs.recovery_hist_1 (6 regs) */
	0x28000f3b,		/* pglcs.pgl_cs.phy_ltssm_hist_0 .. pglcs.pgl_cs.phy_dbg_sed_extcfg_74 (40 regs) */
	0x01000f64,		/* pglcs.pgl_cs.phy_dbg_preset_lut (1 regs) */
	0x01000f80,		/* pglcs.pgl_cs.phy_dbg_muxed_sigs (1 regs) */
	0x05000f84,		/* pglcs.pgl_cs.phy_dbg_clkreq_0 .. pglcs.pgl_cs.misc_dbg_status (5 regs) */
	0x01400000,		/* block igu */
	0x03060203,		/* igu.pci_pf_msi_en .. igu.pci_pf_msix_func_mask (3 regs) */
	0x044700d3,		/* mode (!bb)&(!(!asic)), block wol */
	0x12182040,		/* wol.acpi_enable .. wol.mpkt_enable (18 regs) */
	0x02982052,		/* wol.mpkt_mac_addr (2 regs, WB) */
	0x01182054,		/* wol.force_wol (1 regs) */
	0x01182066,		/* wol.tcp_syn_enable (1 regs) */
	0x031f0151,		/* mode !fpga, block tm */
	0x010b010f,		/* tm.pf_enable_conn (1 regs) */
	0x010b0111,		/* tm.pf_enable_task (1 regs) */
	0x020b013f,		/* tm.pf_scan_active_conn .. tm.pf_scan_active_task (2 regs) */
	0x05220000,		/* block src */
	0x0208e127,		/* src.NumIpv4Conn .. src.NumIpv6Conn (2 regs) */
	0x0288e140,		/* src.FirstFree (2 regs, WB) */
	0x0288e148,		/* src.LastFree (2 regs, WB) */
	0x0108e150,		/* src.CountFree (1 regs) */
	0x0108e181,		/* src.number_hash_bits (1 regs) */
	0x063f0000,		/* block tcfc */
	0x010b4176,		/* tcfc.robustwb_pf (1 regs) */
	0x010b41c0,		/* tcfc.weak_enable_pf (1 regs) */
	0x010b41c2,		/* tcfc.strong_enable_pf (1 regs) */
	0x010b41c6,		/* tcfc.pf_minicache_enable (1 regs) */
	0x010b7403,		/* tcfc.pf_lstate_cnt1 (1 regs) */
	0x010b7405,		/* tcfc.pf_lstate_cnt2 (1 regs) */
	0x0300000e,		/* split PORT_PF */
	0x01230000,		/* block prs */
	0x0207c1f3,		/* prs.first_hdr_dst_mac_0 .. prs.first_hdr_dst_mac_1 (2 regs) */
	0x05460000,		/* block nig */
	0x01140669,		/* nig.llh_func_tagmac_cls_type (1 regs) */
	0x0d14066c,		/* nig.llh_func_tag_en .. nig.llh_func_no_tag (13 regs) */
	0x20940680,		/* nig.llh_func_filter_value (32 regs, WB) */
	0x401406a0,		/* nig.llh_func_filter_en .. nig.llh_func_filter_hdr_sel (64 regs) */
	0x0114223e,		/* nig.dscp_to_tc_map_enable (1 regs) */
	0x05460081,		/* mode !k2, block nig */
	0x02142000,		/* nig.acpi_tag_rm .. nig.acpi_prop_hdr_rm (2 regs) */
	0x11142040,		/* nig.acpi_enable .. nig.acpi_pat_7_len (17 regs) */
	0x01142062,		/* nig.mpkt_enable (1 regs) */
	0x02942064,		/* nig.mpkt_mac_addr (2 regs, WB) */
	0x01142066,		/* nig.force_wol (1 regs) */
	0x0400002a,		/* split VF */
	0x03040000,		/* block pglue_b */
	0x010aa85a,		/* pglue_b.internal_vfid_enable (1 regs) */
	0x010aa95f,		/* pglue_b.fid_channel_enable (1 regs) */
	0x010aa966,		/* pglue_b.mask_block_discard_attn_vf (1 regs) */
	0x021e0000,		/* block qm */
	0x010bcbaa,		/* qm.vf_en (1 regs) */
	0x020bcbad,		/* qm.usg_cnt_vf_tx .. qm.usg_cnt_vf_other (2 regs) */
	0x03200000,		/* block dorq */
	0x01040143,		/* dorq.vf_db_enable (1 regs) */
	0x01040271,		/* dorq.vf_usage_cnt (1 regs) */
	0x01040275,		/* dorq.vf_ovfl_sticky (1 regs) */
	0x043e0000,		/* block ccfc */
	0x010b81c1,		/* ccfc.weak_enable_vf (1 regs) */
	0x010b81c3,		/* ccfc.strong_enable_vf (1 regs) */
	0x010bb402,		/* ccfc.vf_lstate_cnt1 (1 regs) */
	0x010bb404,		/* ccfc.vf_lstate_cnt2 (1 regs) */
	0x05400000,		/* block igu */
	0x01060102,		/* igu.statistic_num_vf_msg_sent (1 regs) */
	0x01060201,		/* igu.vf_configuration (1 regs) */
	0x02060330,		/* igu.int_before_mask_sts_vf_lsb .. igu.int_before_mask_sts_vf_msb (2 regs) */
	0x02060340,		/* igu.int_mask_sts_vf_lsb .. igu.int_mask_sts_vf_msb (2 regs) */
	0x02060350,		/* igu.pba_sts_vf_lsb .. igu.pba_sts_vf_msb (2 regs) */
	0x08150023,		/* mode !bb, block pglcs */
	0x0c000c00,		/* pglcs.pgl_cs_vf_1.VF_DEVICE_ID_VENDOR_ID_REG .. pglcs.pgl_cs_vf_1.VF_SUBSYSTEM_ID_SUBSYSTEM_VENDOR_ID_REG (12 regs) */
	0x01000c0d,		/* pglcs.pgl_cs_vf_1.VF_PCI_CAP_PTR_REG (1 regs) */
	0x01000c0f,		/* pglcs.pgl_cs_vf_1.VF_MAX_LATENCY_MIN_GRANT_INTERRUPT_PIN_INTERRUPT_LINE_REG (1 regs) */
	0x05000c1c,		/* pglcs.pgl_cs_vf_1.VF_PCIE_CAP_ID_PCIE_NEXT_CAP_PTR_PCIE_CAP_REG .. pglcs.pgl_cs_vf_1.VF_LINK_CONTROL_LINK_STATUS_REG (5 regs) */
	0x04000c25,		/* pglcs.pgl_cs_vf_1.VF_DEVICE_CAPABILITIES2_REG .. pglcs.pgl_cs_vf_1.VF_LINK_CONTROL2_LINK_STATUS2_REG (4 regs) */
	0x03000c2c,		/* pglcs.pgl_cs_vf_1.VF_PCI_MSIX_CAP_ID_NEXT_CTRL_REG .. pglcs.pgl_cs_vf_1.VF_MSIX_PBA_OFFSET_REG (3 regs) */
	0x02000c40,		/* pglcs.pgl_cs_vf_1.VF_ARI_BASE .. pglcs.pgl_cs_vf_1.VF_CAP_REG (2 regs) */
	0x04000c44,		/* pglcs.pgl_cs_vf_1.VF_TPH_EXT_CAP_HDR_REG .. pglcs.pgl_cs_vf_1.VF_TPH_ST_TABLE_REG_0 (4 regs) */
	0x01400081,		/* mode !k2, block igu */
	0x02060206,		/* igu.pci_vf_msix_en .. igu.pci_vf_msix_func_mask (2 regs) */
	0x031f0151,		/* mode !fpga, block tm */
	0x010b010e,		/* tm.vf_enable_conn (1 regs) */
	0x010b0110,		/* tm.vf_enable_task (1 regs) */
	0x020b0141,		/* tm.vf_scan_active_conn .. tm.vf_scan_active_task (2 regs) */
	0x043f0000,		/* block tcfc */
	0x010b41c1,		/* tcfc.weak_enable_vf (1 regs) */
	0x010b41c3,		/* tcfc.strong_enable_vf (1 regs) */
	0x010b7402,		/* tcfc.vf_lstate_cnt1 (1 regs) */
	0x010b7404,		/* tcfc.vf_lstate_cnt2 (1 regs) */
};

/* Data size: 11076 bytes */

#ifndef __PREVENT_DUMP_MEM_ARR__

/* Array of memories to be dumped */
static const u32 dump_mem[] = {
	0x000001f9,		/* split NONE */
	0x14040000,		/* block pglue_b */
	0x000aa910, 0x00000006,	/* pglue_b.sdm_inb_int_b_pf_0, group=PXP_MEM, size=6 regs */
	0x000aa918, 0x00000006,	/* pglue_b.sdm_inb_int_b_pf_1, group=PXP_MEM, size=6 regs */
	0x000aa920, 0x00000006,	/* pglue_b.sdm_inb_int_b_pf_2, group=PXP_MEM, size=6 regs */
	0x000aa928, 0x00000006,	/* pglue_b.sdm_inb_int_b_pf_3, group=PXP_MEM, size=6 regs */
	0x000aa930, 0x00000006,	/* pglue_b.sdm_inb_int_b_pf_4, group=PXP_MEM, size=6 regs */
	0x000aa938, 0x00000006,	/* pglue_b.sdm_inb_int_b_pf_5, group=PXP_MEM, size=6 regs */
	0x000aa940, 0x00000006,	/* pglue_b.sdm_inb_int_b_pf_6, group=PXP_MEM, size=6 regs */
	0x000aa948, 0x00000006,	/* pglue_b.sdm_inb_int_b_pf_7, group=PXP_MEM, size=6 regs */
	0x000aaa00, 0x010000b0,	/* pglue_b.write_fifo_queue, group=PXP_MEM, size=176 regs, WB */
	0x000aab00, 0x01000070,	/* pglue_b.read_fifo_queue, group=PXP_MEM, size=112 regs, WB */
	0x020d0000,		/* block pswhst */
	0x000a8100, 0x00000048,	/* pswhst.inbound_int, group=PXP_MEM, size=72 regs */
	0x02160000,		/* block dmae */
	0x01003200, 0x000001c0,	/* dmae.cmd_mem, group=DMAE_MEM, size=448 regs */
	0x0a180000,		/* block tcm */
	0x02460538, 0x00000008,	/* tcm.xx_cbyp_tbl, group=CM_MEM, size=8 regs */
	0x02460580, 0x00000020,	/* tcm.xx_lcid_cam, group=CM_MEM, size=32 regs */
	0x024605c0, 0x00000020,	/* tcm.xx_tbl, group=CM_MEM, size=32 regs */
	0x02460600, 0x00000040,	/* tcm.xx_dscr_tbl, group=CM_MEM, size=64 regs */
	0x02462000, 0x00001600,	/* tcm.xx_msg_ram, group=CM_MEM, size=5632 regs */
	0x0a190000,		/* block mcm */
	0x02480608, 0x00000008,	/* mcm.xx_cbyp_tbl, group=CM_MEM, size=8 regs */
	0x02480680, 0x00000016,	/* mcm.xx_lcid_cam, group=CM_MEM, size=22 regs */
	0x024806c0, 0x00000016,	/* mcm.xx_tbl, group=CM_MEM, size=22 regs */
	0x02480700, 0x00000040,	/* mcm.xx_dscr_tbl, group=CM_MEM, size=64 regs */
	0x02482000, 0x00001a00,	/* mcm.xx_msg_ram, group=CM_MEM, size=6656 regs */
	0x0a1a0000,		/* block ucm */
	0x024a05c8, 0x00000008,	/* ucm.xx_cbyp_tbl, group=CM_MEM, size=8 regs */
	0x024a0640, 0x00000018,	/* ucm.xx_lcid_cam, group=CM_MEM, size=24 regs */
	0x024a0680, 0x00000018,	/* ucm.xx_tbl, group=CM_MEM, size=24 regs */
	0x024a06c0, 0x00000040,	/* ucm.xx_dscr_tbl, group=CM_MEM, size=64 regs */
	0x024a2000, 0x00001a00,	/* ucm.xx_msg_ram, group=CM_MEM, size=6656 regs */
	0x0a1b0000,		/* block xcm */
	0x02400608, 0x00000008,	/* xcm.xx_cbyp_tbl, group=CM_MEM, size=8 regs */
	0x02400640, 0x0000001e,	/* xcm.xx_lcid_cam, group=CM_MEM, size=30 regs */
	0x02400680, 0x0000001e,	/* xcm.xx_tbl, group=CM_MEM, size=30 regs */
	0x024006c0, 0x00000040,	/* xcm.xx_dscr_tbl, group=CM_MEM, size=64 regs */
	0x02400800, 0x00000400,	/* xcm.xx_msg_ram, group=CM_MEM, size=1024 regs */
	0x0a1c0000,		/* block ycm */
	0x02420608, 0x00000008,	/* ycm.xx_cbyp_tbl, group=CM_MEM, size=8 regs */
	0x02420680, 0x00000016,	/* ycm.xx_lcid_cam, group=CM_MEM, size=22 regs */
	0x024206c0, 0x00000016,	/* ycm.xx_tbl, group=CM_MEM, size=22 regs */
	0x02420700, 0x00000040,	/* ycm.xx_dscr_tbl, group=CM_MEM, size=64 regs */
	0x02422000, 0x00001860,	/* ycm.xx_msg_ram, group=CM_MEM, size=6240 regs */
	0x081d0000,		/* block pcm */
	0x02440540, 0x00000002,	/* pcm.xx_lcid_cam, group=CM_MEM, size=2 regs */
	0x02440580, 0x00000002,	/* pcm.xx_tbl, group=CM_MEM, size=2 regs */
	0x024405c0, 0x00000004,	/* pcm.xx_dscr_tbl, group=CM_MEM, size=4 regs */
	0x02440800, 0x000002c0,	/* pcm.xx_msg_ram, group=CM_MEM, size=704 regs */
	0x081e0000,		/* block qm */
	0x030bc700, 0x00000048,	/* qm.CMIntQMask, group=QM_MEM, size=72 regs */
	0x030bcd00, 0x00000100,	/* qm.RlGlblIncVal, group=QM_MEM, size=256 regs */
	0x030bcf00, 0x00000100,	/* qm.RlGlblUpperBound, group=QM_MEM, size=256 regs */
	0x030bd100, 0x00000100,	/* qm.RlGlblCrd, group=QM_MEM, size=256 regs */
	0x02200000,		/* block dorq */
	0x04040400, 0x00000280,	/* dorq.wqe_buf, group=DORQ_MEM, size=640 regs */
	0x0c210000,		/* block brb */
	0x050d0204, 0x00000004,	/* brb.free_list_head, group=BRB_RAM, size=4 regs */
	0x050d0208, 0x00000004,	/* brb.free_list_tail, group=BRB_RAM, size=4 regs */
	0x050d020c, 0x00000004,	/* brb.free_list_size, group=BRB_RAM, size=4 regs */
	0x060d0600, 0x01000014,	/* brb.stopped_rd_req, group=BRB_MEM, size=20 regs, WB */
	0x060d0640, 0x01000014,	/* brb.stopped_rls_req, group=BRB_MEM, size=20 regs, WB */
	0x060d0680, 0x00000022,	/* brb.per_tc_counters, group=BRB_MEM, size=34 regs */
	0x06230000,		/* block prs */
	0x0707c300, 0x00000080,	/* prs.last_pkt_list, group=PRS_MEM, size=128 regs */
	0x0707c400, 0x01000040,	/* prs.gft_profile_mask_ram, group=PRS_MEM, size=64 regs, WB */
	0x0707c440, 0x0000001f,	/* prs.gft_cam, group=PRS_MEM, size=31 regs */
	0x04240000,		/* block tsdm */
	0x083ec200, 0x00000020,	/* tsdm.agg_int_ctrl, group=SDM_MEM, size=32 regs */
	0x083ec280, 0x00000020,	/* tsdm.agg_int_state, group=SDM_MEM, size=32 regs */
	0x04250000,		/* block msdm */
	0x083f0200, 0x00000020,	/* msdm.agg_int_ctrl, group=SDM_MEM, size=32 regs */
	0x083f0280, 0x00000020,	/* msdm.agg_int_state, group=SDM_MEM, size=32 regs */
	0x04260000,		/* block usdm */
	0x083f4200, 0x00000020,	/* usdm.agg_int_ctrl, group=SDM_MEM, size=32 regs */
	0x083f4280, 0x00000020,	/* usdm.agg_int_state, group=SDM_MEM, size=32 regs */
	0x04270000,		/* block xsdm */
	0x083e0200, 0x00000020,	/* xsdm.agg_int_ctrl, group=SDM_MEM, size=32 regs */
	0x083e0280, 0x00000020,	/* xsdm.agg_int_state, group=SDM_MEM, size=32 regs */
	0x04280000,		/* block ysdm */
	0x083e4200, 0x00000020,	/* ysdm.agg_int_ctrl, group=SDM_MEM, size=32 regs */
	0x083e4280, 0x00000020,	/* ysdm.agg_int_state, group=SDM_MEM, size=32 regs */
	0x04290000,		/* block psdm */
	0x083e8200, 0x00000020,	/* psdm.agg_int_ctrl, group=SDM_MEM, size=32 regs */
	0x083e8280, 0x00000020,	/* psdm.agg_int_state, group=SDM_MEM, size=32 regs */
	0x0e2a0000,		/* block tsem */
	0x095c02c4, 0x00000001,	/* tsem.thread_valid, group=PBUF, size=1 regs */
	0x095c8000, 0x010010e0,	/* tsem.passive_buffer, group=PBUF, size=4320 regs, WB */
	0x0a5d0100, 0x00000020,	/* tsem.fast_memory.gpre, group=IOR, size=32 regs */
	0x0a5d0130, 0x00000001,	/* tsem.fast_memory.active_reg_set, group=IOR, size=1 regs */
	0x0b5d0180, 0x0000000a,	/* tsem.fast_memory.real_time_cnt, group=SEM_MEM, size=10 regs */
	0x0a5d2000, 0x00000200,	/* tsem.fast_memory.storm_reg_file, group=IOR, size=512 regs */
	0x0c5d8000, 0x00005000,	/* tsem.fast_memory.int_ram, group=RAM, size=20480 regs */
	0x0e2b0000,		/* block msem */
	0x096002c4, 0x00000001,	/* msem.thread_valid, group=PBUF, size=1 regs */
	0x09608000, 0x010010e0,	/* msem.passive_buffer, group=PBUF, size=4320 regs, WB */
	0x0a610100, 0x00000020,	/* msem.fast_memory.gpre, group=IOR, size=32 regs */
	0x0a610130, 0x00000001,	/* msem.fast_memory.active_reg_set, group=IOR, size=1 regs */
	0x0b610180, 0x0000000a,	/* msem.fast_memory.real_time_cnt, group=SEM_MEM, size=10 regs */
	0x0a612000, 0x00000200,	/* msem.fast_memory.storm_reg_file, group=IOR, size=512 regs */
	0x0c618000, 0x00005000,	/* msem.fast_memory.int_ram, group=RAM, size=20480 regs */
	0x0e2c0000,		/* block usem */
	0x096402c4, 0x00000001,	/* usem.thread_valid, group=PBUF, size=1 regs */
	0x09648000, 0x01000b40,	/* usem.passive_buffer, group=PBUF, size=2880 regs, WB */
	0x0a650100, 0x00000020,	/* usem.fast_memory.gpre, group=IOR, size=32 regs */
	0x0a650130, 0x00000001,	/* usem.fast_memory.active_reg_set, group=IOR, size=1 regs */
	0x0b650180, 0x0000000a,	/* usem.fast_memory.real_time_cnt, group=SEM_MEM, size=10 regs */
	0x0a652000, 0x00000200,	/* usem.fast_memory.storm_reg_file, group=IOR, size=512 regs */
	0x0c658000, 0x00005000,	/* usem.fast_memory.int_ram, group=RAM, size=20480 regs */
	0x0e2d0000,		/* block xsem */
	0x095002c4, 0x00000001,	/* xsem.thread_valid, group=PBUF, size=1 regs */
	0x09508000, 0x010010e0,	/* xsem.passive_buffer, group=PBUF, size=4320 regs, WB */
	0x0a510100, 0x00000020,	/* xsem.fast_memory.gpre, group=IOR, size=32 regs */
	0x0a510130, 0x00000001,	/* xsem.fast_memory.active_reg_set, group=IOR, size=1 regs */
	0x0b510180, 0x0000000a,	/* xsem.fast_memory.real_time_cnt, group=SEM_MEM, size=10 regs */
	0x0a512000, 0x00000200,	/* xsem.fast_memory.storm_reg_file, group=IOR, size=512 regs */
	0x0c518000, 0x00005000,	/* xsem.fast_memory.int_ram, group=RAM, size=20480 regs */
	0x0e2e0000,		/* block ysem */
	0x095402c4, 0x00000001,	/* ysem.thread_valid, group=PBUF, size=1 regs */
	0x09548000, 0x010009d8,	/* ysem.passive_buffer, group=PBUF, size=2520 regs, WB */
	0x0a550100, 0x00000020,	/* ysem.fast_memory.gpre, group=IOR, size=32 regs */
	0x0a550130, 0x00000001,	/* ysem.fast_memory.active_reg_set, group=IOR, size=1 regs */
	0x0b550180, 0x0000000a,	/* ysem.fast_memory.real_time_cnt, group=SEM_MEM, size=10 regs */
	0x0a552000, 0x00000200,	/* ysem.fast_memory.storm_reg_file, group=IOR, size=512 regs */
	0x0c558000, 0x00005000,	/* ysem.fast_memory.int_ram, group=RAM, size=20480 regs */
	0x0e2f0000,		/* block psem */
	0x095802c4, 0x00000001,	/* psem.thread_valid, group=PBUF, size=1 regs */
	0x09588000, 0x010002d0,	/* psem.passive_buffer, group=PBUF, size=720 regs, WB */
	0x0a590100, 0x00000020,	/* psem.fast_memory.gpre, group=IOR, size=32 regs */
	0x0a590130, 0x00000001,	/* psem.fast_memory.active_reg_set, group=IOR, size=1 regs */
	0x0b590180, 0x0000000a,	/* psem.fast_memory.real_time_cnt, group=SEM_MEM, size=10 regs */
	0x0a592000, 0x00000200,	/* psem.fast_memory.storm_reg_file, group=IOR, size=512 regs */
	0x0c598000, 0x00005000,	/* psem.fast_memory.int_ram, group=RAM, size=20480 regs */
	0x06390000,		/* block btb */
	0x0d36c204, 0x00000004,	/* btb.free_list_head, group=BTB_RAM, size=4 regs */
	0x0d36c208, 0x00000004,	/* btb.free_list_tail, group=BTB_RAM, size=4 regs */
	0x0d36c20c, 0x00000004,	/* btb.free_list_size, group=BTB_RAM, size=4 regs */
	0x023b0000,		/* block rdif */
	0x0e0c1000, 0x01000a00,	/* rdif.l1_task_context, group=RDIF_CTX, size=2560 regs, WB */
	0x023c0000,		/* block tdif */
	0x0f0c6000, 0x01001400,	/* tdif.l1_task_context, group=TDIF_CTX, size=5120 regs, WB */
	0x0a3e0000,		/* block ccfc */
	0x100b81d0, 0x0000000e,	/* ccfc.lcreq_credit, group=CFC_MEM, size=14 regs */
	0x110ba200, 0x00000140,	/* ccfc.activity_counter, group=CONN_CFC_MEM, size=320 regs */
	0x110ba400, 0x00000140,	/* ccfc.info_state, group=CONN_CFC_MEM, size=320 regs */
	0x110ba600, 0x00000140,	/* ccfc.info_reg, group=CONN_CFC_MEM, size=320 regs */
	0x110bac00, 0x01000280,	/* ccfc.cid_cam, group=CONN_CFC_MEM, size=640 regs, WB */
	0x0e410000,		/* block cau */
	0x12070100, 0x00000001,	/* cau.num_pi_per_sb, group=CAU_PI, size=1 regs */
	0x13070800, 0x01000078,	/* cau.cqe_fifo, group=CAU_MEM, size=120 regs, WB */
	0x13070880, 0x01000010,	/* cau.igu_cmd_fifo, group=CAU_MEM, size=16 regs, WB */
	0x130708c0, 0x01000020,	/* cau.pxp_req_fifo, group=CAU_MEM, size=32 regs, WB */
	0x13070900, 0x01000100,	/* cau.pxp_wdata_fifo, group=CAU_MEM, size=256 regs, WB */
	0x14071100, 0x00000100,	/* cau.fsm_table, group=CAU_MEM_EXT, size=256 regs */
	0x14077000, 0x01000100,	/* cau.agg_unit_descriptor, group=CAU_MEM_EXT, size=256 regs, WB */
	0x020d0005,		/* mode bb, block pswhst */
	0x000a8200, 0x00000100,	/* pswhst.zone_permission_table, group=PXP_MEM, size=256 regs */
	0x02140000,		/* block pswrq2 */
	0x15098000, 0x01003b60,	/* pswrq2.ilt_memory, group=PXP_ILT, size=15200 regs, WB */
	0x021b0000,		/* block xcm */
	0x02401000, 0x000001c0,	/* xcm.qm_act_st_cnt, group=CM_MEM, size=448 regs */
	0x2e1e0000,		/* block qm */
	0x030bc180, 0x00000040,	/* qm.BaseAddrOtherPq, group=QM_MEM, size=64 regs */
	0x030bc300, 0x01000080,	/* qm.PtrTblOther, group=QM_MEM, size=128 regs, WB */
	0x030bc560, 0x00000012,	/* qm.VoqCrdLine, group=QM_MEM, size=18 regs */
	0x030bc5a0, 0x00000012,	/* qm.VoqInitCrdLine, group=QM_MEM, size=18 regs */
	0x030bc5e0, 0x00000012,	/* qm.VoqCrdByte, group=QM_MEM, size=18 regs */
	0x030bc620, 0x00000012,	/* qm.VoqInitCrdByte, group=QM_MEM, size=18 regs */
	0x030bc800, 0x00000040,	/* qm.PqFillLvlOther, group=QM_MEM, size=64 regs */
	0x030bca00, 0x00000040,	/* qm.PqStsOther, group=QM_MEM, size=64 regs */
	0x030bd320, 0x00000008,	/* qm.RlPfIncVal, group=QM_MEM, size=8 regs */
	0x030bd340, 0x00000008,	/* qm.RlPfUpperBound, group=QM_MEM, size=8 regs */
	0x030bd360, 0x00000008,	/* qm.RlPfCrd, group=QM_MEM, size=8 regs */
	0x030bd3a0, 0x00000008,	/* qm.WfqPfWeight, group=QM_MEM, size=8 regs */
	0x030bd3c0, 0x00000008,	/* qm.WfqPfUpperBound, group=QM_MEM, size=8 regs */
	0x030bd500, 0x00000090,	/* qm.WfqPfCrd, group=QM_MEM, size=144 regs */
	0x030bd800, 0x000001c0,	/* qm.BaseAddrTxPq, group=QM_MEM, size=448 regs */
	0x030bdc00, 0x000001c0,	/* qm.PqFillLvlTx, group=QM_MEM, size=448 regs */
	0x030be000, 0x000001c0,	/* qm.PqStsTx, group=QM_MEM, size=448 regs */
	0x030be400, 0x000001c0,	/* qm.TxPqMap, group=QM_MEM, size=448 regs */
	0x030be800, 0x000001c0,	/* qm.WfqVpWeight, group=QM_MEM, size=448 regs */
	0x030bec00, 0x000001c0,	/* qm.WfqVpUpperBound, group=QM_MEM, size=448 regs */
	0x030bf000, 0x000001c0,	/* qm.WfqVpCrd, group=QM_MEM, size=448 regs */
	0x030bf400, 0x000001c0,	/* qm.WfqVpMap, group=QM_MEM, size=448 regs */
	0x030bf800, 0x01000380,	/* qm.PtrTblTx, group=QM_MEM, size=896 regs, WB */
	0x02210000,		/* block brb */
	0x050d2000, 0x000012c0,	/* brb.link_list, group=BRB_RAM, size=4800 regs */
	0x04320000,		/* block muld */
	0x16139000, 0x01000800,	/* muld.bd_db_arr_dw, group=MULD_MEM, size=2048 regs, WB */
	0x1613a000, 0x01000800,	/* muld.sge_db_arr_dw, group=MULD_MEM, size=2048 regs, WB */
	0x06390000,		/* block btb */
	0x1736c400, 0x01000008,	/* btb.stopped_rd_req, group=BTB_MEM, size=8 regs, WB */
	0x1736c440, 0x01000008,	/* btb.stopped_rls_req, group=BTB_MEM, size=8 regs, WB */
	0x0d36d000, 0x00000b40,	/* btb.link_list, group=BTB_RAM, size=2880 regs */
	0x0c400000,		/* block igu */
	0x18060220, 0x00000009,	/* igu.pending_bits_status, group=IGU_MEM, size=9 regs */
	0x18060240, 0x00000009,	/* igu.write_done_pending, group=IGU_MEM, size=9 regs */
	0x18060800, 0x00000128,	/* igu.producer_memory, group=IGU_MEM, size=296 regs */
	0x18060c00, 0x00000128,	/* igu.consumer_mem, group=IGU_MEM, size=296 regs */
	0x18061000, 0x00000120,	/* igu.mapping_memory, group=IGU_MEM, size=288 regs */
	0x19061800, 0x01000480,	/* igu.msix_memory, group=IGU_MSIX, size=1152 regs, WB */
	0x06410000,		/* block cau */
	0x1a071800, 0x01000240,	/* cau.sb_var_memory, group=CAU_SB, size=576 regs, WB */
	0x1a072000, 0x01000240,	/* cau.sb_addr_memory, group=CAU_SB, size=576 regs, WB */
	0x12074000, 0x00000d80,	/* cau.pi_memory, group=CAU_PI, size=3456 regs */
	0x020d0023,		/* mode !bb, block pswhst */
	0x000a8200, 0x00000140,	/* pswhst.zone_permission_table, group=PXP_MEM, size=320 regs */
	0x02140000,		/* block pswrq2 */
	0x15098000, 0x010055f0,	/* pswrq2.ilt_memory, group=PXP_ILT, size=22000 regs, WB */
	0x021b0000,		/* block xcm */
	0x02401000, 0x00000200,	/* xcm.qm_act_st_cnt, group=CM_MEM, size=512 regs */
	0x321e0000,		/* block qm */
	0x030bc180, 0x00000080,	/* qm.BaseAddrOtherPq, group=QM_MEM, size=128 regs */
	0x030bc300, 0x01000100,	/* qm.PtrTblOther, group=QM_MEM, size=256 regs, WB */
	0x030bc560, 0x00000014,	/* qm.VoqCrdLine, group=QM_MEM, size=20 regs */
	0x030bc5a0, 0x00000014,	/* qm.VoqInitCrdLine, group=QM_MEM, size=20 regs */
	0x030bc5e0, 0x00000014,	/* qm.VoqCrdByte, group=QM_MEM, size=20 regs */
	0x030bc620, 0x00000014,	/* qm.VoqInitCrdByte, group=QM_MEM, size=20 regs */
	0x030bc800, 0x00000080,	/* qm.PqFillLvlOther, group=QM_MEM, size=128 regs */
	0x030bca00, 0x00000080,	/* qm.PqStsOther, group=QM_MEM, size=128 regs */
	0x030bd320, 0x00000010,	/* qm.RlPfIncVal, group=QM_MEM, size=16 regs */
	0x030bd340, 0x00000010,	/* qm.RlPfUpperBound, group=QM_MEM, size=16 regs */
	0x030bd360, 0x00000010,	/* qm.RlPfCrd, group=QM_MEM, size=16 regs */
	0x030bd3a0, 0x00000010,	/* qm.WfqPfWeight, group=QM_MEM, size=16 regs */
	0x030bd3c0, 0x00000010,	/* qm.WfqPfUpperBound, group=QM_MEM, size=16 regs */
	0x030bd500, 0x000000a0,	/* qm.WfqPfCrd, group=QM_MEM, size=160 regs */
	0x030bd800, 0x00000200,	/* qm.BaseAddrTxPq, group=QM_MEM, size=512 regs */
	0x030bdc00, 0x00000200,	/* qm.PqFillLvlTx, group=QM_MEM, size=512 regs */
	0x030be000, 0x00000200,	/* qm.PqStsTx, group=QM_MEM, size=512 regs */
	0x030be400, 0x00000200,	/* qm.TxPqMap, group=QM_MEM, size=512 regs */
	0x030be800, 0x00000200,	/* qm.WfqVpWeight, group=QM_MEM, size=512 regs */
	0x030bec00, 0x00000200,	/* qm.WfqVpUpperBound, group=QM_MEM, size=512 regs */
	0x030bf000, 0x00000200,	/* qm.WfqVpCrd, group=QM_MEM, size=512 regs */
	0x030bf400, 0x00000200,	/* qm.WfqVpMap, group=QM_MEM, size=512 regs */
	0x030bf800, 0x01000400,	/* qm.PtrTblTx, group=QM_MEM, size=1024 regs, WB */
	0x030bfc00, 0x00000048,	/* qm.CMIntQMask_msb, group=QM_MEM, size=72 regs */
	0x030bfd00, 0x000000a0,	/* qm.WfqPfCrd_msb, group=QM_MEM, size=160 regs */
	0x02210000,		/* block brb */
	0x050d2000, 0x00001e00,	/* brb.link_list, group=BRB_RAM, size=7680 regs */
	0x04320000,		/* block muld */
	0x16139000, 0x01000a00,	/* muld.bd_db_arr_dw, group=MULD_MEM, size=2560 regs, WB */
	0x1613a000, 0x01000a00,	/* muld.sge_db_arr_dw, group=MULD_MEM, size=2560 regs, WB */
	0x06390000,		/* block btb */
	0x1736c400, 0x01000010,	/* btb.stopped_rd_req, group=BTB_MEM, size=16 regs, WB */
	0x1736c440, 0x01000010,	/* btb.stopped_rls_req, group=BTB_MEM, size=16 regs, WB */
	0x0d36d000, 0x00000e60,	/* btb.link_list, group=BTB_RAM, size=3680 regs */
	0x0c400000,		/* block igu */
	0x18060220, 0x0000000c,	/* igu.pending_bits_status, group=IGU_MEM, size=12 regs */
	0x18060240, 0x0000000c,	/* igu.write_done_pending, group=IGU_MEM, size=12 regs */
	0x18060800, 0x00000180,	/* igu.producer_memory, group=IGU_MEM, size=384 regs */
	0x18060c00, 0x00000180,	/* igu.consumer_mem, group=IGU_MEM, size=384 regs */
	0x18061000, 0x00000170,	/* igu.mapping_memory, group=IGU_MEM, size=368 regs */
	0x19061800, 0x010005c0,	/* igu.msix_memory, group=IGU_MSIX, size=1472 regs, WB */
	0x06410000,		/* block cau */
	0x1a071800, 0x010002e0,	/* cau.sb_var_memory, group=CAU_SB, size=736 regs, WB */
	0x1a072000, 0x010002e0,	/* cau.sb_addr_memory, group=CAU_SB, size=736 regs, WB */
	0x12074000, 0x00001140,	/* cau.pi_memory, group=CAU_PI, size=4416 regs */
	0x0c0900d9,		/* mode !(!asic), block bmb */
	0x1b150204, 0x00000004,	/* bmb.free_list_head, group=BMB_RAM, size=4 regs */
	0x1b150208, 0x00000004,	/* bmb.free_list_tail, group=BMB_RAM, size=4 regs */
	0x1b15020c, 0x00000004,	/* bmb.free_list_size, group=BMB_RAM, size=4 regs */
	0x1c150480, 0x01000014,	/* bmb.stopped_rd_req, group=BMB_MEM, size=20 regs, WB */
	0x1c1504c0, 0x01000028,	/* bmb.stopped_rls_req, group=BMB_MEM, size=40 regs, WB */
	0x1b150800, 0x00000480,	/* bmb.link_list, group=BMB_RAM, size=1152 regs */
	0x021f0151,		/* mode !fpga, block tm */
	0x1d0b2000, 0x01000a00,	/* tm.context_mem, group=TM_MEM, size=2560 regs, WB */
	0x0a3f0000,		/* block tcfc */
	0x100b41d0, 0x0000000e,	/* tcfc.lcreq_credit, group=CFC_MEM, size=14 regs */
	0x1e0b6200, 0x00000140,	/* tcfc.activity_counter, group=TASK_CFC_MEM, size=320 regs */
	0x1e0b6400, 0x00000140,	/* tcfc.info_state, group=TASK_CFC_MEM, size=320 regs */
	0x1e0b6600, 0x00000140,	/* tcfc.info_reg, group=TASK_CFC_MEM, size=320 regs */
	0x1e0b6c00, 0x01000280,	/* tcfc.cid_cam, group=TASK_CFC_MEM, size=640 regs, WB */
	0x041f01e5,		/* mode bb&(!fpga), block tm */
	0x1d0b0400, 0x01000100,	/* tm.config_conn_mem, group=TM_MEM, size=256 regs, WB */
	0x1d0b0800, 0x01000130,	/* tm.config_task_mem, group=TM_MEM, size=304 regs, WB */
	0x041f014b,		/* mode (!bb)&(!fpga), block tm */
	0x1d0b0400, 0x010001a0,	/* tm.config_conn_mem, group=TM_MEM, size=416 regs, WB */
	0x1d0b0800, 0x01000200,	/* tm.config_task_mem, group=TM_MEM, size=512 regs, WB */
};

/* Data size: 2024 bytes */

#endif /* __PREVENT_DUMP_MEM_ARR__ */

/* Idle check registers */
static const u32 idle_chk_regs[] = {
	0x15000740, 0x00010001,	/* cond: pglcs.INT_STS */
	0x15000741, 0x00010001,	/* cond: pglcs.INT_MASK */
	0x02002060, 0x00010001,	/* cond: misc.INT_STS */
	0x02002061, 0x00010001,	/* cond: misc.INT_MASK */
	0x020021ed, 0x00010001,	/* cond: misc.aeu_after_invert_1_igu */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021ee, 0x00010001,	/* cond: misc.aeu_after_invert_2_igu */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021f0, 0x00010001,	/* cond: misc.aeu_after_invert_4_igu */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021f1, 0x00010001,	/* cond: misc.aeu_after_invert_5_igu */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021f2, 0x00010001,	/* cond: misc.aeu_after_invert_6_igu */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021f3, 0x00010001,	/* cond: misc.aeu_after_invert_7_igu */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021f4, 0x00010001,	/* cond: misc.aeu_after_invert_8_igu */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021f5, 0x00010001,	/* cond: misc.aeu_after_invert_9_igu */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021f6, 0x00010001,	/* cond: misc.aeu_after_invert_1_mcp */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021f7, 0x00010001,	/* cond: misc.aeu_after_invert_2_mcp */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021f9, 0x00010001,	/* cond: misc.aeu_after_invert_4_mcp */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021fa, 0x00010001,	/* cond: misc.aeu_after_invert_5_mcp */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021fb, 0x00010001,	/* cond: misc.aeu_after_invert_6_mcp */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021fc, 0x00010001,	/* cond: misc.aeu_after_invert_7_mcp */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021fd, 0x00010001,	/* cond: misc.aeu_after_invert_8_mcp */
	0x2e55012b, 0x00000001,	/* info: ysem.fast_memory.pram_prty_addr_high mode=all */
	0x2e55012a, 0x00000001,	/* info: ysem.fast_memory.pram_prty_addr_low mode=all */
	0x020021fd, 0x00010001,	/* cond: misc.aeu_after_invert_8_mcp */
	0x2c65012b, 0x00000001,	/* info: usem.fast_memory.pram_prty_addr_high mode=all */
	0x2c65012a, 0x00000001,	/* info: usem.fast_memory.pram_prty_addr_low mode=all */
	0x020021fd, 0x00010001,	/* cond: misc.aeu_after_invert_8_mcp */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021fd, 0x00010001,	/* cond: misc.aeu_after_invert_8_mcp */
	0x2d51012b, 0x00000001,	/* info: xsem.fast_memory.pram_prty_addr_high mode=all */
	0x2d51012a, 0x00000001,	/* info: xsem.fast_memory.pram_prty_addr_low mode=all */
	0x020021fd, 0x00010001,	/* cond: misc.aeu_after_invert_8_mcp */
	0x2b61012b, 0x00000001,	/* info: msem.fast_memory.pram_prty_addr_high mode=all */
	0x2b61012a, 0x00000001,	/* info: msem.fast_memory.pram_prty_addr_low mode=all */
	0x020021fd, 0x00010001,	/* cond: misc.aeu_after_invert_8_mcp */
	0x2a5d012b, 0x00000001,	/* info: tsem.fast_memory.pram_prty_addr_high mode=all */
	0x2a5d012a, 0x00000001,	/* info: tsem.fast_memory.pram_prty_addr_low mode=all */
	0x020021fd, 0x00010001,	/* cond: misc.aeu_after_invert_8_mcp */
	0x2f59012b, 0x00000001,	/* info: psem.fast_memory.pram_prty_addr_high mode=all */
	0x2f59012a, 0x00000001,	/* info: psem.fast_memory.pram_prty_addr_low mode=all */
	0x020021fe, 0x00010001,	/* cond: misc.aeu_after_invert_9_mcp */
	0x0200220f, 0x00000001,	/* info: misc.attn_num_st mode=all */
	0x020021ff, 0x00010001,	/* cond: misc.aeu_sys_kill_occurred */
	0x01002460, 0x00010001,	/* cond: miscs.INT_STS_0 */
	0x01002461, 0x00010001,	/* cond: miscs.INT_MASK_0 */
	0x010025bf, 0x00010001,	/* cond: miscs.pcie_hot_reset */
	0x16003012, 0x00010001,	/* cond: dmae.go_c0 */
	0x16003013, 0x00010001,	/* cond: dmae.go_c1 */
	0x16003014, 0x00010001,	/* cond: dmae.go_c2 */
	0x16003015, 0x00010001,	/* cond: dmae.go_c3 */
	0x16003016, 0x00010001,	/* cond: dmae.go_c4 */
	0x16003017, 0x00010001,	/* cond: dmae.go_c5 */
	0x16003018, 0x00010001,	/* cond: dmae.go_c6 */
	0x16003019, 0x00010001,	/* cond: dmae.go_c7 */
	0x1600301a, 0x00010001,	/* cond: dmae.go_c8 */
	0x1600301b, 0x00010001,	/* cond: dmae.go_c9 */
	0x1600301c, 0x00010001,	/* cond: dmae.go_c10 */
	0x1600301d, 0x00010001,	/* cond: dmae.go_c11 */
	0x1600301e, 0x00010001,	/* cond: dmae.go_c12 */
	0x1600301f, 0x00010001,	/* cond: dmae.go_c13 */
	0x16003020, 0x00010001,	/* cond: dmae.go_c14 */
	0x16003021, 0x00010001,	/* cond: dmae.go_c15 */
	0x16003022, 0x00010001,	/* cond: dmae.go_c16 */
	0x16003023, 0x00010001,	/* cond: dmae.go_c17 */
	0x16003024, 0x00010001,	/* cond: dmae.go_c18 */
	0x16003025, 0x00010001,	/* cond: dmae.go_c19 */
	0x16003026, 0x00010001,	/* cond: dmae.go_c20 */
	0x16003027, 0x00010001,	/* cond: dmae.go_c21 */
	0x16003028, 0x00010001,	/* cond: dmae.go_c22 */
	0x16003029, 0x00010001,	/* cond: dmae.go_c23 */
	0x1600302a, 0x00010001,	/* cond: dmae.go_c24 */
	0x1600302b, 0x00010001,	/* cond: dmae.go_c25 */
	0x1600302c, 0x00010001,	/* cond: dmae.go_c26 */
	0x1600302d, 0x00010001,	/* cond: dmae.go_c27 */
	0x1600302e, 0x00010001,	/* cond: dmae.go_c28 */
	0x1600302f, 0x00010001,	/* cond: dmae.go_c29 */
	0x16003030, 0x00010001,	/* cond: dmae.go_c30 */
	0x16003031, 0x00010001,	/* cond: dmae.go_c31 */
	0x16003060, 0x00010001,	/* cond: dmae.INT_STS */
	0x16003061, 0x00010001,	/* cond: dmae.INT_MASK */
	0x16003080, 0x00010001,	/* cond: dmae.PRTY_STS_H_0 */
	0x16003081, 0x00010001,	/* cond: dmae.PRTY_MASK_H_0 */
	0x45004060, 0x00010001,	/* cond: dbg.INT_STS */
	0x45004061, 0x00010001,	/* cond: dbg.INT_MASK */
	0x45004080, 0x00010001,	/* cond: dbg.PRTY_STS_H_0 */
	0x45004081, 0x00010001,	/* cond: dbg.PRTY_MASK_H_0 */
	0x00014019, 0x00010001,	/* cond: grc.trace_fifo_valid_data */
	0x00014060, 0x00010001,	/* cond: grc.INT_STS_0 */
	0x00014061, 0x00010001,	/* cond: grc.INT_MASK_0 */
	0x00014080, 0x00010001,	/* cond: grc.PRTY_STS_H_0 */
	0x00014081, 0x00010001,	/* cond: grc.PRTY_MASK_H_0 */
	0x0c014810, 0x00010001,	/* cond: mcp2.PRTY_STS */
	0x0c014811, 0x00010001,	/* cond: mcp2.PRTY_MASK */
	0x0c014881, 0x00010001,	/* cond: mcp2.PRTY_STS_H_0 */
	0x0c014882, 0x00010001,	/* cond: mcp2.PRTY_MASK_H_0 */
	0x20040060, 0x00010001,	/* cond: dorq.INT_STS */
	0x20040061, 0x00010001,	/* cond: dorq.INT_MASK */
	0x20040064, 0x00010001,	/* cond: dorq.PRTY_STS */
	0x20040065, 0x00010001,	/* cond: dorq.PRTY_MASK */
	0x20040080, 0x00010001,	/* cond: dorq.PRTY_STS_H_0 */
	0x20040081, 0x00010001,	/* cond: dorq.PRTY_MASK_H_0 */
	0x20040260, 0x00010001,	/* cond: dorq.xcm_msg_init_crd */
	0x20040261, 0x00010001,	/* cond: dorq.tcm_msg_init_crd */
	0x20040262, 0x00010001,	/* cond: dorq.ucm_msg_init_crd */
	0x20040263, 0x00010001,	/* cond: dorq.pbf_cmd_init_crd */
	0x20040270, 0x00010001,	/* cond: dorq.pf_usage_cnt */
	0x20040271, 0x00010001,	/* cond: dorq.vf_usage_cnt */
	0x20040282, 0x00010001,	/* cond: dorq.cfc_ld_req_fifo_fill_lvl */
	0x20040283, 0x00010001,	/* cond: dorq.dorq_fifo_fill_lvl */
	0x20040286, 0x00010001,	/* cond: dorq.db_drop_cnt */
	0x20040289, 0x00000001,	/* info: dorq.db_drop_details mode=all */
	0x20040288, 0x00000001,	/* info: dorq.db_drop_details_reason mode=all */
	0x2004028c, 0x00010001,	/* cond: dorq.dpm_abort_cnt */
	0x20040293, 0x00000001,	/* info: dorq.dpm_abort_reason mode=all */
	0x20040291, 0x00000001,	/* info: dorq.dpm_abort_details_reason mode=all */
	0x20040296, 0x00010001,	/* cond: dorq.dpm_timer_expir_abort_cnt */
	0x200402a2, 0x00010001,	/* cond: dorq.dpm_tbl_fill_lvl */
	0x40060060, 0x00010001,	/* cond: igu.INT_STS */
	0x40060061, 0x00010001,	/* cond: igu.INT_MASK */
	0x40060064, 0x00010001,	/* cond: igu.PRTY_STS */
	0x40060065, 0x00010001,	/* cond: igu.PRTY_MASK */
	0x40060080, 0x00010001,	/* cond: igu.PRTY_STS_H_0 */
	0x40060081, 0x00010001,	/* cond: igu.PRTY_MASK_H_0 */
	0x40060545, 0x00010001,	/* cond: igu.attn_write_done_pending */
	0x40060547, 0x00010001,	/* cond: igu.Interrupt_status */
	0x4006054c, 0x00010001,	/* cond: igu.error_handling_data_valid */
	0x4006054d, 0x00010001,	/* cond: igu.silent_drop */
	0x4006054f, 0x00010001,	/* cond: igu.sb_ctrl_fsm */
	0x40060550, 0x00010001,	/* cond: igu.int_handle_fsm */
	0x40060551, 0x00010001,	/* cond: igu.attn_fsm */
	0x40060555, 0x00010001,	/* cond: igu.ctrl_fsm */
	0x40060556, 0x00010001,	/* cond: igu.pxp_arb_fsm */
	0x41070035, 0x00010001,	/* cond: cau.INT_STS */
	0x41070038, 0x00010001,	/* cond: cau.INT_MASK */
	0x41070080, 0x00010001,	/* cond: cau.PRTY_STS_H_0 */
	0x41070081, 0x00010001,	/* cond: cau.PRTY_MASK_H_0 */
	0x41070260, 0x00010001,	/* cond: cau.igu_req_credit_status */
	0x41070261, 0x00010001,	/* cond: cau.igu_cmd_credit_status */
	0x41070320, 0x00010001,	/* cond: cau.debug_fifo_status */
	0x41070321, 0x00010001,	/* cond: cau.error_pxp_req */
	0x41070322, 0x00010001,	/* cond: cau.error_fsm_line */
	0x41070323, 0x00000001,	/* info: cau.error_fsm_line_pre mode=all */
	0x41070324, 0x00010001,	/* cond: cau.parity_latch_status */
	0x41070325, 0x00010001,	/* cond: cau.error_cleanup_cmd_reg */
	0x41070327, 0x00010001,	/* cond: cau.agg_units_0to15_state */
	0x41070328, 0x00010001,	/* cond: cau.agg_units_16to31_state */
	0x41070329, 0x00010001,	/* cond: cau.agg_units_32to47_state */
	0x4107032a, 0x00010001,	/* cond: cau.agg_units_48to63_state */
	0x41070380, 0x00010001,	/* cond: cau.req_counter */
	0x41070382, 0x00010001,	/* cond: cau.wdone_counter */
	0x41070380, 0x00010001,	/* cond: cau.req_counter */
	0x41070381, 0x00010001,	/* cond: cau.ack_counter */
	0x410703c0, 0x00010001,	/* cond: cau.main_fsm_status */
	0x410703c1, 0x00010001,	/* cond: cau.var_read_fsm_status */
	0x410703c2, 0x00010001,	/* cond: cau.igu_dma_fsm_status */
	0x410703c3, 0x00010001,	/* cond: cau.igu_cqe_cmd_fsm_status */
	0x410703c4, 0x00010001,	/* cond: cau.igu_cqe_agg_fsm_status */
	0x2307c002, 0x00010001,	/* cond: prs.mac_vlan_cache_init_done */
	0x2307c010, 0x00010001,	/* cond: prs.INT_STS_0 */
	0x2307c011, 0x00010001,	/* cond: prs.INT_MASK_0 */
	0x2307c2da, 0x00000001,	/* info: prs.mini_cache_failed_response mode=all */
	0x2387c2d8, 0x00000002,	/* info: prs.mini_cache_entry width=2 access=WB mode=all */
	0x00014019, 0x00000001,	/* info: grc.trace_fifo_valid_data mode=all */
	0x2307c014, 0x00010001,	/* cond: prs.PRTY_STS */
	0x2307c015, 0x00010001,	/* cond: prs.PRTY_MASK */
	0x2307c081, 0x00010001,	/* cond: prs.PRTY_STS_H_0 */
	0x2307c082, 0x00010001,	/* cond: prs.PRTY_MASK_H_0 */
	0x2307c085, 0x00010001,	/* cond: prs.PRTY_STS_H_1 */
	0x2307c086, 0x00010001,	/* cond: prs.PRTY_MASK_H_1 */
	0x2307c2d4, 0x00010001,	/* cond: prs.prs_pkt_ct */
	0x2307c2d5, 0x00010001,	/* cond: prs.queue_pkt_avail_status */
	0x2307c2d6, 0x00010001,	/* cond: prs.storm_bkprs_status */
	0x2307c2d7, 0x00010001,	/* cond: prs.stop_parsing_status */
	0x2307c3c3, 0x00010001,	/* cond: prs.tcm_current_credit */
	0x2307c3c4, 0x00010001,	/* cond: prs.ccfc_search_current_credit */
	0x2307c3c5, 0x00010001,	/* cond: prs.tcfc_search_current_credit */
	0x2307c3c6, 0x00010001,	/* cond: prs.ccfc_load_current_credit */
	0x2307c3c7, 0x00010001,	/* cond: prs.tcfc_load_current_credit */
	0x2307c3c8, 0x00010001,	/* cond: prs.ccfc_search_req_ct */
	0x2307c3c9, 0x00010001,	/* cond: prs.tcfc_search_req_ct */
	0x2307c3ca, 0x00010001,	/* cond: prs.ccfc_load_req_ct */
	0x2307c3cb, 0x00010001,	/* cond: prs.tcfc_load_req_ct */
	0x2307c3cc, 0x00010001,	/* cond: prs.sop_req_ct */
	0x2307c3cd, 0x00010001,	/* cond: prs.eop_req_ct */
	0x3508c010, 0x00010001,	/* cond: prm.INT_STS */
	0x3508c011, 0x00010001,	/* cond: prm.INT_MASK */
	0x3508c014, 0x00010001,	/* cond: prm.PRTY_STS */
	0x3508c015, 0x00010001,	/* cond: prm.PRTY_MASK */
	0x3508c080, 0x00010001,	/* cond: prm.PRTY_STS_H_0 */
	0x3508c081, 0x00010001,	/* cond: prm.PRTY_MASK_H_0 */
	0x3008e202, 0x00010001,	/* cond: rss.rss_init_done */
	0x3008e260, 0x00010001,	/* cond: rss.INT_STS */
	0x3008e261, 0x00010001,	/* cond: rss.INT_MASK */
	0x3008e280, 0x00010001,	/* cond: rss.PRTY_STS_H_0 */
	0x3008e281, 0x00010001,	/* cond: rss.PRTY_MASK_H_0 */
	0x3008e301, 0x00010001,	/* cond: rss.tmld_credit */
	0x3008e30e, 0x00010001,	/* cond: rss.empty_status */
	0x3008e30f, 0x00010001,	/* cond: rss.full_status */
	0x3008e310, 0x00010001,	/* cond: rss.counters_status */
	0x3008e311, 0x00010001,	/* cond: rss.state_machines */
	0x3808f010, 0x00010001,	/* cond: rpb.INT_STS */
	0x3808f011, 0x00010001,	/* cond: rpb.INT_MASK */
	0x3808f014, 0x00010001,	/* cond: rpb.PRTY_STS */
	0x3808f015, 0x00010001,	/* cond: rpb.PRTY_MASK */
	0x3808f140, 0x00010001,	/* cond: rpb.db_empty */
	0x3808f141, 0x00000001,	/* info: rpb.db_full mode=all */
	0x3808f142, 0x00010001,	/* cond: rpb.tq_empty */
	0x3808f143, 0x00000001,	/* info: rpb.tq_full mode=all */
	0x3808f144, 0x00010001,	/* cond: rpb.ififo_empty */
	0x3808f145, 0x00000001,	/* info: rpb.ififo_full mode=all */
	0x3808f146, 0x00010001,	/* cond: rpb.pfifo_empty */
	0x3808f147, 0x00000001,	/* info: rpb.pfifo_full mode=all */
	0x3808f148, 0x00010001,	/* cond: rpb.tq_th_empty */
	0x14090000, 0x00010001,	/* cond: pswrq2.rbc_done */
	0x14090001, 0x00010001,	/* cond: pswrq2.cfg_done */
	0x14090060, 0x00010001,	/* cond: pswrq2.INT_STS */
	0x14090061, 0x00010001,	/* cond: pswrq2.INT_MASK */
	0x14090080, 0x00010001,	/* cond: pswrq2.PRTY_STS_H_0 */
	0x14090081, 0x00010001,	/* cond: pswrq2.PRTY_MASK_H_0 */
	0x14090115, 0x00010001,	/* cond: pswrq2.vq0_entry_cnt */
	0x14090116, 0x00010001,	/* cond: pswrq2.vq1_entry_cnt */
	0x14090117, 0x00010001,	/* cond: pswrq2.vq2_entry_cnt */
	0x14090118, 0x00010001,	/* cond: pswrq2.vq3_entry_cnt */
	0x14090119, 0x00010001,	/* cond: pswrq2.vq4_entry_cnt */
	0x1409011a, 0x00010001,	/* cond: pswrq2.vq5_entry_cnt */
	0x1409011b, 0x00010001,	/* cond: pswrq2.vq6_entry_cnt */
	0x1409011c, 0x00010001,	/* cond: pswrq2.vq7_entry_cnt */
	0x1409011d, 0x00010001,	/* cond: pswrq2.vq8_entry_cnt */
	0x1409011e, 0x00010001,	/* cond: pswrq2.vq9_entry_cnt */
	0x1409011f, 0x00010001,	/* cond: pswrq2.vq10_entry_cnt */
	0x14090120, 0x00010001,	/* cond: pswrq2.vq11_entry_cnt */
	0x14090121, 0x00010001,	/* cond: pswrq2.vq12_entry_cnt */
	0x14090122, 0x00010001,	/* cond: pswrq2.vq13_entry_cnt */
	0x14090123, 0x00010001,	/* cond: pswrq2.vq14_entry_cnt */
	0x14090124, 0x00010001,	/* cond: pswrq2.vq15_entry_cnt */
	0x14090125, 0x00010001,	/* cond: pswrq2.vq16_entry_cnt */
	0x14090126, 0x00010001,	/* cond: pswrq2.vq17_entry_cnt */
	0x14090127, 0x00010001,	/* cond: pswrq2.vq18_entry_cnt */
	0x14090128, 0x00010001,	/* cond: pswrq2.vq19_entry_cnt */
	0x14090129, 0x00010001,	/* cond: pswrq2.vq20_entry_cnt */
	0x1409012a, 0x00010001,	/* cond: pswrq2.vq21_entry_cnt */
	0x1409012b, 0x00010001,	/* cond: pswrq2.vq22_entry_cnt */
	0x1409012c, 0x00010001,	/* cond: pswrq2.vq23_entry_cnt */
	0x1409012d, 0x00010001,	/* cond: pswrq2.vq24_entry_cnt */
	0x1409012e, 0x00010001,	/* cond: pswrq2.vq25_entry_cnt */
	0x1409012f, 0x00010001,	/* cond: pswrq2.vq26_entry_cnt */
	0x14090130, 0x00010001,	/* cond: pswrq2.vq27_entry_cnt */
	0x14090131, 0x00010001,	/* cond: pswrq2.vq28_entry_cnt */
	0x14090132, 0x00010001,	/* cond: pswrq2.vq29_entry_cnt */
	0x14090133, 0x00010001,	/* cond: pswrq2.vq30_entry_cnt */
	0x14090134, 0x00010001,	/* cond: pswrq2.vq31_entry_cnt */
	0x140901c5, 0x00010001,	/* cond: pswrq2.BW_CREDIT */
	0x14090206, 0x00010001,	/* cond: pswrq2.treq_fifo_fill_lvl */
	0x14090207, 0x00010001,	/* cond: pswrq2.icpl_fifo_fill_lvl */
	0x1409024c, 0x00010001,	/* cond: pswrq2.l2p_err_details */
	0x1409024b, 0x00000001,	/* info: pswrq2.l2p_err_add_63_32 mode=all */
	0x1409024a, 0x00000001,	/* info: pswrq2.l2p_err_add_31_0 mode=all */
	0x1409024d, 0x00010001,	/* cond: pswrq2.l2p_err_details2 */
	0x14090271, 0x00010001,	/* cond: pswrq2.sr_cnt */
	0x1409024f, 0x00010001,	/* cond: pswrq2.sr_num_cfg */
	0x14090271, 0x00010001,	/* cond: pswrq2.sr_cnt */
	0x1409024f, 0x00010001,	/* cond: pswrq2.sr_num_cfg */
	0x14090272, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_0 */
	0x140902d1, 0x00010001,	/* cond: pswrq2.max_srs_vq0 */
	0x14090273, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_1 */
	0x140902d2, 0x00010001,	/* cond: pswrq2.max_srs_vq1 */
	0x14090274, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_2 */
	0x140902d3, 0x00010001,	/* cond: pswrq2.max_srs_vq2 */
	0x14090275, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_3 */
	0x140902d4, 0x00010001,	/* cond: pswrq2.max_srs_vq3 */
	0x14090276, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_4 */
	0x140902d5, 0x00010001,	/* cond: pswrq2.max_srs_vq4 */
	0x14090277, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_5 */
	0x140902d6, 0x00010001,	/* cond: pswrq2.max_srs_vq5 */
	0x14090278, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_6 */
	0x140902d7, 0x00010001,	/* cond: pswrq2.max_srs_vq6 */
	0x1409027b, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_9 */
	0x140902da, 0x00010001,	/* cond: pswrq2.max_srs_vq9 */
	0x1409027d, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_11 */
	0x140902dc, 0x00010001,	/* cond: pswrq2.max_srs_vq11 */
	0x1409027e, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_12 */
	0x140902dd, 0x00010001,	/* cond: pswrq2.max_srs_vq12 */
	0x14090281, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_15 */
	0x140902e0, 0x00010001,	/* cond: pswrq2.max_srs_vq15 */
	0x14090282, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_16 */
	0x140902e1, 0x00010001,	/* cond: pswrq2.max_srs_vq16 */
	0x14090283, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_17 */
	0x140902e2, 0x00010001,	/* cond: pswrq2.max_srs_vq17 */
	0x14090284, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_18 */
	0x140902e3, 0x00010001,	/* cond: pswrq2.max_srs_vq18 */
	0x14090285, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_19 */
	0x140902e4, 0x00010001,	/* cond: pswrq2.max_srs_vq19 */
	0x14090287, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_21 */
	0x140902e6, 0x00010001,	/* cond: pswrq2.max_srs_vq21 */
	0x14090288, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_22 */
	0x140902e7, 0x00010001,	/* cond: pswrq2.max_srs_vq22 */
	0x14090289, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_23 */
	0x140902e8, 0x00010001,	/* cond: pswrq2.max_srs_vq23 */
	0x1409028a, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_24 */
	0x140902e9, 0x00010001,	/* cond: pswrq2.max_srs_vq24 */
	0x1409028c, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_26 */
	0x140902eb, 0x00010001,	/* cond: pswrq2.max_srs_vq26 */
	0x1409028e, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_28 */
	0x140902ed, 0x00010001,	/* cond: pswrq2.max_srs_vq28 */
	0x1409028f, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_29 */
	0x140902ee, 0x00010001,	/* cond: pswrq2.max_srs_vq29 */
	0x14090290, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_30 */
	0x140902ef, 0x00010001,	/* cond: pswrq2.max_srs_vq30 */
	0x14090292, 0x00010001,	/* cond: pswrq2.blk_cnt */
	0x14090250, 0x00010001,	/* cond: pswrq2.blk_num_cfg */
	0x14090292, 0x00010001,	/* cond: pswrq2.blk_cnt */
	0x14090250, 0x00010001,	/* cond: pswrq2.blk_num_cfg */
	0x14090293, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_0 */
	0x14090251, 0x00010001,	/* cond: pswrq2.max_blks_vq0 */
	0x14090294, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_1 */
	0x14090252, 0x00010001,	/* cond: pswrq2.max_blks_vq1 */
	0x14090295, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_2 */
	0x14090253, 0x00010001,	/* cond: pswrq2.max_blks_vq2 */
	0x14090296, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_3 */
	0x14090254, 0x00010001,	/* cond: pswrq2.max_blks_vq3 */
	0x14090297, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_4 */
	0x14090255, 0x00010001,	/* cond: pswrq2.max_blks_vq4 */
	0x14090298, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_5 */
	0x14090256, 0x00010001,	/* cond: pswrq2.max_blks_vq5 */
	0x14090299, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_6 */
	0x14090257, 0x00010001,	/* cond: pswrq2.max_blks_vq6 */
	0x1409029c, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_9 */
	0x1409025a, 0x00010001,	/* cond: pswrq2.max_blks_vq9 */
	0x1409029e, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_11 */
	0x1409025c, 0x00010001,	/* cond: pswrq2.max_blks_vq11 */
	0x1409029f, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_12 */
	0x1409025d, 0x00010001,	/* cond: pswrq2.max_blks_vq12 */
	0x140902a2, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_15 */
	0x14090260, 0x00010001,	/* cond: pswrq2.max_blks_vq15 */
	0x140902a3, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_16 */
	0x14090261, 0x00010001,	/* cond: pswrq2.max_blks_vq16 */
	0x140902a4, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_17 */
	0x14090262, 0x00010001,	/* cond: pswrq2.max_blks_vq17 */
	0x140902a5, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_18 */
	0x14090263, 0x00010001,	/* cond: pswrq2.max_blks_vq18 */
	0x140902a6, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_19 */
	0x14090264, 0x00010001,	/* cond: pswrq2.max_blks_vq19 */
	0x140902a8, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_21 */
	0x14090266, 0x00010001,	/* cond: pswrq2.max_blks_vq21 */
	0x140902a9, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_22 */
	0x14090267, 0x00010001,	/* cond: pswrq2.max_blks_vq22 */
	0x140902aa, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_23 */
	0x14090268, 0x00010001,	/* cond: pswrq2.max_blks_vq23 */
	0x140902ab, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_24 */
	0x14090269, 0x00010001,	/* cond: pswrq2.max_blks_vq24 */
	0x140902ad, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_26 */
	0x1409026b, 0x00010001,	/* cond: pswrq2.max_blks_vq26 */
	0x140902af, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_28 */
	0x1409026d, 0x00010001,	/* cond: pswrq2.max_blks_vq28 */
	0x140902b0, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_29 */
	0x1409026e, 0x00010001,	/* cond: pswrq2.max_blks_vq29 */
	0x140902b1, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_30 */
	0x1409026f, 0x00010001,	/* cond: pswrq2.max_blks_vq30 */
	0x140902f2, 0x00010001,	/* cond: pswrq2.l2p_close_gate_sts */
	0x140902f3, 0x00010001,	/* cond: pswrq2.misc_close_gate_sts */
	0x140902f4, 0x00010001,	/* cond: pswrq2.misc_stall_mem_sts */
	0x130a0060, 0x00010001,	/* cond: pswrq.INT_STS */
	0x130a0061, 0x00010001,	/* cond: pswrq.INT_MASK */
	0x130a0064, 0x00010001,	/* cond: pswrq.PRTY_STS */
	0x130a0065, 0x00010001,	/* cond: pswrq.PRTY_MASK */
	0x50015800, 0x00230001,	/* info: pxpreqbus.PRTY_STS_H_0 mode=!bb */
	0x50015801, 0x00230001,	/* info: pxpreqbus.PRTY_MASK_H_0 mode=!bb */
	0x110a6860, 0x00010001,	/* cond: pswwr.INT_STS */
	0x110a6861, 0x00010001,	/* cond: pswwr.INT_MASK */
	0x110a6864, 0x00010001,	/* cond: pswwr.PRTY_STS */
	0x110a6864, 0x00010001,	/* cond: pswwr.PRTY_STS */
	0x120a6c12, 0x00010001,	/* cond: pswwr2.pglue_eop_err_details */
	0x120a6c14, 0x00010001,	/* cond: pswwr2.prm_curr_fill_level */
	0x120a6c16, 0x00010001,	/* cond: pswwr2.cdu_curr_fill_level */
	0x120a6c60, 0x00010001,	/* cond: pswwr2.INT_STS */
	0x120a6c61, 0x00010001,	/* cond: pswwr2.INT_MASK */
	0x120a6c64, 0x00010001,	/* cond: pswwr2.PRTY_STS */
	0x120a6c64, 0x00010001,	/* cond: pswwr2.PRTY_STS */
	0x120a6c80, 0x00010001,	/* cond: pswwr2.PRTY_STS_H_0 */
	0x120a6c81, 0x00010001,	/* cond: pswwr2.PRTY_MASK_H_0 */
	0x120a6c84, 0x00010001,	/* cond: pswwr2.PRTY_STS_H_1 */
	0x120a6c85, 0x00010001,	/* cond: pswwr2.PRTY_MASK_H_1 */
	0x120a6c88, 0x00010001,	/* cond: pswwr2.PRTY_STS_H_2 */
	0x120a6c89, 0x00010001,	/* cond: pswwr2.PRTY_MASK_H_2 */
	0x120a6c8c, 0x00010001,	/* cond: pswwr2.PRTY_STS_H_3 */
	0x120a6c8d, 0x00010001,	/* cond: pswwr2.PRTY_MASK_H_3 */
	0x0f0a7029, 0x00010001,	/* cond: pswrd.fifo_full_status */
	0x0f0a7060, 0x00010001,	/* cond: pswrd.INT_STS */
	0x0f0a7061, 0x00010001,	/* cond: pswrd.INT_MASK */
	0x0f0a7064, 0x00010001,	/* cond: pswrd.PRTY_STS */
	0x0f0a7065, 0x00010001,	/* cond: pswrd.PRTY_MASK */
	0x100a7400, 0x00010001,	/* cond: pswrd2.start_init */
	0x100a7401, 0x00010001,	/* cond: pswrd2.init_done */
	0x100a741a, 0x00010001,	/* cond: pswrd2.cpl_err_details */
	0x100a741b, 0x00010001,	/* cond: pswrd2.cpl_err_details2 */
	0x100a741f, 0x00010001,	/* cond: pswrd2.port_is_idle_0 */
	0x100a7420, 0x00010001,	/* cond: pswrd2.port_is_idle_1 */
	0x100a7438, 0x00010010,	/* cond: pswrd2.almost_full_0[0:15] */
	0x100a7460, 0x00010001,	/* cond: pswrd2.INT_STS */
	0x100a7461, 0x00010001,	/* cond: pswrd2.INT_MASK */
	0x100a7464, 0x00010001,	/* cond: pswrd2.PRTY_STS */
	0x100a7465, 0x00010001,	/* cond: pswrd2.PRTY_MASK */
	0x100a7480, 0x00010001,	/* cond: pswrd2.PRTY_STS_H_0 */
	0x100a7481, 0x00010001,	/* cond: pswrd2.PRTY_MASK_H_0 */
	0x100a7484, 0x00010001,	/* cond: pswrd2.PRTY_STS_H_1 */
	0x100a7485, 0x00010001,	/* cond: pswrd2.PRTY_MASK_H_1 */
	0x100a7518, 0x00010001,	/* cond: pswrd2.disable_inputs */
	0x0e0a7810, 0x00010001,	/* cond: pswhst2.header_fifo_status */
	0x0e0a7811, 0x00010001,	/* cond: pswhst2.data_fifo_status */
	0x0e0a7860, 0x00010001,	/* cond: pswhst2.INT_STS */
	0x0e0a7861, 0x00010001,	/* cond: pswhst2.INT_MASK */
	0x0d0a8013, 0x00010001,	/* cond: pswhst.discard_internal_writes_status */
	0x0d0a8014, 0x00010001,	/* cond: pswhst.discard_doorbells_status */
	0x0d0a8016, 0x00010001,	/* cond: pswhst.arb_is_idle */
	0x0d0a801c, 0x00010001,	/* cond: pswhst.incorrect_access_valid */
	0x0d0a801a, 0x00000001,	/* info: pswhst.incorrect_access_data mode=all */
	0x0d0a801b, 0x00000001,	/* info: pswhst.incorrect_access_length mode=all */
	0x0d0a801d, 0x00000001,	/* info: pswhst.incorrect_access_address mode=all */
	0x0d0a801e, 0x00010001,	/* cond: pswhst.per_violation_valid */
	0x0d0a801f, 0x00000001,	/* info: pswhst.per_violation_data mode=all */
	0x0d0a8023, 0x00010001,	/* cond: pswhst.source_credits_avail */
	0x0d0a8025, 0x00010001,	/* cond: pswhst.source_credit_viol_valid */
	0x0d0a8024, 0x00000001,	/* info: pswhst.source_credit_viol_data mode=all */
	0x0d0a8029, 0x00010001,	/* cond: pswhst.dest_credits_avail */
	0x0d0a802b, 0x00010001,	/* cond: pswhst.is_in_drain_mode */
	0x0d0a802e, 0x00010001,	/* cond: pswhst.timeout_valid */
	0x0d0a802d, 0x00000001,	/* info: pswhst.timeout_data mode=all */
	0x0d0a802e, 0x00010001,	/* cond: pswhst.timeout_valid */
	0x0d0a802f, 0x00000001,	/* info: pswhst.timeout_address mode=all */
	0x0d0a8058, 0x00010008,	/* cond: pswhst.clients_waiting_to_source_arb[0:7] */
	0x0d0a8060, 0x00010001,	/* cond: pswhst.INT_STS */
	0x0d0a8061, 0x00010001,	/* cond: pswhst.INT_MASK */
	0x0d0a8064, 0x00010001,	/* cond: pswhst.PRTY_STS */
	0x0d0a8065, 0x00010001,	/* cond: pswhst.PRTY_MASK */
	0x0d0a8080, 0x00010001,	/* cond: pswhst.PRTY_STS_H_0 */
	0x0d0a8081, 0x00010001,	/* cond: pswhst.PRTY_MASK_H_0 */
	0x040aa003, 0x00010001,	/* cond: pglue_b.init_done_ptt_gtt */
	0x040aa064, 0x00010001,	/* cond: pglue_b.PRTY_STS */
	0x040aa065, 0x00010001,	/* cond: pglue_b.PRTY_MASK */
	0x040aa080, 0x00010001,	/* cond: pglue_b.PRTY_STS_H_0 */
	0x040aa081, 0x00010001,	/* cond: pglue_b.PRTY_MASK_H_0 */
	0x040aa120, 0x00010001,	/* cond: pglue_b.pgl_write_blocked */
	0x040aa121, 0x00010001,	/* cond: pglue_b.pgl_read_blocked */
	0x040aa122, 0x00010001,	/* cond: pglue_b.read_fifo_occupancy_level */
	0x040aa12b, 0x00010001,	/* cond: pglue_b.rx_legacy_errors */
	0x040aa159, 0x00010001,	/* cond: pglue_b.pgl_txw_cdts */
	0x040aa804, 0x00010001,	/* cond: pglue_b.cfg_space_a_request */
	0x040aa806, 0x00010001,	/* cond: pglue_b.cfg_space_b_request */
	0x040aa808, 0x00010001,	/* cond: pglue_b.flr_request_vf_31_0 */
	0x040aa809, 0x00010001,	/* cond: pglue_b.flr_request_vf_63_32 */
	0x040aa80a, 0x00010001,	/* cond: pglue_b.flr_request_vf_95_64 */
	0x040aa80b, 0x00010001,	/* cond: pglue_b.flr_request_vf_127_96 */
	0x040aa80c, 0x00010001,	/* cond: pglue_b.flr_request_vf_159_128 */
	0x040aa80d, 0x00010001,	/* cond: pglue_b.flr_request_vf_191_160 */
	0x040aa810, 0x00010001,	/* cond: pglue_b.flr_request_pf_31_0 */
	0x040aa81b, 0x00010001,	/* cond: pglue_b.sr_iov_disabled_request */
	0x040aa83c, 0x00010001,	/* cond: pglue_b.incorrect_rcv_details */
	0x040aa83d, 0x00010001,	/* cond: pglue_b.was_error_vf_31_0 */
	0x040aa83e, 0x00010001,	/* cond: pglue_b.was_error_vf_63_32 */
	0x040aa83f, 0x00010001,	/* cond: pglue_b.was_error_vf_95_64 */
	0x040aa840, 0x00010001,	/* cond: pglue_b.was_error_vf_127_96 */
	0x040aa841, 0x00010001,	/* cond: pglue_b.was_error_vf_159_128 */
	0x040aa842, 0x00010001,	/* cond: pglue_b.was_error_vf_191_160 */
	0x040aa845, 0x00010001,	/* cond: pglue_b.was_error_pf_31_0 */
	0x040aa84f, 0x00010001,	/* cond: pglue_b.rx_err_details */
	0x040aa850, 0x00010001,	/* cond: pglue_b.rx_tcpl_err_details */
	0x040aa853, 0x00010001,	/* cond: pglue_b.tx_err_wr_details */
	0x040aa852, 0x00000001,	/* info: pglue_b.tx_err_wr_add_63_32 mode=all */
	0x040aa851, 0x00000001,	/* info: pglue_b.tx_err_wr_add_31_0 mode=all */
	0x040aa854, 0x00010001,	/* cond: pglue_b.tx_err_wr_details2 */
	0x040aa857, 0x00010001,	/* cond: pglue_b.tx_err_rd_details */
	0x040aa856, 0x00000001,	/* info: pglue_b.tx_err_rd_add_63_32 mode=all */
	0x040aa855, 0x00000001,	/* info: pglue_b.tx_err_rd_add_31_0 mode=all */
	0x040aa858, 0x00010001,	/* cond: pglue_b.tx_err_rd_details2 */
	0x040aa8ec, 0x00010001,	/* cond: pglue_b.vf_length_violation_details */
	0x040aa8ed, 0x00010001,	/* cond: pglue_b.vf_length_violation_details2 */
	0x040aa8ee, 0x00010001,	/* cond: pglue_b.vf_grc_space_violation_details */
	0x040aa8f6, 0x00010001,	/* cond: pglue_b.pcie_err_status */
	0x040aa953, 0x00010001,	/* cond: pglue_b.master_zlr_err_details */
	0x040aa952, 0x00000001,	/* info: pglue_b.master_zlr_err_add_63_32 mode=all */
	0x040aa951, 0x00000001,	/* info: pglue_b.master_zlr_err_add_31_0 mode=all */
	0x040aa955, 0x00010001,	/* cond: pglue_b.admin_window_violation_details */
	0x040aa956, 0x00010001,	/* cond: pglue_b.out_of_range_function_in_pretend_details */
	0x040aa957, 0x00010001,	/* cond: pglue_b.out_of_range_function_in_pretend_address */
	0x040aab80, 0x00010001,	/* cond: pglue_b.write_fifo_occupancy_level */
	0x040aab86, 0x00010001,	/* cond: pglue_b.illegal_address_details */
	0x040aab85, 0x00000001,	/* info: pglue_b.illegal_address_add_63_32 mode=all */
	0x040aab84, 0x00000001,	/* info: pglue_b.illegal_address_add_31_0 mode=all */
	0x040aab87, 0x00010001,	/* cond: pglue_b.illegal_address_details2 */
	0x040aab8e, 0x00010001,	/* cond: pglue_b.tags_31_0 */
	0x040aab8f, 0x00010001,	/* cond: pglue_b.tags_63_32 */
	0x040aab90, 0x00010001,	/* cond: pglue_b.tags_95_64 */
	0x040aab91, 0x00010001,	/* cond: pglue_b.tags_127_96 */
	0x040aab9f, 0x00010001,	/* cond: pglue_b.vf_ilt_err_details */
	0x040aab9e, 0x00000001,	/* info: pglue_b.vf_ilt_err_add_63_32 mode=all */
	0x040aab9d, 0x00000001,	/* info: pglue_b.vf_ilt_err_add_31_0 mode=all */
	0x040aaba0, 0x00010001,	/* cond: pglue_b.vf_ilt_err_details2 */
	0x3e0b8001, 0x00010001,	/* cond: ccfc.ll_init_done */
	0x3e0b8002, 0x00010001,	/* cond: ccfc.ac_init_done */
	0x3e0b8003, 0x00010001,	/* cond: ccfc.cam_init_done */
	0x3e0b8004, 0x00010001,	/* cond: ccfc.tidram_init_done */
	0x3e0b8060, 0x00010001,	/* cond: ccfc.INT_STS_0 */
	0x3e0b8061, 0x00010001,	/* cond: ccfc.INT_MASK_0 */
	0x3e0b8153, 0x00000001,	/* info: ccfc.error_vector mode=all */
	0x3e0b8156, 0x00000001,	/* info: ccfc.error_data1 mode=all */
	0x3e0b8157, 0x00000001,	/* info: ccfc.error_data2 mode=all */
	0x3e0b8158, 0x00000001,	/* info: ccfc.error_data3 mode=all */
	0x3e0b8159, 0x00000001,	/* info: ccfc.error_data4 mode=all */
	0x3e0b8080, 0x00010001,	/* cond: ccfc.PRTY_STS_H_0 */
	0x3e0b8081, 0x00010001,	/* cond: ccfc.PRTY_MASK_H_0 */
	0x3e0b8179, 0x00010001,	/* cond: ccfc.PRTY_STS */
	0x3e0b817a, 0x00010001,	/* cond: ccfc.PRTY_MASK */
	0x3e0b8184, 0x00010001,	/* cond: ccfc.lstate_arriving */
	0x3e0b8187, 0x00010001,	/* cond: ccfc.lstate_leaving */
	0x3e0b81e2, 0x00010001,	/* cond: ccfc.cduld_credit */
	0x3e0b81e3, 0x00010001,	/* cond: ccfc.cduwb_credit */
	0x1e0bc060, 0x00010001,	/* cond: qm.INT_STS */
	0x1e0bc061, 0x00010001,	/* cond: qm.INT_MASK */
	0x1e0bc064, 0x00010001,	/* cond: qm.PRTY_STS */
	0x1e0bc065, 0x00010001,	/* cond: qm.PRTY_MASK */
	0x1e0bc080, 0x00010001,	/* cond: qm.PRTY_STS_H_0 */
	0x1e0bc081, 0x00010001,	/* cond: qm.PRTY_MASK_H_0 */
	0x1e0bc084, 0x00010001,	/* cond: qm.PRTY_STS_H_1 */
	0x1e0bc085, 0x00010001,	/* cond: qm.PRTY_MASK_H_1 */
	0x1e0bc088, 0x00010001,	/* cond: qm.PRTY_STS_H_2 */
	0x1e0bc089, 0x00010001,	/* cond: qm.PRTY_MASK_H_2 */
	0x1e0bc106, 0x00010006,	/* cond: qm.wrc_fifolvl_0[0:5] */
	0x1e0bc202, 0x00010001,	/* cond: qm.OutLdReqCrdConnTx */
	0x1e0bc200, 0x00010001,	/* cond: qm.OutLdReqSizeConnTx */
	0x1e0bc203, 0x00010001,	/* cond: qm.OutLdReqCrdConnOther */
	0x1e0bc201, 0x00010001,	/* cond: qm.OutLdReqSizeConnOther */
	0x1e0bc54e, 0x00010001,	/* cond: qm.OvfQNumTx */
	0x1e0bc54f, 0x00010001,	/* cond: qm.OvfErrorTx */
	0x1e0bc54e, 0x00000001,	/* info: qm.OvfQNumTx mode=all */
	0x1e0bc550, 0x00010001,	/* cond: qm.OvfQNumOther */
	0x1e0bc551, 0x00010001,	/* cond: qm.OvfErrorOther */
	0x1e0bc550, 0x00000001,	/* info: qm.OvfQNumOther mode=all */
	0x1e0bc68e, 0x00010001,	/* cond: qm.CmCrd_0 */
	0x1e0bc684, 0x00010001,	/* cond: qm.CmInitCrd_0 */
	0x1e0bc68f, 0x00010001,	/* cond: qm.CmCrd_1 */
	0x1e0bc685, 0x00010001,	/* cond: qm.CmInitCrd_1 */
	0x1e0bc690, 0x00010001,	/* cond: qm.CmCrd_2 */
	0x1e0bc686, 0x00010001,	/* cond: qm.CmInitCrd_2 */
	0x1e0bc691, 0x00010001,	/* cond: qm.CmCrd_3 */
	0x1e0bc687, 0x00010001,	/* cond: qm.CmInitCrd_3 */
	0x1e0bc692, 0x00010001,	/* cond: qm.CmCrd_4 */
	0x1e0bc688, 0x00010001,	/* cond: qm.CmInitCrd_4 */
	0x1e0bc693, 0x00010001,	/* cond: qm.CmCrd_5 */
	0x1e0bc689, 0x00010001,	/* cond: qm.CmInitCrd_5 */
	0x1e0bc694, 0x00010001,	/* cond: qm.CmCrd_6 */
	0x1e0bc68a, 0x00010001,	/* cond: qm.CmInitCrd_6 */
	0x1e0bc695, 0x00010001,	/* cond: qm.CmCrd_7 */
	0x1e0bc68b, 0x00010001,	/* cond: qm.CmInitCrd_7 */
	0x1e0bc696, 0x00010001,	/* cond: qm.CmCrd_8 */
	0x1e0bc68c, 0x00010001,	/* cond: qm.CmInitCrd_8 */
	0x1e0bc697, 0x00010001,	/* cond: qm.CmCrd_9 */
	0x1e0bc68d, 0x00010001,	/* cond: qm.CmInitCrd_9 */
	0x1e0bd303, 0x00010001,	/* cond: qm.Err_Inc0_RlGlblCrd */
	0x1e0bd306, 0x00010001,	/* cond: qm.Err_Mask_RlGlblCrd */
	0x1e0bd304, 0x00010001,	/* cond: qm.Err_Dec0_RlGlblCrd */
	0x1e0bd306, 0x00010001,	/* cond: qm.Err_Mask_RlGlblCrd */
	0x1e0bd305, 0x00010001,	/* cond: qm.Err_Dec1_RlGlblCrd */
	0x1e0bd306, 0x00010001,	/* cond: qm.Err_Mask_RlGlblCrd */
	0x1e0bd382, 0x00010001,	/* cond: qm.Err_Inc0_RlPfCrd */
	0x1e0bd385, 0x00010001,	/* cond: qm.Err_Mask_RlPfCrd */
	0x1e0bd383, 0x00010001,	/* cond: qm.Err_Dec0_RlPfCrd */
	0x1e0bd385, 0x00010001,	/* cond: qm.Err_Mask_RlPfCrd */
	0x1e0bd384, 0x00010001,	/* cond: qm.Err_Dec1_RlPfCrd */
	0x1e0bd385, 0x00010001,	/* cond: qm.Err_Mask_RlPfCrd */
	0x1e0bd701, 0x00010001,	/* cond: qm.Err_Inc0_WfqPfCrd */
	0x1e0bd704, 0x00010001,	/* cond: qm.Err_Mask_WfqPfCrd */
	0x1e0bd702, 0x00010001,	/* cond: qm.Err_Dec0_WfqPfCrd */
	0x1e0bd704, 0x00010001,	/* cond: qm.Err_Mask_WfqPfCrd */
	0x1e0bd703, 0x00010001,	/* cond: qm.Err_Dec1_WfqPfCrd */
	0x1e0bd704, 0x00010001,	/* cond: qm.Err_Mask_WfqPfCrd */
	0x1e0bd74f, 0x00010001,	/* cond: qm.Err_Inc0_VoqLineCrd */
	0x1e0bd753, 0x00010001,	/* cond: qm.Err_Mask_VoqLineCrd */
	0x1e0bd750, 0x00010001,	/* cond: qm.Err_Inc1_VoqLineCrd */
	0x1e0bd753, 0x00010001,	/* cond: qm.Err_Mask_VoqLineCrd */
	0x1e0bd751, 0x00010001,	/* cond: qm.Err_Dec0_VoqLineCrd */
	0x1e0bd753, 0x00010001,	/* cond: qm.Err_Mask_VoqLineCrd */
	0x1e0bd752, 0x00010001,	/* cond: qm.Err_Dec1_VoqLineCrd */
	0x1e0bd753, 0x00010001,	/* cond: qm.Err_Mask_VoqLineCrd */
	0x1e0bd754, 0x00010001,	/* cond: qm.Err_Inc0_VoqByteCrd */
	0x1e0bd758, 0x00010001,	/* cond: qm.Err_Mask_VoqByteCrd */
	0x1e0bd755, 0x00010001,	/* cond: qm.Err_Inc1_VoqByteCrd */
	0x1e0bd758, 0x00010001,	/* cond: qm.Err_Mask_VoqByteCrd */
	0x1e0bd756, 0x00010001,	/* cond: qm.Err_Dec0_VoqByteCrd */
	0x1e0bd758, 0x00010001,	/* cond: qm.Err_Mask_VoqByteCrd */
	0x1e0bd757, 0x00010001,	/* cond: qm.Err_Dec1_VoqByteCrd */
	0x1e0bd758, 0x00010001,	/* cond: qm.Err_Mask_VoqByteCrd */
	0x1e0bd759, 0x00010001,	/* cond: qm.fifo_empty0 */
	0x1e0bd75b, 0x00000001,	/* info: qm.fifo_full0 mode=all */
	0x1e0bd75a, 0x00010001,	/* cond: qm.fifo_empty1 */
	0x1e0bd75c, 0x00000001,	/* info: qm.fifo_full1 mode=all */
	0x1e0bd75d, 0x00010001,	/* cond: qm.fifo_error0 */
	0x1e0bd75e, 0x00010001,	/* cond: qm.fifo_error1 */
	0x3b0c0060, 0x00010001,	/* cond: rdif.INT_STS */
	0x3b0c0061, 0x00010001,	/* cond: rdif.INT_MASK */
	0x3b0c0064, 0x00010001,	/* cond: rdif.PRTY_STS */
	0x3b0c0064, 0x00010001,	/* cond: rdif.PRTY_STS */
	0x3c0c4060, 0x00010001,	/* cond: tdif.INT_STS */
	0x3c0c4061, 0x00010001,	/* cond: tdif.INT_MASK */
	0x3c0c4064, 0x00010001,	/* cond: tdif.PRTY_STS */
	0x3c0c4064, 0x00010001,	/* cond: tdif.PRTY_STS */
	0x3c0c4080, 0x00010001,	/* cond: tdif.PRTY_STS_H_0 */
	0x3c0c4081, 0x00010001,	/* cond: tdif.PRTY_MASK_H_0 */
	0x210d0002, 0x00010001,	/* cond: brb.init_done */
	0x210d0030, 0x00010001,	/* cond: brb.INT_STS_0 */
	0x210d0031, 0x00010001,	/* cond: brb.INT_MASK_0 */
	0x218d0600, 0x00000014,	/* info: brb.stopped_rd_req[0:4] width=3 access=WB mode=all */
	0x218d0640, 0x00000014,	/* info: brb.stopped_rls_req[0:4] width=3 access=WB mode=all */
	0x210d0036, 0x00010001,	/* cond: brb.INT_STS_1 */
	0x210d0037, 0x00010001,	/* cond: brb.INT_MASK_1 */
	0x210d003c, 0x00010001,	/* cond: brb.INT_STS_2 */
	0x210d003d, 0x00010001,	/* cond: brb.INT_MASK_2 */
	0x210d0042, 0x00010001,	/* cond: brb.INT_STS_3 */
	0x210d0043, 0x00010001,	/* cond: brb.INT_MASK_3 */
	0x210d0048, 0x00010001,	/* cond: brb.INT_STS_4 */
	0x210d0049, 0x00010001,	/* cond: brb.INT_MASK_4 */
	0x218d0600, 0x00000014,	/* info: brb.stopped_rd_req[0:4] width=3 access=WB mode=all */
	0x218d0640, 0x00000014,	/* info: brb.stopped_rls_req[0:4] width=3 access=WB mode=all */
	0x210d004e, 0x00010001,	/* cond: brb.INT_STS_5 */
	0x210d004f, 0x00010001,	/* cond: brb.INT_MASK_5 */
	0x210d0054, 0x00010001,	/* cond: brb.INT_STS_6 */
	0x210d0055, 0x00010001,	/* cond: brb.INT_MASK_6 */
	0x210d005a, 0x00010001,	/* cond: brb.INT_STS_7 */
	0x210d005b, 0x00010001,	/* cond: brb.INT_MASK_7 */
	0x210d0061, 0x00010001,	/* cond: brb.INT_STS_8 */
	0x210d0062, 0x00010001,	/* cond: brb.INT_MASK_8 */
	0x210d0067, 0x00010001,	/* cond: brb.INT_STS_9 */
	0x210d0068, 0x00010001,	/* cond: brb.INT_MASK_9 */
	0x210d006d, 0x00010001,	/* cond: brb.INT_STS_10 */
	0x210d006e, 0x00010001,	/* cond: brb.INT_MASK_10 */
	0x210d0073, 0x00010001,	/* cond: brb.INT_STS_11 */
	0x210d0074, 0x00010001,	/* cond: brb.INT_MASK_11 */
	0x210d0077, 0x00010001,	/* cond: brb.PRTY_STS */
	0x210d0078, 0x00010001,	/* cond: brb.PRTY_MASK */
	0x210d0100, 0x00010001,	/* cond: brb.PRTY_STS_H_0 */
	0x210d0101, 0x00010001,	/* cond: brb.PRTY_MASK_H_0 */
	0x210d0104, 0x00010001,	/* cond: brb.PRTY_STS_H_1 */
	0x210d0105, 0x00010001,	/* cond: brb.PRTY_MASK_H_1 */
	0x210d03ec, 0x00010001,	/* cond: brb.wc_bandwidth_if_full */
	0x210d03ed, 0x00010001,	/* cond: brb.rc_pkt_if_full */
	0x210d03ee, 0x00010005,	/* cond: brb.rc_pkt_empty_0[0:4] */
	0x210d041b, 0x00010001,	/* cond: brb.rc_sop_empty */
	0x210d0421, 0x00010001,	/* cond: brb.ll_arb_empty */
	0x210d0468, 0x00010001,	/* cond: brb.stop_packet_counter */
	0x210d0469, 0x00010001,	/* cond: brb.stop_byte_counter */
	0x210d046a, 0x00010001,	/* cond: brb.rc_pkt_state */
	0x210d0474, 0x00010001,	/* cond: brb.mac0_tc_occupancy_0 */
	0x210d0475, 0x00010001,	/* cond: brb.mac0_tc_occupancy_1 */
	0x210d0476, 0x00010001,	/* cond: brb.mac0_tc_occupancy_2 */
	0x210d0477, 0x00010001,	/* cond: brb.mac0_tc_occupancy_3 */
	0x210d0478, 0x00010001,	/* cond: brb.mac0_tc_occupancy_4 */
	0x210d0479, 0x00010001,	/* cond: brb.mac0_tc_occupancy_5 */
	0x210d047a, 0x00010001,	/* cond: brb.mac0_tc_occupancy_6 */
	0x210d047b, 0x00010001,	/* cond: brb.mac0_tc_occupancy_7 */
	0x210d047c, 0x00010001,	/* cond: brb.mac0_tc_occupancy_8 */
	0x210d0484, 0x00010001,	/* cond: brb.mac1_tc_occupancy_0 */
	0x210d0485, 0x00010001,	/* cond: brb.mac1_tc_occupancy_1 */
	0x210d0486, 0x00010001,	/* cond: brb.mac1_tc_occupancy_2 */
	0x210d0487, 0x00010001,	/* cond: brb.mac1_tc_occupancy_3 */
	0x210d0488, 0x00010001,	/* cond: brb.mac1_tc_occupancy_4 */
	0x210d0489, 0x00010001,	/* cond: brb.mac1_tc_occupancy_5 */
	0x210d048a, 0x00010001,	/* cond: brb.mac1_tc_occupancy_6 */
	0x210d048b, 0x00010001,	/* cond: brb.mac1_tc_occupancy_7 */
	0x210d048c, 0x00010001,	/* cond: brb.mac1_tc_occupancy_8 */
	0x34130017, 0x00010001,	/* cond: xyld.pending_msg_to_ext_ev_1_ctr */
	0x34130018, 0x00010001,	/* cond: xyld.pending_msg_to_ext_ev_2_ctr */
	0x34130019, 0x00010001,	/* cond: xyld.pending_msg_to_ext_ev_3_ctr */
	0x3413001a, 0x00010001,	/* cond: xyld.pending_msg_to_ext_ev_4_ctr */
	0x3413001b, 0x00010001,	/* cond: xyld.pending_msg_to_ext_ev_5_ctr */
	0x3413001c, 0x00010001,	/* cond: xyld.foc_remain_credits */
	0x34130003, 0x00010001,	/* cond: xyld.foci_foc_credits */
	0x3413001f, 0x00010001,	/* cond: xyld.pci_pending_msg_ctr */
	0x34130039, 0x00010001,	/* cond: xyld.dbg_pending_ccfc_req */
	0x3413003a, 0x00010001,	/* cond: xyld.dbg_pending_tcfc_req */
	0x34130060, 0x00010001,	/* cond: xyld.INT_STS */
	0x34130061, 0x00010001,	/* cond: xyld.INT_MASK */
	0x34130080, 0x00010001,	/* cond: xyld.PRTY_STS_H_0 */
	0x34130081, 0x00010001,	/* cond: xyld.PRTY_MASK_H_0 */
	0x33132013, 0x00010001,	/* cond: yuld.pending_msg_to_ext_ev_1_ctr */
	0x33132014, 0x00010001,	/* cond: yuld.pending_msg_to_ext_ev_2_ctr */
	0x33132015, 0x00010001,	/* cond: yuld.pending_msg_to_ext_ev_3_ctr */
	0x33132016, 0x00010001,	/* cond: yuld.pending_msg_to_ext_ev_4_ctr */
	0x33132017, 0x00010001,	/* cond: yuld.pending_msg_to_ext_ev_5_ctr */
	0x33132018, 0x00010001,	/* cond: yuld.foc_remain_credits */
	0x33132003, 0x00010001,	/* cond: yuld.foci_foc_credits */
	0x3313202c, 0x00010001,	/* cond: yuld.dbg_pending_ccfc_req */
	0x3313202d, 0x00010001,	/* cond: yuld.dbg_pending_tcfc_req */
	0x33132080, 0x00010001,	/* cond: yuld.PRTY_STS_H_0 */
	0x33132081, 0x00010001,	/* cond: yuld.PRTY_MASK_H_0 */
	0x31134016, 0x00010001,	/* cond: tmld.pending_msg_to_ext_ev_1_ctr */
	0x31134017, 0x00010001,	/* cond: tmld.pending_msg_to_ext_ev_2_ctr */
	0x31134018, 0x00010001,	/* cond: tmld.pending_msg_to_ext_ev_3_ctr */
	0x31134019, 0x00010001,	/* cond: tmld.pending_msg_to_ext_ev_4_ctr */
	0x3113401a, 0x00010001,	/* cond: tmld.pending_msg_to_ext_ev_5_ctr */
	0x3113401b, 0x00010001,	/* cond: tmld.foc_remain_credits */
	0x31134003, 0x00010001,	/* cond: tmld.foci_foc_credits */
	0x31134031, 0x00010001,	/* cond: tmld.dbg_pending_ccfc_req */
	0x31134032, 0x00010001,	/* cond: tmld.dbg_pending_tcfc_req */
	0x31134060, 0x00010001,	/* cond: tmld.INT_STS */
	0x31134061, 0x00010001,	/* cond: tmld.INT_MASK */
	0x31134080, 0x00010001,	/* cond: tmld.PRTY_STS_H_0 */
	0x31134081, 0x00010001,	/* cond: tmld.PRTY_MASK_H_0 */
	0x3213801c, 0x00010001,	/* cond: muld.pending_msg_to_ext_ev_1_ctr */
	0x3213801d, 0x00010001,	/* cond: muld.pending_msg_to_ext_ev_2_ctr */
	0x3213801e, 0x00010001,	/* cond: muld.pending_msg_to_ext_ev_3_ctr */
	0x3213801f, 0x00010001,	/* cond: muld.pending_msg_to_ext_ev_4_ctr */
	0x32138020, 0x00010001,	/* cond: muld.pending_msg_to_ext_ev_5_ctr */
	0x32138021, 0x00010001,	/* cond: muld.foc_remain_credits */
	0x32138009, 0x00010001,	/* cond: muld.foci_foc_credits */
	0x32138022, 0x00010001,	/* cond: muld.bd_pending_msg_ctr */
	0x32138023, 0x00010001,	/* cond: muld.sge_pending_msg_ctr */
	0x32138026, 0x00010001,	/* cond: muld.pci_pending_msg_ctr */
	0x3213803a, 0x00010001,	/* cond: muld.dbg_pending_ccfc_req */
	0x3213803b, 0x00010001,	/* cond: muld.dbg_pending_tcfc_req */
	0x32138060, 0x00010001,	/* cond: muld.INT_STS */
	0x32138061, 0x00010001,	/* cond: muld.INT_MASK */
	0x32138080, 0x00010001,	/* cond: muld.PRTY_STS_H_0 */
	0x32138081, 0x00010001,	/* cond: muld.PRTY_MASK_H_0 */
	0x46140010, 0x00010001,	/* cond: nig.INT_STS_0 */
	0x46140011, 0x00010001,	/* cond: nig.INT_MASK_0 */
	0x46140014, 0x00010001,	/* cond: nig.INT_STS_1 */
	0x46140015, 0x00010001,	/* cond: nig.INT_MASK_1 */
	0x46140018, 0x00010001,	/* cond: nig.INT_STS_2 */
	0x46140019, 0x00010001,	/* cond: nig.INT_MASK_2 */
	0x4614001c, 0x00010001,	/* cond: nig.INT_STS_3 */
	0x4614001d, 0x00010001,	/* cond: nig.INT_MASK_3 */
	0x461406e8, 0x00000001,	/* info: nig.flowctrl_mode mode=all */
	0x46140713, 0x00000001,	/* info: nig.rx_flowctrl_status mode=all */
	0x46140020, 0x00010001,	/* cond: nig.INT_STS_4 */
	0x46140021, 0x00010001,	/* cond: nig.INT_MASK_4 */
	0x46140024, 0x00010001,	/* cond: nig.INT_STS_5 */
	0x46140025, 0x00010001,	/* cond: nig.INT_MASK_5 */
	0x461406e8, 0x00000001,	/* info: nig.flowctrl_mode mode=all */
	0x46140713, 0x00000001,	/* info: nig.rx_flowctrl_status mode=all */
	0x46140080, 0x00010001,	/* cond: nig.PRTY_STS_H_0 */
	0x46140081, 0x00010001,	/* cond: nig.PRTY_MASK_H_0 */
	0x46140084, 0x00010001,	/* cond: nig.PRTY_STS_H_1 */
	0x46140085, 0x00010001,	/* cond: nig.PRTY_MASK_H_1 */
	0x46140088, 0x00010001,	/* cond: nig.PRTY_STS_H_2 */
	0x46140089, 0x00010001,	/* cond: nig.PRTY_MASK_H_2 */
	0x4614008c, 0x00010001,	/* cond: nig.PRTY_STS_H_3 */
	0x4614008d, 0x00010001,	/* cond: nig.PRTY_MASK_H_3 */
	0x46140301, 0x00010001,	/* cond: nig.lb_sopq_empty */
	0x46140303, 0x00010001,	/* cond: nig.tx_sopq_empty */
	0x46140394, 0x00010001,	/* cond: nig.tx_macfifo_empty */
	0x461404c2, 0x00010001,	/* cond: nig.rx_llh_dfifo_empty */
	0x461404c3, 0x00000001,	/* info: nig.rx_llh_dfifo_full mode=all */
	0x461404c4, 0x00010001,	/* cond: nig.rx_llh_hfifo_empty */
	0x461404c5, 0x00000001,	/* info: nig.rx_llh_hfifo_full mode=all */
	0x461404c6, 0x00010001,	/* cond: nig.rx_llh_rfifo_empty */
	0x461404c6, 0x00010001,	/* cond: nig.rx_llh_rfifo_empty */
	0x461404c7, 0x00000001,	/* info: nig.rx_llh_rfifo_full mode=all */
	0x46140500, 0x00010001,	/* cond: nig.storm_status */
	0x4614020f, 0x00000001,	/* info: nig.storm_credit mode=all */
	0x461405db, 0x00010001,	/* cond: nig.lb_btb_fifo_empty */
	0x461405e0, 0x00010001,	/* cond: nig.lb_llh_dfifo_empty */
	0x461405e2, 0x00000001,	/* info: nig.lb_llh_dfifo_full mode=all */
	0x461405e3, 0x00010001,	/* cond: nig.lb_llh_hfifo_empty */
	0x461405e5, 0x00000001,	/* info: nig.lb_llh_hfifo_full mode=all */
	0x461405e6, 0x00010001,	/* cond: nig.lb_llh_rfifo_empty */
	0x461405e6, 0x00010001,	/* cond: nig.lb_llh_rfifo_empty */
	0x46140600, 0x00000001,	/* info: nig.lb_llh_rfifo_full mode=all */
	0x46140655, 0x00010001,	/* cond: nig.rx_ptp_ts_msb_err */
	0x46140640, 0x00010001,	/* cond: nig.rx_ptp_en */
	0x461407ff, 0x00010001,	/* cond: nig.tx_btb_fifo_empty */
	0x46140804, 0x00010001,	/* cond: nig.tx_llh_dfifo_empty */
	0x46140806, 0x00000001,	/* info: nig.tx_llh_dfifo_full mode=all */
	0x46140807, 0x00010001,	/* cond: nig.tx_llh_hfifo_empty */
	0x46140809, 0x00000001,	/* info: nig.tx_llh_hfifo_full mode=all */
	0x4614080a, 0x00010001,	/* cond: nig.tx_llh_rfifo_empty */
	0x4614080c, 0x00000001,	/* info: nig.tx_llh_rfifo_full mode=all */
	0x4614080d, 0x00010001,	/* cond: nig.tx_gnt_fifo_empty */
	0x4614080e, 0x00000001,	/* info: nig.tx_gnt_fifo_full mode=all */
	0x46140831, 0x00010001,	/* cond: nig.dorq_fifo_empty */
	0x46140832, 0x00000001,	/* info: nig.dorq_fifo_full mode=all */
	0x46140838, 0x00010001,	/* cond: nig.debug_fifo_empty */
	0x461422d9, 0x00010001,	/* cond: nig.lb_gnt_fifo_empty */
	0x461422da, 0x00000001,	/* info: nig.lb_gnt_fifo_full mode=all */
	0x17158001, 0x00010001,	/* cond: ptu.atc_init_done */
	0x1715801c, 0x00010001,	/* cond: ptu.pxp_err_ctr */
	0x1715801d, 0x00010001,	/* cond: ptu.inv_err_ctr */
	0x17158032, 0x00010001,	/* cond: ptu.pbf_fill_level */
	0x17158033, 0x00010001,	/* cond: ptu.prm_fill_level */
	0x17158060, 0x00010001,	/* cond: ptu.INT_STS */
	0x17158061, 0x00010001,	/* cond: ptu.INT_MASK */
	0x17158080, 0x00010001,	/* cond: ptu.PRTY_STS_H_0 */
	0x17158081, 0x00010001,	/* cond: ptu.PRTY_MASK_H_0 */
	0x3d160070, 0x00010001,	/* cond: cdu.INT_STS */
	0x3d160073, 0x00010001,	/* cond: cdu.INT_MASK */
	0x3d160200, 0x00000001,	/* info: cdu.ccfc_cvld_error_data mode=all */
	0x3d160201, 0x00000001,	/* info: cdu.tcfc_cvld_error_data mode=all */
	0x3d160202, 0x00000001,	/* info: cdu.ccfc_ld_l1_num_error_data mode=all */
	0x3d160203, 0x00000001,	/* info: cdu.tcfc_ld_l1_num_error_data mode=all */
	0x3d160204, 0x00000001,	/* info: cdu.ccfc_wb_l1_num_error_data mode=all */
	0x3d160205, 0x00000001,	/* info: cdu.tcfc_wb_l1_num_error_data mode=all */
	0x3d160080, 0x00010001,	/* cond: cdu.PRTY_STS_H_0 */
	0x3d160081, 0x00010001,	/* cond: cdu.PRTY_MASK_H_0 */
	0x3a360060, 0x00010001,	/* cond: pbf.INT_STS */
	0x3a360061, 0x00010001,	/* cond: pbf.INT_MASK */
	0x3a360064, 0x00010001,	/* cond: pbf.PRTY_STS */
	0x3a360065, 0x00010001,	/* cond: pbf.PRTY_MASK */
	0x3a360080, 0x00010001,	/* cond: pbf.PRTY_STS_H_0 */
	0x3a360081, 0x00010001,	/* cond: pbf.PRTY_MASK_H_0 */
	0x3a360084, 0x00010001,	/* cond: pbf.PRTY_STS_H_1 */
	0x3a360085, 0x00010001,	/* cond: pbf.PRTY_MASK_H_1 */
	0x3a36019a, 0x00010001,	/* cond: pbf.num_pkts_received_with_error */
	0x3a36019b, 0x00010001,	/* cond: pbf.num_pkts_sent_with_error_to_btb */
	0x3a36019c, 0x00010001,	/* cond: pbf.num_pkts_sent_with_drop_to_btb */
	0x3a3601ac, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq0 */
	0x3a3601ae, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq0 */
	0x3a3601b1, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq0 */
	0x3a3601bc, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq1 */
	0x3a3601be, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq1 */
	0x3a3601c1, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq1 */
	0x3a3601cc, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq2 */
	0x3a3601ce, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq2 */
	0x3a3601d1, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq2 */
	0x3a3601dc, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq3 */
	0x3a3601de, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq3 */
	0x3a3601e1, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq3 */
	0x3a3601ec, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq4 */
	0x3a3601ee, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq4 */
	0x3a3601f1, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq4 */
	0x3a3601fc, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq5 */
	0x3a3601fe, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq5 */
	0x3a360201, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq5 */
	0x3a36020c, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq6 */
	0x3a36020e, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq6 */
	0x3a360211, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq6 */
	0x3a36021c, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq7 */
	0x3a36021e, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq7 */
	0x3a360221, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq7 */
	0x3a36022c, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq8 */
	0x3a36022e, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq8 */
	0x3a360231, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq8 */
	0x3a36023c, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq9 */
	0x3a36023e, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq9 */
	0x3a360241, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq9 */
	0x3a36024c, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq10 */
	0x3a36024e, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq10 */
	0x3a360251, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq10 */
	0x3a36025c, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq11 */
	0x3a36025e, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq11 */
	0x3a360261, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq11 */
	0x3a36026c, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq12 */
	0x3a36026e, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq12 */
	0x3a360271, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq12 */
	0x3a36027c, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq13 */
	0x3a36027e, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq13 */
	0x3a360281, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq13 */
	0x3a36028c, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq14 */
	0x3a36028e, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq14 */
	0x3a360291, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq14 */
	0x3a36029c, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq15 */
	0x3a36029e, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq15 */
	0x3a3602a1, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq15 */
	0x3a3602ac, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq16 */
	0x3a3602ae, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq16 */
	0x3a3602b1, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq16 */
	0x3a3602bc, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq17 */
	0x3a3602be, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq17 */
	0x3a3602c1, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq17 */
	0x3a3602cc, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq18 */
	0x3a3602ce, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq18 */
	0x3a3602d1, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq18 */
	0x3a3602dc, 0x00010001,	/* cond: pbf.ycmd_qs_cmd_cnt_voq19 */
	0x3a3602de, 0x00010001,	/* cond: pbf.ycmd_qs_occupancy_voq19 */
	0x3a3602e1, 0x00010001,	/* cond: pbf.btb_allocated_blocks_voq19 */
	0x36368010, 0x00010001,	/* cond: pbf_pb1.INT_STS */
	0x36368011, 0x00010001,	/* cond: pbf_pb1.INT_MASK */
	0x36368014, 0x00010001,	/* cond: pbf_pb1.PRTY_STS */
	0x36368015, 0x00010001,	/* cond: pbf_pb1.PRTY_MASK */
	0x36368140, 0x00010001,	/* cond: pbf_pb1.db_empty */
	0x36368141, 0x00000001,	/* info: pbf_pb1.db_full mode=all */
	0x36368142, 0x00010001,	/* cond: pbf_pb1.tq_empty */
	0x36368143, 0x00000001,	/* info: pbf_pb1.tq_full mode=all */
	0x36368144, 0x00010001,	/* cond: pbf_pb1.ififo_empty */
	0x36368145, 0x00000001,	/* info: pbf_pb1.ififo_full mode=all */
	0x36368146, 0x00010001,	/* cond: pbf_pb1.pfifo_empty */
	0x36368147, 0x00000001,	/* info: pbf_pb1.pfifo_full mode=all */
	0x36368148, 0x00010001,	/* cond: pbf_pb1.tq_th_empty */
	0x36368181, 0x00010001,	/* cond: pbf_pb1.errored_instr */
	0x36368182, 0x00000001,	/* info: pbf_pb1.errored_hdr_low mode=all */
	0x36368183, 0x00000001,	/* info: pbf_pb1.errored_hdr_high mode=all */
	0x36368184, 0x00000001,	/* info: pbf_pb1.errored_length mode=all */
	0x37369010, 0x00010001,	/* cond: pbf_pb2.INT_STS */
	0x37369011, 0x00010001,	/* cond: pbf_pb2.INT_MASK */
	0x37369014, 0x00010001,	/* cond: pbf_pb2.PRTY_STS */
	0x37369015, 0x00010001,	/* cond: pbf_pb2.PRTY_MASK */
	0x37369140, 0x00010001,	/* cond: pbf_pb2.db_empty */
	0x37369141, 0x00000001,	/* info: pbf_pb2.db_full mode=all */
	0x37369142, 0x00010001,	/* cond: pbf_pb2.tq_empty */
	0x37369143, 0x00000001,	/* info: pbf_pb2.tq_full mode=all */
	0x37369144, 0x00010001,	/* cond: pbf_pb2.ififo_empty */
	0x37369145, 0x00000001,	/* info: pbf_pb2.ififo_full mode=all */
	0x37369146, 0x00010001,	/* cond: pbf_pb2.pfifo_empty */
	0x37369147, 0x00000001,	/* info: pbf_pb2.pfifo_full mode=all */
	0x37369148, 0x00010001,	/* cond: pbf_pb2.tq_th_empty */
	0x37369181, 0x00010001,	/* cond: pbf_pb2.errored_instr */
	0x37369182, 0x00000001,	/* info: pbf_pb2.errored_hdr_low mode=all */
	0x37369183, 0x00000001,	/* info: pbf_pb2.errored_hdr_high mode=all */
	0x37369184, 0x00000001,	/* info: pbf_pb2.errored_length mode=all */
	0x3936c002, 0x00010001,	/* cond: btb.init_done */
	0x3936c036, 0x00010001,	/* cond: btb.INT_STS_1 */
	0x3936c037, 0x00010001,	/* cond: btb.INT_MASK_1 */
	0x39b6c480, 0x00000004,	/* info: btb.wc_status_0 width=3 access=WB mode=all */
	0x3936c03c, 0x00010001,	/* cond: btb.INT_STS_2 */
	0x3936c03d, 0x00010001,	/* cond: btb.INT_MASK_2 */
	0x39b6c480, 0x00000004,	/* info: btb.wc_status_0 width=3 access=WB mode=all */
	0x3936c042, 0x00010001,	/* cond: btb.INT_STS_3 */
	0x3936c043, 0x00010001,	/* cond: btb.INT_MASK_3 */
	0x3936c04e, 0x00010001,	/* cond: btb.INT_STS_5 */
	0x3936c04f, 0x00010001,	/* cond: btb.INT_MASK_5 */
	0x3936c054, 0x00010001,	/* cond: btb.INT_STS_6 */
	0x3936c055, 0x00010001,	/* cond: btb.INT_MASK_6 */
	0x3936c061, 0x00010001,	/* cond: btb.INT_STS_8 */
	0x3936c062, 0x00010001,	/* cond: btb.INT_MASK_8 */
	0x39b6c480, 0x00000004,	/* info: btb.wc_status_0 width=3 access=WB mode=all */
	0x3936c067, 0x00010001,	/* cond: btb.INT_STS_9 */
	0x3936c068, 0x00010001,	/* cond: btb.INT_MASK_9 */
	0x39b6c480, 0x00000004,	/* info: btb.wc_status_0 width=3 access=WB mode=all */
	0x3936c06d, 0x00010001,	/* cond: btb.INT_STS_10 */
	0x3936c06e, 0x00010001,	/* cond: btb.INT_MASK_10 */
	0x39b6c480, 0x00000004,	/* info: btb.wc_status_0 width=3 access=WB mode=all */
	0x3936c073, 0x00010001,	/* cond: btb.INT_STS_11 */
	0x3936c074, 0x00010001,	/* cond: btb.INT_MASK_11 */
	0x3936c077, 0x00010001,	/* cond: btb.PRTY_STS */
	0x3936c078, 0x00010001,	/* cond: btb.PRTY_MASK */
	0x3936c100, 0x00010001,	/* cond: btb.PRTY_STS_H_0 */
	0x3936c101, 0x00010001,	/* cond: btb.PRTY_MASK_H_0 */
	0x3936c244, 0x00010001,	/* cond: btb.wc_dup_empty */
	0x3936c246, 0x00010001,	/* cond: btb.wc_dup_status */
	0x3936c247, 0x00010001,	/* cond: btb.wc_empty_0 */
	0x3936c267, 0x00010001,	/* cond: btb.wc_bandwidth_if_full */
	0x3936c268, 0x00010001,	/* cond: btb.rc_pkt_if_full */
	0x3936c269, 0x00010001,	/* cond: btb.rc_pkt_empty_0 */
	0x3936c26a, 0x00010001,	/* cond: btb.rc_pkt_empty_1 */
	0x3936c26b, 0x00010001,	/* cond: btb.rc_pkt_empty_2 */
	0x3936c26c, 0x00010001,	/* cond: btb.rc_pkt_empty_3 */
	0x3936c296, 0x00010001,	/* cond: btb.rc_sop_empty */
	0x3936c299, 0x00010001,	/* cond: btb.ll_arb_empty */
	0x3936c29c, 0x00010001,	/* cond: btb.block_occupancy */
	0x3936c2ae, 0x00010001,	/* cond: btb.rc_pkt_state */
	0x39b6c480, 0x00030001,	/* cond: btb.wc_status_0 width=3 access=WB */
	0x273e0010, 0x00010001,	/* cond: xsdm.INT_STS */
	0x273e0011, 0x00010001,	/* cond: xsdm.INT_MASK */
	0x273e0109, 0x00000001,	/* info: xsdm.inp_queue_err_vect mode=all */
	0x273e0080, 0x00010001,	/* cond: xsdm.PRTY_STS_H_0 */
	0x273e0081, 0x00010001,	/* cond: xsdm.PRTY_MASK_H_0 */
	0x273e0303, 0x00010001,	/* cond: xsdm.qm_full */
	0x273e030c, 0x00010001,	/* cond: xsdm.rsp_brb_if_full */
	0x273e030d, 0x00010001,	/* cond: xsdm.rsp_pxp_if_full */
	0x273e0316, 0x00010001,	/* cond: xsdm.dst_pxp_if_full */
	0x273e0317, 0x00010001,	/* cond: xsdm.dst_int_ram_if_full */
	0x273e0318, 0x00010001,	/* cond: xsdm.dst_pas_buf_if_full */
	0x273e0340, 0x00010001,	/* cond: xsdm.int_cmpl_pend_empty */
	0x273e0341, 0x00010001,	/* cond: xsdm.int_cprm_pend_empty */
	0x273e0342, 0x00010001,	/* cond: xsdm.queue_empty */
	0x273e0343, 0x00010001,	/* cond: xsdm.delay_fifo_empty */
	0x273e0346, 0x00010001,	/* cond: xsdm.rsp_pxp_rdata_empty */
	0x273e0347, 0x00010001,	/* cond: xsdm.rsp_brb_rdata_empty */
	0x273e0348, 0x00010001,	/* cond: xsdm.rsp_int_ram_rdata_empty */
	0x273e0349, 0x00010001,	/* cond: xsdm.rsp_brb_pend_empty */
	0x273e034a, 0x00010001,	/* cond: xsdm.rsp_int_ram_pend_empty */
	0x273e034b, 0x00010001,	/* cond: xsdm.dst_pxp_immed_empty */
	0x273e034c, 0x00010001,	/* cond: xsdm.dst_pxp_dst_pend_empty */
	0x273e034d, 0x00010001,	/* cond: xsdm.dst_pxp_src_pend_empty */
	0x273e034e, 0x00010001,	/* cond: xsdm.dst_brb_src_pend_empty */
	0x273e034f, 0x00010001,	/* cond: xsdm.dst_brb_src_addr_empty */
	0x273e0350, 0x00010001,	/* cond: xsdm.dst_pxp_link_empty */
	0x273e0351, 0x00010001,	/* cond: xsdm.dst_int_ram_wait_empty */
	0x273e0352, 0x00010001,	/* cond: xsdm.dst_pas_buf_wait_empty */
	0x273e0353, 0x00010001,	/* cond: xsdm.sh_delay_empty */
	0x273e0354, 0x00010001,	/* cond: xsdm.cm_delay_empty */
	0x273e0355, 0x00010001,	/* cond: xsdm.cmsg_que_empty */
	0x273e0356, 0x00010001,	/* cond: xsdm.ccfc_load_pend_empty */
	0x273e0357, 0x00010001,	/* cond: xsdm.tcfc_load_pend_empty */
	0x273e0358, 0x00010001,	/* cond: xsdm.async_host_empty */
	0x273e0359, 0x00010001,	/* cond: xsdm.prm_fifo_empty */
	0x283e4010, 0x00010001,	/* cond: ysdm.INT_STS */
	0x283e4011, 0x00010001,	/* cond: ysdm.INT_MASK */
	0x283e4109, 0x00000001,	/* info: ysdm.inp_queue_err_vect mode=all */
	0x283e4080, 0x00010001,	/* cond: ysdm.PRTY_STS_H_0 */
	0x283e4081, 0x00010001,	/* cond: ysdm.PRTY_MASK_H_0 */
	0x283e4303, 0x00010001,	/* cond: ysdm.qm_full */
	0x283e430c, 0x00010001,	/* cond: ysdm.rsp_brb_if_full */
	0x283e430d, 0x00010001,	/* cond: ysdm.rsp_pxp_if_full */
	0x283e4316, 0x00010001,	/* cond: ysdm.dst_pxp_if_full */
	0x283e4317, 0x00010001,	/* cond: ysdm.dst_int_ram_if_full */
	0x283e4318, 0x00010001,	/* cond: ysdm.dst_pas_buf_if_full */
	0x283e4340, 0x00010001,	/* cond: ysdm.int_cmpl_pend_empty */
	0x283e4341, 0x00010001,	/* cond: ysdm.int_cprm_pend_empty */
	0x283e4342, 0x00010001,	/* cond: ysdm.queue_empty */
	0x283e4343, 0x00010001,	/* cond: ysdm.delay_fifo_empty */
	0x283e4346, 0x00010001,	/* cond: ysdm.rsp_pxp_rdata_empty */
	0x283e4347, 0x00010001,	/* cond: ysdm.rsp_brb_rdata_empty */
	0x283e4348, 0x00010001,	/* cond: ysdm.rsp_int_ram_rdata_empty */
	0x283e4349, 0x00010001,	/* cond: ysdm.rsp_brb_pend_empty */
	0x283e434a, 0x00010001,	/* cond: ysdm.rsp_int_ram_pend_empty */
	0x283e434b, 0x00010001,	/* cond: ysdm.dst_pxp_immed_empty */
	0x283e434c, 0x00010001,	/* cond: ysdm.dst_pxp_dst_pend_empty */
	0x283e434d, 0x00010001,	/* cond: ysdm.dst_pxp_src_pend_empty */
	0x283e434e, 0x00010001,	/* cond: ysdm.dst_brb_src_pend_empty */
	0x283e434f, 0x00010001,	/* cond: ysdm.dst_brb_src_addr_empty */
	0x283e4350, 0x00010001,	/* cond: ysdm.dst_pxp_link_empty */
	0x283e4351, 0x00010001,	/* cond: ysdm.dst_int_ram_wait_empty */
	0x283e4352, 0x00010001,	/* cond: ysdm.dst_pas_buf_wait_empty */
	0x283e4353, 0x00010001,	/* cond: ysdm.sh_delay_empty */
	0x283e4354, 0x00010001,	/* cond: ysdm.cm_delay_empty */
	0x283e4355, 0x00010001,	/* cond: ysdm.cmsg_que_empty */
	0x283e4356, 0x00010001,	/* cond: ysdm.ccfc_load_pend_empty */
	0x283e4357, 0x00010001,	/* cond: ysdm.tcfc_load_pend_empty */
	0x283e4358, 0x00010001,	/* cond: ysdm.async_host_empty */
	0x283e4359, 0x00010001,	/* cond: ysdm.prm_fifo_empty */
	0x293e8010, 0x00010001,	/* cond: psdm.INT_STS */
	0x293e8011, 0x00010001,	/* cond: psdm.INT_MASK */
	0x293e8109, 0x00000001,	/* info: psdm.inp_queue_err_vect mode=all */
	0x293e8080, 0x00010001,	/* cond: psdm.PRTY_STS_H_0 */
	0x293e8081, 0x00010001,	/* cond: psdm.PRTY_MASK_H_0 */
	0x293e8303, 0x00010001,	/* cond: psdm.qm_full */
	0x293e830c, 0x00010001,	/* cond: psdm.rsp_brb_if_full */
	0x293e830d, 0x00010001,	/* cond: psdm.rsp_pxp_if_full */
	0x293e8316, 0x00010001,	/* cond: psdm.dst_pxp_if_full */
	0x293e8317, 0x00010001,	/* cond: psdm.dst_int_ram_if_full */
	0x293e8318, 0x00010001,	/* cond: psdm.dst_pas_buf_if_full */
	0x293e8340, 0x00010001,	/* cond: psdm.int_cmpl_pend_empty */
	0x293e8341, 0x00010001,	/* cond: psdm.int_cprm_pend_empty */
	0x293e8342, 0x00010001,	/* cond: psdm.queue_empty */
	0x293e8343, 0x00010001,	/* cond: psdm.delay_fifo_empty */
	0x293e8346, 0x00010001,	/* cond: psdm.rsp_pxp_rdata_empty */
	0x293e8347, 0x00010001,	/* cond: psdm.rsp_brb_rdata_empty */
	0x293e8348, 0x00010001,	/* cond: psdm.rsp_int_ram_rdata_empty */
	0x293e8349, 0x00010001,	/* cond: psdm.rsp_brb_pend_empty */
	0x293e834a, 0x00010001,	/* cond: psdm.rsp_int_ram_pend_empty */
	0x293e834b, 0x00010001,	/* cond: psdm.dst_pxp_immed_empty */
	0x293e834c, 0x00010001,	/* cond: psdm.dst_pxp_dst_pend_empty */
	0x293e834d, 0x00010001,	/* cond: psdm.dst_pxp_src_pend_empty */
	0x293e834e, 0x00010001,	/* cond: psdm.dst_brb_src_pend_empty */
	0x293e834f, 0x00010001,	/* cond: psdm.dst_brb_src_addr_empty */
	0x293e8350, 0x00010001,	/* cond: psdm.dst_pxp_link_empty */
	0x293e8351, 0x00010001,	/* cond: psdm.dst_int_ram_wait_empty */
	0x293e8352, 0x00010001,	/* cond: psdm.dst_pas_buf_wait_empty */
	0x293e8353, 0x00010001,	/* cond: psdm.sh_delay_empty */
	0x293e8354, 0x00010001,	/* cond: psdm.cm_delay_empty */
	0x293e8355, 0x00010001,	/* cond: psdm.cmsg_que_empty */
	0x293e8356, 0x00010001,	/* cond: psdm.ccfc_load_pend_empty */
	0x293e8357, 0x00010001,	/* cond: psdm.tcfc_load_pend_empty */
	0x293e8358, 0x00010001,	/* cond: psdm.async_host_empty */
	0x293e8359, 0x00010001,	/* cond: psdm.prm_fifo_empty */
	0x243ec010, 0x00010001,	/* cond: tsdm.INT_STS */
	0x243ec011, 0x00010001,	/* cond: tsdm.INT_MASK */
	0x243ec109, 0x00000001,	/* info: tsdm.inp_queue_err_vect mode=all */
	0x243ec080, 0x00010001,	/* cond: tsdm.PRTY_STS_H_0 */
	0x243ec081, 0x00010001,	/* cond: tsdm.PRTY_MASK_H_0 */
	0x243ec303, 0x00010001,	/* cond: tsdm.qm_full */
	0x243ec30c, 0x00010001,	/* cond: tsdm.rsp_brb_if_full */
	0x243ec30d, 0x00010001,	/* cond: tsdm.rsp_pxp_if_full */
	0x243ec316, 0x00010001,	/* cond: tsdm.dst_pxp_if_full */
	0x243ec317, 0x00010001,	/* cond: tsdm.dst_int_ram_if_full */
	0x243ec318, 0x00010001,	/* cond: tsdm.dst_pas_buf_if_full */
	0x243ec340, 0x00010001,	/* cond: tsdm.int_cmpl_pend_empty */
	0x243ec341, 0x00010001,	/* cond: tsdm.int_cprm_pend_empty */
	0x243ec342, 0x00010001,	/* cond: tsdm.queue_empty */
	0x243ec343, 0x00010001,	/* cond: tsdm.delay_fifo_empty */
	0x243ec346, 0x00010001,	/* cond: tsdm.rsp_pxp_rdata_empty */
	0x243ec347, 0x00010001,	/* cond: tsdm.rsp_brb_rdata_empty */
	0x243ec348, 0x00010001,	/* cond: tsdm.rsp_int_ram_rdata_empty */
	0x243ec349, 0x00010001,	/* cond: tsdm.rsp_brb_pend_empty */
	0x243ec34a, 0x00010001,	/* cond: tsdm.rsp_int_ram_pend_empty */
	0x243ec34b, 0x00010001,	/* cond: tsdm.dst_pxp_immed_empty */
	0x243ec34c, 0x00010001,	/* cond: tsdm.dst_pxp_dst_pend_empty */
	0x243ec34d, 0x00010001,	/* cond: tsdm.dst_pxp_src_pend_empty */
	0x243ec34e, 0x00010001,	/* cond: tsdm.dst_brb_src_pend_empty */
	0x243ec34f, 0x00010001,	/* cond: tsdm.dst_brb_src_addr_empty */
	0x243ec350, 0x00010001,	/* cond: tsdm.dst_pxp_link_empty */
	0x243ec351, 0x00010001,	/* cond: tsdm.dst_int_ram_wait_empty */
	0x243ec352, 0x00010001,	/* cond: tsdm.dst_pas_buf_wait_empty */
	0x243ec353, 0x00010001,	/* cond: tsdm.sh_delay_empty */
	0x243ec354, 0x00010001,	/* cond: tsdm.cm_delay_empty */
	0x243ec355, 0x00010001,	/* cond: tsdm.cmsg_que_empty */
	0x243ec356, 0x00010001,	/* cond: tsdm.ccfc_load_pend_empty */
	0x243ec357, 0x00010001,	/* cond: tsdm.tcfc_load_pend_empty */
	0x243ec358, 0x00010001,	/* cond: tsdm.async_host_empty */
	0x243ec359, 0x00010001,	/* cond: tsdm.prm_fifo_empty */
	0x253f0010, 0x00010001,	/* cond: msdm.INT_STS */
	0x253f0011, 0x00010001,	/* cond: msdm.INT_MASK */
	0x253f0109, 0x00000001,	/* info: msdm.inp_queue_err_vect mode=all */
	0x253f0080, 0x00010001,	/* cond: msdm.PRTY_STS_H_0 */
	0x253f0081, 0x00010001,	/* cond: msdm.PRTY_MASK_H_0 */
	0x253f0303, 0x00010001,	/* cond: msdm.qm_full */
	0x253f030c, 0x00010001,	/* cond: msdm.rsp_brb_if_full */
	0x253f030d, 0x00010001,	/* cond: msdm.rsp_pxp_if_full */
	0x253f0316, 0x00010001,	/* cond: msdm.dst_pxp_if_full */
	0x253f0317, 0x00010001,	/* cond: msdm.dst_int_ram_if_full */
	0x253f0318, 0x00010001,	/* cond: msdm.dst_pas_buf_if_full */
	0x253f0340, 0x00010001,	/* cond: msdm.int_cmpl_pend_empty */
	0x253f0341, 0x00010001,	/* cond: msdm.int_cprm_pend_empty */
	0x253f0342, 0x00010001,	/* cond: msdm.queue_empty */
	0x253f0343, 0x00010001,	/* cond: msdm.delay_fifo_empty */
	0x253f0346, 0x00010001,	/* cond: msdm.rsp_pxp_rdata_empty */
	0x253f0347, 0x00010001,	/* cond: msdm.rsp_brb_rdata_empty */
	0x253f0348, 0x00010001,	/* cond: msdm.rsp_int_ram_rdata_empty */
	0x253f0349, 0x00010001,	/* cond: msdm.rsp_brb_pend_empty */
	0x253f034a, 0x00010001,	/* cond: msdm.rsp_int_ram_pend_empty */
	0x253f034b, 0x00010001,	/* cond: msdm.dst_pxp_immed_empty */
	0x253f034c, 0x00010001,	/* cond: msdm.dst_pxp_dst_pend_empty */
	0x253f034d, 0x00010001,	/* cond: msdm.dst_pxp_src_pend_empty */
	0x253f034e, 0x00010001,	/* cond: msdm.dst_brb_src_pend_empty */
	0x253f034f, 0x00010001,	/* cond: msdm.dst_brb_src_addr_empty */
	0x253f0350, 0x00010001,	/* cond: msdm.dst_pxp_link_empty */
	0x253f0351, 0x00010001,	/* cond: msdm.dst_int_ram_wait_empty */
	0x253f0352, 0x00010001,	/* cond: msdm.dst_pas_buf_wait_empty */
	0x253f0353, 0x00010001,	/* cond: msdm.sh_delay_empty */
	0x253f0354, 0x00010001,	/* cond: msdm.cm_delay_empty */
	0x253f0355, 0x00010001,	/* cond: msdm.cmsg_que_empty */
	0x253f0356, 0x00010001,	/* cond: msdm.ccfc_load_pend_empty */
	0x253f0357, 0x00010001,	/* cond: msdm.tcfc_load_pend_empty */
	0x253f0358, 0x00010001,	/* cond: msdm.async_host_empty */
	0x253f0359, 0x00010001,	/* cond: msdm.prm_fifo_empty */
	0x263f4010, 0x00010001,	/* cond: usdm.INT_STS */
	0x263f4011, 0x00010001,	/* cond: usdm.INT_MASK */
	0x263f4109, 0x00000001,	/* info: usdm.inp_queue_err_vect mode=all */
	0x263f4080, 0x00010001,	/* cond: usdm.PRTY_STS_H_0 */
	0x263f4081, 0x00010001,	/* cond: usdm.PRTY_MASK_H_0 */
	0x263f4303, 0x00010001,	/* cond: usdm.qm_full */
	0x263f430c, 0x00010001,	/* cond: usdm.rsp_brb_if_full */
	0x263f430d, 0x00010001,	/* cond: usdm.rsp_pxp_if_full */
	0x263f4316, 0x00010001,	/* cond: usdm.dst_pxp_if_full */
	0x263f4317, 0x00010001,	/* cond: usdm.dst_int_ram_if_full */
	0x263f4318, 0x00010001,	/* cond: usdm.dst_pas_buf_if_full */
	0x263f4340, 0x00010001,	/* cond: usdm.int_cmpl_pend_empty */
	0x263f4341, 0x00010001,	/* cond: usdm.int_cprm_pend_empty */
	0x263f4342, 0x00010001,	/* cond: usdm.queue_empty */
	0x263f4343, 0x00010001,	/* cond: usdm.delay_fifo_empty */
	0x263f4346, 0x00010001,	/* cond: usdm.rsp_pxp_rdata_empty */
	0x263f4347, 0x00010001,	/* cond: usdm.rsp_brb_rdata_empty */
	0x263f4348, 0x00010001,	/* cond: usdm.rsp_int_ram_rdata_empty */
	0x263f4349, 0x00010001,	/* cond: usdm.rsp_brb_pend_empty */
	0x263f434a, 0x00010001,	/* cond: usdm.rsp_int_ram_pend_empty */
	0x263f434b, 0x00010001,	/* cond: usdm.dst_pxp_immed_empty */
	0x263f434c, 0x00010001,	/* cond: usdm.dst_pxp_dst_pend_empty */
	0x263f434d, 0x00010001,	/* cond: usdm.dst_pxp_src_pend_empty */
	0x263f434e, 0x00010001,	/* cond: usdm.dst_brb_src_pend_empty */
	0x263f434f, 0x00010001,	/* cond: usdm.dst_brb_src_addr_empty */
	0x263f4350, 0x00010001,	/* cond: usdm.dst_pxp_link_empty */
	0x263f4351, 0x00010001,	/* cond: usdm.dst_int_ram_wait_empty */
	0x263f4352, 0x00010001,	/* cond: usdm.dst_pas_buf_wait_empty */
	0x263f4353, 0x00010001,	/* cond: usdm.sh_delay_empty */
	0x263f4354, 0x00010001,	/* cond: usdm.cm_delay_empty */
	0x263f4355, 0x00010001,	/* cond: usdm.cmsg_que_empty */
	0x263f4356, 0x00010001,	/* cond: usdm.ccfc_load_pend_empty */
	0x263f4357, 0x00010001,	/* cond: usdm.tcfc_load_pend_empty */
	0x263f4358, 0x00010001,	/* cond: usdm.async_host_empty */
	0x263f4359, 0x00010001,	/* cond: usdm.prm_fifo_empty */
	0x1b400060, 0x00010001,	/* cond: xcm.INT_STS_0 */
	0x1b400061, 0x00010001,	/* cond: xcm.INT_MASK_0 */
	0x1b400064, 0x00010001,	/* cond: xcm.INT_STS_1 */
	0x1b400065, 0x00010001,	/* cond: xcm.INT_MASK_1 */
	0x1b400068, 0x00010001,	/* cond: xcm.INT_STS_2 */
	0x1b400069, 0x00010001,	/* cond: xcm.INT_MASK_2 */
	0x1b4002d4, 0x00000001,	/* info: xcm.qm_act_st_cnt_err_details mode=all */
	0x1b400080, 0x00010001,	/* cond: xcm.PRTY_STS_H_0 */
	0x1b400081, 0x00010001,	/* cond: xcm.PRTY_MASK_H_0 */
	0x1b400084, 0x00010001,	/* cond: xcm.PRTY_STS_H_1 */
	0x1b400085, 0x00010001,	/* cond: xcm.PRTY_MASK_H_1 */
	0x1b4001a2, 0x00010001,	/* cond: xcm.fi_desc_input_violate */
	0x1b4001a3, 0x00010001,	/* cond: xcm.ia_agg_con_part_fill_lvl */
	0x1b4001a4, 0x00010001,	/* cond: xcm.ia_sm_con_part_fill_lvl */
	0x1b4001a5, 0x00010001,	/* cond: xcm.ia_trans_part_fill_lvl */
	0x1b4001c4, 0x00010001,	/* cond: xcm.xx_free_cnt */
	0x1b4001c5, 0x00010001,	/* cond: xcm.xx_lcid_cam_fill_lvl */
	0x1b4001ce, 0x00010001,	/* cond: xcm.xx_non_lock_cnt */
	0x1b4001cf, 0x00010001,	/* cond: xcm.xx_lock_cnt */
	0x1b4001d6, 0x00010001,	/* cond: xcm.xx_cbyp_tbl_fill_lvl */
	0x1b4001e2, 0x00010001,	/* cond: xcm.xx_pref_dir_fill_lvl */
	0x1b4001e3, 0x00010001,	/* cond: xcm.xx_pref_aggst_fill_lvl */
	0x1b4001e4, 0x00010001,	/* cond: xcm.xx_pref_byp_fill_lvl */
	0x1b400201, 0x00010001,	/* cond: xcm.prcs_agg_con_curr_st */
	0x1b400202, 0x00010001,	/* cond: xcm.prcs_sm_con_curr_st */
	0x1b40020b, 0x00010001,	/* cond: xcm.agg_con_fic_buf_fill_lvl */
	0x1b40020c, 0x00010001,	/* cond: xcm.sm_con_fic_buf_fill_lvl */
	0x1b400283, 0x00010001,	/* cond: xcm.in_prcs_tbl_fill_lvl */
	0x1b400288, 0x00010001,	/* cond: xcm.cmpl_dir_curr_st */
	0x1b4002a1, 0x00010001,	/* cond: xcm.ccfc_init_crd */
	0x1b4002a2, 0x00010001,	/* cond: xcm.qm_init_crd0 */
	0x1b4002a3, 0x00010001,	/* cond: xcm.qm_init_crd1 */
	0x1b4002a4, 0x00010001,	/* cond: xcm.tm_init_crd */
	0x1b4002a5, 0x00010001,	/* cond: xcm.fic_init_crd */
	0x1b4002aa, 0x00010001,	/* cond: xcm.msdm_length_mis */
	0x1b4002ab, 0x00010001,	/* cond: xcm.xsdm_length_mis */
	0x1b4002ac, 0x00010001,	/* cond: xcm.ysdm_length_mis */
	0x1b4002ae, 0x00010001,	/* cond: xcm.dorq_length_mis */
	0x1b4002af, 0x00010001,	/* cond: xcm.pbf_length_mis */
	0x1b4002b0, 0x00010001,	/* cond: xcm.grc_buf_empty */
	0x1b4002b1, 0x00000001,	/* info: xcm.grc_buf_status mode=all */
	0x1b4002c0, 0x00010001,	/* cond: xcm.is_qm_p_fill_lvl */
	0x1b4002c1, 0x00010001,	/* cond: xcm.is_qm_s_fill_lvl */
	0x1b4002c2, 0x00010001,	/* cond: xcm.is_tm_fill_lvl */
	0x1b4002c3, 0x00010001,	/* cond: xcm.is_storm_fill_lvl */
	0x1b4002c4, 0x00010001,	/* cond: xcm.is_msdm_fill_lvl */
	0x1b4002c5, 0x00010001,	/* cond: xcm.is_xsdm_fill_lvl */
	0x1b4002c6, 0x00010001,	/* cond: xcm.is_ysdm_fill_lvl */
	0x1b4002c8, 0x00010001,	/* cond: xcm.is_msem_fill_lvl */
	0x1b4002c9, 0x00010001,	/* cond: xcm.is_usem_fill_lvl */
	0x1b4002ca, 0x00010001,	/* cond: xcm.is_ysem_fill_lvl */
	0x1b4002cb, 0x00010001,	/* cond: xcm.is_dorq_fill_lvl */
	0x1b4002cc, 0x00010001,	/* cond: xcm.is_pbf_fill_lvl */
	0x1c420060, 0x00010001,	/* cond: ycm.INT_STS_0 */
	0x1c420061, 0x00010001,	/* cond: ycm.INT_MASK_0 */
	0x1c420064, 0x00010001,	/* cond: ycm.INT_STS_1 */
	0x1c420065, 0x00010001,	/* cond: ycm.INT_MASK_1 */
	0x1c420068, 0x00010001,	/* cond: ycm.INT_STS_2 */
	0x1c420069, 0x00010001,	/* cond: ycm.INT_MASK_2 */
	0x1c420080, 0x00010001,	/* cond: ycm.PRTY_STS_H_0 */
	0x1c420081, 0x00010001,	/* cond: ycm.PRTY_MASK_H_0 */
	0x1c420084, 0x00010001,	/* cond: ycm.PRTY_STS_H_1 */
	0x1c420085, 0x00010001,	/* cond: ycm.PRTY_MASK_H_1 */
	0x1c42019b, 0x00010001,	/* cond: ycm.fi_desc_input_violate */
	0x1c42019c, 0x00010001,	/* cond: ycm.se_desc_input_violate */
	0x1c42019e, 0x00010001,	/* cond: ycm.ia_sm_con_part_fill_lvl */
	0x1c42019f, 0x00010001,	/* cond: ycm.ia_agg_task_part_fill_lvl */
	0x1c4201a0, 0x00010001,	/* cond: ycm.ia_sm_task_part_fill_lvl */
	0x1c4201a1, 0x00010001,	/* cond: ycm.ia_trans_part_fill_lvl */
	0x1c4201c4, 0x00010001,	/* cond: ycm.xx_free_cnt */
	0x1c4201c5, 0x00010001,	/* cond: ycm.xx_lcid_cam_fill_lvl */
	0x1c4201ce, 0x00010001,	/* cond: ycm.xx_non_lock_cnt */
	0x1c4201cf, 0x00010001,	/* cond: ycm.xx_lock_cnt */
	0x1c4201d6, 0x00010001,	/* cond: ycm.xx_cbyp_tbl_fill_lvl */
	0x1c4201d9, 0x00010001,	/* cond: ycm.xx_tbyp_tbl_fill_lvl */
	0x1c4201d9, 0x00010001,	/* cond: ycm.xx_tbyp_tbl_fill_lvl */
	0x1c4201e5, 0x00010001,	/* cond: ycm.xx_pref_dir_fill_lvl */
	0x1c4201e6, 0x00010001,	/* cond: ycm.xx_pref_aggst_fill_lvl */
	0x1c4201e7, 0x00010001,	/* cond: ycm.xx_pref_byp_fill_lvl */
	0x1c420201, 0x00010001,	/* cond: ycm.prcs_agg_con_curr_st */
	0x1c420202, 0x00010001,	/* cond: ycm.prcs_sm_con_curr_st */
	0x1c420203, 0x00010001,	/* cond: ycm.prcs_agg_task_curr_st */
	0x1c420204, 0x00010001,	/* cond: ycm.prcs_sm_task_curr_st */
	0x1c420216, 0x00010001,	/* cond: ycm.sm_con_fic_buf_fill_lvl */
	0x1c42021e, 0x00010001,	/* cond: ycm.agg_task_fic_buf_fill_lvl */
	0x1c42021f, 0x00010001,	/* cond: ycm.sm_task_fic_buf_fill_lvl */
	0x1c420283, 0x00010001,	/* cond: ycm.in_prcs_tbl_fill_lvl */
	0x1c420289, 0x00010001,	/* cond: ycm.cmpl_dir_curr_st */
	0x1c4202a1, 0x00010001,	/* cond: ycm.ccfc_init_crd */
	0x1c4202a2, 0x00010001,	/* cond: ycm.tcfc_init_crd */
	0x1c4202a3, 0x00010001,	/* cond: ycm.qm_init_crd0 */
	0x1c4202a6, 0x00010001,	/* cond: ycm.fic_init_crd */
	0x1c4202aa, 0x00010001,	/* cond: ycm.msdm_length_mis */
	0x1c4202ab, 0x00010001,	/* cond: ycm.ysdm_length_mis */
	0x1c4202ac, 0x00010001,	/* cond: ycm.pbf_length_mis */
	0x1c4202ad, 0x00010001,	/* cond: ycm.xyld_length_mis */
	0x1c4202ae, 0x00010001,	/* cond: ycm.grc_buf_empty */
	0x1c4202af, 0x00000001,	/* info: ycm.grc_buf_status mode=all */
	0x1c4202ba, 0x00010001,	/* cond: ycm.is_qm_p_fill_lvl */
	0x1c4202bb, 0x00010001,	/* cond: ycm.is_qm_s_fill_lvl */
	0x1c4202bc, 0x00010001,	/* cond: ycm.is_storm_fill_lvl */
	0x1c4202bd, 0x00010001,	/* cond: ycm.is_msdm_fill_lvl */
	0x1c4202be, 0x00010001,	/* cond: ycm.is_ysdm_fill_lvl */
	0x1c4202bf, 0x00010001,	/* cond: ycm.is_xyld_fill_lvl */
	0x1c4202c0, 0x00010001,	/* cond: ycm.is_msem_fill_lvl */
	0x1c4202c1, 0x00010001,	/* cond: ycm.is_usem_fill_lvl */
	0x1c4202c2, 0x00010001,	/* cond: ycm.is_pbf_fill_lvl */
	0x1d440060, 0x00010001,	/* cond: pcm.INT_STS_0 */
	0x1d440061, 0x00010001,	/* cond: pcm.INT_MASK_0 */
	0x1d440064, 0x00010001,	/* cond: pcm.INT_STS_1 */
	0x1d440065, 0x00010001,	/* cond: pcm.INT_MASK_1 */
	0x1d440068, 0x00010001,	/* cond: pcm.INT_STS_2 */
	0x1d440069, 0x00010001,	/* cond: pcm.INT_MASK_2 */
	0x1d440080, 0x00010001,	/* cond: pcm.PRTY_STS_H_0 */
	0x1d440081, 0x00010001,	/* cond: pcm.PRTY_MASK_H_0 */
	0x1d440191, 0x00010001,	/* cond: pcm.fi_desc_input_violate */
	0x1d440192, 0x00010001,	/* cond: pcm.ia_sm_con_part_fill_lvl */
	0x1d440193, 0x00010001,	/* cond: pcm.ia_trans_part_fill_lvl */
	0x1d4401c4, 0x00010001,	/* cond: pcm.xx_free_cnt */
	0x1d4401c5, 0x00010001,	/* cond: pcm.xx_lcid_cam_fill_lvl */
	0x1d4401ce, 0x00010001,	/* cond: pcm.xx_non_lock_cnt */
	0x1d4401cf, 0x00010001,	/* cond: pcm.xx_lock_cnt */
	0x1d4401d6, 0x00010001,	/* cond: pcm.xx_pref_dir_fill_lvl */
	0x1d4401d7, 0x00010001,	/* cond: pcm.xx_pref_aggst_fill_lvl */
	0x1d440201, 0x00010001,	/* cond: pcm.prcs_sm_con_curr_st */
	0x1d44020a, 0x00010001,	/* cond: pcm.sm_con_fic_buf_fill_lvl */
	0x1d440283, 0x00010001,	/* cond: pcm.in_prcs_tbl_fill_lvl */
	0x1d440286, 0x00010001,	/* cond: pcm.cmpl_dir_curr_st */
	0x1d4402a1, 0x00010001,	/* cond: pcm.ccfc_init_crd */
	0x1d4402a2, 0x00010001,	/* cond: pcm.fic_init_crd */
	0x1d4402aa, 0x00010001,	/* cond: pcm.psdm_length_mis */
	0x1d4402ab, 0x00010001,	/* cond: pcm.pbf_length_mis */
	0x1d4402ac, 0x00010001,	/* cond: pcm.grc_buf_empty */
	0x1d4402ad, 0x00000001,	/* info: pcm.grc_buf_status mode=all */
	0x1d4402b2, 0x00010001,	/* cond: pcm.is_storm_fill_lvl */
	0x1d4402b3, 0x00010001,	/* cond: pcm.is_psdm_fill_lvl */
	0x18460060, 0x00010001,	/* cond: tcm.INT_STS_0 */
	0x18460061, 0x00010001,	/* cond: tcm.INT_MASK_0 */
	0x18460064, 0x00010001,	/* cond: tcm.INT_STS_1 */
	0x18460065, 0x00010001,	/* cond: tcm.INT_MASK_1 */
	0x18460068, 0x00010001,	/* cond: tcm.INT_STS_2 */
	0x18460069, 0x00010001,	/* cond: tcm.INT_MASK_2 */
	0x18460080, 0x00010001,	/* cond: tcm.PRTY_STS_H_0 */
	0x18460081, 0x00010001,	/* cond: tcm.PRTY_MASK_H_0 */
	0x18460084, 0x00010001,	/* cond: tcm.PRTY_STS_H_1 */
	0x18460085, 0x00010001,	/* cond: tcm.PRTY_MASK_H_1 */
	0x1846019b, 0x00010001,	/* cond: tcm.fi_desc_input_violate */
	0x1846019c, 0x00010001,	/* cond: tcm.se_desc_input_violate */
	0x1846019d, 0x00010001,	/* cond: tcm.ia_agg_con_part_fill_lvl */
	0x1846019e, 0x00010001,	/* cond: tcm.ia_sm_con_part_fill_lvl */
	0x1846019f, 0x00010001,	/* cond: tcm.ia_agg_task_part_fill_lvl */
	0x184601a0, 0x00010001,	/* cond: tcm.ia_sm_task_part_fill_lvl */
	0x184601a1, 0x00010001,	/* cond: tcm.ia_trans_part_fill_lvl */
	0x184601c4, 0x00010001,	/* cond: tcm.xx_free_cnt */
	0x184601c5, 0x00010001,	/* cond: tcm.xx_lcid_cam_fill_lvl */
	0x184601ce, 0x00010001,	/* cond: tcm.xx_non_lock_cnt */
	0x184601cf, 0x00010001,	/* cond: tcm.xx_lock_cnt */
	0x184601d6, 0x00010001,	/* cond: tcm.xx_cbyp_tbl_fill_lvl */
	0x184601d9, 0x00010001,	/* cond: tcm.xx_tbyp_tbl_fill_lvl */
	0x184601d9, 0x00010001,	/* cond: tcm.xx_tbyp_tbl_fill_lvl */
	0x184601e5, 0x00010001,	/* cond: tcm.xx_pref_dir_fill_lvl */
	0x184601e6, 0x00010001,	/* cond: tcm.xx_pref_aggst_fill_lvl */
	0x184601e7, 0x00010001,	/* cond: tcm.xx_pref_byp_fill_lvl */
	0x18460201, 0x00010001,	/* cond: tcm.prcs_agg_con_curr_st */
	0x18460202, 0x00010001,	/* cond: tcm.prcs_sm_con_curr_st */
	0x18460203, 0x00010001,	/* cond: tcm.prcs_agg_task_curr_st */
	0x18460204, 0x00010001,	/* cond: tcm.prcs_sm_task_curr_st */
	0x18460215, 0x00010001,	/* cond: tcm.agg_con_fic_buf_fill_lvl */
	0x18460216, 0x00010001,	/* cond: tcm.sm_con_fic_buf_fill_lvl */
	0x1846021e, 0x00010001,	/* cond: tcm.agg_task_fic_buf_fill_lvl */
	0x1846021f, 0x00010001,	/* cond: tcm.sm_task_fic_buf_fill_lvl */
	0x18460283, 0x00010001,	/* cond: tcm.in_prcs_tbl_fill_lvl */
	0x1846028b, 0x00010001,	/* cond: tcm.cmpl_dir_curr_st */
	0x184602a1, 0x00010001,	/* cond: tcm.ccfc_init_crd */
	0x184602a2, 0x00010001,	/* cond: tcm.tcfc_init_crd */
	0x184602a3, 0x00010001,	/* cond: tcm.qm_init_crd0 */
	0x184602a4, 0x00010001,	/* cond: tcm.tm_init_crd */
	0x184602a5, 0x00010001,	/* cond: tcm.fic_init_crd */
	0x184602aa, 0x00010001,	/* cond: tcm.tsdm_length_mis */
	0x184602ab, 0x00010001,	/* cond: tcm.dorq_length_mis */
	0x184602ac, 0x00010001,	/* cond: tcm.pbf_length_mis */
	0x184602ad, 0x00010001,	/* cond: tcm.prs_length_mis */
	0x184602ae, 0x00010001,	/* cond: tcm.grc_buf_empty */
	0x184602af, 0x00000001,	/* info: tcm.grc_buf_status mode=all */
	0x184602bb, 0x00010001,	/* cond: tcm.is_qm_p_fill_lvl */
	0x184602bc, 0x00010001,	/* cond: tcm.is_qm_s_fill_lvl */
	0x184602bd, 0x00010001,	/* cond: tcm.is_tm_fill_lvl */
	0x184602be, 0x00010001,	/* cond: tcm.is_storm_fill_lvl */
	0x184602bf, 0x00010001,	/* cond: tcm.is_tsdm_fill_lvl */
	0x184602c0, 0x00010001,	/* cond: tcm.is_msem_fill_lvl */
	0x184602c2, 0x00010001,	/* cond: tcm.is_dorq_fill_lvl */
	0x184602c3, 0x00010001,	/* cond: tcm.is_pbf_fill_lvl */
	0x19480060, 0x00010001,	/* cond: mcm.INT_STS_0 */
	0x19480061, 0x00010001,	/* cond: mcm.INT_MASK_0 */
	0x19480064, 0x00010001,	/* cond: mcm.INT_STS_1 */
	0x19480065, 0x00010001,	/* cond: mcm.INT_MASK_1 */
	0x19480068, 0x00010001,	/* cond: mcm.INT_STS_2 */
	0x19480069, 0x00010001,	/* cond: mcm.INT_MASK_2 */
	0x19480080, 0x00010001,	/* cond: mcm.PRTY_STS_H_0 */
	0x19480081, 0x00010001,	/* cond: mcm.PRTY_MASK_H_0 */
	0x19480084, 0x00010001,	/* cond: mcm.PRTY_STS_H_1 */
	0x19480085, 0x00010001,	/* cond: mcm.PRTY_MASK_H_1 */
	0x1948019d, 0x00010001,	/* cond: mcm.fi_desc_input_violate */
	0x1948019e, 0x00010001,	/* cond: mcm.se_desc_input_violate */
	0x1948019f, 0x00010001,	/* cond: mcm.ia_agg_con_part_fill_lvl */
	0x194801a0, 0x00010001,	/* cond: mcm.ia_sm_con_part_fill_lvl */
	0x194801a1, 0x00010001,	/* cond: mcm.ia_agg_task_part_fill_lvl */
	0x194801a2, 0x00010001,	/* cond: mcm.ia_sm_task_part_fill_lvl */
	0x194801a3, 0x00010001,	/* cond: mcm.ia_trans_part_fill_lvl */
	0x194801c4, 0x00010001,	/* cond: mcm.xx_free_cnt */
	0x194801c5, 0x00010001,	/* cond: mcm.xx_lcid_cam_fill_lvl */
	0x194801ce, 0x00010001,	/* cond: mcm.xx_non_lock_cnt */
	0x194801cf, 0x00010001,	/* cond: mcm.xx_lock_cnt */
	0x194801d6, 0x00010001,	/* cond: mcm.xx_cbyp_tbl_fill_lvl */
	0x194801d9, 0x00010001,	/* cond: mcm.xx_tbyp_tbl_fill_lvl */
	0x194801d9, 0x00010001,	/* cond: mcm.xx_tbyp_tbl_fill_lvl */
	0x194801e5, 0x00010001,	/* cond: mcm.xx_pref_dir_fill_lvl */
	0x194801e6, 0x00010001,	/* cond: mcm.xx_pref_aggst_fill_lvl */
	0x194801e7, 0x00010001,	/* cond: mcm.xx_pref_byp_fill_lvl */
	0x19480201, 0x00010001,	/* cond: mcm.prcs_agg_con_curr_st */
	0x19480202, 0x00010001,	/* cond: mcm.prcs_sm_con_curr_st */
	0x19480203, 0x00010001,	/* cond: mcm.prcs_agg_task_curr_st */
	0x19480204, 0x00010001,	/* cond: mcm.prcs_sm_task_curr_st */
	0x19480215, 0x00010001,	/* cond: mcm.agg_con_fic_buf_fill_lvl */
	0x19480216, 0x00010001,	/* cond: mcm.sm_con_fic_buf_fill_lvl */
	0x1948021e, 0x00010001,	/* cond: mcm.agg_task_fic_buf_fill_lvl */
	0x1948021f, 0x00010001,	/* cond: mcm.sm_task_fic_buf_fill_lvl */
	0x19480283, 0x00010001,	/* cond: mcm.in_prcs_tbl_fill_lvl */
	0x19480289, 0x00010001,	/* cond: mcm.cmpl_dir_curr_st */
	0x194802a1, 0x00010001,	/* cond: mcm.ccfc_init_crd */
	0x194802a2, 0x00010001,	/* cond: mcm.tcfc_init_crd */
	0x194802a3, 0x00010001,	/* cond: mcm.qm_init_crd0 */
	0x194802a6, 0x00010001,	/* cond: mcm.fic_init_crd */
	0x194802aa, 0x00010001,	/* cond: mcm.msdm_length_mis */
	0x194802ab, 0x00010001,	/* cond: mcm.ysdm_length_mis */
	0x194802ac, 0x00010001,	/* cond: mcm.usdm_length_mis */
	0x194802ad, 0x00010001,	/* cond: mcm.pbf_length_mis */
	0x194802ae, 0x00010001,	/* cond: mcm.tmld_length_mis */
	0x194802af, 0x00010001,	/* cond: mcm.grc_buf_empty */
	0x194802b0, 0x00000001,	/* info: mcm.grc_buf_status mode=all */
	0x194802bc, 0x00010001,	/* cond: mcm.is_qm_p_fill_lvl */
	0x194802bd, 0x00010001,	/* cond: mcm.is_qm_s_fill_lvl */
	0x194802be, 0x00010001,	/* cond: mcm.is_storm_fill_lvl */
	0x194802bf, 0x00010001,	/* cond: mcm.is_msdm_fill_lvl */
	0x194802c0, 0x00010001,	/* cond: mcm.is_ysdm_fill_lvl */
	0x194802c1, 0x00010001,	/* cond: mcm.is_usdm_fill_lvl */
	0x194802c2, 0x00010001,	/* cond: mcm.is_tmld_fill_lvl */
	0x194802c3, 0x00010001,	/* cond: mcm.is_usem_fill_lvl */
	0x194802c4, 0x00010001,	/* cond: mcm.is_ysem_fill_lvl */
	0x194802c5, 0x00010001,	/* cond: mcm.is_pbf_fill_lvl */
	0x1a4a0060, 0x00010001,	/* cond: ucm.INT_STS_0 */
	0x1a4a0061, 0x00010001,	/* cond: ucm.INT_MASK_0 */
	0x1a4a0064, 0x00010001,	/* cond: ucm.INT_STS_1 */
	0x1a4a0065, 0x00010001,	/* cond: ucm.INT_MASK_1 */
	0x1a4a0068, 0x00010001,	/* cond: ucm.INT_STS_2 */
	0x1a4a0069, 0x00010001,	/* cond: ucm.INT_MASK_2 */
	0x1a4a0080, 0x00010001,	/* cond: ucm.PRTY_STS_H_0 */
	0x1a4a0081, 0x00010001,	/* cond: ucm.PRTY_MASK_H_0 */
	0x1a4a0084, 0x00010001,	/* cond: ucm.PRTY_STS_H_1 */
	0x1a4a0085, 0x00010001,	/* cond: ucm.PRTY_MASK_H_1 */
	0x1a4a01a3, 0x00010001,	/* cond: ucm.fi_desc_input_violate */
	0x1a4a01a4, 0x00010001,	/* cond: ucm.se_desc_input_violate */
	0x1a4a01a5, 0x00010001,	/* cond: ucm.ia_agg_con_part_fill_lvl */
	0x1a4a01a6, 0x00010001,	/* cond: ucm.ia_sm_con_part_fill_lvl */
	0x1a4a01a7, 0x00010001,	/* cond: ucm.ia_agg_task_part_fill_lvl */
	0x1a4a01a8, 0x00010001,	/* cond: ucm.ia_sm_task_part_fill_lvl */
	0x1a4a01a9, 0x00010001,	/* cond: ucm.ia_trans_part_fill_lvl */
	0x1a4a01c4, 0x00010001,	/* cond: ucm.xx_free_cnt */
	0x1a4a01c5, 0x00010001,	/* cond: ucm.xx_lcid_cam_fill_lvl */
	0x1a4a01ce, 0x00010001,	/* cond: ucm.xx_non_lock_cnt */
	0x1a4a01cf, 0x00010001,	/* cond: ucm.xx_lock_cnt */
	0x1a4a01d6, 0x00010001,	/* cond: ucm.xx_cbyp_tbl_fill_lvl */
	0x1a4a01d9, 0x00010001,	/* cond: ucm.xx_tbyp_tbl_fill_lvl */
	0x1a4a01d9, 0x00010001,	/* cond: ucm.xx_tbyp_tbl_fill_lvl */
	0x1a4a01e5, 0x00010001,	/* cond: ucm.xx_pref_dir_fill_lvl */
	0x1a4a01e6, 0x00010001,	/* cond: ucm.xx_pref_aggst_fill_lvl */
	0x1a4a01e7, 0x00010001,	/* cond: ucm.xx_pref_byp_fill_lvl */
	0x1a4a0201, 0x00010001,	/* cond: ucm.prcs_agg_con_curr_st */
	0x1a4a0202, 0x00010001,	/* cond: ucm.prcs_sm_con_curr_st */
	0x1a4a0203, 0x00010001,	/* cond: ucm.prcs_agg_task_curr_st */
	0x1a4a0204, 0x00010001,	/* cond: ucm.prcs_sm_task_curr_st */
	0x1a4a0215, 0x00010001,	/* cond: ucm.agg_con_fic_buf_fill_lvl */
	0x1a4a0216, 0x00010001,	/* cond: ucm.sm_con_fic_buf_fill_lvl */
	0x1a4a021e, 0x00010001,	/* cond: ucm.agg_task_fic_buf_fill_lvl */
	0x1a4a021f, 0x00010001,	/* cond: ucm.sm_task_fic_buf_fill_lvl */
	0x1a4a0283, 0x00010001,	/* cond: ucm.in_prcs_tbl_fill_lvl */
	0x1a4a028b, 0x00010001,	/* cond: ucm.cmpl_dir_curr_st */
	0x1a4a02a1, 0x00010001,	/* cond: ucm.ccfc_init_crd */
	0x1a4a02a2, 0x00010001,	/* cond: ucm.tcfc_init_crd */
	0x1a4a02a3, 0x00010001,	/* cond: ucm.qm_init_crd0 */
	0x1a4a02a4, 0x00010001,	/* cond: ucm.tm_init_crd */
	0x1a4a02a5, 0x00010001,	/* cond: ucm.fic_init_crd */
	0x1a4a02ab, 0x00010001,	/* cond: ucm.ysdm_length_mis */
	0x1a4a02ac, 0x00010001,	/* cond: ucm.usdm_length_mis */
	0x1a4a02ad, 0x00010001,	/* cond: ucm.dorq_length_mis */
	0x1a4a02ae, 0x00010001,	/* cond: ucm.pbf_length_mis */
	0x1a4a02af, 0x00010001,	/* cond: ucm.rdif_length_mis */
	0x1a4a02b0, 0x00010001,	/* cond: ucm.tdif_length_mis */
	0x1a4a02b1, 0x00010001,	/* cond: ucm.muld_length_mis */
	0x1a4a02b2, 0x00010001,	/* cond: ucm.yuld_length_mis */
	0x1a4a02b3, 0x00010001,	/* cond: ucm.grc_buf_empty */
	0x1a4a02b4, 0x00000001,	/* info: ucm.grc_buf_status mode=all */
	0x1a4a02c3, 0x00010001,	/* cond: ucm.is_qm_p_fill_lvl */
	0x1a4a02c4, 0x00010001,	/* cond: ucm.is_qm_s_fill_lvl */
	0x1a4a02c5, 0x00010001,	/* cond: ucm.is_tm_fill_lvl */
	0x1a4a02c6, 0x00010001,	/* cond: ucm.is_storm_fill_lvl */
	0x1a4a02c8, 0x00010001,	/* cond: ucm.is_ysdm_fill_lvl */
	0x1a4a02c9, 0x00010001,	/* cond: ucm.is_usdm_fill_lvl */
	0x1a4a02ca, 0x00010001,	/* cond: ucm.is_rdif_fill_lvl */
	0x1a4a02cb, 0x00010001,	/* cond: ucm.is_tdif_fill_lvl */
	0x1a4a02cc, 0x00010001,	/* cond: ucm.is_muld_fill_lvl */
	0x1a4a02ce, 0x00010001,	/* cond: ucm.is_dorq_fill_lvl */
	0x1a4a02cf, 0x00010001,	/* cond: ucm.is_pbf_fill_lvl */
	0x2d500010, 0x00010001,	/* cond: xsem.INT_STS_0 */
	0x2d500011, 0x00010001,	/* cond: xsem.INT_MASK_0 */
	0x2d500014, 0x00010001,	/* cond: xsem.INT_STS_1 */
	0x2d500015, 0x00010001,	/* cond: xsem.INT_MASK_1 */
	0x2d500032, 0x00010001,	/* cond: xsem.PRTY_STS */
	0x2d500033, 0x00010001,	/* cond: xsem.PRTY_MASK */
	0x2d500080, 0x00010001,	/* cond: xsem.PRTY_STS_H_0 */
	0x2d500081, 0x00010001,	/* cond: xsem.PRTY_MASK_H_0 */
	0x2d500110, 0x00010001,	/* cond: xsem.pf_err_vector */
	0x2d5001a0, 0x01010001,	/* cond: xsem.foc_credit[1] */
	0x2d5001a0, 0x00010001,	/* cond: xsem.foc_credit */
	0x2d5002c0, 0x00010001,	/* cond: xsem.num_of_threads */
	0x2d5002c1, 0x00010001,	/* cond: xsem.thread_error */
	0x2d500440, 0x00010001,	/* cond: xsem.dra_empty */
	0x2d500441, 0x00010001,	/* cond: xsem.ext_pas_empty */
	0x2d500480, 0x00000001,	/* info: xsem.ext_pas_full mode=all */
	0x2d500448, 0x00010002,	/* cond: xsem.fic_empty[0:1] */
	0x2d500488, 0x00000002,	/* info: xsem.fic_full[0:1] mode=all */
	0x2d500451, 0x00010001,	/* cond: xsem.slow_dra_fin_empty */
	0x2d500494, 0x00000001,	/* info: xsem.slow_dra_fin_full mode=all */
	0x2d500452, 0x00010001,	/* cond: xsem.slow_dra_rd_empty */
	0x2d500495, 0x00000001,	/* info: xsem.slow_dra_rd_full mode=all */
	0x2d500453, 0x00010001,	/* cond: xsem.slow_dra_wr_empty */
	0x2d500496, 0x00000001,	/* info: xsem.slow_dra_wr_full mode=all */
	0x2d500454, 0x00010001,	/* cond: xsem.slow_ext_store_empty */
	0x2d500497, 0x00000001,	/* info: xsem.slow_ext_store_full mode=all */
	0x2d500455, 0x00010001,	/* cond: xsem.slow_ext_load_empty */
	0x2d500498, 0x00000001,	/* info: xsem.slow_ext_load_full mode=all */
	0x2d500458, 0x00010001,	/* cond: xsem.sync_dbg_empty */
	0x2d500481, 0x00010001,	/* cond: xsem.ext_store_if_full */
	0x2d500491, 0x00010001,	/* cond: xsem.ram_if_full */
	0x2d510010, 0x00010001,	/* cond: xsem.fast_memory.INT_STS */
	0x2d510011, 0x00010001,	/* cond: xsem.fast_memory.INT_MASK */
	0x2d510125, 0x00010001,	/* cond: xsem.fast_memory.stalled */
	0x2e540010, 0x00010001,	/* cond: ysem.INT_STS_0 */
	0x2e540011, 0x00010001,	/* cond: ysem.INT_MASK_0 */
	0x2e540014, 0x00010001,	/* cond: ysem.INT_STS_1 */
	0x2e540015, 0x00010001,	/* cond: ysem.INT_MASK_1 */
	0x2e540032, 0x00010001,	/* cond: ysem.PRTY_STS */
	0x2e540033, 0x00010001,	/* cond: ysem.PRTY_MASK */
	0x2e540080, 0x00010001,	/* cond: ysem.PRTY_STS_H_0 */
	0x2e540081, 0x00010001,	/* cond: ysem.PRTY_MASK_H_0 */
	0x2e540110, 0x00010001,	/* cond: ysem.pf_err_vector */
	0x2e5401a0, 0x05010001,	/* cond: ysem.foc_credit[5] */
	0x2e5401a0, 0x04010001,	/* cond: ysem.foc_credit[4] */
	0x2e5401a0, 0x03010001,	/* cond: ysem.foc_credit[3] */
	0x2e5401a0, 0x02010001,	/* cond: ysem.foc_credit[2] */
	0x2e5401a0, 0x01010001,	/* cond: ysem.foc_credit[1] */
	0x2e5401a0, 0x00010001,	/* cond: ysem.foc_credit */
	0x2e5402c0, 0x00010001,	/* cond: ysem.num_of_threads */
	0x2e5402c1, 0x00010001,	/* cond: ysem.thread_error */
	0x2e540440, 0x00010001,	/* cond: ysem.dra_empty */
	0x2e540441, 0x00010001,	/* cond: ysem.ext_pas_empty */
	0x2e540480, 0x00000001,	/* info: ysem.ext_pas_full mode=all */
	0x2e540448, 0x00010002,	/* cond: ysem.fic_empty[0:1] */
	0x2e540488, 0x00000002,	/* info: ysem.fic_full[0:1] mode=all */
	0x2e540451, 0x00010001,	/* cond: ysem.slow_dra_fin_empty */
	0x2e540494, 0x00000001,	/* info: ysem.slow_dra_fin_full mode=all */
	0x2e540452, 0x00010001,	/* cond: ysem.slow_dra_rd_empty */
	0x2e540495, 0x00000001,	/* info: ysem.slow_dra_rd_full mode=all */
	0x2e540453, 0x00010001,	/* cond: ysem.slow_dra_wr_empty */
	0x2e540496, 0x00000001,	/* info: ysem.slow_dra_wr_full mode=all */
	0x2e540454, 0x00010001,	/* cond: ysem.slow_ext_store_empty */
	0x2e540497, 0x00000001,	/* info: ysem.slow_ext_store_full mode=all */
	0x2e540455, 0x00010001,	/* cond: ysem.slow_ext_load_empty */
	0x2e540498, 0x00000001,	/* info: ysem.slow_ext_load_full mode=all */
	0x2e540458, 0x00010001,	/* cond: ysem.sync_dbg_empty */
	0x2e540481, 0x00010001,	/* cond: ysem.ext_store_if_full */
	0x2e540491, 0x00010001,	/* cond: ysem.ram_if_full */
	0x2e550010, 0x00010001,	/* cond: ysem.fast_memory.INT_STS */
	0x2e550011, 0x00010001,	/* cond: ysem.fast_memory.INT_MASK */
	0x2e550125, 0x00010001,	/* cond: ysem.fast_memory.stalled */
	0x2f580010, 0x00010001,	/* cond: psem.INT_STS_0 */
	0x2f580011, 0x00010001,	/* cond: psem.INT_MASK_0 */
	0x2f580014, 0x00010001,	/* cond: psem.INT_STS_1 */
	0x2f580015, 0x00010001,	/* cond: psem.INT_MASK_1 */
	0x2f580032, 0x00010001,	/* cond: psem.PRTY_STS */
	0x2f580033, 0x00010001,	/* cond: psem.PRTY_MASK */
	0x2f580080, 0x00010001,	/* cond: psem.PRTY_STS_H_0 */
	0x2f580081, 0x00010001,	/* cond: psem.PRTY_MASK_H_0 */
	0x2f580110, 0x00010001,	/* cond: psem.pf_err_vector */
	0x2f5801a0, 0x00010001,	/* cond: psem.foc_credit */
	0x2f5801a0, 0x01010001,	/* cond: psem.foc_credit[1] */
	0x2f5802c0, 0x00010001,	/* cond: psem.num_of_threads */
	0x2f5802c1, 0x00010001,	/* cond: psem.thread_error */
	0x2f580440, 0x00010001,	/* cond: psem.dra_empty */
	0x2f580441, 0x00010001,	/* cond: psem.ext_pas_empty */
	0x2f580480, 0x00000001,	/* info: psem.ext_pas_full mode=all */
	0x2f580448, 0x00010001,	/* cond: psem.fic_empty */
	0x2f580488, 0x00000001,	/* info: psem.fic_full mode=all */
	0x2f580451, 0x00010001,	/* cond: psem.slow_dra_fin_empty */
	0x2f580494, 0x00000001,	/* info: psem.slow_dra_fin_full mode=all */
	0x2f580452, 0x00010001,	/* cond: psem.slow_dra_rd_empty */
	0x2f580495, 0x00000001,	/* info: psem.slow_dra_rd_full mode=all */
	0x2f580453, 0x00010001,	/* cond: psem.slow_dra_wr_empty */
	0x2f580496, 0x00000001,	/* info: psem.slow_dra_wr_full mode=all */
	0x2f580454, 0x00010001,	/* cond: psem.slow_ext_store_empty */
	0x2f580497, 0x00000001,	/* info: psem.slow_ext_store_full mode=all */
	0x2f580455, 0x00010001,	/* cond: psem.slow_ext_load_empty */
	0x2f580498, 0x00000001,	/* info: psem.slow_ext_load_full mode=all */
	0x2f580458, 0x00010001,	/* cond: psem.sync_dbg_empty */
	0x2f580481, 0x00010001,	/* cond: psem.ext_store_if_full */
	0x2f580491, 0x00010001,	/* cond: psem.ram_if_full */
	0x2f590010, 0x00010001,	/* cond: psem.fast_memory.INT_STS */
	0x2f590011, 0x00010001,	/* cond: psem.fast_memory.INT_MASK */
	0x2f590125, 0x00010001,	/* cond: psem.fast_memory.stalled */
	0x2f590125, 0x00010001,	/* cond: psem.fast_memory.stalled */
	0x2a5c0010, 0x00010001,	/* cond: tsem.INT_STS_0 */
	0x2a5c0011, 0x00010001,	/* cond: tsem.INT_MASK_0 */
	0x2a5c0014, 0x00010001,	/* cond: tsem.INT_STS_1 */
	0x2a5c0015, 0x00010001,	/* cond: tsem.INT_MASK_1 */
	0x2a5c0032, 0x00010001,	/* cond: tsem.PRTY_STS */
	0x2a5c0033, 0x00010001,	/* cond: tsem.PRTY_MASK */
	0x2a5c0080, 0x00010001,	/* cond: tsem.PRTY_STS_H_0 */
	0x2a5c0081, 0x00010001,	/* cond: tsem.PRTY_MASK_H_0 */
	0x2a5c0110, 0x00010001,	/* cond: tsem.pf_err_vector */
	0x2a5c01a0, 0x00010001,	/* cond: tsem.foc_credit */
	0x2a5c01a0, 0x01010001,	/* cond: tsem.foc_credit[1] */
	0x2a5c02c0, 0x00010001,	/* cond: tsem.num_of_threads */
	0x2a5c02c1, 0x00010001,	/* cond: tsem.thread_error */
	0x2a5c0440, 0x00010001,	/* cond: tsem.dra_empty */
	0x2a5c0441, 0x00010001,	/* cond: tsem.ext_pas_empty */
	0x2a5c0480, 0x00000001,	/* info: tsem.ext_pas_full mode=all */
	0x2a5c0448, 0x00010001,	/* cond: tsem.fic_empty */
	0x2a5c0488, 0x00000001,	/* info: tsem.fic_full mode=all */
	0x2a5c0451, 0x00010001,	/* cond: tsem.slow_dra_fin_empty */
	0x2a5c0494, 0x00000001,	/* info: tsem.slow_dra_fin_full mode=all */
	0x2a5c0452, 0x00010001,	/* cond: tsem.slow_dra_rd_empty */
	0x2a5c0495, 0x00000001,	/* info: tsem.slow_dra_rd_full mode=all */
	0x2a5c0453, 0x00010001,	/* cond: tsem.slow_dra_wr_empty */
	0x2a5c0496, 0x00000001,	/* info: tsem.slow_dra_wr_full mode=all */
	0x2a5c0454, 0x00010001,	/* cond: tsem.slow_ext_store_empty */
	0x2a5c0497, 0x00000001,	/* info: tsem.slow_ext_store_full mode=all */
	0x2a5c0455, 0x00010001,	/* cond: tsem.slow_ext_load_empty */
	0x2a5c0498, 0x00000001,	/* info: tsem.slow_ext_load_full mode=all */
	0x2a5c0458, 0x00010001,	/* cond: tsem.sync_dbg_empty */
	0x2a5c0481, 0x00010001,	/* cond: tsem.ext_store_if_full */
	0x2a5c0491, 0x00010001,	/* cond: tsem.ram_if_full */
	0x2a5d0010, 0x00010001,	/* cond: tsem.fast_memory.INT_STS */
	0x2a5d0011, 0x00010001,	/* cond: tsem.fast_memory.INT_MASK */
	0x2a5d0125, 0x00010001,	/* cond: tsem.fast_memory.stalled */
	0x2b600010, 0x00010001,	/* cond: msem.INT_STS_0 */
	0x2b600011, 0x00010001,	/* cond: msem.INT_MASK_0 */
	0x2b600014, 0x00010001,	/* cond: msem.INT_STS_1 */
	0x2b600015, 0x00010001,	/* cond: msem.INT_MASK_1 */
	0x2b600032, 0x00010001,	/* cond: msem.PRTY_STS */
	0x2b600033, 0x00010001,	/* cond: msem.PRTY_MASK */
	0x2b600080, 0x00010001,	/* cond: msem.PRTY_STS_H_0 */
	0x2b600081, 0x00010001,	/* cond: msem.PRTY_MASK_H_0 */
	0x2b600110, 0x00010001,	/* cond: msem.pf_err_vector */
	0x2b6001a0, 0x05010001,	/* cond: msem.foc_credit[5] */
	0x2b6001a0, 0x04010001,	/* cond: msem.foc_credit[4] */
	0x2b6001a0, 0x02010001,	/* cond: msem.foc_credit[2] */
	0x2b6001a0, 0x01010001,	/* cond: msem.foc_credit[1] */
	0x2b6001a0, 0x03010001,	/* cond: msem.foc_credit[3] */
	0x2b6001a0, 0x00010001,	/* cond: msem.foc_credit */
	0x2b6002c0, 0x00010001,	/* cond: msem.num_of_threads */
	0x2b6002c1, 0x00010001,	/* cond: msem.thread_error */
	0x2b600440, 0x00010001,	/* cond: msem.dra_empty */
	0x2b600441, 0x00010001,	/* cond: msem.ext_pas_empty */
	0x2b600480, 0x00000001,	/* info: msem.ext_pas_full mode=all */
	0x2b600448, 0x00010001,	/* cond: msem.fic_empty */
	0x2b600488, 0x00000001,	/* info: msem.fic_full mode=all */
	0x2b600451, 0x00010001,	/* cond: msem.slow_dra_fin_empty */
	0x2b600494, 0x00000001,	/* info: msem.slow_dra_fin_full mode=all */
	0x2b600452, 0x00010001,	/* cond: msem.slow_dra_rd_empty */
	0x2b600495, 0x00000001,	/* info: msem.slow_dra_rd_full mode=all */
	0x2b600453, 0x00010001,	/* cond: msem.slow_dra_wr_empty */
	0x2b600496, 0x00000001,	/* info: msem.slow_dra_wr_full mode=all */
	0x2b600454, 0x00010001,	/* cond: msem.slow_ext_store_empty */
	0x2b600497, 0x00000001,	/* info: msem.slow_ext_store_full mode=all */
	0x2b600455, 0x00010001,	/* cond: msem.slow_ext_load_empty */
	0x2b600498, 0x00000001,	/* info: msem.slow_ext_load_full mode=all */
	0x2b600458, 0x00010001,	/* cond: msem.sync_dbg_empty */
	0x2b600481, 0x00010001,	/* cond: msem.ext_store_if_full */
	0x2b600491, 0x00010001,	/* cond: msem.ram_if_full */
	0x2b610010, 0x00010001,	/* cond: msem.fast_memory.INT_STS */
	0x2b610011, 0x00010001,	/* cond: msem.fast_memory.INT_MASK */
	0x2b610125, 0x00010001,	/* cond: msem.fast_memory.stalled */
	0x2c640010, 0x00010001,	/* cond: usem.INT_STS_0 */
	0x2c640011, 0x00010001,	/* cond: usem.INT_MASK_0 */
	0x2c640014, 0x00010001,	/* cond: usem.INT_STS_1 */
	0x2c640015, 0x00010001,	/* cond: usem.INT_MASK_1 */
	0x2c640032, 0x00010001,	/* cond: usem.PRTY_STS */
	0x2c640033, 0x00010001,	/* cond: usem.PRTY_MASK */
	0x2c640080, 0x00010001,	/* cond: usem.PRTY_STS_H_0 */
	0x2c640081, 0x00010001,	/* cond: usem.PRTY_MASK_H_0 */
	0x2c640110, 0x00010001,	/* cond: usem.pf_err_vector */
	0x2c6401a0, 0x01010001,	/* cond: usem.foc_credit[1] */
	0x2c6401a0, 0x02010001,	/* cond: usem.foc_credit[2] */
	0x2c6401a0, 0x03010001,	/* cond: usem.foc_credit[3] */
	0x2c6401a0, 0x04010001,	/* cond: usem.foc_credit[4] */
	0x2c6401a0, 0x00010001,	/* cond: usem.foc_credit */
	0x2c6402c0, 0x00010001,	/* cond: usem.num_of_threads */
	0x2c6402c1, 0x00010001,	/* cond: usem.thread_error */
	0x2c640440, 0x00010001,	/* cond: usem.dra_empty */
	0x2c640441, 0x00010001,	/* cond: usem.ext_pas_empty */
	0x2c640480, 0x00000001,	/* info: usem.ext_pas_full mode=all */
	0x2c640448, 0x00010001,	/* cond: usem.fic_empty */
	0x2c640488, 0x00000001,	/* info: usem.fic_full mode=all */
	0x2c640451, 0x00010001,	/* cond: usem.slow_dra_fin_empty */
	0x2c640494, 0x00000001,	/* info: usem.slow_dra_fin_full mode=all */
	0x2c640452, 0x00010001,	/* cond: usem.slow_dra_rd_empty */
	0x2c640495, 0x00000001,	/* info: usem.slow_dra_rd_full mode=all */
	0x2c640453, 0x00010001,	/* cond: usem.slow_dra_wr_empty */
	0x2c640496, 0x00000001,	/* info: usem.slow_dra_wr_full mode=all */
	0x2c640454, 0x00010001,	/* cond: usem.slow_ext_store_empty */
	0x2c640497, 0x00000001,	/* info: usem.slow_ext_store_full mode=all */
	0x2c640455, 0x00010001,	/* cond: usem.slow_ext_load_empty */
	0x2c640498, 0x00000001,	/* info: usem.slow_ext_load_full mode=all */
	0x2c640458, 0x00010001,	/* cond: usem.sync_dbg_empty */
	0x2c640481, 0x00010001,	/* cond: usem.ext_store_if_full */
	0x2c640491, 0x00010001,	/* cond: usem.ram_if_full */
	0x2c650010, 0x00010001,	/* cond: usem.fast_memory.INT_STS */
	0x2c650011, 0x00010001,	/* cond: usem.fast_memory.INT_MASK */
	0x2c650125, 0x00010001,	/* cond: usem.fast_memory.stalled */
	0x0a015000, 0x00010001,	/* cond: pcie.PRTY_STS_H_0 */
	0x40060220, 0x00010009,	/* cond: igu.pending_bits_status[0:8] */
	0x40060240, 0x00010009,	/* cond: igu.write_done_pending[0:8] */
	0x050860ba, 0x00010001,	/* cond: cnig.INT_STS */
	0x050860bb, 0x00010001,	/* cond: cnig.INT_MASK */
	0x050860a7, 0x00810001,	/* info: cnig.cnig_dbg_fifo_error mode=!k2 */
	0x050860d2, 0x00010001,	/* cond: cnig.PRTY_STS */
	0x050860d3, 0x00010001,	/* cond: cnig.PRTY_MASK */
	0x0d0a8018, 0x00010001,	/* cond: pswhst.vf_disabled_error_valid */
	0x0d0a8017, 0x00000001,	/* info: pswhst.vf_disabled_error_data mode=all */
	0x0d0a8019, 0x00000001,	/* info: pswhst.vf_disabled_error_address mode=all */
	0x040aa001, 0x00010001,	/* cond: pglue_b.init_done_inb_int_mem */
	0x040aa005, 0x00010001,	/* cond: pglue_b.init_done_zone_a */
	0x040aa060, 0x00010001,	/* cond: pglue_b.INT_STS */
	0x040aa060, 0x00010001,	/* cond: pglue_b.INT_STS */
	0x040aa80e, 0x00010001,	/* cond: pglue_b.flr_request_vf_223_192 */
	0x040aa80f, 0x00010001,	/* cond: pglue_b.flr_request_vf_255_224 */
	0x040aa843, 0x00010001,	/* cond: pglue_b.was_error_vf_223_192 */
	0x040aa844, 0x00010001,	/* cond: pglue_b.was_error_vf_255_224 */
	0x040aab92, 0x00010001,	/* cond: pglue_b.tags_159_128 */
	0x040aab93, 0x00010001,	/* cond: pglue_b.tags_191_160 */
	0x040aab94, 0x00010001,	/* cond: pglue_b.tags_223_192 */
	0x040aab95, 0x00010001,	/* cond: pglue_b.tags_255_224 */
	0x1e8bc300, 0x00020040,	/* cond: qm.PtrTblOther[0:63] width=2 access=WB */
	0x1e8bc300, 0x00020040,	/* cond: qm.PtrTblOther[0:63] width=2 access=WB */
	0x1e0bc410, 0x0001000e,	/* cond: qm.QstatusTx_0[0:13] */
	0x1e0bc430, 0x00010002,	/* cond: qm.QstatusOther_0[0:1] */
	0x1e0bc560, 0x00010008,	/* cond: qm.VoqCrdLine[0:7] */
	0x1e0bc5a0, 0x00010008,	/* cond: qm.VoqInitCrdLine[0:7] */
	0x02002300, 0x00010001,	/* cond: misc.port_mode */
	0x1e0bc560, 0x10010001,	/* cond: qm.VoqCrdLine[16] */
	0x1e0bc5a0, 0x10010001,	/* cond: qm.VoqInitCrdLine[16] */
	0x02002300, 0x00010001,	/* cond: misc.port_mode */
	0x1e0bc560, 0x00010012,	/* cond: qm.VoqCrdLine[0:17] */
	0x1e0bc5a0, 0x00010012,	/* cond: qm.VoqInitCrdLine[0:17] */
	0x02002300, 0x00010001,	/* cond: misc.port_mode */
	0x1e0bc5e0, 0x00010008,	/* cond: qm.VoqCrdByte[0:7] */
	0x1e0bc620, 0x00010008,	/* cond: qm.VoqInitCrdByte[0:7] */
	0x02002300, 0x00010001,	/* cond: misc.port_mode */
	0x1e0bc5e0, 0x10010001,	/* cond: qm.VoqCrdByte[16] */
	0x1e0bc620, 0x10010001,	/* cond: qm.VoqInitCrdByte[16] */
	0x02002300, 0x00010001,	/* cond: misc.port_mode */
	0x1e0bc5e0, 0x00010012,	/* cond: qm.VoqCrdByte[0:17] */
	0x1e0bc620, 0x00010012,	/* cond: qm.VoqInitCrdByte[0:17] */
	0x02002300, 0x00010001,	/* cond: misc.port_mode */
	0x1e8bf800, 0x000201c0,	/* cond: qm.PtrTblTx[0:447] width=2 access=WB */
	0x1e8bf800, 0x000201c0,	/* cond: qm.PtrTblTx[0:447] width=2 access=WB */
	0x210d03cc, 0x00010004,	/* cond: brb.wc_empty_0[0:3] */
	0x210d041e, 0x00010001,	/* cond: brb.rc_eop_empty */
	0x218d06c0, 0x00030004,	/* cond: brb.wc_status_0[0:3] width=3 access=WB */
	0x46140028, 0x00010001,	/* cond: nig.PRTY_STS */
	0x46140029, 0x00010001,	/* cond: nig.PRTY_MASK */
	0x461422db, 0x00000001,	/* info: nig.rx_parity_err mode=all */
	0x461422dc, 0x00000001,	/* info: nig.tx_parity_err mode=all */
	0x461422dd, 0x00000001,	/* info: nig.lb_parity_err mode=all */
	0x3936c030, 0x00010001,	/* cond: btb.INT_STS_0 */
	0x3936c031, 0x00010001,	/* cond: btb.INT_MASK_0 */
	0x39b6c400, 0x00000008,	/* info: btb.stopped_rd_req[0:3] width=2 access=WB mode=all */
	0x39b6c440, 0x00000008,	/* info: btb.stopped_rls_req[0:3] width=2 access=WB mode=all */
	0x3936c048, 0x00010001,	/* cond: btb.INT_STS_4 */
	0x3936c049, 0x00010001,	/* cond: btb.INT_MASK_4 */
	0x39b6c400, 0x00000008,	/* info: btb.stopped_rd_req[0:3] width=2 access=WB mode=all */
	0x39b6c440, 0x00000008,	/* info: btb.stopped_rls_req[0:3] width=2 access=WB mode=all */
	0x1b401000, 0x000101c0,	/* cond: xcm.qm_act_st_cnt[0:447] */
	0x2dd00108, 0x00040001,	/* cond: xsem.vf_err_vector width=4 access=WB */
	0x2ed40108, 0x00040001,	/* cond: ysem.vf_err_vector width=4 access=WB */
	0x2fd80108, 0x00040001,	/* cond: psem.vf_err_vector width=4 access=WB */
	0x2adc0108, 0x00040001,	/* cond: tsem.vf_err_vector width=4 access=WB */
	0x2be00108, 0x00040001,	/* cond: msem.vf_err_vector width=4 access=WB */
	0x2ce40108, 0x00040001,	/* cond: usem.vf_err_vector width=4 access=WB */
	0x0a015000, 0x00010001,	/* cond: pcie.PRTY_STS_H_0 */
	0x0a015001, 0x00010001,	/* cond: pcie.PRTY_MASK_H_0 */
	0x0d0a8018, 0x00010001,	/* cond: pswhst.vf_disabled_error_valid */
	0x0d0a8017, 0x00000001,	/* info: pswhst.vf_disabled_error_data mode=all */
	0x0d0a8019, 0x00000001,	/* info: pswhst.vf_disabled_error_address mode=all */
	0x040aa001, 0x00010001,	/* cond: pglue_b.init_done_inb_int_mem */
	0x040aa005, 0x00010001,	/* cond: pglue_b.init_done_zone_a */
	0x040aa060, 0x00010001,	/* cond: pglue_b.INT_STS */
	0x040aa060, 0x00010001,	/* cond: pglue_b.INT_STS */
	0x210d03cc, 0x00010008,	/* cond: brb.wc_empty_0[0:7] */
	0x218d06c0, 0x00030008,	/* cond: brb.wc_status_0[0:7] width=3 access=WB */
	0x14090279, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_7 */
	0x140902d8, 0x00010001,	/* cond: pswrq2.max_srs_vq7 */
	0x1409027a, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_8 */
	0x140902d9, 0x00010001,	/* cond: pswrq2.max_srs_vq8 */
	0x1409027c, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_10 */
	0x140902db, 0x00010001,	/* cond: pswrq2.max_srs_vq10 */
	0x1409027f, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_13 */
	0x140902de, 0x00010001,	/* cond: pswrq2.max_srs_vq13 */
	0x14090280, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_14 */
	0x140902df, 0x00010001,	/* cond: pswrq2.max_srs_vq14 */
	0x14090286, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_20 */
	0x140902e5, 0x00010001,	/* cond: pswrq2.max_srs_vq20 */
	0x1409028b, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_25 */
	0x140902ea, 0x00010001,	/* cond: pswrq2.max_srs_vq25 */
	0x1409028d, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_27 */
	0x140902ec, 0x00010001,	/* cond: pswrq2.max_srs_vq27 */
	0x14090291, 0x00010001,	/* cond: pswrq2.sr_cnt_per_vq_31 */
	0x140902f0, 0x00010001,	/* cond: pswrq2.max_srs_vq31 */
	0x1409029a, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_7 */
	0x14090258, 0x00010001,	/* cond: pswrq2.max_blks_vq7 */
	0x1409029b, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_8 */
	0x14090259, 0x00010001,	/* cond: pswrq2.max_blks_vq8 */
	0x1409029d, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_10 */
	0x1409025b, 0x00010001,	/* cond: pswrq2.max_blks_vq10 */
	0x140902a0, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_13 */
	0x1409025e, 0x00010001,	/* cond: pswrq2.max_blks_vq13 */
	0x140902a1, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_14 */
	0x1409025f, 0x00010001,	/* cond: pswrq2.max_blks_vq14 */
	0x140902a7, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_20 */
	0x14090265, 0x00010001,	/* cond: pswrq2.max_blks_vq20 */
	0x140902ac, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_25 */
	0x1409026a, 0x00010001,	/* cond: pswrq2.max_blks_vq25 */
	0x140902ae, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_27 */
	0x1409026c, 0x00010001,	/* cond: pswrq2.max_blks_vq27 */
	0x140902b2, 0x00010001,	/* cond: pswrq2.blk_cnt_per_vq_31 */
	0x14090270, 0x00010001,	/* cond: pswrq2.max_blks_vq31 */
	0x1f0b0064, 0x00010001,	/* cond: tm.INT_STS_1 */
	0x09150100, 0x00010001,	/* cond: bmb.PRTY_STS_H_0 */
	0x09150101, 0x00010001,	/* cond: bmb.PRTY_MASK_H_0 */
	0x09150104, 0x00010001,	/* cond: bmb.PRTY_STS_H_1 */
	0x09150105, 0x00010001,	/* cond: bmb.PRTY_MASK_H_1 */
	0x1f0b0064, 0x00010001,	/* cond: tm.INT_STS_1 */
	0x15000841, 0x00010001,	/* cond: pglcs.pgl_cs.UNCORR_ERR_STATUS_OFF */
	0x15000841, 0x00010001,	/* cond: pglcs.pgl_cs.UNCORR_ERR_STATUS_OFF */
	0x15000841, 0x00010001,	/* cond: pglcs.pgl_cs.UNCORR_ERR_STATUS_OFF */
	0x15000847, 0x00010004,	/* cond: pglcs.pgl_cs.HDR_LOG_0_OFF[0:3] */
	0x15000e0c, 0x00010001,	/* cond: pglcs.syncfifo_pop_underflow */
	0x15000e0d, 0x00010001,	/* cond: pglcs.syncfifo_push_overflow */
	0x15000e0e, 0x00010001,	/* cond: pglcs.tx_syncfifo_pop_status */
	0x0a0151e8, 0x00010001,	/* cond: pcie.INT_STS */
	0x40060220, 0x0001000c,	/* cond: igu.pending_bits_status[0:11] */
	0x40060240, 0x0001000c,	/* cond: igu.write_done_pending[0:11] */
	0x05086086, 0x00010001,	/* cond: cnig.INT_STS */
	0x05086087, 0x00010001,	/* cond: cnig.INT_MASK */
	0x050860a7, 0x00810001,	/* info: cnig.cnig_dbg_fifo_error mode=!k2 */
	0x0508608b, 0x00010001,	/* cond: cnig.PRTY_STS */
	0x0508608c, 0x00010001,	/* cond: cnig.PRTY_MASK */
	0x040aabf1, 0x00010001,	/* cond: pglue_b.syncfifo_push_overflow */
	0x040aabf2, 0x00010001,	/* cond: pglue_b.syncfifo_pop_underflow */
	0x040aabf3, 0x00010001,	/* cond: pglue_b.rxh_syncfifo_pop_status */
	0x040aabf4, 0x00010001,	/* cond: pglue_b.rxd_syncfifo_pop_status */
	0x040aabf3, 0x00230001,	/* info: pglue_b.rxh_syncfifo_pop_status mode=!bb */
	0x1e8bc300, 0x00020080,	/* cond: qm.PtrTblOther[0:127] width=2 access=WB */
	0x1e8bc300, 0x00020080,	/* cond: qm.PtrTblOther[0:127] width=2 access=WB */
	0x1e0bc410, 0x00010010,	/* cond: qm.QstatusTx_0[0:15] */
	0x1e0bc430, 0x00010004,	/* cond: qm.QstatusOther_0[0:3] */
	0x1e0bc560, 0x00010014,	/* cond: qm.VoqCrdLine[0:19] */
	0x1e0bc5a0, 0x00010014,	/* cond: qm.VoqInitCrdLine[0:19] */
	0x02002300, 0x00010001,	/* cond: misc.port_mode */
	0x1e0bc560, 0x10010001,	/* cond: qm.VoqCrdLine[16] */
	0x1e0bc5a0, 0x10010001,	/* cond: qm.VoqInitCrdLine[16] */
	0x02002300, 0x00010001,	/* cond: misc.port_mode */
	0x1e0bc560, 0x00010008,	/* cond: qm.VoqCrdLine[0:7] */
	0x1e0bc5a0, 0x00010008,	/* cond: qm.VoqInitCrdLine[0:7] */
	0x02002300, 0x00010001,	/* cond: misc.port_mode */
	0x1e0bc5e0, 0x00010014,	/* cond: qm.VoqCrdByte[0:19] */
	0x1e0bc620, 0x00010014,	/* cond: qm.VoqInitCrdByte[0:19] */
	0x02002300, 0x00010001,	/* cond: misc.port_mode */
	0x1e0bc5e0, 0x00010008,	/* cond: qm.VoqCrdByte[0:7] */
	0x1e0bc620, 0x00010008,	/* cond: qm.VoqInitCrdByte[0:7] */
	0x02002300, 0x00010001,	/* cond: misc.port_mode */
	0x1e0bc5e0, 0x10010001,	/* cond: qm.VoqCrdByte[16] */
	0x1e0bc620, 0x10010001,	/* cond: qm.VoqInitCrdByte[16] */
	0x02002300, 0x00010001,	/* cond: misc.port_mode */
	0x1e8bf800, 0x00020200,	/* cond: qm.PtrTblTx[0:511] width=2 access=WB */
	0x1e8bf800, 0x00020200,	/* cond: qm.PtrTblTx[0:511] width=2 access=WB */
	0x210d041e, 0x00010001,	/* cond: brb.rc_eop_empty */
	0x210d0494, 0x00010001,	/* cond: brb.mac2_tc_occupancy_0 */
	0x210d0495, 0x00010001,	/* cond: brb.mac2_tc_occupancy_1 */
	0x210d0496, 0x00010001,	/* cond: brb.mac2_tc_occupancy_2 */
	0x210d0497, 0x00010001,	/* cond: brb.mac2_tc_occupancy_3 */
	0x210d0498, 0x00010001,	/* cond: brb.mac2_tc_occupancy_4 */
	0x210d04a4, 0x00010001,	/* cond: brb.mac3_tc_occupancy_0 */
	0x210d04a5, 0x00010001,	/* cond: brb.mac3_tc_occupancy_1 */
	0x210d04a6, 0x00010001,	/* cond: brb.mac3_tc_occupancy_2 */
	0x210d04a7, 0x00010001,	/* cond: brb.mac3_tc_occupancy_3 */
	0x210d04a8, 0x00010001,	/* cond: brb.mac3_tc_occupancy_4 */
	0x46140028, 0x00010001,	/* cond: nig.INT_STS_6 */
	0x46140029, 0x00010001,	/* cond: nig.INT_MASK_6 */
	0x4614002c, 0x00010001,	/* cond: nig.INT_STS_7 */
	0x4614002d, 0x00010001,	/* cond: nig.INT_MASK_7 */
	0x461406e8, 0x00000001,	/* info: nig.flowctrl_mode mode=all */
	0x46140713, 0x00000001,	/* info: nig.rx_flowctrl_status mode=all */
	0x46140030, 0x00010001,	/* cond: nig.INT_STS_8 */
	0x46140031, 0x00010001,	/* cond: nig.INT_MASK_8 */
	0x46140034, 0x00010001,	/* cond: nig.INT_STS_9 */
	0x46140035, 0x00010001,	/* cond: nig.INT_MASK_9 */
	0x461406e8, 0x00000001,	/* info: nig.flowctrl_mode mode=all */
	0x46140713, 0x00000001,	/* info: nig.rx_flowctrl_status mode=all */
	0x46140038, 0x00010001,	/* cond: nig.PRTY_STS */
	0x46140039, 0x00010001,	/* cond: nig.PRTY_MASK */
	0x461422db, 0x00000001,	/* info: nig.rx_parity_err mode=all */
	0x461422dc, 0x00000001,	/* info: nig.tx_parity_err mode=all */
	0x461422dd, 0x00000001,	/* info: nig.lb_parity_err mode=all */
	0x3936c030, 0x00010001,	/* cond: btb.INT_STS_0 */
	0x3936c031, 0x00010001,	/* cond: btb.INT_MASK_0 */
	0x39b6c400, 0x00000010,	/* info: btb.stopped_rd_req[0:7] width=2 access=WB mode=all */
	0x39b6c440, 0x00000010,	/* info: btb.stopped_rls_req[0:7] width=2 access=WB mode=all */
	0x3936c048, 0x00010001,	/* cond: btb.INT_STS_4 */
	0x3936c049, 0x00010001,	/* cond: btb.INT_MASK_4 */
	0x39b6c400, 0x00000010,	/* info: btb.stopped_rd_req[0:7] width=2 access=WB mode=all */
	0x39b6c440, 0x00000010,	/* info: btb.stopped_rls_req[0:7] width=2 access=WB mode=all */
	0x3936c26d, 0x00010001,	/* cond: btb.rc_pkt_empty_4 */
	0x3936c26e, 0x00010001,	/* cond: btb.rc_pkt_empty_5 */
	0x3936c26f, 0x00010001,	/* cond: btb.rc_pkt_empty_6 */
	0x3936c270, 0x00010001,	/* cond: btb.rc_pkt_empty_7 */
	0x253f035a, 0x00010001,	/* cond: msdm.rmt_xcm_fifo_empty */
	0x253f0320, 0x00230001,	/* info: msdm.rmt_xcm_fifo_full mode=!bb */
	0x253f035b, 0x00010001,	/* cond: msdm.rmt_ycm_fifo_empty */
	0x253f0321, 0x00230001,	/* info: msdm.rmt_ycm_fifo_full mode=!bb */
	0x1b401000, 0x00010200,	/* cond: xcm.qm_act_st_cnt[0:511] */
	0x2dd00108, 0x00060001,	/* cond: xsem.vf_err_vector width=6 access=WB */
	0x2d510080, 0x00010001,	/* cond: xsem.fast_memory.PRTY_STS_H_0 */
	0x2d510081, 0x00010001,	/* cond: xsem.fast_memory.PRTY_MASK_H_0 */
	0x2ed40108, 0x00060001,	/* cond: ysem.vf_err_vector width=6 access=WB */
	0x2e550080, 0x00010001,	/* cond: ysem.fast_memory.PRTY_STS_H_0 */
	0x2e550081, 0x00010001,	/* cond: ysem.fast_memory.PRTY_MASK_H_0 */
	0x2fd80108, 0x00060001,	/* cond: psem.vf_err_vector width=6 access=WB */
	0x2f590080, 0x00010001,	/* cond: psem.fast_memory.PRTY_STS_H_0 */
	0x2f590081, 0x00010001,	/* cond: psem.fast_memory.PRTY_MASK_H_0 */
	0x2adc0108, 0x00060001,	/* cond: tsem.vf_err_vector width=6 access=WB */
	0x2a5d0080, 0x00010001,	/* cond: tsem.fast_memory.PRTY_STS_H_0 */
	0x2a5d0081, 0x00010001,	/* cond: tsem.fast_memory.PRTY_MASK_H_0 */
	0x2be00108, 0x00060001,	/* cond: msem.vf_err_vector width=6 access=WB */
	0x2b610080, 0x00010001,	/* cond: msem.fast_memory.PRTY_STS_H_0 */
	0x2b610081, 0x00010001,	/* cond: msem.fast_memory.PRTY_MASK_H_0 */
	0x2ce40108, 0x00060001,	/* cond: usem.vf_err_vector width=6 access=WB */
	0x2c650080, 0x00010001,	/* cond: usem.fast_memory.PRTY_STS_H_0 */
	0x2c650081, 0x00010001,	/* cond: usem.fast_memory.PRTY_MASK_H_0 */
	0x15000841, 0x00010001,	/* cond: pglcs.pgl_cs.uc_err_status */
	0x15000841, 0x00010001,	/* cond: pglcs.pgl_cs.uc_err_status */
	0x15000841, 0x00010001,	/* cond: pglcs.pgl_cs.uc_err_status */
	0x15000847, 0x00010004,	/* cond: pglcs.pgl_cs.header_log1[0:3] */
	0x15000a05, 0x00010001,	/* cond: pglcs.pgl_cs.tl_ctrlstat_5 */
	0x15000a05, 0x00010001,	/* cond: pglcs.pgl_cs.tl_ctrlstat_5 */
	0x15000a15, 0x00010001,	/* cond: pglcs.pgl_cs.tl_func345_stat */
	0x15000a17, 0x00010001,	/* cond: pglcs.pgl_cs.tl_func678_stat */
	0x15000c10, 0x00010001,	/* cond: pglcs.pgl_cs.DLATTN_VEC */
	0x01002464, 0x00010001,	/* cond: miscs.INT_STS_1 */
	0x01002465, 0x00010001,	/* cond: miscs.INT_MASK_1 */
	0x08014c00, 0x00010001,	/* cond: opte.PRTY_STS_H_0 */
	0x08014c01, 0x00010001,	/* cond: opte.PRTY_MASK_H_0 */
	0x40060084, 0x00010001,	/* cond: igu.PRTY_STS_H_1 */
	0x40060085, 0x00010001,	/* cond: igu.PRTY_MASK_H_1 */
	0x46140391, 0x00010001,	/* cond: nig.rx_macfifo_empty */
	0x07010000, 0x00010001,	/* cond: ncsi.PRTY_STS_H_0 */
	0x07010001, 0x00010001,	/* cond: ncsi.PRTY_MASK_H_0 */
	0x490080b7, 0x00010001,	/* cond: ipc.INT_STS_0 */
	0x490080b8, 0x00010001,	/* cond: ipc.INT_MASK_0 */
	0x42014460, 0x00010001,	/* cond: umac.INT_STS */
	0x42014460, 0x00010001,	/* cond: umac.INT_STS */
	0x4b1c0003, 0x00010001,	/* cond: nws.common_status */
	0x4b1c0003, 0x00010001,	/* cond: nws.common_status */
	0x4b1c0003, 0x00010001,	/* cond: nws.common_status */
	0x4b1c0060, 0x00010001,	/* cond: nws.INT_STS_0 */
	0x4b1c0080, 0x00010001,	/* cond: nws.PRTY_STS_H_0 */
	0x4b1c0081, 0x00010001,	/* cond: nws.PRTY_MASK_H_0 */
	0x4b1c8180, 0x00010001,	/* cond: nws.nws_cmu.phy0_top_err_ctrl0 */
	0x4b1c8181, 0x00010001,	/* cond: nws.nws_cmu.phy0_top_err_ctrl1 */
	0x4b1c8182, 0x00010001,	/* cond: nws.nws_cmu.phy0_top_err_ctrl2 */
	0x4b1c8188, 0x00010001,	/* cond: nws.nws_cmu.phy0_top_regbus_err_info_status0 */
	0x4b1c8188, 0x00010001,	/* cond: nws.nws_cmu.phy0_top_regbus_err_info_status0 */
	0x4b1c8188, 0x00010001,	/* cond: nws.nws_cmu.phy0_top_regbus_err_info_status0 */
	0x4b1c8188, 0x00010001,	/* cond: nws.nws_cmu.phy0_top_regbus_err_info_status0 */
	0x4b1c8189, 0x00010001,	/* cond: nws.nws_cmu.phy0_top_regbus_err_info_status1 */
	0x4b1c818a, 0x00010001,	/* cond: nws.nws_cmu.phy0_top_regbus_err_info_status2 */
	0x4b1c818b, 0x00010001,	/* cond: nws.nws_cmu.phy0_top_regbus_err_info_status3 */
	0x4b1c8852, 0x00010001,	/* cond: nws.nws_cmu.cmu_lc0_top_phy_if_status */
	0x4b1c8880, 0x00010001,	/* cond: nws.nws_cmu.cmu_lc0_top_err_ctrl1 */
	0x4b1c8881, 0x00010001,	/* cond: nws.nws_cmu.cmu_lc0_top_err_ctrl2 */
	0x4b1c8882, 0x00010001,	/* cond: nws.nws_cmu.cmu_lc0_top_err_ctrl3 */
	0x4b1c8c52, 0x00010001,	/* cond: nws.nws_cmu.cmu_r0_top_phy_if_status */
	0x4b1c8c80, 0x00010001,	/* cond: nws.nws_cmu.cmu_r0_top_err_ctrl1 */
	0x4b1c8c81, 0x00010001,	/* cond: nws.nws_cmu.cmu_r0_top_err_ctrl2 */
	0x4b1c8c82, 0x00010001,	/* cond: nws.nws_cmu.cmu_r0_top_err_ctrl3 */
	0x4b1c9850, 0x00010001,	/* cond: nws.nws_cmu.ln0_top_err_ctrl1 */
	0x4b1c9851, 0x00010001,	/* cond: nws.nws_cmu.ln0_top_err_ctrl2 */
	0x4b1c9852, 0x00010001,	/* cond: nws.nws_cmu.ln0_top_err_ctrl3 */
	0x4b1ca050, 0x00010001,	/* cond: nws.nws_cmu.ln1_top_err_ctrl1 */
	0x4b1ca051, 0x00010001,	/* cond: nws.nws_cmu.ln1_top_err_ctrl2 */
	0x4b1ca052, 0x00010001,	/* cond: nws.nws_cmu.ln1_top_err_ctrl3 */
	0x4b1ca850, 0x00010001,	/* cond: nws.nws_cmu.ln2_top_err_ctrl1 */
	0x4b1ca851, 0x00010001,	/* cond: nws.nws_cmu.ln2_top_err_ctrl2 */
	0x4b1ca852, 0x00010001,	/* cond: nws.nws_cmu.ln2_top_err_ctrl3 */
	0x4b1cb050, 0x00010001,	/* cond: nws.nws_cmu.ln3_top_err_ctrl1 */
	0x4b1cb051, 0x00010001,	/* cond: nws.nws_cmu.ln3_top_err_ctrl2 */
	0x4b1cb052, 0x00010001,	/* cond: nws.nws_cmu.ln3_top_err_ctrl3 */
	0x2208e076, 0x00010001,	/* cond: src.INT_STS */
	0x2208e079, 0x00010001,	/* cond: src.INT_MASK */
	0x1f0b0060, 0x00010001,	/* cond: tm.INT_STS_0 */
	0x1f0b0060, 0x00010001,	/* cond: tm.INT_STS_0 */
	0x1f0b0060, 0x00010001,	/* cond: tm.INT_STS_0 */
	0x1f0b0064, 0x00010001,	/* cond: tm.INT_STS_1 */
	0x1f0b0080, 0x00010001,	/* cond: tm.PRTY_STS_H_0 */
	0x1f0b0081, 0x00010001,	/* cond: tm.PRTY_MASK_H_0 */
	0x1f0b01c4, 0x00010001,	/* cond: tm.pxp_read_data_fifo_status */
	0x1f0b01c6, 0x00010001,	/* cond: tm.pxp_read_ctrl_fifo_status */
	0x1f0b01c8, 0x00010001,	/* cond: tm.cfc_load_echo_fifo_status */
	0x1f0b01c9, 0x00010001,	/* cond: tm.client_out_fifo_status */
	0x1f0b01ca, 0x00010001,	/* cond: tm.client_in_pbf_fifo_status */
	0x1f0b01cb, 0x00010001,	/* cond: tm.client_in_xcm_fifo_status */
	0x1f0b01cc, 0x00010001,	/* cond: tm.client_in_tcm_fifo_status */
	0x1f0b01cd, 0x00010001,	/* cond: tm.client_in_ucm_fifo_status */
	0x1f0b01ce, 0x00010001,	/* cond: tm.expiration_cmd_fifo_status */
	0x1f0b01cf, 0x00010001,	/* cond: tm.ac_command_fifo_status */
	0x1f0b01d5, 0x00010001,	/* cond: tm.debug_0_error_valid */
	0x1f0b01d6, 0x01510001,	/* info: tm.debug_0_cid mode=!fpga */
	0x1f0b01d7, 0x01510001,	/* info: tm.debug_0_lcid mode=!fpga */
	0x1f0b01d8, 0x01510001,	/* info: tm.debug_0_dont_dec_ac mode=!fpga */
	0x1f0b01d9, 0x01510001,	/* info: tm.debug_0_command mode=!fpga */
	0x1f0b01da, 0x01510001,	/* info: tm.debug_0_log_client_num mode=!fpga */
	0x1f0b01db, 0x01510001,	/* info: tm.debug_0_type mode=!fpga */
	0x1f0b01dc, 0x01510001,	/* info: tm.debug_0_leader_type mode=!fpga */
	0x1f0b01dd, 0x01510001,	/* info: tm.debug_0_source mode=!fpga */
	0x1f0b01de, 0x01510001,	/* info: tm.debug_0_context_status mode=!fpga */
	0x1f0b01df, 0x01510001,	/* info: tm.debug_0_error_type mode=!fpga */
	0x1f0b01e0, 0x00010001,	/* cond: tm.debug_1_error_valid */
	0x1f0b01e1, 0x01510001,	/* info: tm.debug_1_cid mode=!fpga */
	0x1f0b01e2, 0x01510001,	/* info: tm.debug_1 mode=!fpga */
	0x1f0b01e3, 0x00010001,	/* cond: tm.debug_2_error_valid */
	0x1f0b01e4, 0x01510001,	/* info: tm.debug_2 mode=!fpga */
	0x1f0b01e5, 0x00010001,	/* cond: tm.debug_3_error_valid */
	0x1f0b01e6, 0x01510001,	/* info: tm.debug_3 mode=!fpga */
	0x1f0b01e8, 0x00010001,	/* cond: tm.debug_4_error_valid */
	0x1f0b01e9, 0x01510001,	/* info: tm.debug_4 mode=!fpga */
	0x3f0b4001, 0x00010001,	/* cond: tcfc.ll_init_done */
	0x3f0b4002, 0x00010001,	/* cond: tcfc.ac_init_done */
	0x3f0b4003, 0x00010001,	/* cond: tcfc.cam_init_done */
	0x3f0b4004, 0x00010001,	/* cond: tcfc.tidram_init_done */
	0x3f0b4060, 0x00010001,	/* cond: tcfc.INT_STS_0 */
	0x3f0b4061, 0x00010001,	/* cond: tcfc.INT_MASK_0 */
	0x3f0b4153, 0x01510001,	/* info: tcfc.error_vector mode=!fpga */
	0x3f0b4156, 0x01510001,	/* info: tcfc.error_data1 mode=!fpga */
	0x3f0b4157, 0x01510001,	/* info: tcfc.error_data2 mode=!fpga */
	0x3f0b4158, 0x01510001,	/* info: tcfc.error_data3 mode=!fpga */
	0x3f0b4159, 0x01510001,	/* info: tcfc.error_data4 mode=!fpga */
	0x3f0b4080, 0x00010001,	/* cond: tcfc.PRTY_STS_H_0 */
	0x3f0b4081, 0x00010001,	/* cond: tcfc.PRTY_MASK_H_0 */
	0x3f0b4179, 0x00010001,	/* cond: tcfc.PRTY_STS */
	0x3f0b4179, 0x00010001,	/* cond: tcfc.PRTY_STS */
	0x3f0b4184, 0x00010001,	/* cond: tcfc.lstate_arriving */
	0x3f0b4187, 0x00010001,	/* cond: tcfc.lstate_leaving */
	0x3f0b41e2, 0x00010001,	/* cond: tcfc.cduld_credit */
	0x3f0b41e3, 0x00010001,	/* cond: tcfc.cduwb_credit */
	0x49008143, 0x00010001,	/* cond: ipc.INT_STS_0 */
	0x49008144, 0x00010001,	/* cond: ipc.INT_MASK_0 */
	0x4a200001, 0x00010001,	/* cond: nwm.INT_STS */
	0x4a200001, 0x00010001,	/* cond: nwm.INT_STS */
	0x4a200080, 0x00010001,	/* cond: nwm.PRTY_STS_H_0 */
	0x4a200081, 0x00010001,	/* cond: nwm.PRTY_MASK_H_0 */
	0x4a200084, 0x00010001,	/* cond: nwm.PRTY_STS_H_1 */
	0x4a200085, 0x00010001,	/* cond: nwm.PRTY_MASK_H_1 */
	0x4a200088, 0x00010001,	/* cond: nwm.PRTY_STS_H_2 */
	0x4a200089, 0x00010001,	/* cond: nwm.PRTY_MASK_H_2 */
	0x49008147, 0x00010001,	/* cond: ipc.PRTY_STS */
	0x49008148, 0x00010001,	/* cond: ipc.PRTY_MASK */
};

/* Data size: 16448 bytes */

/* Idle check immediates */
static const u32 idle_chk_imms[] = {
	0x00000000, 0x000000b0, 0x00000000, 0x0000000e, 0x00000000, 0x34000000,
	0x00000000, 0x00000060, 0x00000000, 0x00100000, 0x00000000, 0x00400000,
	0x00000000, 0x37f00000, 0x00000000, 0x00200000, 0x00000000, 0x02000000,
	0x00000000, 0x00800000, 0x00000000, 0x01000000, 0x00000000, 0x00000005,
	0x00000010, 0x00000002, 0x00000000, 0x00000001, 0x00000020, 0x0000001f,
	0x000000b7, 0x00000004, 0x00000008, 0x00000003, 0x00022aab, 0x0000eaaa,
	0x00000011, 0x00000001, 0x00000000, 0x0000000f, 0xffffffff, 0x00000030,
	0x00000001, 0x00000000, 0x00000001, 0x00000000, 0x00000001, 0x00000000,
	0x00000004, 0x00000000, 0x00000001, 0x00000000, 0x00000008, 0x00000000,
	0x00000001, 0x00000000, 0x00000002, 0x00000000, 0x00000100, 0xffffffff,
	0x000000ff, 0x000fffff, 0x0000ffff, 0x000000ff, 0x00000000, 0x00000007,
	0x00000000, 0x80000000, 0x00000000, 0x00001ffe, 0x0000002e, 0x000001ff,
	0x00000040, 0x00000028, 0x0000002a, 0x00000034, 0x0000001d, 0x0000003f,
	0x00000018, 0x00000026, 0x0000000a, 0x0000001a, 0x00000017, 0x0000000f,
	0x00000000, 0x0000002c, 0x0000003c, 0x00000032, 0x00003f02, 0x00000000,
	0x0076417c, 0x00000000, 0x0001bc01, 0x00000000, 0x00000030, 0x00000004,
	0x00000003, 0x3fffffc0, 0x00000006, 0xc0000000, 0x0000001e, 0x003fffff,
	0x00000002, 0x0070417c, 0x00000000, 0x000003e1, 0x00000381, 0x000fd010,
	0x00000000, 0x00002000, 0x00000000, 0x0000e001, 0x00000000, 0x02040902,
	0x00000000, 0x10240902, 0x00000000, 0x00000004, 0x00000004, 0x00000002,
	0x00000002, 0x007fffff, 0x7f800000, 0x0000041e, 0x000001fe, 0x00000000,
};

/* Data size: 504 bytes */

/* Idle check rules */
static const u32 idle_chk_rules[] = {
	0x0ea30000,		/* mode all */
	0x00020000, 0x00010002, 0x00000000,	/* ((r1&~r2)!=0), r1=pglcs.INT_STS, r2=pglcs.INT_MASK,  */
	0x00000001, 0x00010002, 0x00000002,	/* ((r1&~r2)!=0), r1=misc.INT_STS, r2=misc.INT_MASK,  */
	0x01000002, 0x00010101, 0x00000004,	/* (r1!=0), r1=misc.aeu_after_invert_1_igu,  */
	0x04000003, 0x00020101, 0x00010006,	/* ((r1&~0xB0)!=0), r1=misc.aeu_after_invert_2_igu,  */
	0x04000004, 0x00020101, 0x00030008,	/* ((r1&~0xE)!=0), r1=misc.aeu_after_invert_4_igu,  */
	0x01000005, 0x00010101, 0x0000000a,	/* (r1!=0), r1=misc.aeu_after_invert_5_igu,  */
	0x01000006, 0x00010101, 0x0000000c,	/* (r1!=0), r1=misc.aeu_after_invert_6_igu,  */
	0x01000007, 0x00010101, 0x0000000e,	/* (r1!=0), r1=misc.aeu_after_invert_7_igu,  */
	0x04000008, 0x00020101, 0x00050010,	/* ((r1&~0x34000000)!=0), r1=misc.aeu_after_invert_8_igu,  */
	0x04000009, 0x00020101, 0x00070012,	/* ((r1&~0x60)!=0), r1=misc.aeu_after_invert_9_igu,  */
	0x0100000a, 0x00010101, 0x00000014,	/* (r1!=0), r1=misc.aeu_after_invert_1_mcp,  */
	0x0400000b, 0x00020101, 0x00010016,	/* ((r1&~0xB0)!=0), r1=misc.aeu_after_invert_2_mcp,  */
	0x0400000c, 0x00020101, 0x00030018,	/* ((r1&~0xE)!=0), r1=misc.aeu_after_invert_4_mcp,  */
	0x0100000d, 0x00010101, 0x0000001a,	/* (r1!=0), r1=misc.aeu_after_invert_5_mcp,  */
	0x0100000e, 0x00010101, 0x0000001c,	/* (r1!=0), r1=misc.aeu_after_invert_6_mcp,  */
	0x0100000f, 0x00010101, 0x0000001e,	/* (r1!=0), r1=misc.aeu_after_invert_7_mcp,  */
	0x06000010, 0x00020201, 0x00090020,	/* ((r1&0x00100000)!=0), r1=misc.aeu_after_invert_8_mcp,  */
	0x06000011, 0x00020201, 0x000b0023,	/* ((r1&0x00400000)!=0), r1=misc.aeu_after_invert_8_mcp,  */
	0x04000012, 0x00020101, 0x000d0026,	/* ((r1&~0x37f00000)!=0), r1=misc.aeu_after_invert_8_mcp,  */
	0x06000013, 0x00020201, 0x000f0028,	/* ((r1&0x00200000)!=0), r1=misc.aeu_after_invert_8_mcp,  */
	0x06000014, 0x00020201, 0x0011002b,	/* ((r1&0x02000000)!=0), r1=misc.aeu_after_invert_8_mcp,  */
	0x06000015, 0x00020201, 0x0013002e,	/* ((r1&0x00800000)!=0), r1=misc.aeu_after_invert_8_mcp,  */
	0x06000016, 0x00020201, 0x00150031,	/* ((r1&0x01000000)!=0), r1=misc.aeu_after_invert_8_mcp,  */
	0x04000017, 0x00020101, 0x00070034,	/* ((r1&~0x60)!=0), r1=misc.aeu_after_invert_9_mcp,  */
	0x01000018, 0x00010001, 0x00000036,	/* (r1!=0), r1=misc.aeu_sys_kill_occurred,  */
	0x00000019, 0x00010002, 0x00000037,	/* ((r1&~r2)!=0), r1=miscs.INT_STS_0, r2=miscs.INT_MASK_0,  */
	0x0102001a, 0x00010001, 0x00000039,	/* (r1!=0), r1=miscs.pcie_hot_reset,  */
	0x0101001b, 0x00010001, 0x0000003a,	/* (r1!=0), r1=dmae.go_c0,  */
	0x0101001c, 0x00010001, 0x0000003b,	/* (r1!=0), r1=dmae.go_c1,  */
	0x0101001d, 0x00010001, 0x0000003c,	/* (r1!=0), r1=dmae.go_c2,  */
	0x0101001e, 0x00010001, 0x0000003d,	/* (r1!=0), r1=dmae.go_c3,  */
	0x0101001f, 0x00010001, 0x0000003e,	/* (r1!=0), r1=dmae.go_c4,  */
	0x01010020, 0x00010001, 0x0000003f,	/* (r1!=0), r1=dmae.go_c5,  */
	0x01010021, 0x00010001, 0x00000040,	/* (r1!=0), r1=dmae.go_c6,  */
	0x01010022, 0x00010001, 0x00000041,	/* (r1!=0), r1=dmae.go_c7,  */
	0x01010023, 0x00010001, 0x00000042,	/* (r1!=0), r1=dmae.go_c8,  */
	0x01010024, 0x00010001, 0x00000043,	/* (r1!=0), r1=dmae.go_c9,  */
	0x01010025, 0x00010001, 0x00000044,	/* (r1!=0), r1=dmae.go_c10,  */
	0x01010026, 0x00010001, 0x00000045,	/* (r1!=0), r1=dmae.go_c11,  */
	0x01010027, 0x00010001, 0x00000046,	/* (r1!=0), r1=dmae.go_c12,  */
	0x01010028, 0x00010001, 0x00000047,	/* (r1!=0), r1=dmae.go_c13,  */
	0x01010029, 0x00010001, 0x00000048,	/* (r1!=0), r1=dmae.go_c14,  */
	0x0101002a, 0x00010001, 0x00000049,	/* (r1!=0), r1=dmae.go_c15,  */
	0x0101002b, 0x00010001, 0x0000004a,	/* (r1!=0), r1=dmae.go_c16,  */
	0x0101002c, 0x00010001, 0x0000004b,	/* (r1!=0), r1=dmae.go_c17,  */
	0x0101002d, 0x00010001, 0x0000004c,	/* (r1!=0), r1=dmae.go_c18,  */
	0x0101002e, 0x00010001, 0x0000004d,	/* (r1!=0), r1=dmae.go_c19,  */
	0x0101002f, 0x00010001, 0x0000004e,	/* (r1!=0), r1=dmae.go_c20,  */
	0x01010030, 0x00010001, 0x0000004f,	/* (r1!=0), r1=dmae.go_c21,  */
	0x01010031, 0x00010001, 0x00000050,	/* (r1!=0), r1=dmae.go_c22,  */
	0x01010032, 0x00010001, 0x00000051,	/* (r1!=0), r1=dmae.go_c23,  */
	0x01010033, 0x00010001, 0x00000052,	/* (r1!=0), r1=dmae.go_c24,  */
	0x01010034, 0x00010001, 0x00000053,	/* (r1!=0), r1=dmae.go_c25,  */
	0x01010035, 0x00010001, 0x00000054,	/* (r1!=0), r1=dmae.go_c26,  */
	0x01010036, 0x00010001, 0x00000055,	/* (r1!=0), r1=dmae.go_c27,  */
	0x01010037, 0x00010001, 0x00000056,	/* (r1!=0), r1=dmae.go_c28,  */
	0x01010038, 0x00010001, 0x00000057,	/* (r1!=0), r1=dmae.go_c29,  */
	0x01010039, 0x00010001, 0x00000058,	/* (r1!=0), r1=dmae.go_c30,  */
	0x0101003a, 0x00010001, 0x00000059,	/* (r1!=0), r1=dmae.go_c31,  */
	0x0000003b, 0x00010002, 0x0000005a,	/* ((r1&~r2)!=0), r1=dmae.INT_STS, r2=dmae.INT_MASK,  */
	0x0000003c, 0x00010002, 0x0000005c,	/* ((r1&~r2)!=0), r1=dmae.PRTY_STS_H_0, r2=dmae.PRTY_MASK_H_0,  */
	0x0000003d, 0x00010002, 0x0000005e,	/* ((r1&~r2)!=0), r1=dbg.INT_STS, r2=dbg.INT_MASK,  */
	0x0000003e, 0x00010002, 0x00000060,	/* ((r1&~r2)!=0), r1=dbg.PRTY_STS_H_0, r2=dbg.PRTY_MASK_H_0,  */
	0x0102003f, 0x00010001, 0x00000062,	/* (r1!=0), r1=grc.trace_fifo_valid_data,  */
	0x00000040, 0x00010002, 0x00000063,	/* ((r1&~r2)!=0), r1=grc.INT_STS_0, r2=grc.INT_MASK_0,  */
	0x00000041, 0x00010002, 0x00000065,	/* ((r1&~r2)!=0), r1=grc.PRTY_STS_H_0, r2=grc.PRTY_MASK_H_0,  */
	0x00000042, 0x00010002, 0x00000067,	/* ((r1&~r2)!=0), r1=mcp2.PRTY_STS, r2=mcp2.PRTY_MASK,  */
	0x00000043, 0x00010002, 0x00000069,	/* ((r1&~r2)!=0), r1=mcp2.PRTY_STS_H_0, r2=mcp2.PRTY_MASK_H_0,  */
	0x00000044, 0x00010002, 0x0000006b,	/* ((r1&~r2)!=0), r1=dorq.INT_STS, r2=dorq.INT_MASK,  */
	0x00000045, 0x00010002, 0x0000006d,	/* ((r1&~r2)!=0), r1=dorq.PRTY_STS, r2=dorq.PRTY_MASK,  */
	0x00000046, 0x00010002, 0x0000006f,	/* ((r1&~r2)!=0), r1=dorq.PRTY_STS_H_0, r2=dorq.PRTY_MASK_H_0,  */
	0x01000047, 0x00010001, 0x00170071,	/* (r1!=reset1), r1=dorq.xcm_msg_init_crd,  */
	0x01000048, 0x00010001, 0x00170072,	/* (r1!=reset1), r1=dorq.tcm_msg_init_crd,  */
	0x01000049, 0x00010001, 0x00170073,	/* (r1!=reset1), r1=dorq.ucm_msg_init_crd,  */
	0x0100004a, 0x00010001, 0x00180074,	/* (r1!=reset1), r1=dorq.pbf_cmd_init_crd,  */
	0x0100004b, 0x00010001, 0x00000075,	/* (r1!=0), r1=dorq.pf_usage_cnt,  */
	0x0100004c, 0x00010001, 0x00000076,	/* (r1!=0), r1=dorq.vf_usage_cnt,  */
	0x0100004d, 0x00010001, 0x00000077,	/* (r1!=0), r1=dorq.cfc_ld_req_fifo_fill_lvl,  */
	0x0100004e, 0x00010001, 0x00000078,	/* (r1!=0), r1=dorq.dorq_fifo_fill_lvl,  */
	0x0102004f, 0x00010201, 0x00000079,	/* (r1!=0), r1=dorq.db_drop_cnt,  */
	0x01020050, 0x00010201, 0x0000007c,	/* (r1!=0), r1=dorq.dpm_abort_cnt,  */
	0x01000051, 0x00010001, 0x0000007f,	/* (r1!=0), r1=dorq.dpm_timer_expir_abort_cnt,  */
	0x01000052, 0x00010001, 0x00000080,	/* (r1!=0), r1=dorq.dpm_tbl_fill_lvl,  */
	0x00000053, 0x00010002, 0x00000081,	/* ((r1&~r2)!=0), r1=igu.INT_STS, r2=igu.INT_MASK,  */
	0x00000054, 0x00010002, 0x00000083,	/* ((r1&~r2)!=0), r1=igu.PRTY_STS, r2=igu.PRTY_MASK,  */
	0x00000055, 0x00010002, 0x00000085,	/* ((r1&~r2)!=0), r1=igu.PRTY_STS_H_0, r2=igu.PRTY_MASK_H_0,  */
	0x01010056, 0x00010001, 0x00000087,	/* (r1!=0), r1=igu.attn_write_done_pending,  */
	0x01020057, 0x00010001, 0x00000088,	/* (r1!=0), r1=igu.Interrupt_status,  */
	0x01000058, 0x00010001, 0x00000089,	/* (r1!=0), r1=igu.error_handling_data_valid,  */
	0x01000059, 0x00010001, 0x0000008a,	/* (r1!=0), r1=igu.silent_drop,  */
	0x0102005a, 0x00010001, 0x0000008b,	/* (r1!=0), r1=igu.sb_ctrl_fsm,  */
	0x0102005b, 0x00010001, 0x0000008c,	/* (r1!=0), r1=igu.int_handle_fsm,  */
	0x0402005c, 0x00020001, 0x0019008d,	/* ((r1&~0x2)!=0), r1=igu.attn_fsm,  */
	0x0102005d, 0x00010001, 0x0000008e,	/* (r1!=0), r1=igu.ctrl_fsm,  */
	0x0102005e, 0x00010001, 0x0000008f,	/* (r1!=0), r1=igu.pxp_arb_fsm,  */
	0x0002005f, 0x00010002, 0x00000090,	/* ((r1&~r2)!=0), r1=cau.INT_STS, r2=cau.INT_MASK,  */
	0x00000060, 0x00010002, 0x00000092,	/* ((r1&~r2)!=0), r1=cau.PRTY_STS_H_0, r2=cau.PRTY_MASK_H_0,  */
	0x01010061, 0x00010001, 0x001b0094,	/* (r1!=1), r1=cau.igu_req_credit_status,  */
	0x01010062, 0x00010001, 0x001b0095,	/* (r1!=1), r1=cau.igu_cmd_credit_status,  */
	0x01010063, 0x00010001, 0x00000096,	/* (r1!=0), r1=cau.debug_fifo_status,  */
	0x01000064, 0x00010001, 0x00000097,	/* (r1!=0), r1=cau.error_pxp_req,  */
	0x01000065, 0x00010101, 0x00000098,	/* (r1!=0), r1=cau.error_fsm_line,  */
	0x01000066, 0x00010001, 0x0000009a,	/* (r1!=0), r1=cau.parity_latch_status,  */
	0x01000067, 0x00010001, 0x0000009b,	/* (r1!=0), r1=cau.error_cleanup_cmd_reg,  */
	0x01020068, 0x00010001, 0x0000009c,	/* (r1!=0), r1=cau.agg_units_0to15_state,  */
	0x01020069, 0x00010001, 0x0000009d,	/* (r1!=0), r1=cau.agg_units_16to31_state,  */
	0x0102006a, 0x00010001, 0x0000009e,	/* (r1!=0), r1=cau.agg_units_32to47_state,  */
	0x0102006b, 0x00010001, 0x0000009f,	/* (r1!=0), r1=cau.agg_units_48to63_state,  */
	0x0301006c, 0x00000002, 0x000000a0,	/* (r1!=r2), r1=cau.req_counter, r2=cau.wdone_counter,  */
	0x0301006d, 0x00000002, 0x000000a2,	/* (r1!=r2), r1=cau.req_counter, r2=cau.ack_counter,  */
	0x0102006e, 0x00010001, 0x000000a4,	/* (r1!=0), r1=cau.main_fsm_status,  */
	0x0102006f, 0x00010001, 0x000000a5,	/* (r1!=0), r1=cau.var_read_fsm_status,  */
	0x01020070, 0x00010001, 0x000000a6,	/* (r1!=0), r1=cau.igu_dma_fsm_status,  */
	0x01020071, 0x00010001, 0x000000a7,	/* (r1!=0), r1=cau.igu_cqe_cmd_fsm_status,  */
	0x01020072, 0x00010001, 0x000000a8,	/* (r1!=0), r1=cau.igu_cqe_agg_fsm_status,  */
	0x01020073, 0x00010001, 0x001b00a9,	/* (r1!=0x01), r1=prs.mac_vlan_cache_init_done,  */
	0x00000074, 0x00010302, 0x000000aa,	/* ((r1&~r2)!=0), r1=prs.INT_STS_0, r2=prs.INT_MASK_0,  */
	0x00000075, 0x00010002, 0x000000af,	/* ((r1&~r2)!=0), r1=prs.PRTY_STS, r2=prs.PRTY_MASK,  */
	0x00000076, 0x00010002, 0x000000b1,	/* ((r1&~r2)!=0), r1=prs.PRTY_STS_H_0, r2=prs.PRTY_MASK_H_0,  */
	0x00000077, 0x00010002, 0x000000b3,	/* ((r1&~r2)!=0), r1=prs.PRTY_STS_H_1, r2=prs.PRTY_MASK_H_1,  */
	0x01010078, 0x00010001, 0x000000b5,	/* (r1!=0), r1=prs.prs_pkt_ct,  */
	0x01010079, 0x00010001, 0x000000b6,	/* (r1!=0), r1=prs.queue_pkt_avail_status,  */
	0x0101007a, 0x00010001, 0x000000b7,	/* (r1!=0), r1=prs.storm_bkprs_status,  */
	0x0101007b, 0x00010001, 0x000000b8,	/* (r1!=0), r1=prs.stop_parsing_status,  */
	0x0101007c, 0x00010001, 0x000000b9,	/* (r1!=0), r1=prs.tcm_current_credit,  */
	0x0101007d, 0x00010001, 0x000000ba,	/* (r1!=0), r1=prs.ccfc_search_current_credit,  */
	0x0101007e, 0x00010001, 0x000000bb,	/* (r1!=0), r1=prs.tcfc_search_current_credit,  */
	0x0101007f, 0x00010001, 0x000000bc,	/* (r1!=0), r1=prs.ccfc_load_current_credit,  */
	0x01010080, 0x00010001, 0x000000bd,	/* (r1!=0), r1=prs.tcfc_load_current_credit,  */
	0x01010081, 0x00010001, 0x000000be,	/* (r1!=0), r1=prs.ccfc_search_req_ct,  */
	0x01010082, 0x00010001, 0x000000bf,	/* (r1!=0), r1=prs.tcfc_search_req_ct,  */
	0x01010083, 0x00010001, 0x000000c0,	/* (r1!=0), r1=prs.ccfc_load_req_ct,  */
	0x01010084, 0x00010001, 0x000000c1,	/* (r1!=0), r1=prs.tcfc_load_req_ct,  */
	0x01010085, 0x00010001, 0x000000c2,	/* (r1!=0), r1=prs.sop_req_ct,  */
	0x01010086, 0x00010001, 0x000000c3,	/* (r1!=0), r1=prs.eop_req_ct,  */
	0x00000087, 0x00010002, 0x000000c4,	/* ((r1&~r2)!=0), r1=prm.INT_STS, r2=prm.INT_MASK,  */
	0x00000088, 0x00010002, 0x000000c6,	/* ((r1&~r2)!=0), r1=prm.PRTY_STS, r2=prm.PRTY_MASK,  */
	0x00000089, 0x00010002, 0x000000c8,	/* ((r1&~r2)!=0), r1=prm.PRTY_STS_H_0, r2=prm.PRTY_MASK_H_0,  */
	0x0100008a, 0x00010001, 0x001b00ca,	/* (r1!=1), r1=rss.rss_init_done,  */
	0x0000008b, 0x00010002, 0x000000cb,	/* ((r1&~r2)!=0), r1=rss.INT_STS, r2=rss.INT_MASK,  */
	0x0000008c, 0x00010002, 0x000000cd,	/* ((r1&~r2)!=0), r1=rss.PRTY_STS_H_0, r2=rss.PRTY_MASK_H_0,  */
	0x0101008d, 0x00010001, 0x001c00cf,	/* (r1!=reset1), r1=rss.tmld_credit,  */
	0x0101008e, 0x00010001, 0x001d00d0,	/* (r1!=31), r1=rss.empty_status,  */
	0x0101008f, 0x00010001, 0x000000d1,	/* (r1!=0), r1=rss.full_status,  */
	0x01010090, 0x00010001, 0x000000d2,	/* (r1!=0), r1=rss.counters_status,  */
	0x01010091, 0x00010001, 0x000000d3,	/* (r1!=0), r1=rss.state_machines,  */
	0x00020092, 0x00010002, 0x000000d4,	/* ((r1&~r2)!=0), r1=rpb.INT_STS, r2=rpb.INT_MASK,  */
	0x00000093, 0x00010002, 0x000000d6,	/* ((r1&~r2)!=0), r1=rpb.PRTY_STS, r2=rpb.PRTY_MASK,  */
	0x01020094, 0x00010101, 0x001b00d8,	/* (r1!=0x01), r1=rpb.db_empty,  */
	0x01020095, 0x00010101, 0x001b00da,	/* (r1!=0x01), r1=rpb.tq_empty,  */
	0x01020096, 0x00010101, 0x001b00dc,	/* (r1!=0x01), r1=rpb.ififo_empty,  */
	0x01020097, 0x00010101, 0x001b00de,	/* (r1!=0x01), r1=rpb.pfifo_empty,  */
	0x01020098, 0x00010001, 0x001b00e0,	/* (r1!=0x01), r1=rpb.tq_th_empty,  */
	0x01000099, 0x00010001, 0x001b00e1,	/* (r1!=1), r1=pswrq2.rbc_done,  */
	0x0100009a, 0x00010001, 0x001b00e2,	/* (r1!=1), r1=pswrq2.cfg_done,  */
	0x0002009b, 0x00010002, 0x000000e3,	/* ((r1&~r2)!=0), r1=pswrq2.INT_STS, r2=pswrq2.INT_MASK,  */
	0x0000009c, 0x00010002, 0x000000e5,	/* ((r1&~r2)!=0), r1=pswrq2.PRTY_STS_H_0, r2=pswrq2.PRTY_MASK_H_0,  */
	0x0101009d, 0x00010001, 0x000000e7,	/* (r1!=0), r1=pswrq2.vq0_entry_cnt,  */
	0x0101009e, 0x00010001, 0x000000e8,	/* (r1!=0), r1=pswrq2.vq1_entry_cnt,  */
	0x0101009f, 0x00010001, 0x000000e9,	/* (r1!=0), r1=pswrq2.vq2_entry_cnt,  */
	0x010100a0, 0x00010001, 0x000000ea,	/* (r1!=0), r1=pswrq2.vq3_entry_cnt,  */
	0x010100a1, 0x00010001, 0x000000eb,	/* (r1!=0), r1=pswrq2.vq4_entry_cnt,  */
	0x010100a2, 0x00010001, 0x000000ec,	/* (r1!=0), r1=pswrq2.vq5_entry_cnt,  */
	0x010100a3, 0x00010001, 0x000000ed,	/* (r1!=0), r1=pswrq2.vq6_entry_cnt,  */
	0x010100a4, 0x00010001, 0x000000ee,	/* (r1!=0), r1=pswrq2.vq7_entry_cnt,  */
	0x010100a5, 0x00010001, 0x000000ef,	/* (r1!=0), r1=pswrq2.vq8_entry_cnt,  */
	0x010100a6, 0x00010001, 0x000000f0,	/* (r1!=0), r1=pswrq2.vq9_entry_cnt,  */
	0x010100a7, 0x00010001, 0x000000f1,	/* (r1!=0), r1=pswrq2.vq10_entry_cnt,  */
	0x010100a8, 0x00010001, 0x000000f2,	/* (r1!=0), r1=pswrq2.vq11_entry_cnt,  */
	0x010100a9, 0x00010001, 0x000000f3,	/* (r1!=0), r1=pswrq2.vq12_entry_cnt,  */
	0x010100aa, 0x00010001, 0x000000f4,	/* (r1!=0), r1=pswrq2.vq13_entry_cnt,  */
	0x010100ab, 0x00010001, 0x000000f5,	/* (r1!=0), r1=pswrq2.vq14_entry_cnt,  */
	0x010100ac, 0x00010001, 0x000000f6,	/* (r1!=0), r1=pswrq2.vq15_entry_cnt,  */
	0x010100ad, 0x00010001, 0x000000f7,	/* (r1!=0), r1=pswrq2.vq16_entry_cnt,  */
	0x010100ae, 0x00010001, 0x000000f8,	/* (r1!=0), r1=pswrq2.vq17_entry_cnt,  */
	0x010100af, 0x00010001, 0x000000f9,	/* (r1!=0), r1=pswrq2.vq18_entry_cnt,  */
	0x010100b0, 0x00010001, 0x000000fa,	/* (r1!=0), r1=pswrq2.vq19_entry_cnt,  */
	0x010100b1, 0x00010001, 0x000000fb,	/* (r1!=0), r1=pswrq2.vq20_entry_cnt,  */
	0x010100b2, 0x00010001, 0x000000fc,	/* (r1!=0), r1=pswrq2.vq21_entry_cnt,  */
	0x010100b3, 0x00010001, 0x000000fd,	/* (r1!=0), r1=pswrq2.vq22_entry_cnt,  */
	0x010100b4, 0x00010001, 0x000000fe,	/* (r1!=0), r1=pswrq2.vq23_entry_cnt,  */
	0x010100b5, 0x00010001, 0x000000ff,	/* (r1!=0), r1=pswrq2.vq24_entry_cnt,  */
	0x010100b6, 0x00010001, 0x00000100,	/* (r1!=0), r1=pswrq2.vq25_entry_cnt,  */
	0x010100b7, 0x00010001, 0x00000101,	/* (r1!=0), r1=pswrq2.vq26_entry_cnt,  */
	0x010100b8, 0x00010001, 0x00000102,	/* (r1!=0), r1=pswrq2.vq27_entry_cnt,  */
	0x010100b9, 0x00010001, 0x00000103,	/* (r1!=0), r1=pswrq2.vq28_entry_cnt,  */
	0x010100ba, 0x00010001, 0x00000104,	/* (r1!=0), r1=pswrq2.vq29_entry_cnt,  */
	0x010100bb, 0x00010001, 0x00000105,	/* (r1!=0), r1=pswrq2.vq30_entry_cnt,  */
	0x010100bc, 0x00010001, 0x00000106,	/* (r1!=0), r1=pswrq2.vq31_entry_cnt,  */
	0x010000bd, 0x00010001, 0x001e0107,	/* (r1!=reset1), r1=pswrq2.BW_CREDIT,  */
	0x010100be, 0x00010001, 0x00000108,	/* (r1!=0), r1=pswrq2.treq_fifo_fill_lvl,  */
	0x010100bf, 0x00010001, 0x00000109,	/* (r1!=0), r1=pswrq2.icpl_fifo_fill_lvl,  */
	0x010000c0, 0x00010201, 0x0000010a,	/* (r1!=0), r1=pswrq2.l2p_err_details,  */
	0x010000c1, 0x00010001, 0x0000010d,	/* (r1!=0), r1=pswrq2.l2p_err_details2,  */
	0x030100c2, 0x00000002, 0x0000010e,	/* (r1!=r2), r1=pswrq2.sr_cnt, r2=pswrq2.sr_num_cfg,  */
	0x080100c3, 0x00010002, 0x001f0110,	/* (r1<(r2-4)), r1=pswrq2.sr_cnt, r2=pswrq2.sr_num_cfg,  */
	0x030100c4, 0x00000002, 0x00000112,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_0, r2=pswrq2.max_srs_vq0,  */
	0x030100c5, 0x00000002, 0x00000114,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_1, r2=pswrq2.max_srs_vq1,  */
	0x030100c6, 0x00000002, 0x00000116,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_2, r2=pswrq2.max_srs_vq2,  */
	0x030100c7, 0x00000002, 0x00000118,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_3, r2=pswrq2.max_srs_vq3,  */
	0x030100c8, 0x00000002, 0x0000011a,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_4, r2=pswrq2.max_srs_vq4,  */
	0x030100c9, 0x00000002, 0x0000011c,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_5, r2=pswrq2.max_srs_vq5,  */
	0x030100ca, 0x00000002, 0x0000011e,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_6, r2=pswrq2.max_srs_vq6,  */
	0x030100cb, 0x00000002, 0x00000120,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_9, r2=pswrq2.max_srs_vq9,  */
	0x030100cc, 0x00000002, 0x00000122,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_11, r2=pswrq2.max_srs_vq11,  */
	0x030100cd, 0x00000002, 0x00000124,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_12, r2=pswrq2.max_srs_vq12,  */
	0x030100ce, 0x00000002, 0x00000126,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_15, r2=pswrq2.max_srs_vq15,  */
	0x030100cf, 0x00000002, 0x00000128,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_16, r2=pswrq2.max_srs_vq16,  */
	0x030100d0, 0x00000002, 0x0000012a,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_17, r2=pswrq2.max_srs_vq17,  */
	0x030100d1, 0x00000002, 0x0000012c,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_18, r2=pswrq2.max_srs_vq18,  */
	0x030100d2, 0x00000002, 0x0000012e,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_19, r2=pswrq2.max_srs_vq19,  */
	0x030100d3, 0x00000002, 0x00000130,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_21, r2=pswrq2.max_srs_vq21,  */
	0x030100d4, 0x00000002, 0x00000132,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_22, r2=pswrq2.max_srs_vq22,  */
	0x030100d5, 0x00000002, 0x00000134,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_23, r2=pswrq2.max_srs_vq23,  */
	0x030100d6, 0x00000002, 0x00000136,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_24, r2=pswrq2.max_srs_vq24,  */
	0x030100d7, 0x00000002, 0x00000138,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_26, r2=pswrq2.max_srs_vq26,  */
	0x030100d8, 0x00000002, 0x0000013a,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_28, r2=pswrq2.max_srs_vq28,  */
	0x030100d9, 0x00000002, 0x0000013c,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_29, r2=pswrq2.max_srs_vq29,  */
	0x030100da, 0x00000002, 0x0000013e,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_30, r2=pswrq2.max_srs_vq30,  */
	0x030100db, 0x00000002, 0x00000140,	/* (r1!=r2), r1=pswrq2.blk_cnt, r2=pswrq2.blk_num_cfg,  */
	0x080100dc, 0x00010002, 0x00200142,	/* (r1<(r2-8)), r1=pswrq2.blk_cnt, r2=pswrq2.blk_num_cfg,  */
	0x030100dd, 0x00000002, 0x00000144,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_0, r2=pswrq2.max_blks_vq0,  */
	0x030100de, 0x00000002, 0x00000146,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_1, r2=pswrq2.max_blks_vq1,  */
	0x030100df, 0x00000002, 0x00000148,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_2, r2=pswrq2.max_blks_vq2,  */
	0x030100e0, 0x00000002, 0x0000014a,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_3, r2=pswrq2.max_blks_vq3,  */
	0x030100e1, 0x00000002, 0x0000014c,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_4, r2=pswrq2.max_blks_vq4,  */
	0x030100e2, 0x00000002, 0x0000014e,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_5, r2=pswrq2.max_blks_vq5,  */
	0x030100e3, 0x00000002, 0x00000150,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_6, r2=pswrq2.max_blks_vq6,  */
	0x030100e4, 0x00000002, 0x00000152,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_9, r2=pswrq2.max_blks_vq9,  */
	0x030100e5, 0x00000002, 0x00000154,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_11, r2=pswrq2.max_blks_vq11,  */
	0x030100e6, 0x00000002, 0x00000156,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_12, r2=pswrq2.max_blks_vq12,  */
	0x030100e7, 0x00000002, 0x00000158,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_15, r2=pswrq2.max_blks_vq15,  */
	0x030100e8, 0x00000002, 0x0000015a,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_16, r2=pswrq2.max_blks_vq16,  */
	0x030100e9, 0x00000002, 0x0000015c,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_17, r2=pswrq2.max_blks_vq17,  */
	0x030100ea, 0x00000002, 0x0000015e,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_18, r2=pswrq2.max_blks_vq18,  */
	0x030100eb, 0x00000002, 0x00000160,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_19, r2=pswrq2.max_blks_vq19,  */
	0x030100ec, 0x00000002, 0x00000162,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_21, r2=pswrq2.max_blks_vq21,  */
	0x030100ed, 0x00000002, 0x00000164,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_22, r2=pswrq2.max_blks_vq22,  */
	0x030100ee, 0x00000002, 0x00000166,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_23, r2=pswrq2.max_blks_vq23,  */
	0x030100ef, 0x00000002, 0x00000168,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_24, r2=pswrq2.max_blks_vq24,  */
	0x030100f0, 0x00000002, 0x0000016a,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_26, r2=pswrq2.max_blks_vq26,  */
	0x030100f1, 0x00000002, 0x0000016c,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_28, r2=pswrq2.max_blks_vq28,  */
	0x030100f2, 0x00000002, 0x0000016e,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_29, r2=pswrq2.max_blks_vq29,  */
	0x030100f3, 0x00000002, 0x00000170,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_30, r2=pswrq2.max_blks_vq30,  */
	0x010000f4, 0x00010001, 0x00000172,	/* (r1!=0), r1=pswrq2.l2p_close_gate_sts,  */
	0x010000f5, 0x00010001, 0x00000173,	/* (r1!=0), r1=pswrq2.misc_close_gate_sts,  */
	0x010200f6, 0x00010001, 0x00000174,	/* (r1!=0), r1=pswrq2.misc_stall_mem_sts,  */
	0x000200f7, 0x00010002, 0x00000175,	/* ((r1&~r2)!=0), r1=pswrq.INT_STS, r2=pswrq.INT_MASK,  */
	0x000000f8, 0x00010202, 0x00000177,	/* ((r1&~r2)!=0), r1=pswrq.PRTY_STS, r2=pswrq.PRTY_MASK,  */
	0x000200f9, 0x00010002, 0x0000017b,	/* ((r1&~r2)!=0), r1=pswwr.INT_STS, r2=pswwr.INT_MASK,  */
	0x000000fa, 0x00010002, 0x0000017d,	/* ((r1&~r2)!=0), r1=pswwr.PRTY_STS, r2=pswwr.PRTY_STS,  */
	0x010000fb, 0x00010001, 0x0000017f,	/* (r1!=0), r1=pswwr2.pglue_eop_err_details,  */
	0x010100fc, 0x00010001, 0x00000180,	/* (r1!=0), r1=pswwr2.prm_curr_fill_level,  */
	0x010100fd, 0x00010001, 0x00000181,	/* (r1!=0), r1=pswwr2.cdu_curr_fill_level,  */
	0x000200fe, 0x00010002, 0x00000182,	/* ((r1&~r2)!=0), r1=pswwr2.INT_STS, r2=pswwr2.INT_MASK,  */
	0x000000ff, 0x00010002, 0x00000184,	/* ((r1&~r2)!=0), r1=pswwr2.PRTY_STS, r2=pswwr2.PRTY_STS,  */
	0x00000100, 0x00010002, 0x00000186,	/* ((r1&~r2)!=0), r1=pswwr2.PRTY_STS_H_0, r2=pswwr2.PRTY_MASK_H_0,  */
	0x00000101, 0x00010002, 0x00000188,	/* ((r1&~r2)!=0), r1=pswwr2.PRTY_STS_H_1, r2=pswwr2.PRTY_MASK_H_1,  */
	0x00000102, 0x00010002, 0x0000018a,	/* ((r1&~r2)!=0), r1=pswwr2.PRTY_STS_H_2, r2=pswwr2.PRTY_MASK_H_2,  */
	0x00000103, 0x00010002, 0x0000018c,	/* ((r1&~r2)!=0), r1=pswwr2.PRTY_STS_H_3, r2=pswwr2.PRTY_MASK_H_3,  */
	0x01000104, 0x00010001, 0x0000018e,	/* (r1!=0), r1=pswrd.fifo_full_status,  */
	0x00020105, 0x00010002, 0x0000018f,	/* ((r1&~r2)!=0), r1=pswrd.INT_STS, r2=pswrd.INT_MASK,  */
	0x00000106, 0x00010002, 0x00000191,	/* ((r1&~r2)!=0), r1=pswrd.PRTY_STS, r2=pswrd.PRTY_MASK,  */
	0x01000107, 0x00010001, 0x001b0193,	/* (r1!=1), r1=pswrd2.start_init,  */
	0x01000108, 0x00010001, 0x001b0194,	/* (r1!=1), r1=pswrd2.init_done,  */
	0x01020109, 0x00010001, 0x00000195,	/* (r1!=0), r1=pswrd2.cpl_err_details,  */
	0x0102010a, 0x00010001, 0x00000196,	/* (r1!=0), r1=pswrd2.cpl_err_details2,  */
	0x0101010b, 0x00010001, 0x001b0197,	/* (r1!=1), r1=pswrd2.port_is_idle_0,  */
	0x0101010c, 0x00010001, 0x001b0198,	/* (r1!=1), r1=pswrd2.port_is_idle_1,  */
	0x0100010d, 0x00010001, 0x00000199,	/* (r1!=0), r1=pswrd2.almost_full_0[0:15],  */
	0x0002010e, 0x00010002, 0x0000019a,	/* ((r1&~r2)!=0), r1=pswrd2.INT_STS, r2=pswrd2.INT_MASK,  */
	0x0000010f, 0x00010002, 0x0000019c,	/* ((r1&~r2)!=0), r1=pswrd2.PRTY_STS, r2=pswrd2.PRTY_MASK,  */
	0x00000110, 0x00010002, 0x0000019e,	/* ((r1&~r2)!=0), r1=pswrd2.PRTY_STS_H_0, r2=pswrd2.PRTY_MASK_H_0,  */
	0x00000111, 0x00010002, 0x000001a0,	/* ((r1&~r2)!=0), r1=pswrd2.PRTY_STS_H_1, r2=pswrd2.PRTY_MASK_H_1,  */
	0x01000112, 0x00010001, 0x000001a2,	/* (r1!=0), r1=pswrd2.disable_inputs,  */
	0x01010113, 0x00010001, 0x000001a3,	/* (r1!=0), r1=pswhst2.header_fifo_status,  */
	0x01010114, 0x00010001, 0x000001a4,	/* (r1!=0), r1=pswhst2.data_fifo_status,  */
	0x00020115, 0x00010002, 0x000001a5,	/* ((r1&~r2)!=0), r1=pswhst2.INT_STS, r2=pswhst2.INT_MASK,  */
	0x01020116, 0x00010001, 0x000001a7,	/* (r1!=0), r1=pswhst.discard_internal_writes_status,  */
	0x01020117, 0x00010001, 0x000001a8,	/* (r1!=0), r1=pswhst.discard_doorbells_status,  */
	0x01020118, 0x00010001, 0x002101a9,	/* (r1!=3), r1=pswhst.arb_is_idle,  */
	0x01020119, 0x00010301, 0x000001aa,	/* (r1!=0), r1=pswhst.incorrect_access_valid,  */
	0x0102011a, 0x00010101, 0x000001ae,	/* (r1!=0), r1=pswhst.per_violation_valid,  */
	0x0102011b, 0x00010001, 0x002201b0,	/* (r1!=reset1), r1=pswhst.source_credits_avail,  */
	0x0102011c, 0x00010101, 0x000001b1,	/* (r1!=0), r1=pswhst.source_credit_viol_valid,  */
	0x0102011d, 0x00010001, 0x002301b3,	/* (r1!=reset1), r1=pswhst.dest_credits_avail,  */
	0x0102011e, 0x00010001, 0x000001b4,	/* (r1!=0), r1=pswhst.is_in_drain_mode,  */
	0x0102011f, 0x00010101, 0x000001b5,	/* (r1!=0), r1=pswhst.timeout_valid,  */
	0x01020120, 0x00010101, 0x000001b7,	/* (r1!=0), r1=pswhst.timeout_valid,  */
	0x01020121, 0x00010001, 0x000001b9,	/* (r1!=0), r1=pswhst.clients_waiting_to_source_arb[0:7],  */
	0x00020122, 0x00010002, 0x000001ba,	/* ((r1&~r2)!=0), r1=pswhst.INT_STS, r2=pswhst.INT_MASK,  */
	0x00000123, 0x00010002, 0x000001bc,	/* ((r1&~r2)!=0), r1=pswhst.PRTY_STS, r2=pswhst.PRTY_MASK,  */
	0x00000124, 0x00010002, 0x000001be,	/* ((r1&~r2)!=0), r1=pswhst.PRTY_STS_H_0, r2=pswhst.PRTY_MASK_H_0,  */
	0x01020125, 0x00010001, 0x001b01c0,	/* (r1!=0x01), r1=pglue_b.init_done_ptt_gtt,  */
	0x00000126, 0x00010002, 0x000001c1,	/* ((r1&~r2)!=0), r1=pglue_b.PRTY_STS, r2=pglue_b.PRTY_MASK,  */
	0x00000127, 0x00010002, 0x000001c3,	/* ((r1&~r2)!=0), r1=pglue_b.PRTY_STS_H_0, r2=pglue_b.PRTY_MASK_H_0,  */
	0x01000128, 0x00010001, 0x000001c5,	/* (r1!=0), r1=pglue_b.pgl_write_blocked,  */
	0x01000129, 0x00010001, 0x000001c6,	/* (r1!=0), r1=pglue_b.pgl_read_blocked,  */
	0x0101012a, 0x00010001, 0x000001c7,	/* (r1!=0), r1=pglue_b.read_fifo_occupancy_level,  */
	0x0102012b, 0x00010001, 0x000001c8,	/* (r1!=0), r1=pglue_b.rx_legacy_errors,  */
	0x0701012c, 0x00030001, 0x002401c9,	/* (((r1>>17)&1)!=0), r1=pglue_b.pgl_txw_cdts,  */
	0x0102012d, 0x00010001, 0x000001ca,	/* (r1!=0), r1=pglue_b.cfg_space_a_request,  */
	0x0102012e, 0x00010001, 0x000001cb,	/* (r1!=0), r1=pglue_b.cfg_space_b_request,  */
	0x0102012f, 0x00010001, 0x000001cc,	/* (r1!=0), r1=pglue_b.flr_request_vf_31_0,  */
	0x01020130, 0x00010001, 0x000001cd,	/* (r1!=0), r1=pglue_b.flr_request_vf_63_32,  */
	0x01020131, 0x00010001, 0x000001ce,	/* (r1!=0), r1=pglue_b.flr_request_vf_95_64,  */
	0x01020132, 0x00010001, 0x000001cf,	/* (r1!=0), r1=pglue_b.flr_request_vf_127_96,  */
	0x01020133, 0x00010001, 0x000001d0,	/* (r1!=0), r1=pglue_b.flr_request_vf_159_128,  */
	0x01020134, 0x00010001, 0x000001d1,	/* (r1!=0), r1=pglue_b.flr_request_vf_191_160,  */
	0x01020135, 0x00010001, 0x000001d2,	/* (r1!=0), r1=pglue_b.flr_request_pf_31_0,  */
	0x01020136, 0x00010001, 0x000001d3,	/* (r1!=0), r1=pglue_b.sr_iov_disabled_request,  */
	0x01000137, 0x00010001, 0x000001d4,	/* (r1!=0), r1=pglue_b.incorrect_rcv_details,  */
	0x01020138, 0x00010001, 0x000001d5,	/* (r1!=0), r1=pglue_b.was_error_vf_31_0,  */
	0x01020139, 0x00010001, 0x000001d6,	/* (r1!=0), r1=pglue_b.was_error_vf_63_32,  */
	0x0102013a, 0x00010001, 0x000001d7,	/* (r1!=0), r1=pglue_b.was_error_vf_95_64,  */
	0x0102013b, 0x00010001, 0x000001d8,	/* (r1!=0), r1=pglue_b.was_error_vf_127_96,  */
	0x0102013c, 0x00010001, 0x000001d9,	/* (r1!=0), r1=pglue_b.was_error_vf_159_128,  */
	0x0102013d, 0x00010001, 0x000001da,	/* (r1!=0), r1=pglue_b.was_error_vf_191_160,  */
	0x0102013e, 0x00010001, 0x000001db,	/* (r1!=0), r1=pglue_b.was_error_pf_31_0,  */
	0x0102013f, 0x00010001, 0x000001dc,	/* (r1!=0), r1=pglue_b.rx_err_details,  */
	0x01020140, 0x00010001, 0x000001dd,	/* (r1!=0), r1=pglue_b.rx_tcpl_err_details,  */
	0x01020141, 0x00010201, 0x000001de,	/* (r1!=0), r1=pglue_b.tx_err_wr_details,  */
	0x01020142, 0x00010001, 0x000001e1,	/* (r1!=0), r1=pglue_b.tx_err_wr_details2,  */
	0x01020143, 0x00010201, 0x000001e2,	/* (r1!=0), r1=pglue_b.tx_err_rd_details,  */
	0x01020144, 0x00010001, 0x000001e5,	/* (r1!=0), r1=pglue_b.tx_err_rd_details2,  */
	0x01020145, 0x00010001, 0x000001e6,	/* (r1!=0), r1=pglue_b.vf_length_violation_details,  */
	0x01020146, 0x00010001, 0x000001e7,	/* (r1!=0), r1=pglue_b.vf_length_violation_details2,  */
	0x01020147, 0x00010001, 0x000001e8,	/* (r1!=0), r1=pglue_b.vf_grc_space_violation_details,  */
	0x01000148, 0x00010001, 0x002701e9,	/* (r1!=0x0f), r1=pglue_b.pcie_err_status,  */
	0x01020149, 0x00010201, 0x000001ea,	/* (r1!=0), r1=pglue_b.master_zlr_err_details,  */
	0x0102014a, 0x00010001, 0x000001ed,	/* (r1!=0), r1=pglue_b.admin_window_violation_details,  */
	0x0100014b, 0x00010001, 0x000001ee,	/* (r1!=0), r1=pglue_b.out_of_range_function_in_pretend_details,  */
	0x0100014c, 0x00010001, 0x000001ef,	/* (r1!=0), r1=pglue_b.out_of_range_function_in_pretend_address,  */
	0x0101014d, 0x00010001, 0x000001f0,	/* (r1!=0), r1=pglue_b.write_fifo_occupancy_level,  */
	0x0102014e, 0x00010201, 0x000001f1,	/* (r1!=0), r1=pglue_b.illegal_address_details,  */
	0x0102014f, 0x00010001, 0x000001f4,	/* (r1!=0), r1=pglue_b.illegal_address_details2,  */
	0x01020150, 0x00010001, 0x002801f5,	/* (r1!=0xffffffff), r1=pglue_b.tags_31_0,  */
	0x01020151, 0x00010001, 0x002801f6,	/* (r1!=0xffffffff), r1=pglue_b.tags_63_32,  */
	0x01020152, 0x00010001, 0x002801f7,	/* (r1!=0xffffffff), r1=pglue_b.tags_95_64,  */
	0x01020153, 0x00010001, 0x002801f8,	/* (r1!=0xffffffff), r1=pglue_b.tags_127_96,  */
	0x01020154, 0x00010201, 0x000001f9,	/* (r1!=0), r1=pglue_b.vf_ilt_err_details,  */
	0x01020155, 0x00010001, 0x000001fc,	/* (r1!=0), r1=pglue_b.vf_ilt_err_details2,  */
	0x01000156, 0x00010001, 0x001b01fd,	/* (r1!=1), r1=ccfc.ll_init_done,  */
	0x01000157, 0x00010001, 0x001b01fe,	/* (r1!=1), r1=ccfc.ac_init_done,  */
	0x01000158, 0x00010001, 0x001b01ff,	/* (r1!=1), r1=ccfc.cam_init_done,  */
	0x01000159, 0x00010001, 0x001b0200,	/* (r1!=1), r1=ccfc.tidram_init_done,  */
	0x0000015a, 0x00010502, 0x00000201,	/* ((r1&~r2)!=0), r1=ccfc.INT_STS_0, r2=ccfc.INT_MASK_0,  */
	0x0000015b, 0x00010002, 0x00000208,	/* ((r1&~r2)!=0), r1=ccfc.PRTY_STS_H_0, r2=ccfc.PRTY_MASK_H_0,  */
	0x0000015c, 0x00010002, 0x0000020a,	/* ((r1&~r2)!=0), r1=ccfc.PRTY_STS, r2=ccfc.PRTY_MASK,  */
	0x0101015d, 0x00010001, 0x0000020c,	/* (r1!=0x0), r1=ccfc.lstate_arriving,  */
	0x0101015e, 0x00010001, 0x0000020d,	/* (r1!=0x0), r1=ccfc.lstate_leaving,  */
	0x0101015f, 0x00010001, 0x0029020e,	/* (r1!=reset1), r1=ccfc.cduld_credit,  */
	0x01010160, 0x00010001, 0x0018020f,	/* (r1!=reset1), r1=ccfc.cduwb_credit,  */
	0x00000161, 0x00010002, 0x00000210,	/* ((r1&~r2)!=0), r1=qm.INT_STS, r2=qm.INT_MASK,  */
	0x00000162, 0x00010002, 0x00000212,	/* ((r1&~r2)!=0), r1=qm.PRTY_STS, r2=qm.PRTY_MASK,  */
	0x00000163, 0x00010002, 0x00000214,	/* ((r1&~r2)!=0), r1=qm.PRTY_STS_H_0, r2=qm.PRTY_MASK_H_0,  */
	0x00000164, 0x00010002, 0x00000216,	/* ((r1&~r2)!=0), r1=qm.PRTY_STS_H_1, r2=qm.PRTY_MASK_H_1,  */
	0x00000165, 0x00010002, 0x00000218,	/* ((r1&~r2)!=0), r1=qm.PRTY_STS_H_2, r2=qm.PRTY_MASK_H_2,  */
	0x01000166, 0x00010001, 0x0000021a,	/* (r1!=0), r1=qm.wrc_fifolvl_0[0:5],  */
	0x03000167, 0x00000002, 0x0000021b,	/* (r1!=r2), r1=qm.OutLdReqCrdConnTx, r2=qm.OutLdReqSizeConnTx,  */
	0x03000168, 0x00000002, 0x0000021d,	/* (r1!=r2), r1=qm.OutLdReqCrdConnOther, r2=qm.OutLdReqSizeConnOther,  */
	0x01000169, 0x00010001, 0x0000021f,	/* (r1!=0), r1=qm.OvfQNumTx,  */
	0x0100016a, 0x00010101, 0x00000220,	/* (r1!=0), r1=qm.OvfErrorTx,  */
	0x0100016b, 0x00010001, 0x00000222,	/* (r1!=0), r1=qm.OvfQNumOther,  */
	0x0100016c, 0x00010101, 0x00000223,	/* (r1!=0), r1=qm.OvfErrorOther,  */
	0x0301016d, 0x00000002, 0x00000225,	/* (r1!=r2), r1=qm.CmCrd_0, r2=qm.CmInitCrd_0,  */
	0x0301016e, 0x00000002, 0x00000227,	/* (r1!=r2), r1=qm.CmCrd_1, r2=qm.CmInitCrd_1,  */
	0x0301016f, 0x00000002, 0x00000229,	/* (r1!=r2), r1=qm.CmCrd_2, r2=qm.CmInitCrd_2,  */
	0x03010170, 0x00000002, 0x0000022b,	/* (r1!=r2), r1=qm.CmCrd_3, r2=qm.CmInitCrd_3,  */
	0x03010171, 0x00000002, 0x0000022d,	/* (r1!=r2), r1=qm.CmCrd_4, r2=qm.CmInitCrd_4,  */
	0x03010172, 0x00000002, 0x0000022f,	/* (r1!=r2), r1=qm.CmCrd_5, r2=qm.CmInitCrd_5,  */
	0x03010173, 0x00000002, 0x00000231,	/* (r1!=r2), r1=qm.CmCrd_6, r2=qm.CmInitCrd_6,  */
	0x03010174, 0x00000002, 0x00000233,	/* (r1!=r2), r1=qm.CmCrd_7, r2=qm.CmInitCrd_7,  */
	0x03010175, 0x00000002, 0x00000235,	/* (r1!=r2), r1=qm.CmCrd_8, r2=qm.CmInitCrd_8,  */
	0x03010176, 0x00000002, 0x00000237,	/* (r1!=r2), r1=qm.CmCrd_9, r2=qm.CmInitCrd_9,  */
	0x05000177, 0x00040002, 0x002a0239,	/* (((r1&0x01)!=0)&&((r2&0x01)!=0)), r1=qm.Err_Inc0_RlGlblCrd, r2=qm.Err_Mask_RlGlblCrd,  */
	0x05000178, 0x00040002, 0x002e023b,	/* (((r1&0x01)!=0)&&((r2&0x04)!=0)), r1=qm.Err_Dec0_RlGlblCrd, r2=qm.Err_Mask_RlGlblCrd,  */
	0x05000179, 0x00040002, 0x0032023d,	/* (((r1&0x01)!=0)&&((r2&0x08)!=0)), r1=qm.Err_Dec1_RlGlblCrd, r2=qm.Err_Mask_RlGlblCrd,  */
	0x0500017a, 0x00040002, 0x002a023f,	/* (((r1&0x01)!=0)&&((r2&0x01)!=0)), r1=qm.Err_Inc0_RlPfCrd, r2=qm.Err_Mask_RlPfCrd,  */
	0x0500017b, 0x00040002, 0x002e0241,	/* (((r1&0x01)!=0)&&((r2&0x04)!=0)), r1=qm.Err_Dec0_RlPfCrd, r2=qm.Err_Mask_RlPfCrd,  */
	0x0500017c, 0x00040002, 0x00320243,	/* (((r1&0x01)!=0)&&((r2&0x08)!=0)), r1=qm.Err_Dec1_RlPfCrd, r2=qm.Err_Mask_RlPfCrd,  */
	0x0500017d, 0x00040002, 0x002a0245,	/* (((r1&0x01)!=0)&&((r2&0x01)!=0)), r1=qm.Err_Inc0_WfqPfCrd, r2=qm.Err_Mask_WfqPfCrd,  */
	0x0500017e, 0x00040002, 0x002e0247,	/* (((r1&0x01)!=0)&&((r2&0x04)!=0)), r1=qm.Err_Dec0_WfqPfCrd, r2=qm.Err_Mask_WfqPfCrd,  */
	0x0500017f, 0x00040002, 0x00320249,	/* (((r1&0x01)!=0)&&((r2&0x08)!=0)), r1=qm.Err_Dec1_WfqPfCrd, r2=qm.Err_Mask_WfqPfCrd,  */
	0x05000180, 0x00040002, 0x002a024b,	/* (((r1&0x01)!=0)&&((r2&0x01)!=0)), r1=qm.Err_Inc0_VoqLineCrd, r2=qm.Err_Mask_VoqLineCrd,  */
	0x05000181, 0x00040002, 0x0036024d,	/* (((r1&0x01)!=0)&&((r2&0x02)!=0)), r1=qm.Err_Inc1_VoqLineCrd, r2=qm.Err_Mask_VoqLineCrd,  */
	0x05000182, 0x00040002, 0x002e024f,	/* (((r1&0x01)!=0)&&((r2&0x04)!=0)), r1=qm.Err_Dec0_VoqLineCrd, r2=qm.Err_Mask_VoqLineCrd,  */
	0x05000183, 0x00040002, 0x00320251,	/* (((r1&0x01)!=0)&&((r2&0x08)!=0)), r1=qm.Err_Dec1_VoqLineCrd, r2=qm.Err_Mask_VoqLineCrd,  */
	0x05000184, 0x00040002, 0x002a0253,	/* (((r1&0x01)!=0)&&((r2&0x01)!=0)), r1=qm.Err_Inc0_VoqByteCrd, r2=qm.Err_Mask_VoqByteCrd,  */
	0x05000185, 0x00040002, 0x00360255,	/* (((r1&0x01)!=0)&&((r2&0x02)!=0)), r1=qm.Err_Inc1_VoqByteCrd, r2=qm.Err_Mask_VoqByteCrd,  */
	0x05000186, 0x00040002, 0x002e0257,	/* (((r1&0x01)!=0)&&((r2&0x04)!=0)), r1=qm.Err_Dec0_VoqByteCrd, r2=qm.Err_Mask_VoqByteCrd,  */
	0x05000187, 0x00040002, 0x00320259,	/* (((r1&0x01)!=0)&&((r2&0x08)!=0)), r1=qm.Err_Dec1_VoqByteCrd, r2=qm.Err_Mask_VoqByteCrd,  */
	0x01010188, 0x00010101, 0x0028025b,	/* (r1!=reset1), r1=qm.fifo_empty0,  */
	0x0e010189, 0x00020101, 0x003a025d,	/* ((r1|0x00000100)!=reset1), r1=qm.fifo_empty1,  */
	0x0100018a, 0x00010001, 0x0000025f,	/* (r1!=0), r1=qm.fifo_error0,  */
	0x0100018b, 0x00010001, 0x00000260,	/* (r1!=0), r1=qm.fifo_error1,  */
	0x0000018c, 0x00010002, 0x00000261,	/* ((r1&~r2)!=0), r1=rdif.INT_STS, r2=rdif.INT_MASK,  */
	0x0000018d, 0x00010002, 0x00000263,	/* ((r1&~r2)!=0), r1=rdif.PRTY_STS, r2=rdif.PRTY_STS,  */
	0x0000018e, 0x00010002, 0x00000265,	/* ((r1&~r2)!=0), r1=tdif.INT_STS, r2=tdif.INT_MASK,  */
	0x0000018f, 0x00010002, 0x00000267,	/* ((r1&~r2)!=0), r1=tdif.PRTY_STS, r2=tdif.PRTY_STS,  */
	0x00000190, 0x00010002, 0x00000269,	/* ((r1&~r2)!=0), r1=tdif.PRTY_STS_H_0, r2=tdif.PRTY_MASK_H_0,  */
	0x01020191, 0x00010001, 0x0021026b,	/* (r1!=0x03), r1=brb.init_done,  */
	0x00000192, 0x00010202, 0x0000026c,	/* ((r1&~r2)!=0), r1=brb.INT_STS_0, r2=brb.INT_MASK_0,  */
	0x00000193, 0x00010002, 0x00000270,	/* ((r1&~r2)!=0), r1=brb.INT_STS_1, r2=brb.INT_MASK_1,  */
	0x00000194, 0x00010002, 0x00000272,	/* ((r1&~r2)!=0), r1=brb.INT_STS_2, r2=brb.INT_MASK_2,  */
	0x00000195, 0x00010002, 0x00000274,	/* ((r1&~r2)!=0), r1=brb.INT_STS_3, r2=brb.INT_MASK_3,  */
	0x00000196, 0x00010202, 0x00000276,	/* ((r1&~r2)!=0), r1=brb.INT_STS_4, r2=brb.INT_MASK_4,  */
	0x00020197, 0x00010002, 0x0000027a,	/* ((r1&~r2)!=0), r1=brb.INT_STS_5, r2=brb.INT_MASK_5,  */
	0x00020198, 0x00010002, 0x0000027c,	/* ((r1&~r2)!=0), r1=brb.INT_STS_6, r2=brb.INT_MASK_6,  */
	0x00020199, 0x00010002, 0x0000027e,	/* ((r1&~r2)!=0), r1=brb.INT_STS_7, r2=brb.INT_MASK_7,  */
	0x0002019a, 0x00010002, 0x00000280,	/* ((r1&~r2)!=0), r1=brb.INT_STS_8, r2=brb.INT_MASK_8,  */
	0x0002019b, 0x00010002, 0x00000282,	/* ((r1&~r2)!=0), r1=brb.INT_STS_9, r2=brb.INT_MASK_9,  */
	0x0002019c, 0x00010002, 0x00000284,	/* ((r1&~r2)!=0), r1=brb.INT_STS_10, r2=brb.INT_MASK_10,  */
	0x0002019d, 0x00010002, 0x00000286,	/* ((r1&~r2)!=0), r1=brb.INT_STS_11, r2=brb.INT_MASK_11,  */
	0x0000019e, 0x00010002, 0x00000288,	/* ((r1&~r2)!=0), r1=brb.PRTY_STS, r2=brb.PRTY_MASK,  */
	0x0000019f, 0x00010002, 0x0000028a,	/* ((r1&~r2)!=0), r1=brb.PRTY_STS_H_0, r2=brb.PRTY_MASK_H_0,  */
	0x000001a0, 0x00010002, 0x0000028c,	/* ((r1&~r2)!=0), r1=brb.PRTY_STS_H_1, r2=brb.PRTY_MASK_H_1,  */
	0x010101a1, 0x00010001, 0x0000028e,	/* (r1!=0), r1=brb.wc_bandwidth_if_full,  */
	0x010101a2, 0x00010001, 0x0000028f,	/* (r1!=0), r1=brb.rc_pkt_if_full,  */
	0x010101a3, 0x00010001, 0x003c0290,	/* (r1!=255), r1=brb.rc_pkt_empty_0[0:4],  */
	0x010101a4, 0x00010001, 0x00270291,	/* (r1!=15), r1=brb.rc_sop_empty,  */
	0x010101a5, 0x00010001, 0x00190292,	/* (r1!=2), r1=brb.ll_arb_empty,  */
	0x010101a6, 0x00010001, 0x00000293,	/* (r1!=0), r1=brb.stop_packet_counter,  */
	0x010101a7, 0x00010001, 0x00000294,	/* (r1!=0), r1=brb.stop_byte_counter,  */
	0x010101a8, 0x00010001, 0x00000295,	/* (r1!=0), r1=brb.rc_pkt_state,  */
	0x010101a9, 0x00010001, 0x00000296,	/* (r1!=0), r1=brb.mac0_tc_occupancy_0,  */
	0x010101aa, 0x00010001, 0x00000297,	/* (r1!=0), r1=brb.mac0_tc_occupancy_1,  */
	0x010101ab, 0x00010001, 0x00000298,	/* (r1!=0), r1=brb.mac0_tc_occupancy_2,  */
	0x010101ac, 0x00010001, 0x00000299,	/* (r1!=0), r1=brb.mac0_tc_occupancy_3,  */
	0x010101ad, 0x00010001, 0x0000029a,	/* (r1!=0), r1=brb.mac0_tc_occupancy_4,  */
	0x010101ae, 0x00010001, 0x0000029b,	/* (r1!=0), r1=brb.mac0_tc_occupancy_5,  */
	0x010101af, 0x00010001, 0x0000029c,	/* (r1!=0), r1=brb.mac0_tc_occupancy_6,  */
	0x010101b0, 0x00010001, 0x0000029d,	/* (r1!=0), r1=brb.mac0_tc_occupancy_7,  */
	0x010101b1, 0x00010001, 0x0000029e,	/* (r1!=0), r1=brb.mac0_tc_occupancy_8,  */
	0x010101b2, 0x00010001, 0x0000029f,	/* (r1!=0), r1=brb.mac1_tc_occupancy_0,  */
	0x010101b3, 0x00010001, 0x000002a0,	/* (r1!=0), r1=brb.mac1_tc_occupancy_1,  */
	0x010101b4, 0x00010001, 0x000002a1,	/* (r1!=0), r1=brb.mac1_tc_occupancy_2,  */
	0x010101b5, 0x00010001, 0x000002a2,	/* (r1!=0), r1=brb.mac1_tc_occupancy_3,  */
	0x010101b6, 0x00010001, 0x000002a3,	/* (r1!=0), r1=brb.mac1_tc_occupancy_4,  */
	0x010101b7, 0x00010001, 0x000002a4,	/* (r1!=0), r1=brb.mac1_tc_occupancy_5,  */
	0x010101b8, 0x00010001, 0x000002a5,	/* (r1!=0), r1=brb.mac1_tc_occupancy_6,  */
	0x010101b9, 0x00010001, 0x000002a6,	/* (r1!=0), r1=brb.mac1_tc_occupancy_7,  */
	0x010101ba, 0x00010001, 0x000002a7,	/* (r1!=0), r1=brb.mac1_tc_occupancy_8,  */
	0x010101bb, 0x00010001, 0x000002a8,	/* (r1!=0), r1=xyld.pending_msg_to_ext_ev_1_ctr,  */
	0x010101bc, 0x00010001, 0x000002a9,	/* (r1!=0), r1=xyld.pending_msg_to_ext_ev_2_ctr,  */
	0x010101bd, 0x00010001, 0x000002aa,	/* (r1!=0), r1=xyld.pending_msg_to_ext_ev_3_ctr,  */
	0x010101be, 0x00010001, 0x000002ab,	/* (r1!=0), r1=xyld.pending_msg_to_ext_ev_4_ctr,  */
	0x010101bf, 0x00010001, 0x000002ac,	/* (r1!=0), r1=xyld.pending_msg_to_ext_ev_5_ctr,  */
	0x030101c0, 0x00000002, 0x000002ad,	/* (r1!=r2), r1=xyld.foc_remain_credits, r2=xyld.foci_foc_credits,  */
	0x010101c1, 0x00010001, 0x000002af,	/* (r1!=0), r1=xyld.pci_pending_msg_ctr,  */
	0x010101c2, 0x00010001, 0x000002b0,	/* (r1!=0), r1=xyld.dbg_pending_ccfc_req,  */
	0x010101c3, 0x00010001, 0x000002b1,	/* (r1!=0), r1=xyld.dbg_pending_tcfc_req,  */
	0x000201c4, 0x00010002, 0x000002b2,	/* ((r1&~r2)!=0), r1=xyld.INT_STS, r2=xyld.INT_MASK,  */
	0x000001c5, 0x00010002, 0x000002b4,	/* ((r1&~r2)!=0), r1=xyld.PRTY_STS_H_0, r2=xyld.PRTY_MASK_H_0,  */
	0x010101c6, 0x00010001, 0x000002b6,	/* (r1!=0), r1=yuld.pending_msg_to_ext_ev_1_ctr,  */
	0x010101c7, 0x00010001, 0x000002b7,	/* (r1!=0), r1=yuld.pending_msg_to_ext_ev_2_ctr,  */
	0x010101c8, 0x00010001, 0x000002b8,	/* (r1!=0), r1=yuld.pending_msg_to_ext_ev_3_ctr,  */
	0x010101c9, 0x00010001, 0x000002b9,	/* (r1!=0), r1=yuld.pending_msg_to_ext_ev_4_ctr,  */
	0x010101ca, 0x00010001, 0x000002ba,	/* (r1!=0), r1=yuld.pending_msg_to_ext_ev_5_ctr,  */
	0x030101cb, 0x00000002, 0x000002bb,	/* (r1!=r2), r1=yuld.foc_remain_credits, r2=yuld.foci_foc_credits,  */
	0x010101cc, 0x00010001, 0x000002bd,	/* (r1!=0), r1=yuld.dbg_pending_ccfc_req,  */
	0x010101cd, 0x00010001, 0x000002be,	/* (r1!=0), r1=yuld.dbg_pending_tcfc_req,  */
	0x000001ce, 0x00010002, 0x000002bf,	/* ((r1&~r2)!=0), r1=yuld.PRTY_STS_H_0, r2=yuld.PRTY_MASK_H_0,  */
	0x010101cf, 0x00010001, 0x000002c1,	/* (r1!=0), r1=tmld.pending_msg_to_ext_ev_1_ctr,  */
	0x010101d0, 0x00010001, 0x000002c2,	/* (r1!=0), r1=tmld.pending_msg_to_ext_ev_2_ctr,  */
	0x010101d1, 0x00010001, 0x000002c3,	/* (r1!=0), r1=tmld.pending_msg_to_ext_ev_3_ctr,  */
	0x010101d2, 0x00010001, 0x000002c4,	/* (r1!=0), r1=tmld.pending_msg_to_ext_ev_4_ctr,  */
	0x010101d3, 0x00010001, 0x000002c5,	/* (r1!=0), r1=tmld.pending_msg_to_ext_ev_5_ctr,  */
	0x030101d4, 0x00000002, 0x000002c6,	/* (r1!=r2), r1=tmld.foc_remain_credits, r2=tmld.foci_foc_credits,  */
	0x010101d5, 0x00010001, 0x000002c8,	/* (r1!=0), r1=tmld.dbg_pending_ccfc_req,  */
	0x010101d6, 0x00010001, 0x000002c9,	/* (r1!=0), r1=tmld.dbg_pending_tcfc_req,  */
	0x000201d7, 0x00010002, 0x000002ca,	/* ((r1&~r2)!=0), r1=tmld.INT_STS, r2=tmld.INT_MASK,  */
	0x000001d8, 0x00010002, 0x000002cc,	/* ((r1&~r2)!=0), r1=tmld.PRTY_STS_H_0, r2=tmld.PRTY_MASK_H_0,  */
	0x010101d9, 0x00010001, 0x000002ce,	/* (r1!=0), r1=muld.pending_msg_to_ext_ev_1_ctr,  */
	0x010101da, 0x00010001, 0x000002cf,	/* (r1!=0), r1=muld.pending_msg_to_ext_ev_2_ctr,  */
	0x010101db, 0x00010001, 0x000002d0,	/* (r1!=0), r1=muld.pending_msg_to_ext_ev_3_ctr,  */
	0x010101dc, 0x00010001, 0x000002d1,	/* (r1!=0), r1=muld.pending_msg_to_ext_ev_4_ctr,  */
	0x010101dd, 0x00010001, 0x000002d2,	/* (r1!=0), r1=muld.pending_msg_to_ext_ev_5_ctr,  */
	0x030101de, 0x00000002, 0x000002d3,	/* (r1!=r2), r1=muld.foc_remain_credits, r2=muld.foci_foc_credits,  */
	0x010101df, 0x00010001, 0x000002d5,	/* (r1!=0), r1=muld.bd_pending_msg_ctr,  */
	0x010101e0, 0x00010001, 0x000002d6,	/* (r1!=0), r1=muld.sge_pending_msg_ctr,  */
	0x010101e1, 0x00010001, 0x000002d7,	/* (r1!=0), r1=muld.pci_pending_msg_ctr,  */
	0x010101e2, 0x00010001, 0x000002d8,	/* (r1!=0), r1=muld.dbg_pending_ccfc_req,  */
	0x010101e3, 0x00010001, 0x000002d9,	/* (r1!=0), r1=muld.dbg_pending_tcfc_req,  */
	0x000201e4, 0x00010002, 0x000002da,	/* ((r1&~r2)!=0), r1=muld.INT_STS, r2=muld.INT_MASK,  */
	0x000001e5, 0x00010002, 0x000002dc,	/* ((r1&~r2)!=0), r1=muld.PRTY_STS_H_0, r2=muld.PRTY_MASK_H_0,  */
	0x000001e6, 0x00010002, 0x000002de,	/* ((r1&~r2)!=0), r1=nig.INT_STS_0, r2=nig.INT_MASK_0,  */
	0x000001e7, 0x00010002, 0x000002e0,	/* ((r1&~r2)!=0), r1=nig.INT_STS_1, r2=nig.INT_MASK_1,  */
	0x000001e8, 0x00010002, 0x000002e2,	/* ((r1&~r2)!=0), r1=nig.INT_STS_2, r2=nig.INT_MASK_2,  */
	0x000201e9, 0x00010202, 0x000002e4,	/* ((r1&~r2)!=0), r1=nig.INT_STS_3, r2=nig.INT_MASK_3,  */
	0x000001ea, 0x00010002, 0x000002e8,	/* ((r1&~r2)!=0), r1=nig.INT_STS_4, r2=nig.INT_MASK_4,  */
	0x000201eb, 0x00010202, 0x000002ea,	/* ((r1&~r2)!=0), r1=nig.INT_STS_5, r2=nig.INT_MASK_5,  */
	0x000001ec, 0x00010002, 0x000002ee,	/* ((r1&~r2)!=0), r1=nig.PRTY_STS_H_0, r2=nig.PRTY_MASK_H_0,  */
	0x000001ed, 0x00010002, 0x000002f0,	/* ((r1&~r2)!=0), r1=nig.PRTY_STS_H_1, r2=nig.PRTY_MASK_H_1,  */
	0x000001ee, 0x00010002, 0x000002f2,	/* ((r1&~r2)!=0), r1=nig.PRTY_STS_H_2, r2=nig.PRTY_MASK_H_2,  */
	0x000001ef, 0x00010002, 0x000002f4,	/* ((r1&~r2)!=0), r1=nig.PRTY_STS_H_3, r2=nig.PRTY_MASK_H_3,  */
	0x010101f0, 0x00010001, 0x003d02f6,	/* (r1!=0xfffff), r1=nig.lb_sopq_empty,  */
	0x010101f1, 0x00010001, 0x003e02f7,	/* (r1!=0xffff), r1=nig.tx_sopq_empty,  */
	0x010101f2, 0x00010001, 0x001b02f8,	/* (r1!=1), r1=nig.tx_macfifo_empty,  */
	0x010101f3, 0x00010101, 0x001b02f9,	/* (r1!=0x01), r1=nig.rx_llh_dfifo_empty,  */
	0x010101f4, 0x00010101, 0x001b02fb,	/* (r1!=0x01), r1=nig.rx_llh_hfifo_empty,  */
	0x010101f5, 0x00010001, 0x001b02fd,	/* (r1!=1), r1=nig.rx_llh_rfifo_empty,  */
	0x010101f6, 0x00010101, 0x001b02fe,	/* (r1!=0x01), r1=nig.rx_llh_rfifo_empty,  */
	0x010001f7, 0x00010101, 0x00210300,	/* (r1!=0x03), r1=nig.storm_status,  */
	0x010101f8, 0x00010001, 0x001b0302,	/* (r1!=1), r1=nig.lb_btb_fifo_empty,  */
	0x010101f9, 0x00010101, 0x001b0303,	/* (r1!=0x01), r1=nig.lb_llh_dfifo_empty,  */
	0x010101fa, 0x00010101, 0x001b0305,	/* (r1!=0x01), r1=nig.lb_llh_hfifo_empty,  */
	0x010101fb, 0x00010001, 0x001b0307,	/* (r1!=1), r1=nig.lb_llh_rfifo_empty,  */
	0x010101fc, 0x00010101, 0x001b0308,	/* (r1!=0x01), r1=nig.lb_llh_rfifo_empty,  */
	0x050001fd, 0x00040002, 0x003f030a,	/* (((r1&0xff)!=0)&&((r2&0x7)!=0)), r1=nig.rx_ptp_ts_msb_err, r2=nig.rx_ptp_en,  */
	0x010101fe, 0x00010001, 0x001b030c,	/* (r1!=1), r1=nig.tx_btb_fifo_empty,  */
	0x010101ff, 0x00010101, 0x001b030d,	/* (r1!=0x01), r1=nig.tx_llh_dfifo_empty,  */
	0x01010200, 0x00010101, 0x001b030f,	/* (r1!=0x01), r1=nig.tx_llh_hfifo_empty,  */
	0x01010201, 0x00010101, 0x001b0311,	/* (r1!=0x01), r1=nig.tx_llh_rfifo_empty,  */
	0x01010202, 0x00010101, 0x001b0313,	/* (r1!=0x01), r1=nig.tx_gnt_fifo_empty,  */
	0x01010203, 0x00010101, 0x001b0315,	/* (r1!=0x01), r1=nig.dorq_fifo_empty,  */
	0x01010204, 0x00010001, 0x001b0317,	/* (r1!=1), r1=nig.debug_fifo_empty,  */
	0x01010205, 0x00010101, 0x001b0318,	/* (r1!=0x01), r1=nig.lb_gnt_fifo_empty,  */
	0x01020206, 0x00010001, 0x001b031a,	/* (r1!=0x01), r1=ptu.atc_init_done,  */
	0x01000207, 0x00010001, 0x0000031b,	/* (r1!=0), r1=ptu.pxp_err_ctr,  */
	0x01000208, 0x00010001, 0x0000031c,	/* (r1!=0), r1=ptu.inv_err_ctr,  */
	0x01010209, 0x00010001, 0x0000031d,	/* (r1!=0), r1=ptu.pbf_fill_level,  */
	0x0101020a, 0x00010001, 0x0000031e,	/* (r1!=0), r1=ptu.prm_fill_level,  */
	0x0000020b, 0x00010002, 0x0000031f,	/* ((r1&~r2)!=0), r1=ptu.INT_STS, r2=ptu.INT_MASK,  */
	0x0000020c, 0x00010002, 0x00000321,	/* ((r1&~r2)!=0), r1=ptu.PRTY_STS_H_0, r2=ptu.PRTY_MASK_H_0,  */
	0x0000020d, 0x00010602, 0x00000323,	/* ((r1&~r2)!=0), r1=cdu.INT_STS, r2=cdu.INT_MASK,  */
	0x0000020e, 0x00010002, 0x0000032b,	/* ((r1&~r2)!=0), r1=cdu.PRTY_STS_H_0, r2=cdu.PRTY_MASK_H_0,  */
	0x0002020f, 0x00010002, 0x0000032d,	/* ((r1&~r2)!=0), r1=pbf.INT_STS, r2=pbf.INT_MASK,  */
	0x00000210, 0x00010002, 0x0000032f,	/* ((r1&~r2)!=0), r1=pbf.PRTY_STS, r2=pbf.PRTY_MASK,  */
	0x00000211, 0x00010002, 0x00000331,	/* ((r1&~r2)!=0), r1=pbf.PRTY_STS_H_0, r2=pbf.PRTY_MASK_H_0,  */
	0x00000212, 0x00010002, 0x00000333,	/* ((r1&~r2)!=0), r1=pbf.PRTY_STS_H_1, r2=pbf.PRTY_MASK_H_1,  */
	0x01020213, 0x00010001, 0x00000335,	/* (r1!=0), r1=pbf.num_pkts_received_with_error,  */
	0x01020214, 0x00010001, 0x00000336,	/* (r1!=0), r1=pbf.num_pkts_sent_with_error_to_btb,  */
	0x01020215, 0x00010001, 0x00000337,	/* (r1!=0), r1=pbf.num_pkts_sent_with_drop_to_btb,  */
	0x01010216, 0x00010001, 0x00000338,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq0,  */
	0x01010217, 0x00010001, 0x00000339,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq0,  */
	0x01010218, 0x00010001, 0x0000033a,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq0,  */
	0x01010219, 0x00010001, 0x0000033b,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq1,  */
	0x0101021a, 0x00010001, 0x0000033c,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq1,  */
	0x0101021b, 0x00010001, 0x0000033d,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq1,  */
	0x0101021c, 0x00010001, 0x0000033e,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq2,  */
	0x0101021d, 0x00010001, 0x0000033f,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq2,  */
	0x0101021e, 0x00010001, 0x00000340,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq2,  */
	0x0101021f, 0x00010001, 0x00000341,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq3,  */
	0x01010220, 0x00010001, 0x00000342,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq3,  */
	0x01010221, 0x00010001, 0x00000343,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq3,  */
	0x01010222, 0x00010001, 0x00000344,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq4,  */
	0x01010223, 0x00010001, 0x00000345,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq4,  */
	0x01010224, 0x00010001, 0x00000346,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq4,  */
	0x01010225, 0x00010001, 0x00000347,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq5,  */
	0x01010226, 0x00010001, 0x00000348,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq5,  */
	0x01010227, 0x00010001, 0x00000349,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq5,  */
	0x01010228, 0x00010001, 0x0000034a,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq6,  */
	0x01010229, 0x00010001, 0x0000034b,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq6,  */
	0x0101022a, 0x00010001, 0x0000034c,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq6,  */
	0x0101022b, 0x00010001, 0x0000034d,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq7,  */
	0x0101022c, 0x00010001, 0x0000034e,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq7,  */
	0x0101022d, 0x00010001, 0x0000034f,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq7,  */
	0x0101022e, 0x00010001, 0x00000350,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq8,  */
	0x0101022f, 0x00010001, 0x00000351,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq8,  */
	0x01010230, 0x00010001, 0x00000352,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq8,  */
	0x01010231, 0x00010001, 0x00000353,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq9,  */
	0x01010232, 0x00010001, 0x00000354,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq9,  */
	0x01010233, 0x00010001, 0x00000355,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq9,  */
	0x01010234, 0x00010001, 0x00000356,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq10,  */
	0x01010235, 0x00010001, 0x00000357,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq10,  */
	0x01010236, 0x00010001, 0x00000358,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq10,  */
	0x01010237, 0x00010001, 0x00000359,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq11,  */
	0x01010238, 0x00010001, 0x0000035a,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq11,  */
	0x01010239, 0x00010001, 0x0000035b,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq11,  */
	0x0101023a, 0x00010001, 0x0000035c,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq12,  */
	0x0101023b, 0x00010001, 0x0000035d,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq12,  */
	0x0101023c, 0x00010001, 0x0000035e,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq12,  */
	0x0101023d, 0x00010001, 0x0000035f,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq13,  */
	0x0101023e, 0x00010001, 0x00000360,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq13,  */
	0x0101023f, 0x00010001, 0x00000361,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq13,  */
	0x01010240, 0x00010001, 0x00000362,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq14,  */
	0x01010241, 0x00010001, 0x00000363,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq14,  */
	0x01010242, 0x00010001, 0x00000364,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq14,  */
	0x01010243, 0x00010001, 0x00000365,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq15,  */
	0x01010244, 0x00010001, 0x00000366,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq15,  */
	0x01010245, 0x00010001, 0x00000367,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq15,  */
	0x01010246, 0x00010001, 0x00000368,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq16,  */
	0x01010247, 0x00010001, 0x00000369,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq16,  */
	0x01010248, 0x00010001, 0x0000036a,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq16,  */
	0x01010249, 0x00010001, 0x0000036b,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq17,  */
	0x0101024a, 0x00010001, 0x0000036c,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq17,  */
	0x0101024b, 0x00010001, 0x0000036d,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq17,  */
	0x0101024c, 0x00010001, 0x0000036e,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq18,  */
	0x0101024d, 0x00010001, 0x0000036f,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq18,  */
	0x0101024e, 0x00010001, 0x00000370,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq18,  */
	0x0101024f, 0x00010001, 0x00000371,	/* (r1!=0), r1=pbf.ycmd_qs_cmd_cnt_voq19,  */
	0x01010250, 0x00010001, 0x00000372,	/* (r1!=0), r1=pbf.ycmd_qs_occupancy_voq19,  */
	0x01010251, 0x00010001, 0x00000373,	/* (r1!=0), r1=pbf.btb_allocated_blocks_voq19,  */
	0x00020252, 0x00010002, 0x00000374,	/* ((r1&~r2)!=0), r1=pbf_pb1.INT_STS, r2=pbf_pb1.INT_MASK,  */
	0x00000253, 0x00010002, 0x00000376,	/* ((r1&~r2)!=0), r1=pbf_pb1.PRTY_STS, r2=pbf_pb1.PRTY_MASK,  */
	0x01020254, 0x00010101, 0x001b0378,	/* (r1!=0x01), r1=pbf_pb1.db_empty,  */
	0x01020255, 0x00010101, 0x001b037a,	/* (r1!=0x01), r1=pbf_pb1.tq_empty,  */
	0x01020256, 0x00010101, 0x001b037c,	/* (r1!=0x01), r1=pbf_pb1.ififo_empty,  */
	0x01020257, 0x00010101, 0x001b037e,	/* (r1!=0x01), r1=pbf_pb1.pfifo_empty,  */
	0x01020258, 0x00010001, 0x001b0380,	/* (r1!=0x01), r1=pbf_pb1.tq_th_empty,  */
	0x06000259, 0x00020301, 0x00430381,	/* ((r1&0x80000000)!=0), r1=pbf_pb1.errored_instr,  */
	0x0002025a, 0x00010002, 0x00000385,	/* ((r1&~r2)!=0), r1=pbf_pb2.INT_STS, r2=pbf_pb2.INT_MASK,  */
	0x0000025b, 0x00010002, 0x00000387,	/* ((r1&~r2)!=0), r1=pbf_pb2.PRTY_STS, r2=pbf_pb2.PRTY_MASK,  */
	0x0102025c, 0x00010101, 0x001b0389,	/* (r1!=0x01), r1=pbf_pb2.db_empty,  */
	0x0102025d, 0x00010101, 0x001b038b,	/* (r1!=0x01), r1=pbf_pb2.tq_empty,  */
	0x0102025e, 0x00010101, 0x001b038d,	/* (r1!=0x01), r1=pbf_pb2.ififo_empty,  */
	0x0102025f, 0x00010101, 0x001b038f,	/* (r1!=0x01), r1=pbf_pb2.pfifo_empty,  */
	0x01020260, 0x00010001, 0x001b0391,	/* (r1!=0x01), r1=pbf_pb2.tq_th_empty,  */
	0x06000261, 0x00020301, 0x00430392,	/* ((r1&0x80000000)!=0), r1=pbf_pb2.errored_instr,  */
	0x01020262, 0x00010001, 0x00210396,	/* (r1!=0x03), r1=btb.init_done,  */
	0x00000263, 0x00010102, 0x00000397,	/* ((r1&~r2)!=0), r1=btb.INT_STS_1, r2=btb.INT_MASK_1,  */
	0x00000264, 0x00010102, 0x0000039a,	/* ((r1&~r2)!=0), r1=btb.INT_STS_2, r2=btb.INT_MASK_2,  */
	0x00000265, 0x00010002, 0x0000039d,	/* ((r1&~r2)!=0), r1=btb.INT_STS_3, r2=btb.INT_MASK_3,  */
	0x00000266, 0x00010002, 0x0000039f,	/* ((r1&~r2)!=0), r1=btb.INT_STS_5, r2=btb.INT_MASK_5,  */
	0x00000267, 0x00010002, 0x000003a1,	/* ((r1&~r2)!=0), r1=btb.INT_STS_6, r2=btb.INT_MASK_6,  */
	0x00020268, 0x00010102, 0x000003a3,	/* ((r1&~r2)!=0), r1=btb.INT_STS_8, r2=btb.INT_MASK_8,  */
	0x00020269, 0x00010102, 0x000003a6,	/* ((r1&~r2)!=0), r1=btb.INT_STS_9, r2=btb.INT_MASK_9,  */
	0x0000026a, 0x00010102, 0x000003a9,	/* ((r1&~r2)!=0), r1=btb.INT_STS_10, r2=btb.INT_MASK_10,  */
	0x0000026b, 0x00010002, 0x000003ac,	/* ((r1&~r2)!=0), r1=btb.INT_STS_11, r2=btb.INT_MASK_11,  */
	0x0000026c, 0x00010002, 0x000003ae,	/* ((r1&~r2)!=0), r1=btb.PRTY_STS, r2=btb.PRTY_MASK,  */
	0x0000026d, 0x00010002, 0x000003b0,	/* ((r1&~r2)!=0), r1=btb.PRTY_STS_H_0, r2=btb.PRTY_MASK_H_0,  */
	0x0101026e, 0x00010001, 0x002703b2,	/* (r1!=15), r1=btb.wc_dup_empty,  */
	0x0101026f, 0x00010001, 0x000003b3,	/* (r1!=0), r1=btb.wc_dup_status,  */
	0x01010270, 0x00010001, 0x004503b4,	/* (r1!=8190), r1=btb.wc_empty_0,  */
	0x01020271, 0x00010001, 0x000003b5,	/* (r1!=0), r1=btb.wc_bandwidth_if_full,  */
	0x01020272, 0x00010001, 0x000003b6,	/* (r1!=0), r1=btb.rc_pkt_if_full,  */
	0x01010273, 0x00010001, 0x003c03b7,	/* (r1!=255), r1=btb.rc_pkt_empty_0,  */
	0x01010274, 0x00010001, 0x003c03b8,	/* (r1!=255), r1=btb.rc_pkt_empty_1,  */
	0x01010275, 0x00010001, 0x003c03b9,	/* (r1!=255), r1=btb.rc_pkt_empty_2,  */
	0x01010276, 0x00010001, 0x003c03ba,	/* (r1!=255), r1=btb.rc_pkt_empty_3,  */
	0x01010277, 0x00010001, 0x002703bb,	/* (r1!=15), r1=btb.rc_sop_empty,  */
	0x01010278, 0x00010001, 0x001903bc,	/* (r1!=2), r1=btb.ll_arb_empty,  */
	0x02010279, 0x00010001, 0x004603bd,	/* (r1>46), r1=btb.block_occupancy,  */
	0x0101027a, 0x00010001, 0x000003be,	/* (r1!=0), r1=btb.rc_pkt_state,  */
	0x0101027b, 0x00010001, 0x001903bf,	/* (r1!=2), r1=btb.wc_status_0 width=3 access=WB,  */
	0x0000027c, 0x00010102, 0x000003c0,	/* ((r1&~r2)!=0), r1=xsdm.INT_STS, r2=xsdm.INT_MASK,  */
	0x0000027d, 0x00010002, 0x000003c3,	/* ((r1&~r2)!=0), r1=xsdm.PRTY_STS_H_0, r2=xsdm.PRTY_MASK_H_0,  */
	0x0101027e, 0x00010001, 0x000003c5,	/* (r1!=0), r1=xsdm.qm_full,  */
	0x0101027f, 0x00010001, 0x000003c6,	/* (r1!=0), r1=xsdm.rsp_brb_if_full,  */
	0x01010280, 0x00010001, 0x000003c7,	/* (r1!=0), r1=xsdm.rsp_pxp_if_full,  */
	0x01010281, 0x00010001, 0x000003c8,	/* (r1!=0), r1=xsdm.dst_pxp_if_full,  */
	0x01010282, 0x00010001, 0x000003c9,	/* (r1!=0), r1=xsdm.dst_int_ram_if_full,  */
	0x01010283, 0x00010001, 0x000003ca,	/* (r1!=0), r1=xsdm.dst_pas_buf_if_full,  */
	0x01010284, 0x00010001, 0x001b03cb,	/* (r1!=1), r1=xsdm.int_cmpl_pend_empty,  */
	0x01010285, 0x00010001, 0x001b03cc,	/* (r1!=1), r1=xsdm.int_cprm_pend_empty,  */
	0x01010286, 0x00010001, 0x004703cd,	/* (r1!=511), r1=xsdm.queue_empty,  */
	0x01010287, 0x00010001, 0x001b03ce,	/* (r1!=1), r1=xsdm.delay_fifo_empty,  */
	0x01010288, 0x00010001, 0x001b03cf,	/* (r1!=1), r1=xsdm.rsp_pxp_rdata_empty,  */
	0x01010289, 0x00010001, 0x001b03d0,	/* (r1!=1), r1=xsdm.rsp_brb_rdata_empty,  */
	0x0101028a, 0x00010001, 0x001b03d1,	/* (r1!=1), r1=xsdm.rsp_int_ram_rdata_empty,  */
	0x0101028b, 0x00010001, 0x001b03d2,	/* (r1!=1), r1=xsdm.rsp_brb_pend_empty,  */
	0x0101028c, 0x00010001, 0x001b03d3,	/* (r1!=1), r1=xsdm.rsp_int_ram_pend_empty,  */
	0x0101028d, 0x00010001, 0x001b03d4,	/* (r1!=1), r1=xsdm.dst_pxp_immed_empty,  */
	0x0101028e, 0x00010001, 0x001b03d5,	/* (r1!=1), r1=xsdm.dst_pxp_dst_pend_empty,  */
	0x0101028f, 0x00010001, 0x001b03d6,	/* (r1!=1), r1=xsdm.dst_pxp_src_pend_empty,  */
	0x01010290, 0x00010001, 0x001b03d7,	/* (r1!=1), r1=xsdm.dst_brb_src_pend_empty,  */
	0x01010291, 0x00010001, 0x001b03d8,	/* (r1!=1), r1=xsdm.dst_brb_src_addr_empty,  */
	0x01010292, 0x00010001, 0x001b03d9,	/* (r1!=1), r1=xsdm.dst_pxp_link_empty,  */
	0x01010293, 0x00010001, 0x001b03da,	/* (r1!=1), r1=xsdm.dst_int_ram_wait_empty,  */
	0x01010294, 0x00010001, 0x001b03db,	/* (r1!=1), r1=xsdm.dst_pas_buf_wait_empty,  */
	0x01010295, 0x00010001, 0x001b03dc,	/* (r1!=1), r1=xsdm.sh_delay_empty,  */
	0x01010296, 0x00010001, 0x001b03dd,	/* (r1!=1), r1=xsdm.cm_delay_empty,  */
	0x01010297, 0x00010001, 0x001b03de,	/* (r1!=1), r1=xsdm.cmsg_que_empty,  */
	0x01010298, 0x00010001, 0x001b03df,	/* (r1!=1), r1=xsdm.ccfc_load_pend_empty,  */
	0x01010299, 0x00010001, 0x001b03e0,	/* (r1!=1), r1=xsdm.tcfc_load_pend_empty,  */
	0x0101029a, 0x00010001, 0x001b03e1,	/* (r1!=1), r1=xsdm.async_host_empty,  */
	0x0101029b, 0x00010001, 0x001b03e2,	/* (r1!=1), r1=xsdm.prm_fifo_empty,  */
	0x0000029c, 0x00010102, 0x000003e3,	/* ((r1&~r2)!=0), r1=ysdm.INT_STS, r2=ysdm.INT_MASK,  */
	0x0000029d, 0x00010002, 0x000003e6,	/* ((r1&~r2)!=0), r1=ysdm.PRTY_STS_H_0, r2=ysdm.PRTY_MASK_H_0,  */
	0x0101029e, 0x00010001, 0x000003e8,	/* (r1!=0), r1=ysdm.qm_full,  */
	0x0101029f, 0x00010001, 0x000003e9,	/* (r1!=0), r1=ysdm.rsp_brb_if_full,  */
	0x010102a0, 0x00010001, 0x000003ea,	/* (r1!=0), r1=ysdm.rsp_pxp_if_full,  */
	0x010102a1, 0x00010001, 0x000003eb,	/* (r1!=0), r1=ysdm.dst_pxp_if_full,  */
	0x010102a2, 0x00010001, 0x000003ec,	/* (r1!=0), r1=ysdm.dst_int_ram_if_full,  */
	0x010102a3, 0x00010001, 0x000003ed,	/* (r1!=0), r1=ysdm.dst_pas_buf_if_full,  */
	0x010102a4, 0x00010001, 0x001b03ee,	/* (r1!=1), r1=ysdm.int_cmpl_pend_empty,  */
	0x010102a5, 0x00010001, 0x001b03ef,	/* (r1!=1), r1=ysdm.int_cprm_pend_empty,  */
	0x010102a6, 0x00010001, 0x004703f0,	/* (r1!=511), r1=ysdm.queue_empty,  */
	0x010102a7, 0x00010001, 0x001b03f1,	/* (r1!=1), r1=ysdm.delay_fifo_empty,  */
	0x010102a8, 0x00010001, 0x001b03f2,	/* (r1!=1), r1=ysdm.rsp_pxp_rdata_empty,  */
	0x010102a9, 0x00010001, 0x001b03f3,	/* (r1!=1), r1=ysdm.rsp_brb_rdata_empty,  */
	0x010102aa, 0x00010001, 0x001b03f4,	/* (r1!=1), r1=ysdm.rsp_int_ram_rdata_empty,  */
	0x010102ab, 0x00010001, 0x001b03f5,	/* (r1!=1), r1=ysdm.rsp_brb_pend_empty,  */
	0x010102ac, 0x00010001, 0x001b03f6,	/* (r1!=1), r1=ysdm.rsp_int_ram_pend_empty,  */
	0x010102ad, 0x00010001, 0x001b03f7,	/* (r1!=1), r1=ysdm.dst_pxp_immed_empty,  */
	0x010102ae, 0x00010001, 0x001b03f8,	/* (r1!=1), r1=ysdm.dst_pxp_dst_pend_empty,  */
	0x010102af, 0x00010001, 0x001b03f9,	/* (r1!=1), r1=ysdm.dst_pxp_src_pend_empty,  */
	0x010102b0, 0x00010001, 0x001b03fa,	/* (r1!=1), r1=ysdm.dst_brb_src_pend_empty,  */
	0x010102b1, 0x00010001, 0x001b03fb,	/* (r1!=1), r1=ysdm.dst_brb_src_addr_empty,  */
	0x010102b2, 0x00010001, 0x001b03fc,	/* (r1!=1), r1=ysdm.dst_pxp_link_empty,  */
	0x010102b3, 0x00010001, 0x001b03fd,	/* (r1!=1), r1=ysdm.dst_int_ram_wait_empty,  */
	0x010102b4, 0x00010001, 0x001b03fe,	/* (r1!=1), r1=ysdm.dst_pas_buf_wait_empty,  */
	0x010102b5, 0x00010001, 0x001b03ff,	/* (r1!=1), r1=ysdm.sh_delay_empty,  */
	0x010102b6, 0x00010001, 0x001b0400,	/* (r1!=1), r1=ysdm.cm_delay_empty,  */
	0x010102b7, 0x00010001, 0x001b0401,	/* (r1!=1), r1=ysdm.cmsg_que_empty,  */
	0x010102b8, 0x00010001, 0x001b0402,	/* (r1!=1), r1=ysdm.ccfc_load_pend_empty,  */
	0x010102b9, 0x00010001, 0x001b0403,	/* (r1!=1), r1=ysdm.tcfc_load_pend_empty,  */
	0x010102ba, 0x00010001, 0x001b0404,	/* (r1!=1), r1=ysdm.async_host_empty,  */
	0x010102bb, 0x00010001, 0x001b0405,	/* (r1!=1), r1=ysdm.prm_fifo_empty,  */
	0x000002bc, 0x00010102, 0x00000406,	/* ((r1&~r2)!=0), r1=psdm.INT_STS, r2=psdm.INT_MASK,  */
	0x000002bd, 0x00010002, 0x00000409,	/* ((r1&~r2)!=0), r1=psdm.PRTY_STS_H_0, r2=psdm.PRTY_MASK_H_0,  */
	0x010102be, 0x00010001, 0x0000040b,	/* (r1!=0), r1=psdm.qm_full,  */
	0x010102bf, 0x00010001, 0x0000040c,	/* (r1!=0), r1=psdm.rsp_brb_if_full,  */
	0x010102c0, 0x00010001, 0x0000040d,	/* (r1!=0), r1=psdm.rsp_pxp_if_full,  */
	0x010102c1, 0x00010001, 0x0000040e,	/* (r1!=0), r1=psdm.dst_pxp_if_full,  */
	0x010102c2, 0x00010001, 0x0000040f,	/* (r1!=0), r1=psdm.dst_int_ram_if_full,  */
	0x010102c3, 0x00010001, 0x00000410,	/* (r1!=0), r1=psdm.dst_pas_buf_if_full,  */
	0x010102c4, 0x00010001, 0x001b0411,	/* (r1!=1), r1=psdm.int_cmpl_pend_empty,  */
	0x010102c5, 0x00010001, 0x001b0412,	/* (r1!=1), r1=psdm.int_cprm_pend_empty,  */
	0x010102c6, 0x00010001, 0x00470413,	/* (r1!=511), r1=psdm.queue_empty,  */
	0x010102c7, 0x00010001, 0x001b0414,	/* (r1!=1), r1=psdm.delay_fifo_empty,  */
	0x010102c8, 0x00010001, 0x001b0415,	/* (r1!=1), r1=psdm.rsp_pxp_rdata_empty,  */
	0x010102c9, 0x00010001, 0x001b0416,	/* (r1!=1), r1=psdm.rsp_brb_rdata_empty,  */
	0x010102ca, 0x00010001, 0x001b0417,	/* (r1!=1), r1=psdm.rsp_int_ram_rdata_empty,  */
	0x010102cb, 0x00010001, 0x001b0418,	/* (r1!=1), r1=psdm.rsp_brb_pend_empty,  */
	0x010102cc, 0x00010001, 0x001b0419,	/* (r1!=1), r1=psdm.rsp_int_ram_pend_empty,  */
	0x010102cd, 0x00010001, 0x001b041a,	/* (r1!=1), r1=psdm.dst_pxp_immed_empty,  */
	0x010102ce, 0x00010001, 0x001b041b,	/* (r1!=1), r1=psdm.dst_pxp_dst_pend_empty,  */
	0x010102cf, 0x00010001, 0x001b041c,	/* (r1!=1), r1=psdm.dst_pxp_src_pend_empty,  */
	0x010102d0, 0x00010001, 0x001b041d,	/* (r1!=1), r1=psdm.dst_brb_src_pend_empty,  */
	0x010102d1, 0x00010001, 0x001b041e,	/* (r1!=1), r1=psdm.dst_brb_src_addr_empty,  */
	0x010102d2, 0x00010001, 0x001b041f,	/* (r1!=1), r1=psdm.dst_pxp_link_empty,  */
	0x010102d3, 0x00010001, 0x001b0420,	/* (r1!=1), r1=psdm.dst_int_ram_wait_empty,  */
	0x010102d4, 0x00010001, 0x001b0421,	/* (r1!=1), r1=psdm.dst_pas_buf_wait_empty,  */
	0x010102d5, 0x00010001, 0x001b0422,	/* (r1!=1), r1=psdm.sh_delay_empty,  */
	0x010102d6, 0x00010001, 0x001b0423,	/* (r1!=1), r1=psdm.cm_delay_empty,  */
	0x010102d7, 0x00010001, 0x001b0424,	/* (r1!=1), r1=psdm.cmsg_que_empty,  */
	0x010102d8, 0x00010001, 0x001b0425,	/* (r1!=1), r1=psdm.ccfc_load_pend_empty,  */
	0x010102d9, 0x00010001, 0x001b0426,	/* (r1!=1), r1=psdm.tcfc_load_pend_empty,  */
	0x010102da, 0x00010001, 0x001b0427,	/* (r1!=1), r1=psdm.async_host_empty,  */
	0x010102db, 0x00010001, 0x001b0428,	/* (r1!=1), r1=psdm.prm_fifo_empty,  */
	0x000002dc, 0x00010102, 0x00000429,	/* ((r1&~r2)!=0), r1=tsdm.INT_STS, r2=tsdm.INT_MASK,  */
	0x000002dd, 0x00010002, 0x0000042c,	/* ((r1&~r2)!=0), r1=tsdm.PRTY_STS_H_0, r2=tsdm.PRTY_MASK_H_0,  */
	0x010102de, 0x00010001, 0x0000042e,	/* (r1!=0), r1=tsdm.qm_full,  */
	0x010102df, 0x00010001, 0x0000042f,	/* (r1!=0), r1=tsdm.rsp_brb_if_full,  */
	0x010102e0, 0x00010001, 0x00000430,	/* (r1!=0), r1=tsdm.rsp_pxp_if_full,  */
	0x010102e1, 0x00010001, 0x00000431,	/* (r1!=0), r1=tsdm.dst_pxp_if_full,  */
	0x010102e2, 0x00010001, 0x00000432,	/* (r1!=0), r1=tsdm.dst_int_ram_if_full,  */
	0x010102e3, 0x00010001, 0x00000433,	/* (r1!=0), r1=tsdm.dst_pas_buf_if_full,  */
	0x010102e4, 0x00010001, 0x001b0434,	/* (r1!=1), r1=tsdm.int_cmpl_pend_empty,  */
	0x010102e5, 0x00010001, 0x001b0435,	/* (r1!=1), r1=tsdm.int_cprm_pend_empty,  */
	0x010102e6, 0x00010001, 0x00470436,	/* (r1!=511), r1=tsdm.queue_empty,  */
	0x010102e7, 0x00010001, 0x001b0437,	/* (r1!=1), r1=tsdm.delay_fifo_empty,  */
	0x010102e8, 0x00010001, 0x001b0438,	/* (r1!=1), r1=tsdm.rsp_pxp_rdata_empty,  */
	0x010102e9, 0x00010001, 0x001b0439,	/* (r1!=1), r1=tsdm.rsp_brb_rdata_empty,  */
	0x010102ea, 0x00010001, 0x001b043a,	/* (r1!=1), r1=tsdm.rsp_int_ram_rdata_empty,  */
	0x010102eb, 0x00010001, 0x001b043b,	/* (r1!=1), r1=tsdm.rsp_brb_pend_empty,  */
	0x010102ec, 0x00010001, 0x001b043c,	/* (r1!=1), r1=tsdm.rsp_int_ram_pend_empty,  */
	0x010102ed, 0x00010001, 0x001b043d,	/* (r1!=1), r1=tsdm.dst_pxp_immed_empty,  */
	0x010102ee, 0x00010001, 0x001b043e,	/* (r1!=1), r1=tsdm.dst_pxp_dst_pend_empty,  */
	0x010102ef, 0x00010001, 0x001b043f,	/* (r1!=1), r1=tsdm.dst_pxp_src_pend_empty,  */
	0x010102f0, 0x00010001, 0x001b0440,	/* (r1!=1), r1=tsdm.dst_brb_src_pend_empty,  */
	0x010102f1, 0x00010001, 0x001b0441,	/* (r1!=1), r1=tsdm.dst_brb_src_addr_empty,  */
	0x010102f2, 0x00010001, 0x001b0442,	/* (r1!=1), r1=tsdm.dst_pxp_link_empty,  */
	0x010102f3, 0x00010001, 0x001b0443,	/* (r1!=1), r1=tsdm.dst_int_ram_wait_empty,  */
	0x010102f4, 0x00010001, 0x001b0444,	/* (r1!=1), r1=tsdm.dst_pas_buf_wait_empty,  */
	0x010102f5, 0x00010001, 0x001b0445,	/* (r1!=1), r1=tsdm.sh_delay_empty,  */
	0x010102f6, 0x00010001, 0x001b0446,	/* (r1!=1), r1=tsdm.cm_delay_empty,  */
	0x010102f7, 0x00010001, 0x001b0447,	/* (r1!=1), r1=tsdm.cmsg_que_empty,  */
	0x010102f8, 0x00010001, 0x001b0448,	/* (r1!=1), r1=tsdm.ccfc_load_pend_empty,  */
	0x010102f9, 0x00010001, 0x001b0449,	/* (r1!=1), r1=tsdm.tcfc_load_pend_empty,  */
	0x010102fa, 0x00010001, 0x001b044a,	/* (r1!=1), r1=tsdm.async_host_empty,  */
	0x010102fb, 0x00010001, 0x001b044b,	/* (r1!=1), r1=tsdm.prm_fifo_empty,  */
	0x000002fc, 0x00010102, 0x0000044c,	/* ((r1&~r2)!=0), r1=msdm.INT_STS, r2=msdm.INT_MASK,  */
	0x000002fd, 0x00010002, 0x0000044f,	/* ((r1&~r2)!=0), r1=msdm.PRTY_STS_H_0, r2=msdm.PRTY_MASK_H_0,  */
	0x010102fe, 0x00010001, 0x00000451,	/* (r1!=0), r1=msdm.qm_full,  */
	0x010102ff, 0x00010001, 0x00000452,	/* (r1!=0), r1=msdm.rsp_brb_if_full,  */
	0x01010300, 0x00010001, 0x00000453,	/* (r1!=0), r1=msdm.rsp_pxp_if_full,  */
	0x01010301, 0x00010001, 0x00000454,	/* (r1!=0), r1=msdm.dst_pxp_if_full,  */
	0x01010302, 0x00010001, 0x00000455,	/* (r1!=0), r1=msdm.dst_int_ram_if_full,  */
	0x01010303, 0x00010001, 0x00000456,	/* (r1!=0), r1=msdm.dst_pas_buf_if_full,  */
	0x01010304, 0x00010001, 0x001b0457,	/* (r1!=1), r1=msdm.int_cmpl_pend_empty,  */
	0x01010305, 0x00010001, 0x001b0458,	/* (r1!=1), r1=msdm.int_cprm_pend_empty,  */
	0x01010306, 0x00010001, 0x00470459,	/* (r1!=511), r1=msdm.queue_empty,  */
	0x01010307, 0x00010001, 0x001b045a,	/* (r1!=1), r1=msdm.delay_fifo_empty,  */
	0x01010308, 0x00010001, 0x001b045b,	/* (r1!=1), r1=msdm.rsp_pxp_rdata_empty,  */
	0x01010309, 0x00010001, 0x001b045c,	/* (r1!=1), r1=msdm.rsp_brb_rdata_empty,  */
	0x0101030a, 0x00010001, 0x001b045d,	/* (r1!=1), r1=msdm.rsp_int_ram_rdata_empty,  */
	0x0101030b, 0x00010001, 0x001b045e,	/* (r1!=1), r1=msdm.rsp_brb_pend_empty,  */
	0x0101030c, 0x00010001, 0x001b045f,	/* (r1!=1), r1=msdm.rsp_int_ram_pend_empty,  */
	0x0101030d, 0x00010001, 0x001b0460,	/* (r1!=1), r1=msdm.dst_pxp_immed_empty,  */
	0x0101030e, 0x00010001, 0x001b0461,	/* (r1!=1), r1=msdm.dst_pxp_dst_pend_empty,  */
	0x0101030f, 0x00010001, 0x001b0462,	/* (r1!=1), r1=msdm.dst_pxp_src_pend_empty,  */
	0x01010310, 0x00010001, 0x001b0463,	/* (r1!=1), r1=msdm.dst_brb_src_pend_empty,  */
	0x01010311, 0x00010001, 0x001b0464,	/* (r1!=1), r1=msdm.dst_brb_src_addr_empty,  */
	0x01010312, 0x00010001, 0x001b0465,	/* (r1!=1), r1=msdm.dst_pxp_link_empty,  */
	0x01010313, 0x00010001, 0x001b0466,	/* (r1!=1), r1=msdm.dst_int_ram_wait_empty,  */
	0x01010314, 0x00010001, 0x001b0467,	/* (r1!=1), r1=msdm.dst_pas_buf_wait_empty,  */
	0x01010315, 0x00010001, 0x001b0468,	/* (r1!=1), r1=msdm.sh_delay_empty,  */
	0x01010316, 0x00010001, 0x001b0469,	/* (r1!=1), r1=msdm.cm_delay_empty,  */
	0x01010317, 0x00010001, 0x001b046a,	/* (r1!=1), r1=msdm.cmsg_que_empty,  */
	0x01010318, 0x00010001, 0x001b046b,	/* (r1!=1), r1=msdm.ccfc_load_pend_empty,  */
	0x01010319, 0x00010001, 0x001b046c,	/* (r1!=1), r1=msdm.tcfc_load_pend_empty,  */
	0x0101031a, 0x00010001, 0x001b046d,	/* (r1!=1), r1=msdm.async_host_empty,  */
	0x0101031b, 0x00010001, 0x001b046e,	/* (r1!=1), r1=msdm.prm_fifo_empty,  */
	0x0000031c, 0x00010102, 0x0000046f,	/* ((r1&~r2)!=0), r1=usdm.INT_STS, r2=usdm.INT_MASK,  */
	0x0000031d, 0x00010002, 0x00000472,	/* ((r1&~r2)!=0), r1=usdm.PRTY_STS_H_0, r2=usdm.PRTY_MASK_H_0,  */
	0x0101031e, 0x00010001, 0x00000474,	/* (r1!=0), r1=usdm.qm_full,  */
	0x0101031f, 0x00010001, 0x00000475,	/* (r1!=0), r1=usdm.rsp_brb_if_full,  */
	0x01010320, 0x00010001, 0x00000476,	/* (r1!=0), r1=usdm.rsp_pxp_if_full,  */
	0x01010321, 0x00010001, 0x00000477,	/* (r1!=0), r1=usdm.dst_pxp_if_full,  */
	0x01010322, 0x00010001, 0x00000478,	/* (r1!=0), r1=usdm.dst_int_ram_if_full,  */
	0x01010323, 0x00010001, 0x00000479,	/* (r1!=0), r1=usdm.dst_pas_buf_if_full,  */
	0x01010324, 0x00010001, 0x001b047a,	/* (r1!=1), r1=usdm.int_cmpl_pend_empty,  */
	0x01010325, 0x00010001, 0x001b047b,	/* (r1!=1), r1=usdm.int_cprm_pend_empty,  */
	0x01010326, 0x00010001, 0x0047047c,	/* (r1!=511), r1=usdm.queue_empty,  */
	0x01010327, 0x00010001, 0x001b047d,	/* (r1!=1), r1=usdm.delay_fifo_empty,  */
	0x01010328, 0x00010001, 0x001b047e,	/* (r1!=1), r1=usdm.rsp_pxp_rdata_empty,  */
	0x01010329, 0x00010001, 0x001b047f,	/* (r1!=1), r1=usdm.rsp_brb_rdata_empty,  */
	0x0101032a, 0x00010001, 0x001b0480,	/* (r1!=1), r1=usdm.rsp_int_ram_rdata_empty,  */
	0x0101032b, 0x00010001, 0x001b0481,	/* (r1!=1), r1=usdm.rsp_brb_pend_empty,  */
	0x0101032c, 0x00010001, 0x001b0482,	/* (r1!=1), r1=usdm.rsp_int_ram_pend_empty,  */
	0x0101032d, 0x00010001, 0x001b0483,	/* (r1!=1), r1=usdm.dst_pxp_immed_empty,  */
	0x0101032e, 0x00010001, 0x001b0484,	/* (r1!=1), r1=usdm.dst_pxp_dst_pend_empty,  */
	0x0101032f, 0x00010001, 0x001b0485,	/* (r1!=1), r1=usdm.dst_pxp_src_pend_empty,  */
	0x01010330, 0x00010001, 0x001b0486,	/* (r1!=1), r1=usdm.dst_brb_src_pend_empty,  */
	0x01010331, 0x00010001, 0x001b0487,	/* (r1!=1), r1=usdm.dst_brb_src_addr_empty,  */
	0x01010332, 0x00010001, 0x001b0488,	/* (r1!=1), r1=usdm.dst_pxp_link_empty,  */
	0x01010333, 0x00010001, 0x001b0489,	/* (r1!=1), r1=usdm.dst_int_ram_wait_empty,  */
	0x01010334, 0x00010001, 0x001b048a,	/* (r1!=1), r1=usdm.dst_pas_buf_wait_empty,  */
	0x01010335, 0x00010001, 0x001b048b,	/* (r1!=1), r1=usdm.sh_delay_empty,  */
	0x01010336, 0x00010001, 0x001b048c,	/* (r1!=1), r1=usdm.cm_delay_empty,  */
	0x01010337, 0x00010001, 0x001b048d,	/* (r1!=1), r1=usdm.cmsg_que_empty,  */
	0x01010338, 0x00010001, 0x001b048e,	/* (r1!=1), r1=usdm.ccfc_load_pend_empty,  */
	0x01010339, 0x00010001, 0x001b048f,	/* (r1!=1), r1=usdm.tcfc_load_pend_empty,  */
	0x0101033a, 0x00010001, 0x001b0490,	/* (r1!=1), r1=usdm.async_host_empty,  */
	0x0101033b, 0x00010001, 0x001b0491,	/* (r1!=1), r1=usdm.prm_fifo_empty,  */
	0x0000033c, 0x00010002, 0x00000492,	/* ((r1&~r2)!=0), r1=xcm.INT_STS_0, r2=xcm.INT_MASK_0,  */
	0x0000033d, 0x00010002, 0x00000494,	/* ((r1&~r2)!=0), r1=xcm.INT_STS_1, r2=xcm.INT_MASK_1,  */
	0x0000033e, 0x00010102, 0x00000496,	/* ((r1&~r2)!=0), r1=xcm.INT_STS_2, r2=xcm.INT_MASK_2,  */
	0x0000033f, 0x00010002, 0x00000499,	/* ((r1&~r2)!=0), r1=xcm.PRTY_STS_H_0, r2=xcm.PRTY_MASK_H_0,  */
	0x00000340, 0x00010002, 0x0000049b,	/* ((r1&~r2)!=0), r1=xcm.PRTY_STS_H_1, r2=xcm.PRTY_MASK_H_1,  */
	0x01000341, 0x00010001, 0x0000049d,	/* (r1!=0), r1=xcm.fi_desc_input_violate,  */
	0x01010342, 0x00010001, 0x0000049e,	/* (r1!=0), r1=xcm.ia_agg_con_part_fill_lvl,  */
	0x01010343, 0x00010001, 0x0000049f,	/* (r1!=0), r1=xcm.ia_sm_con_part_fill_lvl,  */
	0x01010344, 0x00010001, 0x000004a0,	/* (r1!=0), r1=xcm.ia_trans_part_fill_lvl,  */
	0x01010345, 0x00010001, 0x004804a1,	/* (r1!=reset1), r1=xcm.xx_free_cnt,  */
	0x01010346, 0x00010001, 0x000004a2,	/* (r1!=0), r1=xcm.xx_lcid_cam_fill_lvl,  */
	0x01010347, 0x00010001, 0x000004a3,	/* (r1!=0), r1=xcm.xx_non_lock_cnt,  */
	0x01010348, 0x00010001, 0x000004a4,	/* (r1!=0), r1=xcm.xx_lock_cnt,  */
	0x01010349, 0x00010001, 0x000004a5,	/* (r1!=0), r1=xcm.xx_cbyp_tbl_fill_lvl,  */
	0x0101034a, 0x00010001, 0x000004a6,	/* (r1!=0), r1=xcm.xx_pref_dir_fill_lvl,  */
	0x0101034b, 0x00010001, 0x000004a7,	/* (r1!=0), r1=xcm.xx_pref_aggst_fill_lvl,  */
	0x0101034c, 0x00010001, 0x000004a8,	/* (r1!=0), r1=xcm.xx_pref_byp_fill_lvl,  */
	0x0101034d, 0x00010001, 0x000004a9,	/* (r1!=0), r1=xcm.prcs_agg_con_curr_st,  */
	0x0101034e, 0x00010001, 0x000004aa,	/* (r1!=0), r1=xcm.prcs_sm_con_curr_st,  */
	0x0101034f, 0x00010001, 0x000004ab,	/* (r1!=0), r1=xcm.agg_con_fic_buf_fill_lvl,  */
	0x01010350, 0x00010001, 0x000004ac,	/* (r1!=0), r1=xcm.sm_con_fic_buf_fill_lvl,  */
	0x01010351, 0x00010001, 0x000004ad,	/* (r1!=0), r1=xcm.in_prcs_tbl_fill_lvl,  */
	0x01010352, 0x00010001, 0x000004ae,	/* (r1!=0), r1=xcm.cmpl_dir_curr_st,  */
	0x01010353, 0x00010001, 0x001b04af,	/* (r1!=reset1), r1=xcm.ccfc_init_crd,  */
	0x01010354, 0x00010001, 0x001804b0,	/* (r1!=reset1), r1=xcm.qm_init_crd0,  */
	0x01010355, 0x00010001, 0x001804b1,	/* (r1!=reset1), r1=xcm.qm_init_crd1,  */
	0x01010356, 0x00010001, 0x001f04b2,	/* (r1!=reset1), r1=xcm.tm_init_crd,  */
	0x01010357, 0x00010001, 0x004904b3,	/* (r1!=reset1), r1=xcm.fic_init_crd,  */
	0x01000358, 0x00010001, 0x000004b4,	/* (r1!=0), r1=xcm.msdm_length_mis,  */
	0x01000359, 0x00010001, 0x000004b5,	/* (r1!=0), r1=xcm.xsdm_length_mis,  */
	0x0100035a, 0x00010001, 0x000004b6,	/* (r1!=0), r1=xcm.ysdm_length_mis,  */
	0x0100035b, 0x00010001, 0x000004b7,	/* (r1!=0), r1=xcm.dorq_length_mis,  */
	0x0100035c, 0x00010001, 0x000004b8,	/* (r1!=0), r1=xcm.pbf_length_mis,  */
	0x0100035d, 0x00010101, 0x001b04b9,	/* (r1!=0x01), r1=xcm.grc_buf_empty,  */
	0x0101035e, 0x00010001, 0x000004bb,	/* (r1!=0), r1=xcm.is_qm_p_fill_lvl,  */
	0x0101035f, 0x00010001, 0x000004bc,	/* (r1!=0), r1=xcm.is_qm_s_fill_lvl,  */
	0x01010360, 0x00010001, 0x000004bd,	/* (r1!=0), r1=xcm.is_tm_fill_lvl,  */
	0x01010361, 0x00010001, 0x000004be,	/* (r1!=0), r1=xcm.is_storm_fill_lvl,  */
	0x01010362, 0x00010001, 0x000004bf,	/* (r1!=0), r1=xcm.is_msdm_fill_lvl,  */
	0x01010363, 0x00010001, 0x000004c0,	/* (r1!=0), r1=xcm.is_xsdm_fill_lvl,  */
	0x01010364, 0x00010001, 0x000004c1,	/* (r1!=0), r1=xcm.is_ysdm_fill_lvl,  */
	0x01010365, 0x00010001, 0x000004c2,	/* (r1!=0), r1=xcm.is_msem_fill_lvl,  */
	0x01010366, 0x00010001, 0x000004c3,	/* (r1!=0), r1=xcm.is_usem_fill_lvl,  */
	0x01010367, 0x00010001, 0x000004c4,	/* (r1!=0), r1=xcm.is_ysem_fill_lvl,  */
	0x01010368, 0x00010001, 0x000004c5,	/* (r1!=0), r1=xcm.is_dorq_fill_lvl,  */
	0x01010369, 0x00010001, 0x000004c6,	/* (r1!=0), r1=xcm.is_pbf_fill_lvl,  */
	0x0000036a, 0x00010002, 0x000004c7,	/* ((r1&~r2)!=0), r1=ycm.INT_STS_0, r2=ycm.INT_MASK_0,  */
	0x0000036b, 0x00010002, 0x000004c9,	/* ((r1&~r2)!=0), r1=ycm.INT_STS_1, r2=ycm.INT_MASK_1,  */
	0x0000036c, 0x00010002, 0x000004cb,	/* ((r1&~r2)!=0), r1=ycm.INT_STS_2, r2=ycm.INT_MASK_2,  */
	0x0000036d, 0x00010002, 0x000004cd,	/* ((r1&~r2)!=0), r1=ycm.PRTY_STS_H_0, r2=ycm.PRTY_MASK_H_0,  */
	0x0000036e, 0x00010002, 0x000004cf,	/* ((r1&~r2)!=0), r1=ycm.PRTY_STS_H_1, r2=ycm.PRTY_MASK_H_1,  */
	0x0100036f, 0x00010001, 0x000004d1,	/* (r1!=0), r1=ycm.fi_desc_input_violate,  */
	0x01000370, 0x00010001, 0x000004d2,	/* (r1!=0), r1=ycm.se_desc_input_violate,  */
	0x01010371, 0x00010001, 0x000004d3,	/* (r1!=0), r1=ycm.ia_sm_con_part_fill_lvl,  */
	0x01010372, 0x00010001, 0x000004d4,	/* (r1!=0), r1=ycm.ia_agg_task_part_fill_lvl,  */
	0x01010373, 0x00010001, 0x000004d5,	/* (r1!=0), r1=ycm.ia_sm_task_part_fill_lvl,  */
	0x01010374, 0x00010001, 0x000004d6,	/* (r1!=0), r1=ycm.ia_trans_part_fill_lvl,  */
	0x01010375, 0x00010001, 0x004804d7,	/* (r1!=reset1), r1=ycm.xx_free_cnt,  */
	0x01010376, 0x00010001, 0x000004d8,	/* (r1!=0), r1=ycm.xx_lcid_cam_fill_lvl,  */
	0x01010377, 0x00010001, 0x000004d9,	/* (r1!=0), r1=ycm.xx_non_lock_cnt,  */
	0x01010378, 0x00010001, 0x000004da,	/* (r1!=0), r1=ycm.xx_lock_cnt,  */
	0x01010379, 0x00010001, 0x000004db,	/* (r1!=0), r1=ycm.xx_cbyp_tbl_fill_lvl,  */
	0x0101037a, 0x00010001, 0x000004dc,	/* (r1!=0), r1=ycm.xx_tbyp_tbl_fill_lvl,  */
	0x0101037b, 0x00010001, 0x000004dd,	/* (r1!=0), r1=ycm.xx_tbyp_tbl_fill_lvl,  */
	0x0101037c, 0x00010001, 0x000004de,	/* (r1!=0), r1=ycm.xx_pref_dir_fill_lvl,  */
	0x0101037d, 0x00010001, 0x000004df,	/* (r1!=0), r1=ycm.xx_pref_aggst_fill_lvl,  */
	0x0101037e, 0x00010001, 0x000004e0,	/* (r1!=0), r1=ycm.xx_pref_byp_fill_lvl,  */
	0x0101037f, 0x00010001, 0x000004e1,	/* (r1!=0), r1=ycm.prcs_agg_con_curr_st,  */
	0x01010380, 0x00010001, 0x000004e2,	/* (r1!=0), r1=ycm.prcs_sm_con_curr_st,  */
	0x01010381, 0x00010001, 0x000004e3,	/* (r1!=0), r1=ycm.prcs_agg_task_curr_st,  */
	0x01010382, 0x00010001, 0x000004e4,	/* (r1!=0), r1=ycm.prcs_sm_task_curr_st,  */
	0x01010383, 0x00010001, 0x000004e5,	/* (r1!=0), r1=ycm.sm_con_fic_buf_fill_lvl,  */
	0x01010384, 0x00010001, 0x000004e6,	/* (r1!=0), r1=ycm.agg_task_fic_buf_fill_lvl,  */
	0x01010385, 0x00010001, 0x000004e7,	/* (r1!=0), r1=ycm.sm_task_fic_buf_fill_lvl,  */
	0x01010386, 0x00010001, 0x000004e8,	/* (r1!=0), r1=ycm.in_prcs_tbl_fill_lvl,  */
	0x01010387, 0x00010001, 0x000004e9,	/* (r1!=0), r1=ycm.cmpl_dir_curr_st,  */
	0x01010388, 0x00010001, 0x001b04ea,	/* (r1!=reset1), r1=ycm.ccfc_init_crd,  */
	0x01010389, 0x00010001, 0x001b04eb,	/* (r1!=reset1), r1=ycm.tcfc_init_crd,  */
	0x0101038a, 0x00010001, 0x001804ec,	/* (r1!=reset1), r1=ycm.qm_init_crd0,  */
	0x0101038b, 0x00010001, 0x004a04ed,	/* (r1!=reset1), r1=ycm.fic_init_crd,  */
	0x0100038c, 0x00010001, 0x000004ee,	/* (r1!=0), r1=ycm.msdm_length_mis,  */
	0x0100038d, 0x00010001, 0x000004ef,	/* (r1!=0), r1=ycm.ysdm_length_mis,  */
	0x0100038e, 0x00010001, 0x000004f0,	/* (r1!=0), r1=ycm.pbf_length_mis,  */
	0x0100038f, 0x00010001, 0x000004f1,	/* (r1!=0), r1=ycm.xyld_length_mis,  */
	0x01000390, 0x00010101, 0x001b04f2,	/* (r1!=0x01), r1=ycm.grc_buf_empty,  */
	0x01010391, 0x00010001, 0x000004f4,	/* (r1!=0), r1=ycm.is_qm_p_fill_lvl,  */
	0x01010392, 0x00010001, 0x000004f5,	/* (r1!=0), r1=ycm.is_qm_s_fill_lvl,  */
	0x01010393, 0x00010001, 0x000004f6,	/* (r1!=0), r1=ycm.is_storm_fill_lvl,  */
	0x01010394, 0x00010001, 0x000004f7,	/* (r1!=0), r1=ycm.is_msdm_fill_lvl,  */
	0x01010395, 0x00010001, 0x000004f8,	/* (r1!=0), r1=ycm.is_ysdm_fill_lvl,  */
	0x01010396, 0x00010001, 0x000004f9,	/* (r1!=0), r1=ycm.is_xyld_fill_lvl,  */
	0x01010397, 0x00010001, 0x000004fa,	/* (r1!=0), r1=ycm.is_msem_fill_lvl,  */
	0x01010398, 0x00010001, 0x000004fb,	/* (r1!=0), r1=ycm.is_usem_fill_lvl,  */
	0x01010399, 0x00010001, 0x000004fc,	/* (r1!=0), r1=ycm.is_pbf_fill_lvl,  */
	0x0000039a, 0x00010002, 0x000004fd,	/* ((r1&~r2)!=0), r1=pcm.INT_STS_0, r2=pcm.INT_MASK_0,  */
	0x0000039b, 0x00010002, 0x000004ff,	/* ((r1&~r2)!=0), r1=pcm.INT_STS_1, r2=pcm.INT_MASK_1,  */
	0x0000039c, 0x00010002, 0x00000501,	/* ((r1&~r2)!=0), r1=pcm.INT_STS_2, r2=pcm.INT_MASK_2,  */
	0x0000039d, 0x00010002, 0x00000503,	/* ((r1&~r2)!=0), r1=pcm.PRTY_STS_H_0, r2=pcm.PRTY_MASK_H_0,  */
	0x0100039e, 0x00010001, 0x00000505,	/* (r1!=0), r1=pcm.fi_desc_input_violate,  */
	0x0101039f, 0x00010001, 0x00000506,	/* (r1!=0), r1=pcm.ia_sm_con_part_fill_lvl,  */
	0x010103a0, 0x00010001, 0x00000507,	/* (r1!=0), r1=pcm.ia_trans_part_fill_lvl,  */
	0x010103a1, 0x00010001, 0x001f0508,	/* (r1!=reset1), r1=pcm.xx_free_cnt,  */
	0x010103a2, 0x00010001, 0x00000509,	/* (r1!=0), r1=pcm.xx_lcid_cam_fill_lvl,  */
	0x010103a3, 0x00010001, 0x0000050a,	/* (r1!=0), r1=pcm.xx_non_lock_cnt,  */
	0x010103a4, 0x00010001, 0x0000050b,	/* (r1!=0), r1=pcm.xx_lock_cnt,  */
	0x010103a5, 0x00010001, 0x0000050c,	/* (r1!=0), r1=pcm.xx_pref_dir_fill_lvl,  */
	0x010103a6, 0x00010001, 0x0000050d,	/* (r1!=0), r1=pcm.xx_pref_aggst_fill_lvl,  */
	0x010103a7, 0x00010001, 0x0000050e,	/* (r1!=0), r1=pcm.prcs_sm_con_curr_st,  */
	0x010103a8, 0x00010001, 0x0000050f,	/* (r1!=0), r1=pcm.sm_con_fic_buf_fill_lvl,  */
	0x010103a9, 0x00010001, 0x00000510,	/* (r1!=0), r1=pcm.in_prcs_tbl_fill_lvl,  */
	0x010103aa, 0x00010001, 0x00000511,	/* (r1!=0), r1=pcm.cmpl_dir_curr_st,  */
	0x010103ab, 0x00010001, 0x001b0512,	/* (r1!=reset1), r1=pcm.ccfc_init_crd,  */
	0x010103ac, 0x00010001, 0x00490513,	/* (r1!=reset1), r1=pcm.fic_init_crd,  */
	0x010003ad, 0x00010001, 0x00000514,	/* (r1!=0), r1=pcm.psdm_length_mis,  */
	0x010003ae, 0x00010001, 0x00000515,	/* (r1!=0), r1=pcm.pbf_length_mis,  */
	0x010003af, 0x00010101, 0x001b0516,	/* (r1!=0x01), r1=pcm.grc_buf_empty,  */
	0x010103b0, 0x00010001, 0x00000518,	/* (r1!=0), r1=pcm.is_storm_fill_lvl,  */
	0x010103b1, 0x00010001, 0x00000519,	/* (r1!=0), r1=pcm.is_psdm_fill_lvl,  */
	0x000003b2, 0x00010002, 0x0000051a,	/* ((r1&~r2)!=0), r1=tcm.INT_STS_0, r2=tcm.INT_MASK_0,  */
	0x000003b3, 0x00010002, 0x0000051c,	/* ((r1&~r2)!=0), r1=tcm.INT_STS_1, r2=tcm.INT_MASK_1,  */
	0x000203b4, 0x00010002, 0x0000051e,	/* ((r1&~r2)!=0), r1=tcm.INT_STS_2, r2=tcm.INT_MASK_2,  */
	0x000003b5, 0x00010002, 0x00000520,	/* ((r1&~r2)!=0), r1=tcm.PRTY_STS_H_0, r2=tcm.PRTY_MASK_H_0,  */
	0x000003b6, 0x00010002, 0x00000522,	/* ((r1&~r2)!=0), r1=tcm.PRTY_STS_H_1, r2=tcm.PRTY_MASK_H_1,  */
	0x010003b7, 0x00010001, 0x00000524,	/* (r1!=0), r1=tcm.fi_desc_input_violate,  */
	0x010003b8, 0x00010001, 0x00000525,	/* (r1!=0), r1=tcm.se_desc_input_violate,  */
	0x010103b9, 0x00010001, 0x00000526,	/* (r1!=0), r1=tcm.ia_agg_con_part_fill_lvl,  */
	0x010103ba, 0x00010001, 0x00000527,	/* (r1!=0), r1=tcm.ia_sm_con_part_fill_lvl,  */
	0x010103bb, 0x00010001, 0x00000528,	/* (r1!=0), r1=tcm.ia_agg_task_part_fill_lvl,  */
	0x010103bc, 0x00010001, 0x00000529,	/* (r1!=0), r1=tcm.ia_sm_task_part_fill_lvl,  */
	0x010103bd, 0x00010001, 0x0000052a,	/* (r1!=0), r1=tcm.ia_trans_part_fill_lvl,  */
	0x010103be, 0x00010001, 0x0048052b,	/* (r1!=reset1), r1=tcm.xx_free_cnt,  */
	0x010103bf, 0x00010001, 0x0000052c,	/* (r1!=0), r1=tcm.xx_lcid_cam_fill_lvl,  */
	0x010103c0, 0x00010001, 0x0000052d,	/* (r1!=0), r1=tcm.xx_non_lock_cnt,  */
	0x010103c1, 0x00010001, 0x0000052e,	/* (r1!=0), r1=tcm.xx_lock_cnt,  */
	0x010103c2, 0x00010001, 0x0000052f,	/* (r1!=0), r1=tcm.xx_cbyp_tbl_fill_lvl,  */
	0x010103c3, 0x00010001, 0x00000530,	/* (r1!=0), r1=tcm.xx_tbyp_tbl_fill_lvl,  */
	0x010103c4, 0x00010001, 0x00000531,	/* (r1!=0), r1=tcm.xx_tbyp_tbl_fill_lvl,  */
	0x010103c5, 0x00010001, 0x00000532,	/* (r1!=0), r1=tcm.xx_pref_dir_fill_lvl,  */
	0x010103c6, 0x00010001, 0x00000533,	/* (r1!=0), r1=tcm.xx_pref_aggst_fill_lvl,  */
	0x010103c7, 0x00010001, 0x00000534,	/* (r1!=0), r1=tcm.xx_pref_byp_fill_lvl,  */
	0x010103c8, 0x00010001, 0x00000535,	/* (r1!=0), r1=tcm.prcs_agg_con_curr_st,  */
	0x010103c9, 0x00010001, 0x00000536,	/* (r1!=0), r1=tcm.prcs_sm_con_curr_st,  */
	0x010103ca, 0x00010001, 0x00000537,	/* (r1!=0), r1=tcm.prcs_agg_task_curr_st,  */
	0x010103cb, 0x00010001, 0x00000538,	/* (r1!=0), r1=tcm.prcs_sm_task_curr_st,  */
	0x010103cc, 0x00010001, 0x00000539,	/* (r1!=0), r1=tcm.agg_con_fic_buf_fill_lvl,  */
	0x010103cd, 0x00010001, 0x0000053a,	/* (r1!=0), r1=tcm.sm_con_fic_buf_fill_lvl,  */
	0x010103ce, 0x00010001, 0x0000053b,	/* (r1!=0), r1=tcm.agg_task_fic_buf_fill_lvl,  */
	0x010103cf, 0x00010001, 0x0000053c,	/* (r1!=0), r1=tcm.sm_task_fic_buf_fill_lvl,  */
	0x010103d0, 0x00010001, 0x0000053d,	/* (r1!=0), r1=tcm.in_prcs_tbl_fill_lvl,  */
	0x010103d1, 0x00010001, 0x0000053e,	/* (r1!=0), r1=tcm.cmpl_dir_curr_st,  */
	0x010103d2, 0x00010001, 0x001b053f,	/* (r1!=reset1), r1=tcm.ccfc_init_crd,  */
	0x010103d3, 0x00010001, 0x001b0540,	/* (r1!=reset1), r1=tcm.tcfc_init_crd,  */
	0x010103d4, 0x00010001, 0x00180541,	/* (r1!=reset1), r1=tcm.qm_init_crd0,  */
	0x010103d5, 0x00010001, 0x001f0542,	/* (r1!=reset1), r1=tcm.tm_init_crd,  */
	0x010103d6, 0x00010001, 0x004b0543,	/* (r1!=reset1), r1=tcm.fic_init_crd,  */
	0x010003d7, 0x00010001, 0x00000544,	/* (r1!=0), r1=tcm.tsdm_length_mis,  */
	0x010003d8, 0x00010001, 0x00000545,	/* (r1!=0), r1=tcm.dorq_length_mis,  */
	0x010003d9, 0x00010001, 0x00000546,	/* (r1!=0), r1=tcm.pbf_length_mis,  */
	0x010003da, 0x00010001, 0x00000547,	/* (r1!=0), r1=tcm.prs_length_mis,  */
	0x010003db, 0x00010101, 0x001b0548,	/* (r1!=0x01), r1=tcm.grc_buf_empty,  */
	0x010103dc, 0x00010001, 0x0000054a,	/* (r1!=0), r1=tcm.is_qm_p_fill_lvl,  */
	0x010103dd, 0x00010001, 0x0000054b,	/* (r1!=0), r1=tcm.is_qm_s_fill_lvl,  */
	0x010103de, 0x00010001, 0x0000054c,	/* (r1!=0), r1=tcm.is_tm_fill_lvl,  */
	0x010103df, 0x00010001, 0x0000054d,	/* (r1!=0), r1=tcm.is_storm_fill_lvl,  */
	0x010103e0, 0x00010001, 0x0000054e,	/* (r1!=0), r1=tcm.is_tsdm_fill_lvl,  */
	0x010103e1, 0x00010001, 0x0000054f,	/* (r1!=0), r1=tcm.is_msem_fill_lvl,  */
	0x010103e2, 0x00010001, 0x00000550,	/* (r1!=0), r1=tcm.is_dorq_fill_lvl,  */
	0x010103e3, 0x00010001, 0x00000551,	/* (r1!=0), r1=tcm.is_pbf_fill_lvl,  */
	0x000003e4, 0x00010002, 0x00000552,	/* ((r1&~r2)!=0), r1=mcm.INT_STS_0, r2=mcm.INT_MASK_0,  */
	0x000003e5, 0x00010002, 0x00000554,	/* ((r1&~r2)!=0), r1=mcm.INT_STS_1, r2=mcm.INT_MASK_1,  */
	0x000003e6, 0x00010002, 0x00000556,	/* ((r1&~r2)!=0), r1=mcm.INT_STS_2, r2=mcm.INT_MASK_2,  */
	0x000003e7, 0x00010002, 0x00000558,	/* ((r1&~r2)!=0), r1=mcm.PRTY_STS_H_0, r2=mcm.PRTY_MASK_H_0,  */
	0x000003e8, 0x00010002, 0x0000055a,	/* ((r1&~r2)!=0), r1=mcm.PRTY_STS_H_1, r2=mcm.PRTY_MASK_H_1,  */
	0x010003e9, 0x00010001, 0x0000055c,	/* (r1!=0), r1=mcm.fi_desc_input_violate,  */
	0x010003ea, 0x00010001, 0x0000055d,	/* (r1!=0), r1=mcm.se_desc_input_violate,  */
	0x010103eb, 0x00010001, 0x0000055e,	/* (r1!=0), r1=mcm.ia_agg_con_part_fill_lvl,  */
	0x010103ec, 0x00010001, 0x0000055f,	/* (r1!=0), r1=mcm.ia_sm_con_part_fill_lvl,  */
	0x010103ed, 0x00010001, 0x00000560,	/* (r1!=0), r1=mcm.ia_agg_task_part_fill_lvl,  */
	0x010103ee, 0x00010001, 0x00000561,	/* (r1!=0), r1=mcm.ia_sm_task_part_fill_lvl,  */
	0x010103ef, 0x00010001, 0x00000562,	/* (r1!=0), r1=mcm.ia_trans_part_fill_lvl,  */
	0x010103f0, 0x00010001, 0x00480563,	/* (r1!=reset1), r1=mcm.xx_free_cnt,  */
	0x010103f1, 0x00010001, 0x00000564,	/* (r1!=0), r1=mcm.xx_lcid_cam_fill_lvl,  */
	0x010103f2, 0x00010001, 0x00000565,	/* (r1!=0), r1=mcm.xx_non_lock_cnt,  */
	0x010103f3, 0x00010001, 0x00000566,	/* (r1!=0), r1=mcm.xx_lock_cnt,  */
	0x010103f4, 0x00010001, 0x00000567,	/* (r1!=0), r1=mcm.xx_cbyp_tbl_fill_lvl,  */
	0x010103f5, 0x00010001, 0x00000568,	/* (r1!=0), r1=mcm.xx_tbyp_tbl_fill_lvl,  */
	0x010103f6, 0x00010001, 0x00000569,	/* (r1!=0), r1=mcm.xx_tbyp_tbl_fill_lvl,  */
	0x010103f7, 0x00010001, 0x0000056a,	/* (r1!=0), r1=mcm.xx_pref_dir_fill_lvl,  */
	0x010103f8, 0x00010001, 0x0000056b,	/* (r1!=0), r1=mcm.xx_pref_aggst_fill_lvl,  */
	0x010103f9, 0x00010001, 0x0000056c,	/* (r1!=0), r1=mcm.xx_pref_byp_fill_lvl,  */
	0x010103fa, 0x00010001, 0x0000056d,	/* (r1!=0), r1=mcm.prcs_agg_con_curr_st,  */
	0x010103fb, 0x00010001, 0x0000056e,	/* (r1!=0), r1=mcm.prcs_sm_con_curr_st,  */
	0x010103fc, 0x00010001, 0x0000056f,	/* (r1!=0), r1=mcm.prcs_agg_task_curr_st,  */
	0x010103fd, 0x00010001, 0x00000570,	/* (r1!=0), r1=mcm.prcs_sm_task_curr_st,  */
	0x010103fe, 0x00010001, 0x00000571,	/* (r1!=0), r1=mcm.agg_con_fic_buf_fill_lvl,  */
	0x010103ff, 0x00010001, 0x00000572,	/* (r1!=0), r1=mcm.sm_con_fic_buf_fill_lvl,  */
	0x01010400, 0x00010001, 0x00000573,	/* (r1!=0), r1=mcm.agg_task_fic_buf_fill_lvl,  */
	0x01010401, 0x00010001, 0x00000574,	/* (r1!=0), r1=mcm.sm_task_fic_buf_fill_lvl,  */
	0x01010402, 0x00010001, 0x00000575,	/* (r1!=0), r1=mcm.in_prcs_tbl_fill_lvl,  */
	0x01010403, 0x00010001, 0x00000576,	/* (r1!=0), r1=mcm.cmpl_dir_curr_st,  */
	0x01010404, 0x00010001, 0x001b0577,	/* (r1!=reset1), r1=mcm.ccfc_init_crd,  */
	0x01010405, 0x00010001, 0x001b0578,	/* (r1!=reset1), r1=mcm.tcfc_init_crd,  */
	0x01010406, 0x00010001, 0x00180579,	/* (r1!=reset1), r1=mcm.qm_init_crd0,  */
	0x01010407, 0x00010001, 0x004b057a,	/* (r1!=reset1), r1=mcm.fic_init_crd,  */
	0x01000408, 0x00010001, 0x0000057b,	/* (r1!=0), r1=mcm.msdm_length_mis,  */
	0x01000409, 0x00010001, 0x0000057c,	/* (r1!=0), r1=mcm.ysdm_length_mis,  */
	0x0100040a, 0x00010001, 0x0000057d,	/* (r1!=0), r1=mcm.usdm_length_mis,  */
	0x0100040b, 0x00010001, 0x0000057e,	/* (r1!=0), r1=mcm.pbf_length_mis,  */
	0x0100040c, 0x00010001, 0x0000057f,	/* (r1!=0), r1=mcm.tmld_length_mis,  */
	0x0100040d, 0x00010101, 0x001b0580,	/* (r1!=0x01), r1=mcm.grc_buf_empty,  */
	0x0101040e, 0x00010001, 0x00000582,	/* (r1!=0), r1=mcm.is_qm_p_fill_lvl,  */
	0x0101040f, 0x00010001, 0x00000583,	/* (r1!=0), r1=mcm.is_qm_s_fill_lvl,  */
	0x01010410, 0x00010001, 0x00000584,	/* (r1!=0), r1=mcm.is_storm_fill_lvl,  */
	0x01010411, 0x00010001, 0x00000585,	/* (r1!=0), r1=mcm.is_msdm_fill_lvl,  */
	0x01010412, 0x00010001, 0x00000586,	/* (r1!=0), r1=mcm.is_ysdm_fill_lvl,  */
	0x01010413, 0x00010001, 0x00000587,	/* (r1!=0), r1=mcm.is_usdm_fill_lvl,  */
	0x01010414, 0x00010001, 0x00000588,	/* (r1!=0), r1=mcm.is_tmld_fill_lvl,  */
	0x01010415, 0x00010001, 0x00000589,	/* (r1!=0), r1=mcm.is_usem_fill_lvl,  */
	0x01010416, 0x00010001, 0x0000058a,	/* (r1!=0), r1=mcm.is_ysem_fill_lvl,  */
	0x01010417, 0x00010001, 0x0000058b,	/* (r1!=0), r1=mcm.is_pbf_fill_lvl,  */
	0x00000418, 0x00010002, 0x0000058c,	/* ((r1&~r2)!=0), r1=ucm.INT_STS_0, r2=ucm.INT_MASK_0,  */
	0x00000419, 0x00010002, 0x0000058e,	/* ((r1&~r2)!=0), r1=ucm.INT_STS_1, r2=ucm.INT_MASK_1,  */
	0x0000041a, 0x00010002, 0x00000590,	/* ((r1&~r2)!=0), r1=ucm.INT_STS_2, r2=ucm.INT_MASK_2,  */
	0x0000041b, 0x00010002, 0x00000592,	/* ((r1&~r2)!=0), r1=ucm.PRTY_STS_H_0, r2=ucm.PRTY_MASK_H_0,  */
	0x0000041c, 0x00010002, 0x00000594,	/* ((r1&~r2)!=0), r1=ucm.PRTY_STS_H_1, r2=ucm.PRTY_MASK_H_1,  */
	0x0100041d, 0x00010001, 0x00000596,	/* (r1!=0), r1=ucm.fi_desc_input_violate,  */
	0x0100041e, 0x00010001, 0x00000597,	/* (r1!=0), r1=ucm.se_desc_input_violate,  */
	0x0101041f, 0x00010001, 0x00000598,	/* (r1!=0), r1=ucm.ia_agg_con_part_fill_lvl,  */
	0x01010420, 0x00010001, 0x00000599,	/* (r1!=0), r1=ucm.ia_sm_con_part_fill_lvl,  */
	0x01010421, 0x00010001, 0x0000059a,	/* (r1!=0), r1=ucm.ia_agg_task_part_fill_lvl,  */
	0x01010422, 0x00010001, 0x0000059b,	/* (r1!=0), r1=ucm.ia_sm_task_part_fill_lvl,  */
	0x01010423, 0x00010001, 0x0000059c,	/* (r1!=0), r1=ucm.ia_trans_part_fill_lvl,  */
	0x01010424, 0x00010001, 0x0048059d,	/* (r1!=reset1), r1=ucm.xx_free_cnt,  */
	0x01010425, 0x00010001, 0x0000059e,	/* (r1!=0), r1=ucm.xx_lcid_cam_fill_lvl,  */
	0x01010426, 0x00010001, 0x0000059f,	/* (r1!=0), r1=ucm.xx_non_lock_cnt,  */
	0x01010427, 0x00010001, 0x000005a0,	/* (r1!=0), r1=ucm.xx_lock_cnt,  */
	0x01010428, 0x00010001, 0x000005a1,	/* (r1!=0), r1=ucm.xx_cbyp_tbl_fill_lvl,  */
	0x01010429, 0x00010001, 0x000005a2,	/* (r1!=0), r1=ucm.xx_tbyp_tbl_fill_lvl,  */
	0x0101042a, 0x00010001, 0x000005a3,	/* (r1!=0), r1=ucm.xx_tbyp_tbl_fill_lvl,  */
	0x0101042b, 0x00010001, 0x000005a4,	/* (r1!=0), r1=ucm.xx_pref_dir_fill_lvl,  */
	0x0101042c, 0x00010001, 0x000005a5,	/* (r1!=0), r1=ucm.xx_pref_aggst_fill_lvl,  */
	0x0101042d, 0x00010001, 0x000005a6,	/* (r1!=0), r1=ucm.xx_pref_byp_fill_lvl,  */
	0x0101042e, 0x00010001, 0x000005a7,	/* (r1!=0), r1=ucm.prcs_agg_con_curr_st,  */
	0x0101042f, 0x00010001, 0x000005a8,	/* (r1!=0), r1=ucm.prcs_sm_con_curr_st,  */
	0x01010430, 0x00010001, 0x000005a9,	/* (r1!=0), r1=ucm.prcs_agg_task_curr_st,  */
	0x01010431, 0x00010001, 0x000005aa,	/* (r1!=0), r1=ucm.prcs_sm_task_curr_st,  */
	0x01010432, 0x00010001, 0x000005ab,	/* (r1!=0), r1=ucm.agg_con_fic_buf_fill_lvl,  */
	0x01010433, 0x00010001, 0x000005ac,	/* (r1!=0), r1=ucm.sm_con_fic_buf_fill_lvl,  */
	0x01010434, 0x00010001, 0x000005ad,	/* (r1!=0), r1=ucm.agg_task_fic_buf_fill_lvl,  */
	0x01010435, 0x00010001, 0x000005ae,	/* (r1!=0), r1=ucm.sm_task_fic_buf_fill_lvl,  */
	0x01010436, 0x00010001, 0x000005af,	/* (r1!=0), r1=ucm.in_prcs_tbl_fill_lvl,  */
	0x01010437, 0x00010001, 0x000005b0,	/* (r1!=0), r1=ucm.cmpl_dir_curr_st,  */
	0x01010438, 0x00010001, 0x001b05b1,	/* (r1!=reset1), r1=ucm.ccfc_init_crd,  */
	0x01010439, 0x00010001, 0x001b05b2,	/* (r1!=reset1), r1=ucm.tcfc_init_crd,  */
	0x0101043a, 0x00010001, 0x001805b3,	/* (r1!=reset1), r1=ucm.qm_init_crd0,  */
	0x0101043b, 0x00010001, 0x001f05b4,	/* (r1!=reset1), r1=ucm.tm_init_crd,  */
	0x0101043c, 0x00010001, 0x004a05b5,	/* (r1!=reset1), r1=ucm.fic_init_crd,  */
	0x0100043d, 0x00010001, 0x000005b6,	/* (r1!=0), r1=ucm.ysdm_length_mis,  */
	0x0100043e, 0x00010001, 0x000005b7,	/* (r1!=0), r1=ucm.usdm_length_mis,  */
	0x0100043f, 0x00010001, 0x000005b8,	/* (r1!=0), r1=ucm.dorq_length_mis,  */
	0x01000440, 0x00010001, 0x000005b9,	/* (r1!=0), r1=ucm.pbf_length_mis,  */
	0x01000441, 0x00010001, 0x000005ba,	/* (r1!=0), r1=ucm.rdif_length_mis,  */
	0x01000442, 0x00010001, 0x000005bb,	/* (r1!=0), r1=ucm.tdif_length_mis,  */
	0x01000443, 0x00010001, 0x000005bc,	/* (r1!=0), r1=ucm.muld_length_mis,  */
	0x01000444, 0x00010001, 0x000005bd,	/* (r1!=0), r1=ucm.yuld_length_mis,  */
	0x01000445, 0x00010101, 0x001b05be,	/* (r1!=0x01), r1=ucm.grc_buf_empty,  */
	0x01010446, 0x00010001, 0x000005c0,	/* (r1!=0), r1=ucm.is_qm_p_fill_lvl,  */
	0x01010447, 0x00010001, 0x000005c1,	/* (r1!=0), r1=ucm.is_qm_s_fill_lvl,  */
	0x01010448, 0x00010001, 0x000005c2,	/* (r1!=0), r1=ucm.is_tm_fill_lvl,  */
	0x01010449, 0x00010001, 0x000005c3,	/* (r1!=0), r1=ucm.is_storm_fill_lvl,  */
	0x0101044a, 0x00010001, 0x000005c4,	/* (r1!=0), r1=ucm.is_ysdm_fill_lvl,  */
	0x0101044b, 0x00010001, 0x000005c5,	/* (r1!=0), r1=ucm.is_usdm_fill_lvl,  */
	0x0101044c, 0x00010001, 0x000005c6,	/* (r1!=0), r1=ucm.is_rdif_fill_lvl,  */
	0x0101044d, 0x00010001, 0x000005c7,	/* (r1!=0), r1=ucm.is_tdif_fill_lvl,  */
	0x0101044e, 0x00010001, 0x000005c8,	/* (r1!=0), r1=ucm.is_muld_fill_lvl,  */
	0x0101044f, 0x00010001, 0x000005c9,	/* (r1!=0), r1=ucm.is_dorq_fill_lvl,  */
	0x01010450, 0x00010001, 0x000005ca,	/* (r1!=0), r1=ucm.is_pbf_fill_lvl,  */
	0x00000451, 0x00010002, 0x000005cb,	/* ((r1&~r2)!=0), r1=xsem.INT_STS_0, r2=xsem.INT_MASK_0,  */
	0x00000452, 0x00010002, 0x000005cd,	/* ((r1&~r2)!=0), r1=xsem.INT_STS_1, r2=xsem.INT_MASK_1,  */
	0x00000453, 0x00010002, 0x000005cf,	/* ((r1&~r2)!=0), r1=xsem.PRTY_STS, r2=xsem.PRTY_MASK,  */
	0x00000454, 0x00010002, 0x000005d1,	/* ((r1&~r2)!=0), r1=xsem.PRTY_STS_H_0, r2=xsem.PRTY_MASK_H_0,  */
	0x01020455, 0x00010001, 0x000005d3,	/* (r1!=0), r1=xsem.pf_err_vector,  */
	0x01010456, 0x00010001, 0x004c05d4,	/* (r1!=0x1D), r1=xsem.foc_credit[1],  */
	0x01010457, 0x00010001, 0x004d05d5,	/* (r1!=0x3F), r1=xsem.foc_credit,  */
	0x01010458, 0x00010001, 0x004e05d6,	/* (r1!=reset1), r1=xsem.num_of_threads,  */
	0x01000459, 0x00010001, 0x000005d7,	/* (r1!=0), r1=xsem.thread_error,  */
	0x0100045a, 0x00010001, 0x001b05d8,	/* (r1!=reset1), r1=xsem.dra_empty,  */
	0x0101045b, 0x00010101, 0x001b05d9,	/* (r1!=1), r1=xsem.ext_pas_empty,  */
	0x0101045c, 0x00010101, 0x001b05db,	/* (r1!=1), r1=xsem.fic_empty[0:1],  */
	0x0100045d, 0x00010101, 0x001b05dd,	/* (r1!=reset1), r1=xsem.slow_dra_fin_empty,  */
	0x0100045e, 0x00010101, 0x001b05df,	/* (r1!=reset1), r1=xsem.slow_dra_rd_empty,  */
	0x0100045f, 0x00010101, 0x001b05e1,	/* (r1!=reset1), r1=xsem.slow_dra_wr_empty,  */
	0x01010460, 0x00010101, 0x001b05e3,	/* (r1!=1), r1=xsem.slow_ext_store_empty,  */
	0x01010461, 0x00010101, 0x001b05e5,	/* (r1!=1), r1=xsem.slow_ext_load_empty,  */
	0x01010462, 0x00010001, 0x001b05e7,	/* (r1!=1), r1=xsem.sync_dbg_empty,  */
	0x01010463, 0x00010001, 0x000005e8,	/* (r1!=0), r1=xsem.ext_store_if_full,  */
	0x01010464, 0x00010001, 0x000005e9,	/* (r1!=0), r1=xsem.ram_if_full,  */
	0x00000465, 0x00010002, 0x000005ea,	/* ((r1&~r2)!=0), r1=xsem.fast_memory.INT_STS, r2=xsem.fast_memory.INT_MASK,  */
	0x01000466, 0x00010001, 0x000005ec,	/* (r1!=0), r1=xsem.fast_memory.stalled,  */
	0x00000467, 0x00010002, 0x000005ed,	/* ((r1&~r2)!=0), r1=ysem.INT_STS_0, r2=ysem.INT_MASK_0,  */
	0x00000468, 0x00010002, 0x000005ef,	/* ((r1&~r2)!=0), r1=ysem.INT_STS_1, r2=ysem.INT_MASK_1,  */
	0x00000469, 0x00010002, 0x000005f1,	/* ((r1&~r2)!=0), r1=ysem.PRTY_STS, r2=ysem.PRTY_MASK,  */
	0x0000046a, 0x00010002, 0x000005f3,	/* ((r1&~r2)!=0), r1=ysem.PRTY_STS_H_0, r2=ysem.PRTY_MASK_H_0,  */
	0x0102046b, 0x00010001, 0x000005f5,	/* (r1!=0), r1=ysem.pf_err_vector,  */
	0x0101046c, 0x00010001, 0x004905f6,	/* (r1!=40), r1=ysem.foc_credit[5],  */
	0x0101046d, 0x00010001, 0x004f05f7,	/* (r1!=0x26), r1=ysem.foc_credit[4],  */
	0x0101046e, 0x00010001, 0x004105f8,	/* (r1!=7), r1=ysem.foc_credit[3],  */
	0x0101046f, 0x00010001, 0x005005f9,	/* (r1!=0xA), r1=ysem.foc_credit[2],  */
	0x01010470, 0x00010001, 0x005105fa,	/* (r1!=26), r1=ysem.foc_credit[1],  */
	0x01010471, 0x00010001, 0x000305fb,	/* (r1!=14), r1=ysem.foc_credit,  */
	0x01010472, 0x00010001, 0x000305fc,	/* (r1!=reset1), r1=ysem.num_of_threads,  */
	0x01000473, 0x00010001, 0x000005fd,	/* (r1!=0), r1=ysem.thread_error,  */
	0x01000474, 0x00010001, 0x001b05fe,	/* (r1!=reset1), r1=ysem.dra_empty,  */
	0x01010475, 0x00010101, 0x001b05ff,	/* (r1!=1), r1=ysem.ext_pas_empty,  */
	0x01010476, 0x00010101, 0x001b0601,	/* (r1!=1), r1=ysem.fic_empty[0:1],  */
	0x01000477, 0x00010101, 0x001b0603,	/* (r1!=reset1), r1=ysem.slow_dra_fin_empty,  */
	0x01000478, 0x00010101, 0x001b0605,	/* (r1!=reset1), r1=ysem.slow_dra_rd_empty,  */
	0x01000479, 0x00010101, 0x001b0607,	/* (r1!=reset1), r1=ysem.slow_dra_wr_empty,  */
	0x0101047a, 0x00010101, 0x001b0609,	/* (r1!=1), r1=ysem.slow_ext_store_empty,  */
	0x0101047b, 0x00010101, 0x001b060b,	/* (r1!=1), r1=ysem.slow_ext_load_empty,  */
	0x0101047c, 0x00010001, 0x001b060d,	/* (r1!=1), r1=ysem.sync_dbg_empty,  */
	0x0101047d, 0x00010001, 0x0000060e,	/* (r1!=0), r1=ysem.ext_store_if_full,  */
	0x0101047e, 0x00010001, 0x0000060f,	/* (r1!=0), r1=ysem.ram_if_full,  */
	0x0000047f, 0x00010002, 0x00000610,	/* ((r1&~r2)!=0), r1=ysem.fast_memory.INT_STS, r2=ysem.fast_memory.INT_MASK,  */
	0x01000480, 0x00010001, 0x00000612,	/* (r1!=0), r1=ysem.fast_memory.stalled,  */
	0x00000481, 0x00010002, 0x00000613,	/* ((r1&~r2)!=0), r1=psem.INT_STS_0, r2=psem.INT_MASK_0,  */
	0x00000482, 0x00010002, 0x00000615,	/* ((r1&~r2)!=0), r1=psem.INT_STS_1, r2=psem.INT_MASK_1,  */
	0x00000483, 0x00010002, 0x00000617,	/* ((r1&~r2)!=0), r1=psem.PRTY_STS, r2=psem.PRTY_MASK,  */
	0x00000484, 0x00010002, 0x00000619,	/* ((r1&~r2)!=0), r1=psem.PRTY_STS_H_0, r2=psem.PRTY_MASK_H_0,  */
	0x01020485, 0x00010001, 0x0000061b,	/* (r1!=0), r1=psem.pf_err_vector,  */
	0x01010486, 0x00010001, 0x0052061c,	/* (r1!=0x17), r1=psem.foc_credit,  */
	0x01010487, 0x00010001, 0x0048061d,	/* (r1!=64), r1=psem.foc_credit[1],  */
	0x01010488, 0x00010001, 0x001f061e,	/* (r1!=reset1), r1=psem.num_of_threads,  */
	0x01000489, 0x00010001, 0x0000061f,	/* (r1!=0), r1=psem.thread_error,  */
	0x0100048a, 0x00010001, 0x001b0620,	/* (r1!=reset1), r1=psem.dra_empty,  */
	0x0101048b, 0x00010101, 0x001b0621,	/* (r1!=1), r1=psem.ext_pas_empty,  */
	0x0101048c, 0x00010101, 0x001b0623,	/* (r1!=1), r1=psem.fic_empty,  */
	0x0100048d, 0x00010101, 0x001b0625,	/* (r1!=reset1), r1=psem.slow_dra_fin_empty,  */
	0x0100048e, 0x00010101, 0x001b0627,	/* (r1!=reset1), r1=psem.slow_dra_rd_empty,  */
	0x0100048f, 0x00010101, 0x001b0629,	/* (r1!=reset1), r1=psem.slow_dra_wr_empty,  */
	0x01010490, 0x00010101, 0x001b062b,	/* (r1!=1), r1=psem.slow_ext_store_empty,  */
	0x01010491, 0x00010101, 0x001b062d,	/* (r1!=1), r1=psem.slow_ext_load_empty,  */
	0x01010492, 0x00010001, 0x001b062f,	/* (r1!=1), r1=psem.sync_dbg_empty,  */
	0x01010493, 0x00010001, 0x00000630,	/* (r1!=0), r1=psem.ext_store_if_full,  */
	0x01010494, 0x00010001, 0x00000631,	/* (r1!=0), r1=psem.ram_if_full,  */
	0x00000495, 0x00010002, 0x00000632,	/* ((r1&~r2)!=0), r1=psem.fast_memory.INT_STS, r2=psem.fast_memory.INT_MASK,  */
	0x06000496, 0x00020001, 0x00530634,	/* ((r1&0xf)!=0), r1=psem.fast_memory.stalled,  */
	0x01000497, 0x00010001, 0x00000635,	/* (r1!=0), r1=psem.fast_memory.stalled,  */
	0x00000498, 0x00010002, 0x00000636,	/* ((r1&~r2)!=0), r1=tsem.INT_STS_0, r2=tsem.INT_MASK_0,  */
	0x00000499, 0x00010002, 0x00000638,	/* ((r1&~r2)!=0), r1=tsem.INT_STS_1, r2=tsem.INT_MASK_1,  */
	0x0000049a, 0x00010002, 0x0000063a,	/* ((r1&~r2)!=0), r1=tsem.PRTY_STS, r2=tsem.PRTY_MASK,  */
	0x0000049b, 0x00010002, 0x0000063c,	/* ((r1&~r2)!=0), r1=tsem.PRTY_STS_H_0, r2=tsem.PRTY_MASK_H_0,  */
	0x0102049c, 0x00010001, 0x0000063e,	/* (r1!=0), r1=tsem.pf_err_vector,  */
	0x0101049d, 0x00010001, 0x0055063f,	/* (r1!=44), r1=tsem.foc_credit,  */
	0x0101049e, 0x00010001, 0x00560640,	/* (r1!=60), r1=tsem.foc_credit[1],  */
	0x0101049f, 0x00010001, 0x004e0641,	/* (r1!=reset1), r1=tsem.num_of_threads,  */
	0x010004a0, 0x00010001, 0x00000642,	/* (r1!=0), r1=tsem.thread_error,  */
	0x010004a1, 0x00010001, 0x001b0643,	/* (r1!=reset1), r1=tsem.dra_empty,  */
	0x010104a2, 0x00010101, 0x001b0644,	/* (r1!=1), r1=tsem.ext_pas_empty,  */
	0x010104a3, 0x00010101, 0x001b0646,	/* (r1!=1), r1=tsem.fic_empty,  */
	0x010004a4, 0x00010101, 0x001b0648,	/* (r1!=reset1), r1=tsem.slow_dra_fin_empty,  */
	0x010004a5, 0x00010101, 0x001b064a,	/* (r1!=reset1), r1=tsem.slow_dra_rd_empty,  */
	0x010004a6, 0x00010101, 0x001b064c,	/* (r1!=reset1), r1=tsem.slow_dra_wr_empty,  */
	0x010104a7, 0x00010101, 0x001b064e,	/* (r1!=1), r1=tsem.slow_ext_store_empty,  */
	0x010104a8, 0x00010101, 0x001b0650,	/* (r1!=1), r1=tsem.slow_ext_load_empty,  */
	0x010104a9, 0x00010001, 0x001b0652,	/* (r1!=1), r1=tsem.sync_dbg_empty,  */
	0x010104aa, 0x00010001, 0x00000653,	/* (r1!=0), r1=tsem.ext_store_if_full,  */
	0x010104ab, 0x00010001, 0x00000654,	/* (r1!=0), r1=tsem.ram_if_full,  */
	0x000004ac, 0x00010002, 0x00000655,	/* ((r1&~r2)!=0), r1=tsem.fast_memory.INT_STS, r2=tsem.fast_memory.INT_MASK,  */
	0x010004ad, 0x00010001, 0x00000657,	/* (r1!=0), r1=tsem.fast_memory.stalled,  */
	0x000004ae, 0x00010002, 0x00000658,	/* ((r1&~r2)!=0), r1=msem.INT_STS_0, r2=msem.INT_MASK_0,  */
	0x000004af, 0x00010002, 0x0000065a,	/* ((r1&~r2)!=0), r1=msem.INT_STS_1, r2=msem.INT_MASK_1,  */
	0x000004b0, 0x00010002, 0x0000065c,	/* ((r1&~r2)!=0), r1=msem.PRTY_STS, r2=msem.PRTY_MASK,  */
	0x000004b1, 0x00010002, 0x0000065e,	/* ((r1&~r2)!=0), r1=msem.PRTY_STS_H_0, r2=msem.PRTY_MASK_H_0,  */
	0x010204b2, 0x00010001, 0x00000660,	/* (r1!=0), r1=msem.pf_err_vector,  */
	0x010104b3, 0x00010001, 0x001c0661,	/* (r1!=32), r1=msem.foc_credit[5],  */
	0x010104b4, 0x00010001, 0x00410662,	/* (r1!=7), r1=msem.foc_credit[4],  */
	0x010104b5, 0x00010001, 0x00520663,	/* (r1!=23), r1=msem.foc_credit[2],  */
	0x010104b6, 0x00010001, 0x00550664,	/* (r1!=44), r1=msem.foc_credit[1],  */
	0x010104b7, 0x00010001, 0x00240665,	/* (r1!=17), r1=msem.foc_credit[3],  */
	0x010104b8, 0x00010001, 0x00170666,	/* (r1!=5), r1=msem.foc_credit,  */
	0x010104b9, 0x00010001, 0x004e0667,	/* (r1!=reset1), r1=msem.num_of_threads,  */
	0x010004ba, 0x00010001, 0x00000668,	/* (r1!=0), r1=msem.thread_error,  */
	0x010004bb, 0x00010001, 0x001b0669,	/* (r1!=reset1), r1=msem.dra_empty,  */
	0x010104bc, 0x00010101, 0x001b066a,	/* (r1!=1), r1=msem.ext_pas_empty,  */
	0x010104bd, 0x00010101, 0x001b066c,	/* (r1!=1), r1=msem.fic_empty,  */
	0x010004be, 0x00010101, 0x001b066e,	/* (r1!=reset1), r1=msem.slow_dra_fin_empty,  */
	0x010004bf, 0x00010101, 0x001b0670,	/* (r1!=reset1), r1=msem.slow_dra_rd_empty,  */
	0x010004c0, 0x00010101, 0x001b0672,	/* (r1!=reset1), r1=msem.slow_dra_wr_empty,  */
	0x010104c1, 0x00010101, 0x001b0674,	/* (r1!=1), r1=msem.slow_ext_store_empty,  */
	0x010104c2, 0x00010101, 0x001b0676,	/* (r1!=1), r1=msem.slow_ext_load_empty,  */
	0x010104c3, 0x00010001, 0x001b0678,	/* (r1!=1), r1=msem.sync_dbg_empty,  */
	0x010104c4, 0x00010001, 0x00000679,	/* (r1!=0), r1=msem.ext_store_if_full,  */
	0x010104c5, 0x00010001, 0x0000067a,	/* (r1!=0), r1=msem.ram_if_full,  */
	0x000004c6, 0x00010002, 0x0000067b,	/* ((r1&~r2)!=0), r1=msem.fast_memory.INT_STS, r2=msem.fast_memory.INT_MASK,  */
	0x010004c7, 0x00010001, 0x0000067d,	/* (r1!=0), r1=msem.fast_memory.stalled,  */
	0x000004c8, 0x00010002, 0x0000067e,	/* ((r1&~r2)!=0), r1=usem.INT_STS_0, r2=usem.INT_MASK_0,  */
	0x000004c9, 0x00010002, 0x00000680,	/* ((r1&~r2)!=0), r1=usem.INT_STS_1, r2=usem.INT_MASK_1,  */
	0x000004ca, 0x00010002, 0x00000682,	/* ((r1&~r2)!=0), r1=usem.PRTY_STS, r2=usem.PRTY_MASK,  */
	0x000004cb, 0x00010002, 0x00000684,	/* ((r1&~r2)!=0), r1=usem.PRTY_STS_H_0, r2=usem.PRTY_MASK_H_0,  */
	0x010204cc, 0x00010001, 0x00000686,	/* (r1!=0), r1=usem.pf_err_vector,  */
	0x010104cd, 0x00010001, 0x00570687,	/* (r1!=50), r1=usem.foc_credit[1],  */
	0x010104ce, 0x00010001, 0x00500688,	/* (r1!=10), r1=usem.foc_credit[2],  */
	0x010104cf, 0x00010001, 0x00210689,	/* (r1!=3), r1=usem.foc_credit[3],  */
	0x010104d0, 0x00010001, 0x001c068a,	/* (r1!=32), r1=usem.foc_credit[4],  */
	0x010104d1, 0x00010001, 0x0017068b,	/* (r1!=5), r1=usem.foc_credit,  */
	0x010104d2, 0x00010001, 0x0018068c,	/* (r1!=reset1), r1=usem.num_of_threads,  */
	0x010004d3, 0x00010001, 0x0000068d,	/* (r1!=0), r1=usem.thread_error,  */
	0x010004d4, 0x00010001, 0x001b068e,	/* (r1!=reset1), r1=usem.dra_empty,  */
	0x010104d5, 0x00010101, 0x001b068f,	/* (r1!=1), r1=usem.ext_pas_empty,  */
	0x010104d6, 0x00010101, 0x001b0691,	/* (r1!=1), r1=usem.fic_empty,  */
	0x010004d7, 0x00010101, 0x001b0693,	/* (r1!=reset1), r1=usem.slow_dra_fin_empty,  */
	0x010004d8, 0x00010101, 0x001b0695,	/* (r1!=reset1), r1=usem.slow_dra_rd_empty,  */
	0x010004d9, 0x00010101, 0x001b0697,	/* (r1!=reset1), r1=usem.slow_dra_wr_empty,  */
	0x010104da, 0x00010101, 0x001b0699,	/* (r1!=1), r1=usem.slow_ext_store_empty,  */
	0x010104db, 0x00010101, 0x001b069b,	/* (r1!=1), r1=usem.slow_ext_load_empty,  */
	0x010104dc, 0x00010001, 0x001b069d,	/* (r1!=1), r1=usem.sync_dbg_empty,  */
	0x010104dd, 0x00010001, 0x0000069e,	/* (r1!=0), r1=usem.ext_store_if_full,  */
	0x010104de, 0x00010001, 0x0000069f,	/* (r1!=0), r1=usem.ram_if_full,  */
	0x000004df, 0x00010002, 0x000006a0,	/* ((r1&~r2)!=0), r1=usem.fast_memory.INT_STS, r2=usem.fast_memory.INT_MASK,  */
	0x010004e0, 0x00010001, 0x000006a2,	/* (r1!=0), r1=usem.fast_memory.stalled,  */
	0x00810005,		/* mode bb */
	0x040004e1, 0x00020001, 0x005806a3,	/* ((r1&~0x3f02)!=0), r1=pcie.PRTY_STS_H_0,  */
	0x010204e2, 0x00010001, 0x000006a4,	/* (r1!=0), r1=igu.pending_bits_status[0:8],  */
	0x010104e3, 0x00010001, 0x000006a5,	/* (r1!=0), r1=igu.write_done_pending[0:8],  */
	0x000004e4, 0x00010102, 0x000006a6,	/* ((r1&~r2)!=0), r1=cnig.INT_STS, r2=cnig.INT_MASK,  */
	0x000004e5, 0x00010002, 0x000006a9,	/* ((r1&~r2)!=0), r1=cnig.PRTY_STS, r2=cnig.PRTY_MASK,  */
	0x010204e6, 0x00010201, 0x000006ab,	/* (r1!=0), r1=pswhst.vf_disabled_error_valid,  */
	0x010204e7, 0x00010001, 0x002106ae,	/* (r1!=0x03), r1=pglue_b.init_done_inb_int_mem,  */
	0x010204e8, 0x00010001, 0x002106af,	/* (r1!=0x03), r1=pglue_b.init_done_zone_a,  */
	0x060204e9, 0x00020001, 0x005a06b0,	/* ((r1&0x76417C)!=0), r1=pglue_b.INT_STS,  */
	0x060004ea, 0x00020001, 0x005c06b1,	/* ((r1&0x01BC01)!=0), r1=pglue_b.INT_STS,  */
	0x010204eb, 0x00010001, 0x000006b2,	/* (r1!=0), r1=pglue_b.flr_request_vf_223_192,  */
	0x010204ec, 0x00010001, 0x000006b3,	/* (r1!=0), r1=pglue_b.flr_request_vf_255_224,  */
	0x010204ed, 0x00010001, 0x000006b4,	/* (r1!=0), r1=pglue_b.was_error_vf_223_192,  */
	0x010204ee, 0x00010001, 0x000006b5,	/* (r1!=0), r1=pglue_b.was_error_vf_255_224,  */
	0x010204ef, 0x00010001, 0x002806b6,	/* (r1!=0xffffffff), r1=pglue_b.tags_159_128,  */
	0x010204f0, 0x00010001, 0x002806b7,	/* (r1!=0xffffffff), r1=pglue_b.tags_191_160,  */
	0x010204f1, 0x00010001, 0x002806b8,	/* (r1!=0xffffffff), r1=pglue_b.tags_223_192,  */
	0x010204f2, 0x00010001, 0x002806b9,	/* (r1!=0xffffffff), r1=pglue_b.tags_255_224,  */
	0x0a0104f3, 0x00030001, 0x005e06ba,	/* ((r1&0x30)>>4)!=(r1&0x03), r1=qm.PtrTblOther[0:63] width=2 access=WB,  */
	0x090104f4, 0x00060001, 0x006106bb,	/* ((r1[0]&0x3FFFFFC0)>>6)!=(((r1[0]&0xC0000000)>>30)|((r1[1]&0x3FFFFF)<<2)), r1=qm.PtrTblOther[0:63] width=2 access=WB,  */
	0x010004f5, 0x00010001, 0x000006bc,	/* (r1!=0), r1=qm.QstatusTx_0[0:13],  */
	0x010004f6, 0x00010001, 0x000006bd,	/* (r1!=0), r1=qm.QstatusOther_0[0:1],  */
	0x0b0104f7, 0x00010003, 0x000006be,	/* (r1!=r2&&r3==0), r1=qm.VoqCrdLine[0:7], r2=qm.VoqInitCrdLine[0:7], r3=misc.port_mode,  */
	0x0b0104f8, 0x00010003, 0x000006c1,	/* (r1!=r2&&r3==0), r1=qm.VoqCrdLine[16], r2=qm.VoqInitCrdLine[16], r3=misc.port_mode,  */
	0x0c0104f9, 0x00010003, 0x000006c4,	/* (r1!=r2&&r3>0), r1=qm.VoqCrdLine[0:17], r2=qm.VoqInitCrdLine[0:17], r3=misc.port_mode,  */
	0x0b0104fa, 0x00010003, 0x000006c7,	/* (r1!=r2&&r3==0), r1=qm.VoqCrdByte[0:7], r2=qm.VoqInitCrdByte[0:7], r3=misc.port_mode,  */
	0x0b0104fb, 0x00010003, 0x000006ca,	/* (r1!=r2&&r3==0), r1=qm.VoqCrdByte[16], r2=qm.VoqInitCrdByte[16], r3=misc.port_mode,  */
	0x0c0104fc, 0x00010003, 0x000006cd,	/* (r1!=r2&&r3>0), r1=qm.VoqCrdByte[0:17], r2=qm.VoqInitCrdByte[0:17], r3=misc.port_mode,  */
	0x0a0104fd, 0x00030001, 0x005e06d0,	/* ((r1&0x30)>>4)!=(r1&0x03), r1=qm.PtrTblTx[0:447] width=2 access=WB,  */
	0x090104fe, 0x00060001, 0x006106d1,	/* ((r1[0]&0x3FFFFFC0)>>6)!=(((r1[0]&0xC0000000)>>30)|((r1[1]&0x3FFFFF)<<2)), r1=qm.PtrTblTx[0:447] width=2 access=WB,  */
	0x010104ff, 0x00010001, 0x004506d2,	/* (r1!=8190), r1=brb.wc_empty_0[0:3],  */
	0x01010500, 0x00010001, 0x002106d3,	/* (r1!=reset1), r1=brb.rc_eop_empty,  */
	0x01010501, 0x00010001, 0x001906d4,	/* (r1!=2), r1=brb.wc_status_0[0:3] width=3 access=WB,  */
	0x00000502, 0x00010302, 0x000006d5,	/* ((r1&~r2)!=0), r1=nig.PRTY_STS, r2=nig.PRTY_MASK,  */
	0x00000503, 0x00010202, 0x000006da,	/* ((r1&~r2)!=0), r1=btb.INT_STS_0, r2=btb.INT_MASK_0,  */
	0x00000504, 0x00010202, 0x000006de,	/* ((r1&~r2)!=0), r1=btb.INT_STS_4, r2=btb.INT_MASK_4,  */
	0x01010505, 0x00010001, 0x000006e2,	/* (r1!=0), r1=xcm.qm_act_st_cnt[0:447],  */
	0x01020506, 0x00010001, 0x000006e3,	/* (r1!=0), r1=xsem.vf_err_vector width=4 access=WB,  */
	0x01020507, 0x00010001, 0x000006e4,	/* (r1!=0), r1=ysem.vf_err_vector width=4 access=WB,  */
	0x01020508, 0x00010001, 0x000006e5,	/* (r1!=0), r1=psem.vf_err_vector width=4 access=WB,  */
	0x01020509, 0x00010001, 0x000006e6,	/* (r1!=0), r1=tsem.vf_err_vector width=4 access=WB,  */
	0x0102050a, 0x00010001, 0x000006e7,	/* (r1!=0), r1=msem.vf_err_vector width=4 access=WB,  */
	0x0102050b, 0x00010001, 0x000006e8,	/* (r1!=0), r1=usem.vf_err_vector width=4 access=WB,  */
	0x00180041,		/* mode k2 */
	0x0000050c, 0x00010002, 0x000006e9,	/* ((r1&~r2)!=0), r1=pcie.PRTY_STS_H_0, r2=pcie.PRTY_MASK_H_0,  */
	0x0102050d, 0x00010201, 0x000006eb,	/* (r1!=0), r1=pswhst.vf_disabled_error_valid,  */
	0x0102050e, 0x00010001, 0x001b06ee,	/* (r1!=0x01), r1=pglue_b.init_done_inb_int_mem,  */
	0x0102050f, 0x00010001, 0x001b06ef,	/* (r1!=0x01), r1=pglue_b.init_done_zone_a,  */
	0x06020510, 0x00020001, 0x006706f0,	/* ((r1&0x70417C)!=0), r1=pglue_b.INT_STS,  */
	0x06000511, 0x00020001, 0x005c06f1,	/* ((r1&0x01BC01)!=0), r1=pglue_b.INT_STS,  */
	0x01010512, 0x00010001, 0x004506f2,	/* (r1!=8190), r1=brb.wc_empty_0[0:7],  */
	0x01010513, 0x00010001, 0x001906f3,	/* (r1!=2), r1=brb.wc_status_0[0:7] width=3 access=WB,  */
	0x003f002d,		/* mode asic */
	0x03010514, 0x00000002, 0x000006f4,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_7, r2=pswrq2.max_srs_vq7,  */
	0x03010515, 0x00000002, 0x000006f6,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_8, r2=pswrq2.max_srs_vq8,  */
	0x03010516, 0x00000002, 0x000006f8,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_10, r2=pswrq2.max_srs_vq10,  */
	0x03010517, 0x00000002, 0x000006fa,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_13, r2=pswrq2.max_srs_vq13,  */
	0x03010518, 0x00000002, 0x000006fc,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_14, r2=pswrq2.max_srs_vq14,  */
	0x03010519, 0x00000002, 0x000006fe,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_20, r2=pswrq2.max_srs_vq20,  */
	0x0301051a, 0x00000002, 0x00000700,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_25, r2=pswrq2.max_srs_vq25,  */
	0x0301051b, 0x00000002, 0x00000702,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_27, r2=pswrq2.max_srs_vq27,  */
	0x0301051c, 0x00000002, 0x00000704,	/* (r1!=r2), r1=pswrq2.sr_cnt_per_vq_31, r2=pswrq2.max_srs_vq31,  */
	0x0301051d, 0x00000002, 0x00000706,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_7, r2=pswrq2.max_blks_vq7,  */
	0x0301051e, 0x00000002, 0x00000708,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_8, r2=pswrq2.max_blks_vq8,  */
	0x0301051f, 0x00000002, 0x0000070a,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_10, r2=pswrq2.max_blks_vq10,  */
	0x03010520, 0x00000002, 0x0000070c,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_13, r2=pswrq2.max_blks_vq13,  */
	0x03010521, 0x00000002, 0x0000070e,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_14, r2=pswrq2.max_blks_vq14,  */
	0x03010522, 0x00000002, 0x00000710,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_20, r2=pswrq2.max_blks_vq20,  */
	0x03010523, 0x00000002, 0x00000712,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_25, r2=pswrq2.max_blks_vq25,  */
	0x03010524, 0x00000002, 0x00000714,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_27, r2=pswrq2.max_blks_vq27,  */
	0x03010525, 0x00000002, 0x00000716,	/* (r1!=r2), r1=pswrq2.blk_cnt_per_vq_31, r2=pswrq2.max_blks_vq31,  */
	0x0d020526, 0x00010001, 0x00690718,	/* (r1&0x3E1), r1=tm.INT_STS_1,  */
	0x00000527, 0x00010002, 0x00000719,	/* ((r1&~r2)!=0), r1=bmb.PRTY_STS_H_0, r2=bmb.PRTY_MASK_H_0,  */
	0x00000528, 0x00010002, 0x0000071b,	/* ((r1&~r2)!=0), r1=bmb.PRTY_STS_H_1, r2=bmb.PRTY_MASK_H_1,  */
	0x00030051,		/* mode !asic */
	0x0d020529, 0x00010001, 0x006a071d,	/* (r1&0x381), r1=tm.INT_STS_1,  */
	0x00c30023,		/* mode !bb */
	0x0600052a, 0x00020001, 0x006b071e,	/* ((r1&0x0FD010)!=0), r1=pglcs.pgl_cs.UNCORR_ERR_STATUS_OFF,  */
	0x0602052b, 0x00020001, 0x0009071f,	/* ((r1&0x100000)!=0), r1=pglcs.pgl_cs.UNCORR_ERR_STATUS_OFF,  */
	0x0602052c, 0x00020001, 0x006d0720,	/* ((r1&0x2000)!=0), r1=pglcs.pgl_cs.UNCORR_ERR_STATUS_OFF,  */
	0x0102052d, 0x00010001, 0x00000721,	/* (r1!=0), r1=pglcs.pgl_cs.HDR_LOG_0_OFF[0:3],  */
	0x0100052e, 0x00010001, 0x00000722,	/* (r1!=0), r1=pglcs.syncfifo_pop_underflow,  */
	0x0100052f, 0x00010001, 0x00000723,	/* (r1!=0), r1=pglcs.syncfifo_push_overflow,  */
	0x01010530, 0x00010001, 0x00000724,	/* (r1!=0), r1=pglcs.tx_syncfifo_pop_status,  */
	0x06000531, 0x00020001, 0x006f0725,	/* ((r1&0xE001)!=0), r1=pcie.INT_STS,  */
	0x01020532, 0x00010001, 0x00000726,	/* (r1!=0), r1=igu.pending_bits_status[0:11],  */
	0x01010533, 0x00010001, 0x00000727,	/* (r1!=0), r1=igu.write_done_pending[0:11],  */
	0x00000534, 0x00010102, 0x00000728,	/* ((r1&~r2)!=0), r1=cnig.INT_STS, r2=cnig.INT_MASK,  */
	0x00000535, 0x00010002, 0x0000072b,	/* ((r1&~r2)!=0), r1=cnig.PRTY_STS, r2=cnig.PRTY_MASK,  */
	0x01000536, 0x00010001, 0x0000072d,	/* (r1!=0), r1=pglue_b.syncfifo_push_overflow,  */
	0x01000537, 0x00010001, 0x0000072e,	/* (r1!=0), r1=pglue_b.syncfifo_pop_underflow,  */
	0x01010538, 0x00010001, 0x0000072f,	/* (r1!=0), r1=pglue_b.rxh_syncfifo_pop_status,  */
	0x01010539, 0x00010101, 0x00000730,	/* (r1!=0), r1=pglue_b.rxd_syncfifo_pop_status,  */
	0x0a01053a, 0x00030001, 0x005e0732,	/* ((r1&0x30)>>4)!=(r1&0x03), r1=qm.PtrTblOther[0:127] width=2 access=WB,  */
	0x0901053b, 0x00060001, 0x00610733,	/* ((r1[0]&0x3FFFFFC0)>>6)!=(((r1[0]&0xC0000000)>>30)|((r1[1]&0x3FFFFF)<<2)), r1=qm.PtrTblOther[0:127] width=2 access=WB,  */
	0x0100053c, 0x00010001, 0x00000734,	/* (r1!=0), r1=qm.QstatusTx_0[0:15],  */
	0x0100053d, 0x00010001, 0x00000735,	/* (r1!=0), r1=qm.QstatusOther_0[0:3],  */
	0x0c01053e, 0x00010003, 0x00000736,	/* (r1!=r2&&r3>0), r1=qm.VoqCrdLine[0:19], r2=qm.VoqInitCrdLine[0:19], r3=misc.port_mode,  */
	0x0b01053f, 0x00010003, 0x00000739,	/* (r1!=r2&&r3==0), r1=qm.VoqCrdLine[16], r2=qm.VoqInitCrdLine[16], r3=misc.port_mode,  */
	0x0b010540, 0x00010003, 0x0000073c,	/* (r1!=r2&&r3==0), r1=qm.VoqCrdLine[0:7], r2=qm.VoqInitCrdLine[0:7], r3=misc.port_mode,  */
	0x0c010541, 0x00010003, 0x0000073f,	/* (r1!=r2&&r3>0), r1=qm.VoqCrdByte[0:19], r2=qm.VoqInitCrdByte[0:19], r3=misc.port_mode,  */
	0x0b010542, 0x00010003, 0x00000742,	/* (r1!=r2&&r3==0), r1=qm.VoqCrdByte[0:7], r2=qm.VoqInitCrdByte[0:7], r3=misc.port_mode,  */
	0x0b010543, 0x00010003, 0x00000745,	/* (r1!=r2&&r3==0), r1=qm.VoqCrdByte[16], r2=qm.VoqInitCrdByte[16], r3=misc.port_mode,  */
	0x09010544, 0x00060001, 0x00610748,	/* ((r1[0]&0x3FFFFFC0)>>6)!=(((r1[0]&0xC0000000)>>30)|((r1[1]&0x3FFFFF)<<2)), r1=qm.PtrTblTx[0:511] width=2 access=WB,  */
	0x0a010545, 0x00030001, 0x005e0749,	/* ((r1&0x30)>>4)!=(r1&0x03), r1=qm.PtrTblTx[0:511] width=2 access=WB,  */
	0x01010546, 0x00010001, 0x0027074a,	/* (r1!=reset1), r1=brb.rc_eop_empty,  */
	0x01010547, 0x00010001, 0x0000074b,	/* (r1!=0), r1=brb.mac2_tc_occupancy_0,  */
	0x01010548, 0x00010001, 0x0000074c,	/* (r1!=0), r1=brb.mac2_tc_occupancy_1,  */
	0x01010549, 0x00010001, 0x0000074d,	/* (r1!=0), r1=brb.mac2_tc_occupancy_2,  */
	0x0101054a, 0x00010001, 0x0000074e,	/* (r1!=0), r1=brb.mac2_tc_occupancy_3,  */
	0x0101054b, 0x00010001, 0x0000074f,	/* (r1!=0), r1=brb.mac2_tc_occupancy_4,  */
	0x0101054c, 0x00010001, 0x00000750,	/* (r1!=0), r1=brb.mac3_tc_occupancy_0,  */
	0x0101054d, 0x00010001, 0x00000751,	/* (r1!=0), r1=brb.mac3_tc_occupancy_1,  */
	0x0101054e, 0x00010001, 0x00000752,	/* (r1!=0), r1=brb.mac3_tc_occupancy_2,  */
	0x0101054f, 0x00010001, 0x00000753,	/* (r1!=0), r1=brb.mac3_tc_occupancy_3,  */
	0x01010550, 0x00010001, 0x00000754,	/* (r1!=0), r1=brb.mac3_tc_occupancy_4,  */
	0x00000551, 0x00010002, 0x00000755,	/* ((r1&~r2)!=0), r1=nig.INT_STS_6, r2=nig.INT_MASK_6,  */
	0x00020552, 0x00010202, 0x00000757,	/* ((r1&~r2)!=0), r1=nig.INT_STS_7, r2=nig.INT_MASK_7,  */
	0x00000553, 0x00010002, 0x0000075b,	/* ((r1&~r2)!=0), r1=nig.INT_STS_8, r2=nig.INT_MASK_8,  */
	0x00020554, 0x00010202, 0x0000075d,	/* ((r1&~r2)!=0), r1=nig.INT_STS_9, r2=nig.INT_MASK_9,  */
	0x00000555, 0x00010302, 0x00000761,	/* ((r1&~r2)!=0), r1=nig.PRTY_STS, r2=nig.PRTY_MASK,  */
	0x00000556, 0x00010202, 0x00000766,	/* ((r1&~r2)!=0), r1=btb.INT_STS_0, r2=btb.INT_MASK_0,  */
	0x00000557, 0x00010202, 0x0000076a,	/* ((r1&~r2)!=0), r1=btb.INT_STS_4, r2=btb.INT_MASK_4,  */
	0x01010558, 0x00010001, 0x003c076e,	/* (r1!=255), r1=btb.rc_pkt_empty_4,  */
	0x01010559, 0x00010001, 0x003c076f,	/* (r1!=255), r1=btb.rc_pkt_empty_5,  */
	0x0101055a, 0x00010001, 0x003c0770,	/* (r1!=255), r1=btb.rc_pkt_empty_6,  */
	0x0101055b, 0x00010001, 0x003c0771,	/* (r1!=255), r1=btb.rc_pkt_empty_7,  */
	0x0100055c, 0x00010101, 0x001b0772,	/* (r1!=0x01), r1=msdm.rmt_xcm_fifo_empty,  */
	0x0100055d, 0x00010101, 0x001b0774,	/* (r1!=0x01), r1=msdm.rmt_ycm_fifo_empty,  */
	0x0101055e, 0x00010001, 0x00000776,	/* (r1!=0), r1=xcm.qm_act_st_cnt[0:511],  */
	0x0102055f, 0x00010001, 0x00000777,	/* (r1!=0), r1=xsem.vf_err_vector width=6 access=WB,  */
	0x00000560, 0x00010002, 0x00000778,	/* ((r1&~r2)!=0), r1=xsem.fast_memory.PRTY_STS_H_0, r2=xsem.fast_memory.PRTY_MASK_H_0,  */
	0x01020561, 0x00010001, 0x0000077a,	/* (r1!=0), r1=ysem.vf_err_vector width=6 access=WB,  */
	0x00000562, 0x00010002, 0x0000077b,	/* ((r1&~r2)!=0), r1=ysem.fast_memory.PRTY_STS_H_0, r2=ysem.fast_memory.PRTY_MASK_H_0,  */
	0x01020563, 0x00010001, 0x0000077d,	/* (r1!=0), r1=psem.vf_err_vector width=6 access=WB,  */
	0x00000564, 0x00010002, 0x0000077e,	/* ((r1&~r2)!=0), r1=psem.fast_memory.PRTY_STS_H_0, r2=psem.fast_memory.PRTY_MASK_H_0,  */
	0x01020565, 0x00010001, 0x00000780,	/* (r1!=0), r1=tsem.vf_err_vector width=6 access=WB,  */
	0x00000566, 0x00010002, 0x00000781,	/* ((r1&~r2)!=0), r1=tsem.fast_memory.PRTY_STS_H_0, r2=tsem.fast_memory.PRTY_MASK_H_0,  */
	0x01020567, 0x00010001, 0x00000783,	/* (r1!=0), r1=msem.vf_err_vector width=6 access=WB,  */
	0x00000568, 0x00010002, 0x00000784,	/* ((r1&~r2)!=0), r1=msem.fast_memory.PRTY_STS_H_0, r2=msem.fast_memory.PRTY_MASK_H_0,  */
	0x01020569, 0x00010001, 0x00000786,	/* (r1!=0), r1=usem.vf_err_vector width=6 access=WB,  */
	0x0000056a, 0x00010002, 0x00000787,	/* ((r1&~r2)!=0), r1=usem.fast_memory.PRTY_STS_H_0, r2=usem.fast_memory.PRTY_MASK_H_0,  */
	0x00270081,		/* mode !k2 */
	0x0602056b, 0x00020001, 0x00090789,	/* ((r1&0x100000)!=0), r1=pglcs.pgl_cs.uc_err_status,  */
	0x0600056c, 0x00020001, 0x006b078a,	/* ((r1&0x0FD010)!=0), r1=pglcs.pgl_cs.uc_err_status,  */
	0x0602056d, 0x00020001, 0x006d078b,	/* ((r1&0x2000)!=0), r1=pglcs.pgl_cs.uc_err_status,  */
	0x0102056e, 0x00010001, 0x0000078c,	/* (r1!=0), r1=pglcs.pgl_cs.header_log1[0:3],  */
	0x0602056f, 0x00020001, 0x0011078d,	/* ((r1&0x2000000)!=0), r1=pglcs.pgl_cs.tl_ctrlstat_5,  */
	0x04000570, 0x00020001, 0x0071078e,	/* ((r1&~0x2040902)!=0), r1=pglcs.pgl_cs.tl_ctrlstat_5,  */
	0x04000571, 0x00020001, 0x0073078f,	/* ((r1&~0x10240902)!=0), r1=pglcs.pgl_cs.tl_func345_stat,  */
	0x04000572, 0x00020001, 0x00730790,	/* ((r1&~0x10240902)!=0), r1=pglcs.pgl_cs.tl_func678_stat,  */
	0x06000573, 0x00020001, 0x00190791,	/* ((r1&0x2)!=0), r1=pglcs.pgl_cs.DLATTN_VEC,  */
	0x00000574, 0x00010002, 0x00000792,	/* ((r1&~r2)!=0), r1=miscs.INT_STS_1, r2=miscs.INT_MASK_1,  */
	0x00000575, 0x00010002, 0x00000794,	/* ((r1&~r2)!=0), r1=opte.PRTY_STS_H_0, r2=opte.PRTY_MASK_H_0,  */
	0x00000576, 0x00010002, 0x00000796,	/* ((r1&~r2)!=0), r1=igu.PRTY_STS_H_1, r2=igu.PRTY_MASK_H_1,  */
	0x01010577, 0x00010001, 0x001b0798,	/* (r1!=1), r1=nig.rx_macfifo_empty,  */
	0x000300d9,		/* mode !(!asic) */
	0x00000578, 0x00010002, 0x00000799,	/* ((r1&~r2)!=0), r1=ncsi.PRTY_STS_H_0, r2=ncsi.PRTY_MASK_H_0,  */
	0x007200d3,		/* mode (!bb)&(!(!asic)) */
	0x00000579, 0x00010002, 0x0000079b,	/* ((r1&~r2)!=0), r1=ipc.INT_STS_0, r2=ipc.INT_MASK_0,  */
	0x0600057a, 0x00020001, 0x0019079d,	/* ((r1&0x00000002)!=0), r1=umac.INT_STS,  */
	0x0600057b, 0x00020001, 0x0025079e,	/* ((r1&0x00000001)!=0), r1=umac.INT_STS,  */
	0x0600057c, 0x00020001, 0x0075079f,	/* ((r1&0x00000004)!=4), r1=nws.common_status,  */
	0x0600057d, 0x00020001, 0x007707a0,	/* ((r1&0x00000002)!=2), r1=nws.common_status,  */
	0x0600057e, 0x00020001, 0x002507a1,	/* ((r1&0x00000001)!=0), r1=nws.common_status,  */
	0x0600057f, 0x00020001, 0x002507a2,	/* ((r1&0x00000001)!=0), r1=nws.INT_STS_0,  */
	0x00000580, 0x00010002, 0x000007a3,	/* ((r1&~r2)!=0), r1=nws.PRTY_STS_H_0, r2=nws.PRTY_MASK_H_0,  */
	0x01000581, 0x00010001, 0x000007a5,	/* (r1!=0), r1=nws.nws_cmu.phy0_top_err_ctrl0,  */
	0x01000582, 0x00010001, 0x000007a6,	/* (r1!=0), r1=nws.nws_cmu.phy0_top_err_ctrl1,  */
	0x01000583, 0x00010001, 0x000007a7,	/* (r1!=0), r1=nws.nws_cmu.phy0_top_err_ctrl2,  */
	0x06000584, 0x00020001, 0x003007a8,	/* ((r1&0x00000004)!=0), r1=nws.nws_cmu.phy0_top_regbus_err_info_status0,  */
	0x06000585, 0x00020001, 0x001907a9,	/* ((r1&0x00000002)!=0), r1=nws.nws_cmu.phy0_top_regbus_err_info_status0,  */
	0x06000586, 0x00020001, 0x002507aa,	/* ((r1&0x00000001)!=0), r1=nws.nws_cmu.phy0_top_regbus_err_info_status0,  */
	0x01000587, 0x00010001, 0x000007ab,	/* (r1!=0), r1=nws.nws_cmu.phy0_top_regbus_err_info_status0,  */
	0x01000588, 0x00010001, 0x000007ac,	/* (r1!=0), r1=nws.nws_cmu.phy0_top_regbus_err_info_status1,  */
	0x01000589, 0x00010001, 0x000007ad,	/* (r1!=0), r1=nws.nws_cmu.phy0_top_regbus_err_info_status2,  */
	0x0100058a, 0x00010001, 0x000007ae,	/* (r1!=0), r1=nws.nws_cmu.phy0_top_regbus_err_info_status3,  */
	0x0100058b, 0x00010001, 0x001b07af,	/* (r1!=1), r1=nws.nws_cmu.cmu_lc0_top_phy_if_status,  */
	0x0100058c, 0x00010001, 0x000007b0,	/* (r1!=0), r1=nws.nws_cmu.cmu_lc0_top_err_ctrl1,  */
	0x0100058d, 0x00010001, 0x000007b1,	/* (r1!=0), r1=nws.nws_cmu.cmu_lc0_top_err_ctrl2,  */
	0x0100058e, 0x00010001, 0x000007b2,	/* (r1!=0), r1=nws.nws_cmu.cmu_lc0_top_err_ctrl3,  */
	0x0100058f, 0x00010001, 0x001b07b3,	/* (r1!=1), r1=nws.nws_cmu.cmu_r0_top_phy_if_status,  */
	0x01000590, 0x00010001, 0x000007b4,	/* (r1!=0), r1=nws.nws_cmu.cmu_r0_top_err_ctrl1,  */
	0x01000591, 0x00010001, 0x000007b5,	/* (r1!=0), r1=nws.nws_cmu.cmu_r0_top_err_ctrl2,  */
	0x01000592, 0x00010001, 0x000007b6,	/* (r1!=0), r1=nws.nws_cmu.cmu_r0_top_err_ctrl3,  */
	0x01000593, 0x00010001, 0x000007b7,	/* (r1!=0), r1=nws.nws_cmu.ln0_top_err_ctrl1,  */
	0x01000594, 0x00010001, 0x000007b8,	/* (r1!=0), r1=nws.nws_cmu.ln0_top_err_ctrl2,  */
	0x01000595, 0x00010001, 0x000007b9,	/* (r1!=0), r1=nws.nws_cmu.ln0_top_err_ctrl3,  */
	0x01000596, 0x00010001, 0x000007ba,	/* (r1!=0), r1=nws.nws_cmu.ln1_top_err_ctrl1,  */
	0x01000597, 0x00010001, 0x000007bb,	/* (r1!=0), r1=nws.nws_cmu.ln1_top_err_ctrl2,  */
	0x01000598, 0x00010001, 0x000007bc,	/* (r1!=0), r1=nws.nws_cmu.ln1_top_err_ctrl3,  */
	0x01000599, 0x00010001, 0x000007bd,	/* (r1!=0), r1=nws.nws_cmu.ln2_top_err_ctrl1,  */
	0x0100059a, 0x00010001, 0x000007be,	/* (r1!=0), r1=nws.nws_cmu.ln2_top_err_ctrl2,  */
	0x0100059b, 0x00010001, 0x000007bf,	/* (r1!=0), r1=nws.nws_cmu.ln2_top_err_ctrl3,  */
	0x0100059c, 0x00010001, 0x000007c0,	/* (r1!=0), r1=nws.nws_cmu.ln3_top_err_ctrl1,  */
	0x0100059d, 0x00010001, 0x000007c1,	/* (r1!=0), r1=nws.nws_cmu.ln3_top_err_ctrl2,  */
	0x0100059e, 0x00010001, 0x000007c2,	/* (r1!=0), r1=nws.nws_cmu.ln3_top_err_ctrl3,  */
	0x00600151,		/* mode !fpga */
	0x0000059f, 0x00010002, 0x000007c3,	/* ((r1&~r2)!=0), r1=src.INT_STS, r2=src.INT_MASK,  */
	0x0d0005a0, 0x00010001, 0x007907c5,	/* (r1&0x7FFFFF), r1=tm.INT_STS_0,  */
	0x0d0205a1, 0x00010001, 0x004307c6,	/* (r1&0x80000000), r1=tm.INT_STS_0,  */
	0x0d0005a2, 0x00010001, 0x007a07c7,	/* (r1&0x7F800000), r1=tm.INT_STS_0,  */
	0x0d0005a3, 0x00010001, 0x007b07c8,	/* (r1&0x41E), r1=tm.INT_STS_1,  */
	0x000005a4, 0x00010002, 0x000007c9,	/* ((r1&~r2)!=0), r1=tm.PRTY_STS_H_0, r2=tm.PRTY_MASK_H_0,  */
	0x010105a5, 0x00010001, 0x000007cb,	/* (r1!=0), r1=tm.pxp_read_data_fifo_status,  */
	0x010105a6, 0x00010001, 0x000007cc,	/* (r1!=0), r1=tm.pxp_read_ctrl_fifo_status,  */
	0x010105a7, 0x00010001, 0x000007cd,	/* (r1!=0), r1=tm.cfc_load_echo_fifo_status,  */
	0x010105a8, 0x00010001, 0x000007ce,	/* (r1!=0), r1=tm.client_out_fifo_status,  */
	0x010105a9, 0x00010001, 0x000007cf,	/* (r1!=0), r1=tm.client_in_pbf_fifo_status,  */
	0x010105aa, 0x00010001, 0x000007d0,	/* (r1!=0), r1=tm.client_in_xcm_fifo_status,  */
	0x010105ab, 0x00010001, 0x000007d1,	/* (r1!=0), r1=tm.client_in_tcm_fifo_status,  */
	0x010105ac, 0x00010001, 0x000007d2,	/* (r1!=0), r1=tm.client_in_ucm_fifo_status,  */
	0x010105ad, 0x00010001, 0x000007d3,	/* (r1!=0), r1=tm.expiration_cmd_fifo_status,  */
	0x010105ae, 0x00010001, 0x000007d4,	/* (r1!=0), r1=tm.ac_command_fifo_status,  */
	0x060005af, 0x00020a01, 0x002507d5,	/* ((r1&0x01)!=0), r1=tm.debug_0_error_valid,  */
	0x060005b0, 0x00020201, 0x002507e0,	/* ((r1&0x01)!=0), r1=tm.debug_1_error_valid,  */
	0x060005b1, 0x00020101, 0x002507e3,	/* ((r1&0x01)!=0), r1=tm.debug_2_error_valid,  */
	0x060005b2, 0x00020101, 0x002507e5,	/* ((r1&0x01)!=0), r1=tm.debug_3_error_valid,  */
	0x060005b3, 0x00020101, 0x002507e7,	/* ((r1&0x01)!=0), r1=tm.debug_4_error_valid,  */
	0x010005b4, 0x00010001, 0x001b07e9,	/* (r1!=1), r1=tcfc.ll_init_done,  */
	0x010005b5, 0x00010001, 0x001b07ea,	/* (r1!=1), r1=tcfc.ac_init_done,  */
	0x010005b6, 0x00010001, 0x001b07eb,	/* (r1!=1), r1=tcfc.cam_init_done,  */
	0x010005b7, 0x00010001, 0x001b07ec,	/* (r1!=1), r1=tcfc.tidram_init_done,  */
	0x000005b8, 0x00010502, 0x000007ed,	/* ((r1&~r2)!=0), r1=tcfc.INT_STS_0, r2=tcfc.INT_MASK_0,  */
	0x000005b9, 0x00010002, 0x000007f4,	/* ((r1&~r2)!=0), r1=tcfc.PRTY_STS_H_0, r2=tcfc.PRTY_MASK_H_0,  */
	0x000005ba, 0x00010002, 0x000007f6,	/* ((r1&~r2)!=0), r1=tcfc.PRTY_STS, r2=tcfc.PRTY_STS,  */
	0x010105bb, 0x00010001, 0x000007f8,	/* (r1!=0x0), r1=tcfc.lstate_arriving,  */
	0x010105bc, 0x00010001, 0x000007f9,	/* (r1!=0x0), r1=tcfc.lstate_leaving,  */
	0x010105bd, 0x00010001, 0x002907fa,	/* (r1!=reset1), r1=tcfc.cduld_credit,  */
	0x010105be, 0x00010001, 0x002907fb,	/* (r1!=reset1), r1=tcfc.cduwb_credit,  */
	0x0003017d,		/* mode bb&(!(!asic)) */
	0x000005bf, 0x00010002, 0x000007fc,	/* ((r1&~r2)!=0), r1=ipc.INT_STS_0, r2=ipc.INT_MASK_0,  */
	0x000f0021,		/* mode (!bb)&(!(!(asic|emul_mac))) */
	0x060005c0, 0x00020001, 0x002507fe,	/* ((r1&0x00000001)!=0), r1=nwm.INT_STS,  */
	0x060005c1, 0x00020001, 0x007c07ff,	/* ((r1&0x000001FE)!=0), r1=nwm.INT_STS,  */
	0x000005c2, 0x00010002, 0x00000800,	/* ((r1&~r2)!=0), r1=nwm.PRTY_STS_H_0, r2=nwm.PRTY_MASK_H_0,  */
	0x000005c3, 0x00010002, 0x00000802,	/* ((r1&~r2)!=0), r1=nwm.PRTY_STS_H_1, r2=nwm.PRTY_MASK_H_1,  */
	0x000005c4, 0x00010002, 0x00000804,	/* ((r1&~r2)!=0), r1=nwm.PRTY_STS_H_2, r2=nwm.PRTY_MASK_H_2,  */
	0x00030203,		/* mode bb&asic */
	0x000005c5, 0x00010002, 0x00000806,	/* ((r1&~r2)!=0), r1=ipc.PRTY_STS, r2=ipc.PRTY_MASK,  */
};

/* Data size: 17788 bytes */

/* Array of attentions data per register */
static const u32 attn_reg[] = {
	0x00000000, 0x05014060, 0x00014063, 0x00014061,	/* grc.INT_STS_0 */
	0x00000000, 0x02014080, 0x00014083, 0x00014081,	/* grc.PRTY_STS_H_0 */
	0x00000000, 0x03002460, 0x00002463, 0x00002461,	/* miscs.INT_STS_0 */
	0x00030081, 0x0b002464, 0x00002467, 0x00002465,	/* miscs.INT_STS_1, mode !k2 */
	0x00000000, 0x01002468, 0x0000246b, 0x00002469,	/* miscs.PRTY_STS_0 */
	0x00000000, 0x01002060, 0x00002063, 0x00002061,	/* misc.INT_STS */
	0x00000000, 0x180aa060, 0x000aa063, 0x000aa061,	/* pglue_b.INT_STS */
	0x00000000, 0x010aa064, 0x000aa067, 0x000aa065,	/* pglue_b.PRTY_STS */
	0x01220005, 0x160aa080, 0x000aa083, 0x000aa081,	/* pglue_b.PRTY_STS_H_0, mode bb */
	0x00010023, 0x1f0aa080, 0x000aa083, 0x000aa081,	/* pglue_b.PRTY_STS_H_0, mode !bb */
	0x00200023, 0x030aa084, 0x000aa087, 0x000aa085,	/* pglue_b.PRTY_STS_H_1, mode !bb */
	0x01380005, 0x060860ba, 0x000860bd, 0x000860bb,	/* cnig.INT_STS, mode bb */
	0x00000023, 0x07086086, 0x00086089, 0x00086087,	/* cnig.INT_STS, mode !bb */
	0x013e0005, 0x020860d2, 0x000860d5, 0x000860d3,	/* cnig.PRTY_STS, mode bb */
	0x01400023, 0x0208608b, 0x0008608e, 0x0008608c,	/* cnig.PRTY_STS, mode !bb */
	0x0000006f, 0x0100c0f8, 0x0000c0fb, 0x0000c0f9,	/* cpmu.INT_STS_0, mode !(emul_reduced|fpga) */
	0x000000d9, 0x01010133, 0x00010136, 0x00010134,	/* ncsi.INT_STS_0, mode !(!asic) */
	0x000000d9, 0x01010000, 0x00010003, 0x00010001,	/* ncsi.PRTY_STS_H_0, mode !(!asic) */
	0x00000081, 0x01014c82, 0x00014c85, 0x00014c83,	/* opte.PRTY_STS, mode !k2 */
	0x00010081, 0x0b014c00, 0x00014c03, 0x00014c01,	/* opte.PRTY_STS_H_0, mode !k2 */
	0x014200d9, 0x17150030, 0x00150033, 0x00150031,	/* bmb.INT_STS_0, mode !(!asic) */
	0x015900d9, 0x20150036, 0x00150039, 0x00150037,	/* bmb.INT_STS_1, mode !(!asic) */
	0x017900d9, 0x1c15003c, 0x0015003f, 0x0015003d,	/* bmb.INT_STS_2, mode !(!asic) */
	0x019500d9, 0x20150042, 0x00150045, 0x00150043,	/* bmb.INT_STS_3, mode !(!asic) */
	0x01b500d9, 0x1d150048, 0x0015004b, 0x00150049,	/* bmb.INT_STS_4, mode !(!asic) */
	0x01d200d9, 0x2015004e, 0x00150051, 0x0015004f,	/* bmb.INT_STS_5, mode !(!asic) */
	0x01f200d9, 0x20150054, 0x00150057, 0x00150055,	/* bmb.INT_STS_6, mode !(!asic) */
	0x00bb00d9, 0x2015005a, 0x0015005d, 0x0015005b,	/* bmb.INT_STS_7, mode !(!asic) */
	0x00db00d9, 0x20150061, 0x00150064, 0x00150062,	/* bmb.INT_STS_8, mode !(!asic) */
	0x00fb00d9, 0x20150067, 0x0015006a, 0x00150068,	/* bmb.INT_STS_9, mode !(!asic) */
	0x021200d9, 0x0415006d, 0x00150070, 0x0015006e,	/* bmb.INT_STS_10, mode !(!asic) */
	0x021600d9, 0x07150073, 0x00150076, 0x00150074,	/* bmb.INT_STS_11, mode !(!asic) */
	0x000000d9, 0x05150077, 0x0015007a, 0x00150078,	/* bmb.PRTY_STS, mode !(!asic) */
	0x000500d9, 0x1f150100, 0x00150103, 0x00150101,	/* bmb.PRTY_STS_H_0, mode !(!asic) */
	0x002400d9, 0x0f150104, 0x00150107, 0x00150105,	/* bmb.PRTY_STS_H_1, mode !(!asic) */
	0x00000023, 0x110151e8, 0x000151eb, 0x000151e9,	/* pcie.INT_STS, mode !bb */
	0x00000023, 0x030151ec, 0x000151ef, 0x000151ed,	/* pcie.PRTY_STS, mode !bb */
	0x021d0005, 0x11015000, 0x00015003, 0x00015001,	/* pcie.PRTY_STS_H_0, mode bb */
	0x00030023, 0x08015000, 0x00015003, 0x00015001,	/* pcie.PRTY_STS_H_0, mode !bb */
	0x00000000, 0x01014810, 0x00014813, 0x00014811,	/* mcp2.PRTY_STS */
	0x00010000, 0x0c014881, 0x00014884, 0x00014882,	/* mcp2.PRTY_STS_H_0 */
	0x00000000, 0x120a8060, 0x000a8063, 0x000a8061,	/* pswhst.INT_STS */
	0x00000000, 0x010a8064, 0x000a8067, 0x000a8065,	/* pswhst.PRTY_STS */
	0x00010000, 0x110a8080, 0x000a8083, 0x000a8081,	/* pswhst.PRTY_STS_H_0 */
	0x00000000, 0x050a7860, 0x000a7863, 0x000a7861,	/* pswhst2.INT_STS */
	0x00000000, 0x010a7864, 0x000a7867, 0x000a7865,	/* pswhst2.PRTY_STS */
	0x00000000, 0x030a7060, 0x000a7063, 0x000a7061,	/* pswrd.INT_STS */
	0x00000000, 0x010a7064, 0x000a7067, 0x000a7065,	/* pswrd.PRTY_STS */
	0x00000000, 0x050a7460, 0x000a7463, 0x000a7461,	/* pswrd2.INT_STS */
	0x00000000, 0x010a7464, 0x000a7467, 0x000a7465,	/* pswrd2.PRTY_STS */
	0x00010000, 0x1f0a7480, 0x000a7483, 0x000a7481,	/* pswrd2.PRTY_STS_H_0 */
	0x00200000, 0x030a7484, 0x000a7487, 0x000a7485,	/* pswrd2.PRTY_STS_H_1 */
	0x00000000, 0x100a6860, 0x000a6863, 0x000a6861,	/* pswwr.INT_STS */
	0x00000000, 0x010a6864, 0x000a6867, 0x000a6865,	/* pswwr.PRTY_STS */
	0x00000000, 0x130a6c60, 0x000a6c63, 0x000a6c61,	/* pswwr2.INT_STS */
	0x00000000, 0x010a6c64, 0x000a6c67, 0x000a6c65,	/* pswwr2.PRTY_STS */
	0x00010000, 0x1f0a6c80, 0x000a6c83, 0x000a6c81,	/* pswwr2.PRTY_STS_H_0 */
	0x00200000, 0x1f0a6c84, 0x000a6c87, 0x000a6c85,	/* pswwr2.PRTY_STS_H_1 */
	0x003f0000, 0x1f0a6c88, 0x000a6c8b, 0x000a6c89,	/* pswwr2.PRTY_STS_H_2 */
	0x005e0000, 0x140a6c8c, 0x000a6c8f, 0x000a6c8d,	/* pswwr2.PRTY_STS_H_3 */
	0x00000000, 0x150a0060, 0x000a0063, 0x000a0061,	/* pswrq.INT_STS */
	0x00000000, 0x010a0064, 0x000a0067, 0x000a0065,	/* pswrq.PRTY_STS */
	0x00000000, 0x0f090060, 0x00090063, 0x00090061,	/* pswrq2.INT_STS */
	0x022e0005, 0x09090080, 0x00090083, 0x00090081,	/* pswrq2.PRTY_STS_H_0, mode bb */
	0x00000023, 0x0a090080, 0x00090083, 0x00090081,	/* pswrq2.PRTY_STS_H_0, mode !bb */
	0x00000000, 0x02000740, 0x00000743, 0x00000741,	/* pglcs.INT_STS */
	0x00000000, 0x02003060, 0x00003063, 0x00003061,	/* dmae.INT_STS */
	0x00000000, 0x03003080, 0x00003083, 0x00003081,	/* dmae.PRTY_STS_H_0 */
	0x00000000, 0x08158060, 0x00158063, 0x00158061,	/* ptu.INT_STS */
	0x00000000, 0x12158080, 0x00158083, 0x00158081,	/* ptu.PRTY_STS_H_0 */
	0x00000000, 0x08460060, 0x00460063, 0x00460061,	/* tcm.INT_STS_0 */
	0x00080000, 0x20460064, 0x00460067, 0x00460065,	/* tcm.INT_STS_1 */
	0x00280000, 0x01460068, 0x0046006b, 0x00460069,	/* tcm.INT_STS_2 */
	0x02370005, 0x1f460080, 0x00460083, 0x00460081,	/* tcm.PRTY_STS_H_0, mode bb */
	0x00000023, 0x1f460080, 0x00460083, 0x00460081,	/* tcm.PRTY_STS_H_0, mode !bb */
	0x00280005, 0x02460084, 0x00460087, 0x00460085,	/* tcm.PRTY_STS_H_1, mode bb */
	0x00270023, 0x03460084, 0x00460087, 0x00460085,	/* tcm.PRTY_STS_H_1, mode !bb */
	0x00000000, 0x0e480060, 0x00480063, 0x00480061,	/* mcm.INT_STS_0 */
	0x000e0000, 0x1a480064, 0x00480067, 0x00480065,	/* mcm.INT_STS_1 */
	0x00280000, 0x01480068, 0x0048006b, 0x00480069,	/* mcm.INT_STS_2 */
	0x00000000, 0x1f480080, 0x00480083, 0x00480081,	/* mcm.PRTY_STS_H_0 */
	0x001f0000, 0x04480084, 0x00480087, 0x00480085,	/* mcm.PRTY_STS_H_1 */
	0x00000000, 0x114a0060, 0x004a0063, 0x004a0061,	/* ucm.INT_STS_0 */
	0x00110000, 0x1d4a0064, 0x004a0067, 0x004a0065,	/* ucm.INT_STS_1 */
	0x002e0000, 0x014a0068, 0x004a006b, 0x004a0069,	/* ucm.INT_STS_2 */
	0x00000000, 0x1f4a0080, 0x004a0083, 0x004a0081,	/* ucm.PRTY_STS_H_0 */
	0x001f0000, 0x074a0084, 0x004a0087, 0x004a0085,	/* ucm.PRTY_STS_H_1 */
	0x00000000, 0x10400060, 0x00400063, 0x00400061,	/* xcm.INT_STS_0 */
	0x00100000, 0x19400064, 0x00400067, 0x00400065,	/* xcm.INT_STS_1 */
	0x00290000, 0x08400068, 0x0040006b, 0x00400069,	/* xcm.INT_STS_2 */
	0x02560005, 0x1f400080, 0x00400083, 0x00400081,	/* xcm.PRTY_STS_H_0, mode bb */
	0x00000023, 0x1f400080, 0x00400083, 0x00400081,	/* xcm.PRTY_STS_H_0, mode !bb */
	0x02750005, 0x0b400084, 0x00400087, 0x00400085,	/* xcm.PRTY_STS_H_1, mode bb */
	0x00250023, 0x0c400084, 0x00400087, 0x00400085,	/* xcm.PRTY_STS_H_1, mode !bb */
	0x00000000, 0x0d420060, 0x00420063, 0x00420061,	/* ycm.INT_STS_0 */
	0x000d0000, 0x17420064, 0x00420067, 0x00420065,	/* ycm.INT_STS_1 */
	0x00240000, 0x01420068, 0x0042006b, 0x00420069,	/* ycm.INT_STS_2 */
	0x02800005, 0x1f420080, 0x00420083, 0x00420081,	/* ycm.PRTY_STS_H_0, mode bb */
	0x00000023, 0x1f420080, 0x00420083, 0x00420081,	/* ycm.PRTY_STS_H_0, mode !bb */
	0x00290005, 0x03420084, 0x00420087, 0x00420085,	/* ycm.PRTY_STS_H_1, mode bb */
	0x00280023, 0x04420084, 0x00420087, 0x00420085,	/* ycm.PRTY_STS_H_1, mode !bb */
	0x00000000, 0x05440060, 0x00440063, 0x00440061,	/* pcm.INT_STS_0 */
	0x00050000, 0x0e440064, 0x00440067, 0x00440065,	/* pcm.INT_STS_1 */
	0x00130000, 0x01440068, 0x0044006b, 0x00440069,	/* pcm.INT_STS_2 */
	0x029f0005, 0x0b440080, 0x00440083, 0x00440081,	/* pcm.PRTY_STS_H_0, mode bb */
	0x00000023, 0x0f440080, 0x00440083, 0x00440081,	/* pcm.PRTY_STS_H_0, mode !bb */
	0x00000000, 0x160bc060, 0x000bc063, 0x000bc061,	/* qm.INT_STS */
	0x00000000, 0x0b0bc064, 0x000bc067, 0x000bc065,	/* qm.PRTY_STS */
	0x000b0000, 0x1f0bc080, 0x000bc083, 0x000bc081,	/* qm.PRTY_STS_H_0 */
	0x002a0000, 0x1f0bc084, 0x000bc087, 0x000bc085,	/* qm.PRTY_STS_H_1 */
	0x02aa0005, 0x0b0bc088, 0x000bc08b, 0x000bc089,	/* qm.PRTY_STS_H_2, mode bb */
	0x00490023, 0x130bc088, 0x000bc08b, 0x000bc089,	/* qm.PRTY_STS_H_2, mode !bb */
	0x02b50151, 0x200b0060, 0x000b0063, 0x000b0061,	/* tm.INT_STS_0, mode !fpga */
	0x001f0151, 0x0b0b0064, 0x000b0067, 0x000b0065,	/* tm.INT_STS_1, mode !fpga */
	0x00000151, 0x110b0080, 0x000b0083, 0x000b0081,	/* tm.PRTY_STS_H_0, mode !fpga */
	0x00000000, 0x09040060, 0x00040063, 0x00040061,	/* dorq.INT_STS */
	0x00000000, 0x01040064, 0x00040067, 0x00040065,	/* dorq.PRTY_STS */
	0x00010000, 0x06040080, 0x00040083, 0x00040081,	/* dorq.PRTY_STS_H_0 */
	0x00000000, 0x200d0030, 0x000d0033, 0x000d0031,	/* brb.INT_STS_0 */
	0x02d50000, 0x200d0036, 0x000d0039, 0x000d0037,	/* brb.INT_STS_1 */
	0x003e0000, 0x1c0d003c, 0x000d003f, 0x000d003d,	/* brb.INT_STS_2 */
	0x02f50000, 0x200d0042, 0x000d0045, 0x000d0043,	/* brb.INT_STS_3 */
	0x03150000, 0x1c0d0048, 0x000d004b, 0x000d0049,	/* brb.INT_STS_4 */
	0x00940000, 0x010d004e, 0x000d0051, 0x000d004f,	/* brb.INT_STS_5 */
	0x03310000, 0x0a0d0054, 0x000d0057, 0x000d0055,	/* brb.INT_STS_6 */
	0x009d0000, 0x200d005a, 0x000d005d, 0x000d005b,	/* brb.INT_STS_7 */
	0x00bd0000, 0x110d0061, 0x000d0064, 0x000d0062,	/* brb.INT_STS_8 */
	0x00ce0000, 0x010d0067, 0x000d006a, 0x000d0068,	/* brb.INT_STS_9 */
	0x033b0000, 0x150d006d, 0x000d0070, 0x000d006e,	/* brb.INT_STS_10 */
	0x03500000, 0x090d0073, 0x000d0076, 0x000d0074,	/* brb.INT_STS_11 */
	0x00000000, 0x050d0077, 0x000d007a, 0x000d0078,	/* brb.PRTY_STS */
	0x03590005, 0x1f0d0100, 0x000d0103, 0x000d0101,	/* brb.PRTY_STS_H_0, mode bb */
	0x00050023, 0x1f0d0100, 0x000d0103, 0x000d0101,	/* brb.PRTY_STS_H_0, mode !bb */
	0x03780005, 0x0e0d0104, 0x000d0107, 0x000d0105,	/* brb.PRTY_STS_H_1, mode bb */
	0x03860023, 0x1e0d0104, 0x000d0107, 0x000d0105,	/* brb.PRTY_STS_H_1, mode !bb */
	0x00000151, 0x0108e076, 0x0008e077, 0x0008e079,	/* src.INT_STS, mode !fpga */
	0x00000000, 0x0207c010, 0x0007c013, 0x0007c011,	/* prs.INT_STS_0 */
	0x00000000, 0x0207c014, 0x0007c017, 0x0007c015,	/* prs.PRTY_STS */
	0x03a40005, 0x1f07c081, 0x0007c084, 0x0007c082,	/* prs.PRTY_STS_H_0, mode bb */
	0x00020023, 0x1f07c081, 0x0007c084, 0x0007c082,	/* prs.PRTY_STS_H_0, mode !bb */
	0x03c30005, 0x0507c085, 0x0007c088, 0x0007c086,	/* prs.PRTY_STS_H_1, mode bb */
	0x03c80023, 0x1f07c085, 0x0007c088, 0x0007c086,	/* prs.PRTY_STS_H_1, mode !bb */
	0x00000000, 0x1c3ec010, 0x003ec013, 0x003ec011,	/* tsdm.INT_STS */
	0x00000000, 0x0a3ec080, 0x003ec083, 0x003ec081,	/* tsdm.PRTY_STS_H_0 */
	0x00000000, 0x1c3f0010, 0x003f0013, 0x003f0011,	/* msdm.INT_STS */
	0x00000000, 0x0b3f0080, 0x003f0083, 0x003f0081,	/* msdm.PRTY_STS_H_0 */
	0x00000000, 0x1c3f4010, 0x003f4013, 0x003f4011,	/* usdm.INT_STS */
	0x00000000, 0x0a3f4080, 0x003f4083, 0x003f4081,	/* usdm.PRTY_STS_H_0 */
	0x00000000, 0x1c3e0010, 0x003e0013, 0x003e0011,	/* xsdm.INT_STS */
	0x00000000, 0x0a3e0080, 0x003e0083, 0x003e0081,	/* xsdm.PRTY_STS_H_0 */
	0x00000000, 0x1c3e4010, 0x003e4013, 0x003e4011,	/* ysdm.INT_STS */
	0x00000000, 0x093e4080, 0x003e4083, 0x003e4081,	/* ysdm.PRTY_STS_H_0 */
	0x00000000, 0x1c3e8010, 0x003e8013, 0x003e8011,	/* psdm.INT_STS */
	0x00000000, 0x093e8080, 0x003e8083, 0x003e8081,	/* psdm.PRTY_STS_H_0 */
	0x00000000, 0x205c0010, 0x005c0013, 0x005c0011,	/* tsem.INT_STS_0 */
	0x00200000, 0x0d5c0014, 0x005c0017, 0x005c0015,	/* tsem.INT_STS_1 */
	0x00000000, 0x015d0010, 0x005d0013, 0x005d0011,	/* tsem.fast_memory.INT_STS */
	0x00000000, 0x035c0032, 0x005c0035, 0x005c0033,	/* tsem.PRTY_STS */
	0x00030000, 0x065c0080, 0x005c0083, 0x005c0081,	/* tsem.PRTY_STS_H_0 */
	0x00090023, 0x075d0080, 0x005d0083, 0x005d0081,	/* tsem.fast_memory.PRTY_STS_H_0, mode !bb */
	0x03e70151, 0x065d2880, 0x005d2883, 0x005d2881,	/* tsem.fast_memory.vfc_config.PRTY_STS_H_0, mode !fpga */
	0x00000000, 0x20600010, 0x00600013, 0x00600011,	/* msem.INT_STS_0 */
	0x00200000, 0x0d600014, 0x00600017, 0x00600015,	/* msem.INT_STS_1 */
	0x00000000, 0x01610010, 0x00610013, 0x00610011,	/* msem.fast_memory.INT_STS */
	0x00000000, 0x03600032, 0x00600035, 0x00600033,	/* msem.PRTY_STS */
	0x00030000, 0x06600080, 0x00600083, 0x00600081,	/* msem.PRTY_STS_H_0 */
	0x00090023, 0x07610080, 0x00610083, 0x00610081,	/* msem.fast_memory.PRTY_STS_H_0, mode !bb */
	0x00000000, 0x20640010, 0x00640013, 0x00640011,	/* usem.INT_STS_0 */
	0x00200000, 0x0d640014, 0x00640017, 0x00640015,	/* usem.INT_STS_1 */
	0x00000000, 0x01650010, 0x00650013, 0x00650011,	/* usem.fast_memory.INT_STS */
	0x00000000, 0x03640032, 0x00640035, 0x00640033,	/* usem.PRTY_STS */
	0x00030000, 0x06640080, 0x00640083, 0x00640081,	/* usem.PRTY_STS_H_0 */
	0x00090023, 0x07650080, 0x00650083, 0x00650081,	/* usem.fast_memory.PRTY_STS_H_0, mode !bb */
	0x00000000, 0x20500010, 0x00500013, 0x00500011,	/* xsem.INT_STS_0 */
	0x00200000, 0x0d500014, 0x00500017, 0x00500015,	/* xsem.INT_STS_1 */
	0x00000000, 0x01510010, 0x00510013, 0x00510011,	/* xsem.fast_memory.INT_STS */
	0x00000000, 0x03500032, 0x00500035, 0x00500033,	/* xsem.PRTY_STS */
	0x00030000, 0x07500080, 0x00500083, 0x00500081,	/* xsem.PRTY_STS_H_0 */
	0x000a0023, 0x07510080, 0x00510083, 0x00510081,	/* xsem.fast_memory.PRTY_STS_H_0, mode !bb */
	0x00000000, 0x20540010, 0x00540013, 0x00540011,	/* ysem.INT_STS_0 */
	0x00200000, 0x0d540014, 0x00540017, 0x00540015,	/* ysem.INT_STS_1 */
	0x00000000, 0x01550010, 0x00550013, 0x00550011,	/* ysem.fast_memory.INT_STS */
	0x00000000, 0x03540032, 0x00540035, 0x00540033,	/* ysem.PRTY_STS */
	0x00030000, 0x07540080, 0x00540083, 0x00540081,	/* ysem.PRTY_STS_H_0 */
	0x000a0023, 0x07550080, 0x00550083, 0x00550081,	/* ysem.fast_memory.PRTY_STS_H_0, mode !bb */
	0x00000000, 0x20580010, 0x00580013, 0x00580011,	/* psem.INT_STS_0 */
	0x00200000, 0x0d580014, 0x00580017, 0x00580015,	/* psem.INT_STS_1 */
	0x00000000, 0x01590010, 0x00590013, 0x00590011,	/* psem.fast_memory.INT_STS */
	0x00000000, 0x03580032, 0x00580035, 0x00580033,	/* psem.PRTY_STS */
	0x00030000, 0x06580080, 0x00580083, 0x00580081,	/* psem.PRTY_STS_H_0 */
	0x00090023, 0x07590080, 0x00590083, 0x00590081,	/* psem.fast_memory.PRTY_STS_H_0, mode !bb */
	0x03e7006f, 0x06592880, 0x00592883, 0x00592881,	/* psem.fast_memory.vfc_config.PRTY_STS_H_0, mode !(emul_reduced|fpga) */
	0x00000000, 0x0c08e260, 0x0008e263, 0x0008e261,	/* rss.INT_STS */
	0x00000000, 0x0408e280, 0x0008e283, 0x0008e281,	/* rss.PRTY_STS_H_0 */
	0x00000000, 0x06134060, 0x00134063, 0x00134061,	/* tmld.INT_STS */
	0x00000000, 0x08134080, 0x00134083, 0x00134081,	/* tmld.PRTY_STS_H_0 */
	0x00000000, 0x06138060, 0x00138063, 0x00138061,	/* muld.INT_STS */
	0x00000000, 0x0a138080, 0x00138083, 0x00138081,	/* muld.PRTY_STS_H_0 */
	0x00000000, 0x06132060, 0x00132063, 0x00132061,	/* yuld.INT_STS */
	0x00000000, 0x06132080, 0x00132083, 0x00132081,	/* yuld.PRTY_STS_H_0 */
	0x00000000, 0x06130060, 0x00130063, 0x00130061,	/* xyld.INT_STS */
	0x00000000, 0x09130080, 0x00130083, 0x00130081,	/* xyld.PRTY_STS_H_0 */
	0x00000000, 0x0b08c010, 0x0008c013, 0x0008c011,	/* prm.INT_STS */
	0x00000000, 0x0108c014, 0x0008c017, 0x0008c015,	/* prm.PRTY_STS */
	0x03ed0005, 0x1808c080, 0x0008c083, 0x0008c081,	/* prm.PRTY_STS_H_0, mode bb */
	0x00010023, 0x1708c080, 0x0008c083, 0x0008c081,	/* prm.PRTY_STS_H_0, mode !bb */
	0x00000000, 0x09368010, 0x00368013, 0x00368011,	/* pbf_pb1.INT_STS */
	0x00000000, 0x01368014, 0x00368017, 0x00368015,	/* pbf_pb1.PRTY_STS */
	0x00000000, 0x09369010, 0x00369013, 0x00369011,	/* pbf_pb2.INT_STS */
	0x00000000, 0x01369014, 0x00369017, 0x00369015,	/* pbf_pb2.PRTY_STS */
	0x00000000, 0x0908f010, 0x0008f013, 0x0008f011,	/* rpb.INT_STS */
	0x00000000, 0x0108f014, 0x0008f017, 0x0008f015,	/* rpb.PRTY_STS */
	0x04050000, 0x1a36c030, 0x0036c033, 0x0036c031,	/* btb.INT_STS_0 */
	0x041f0000, 0x1236c036, 0x0036c039, 0x0036c037,	/* btb.INT_STS_1 */
	0x04310000, 0x0536c03c, 0x0036c03f, 0x0036c03d,	/* btb.INT_STS_2 */
	0x00240000, 0x2036c042, 0x0036c045, 0x0036c043,	/* btb.INT_STS_3 */
	0x04360000, 0x1c36c048, 0x0036c04b, 0x0036c049,	/* btb.INT_STS_4 */
	0x005b0000, 0x2036c04e, 0x0036c051, 0x0036c04f,	/* btb.INT_STS_5 */
	0x007b0000, 0x0136c054, 0x0036c057, 0x0036c055,	/* btb.INT_STS_6 */
	0x007c0000, 0x0136c061, 0x0036c064, 0x0036c062,	/* btb.INT_STS_8 */
	0x007d0000, 0x0136c067, 0x0036c06a, 0x0036c068,	/* btb.INT_STS_9 */
	0x04520000, 0x0236c06d, 0x0036c070, 0x0036c06e,	/* btb.INT_STS_10 */
	0x04540000, 0x0436c073, 0x0036c076, 0x0036c074,	/* btb.INT_STS_11 */
	0x00000000, 0x0536c077, 0x0036c07a, 0x0036c078,	/* btb.PRTY_STS */
	0x04580005, 0x1736c100, 0x0036c103, 0x0036c101,	/* btb.PRTY_STS_H_0, mode bb */
	0x00050023, 0x1f36c100, 0x0036c103, 0x0036c101,	/* btb.PRTY_STS_H_0, mode !bb */
	0x00000000, 0x01360060, 0x00360063, 0x00360061,	/* pbf.INT_STS */
	0x00000000, 0x01360064, 0x00360067, 0x00360065,	/* pbf.PRTY_STS */
	0x00010000, 0x1f360080, 0x00360083, 0x00360081,	/* pbf.PRTY_STS_H_0 */
	0x00200000, 0x1b360084, 0x00360087, 0x00360085,	/* pbf.PRTY_STS_H_1 */
	0x00000000, 0x090c0060, 0x000c0063, 0x000c0061,	/* rdif.INT_STS */
	0x01400000, 0x020c0064, 0x000c0067, 0x000c0065,	/* rdif.PRTY_STS */
	0x00000000, 0x090c4060, 0x000c4063, 0x000c4061,	/* tdif.INT_STS */
	0x01400000, 0x020c4064, 0x000c4067, 0x000c4065,	/* tdif.PRTY_STS */
	0x00010000, 0x0b0c4080, 0x000c4083, 0x000c4081,	/* tdif.PRTY_STS_H_0 */
	0x00000000, 0x08160070, 0x00160071, 0x00160073,	/* cdu.INT_STS */
	0x00000000, 0x05160080, 0x00160083, 0x00160081,	/* cdu.PRTY_STS_H_0 */
	0x00000000, 0x020b8060, 0x000b8063, 0x000b8061,	/* ccfc.INT_STS_0 */
	0x00000000, 0x060b8179, 0x000b817c, 0x000b817a,	/* ccfc.PRTY_STS */
	0x00060000, 0x020b8080, 0x000b8083, 0x000b8081,	/* ccfc.PRTY_STS_H_0 */
	0x00000151, 0x020b4060, 0x000b4063, 0x000b4061,	/* tcfc.INT_STS_0, mode !fpga */
	0x00000151, 0x060b4179, 0x000b417c, 0x000b417a,	/* tcfc.PRTY_STS, mode !fpga */
	0x00060151, 0x020b4080, 0x000b4083, 0x000b4081,	/* tcfc.PRTY_STS_H_0, mode !fpga */
	0x00000000, 0x0b060060, 0x00060063, 0x00060061,	/* igu.INT_STS */
	0x00000000, 0x01060064, 0x00060067, 0x00060065,	/* igu.PRTY_STS */
	0x046f0005, 0x1f060080, 0x00060083, 0x00060081,	/* igu.PRTY_STS_H_0, mode bb */
	0x00010023, 0x1c060080, 0x00060083, 0x00060081,	/* igu.PRTY_STS_H_0, mode !bb */
	0x00020081, 0x01060084, 0x00060087, 0x00060085,	/* igu.PRTY_STS_H_1, mode !k2 */
	0x00000000, 0x0b070035, 0x00070036, 0x00070038,	/* cau.INT_STS */
	0x048e0005, 0x0d070080, 0x00070083, 0x00070081,	/* cau.PRTY_STS_H_0, mode bb */
	0x00000023, 0x0d070080, 0x00070083, 0x00070081,	/* cau.PRTY_STS_H_0, mode !bb */
	0x000000d3, 0x02014460, 0x00014463, 0x00014461,	/* umac.INT_STS, mode (!bb)&(!(!asic)) */
	0x000000d9, 0x010853fc, 0x000853fd, 0x000853ff,	/* mstat.INT_STS, mode !(!asic) */
	0x00000000, 0x01004060, 0x00004063, 0x00004061,	/* dbg.INT_STS */
	0x00000000, 0x01004080, 0x00004083, 0x00004081,	/* dbg.PRTY_STS_H_0 */
	0x00000000, 0x0c140010, 0x00140013, 0x00140011,	/* nig.INT_STS_0 */
	0x000c0000, 0x20140014, 0x00140017, 0x00140015,	/* nig.INT_STS_1 */
	0x002c0000, 0x14140018, 0x0014001b, 0x00140019,	/* nig.INT_STS_2 */
	0x00400000, 0x1214001c, 0x0014001f, 0x0014001d,	/* nig.INT_STS_3 */
	0x00520000, 0x14140020, 0x00140023, 0x00140021,	/* nig.INT_STS_4 */
	0x00660000, 0x12140024, 0x00140027, 0x00140025,	/* nig.INT_STS_5 */
	0x00780023, 0x14140028, 0x0014002b, 0x00140029,	/* nig.INT_STS_6, mode !bb */
	0x008c0023, 0x1214002c, 0x0014002f, 0x0014002d,	/* nig.INT_STS_7, mode !bb */
	0x009e0023, 0x14140030, 0x00140033, 0x00140031,	/* nig.INT_STS_8, mode !bb */
	0x00b20023, 0x12140034, 0x00140037, 0x00140035,	/* nig.INT_STS_9, mode !bb */
	0x00000005, 0x01140028, 0x0014002b, 0x00140029,	/* nig.PRTY_STS, mode bb */
	0x00000023, 0x01140038, 0x0014003b, 0x00140039,	/* nig.PRTY_STS, mode !bb */
	0x049b0005, 0x1f140080, 0x00140083, 0x00140081,	/* nig.PRTY_STS_H_0, mode bb */
	0x00010023, 0x1f140080, 0x00140083, 0x00140081,	/* nig.PRTY_STS_H_0, mode !bb */
	0x04ba0000, 0x3e140084, 0x00140087, 0x00140085,	/* nig.PRTY_STS_H_1 */
	0x04f80005, 0x1f140088, 0x0014008b, 0x00140089,	/* nig.PRTY_STS_H_2, mode bb */
	0x05170023, 0x1f140088, 0x0014008b, 0x00140089,	/* nig.PRTY_STS_H_2, mode !bb */
	0x05360000, 0x1f14008c, 0x0014008f, 0x0014008d,	/* nig.PRTY_STS_H_3 */
	0x000000d3, 0x01180010, 0x00180013, 0x00180011,	/* wol.INT_STS_0, mode (!bb)&(!(!asic)) */
	0x000000d3, 0x18180080, 0x00180083, 0x00180081,	/* wol.PRTY_STS_H_0, mode (!bb)&(!(!asic)) */
	0x00000023, 0x01184010, 0x00184013, 0x00184011,	/* bmbn.INT_STS_0, mode !bb */
	0x0555017d, 0x0e008143, 0x00008146, 0x00008144,	/* ipc.INT_STS_0, mode bb&(!(!asic)) */
	0x055500d3, 0x060080b7, 0x000080ba, 0x000080b8,	/* ipc.INT_STS_0, mode (!bb)&(!(!asic)) */
	0x000000eb, 0x01008147, 0x0000814a, 0x00008148,	/* ipc.PRTY_STS, mode (!k2)&(!(!asic)) */
	0x05630021, 0x12200001, 0x00200004, 0x00200002,	/* nwm.INT_STS, mode (!bb)&(!(!(asic|emul_mac))) */
	0x00000021, 0x1f200080, 0x00200083, 0x00200081,	/* nwm.PRTY_STS_H_0, mode (!bb)&(!(!(asic|emul_mac))) */
	0x001f0021, 0x1f200084, 0x00200087, 0x00200085,	/* nwm.PRTY_STS_H_1, mode (!bb)&(!(!(asic|emul_mac))) */
	0x003e0021, 0x0a200088, 0x0020008b, 0x00200089,	/* nwm.PRTY_STS_H_2, mode (!bb)&(!(!(asic|emul_mac))) */
	0x000000d3, 0x0a1c0060, 0x001c0063, 0x001c0061,	/* nws.INT_STS_0, mode (!bb)&(!(!asic)) */
	0x057500d3, 0x0a1c0064, 0x001c0067, 0x001c0065,	/* nws.INT_STS_1, mode (!bb)&(!(!asic)) */
	0x057f00d3, 0x0a1c0068, 0x001c006b, 0x001c0069,	/* nws.INT_STS_2, mode (!bb)&(!(!asic)) */
	0x058900d3, 0x0a1c006c, 0x001c006f, 0x001c006d,	/* nws.INT_STS_3, mode (!bb)&(!(!asic)) */
	0x000000d3, 0x041c0080, 0x001c0083, 0x001c0081,	/* nws.PRTY_STS_H_0, mode (!bb)&(!(!asic)) */
	0x000000d3, 0x011a8060, 0x001a8063, 0x001a8061,	/* ms.INT_STS, mode (!bb)&(!(!asic)) */
	0x00000023, 0x011ae060, 0x001ae063, 0x001ae061,	/* led.INT_STS_0, mode !bb */
	0x05930023, 0x031ad002, 0x001ad005, 0x001ad003,	/* avs_wrap.INT_STS, mode !bb */
	0x00000023, 0x031ad006, 0x001ad009, 0x001ad007,	/* avs_wrap.PRTY_STS, mode !bb */
	0x00000023, 0x14015800, 0x00015803, 0x00015801,	/* pxpreqbus.PRTY_STS_H_0, mode !bb */
};

/* Data size: 4688 bytes */

/* Array of attentions data per block */
static const u32 attn_block[] = {
	0x00000000, 0x00000001, 0x00000005, 0x00010001,	/* block grc, 1 interrupt regs (5 attentions), 1 parity regs (2 attentions) */
	0x00000007, 0x00020002, 0x00000015, 0x00040001,	/* block miscs, 2 interrupt regs (14 attentions), 1 parity regs (1 attentions) */
	0x00000000, 0x00050001, 0x00000000, 0x00060000,	/* block misc, 1 interrupt regs (1 attentions) */
	0x00000000, 0x00060000, 0x00000000, 0x00060000,	/* block dbu */
	0x00000016, 0x00060001, 0x0000002e, 0x00070004,	/* block pglue_b, 1 interrupt regs (24 attentions), 4 parity regs (35 attentions) */
	0x00000051, 0x000b0002, 0x0000005b, 0x000d0002,	/* block cnig, 2 interrupt regs (10 attentions), 2 parity regs (2 attentions) */
	0x00000000, 0x000f0001, 0x00000000, 0x00100000,	/* block cpmu, 1 interrupt regs (1 attentions) */
	0x00000000, 0x00100001, 0x00000006, 0x00110001,	/* block ncsi, 1 interrupt regs (1 attentions), 1 parity regs (1 attentions) */
	0x00000000, 0x00120000, 0x0000005d, 0x00120002,	/* block opte, 2 parity regs (12 attentions) */
	0x00000069, 0x0014000c, 0x0000018b, 0x00200003,	/* block bmb, 12 interrupt regs (290 attentions), 3 parity regs (51 attentions) */
	0x000001be, 0x00230001, 0x000001cf, 0x00240003,	/* block pcie, 1 interrupt regs (17 attentions), 3 parity regs (24 attentions) */
	0x00000000, 0x00270000, 0x00000000, 0x00270000,	/* block mcp */
	0x00000000, 0x00270000, 0x000001e7, 0x00270002,	/* block mcp2, 2 parity regs (13 attentions) */
	0x000001f4, 0x00290001, 0x00000206, 0x002a0002,	/* block pswhst, 1 interrupt regs (18 attentions), 2 parity regs (18 attentions) */
	0x00000218, 0x002c0001, 0x0000002e, 0x002d0001,	/* block pswhst2, 1 interrupt regs (5 attentions), 1 parity regs (1 attentions) */
	0x0000021d, 0x002e0001, 0x0000002e, 0x002f0001,	/* block pswrd, 1 interrupt regs (3 attentions), 1 parity regs (1 attentions) */
	0x00000220, 0x00300001, 0x00000225, 0x00310003,	/* block pswrd2, 1 interrupt regs (5 attentions), 3 parity regs (35 attentions) */
	0x00000248, 0x00340001, 0x0000002e, 0x00350001,	/* block pswwr, 1 interrupt regs (16 attentions), 1 parity regs (1 attentions) */
	0x00000258, 0x00360001, 0x0000026b, 0x00370005,	/* block pswwr2, 1 interrupt regs (19 attentions), 5 parity regs (114 attentions) */
	0x000002dd, 0x003c0001, 0x000002f2, 0x003d0001,	/* block pswrq, 1 interrupt regs (21 attentions), 1 parity regs (1 attentions) */
	0x000002f3, 0x003e0001, 0x00000302, 0x003f0002,	/* block pswrq2, 1 interrupt regs (15 attentions), 2 parity regs (11 attentions) */
	0x0000030d, 0x00410001, 0x00000000, 0x00420000,	/* block pglcs, 1 interrupt regs (2 attentions) */
	0x0000030f, 0x00420001, 0x00000311, 0x00430001,	/* block dmae, 1 interrupt regs (2 attentions), 1 parity regs (3 attentions) */
	0x00000314, 0x00440001, 0x0000031c, 0x00450001,	/* block ptu, 1 interrupt regs (8 attentions), 1 parity regs (18 attentions) */
	0x0000032e, 0x00460003, 0x00000357, 0x00490004,	/* block tcm, 3 interrupt regs (41 attentions), 4 parity regs (42 attentions) */
	0x00000381, 0x004d0003, 0x000003aa, 0x00500002,	/* block mcm, 3 interrupt regs (41 attentions), 2 parity regs (35 attentions) */
	0x000003cd, 0x00520003, 0x000003fc, 0x00550002,	/* block ucm, 3 interrupt regs (47 attentions), 2 parity regs (38 attentions) */
	0x00000422, 0x00570003, 0x00000453, 0x005a0004,	/* block xcm, 3 interrupt regs (49 attentions), 4 parity regs (49 attentions) */
	0x00000484, 0x005e0003, 0x000004a9, 0x00610004,	/* block ycm, 3 interrupt regs (37 attentions), 4 parity regs (44 attentions) */
	0x000004d5, 0x00650003, 0x000004e9, 0x00680002,	/* block pcm, 3 interrupt regs (20 attentions), 2 parity regs (19 attentions) */
	0x000004fc, 0x006a0001, 0x00000512, 0x006b0005,	/* block qm, 1 interrupt regs (22 attentions), 5 parity regs (92 attentions) */
	0x0000056e, 0x00700002, 0x00000598, 0x00720001,	/* block tm, 2 interrupt regs (42 attentions), 1 parity regs (17 attentions) */
	0x000005a9, 0x00730001, 0x000005b2, 0x00740002,	/* block dorq, 1 interrupt regs (9 attentions), 2 parity regs (7 attentions) */
	0x000005b9, 0x0076000c, 0x000006a2, 0x00820005,	/* block brb, 12 interrupt regs (233 attentions), 5 parity regs (74 attentions) */
	0x00000000, 0x00870001, 0x00000000, 0x00880000,	/* block src, 1 interrupt regs (1 attentions) */
	0x000006ec, 0x00880001, 0x000006ee, 0x00890005,	/* block prs, 1 interrupt regs (2 attentions), 5 parity regs (75 attentions) */
	0x00000739, 0x008e0001, 0x00000755, 0x008f0001,	/* block tsdm, 1 interrupt regs (28 attentions), 1 parity regs (10 attentions) */
	0x00000739, 0x00900001, 0x0000075f, 0x00910001,	/* block msdm, 1 interrupt regs (28 attentions), 1 parity regs (11 attentions) */
	0x00000739, 0x00920001, 0x00000756, 0x00930001,	/* block usdm, 1 interrupt regs (28 attentions), 1 parity regs (10 attentions) */
	0x00000739, 0x00940001, 0x0000076a, 0x00950001,	/* block xsdm, 1 interrupt regs (28 attentions), 1 parity regs (10 attentions) */
	0x00000739, 0x00960001, 0x00000774, 0x00970001,	/* block ysdm, 1 interrupt regs (28 attentions), 1 parity regs (9 attentions) */
	0x00000739, 0x00980001, 0x00000774, 0x00990001,	/* block psdm, 1 interrupt regs (28 attentions), 1 parity regs (9 attentions) */
	0x0000077d, 0x009a0003, 0x000007aa, 0x009d0004,	/* block tsem, 3 interrupt regs (45 attentions), 4 parity regs (20 attentions) */
	0x0000077d, 0x00a10003, 0x000007aa, 0x00a40003,	/* block msem, 3 interrupt regs (45 attentions), 3 parity regs (16 attentions) */
	0x0000077d, 0x00a70003, 0x000007aa, 0x00aa0003,	/* block usem, 3 interrupt regs (45 attentions), 3 parity regs (16 attentions) */
	0x0000077d, 0x00ad0003, 0x000007be, 0x00b00003,	/* block xsem, 3 interrupt regs (45 attentions), 3 parity regs (17 attentions) */
	0x0000077d, 0x00b30003, 0x000007be, 0x00b60003,	/* block ysem, 3 interrupt regs (45 attentions), 3 parity regs (17 attentions) */
	0x0000077d, 0x00b90003, 0x000007aa, 0x00bc0004,	/* block psem, 3 interrupt regs (45 attentions), 4 parity regs (20 attentions) */
	0x000007cf, 0x00c00001, 0x000007db, 0x00c10001,	/* block rss, 1 interrupt regs (12 attentions), 1 parity regs (4 attentions) */
	0x000007df, 0x00c20001, 0x000007e5, 0x00c30001,	/* block tmld, 1 interrupt regs (6 attentions), 1 parity regs (8 attentions) */
	0x000007df, 0x00c40001, 0x000007ed, 0x00c50001,	/* block muld, 1 interrupt regs (6 attentions), 1 parity regs (10 attentions) */
	0x000007df, 0x00c60001, 0x000007f7, 0x00c70001,	/* block yuld, 1 interrupt regs (6 attentions), 1 parity regs (6 attentions) */
	0x000007df, 0x00c80001, 0x000007fd, 0x00c90001,	/* block xyld, 1 interrupt regs (6 attentions), 1 parity regs (9 attentions) */
	0x00000806, 0x00ca0001, 0x00000811, 0x00cb0003,	/* block prm, 1 interrupt regs (11 attentions), 3 parity regs (30 attentions) */
	0x0000082f, 0x00ce0001, 0x0000002e, 0x00cf0001,	/* block pbf_pb1, 1 interrupt regs (9 attentions), 1 parity regs (1 attentions) */
	0x0000082f, 0x00d00001, 0x0000002e, 0x00d10001,	/* block pbf_pb2, 1 interrupt regs (9 attentions), 1 parity regs (1 attentions) */
	0x0000082f, 0x00d20001, 0x0000002e, 0x00d30001,	/* block rpb, 1 interrupt regs (9 attentions), 1 parity regs (1 attentions) */
	0x00000838, 0x00d4000b, 0x000008b9, 0x00df0003,	/* block btb, 11 interrupt regs (129 attentions), 3 parity regs (36 attentions) */
	0x00000000, 0x00e20001, 0x000008dd, 0x00e30003,	/* block pbf, 1 interrupt regs (1 attentions), 3 parity regs (59 attentions) */
	0x00000918, 0x00e60001, 0x0000002e, 0x00e70001,	/* block rdif, 1 interrupt regs (9 attentions), 1 parity regs (1 attentions) */
	0x00000918, 0x00e80001, 0x00000921, 0x00e90002,	/* block tdif, 1 interrupt regs (9 attentions), 2 parity regs (12 attentions) */
	0x0000092d, 0x00eb0001, 0x00000935, 0x00ec0001,	/* block cdu, 1 interrupt regs (8 attentions), 1 parity regs (5 attentions) */
	0x0000093a, 0x00ed0001, 0x0000093c, 0x00ee0002,	/* block ccfc, 1 interrupt regs (2 attentions), 2 parity regs (8 attentions) */
	0x0000093a, 0x00f00001, 0x00000944, 0x00f10002,	/* block tcfc, 1 interrupt regs (2 attentions), 2 parity regs (8 attentions) */
	0x0000094c, 0x00f30001, 0x00000957, 0x00f40004,	/* block igu, 1 interrupt regs (11 attentions), 4 parity regs (42 attentions) */
	0x00000981, 0x00f80001, 0x0000098c, 0x00f90002,	/* block cau, 1 interrupt regs (11 attentions), 2 parity regs (15 attentions) */
	0x0000099b, 0x00fb0001, 0x00000000, 0x00fc0000,	/* block umac, 1 interrupt regs (2 attentions) */
	0x00000000, 0x00fc0000, 0x00000000, 0x00fc0000,	/* block xmac */
	0x00000000, 0x00fc0001, 0x00000000, 0x00fd0000,	/* block mstat, 1 interrupt regs (1 attentions) */
	0x00000000, 0x00fd0001, 0x00000050, 0x00fe0001,	/* block dbg, 1 interrupt regs (1 attentions), 1 parity regs (1 attentions) */
	0x0000099d, 0x00ff000a, 0x00000a61, 0x01090008,	/* block nig, 10 interrupt regs (196 attentions), 8 parity regs (113 attentions) */
	0x00000000, 0x01110001, 0x00000ad2, 0x01120001,	/* block wol, 1 interrupt regs (1 attentions), 1 parity regs (24 attentions) */
	0x00000000, 0x01130001, 0x00000000, 0x01140000,	/* block bmbn, 1 interrupt regs (1 attentions) */
	0x00000aea, 0x01140002, 0x00000af7, 0x01160001,	/* block ipc, 2 interrupt regs (13 attentions), 1 parity regs (1 attentions) */
	0x00000af8, 0x01170001, 0x00000b09, 0x01180003,	/* block nwm, 1 interrupt regs (17 attentions), 3 parity regs (72 attentions) */
	0x00000b51, 0x011b0004, 0x00000b76, 0x011f0001,	/* block nws, 4 interrupt regs (37 attentions), 1 parity regs (4 attentions) */
	0x00000000, 0x01200001, 0x00000000, 0x01210000,	/* block ms, 1 interrupt regs (1 attentions) */
	0x00000000, 0x01210000, 0x00000000, 0x01210000,	/* block phy_pcie */
	0x00000000, 0x01210001, 0x00000000, 0x01220000,	/* block led, 1 interrupt regs (1 attentions) */
	0x00000b7a, 0x01220001, 0x00000b7c, 0x01230001,	/* block avs_wrap, 1 interrupt regs (2 attentions), 1 parity regs (3 attentions) */
	0x00000000, 0x01240000, 0x00000b7f, 0x01240001,	/* block pxpreqbus, 1 parity regs (20 attentions) */
	0x00000000, 0x01250000, 0x00000000, 0x01250000,	/* block bar0_map */
	0x00000000, 0x01250000, 0x00000000, 0x01250000,	/* block mcp_fio */
	0x00000000, 0x01250000, 0x00000000, 0x01250000,	/* block last_init */
};

/* Data size: 1344 bytes */

/* Blocks debug data */
static const u32 dbg_blocks[] = {
	0x00637267, 0x00000000, 0x00000000, 0x00000000,	/* grc */
	0x6373696d, 0x00000073, 0x00000000, 0x00000000,	/* miscs */
	0x6373696d, 0x00000000, 0x00000000, 0x00000000,	/* misc */
	0x00756264, 0x00000000, 0x00000000, 0x00000000,	/* dbu */
	0x756c6770, 0x00625f65, 0x00000000, 0x00000000,	/* pglue_b */
	0x67696e63, 0x00000000, 0x00000000, 0x00000000,	/* cnig */
	0x756d7063, 0x00000000, 0x00000000, 0x00000000,	/* cpmu */
	0x6973636e, 0x00000000, 0x00000000, 0x00000000,	/* ncsi */
	0x6574706f, 0x00000000, 0x00000000, 0x00000000,	/* opte */
	0x00626d62, 0x00000000, 0x00000000, 0x00000000,	/* bmb */
	0x65696370, 0x00000000, 0x00000000, 0x00000000,	/* pcie */
	0x0070636d, 0x00000000, 0x00000000, 0x00000000,	/* mcp */
	0x3270636d, 0x00000000, 0x00000000, 0x00000000,	/* mcp2 */
	0x68777370, 0x00007473, 0x00000000, 0x00000000,	/* pswhst */
	0x68777370, 0x00327473, 0x00000000, 0x00000000,	/* pswhst2 */
	0x72777370, 0x00000064, 0x00000000, 0x00000000,	/* pswrd */
	0x72777370, 0x00003264, 0x00000000, 0x00000000,	/* pswrd2 */
	0x77777370, 0x00000072, 0x00000000, 0x00000000,	/* pswwr */
	0x77777370, 0x00003272, 0x00000000, 0x00000000,	/* pswwr2 */
	0x72777370, 0x00000071, 0x00000000, 0x00000000,	/* pswrq */
	0x72777370, 0x00003271, 0x00000000, 0x00000000,	/* pswrq2 */
	0x636c6770, 0x00000073, 0x00000000, 0x00000000,	/* pglcs */
	0x65616d64, 0x00000000, 0x00000000, 0x00000000,	/* dmae */
	0x00757470, 0x00000000, 0x00000000, 0x00000000,	/* ptu */
	0x006d6374, 0x00000000, 0x00000000, 0x54000000,	/* tcm */
	0x006d636d, 0x00000000, 0x00000000, 0x4d000000,	/* mcm */
	0x006d6375, 0x00000000, 0x00000000, 0x55000000,	/* ucm */
	0x006d6378, 0x00000000, 0x00000000, 0x58000000,	/* xcm */
	0x006d6379, 0x00000000, 0x00000000, 0x59000000,	/* ycm */
	0x006d6370, 0x00000000, 0x00000000, 0x50000000,	/* pcm */
	0x00006d71, 0x00000000, 0x00000000, 0x00000000,	/* qm */
	0x00006d74, 0x00000000, 0x00000000, 0x00000000,	/* tm */
	0x71726f64, 0x00000000, 0x00000000, 0x00000000,	/* dorq */
	0x00627262, 0x00000000, 0x00000000, 0x00000000,	/* brb */
	0x00637273, 0x00000000, 0x00000000, 0x00000000,	/* src */
	0x00737270, 0x00000000, 0x00000000, 0x00000000,	/* prs */
	0x6d647374, 0x00000000, 0x00000000, 0x54000000,	/* tsdm */
	0x6d64736d, 0x00000000, 0x00000000, 0x4d000000,	/* msdm */
	0x6d647375, 0x00000000, 0x00000000, 0x55000000,	/* usdm */
	0x6d647378, 0x00000000, 0x00000000, 0x58000000,	/* xsdm */
	0x6d647379, 0x00000000, 0x00000000, 0x59000000,	/* ysdm */
	0x6d647370, 0x00000000, 0x00000000, 0x50000000,	/* psdm */
	0x6d657374, 0x00000000, 0x00000000, 0x54000000,	/* tsem */
	0x6d65736d, 0x00000000, 0x00000000, 0x4d000000,	/* msem */
	0x6d657375, 0x00000000, 0x00000000, 0x55000000,	/* usem */
	0x6d657378, 0x00000000, 0x00000000, 0x58000000,	/* xsem */
	0x6d657379, 0x00000000, 0x00000000, 0x59000000,	/* ysem */
	0x6d657370, 0x00000000, 0x00000000, 0x50000000,	/* psem */
	0x00737372, 0x00000000, 0x00000000, 0x00000000,	/* rss */
	0x646c6d74, 0x00000000, 0x00000000, 0x00000000,	/* tmld */
	0x646c756d, 0x00000000, 0x00000000, 0x00000000,	/* muld */
	0x646c7579, 0x00000000, 0x00000000, 0x00000000,	/* yuld */
	0x646c7978, 0x00000000, 0x00000000, 0x00000000,	/* xyld */
	0x006d7270, 0x00000000, 0x00000000, 0x00000000,	/* prm */
	0x5f666270, 0x00316270, 0x00000000, 0x00000000,	/* pbf_pb1 */
	0x5f666270, 0x00326270, 0x00000000, 0x00000000,	/* pbf_pb2 */
	0x00627072, 0x00000000, 0x00000000, 0x00000000,	/* rpb */
	0x00627462, 0x00000000, 0x00000000, 0x00000000,	/* btb */
	0x00666270, 0x00000000, 0x00000000, 0x00000000,	/* pbf */
	0x66696472, 0x00000000, 0x00000000, 0x00000000,	/* rdif */
	0x66696474, 0x00000000, 0x00000000, 0x00000000,	/* tdif */
	0x00756463, 0x00000000, 0x00000000, 0x00000000,	/* cdu */
	0x63666363, 0x00000000, 0x00000000, 0x00000000,	/* ccfc */
	0x63666374, 0x00000000, 0x00000000, 0x00000000,	/* tcfc */
	0x00756769, 0x00000000, 0x00000000, 0x00000000,	/* igu */
	0x00756163, 0x00000000, 0x00000000, 0x00000000,	/* cau */
	0x63616d75, 0x00000000, 0x00000000, 0x00000000,	/* umac */
	0x63616d78, 0x00000000, 0x00000000, 0x00000000,	/* xmac */
	0x6174736d, 0x00000074, 0x00000000, 0x00000000,	/* mstat */
	0x00676264, 0x00000000, 0x00000000, 0x00000000,	/* dbg */
	0x0067696e, 0x00000000, 0x00000000, 0x00000000,	/* nig */
	0x006c6f77, 0x00000000, 0x00000000, 0x00000000,	/* wol */
	0x6e626d62, 0x00000000, 0x00000000, 0x00000000,	/* bmbn */
	0x00637069, 0x00000000, 0x00000000, 0x00000000,	/* ipc */
	0x006d776e, 0x00000000, 0x00000000, 0x00000000,	/* nwm */
	0x0073776e, 0x00000000, 0x00000000, 0x00000000,	/* nws */
	0x0000736d, 0x00000000, 0x00000000, 0x00000000,	/* ms */
	0x5f796870, 0x65696370, 0x00000000, 0x00000000,	/* phy_pcie */
	0x0064656c, 0x00000000, 0x00000000, 0x00000000,	/* led */
	0x5f737661, 0x70617277, 0x00000000, 0x00000000,	/* avs_wrap */
	0x72707870, 0x75627165, 0x00000073, 0x00000000,	/* pxpreqbus */
	0x30726162, 0x70616d5f, 0x00000000, 0x00000000,	/* bar0_map */
	0x5f70636d, 0x006f6966, 0x00000000, 0x00000000,	/* mcp_fio */
	0x7473616c, 0x696e695f, 0x00000074, 0x00000000,	/* last_init */
	0x5f737270, 0x00006366, 0x00000000, 0x00000000,	/* prs_fc */
	0x5f666270, 0x00006366, 0x00000000, 0x00000000,	/* pbf_fc */
	0x5f67696e, 0x665f626c, 0x00000063, 0x00000000,	/* nig_lb_fc */
	0x5f67696e, 0x665f626c, 0x6c705f63, 0x0000686c,	/* nig_lb_fc_pllh */
	0x5f67696e, 0x665f7874, 0x6c705f63, 0x0000686c,	/* nig_tx_fc_pllh */
	0x5f67696e, 0x665f7874, 0x00000063, 0x00000000,	/* nig_tx_fc */
	0x5f67696e, 0x665f7872, 0x6c705f63, 0x0000686c,	/* nig_rx_fc_pllh */
	0x5f67696e, 0x665f7872, 0x00000063, 0x00000000,	/* nig_rx_fc */
};

/* Data size: 1472 bytes */

/* Blocks debug data per chip */
static const u32 dbg_blocks_per_chip[] = {
	0x0100000a, 0x00000000, 0x00000f00, 0x00014029, 0x0001402a, 0x0001402b,
	0x0001402c, 0x0001402d,	/* grc, bb, misc.reset_pl_ua[1], rbcn, 15 lines */
	0x0100000a, 0x00000000, 0x00000f00, 0x00014029, 0x0001402a, 0x0001402b,
	0x0001402c, 0x0001402d,	/* grc, k2, misc.reset_pl_ua[1], rbcn, 15 lines */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* miscs, bb */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* miscs, k2 */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* misc, bb */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* misc, k2 */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* dbu, bb */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* dbu, k2 */
	0x0101081a, 0x00000000, 0x000f2700, 0x000aa100, 0x000aa101, 0x000aa102,
	0x000aa103, 0x000aa104,	/* pglue_b, bb, miscs.reset_pl_hv[1], rbch, 39 lines */
	0x0101081a, 0x00000000, 0x00362a00, 0x000aa100, 0x000aa101, 0x000aa102,
	0x000aa103, 0x000aa104,	/* pglue_b, k2, miscs.reset_pl_hv[1], rbch, 42 lines */
	0x00010002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* cnig, bb, miscs.reset_pl_hv[0] */
	0x0001111a, 0x00000041, 0x00120200, 0x00086095, 0x00086096, 0x00086097,
	0x00086098, 0x00086099,	/* cnig, k2, miscs.reset_pl_hv[0], rbcw, 2 lines, mode k2 */
	0x08010002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* cpmu, bb, miscs.reset_pl_hv[8] */
	0x08010002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* cpmu, k2, miscs.reset_pl_hv[8] */
	0x0501091a, 0x000000d9, 0x00000100, 0x0001011d, 0x0001011e, 0x0001011f,
	0x00010120, 0x00010121,	/* ncsi, bb, miscs.reset_pl_hv[5], rbcz, 1 lines, mode !(!asic) */
	0x0501091a, 0x000000d9, 0x00000100, 0x0001011d, 0x0001011e, 0x0001011f,
	0x00010120, 0x00010121,	/* ncsi, k2, miscs.reset_pl_hv[5], rbcz, 1 lines, mode !(!asic) */
	0x04010002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* opte, bb, miscs.reset_pl_hv[4] */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* opte, k2 (removed) */
	0x0702090a, 0x000000d9, 0x00606a00, 0x0015029f, 0x001502a0, 0x001502a1,
	0x001502a2, 0x001502a3,	/* bmb, bb, miscs.reset_pl_ua[7], rbcz, 106 lines, mode !(!asic) */
	0x0702100a, 0x000000d9, 0x00606a00, 0x0015029f, 0x001502a0, 0x001502a1,
	0x001502a2, 0x001502a3,	/* bmb, k2, miscs.reset_pl_ua[7], rbcb, 106 lines, mode !(!asic) */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* pcie, bb */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* pcie, k2 */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* mcp, bb */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* mcp, k2 */
	0x00000908, 0x00000000, 0x00ca0900, 0x00014900, 0x00014901, 0x00014902,
	0x00014910, 0x00014911,	/* mcp2, bb, rbcz, 9 lines */
	0x00000908, 0x00000000, 0x00ca0900, 0x00014900, 0x00014901, 0x00014902,
	0x00014910, 0x00014911,	/* mcp2, k2, rbcz, 9 lines */
	0x0003011a, 0x00000000, 0x00d30400, 0x000a8040, 0x000a8041, 0x000a8042,
	0x000a8043, 0x000a8044,	/* pswhst, bb, misc.reset_pl_hv[0], rbcp, 4 lines */
	0x0003011a, 0x00000000, 0x00d30400, 0x000a8040, 0x000a8041, 0x000a8042,
	0x000a8043, 0x000a8044,	/* pswhst, k2, misc.reset_pl_hv[0], rbcp, 4 lines */
	0x0003011a, 0x00000000, 0x00d70300, 0x000a7816, 0x000a7817, 0x000a7818,
	0x000a7819, 0x000a781a,	/* pswhst2, bb, misc.reset_pl_hv[0], rbcp, 3 lines */
	0x0003011a, 0x00000000, 0x00d70300, 0x000a7816, 0x000a7817, 0x000a7818,
	0x000a7819, 0x000a781a,	/* pswhst2, k2, misc.reset_pl_hv[0], rbcp, 3 lines */
	0x0203011a, 0x00000000, 0x00340100, 0x000a7010, 0x000a7011, 0x000a7012,
	0x000a7013, 0x000a7014,	/* pswrd, bb, misc.reset_pl_hv[2], rbcp, 1 lines */
	0x0203011a, 0x00000000, 0x00340100, 0x000a7010, 0x000a7011, 0x000a7012,
	0x000a7013, 0x000a7014,	/* pswrd, k2, misc.reset_pl_hv[2], rbcp, 1 lines */
	0x0203011a, 0x00000000, 0x00da1900, 0x000a7500, 0x000a7501, 0x000a7502,
	0x000a7503, 0x000a7504,	/* pswrd2, bb, misc.reset_pl_hv[2], rbcp, 25 lines */
	0x0203011a, 0x00000000, 0x00da1900, 0x000a7500, 0x000a7501, 0x000a7502,
	0x000a7503, 0x000a7504,	/* pswrd2, k2, misc.reset_pl_hv[2], rbcp, 25 lines */
	0x0303011a, 0x00000000, 0x00f30900, 0x000a6821, 0x000a6822, 0x000a6823,
	0x000a6824, 0x000a6825,	/* pswwr, bb, misc.reset_pl_hv[3], rbcp, 9 lines */
	0x0303011a, 0x00000000, 0x00f30900, 0x000a6821, 0x000a6822, 0x000a6823,
	0x000a6824, 0x000a6825,	/* pswwr, k2, misc.reset_pl_hv[3], rbcp, 9 lines */
	0x03030002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* pswwr2, bb, misc.reset_pl_hv[3] */
	0x03030002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* pswwr2, k2, misc.reset_pl_hv[3] */
	0x0103010a, 0x00000000, 0x001c0100, 0x000a0008, 0x000a0009, 0x000a000a,
	0x000a000b, 0x000a000c,	/* pswrq, bb, misc.reset_pl_hv[1], rbcp, 1 lines */
	0x0103010a, 0x00000000, 0x001c0100, 0x000a0008, 0x000a0009, 0x000a000a,
	0x000a000b, 0x000a000c,	/* pswrq, k2, misc.reset_pl_hv[1], rbcp, 1 lines */
	0x0103010a, 0x00000000, 0x00fc1500, 0x00090040, 0x00090041, 0x00090042,
	0x00090043, 0x00090044,	/* pswrq2, bb, misc.reset_pl_hv[1], rbcp, 21 lines */
	0x0103010a, 0x00000000, 0x00fc1500, 0x00090040, 0x00090041, 0x00090042,
	0x00090043, 0x00090044,	/* pswrq2, k2, misc.reset_pl_hv[1], rbcp, 21 lines */
	0x02010002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* pglcs, bb, miscs.reset_pl_hv[2] */
	0x0201080a, 0x00000041, 0x00120600, 0x00000745, 0x00000746, 0x00000747,
	0x00000748, 0x00000749,	/* pglcs, k2, miscs.reset_pl_hv[2], rbch, 6 lines, mode k2 */
	0x1c04010e, 0x00000000, 0x00100100, 0x00003144, 0x00003145, 0x00003146,
	0x00003147, 0x00003148,	/* dmae, bb, misc.reset_pl_pda_vmain_1[28] (unreset), rbcp, 1 lines */
	0x1c04010e, 0x00000000, 0x00100100, 0x00003144, 0x00003145, 0x00003146,
	0x00003147, 0x00003148,	/* dmae, k2, misc.reset_pl_pda_vmain_1[28] (unreset), rbcp, 1 lines */
	0x1405011e, 0x00000000, 0x01110500, 0x00158040, 0x00158041, 0x00158042,
	0x00158043, 0x00158044,	/* ptu, bb, misc.reset_pl_pda_vmain_2[20] (unreset), rbcp, 5 lines */
	0x1405011e, 0x00000000, 0x01110500, 0x00158040, 0x00158041, 0x00158042,
	0x00158043, 0x00158044,	/* ptu, k2, misc.reset_pl_pda_vmain_2[20] (unreset), rbcp, 5 lines */
	0x0504031e, 0x00000000, 0x01161300, 0x00460010, 0x00460011, 0x00460012,
	0x00460013, 0x00460014,	/* tcm, bb, misc.reset_pl_pda_vmain_1[5] (unreset), rbct, 19 lines */
	0x0504031e, 0x00000000, 0x01161300, 0x00460010, 0x00460011, 0x00460012,
	0x00460013, 0x00460014,	/* tcm, k2, misc.reset_pl_pda_vmain_1[5] (unreset), rbct, 19 lines */
	0x0305031e, 0x00000000, 0x01291300, 0x00480010, 0x00480011, 0x00480012,
	0x00480013, 0x00480014,	/* mcm, bb, misc.reset_pl_pda_vmain_2[3] (unreset), rbct, 19 lines */
	0x03050f1e, 0x00000000, 0x01291300, 0x00480010, 0x00480011, 0x00480012,
	0x00480013, 0x00480014,	/* mcm, k2, misc.reset_pl_pda_vmain_2[3] (unreset), rbcm, 19 lines */
	0x0804041e, 0x00000000, 0x013c1600, 0x004a0014, 0x004a0015, 0x004a0016,
	0x004a0017, 0x004a0018,	/* ucm, bb, misc.reset_pl_pda_vmain_1[8] (unreset), rbcu, 22 lines */
	0x0804041e, 0x00000000, 0x013c1600, 0x004a0014, 0x004a0015, 0x004a0016,
	0x004a0017, 0x004a0018,	/* ucm, k2, misc.reset_pl_pda_vmain_1[8] (unreset), rbcu, 22 lines */
	0x1304061e, 0x00000000, 0x01521400, 0x00400010, 0x00400011, 0x00400012,
	0x00400013, 0x00400014,	/* xcm, bb, misc.reset_pl_pda_vmain_1[19] (unreset), rbcx, 20 lines */
	0x1304061e, 0x00000000, 0x01521400, 0x00400010, 0x00400011, 0x00400012,
	0x00400013, 0x00400014,	/* xcm, k2, misc.reset_pl_pda_vmain_1[19] (unreset), rbcx, 20 lines */
	0x0505061e, 0x00000000, 0x012a1200, 0x00420010, 0x00420011, 0x00420012,
	0x00420013, 0x00420014,	/* ycm, bb, misc.reset_pl_pda_vmain_2[5] (unreset), rbcx, 18 lines */
	0x05050d1e, 0x00000000, 0x012a1200, 0x00420010, 0x00420011, 0x00420012,
	0x00420013, 0x00420014,	/* ycm, k2, misc.reset_pl_pda_vmain_2[5] (unreset), rbcy, 18 lines */
	0x0405071e, 0x00000000, 0x01660a00, 0x00440010, 0x00440011, 0x00440012,
	0x00440013, 0x00440014,	/* pcm, bb, misc.reset_pl_pda_vmain_2[4] (unreset), rbcs, 10 lines */
	0x0405071e, 0x00000000, 0x01660a00, 0x00440010, 0x00440011, 0x00440012,
	0x00440013, 0x00440014,	/* pcm, k2, misc.reset_pl_pda_vmain_2[4] (unreset), rbcs, 10 lines */
	0x1004010e, 0x00000000, 0x01706200, 0x000bcb9d, 0x000bcb9e, 0x000bcb9f,
	0x000bcba0, 0x000bcba1,	/* qm, bb, misc.reset_pl_pda_vmain_1[16] (unreset), rbcp, 98 lines */
	0x10040e0e, 0x00000000, 0x01706200, 0x000bcb9d, 0x000bcb9e, 0x000bcb9f,
	0x000bcba0, 0x000bcba1,	/* qm, k2, misc.reset_pl_pda_vmain_1[16] (unreset), rbcq, 98 lines */
	0x1104070e, 0x00000000, 0x01d22100, 0x000b01ea, 0x000b01eb, 0x000b01ec,
	0x000b01ed, 0x000b01ee,	/* tm, bb, misc.reset_pl_pda_vmain_1[17] (unreset), rbcs, 33 lines */
	0x1104070e, 0x00000000, 0x01d22100, 0x000b01ea, 0x000b01eb, 0x000b01ec,
	0x000b01ed, 0x000b01ee,	/* tm, k2, misc.reset_pl_pda_vmain_1[17] (unreset), rbcs, 33 lines */
	0x1204061e, 0x00000000, 0x01f30700, 0x000402b4, 0x000402b5, 0x000402b6,
	0x000402b7, 0x000402b8,	/* dorq, bb, misc.reset_pl_pda_vmain_1[18] (unreset), rbcx, 7 lines */
	0x12040d1e, 0x00000000, 0x01f30700, 0x000402b4, 0x000402b5, 0x000402b6,
	0x000402b7, 0x000402b8,	/* dorq, k2, misc.reset_pl_pda_vmain_1[18] (unreset), rbcy, 7 lines */
	0x0004021e, 0x00000000, 0x01fa6200, 0x000d03b4, 0x000d03b5, 0x000d03b6,
	0x000d03b7, 0x000d03b8,	/* brb, bb, misc.reset_pl_pda_vmain_1[0] (unreset), rbcr, 98 lines */
	0x0004021e, 0x00000000, 0x01fa6200, 0x000d03b4, 0x000d03b5, 0x000d03b6,
	0x000d03b7, 0x000d03b8,	/* brb, k2, misc.reset_pl_pda_vmain_1[0] (unreset), rbcr, 98 lines */
	0x0204050e, 0x00000000, 0x025c1900, 0x0008e1c0, 0x0008e1c1, 0x0008e1c2,
	0x0008e1c3, 0x0008e1c4,	/* src, bb, misc.reset_pl_pda_vmain_1[2] (unreset), rbcf, 25 lines */
	0x0204050e, 0x00000000, 0x02751a00, 0x0008e1c0, 0x0008e1c1, 0x0008e1c2,
	0x0008e1c3, 0x0008e1c4,	/* src, k2, misc.reset_pl_pda_vmain_1[2] (unreset), rbcf, 26 lines */
	0x0104021e, 0x00000000, 0x028f0400, 0x0007c2db, 0x0007c2dc, 0x0007c2dd,
	0x0007c2e8, 0x0007c2e9,	/* prs, bb, misc.reset_pl_pda_vmain_1[1] (unreset), rbcr, 4 lines */
	0x0104021e, 0x00000000, 0x028f0400, 0x0007c2db, 0x0007c2dc, 0x0007c2dd,
	0x0007c2e8, 0x0007c2e9,	/* prs, k2, misc.reset_pl_pda_vmain_1[1] (unreset), rbcr, 4 lines */
	0x0304031e, 0x00000000, 0x02933300, 0x003ec38a, 0x003ec38b, 0x003ec38c,
	0x003ec38d, 0x003ec38e,	/* tsdm, bb, misc.reset_pl_pda_vmain_1[3] (unreset), rbct, 51 lines */
	0x0304031e, 0x00000000, 0x02933300, 0x003ec38a, 0x003ec38b, 0x003ec38c,
	0x003ec38d, 0x003ec38e,	/* tsdm, k2, misc.reset_pl_pda_vmain_1[3] (unreset), rbct, 51 lines */
	0x0605031e, 0x00000000, 0x02933300, 0x003f038a, 0x003f038b, 0x003f038c,
	0x003f038d, 0x003f038e,	/* msdm, bb, misc.reset_pl_pda_vmain_2[6] (unreset), rbct, 51 lines */
	0x06050f1e, 0x00000000, 0x02933300, 0x003f038a, 0x003f038b, 0x003f038c,
	0x003f038d, 0x003f038e,	/* msdm, k2, misc.reset_pl_pda_vmain_2[6] (unreset), rbcm, 51 lines */
	0x0704041e, 0x00000000, 0x02933300, 0x003f438a, 0x003f438b, 0x003f438c,
	0x003f438d, 0x003f438e,	/* usdm, bb, misc.reset_pl_pda_vmain_1[7] (unreset), rbcu, 51 lines */
	0x0704041e, 0x00000000, 0x02933300, 0x003f438a, 0x003f438b, 0x003f438c,
	0x003f438d, 0x003f438e,	/* usdm, k2, misc.reset_pl_pda_vmain_1[7] (unreset), rbcu, 51 lines */
	0x1404061e, 0x00000000, 0x02933300, 0x003e038a, 0x003e038b, 0x003e038c,
	0x003e038d, 0x003e038e,	/* xsdm, bb, misc.reset_pl_pda_vmain_1[20] (unreset), rbcx, 51 lines */
	0x1404061e, 0x00000000, 0x02933300, 0x003e038a, 0x003e038b, 0x003e038c,
	0x003e038d, 0x003e038e,	/* xsdm, k2, misc.reset_pl_pda_vmain_1[20] (unreset), rbcx, 51 lines */
	0x0805061e, 0x00000000, 0x02933300, 0x003e438a, 0x003e438b, 0x003e438c,
	0x003e438d, 0x003e438e,	/* ysdm, bb, misc.reset_pl_pda_vmain_2[8] (unreset), rbcx, 51 lines */
	0x08050d1e, 0x00000000, 0x02933300, 0x003e438a, 0x003e438b, 0x003e438c,
	0x003e438d, 0x003e438e,	/* ysdm, k2, misc.reset_pl_pda_vmain_2[8] (unreset), rbcy, 51 lines */
	0x0705071e, 0x00000000, 0x02933300, 0x003e838a, 0x003e838b, 0x003e838c,
	0x003e838d, 0x003e838e,	/* psdm, bb, misc.reset_pl_pda_vmain_2[7] (unreset), rbcs, 51 lines */
	0x0705071e, 0x00000000, 0x02933300, 0x003e838a, 0x003e838b, 0x003e838c,
	0x003e838d, 0x003e838e,	/* psdm, k2, misc.reset_pl_pda_vmain_2[7] (unreset), rbcs, 51 lines */
	0x0404031e, 0x00000000, 0x02c60c00, 0x005c054a, 0x005c054b, 0x005c054c,
	0x005c054d, 0x005c054e,	/* tsem, bb, misc.reset_pl_pda_vmain_1[4] (unreset), rbct, 12 lines */
	0x0404031e, 0x00000000, 0x02c60c00, 0x005c054a, 0x005c054b, 0x005c054c,
	0x005c054d, 0x005c054e,	/* tsem, k2, misc.reset_pl_pda_vmain_1[4] (unreset), rbct, 12 lines */
	0x0905031e, 0x00000000, 0x02c60c00, 0x0060054a, 0x0060054b, 0x0060054c,
	0x0060054d, 0x0060054e,	/* msem, bb, misc.reset_pl_pda_vmain_2[9] (unreset), rbct, 12 lines */
	0x09050f1e, 0x00000000, 0x02c60c00, 0x0060054a, 0x0060054b, 0x0060054c,
	0x0060054d, 0x0060054e,	/* msem, k2, misc.reset_pl_pda_vmain_2[9] (unreset), rbcm, 12 lines */
	0x0904041e, 0x00000000, 0x02c60c00, 0x0064054a, 0x0064054b, 0x0064054c,
	0x0064054d, 0x0064054e,	/* usem, bb, misc.reset_pl_pda_vmain_1[9] (unreset), rbcu, 12 lines */
	0x0904041e, 0x00000000, 0x02c60c00, 0x0064054a, 0x0064054b, 0x0064054c,
	0x0064054d, 0x0064054e,	/* usem, k2, misc.reset_pl_pda_vmain_1[9] (unreset), rbcu, 12 lines */
	0x1504061e, 0x00000000, 0x02c60c00, 0x0050054a, 0x0050054b, 0x0050054c,
	0x0050054d, 0x0050054e,	/* xsem, bb, misc.reset_pl_pda_vmain_1[21] (unreset), rbcx, 12 lines */
	0x1504061e, 0x00000000, 0x02c60c00, 0x0050054a, 0x0050054b, 0x0050054c,
	0x0050054d, 0x0050054e,	/* xsem, k2, misc.reset_pl_pda_vmain_1[21] (unreset), rbcx, 12 lines */
	0x0b05061e, 0x00000000, 0x02c60c00, 0x0054054a, 0x0054054b, 0x0054054c,
	0x0054054d, 0x0054054e,	/* ysem, bb, misc.reset_pl_pda_vmain_2[11] (unreset), rbcx, 12 lines */
	0x0b050d1e, 0x00000000, 0x02c60c00, 0x0054054a, 0x0054054b, 0x0054054c,
	0x0054054d, 0x0054054e,	/* ysem, k2, misc.reset_pl_pda_vmain_2[11] (unreset), rbcy, 12 lines */
	0x0a05071e, 0x00000000, 0x02c60c00, 0x0058054a, 0x0058054b, 0x0058054c,
	0x0058054d, 0x0058054e,	/* psem, bb, misc.reset_pl_pda_vmain_2[10] (unreset), rbcs, 12 lines */
	0x0a05071e, 0x00000000, 0x02c60c00, 0x0058054a, 0x0058054b, 0x0058054c,
	0x0058054d, 0x0058054e,	/* psem, k2, misc.reset_pl_pda_vmain_2[10] (unreset), rbcs, 12 lines */
	0x1205030e, 0x00000000, 0x02d20d00, 0x0008e313, 0x0008e314, 0x0008e315,
	0x0008e316, 0x0008e317,	/* rss, bb, misc.reset_pl_pda_vmain_2[18] (unreset), rbct, 13 lines */
	0x1205030e, 0x00000000, 0x02d20d00, 0x0008e313, 0x0008e314, 0x0008e315,
	0x0008e316, 0x0008e317,	/* rss, k2, misc.reset_pl_pda_vmain_2[18] (unreset), rbct, 13 lines */
	0x0d05031e, 0x00000000, 0x02df0600, 0x00134580, 0x00134581, 0x00134582,
	0x00134583, 0x00134584,	/* tmld, bb, misc.reset_pl_pda_vmain_2[13] (unreset), rbct, 6 lines */
	0x0d050f1e, 0x00000000, 0x02df0600, 0x00134580, 0x00134581, 0x00134582,
	0x00134583, 0x00134584,	/* tmld, k2, misc.reset_pl_pda_vmain_2[13] (unreset), rbcm, 6 lines */
	0x0e05041e, 0x00000000, 0x02e50600, 0x00138580, 0x00138581, 0x00138582,
	0x00138583, 0x00138584,	/* muld, bb, misc.reset_pl_pda_vmain_2[14] (unreset), rbcu, 6 lines */
	0x0e05041e, 0x00000000, 0x02e50600, 0x00138580, 0x00138581, 0x00138582,
	0x00138583, 0x00138584,	/* muld, k2, misc.reset_pl_pda_vmain_2[14] (unreset), rbcu, 6 lines */
	0x0f05040e, 0x00000000, 0x02df0500, 0x00132580, 0x00132581, 0x00132582,
	0x00132583, 0x00132584,	/* yuld, bb, misc.reset_pl_pda_vmain_2[15] (unreset), rbcu, 5 lines */
	0x0f05040e, 0x00000000, 0x02df0500, 0x00132580, 0x00132581, 0x00132582,
	0x00132583, 0x00132584,	/* yuld, k2, misc.reset_pl_pda_vmain_2[15] (unreset), rbcu, 5 lines */
	0x0c05061e, 0x00000000, 0x02eb0700, 0x00130580, 0x00130581, 0x00130582,
	0x00130583, 0x00130584,	/* xyld, bb, misc.reset_pl_pda_vmain_2[12] (unreset), rbcx, 7 lines */
	0x0c05061e, 0x00000000, 0x02e50700, 0x00130580, 0x00130581, 0x00130582,
	0x00130583, 0x00130584,	/* xyld, k2, misc.reset_pl_pda_vmain_2[12] (unreset), rbcx, 7 lines */
	0x1505031e, 0x00000000, 0x02f20e00, 0x0008c1aa, 0x0008c1ab, 0x0008c1ac,
	0x0008c1ad, 0x0008c1ae,	/* prm, bb, misc.reset_pl_pda_vmain_2[21] (unreset), rbct, 14 lines */
	0x15050f1e, 0x00000000, 0x03001000, 0x0008c1aa, 0x0008c1ab, 0x0008c1ac,
	0x0008c1ad, 0x0008c1ae,	/* prm, k2, misc.reset_pl_pda_vmain_2[21] (unreset), rbcm, 16 lines */
	0x0b04070e, 0x00000000, 0x03100d00, 0x003681ca, 0x003681cb, 0x003681cc,
	0x003681cd, 0x003681ce,	/* pbf_pb1, bb, misc.reset_pl_pda_vmain_1[11] (unreset), rbcs, 13 lines */
	0x0b04120e, 0x00000000, 0x03100d00, 0x003681ca, 0x003681cb, 0x003681cc,
	0x003681cd, 0x003681ce,	/* pbf_pb1, k2, misc.reset_pl_pda_vmain_1[11] (unreset), rbcv, 13 lines */
	0x0c04070e, 0x00000000, 0x03100d00, 0x003691ca, 0x003691cb, 0x003691cc,
	0x003691cd, 0x003691ce,	/* pbf_pb2, bb, misc.reset_pl_pda_vmain_1[12] (unreset), rbcs, 13 lines */
	0x0c04120e, 0x00000000, 0x03100d00, 0x003691ca, 0x003691cb, 0x003691cc,
	0x003691cd, 0x003691ce,	/* pbf_pb2, k2, misc.reset_pl_pda_vmain_1[12] (unreset), rbcv, 13 lines */
	0x0d04030e, 0x00000000, 0x03100d00, 0x0008f1ca, 0x0008f1cb, 0x0008f1cc,
	0x0008f1cd, 0x0008f1ce,	/* rpb, bb, misc.reset_pl_pda_vmain_1[13] (unreset), rbct, 13 lines */
	0x0d040f0e, 0x00000000, 0x03100d00, 0x0008f1ca, 0x0008f1cb, 0x0008f1cc,
	0x0008f1cd, 0x0008f1ce,	/* rpb, k2, misc.reset_pl_pda_vmain_1[13] (unreset), rbcm, 13 lines */
	0x0a04021e, 0x00000000, 0x031d4000, 0x0036c232, 0x0036c233, 0x0036c234,
	0x0036c235, 0x0036c236,	/* btb, bb, misc.reset_pl_pda_vmain_1[10] (unreset), rbcr, 64 lines */
	0x0a04121e, 0x00000000, 0x031d4000, 0x0036c232, 0x0036c233, 0x0036c234,
	0x0036c235, 0x0036c236,	/* btb, k2, misc.reset_pl_pda_vmain_1[10] (unreset), rbcv, 64 lines */
	0x0f04071e, 0x00000000, 0x035d1700, 0x00360018, 0x00360019, 0x0036001a,
	0x0036001b, 0x0036001c,	/* pbf, bb, misc.reset_pl_pda_vmain_1[15] (unreset), rbcs, 23 lines */
	0x0f04121e, 0x00000000, 0x035d1700, 0x00360018, 0x00360019, 0x0036001a,
	0x0036001b, 0x0036001c,	/* pbf, k2, misc.reset_pl_pda_vmain_1[15] (unreset), rbcv, 23 lines */
	0x1005030e, 0x00000000, 0x03740600, 0x000c0140, 0x000c0141, 0x000c0142,
	0x000c0143, 0x000c0144,	/* rdif, bb, misc.reset_pl_pda_vmain_2[16] (unreset), rbct, 6 lines */
	0x10050f0e, 0x00000000, 0x03740600, 0x000c0140, 0x000c0141, 0x000c0142,
	0x000c0143, 0x000c0144,	/* rdif, k2, misc.reset_pl_pda_vmain_2[16] (unreset), rbcm, 6 lines */
	0x1105070e, 0x00000000, 0x037a0600, 0x000c4140, 0x000c4141, 0x000c4142,
	0x000c4143, 0x000c4144,	/* tdif, bb, misc.reset_pl_pda_vmain_2[17] (unreset), rbcs, 6 lines */
	0x1105070e, 0x00000000, 0x037a0600, 0x000c4140, 0x000c4141, 0x000c4142,
	0x000c4143, 0x000c4144,	/* tdif, k2, misc.reset_pl_pda_vmain_2[17] (unreset), rbcs, 6 lines */
	0x1704050e, 0x00000000, 0x03800300, 0x001601c1, 0x001601c2, 0x001601c3,
	0x001601c4, 0x001601c5,	/* cdu, bb, misc.reset_pl_pda_vmain_1[23] (unreset), rbcf, 3 lines */
	0x1704050e, 0x00000000, 0x03830e00, 0x001601c1, 0x001601c2, 0x001601c3,
	0x001601c4, 0x001601c5,	/* cdu, k2, misc.reset_pl_pda_vmain_1[23] (unreset), rbcf, 14 lines */
	0x1804051e, 0x00000000, 0x03910d00, 0x000b8140, 0x000b8141, 0x000b8142,
	0x000b8143, 0x000b8144,	/* ccfc, bb, misc.reset_pl_pda_vmain_1[24] (unreset), rbcf, 13 lines */
	0x1804051e, 0x00000000, 0x039e1700, 0x000b8140, 0x000b8141, 0x000b8142,
	0x000b8143, 0x000b8144,	/* ccfc, k2, misc.reset_pl_pda_vmain_1[24] (unreset), rbcf, 23 lines */
	0x1904051e, 0x00000000, 0x03910d00, 0x000b4140, 0x000b4141, 0x000b4142,
	0x000b4143, 0x000b4144,	/* tcfc, bb, misc.reset_pl_pda_vmain_1[25] (unreset), rbcf, 13 lines */
	0x1904051e, 0x00000000, 0x039e1700, 0x000b4140, 0x000b4141, 0x000b4142,
	0x000b4143, 0x000b4144,	/* tcfc, k2, misc.reset_pl_pda_vmain_1[25] (unreset), rbcf, 23 lines */
	0x1b04011e, 0x00000000, 0x03b53300, 0x0006055e, 0x0006055f, 0x00060560,
	0x00060561, 0x00060562,	/* igu, bb, misc.reset_pl_pda_vmain_1[27] (unreset), rbcp, 51 lines */
	0x1b04011e, 0x00000000, 0x03b53300, 0x0006055e, 0x0006055f, 0x00060560,
	0x00060561, 0x00060562,	/* igu, k2, misc.reset_pl_pda_vmain_1[27] (unreset), rbcp, 51 lines */
	0x1305011e, 0x00000000, 0x03e80600, 0x000703aa, 0x000703ab, 0x000703ac,
	0x000703ad, 0x000703ae,	/* cau, bb, misc.reset_pl_pda_vmain_2[19] (unreset), rbcp, 6 lines */
	0x1305011e, 0x00000000, 0x03e80600, 0x000703aa, 0x000703ab, 0x000703ac,
	0x000703ad, 0x000703ae,	/* cau, k2, misc.reset_pl_pda_vmain_2[19] (unreset), rbcp, 6 lines */
	0x06010002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* umac, bb, miscs.reset_pl_hv[6] */
	0x0601090a, 0x0000019b, 0x00120600, 0x00014425, 0x00014426, 0x00014427,
	0x00014428, 0x00014429,	/* umac, k2, miscs.reset_pl_hv[6], rbcz, 6 lines, mode k2&(!(!asic)) */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* xmac, bb */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* xmac, k2 (removed) */
	0x07010002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* mstat, bb, miscs.reset_pl_hv[7] */
	0x07010002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* mstat, k2, miscs.reset_pl_hv[7] */
	0x03060006, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* dbg, bb, misc.reset_pl_pda_vaux[3] (unreset) */
	0x03060006, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* dbg, k2, misc.reset_pl_pda_vaux[3] (unreset) */
	0x0006001e, 0x00000000, 0x03ee2b00, 0x00140850, 0x00140851, 0x00140852,
	0x00140853, 0x00140854,	/* nig, bb, misc.reset_pl_pda_vaux[0] (unreset), rbcn, 43 lines */
	0x0006001e, 0x00000000, 0x03ee1d00, 0x00140850, 0x00140851, 0x00140852,
	0x00140853, 0x00140854,	/* nig, k2, misc.reset_pl_pda_vaux[0] (unreset), rbcn, 29 lines */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* wol, bb (removed) */
	0x0706090e, 0x0000019b, 0x001c0200, 0x00180050, 0x00180051, 0x00180052,
	0x00180053, 0x00180054,	/* wol, k2, misc.reset_pl_pda_vaux[7] (unreset), rbcz, 2 lines, mode k2&(!(!asic)) */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* bmbn, bb (removed) */
	0x00001008, 0x00000041, 0x00210800, 0x00184050, 0x00184051, 0x00184052,
	0x00184053, 0x00184054,	/* bmbn, k2, rbcb, 8 lines, mode k2 */
	0x08020002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* ipc, bb, miscs.reset_pl_ua[8] */
	0x08020002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* ipc, k2, miscs.reset_pl_ua[8] */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* nwm, bb (removed) */
	0x0007110a, 0x0000003f, 0x04190b00, 0x0020003b, 0x0020003c, 0x0020003d,
	0x0020003e, 0x0020003f,	/* nwm, k2, miscs.reset_pl_hv_2[0], rbcw, 11 lines, mode k2&(!(!(asic|emul_mac))) */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* nws, bb (removed) */
	0x0c010002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* nws, k2, miscs.reset_pl_hv[12] */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* ms, bb (removed) */
	0x0d010002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* ms, k2, miscs.reset_pl_hv[13] */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* phy_pcie, bb (removed) */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* phy_pcie, k2 */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* led, bb (removed) */
	0x0e010002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* led, k2, miscs.reset_pl_hv[14] */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* avs_wrap, bb (removed) */
	0x0b020002, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* avs_wrap, k2, miscs.reset_pl_ua[11] */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* pxpreqbus, bb (removed) */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* pxpreqbus, k2 */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* bar0_map, bb */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* bar0_map, k2 */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* mcp_fio, bb */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* mcp_fio, k2 */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* last_init, bb */
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* last_init, k2 */
	0x00000208, 0x00000000, 0x04240200, 0x0007c2ec, 0x0007c2ed, 0x0007c2ee,
	0x0007c2f8, 0x0007c2f9,	/* prs_fc, bb, rbcr, 2 lines */
	0x00000208, 0x00000000, 0x04240200, 0x0007c2ec, 0x0007c2ed, 0x0007c2ee,
	0x0007c2f8, 0x0007c2f9,	/* prs_fc, k2, rbcr, 2 lines */
	0x00000708, 0x00000000, 0x04240200, 0x0036002a, 0x0036002b, 0x0036002c,
	0x00360038, 0x00360039,	/* pbf_fc, bb, rbcs, 2 lines */
	0x00001208, 0x00000000, 0x04240200, 0x0036002a, 0x0036002b, 0x0036002c,
	0x00360038, 0x00360039,	/* pbf_fc, k2, rbcv, 2 lines */
	0x00000008, 0x00000000, 0x04240200, 0x0014086c, 0x0014086d, 0x0014086e,
	0x0014086f, 0x00140870,	/* nig_lb_fc, bb, rbcn, 2 lines */
	0x00000008, 0x00000000, 0x04240200, 0x0014086c, 0x0014086d, 0x0014086e,
	0x0014086f, 0x00140870,	/* nig_lb_fc, k2, rbcn, 2 lines */
	0x00000008, 0x00000005, 0x04240200, 0x0014087b, 0x0014087c, 0x0014087d,
	0x0014087e, 0x0014087f,	/* nig_lb_fc_pllh, bb, rbcn, 2 lines, mode bb */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* nig_lb_fc_pllh, k2 (removed) */
	0x00000008, 0x00000005, 0x04240200, 0x00140876, 0x00140877, 0x00140878,
	0x00140879, 0x0014087a,	/* nig_tx_fc_pllh, bb, rbcn, 2 lines, mode bb */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* nig_tx_fc_pllh, k2 (removed) */
	0x00000008, 0x00000000, 0x04240200, 0x00140867, 0x00140868, 0x00140869,
	0x0014086a, 0x0014086b,	/* nig_tx_fc, bb, rbcn, 2 lines */
	0x00000008, 0x00000000, 0x04240200, 0x00140867, 0x00140868, 0x00140869,
	0x0014086a, 0x0014086b,	/* nig_tx_fc, k2, rbcn, 2 lines */
	0x00000008, 0x00000005, 0x04240200, 0x00140871, 0x00140872, 0x00140873,
	0x00140874, 0x00140875,	/* nig_rx_fc_pllh, bb, rbcn, 2 lines, mode bb */
	0x00000001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000,	/* nig_rx_fc_pllh, k2 (removed) */
	0x00000008, 0x00000000, 0x04240200, 0x00140862, 0x00140863, 0x00140864,
	0x00140865, 0x00140866,	/* nig_rx_fc, bb, rbcn, 2 lines */
	0x00000008, 0x00000000, 0x04240200, 0x00140862, 0x00140863, 0x00140864,
	0x00140865, 0x00140866,	/* nig_rx_fc, k2, rbcn, 2 lines */
};

/* Data size: 5888 bytes */

/* Debug Bus lines */
static const u16 dbg_bus_lines[] = {
	0x0301, 0x0202, 0x0202, 0x0202, 0x0202, 0x0202, 0x0202, 0x0202, 0x0202,
	0x0202, 0x0202, 0x0301, 0x0103, 0x0004, 0x0004, 0x0301, 0x0502, 0x1003,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0002, 0x0301, 0x0301,
	0x0301, 0x0311, 0x0311, 0x0311, 0x0311, 0x0502, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0502, 0x0301, 0x0301, 0x0003, 0x0301, 0x0101, 0x0102,
	0x0301, 0x0502, 0x1003, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0002, 0x0301, 0x0301, 0x0301, 0x0311, 0x0311, 0x0311, 0x0311, 0x0502,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0502, 0x0301, 0x0301, 0x0003,
	0x0301, 0x0101, 0x0102, 0x0502, 0x0502, 0x0502, 0x0502, 0x0502, 0x0101,
	0x0101, 0x0101, 0x0101, 0x0101, 0x0101, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x1003, 0x1003, 0x1003,
	0x1003, 0x1003, 0x1003, 0x1003, 0x1003, 0x1003, 0x1003, 0x0101, 0x0101,
	0x0101, 0x0101, 0x0101, 0x0101, 0x0101, 0x0101, 0x0101, 0x0101, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0101, 0x0101, 0x0101, 0x0101, 0x0101, 0x0101, 0x0101,
	0x0101, 0x0101, 0x0101, 0x0101, 0x0001, 0x0001, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0001, 0x0301, 0x0301, 0x0502, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0001, 0x0502, 0x0301, 0x0301, 0x0502, 0x0301,
	0x0301, 0x0502, 0x0502, 0x0301, 0x0014, 0x0502, 0x0512, 0x0301, 0x0301,
	0x0512, 0x0502, 0x0402, 0x0502, 0x0001, 0x0311, 0x0311, 0x0502, 0x0301,
	0x0002, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0202,
	0x0502, 0x0502, 0x0502, 0x0502, 0x0502, 0x0502, 0x0301, 0x0311, 0x0311,
	0x0001, 0x0301, 0x0001, 0x0101, 0x0301, 0x0103, 0x0102, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0101, 0x0301, 0x0301, 0x0301, 0x0301, 0x0512, 0x0512,
	0x0502, 0x0301, 0x0311, 0x0512, 0x0512, 0x0502, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0101,
	0x0301, 0x1003, 0x0502, 0x0202, 0x0202, 0x1003, 0x0004, 0x1003, 0x0004,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x1003, 0x0502, 0x0202, 0x0202, 0x1003, 0x0004, 0x1003,
	0x0004, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0101, 0x0301, 0x1003, 0x0502, 0x0202,
	0x0202, 0x1003, 0x0004, 0x1003, 0x0004, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0101, 0x0301,
	0x1003, 0x0502, 0x0502, 0x0202, 0x0202, 0x1003, 0x0004, 0x0301, 0x0301,
	0x0301, 0x0301, 0x1003, 0x0502, 0x0502, 0x0202, 0x0202, 0x0004, 0x0001,
	0x0301, 0x0001, 0x0002, 0x0002, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0101, 0x0301, 0x0301, 0x0101, 0x0001, 0x0301, 0x0502, 0x0502, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0502, 0x0301, 0x0301, 0x0402, 0x0502, 0x0502,
	0x0402, 0x0502, 0x0502, 0x0502, 0x0502, 0x0002, 0x0102, 0x0502, 0x0002,
	0x0102, 0x0502, 0x0002, 0x0102, 0x0502, 0x1003, 0x0301, 0x0301, 0x0301,
	0x0101, 0x0301, 0x0002, 0x0502, 0x0502, 0x0101, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0101, 0x0301, 0x0301, 0x0301, 0x0101, 0x0301, 0x0301, 0x0301,
	0x0101, 0x0301, 0x0301, 0x0301, 0x0101, 0x0301, 0x0301, 0x0301, 0x0101,
	0x0502, 0x0002, 0x0001, 0x0301, 0x0502, 0x0301, 0x0301, 0x0301, 0x0001,
	0x1003, 0x0003, 0x0301, 0x0502, 0x0402, 0x0001, 0x0101, 0x0402, 0x0102,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0002, 0x0201, 0x0201,
	0x0201, 0x0201, 0x0201, 0x0004, 0x0201, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0502, 0x0502,
	0x0201, 0x0301, 0x0502, 0x0201, 0x0201, 0x0201, 0x0301, 0x0301, 0x0201,
	0x0301, 0x0201, 0x0301, 0x0502, 0x0202, 0x0103, 0x0103, 0x0301, 0x0103,
	0x0301, 0x0301, 0x0502, 0x0502, 0x0101, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x1003, 0x1003, 0x1003, 0x1003, 0x1003, 0x0101, 0x0101, 0x0101,
	0x0101, 0x0101, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0101, 0x0101, 0x0101,
	0x0101, 0x0101, 0x0101, 0x0101, 0x0101, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0101, 0x0101, 0x0101, 0x0101, 0x0101,
	0x0101, 0x0101, 0x0101, 0x0101, 0x0001, 0x0001, 0x0301, 0x0101, 0x0101,
	0x0101, 0x0101, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
	0x0001, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0001, 0x0301,
	0x0301, 0x0502, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0001, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0502, 0x0014,
	0x0311, 0x0014, 0x0011, 0x0101, 0x0301, 0x0201, 0x0301, 0x0101, 0x0301,
	0x0101, 0x0001, 0x0001, 0x0301, 0x0101, 0x0001, 0x0001, 0x0301, 0x0101,
	0x0301, 0x0101, 0x0001, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0101, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0201, 0x0101, 0x0301, 0x0201, 0x0201, 0x0101, 0x0301, 0x0101, 0x0301,
	0x0201, 0x0101, 0x0301, 0x0201, 0x0301, 0x0301, 0x0301, 0x0301, 0x0001,
	0x0001, 0x0001, 0x0111, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0003, 0x0301, 0x0301, 0x0301, 0x0001, 0x0301, 0x0001,
	0x0301, 0x0001, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0502,
	0x0004, 0x0512, 0x0512, 0x0301, 0x0301, 0x0502, 0x0004, 0x0301, 0x0512,
	0x0301, 0x0301, 0x0502, 0x0004, 0x0301, 0x0512, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0111, 0x0311, 0x0301, 0x0101, 0x0301, 0x0201, 0x0001, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0301, 0x0301, 0x0301, 0x0111, 0x0311, 0x0301,
	0x0101, 0x0301, 0x0201, 0x0001, 0x0311, 0x0311, 0x0311, 0x0311, 0x0301,
	0x0301, 0x0311, 0x0311, 0x0301, 0x0301, 0x0001, 0x0001, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0101, 0x0301, 0x0502, 0x0502, 0x0101, 0x0101,
	0x0101, 0x0101, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x1003, 0x1003, 0x1003, 0x1003, 0x1003, 0x1003, 0x1003, 0x1003,
	0x0101, 0x0101, 0x0101, 0x0101, 0x0101, 0x0101, 0x0101, 0x0101, 0x0311,
	0x0301, 0x0101, 0x0301, 0x0101, 0x0301, 0x0101, 0x0101, 0x0001, 0x0001,
	0x0301, 0x0301, 0x0301, 0x0001, 0x0301, 0x0301, 0x0502, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0001, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0311,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0311, 0x0311, 0x0301, 0x0301,
	0x0311, 0x0301, 0x0301, 0x0502, 0x0311, 0x0502, 0x0301, 0x0301, 0x0301,
	0x0502, 0x0311, 0x0301, 0x0502, 0x0301, 0x0014, 0x0502, 0x0311, 0x0502,
	0x0502, 0x0502, 0x0502, 0x0502, 0x0502, 0x0502, 0x0502, 0x0502, 0x0502,
	0x0502, 0x0014, 0x0502, 0x0311, 0x0004, 0x0004, 0x0004, 0x0001, 0x0014,
	0x0014, 0x0014, 0x0012, 0x0101, 0x0512, 0x0512, 0x0502, 0x0301, 0x0004,
	0x0004, 0x0004, 0x0001, 0x0014, 0x0014, 0x0014, 0x0012, 0x0101, 0x0512,
	0x0512, 0x0502, 0x0301, 0x0004, 0x0004, 0x0004, 0x0002, 0x0301, 0x0102,
	0x0502, 0x0301, 0x0311, 0x0402, 0x0201, 0x0502, 0x0502, 0x0402, 0x0301,
	0x0201, 0x0101, 0x0802, 0x0301, 0x0201, 0x0301, 0x0502, 0x0502, 0x0301,
	0x0301, 0x0101, 0x0101, 0x0201, 0x0201, 0x0301, 0x0802, 0x0301, 0x0202,
	0x0301, 0x0202, 0x0301, 0x0301, 0x0802, 0x0301, 0x0802, 0x0301, 0x0301,
	0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0301, 0x0402,
	0x0102, 0x0502, 0x0502, 0x0502, 0x0201, 0x0301, 0x0301, 0x0301, 0x0301,
	0x0301, 0x0502, 0x0502, 0x0502, 0x0502, 0x0502, 0x0111, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311, 0x0311,
	0x0311, 0x0311, 0x0311, 0x0311, 0x0004, 0x0004, 0x0301, 0x0311, 0x0003,
};

/* Data size: 2124 bytes */

/* Reset registers */
static const u32 dbg_reset_regs[] = {
	0x00002014,		/* bb, misc.reset_pl_ua */
	0x00002014,		/* k2, misc.reset_pl_ua */
	0x00002418,		/* bb, miscs.reset_pl_hv */
	0x00002418,		/* k2, miscs.reset_pl_hv */
	0x00002414,		/* bb, miscs.reset_pl_ua */
	0x00002414,		/* k2, miscs.reset_pl_ua */
	0x00002018,		/* bb, misc.reset_pl_hv */
	0x00002018,		/* k2, misc.reset_pl_hv */
	0x0000201c,		/* bb, misc.reset_pl_pda_vmain_1 */
	0x0000201c,		/* k2, misc.reset_pl_pda_vmain_1 */
	0x00002020,		/* bb, misc.reset_pl_pda_vmain_2 */
	0x00002020,		/* k2, misc.reset_pl_pda_vmain_2 */
	0x00002024,		/* bb, misc.reset_pl_pda_vaux */
	0x00002024,		/* k2, misc.reset_pl_pda_vaux */
	0x01000000,		/* bb, miscs.reset_pl_hv_2 (removed) */
	0x00002454,		/* k2, miscs.reset_pl_hv_2 */
};

/* Data size: 64 bytes */

#endif /* __DBG_VALUES_H__ */

#ifndef __DBG_VALUES_USER_H__
#define __DBG_VALUES_USER_H__

/* Idle Check rule parsing data */
static const u32 idle_chk_rule_parsing_data[] = {
	0x00000000, 0x0000007e, 0x0000010a, 0x000001b8, 0x0000026c, 0x0000031a,
	0x000003c8, 0x00000476, 0x00000524, 0x000005e4, 0x00000698, 0x00000744,
	0x000007f6, 0x000008a2, 0x0000094e, 0x000009fa, 0x00000aa6, 0x00000bb8,
	0x00000cca, 0x00000d88, 0x00000e9a, 0x00000fac, 0x000010be, 0x000011d0,
	0x00001282, 0x000012f8, 0x00001392, 0x00001432, 0x00001488, 0x000014de,
	0x00001534, 0x0000158a, 0x000015e0, 0x00001636, 0x0000168c, 0x000016e2,
	0x00001738, 0x0000178e, 0x000017e8, 0x00001842, 0x0000189c, 0x000018f6,
	0x00001950, 0x000019aa, 0x00001a04, 0x00001a5e, 0x00001ab8, 0x00001b12,
	0x00001b6c, 0x00001bc6, 0x00001c20, 0x00001c7a, 0x00001cd4, 0x00001d2e,
	0x00001d88, 0x00001de2, 0x00001e3c, 0x00001e96, 0x00001ef0, 0x00001f4a,
	0x00001fc4, 0x00002058, 0x000020cc, 0x0000215a, 0x00002249, 0x000023fe,
	0x0000248c, 0x00002518, 0x000025ad, 0x00002686, 0x00002734, 0x000027c9,
	0x000028d5, 0x000029e1, 0x00002aed, 0x00002bf9, 0x00002cd1, 0x00002da9,
	0x00002ecf, 0x00002f99, 0x000030c1, 0x000031f7, 0x00003317, 0x000033f8,
	0x0000346c, 0x000034f2, 0x00003585, 0x00003674, 0x00003716, 0x0000381b,
	0x000038da, 0x0000398a, 0x00003a46, 0x00003afe, 0x00003ba2, 0x00003c53,
	0x00003d00, 0x00003d8f, 0x00003e31, 0x00003ed3, 0x00003f6d, 0x00004058,
	0x00004116, 0x0000419e, 0x0000424d, 0x000042f7, 0x000043a3, 0x0000444f,
	0x000044fb, 0x000045d9, 0x000046af, 0x00004747, 0x000047ef, 0x00004893,
	0x00004947, 0x000049fa, 0x00004a91, 0x00004bee, 0x00004c74, 0x00004d06,
	0x00004d99, 0x00004ea3, 0x00005031, 0x00005211, 0x00005397, 0x0000546b,
	0x0000556f, 0x00005673, 0x0000576b, 0x00005863, 0x00005973, 0x00005a83,
	0x00005b87, 0x00005c8b, 0x00005d7b, 0x00005e6a, 0x00005eea, 0x00005f68,
	0x00005ff7, 0x00006096, 0x00006112, 0x000061a1, 0x000062a8, 0x00006304,
	0x00006354, 0x000063b4, 0x00006438, 0x000064ac, 0x0000652a, 0x000065a8,
	0x00006624, 0x000066b8, 0x00006748, 0x000067cd, 0x00006871, 0x00006914,
	0x000069a6, 0x00006a46, 0x00006abe, 0x00006b34, 0x00006ba6, 0x00006c2e,
	0x00006cb8, 0x00006d2c, 0x00006db8, 0x00006e50, 0x00006ee8, 0x00006f6c,
	0x00007014, 0x000070bc, 0x00007138, 0x000071b4, 0x0000724e, 0x000072ca,
	0x00007346, 0x000073c2, 0x0000743e, 0x000074c0, 0x0000753c, 0x000075b8,
	0x00007634, 0x000076b0, 0x00007726, 0x000077a8, 0x00007832, 0x000078ae,
	0x00007926, 0x000079a2, 0x00007a2e, 0x00007abc, 0x00007b6c, 0x00007be6,
	0x00007c60, 0x00007e04, 0x00007ff2, 0x00008096, 0x0000813a, 0x00008218,
	0x000082f4, 0x000083cc, 0x000084bc, 0x000085ac, 0x00008686, 0x00008778,
	0x0000888e, 0x000089a0, 0x00008a86, 0x00008b6c, 0x00008c52, 0x00008d38,
	0x00008e1e, 0x00008f0a, 0x00008ff0, 0x000090d6, 0x000091bc, 0x0000929c,
	0x00009390, 0x00009472, 0x00009558, 0x00009650, 0x000096fc, 0x000097ac,
	0x00009892, 0x00009976, 0x00009a56, 0x00009b4c, 0x00009c44, 0x00009d26,
	0x00009e20, 0x00009f3e, 0x0000a058, 0x0000a146, 0x0000a234, 0x0000a322,
	0x0000a410, 0x0000a4fe, 0x0000a5f2, 0x0000a6e0, 0x0000a7ce, 0x0000a8bc,
	0x0000a9a4, 0x0000aaa0, 0x0000ab8a, 0x0000ac78, 0x0000ad76, 0x0000ae00,
	0x0000ae8e, 0x0000af10, 0x0000af9c, 0x0000b0a6, 0x0000b132, 0x0000b1cc,
	0x0000b658, 0x0000b6e2, 0x0000b76c, 0x0000b7fe, 0x0000b89e, 0x0000b942,
	0x0000b9e6, 0x0000ba8a, 0x0000bb2e, 0x0000bbc8, 0x0000bc54, 0x0000bcf1,
	0x0000bd9b, 0x0000be42, 0x0000bfb6, 0x0000c0f0, 0x0000c17a, 0x0000c204,
	0x0000c276, 0x0000c308, 0x0000c3aa, 0x0000c44e, 0x0000c4f2, 0x0000c56a,
	0x0000c5fa, 0x0000c682, 0x0000c71a, 0x0000c7de, 0x0000c88a, 0x0000c8f8,
	0x0000ca5e, 0x0000cb32, 0x0000ccd8, 0x0000cea6, 0x0000d052, 0x0000d0f6,
	0x0000d370, 0x0000d422, 0x0000d74c, 0x0000d7de, 0x0000d880, 0x0000d920,
	0x0000d9b8, 0x0000da4e, 0x0000daf5, 0x0000dc4d, 0x0000dd9e, 0x0000de28,
	0x0000e1ee, 0x0000e26c, 0x0000e2ec, 0x0000e36c, 0x0000e408, 0x0000e4a8,
	0x0000e548, 0x0000e5ec, 0x0000e694, 0x0000e73c, 0x0000e7d6, 0x0000e864,
	0x0000e926, 0x0000e9b4, 0x0000ea46, 0x0000ead8, 0x0000eb6e, 0x0000ec08,
	0x0000eca2, 0x0000ed30, 0x0000ef4c, 0x0000f144, 0x0000f334, 0x0000f8a8,
	0x0000faae, 0x0000fed2, 0x0001003a, 0x00010130, 0x000102c0, 0x00010338,
	0x0001055c, 0x000107d0, 0x00010a8c, 0x00010bbe, 0x00010c4c, 0x00010e50,
	0x000110ce, 0x00011216, 0x00011362, 0x000114ae, 0x000115fe, 0x000117ee,
	0x0001196b, 0x00011a07, 0x00011aa3, 0x00011b45, 0x00011bf9, 0x00011d5e,
	0x00011df2, 0x00011e7e, 0x00011f00, 0x00011f7e, 0x00012022, 0x000120d3,
	0x0001219c, 0x00012238, 0x000122c4, 0x00012350, 0x000123dd, 0x000124e3,
	0x0001266f, 0x00012826, 0x00012877, 0x0001295b, 0x000129d7, 0x00012ad3,
	0x00012c5b, 0x00012de3, 0x00012f6b, 0x000130f3, 0x0001327b, 0x00013403,
	0x0001358b, 0x00013713, 0x0001389b, 0x00013a22, 0x00013ae6, 0x00013baa,
	0x00013c6e, 0x00013d22, 0x00013dd6, 0x00013e8a, 0x00013f44, 0x00013ffe,
	0x000140b8, 0x0001417e, 0x00014244, 0x0001430a, 0x000143d0, 0x00014496,
	0x0001455c, 0x00014622, 0x000146e8, 0x0001475e, 0x000147d4, 0x0001482c,
	0x00014885, 0x00014962, 0x000149ff, 0x00014adc, 0x00014b78, 0x00014c0c,
	0x00014c71, 0x00014d81, 0x00014e33, 0x00014ee5, 0x00014f97, 0x000150e6,
	0x00015216, 0x0001529e, 0x00015326, 0x000153ae, 0x00015436, 0x000154c4,
	0x00015552, 0x000155fa, 0x0001568c, 0x0001571f, 0x00015877, 0x0001595b,
	0x00015a3d, 0x00015b0b, 0x00015bc5, 0x00015ceb, 0x00015e05, 0x00015eff,
	0x00015ff5, 0x000160eb, 0x000161e1, 0x000162d7, 0x000163cd, 0x000164c3,
	0x000165b9, 0x000166af, 0x000167a5, 0x0001689b, 0x00016991, 0x00016a87,
	0x00016b7d, 0x00016c73, 0x00016d69, 0x00016e5f, 0x00016f55, 0x0001704a,
	0x0001710a, 0x000171ca, 0x0001728b, 0x000173e5, 0x0001753d, 0x000176a1,
	0x00017785, 0x0001787f, 0x00017978, 0x000179fa, 0x00017a8e, 0x00017b4e,
	0x00017c0f, 0x00017d67, 0x00017ebe, 0x00017f7f, 0x000180e3, 0x000181dd,
	0x000182d6, 0x0001836a, 0x0001842a, 0x000184ea, 0x000185ab, 0x00018705,
	0x0001885d, 0x000189c1, 0x00018abb, 0x00018bb4, 0x00018c36, 0x00018ccb,
	0x00018e2f, 0x00018f91, 0x000190e9, 0x00019241, 0x00019399, 0x000194fd,
	0x000195e5, 0x000196d1, 0x000197b5, 0x000198af, 0x000199a8, 0x00019a22,
	0x00019ab7, 0x00019b7d, 0x00019c1b, 0x00019ce1, 0x00019e85, 0x00019f4b,
	0x0001a0ee, 0x0001a180, 0x0001a212, 0x0001a2a4, 0x0001a337, 0x0001a3f3,
	0x0001a4af, 0x0001a5a4, 0x0001a654, 0x0001a709, 0x0001a7e0, 0x0001a894,
	0x0001a96f, 0x0001aa42, 0x0001aaf2, 0x0001aba7, 0x0001ac80, 0x0001ad35,
	0x0001ae0b, 0x0001aede, 0x0001af8e, 0x0001b042, 0x0001b0f6, 0x0001b198,
	0x0001b22f, 0x0001b2e4, 0x0001b384, 0x0001b3f9, 0x0001b4b3, 0x0001b585,
	0x0001b64d, 0x0001b715, 0x0001b7e6, 0x0001b875, 0x0001ba8c, 0x0001bb1a,
	0x0001bb8e, 0x0001bc5c, 0x0001bcee, 0x0001bd81, 0x0001bebb, 0x0001bfe7,
	0x0001c10d, 0x0001c25d, 0x0001c375, 0x0001c47d, 0x0001c5cb, 0x0001c6e1,
	0x0001c7e9, 0x0001c937, 0x0001ca4d, 0x0001cb55, 0x0001cca3, 0x0001cdb9,
	0x0001cec1, 0x0001d00f, 0x0001d125, 0x0001d22d, 0x0001d37b, 0x0001d491,
	0x0001d599, 0x0001d6e7, 0x0001d7fd, 0x0001d905, 0x0001da53, 0x0001db69,
	0x0001dc71, 0x0001ddbf, 0x0001ded5, 0x0001dfdd, 0x0001e12b, 0x0001e241,
	0x0001e349, 0x0001e49d, 0x0001e5b9, 0x0001e6c7, 0x0001e81b, 0x0001e937,
	0x0001ea45, 0x0001eb99, 0x0001ecb5, 0x0001edc3, 0x0001ef17, 0x0001f033,
	0x0001f141, 0x0001f295, 0x0001f3b1, 0x0001f4bf, 0x0001f613, 0x0001f72f,
	0x0001f83d, 0x0001f991, 0x0001faad, 0x0001fbbb, 0x0001fd0f, 0x0001fe2b,
	0x0001ff39, 0x0002008d, 0x000201a9, 0x000202b7, 0x0002040b, 0x00020527,
	0x00020634, 0x000206ba, 0x00020750, 0x000207e6, 0x0002087a, 0x00020926,
	0x000209ce, 0x00020a62, 0x00020b7e, 0x00020c04, 0x00020c9a, 0x00020d30,
	0x00020dc4, 0x00020e70, 0x00020f18, 0x00020fac, 0x000210c8, 0x0002112d,
	0x00021207, 0x000212e1, 0x00021393, 0x00021461, 0x00021535, 0x00021645,
	0x00021757, 0x00021849, 0x00021914, 0x000219bc, 0x00021a49, 0x00021b37,
	0x00021c17, 0x00021cfd, 0x00021e55, 0x00021f39, 0x00022027, 0x00022123,
	0x00022211, 0x000222fb, 0x000223c9, 0x00022483, 0x0002251f, 0x00022619,
	0x000226c9, 0x000227a6, 0x0002283b, 0x000228d7, 0x00022985, 0x00022a35,
	0x00022ae9, 0x00022bb1, 0x00022c81, 0x00022d95, 0x00022ea9, 0x00022f7b,
	0x00023047, 0x0002313d, 0x00023227, 0x0002333d, 0x00023413, 0x0002350b,
	0x0002362f, 0x0002373f, 0x00023825, 0x00023935, 0x00023a55, 0x00023b39,
	0x00023c59, 0x00023d7f, 0x00023ea3, 0x00023fbf, 0x000240af, 0x00024187,
	0x00024263, 0x0002432d, 0x000243e3, 0x000244c0, 0x00024555, 0x000245f1,
	0x0002469f, 0x0002474f, 0x00024803, 0x000248cb, 0x0002499b, 0x00024aaf,
	0x00024bc3, 0x00024c95, 0x00024d61, 0x00024e57, 0x00024f41, 0x00025057,
	0x0002512d, 0x00025225, 0x00025349, 0x00025459, 0x0002553f, 0x0002564f,
	0x0002576f, 0x00025853, 0x00025973, 0x00025a99, 0x00025bbd, 0x00025cd9,
	0x00025dc9, 0x00025ea1, 0x00025f7d, 0x00026047, 0x000260fd, 0x000261da,
	0x0002626f, 0x0002630b, 0x000263b9, 0x00026469, 0x0002651d, 0x000265e5,
	0x000266b5, 0x000267c9, 0x000268dd, 0x000269af, 0x00026a7b, 0x00026b71,
	0x00026c5b, 0x00026d71, 0x00026e47, 0x00026f3f, 0x00027063, 0x00027173,
	0x00027259, 0x00027369, 0x00027489, 0x0002756d, 0x0002768d, 0x000277b3,
	0x000278d7, 0x000279f3, 0x00027ae3, 0x00027bbb, 0x00027c97, 0x00027d61,
	0x00027e17, 0x00027ef4, 0x00027f89, 0x00028025, 0x000280d3, 0x00028183,
	0x00028237, 0x000282ff, 0x000283cf, 0x000284e3, 0x000285f7, 0x000286c9,
	0x00028795, 0x0002888b, 0x00028975, 0x00028a8b, 0x00028b61, 0x00028c59,
	0x00028d7d, 0x00028e8d, 0x00028f73, 0x00029083, 0x000291a3, 0x00029287,
	0x000293a7, 0x000294cd, 0x000295f1, 0x0002970d, 0x000297fd, 0x000298d5,
	0x000299b1, 0x00029a7b, 0x00029b31, 0x00029c0e, 0x00029ca3, 0x00029d3f,
	0x00029ded, 0x00029e9d, 0x00029f51, 0x0002a019, 0x0002a0e9, 0x0002a1fd,
	0x0002a311, 0x0002a3e3, 0x0002a4af, 0x0002a5a5, 0x0002a68f, 0x0002a7a5,
	0x0002a87b, 0x0002a973, 0x0002aa97, 0x0002aba7, 0x0002ac8d, 0x0002ad9d,
	0x0002aebd, 0x0002afa1, 0x0002b0c1, 0x0002b1e7, 0x0002b30b, 0x0002b427,
	0x0002b517, 0x0002b5ef, 0x0002b6cb, 0x0002b795, 0x0002b84b, 0x0002b928,
	0x0002b9bd, 0x0002ba59, 0x0002bb07, 0x0002bbb7, 0x0002bc6b, 0x0002bd33,
	0x0002be03, 0x0002bf17, 0x0002c02b, 0x0002c0fd, 0x0002c1c9, 0x0002c2bf,
	0x0002c3a9, 0x0002c4bf, 0x0002c595, 0x0002c68d, 0x0002c7b1, 0x0002c8c1,
	0x0002c9a7, 0x0002cab7, 0x0002cbd7, 0x0002ccbb, 0x0002cddb, 0x0002cf01,
	0x0002d025, 0x0002d141, 0x0002d231, 0x0002d309, 0x0002d3e5, 0x0002d4af,
	0x0002d565, 0x0002d643, 0x0002d721, 0x0002d842, 0x0002d8d4, 0x0002d966,
	0x0002da04, 0x0002dab6, 0x0002db5a, 0x0002dbed, 0x0002dcbf, 0x0002dd7b,
	0x0002de33, 0x0002deda, 0x0002df6e, 0x0002dffa, 0x0002e0a0, 0x0002e12c,
	0x0002e1e6, 0x0002e292, 0x0002e33c, 0x0002e3d8, 0x0002e458, 0x0002e4db,
	0x0002e5d7, 0x0002e6cd, 0x0002e7c3, 0x0002e8b3, 0x0002e9aa, 0x0002ea42,
	0x0002eada, 0x0002eb72, 0x0002ec0a, 0x0002ec9e, 0x0002ed4b, 0x0002ee4d,
	0x0002ef53, 0x0002f03b, 0x0002f13f, 0x0002f233, 0x0002f327, 0x0002f41b,
	0x0002f50f, 0x0002f603, 0x0002f6f7, 0x0002f7eb, 0x0002f8d9, 0x0002f9b7,
	0x0002fa95, 0x0002fb62, 0x0002fbf4, 0x0002fc86, 0x0002fd24, 0x0002fdc4,
	0x0002fe68, 0x0002ff10, 0x0002ffaa, 0x0003003d, 0x0003010f, 0x000301cb,
	0x00030283, 0x0003032a, 0x000303be, 0x0003043e, 0x000304c6, 0x00030552,
	0x000305f8, 0x00030684, 0x0003073e, 0x000307ea, 0x0003089a, 0x0003093c,
	0x000309d8, 0x00030a78, 0x00030b0a, 0x00030b8a, 0x00030c0d, 0x00030d09,
	0x00030e05, 0x00030ef9, 0x00030ff0, 0x00031088, 0x00031120, 0x000311b4,
	0x0003124c, 0x000312f9, 0x000313fb, 0x00031501, 0x00031605, 0x000316f9,
	0x000317ed, 0x000318e1, 0x000319d5, 0x00031ac9, 0x00031bb7, 0x00031c95,
	0x00031d73, 0x00031e50, 0x00031ede, 0x00031f7c, 0x00032020, 0x000320b3,
	0x00032185, 0x00032241, 0x000322f9, 0x000323a0, 0x0003242c, 0x000324d2,
	0x0003257e, 0x0003261a, 0x0003269a, 0x0003271d, 0x00032819, 0x00032910,
	0x000329a8, 0x00032a3c, 0x00032ae8, 0x00032b7e, 0x00032c05, 0x00032ce3,
	0x00032dc1, 0x00032e84, 0x00032f16, 0x00032fa8, 0x00033046, 0x000330e6,
	0x00033198, 0x0003323c, 0x000332e4, 0x0003337e, 0x00033411, 0x000334e3,
	0x0003359f, 0x00033657, 0x000336fe, 0x00033792, 0x00033812, 0x0003389a,
	0x00033926, 0x000339cc, 0x00033a58, 0x00033b12, 0x00033bbe, 0x00033c6e,
	0x00033d10, 0x00033dba, 0x00033e56, 0x00033ef6, 0x00033f88, 0x00034008,
	0x0003408b, 0x00034187, 0x00034283, 0x00034377, 0x00034467, 0x0003455e,
	0x000345f6, 0x0003468e, 0x00034722, 0x000347b6, 0x00034863, 0x00034965,
	0x00034a6b, 0x00034b53, 0x00034c57, 0x00034d4b, 0x00034e3f, 0x00034f33,
	0x00035021, 0x000350ff, 0x000351dd, 0x000352ba, 0x0003534c, 0x000353de,
	0x0003547c, 0x0003551c, 0x000355ce, 0x00035672, 0x0003571a, 0x000357b4,
	0x00035847, 0x00035919, 0x000359d5, 0x00035a8d, 0x00035b34, 0x00035bc8,
	0x00035c48, 0x00035cd0, 0x00035d5c, 0x00035e02, 0x00035e8e, 0x00035f48,
	0x00035ff4, 0x000360a4, 0x00036146, 0x000361f0, 0x0003628c, 0x0003632c,
	0x000363be, 0x0003643e, 0x000364c1, 0x000365bd, 0x000366b9, 0x000367ad,
	0x000368a4, 0x0003693c, 0x000369d4, 0x00036a6c, 0x00036b00, 0x00036b98,
	0x00036c45, 0x00036d47, 0x00036e4d, 0x00036f51, 0x00037045, 0x00037139,
	0x0003722d, 0x00037321, 0x00037415, 0x00037509, 0x000375f7, 0x000376d5,
	0x000377b3, 0x00037890, 0x00037922, 0x000379b4, 0x00037a52, 0x00037af2,
	0x00037ba4, 0x00037c48, 0x00037cf0, 0x00037d8a, 0x00037e1d, 0x00037eef,
	0x00037fab, 0x00038063, 0x0003810a, 0x0003819e, 0x00038226, 0x000382a6,
	0x00038332, 0x000383d8, 0x00038464, 0x0003851e, 0x000385ca, 0x0003867a,
	0x0003871c, 0x000387c6, 0x00038862, 0x00038902, 0x00038994, 0x00038a14,
	0x00038a97, 0x00038b93, 0x00038c8f, 0x00038d83, 0x00038e73, 0x00038f6a,
	0x00039002, 0x0003909a, 0x00039132, 0x000391c6, 0x0003925e, 0x000392f6,
	0x0003938e, 0x00039426, 0x000394d3, 0x000395d5, 0x000396db, 0x000397c3,
	0x000398c7, 0x000399bb, 0x00039aaf, 0x00039ba3, 0x00039c97, 0x00039d8b,
	0x00039e7f, 0x00039f6d, 0x0003a02d, 0x0003a0ec, 0x0003a182, 0x0003a217,
	0x0003a2e5, 0x0003a3c3, 0x0003a494, 0x0003a4f2, 0x0003a544, 0x0003a593,
	0x0003a6cd, 0x0003a77c, 0x0003a81a, 0x0003a8b6, 0x0003a955, 0x0003aa67,
	0x0003ab71, 0x0003ac27, 0x0003ad17, 0x0003add2, 0x0003ae7c, 0x0003aee5,
	0x0003afa5, 0x0003b064, 0x0003b0fa, 0x0003b18f, 0x0003b25d, 0x0003b32f,
	0x0003b401, 0x0003b4d3, 0x0003b5b1, 0x0003b683, 0x0003b754, 0x0003b7b2,
	0x0003b804, 0x0003b853, 0x0003b98d, 0x0003ba3c, 0x0003bada, 0x0003bb76,
	0x0003bc15, 0x0003bd27, 0x0003be31, 0x0003bee7, 0x0003bfd7, 0x0003c092,
	0x0003c13c, 0x0003c1a5, 0x0003c265, 0x0003c324, 0x0003c3ba, 0x0003c44f,
	0x0003c51d, 0x0003c5ef, 0x0003c6c0, 0x0003c71e, 0x0003c770, 0x0003c7bf,
	0x0003c8f9, 0x0003c9a8, 0x0003ca46, 0x0003cae2, 0x0003cb81, 0x0003cc93,
	0x0003cd9d, 0x0003ce53, 0x0003cf43, 0x0003cffe, 0x0003d0a8, 0x0003d110,
	0x0003d179, 0x0003d239, 0x0003d2f8, 0x0003d38e, 0x0003d423, 0x0003d4f1,
	0x0003d5c3, 0x0003d6a0, 0x0003d6fe, 0x0003d750, 0x0003d79f, 0x0003d8d9,
	0x0003d988, 0x0003da26, 0x0003dac2, 0x0003db61, 0x0003dc73, 0x0003dd7d,
	0x0003de33, 0x0003df23, 0x0003dfde, 0x0003e088, 0x0003e0f1, 0x0003e1b1,
	0x0003e270, 0x0003e306, 0x0003e39b, 0x0003e469, 0x0003e53b, 0x0003e60d,
	0x0003e6eb, 0x0003e7bd, 0x0003e88f, 0x0003e960, 0x0003e9be, 0x0003ea10,
	0x0003ea5f, 0x0003eb99, 0x0003ec48, 0x0003ece6, 0x0003ed82, 0x0003ee21,
	0x0003ef33, 0x0003f03d, 0x0003f0f3, 0x0003f1e3, 0x0003f29e, 0x0003f348,
	0x0003f3b1, 0x0003f471, 0x0003f530, 0x0003f5c6, 0x0003f65b, 0x0003f729,
	0x0003f7fb, 0x0003f8cd, 0x0003f99f, 0x0003fa71, 0x0003fb42, 0x0003fba0,
	0x0003fbf2, 0x0003fc41, 0x0003fd7b, 0x0003fe2a, 0x0003fec8, 0x0003ff64,
	0x00040003, 0x00040115, 0x0004021f, 0x000402d5, 0x000403c5, 0x00040480,
	0x0004052a, 0x00040592, 0x000405f8, 0x000406af, 0x00040791, 0x00040874,
	0x000408f6, 0x00040b80, 0x00040c60, 0x00040d26, 0x00040daa, 0x00040e2e,
	0x00040ed6, 0x00040f7e, 0x00041018, 0x000410b2, 0x00041206, 0x0004135a,
	0x000414ae, 0x00041603, 0x0004172f, 0x00041830, 0x0004188e, 0x000418f9,
	0x00041a49, 0x00041b99, 0x00041ccf, 0x00041e1f, 0x00041f6f, 0x000420a5,
	0x000421b3, 0x000422a3, 0x000423a9, 0x0004247b, 0x0004252b, 0x0004268d,
	0x0004279d, 0x000428ac, 0x00042925, 0x000429f3, 0x00042ac1, 0x00042b8f,
	0x00042c5d, 0x00042d2b, 0x00042df8, 0x00042e8c, 0x00043116, 0x000431f6,
	0x000432bc, 0x00043340, 0x000433c5, 0x000434cb, 0x0004357a, 0x00043676,
	0x00043772, 0x00043884, 0x0004396a, 0x00043a6c, 0x00043b52, 0x00043c3e,
	0x00043d24, 0x00043e22, 0x00043f26, 0x0004402a, 0x00044144, 0x00044232,
	0x0004433c, 0x0004442a, 0x0004451e, 0x0004460c, 0x0004473f, 0x00044962,
	0x000449f4, 0x00044a87, 0x00044c62, 0x00044cf0, 0x00044da4, 0x00044e5c,
	0x00044ee1, 0x0004505b, 0x000451b5, 0x00045372, 0x000453d8, 0x0004548f,
	0x00045571, 0x00045654, 0x000456d7, 0x00045851, 0x000459b3, 0x00045b0d,
	0x00045d51, 0x00045e7d, 0x00045f7e, 0x00045fdc, 0x00046047, 0x0004617d,
	0x000462cd, 0x0004641d, 0x00046553, 0x000466a3, 0x000467f3, 0x000468e3,
	0x000469f1, 0x00046ac3, 0x00046bb9, 0x00046caf, 0x00046da5, 0x00046e9b,
	0x00046f91, 0x00047087, 0x0004717d, 0x00047273, 0x00047369, 0x0004745f,
	0x00047525, 0x000476c9, 0x0004778f, 0x00047933, 0x00047a95, 0x00047ba5,
	0x00047cb5, 0x00047da3, 0x00047e9f, 0x00047f8d, 0x00048088, 0x00048136,
	0x000481e4, 0x0004825d, 0x0004832a, 0x000483f3, 0x000484c0, 0x00048589,
	0x00048656, 0x0004871f, 0x000487ec, 0x000488b5, 0x00048982, 0x00048a4b,
	0x00048b18, 0x00048be0, 0x00048c84, 0x00048d02, 0x00048daa, 0x00048e2a,
	0x00048ef8, 0x00048fa0, 0x00049052, 0x00049104, 0x0004917e, 0x00049218,
	0x000492ac, 0x0004933f, 0x00049432, 0x000494c7, 0x000495ce, 0x0004964c,
	0x000496ce, 0x00049740, 0x000497b2, 0x00049822, 0x000498a4, 0x00049936,
	0x000499d4, 0x00049a68, 0x00049afe, 0x00049be8, 0x00049cbc, 0x00049d90,
	0x00049eca, 0x00049fc8, 0x0004a0c6, 0x0004a1c2, 0x0004a24a, 0x0004a2ee,
	0x0004a394, 0x0004a498, 0x0004a51e, 0x0004a5c0, 0x0004a664, 0x0004a766,
	0x0004a800, 0x0004a89c, 0x0004a926, 0x0004a9c0, 0x0004aa5c, 0x0004aae6,
	0x0004ab80, 0x0004ac1c, 0x0004aca6, 0x0004ad40, 0x0004addc, 0x0004ae66,
	0x0004aeeb, 0x0004b073, 0x0004b19f, 0x0004b2e5, 0x0004b4cc, 0x0004b555,
	0x0004b67d, 0x0004b7b5, 0x0004b8c9, 0x0004ba01, 0x0004bb2d, 0x0004bc59,
	0x0004bd85, 0x0004beb1, 0x0004bff7, 0x0004c110, 0x0004c360, 0x0004c40c,
	0x0004c496, 0x0004c522, 0x0004c5a7, 0x0004c643, 0x0004c6df, 0x0004c781,
	0x0004c835, 0x0004c99a, 0x0004ca2e, 0x0004caca, 0x0004cb4c, 0x0004cbca,
	0x0004cc6e, 0x0004cd1f, 0x0004ce26, 0x0004cea4, 0x0004cf4a, 0x0004cfdc,
	0x0004d06e, 0x0004d100,
};

/* Data size: 5912 bytes */

/* Array of attention name offsets */
static const u32 attn_name_offsets[] = {
	0x000268bf, 0x000268cd, 0x000268db, 0x000268f3, 0x00026908, 0x0002691e,
	0x00026930, 0x000268bf, 0x00026942, 0x0002694d, 0x0002695c, 0x00026974,
	0x0002698c, 0x000269a3, 0x000269ba, 0x000269d5, 0x000269f0, 0x00026a0b,
	0x00026a26, 0x00026a41, 0x00026a5c, 0x00026a75, 0x000268bf, 0x00026a81,
	0x00026a98, 0x00026aa7, 0x00026ac0, 0x00026adc, 0x00026aec, 0x00026b02,
	0x00026b18, 0x00026b38, 0x00026b4c, 0x00026b5c, 0x00026b78, 0x00026b99,
	0x00026ba9, 0x00026bb5, 0x00026bc0, 0x00026bcc, 0x00026bd7, 0x00026be4,
	0x00026bf2, 0x00026c03, 0x00026c15, 0x00026c20, 0x00026c35, 0x00026c48,
	0x00026c5a, 0x00026c6c, 0x00026c7e, 0x00026c90, 0x00026ca2, 0x00026cb4,
	0x00026cc6, 0x00026cd8, 0x00026cea, 0x00026cfc, 0x00026d0e, 0x00026d20,
	0x00026d32, 0x00026d44, 0x00026d56, 0x00026d68, 0x00026d7a, 0x00026d8c,
	0x00026d9e, 0x00026db0, 0x00026dc2, 0x00026dd4, 0x00026de6, 0x0002691e,
	0x00026df8, 0x00026e0c, 0x00026e20, 0x00026e34, 0x00026e48, 0x00026e5c,
	0x00026e70, 0x00026e84, 0x00026e98, 0x000268bf, 0x00026eaa, 0x00026ebf,
	0x00026ed4, 0x00026ee9, 0x00026efe, 0x00026f19, 0x00026f34, 0x00026f3e,
	0x00026f48, 0x00026f53, 0x00026f5f, 0x00026f6b, 0x00026c6c, 0x00026c7e,
	0x00026cfc, 0x00026de6, 0x00026c5a, 0x00026c90, 0x00026e98, 0x00026930,
	0x0002691e, 0x00026cea, 0x00026d0e, 0x000268bf, 0x00026f81, 0x00026f93,
	0x00026faa, 0x00026fbc, 0x00026fd3, 0x00026fe5, 0x00026ffc, 0x0002700e,
	0x00027025, 0x0002703e, 0x00027051, 0x00027064, 0x00027077, 0x0002708a,
	0x00027097, 0x000270a9, 0x000270bb, 0x000270ce, 0x000270e1, 0x000270f4,
	0x00027109, 0x00027123, 0x0002713d, 0x00027151, 0x0002716c, 0x00027185,
	0x0002719c, 0x000271b2, 0x000271c8, 0x000271dc, 0x000271f0, 0x00027203,
	0x00027216, 0x0002722b, 0x00027245, 0x0002725f, 0x00027273, 0x0002728e,
	0x000272a7, 0x000272be, 0x000272d4, 0x000272ea, 0x000272fe, 0x00027312,
	0x00027325, 0x00027338, 0x0002734d, 0x00027367, 0x00027381, 0x00027395,
	0x000273b0, 0x000273c9, 0x000273e0, 0x000273f6, 0x0002740c, 0x00027420,
	0x00027434, 0x00027447, 0x0002745a, 0x0002746f, 0x00027489, 0x000274a3,
	0x000274b7, 0x000274d2, 0x000274eb, 0x00027502, 0x00027518, 0x0002752e,
	0x00027542, 0x00027556, 0x0002756e, 0x00027585, 0x0002759c, 0x000275b8,
	0x000275d4, 0x000275f2, 0x00027609, 0x00027621, 0x00027639, 0x00027650,
	0x00027667, 0x00027683, 0x0002769f, 0x000276bd, 0x000276d4, 0x000276ec,
	0x00027704, 0x0002771b, 0x00027732, 0x0002774e, 0x0002776a, 0x00027788,
	0x0002779f, 0x000277b7, 0x000277cf, 0x000277e6, 0x000277fd, 0x00027819,
	0x00027835, 0x00027853, 0x0002786a, 0x00027882, 0x00027899, 0x000278af,
	0x000278c6, 0x000278de, 0x000278f4, 0x0002790f, 0x00027926, 0x0002793d,
	0x00027954, 0x0002796b, 0x00027982, 0x00027999, 0x000279b0, 0x000279c7,
	0x000279de, 0x000279f5, 0x00027a07, 0x00027a1e, 0x00027a36, 0x00027a4d,
	0x00027a64, 0x00027a80, 0x00027a9c, 0x00027aba, 0x00027ad1, 0x00027ae9,
	0x00027afb, 0x00027b12, 0x00027b2a, 0x00027b41, 0x00027b58, 0x00027b74,
	0x00027b90, 0x00027bae, 0x00027bc5, 0x00027bdd, 0x00027bef, 0x00027c06,
	0x00027c1e, 0x00027c35, 0x00027c4c, 0x00027c68, 0x00027c84, 0x00027ca2,
	0x00027cb9, 0x00027cd1, 0x00027ce3, 0x00027cfa, 0x00027d12, 0x00027d29,
	0x00027d40, 0x00027d5c, 0x00027d78, 0x00027d96, 0x00027dad, 0x00027dd3,
	0x00027de5, 0x00027dfc, 0x00027e14, 0x00027e2b, 0x00027e42, 0x00027e5e,
	0x00027e7a, 0x00027e98, 0x00027eaf, 0x00027ec7, 0x00027ed9, 0x00027ef0,
	0x00027f08, 0x00027f1f, 0x00027f36, 0x00027f52, 0x00027f6e, 0x00027f8c,
	0x00027fa3, 0x00027fbb, 0x00027fce, 0x00027fe1, 0x00027ff4, 0x00028007,
	0x0002801a, 0x0002802d, 0x00028040, 0x00028053, 0x00028068, 0x00028082,
	0x0002809c, 0x000280b0, 0x000280cb, 0x000280e4, 0x000280fb, 0x00028111,
	0x00028127, 0x0002813b, 0x0002814f, 0x00028162, 0x00028175, 0x0002818a,
	0x000281a4, 0x000281be, 0x000281d2, 0x000281ed, 0x00028206, 0x0002821d,
	0x00028233, 0x00028249, 0x0002825d, 0x00028271, 0x00028284, 0x00028297,
	0x000282ac, 0x000282c6, 0x000282e0, 0x000282f4, 0x0002830f, 0x00028328,
	0x0002833f, 0x00028355, 0x0002836b, 0x0002837f, 0x00028393, 0x000283a6,
	0x000283b9, 0x000283ce, 0x000283e8, 0x00028402, 0x00028416, 0x00028431,
	0x0002844a, 0x00028461, 0x00028477, 0x0002848d, 0x000284a1, 0x000284b5,
	0x000284c8, 0x000284db, 0x000284f0, 0x0002850a, 0x00028524, 0x00028538,
	0x00028553, 0x0002856c, 0x00028583, 0x00028599, 0x000285af, 0x000285c3,
	0x000285d7, 0x000285ea, 0x000285fd, 0x00028612, 0x0002862c, 0x00028646,
	0x0002865a, 0x00028675, 0x0002868e, 0x000286a5, 0x000286bb, 0x000286d1,
	0x000286e5, 0x000286f9, 0x00028718, 0x00028739, 0x00028751, 0x00028769,
	0x00028781, 0x00028799, 0x000287b1, 0x000287c9, 0x000287e1, 0x000287f9,
	0x00028812, 0x0002882b, 0x00028844, 0x0002885d, 0x00028876, 0x0002888f,
	0x000288a8, 0x000288c1, 0x000288e2, 0x00028903, 0x00028924, 0x0002893b,
	0x00028958, 0x00028975, 0x0002898e, 0x000289a7, 0x000289c6, 0x000289de,
	0x000289f0, 0x00028a02, 0x00028a14, 0x00026c35, 0x00028a26, 0x00028a3a,
	0x00028a4e, 0x00028a62, 0x00028a76, 0x00028a8a, 0x00028a9e, 0x00028ab2,
	0x00028ac6, 0x00028ada, 0x00028aee, 0x00028b02, 0x00028b16, 0x00028b2a,
	0x00028b3e, 0x00028b52, 0x00028b66, 0x00028b78, 0x00028b8a, 0x00028b9c,
	0x00028bae, 0x00028bc0, 0x00028bd2, 0x00028be4, 0x00028bf6, 0x00028c08,
	0x00028c1a, 0x00028c2c, 0x00028c3e, 0x00028c50, 0x00028c62, 0x00028c74,
	0x00028c86, 0x00028c98, 0x00028caa, 0x00028cbc, 0x00028cce, 0x00028ce0,
	0x00028cf2, 0x00028d04, 0x00028d16, 0x00028d28, 0x00028d3a, 0x00028d4c,
	0x00026db0, 0x00026ca2, 0x000268bf, 0x00028d5e, 0x00028d6f, 0x00028d7e,
	0x00028d92, 0x00028daf, 0x00028dc6, 0x00028dd7, 0x00028de7, 0x00028df6,
	0x00028e0f, 0x00028e21, 0x00028e3c, 0x00028e4d, 0x00028e5d, 0x00028e6c,
	0x00028e7d, 0x00028e88, 0x00028e9a, 0x00028eac, 0x00028b02, 0x00028b16,
	0x00026c90, 0x00026c5a, 0x00026cfc, 0x00026de6, 0x00026e98, 0x00026930,
	0x00028a26, 0x00028b2a, 0x00028a62, 0x00028a4e, 0x00028b52, 0x00028ebe,
	0x00028ed2, 0x00028ee6, 0x00028efa, 0x00028f0e, 0x00028f22, 0x00028f36,
	0x00028f4a, 0x00028f5e, 0x00028a26, 0x00028f69, 0x00028f7f, 0x00028f95,
	0x00028fab, 0x00028b52, 0x00026cea, 0x0002691e, 0x00026930, 0x00026c6c,
	0x00026c90, 0x00026cfc, 0x000268bf, 0x00028fc1, 0x00028fd3, 0x00028fe5,
	0x00028ff7, 0x00029009, 0x0002901b, 0x00029031, 0x00029048, 0x0002905e,
	0x00029075, 0x0002908e, 0x000290a3, 0x000290b5, 0x000290c7, 0x000290d9,
	0x000290eb, 0x00029107, 0x00026c35, 0x00026de6, 0x00026c5a, 0x00026cfc,
	0x00026930, 0x0002691e, 0x00026e98, 0x00026c90, 0x00026cea, 0x00026c6c,
	0x00026c7e, 0x00026d20, 0x00026d44, 0x00026d56, 0x00026d68, 0x00026d7a,
	0x00026d0e, 0x00026d32, 0x000268bf, 0x00029113, 0x00029127, 0x00029139,
	0x0002914a, 0x000268bf, 0x0002915c, 0x00029166, 0x000268bf, 0x00029174,
	0x00029182, 0x00029191, 0x0002919c, 0x00026c35, 0x000291ab, 0x000291bf,
	0x000291d3, 0x000291e7, 0x000291fb, 0x0002920f, 0x00029223, 0x00029237,
	0x0002924b, 0x00028ac6, 0x00028d28, 0x0002925f, 0x00029271, 0x00028d16,
	0x00029283, 0x00029295, 0x000292a7, 0x00026c48, 0x00026dd4, 0x00026e98,
	0x00026c5a, 0x00026c90, 0x00026c6c, 0x00026c7e, 0x00026d0e, 0x00026d44,
	0x00026d56, 0x00026d68, 0x00026930, 0x0002691e, 0x00026cea, 0x00026cfc,
	0x00026de6, 0x00026d20, 0x000268bf, 0x000292b9, 0x000292cb, 0x000292dc,
	0x000292ed, 0x00029300, 0x00029315, 0x00029328, 0x0002933b, 0x0002934f,
	0x00029361, 0x00029374, 0x00029385, 0x00029398, 0x000293ab, 0x000293be,
	0x000268bf, 0x000293d0, 0x000293e0, 0x000293f0, 0x000293fd, 0x0002940a,
	0x00029418, 0x00029427, 0x00029436, 0x00029445, 0x00029456, 0x00029464,
	0x00029472, 0x00029481, 0x0002948e, 0x0002949d, 0x000294ac, 0x000294bb,
	0x000294c9, 0x00026c35, 0x00028a3a, 0x00026e98, 0x000294e1, 0x000294f5,
	0x00029509, 0x0002951d, 0x00029531, 0x00029545, 0x00029559, 0x0002956d,
	0x00029581, 0x00029595, 0x000295a9, 0x000295bd, 0x000295d1, 0x000295e5,
	0x000295f9, 0x0002960d, 0x00029621, 0x00029635, 0x00029649, 0x0002965d,
	0x00029671, 0x00029685, 0x00029699, 0x000296ad, 0x000296c1, 0x000296d5,
	0x000296e9, 0x000296fd, 0x00029711, 0x00029725, 0x00029739, 0x0002974d,
	0x00029761, 0x00029775, 0x00029789, 0x0002979d, 0x000297b1, 0x000297c5,
	0x000297d9, 0x000297ed, 0x00029801, 0x00029815, 0x00029829, 0x0002983d,
	0x00029851, 0x00029865, 0x00029879, 0x0002988d, 0x000298a1, 0x000298b5,
	0x000298c9, 0x000298dd, 0x000298f1, 0x00029905, 0x00029919, 0x0002992d,
	0x00029941, 0x00029955, 0x00029969, 0x0002997d, 0x00029991, 0x000299a5,
	0x000299b9, 0x000299cd, 0x000299e1, 0x000299f5, 0x00029a09, 0x00029a1d,
	0x00029a31, 0x00029a45, 0x00029a59, 0x00029a6d, 0x00026d44, 0x00029a81,
	0x00028f0e, 0x00028f22, 0x00029a95, 0x00029aa9, 0x00029abd, 0x00029ad1,
	0x00029ae5, 0x00029af9, 0x00028ebe, 0x00028ed2, 0x00028ee6, 0x00028efa,
	0x00029b0d, 0x00029b21, 0x00029b35, 0x00029b49, 0x00029b5d, 0x00029b71,
	0x00029b85, 0x00029b99, 0x00029bad, 0x00029bc1, 0x00029bd5, 0x00029be9,
	0x00029bfd, 0x00029c11, 0x00029c25, 0x00029c39, 0x00029c4d, 0x00026df8,
	0x00026e0c, 0x00026e20, 0x00026e34, 0x00026e48, 0x00029c61, 0x00029c75,
	0x00029c89, 0x000268bf, 0x00029c9d, 0x000292b9, 0x000292cb, 0x000292dc,
	0x000292ed, 0x000293be, 0x00029315, 0x00029328, 0x00029caf, 0x0002933b,
	0x00029cc1, 0x00029361, 0x00029374, 0x0002934f, 0x00029385, 0x00029398,
	0x000293ab, 0x00029cd5, 0x00029ce7, 0x00029cfa, 0x00029d0d, 0x000268bf,
	0x00029d1e, 0x00029d30, 0x00029d40, 0x00029d50, 0x00029d60, 0x00029d70,
	0x00029d80, 0x00029d89, 0x00029d94, 0x00029da8, 0x00029dbc, 0x00029dcf,
	0x00029de3, 0x00029df6, 0x00028b16, 0x00028b2a, 0x00028a26, 0x00026de6,
	0x00026c90, 0x00026c6c, 0x0002691e, 0x00026930, 0x00026c7e, 0x00026c5a,
	0x00026cfc, 0x000268bf, 0x00029e0d, 0x000268bf, 0x00029e19, 0x00026930,
	0x00026e98, 0x0002691e, 0x000268bf, 0x00029e28, 0x00029e3d, 0x00029e53,
	0x00029e69, 0x00029e78, 0x00029e85, 0x00029e99, 0x000291ab, 0x00026d8c,
	0x00026de6, 0x00026e98, 0x00026930, 0x0002691e, 0x00026cea, 0x00026cfc,
	0x00026c6c, 0x00026c7e, 0x00026d20, 0x00026c5a, 0x00026d7a, 0x00026d56,
	0x00026d44, 0x00026d68, 0x00026d0e, 0x00026c90, 0x000268bf, 0x00029eb0,
	0x00029ec2, 0x00029ed5, 0x00029ee6, 0x00029ef8, 0x00029f09, 0x00029f1b,
	0x00029f2c, 0x00029f3e, 0x00029f4f, 0x00029f61, 0x00029f71, 0x00029f82,
	0x00029f92, 0x00029fa3, 0x00029fb2, 0x00029fc2, 0x00029fd3, 0x00029fe5,
	0x00029ff6, 0x0002a008, 0x0002a019, 0x0002a02b, 0x0002a03c, 0x0002a04e,
	0x0002a05f, 0x0002a071, 0x0002a082, 0x0002a094, 0x0002a0a5, 0x0002a0bb,
	0x0002a0d0, 0x0002a0e5, 0x0002a0f9, 0x0002a110, 0x0002a126, 0x0002a13c,
	0x0002a151, 0x0002a167, 0x0002a17d, 0x0002a189, 0x0002a19d, 0x0002a1b3,
	0x0002a1c9, 0x0002a1df, 0x0002a1f5, 0x0002a20b, 0x0002a221, 0x0002a237,
	0x00026d8c, 0x00026dc2, 0x00026d7a, 0x00026d20, 0x00026d32, 0x00026c7e,
	0x00026d9e, 0x00026d0e, 0x00026d44, 0x00026d56, 0x00026d68, 0x00029295,
	0x00029271, 0x00026c48, 0x00026cea, 0x00026cb4, 0x00026de6, 0x00026cd8,
	0x00026db0, 0x00029649, 0x0002965d, 0x00026c90, 0x0002924b, 0x0002a24d,
	0x0002a263, 0x0002a279, 0x0002a28f, 0x00026dd4, 0x00026ca2, 0x00026cc6,
	0x00026c6c, 0x00026e98, 0x00026930, 0x000268bf, 0x00029eb0, 0x00029ec2,
	0x0002a2a5, 0x0002a2b6, 0x0002a2c8, 0x0002a2d9, 0x0002a2eb, 0x0002a2fc,
	0x0002a30e, 0x0002a31f, 0x0002a331, 0x0002a342, 0x00029f1b, 0x00029f2c,
	0x00029f61, 0x00029f71, 0x00029fc2, 0x00029fd3, 0x00029fe5, 0x00029ff6,
	0x0002a008, 0x0002a019, 0x0002a02b, 0x0002a03c, 0x0002a04e, 0x0002a05f,
	0x0002a071, 0x0002a082, 0x0002a094, 0x0002a0a5, 0x0002a0bb, 0x0002a0d0,
	0x0002a0e5, 0x0002a0f9, 0x0002a110, 0x0002a126, 0x0002a13c, 0x0002a151,
	0x0002a167, 0x0002a17d, 0x0002a354, 0x00028b02, 0x0002a279, 0x0002a28f,
	0x0002a1f5, 0x0002a20b, 0x0002a368, 0x0002a37e, 0x0002a189, 0x00026d32,
	0x00026dc2, 0x00026d20, 0x00026d7a, 0x00026d9e, 0x00026db0, 0x00026d8c,
	0x00026d0e, 0x00026d44, 0x00026d56, 0x00026d68, 0x000292a7, 0x00029283,
	0x00029295, 0x00026cea, 0x00026cc6, 0x00026de6, 0x00026c48, 0x00026ca2,
	0x00029649, 0x0002965d, 0x00026c90, 0x00026c6c, 0x00026c7e, 0x00026e98,
	0x00026930, 0x000268bf, 0x00029eb0, 0x00029ec2, 0x0002a394, 0x0002a3a5,
	0x0002a2c8, 0x0002a2d9, 0x0002a2eb, 0x0002a2fc, 0x0002a3b7, 0x0002a3c8,
	0x0002a3da, 0x0002a3eb, 0x0002a3fd, 0x0002a40e, 0x0002a420, 0x0002a431,
	0x00029f3e, 0x00029f4f, 0x00029f61, 0x00029f71, 0x00029fa3, 0x00029fb2,
	0x00029fc2, 0x00029fd3, 0x00029fe5, 0x00029ff6, 0x0002a008, 0x0002a019,
	0x0002a02b, 0x0002a03c, 0x0002a04e, 0x0002a05f, 0x0002a071, 0x0002a082,
	0x0002a094, 0x0002a0a5, 0x0002a0bb, 0x0002a0d0, 0x0002a0e5, 0x0002a0f9,
	0x0002a110, 0x0002a126, 0x0002a13c, 0x0002a151, 0x0002a167, 0x0002a17d,
	0x0002a443, 0x0002a1f5, 0x0002a20b, 0x0002a221, 0x0002a237, 0x0002924b,
	0x0002a457, 0x0002a46d, 0x00028a3a, 0x0002a483, 0x0002a499, 0x0002a354,
	0x00026d9e, 0x00026db0, 0x00026dc2, 0x00026d56, 0x00026d8c, 0x00026ca2,
	0x00026d68, 0x00026d7a, 0x00026d20, 0x00026d32, 0x00028d16, 0x0002925f,
	0x000292a7, 0x00026de6, 0x00026dd4, 0x00026c6c, 0x00029295, 0x00026cb4,
	0x000299cd, 0x000299e1, 0x00026d0e, 0x00026d44, 0x0002691e, 0x00026cea,
	0x00026e98, 0x00026930, 0x000268bf, 0x00029eb0, 0x00029ec2, 0x0002a2a5,
	0x0002a2b6, 0x0002a394, 0x0002a3a5, 0x0002a2c8, 0x0002a2d9, 0x0002a2eb,
	0x0002a2fc, 0x00029ef8, 0x00029f09, 0x0002a331, 0x0002a342, 0x00029f1b,
	0x00029f2c, 0x00029f3e, 0x00029f4f, 0x00029f61, 0x00029f71, 0x00029fa3,
	0x00029fb2, 0x00029fc2, 0x00029fd3, 0x00029fe5, 0x00029ff6, 0x0002a008,
	0x0002a019, 0x0002a02b, 0x0002a03c, 0x0002a04e, 0x0002a05f, 0x0002a071,
	0x0002a082, 0x0002a094, 0x0002a0a5, 0x0002a0bb, 0x0002a0d0, 0x0002a0e5,
	0x0002a151, 0x0002a4af, 0x0002a4cc, 0x0002a4e8, 0x0002a503, 0x0002a51d,
	0x0002a535, 0x0002a54c, 0x0002a565, 0x0002a57f, 0x0002a19d, 0x0002a1b3,
	0x0002a593, 0x0002a5a9, 0x00028b16, 0x0002a5bf, 0x0002a5d5, 0x0002a5eb,
	0x00026dd4, 0x00026cd8, 0x00026ca2, 0x00029295, 0x00026cb4, 0x00029271,
	0x00029283, 0x00026d32, 0x00026cc6, 0x00026c48, 0x00026d8c, 0x00026dc2,
	0x00026d9e, 0x00026db0, 0x00028bae, 0x00028b9c, 0x00028b8a, 0x00026cfc,
	0x00028d3a, 0x000292a7, 0x00026de6, 0x00026d7a, 0x0002a5ff, 0x0002a613,
	0x0002a629, 0x0002a63f, 0x00028d4c, 0x00028d28, 0x00026d20, 0x0002925f,
	0x00026c5a, 0x00026c90, 0x00026c6c, 0x00026c7e, 0x00026d0e, 0x00026d44,
	0x00026d56, 0x00026d68, 0x00026e98, 0x00026930, 0x000268bf, 0x00029eb0,
	0x00029ec2, 0x0002a2a5, 0x0002a2b6, 0x0002a2c8, 0x0002a2d9, 0x0002a653,
	0x0002a664, 0x00029ef8, 0x00029f09, 0x0002a331, 0x0002a342, 0x00029f61,
	0x00029f71, 0x00029fc2, 0x00029fd3, 0x00029fe5, 0x00029ff6, 0x0002a008,
	0x0002a019, 0x0002a02b, 0x0002a03c, 0x0002a04e, 0x0002a05f, 0x0002a071,
	0x0002a082, 0x0002a094, 0x0002a0d0, 0x0002a0e5, 0x0002a0f9, 0x0002a110,
	0x0002a126, 0x0002a13c, 0x0002a151, 0x0002a167, 0x0002a17d, 0x0002a676,
	0x0002a19d, 0x0002a1b3, 0x0002a1c9, 0x0002a1df, 0x00029223, 0x0002a1f5,
	0x0002a20b, 0x0002a368, 0x0002a37e, 0x00026d8c, 0x00026d9e, 0x00026d32,
	0x00026d20, 0x00026dc2, 0x00026d7a, 0x00026d0e, 0x00026d44, 0x00026d56,
	0x00026d68, 0x00029283, 0x00029295, 0x00029271, 0x00026cea, 0x00026cc6,
	0x00026de6, 0x00026dd4, 0x00026db0, 0x00029649, 0x0002965d, 0x00026c90,
	0x0002a189, 0x0002a24d, 0x0002a263, 0x0002920f, 0x0002a221, 0x0002a237,
	0x00026c48, 0x00026cb4, 0x00026cd8, 0x00026c6c, 0x00026c7e, 0x00026e98,
	0x00026930, 0x000268bf, 0x00029eb0, 0x00029ec2, 0x0002a68a, 0x0002a69b,
	0x00029f61, 0x00029f71, 0x0002a008, 0x0002a019, 0x0002a02b, 0x0002a03c,
	0x0002a04e, 0x0002a05f, 0x0002a071, 0x0002a082, 0x0002a094, 0x0002a0d0,
	0x0002a0e5, 0x0002a151, 0x0002a17d, 0x00028a8a, 0x0002a6ad, 0x0002a6c3,
	0x00026c90, 0x00026c5a, 0x00026de6, 0x00026930, 0x0002691e, 0x00026cea,
	0x00026cfc, 0x00026d68, 0x00026d56, 0x00026d0e, 0x00026c6c, 0x00026e98,
	0x00028a76, 0x0002a6d9, 0x0002a6ef, 0x00026c7e, 0x000268bf, 0x0002a705,
	0x0002a710, 0x0002a71e, 0x0002a72d, 0x0002a73c, 0x0002a74c, 0x0002a75c,
	0x0002a76d, 0x0002a77e, 0x0002a793, 0x0002a7a6, 0x0002a7ba, 0x0002a7ce,
	0x0002a7e4, 0x0002a7fa, 0x0002a806, 0x0002a82a, 0x0002a84f, 0x0002a873,
	0x0002a898, 0x0002a8bc, 0x0002a8e1, 0x0002a8ee, 0x0002a8fb, 0x0002a908,
	0x0002a915, 0x0002a920, 0x0002a92a, 0x0002a937, 0x0002a93e, 0x0002a94f,
	0x0002a95f, 0x00028f69, 0x00028f7f, 0x0002a1f5, 0x0002a20b, 0x00028a8a,
	0x00028b8a, 0x00028d4c, 0x00028bae, 0x00028b9c, 0x00028bc0, 0x00028be4,
	0x00028bd2, 0x00028ce0, 0x00028cce, 0x00028caa, 0x00028cbc, 0x00028cf2,
	0x00028d04, 0x0002a972, 0x0002a984, 0x00028b66, 0x00028b78, 0x0002a996,
	0x0002a9a8, 0x00028d16, 0x0002925f, 0x00029283, 0x000292a7, 0x00028d28,
	0x00028d3a, 0x00028c86, 0x00028c3e, 0x00028c62, 0x00028c50, 0x00028c98,
	0x00028c74, 0x00028c1a, 0x00028c2c, 0x00028bf6, 0x00028c08, 0x00029295,
	0x00026cea, 0x0002691e, 0x00026d0e, 0x00026d32, 0x00026d20, 0x00026d56,
	0x00026db0, 0x00026cc6, 0x00026dc2, 0x00026d8c, 0x00026d7a, 0x00026d68,
	0x00026cb4, 0x00026dd4, 0x00026ca2, 0x00026cd8, 0x00026d9e, 0x00026c90,
	0x00029649, 0x0002965d, 0x00029671, 0x00029685, 0x00029699, 0x000296ad,
	0x000296c1, 0x000296d5, 0x000296e9, 0x0002a9ba, 0x0002a9ce, 0x0002a9e3,
	0x0002a9f8, 0x0002aa0d, 0x0002aa22, 0x0002aa37, 0x00026c48, 0x00026e98,
	0x00029271, 0x00026930, 0x00026c7e, 0x00026c6c, 0x000268bf, 0x0002aa4c,
	0x0002aa62, 0x0002aa78, 0x0002aa8e, 0x0002aaa4, 0x0002aabd, 0x0002aad6,
	0x0002aaec, 0x0002ab02, 0x0002ab15, 0x0002ab28, 0x0002ab3b, 0x0002ab4e,
	0x0002ab64, 0x0002ab7a, 0x0002ab90, 0x0002aba6, 0x0002abbc, 0x0002abd2,
	0x0002abe8, 0x0002abfe, 0x0002ac15, 0x0002ac2c, 0x0002ac40, 0x0002ac55,
	0x0002ac6a, 0x0002ac80, 0x0002ac96, 0x0002acac, 0x0002acc2, 0x0002acd9,
	0x0002acf3, 0x0002ad0b, 0x0002ad1b, 0x0002ad2b, 0x0002ad42, 0x0002ad51,
	0x0002ad60, 0x0002ad76, 0x0002ad8d, 0x0002ada7, 0x0002adbf, 0x0002add5,
	0x00028b02, 0x00026d20, 0x00026c5a, 0x00026c7e, 0x00026c90, 0x00026c6c,
	0x00026d56, 0x00026d7a, 0x00026d68, 0x00026cea, 0x00026cfc, 0x00026de6,
	0x00026d0e, 0x00026e98, 0x00026930, 0x000268bf, 0x0002adeb, 0x0002adf3,
	0x0002ae06, 0x0002ae16, 0x0002ae2d, 0x0002ae3d, 0x0002ae4e, 0x0002ae67,
	0x00026c35, 0x00028aee, 0x00026e98, 0x0002691e, 0x00026cea, 0x00026cfc,
	0x00026de6, 0x000268bf, 0x00026f81, 0x0002ae81, 0x0002ae93, 0x0002aea5,
	0x00026f93, 0x00026faa, 0x0002aeba, 0x0002aecc, 0x0002aede, 0x00026fbc,
	0x00026fd3, 0x0002aef3, 0x0002af05, 0x0002af17, 0x00026fe5, 0x00026ffc,
	0x0002af2c, 0x0002af3e, 0x0002af50, 0x0002700e, 0x00027025, 0x0002af65,
	0x0002703e, 0x00027051, 0x00027064, 0x00027077, 0x0002af80, 0x0002708a,
	0x0002af9a, 0x0002afaf, 0x00027097, 0x0002afc2, 0x000270a9, 0x000270bb,
	0x000270ce, 0x0002afd4, 0x000270f4, 0x00027109, 0x00027123, 0x0002713d,
	0x00027151, 0x0002716c, 0x00027185, 0x0002719c, 0x000271b2, 0x000271c8,
	0x000271dc, 0x000271f0, 0x00027203, 0x0002afe7, 0x00027216, 0x0002722b,
	0x00027245, 0x0002725f, 0x00027273, 0x0002728e, 0x000272a7, 0x000272be,
	0x000272d4, 0x000272ea, 0x000272fe, 0x00027312, 0x00027325, 0x0002affa,
	0x00027338, 0x0002734d, 0x00027367, 0x00027381, 0x00027395, 0x000273b0,
	0x000273c9, 0x000273e0, 0x000273f6, 0x0002740c, 0x00027420, 0x00027434,
	0x00027447, 0x0002b00d, 0x0002745a, 0x0002746f, 0x00027489, 0x000274a3,
	0x000274b7, 0x000274d2, 0x000274eb, 0x00027502, 0x00027518, 0x0002752e,
	0x00027542, 0x00027556, 0x0002756e, 0x00027585, 0x0002759c, 0x000275b8,
	0x000275d4, 0x000275f2, 0x00027609, 0x00027621, 0x00027639, 0x00027650,
	0x00027667, 0x00027683, 0x0002769f, 0x000276bd, 0x000276d4, 0x000276ec,
	0x00027704, 0x0002771b, 0x00027732, 0x0002774e, 0x0002776a, 0x00027788,
	0x0002779f, 0x000277b7, 0x000277cf, 0x000277e6, 0x000277fd, 0x00027819,
	0x00027835, 0x00027853, 0x0002786a, 0x00027882, 0x00027899, 0x000278af,
	0x000278c6, 0x0002b020, 0x0002b02e, 0x000278de, 0x000278f4, 0x0002790f,
	0x00027926, 0x0002793d, 0x00027954, 0x0002796b, 0x000279f5, 0x0002b03c,
	0x0002b04e, 0x0002b060, 0x00027a07, 0x00027a1e, 0x00027a36, 0x00027a4d,
	0x00027a64, 0x00027a80, 0x00027a9c, 0x00027aba, 0x00027ad1, 0x00027ae9,
	0x00027dad, 0x00027fbb, 0x00027fce, 0x00027fe1, 0x00027ff4, 0x0002802d,
	0x00028040, 0x00028053, 0x00028068, 0x00028082, 0x0002809c, 0x000280b0,
	0x000280cb, 0x000280e4, 0x000280fb, 0x00028111, 0x00028127, 0x0002813b,
	0x0002814f, 0x00028162, 0x00028175, 0x0002818a, 0x000281a4, 0x000281be,
	0x000281d2, 0x000281ed, 0x00028206, 0x0002821d, 0x00028233, 0x00028249,
	0x0002825d, 0x00028271, 0x00028284, 0x00028297, 0x000282ac, 0x000282c6,
	0x000282e0, 0x000282f4, 0x0002830f, 0x00028328, 0x0002833f, 0x00028355,
	0x0002836b, 0x0002837f, 0x00028393, 0x000283a6, 0x000283b9, 0x000283ce,
	0x000283e8, 0x00028402, 0x00028416, 0x00028431, 0x0002844a, 0x00028461,
	0x00028477, 0x0002848d, 0x000284a1, 0x000285fd, 0x0002b075, 0x0002b095,
	0x0002b0b2, 0x0002b0cf, 0x0002b0ec, 0x0002b109, 0x0002b126, 0x0002b143,
	0x0002b160, 0x0002b17d, 0x0002b19a, 0x0002b1bb, 0x0002b1dc, 0x0002b1fd,
	0x0002b21e, 0x0002b23f, 0x0002b260, 0x0002b281, 0x0002b2a2, 0x0002b2b0,
	0x0002b2be, 0x0002b2d0, 0x0002b2e2, 0x0002b2f5, 0x0002b308, 0x0002b31b,
	0x000289de, 0x000289f0, 0x00028a02, 0x00028a14, 0x00026c35, 0x00028a26,
	0x00028a3a, 0x00028a4e, 0x00028a62, 0x00028a76, 0x00028a8a, 0x00028a9e,
	0x00028ab2, 0x00028ac6, 0x00028ada, 0x00028aee, 0x00028b02, 0x00028b16,
	0x00028b2a, 0x00028b3e, 0x00028b52, 0x0002b32e, 0x0002b340, 0x00028caa,
	0x00028cbc, 0x00028cce, 0x00028ce0, 0x00028cf2, 0x00028d04, 0x00028b66,
	0x00028b78, 0x0002a984, 0x0002a972, 0x0002a996, 0x0002a9a8, 0x0002b352,
	0x00028c1a, 0x00028c2c, 0x00028c3e, 0x00028c50, 0x00028c62, 0x00028c74,
	0x00028c86, 0x00028c98, 0x00028bd2, 0x00028be4, 0x00028bf6, 0x00028c08,
	0x00028bc0, 0x0002b364, 0x0002b376, 0x0002b388, 0x00029283, 0x000292a7,
	0x0002925f, 0x00028d16, 0x00028b8a, 0x00028b9c, 0x00028d28, 0x00028d3a,
	0x00028d4c, 0x00026d32, 0x00026d8c, 0x00026dc2, 0x00026d9e, 0x00026db0,
	0x00026ca2, 0x00026cb4, 0x00026cc6, 0x00029295, 0x00026dd4, 0x00026c48,
	0x00029271, 0x00026cd8, 0x000268bf, 0x0002b39a, 0x0002b3ae, 0x0002b3b9,
	0x00028a76, 0x00028a8a, 0x00028ada, 0x000291ab, 0x000291fb, 0x0002920f,
	0x0002a189, 0x0002a676, 0x0002a9a8, 0x00028c08, 0x00028bf6, 0x00028b8a,
	0x00028d16, 0x00028d28, 0x00028d3a, 0x00028d4c, 0x00029295, 0x00029283,
	0x000292a7, 0x0002925f, 0x00026c5a, 0x00029271, 0x00028bae, 0x00028bc0,
	0x00028d04, 0x00028b66, 0x00028bd2, 0x00028be4, 0x00028b78, 0x0002a984,
	0x00026c6c, 0x00028a4e, 0x00028a62, 0x00028ab2, 0x00028ac6, 0x00026dd4,
	0x00026cd8, 0x00026db0, 0x00026dc2, 0x00026d9e, 0x00026d32, 0x00026d8c,
	0x00026cfc, 0x00026d20, 0x00026cb4, 0x00026cc6, 0x00026c90, 0x00026d44,
	0x00026d56, 0x00026de6, 0x00026d0e, 0x0002691e, 0x00026cea, 0x00026c7e,
	0x00026d68, 0x00026d7a, 0x00028cbc, 0x00028cce, 0x00028ce0, 0x00028cf2,
	0x00028c2c, 0x00028c3e, 0x00028c50, 0x00028c62, 0x00028c74, 0x00028c86,
	0x00028c98, 0x00028caa, 0x0002a972, 0x00028c1a, 0x00026930, 0x00026e98,
	0x00026c48, 0x000268bf, 0x0002b3c8, 0x0002b3d8, 0x0002b3e9, 0x0002b3fa,
	0x0002b409, 0x0002b41e, 0x0002b433, 0x0002b44a, 0x0002b461, 0x0002b475,
	0x0002b48c, 0x0002b4a3, 0x0002b4ba, 0x0002b4cd, 0x0002b4e4, 0x0002b4fa,
	0x0002b514, 0x0002b52a, 0x0002b539, 0x0002b548, 0x0002b558, 0x0002b568,
	0x0002b579, 0x0002b58a, 0x0002b599, 0x0002b5ac, 0x0002b5c1, 0x00026c6c,
	0x00026c90, 0x00026c5a, 0x00026de6, 0x00026cfc, 0x00026930, 0x00026c7e,
	0x00026e98, 0x0002691e, 0x00026cea, 0x00026c6c, 0x00026c90, 0x00026c5a,
	0x00026de6, 0x00026cfc, 0x00026930, 0x00026d0e, 0x00026e98, 0x0002691e,
	0x00026cea, 0x00026c7e, 0x00026c6c, 0x00026c90, 0x00026c5a, 0x00026de6,
	0x0002691e, 0x00026c7e, 0x00026930, 0x00026cea, 0x00026cfc, 0x00026e98,
	0x00026c90, 0x00026c5a, 0x00026de6, 0x00026cfc, 0x00026930, 0x00026c6c,
	0x00026e98, 0x0002691e, 0x00026cea, 0x000268bf, 0x0002b5d6, 0x0002b5e5,
	0x0002b5f6, 0x0002b605, 0x0002b618, 0x0002b62b, 0x0002b642, 0x0002b658,
	0x0002b66f, 0x0002b685, 0x0002b699, 0x0002b6b0, 0x0002b6c1, 0x0002b6d2,
	0x0002b6df, 0x0002b6e8, 0x0002b6fa, 0x0002b709, 0x0002b723, 0x0002b73c,
	0x0002b755, 0x0002b76d, 0x0002b784, 0x0002b79a, 0x0002b7b0, 0x0002b7c7,
	0x0002b7db, 0x0002b7ee, 0x0002b7fd, 0x0002b80f, 0x0002b81d, 0x0002b830,
	0x0002b844, 0x0002b858, 0x0002b86c, 0x0002b883, 0x0002b899, 0x0002b8af,
	0x0002b8c4, 0x0002b8d9, 0x0002b8eb, 0x0002b8fb, 0x0002b90c, 0x0002b920,
	0x0002b935, 0x0002b94a, 0x0002b960, 0x0002a1f5, 0x0002a20b, 0x00026cea,
	0x00026930, 0x0002691e, 0x00026e98, 0x00026cc6, 0x00026cb4, 0x00026ca2,
	0x00026db0, 0x00026d9e, 0x00026dc2, 0x00026d8c, 0x00028b2a, 0x00028aee,
	0x00026de6, 0x00026c5a, 0x0002b935, 0x0002b94a, 0x0002b960, 0x00028f69,
	0x00028f7f, 0x00026cfc, 0x00026930, 0x00026cea, 0x0002691e, 0x00026e98,
	0x00026cc6, 0x00026cb4, 0x00026ca2, 0x00026db0, 0x00026d9e, 0x00026dc2,
	0x00026d8c, 0x000268bf, 0x0002b975, 0x0002b987, 0x0002b999, 0x0002b9a9,
	0x0002b9b9, 0x0002b9ca, 0x0002b9db, 0x0002b9ea, 0x0002b9f9, 0x0002ba08,
	0x0002ba17, 0x00028aee, 0x00028a26, 0x0002691e, 0x00026cea, 0x000268bf,
	0x0002ba26, 0x0002ba31, 0x0002ba40, 0x0002ba56, 0x0002ba6c, 0x00028b3e,
	0x00028aee, 0x0002691e, 0x00026cea, 0x00026c5a, 0x00026c90, 0x00026cfc,
	0x00026e98, 0x00028b2a, 0x00028a26, 0x00028a3a, 0x00028b52, 0x00026930,
	0x0002691e, 0x00026c6c, 0x00026c7e, 0x00026cea, 0x00026de6, 0x00026e98,
	0x00026930, 0x00026cfc, 0x00026de6, 0x00026cea, 0x0002691e, 0x00028b16,
	0x00028b3e, 0x00026e98, 0x00026930, 0x00026c90, 0x00026c6c, 0x0002691e,
	0x00026cfc, 0x00026c5a, 0x000268bf, 0x0002ba7c, 0x0002ba88, 0x0002ba99,
	0x0002baa9, 0x0002bab8, 0x0002bac9, 0x0002bad8, 0x0002bae7, 0x0002baf6,
	0x0002bb05, 0x00026c35, 0x00028a8a, 0x00028a9e, 0x00028ab2, 0x000291e7,
	0x00026cea, 0x00026cc6, 0x00026d20, 0x00026d32, 0x00026c90, 0x00026c6c,
	0x00026c7e, 0x00026d7a, 0x00026d0e, 0x0002691e, 0x00026930, 0x00026cfc,
	0x00026cb4, 0x00026de6, 0x00026c5a, 0x00026e98, 0x00026ca2, 0x00026db0,
	0x00026dc2, 0x00028ac6, 0x000291fb, 0x00026cd8, 0x00026d8c, 0x00026d44,
	0x00026d9e, 0x000268bf, 0x0002bb14, 0x0002ba7c, 0x0002bb1e, 0x0002bb2a,
	0x0002bb37, 0x0002bb45, 0x0002bb51, 0x0002bb60, 0x000268bf, 0x00026f81,
	0x0002ae93, 0x00026f93, 0x00026faa, 0x0002aecc, 0x00026fbc, 0x00026fd3,
	0x0002af05, 0x00026fe5, 0x00026ffc, 0x0002af3e, 0x0002700e, 0x00027025,
	0x0002703e, 0x0002708a, 0x000270a9, 0x0002bb6f, 0x000270bb, 0x000270ce,
	0x000270e1, 0x0002afd4, 0x000270f4, 0x00027109, 0x00027123, 0x0002713d,
	0x00027151, 0x0002716c, 0x0002719c, 0x000271b2, 0x000271c8, 0x000271dc,
	0x0002bb81, 0x0002bb9c, 0x0002bbb7, 0x0002bbd3, 0x0002bbef, 0x00027556,
	0x0002756e, 0x00027585, 0x0002759c, 0x000275b8, 0x000275d4, 0x000275f2,
	0x00027609, 0x00027621, 0x00027639, 0x00027650, 0x00027667, 0x00027683,
	0x0002769f, 0x000276bd, 0x000276d4, 0x000276ec, 0x00027704, 0x0002771b,
	0x00027732, 0x0002774e, 0x0002776a, 0x00027788, 0x0002779f, 0x000277b7,
	0x000277cf, 0x000277e6, 0x000277fd, 0x00027819, 0x00027835, 0x00027853,
	0x0002786a, 0x000278c6, 0x000278de, 0x000278f4, 0x0002790f, 0x00027926,
	0x0002793d, 0x00027954, 0x0002796b, 0x00027982, 0x00027999, 0x000279b0,
	0x000279f5, 0x0002b04e, 0x00027a07, 0x00027a1e, 0x00027a36, 0x00027a4d,
	0x00027a64, 0x00027a80, 0x00027a9c, 0x00027aba, 0x00027ad1, 0x00027ae9,
	0x0002bc0a, 0x00027afb, 0x00027b12, 0x00027b2a, 0x00027b41, 0x00027b58,
	0x00027b74, 0x00027b90, 0x00027bae, 0x00027bc5, 0x00027bdd, 0x0002bc1c,
	0x00027bef, 0x00027c06, 0x00027c1e, 0x00027c35, 0x00027c4c, 0x00027c68,
	0x00027c84, 0x00027ca2, 0x00027cb9, 0x00027cd1, 0x0002bc2e, 0x00027ce3,
	0x00027cfa, 0x00027d12, 0x00027d29, 0x00027d40, 0x00027d5c, 0x00027d78,
	0x00027d96, 0x00027dad, 0x0002833f, 0x000285fd, 0x0002bc40, 0x0002bc59,
	0x000289c6, 0x000289de, 0x000289f0, 0x00028a02, 0x00028a14, 0x00026c35,
	0x00028a26, 0x00028a3a, 0x00028a4e, 0x00028a62, 0x00028a76, 0x00028a8a,
	0x00028a9e, 0x00028ab2, 0x00028ac6, 0x00028ada, 0x00028aee, 0x00028b02,
	0x00028b16, 0x00028b2a, 0x00028b3e, 0x00028b52, 0x00028d16, 0x00028d3a,
	0x00028d28, 0x0002925f, 0x000292a7, 0x00026db0, 0x00026ca2, 0x00026cb4,
	0x00026cc6, 0x00026cd8, 0x00026dd4, 0x00026c48, 0x00029271, 0x00029283,
	0x00029295, 0x00026c35, 0x0002bc72, 0x0002bc86, 0x0002a63f, 0x00028b02,
	0x000291bf, 0x0002a6d9, 0x0002a6ef, 0x0002adbf, 0x0002add5, 0x0002bc9a,
	0x0002bcb0, 0x0002bcc6, 0x0002bcdc, 0x0002bcf2, 0x0002bd08, 0x0002bd1e,
	0x0002bd34, 0x0002bd4a, 0x0002bd61, 0x0002bd78, 0x0002bd8f, 0x0002bda6,
	0x0002bdbd, 0x00028bc0, 0x00028bae, 0x00028b9c, 0x00028d28, 0x0002925f,
	0x000292a7, 0x00029283, 0x00029295, 0x00026ca2, 0x00026cb4, 0x00026db0,
	0x00026d9e, 0x00026e98, 0x00026930, 0x00026de6, 0x00026c5a, 0x00026cfc,
	0x00026cea, 0x00029271, 0x00026dd4, 0x00026c48, 0x00026dc2, 0x00026d20,
	0x00026d32, 0x00026c90, 0x00026d0e, 0x00026c7e, 0x00026cc6, 0x00026cd8,
	0x00028b8a, 0x00028d4c, 0x00028d3a, 0x00026d68, 0x00026d7a, 0x00026d56,
	0x000268bf, 0x0002bdd4, 0x0002bde2, 0x0002bdf3, 0x0002be00, 0x0002be0f,
	0x0002be1e, 0x0002be2b, 0x0002be3d, 0x00026c35, 0x00028b2a, 0x00028a4e,
	0x00028a62, 0x00028a76, 0x00026e98, 0x0002691e, 0x00026930, 0x00026de6,
	0x00026c5a, 0x00026c90, 0x00026cea, 0x000268bf, 0x0002be4a, 0x0002be5f,
	0x0002be74, 0x0002be89, 0x0002be9e, 0x0002beae, 0x0002bebe, 0x00026e98,
	0x00026cea, 0x00026930, 0x00026cfc, 0x0002691e, 0x000268bf, 0x0002becb,
	0x0002bed5, 0x0002bee2, 0x0002beef, 0x0002bf0c, 0x0002bf29, 0x0002bf46,
	0x00028a26, 0x0002691e, 0x0002bed5, 0x0002bee2, 0x0002beef, 0x0002bf0c,
	0x0002bf29, 0x0002bf46, 0x00026930, 0x00026e98, 0x000268bf, 0x0002bf63,
	0x0002bf77, 0x0002bf8e, 0x0002bfa9, 0x0002bfbf, 0x0002bfd1, 0x0002bfe7,
	0x0002c004, 0x0002c016, 0x0002c02d, 0x0002b3ae, 0x00028a4e, 0x00026d7a,
	0x00026d20, 0x00026d32, 0x00026d8c, 0x00026dc2, 0x00026e98, 0x00026df8,
	0x00026e0c, 0x00028ebe, 0x00028ed2, 0x00028ee6, 0x0002691e, 0x00026cfc,
	0x00029919, 0x0002992d, 0x0002c03b, 0x0002c04f, 0x0002c063, 0x00026c5a,
	0x000299cd, 0x000299e1, 0x0002c077, 0x00028f36, 0x00028f4a, 0x00026d0e,
	0x00026d56, 0x00026d68, 0x00026d9e, 0x00029c61, 0x00029c75, 0x00029c89,
	0x00026930, 0x00029649, 0x0002965d, 0x00029671, 0x00026de6, 0x000299f5,
	0x00029a09, 0x00029865, 0x00029879, 0x000268bf, 0x0002c08b, 0x0002c0a3,
	0x0002c0bf, 0x0002c0d4, 0x0002c0e8, 0x0002c101, 0x0002c112, 0x0002c11f,
	0x0002c132, 0x0002c143, 0x00028b3e, 0x0002c154, 0x0002c16a, 0x00028aee,
	0x00028b16, 0x00026cfc, 0x00026c5a, 0x00026c90, 0x00026c6c, 0x00026c7e,
	0x00026d0e, 0x00029c61, 0x00029c75, 0x00026930, 0x00026cea, 0x000268bf,
	0x0002c180, 0x000268bf, 0x0002c18c, 0x0002c19d, 0x0002c1ad, 0x0002c1c3,
	0x0002c1da, 0x0002c1f2, 0x0002c20b, 0x0002c222, 0x0002c239, 0x0002c250,
	0x0002c267, 0x0002c27e, 0x0002c28d, 0x0002c29c, 0x0002c2ab, 0x0002c2ba,
	0x0002c2c9, 0x0002c2d8, 0x0002c2e7, 0x0002c2f6, 0x0002c305, 0x0002c314,
	0x0002c324, 0x0002c334, 0x0002c344, 0x0002c354, 0x0002c364, 0x0002c374,
	0x0002c383, 0x0002c392, 0x0002c3a1, 0x0002c3b0, 0x0002c3bf, 0x0002c3ce,
	0x0002c3dd, 0x0002c3ec, 0x0002c3fb, 0x0002c40a, 0x0002c41a, 0x0002c42a,
	0x0002c43a, 0x0002c44a, 0x0002c45a, 0x0002c46a, 0x0002c47f, 0x0002c493,
	0x0002c4a7, 0x0002c4bc, 0x0002c4d1, 0x0002c4e6, 0x0002c4fb, 0x0002c511,
	0x0002c527, 0x0002c53d, 0x0002c553, 0x0002c569, 0x0002c57f, 0x0002c595,
	0x0002c5ab, 0x0002c5c1, 0x0002c5d5, 0x0002c5ee, 0x0002c603, 0x0002c618,
	0x0002c631, 0x0002c64b, 0x0002c665, 0x0002c67f, 0x0002c699, 0x0002c6b3,
	0x0002c6cd, 0x0002c6e7, 0x0002c701, 0x0002c71e, 0x0002c73b, 0x0002c758,
	0x0002c775, 0x0002c792, 0x0002c7af, 0x0002c7cc, 0x0002c7e9, 0x0002c806,
	0x0002c81b, 0x0002c82f, 0x0002c843, 0x0002c858, 0x0002c86d, 0x0002c882,
	0x0002c897, 0x0002c8ad, 0x0002c8c3, 0x0002c8d9, 0x0002c8ef, 0x0002c905,
	0x0002c91b, 0x0002c931, 0x0002c947, 0x0002c95d, 0x0002c971, 0x0002c98a,
	0x0002c99f, 0x0002c9b4, 0x0002c9cd, 0x0002c9e7, 0x0002ca01, 0x0002ca1b,
	0x0002ca35, 0x0002ca4f, 0x0002ca69, 0x0002ca83, 0x0002ca9d, 0x0002caba,
	0x0002cad7, 0x0002caf4, 0x0002cb11, 0x0002cb2e, 0x0002cb4b, 0x0002cb68,
	0x0002cb85, 0x0002cba2, 0x0002cbb7, 0x0002cbcb, 0x0002cbdf, 0x0002cbf4,
	0x0002cc09, 0x0002cc1e, 0x0002cc33, 0x0002cc49, 0x0002cc5f, 0x0002cc75,
	0x0002cc8b, 0x0002cca1, 0x0002ccb7, 0x0002cccd, 0x0002cce3, 0x0002ccf9,
	0x0002cd0d, 0x0002cd26, 0x0002cd3b, 0x0002cd50, 0x0002cd69, 0x0002cd83,
	0x0002cd9d, 0x0002cdb7, 0x0002cdd1, 0x0002cdeb, 0x0002ce05, 0x0002ce1f,
	0x0002ce39, 0x0002ce56, 0x0002ce73, 0x0002ce90, 0x0002cead, 0x0002ceca,
	0x0002cee7, 0x0002cf04, 0x0002cf21, 0x0002cf3e, 0x0002cf53, 0x0002cf67,
	0x0002cf7b, 0x0002cf90, 0x0002cfa5, 0x0002cfba, 0x0002cfcf, 0x0002cfe5,
	0x0002cffb, 0x0002d011, 0x0002d027, 0x0002d03d, 0x0002d053, 0x0002d069,
	0x0002d07f, 0x0002d095, 0x0002d0a9, 0x0002d0c2, 0x0002d0d7, 0x0002d0ec,
	0x0002d105, 0x0002d11f, 0x0002d139, 0x0002d153, 0x0002d16d, 0x0002d187,
	0x0002d1a1, 0x0002d1bb, 0x0002d1d5, 0x0002d1f2, 0x0002d20f, 0x0002d22c,
	0x0002d249, 0x0002d266, 0x0002d283, 0x0002d2a0, 0x0002d2bd, 0x00026f6b,
	0x0002d2da, 0x0002d2ec, 0x0002d2fe, 0x0002d310, 0x0002d322, 0x0002d334,
	0x0002d346, 0x0002d358, 0x0002d36a, 0x0002d37c, 0x0002d38e, 0x0002d3a0,
	0x0002d3b2, 0x00028cce, 0x0002a972, 0x0002a996, 0x0002a9a8, 0x0002b352,
	0x0002b364, 0x0002b376, 0x0002b388, 0x0002b340, 0x0002b32e, 0x00028ce0,
	0x00028cf2, 0x00028d04, 0x00028b66, 0x00028b78, 0x0002a984, 0x00028d3a,
	0x00028c2c, 0x00028c86, 0x00028c98, 0x0002d3c4, 0x0002d3d6, 0x0002d3e8,
	0x0002d3fa, 0x0002d40c, 0x0002d41e, 0x00026e98, 0x00026c90, 0x00026c6c,
	0x00026c7e, 0x00026d0e, 0x00026d44, 0x00026d56, 0x00026d68, 0x00026d7a,
	0x00026d20, 0x00026930, 0x0002691e, 0x00026cea, 0x00026cfc, 0x00026de6,
	0x00026c5a, 0x0002d430, 0x0002d442, 0x0002d454, 0x0002d466, 0x00028c3e,
	0x00028c50, 0x00028c62, 0x00028c74, 0x00028caa, 0x00028cbc, 0x00028d4c,
	0x00028b8a, 0x00028b9c, 0x00028bae, 0x00028bc0, 0x00028bd2, 0x00028be4,
	0x00028bf6, 0x00028c08, 0x00028c1a, 0x0002d478, 0x0002d48a, 0x00026c48,
	0x00029271, 0x00029295, 0x00029283, 0x00026d32, 0x00026d8c, 0x0002d49c,
	0x0002d4ae, 0x0002d4c0, 0x0002d4d2, 0x0002d4e4, 0x0002d4f6, 0x0002d508,
	0x0002d51a, 0x0002d52c, 0x0002d53e, 0x0002d550, 0x0002d562, 0x000292a7,
	0x0002925f, 0x00028d16, 0x00028d28, 0x00026dc2, 0x00026d9e, 0x00026db0,
	0x00026ca2, 0x0002d574, 0x0002d586, 0x00026cb4, 0x00026cc6, 0x00026cd8,
	0x00026dd4, 0x0002d598, 0x0002d5aa, 0x0002d5c0, 0x00026d32, 0x00026d8c,
	0x00026dc2, 0x00026d9e, 0x00026db0, 0x00026ca2, 0x00026cb4, 0x00026cc6,
	0x00026e98, 0x00026c90, 0x00026c6c, 0x00026c7e, 0x00026d0e, 0x00026d44,
	0x00026d56, 0x00026d68, 0x00026d7a, 0x00026d20, 0x00026930, 0x0002691e,
	0x00026cea, 0x00026cfc, 0x00026de6, 0x00026c5a, 0x000268bf, 0x0002d5d6,
	0x0002d5e7, 0x0002d5fa, 0x0002d607, 0x0002d616, 0x0002d624, 0x0002d632,
	0x0002d640, 0x0002d64e, 0x0002d65c, 0x0002d66a, 0x0002d678, 0x0002d686,
	0x000268bf, 0x0002d693, 0x0002d6a1, 0x0002d6b0, 0x0002d6be, 0x0002d6cd,
	0x0002d6db, 0x0002d6ea, 0x0002d6f8, 0x0002d707, 0x0002d712, 0x0002d71e,
	0x0002d729, 0x0002d735, 0x0002d740, 0x0002d74c, 0x0002d757, 0x00026d9e,
	0x00029271, 0x00028d4c, 0x00028c08, 0x00026cb4, 0x000292a7, 0x00028bae,
	0x00028c3e, 0x00026cc6, 0x0002925f, 0x00028bc0, 0x00028c50, 0x00026d8c,
	0x00026dd4, 0x00028d28, 0x00028be4, 0x00026d32, 0x00026cd8, 0x00028d16,
	0x00028bd2, 0x00026db0, 0x00029295, 0x00028b8a, 0x00028c1a, 0x00026dc2,
	0x00026c48, 0x00028d3a, 0x00028bf6, 0x00026ca2, 0x00029283, 0x00028b9c,
	0x00028c2c, 0x00028cf2, 0x00028b66, 0x0002a984, 0x0002a996, 0x00028d04,
	0x00028b78, 0x0002a972, 0x0002a9a8, 0x00026c6c, 0x00026c7e, 0x00026d0e,
	0x00026d44, 0x00026d56, 0x00026d68, 0x00026d7a, 0x00026d20, 0x00026e98,
	0x00026930, 0x0002691e, 0x00026cea, 0x00026cfc, 0x00026de6, 0x00026c5a,
	0x00026c90, 0x00028c62, 0x00028caa, 0x00028c74, 0x00028cbc, 0x00028c86,
	0x00028cce, 0x00028c98, 0x00028ce0, 0x0002b364, 0x0002b388, 0x0002b32e,
	0x0002d334, 0x0002b352, 0x0002b376, 0x0002b340, 0x0002d346, 0x000268bf,
	0x0002d763, 0x0002d77a, 0x0002d791, 0x0002d7a8, 0x0002d7bf, 0x0002d7d5,
	0x0002d7eb, 0x0002d801, 0x0002d817, 0x0002d82c, 0x0002d843, 0x0002d85a,
	0x0002d871, 0x0002d888, 0x0002d89e, 0x0002d8b4, 0x0002d8ca, 0x0002d8e0,
	0x0002d8f5, 0x0002d90c, 0x0002d923, 0x0002d93a, 0x0002d951, 0x0002d967,
	0x0002d97d, 0x0002d993, 0x0002d9a9, 0x0002d9be, 0x0002d9d5, 0x0002d9ec,
	0x0002da03, 0x0002da1a, 0x0002da30, 0x0002da46, 0x0002da5c, 0x0002da72,
	0x0002691e, 0x00026e98, 0x00026cea, 0x00026930, 0x000268bf, 0x0002da87,
	0x0002da8f, 0x0002daa3, 0x0002dab1, 0x00026d20, 0x00026c5a, 0x00026d32,
	0x00026d8c, 0x00026d9e, 0x00026d0e, 0x00026c6c, 0x00026d56, 0x00026d7a,
	0x00026d68, 0x00026e98, 0x00026cea, 0x00026c90, 0x00026dc2, 0x00026d44,
	0x00026de6, 0x00026c7e, 0x00026cfc, 0x00026930, 0x0002691e,
};

/* Data size: 11852 bytes */

/* Array of attention indexes */
static const u16 attn_idx[] = {
	0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007, 0x0008,
	0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f, 0x0010, 0x0011,
	0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017, 0x0018, 0x0019, 0x001a,
	0x001b, 0x001c, 0x001d, 0x001e, 0x001f, 0x0020, 0x0021, 0x0022, 0x0023,
	0x0024, 0x0025, 0x0026, 0x0027, 0x0028, 0x0029, 0x002a, 0x002b, 0x002c,
	0x002d, 0x002e, 0x002f, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035,
	0x0036, 0x0037, 0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e,
	0x003f, 0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047,
	0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050,
	0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059,
	0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f, 0x0060, 0x0061, 0x0062,
	0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x006a, 0x006b,
	0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072, 0x0073, 0x0074,
	0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x007b, 0x007c, 0x007d,
	0x007e, 0x007f, 0x0080, 0x0081, 0x0082, 0x0083, 0x0084, 0x0085, 0x0086,
	0x0087, 0x0088, 0x0089, 0x008a, 0x008b, 0x008c, 0x008d, 0x008e, 0x008f,
	0x0090, 0x0091, 0x0092, 0x0093, 0x0094, 0x0095, 0x0096, 0x0097, 0x0098,
	0x0099, 0x009a, 0x009b, 0x009c, 0x009d, 0x009e, 0x009f, 0x00a0, 0x00a1,
	0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7, 0x00a8, 0x00a9, 0x00aa,
	0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x00af, 0x00b0, 0x00b1, 0x00b2, 0x00b3,
	0x00b4, 0x00b5, 0x00b6, 0x00b7, 0x00b8, 0x00b9, 0x00ba, 0x00bb, 0x00bc,
	0x00bd, 0x00be, 0x00bf, 0x00c0, 0x00c1, 0x00c2, 0x00c3, 0x00c4, 0x00c5,
	0x00c6, 0x00c7, 0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00cd, 0x00ce,
	0x00cf, 0x00d0, 0x00d1, 0x00d2, 0x00d3, 0x00d4, 0x00d5, 0x00d6, 0x00d7,
	0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de, 0x00df, 0x00e0,
	0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6, 0x00e7, 0x00e8, 0x00e9,
	0x00ea, 0x00eb, 0x00ec, 0x00ed, 0x00ee, 0x00ef, 0x00f0, 0x00f1, 0x00f2,
	0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7, 0x00f8, 0x00f9, 0x00fa, 0x00fb,
	0x00fc, 0x00fd, 0x00fe, 0x00ff, 0x0100, 0x0101, 0x0102, 0x0103, 0x0104,
	0x0105, 0x0106, 0x0107, 0x0108, 0x0109, 0x010a, 0x010b, 0x010c, 0x010d,
	0x010e, 0x010f, 0x0110, 0x0111, 0x0112, 0x0113, 0x0114, 0x0115, 0x0116,
	0x0117, 0x0118, 0x0119, 0x011a, 0x011b, 0x011c, 0x011d, 0x011e, 0x011f,
	0x0120, 0x0121, 0x0012, 0x0018, 0x0003, 0x0004, 0x0002, 0x0005, 0x000c,
	0x000f, 0x0010, 0x0011, 0x000a, 0x000b, 0x0019, 0x001a, 0x001b, 0x001c,
	0x001d, 0x001e, 0x001f, 0x0020, 0x0021, 0x0022, 0x0000, 0x0007, 0x0008,
	0x0009, 0x0001, 0x0003, 0x0001, 0x0000, 0x8001, 0x0000, 0x0000, 0x0001,
	0x8003, 0x0002, 0x0003, 0x8003, 0x0004, 0x0005, 0x8003, 0x0006, 0x0007,
	0x8003, 0x0008, 0x0009, 0x8001, 0x000a, 0x000b, 0x000c, 0x000d, 0x8001,
	0x000e, 0x8002, 0x000f, 0x8001, 0x0010, 0x8001, 0x0011, 0x0012, 0x0013,
	0x8001, 0x0014, 0x0015, 0x0016, 0x0017, 0x0018, 0x0019, 0x001a, 0x001b,
	0x001c, 0x001d, 0x001e, 0x001f, 0x0020, 0x8001, 0x0021, 0x0022, 0x0023,
	0x0024, 0x0025, 0x0026, 0x0027, 0x0028, 0x0029, 0x002a, 0x002b, 0x002c,
	0x002d, 0x8001, 0x002e, 0x002f, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034,
	0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x003a, 0x8001, 0x003b, 0x003c,
	0x003d, 0x003e, 0x003f, 0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045,
	0x8001, 0x0046, 0x0047, 0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d,
	0x004e, 0x004f, 0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056,
	0x0057, 0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f,
	0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068,
	0x0069, 0x8002, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070,
	0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x8003, 0x0077, 0x0078,
	0x0079, 0x007a, 0x007b, 0x007c, 0x007d, 0x007e, 0x007f, 0x0080, 0x8001,
	0x0081, 0x0082, 0x0083, 0x0084, 0x0085, 0x0086, 0x0087, 0x0088, 0x0089,
	0x008a, 0x8001, 0x008b, 0x008c, 0x008d, 0x008e, 0x008f, 0x0090, 0x0091,
	0x0092, 0x0093, 0x0094, 0x8001, 0x0095, 0x0096, 0x0097, 0x0098, 0x0099,
	0x009a, 0x009b, 0x009c, 0x009d, 0x009e, 0x8001, 0x009f, 0x00a0, 0x00a1,
	0x00a2, 0x00a3, 0x00a4, 0x00a5, 0x00a6, 0x00a7, 0x00a8, 0x8001, 0x00a9,
	0x00aa, 0x00ab, 0x00ac, 0x00ad, 0x00ae, 0x00af, 0x00b0, 0x00b1, 0x00b2,
	0x00b3, 0x00b4, 0x00b5, 0x00b6, 0x00b7, 0x00b8, 0x00b9, 0x00ba, 0x011b,
	0x8013, 0x011c, 0x011d, 0x8006, 0x011e, 0x011f, 0x8001, 0x0120, 0x8008,
	0x0121, 0x000b, 0x000c, 0x000d, 0x000e, 0x0003, 0x000f, 0x000a, 0x0005,
	0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017, 0x0008,
	0x0000, 0x0002, 0x000a, 0x0009, 0x0004, 0x0006, 0x0007, 0x0005, 0x0003,
	0x001f, 0x0001, 0x0002, 0x0020, 0x0021, 0x0005, 0x0006, 0x0022, 0x0023,
	0x0009, 0x000b, 0x000c, 0x000d, 0x000e, 0x000a, 0x0010, 0x0011, 0x0012,
	0x0013, 0x0015, 0x0016, 0x0024, 0x0017, 0x0025, 0x0019, 0x0026, 0x000f,
	0x001c, 0x001d, 0x001e, 0x0027, 0x001f, 0x0001, 0x0002, 0x0003, 0x0004,
	0x0005, 0x0020, 0x0021, 0x0022, 0x0009, 0x000a, 0x000b, 0x000d, 0x000e,
	0x000c, 0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0018,
	0x0019, 0x0023, 0x001a, 0x0024, 0x000f, 0x001d, 0x001e, 0x0025, 0x001c,
	0x0027, 0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
	0x0030, 0x001f, 0x0001, 0x0002, 0x0020, 0x0021, 0x0022, 0x0006, 0x0007,
	0x0023, 0x0024, 0x000a, 0x000c, 0x000d, 0x000e, 0x000f, 0x0010, 0x0011,
	0x0012, 0x0013, 0x0015, 0x0016, 0x0025, 0x0017, 0x0026, 0x0019, 0x0027,
	0x000b, 0x001c, 0x001d, 0x001e, 0x0028, 0x000f, 0x0010, 0x0011, 0x0004,
	0x0005, 0x0006, 0x0007, 0x0008, 0x0009, 0x0012, 0x000e, 0x0049, 0x004a,
	0x004b, 0x004c, 0x004d, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x005b,
	0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007, 0x0008,
	0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f, 0x0010, 0x0011,
	0x0012, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017, 0x0018, 0x0019, 0x001a,
	0x001b, 0x001c, 0x8001, 0x001d, 0x001e, 0x0020, 0x0021, 0x8001, 0x0022,
	0x0023, 0x8001, 0x0024, 0x0025, 0x0026, 0x0027, 0x0028, 0x0029, 0x002a,
	0x002b, 0x002c, 0x002d, 0x002e, 0x002f, 0x0030, 0x0031, 0x0032, 0x0033,
	0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x003a, 0x003b, 0x003c,
	0x003d, 0x8001, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f, 0x0060,
	0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069,
	0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, 0x0070, 0x0071, 0x0072,
	0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x007b,
	0x007c, 0x007d, 0x007e, 0x007f, 0x0080, 0x0081, 0x0082, 0x0083, 0x0084,
	0x0085, 0x0086, 0x8005, 0x0087, 0x0088, 0x0089, 0x008a, 0x008b, 0x008c,
	0x008d, 0x008e, 0x008f, 0x0090, 0x0091, 0x0092, 0x0093, 0x0095, 0x8016,
	0x0096, 0x0097, 0x0098, 0x0099, 0x8002, 0x009a, 0x009b, 0x009c, 0x8001,
	0x00cf, 0x00d0, 0x00d1, 0x00d2, 0x00d3, 0x8006, 0x00d4, 0x00d5, 0x00d6,
	0x00d7, 0x00d8, 0x8005, 0x00d9, 0x00da, 0x00db, 0x00dc, 0x00dd, 0x00de,
	0x00df, 0x00e0, 0x800a, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e6,
	0x00e7, 0x00e8, 0x0005, 0x0006, 0x0007, 0x0008, 0x0009, 0x000a, 0x000b,
	0x000c, 0x000d, 0x000e, 0x000f, 0x0010, 0x0011, 0x0012, 0x0013, 0x0014,
	0x0018, 0x0017, 0x0024, 0x0025, 0x0026, 0x0027, 0x0028, 0x0029, 0x002a,
	0x002b, 0x002c, 0x002d, 0x002e, 0x002f, 0x0030, 0x0046, 0x0047, 0x0048,
	0x0045, 0x0037, 0x003a, 0x0034, 0x0035, 0x0036, 0x003d, 0x003e, 0x003f,
	0x0040, 0x0049, 0x0031, 0x0032, 0x0033, 0x0024, 0x0025, 0x0026, 0x0027,
	0x0028, 0x0029, 0x002a, 0x002b, 0x002f, 0x0034, 0x0035, 0x0036, 0x0037,
	0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f, 0x0040,
	0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0021, 0x0022, 0x0023, 0x0024,
	0x000d, 0x0025, 0x0026, 0x0027, 0x0028, 0x0029, 0x002a, 0x002b, 0x002c,
	0x002d, 0x002e, 0x002f, 0x000f, 0x0010, 0x0016, 0x0030, 0x0031, 0x0032,
	0x0033, 0x0034, 0x0015, 0x000e, 0x0035, 0x0036, 0x0017, 0x0012, 0x0013,
	0x0014, 0x0011, 0x004a, 0x0048, 0x0049, 0x0037, 0x0038, 0x0039, 0x0028,
	0x0029, 0x002f, 0x0026, 0x0030, 0x0032, 0x002b, 0x002e, 0x003a, 0x003b,
	0x003c, 0x003d, 0x0035, 0x0036, 0x002c, 0x0033, 0x003e, 0x003f, 0x0040,
	0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049,
	0x0010, 0x0011, 0x8001, 0x0008, 0x0005, 0x0007, 0x0002, 0x0003, 0x0018,
	0x0019, 0x0010, 0x001a, 0x0008, 0x001b, 0x000a, 0x000b, 0x000d, 0x0007,
	0x001c, 0x0005, 0x000e, 0x0012, 0x0006, 0x0013, 0x0009, 0x0014, 0x000f,
	0x0011, 0x0015, 0x001d, 0x0000, 0x0001, 0x8001, 0x0002, 0x8001, 0x0003,
	0x0004, 0x8001, 0x0005, 0x8001, 0x0006, 0x0007, 0x8001, 0x0008, 0x8001,
	0x0009, 0x000a, 0x8001, 0x000b, 0x8001, 0x000c, 0x000d, 0x8001, 0x000e,
	0x8004, 0x000f, 0x8001, 0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015,
	0x0016, 0x0017, 0x0018, 0x0019, 0x001a, 0x001b, 0x8001, 0x001c, 0x001d,
	0x001e, 0x001f, 0x801c, 0x0020, 0x0021, 0x0022, 0x0023, 0x0044, 0x8003,
	0x0045, 0x8002, 0x0046, 0x0047, 0x0048, 0x0049, 0x004a, 0x004b, 0x004c,
	0x004d, 0x004e, 0x004f, 0x8002, 0x0050, 0x8001, 0x0051, 0x8001, 0x0052,
	0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x801e,
	0x007e, 0x8008, 0x007f, 0x8009, 0x0080, 0x0005, 0x0006, 0x0007, 0x0008,
	0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f, 0x0010, 0x0011,
	0x0012, 0x0013, 0x0014, 0x0023, 0x0019, 0x0022, 0x0021, 0x0020, 0x001f,
	0x001e, 0x0001, 0x0003, 0x0004, 0x0005, 0x0006, 0x001d, 0x0007, 0x000a,
	0x000b, 0x001e, 0x001f, 0x0020, 0x0021, 0x000e, 0x0011, 0x0012, 0x0022,
	0x0023, 0x0024, 0x0025, 0x0015, 0x0016, 0x0026, 0x0027, 0x0028, 0x0029,
	0x0017, 0x0018, 0x0019, 0x001a, 0x001c, 0x0000, 0x0001, 0x0002, 0x0005,
	0x0006, 0x0007, 0x0008, 0x0009, 0x000a, 0x000d, 0x000b, 0x000c, 0x000e,
	0x0020, 0x0021, 0x0004, 0x0005, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026,
	0x0027, 0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f,
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0009,
	0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x0055, 0x003d, 0x0056, 0x003e,
	0x0057, 0x003f, 0x0058, 0x0020, 0x0059, 0x0021, 0x0008, 0x0040, 0x000b,
	0x0041, 0x000a, 0x0042, 0x000d, 0x0043, 0x000c, 0x0044, 0x005a, 0x0045,
	0x000e, 0x0046, 0x0011, 0x0047, 0x0012, 0x0048, 0x0013, 0x0049, 0x0014,
	0x004a, 0x0015, 0x004b, 0x0016, 0x0025, 0x0017, 0x0024, 0x0007, 0x004c,
	0x0006, 0x004d, 0x0018, 0x004e, 0x0019, 0x004f, 0x001a, 0x0050, 0x001b,
	0x0051, 0x001c, 0x0030, 0x001d, 0x0031, 0x000f, 0x0052, 0x0010, 0x0053,
	0x005b, 0x0054, 0x005c, 0x004b, 0x001f, 0x006c, 0x006d, 0x0068, 0x0069,
	0x0001, 0x003c, 0x003d, 0x0040, 0x004e, 0x004f, 0x0062, 0x0002, 0x0003,
	0x006e, 0x003e, 0x003f, 0x0041, 0x0050, 0x0051, 0x0060, 0x006f, 0x0063,
	0x0070, 0x0054, 0x005d, 0x0066, 0x0067, 0x005e, 0x005f, 0x005d, 0x005e,
	0x005f, 0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067,
	0x005b, 0x005c, 0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x003b,
	0x0055, 0x0056, 0x0057, 0x0037, 0x0029, 0x002a, 0x002b, 0x0058, 0x0059,
	0x0023, 0x0022, 0x002c, 0x006a, 0x002d, 0x006b, 0x002e, 0x0052, 0x002f,
	0x0053, 0x0028, 0x004c, 0x0032, 0x004d, 0x005a, 0x0064, 0x0038, 0x0065,
	0x0039, 0x0046, 0x003a, 0x0042, 0x0033, 0x0045, 0x0034, 0x0047, 0x0035,
	0x0048, 0x0036, 0x0049, 0x004a, 0x0044, 0x0043, 0x0000, 0x8003, 0x0001,
	0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007, 0x0008, 0x0009, 0x000a,
	0x000b, 0x000c, 0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006,
	0x0007, 0x0008, 0x8007, 0x0009, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e,
	0x000f, 0x0010, 0x8001, 0x000a, 0x000b, 0x000c, 0x000d, 0x000e, 0x000f,
	0x0010, 0x0011, 0x0012, 0x8001, 0x0013, 0x0014, 0x0015, 0x0016, 0x0017,
	0x0018, 0x0019, 0x001a, 0x001b, 0x8001, 0x001c, 0x001d, 0x001e, 0x001f,
	0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0000, 0x8001, 0x0001,
};

/* Data size: 2860 bytes */

/* Blocks debug user data */
static const u32 dbg_blocks_user[] = {
	0x00637267, 0x00000000, 0x00000000, 0x00000000,	/* grc */
	0x6373696d, 0x00000073, 0x00000000, 0x00000000,	/* miscs */
	0x6373696d, 0x00000000, 0x00000000, 0x00000000,	/* misc */
	0x00756264, 0x00000000, 0x00000000, 0x00000000,	/* dbu */
	0x756c6770, 0x00625f65, 0x00000000, 0x00000000,	/* pglue_b */
	0x67696e63, 0x00000000, 0x00000000, 0x00000000,	/* cnig */
	0x756d7063, 0x00000000, 0x00000000, 0x00000000,	/* cpmu */
	0x6973636e, 0x00000000, 0x00000000, 0x00000000,	/* ncsi */
	0x6574706f, 0x00000000, 0x00000000, 0x00000000,	/* opte */
	0x00626d62, 0x00000000, 0x00000000, 0x00000000,	/* bmb */
	0x65696370, 0x00000000, 0x00000000, 0x00000000,	/* pcie */
	0x0070636d, 0x00000000, 0x00000000, 0x00000000,	/* mcp */
	0x3270636d, 0x00000000, 0x00000000, 0x00000000,	/* mcp2 */
	0x68777370, 0x00007473, 0x00000000, 0x00000000,	/* pswhst */
	0x68777370, 0x00327473, 0x00000000, 0x00000000,	/* pswhst2 */
	0x72777370, 0x00000064, 0x00000000, 0x00000000,	/* pswrd */
	0x72777370, 0x00003264, 0x00000000, 0x00000000,	/* pswrd2 */
	0x77777370, 0x00000072, 0x00000000, 0x00000000,	/* pswwr */
	0x77777370, 0x00003272, 0x00000000, 0x00000000,	/* pswwr2 */
	0x72777370, 0x00000071, 0x00000000, 0x00000000,	/* pswrq */
	0x72777370, 0x00003271, 0x00000000, 0x00000000,	/* pswrq2 */
	0x636c6770, 0x00000073, 0x00000000, 0x00000000,	/* pglcs */
	0x65616d64, 0x00000000, 0x00000000, 0x00000000,	/* dmae */
	0x00757470, 0x00000000, 0x00000000, 0x00000000,	/* ptu */
	0x006d6374, 0x00000000, 0x00000000, 0x00000000,	/* tcm */
	0x006d636d, 0x00000000, 0x00000000, 0x00000000,	/* mcm */
	0x006d6375, 0x00000000, 0x00000000, 0x00000000,	/* ucm */
	0x006d6378, 0x00000000, 0x00000000, 0x00000000,	/* xcm */
	0x006d6379, 0x00000000, 0x00000000, 0x00000000,	/* ycm */
	0x006d6370, 0x00000000, 0x00000000, 0x00000000,	/* pcm */
	0x00006d71, 0x00000000, 0x00000000, 0x00000000,	/* qm */
	0x00006d74, 0x00000000, 0x00000000, 0x00000000,	/* tm */
	0x71726f64, 0x00000000, 0x00000000, 0x00000000,	/* dorq */
	0x00627262, 0x00000000, 0x00000000, 0x00000000,	/* brb */
	0x00637273, 0x00000000, 0x00000000, 0x00000000,	/* src */
	0x00737270, 0x00000000, 0x00000000, 0x00000000,	/* prs */
	0x6d647374, 0x00000000, 0x00000000, 0x00000000,	/* tsdm */
	0x6d64736d, 0x00000000, 0x00000000, 0x00000000,	/* msdm */
	0x6d647375, 0x00000000, 0x00000000, 0x00000000,	/* usdm */
	0x6d647378, 0x00000000, 0x00000000, 0x00000000,	/* xsdm */
	0x6d647379, 0x00000000, 0x00000000, 0x00000000,	/* ysdm */
	0x6d647370, 0x00000000, 0x00000000, 0x00000000,	/* psdm */
	0x6d657374, 0x00000000, 0x00000000, 0x00000000,	/* tsem */
	0x6d65736d, 0x00000000, 0x00000000, 0x00000000,	/* msem */
	0x6d657375, 0x00000000, 0x00000000, 0x00000000,	/* usem */
	0x6d657378, 0x00000000, 0x00000000, 0x00000000,	/* xsem */
	0x6d657379, 0x00000000, 0x00000000, 0x00000000,	/* ysem */
	0x6d657370, 0x00000000, 0x00000000, 0x00000000,	/* psem */
	0x00737372, 0x00000000, 0x00000000, 0x00000000,	/* rss */
	0x646c6d74, 0x00000000, 0x00000000, 0x00000000,	/* tmld */
	0x646c756d, 0x00000000, 0x00000000, 0x00000000,	/* muld */
	0x646c7579, 0x00000000, 0x00000000, 0x00000000,	/* yuld */
	0x646c7978, 0x00000000, 0x00000000, 0x00000000,	/* xyld */
	0x006d7270, 0x00000000, 0x00000000, 0x00000000,	/* prm */
	0x5f666270, 0x00316270, 0x00000000, 0x00000000,	/* pbf_pb1 */
	0x5f666270, 0x00326270, 0x00000000, 0x00000000,	/* pbf_pb2 */
	0x00627072, 0x00000000, 0x00000000, 0x00000000,	/* rpb */
	0x00627462, 0x00000000, 0x00000000, 0x00000000,	/* btb */
	0x00666270, 0x00000000, 0x00000000, 0x00000000,	/* pbf */
	0x66696472, 0x00000000, 0x00000000, 0x00000000,	/* rdif */
	0x66696474, 0x00000000, 0x00000000, 0x00000000,	/* tdif */
	0x00756463, 0x00000000, 0x00000000, 0x00000000,	/* cdu */
	0x63666363, 0x00000000, 0x00000000, 0x00000000,	/* ccfc */
	0x63666374, 0x00000000, 0x00000000, 0x00000000,	/* tcfc */
	0x00756769, 0x00000000, 0x00000000, 0x00000000,	/* igu */
	0x00756163, 0x00000000, 0x00000000, 0x00000000,	/* cau */
	0x63616d75, 0x00000000, 0x00000000, 0x00000000,	/* umac */
	0x63616d78, 0x00000000, 0x00000000, 0x00000000,	/* xmac */
	0x6174736d, 0x00000074, 0x00000000, 0x00000000,	/* mstat */
	0x00676264, 0x00000000, 0x00000000, 0x00000000,	/* dbg */
	0x0067696e, 0x00000000, 0x00000000, 0x00000000,	/* nig */
	0x006c6f77, 0x00000000, 0x00000000, 0x00000000,	/* wol */
	0x6e626d62, 0x00000000, 0x00000000, 0x00000000,	/* bmbn */
	0x00637069, 0x00000000, 0x00000000, 0x00000000,	/* ipc */
	0x006d776e, 0x00000000, 0x00000000, 0x00000000,	/* nwm */
	0x0073776e, 0x00000000, 0x00000000, 0x00000000,	/* nws */
	0x0000736d, 0x00000000, 0x00000000, 0x00000000,	/* ms */
	0x5f796870, 0x65696370, 0x00000000, 0x00000000,	/* phy_pcie */
	0x0064656c, 0x00000000, 0x00000000, 0x00000000,	/* led */
	0x5f737661, 0x70617277, 0x00000000, 0x00000000,	/* avs_wrap */
	0x72707870, 0x75627165, 0x00000073, 0x00000000,	/* pxpreqbus */
	0x30726162, 0x70616d5f, 0x00000000, 0x00000000,	/* bar0_map */
	0x5f70636d, 0x006f6966, 0x00000000, 0x00000000,	/* mcp_fio */
	0x7473616c, 0x696e695f, 0x00000074, 0x00000000,	/* last_init */
	0x5f737270, 0x00006366, 0x00000000, 0x00000000,	/* prs_fc */
	0x5f666270, 0x00006366, 0x00000000, 0x00000000,	/* pbf_fc */
	0x5f67696e, 0x665f626c, 0x00000063, 0x00000000,	/* nig_lb_fc */
	0x5f67696e, 0x665f626c, 0x6c705f63, 0x0000686c,	/* nig_lb_fc_pllh */
	0x5f67696e, 0x665f7874, 0x6c705f63, 0x0000686c,	/* nig_tx_fc_pllh */
	0x5f67696e, 0x665f7874, 0x00000063, 0x00000000,	/* nig_tx_fc */
	0x5f67696e, 0x665f7872, 0x6c705f63, 0x0000686c,	/* nig_rx_fc_pllh */
	0x5f67696e, 0x665f7872, 0x00000063, 0x00000000,	/* nig_rx_fc */
};

/* Data size: 1472 bytes */

/* Blocks debug user data per chip */
static const u32 dbg_blocks_user_per_chip[] = {
	0x0000000f,		/* grc, bb, 15 lines */
	0x0000000f,		/* grc, k2, 15 lines */
	0x00000000,		/* miscs, bb */
	0x00000000,		/* miscs, k2 */
	0x00000000,		/* misc, bb */
	0x00000000,		/* misc, k2 */
	0x00000000,		/* dbu, bb */
	0x00000000,		/* dbu, k2 */
	0x000f0127,		/* pglue_b, bb, 39 lines */
	0x0036012a,		/* pglue_b, k2, 42 lines */
	0x00000000,		/* cnig, bb */
	0x00000102,		/* cnig, k2, 2 lines */
	0x00000000,		/* cpmu, bb */
	0x00000000,		/* cpmu, k2 */
	0x00600101,		/* ncsi, bb, 1 lines */
	0x00600101,		/* ncsi, k2, 1 lines */
	0x00000000,		/* opte, bb */
	0x00000000,		/* opte, k2 (removed) */
	0x0061006a,		/* bmb, bb, 106 lines */
	0x0061006a,		/* bmb, k2, 106 lines */
	0x00000000,		/* pcie, bb */
	0x00000000,		/* pcie, k2 */
	0x00000000,		/* mcp, bb */
	0x00000000,		/* mcp, k2 */
	0x00cb0009,		/* mcp2, bb, 9 lines */
	0x00cb0009,		/* mcp2, k2, 9 lines */
	0x00d40104,		/* pswhst, bb, 4 lines */
	0x00d40104,		/* pswhst, k2, 4 lines */
	0x00d40103,		/* pswhst2, bb, 3 lines */
	0x00d40103,		/* pswhst2, k2, 3 lines */
	0x00000101,		/* pswrd, bb, 1 lines */
	0x00000101,		/* pswrd, k2, 1 lines */
	0x00d80119,		/* pswrd2, bb, 25 lines */
	0x00d80119,		/* pswrd2, k2, 25 lines */
	0x00f10109,		/* pswwr, bb, 9 lines */
	0x00f10109,		/* pswwr, k2, 9 lines */
	0x00000000,		/* pswwr2, bb */
	0x00000000,		/* pswwr2, k2 */
	0x00fa0001,		/* pswrq, bb, 1 lines */
	0x00fa0001,		/* pswrq, k2, 1 lines */
	0x00fb0015,		/* pswrq2, bb, 21 lines */
	0x00fb0015,		/* pswrq2, k2, 21 lines */
	0x00000000,		/* pglcs, bb */
	0x00000006,		/* pglcs, k2, 6 lines */
	0x01100001,		/* dmae, bb, 1 lines */
	0x01100001,		/* dmae, k2, 1 lines */
	0x00000105,		/* ptu, bb, 5 lines */
	0x00000105,		/* ptu, k2, 5 lines */
	0x01110113,		/* tcm, bb, 19 lines */
	0x01110113,		/* tcm, k2, 19 lines */
	0x01240113,		/* mcm, bb, 19 lines */
	0x01240113,		/* mcm, k2, 19 lines */
	0x01370116,		/* ucm, bb, 22 lines */
	0x01370116,		/* ucm, k2, 22 lines */
	0x014d0114,		/* xcm, bb, 20 lines */
	0x014d0114,		/* xcm, k2, 20 lines */
	0x01610112,		/* ycm, bb, 18 lines */
	0x01610112,		/* ycm, k2, 18 lines */
	0x0173010a,		/* pcm, bb, 10 lines */
	0x0173010a,		/* pcm, k2, 10 lines */
	0x017d0062,		/* qm, bb, 98 lines */
	0x017d0062,		/* qm, k2, 98 lines */
	0x01df0021,		/* tm, bb, 33 lines */
	0x01df0021,		/* tm, k2, 33 lines */
	0x02000107,		/* dorq, bb, 7 lines */
	0x02000107,		/* dorq, k2, 7 lines */
	0x02070162,		/* brb, bb, 98 lines */
	0x02070162,		/* brb, k2, 98 lines */
	0x02690019,		/* src, bb, 25 lines */
	0x0282001a,		/* src, k2, 26 lines */
	0x029c0104,		/* prs, bb, 4 lines */
	0x029c0104,		/* prs, k2, 4 lines */
	0x02a00133,		/* tsdm, bb, 51 lines */
	0x02a00133,		/* tsdm, k2, 51 lines */
	0x02a00133,		/* msdm, bb, 51 lines */
	0x02a00133,		/* msdm, k2, 51 lines */
	0x02a00133,		/* usdm, bb, 51 lines */
	0x02a00133,		/* usdm, k2, 51 lines */
	0x02a00133,		/* xsdm, bb, 51 lines */
	0x02a00133,		/* xsdm, k2, 51 lines */
	0x02a00133,		/* ysdm, bb, 51 lines */
	0x02a00133,		/* ysdm, k2, 51 lines */
	0x02a00133,		/* psdm, bb, 51 lines */
	0x02a00133,		/* psdm, k2, 51 lines */
	0x0000010c,		/* tsem, bb, 12 lines */
	0x0000010c,		/* tsem, k2, 12 lines */
	0x0000010c,		/* msem, bb, 12 lines */
	0x0000010c,		/* msem, k2, 12 lines */
	0x0000010c,		/* usem, bb, 12 lines */
	0x0000010c,		/* usem, k2, 12 lines */
	0x0000010c,		/* xsem, bb, 12 lines */
	0x0000010c,		/* xsem, k2, 12 lines */
	0x0000010c,		/* ysem, bb, 12 lines */
	0x0000010c,		/* ysem, k2, 12 lines */
	0x0000010c,		/* psem, bb, 12 lines */
	0x0000010c,		/* psem, k2, 12 lines */
	0x0000000d,		/* rss, bb, 13 lines */
	0x0000000d,		/* rss, k2, 13 lines */
	0x00000106,		/* tmld, bb, 6 lines */
	0x00000106,		/* tmld, k2, 6 lines */
	0x00000106,		/* muld, bb, 6 lines */
	0x00000106,		/* muld, k2, 6 lines */
	0x02d30005,		/* yuld, bb, 5 lines */
	0x02d30005,		/* yuld, k2, 5 lines */
	0x00000107,		/* xyld, bb, 7 lines */
	0x00000107,		/* xyld, k2, 7 lines */
	0x0000010e,		/* prm, bb, 14 lines */
	0x00fb0110,		/* prm, k2, 16 lines */
	0x0000000d,		/* pbf_pb1, bb, 13 lines */
	0x0000000d,		/* pbf_pb1, k2, 13 lines */
	0x0000000d,		/* pbf_pb2, bb, 13 lines */
	0x0000000d,		/* pbf_pb2, k2, 13 lines */
	0x0000000d,		/* rpb, bb, 13 lines */
	0x0000000d,		/* rpb, k2, 13 lines */
	0x02d80140,		/* btb, bb, 64 lines */
	0x02d80140,		/* btb, k2, 64 lines */
	0x02a00117,		/* pbf, bb, 23 lines */
	0x02a00117,		/* pbf, k2, 23 lines */
	0x00000006,		/* rdif, bb, 6 lines */
	0x00000006,		/* rdif, k2, 6 lines */
	0x00000006,		/* tdif, bb, 6 lines */
	0x00000006,		/* tdif, k2, 6 lines */
	0x03180003,		/* cdu, bb, 3 lines */
	0x031b000e,		/* cdu, k2, 14 lines */
	0x0329010d,		/* ccfc, bb, 13 lines */
	0x03360117,		/* ccfc, k2, 23 lines */
	0x0329010d,		/* tcfc, bb, 13 lines */
	0x03360117,		/* tcfc, k2, 23 lines */
	0x02a00133,		/* igu, bb, 51 lines */
	0x02a00133,		/* igu, k2, 51 lines */
	0x00000106,		/* cau, bb, 6 lines */
	0x00000106,		/* cau, k2, 6 lines */
	0x00000000,		/* umac, bb */
	0x00000006,		/* umac, k2, 6 lines */
	0x00000000,		/* xmac, bb */
	0x00000000,		/* xmac, k2 (removed) */
	0x00000000,		/* mstat, bb */
	0x00000000,		/* mstat, k2 */
	0x00000000,		/* dbg, bb */
	0x00000000,		/* dbg, k2 */
	0x02a0012b,		/* nig, bb, 43 lines */
	0x034d011d,		/* nig, k2, 29 lines */
	0x00000000,		/* wol, bb (removed) */
	0x00000002,		/* wol, k2, 2 lines */
	0x00000000,		/* bmbn, bb (removed) */
	0x00000008,		/* bmbn, k2, 8 lines */
	0x00000000,		/* ipc, bb */
	0x00000000,		/* ipc, k2 */
	0x00000000,		/* nwm, bb (removed) */
	0x0000000b,		/* nwm, k2, 11 lines */
	0x00000000,		/* nws, bb (removed) */
	0x00000000,		/* nws, k2 */
	0x00000000,		/* ms, bb (removed) */
	0x00000000,		/* ms, k2 */
	0x00000000,		/* phy_pcie, bb (removed) */
	0x00000000,		/* phy_pcie, k2 */
	0x00000000,		/* led, bb (removed) */
	0x00000000,		/* led, k2 */
	0x00000000,		/* avs_wrap, bb (removed) */
	0x00000000,		/* avs_wrap, k2 */
	0x00000000,		/* pxpreqbus, bb (removed) */
	0x00000000,		/* pxpreqbus, k2 */
	0x00000000,		/* bar0_map, bb */
	0x00000000,		/* bar0_map, k2 */
	0x00000000,		/* mcp_fio, bb */
	0x00000000,		/* mcp_fio, k2 */
	0x00000000,		/* last_init, bb */
	0x00000000,		/* last_init, k2 */
	0x036a0002,		/* prs_fc, bb, 2 lines */
	0x036a0002,		/* prs_fc, k2, 2 lines */
	0x036a0002,		/* pbf_fc, bb, 2 lines */
	0x036a0002,		/* pbf_fc, k2, 2 lines */
	0x036a0002,		/* nig_lb_fc, bb, 2 lines */
	0x036a0002,		/* nig_lb_fc, k2, 2 lines */
	0x036a0002,		/* nig_lb_fc_pllh, bb, 2 lines */
	0x00000000,		/* nig_lb_fc_pllh, k2 (removed) */
	0x036a0002,		/* nig_tx_fc_pllh, bb, 2 lines */
	0x00000000,		/* nig_tx_fc_pllh, k2 (removed) */
	0x036a0002,		/* nig_tx_fc, bb, 2 lines */
	0x036a0002,		/* nig_tx_fc, k2, 2 lines */
	0x036a0002,		/* nig_rx_fc_pllh, bb, 2 lines */
	0x00000000,		/* nig_rx_fc_pllh, k2 (removed) */
	0x036a0002,		/* nig_rx_fc, bb, 2 lines */
	0x036a0002,		/* nig_rx_fc, k2, 2 lines */
};

/* Data size: 736 bytes */

/* Debug Bus line name offsets */
static const u32 dbg_bus_line_name_offsets[] = {
	0x0002dabd, 0x0002dac3, 0x0002dac9, 0x0002dacf, 0x0002dad5, 0x0002dadb,
	0x0002dae1, 0x0002dae7, 0x0002daed, 0x0002daf3, 0x0002daf9, 0x0002db00,
	0x0002db07, 0x0002db0e, 0x0002db15, 0x0002dabd, 0x0002dac3, 0x0002dac9,
	0x0002dacf, 0x0002dad5, 0x0002dadb, 0x0002dae1, 0x0002dae7, 0x0002daed,
	0x0002db1c, 0x0002daf3, 0x0002daf9, 0x0002db00, 0x0002db07, 0x0002db0e,
	0x0002db15, 0x0002db23, 0x0002db2a, 0x0002db31, 0x0002db38, 0x0002db3f,
	0x0002db46, 0x0002db4d, 0x0002db54, 0x0002db5b, 0x0002db62, 0x0002db69,
	0x0002db70, 0x0002db77, 0x0002db7e, 0x0002db85, 0x0002db8c, 0x0002db93,
	0x0002db9a, 0x0002dba1, 0x0002dba8, 0x0002dbaf, 0x0002dbb6, 0x0002dbbd,
	0x0002dabd, 0x0002dac3, 0x0002dac9, 0x0002dacf, 0x0002dad5, 0x0002dadb,
	0x0002dae1, 0x0002dae7, 0x0002daed, 0x0002db1c, 0x0002daf3, 0x0002daf9,
	0x0002db00, 0x0002db07, 0x0002db0e, 0x0002db15, 0x0002db23, 0x0002db2a,
	0x0002db31, 0x0002db38, 0x0002db3f, 0x0002db46, 0x0002db4d, 0x0002db54,
	0x0002db5b, 0x0002db62, 0x0002db69, 0x0002db70, 0x0002db77, 0x0002db7e,
	0x0002db85, 0x0002db8c, 0x0002db93, 0x0002db9a, 0x0002dba1, 0x0002dba8,
	0x0002dbaf, 0x0002dbb6, 0x0002dbbd, 0x0002dbc4, 0x0002dbcb, 0x0002dbd2,
	0x0002dbd9, 0x0002dabd, 0x0002dac3, 0x0002dac9, 0x0002dacf, 0x0002dad5,
	0x0002dadb, 0x0002dae1, 0x0002dae7, 0x0002db1c, 0x0002dbe0, 0x0002dbe7,
	0x0002dbee, 0x0002daf9, 0x0002db00, 0x0002db07, 0x0002db0e, 0x0002db15,
	0x0002db23, 0x0002db2a, 0x0002db31, 0x0002db38, 0x0002db3f, 0x0002db46,
	0x0002db4d, 0x0002db54, 0x0002db5b, 0x0002db62, 0x0002db69, 0x0002db70,
	0x0002db77, 0x0002db7e, 0x0002db85, 0x0002db8c, 0x0002db93, 0x0002db9a,
	0x0002dba1, 0x0002dba8, 0x0002dbaf, 0x0002dbb6, 0x0002dbbd, 0x0002dbc4,
	0x0002dbcb, 0x0002dbd2, 0x0002dbf5, 0x0002dbfc, 0x0002dc03, 0x0002dc0a,
	0x0002dc11, 0x0002dc18, 0x0002dc1f, 0x0002dc26, 0x0002dc2d, 0x0002dc34,
	0x0002dc3b, 0x0002dc42, 0x0002dc49, 0x0002dc50, 0x0002dc57, 0x0002dc5e,
	0x0002dc65, 0x0002dc6c, 0x0002dc73, 0x0002dc7a, 0x0002dc81, 0x0002dc88,
	0x0002dc8f, 0x0002dc96, 0x0002dc9d, 0x0002dca4, 0x0002dcab, 0x0002dcb2,
	0x0002dcb9, 0x0002dcc0, 0x0002dcc7, 0x0002dcce, 0x0002dcd5, 0x0002dcdc,
	0x0002dce3, 0x0002dcea, 0x0002dcf1, 0x0002dcf8, 0x0002dcff, 0x0002dd06,
	0x0002dd0e, 0x0002dd16, 0x0002dd1e, 0x0002dd26, 0x0002dd2e, 0x0002dd36,
	0x0002dd3e, 0x0002dd46, 0x0002dd4e, 0x0002dd56, 0x0002dd5e, 0x0002dd66,
	0x0002dd6e, 0x0002dd76, 0x0002dd7e, 0x0002dd86, 0x0002dd8e, 0x0002dd96,
	0x0002dd9e, 0x0002dda6, 0x0002ddae, 0x0002ddb6, 0x0002ddbe, 0x0002ddc6,
	0x0002ddd2, 0x0002ddde, 0x0002dde5, 0x0002ddef, 0x0002ddfc, 0x0002de09,
	0x0002de12, 0x0002de1e, 0x0002dabd, 0x0002dac3, 0x0002dacf, 0x0002dad5,
	0x0002dabd, 0x0002dac3, 0x0002dac9, 0x0002dacf, 0x0002dad5, 0x0002dadb,
	0x0002de2b, 0x0002de32, 0x0002dae1, 0x0002dae7, 0x0002daed, 0x0002daf3,
	0x0002daf9, 0x0002db00, 0x0002db07, 0x0002db0e, 0x0002db15, 0x0002db23,
	0x0002db2a, 0x0002db31, 0x0002db38, 0x0002db3f, 0x0002db46, 0x0002db4d,
	0x0002db54, 0x0002de39, 0x0002de43, 0x0002de4d, 0x0002de57, 0x0002de62,
	0x0002de6b, 0x0002de76, 0x0002de81, 0x0002de8c, 0x0002de97, 0x0002dabd,
	0x0002dac3, 0x0002dac9, 0x0002dacf, 0x0002dad5, 0x0002dadb, 0x0002dae1,
	0x0002dae7, 0x0002daed, 0x0002daf3, 0x0002daf9, 0x0002db00, 0x0002db07,
	0x0002db0e, 0x0002db15, 0x0002db23, 0x0002db2a, 0x0002db31, 0x0002db38,
	0x0002db3f, 0x0002db46, 0x0002dea8, 0x0002deb0, 0x0002deb8, 0x0002dec2,
	0x0002decc, 0x0002ded2, 0x0002ded9, 0x0002dee0, 0x0002dee7, 0x0002deed,
	0x0002def4, 0x0002db54, 0x0002db5b, 0x0002db69, 0x0002db70, 0x0002db77,
	0x0002db7e, 0x0002db85, 0x0002db8c, 0x0002db93, 0x0002deb0, 0x0002deb8,
	0x0002dec2, 0x0002def9, 0x0002df00, 0x0002decc, 0x0002df07, 0x0002df0e,
	0x0002ded2, 0x0002df15, 0x0002db54, 0x0002db5b, 0x0002db69, 0x0002db70,
	0x0002db77, 0x0002db7e, 0x0002db85, 0x0002db8c, 0x0002db93, 0x0002deb0,
	0x0002deb8, 0x0002dec2, 0x0002def9, 0x0002df00, 0x0002decc, 0x0002deed,
	0x0002df1c, 0x0002df23, 0x0002df2a, 0x0002df31, 0x0002df38, 0x0002def4,
	0x0002db54, 0x0002db5b, 0x0002db69, 0x0002db70, 0x0002db77, 0x0002db7e,
	0x0002db85, 0x0002db8c, 0x0002db93, 0x0002deb0, 0x0002deb8, 0x0002dec2,
	0x0002def9, 0x0002df00, 0x0002decc, 0x0002df0e, 0x0002ded2, 0x0002dee0,
	0x0002deed, 0x0002df38, 0x0002def4, 0x0002db54, 0x0002db5b, 0x0002db62,
	0x0002db69, 0x0002db70, 0x0002db77, 0x0002db7e, 0x0002db85, 0x0002deb0,
	0x0002deb8, 0x0002dec2, 0x0002df00, 0x0002decc, 0x0002df07, 0x0002df0e,
	0x0002dee0, 0x0002df3f, 0x0002db54, 0x0002db5b, 0x0002db69, 0x0002db70,
	0x0002db77, 0x0002db7e, 0x0002db85, 0x0002db8c, 0x0002db93, 0x0002deb0,
	0x0002decc, 0x0002df46, 0x0002db54, 0x0002db5b, 0x0002db62, 0x0002db69,
	0x0002db70, 0x0002db77, 0x0002db85, 0x0002dabd, 0x0002dac3, 0x0002dac9,
	0x0002dacf, 0x0002dad5, 0x0002de2b, 0x0002de32, 0x0002df4d, 0x0002df54,
	0x0002df5b, 0x0002df62, 0x0002db1c, 0x0002dbe0, 0x0002daf3, 0x0002daf9,
	0x0002db00, 0x0002db0e, 0x0002db15, 0x0002db23, 0x0002db2a, 0x0002db31,
	0x0002db38, 0x0002db3f, 0x0002db46, 0x0002db4d, 0x0002db54, 0x0002db5b,
	0x0002db62, 0x0002db69, 0x0002db70, 0x0002db77, 0x0002db7e, 0x0002db85,
	0x0002db8c, 0x0002db93, 0x0002db9a, 0x0002dba1, 0x0002dba8, 0x0002dbaf,
	0x0002dbb6, 0x0002dbbd, 0x0002dbc4, 0x0002dbcb, 0x0002dbd2, 0x0002dbf5,
	0x0002dbfc, 0x0002dc03, 0x0002dc0a, 0x0002dc11, 0x0002dc18, 0x0002dc1f,
	0x0002dc26, 0x0002dc2d, 0x0002dc34, 0x0002dc3b, 0x0002dc42, 0x0002dc49,
	0x0002dc50, 0x0002dc57, 0x0002df69, 0x0002df70, 0x0002df77, 0x0002df7e,
	0x0002df85, 0x0002df8c, 0x0002df93, 0x0002df9a, 0x0002dfa1, 0x0002dfa8,
	0x0002dc5e, 0x0002dc65, 0x0002dc6c, 0x0002dc73, 0x0002dc7a, 0x0002dc81,
	0x0002dc88, 0x0002dc8f, 0x0002dc96, 0x0002dc9d, 0x0002dca4, 0x0002dcab,
	0x0002dcb2, 0x0002dcb9, 0x0002dcc0, 0x0002dcc7, 0x0002dcce, 0x0002dcd5,
	0x0002dcdc, 0x0002dce3, 0x0002dfaf, 0x0002dfb6, 0x0002dfbd, 0x0002dfc5,
	0x0002dfcd, 0x0002dfd5, 0x0002dfdd, 0x0002dfe5, 0x0002dcff, 0x0002dabd,
	0x0002dac3, 0x0002dac9, 0x0002dacf, 0x0002dad5, 0x0002dadb, 0x0002dae1,
	0x0002dae7, 0x0002daed, 0x0002daf3, 0x0002daf9, 0x0002db00, 0x0002db07,
	0x0002db0e, 0x0002db15, 0x0002db23, 0x0002db2a, 0x0002db31, 0x0002db3f,
	0x0002db46, 0x0002db4d, 0x0002db54, 0x0002db5b, 0x0002db62, 0x0002db69,
	0x0002db70, 0x0002db77, 0x0002db7e, 0x0002db85, 0x0002db8c, 0x0002db93,
	0x0002db9a, 0x0002dba1, 0x0002dfed, 0x0002dffa, 0x0002e001, 0x0002e012,
	0x0002e025, 0x0002e02a, 0x0002e039, 0x0002dabd, 0x0002dac3, 0x0002dac9,
	0x0002db1c, 0x0002dbe0, 0x0002dbe7, 0x0002dbee, 0x0002daf9, 0x0002db2a,
	0x0002db31, 0x0002db38, 0x0002db3f, 0x0002db46, 0x0002db70, 0x0002db77,
	0x0002db7e, 0x0002db85, 0x0002db8c, 0x0002dbb6, 0x0002dbbd, 0x0002dbc4,
	0x0002dbcb, 0x0002dbd2, 0x0002dc18, 0x0002dc1f, 0x0002dc26, 0x0002dc2d,
	0x0002dc34, 0x0002dc3b, 0x0002dc42, 0x0002dc49, 0x0002df69, 0x0002df70,
	0x0002df77, 0x0002df7e, 0x0002df85, 0x0002df8c, 0x0002df93, 0x0002df9a,
	0x0002dc5e, 0x0002dc65, 0x0002dc6c, 0x0002dc73, 0x0002dc7a, 0x0002dc81,
	0x0002dc88, 0x0002dc8f, 0x0002dca4, 0x0002dcab, 0x0002dcb2, 0x0002dcb9,
	0x0002dcc0, 0x0002dcc7, 0x0002dcce, 0x0002dcd5, 0x0002dcea, 0x0002dcf1,
	0x0002dcf8, 0x0002dcff, 0x0002e043, 0x0002e04a, 0x0002e051, 0x0002e058,
	0x0002e05f, 0x0002e066, 0x0002e06d, 0x0002e074, 0x0002e07b, 0x0002e083,
	0x0002e08b, 0x0002e093, 0x0002dd06, 0x0002e09b, 0x0002e0a3, 0x0002e0ab,
	0x0002dd0e, 0x0002dd16, 0x0002dd1e, 0x0002dd26, 0x0002dd2e, 0x0002dd36,
	0x0002dd3e, 0x0002dd46, 0x0002dd4e, 0x0002dd56, 0x0002dd5e, 0x0002dd66,
	0x0002dd6e, 0x0002dd76, 0x0002dd7e, 0x0002dd86, 0x0002dd8e, 0x0002dd96,
	0x0002dd9e, 0x0002dda6, 0x0002ddae, 0x0002ddb6, 0x0002ddbe, 0x0002e0b3,
	0x0002e0b9, 0x0002e0bf, 0x0002e0c5, 0x0002e0cb, 0x0002e0d1, 0x0002e0d7,
	0x0002e0dd, 0x0002e0e3, 0x0002e0e9, 0x0002e0ef, 0x0002e0f6, 0x0002e0fd,
	0x0002e104, 0x0002e10b, 0x0002e112, 0x0002e119, 0x0002e120, 0x0002e127,
	0x0002e12e, 0x0002e135, 0x0002e13c, 0x0002e143, 0x0002e14a, 0x0002e151,
	0x0002e158, 0x0002e0b3, 0x0002e0b9, 0x0002e0bf, 0x0002e0c5, 0x0002e0cb,
	0x0002e0d1, 0x0002e0d7, 0x0002e0dd, 0x0002e0e3, 0x0002e0e9, 0x0002e0ef,
	0x0002e0f6, 0x0002e0fd, 0x0002e104, 0x0002e10b, 0x0002e112, 0x0002e119,
	0x0002e120, 0x0002e127, 0x0002e12e, 0x0002e135, 0x0002e13c, 0x0002e143,
	0x0002e14a, 0x0002e151, 0x0002e166, 0x0002e175, 0x0002e17e, 0x0002e18b,
	0x0002dabd, 0x0002dac3, 0x0002dac9, 0x0002dacf, 0x0002dad5, 0x0002dadb,
	0x0002dae1, 0x0002dae7, 0x0002daed, 0x0002daf3, 0x0002daf9, 0x0002db00,
	0x0002db07, 0x0002db0e, 0x0002db15, 0x0002db23, 0x0002db2a, 0x0002db31,
	0x0002db38, 0x0002db3f, 0x0002db46, 0x0002db4d, 0x0002db54, 0x0002db5b,
	0x0002db62, 0x0002db69, 0x0002db70, 0x0002db77, 0x0002db7e, 0x0002db85,
	0x0002db8c, 0x0002db93, 0x0002db9a, 0x0002dba1, 0x0002dba8, 0x0002dbaf,
	0x0002dbb6, 0x0002dbbd, 0x0002dbc4, 0x0002dbcb, 0x0002dbd2, 0x0002dbf5,
	0x0002dbfc, 0x0002dc03, 0x0002dc0a, 0x0002dc11, 0x0002dc18, 0x0002dc1f,
	0x0002dc26, 0x0002dc2d, 0x0002dc34, 0x0002dabd, 0x0002dac3, 0x0002dac9,
	0x0002dacf, 0x0002dadb, 0x0002dabd, 0x0002dac3, 0x0002dac9, 0x0002dacf,
	0x0002dad5, 0x0002dadb, 0x0002db1c, 0x0002dbe0, 0x0002dbe7, 0x0002dbee,
	0x0002daf9, 0x0002db00, 0x0002db07, 0x0002db0e, 0x0002db2a, 0x0002db31,
	0x0002db38, 0x0002db3f, 0x0002db46, 0x0002db4d, 0x0002db54, 0x0002db5b,
	0x0002db70, 0x0002db77, 0x0002db7e, 0x0002db85, 0x0002db8c, 0x0002db93,
	0x0002db9a, 0x0002dba1, 0x0002dbb6, 0x0002dc18, 0x0002df69, 0x0002dc5e,
	0x0002dca4, 0x0002dfaf, 0x0002dfb6, 0x0002dcea, 0x0002dcf1, 0x0002dcf8,
	0x0002dcff, 0x0002dd0e, 0x0002dd16, 0x0002dd1e, 0x0002dd26, 0x0002dd2e,
	0x0002dd36, 0x0002dd3e, 0x0002dd46, 0x0002dd4e, 0x0002dd56, 0x0002dd5e,
	0x0002dd66, 0x0002dd6e, 0x0002dd76, 0x0002dd7e, 0x0002dd86, 0x0002dd8e,
	0x0002dd96, 0x0002dd9e, 0x0002dda6, 0x0002ddae, 0x0002ddb6, 0x0002ddbe,
	0x0002e193, 0x0002e19f, 0x0002e1ab, 0x0002e1b6, 0x0002e1c5, 0x0002e1d4,
	0x0002e1e3, 0x0002e1f2, 0x0002e201, 0x0002e210, 0x0002e21f, 0x0002e22e,
	0x0002e23c, 0x0002e24c, 0x0002e193, 0x0002e19f, 0x0002e1ab, 0x0002e25c,
	0x0002e269, 0x0002e276, 0x0002e283, 0x0002e290, 0x0002e29d, 0x0002e2aa,
	0x0002e2b7, 0x0002e2c4, 0x0002e2d1, 0x0002e2e2, 0x0002e2f3, 0x0002e301,
	0x0002e25c, 0x0002e269, 0x0002e276, 0x0002e283, 0x0002e290, 0x0002e29d,
	0x0002e2aa, 0x0002e2b7, 0x0002e2c4, 0x0002e2d1, 0x0002e2e2, 0x0002e2f3,
	0x0002e301, 0x0002e30a, 0x0002e317, 0x0002e324, 0x0002e331, 0x0002e33e,
	0x0002e34d, 0x0002e35d, 0x0002e36f, 0x0002e37c, 0x0002e38d, 0x0002dabd,
	0x0002dac3, 0x0002dac9, 0x0002dacf, 0x0002dad5, 0x0002dadb, 0x0002dae1,
	0x0002dae7, 0x0002daed, 0x0002daf3, 0x0002daf9, 0x0002db00, 0x0002db07,
	0x0002db0e, 0x0002db15, 0x0002db23, 0x0002db2a, 0x0002db31, 0x0002db3f,
	0x0002db46, 0x0002db4d, 0x0002db54, 0x0002db62, 0x0002db69, 0x0002db70,
	0x0002db77, 0x0002db85, 0x0002db8c, 0x0002db93, 0x0002e39a, 0x0002e3a4,
};

/* Data size: 3504 bytes */

/* Debug strings */
static const u8 dbg_strings[] = {
	/* PGLCS: INT_STS reg is set */
	0x50, 0x47, 0x4c, 0x43, 0x53, 0x3a, 0x20, 0x49, 0x4e, 0x54, 0x5f, 0x53,
	0x54, 0x53, 0x20, 0x72, 0x65, 0x67, 0x20, 0x69, 0x73, 0x20, 0x73, 0x65,
	0x74, 0x00,
	/* PGLCS_REG_INT_STS */
	0x50, 0x47, 0x4c, 0x43, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e,
	0x54, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PGLCS_REG_INT_MASK */
	0x50, 0x47, 0x4c, 0x43, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e,
	0x54, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* MISC: interrupt status 0 is not 0. */
	0x4d, 0x49, 0x53, 0x43, 0x3a, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x72,
	0x75, 0x70, 0x74, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x30,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x2e, 0x00,
	/* MISC_REG_INT_STS */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54,
	0x5f, 0x53, 0x54, 0x53, 0x00,
	/* MISC_REG_INT_MASK */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* AEU: IGU: AFTER_INVERT_1 is not 0 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x49, 0x47, 0x55, 0x3a, 0x20, 0x41, 0x46,
	0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x31,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_1_IGU */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x31, 0x5f, 0x49, 0x47, 0x55, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* AEU: IGU: AFTER_INVERT_2 is not 0x90 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x49, 0x47, 0x55, 0x3a, 0x20, 0x41, 0x46,
	0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x32,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x78, 0x39, 0x30,
	0x00,
	/* MISC_REG_AEU_AFTER_INVERT_2_IGU */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x32, 0x5f, 0x49, 0x47, 0x55, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* AEU: IGU: AFTER_INVERT_4 is not 0 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x49, 0x47, 0x55, 0x3a, 0x20, 0x41, 0x46,
	0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x34,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_4_IGU */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x34, 0x5f, 0x49, 0x47, 0x55, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* AEU: IGU: AFTER_INVERT_5 is not 0 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x49, 0x47, 0x55, 0x3a, 0x20, 0x41, 0x46,
	0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x35,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_5_IGU */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x35, 0x5f, 0x49, 0x47, 0x55, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* AEU: IGU: AFTER_INVERT_6 is not 0 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x49, 0x47, 0x55, 0x3a, 0x20, 0x41, 0x46,
	0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x36,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_6_IGU */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x36, 0x5f, 0x49, 0x47, 0x55, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* AEU: IGU: AFTER_INVERT_7 is not 0 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x49, 0x47, 0x55, 0x3a, 0x20, 0x41, 0x46,
	0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x37,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_7_IGU */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x37, 0x5f, 0x49, 0x47, 0x55, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* AEU: IGU: AFTER_INVERT_8 is not 0x34000000 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x49, 0x47, 0x55, 0x3a, 0x20, 0x41, 0x46,
	0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x38,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x78, 0x33, 0x34,
	0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_8_IGU */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x38, 0x5f, 0x49, 0x47, 0x55, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* AEU: IGU: AFTER_INVERT_9 is not 0x60 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x49, 0x47, 0x55, 0x3a, 0x20, 0x41, 0x46,
	0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x39,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x78, 0x36, 0x30,
	0x00,
	/* MISC_REG_AEU_AFTER_INVERT_9_IGU */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x39, 0x5f, 0x49, 0x47, 0x55, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* AEU: MCP AFTER_INVERT_1 is not 0 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x4d, 0x43, 0x50, 0x20, 0x41, 0x46, 0x54,
	0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x31, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_1_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x31, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* AEU: MCP AFTER_INVERT_2 is not 0x90 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x4d, 0x43, 0x50, 0x20, 0x41, 0x46, 0x54,
	0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x32, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x78, 0x39, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_2_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x32, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* AEU: MCP AFTER_INVERT_4 is not 0 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x4d, 0x43, 0x50, 0x20, 0x41, 0x46, 0x54,
	0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x34, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_4_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x34, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* AEU: MCP AFTER_INVERT_5 is not 0 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x4d, 0x43, 0x50, 0x20, 0x41, 0x46, 0x54,
	0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x35, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_5_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x35, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* AEU: MCP AFTER_INVERT_6 is not 0 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x4d, 0x43, 0x50, 0x20, 0x41, 0x46, 0x54,
	0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x36, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_6_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x36, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* AEU: MCP AFTER_INVERT_7 is not 0 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x4d, 0x43, 0x50, 0x20, 0x41, 0x46, 0x54,
	0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x37, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_7_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x37, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* YSEM: PRAM Parity error */
	0x59, 0x53, 0x45, 0x4d, 0x3a, 0x20, 0x50, 0x52, 0x41, 0x4d, 0x20, 0x50,
	0x61, 0x72, 0x69, 0x74, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_8_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x38, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* YSEM_REG_FAST_MEMORY_PRAM_PRTY_ADDR_HIGH */
	0x59, 0x53, 0x45, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x41, 0x53,
	0x54, 0x5f, 0x4d, 0x45, 0x4d, 0x4f, 0x52, 0x59, 0x5f, 0x50, 0x52, 0x41,
	0x4d, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x5f,
	0x48, 0x49, 0x47, 0x48, 0x00,
	/* YSEM_REG_FAST_MEMORY_PRAM_PRTY_ADDR_LOW */
	0x59, 0x53, 0x45, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x41, 0x53,
	0x54, 0x5f, 0x4d, 0x45, 0x4d, 0x4f, 0x52, 0x59, 0x5f, 0x50, 0x52, 0x41,
	0x4d, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x5f,
	0x4c, 0x4f, 0x57, 0x00,
	/* USEM: PRAM Parity error */
	0x55, 0x53, 0x45, 0x4d, 0x3a, 0x20, 0x50, 0x52, 0x41, 0x4d, 0x20, 0x50,
	0x61, 0x72, 0x69, 0x74, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_8_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x38, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* USEM_REG_FAST_MEMORY_PRAM_PRTY_ADDR_HIGH */
	0x55, 0x53, 0x45, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x41, 0x53,
	0x54, 0x5f, 0x4d, 0x45, 0x4d, 0x4f, 0x52, 0x59, 0x5f, 0x50, 0x52, 0x41,
	0x4d, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x5f,
	0x48, 0x49, 0x47, 0x48, 0x00,
	/* USEM_REG_FAST_MEMORY_PRAM_PRTY_ADDR_LOW */
	0x55, 0x53, 0x45, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x41, 0x53,
	0x54, 0x5f, 0x4d, 0x45, 0x4d, 0x4f, 0x52, 0x59, 0x5f, 0x50, 0x52, 0x41,
	0x4d, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x5f,
	0x4c, 0x4f, 0x57, 0x00,
	/* AEU: MCP AFTER_INVERT_8 is not 0x37f00000 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x4d, 0x43, 0x50, 0x20, 0x41, 0x46, 0x54,
	0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x38, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x78, 0x33, 0x37, 0x66,
	0x30, 0x30, 0x30, 0x30, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_8_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x38, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* XSEM: PRAM Parity error */
	0x58, 0x53, 0x45, 0x4d, 0x3a, 0x20, 0x50, 0x52, 0x41, 0x4d, 0x20, 0x50,
	0x61, 0x72, 0x69, 0x74, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_8_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x38, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* XSEM_REG_FAST_MEMORY_PRAM_PRTY_ADDR_HIGH */
	0x58, 0x53, 0x45, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x41, 0x53,
	0x54, 0x5f, 0x4d, 0x45, 0x4d, 0x4f, 0x52, 0x59, 0x5f, 0x50, 0x52, 0x41,
	0x4d, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x5f,
	0x48, 0x49, 0x47, 0x48, 0x00,
	/* XSEM_REG_FAST_MEMORY_PRAM_PRTY_ADDR_LOW */
	0x58, 0x53, 0x45, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x41, 0x53,
	0x54, 0x5f, 0x4d, 0x45, 0x4d, 0x4f, 0x52, 0x59, 0x5f, 0x50, 0x52, 0x41,
	0x4d, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x5f,
	0x4c, 0x4f, 0x57, 0x00,
	/* MSEM: PRAM Parity error */
	0x4d, 0x53, 0x45, 0x4d, 0x3a, 0x20, 0x50, 0x52, 0x41, 0x4d, 0x20, 0x50,
	0x61, 0x72, 0x69, 0x74, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_8_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x38, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* MSEM_REG_FAST_MEMORY_PRAM_PRTY_ADDR_HIGH */
	0x4d, 0x53, 0x45, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x41, 0x53,
	0x54, 0x5f, 0x4d, 0x45, 0x4d, 0x4f, 0x52, 0x59, 0x5f, 0x50, 0x52, 0x41,
	0x4d, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x5f,
	0x48, 0x49, 0x47, 0x48, 0x00,
	/* MSEM_REG_FAST_MEMORY_PRAM_PRTY_ADDR_LOW */
	0x4d, 0x53, 0x45, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x41, 0x53,
	0x54, 0x5f, 0x4d, 0x45, 0x4d, 0x4f, 0x52, 0x59, 0x5f, 0x50, 0x52, 0x41,
	0x4d, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x5f,
	0x4c, 0x4f, 0x57, 0x00,
	/* TSEM: PRAM Parity error */
	0x54, 0x53, 0x45, 0x4d, 0x3a, 0x20, 0x50, 0x52, 0x41, 0x4d, 0x20, 0x50,
	0x61, 0x72, 0x69, 0x74, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_8_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x38, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* TSEM_REG_FAST_MEMORY_PRAM_PRTY_ADDR_HIGH */
	0x54, 0x53, 0x45, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x41, 0x53,
	0x54, 0x5f, 0x4d, 0x45, 0x4d, 0x4f, 0x52, 0x59, 0x5f, 0x50, 0x52, 0x41,
	0x4d, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x5f,
	0x48, 0x49, 0x47, 0x48, 0x00,
	/* TSEM_REG_FAST_MEMORY_PRAM_PRTY_ADDR_LOW */
	0x54, 0x53, 0x45, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x41, 0x53,
	0x54, 0x5f, 0x4d, 0x45, 0x4d, 0x4f, 0x52, 0x59, 0x5f, 0x50, 0x52, 0x41,
	0x4d, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x5f,
	0x4c, 0x4f, 0x57, 0x00,
	/* PSEM: PRAM Parity error */
	0x50, 0x53, 0x45, 0x4d, 0x3a, 0x20, 0x50, 0x52, 0x41, 0x4d, 0x20, 0x50,
	0x61, 0x72, 0x69, 0x74, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_8_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x38, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* PSEM_REG_FAST_MEMORY_PRAM_PRTY_ADDR_HIGH */
	0x50, 0x53, 0x45, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x41, 0x53,
	0x54, 0x5f, 0x4d, 0x45, 0x4d, 0x4f, 0x52, 0x59, 0x5f, 0x50, 0x52, 0x41,
	0x4d, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x5f,
	0x48, 0x49, 0x47, 0x48, 0x00,
	/* PSEM_REG_FAST_MEMORY_PRAM_PRTY_ADDR_LOW */
	0x50, 0x53, 0x45, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x41, 0x53,
	0x54, 0x5f, 0x4d, 0x45, 0x4d, 0x4f, 0x52, 0x59, 0x5f, 0x50, 0x52, 0x41,
	0x4d, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x5f,
	0x4c, 0x4f, 0x57, 0x00,
	/* AEU: MCP AFTER_INVERT_9 is not 0x60 */
	0x41, 0x45, 0x55, 0x3a, 0x20, 0x4d, 0x43, 0x50, 0x20, 0x41, 0x46, 0x54,
	0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52, 0x54, 0x5f, 0x39, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x78, 0x36, 0x30, 0x00,
	/* MISC_REG_AEU_AFTER_INVERT_9_MCP */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x41, 0x46, 0x54, 0x45, 0x52, 0x5f, 0x49, 0x4e, 0x56, 0x45, 0x52,
	0x54, 0x5f, 0x39, 0x5f, 0x4d, 0x43, 0x50, 0x00,
	/* MISC_REG_ATTN_NUM_ST */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54,
	0x4e, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x53, 0x54, 0x00,
	/* MISC: system kill occurred; */
	0x4d, 0x49, 0x53, 0x43, 0x3a, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6d,
	0x20, 0x6b, 0x69, 0x6c, 0x6c, 0x20, 0x6f, 0x63, 0x63, 0x75, 0x72, 0x72,
	0x65, 0x64, 0x3b, 0x00,
	/* MISC_REG_AEU_SYS_KILL_OCCURRED */
	0x4d, 0x49, 0x53, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x45, 0x55,
	0x5f, 0x53, 0x59, 0x53, 0x5f, 0x4b, 0x49, 0x4c, 0x4c, 0x5f, 0x4f, 0x43,
	0x43, 0x55, 0x52, 0x52, 0x45, 0x44, 0x00,
	/* MISCS: interrupt status 0 is not 0. */
	0x4d, 0x49, 0x53, 0x43, 0x53, 0x3a, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72,
	0x72, 0x75, 0x70, 0x74, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20,
	0x30, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x2e, 0x00,
	/* MISCS_REG_INT_STS_0 */
	0x4d, 0x49, 0x53, 0x43, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e,
	0x54, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x30, 0x00,
	/* MISCS_REG_INT_MASK_0 */
	0x4d, 0x49, 0x53, 0x43, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e,
	0x54, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x30, 0x00,
	/* MISCS: pcie_rst_b was asserted without perst assertion */
	0x4d, 0x49, 0x53, 0x43, 0x53, 0x3a, 0x20, 0x70, 0x63, 0x69, 0x65, 0x5f,
	0x72, 0x73, 0x74, 0x5f, 0x62, 0x20, 0x77, 0x61, 0x73, 0x20, 0x61, 0x73,
	0x73, 0x65, 0x72, 0x74, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x6f,
	0x75, 0x74, 0x20, 0x70, 0x65, 0x72, 0x73, 0x74, 0x20, 0x61, 0x73, 0x73,
	0x65, 0x72, 0x74, 0x69, 0x6f, 0x6e, 0x00,
	/* MISCS_REG_PCIE_HOT_RESET */
	0x4d, 0x49, 0x53, 0x43, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x43,
	0x49, 0x45, 0x5f, 0x48, 0x4f, 0x54, 0x5f, 0x52, 0x45, 0x53, 0x45, 0x54,
	0x00,
	/* DMAE: command 0 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x30, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C0 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x30, 0x00,
	/* DMAE: command 1 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x31, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C1 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x31, 0x00,
	/* DMAE: command 2 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x32, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C2 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x32, 0x00,
	/* DMAE: command 3 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x33, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C3 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x33, 0x00,
	/* DMAE: command 4 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x34, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C4 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x34, 0x00,
	/* DMAE: command 5 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x35, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C5 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x35, 0x00,
	/* DMAE: command 6 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x36, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C6 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x36, 0x00,
	/* DMAE: command 7 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x37, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C7 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x37, 0x00,
	/* DMAE: command 8 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x38, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C8 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x38, 0x00,
	/* DMAE: command 9 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x39, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C9 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x39, 0x00,
	/* DMAE: command 10 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x31, 0x30, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C10 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x31, 0x30, 0x00,
	/* DMAE: command 11 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x31, 0x31, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C11 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x31, 0x31, 0x00,
	/* DMAE: command 12 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x31, 0x32, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C12 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x31, 0x32, 0x00,
	/* DMAE: command 13 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x31, 0x33, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C13 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x31, 0x33, 0x00,
	/* DMAE: command 14 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x31, 0x34, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C14 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x31, 0x34, 0x00,
	/* DMAE: command 15 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x31, 0x35, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C15 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x31, 0x35, 0x00,
	/* DMAE: command 16 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x31, 0x36, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C16 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x31, 0x36, 0x00,
	/* DMAE: command 17 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x31, 0x37, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C17 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x31, 0x37, 0x00,
	/* DMAE: command 18 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x31, 0x38, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C18 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x31, 0x38, 0x00,
	/* DMAE: command 19 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x31, 0x39, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C19 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x31, 0x39, 0x00,
	/* DMAE: command 20 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x32, 0x30, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C20 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x32, 0x30, 0x00,
	/* DMAE: command 21 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x32, 0x31, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C21 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x32, 0x31, 0x00,
	/* DMAE: command 22 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x32, 0x32, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C22 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x32, 0x32, 0x00,
	/* DMAE: command 23 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x32, 0x33, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C23 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x32, 0x33, 0x00,
	/* DMAE: command 24 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x32, 0x34, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C24 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x32, 0x34, 0x00,
	/* DMAE: command 25 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x32, 0x35, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C25 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x32, 0x35, 0x00,
	/* DMAE: command 26 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x32, 0x36, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C26 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x32, 0x36, 0x00,
	/* DMAE: command 27 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x32, 0x37, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C27 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x32, 0x37, 0x00,
	/* DMAE: command 28 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x32, 0x38, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C28 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x32, 0x38, 0x00,
	/* DMAE: command 29 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x32, 0x39, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C29 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x32, 0x39, 0x00,
	/* DMAE: command 30 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x33, 0x30, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C30 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x33, 0x30, 0x00,
	/* DMAE: command 31 go is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e,
	0x64, 0x20, 0x33, 0x31, 0x20, 0x67, 0x6f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_GO_C31 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x47, 0x4f, 0x5f,
	0x43, 0x33, 0x31, 0x00,
	/* DMAE: INT status is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x49, 0x4e, 0x54, 0x20, 0x73, 0x74,
	0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20,
	0x30, 0x00,
	/* DMAE_REG_INT_STS */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54,
	0x5f, 0x53, 0x54, 0x53, 0x00,
	/* DMAE_REG_INT_MASK */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* DMAE: Parity status is not 0 */
	0x44, 0x4d, 0x41, 0x45, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79,
	0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DMAE_REG_PRTY_STS_H_0 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54,
	0x59, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* DMAE_REG_PRTY_MASK_H_0 */
	0x44, 0x4d, 0x41, 0x45, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54,
	0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* DBG: INT_STS is is not 0 */
	0x44, 0x42, 0x47, 0x3a, 0x20, 0x49, 0x4e, 0x54, 0x5f, 0x53, 0x54, 0x53,
	0x20, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30,
	0x00,
	/* DBG_REG_INT_STS */
	0x44, 0x42, 0x47, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x53, 0x54, 0x53, 0x00,
	/* DBG_REG_INT_MASK */
	0x44, 0x42, 0x47, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* DBG: Parity status is not 0 */
	0x44, 0x42, 0x47, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79, 0x20,
	0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* DBG_REG_PRTY_STS_H_0 */
	0x44, 0x42, 0x47, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* DBG_REG_PRTY_MASK_H_0 */
	0x44, 0x42, 0x47, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* GRC: GRC_REG_TRACE_FIFO_VALID_DATA = 1, the trace FIFO contains at least one valid data. */
	0x47, 0x52, 0x43, 0x3a, 0x20, 0x47, 0x52, 0x43, 0x5f, 0x52, 0x45, 0x47,
	0x5f, 0x54, 0x52, 0x41, 0x43, 0x45, 0x5f, 0x46, 0x49, 0x46, 0x4f, 0x5f,
	0x56, 0x41, 0x4c, 0x49, 0x44, 0x5f, 0x44, 0x41, 0x54, 0x41, 0x20, 0x3d,
	0x20, 0x31, 0x2c, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x72, 0x61, 0x63,
	0x65, 0x20, 0x46, 0x49, 0x46, 0x4f, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x61,
	0x69, 0x6e, 0x73, 0x20, 0x61, 0x74, 0x20, 0x6c, 0x65, 0x61, 0x73, 0x74,
	0x20, 0x6f, 0x6e, 0x65, 0x20, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20, 0x64,
	0x61, 0x74, 0x61, 0x2e, 0x00,
	/* GRC_REG_TRACE_FIFO_VALID_DATA */
	0x47, 0x52, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54, 0x52, 0x41, 0x43,
	0x45, 0x5f, 0x46, 0x49, 0x46, 0x4f, 0x5f, 0x56, 0x41, 0x4c, 0x49, 0x44,
	0x5f, 0x44, 0x41, 0x54, 0x41, 0x00,
	/* GRC: GRC_REG_INT_STS_0, one of the interrupts is set, bits 3:0: path_isolation_error, global_reserved_address, timeout_event, address_error. */
	0x47, 0x52, 0x43, 0x3a, 0x20, 0x47, 0x52, 0x43, 0x5f, 0x52, 0x45, 0x47,
	0x5f, 0x49, 0x4e, 0x54, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x30, 0x2c, 0x20,
	0x6f, 0x6e, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x69,
	0x6e, 0x74, 0x65, 0x72, 0x72, 0x75, 0x70, 0x74, 0x73, 0x20, 0x69, 0x73,
	0x20, 0x73, 0x65, 0x74, 0x2c, 0x20, 0x62, 0x69, 0x74, 0x73, 0x20, 0x33,
	0x3a, 0x30, 0x3a, 0x20, 0x70, 0x61, 0x74, 0x68, 0x5f, 0x69, 0x73, 0x6f,
	0x6c, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x65, 0x72, 0x72, 0x6f, 0x72,
	0x2c, 0x20, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x5f, 0x72, 0x65, 0x73,
	0x65, 0x72, 0x76, 0x65, 0x64, 0x5f, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73,
	0x73, 0x2c, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x6f, 0x75, 0x74, 0x5f, 0x65,
	0x76, 0x65, 0x6e, 0x74, 0x2c, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73,
	0x73, 0x5f, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x2e, 0x00,
	/* GRC: There is an interrupt on register 0 */
	0x47, 0x52, 0x43, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x69,
	0x73, 0x20, 0x61, 0x6e, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x72, 0x75,
	0x70, 0x74, 0x20, 0x6f, 0x6e, 0x20, 0x72, 0x65, 0x67, 0x69, 0x73, 0x74,
	0x65, 0x72, 0x20, 0x30, 0x00,
	/* GRC_REG_INT_STS_0 */
	0x47, 0x52, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x53, 0x54, 0x53, 0x5f, 0x30, 0x00,
	/* GRC_REG_INT_MASK_0 */
	0x47, 0x52, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x30, 0x00,
	/* GRC: Parity status is not 0 */
	0x47, 0x52, 0x43, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79, 0x20,
	0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* GRC_REG_PRTY_STS_H_0 */
	0x47, 0x52, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* GRC_REG_PRTY_MASK_H_0 */
	0x47, 0x52, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* MCP2: ROM Parity status is not 0 */
	0x4d, 0x43, 0x50, 0x32, 0x3a, 0x20, 0x52, 0x4f, 0x4d, 0x20, 0x50, 0x61,
	0x72, 0x69, 0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* MCP2_REG_PRTY_STS */
	0x4d, 0x43, 0x50, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54,
	0x59, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* MCP2_REG_PRTY_MASK */
	0x4d, 0x43, 0x50, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54,
	0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* MCP2: Parity status is not 0 */
	0x4d, 0x43, 0x50, 0x32, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79,
	0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* MCP2_REG_PRTY_STS_H_0 */
	0x4d, 0x43, 0x50, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54,
	0x59, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* MCP2_REG_PRTY_MASK_H_0 */
	0x4d, 0x43, 0x50, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54,
	0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* DORQ: Interrupt Status is not 0 */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x72,
	0x75, 0x70, 0x74, 0x20, 0x53, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DORQ: There is an interrupt on register 0 */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x69, 0x73, 0x20, 0x61, 0x6e, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x72,
	0x75, 0x70, 0x74, 0x20, 0x6f, 0x6e, 0x20, 0x72, 0x65, 0x67, 0x69, 0x73,
	0x74, 0x65, 0x72, 0x20, 0x30, 0x00,
	/* DORQ_REG_INT_STS */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54,
	0x5f, 0x53, 0x54, 0x53, 0x00,
	/* DORQ_REG_INT_MASK */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* DORQ: Parity error. parity status register is set */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79,
	0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x2e, 0x20, 0x70, 0x61, 0x72, 0x69,
	0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x72, 0x65,
	0x67, 0x69, 0x73, 0x74, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20, 0x73, 0x65,
	0x74, 0x00,
	/* DORQ_REG_PRTY_STS */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54,
	0x59, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* DORQ_REG_PRTY_MASK */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54,
	0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* DORQ: Parity status is not 0 */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79,
	0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DORQ_REG_PRTY_STS_H_0 */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54,
	0x59, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* DORQ_REG_PRTY_MASK_H_0 */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54,
	0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* DORQ: Credit to XCM is different than default value */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x43, 0x72, 0x65, 0x64, 0x69, 0x74,
	0x20, 0x74, 0x6f, 0x20, 0x58, 0x43, 0x4d, 0x20, 0x69, 0x73, 0x20, 0x64,
	0x69, 0x66, 0x66, 0x65, 0x72, 0x65, 0x6e, 0x74, 0x20, 0x74, 0x68, 0x61,
	0x6e, 0x20, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x20, 0x76, 0x61,
	0x6c, 0x75, 0x65, 0x00,
	/* DORQ: There are pending messages in DORQ->XCM interface */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20,
	0x6d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x73, 0x20, 0x69, 0x6e, 0x20,
	0x44, 0x4f, 0x52, 0x51, 0x2d, 0x3e, 0x58, 0x43, 0x4d, 0x20, 0x69, 0x6e,
	0x74, 0x65, 0x72, 0x66, 0x61, 0x63, 0x65, 0x00,
	/* DORQ_REG_XCM_MSG_INIT_CRD */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x58, 0x43, 0x4d,
	0x5f, 0x4d, 0x53, 0x47, 0x5f, 0x49, 0x4e, 0x49, 0x54, 0x5f, 0x43, 0x52,
	0x44, 0x00,
	/* DORQ: Credit to TCM is different than default value */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x43, 0x72, 0x65, 0x64, 0x69, 0x74,
	0x20, 0x74, 0x6f, 0x20, 0x54, 0x43, 0x4d, 0x20, 0x69, 0x73, 0x20, 0x64,
	0x69, 0x66, 0x66, 0x65, 0x72, 0x65, 0x6e, 0x74, 0x20, 0x74, 0x68, 0x61,
	0x6e, 0x20, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x20, 0x76, 0x61,
	0x6c, 0x75, 0x65, 0x00,
	/* DORQ: There are pending messages in DORQ->TCM interface */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20,
	0x6d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x73, 0x20, 0x69, 0x6e, 0x20,
	0x44, 0x4f, 0x52, 0x51, 0x2d, 0x3e, 0x54, 0x43, 0x4d, 0x20, 0x69, 0x6e,
	0x74, 0x65, 0x72, 0x66, 0x61, 0x63, 0x65, 0x00,
	/* DORQ_REG_TCM_MSG_INIT_CRD */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54, 0x43, 0x4d,
	0x5f, 0x4d, 0x53, 0x47, 0x5f, 0x49, 0x4e, 0x49, 0x54, 0x5f, 0x43, 0x52,
	0x44, 0x00,
	/* DORQ: Credit to UCM is different than default value */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x43, 0x72, 0x65, 0x64, 0x69, 0x74,
	0x20, 0x74, 0x6f, 0x20, 0x55, 0x43, 0x4d, 0x20, 0x69, 0x73, 0x20, 0x64,
	0x69, 0x66, 0x66, 0x65, 0x72, 0x65, 0x6e, 0x74, 0x20, 0x74, 0x68, 0x61,
	0x6e, 0x20, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x20, 0x76, 0x61,
	0x6c, 0x75, 0x65, 0x00,
	/* DORQ: There are pending messages in DORQ->UCM interface */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20,
	0x6d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x73, 0x20, 0x69, 0x6e, 0x20,
	0x44, 0x4f, 0x52, 0x51, 0x2d, 0x3e, 0x55, 0x43, 0x4d, 0x20, 0x69, 0x6e,
	0x74, 0x65, 0x72, 0x66, 0x61, 0x63, 0x65, 0x00,
	/* DORQ_REG_UCM_MSG_INIT_CRD */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x55, 0x43, 0x4d,
	0x5f, 0x4d, 0x53, 0x47, 0x5f, 0x49, 0x4e, 0x49, 0x54, 0x5f, 0x43, 0x52,
	0x44, 0x00,
	/* DORQ: Credit to PBF is different than default value */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x43, 0x72, 0x65, 0x64, 0x69, 0x74,
	0x20, 0x74, 0x6f, 0x20, 0x50, 0x42, 0x46, 0x20, 0x69, 0x73, 0x20, 0x64,
	0x69, 0x66, 0x66, 0x65, 0x72, 0x65, 0x6e, 0x74, 0x20, 0x74, 0x68, 0x61,
	0x6e, 0x20, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x20, 0x76, 0x61,
	0x6c, 0x75, 0x65, 0x00,
	/* DORQ: There are pending messages in DORQ->PBF interface */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20,
	0x6d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x73, 0x20, 0x69, 0x6e, 0x20,
	0x44, 0x4f, 0x52, 0x51, 0x2d, 0x3e, 0x50, 0x42, 0x46, 0x20, 0x69, 0x6e,
	0x74, 0x65, 0x72, 0x66, 0x61, 0x63, 0x65, 0x00,
	/* DORQ_REG_PBF_CMD_INIT_CRD */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x42, 0x46,
	0x5f, 0x43, 0x4d, 0x44, 0x5f, 0x49, 0x4e, 0x49, 0x54, 0x5f, 0x43, 0x52,
	0x44, 0x00,
	/* DORQ: PF Usage counter is not 0 */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x50, 0x46, 0x20, 0x55, 0x73, 0x61,
	0x67, 0x65, 0x20, 0x63, 0x6f, 0x75, 0x6e, 0x74, 0x65, 0x72, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DORQ: There are currently doorbells pending on the PF */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x61, 0x72, 0x65, 0x20, 0x63, 0x75, 0x72, 0x72, 0x65, 0x6e, 0x74, 0x6c,
	0x79, 0x20, 0x64, 0x6f, 0x6f, 0x72, 0x62, 0x65, 0x6c, 0x6c, 0x73, 0x20,
	0x70, 0x65, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x6f, 0x6e, 0x20, 0x74,
	0x68, 0x65, 0x20, 0x50, 0x46, 0x00,
	/* DORQ_REG_PF_USAGE_CNT */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x46, 0x5f,
	0x55, 0x53, 0x41, 0x47, 0x45, 0x5f, 0x43, 0x4e, 0x54, 0x00,
	/* DORQ: VF Usage counter is not 0 */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x56, 0x46, 0x20, 0x55, 0x73, 0x61,
	0x67, 0x65, 0x20, 0x63, 0x6f, 0x75, 0x6e, 0x74, 0x65, 0x72, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DORQ: There are currently doorbells pending on the VF */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x61, 0x72, 0x65, 0x20, 0x63, 0x75, 0x72, 0x72, 0x65, 0x6e, 0x74, 0x6c,
	0x79, 0x20, 0x64, 0x6f, 0x6f, 0x72, 0x62, 0x65, 0x6c, 0x6c, 0x73, 0x20,
	0x70, 0x65, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x6f, 0x6e, 0x20, 0x74,
	0x68, 0x65, 0x20, 0x56, 0x46, 0x00,
	/* DORQ_REG_VF_USAGE_CNT */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56, 0x46, 0x5f,
	0x55, 0x53, 0x41, 0x47, 0x45, 0x5f, 0x43, 0x4e, 0x54, 0x00,
	/* DORQ: CFC load request FIFO current fill level is not 0 */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x43, 0x46, 0x43, 0x20, 0x6c, 0x6f,
	0x61, 0x64, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x20, 0x46,
	0x49, 0x46, 0x4f, 0x20, 0x63, 0x75, 0x72, 0x72, 0x65, 0x6e, 0x74, 0x20,
	0x66, 0x69, 0x6c, 0x6c, 0x20, 0x6c, 0x65, 0x76, 0x65, 0x6c, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DORQ: There are pending messages in DORQ->CCFC interface */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x61, 0x72, 0x65, 0x20, 0x70, 0x65, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20,
	0x6d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x73, 0x20, 0x69, 0x6e, 0x20,
	0x44, 0x4f, 0x52, 0x51, 0x2d, 0x3e, 0x43, 0x43, 0x46, 0x43, 0x20, 0x69,
	0x6e, 0x74, 0x65, 0x72, 0x66, 0x61, 0x63, 0x65, 0x00,
	/* DORQ_REG_CFC_LD_REQ_FIFO_FILL_LVL */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x43, 0x46, 0x43,
	0x5f, 0x4c, 0x44, 0x5f, 0x52, 0x45, 0x51, 0x5f, 0x46, 0x49, 0x46, 0x4f,
	0x5f, 0x46, 0x49, 0x4c, 0x4c, 0x5f, 0x4c, 0x56, 0x4c, 0x00,
	/* DORQ: DORQ FIFO is not empty */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x44, 0x4f, 0x52, 0x51, 0x20, 0x46,
	0x49, 0x46, 0x4f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65,
	0x6d, 0x70, 0x74, 0x79, 0x00,
	/* DORQ: There are currently doorbells pending */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x61, 0x72, 0x65, 0x20, 0x63, 0x75, 0x72, 0x72, 0x65, 0x6e, 0x74, 0x6c,
	0x79, 0x20, 0x64, 0x6f, 0x6f, 0x72, 0x62, 0x65, 0x6c, 0x6c, 0x73, 0x20,
	0x70, 0x65, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x00,
	/* DORQ_REG_DORQ_FIFO_FILL_LVL */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44, 0x4f, 0x52,
	0x51, 0x5f, 0x46, 0x49, 0x46, 0x4f, 0x5f, 0x46, 0x49, 0x4c, 0x4c, 0x5f,
	0x4c, 0x56, 0x4c, 0x00,
	/* DORQ: Doorbell drops counter is not 0 */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x44, 0x6f, 0x6f, 0x72, 0x62, 0x65,
	0x6c, 0x6c, 0x20, 0x64, 0x72, 0x6f, 0x70, 0x73, 0x20, 0x63, 0x6f, 0x75,
	0x6e, 0x74, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20,
	0x30, 0x00,
	/* DORQ: there were doorbell drops */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x74, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x77, 0x65, 0x72, 0x65, 0x20, 0x64, 0x6f, 0x6f, 0x72, 0x62, 0x65, 0x6c,
	0x6c, 0x20, 0x64, 0x72, 0x6f, 0x70, 0x73, 0x00,
	/* DORQ_REG_DB_DROP_CNT */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44, 0x42, 0x5f,
	0x44, 0x52, 0x4f, 0x50, 0x5f, 0x43, 0x4e, 0x54, 0x00,
	/* DORQ_REG_DB_DROP_DETAILS */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44, 0x42, 0x5f,
	0x44, 0x52, 0x4f, 0x50, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x53,
	0x00,
	/* DORQ_REG_DB_DROP_DETAILS_REASON */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44, 0x42, 0x5f,
	0x44, 0x52, 0x4f, 0x50, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x53,
	0x5f, 0x52, 0x45, 0x41, 0x53, 0x4f, 0x4e, 0x00,
	/* DORQ: Doorbell aborts counter is not 0 */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x44, 0x6f, 0x6f, 0x72, 0x62, 0x65,
	0x6c, 0x6c, 0x20, 0x61, 0x62, 0x6f, 0x72, 0x74, 0x73, 0x20, 0x63, 0x6f,
	0x75, 0x6e, 0x74, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74,
	0x20, 0x30, 0x00,
	/* DORQ: there were doorbell aborts */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x74, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x77, 0x65, 0x72, 0x65, 0x20, 0x64, 0x6f, 0x6f, 0x72, 0x62, 0x65, 0x6c,
	0x6c, 0x20, 0x61, 0x62, 0x6f, 0x72, 0x74, 0x73, 0x00,
	/* DORQ_REG_DPM_ABORT_CNT */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44, 0x50, 0x4d,
	0x5f, 0x41, 0x42, 0x4f, 0x52, 0x54, 0x5f, 0x43, 0x4e, 0x54, 0x00,
	/* DORQ_REG_DPM_ABORT_REASON */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44, 0x50, 0x4d,
	0x5f, 0x41, 0x42, 0x4f, 0x52, 0x54, 0x5f, 0x52, 0x45, 0x41, 0x53, 0x4f,
	0x4e, 0x00,
	/* DORQ_REG_DPM_ABORT_DETAILS_REASON */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44, 0x50, 0x4d,
	0x5f, 0x41, 0x42, 0x4f, 0x52, 0x54, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49,
	0x4c, 0x53, 0x5f, 0x52, 0x45, 0x41, 0x53, 0x4f, 0x4e, 0x00,
	/* DORQ: Timer expiration of external doorbell counter is not 0 */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x72, 0x20,
	0x65, 0x78, 0x70, 0x69, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x6f,
	0x66, 0x20, 0x65, 0x78, 0x74, 0x65, 0x72, 0x6e, 0x61, 0x6c, 0x20, 0x64,
	0x6f, 0x6f, 0x72, 0x62, 0x65, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x75, 0x6e,
	0x74, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30,
	0x00,
	/* DORQ: There were aborts due to Timer expiration */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x77, 0x65, 0x72, 0x65, 0x20, 0x61, 0x62, 0x6f, 0x72, 0x74, 0x73, 0x20,
	0x64, 0x75, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x72,
	0x20, 0x65, 0x78, 0x70, 0x69, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x00,
	/* DORQ_REG_DPM_TIMER_EXPIR_ABORT_CNT */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44, 0x50, 0x4d,
	0x5f, 0x54, 0x49, 0x4d, 0x45, 0x52, 0x5f, 0x45, 0x58, 0x50, 0x49, 0x52,
	0x5f, 0x41, 0x42, 0x4f, 0x52, 0x54, 0x5f, 0x43, 0x4e, 0x54, 0x00,
	/* DORQ: DPM Table  current fill level is not 0 */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x44, 0x50, 0x4d, 0x20, 0x54, 0x61,
	0x62, 0x6c, 0x65, 0x20, 0x20, 0x63, 0x75, 0x72, 0x72, 0x65, 0x6e, 0x74,
	0x20, 0x66, 0x69, 0x6c, 0x6c, 0x20, 0x6c, 0x65, 0x76, 0x65, 0x6c, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* DORQ: There are DPMs currently in process */
	0x44, 0x4f, 0x52, 0x51, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x61, 0x72, 0x65, 0x20, 0x44, 0x50, 0x4d, 0x73, 0x20, 0x63, 0x75, 0x72,
	0x72, 0x65, 0x6e, 0x74, 0x6c, 0x79, 0x20, 0x69, 0x6e, 0x20, 0x70, 0x72,
	0x6f, 0x63, 0x65, 0x73, 0x73, 0x00,
	/* DORQ_REG_DPM_TBL_FILL_LVL */
	0x44, 0x4f, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44, 0x50, 0x4d,
	0x5f, 0x54, 0x42, 0x4c, 0x5f, 0x46, 0x49, 0x4c, 0x4c, 0x5f, 0x4c, 0x56,
	0x4c, 0x00,
	/* IGU: INT status is not 0 */
	0x49, 0x47, 0x55, 0x3a, 0x20, 0x49, 0x4e, 0x54, 0x20, 0x73, 0x74, 0x61,
	0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30,
	0x00,
	/* IGU_REG_INT_STS */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x53, 0x54, 0x53, 0x00,
	/* IGU_REG_INT_MASK */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* IGU: CAM Parity status is not 0 */
	0x49, 0x47, 0x55, 0x3a, 0x20, 0x43, 0x41, 0x4d, 0x20, 0x50, 0x61, 0x72,
	0x69, 0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* IGU_REG_PRTY_STS */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x53, 0x54, 0x53, 0x00,
	/* IGU_REG_PRTY_MASK */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* IGU: Parity status 0 is not 0 */
	0x49, 0x47, 0x55, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79, 0x20,
	0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x30, 0x20, 0x69, 0x73, 0x20,
	0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* IGU_REG_PRTY_STS_H_0 */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* IGU_REG_PRTY_MASK_H_0 */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* IGU attention message write done pending is not empty */
	0x49, 0x47, 0x55, 0x20, 0x61, 0x74, 0x74, 0x65, 0x6e, 0x74, 0x69, 0x6f,
	0x6e, 0x20, 0x6d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x20, 0x77, 0x72,
	0x69, 0x74, 0x65, 0x20, 0x64, 0x6f, 0x6e, 0x65, 0x20, 0x70, 0x65, 0x6e,
	0x64, 0x69, 0x6e, 0x67, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20,
	0x65, 0x6d, 0x70, 0x74, 0x79, 0x00,
	/* IGU: attention write done pending */
	0x49, 0x47, 0x55, 0x3a, 0x20, 0x61, 0x74, 0x74, 0x65, 0x6e, 0x74, 0x69,
	0x6f, 0x6e, 0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x20, 0x64, 0x6f, 0x6e,
	0x65, 0x20, 0x70, 0x65, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x00,
	/* IGU_REG_ATTN_WRITE_DONE_PENDING */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54, 0x4e,
	0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x5f, 0x44, 0x4f, 0x4e, 0x45, 0x5f,
	0x50, 0x45, 0x4e, 0x44, 0x49, 0x4e, 0x47, 0x00,
	/* IGU: block is not in idle. Some interrupts are asserted */
	0x49, 0x47, 0x55, 0x3a, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x2e, 0x20, 0x53, 0x6f, 0x6d, 0x65, 0x20, 0x69, 0x6e, 0x74, 0x65,
	0x72, 0x72, 0x75, 0x70, 0x74, 0x73, 0x20, 0x61, 0x72, 0x65, 0x20, 0x61,
	0x73, 0x73, 0x65, 0x72, 0x74, 0x65, 0x64, 0x00,
	/* IGU_REG_INTERRUPT_STATUS */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x45,
	0x52, 0x52, 0x55, 0x50, 0x54, 0x5f, 0x53, 0x54, 0x41, 0x54, 0x55, 0x53,
	0x00,
	/* IGU: some unauthorized commands arrived to the IGU. Use igu_dump_fifo utility for more details. */
	0x49, 0x47, 0x55, 0x3a, 0x20, 0x73, 0x6f, 0x6d, 0x65, 0x20, 0x75, 0x6e,
	0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x65, 0x64, 0x20, 0x63,
	0x6f, 0x6d, 0x6d, 0x61, 0x6e, 0x64, 0x73, 0x20, 0x61, 0x72, 0x72, 0x69,
	0x76, 0x65, 0x64, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x49,
	0x47, 0x55, 0x2e, 0x20, 0x55, 0x73, 0x65, 0x20, 0x69, 0x67, 0x75, 0x5f,
	0x64, 0x75, 0x6d, 0x70, 0x5f, 0x66, 0x69, 0x66, 0x6f, 0x20, 0x75, 0x74,
	0x69, 0x6c, 0x69, 0x74, 0x79, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x6d, 0x6f,
	0x72, 0x65, 0x20, 0x64, 0x65, 0x74, 0x61, 0x69, 0x6c, 0x73, 0x2e, 0x00,
	/* IGU_REG_ERROR_HANDLING_DATA_VALID */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x45, 0x52, 0x52, 0x4f,
	0x52, 0x5f, 0x48, 0x41, 0x4e, 0x44, 0x4c, 0x49, 0x4e, 0x47, 0x5f, 0x44,
	0x41, 0x54, 0x41, 0x5f, 0x56, 0x41, 0x4c, 0x49, 0x44, 0x00,
	/* IGU: Some messages were not executed in the IGU. */
	0x49, 0x47, 0x55, 0x3a, 0x20, 0x53, 0x6f, 0x6d, 0x65, 0x20, 0x6d, 0x65,
	0x73, 0x73, 0x61, 0x67, 0x65, 0x73, 0x20, 0x77, 0x65, 0x72, 0x65, 0x20,
	0x6e, 0x6f, 0x74, 0x20, 0x65, 0x78, 0x65, 0x63, 0x75, 0x74, 0x65, 0x64,
	0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x49, 0x47, 0x55, 0x2e,
	0x00,
	/* IGU: Commands were dropped */
	0x49, 0x47, 0x55, 0x3a, 0x20, 0x43, 0x6f, 0x6d, 0x6d, 0x61, 0x6e, 0x64,
	0x73, 0x20, 0x77, 0x65, 0x72, 0x65, 0x20, 0x64, 0x72, 0x6f, 0x70, 0x70,
	0x65, 0x64, 0x00,
	/* IGU_REG_SILENT_DROP */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53, 0x49, 0x4c, 0x45,
	0x4e, 0x54, 0x5f, 0x44, 0x52, 0x4f, 0x50, 0x00,
	/* IGU: block is not in idle. SB_CTRL_FSM should be zero in idle state */
	0x49, 0x47, 0x55, 0x3a, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x2e, 0x20, 0x53, 0x42, 0x5f, 0x43, 0x54, 0x52, 0x4c, 0x5f, 0x46,
	0x53, 0x4d, 0x20, 0x73, 0x68, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x62, 0x65,
	0x20, 0x7a, 0x65, 0x72, 0x6f, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x20, 0x73, 0x74, 0x61, 0x74, 0x65, 0x00,
	/* IGU_REG_SB_CTRL_FSM */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53, 0x42, 0x5f, 0x43,
	0x54, 0x52, 0x4c, 0x5f, 0x46, 0x53, 0x4d, 0x00,
	/* IGU: block is not in idle. INT_HANDLE_FSM should be zero in idle state */
	0x49, 0x47, 0x55, 0x3a, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x2e, 0x20, 0x49, 0x4e, 0x54, 0x5f, 0x48, 0x41, 0x4e, 0x44, 0x4c,
	0x45, 0x5f, 0x46, 0x53, 0x4d, 0x20, 0x73, 0x68, 0x6f, 0x75, 0x6c, 0x64,
	0x20, 0x62, 0x65, 0x20, 0x7a, 0x65, 0x72, 0x6f, 0x20, 0x69, 0x6e, 0x20,
	0x69, 0x64, 0x6c, 0x65, 0x20, 0x73, 0x74, 0x61, 0x74, 0x65, 0x00,
	/* IGU_REG_INT_HANDLE_FSM */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x48, 0x41, 0x4e, 0x44, 0x4c, 0x45, 0x5f, 0x46, 0x53, 0x4d, 0x00,
	/* IGU: block is not in idle. SB_ATTN_FSM should be zero or two in idle state */
	0x49, 0x47, 0x55, 0x3a, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x2e, 0x20, 0x53, 0x42, 0x5f, 0x41, 0x54, 0x54, 0x4e, 0x5f, 0x46,
	0x53, 0x4d, 0x20, 0x73, 0x68, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x62, 0x65,
	0x20, 0x7a, 0x65, 0x72, 0x6f, 0x20, 0x6f, 0x72, 0x20, 0x74, 0x77, 0x6f,
	0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c, 0x65, 0x20, 0x73, 0x74, 0x61,
	0x74, 0x65, 0x00,
	/* IGU_REG_ATTN_FSM */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x54, 0x54, 0x4e,
	0x5f, 0x46, 0x53, 0x4d, 0x00,
	/* IGU: block is not in idle. CTRL_FSM should be zero in idle state */
	0x49, 0x47, 0x55, 0x3a, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x2e, 0x20, 0x43, 0x54, 0x52, 0x4c, 0x5f, 0x46, 0x53, 0x4d, 0x20,
	0x73, 0x68, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x62, 0x65, 0x20, 0x7a, 0x65,
	0x72, 0x6f, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c, 0x65, 0x20, 0x73,
	0x74, 0x61, 0x74, 0x65, 0x00,
	/* IGU_REG_CTRL_FSM */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x43, 0x54, 0x52, 0x4c,
	0x5f, 0x46, 0x53, 0x4d, 0x00,
	/* IGU: block is not in idle. PXP_ARB_FSM should be zero in idle state */
	0x49, 0x47, 0x55, 0x3a, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x2e, 0x20, 0x50, 0x58, 0x50, 0x5f, 0x41, 0x52, 0x42, 0x5f, 0x46,
	0x53, 0x4d, 0x20, 0x73, 0x68, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x62, 0x65,
	0x20, 0x7a, 0x65, 0x72, 0x6f, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x20, 0x73, 0x74, 0x61, 0x74, 0x65, 0x00,
	/* IGU_REG_PXP_ARB_FSM */
	0x49, 0x47, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x58, 0x50, 0x5f,
	0x41, 0x52, 0x42, 0x5f, 0x46, 0x53, 0x4d, 0x00,
	/* CAU: R/W to CAU FIFO failed */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x52, 0x2f, 0x57, 0x20, 0x74, 0x6f, 0x20,
	0x43, 0x41, 0x55, 0x20, 0x46, 0x49, 0x46, 0x4f, 0x20, 0x66, 0x61, 0x69,
	0x6c, 0x65, 0x64, 0x00,
	/* CAU: FIFO commands failed */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x46, 0x49, 0x46, 0x4f, 0x20, 0x63, 0x6f,
	0x6d, 0x6d, 0x61, 0x6e, 0x64, 0x73, 0x20, 0x66, 0x61, 0x69, 0x6c, 0x65,
	0x64, 0x00,
	/* CAU_REG_INT_STS */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x53, 0x54, 0x53, 0x00,
	/* CAU_REG_INT_MASK */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* CAU: Parity status is not 0 */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79, 0x20,
	0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* CAU_REG_PRTY_STS_H_0 */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* CAU_REG_PRTY_MASK_H_0 */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: Pending IGU Request */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x50, 0x65, 0x6e, 0x64, 0x69, 0x6e, 0x67,
	0x20, 0x49, 0x47, 0x55, 0x20, 0x52, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74,
	0x00,
	/* CAU_REG_IGU_REQ_CREDIT_STATUS */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x47, 0x55, 0x5f,
	0x52, 0x45, 0x51, 0x5f, 0x43, 0x52, 0x45, 0x44, 0x49, 0x54, 0x5f, 0x53,
	0x54, 0x41, 0x54, 0x55, 0x53, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: Pending IGU Command */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x50, 0x65, 0x6e, 0x64, 0x69, 0x6e, 0x67,
	0x20, 0x49, 0x47, 0x55, 0x20, 0x43, 0x6f, 0x6d, 0x6d, 0x61, 0x6e, 0x64,
	0x00,
	/* CAU_REG_IGU_CMD_CREDIT_STATUS */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x47, 0x55, 0x5f,
	0x43, 0x4d, 0x44, 0x5f, 0x43, 0x52, 0x45, 0x44, 0x49, 0x54, 0x5f, 0x53,
	0x54, 0x41, 0x54, 0x55, 0x53, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: FIFOs are not empty */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x46, 0x49, 0x46, 0x4f, 0x73, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79,
	0x00,
	/* CAU_REG_DEBUG_FIFO_STATUS */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44, 0x45, 0x42, 0x55,
	0x47, 0x5f, 0x46, 0x49, 0x46, 0x4f, 0x5f, 0x53, 0x54, 0x41, 0x54, 0x55,
	0x53, 0x00,
	/* CAU: Fatal unauthorized command arrived from PXP */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x46, 0x61, 0x74, 0x61, 0x6c, 0x20, 0x75,
	0x6e, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x65, 0x64, 0x20,
	0x63, 0x6f, 0x6d, 0x6d, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x72, 0x72, 0x69,
	0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x50, 0x58, 0x50,
	0x00,
	/* Invalid command arrived to CAU from PXP switch */
	0x49, 0x6e, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20, 0x63, 0x6f, 0x6d, 0x6d,
	0x61, 0x6e, 0x64, 0x20, 0x61, 0x72, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20,
	0x74, 0x6f, 0x20, 0x43, 0x41, 0x55, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20,
	0x50, 0x58, 0x50, 0x20, 0x73, 0x77, 0x69, 0x74, 0x63, 0x68, 0x00,
	/* CAU_REG_ERROR_PXP_REQ */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x45, 0x52, 0x52, 0x4f,
	0x52, 0x5f, 0x50, 0x58, 0x50, 0x5f, 0x52, 0x45, 0x51, 0x00,
	/* CAU: Fatal. CAU arrived to invalid FSM line. */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x46, 0x61, 0x74, 0x61, 0x6c, 0x2e, 0x20,
	0x43, 0x41, 0x55, 0x20, 0x61, 0x72, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20,
	0x74, 0x6f, 0x20, 0x69, 0x6e, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20, 0x46,
	0x53, 0x4d, 0x20, 0x6c, 0x69, 0x6e, 0x65, 0x2e, 0x00,
	/* CAU_REG_ERROR_FSM_LINE */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x45, 0x52, 0x52, 0x4f,
	0x52, 0x5f, 0x46, 0x53, 0x4d, 0x5f, 0x4c, 0x49, 0x4e, 0x45, 0x00,
	/* CAU_REG_ERROR_FSM_LINE_PRE */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x45, 0x52, 0x52, 0x4f,
	0x52, 0x5f, 0x46, 0x53, 0x4d, 0x5f, 0x4c, 0x49, 0x4e, 0x45, 0x5f, 0x50,
	0x52, 0x45, 0x00,
	/* CAU: Fatal parity error occurred in CAU */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x46, 0x61, 0x74, 0x61, 0x6c, 0x20, 0x70,
	0x61, 0x72, 0x69, 0x74, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20,
	0x6f, 0x63, 0x63, 0x75, 0x72, 0x72, 0x65, 0x64, 0x20, 0x69, 0x6e, 0x20,
	0x43, 0x41, 0x55, 0x00,
	/* CAU_REG_PARITY_LATCH_STATUS */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x41, 0x52, 0x49,
	0x54, 0x59, 0x5f, 0x4c, 0x41, 0x54, 0x43, 0x48, 0x5f, 0x53, 0x54, 0x41,
	0x54, 0x55, 0x53, 0x00,
	/* CAU: Fatal unauthorized cleanup command arrived from GRC */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x46, 0x61, 0x74, 0x61, 0x6c, 0x20, 0x75,
	0x6e, 0x61, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x7a, 0x65, 0x64, 0x20,
	0x63, 0x6c, 0x65, 0x61, 0x6e, 0x75, 0x70, 0x20, 0x63, 0x6f, 0x6d, 0x6d,
	0x61, 0x6e, 0x64, 0x20, 0x61, 0x72, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20,
	0x66, 0x72, 0x6f, 0x6d, 0x20, 0x47, 0x52, 0x43, 0x00,
	/* CAU_REG_ERROR_CLEANUP_CMD_REG */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x45, 0x52, 0x52, 0x4f,
	0x52, 0x5f, 0x43, 0x4c, 0x45, 0x41, 0x4e, 0x55, 0x50, 0x5f, 0x43, 0x4d,
	0x44, 0x5f, 0x52, 0x45, 0x47, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: Agg Unit is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x41, 0x67, 0x67, 0x20, 0x55, 0x6e, 0x69,
	0x74, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20,
	0x69, 0x64, 0x6c, 0x65, 0x00,
	/* CAU_REG_AGG_UNITS_0TO15_STATE */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x47, 0x47, 0x5f,
	0x55, 0x4e, 0x49, 0x54, 0x53, 0x5f, 0x30, 0x54, 0x4f, 0x31, 0x35, 0x5f,
	0x53, 0x54, 0x41, 0x54, 0x45, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: Agg Unit is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x41, 0x67, 0x67, 0x20, 0x55, 0x6e, 0x69,
	0x74, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20,
	0x69, 0x64, 0x6c, 0x65, 0x00,
	/* CAU_REG_AGG_UNITS_16TO31_STATE */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x47, 0x47, 0x5f,
	0x55, 0x4e, 0x49, 0x54, 0x53, 0x5f, 0x31, 0x36, 0x54, 0x4f, 0x33, 0x31,
	0x5f, 0x53, 0x54, 0x41, 0x54, 0x45, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: Agg Unit is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x41, 0x67, 0x67, 0x20, 0x55, 0x6e, 0x69,
	0x74, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20,
	0x69, 0x64, 0x6c, 0x65, 0x00,
	/* CAU_REG_AGG_UNITS_32TO47_STATE */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x47, 0x47, 0x5f,
	0x55, 0x4e, 0x49, 0x54, 0x53, 0x5f, 0x33, 0x32, 0x54, 0x4f, 0x34, 0x37,
	0x5f, 0x53, 0x54, 0x41, 0x54, 0x45, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: Agg Unit is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x41, 0x67, 0x67, 0x20, 0x55, 0x6e, 0x69,
	0x74, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20,
	0x69, 0x64, 0x6c, 0x65, 0x00,
	/* CAU_REG_AGG_UNITS_48TO63_STATE */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x47, 0x47, 0x5f,
	0x55, 0x4e, 0x49, 0x54, 0x53, 0x5f, 0x34, 0x38, 0x54, 0x4f, 0x36, 0x33,
	0x5f, 0x53, 0x54, 0x41, 0x54, 0x45, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: Not all WDones were received from IGU */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x4e, 0x6f, 0x74, 0x20, 0x61, 0x6c, 0x6c,
	0x20, 0x57, 0x44, 0x6f, 0x6e, 0x65, 0x73, 0x20, 0x77, 0x65, 0x72, 0x65,
	0x20, 0x72, 0x65, 0x63, 0x65, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x72,
	0x6f, 0x6d, 0x20, 0x49, 0x47, 0x55, 0x00,
	/* CAU_REG_REQ_COUNTER */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x52, 0x45, 0x51, 0x5f,
	0x43, 0x4f, 0x55, 0x4e, 0x54, 0x45, 0x52, 0x00,
	/* CAU_REG_WDONE_COUNTER */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x57, 0x44, 0x4f, 0x4e,
	0x45, 0x5f, 0x43, 0x4f, 0x55, 0x4e, 0x54, 0x45, 0x52, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: Not all ACKs were received from IGU */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x4e, 0x6f, 0x74, 0x20, 0x61, 0x6c, 0x6c,
	0x20, 0x41, 0x43, 0x4b, 0x73, 0x20, 0x77, 0x65, 0x72, 0x65, 0x20, 0x72,
	0x65, 0x63, 0x65, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x72, 0x6f, 0x6d,
	0x20, 0x49, 0x47, 0x55, 0x00,
	/* CAU_REG_REQ_COUNTER */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x52, 0x45, 0x51, 0x5f,
	0x43, 0x4f, 0x55, 0x4e, 0x54, 0x45, 0x52, 0x00,
	/* CAU_REG_ACK_COUNTER */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41, 0x43, 0x4b, 0x5f,
	0x43, 0x4f, 0x55, 0x4e, 0x54, 0x45, 0x52, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: Main FSM is not idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x4d, 0x61, 0x69, 0x6e, 0x20, 0x46, 0x53,
	0x4d, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU_REG_MAIN_FSM_STATUS */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d, 0x41, 0x49, 0x4e,
	0x5f, 0x46, 0x53, 0x4d, 0x5f, 0x53, 0x54, 0x41, 0x54, 0x55, 0x53, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: Var Read FSM is not idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x56, 0x61, 0x72, 0x20, 0x52, 0x65, 0x61,
	0x64, 0x20, 0x46, 0x53, 0x4d, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74,
	0x20, 0x69, 0x64, 0x6c, 0x65, 0x00,
	/* CAU_REG_VAR_READ_FSM_STATUS */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56, 0x41, 0x52, 0x5f,
	0x52, 0x45, 0x41, 0x44, 0x5f, 0x46, 0x53, 0x4d, 0x5f, 0x53, 0x54, 0x41,
	0x54, 0x55, 0x53, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: IGU DMA FSM is not idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x49, 0x47, 0x55, 0x20, 0x44, 0x4d, 0x41,
	0x20, 0x46, 0x53, 0x4d, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20,
	0x69, 0x64, 0x6c, 0x65, 0x00,
	/* CAU_REG_IGU_DMA_FSM_STATUS */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x47, 0x55, 0x5f,
	0x44, 0x4d, 0x41, 0x5f, 0x46, 0x53, 0x4d, 0x5f, 0x53, 0x54, 0x41, 0x54,
	0x55, 0x53, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: IGU CQE AMD FSM is not idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x49, 0x47, 0x55, 0x20, 0x43, 0x51, 0x45,
	0x20, 0x41, 0x4d, 0x44, 0x20, 0x46, 0x53, 0x4d, 0x20, 0x69, 0x73, 0x20,
	0x6e, 0x6f, 0x74, 0x20, 0x69, 0x64, 0x6c, 0x65, 0x00,
	/* CAU_REG_IGU_CQE_CMD_FSM_STATUS */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x47, 0x55, 0x5f,
	0x43, 0x51, 0x45, 0x5f, 0x43, 0x4d, 0x44, 0x5f, 0x46, 0x53, 0x4d, 0x5f,
	0x53, 0x54, 0x41, 0x54, 0x55, 0x53, 0x00,
	/* CAU: Block is not in idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x69, 0x64, 0x6c,
	0x65, 0x00,
	/* CAU: IGU CQE AGG FSM is not idle */
	0x43, 0x41, 0x55, 0x3a, 0x20, 0x49, 0x47, 0x55, 0x20, 0x43, 0x51, 0x45,
	0x20, 0x41, 0x47, 0x47, 0x20, 0x46, 0x53, 0x4d, 0x20, 0x69, 0x73, 0x20,
	0x6e, 0x6f, 0x74, 0x20, 0x69, 0x64, 0x6c, 0x65, 0x00,
	/* CAU_REG_IGU_CQE_AGG_FSM_STATUS */
	0x43, 0x41, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x47, 0x55, 0x5f,
	0x43, 0x51, 0x45, 0x5f, 0x41, 0x47, 0x47, 0x5f, 0x46, 0x53, 0x4d, 0x5f,
	0x53, 0x54, 0x41, 0x54, 0x55, 0x53, 0x00,
	/* PRS: SAL Cache initialization is not done */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x53, 0x41, 0x4c, 0x20, 0x43, 0x61, 0x63,
	0x68, 0x65, 0x20, 0x69, 0x6e, 0x69, 0x74, 0x69, 0x61, 0x6c, 0x69, 0x7a,
	0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74,
	0x20, 0x64, 0x6f, 0x6e, 0x65, 0x00,
	/* PRS_REG_MAC_VLAN_CACHE_INIT_DONE */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d, 0x41, 0x43, 0x5f,
	0x56, 0x4c, 0x41, 0x4e, 0x5f, 0x43, 0x41, 0x43, 0x48, 0x45, 0x5f, 0x49,
	0x4e, 0x49, 0x54, 0x5f, 0x44, 0x4f, 0x4e, 0x45, 0x00,
	/* PRS: Interrupt Status is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x72, 0x75,
	0x70, 0x74, 0x20, 0x53, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73,
	0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PRS: Fatal error */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x46, 0x61, 0x74, 0x61, 0x6c, 0x20, 0x65,
	0x72, 0x72, 0x6f, 0x72, 0x00,
	/* PRS_REG_INT_STS_0 */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x53, 0x54, 0x53, 0x5f, 0x30, 0x00,
	/* PRS_REG_INT_MASK_0 */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x30, 0x00,
	/* PRS_REG_MINI_CACHE_FAILED_RESPONSE */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d, 0x49, 0x4e, 0x49,
	0x5f, 0x43, 0x41, 0x43, 0x48, 0x45, 0x5f, 0x46, 0x41, 0x49, 0x4c, 0x45,
	0x44, 0x5f, 0x52, 0x45, 0x53, 0x50, 0x4f, 0x4e, 0x53, 0x45, 0x00,
	/* PRS_REG_MINI_CACHE_ENTRY */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d, 0x49, 0x4e, 0x49,
	0x5f, 0x43, 0x41, 0x43, 0x48, 0x45, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59,
	0x00,
	/* GRC_REG_TRACE_FIFO_VALID_DATA */
	0x47, 0x52, 0x43, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54, 0x52, 0x41, 0x43,
	0x45, 0x5f, 0x46, 0x49, 0x46, 0x4f, 0x5f, 0x56, 0x41, 0x4c, 0x49, 0x44,
	0x5f, 0x44, 0x41, 0x54, 0x41, 0x00,
	/* PRS: CAM Parity status is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x43, 0x41, 0x4d, 0x20, 0x50, 0x61, 0x72,
	0x69, 0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PRS_REG_PRTY_STS */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PRS_REG_PRTY_MASK */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PRS: Parity status 0 is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79, 0x20,
	0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x30, 0x20, 0x69, 0x73, 0x20,
	0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PRS_REG_PRTY_STS_H_0 */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* PRS_REG_PRTY_MASK_H_0 */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* PRS: Parity status 1 is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79, 0x20,
	0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x31, 0x20, 0x69, 0x73, 0x20,
	0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PRS_REG_PRTY_STS_H_1 */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x31, 0x00,
	/* PRS_REG_PRTY_MASK_H_1 */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x31, 0x00,
	/* PRS: The number of packets in the parser pipeline is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x20, 0x6e, 0x75, 0x6d,
	0x62, 0x65, 0x72, 0x20, 0x6f, 0x66, 0x20, 0x70, 0x61, 0x63, 0x6b, 0x65,
	0x74, 0x73, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x61,
	0x72, 0x73, 0x65, 0x72, 0x20, 0x70, 0x69, 0x70, 0x65, 0x6c, 0x69, 0x6e,
	0x65, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PRS: There are outstanding packets in the PRS pipeline */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74, 0x61, 0x6e, 0x64, 0x69,
	0x6e, 0x67, 0x20, 0x70, 0x61, 0x63, 0x6b, 0x65, 0x74, 0x73, 0x20, 0x69,
	0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x50, 0x52, 0x53, 0x20, 0x70, 0x69,
	0x70, 0x65, 0x6c, 0x69, 0x6e, 0x65, 0x00,
	/* PRS_REG_PRS_PKT_CT */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x53, 0x5f,
	0x50, 0x4b, 0x54, 0x5f, 0x43, 0x54, 0x00,
	/* PRS: The number of available packets waiting for Parser processing is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x20, 0x6e, 0x75, 0x6d,
	0x62, 0x65, 0x72, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x76, 0x61, 0x69, 0x6c,
	0x61, 0x62, 0x6c, 0x65, 0x20, 0x70, 0x61, 0x63, 0x6b, 0x65, 0x74, 0x73,
	0x20, 0x77, 0x61, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x66, 0x6f, 0x72,
	0x20, 0x50, 0x61, 0x72, 0x73, 0x65, 0x72, 0x20, 0x70, 0x72, 0x6f, 0x63,
	0x65, 0x73, 0x73, 0x69, 0x6e, 0x67, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* There are available packets waiting for PRS processing. 16:8 - Loopback TCs, 7:0 - main TCs */
	0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61, 0x72, 0x65, 0x20, 0x61, 0x76,
	0x61, 0x69, 0x6c, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x70, 0x61, 0x63, 0x6b,
	0x65, 0x74, 0x73, 0x20, 0x77, 0x61, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x20,
	0x66, 0x6f, 0x72, 0x20, 0x50, 0x52, 0x53, 0x20, 0x70, 0x72, 0x6f, 0x63,
	0x65, 0x73, 0x73, 0x69, 0x6e, 0x67, 0x2e, 0x20, 0x31, 0x36, 0x3a, 0x38,
	0x20, 0x2d, 0x20, 0x4c, 0x6f, 0x6f, 0x70, 0x62, 0x61, 0x63, 0x6b, 0x20,
	0x54, 0x43, 0x73, 0x2c, 0x20, 0x37, 0x3a, 0x30, 0x20, 0x2d, 0x20, 0x6d,
	0x61, 0x69, 0x6e, 0x20, 0x54, 0x43, 0x73, 0x00,
	/* PRS_REG_QUEUE_PKT_AVAIL_STATUS */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x51, 0x55, 0x45, 0x55,
	0x45, 0x5f, 0x50, 0x4b, 0x54, 0x5f, 0x41, 0x56, 0x41, 0x49, 0x4c, 0x5f,
	0x53, 0x54, 0x41, 0x54, 0x55, 0x53, 0x00,
	/* PRS: There are traffic-classes that are backpressured by the STORM */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x74, 0x72, 0x61, 0x66, 0x66, 0x69, 0x63, 0x2d, 0x63,
	0x6c, 0x61, 0x73, 0x73, 0x65, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20,
	0x61, 0x72, 0x65, 0x20, 0x62, 0x61, 0x63, 0x6b, 0x70, 0x72, 0x65, 0x73,
	0x73, 0x75, 0x72, 0x65, 0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65,
	0x20, 0x53, 0x54, 0x4f, 0x52, 0x4d, 0x00,
	/* PRS: There are traffic-classes that are blocked by the STORM. Each set bit represents a blocked TC (7-0) from MSDM, TSDM, and USDM, respectively. */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x74, 0x72, 0x61, 0x66, 0x66, 0x69, 0x63, 0x2d, 0x63,
	0x6c, 0x61, 0x73, 0x73, 0x65, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20,
	0x61, 0x72, 0x65, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x65, 0x64, 0x20,
	0x62, 0x79, 0x20, 0x74, 0x68, 0x65, 0x20, 0x53, 0x54, 0x4f, 0x52, 0x4d,
	0x2e, 0x20, 0x45, 0x61, 0x63, 0x68, 0x20, 0x73, 0x65, 0x74, 0x20, 0x62,
	0x69, 0x74, 0x20, 0x72, 0x65, 0x70, 0x72, 0x65, 0x73, 0x65, 0x6e, 0x74,
	0x73, 0x20, 0x61, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x65, 0x64, 0x20,
	0x54, 0x43, 0x20, 0x28, 0x37, 0x2d, 0x30, 0x29, 0x20, 0x66, 0x72, 0x6f,
	0x6d, 0x20, 0x4d, 0x53, 0x44, 0x4d, 0x2c, 0x20, 0x54, 0x53, 0x44, 0x4d,
	0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x55, 0x53, 0x44, 0x4d, 0x2c, 0x20,
	0x72, 0x65, 0x73, 0x70, 0x65, 0x63, 0x74, 0x69, 0x76, 0x65, 0x6c, 0x79,
	0x2e, 0x00,
	/* PRS_REG_STORM_BKPRS_STATUS */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53, 0x54, 0x4f, 0x52,
	0x4d, 0x5f, 0x42, 0x4b, 0x50, 0x52, 0x53, 0x5f, 0x53, 0x54, 0x41, 0x54,
	0x55, 0x53, 0x00,
	/* PRS: BRB has asserted Stop Parsing indication to PRS */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x42, 0x52, 0x42, 0x20, 0x68, 0x61, 0x73,
	0x20, 0x61, 0x73, 0x73, 0x65, 0x72, 0x74, 0x65, 0x64, 0x20, 0x53, 0x74,
	0x6f, 0x70, 0x20, 0x50, 0x61, 0x72, 0x73, 0x69, 0x6e, 0x67, 0x20, 0x69,
	0x6e, 0x64, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x6f,
	0x20, 0x50, 0x52, 0x53, 0x00,
	/* PRS: The number of outstanding packets in the pipeline is above a threshold so PRS stopped processing new packets */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x20, 0x6e, 0x75, 0x6d,
	0x62, 0x65, 0x72, 0x20, 0x6f, 0x66, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74,
	0x61, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x70, 0x61, 0x63, 0x6b, 0x65,
	0x74, 0x73, 0x20, 0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x70, 0x69,
	0x70, 0x65, 0x6c, 0x69, 0x6e, 0x65, 0x20, 0x69, 0x73, 0x20, 0x61, 0x62,
	0x6f, 0x76, 0x65, 0x20, 0x61, 0x20, 0x74, 0x68, 0x72, 0x65, 0x73, 0x68,
	0x6f, 0x6c, 0x64, 0x20, 0x73, 0x6f, 0x20, 0x50, 0x52, 0x53, 0x20, 0x73,
	0x74, 0x6f, 0x70, 0x70, 0x65, 0x64, 0x20, 0x70, 0x72, 0x6f, 0x63, 0x65,
	0x73, 0x73, 0x69, 0x6e, 0x67, 0x20, 0x6e, 0x65, 0x77, 0x20, 0x70, 0x61,
	0x63, 0x6b, 0x65, 0x74, 0x73, 0x00,
	/* PRS_REG_STOP_PARSING_STATUS */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53, 0x54, 0x4f, 0x50,
	0x5f, 0x50, 0x41, 0x52, 0x53, 0x49, 0x4e, 0x47, 0x5f, 0x53, 0x54, 0x41,
	0x54, 0x55, 0x53, 0x00,
	/* PRS: TCM Credit Count is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x43, 0x4d, 0x20, 0x43, 0x72, 0x65,
	0x64, 0x69, 0x74, 0x20, 0x43, 0x6f, 0x75, 0x6e, 0x74, 0x20, 0x69, 0x73,
	0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PRS: There are TCM messages that were not ACKed */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x54, 0x43, 0x4d, 0x20, 0x6d, 0x65, 0x73, 0x73, 0x61,
	0x67, 0x65, 0x73, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x77, 0x65, 0x72,
	0x65, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x41, 0x43, 0x4b, 0x65, 0x64, 0x00,
	/* PRS_REG_TCM_CURRENT_CREDIT */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54, 0x43, 0x4d, 0x5f,
	0x43, 0x55, 0x52, 0x52, 0x45, 0x4e, 0x54, 0x5f, 0x43, 0x52, 0x45, 0x44,
	0x49, 0x54, 0x00,
	/* PRS: CCFC Search Credit Count is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x43, 0x43, 0x46, 0x43, 0x20, 0x53, 0x65,
	0x61, 0x72, 0x63, 0x68, 0x20, 0x43, 0x72, 0x65, 0x64, 0x69, 0x74, 0x20,
	0x43, 0x6f, 0x75, 0x6e, 0x74, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74,
	0x20, 0x30, 0x00,
	/* PRS: There are CCFC search requests that were not ACKed */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x43, 0x43, 0x46, 0x43, 0x20, 0x73, 0x65, 0x61, 0x72,
	0x63, 0x68, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x20,
	0x74, 0x68, 0x61, 0x74, 0x20, 0x77, 0x65, 0x72, 0x65, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x41, 0x43, 0x4b, 0x65, 0x64, 0x00,
	/* PRS_REG_CCFC_SEARCH_CURRENT_CREDIT */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x43, 0x43, 0x46, 0x43,
	0x5f, 0x53, 0x45, 0x41, 0x52, 0x43, 0x48, 0x5f, 0x43, 0x55, 0x52, 0x52,
	0x45, 0x4e, 0x54, 0x5f, 0x43, 0x52, 0x45, 0x44, 0x49, 0x54, 0x00,
	/* PRS: TCFC Search Credit Count is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x43, 0x46, 0x43, 0x20, 0x53, 0x65,
	0x61, 0x72, 0x63, 0x68, 0x20, 0x43, 0x72, 0x65, 0x64, 0x69, 0x74, 0x20,
	0x43, 0x6f, 0x75, 0x6e, 0x74, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74,
	0x20, 0x30, 0x00,
	/* PRS: There are TCFC search requests that were not ACKed */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x54, 0x43, 0x46, 0x43, 0x20, 0x73, 0x65, 0x61, 0x72,
	0x63, 0x68, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x20,
	0x74, 0x68, 0x61, 0x74, 0x20, 0x77, 0x65, 0x72, 0x65, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x41, 0x43, 0x4b, 0x65, 0x64, 0x00,
	/* PRS_REG_TCFC_SEARCH_CURRENT_CREDIT */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54, 0x43, 0x46, 0x43,
	0x5f, 0x53, 0x45, 0x41, 0x52, 0x43, 0x48, 0x5f, 0x43, 0x55, 0x52, 0x52,
	0x45, 0x4e, 0x54, 0x5f, 0x43, 0x52, 0x45, 0x44, 0x49, 0x54, 0x00,
	/* PRS: CCFC Load Credit Count is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x43, 0x43, 0x46, 0x43, 0x20, 0x4c, 0x6f,
	0x61, 0x64, 0x20, 0x43, 0x72, 0x65, 0x64, 0x69, 0x74, 0x20, 0x43, 0x6f,
	0x75, 0x6e, 0x74, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30,
	0x00,
	/* PRS: There are CCFC load requests that were not ACKed */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x43, 0x43, 0x46, 0x43, 0x20, 0x6c, 0x6f, 0x61, 0x64,
	0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x20, 0x74, 0x68,
	0x61, 0x74, 0x20, 0x77, 0x65, 0x72, 0x65, 0x20, 0x6e, 0x6f, 0x74, 0x20,
	0x41, 0x43, 0x4b, 0x65, 0x64, 0x00,
	/* PRS_REG_CCFC_LOAD_CURRENT_CREDIT */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x43, 0x43, 0x46, 0x43,
	0x5f, 0x4c, 0x4f, 0x41, 0x44, 0x5f, 0x43, 0x55, 0x52, 0x52, 0x45, 0x4e,
	0x54, 0x5f, 0x43, 0x52, 0x45, 0x44, 0x49, 0x54, 0x00,
	/* PRS: TCFC Load Credit Count is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x43, 0x46, 0x43, 0x20, 0x4c, 0x6f,
	0x61, 0x64, 0x20, 0x43, 0x72, 0x65, 0x64, 0x69, 0x74, 0x20, 0x43, 0x6f,
	0x75, 0x6e, 0x74, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30,
	0x00,
	/* PRS: There are TCFC load requests that were not ACKed */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x54, 0x43, 0x46, 0x43, 0x20, 0x6c, 0x6f, 0x61, 0x64,
	0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x20, 0x74, 0x68,
	0x61, 0x74, 0x20, 0x77, 0x65, 0x72, 0x65, 0x20, 0x6e, 0x6f, 0x74, 0x20,
	0x41, 0x43, 0x4b, 0x65, 0x64, 0x00,
	/* PRS_REG_TCFC_LOAD_CURRENT_CREDIT */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54, 0x43, 0x46, 0x43,
	0x5f, 0x4c, 0x4f, 0x41, 0x44, 0x5f, 0x43, 0x55, 0x52, 0x52, 0x45, 0x4e,
	0x54, 0x5f, 0x43, 0x52, 0x45, 0x44, 0x49, 0x54, 0x00,
	/* PRS: The number of outstanding CCFC search requests is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x20, 0x6e, 0x75, 0x6d,
	0x62, 0x65, 0x72, 0x20, 0x6f, 0x66, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74,
	0x61, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x43, 0x43, 0x46, 0x43, 0x20,
	0x73, 0x65, 0x61, 0x72, 0x63, 0x68, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65,
	0x73, 0x74, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30,
	0x00,
	/* PRS: There are outstanding CCFC search requests */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74, 0x61, 0x6e, 0x64, 0x69,
	0x6e, 0x67, 0x20, 0x43, 0x43, 0x46, 0x43, 0x20, 0x73, 0x65, 0x61, 0x72,
	0x63, 0x68, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x00,
	/* PRS_REG_CCFC_SEARCH_REQ_CT */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x43, 0x43, 0x46, 0x43,
	0x5f, 0x53, 0x45, 0x41, 0x52, 0x43, 0x48, 0x5f, 0x52, 0x45, 0x51, 0x5f,
	0x43, 0x54, 0x00,
	/* PRS: The number of outstanding TCFC search requests is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x20, 0x6e, 0x75, 0x6d,
	0x62, 0x65, 0x72, 0x20, 0x6f, 0x66, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74,
	0x61, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x54, 0x43, 0x46, 0x43, 0x20,
	0x73, 0x65, 0x61, 0x72, 0x63, 0x68, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65,
	0x73, 0x74, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30,
	0x00,
	/* PRS: There are outstanding TCFC search requests */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74, 0x61, 0x6e, 0x64, 0x69,
	0x6e, 0x67, 0x20, 0x54, 0x43, 0x46, 0x43, 0x20, 0x73, 0x65, 0x61, 0x72,
	0x63, 0x68, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x00,
	/* PRS_REG_TCFC_SEARCH_REQ_CT */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54, 0x43, 0x46, 0x43,
	0x5f, 0x53, 0x45, 0x41, 0x52, 0x43, 0x48, 0x5f, 0x52, 0x45, 0x51, 0x5f,
	0x43, 0x54, 0x00,
	/* PRS: The number of outstanding CCFC load requests is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x20, 0x6e, 0x75, 0x6d,
	0x62, 0x65, 0x72, 0x20, 0x6f, 0x66, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74,
	0x61, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x43, 0x43, 0x46, 0x43, 0x20,
	0x6c, 0x6f, 0x61, 0x64, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74,
	0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PRS: There are outstanding CCFC load requests */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74, 0x61, 0x6e, 0x64, 0x69,
	0x6e, 0x67, 0x20, 0x43, 0x43, 0x46, 0x43, 0x20, 0x6c, 0x6f, 0x61, 0x64,
	0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x00,
	/* PRS_REG_CCFC_LOAD_REQ_CT */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x43, 0x43, 0x46, 0x43,
	0x5f, 0x4c, 0x4f, 0x41, 0x44, 0x5f, 0x52, 0x45, 0x51, 0x5f, 0x43, 0x54,
	0x00,
	/* PRS: The number of outstanding TCFC load requests is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x20, 0x6e, 0x75, 0x6d,
	0x62, 0x65, 0x72, 0x20, 0x6f, 0x66, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74,
	0x61, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x54, 0x43, 0x46, 0x43, 0x20,
	0x6c, 0x6f, 0x61, 0x64, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74,
	0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PRS: There are outstanding TCFC load requests */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74, 0x61, 0x6e, 0x64, 0x69,
	0x6e, 0x67, 0x20, 0x54, 0x43, 0x46, 0x43, 0x20, 0x6c, 0x6f, 0x61, 0x64,
	0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x00,
	/* PRS_REG_TCFC_LOAD_REQ_CT */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54, 0x43, 0x46, 0x43,
	0x5f, 0x4c, 0x4f, 0x41, 0x44, 0x5f, 0x52, 0x45, 0x51, 0x5f, 0x43, 0x54,
	0x00,
	/* PRS: The number of outstanding BRB SOP requests is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x20, 0x6e, 0x75, 0x6d,
	0x62, 0x65, 0x72, 0x20, 0x6f, 0x66, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74,
	0x61, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x42, 0x52, 0x42, 0x20, 0x53,
	0x4f, 0x50, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PRS: There are outstanding BRB SOP requests */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74, 0x61, 0x6e, 0x64, 0x69,
	0x6e, 0x67, 0x20, 0x42, 0x52, 0x42, 0x20, 0x53, 0x4f, 0x50, 0x20, 0x72,
	0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x00,
	/* PRS_REG_SOP_REQ_CT */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53, 0x4f, 0x50, 0x5f,
	0x52, 0x45, 0x51, 0x5f, 0x43, 0x54, 0x00,
	/* PRS: The number of outstanding BRB EOP requests is not 0 */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x20, 0x6e, 0x75, 0x6d,
	0x62, 0x65, 0x72, 0x20, 0x6f, 0x66, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74,
	0x61, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x42, 0x52, 0x42, 0x20, 0x45,
	0x4f, 0x50, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PRS: There are outstanding BRB EOP requests */
	0x50, 0x52, 0x53, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74, 0x61, 0x6e, 0x64, 0x69,
	0x6e, 0x67, 0x20, 0x42, 0x52, 0x42, 0x20, 0x45, 0x4f, 0x50, 0x20, 0x72,
	0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x00,
	/* PRS_REG_EOP_REQ_CT */
	0x50, 0x52, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x45, 0x4f, 0x50, 0x5f,
	0x52, 0x45, 0x51, 0x5f, 0x43, 0x54, 0x00,
	/* PRM: Interrupt status is not 0 */
	0x50, 0x52, 0x4d, 0x3a, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x72, 0x75,
	0x70, 0x74, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73,
	0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PRM_REG_INT_STS */
	0x50, 0x52, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x53, 0x54, 0x53, 0x00,
	/* PRM_REG_INT_MASK */
	0x50, 0x52, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PRM: Parity status is not 0 */
	0x50, 0x52, 0x4d, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79, 0x20,
	0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* PRM_REG_PRTY_STS */
	0x50, 0x52, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PRM_REG_PRTY_MASK */
	0x50, 0x52, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PRM: Parity status is not 0 */
	0x50, 0x52, 0x4d, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79, 0x20,
	0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* PRM_REG_PRTY_STS_H_0 */
	0x50, 0x52, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* PRM_REG_PRTY_MASK_H_0 */
	0x50, 0x52, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* RSS: RSS_INIT_DONE is not 1 */
	0x52, 0x53, 0x53, 0x3a, 0x20, 0x52, 0x53, 0x53, 0x5f, 0x49, 0x4e, 0x49,
	0x54, 0x5f, 0x44, 0x4f, 0x4e, 0x45, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x31, 0x00,
	/* RSS: Block is not initialized */
	0x52, 0x53, 0x53, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x69, 0x74, 0x69, 0x61,
	0x6c, 0x69, 0x7a, 0x65, 0x64, 0x00,
	/* RSS_REG_RSS_INIT_DONE */
	0x52, 0x53, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x52, 0x53, 0x53, 0x5f,
	0x49, 0x4e, 0x49, 0x54, 0x5f, 0x44, 0x4f, 0x4e, 0x45, 0x00,
	/* RSS: Interrupt status is set */
	0x52, 0x53, 0x53, 0x3a, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x72, 0x75,
	0x70, 0x74, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73,
	0x20, 0x73, 0x65, 0x74, 0x00,
	/* RSS_REG_INT_STS */
	0x52, 0x53, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x53, 0x54, 0x53, 0x00,
	/* RSS_REG_INT_MASK */
	0x52, 0x53, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* RSS: Parity status is not 0 */
	0x52, 0x53, 0x53, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79, 0x20,
	0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* RSS_REG_PRTY_STS_H_0 */
	0x52, 0x53, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* RSS_REG_PRTY_MASK_H_0 */
	0x52, 0x53, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* RSS: RSS-TMLD interface credits != reset value (BB/AH=0x20:E5=0x1e) */
	0x52, 0x53, 0x53, 0x3a, 0x20, 0x52, 0x53, 0x53, 0x2d, 0x54, 0x4d, 0x4c,
	0x44, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x66, 0x61, 0x63, 0x65, 0x20,
	0x63, 0x72, 0x65, 0x64, 0x69, 0x74, 0x73, 0x20, 0x21, 0x3d, 0x20, 0x72,
	0x65, 0x73, 0x65, 0x74, 0x20, 0x76, 0x61, 0x6c, 0x75, 0x65, 0x20, 0x28,
	0x42, 0x42, 0x2f, 0x41, 0x48, 0x3d, 0x30, 0x78, 0x32, 0x30, 0x3a, 0x45,
	0x35, 0x3d, 0x30, 0x78, 0x31, 0x65, 0x29, 0x00,
	/* RSS: Pending messages on RSS-TMLD interface */
	0x52, 0x53, 0x53, 0x3a, 0x20, 0x50, 0x65, 0x6e, 0x64, 0x69, 0x6e, 0x67,
	0x20, 0x6d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x73, 0x20, 0x6f, 0x6e,
	0x20, 0x52, 0x53, 0x53, 0x2d, 0x54, 0x4d, 0x4c, 0x44, 0x20, 0x69, 0x6e,
	0x74, 0x65, 0x72, 0x66, 0x61, 0x63, 0x65, 0x00,
	/* RSS_REG_TMLD_CREDIT */
	0x52, 0x53, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54, 0x4d, 0x4c, 0x44,
	0x5f, 0x43, 0x52, 0x45, 0x44, 0x49, 0x54, 0x00,
	/* RSS: FIFOS are not empty */
	0x52, 0x53, 0x53, 0x3a, 0x20, 0x46, 0x49, 0x46, 0x4f, 0x53, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79,
	0x00,
	/* RSS_REG_EMPTY_STATUS */
	0x52, 0x53, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x45, 0x4d, 0x50, 0x54,
	0x59, 0x5f, 0x53, 0x54, 0x41, 0x54, 0x55, 0x53, 0x00,
	/* RSS: FIFOS are full */
	0x52, 0x53, 0x53, 0x3a, 0x20, 0x46, 0x49, 0x46, 0x4f, 0x53, 0x20, 0x61,
	0x72, 0x65, 0x20, 0x66, 0x75, 0x6c, 0x6c, 0x00,
	/* RSS_REG_FULL_STATUS */
	0x52, 0x53, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x55, 0x4c, 0x4c,
	0x5f, 0x53, 0x54, 0x41, 0x54, 0x55, 0x53, 0x00,
	/* RSS: Counters are not 0 */
	0x52, 0x53, 0x53, 0x3a, 0x20, 0x43, 0x6f, 0x75, 0x6e, 0x74, 0x65, 0x72,
	0x73, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* RSS_REG_COUNTERS_STATUS */
	0x52, 0x53, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x43, 0x4f, 0x55, 0x4e,
	0x54, 0x45, 0x52, 0x53, 0x5f, 0x53, 0x54, 0x41, 0x54, 0x55, 0x53, 0x00,
	/* RSS: State machines are not in reset state */
	0x52, 0x53, 0x53, 0x3a, 0x20, 0x53, 0x74, 0x61, 0x74, 0x65, 0x20, 0x6d,
	0x61, 0x63, 0x68, 0x69, 0x6e, 0x65, 0x73, 0x20, 0x61, 0x72, 0x65, 0x20,
	0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x20, 0x72, 0x65, 0x73, 0x65, 0x74,
	0x20, 0x73, 0x74, 0x61, 0x74, 0x65, 0x00,
	/* RSS_REG_STATE_MACHINES */
	0x52, 0x53, 0x53, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53, 0x54, 0x41, 0x54,
	0x45, 0x5f, 0x4d, 0x41, 0x43, 0x48, 0x49, 0x4e, 0x45, 0x53, 0x00,
	/* PTLD: INT_STS reg is set */
	0x50, 0x54, 0x4c, 0x44, 0x3a, 0x20, 0x49, 0x4e, 0x54, 0x5f, 0x53, 0x54,
	0x53, 0x20, 0x72, 0x65, 0x67, 0x20, 0x69, 0x73, 0x20, 0x73, 0x65, 0x74,
	0x00,
	/* RPB_REG_INT_STS */
	0x52, 0x50, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x53, 0x54, 0x53, 0x00,
	/* RPB_REG_INT_MASK */
	0x52, 0x50, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e, 0x54, 0x5f,
	0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* RPB: Parity status is not 0 */
	0x52, 0x50, 0x42, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69, 0x74, 0x79, 0x20,
	0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x00,
	/* RPB_REG_PRTY_STS */
	0x52, 0x50, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x53, 0x54, 0x53, 0x00,
	/* RPB_REG_PRTY_MASK */
	0x52, 0x50, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52, 0x54, 0x59,
	0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* RPB: Data Buffer is not empty */
	0x52, 0x50, 0x42, 0x3a, 0x20, 0x44, 0x61, 0x74, 0x61, 0x20, 0x42, 0x75,
	0x66, 0x66, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20,
	0x65, 0x6d, 0x70, 0x74, 0x79, 0x00,
	/* RPB_REG_DB_EMPTY */
	0x52, 0x50, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44, 0x42, 0x5f, 0x45,
	0x4d, 0x50, 0x54, 0x59, 0x00,
	/* RPB_REG_DB_FULL */
	0x52, 0x50, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44, 0x42, 0x5f, 0x46,
	0x55, 0x4c, 0x4c, 0x00,
	/* RPB: Task Queue is not Empty */
	0x52, 0x50, 0x42, 0x3a, 0x20, 0x54, 0x61, 0x73, 0x6b, 0x20, 0x51, 0x75,
	0x65, 0x75, 0x65, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x45,
	0x6d, 0x70, 0x74, 0x79, 0x00,
	/* RPB_REG_TQ_EMPTY */
	0x52, 0x50, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54, 0x51, 0x5f, 0x45,
	0x4d, 0x50, 0x54, 0x59, 0x00,
	/* RPB_REG_TQ_FULL */
	0x52, 0x50, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54, 0x51, 0x5f, 0x46,
	0x55, 0x4c, 0x4c, 0x00,
	/* RPB: Instruction FIFO is not empty */
	0x52, 0x50, 0x42, 0x3a, 0x20, 0x49, 0x6e, 0x73, 0x74, 0x72, 0x75, 0x63,
	0x74, 0x69, 0x6f, 0x6e, 0x20, 0x46, 0x49, 0x46, 0x4f, 0x20, 0x69, 0x73,
	0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x00,
	/* RPB_REG_IFIFO_EMPTY */
	0x52, 0x50, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x46, 0x49, 0x46,
	0x4f, 0x5f, 0x45, 0x4d, 0x50, 0x54, 0x59, 0x00,
	/* RPB_REG_IFIFO_FULL */
	0x52, 0x50, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x46, 0x49, 0x46,
	0x4f, 0x5f, 0x46, 0x55, 0x4c, 0x4c, 0x00,
	/* RPB: Parameter FIFO is not empty */
	0x52, 0x50, 0x42, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x61, 0x6d, 0x65, 0x74,
	0x65, 0x72, 0x20, 0x46, 0x49, 0x46, 0x4f, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x00,
	/* RPB_REG_PFIFO_EMPTY */
	0x52, 0x50, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x46, 0x49, 0x46,
	0x4f, 0x5f, 0x45, 0x4d, 0x50, 0x54, 0x59, 0x00,
	/* RPB_REG_PFIFO_FULL */
	0x52, 0x50, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x46, 0x49, 0x46,
	0x4f, 0x5f, 0x46, 0x55, 0x4c, 0x4c, 0x00,
	/* RPB: Task Queue for task handler is not empty */
	0x52, 0x50, 0x42, 0x3a, 0x20, 0x54, 0x61, 0x73, 0x6b, 0x20, 0x51, 0x75,
	0x65, 0x75, 0x65, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x74, 0x61, 0x73, 0x6b,
	0x20, 0x68, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20,
	0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x00,
	/* RPB_REG_TQ_TH_EMPTY */
	0x52, 0x50, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54, 0x51, 0x5f, 0x54,
	0x48, 0x5f, 0x45, 0x4d, 0x50, 0x54, 0x59, 0x00,
	/* PSWRQ2: rq_rbc_done is not 1 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x72, 0x71, 0x5f, 0x72,
	0x62, 0x63, 0x5f, 0x64, 0x6f, 0x6e, 0x65, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x31, 0x00,
	/* PSWRQ2: Block is not initialized */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63,
	0x6b, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x69,
	0x74, 0x69, 0x61, 0x6c, 0x69, 0x7a, 0x65, 0x64, 0x00,
	/* PSWRQ2_REG_RBC_DONE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x52,
	0x42, 0x43, 0x5f, 0x44, 0x4f, 0x4e, 0x45, 0x00,
	/* PSWRQ2: rq_cfg_done is not 1 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x72, 0x71, 0x5f, 0x63,
	0x66, 0x67, 0x5f, 0x64, 0x6f, 0x6e, 0x65, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x31, 0x00,
	/* PSWRQ2: Block is not initialized */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63,
	0x6b, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x69,
	0x74, 0x69, 0x61, 0x6c, 0x69, 0x7a, 0x65, 0x64, 0x00,
	/* PSWRQ2_REG_CFG_DONE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x43,
	0x46, 0x47, 0x5f, 0x44, 0x4f, 0x4e, 0x45, 0x00,
	/* PSWRQ2: Interrupt status is not 0 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x49, 0x6e, 0x74, 0x65,
	0x72, 0x72, 0x75, 0x70, 0x74, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWRQ2_REG_INT_STS */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x4e, 0x54, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWRQ2_REG_INT_MASK */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x4e, 0x54, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PSWRQ2: Parity status is not 0 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69,
	0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73,
	0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWRQ2_REG_PRTY_STS_H_0 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* PSWRQ2_REG_PRTY_MASK_H_0 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30,
	0x00,
	/* PSWRQ2: VQ is not empty - CDU_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x43, 0x44, 0x55, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_VQ0_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x30, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e, 0x54,
	0x00,
	/* PSWRQ2: VQ is not empty - SRCH_RW */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x53, 0x52, 0x43, 0x48, 0x5f, 0x52, 0x57, 0x00,
	/* PSWRQ2_REG_VQ1_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x31, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e, 0x54,
	0x00,
	/* PSWRQ2: VQ is not empty - QM RW */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x51, 0x4d, 0x20, 0x52, 0x57, 0x00,
	/* PSWRQ2_REG_VQ2_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x32, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e, 0x54,
	0x00,
	/* PSWRQ2: VQ is not empty - TM_REG_PCI_VQ_ID */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x54, 0x4d, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x43, 0x49,
	0x5f, 0x56, 0x51, 0x5f, 0x49, 0x44, 0x00,
	/* PSWRQ2_REG_VQ3_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x33, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e, 0x54,
	0x00,
	/* PSWRQ2: VQ is not empty - PBF_REG_PCI_VQ_ID */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x50, 0x42, 0x46, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x43,
	0x49, 0x5f, 0x56, 0x51, 0x5f, 0x49, 0x44, 0x00,
	/* PSWRQ2_REG_VQ4_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x34, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e, 0x54,
	0x00,
	/* PSWRQ2: VQ is not empty - MULD R */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x4d, 0x55, 0x4c, 0x44, 0x20, 0x52, 0x00,
	/* PSWRQ2_REG_VQ5_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x35, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e, 0x54,
	0x00,
	/* PSWRQ2: VQ is not empty - ALL_SDM_READ_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x00,
	/* PSWRQ2_REG_VQ6_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x36, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e, 0x54,
	0x00,
	/* PSWRQ2: VQ is not empty - ALL_SDM_READ_WRITE  YSDM */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x20, 0x20, 0x59, 0x53,
	0x44, 0x4d, 0x00,
	/* PSWRQ2_REG_VQ7_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x37, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e, 0x54,
	0x00,
	/* PSWRQ2: VQ is not empty - ALL_SDM_READ_WRITE  PSDM */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x20, 0x20, 0x50, 0x53,
	0x44, 0x4d, 0x00,
	/* PSWRQ2_REG_VQ8_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x38, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e, 0x54,
	0x00,
	/* PSWRQ2: VQ is not empty - PRM_READ_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x50, 0x52, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x5f, 0x57,
	0x52, 0x49, 0x54, 0x45, 0x00,
	/* PSWRQ2_REG_VQ9_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x39, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e, 0x54,
	0x00,
	/* PSWRQ2: VQ is not empty - ALL_SDM_READ_OR_XSDM_READ_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x5f, 0x4f, 0x52, 0x5f, 0x58, 0x53, 0x44, 0x4d, 0x5f, 0x52,
	0x45, 0x41, 0x44, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x00,
	/* PSWRQ2_REG_VQ10_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x31, 0x30, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - ALL_SDM_READ_OR_TSDM_READ_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x5f, 0x4f, 0x52, 0x5f, 0x54, 0x53, 0x44, 0x4d, 0x5f, 0x52,
	0x45, 0x41, 0x44, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x00,
	/* PSWRQ2_REG_VQ11_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x31, 0x31, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - XSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x58, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_VQ12_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x31, 0x32, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - XSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x58, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_VQ13_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x31, 0x33, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - ALL_SDM_READ_WRITE  MSDM */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x20, 0x20, 0x4d, 0x53,
	0x44, 0x4d, 0x00,
	/* PSWRQ2_REG_VQ14_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x31, 0x34, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - MSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x4d, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_VQ15_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x31, 0x35, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - USDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x55, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_VQ16_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x31, 0x36, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - USDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x55, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_VQ17_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x31, 0x37, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - USDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x55, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_VQ18_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x31, 0x38, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - ALL_SDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x00,
	/* PSWRQ2_REG_VQ19_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x31, 0x39, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - PSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x50, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_VQ20_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x32, 0x30, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - IGU_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x49, 0x47, 0x55, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x00,
	/* PSWRQ2_REG_VQ21_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x32, 0x31, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - YSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x59, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_VQ22_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x32, 0x32, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - YSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x59, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_VQ23_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x32, 0x33, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - XYLD R */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x58, 0x59, 0x4c, 0x44, 0x20, 0x52, 0x00,
	/* PSWRQ2_REG_VQ24_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x32, 0x34, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - ALL_SDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x00,
	/* PSWRQ2_REG_VQ25_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x32, 0x35, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - PTU_REG_VQID_CFG */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x50, 0x54, 0x55, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56, 0x51,
	0x49, 0x44, 0x5f, 0x43, 0x46, 0x47, 0x00,
	/* PSWRQ2_REG_VQ26_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x32, 0x36, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - MSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x4d, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_VQ27_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x32, 0x37, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - DMAE RW */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x44, 0x4d, 0x41, 0x45, 0x20, 0x52, 0x57, 0x00,
	/* PSWRQ2_REG_VQ28_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x32, 0x38, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - CDU_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x43, 0x44, 0x55, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x00,
	/* PSWRQ2_REG_VQ29_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x32, 0x39, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - DBG_REG_PCI_VQ_ID */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x44, 0x42, 0x47, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x43,
	0x49, 0x5f, 0x56, 0x51, 0x5f, 0x49, 0x44, 0x00,
	/* PSWRQ2_REG_VQ30_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x33, 0x30, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: VQ is not empty - ALL_SDM_READ_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x56, 0x51, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x20,
	0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x00,
	/* PSWRQ2_REG_VQ31_ENTRY_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x56,
	0x51, 0x33, 0x31, 0x5f, 0x45, 0x4e, 0x54, 0x52, 0x59, 0x5f, 0x43, 0x4e,
	0x54, 0x00,
	/* PSWRQ2: There are used credits on PSWRQ - PGLUE request interface. */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72,
	0x65, 0x20, 0x61, 0x72, 0x65, 0x20, 0x75, 0x73, 0x65, 0x64, 0x20, 0x63,
	0x72, 0x65, 0x64, 0x69, 0x74, 0x73, 0x20, 0x6f, 0x6e, 0x20, 0x50, 0x53,
	0x57, 0x52, 0x51, 0x20, 0x2d, 0x20, 0x50, 0x47, 0x4c, 0x55, 0x45, 0x20,
	0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x20, 0x69, 0x6e, 0x74, 0x65,
	0x72, 0x66, 0x61, 0x63, 0x65, 0x2e, 0x00,
	/* PSWRQ2_REG_BW_CREDIT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x57, 0x5f, 0x43, 0x52, 0x45, 0x44, 0x49, 0x54, 0x00,
	/* PSWRQ2: TREQ FIFO is not empty */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x54, 0x52, 0x45, 0x51,
	0x20, 0x46, 0x49, 0x46, 0x4f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74,
	0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x00,
	/* PSWRQ2_REG_TREQ_FIFO_FILL_LVL */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54,
	0x52, 0x45, 0x51, 0x5f, 0x46, 0x49, 0x46, 0x4f, 0x5f, 0x46, 0x49, 0x4c,
	0x4c, 0x5f, 0x4c, 0x56, 0x4c, 0x00,
	/* PSWRQ2: ICPL FIFO is not empty */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x49, 0x43, 0x50, 0x4c,
	0x20, 0x46, 0x49, 0x46, 0x4f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74,
	0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x00,
	/* PSWRQ2_REG_ICPL_FIFO_FILL_LVL */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x43, 0x50, 0x4c, 0x5f, 0x46, 0x49, 0x46, 0x4f, 0x5f, 0x46, 0x49, 0x4c,
	0x4c, 0x5f, 0x4c, 0x56, 0x4c, 0x00,
	/* PSWRQ: L2P error. Error details register is not 0. (12:0) - Length in bytes. (16:13) - PFID. (17) - VF_VALID. (25:18) - VFID. */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x3a, 0x20, 0x4c, 0x32, 0x50, 0x20, 0x65,
	0x72, 0x72, 0x6f, 0x72, 0x2e, 0x20, 0x45, 0x72, 0x72, 0x6f, 0x72, 0x20,
	0x64, 0x65, 0x74, 0x61, 0x69, 0x6c, 0x73, 0x20, 0x72, 0x65, 0x67, 0x69,
	0x73, 0x74, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20,
	0x30, 0x2e, 0x20, 0x28, 0x31, 0x32, 0x3a, 0x30, 0x29, 0x20, 0x2d, 0x20,
	0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x20, 0x69, 0x6e, 0x20, 0x62, 0x79,
	0x74, 0x65, 0x73, 0x2e, 0x20, 0x28, 0x31, 0x36, 0x3a, 0x31, 0x33, 0x29,
	0x20, 0x2d, 0x20, 0x50, 0x46, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x31, 0x37,
	0x29, 0x20, 0x2d, 0x20, 0x56, 0x46, 0x5f, 0x56, 0x41, 0x4c, 0x49, 0x44,
	0x2e, 0x20, 0x28, 0x32, 0x35, 0x3a, 0x31, 0x38, 0x29, 0x20, 0x2d, 0x20,
	0x56, 0x46, 0x49, 0x44, 0x2e, 0x00,
	/* PSWRQ2_REG_L2P_ERR_DETAILS */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4c,
	0x32, 0x50, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49,
	0x4c, 0x53, 0x00,
	/* PSWRQ2_REG_L2P_ERR_ADD_63_32 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4c,
	0x32, 0x50, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x41, 0x44, 0x44, 0x5f, 0x36,
	0x33, 0x5f, 0x33, 0x32, 0x00,
	/* PSWRQ2_REG_L2P_ERR_ADD_31_0 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4c,
	0x32, 0x50, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x41, 0x44, 0x44, 0x5f, 0x33,
	0x31, 0x5f, 0x30, 0x00,
	/* PSWRQ: L2P error. Error details register 2 is not 0. (15:0) - Request ID. (20:16) - client ID. (21) - Error type - 0 - rq_l2p_vf_err; 1 - rq_elt_addr. (22) - w_nr - 0 - read; 1 - write.(27:23) - VQID. (28) - valid bit. */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x3a, 0x20, 0x4c, 0x32, 0x50, 0x20, 0x65,
	0x72, 0x72, 0x6f, 0x72, 0x2e, 0x20, 0x45, 0x72, 0x72, 0x6f, 0x72, 0x20,
	0x64, 0x65, 0x74, 0x61, 0x69, 0x6c, 0x73, 0x20, 0x72, 0x65, 0x67, 0x69,
	0x73, 0x74, 0x65, 0x72, 0x20, 0x32, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x2e, 0x20, 0x28, 0x31, 0x35, 0x3a, 0x30, 0x29, 0x20,
	0x2d, 0x20, 0x52, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x20, 0x49, 0x44,
	0x2e, 0x20, 0x28, 0x32, 0x30, 0x3a, 0x31, 0x36, 0x29, 0x20, 0x2d, 0x20,
	0x63, 0x6c, 0x69, 0x65, 0x6e, 0x74, 0x20, 0x49, 0x44, 0x2e, 0x20, 0x28,
	0x32, 0x31, 0x29, 0x20, 0x2d, 0x20, 0x45, 0x72, 0x72, 0x6f, 0x72, 0x20,
	0x74, 0x79, 0x70, 0x65, 0x20, 0x2d, 0x20, 0x30, 0x20, 0x2d, 0x20, 0x72,
	0x71, 0x5f, 0x6c, 0x32, 0x70, 0x5f, 0x76, 0x66, 0x5f, 0x65, 0x72, 0x72,
	0x3b, 0x20, 0x31, 0x20, 0x2d, 0x20, 0x72, 0x71, 0x5f, 0x65, 0x6c, 0x74,
	0x5f, 0x61, 0x64, 0x64, 0x72, 0x2e, 0x20, 0x28, 0x32, 0x32, 0x29, 0x20,
	0x2d, 0x20, 0x77, 0x5f, 0x6e, 0x72, 0x20, 0x2d, 0x20, 0x30, 0x20, 0x2d,
	0x20, 0x72, 0x65, 0x61, 0x64, 0x3b, 0x20, 0x31, 0x20, 0x2d, 0x20, 0x77,
	0x72, 0x69, 0x74, 0x65, 0x2e, 0x28, 0x32, 0x37, 0x3a, 0x32, 0x33, 0x29,
	0x20, 0x2d, 0x20, 0x56, 0x51, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x32, 0x38,
	0x29, 0x20, 0x2d, 0x20, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20, 0x62, 0x69,
	0x74, 0x2e, 0x00,
	/* PSWRQ2_REG_L2P_ERR_DETAILS2 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4c,
	0x32, 0x50, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49,
	0x4c, 0x53, 0x32, 0x00,
	/* PSWRQ2: sr_cnt is not equal to sr_num_cfg */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x73, 0x72, 0x5f, 0x63,
	0x6e, 0x74, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71,
	0x75, 0x61, 0x6c, 0x20, 0x74, 0x6f, 0x20, 0x73, 0x72, 0x5f, 0x6e, 0x75,
	0x6d, 0x5f, 0x63, 0x66, 0x67, 0x00,
	/* PSWRQ2_REG_SR_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x00,
	/* PSWRQ2_REG_SR_NUM_CFG */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x43, 0x46, 0x47, 0x00,
	/* PSWRQ2: There are more than four used SRs */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72,
	0x65, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6d, 0x6f, 0x72, 0x65, 0x20, 0x74,
	0x68, 0x61, 0x6e, 0x20, 0x66, 0x6f, 0x75, 0x72, 0x20, 0x75, 0x73, 0x65,
	0x64, 0x20, 0x53, 0x52, 0x73, 0x00,
	/* PSWRQ2_REG_SR_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x00,
	/* PSWRQ2_REG_SR_NUM_CFG */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x43, 0x46, 0x47, 0x00,
	/* PSWRQ2: SR_CNT_per_vq0 is not equal to MAX_SRS_vq - CDU_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x30, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c, 0x20,
	0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x76,
	0x71, 0x20, 0x2d, 0x20, 0x43, 0x44, 0x55, 0x5f, 0x52, 0x45, 0x41, 0x44,
	0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_0 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x30, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ0 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x30, 0x00,
	/* PSWRQ2: SR_CNT_per_vq1 is not equal to MAX_SRS_vq - SRCH_RW */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c, 0x20,
	0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x76,
	0x71, 0x20, 0x2d, 0x20, 0x53, 0x52, 0x43, 0x48, 0x5f, 0x52, 0x57, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_1 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x31, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ1 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x00,
	/* PSWRQ2: SR_CNT_per_vq2 is not equal to MAX_SRS_vq - QM RW */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c, 0x20,
	0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x76,
	0x71, 0x20, 0x2d, 0x20, 0x51, 0x4d, 0x20, 0x52, 0x57, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_2 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x32, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ2 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x00,
	/* PSWRQ2: SR_CNT_per_vq3 is not equal to MAX_SRS_vq - DBG_REG_PCI_VQ_ID */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x33, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c, 0x20,
	0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x76,
	0x71, 0x20, 0x2d, 0x20, 0x44, 0x42, 0x47, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x50, 0x43, 0x49, 0x5f, 0x56, 0x51, 0x5f, 0x49, 0x44, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_3 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x33, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ3 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x33, 0x00,
	/* PSWRQ2: SR_CNT_per_vq4 is not equal to MAX_SRS_vq - PBF_REG_PCI_VQ_ID */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x34, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c, 0x20,
	0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x76,
	0x71, 0x20, 0x2d, 0x20, 0x50, 0x42, 0x46, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x50, 0x43, 0x49, 0x5f, 0x56, 0x51, 0x5f, 0x49, 0x44, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_4 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x34, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ4 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x34, 0x00,
	/* PSWRQ2: SR_CNT_per_vq5 is not equal to MAX_SRS_vq - MULD R */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x35, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c, 0x20,
	0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x76,
	0x71, 0x20, 0x2d, 0x20, 0x4d, 0x55, 0x4c, 0x44, 0x20, 0x52, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_5 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x35, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ5 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x35, 0x00,
	/* PSWRQ2: SR_CNT_per_vq6 is not equal to MAX_SRS_vq - ALL_SDM_READ_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x36, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c, 0x20,
	0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x76,
	0x71, 0x20, 0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d, 0x5f,
	0x52, 0x45, 0x41, 0x44, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_6 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x36, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ6 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x36, 0x00,
	/* PSWRQ2: SR_CNT_per_vq9 is not equal to MAX_SRS_vq - ALL_SDM_READ_WRITE or PRM_READ_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x39, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c, 0x20,
	0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x76,
	0x71, 0x20, 0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d, 0x5f,
	0x52, 0x45, 0x41, 0x44, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x20, 0x6f,
	0x72, 0x20, 0x50, 0x52, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x5f, 0x57,
	0x52, 0x49, 0x54, 0x45, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_9 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x39, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ9 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x39, 0x00,
	/* PSWRQ2: SR_CNT_per_vq11 is not equal to MAX_SRS_vq - ALL_SDM_READ_OR_TSDM_READ_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x31, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d,
	0x5f, 0x52, 0x45, 0x41, 0x44, 0x5f, 0x4f, 0x52, 0x5f, 0x54, 0x53, 0x44,
	0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45,
	0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_11 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x31, 0x31, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ11 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x31, 0x00,
	/* PSWRQ2: SR_CNT_per_vq12 is not equal to MAX_SRS_vq - XSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x32, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x58, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_12 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x31, 0x32, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ12 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x32, 0x00,
	/* PSWRQ2: SR_CNT_per_vq15 is not equal to MAX_SRS_vq - MSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x35, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x4d, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_15 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x31, 0x35, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ15 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x35, 0x00,
	/* PSWRQ2: SR_CNT_per_vq16 is not equal to MAX_SRS_vq - USDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x36, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x55, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_16 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x31, 0x36, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ16 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x36, 0x00,
	/* PSWRQ2: SR_CNT_per_vq17 is not equal to MAX_SRS_vq - USDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x37, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x55, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_17 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x31, 0x37, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ17 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x37, 0x00,
	/* PSWRQ2: SR_CNT_per_vq18 is not equal to MAX_SRS_vq - USDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x38, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x55, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_18 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x31, 0x38, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ18 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x38, 0x00,
	/* PSWRQ2: SR_CNT_per_vq19 is not equal to MAX_SRS_vq - ALL_SDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x39, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d,
	0x5f, 0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_19 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x31, 0x39, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ19 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x39, 0x00,
	/* PSWRQ2: SR_CNT_per_vq21 is not equal to MAX_SRS_vq - IGU_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x31, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x49, 0x47, 0x55, 0x5f, 0x57, 0x52, 0x49,
	0x54, 0x45, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_21 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x32, 0x31, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ21 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x31, 0x00,
	/* PSWRQ2: SR_CNT_per_vq22 is not equal to MAX_SRS_vq - YSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x32, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x59, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_22 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x32, 0x32, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ22 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x32, 0x00,
	/* PSWRQ2: SR_CNT_per_vq23 is not equal to MAX_SRS_vq - YSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x33, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x59, 0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_23 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x32, 0x33, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ23 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x33, 0x00,
	/* PSWRQ2: SR_CNT_per_vq24 is not equal to MAX_SRS_vq - XYLD R */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x34, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x58, 0x59, 0x4c, 0x44, 0x20, 0x52, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_24 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x32, 0x34, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ24 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x34, 0x00,
	/* PSWRQ2: SR_CNT_per_vq26 is not equal to MAX_SRS_vq - PTU_REG_VQID_CFG */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x36, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x50, 0x54, 0x55, 0x5f, 0x52, 0x45, 0x47,
	0x5f, 0x56, 0x51, 0x49, 0x44, 0x5f, 0x43, 0x46, 0x47, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_26 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x32, 0x36, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ26 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x36, 0x00,
	/* PSWRQ2: SR_CNT_per_vq28 is not equal to MAX_SRS_vq - DMAE RW */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x38, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x44, 0x4d, 0x41, 0x45, 0x20, 0x52, 0x57,
	0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_28 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x32, 0x38, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ28 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x38, 0x00,
	/* PSWRQ2: SR_CNT_per_vq29 is not equal to MAX_SRS_vq - CDU_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x39, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x43, 0x44, 0x55, 0x5f, 0x57, 0x52, 0x49,
	0x54, 0x45, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_29 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x32, 0x39, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ29 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x39, 0x00,
	/* PSWRQ2: SR_CNT_per_vq30 is not equal to MAX_SRS_vq - ALL_SDM_READ_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x53, 0x52, 0x5f, 0x43,
	0x4e, 0x54, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x33, 0x30, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x4d, 0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f,
	0x76, 0x71, 0x20, 0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44, 0x4d,
	0x5f, 0x52, 0x45, 0x41, 0x44, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x00,
	/* PSWRQ2_REG_SR_CNT_PER_VQ_30 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x52, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56, 0x51,
	0x5f, 0x33, 0x30, 0x00,
	/* PSWRQ2_REG_MAX_SRS_VQ30 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x53, 0x52, 0x53, 0x5f, 0x56, 0x51, 0x33, 0x30, 0x00,
	/* PSWRQ2: blk_cnt is not equal to blk_num_cfg */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65,
	0x71, 0x75, 0x61, 0x6c, 0x20, 0x74, 0x6f, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x6e, 0x75, 0x6d, 0x5f, 0x63, 0x66, 0x67, 0x00,
	/* PSWRQ2_REG_BLK_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x00,
	/* PSWRQ2_REG_BLK_NUM_CFG */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x43, 0x46, 0x47, 0x00,
	/* PSWRQ2: There are more than eight used blocks */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x54, 0x68, 0x65, 0x72,
	0x65, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6d, 0x6f, 0x72, 0x65, 0x20, 0x74,
	0x68, 0x61, 0x6e, 0x20, 0x65, 0x69, 0x67, 0x68, 0x74, 0x20, 0x75, 0x73,
	0x65, 0x64, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x73, 0x00,
	/* PSWRQ2_REG_BLK_CNT */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x00,
	/* PSWRQ2_REG_BLK_NUM_CFG */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x4e, 0x55, 0x4d, 0x5f, 0x43, 0x46, 0x47, 0x00,
	/* PSWRQ2: blk_cnt_per_vq0 is not equal to max_blks_vq - CDU_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x30, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b, 0x73,
	0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x43, 0x44, 0x55, 0x5f, 0x52, 0x45,
	0x41, 0x44, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_0 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x30, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ0 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x30, 0x00,
	/* PSWRQ2: blk_cnt_per_vq1 is not equal to max_blks_vq - SRCH_RW */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b, 0x73,
	0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x53, 0x52, 0x43, 0x48, 0x5f, 0x52,
	0x57, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_1 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x31, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ1 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x00,
	/* PSWRQ2: blk_cnt_per_vq2 is not equal to max_blks_vq - QM RW */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b, 0x73,
	0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x51, 0x4d, 0x20, 0x52, 0x57, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_2 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x32, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ2 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x00,
	/* PSWRQ2: blk_cnt_per_vq3 is not equal to max_blks_vq - TM_REG_PCI_VQ_ID */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x33, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b, 0x73,
	0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x54, 0x4d, 0x5f, 0x52, 0x45, 0x47,
	0x5f, 0x50, 0x43, 0x49, 0x5f, 0x56, 0x51, 0x5f, 0x49, 0x44, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_3 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x33, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ3 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x33, 0x00,
	/* PSWRQ2: blk_cnt_per_vq4 is not equal to max_blks_vq - PBF_REG_PCI_VQ_ID */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x34, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b, 0x73,
	0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x50, 0x42, 0x46, 0x5f, 0x52, 0x45,
	0x47, 0x5f, 0x50, 0x43, 0x49, 0x5f, 0x56, 0x51, 0x5f, 0x49, 0x44, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_4 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x34, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ4 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x34, 0x00,
	/* PSWRQ2: blk_cnt_per_vq5 is not equal to max_blks_vq - MULD R */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x35, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b, 0x73,
	0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x4d, 0x55, 0x4c, 0x44, 0x20, 0x52,
	0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_5 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x35, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ5 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x35, 0x00,
	/* PSWRQ2: blk_cnt_per_vq6 is not equal to max_blks_vq - ALL_SDM_READ_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x36, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b, 0x73,
	0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44,
	0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45,
	0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_6 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x36, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ6 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x36, 0x00,
	/* PSWRQ2: blk_cnt_per_vq9 is not equal to max_blks_vq - ALL_SDM_READ_WRITE or PRM_READ_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x39, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61, 0x6c,
	0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b, 0x73,
	0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53, 0x44,
	0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45,
	0x20, 0x6f, 0x72, 0x20, 0x50, 0x52, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44,
	0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_9 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x39, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ9 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x39, 0x00,
	/* PSWRQ2: blk_cnt_per_vq11 is not equal to max_blks_vq - ALL_SDM_READ_OR_TSDM_READ_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x31,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53,
	0x44, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x5f, 0x4f, 0x52, 0x5f, 0x54,
	0x53, 0x44, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x5f, 0x57, 0x52, 0x49,
	0x54, 0x45, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_11 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x31, 0x31, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ11 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x31,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq12 is not equal to max_blks_vq - XSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x32,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x58, 0x53, 0x44, 0x4d, 0x5f,
	0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_12 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x31, 0x32, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ12 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x32,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq15 is not equal to max_blks_vq - MSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x35,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x4d, 0x53, 0x44, 0x4d, 0x5f,
	0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_15 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x31, 0x35, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ15 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x35,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq16 is not equal to max_blks_vq - USDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x36,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x55, 0x53, 0x44, 0x4d, 0x5f,
	0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_16 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x31, 0x36, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ16 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x36,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq17 is not equal to max_blks_vq - USDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x37,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x55, 0x53, 0x44, 0x4d, 0x5f,
	0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_17 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x31, 0x37, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ17 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x37,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq18 is not equal to max_blks_vq - USDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x38,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x55, 0x53, 0x44, 0x4d, 0x5f,
	0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_18 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x31, 0x38, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ18 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x38,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq19 is not equal to max_blks_vq - ALL_SDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x31, 0x39,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x41, 0x4c, 0x4c, 0x5f, 0x53,
	0x44, 0x4d, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_19 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x31, 0x39, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ19 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x31, 0x39,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq21 is not equal to max_blks_vq - IGU_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x31,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x49, 0x47, 0x55, 0x5f, 0x57,
	0x52, 0x49, 0x54, 0x45, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_21 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x32, 0x31, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ21 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x31,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq22 is not equal to max_blks_vq - YSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x32,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x59, 0x53, 0x44, 0x4d, 0x5f,
	0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_22 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x32, 0x32, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ22 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x32,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq23 is not equal to max_blks_vq - YSDM_READ */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x33,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x59, 0x53, 0x44, 0x4d, 0x5f,
	0x52, 0x45, 0x41, 0x44, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_23 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x32, 0x33, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ23 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x33,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq24 is not equal to max_blks_vq - XYLD R */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x34,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x58, 0x59, 0x4c, 0x44, 0x20,
	0x52, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_24 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x32, 0x34, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ24 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x34,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq26 is not equal to max_blks_vq - PTU_REG_VQID_CFG */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x36,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x50, 0x54, 0x55, 0x5f, 0x52,
	0x45, 0x47, 0x5f, 0x56, 0x51, 0x49, 0x44, 0x5f, 0x43, 0x46, 0x47, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_26 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x32, 0x36, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ26 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x36,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq28 is not equal to max_blks_vq - DMAE RW */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x38,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x44, 0x4d, 0x41, 0x45, 0x20,
	0x52, 0x57, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_28 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x32, 0x38, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ28 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x38,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq29 is not equal to max_blks_vq - CDU_WRITE */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x32, 0x39,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x43, 0x44, 0x55, 0x5f, 0x57,
	0x52, 0x49, 0x54, 0x45, 0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_29 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x32, 0x39, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ29 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x32, 0x39,
	0x00,
	/* PSWRQ2: blk_cnt_per_vq30 is not equal to max_blks_vq - DBG_REG_PCI_VQ_ID */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x62, 0x6c, 0x6b, 0x5f,
	0x63, 0x6e, 0x74, 0x5f, 0x70, 0x65, 0x72, 0x5f, 0x76, 0x71, 0x33, 0x30,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x71, 0x75, 0x61,
	0x6c, 0x20, 0x74, 0x6f, 0x20, 0x6d, 0x61, 0x78, 0x5f, 0x62, 0x6c, 0x6b,
	0x73, 0x5f, 0x76, 0x71, 0x20, 0x2d, 0x20, 0x44, 0x42, 0x47, 0x5f, 0x52,
	0x45, 0x47, 0x5f, 0x50, 0x43, 0x49, 0x5f, 0x56, 0x51, 0x5f, 0x49, 0x44,
	0x00,
	/* PSWRQ2_REG_BLK_CNT_PER_VQ_30 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x42,
	0x4c, 0x4b, 0x5f, 0x43, 0x4e, 0x54, 0x5f, 0x50, 0x45, 0x52, 0x5f, 0x56,
	0x51, 0x5f, 0x33, 0x30, 0x00,
	/* PSWRQ2_REG_MAX_BLKS_VQ30 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x41, 0x58, 0x5f, 0x42, 0x4c, 0x4b, 0x53, 0x5f, 0x56, 0x51, 0x33, 0x30,
	0x00,
	/* PSWRQ2: L2P close the gate is asserted */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x4c, 0x32, 0x50, 0x20,
	0x63, 0x6c, 0x6f, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x67, 0x61,
	0x74, 0x65, 0x20, 0x69, 0x73, 0x20, 0x61, 0x73, 0x73, 0x65, 0x72, 0x74,
	0x65, 0x64, 0x00,
	/* PSWRQ2_REG_L2P_CLOSE_GATE_STS */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4c,
	0x32, 0x50, 0x5f, 0x43, 0x4c, 0x4f, 0x53, 0x45, 0x5f, 0x47, 0x41, 0x54,
	0x45, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWRQ2: MISC close the gate is asserted */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x4d, 0x49, 0x53, 0x43,
	0x20, 0x63, 0x6c, 0x6f, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x67,
	0x61, 0x74, 0x65, 0x20, 0x69, 0x73, 0x20, 0x61, 0x73, 0x73, 0x65, 0x72,
	0x74, 0x65, 0x64, 0x00,
	/* PSWRQ2_REG_MISC_CLOSE_GATE_STS */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x49, 0x53, 0x43, 0x5f, 0x43, 0x4c, 0x4f, 0x53, 0x45, 0x5f, 0x47, 0x41,
	0x54, 0x45, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWRQ2: MISC stall mem is asserted */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x3a, 0x20, 0x4d, 0x49, 0x53, 0x43,
	0x20, 0x73, 0x74, 0x61, 0x6c, 0x6c, 0x20, 0x6d, 0x65, 0x6d, 0x20, 0x69,
	0x73, 0x20, 0x61, 0x73, 0x73, 0x65, 0x72, 0x74, 0x65, 0x64, 0x00,
	/* PSWRQ2_REG_MISC_STALL_MEM_STS */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x4d,
	0x49, 0x53, 0x43, 0x5f, 0x53, 0x54, 0x41, 0x4c, 0x4c, 0x5f, 0x4d, 0x45,
	0x4d, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWRQ: Interrupt status is not 0 */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x3a, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x72,
	0x72, 0x75, 0x70, 0x74, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWRQ_REG_INT_STS */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e,
	0x54, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWRQ_REG_INT_MASK */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e,
	0x54, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PSWRQ: parity error datapath registers */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x3a, 0x20, 0x70, 0x61, 0x72, 0x69, 0x74,
	0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x64, 0x61, 0x74, 0x61,
	0x70, 0x61, 0x74, 0x68, 0x20, 0x72, 0x65, 0x67, 0x69, 0x73, 0x74, 0x65,
	0x72, 0x73, 0x00,
	/* PSWRQ_REG_PRTY_STS */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52,
	0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWRQ_REG_PRTY_MASK */
	0x50, 0x53, 0x57, 0x52, 0x51, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52,
	0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PXPREQBUS_REG_PRTY_STS_H_0 */
	0x50, 0x58, 0x50, 0x52, 0x45, 0x51, 0x42, 0x55, 0x53, 0x5f, 0x52, 0x45,
	0x47, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48,
	0x5f, 0x30, 0x00,
	/* PXPREQBUS_REG_PRTY_MASK_H_0 */
	0x50, 0x58, 0x50, 0x52, 0x45, 0x51, 0x42, 0x55, 0x53, 0x5f, 0x52, 0x45,
	0x47, 0x5f, 0x50, 0x52, 0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f,
	0x48, 0x5f, 0x30, 0x00,
	/* PSWWR: Interrupt status is not 0 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x3a, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x72,
	0x72, 0x75, 0x70, 0x74, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWWR_REG_INT_STS */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e,
	0x54, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWWR_REG_INT_MASK */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e,
	0x54, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PSWWR: parity error datapath registers */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x3a, 0x20, 0x70, 0x61, 0x72, 0x69, 0x74,
	0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x64, 0x61, 0x74, 0x61,
	0x70, 0x61, 0x74, 0x68, 0x20, 0x72, 0x65, 0x67, 0x69, 0x73, 0x74, 0x65,
	0x72, 0x73, 0x00,
	/* PSWWR_REG_PRTY_STS */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52,
	0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWWR_REG_PRTY_STS */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52,
	0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWWR: Details of first request that triggered any of the 3 EOP interrupts: [4:0] - client ID. [7:5] - (sum1[5:3] + 1) or (sum1[5:4] + 1) according to the definition in the spec. [10:8] - number_of_valid_64bit_words[2:0] or number_of_valid_128bit_words[1:0] according to the definition in the spec. [13:11] - The type of interrupt the logging corresponds to: [11] - pglue_eop_error; [12] - pglue_lsr_error; [13] - pglue_eop_error_in_line. [14] - valid - indicates if there was a request that triggered EOP interrupt since this register was cleared. */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x3a, 0x20, 0x44, 0x65, 0x74, 0x61, 0x69,
	0x6c, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x66, 0x69, 0x72, 0x73, 0x74, 0x20,
	0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x20, 0x74, 0x68, 0x61, 0x74,
	0x20, 0x74, 0x72, 0x69, 0x67, 0x67, 0x65, 0x72, 0x65, 0x64, 0x20, 0x61,
	0x6e, 0x79, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x33, 0x20,
	0x45, 0x4f, 0x50, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x72, 0x75, 0x70,
	0x74, 0x73, 0x3a, 0x20, 0x5b, 0x34, 0x3a, 0x30, 0x5d, 0x20, 0x2d, 0x20,
	0x63, 0x6c, 0x69, 0x65, 0x6e, 0x74, 0x20, 0x49, 0x44, 0x2e, 0x20, 0x5b,
	0x37, 0x3a, 0x35, 0x5d, 0x20, 0x2d, 0x20, 0x28, 0x73, 0x75, 0x6d, 0x31,
	0x5b, 0x35, 0x3a, 0x33, 0x5d, 0x20, 0x2b, 0x20, 0x31, 0x29, 0x20, 0x6f,
	0x72, 0x20, 0x28, 0x73, 0x75, 0x6d, 0x31, 0x5b, 0x35, 0x3a, 0x34, 0x5d,
	0x20, 0x2b, 0x20, 0x31, 0x29, 0x20, 0x61, 0x63, 0x63, 0x6f, 0x72, 0x64,
	0x69, 0x6e, 0x67, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64,
	0x65, 0x66, 0x69, 0x6e, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x6e,
	0x20, 0x74, 0x68, 0x65, 0x20, 0x73, 0x70, 0x65, 0x63, 0x2e, 0x20, 0x5b,
	0x31, 0x30, 0x3a, 0x38, 0x5d, 0x20, 0x2d, 0x20, 0x6e, 0x75, 0x6d, 0x62,
	0x65, 0x72, 0x5f, 0x6f, 0x66, 0x5f, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x5f,
	0x36, 0x34, 0x62, 0x69, 0x74, 0x5f, 0x77, 0x6f, 0x72, 0x64, 0x73, 0x5b,
	0x32, 0x3a, 0x30, 0x5d, 0x20, 0x6f, 0x72, 0x20, 0x6e, 0x75, 0x6d, 0x62,
	0x65, 0x72, 0x5f, 0x6f, 0x66, 0x5f, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x5f,
	0x31, 0x32, 0x38, 0x62, 0x69, 0x74, 0x5f, 0x77, 0x6f, 0x72, 0x64, 0x73,
	0x5b, 0x31, 0x3a, 0x30, 0x5d, 0x20, 0x61, 0x63, 0x63, 0x6f, 0x72, 0x64,
	0x69, 0x6e, 0x67, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x64,
	0x65, 0x66, 0x69, 0x6e, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x6e,
	0x20, 0x74, 0x68, 0x65, 0x20, 0x73, 0x70, 0x65, 0x63, 0x2e, 0x20, 0x5b,
	0x31, 0x33, 0x3a, 0x31, 0x31, 0x5d, 0x20, 0x2d, 0x20, 0x54, 0x68, 0x65,
	0x20, 0x74, 0x79, 0x70, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x69, 0x6e, 0x74,
	0x65, 0x72, 0x72, 0x75, 0x70, 0x74, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6c,
	0x6f, 0x67, 0x67, 0x69, 0x6e, 0x67, 0x20, 0x63, 0x6f, 0x72, 0x72, 0x65,
	0x73, 0x70, 0x6f, 0x6e, 0x64, 0x73, 0x20, 0x74, 0x6f, 0x3a, 0x20, 0x5b,
	0x31, 0x31, 0x5d, 0x20, 0x2d, 0x20, 0x70, 0x67, 0x6c, 0x75, 0x65, 0x5f,
	0x65, 0x6f, 0x70, 0x5f, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x3b, 0x20, 0x5b,
	0x31, 0x32, 0x5d, 0x20, 0x2d, 0x20, 0x70, 0x67, 0x6c, 0x75, 0x65, 0x5f,
	0x6c, 0x73, 0x72, 0x5f, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x3b, 0x20, 0x5b,
	0x31, 0x33, 0x5d, 0x20, 0x2d, 0x20, 0x70, 0x67, 0x6c, 0x75, 0x65, 0x5f,
	0x65, 0x6f, 0x70, 0x5f, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x5f, 0x69, 0x6e,
	0x5f, 0x6c, 0x69, 0x6e, 0x65, 0x2e, 0x20, 0x5b, 0x31, 0x34, 0x5d, 0x20,
	0x2d, 0x20, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20, 0x2d, 0x20, 0x69, 0x6e,
	0x64, 0x69, 0x63, 0x61, 0x74, 0x65, 0x73, 0x20, 0x69, 0x66, 0x20, 0x74,
	0x68, 0x65, 0x72, 0x65, 0x20, 0x77, 0x61, 0x73, 0x20, 0x61, 0x20, 0x72,
	0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20,
	0x74, 0x72, 0x69, 0x67, 0x67, 0x65, 0x72, 0x65, 0x64, 0x20, 0x45, 0x4f,
	0x50, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x72, 0x75, 0x70, 0x74, 0x20,
	0x73, 0x69, 0x6e, 0x63, 0x65, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x72,
	0x65, 0x67, 0x69, 0x73, 0x74, 0x65, 0x72, 0x20, 0x77, 0x61, 0x73, 0x20,
	0x63, 0x6c, 0x65, 0x61, 0x72, 0x65, 0x64, 0x2e, 0x00,
	/* PSWWR2_REG_PGLUE_EOP_ERR_DETAILS */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x47, 0x4c, 0x55, 0x45, 0x5f, 0x45, 0x4f, 0x50, 0x5f, 0x45, 0x52, 0x52,
	0x5f, 0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x53, 0x00,
	/* PSWWR: PRM payload FIFO is not empty. */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x3a, 0x20, 0x50, 0x52, 0x4d, 0x20, 0x70,
	0x61, 0x79, 0x6c, 0x6f, 0x61, 0x64, 0x20, 0x46, 0x49, 0x46, 0x4f, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79,
	0x2e, 0x00,
	/* PSWWR2_REG_PRM_CURR_FILL_LEVEL */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x4d, 0x5f, 0x43, 0x55, 0x52, 0x52, 0x5f, 0x46, 0x49, 0x4c, 0x4c,
	0x5f, 0x4c, 0x45, 0x56, 0x45, 0x4c, 0x00,
	/* PSWWR: CDU payload FIFO is not empty. */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x3a, 0x20, 0x43, 0x44, 0x55, 0x20, 0x70,
	0x61, 0x79, 0x6c, 0x6f, 0x61, 0x64, 0x20, 0x46, 0x49, 0x46, 0x4f, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x65, 0x6d, 0x70, 0x74, 0x79,
	0x2e, 0x00,
	/* PSWWR2_REG_CDU_CURR_FILL_LEVEL */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x43,
	0x44, 0x55, 0x5f, 0x43, 0x55, 0x52, 0x52, 0x5f, 0x46, 0x49, 0x4c, 0x4c,
	0x5f, 0x4c, 0x45, 0x56, 0x45, 0x4c, 0x00,
	/* PSWWR2: Interrupt status is not 0 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x3a, 0x20, 0x49, 0x6e, 0x74, 0x65,
	0x72, 0x72, 0x75, 0x70, 0x74, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWWR2_REG_INT_STS */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x4e, 0x54, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWWR2_REG_INT_MASK */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x4e, 0x54, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PSWWR2: parity error datapath registers */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x3a, 0x20, 0x70, 0x61, 0x72, 0x69,
	0x74, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x64, 0x61, 0x74,
	0x61, 0x70, 0x61, 0x74, 0x68, 0x20, 0x72, 0x65, 0x67, 0x69, 0x73, 0x74,
	0x65, 0x72, 0x73, 0x00,
	/* PSWWR2_REG_PRTY_STS */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWWR2_REG_PRTY_STS */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWWR2: Parity status 0 is not 0 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69,
	0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x30, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWWR2_REG_PRTY_STS_H_0 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* PSWWR2_REG_PRTY_MASK_H_0 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30,
	0x00,
	/* PSWWR2: Parity status 1 is not 0 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69,
	0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x31, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWWR2_REG_PRTY_STS_H_1 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x31, 0x00,
	/* PSWWR2_REG_PRTY_MASK_H_1 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x31,
	0x00,
	/* PSWWR2: Parity status 2 is not 0 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69,
	0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x32, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWWR2_REG_PRTY_STS_H_2 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x32, 0x00,
	/* PSWWR2_REG_PRTY_MASK_H_2 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x32,
	0x00,
	/* PSWWR2: Parity status 3 is not 0 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69,
	0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x33, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWWR2_REG_PRTY_STS_H_3 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x33, 0x00,
	/* PSWWR2_REG_PRTY_MASK_H_3 */
	0x50, 0x53, 0x57, 0x57, 0x52, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x33,
	0x00,
	/* PSWRD: Full is asserted from at least one client. */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x3a, 0x20, 0x46, 0x75, 0x6c, 0x6c, 0x20,
	0x69, 0x73, 0x20, 0x61, 0x73, 0x73, 0x65, 0x72, 0x74, 0x65, 0x64, 0x20,
	0x66, 0x72, 0x6f, 0x6d, 0x20, 0x61, 0x74, 0x20, 0x6c, 0x65, 0x61, 0x73,
	0x74, 0x20, 0x6f, 0x6e, 0x65, 0x20, 0x63, 0x6c, 0x69, 0x65, 0x6e, 0x74,
	0x2e, 0x00,
	/* PSWRD_REG_FIFO_FULL_STATUS */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x46, 0x49,
	0x46, 0x4f, 0x5f, 0x46, 0x55, 0x4c, 0x4c, 0x5f, 0x53, 0x54, 0x41, 0x54,
	0x55, 0x53, 0x00,
	/* PSWRD: Interrupt status is not 0 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x3a, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x72,
	0x72, 0x75, 0x70, 0x74, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWRD_REG_INT_STS */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e,
	0x54, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWRD_REG_INT_MASK */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49, 0x4e,
	0x54, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PSWRD: parity error datapath registers */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x3a, 0x20, 0x70, 0x61, 0x72, 0x69, 0x74,
	0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x64, 0x61, 0x74, 0x61,
	0x70, 0x61, 0x74, 0x68, 0x20, 0x72, 0x65, 0x67, 0x69, 0x73, 0x74, 0x65,
	0x72, 0x73, 0x00,
	/* PSWRD_REG_PRTY_STS */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52,
	0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWRD_REG_PRTY_MASK */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50, 0x52,
	0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PSWRD2: rd_start_init is not 1 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x3a, 0x20, 0x72, 0x64, 0x5f, 0x73,
	0x74, 0x61, 0x72, 0x74, 0x5f, 0x69, 0x6e, 0x69, 0x74, 0x20, 0x69, 0x73,
	0x20, 0x6e, 0x6f, 0x74, 0x20, 0x31, 0x00,
	/* PSWRD2 Block is not initialized */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x20, 0x42, 0x6c, 0x6f, 0x63, 0x6b,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x69, 0x74,
	0x69, 0x61, 0x6c, 0x69, 0x7a, 0x65, 0x64, 0x00,
	/* PSWRD2_REG_START_INIT */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x54, 0x41, 0x52, 0x54, 0x5f, 0x49, 0x4e, 0x49, 0x54, 0x00,
	/* PSWRD2: rd_init_done is not 1 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x3a, 0x20, 0x72, 0x64, 0x5f, 0x69,
	0x6e, 0x69, 0x74, 0x5f, 0x64, 0x6f, 0x6e, 0x65, 0x20, 0x69, 0x73, 0x20,
	0x6e, 0x6f, 0x74, 0x20, 0x31, 0x00,
	/* PSWRD2: Block is not initialized */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x3a, 0x20, 0x42, 0x6c, 0x6f, 0x63,
	0x6b, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x69, 0x6e, 0x69,
	0x74, 0x69, 0x61, 0x6c, 0x69, 0x7a, 0x65, 0x64, 0x00,
	/* PSWRD2_REG_INIT_DONE */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x4e, 0x49, 0x54, 0x5f, 0x44, 0x4f, 0x4e, 0x45, 0x00,
	/* PSWRD2: Completion received with error. Error details register is not 0. (15:0) - ECHO. (28:16) - Sub Request length minus 1. (29) - first SR. (30) - last SR. */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x3a, 0x20, 0x43, 0x6f, 0x6d, 0x70,
	0x6c, 0x65, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x72, 0x65, 0x63, 0x65, 0x69,
	0x76, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x65, 0x72, 0x72,
	0x6f, 0x72, 0x2e, 0x20, 0x45, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x64, 0x65,
	0x74, 0x61, 0x69, 0x6c, 0x73, 0x20, 0x72, 0x65, 0x67, 0x69, 0x73, 0x74,
	0x65, 0x72, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x2e,
	0x20, 0x28, 0x31, 0x35, 0x3a, 0x30, 0x29, 0x20, 0x2d, 0x20, 0x45, 0x43,
	0x48, 0x4f, 0x2e, 0x20, 0x28, 0x32, 0x38, 0x3a, 0x31, 0x36, 0x29, 0x20,
	0x2d, 0x20, 0x53, 0x75, 0x62, 0x20, 0x52, 0x65, 0x71, 0x75, 0x65, 0x73,
	0x74, 0x20, 0x6c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x20, 0x6d, 0x69, 0x6e,
	0x75, 0x73, 0x20, 0x31, 0x2e, 0x20, 0x28, 0x32, 0x39, 0x29, 0x20, 0x2d,
	0x20, 0x66, 0x69, 0x72, 0x73, 0x74, 0x20, 0x53, 0x52, 0x2e, 0x20, 0x28,
	0x33, 0x30, 0x29, 0x20, 0x2d, 0x20, 0x6c, 0x61, 0x73, 0x74, 0x20, 0x53,
	0x52, 0x2e, 0x00,
	/* PSWRD2_REG_CPL_ERR_DETAILS */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x43,
	0x50, 0x4c, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49,
	0x4c, 0x53, 0x00,
	/* PSWRD2: Completion received with error. Error details 2nd register is not 0. (4:0) - VQ ID. (9:5) - client ID. (10) - valid bit. */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x3a, 0x20, 0x43, 0x6f, 0x6d, 0x70,
	0x6c, 0x65, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x72, 0x65, 0x63, 0x65, 0x69,
	0x76, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x65, 0x72, 0x72,
	0x6f, 0x72, 0x2e, 0x20, 0x45, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x64, 0x65,
	0x74, 0x61, 0x69, 0x6c, 0x73, 0x20, 0x32, 0x6e, 0x64, 0x20, 0x72, 0x65,
	0x67, 0x69, 0x73, 0x74, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x30, 0x2e, 0x20, 0x28, 0x34, 0x3a, 0x30, 0x29, 0x20, 0x2d,
	0x20, 0x56, 0x51, 0x20, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x39, 0x3a, 0x35,
	0x29, 0x20, 0x2d, 0x20, 0x63, 0x6c, 0x69, 0x65, 0x6e, 0x74, 0x20, 0x49,
	0x44, 0x2e, 0x20, 0x28, 0x31, 0x30, 0x29, 0x20, 0x2d, 0x20, 0x76, 0x61,
	0x6c, 0x69, 0x64, 0x20, 0x62, 0x69, 0x74, 0x2e, 0x00,
	/* PSWRD2_REG_CPL_ERR_DETAILS2 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x43,
	0x50, 0x4c, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49,
	0x4c, 0x53, 0x32, 0x00,
	/* PSWRD2: P0 All delivery ports are not idle */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x3a, 0x20, 0x50, 0x30, 0x20, 0x41,
	0x6c, 0x6c, 0x20, 0x64, 0x65, 0x6c, 0x69, 0x76, 0x65, 0x72, 0x79, 0x20,
	0x70, 0x6f, 0x72, 0x74, 0x73, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x69, 0x64, 0x6c, 0x65, 0x00,
	/* PSWRD2_REG_PORT_IS_IDLE_0 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x4f, 0x52, 0x54, 0x5f, 0x49, 0x53, 0x5f, 0x49, 0x44, 0x4c, 0x45, 0x5f,
	0x30, 0x00,
	/* PSWRD2: P1 All delivery ports are not idle */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x3a, 0x20, 0x50, 0x31, 0x20, 0x41,
	0x6c, 0x6c, 0x20, 0x64, 0x65, 0x6c, 0x69, 0x76, 0x65, 0x72, 0x79, 0x20,
	0x70, 0x6f, 0x72, 0x74, 0x73, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6e, 0x6f,
	0x74, 0x20, 0x69, 0x64, 0x6c, 0x65, 0x00,
	/* PSWRD2_REG_PORT_IS_IDLE_1 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x4f, 0x52, 0x54, 0x5f, 0x49, 0x53, 0x5f, 0x49, 0x44, 0x4c, 0x45, 0x5f,
	0x31, 0x00,
	/* PSWRD2: rd_almost_full is not 0 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x3a, 0x20, 0x72, 0x64, 0x5f, 0x61,
	0x6c, 0x6d, 0x6f, 0x73, 0x74, 0x5f, 0x66, 0x75, 0x6c, 0x6c, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWRD2_REG_ALMOST_FULL_0 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41,
	0x4c, 0x4d, 0x4f, 0x53, 0x54, 0x5f, 0x46, 0x55, 0x4c, 0x4c, 0x5f, 0x30,
	0x00,
	/* PSWRD2: Interrupt status is not 0 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x3a, 0x20, 0x49, 0x6e, 0x74, 0x65,
	0x72, 0x72, 0x75, 0x70, 0x74, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWRD2_REG_INT_STS */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x4e, 0x54, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWRD2_REG_INT_MASK */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x4e, 0x54, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PSWRD2: parity error datapath registers */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x3a, 0x20, 0x70, 0x61, 0x72, 0x69,
	0x74, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x64, 0x61, 0x74,
	0x61, 0x70, 0x61, 0x74, 0x68, 0x20, 0x72, 0x65, 0x67, 0x69, 0x73, 0x74,
	0x65, 0x72, 0x73, 0x00,
	/* PSWRD2_REG_PRTY_STS */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWRD2_REG_PRTY_MASK */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PSWRD2: Parity status 0 is not 0 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69,
	0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x30, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWRD2_REG_PRTY_STS_H_0 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* PSWRD2_REG_PRTY_MASK_H_0 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30,
	0x00,
	/* PSWRD2: Parity status 1 is not 0 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69,
	0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x31, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWRD2_REG_PRTY_STS_H_1 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x31, 0x00,
	/* PSWRD2_REG_PRTY_MASK_H_1 */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x31,
	0x00,
	/* PSWRD2: PSWRD inputs are disabled */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x3a, 0x20, 0x50, 0x53, 0x57, 0x52,
	0x44, 0x20, 0x69, 0x6e, 0x70, 0x75, 0x74, 0x73, 0x20, 0x61, 0x72, 0x65,
	0x20, 0x64, 0x69, 0x73, 0x61, 0x62, 0x6c, 0x65, 0x64, 0x00,
	/* PSWRD2_REG_DISABLE_INPUTS */
	0x50, 0x53, 0x57, 0x52, 0x44, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44,
	0x49, 0x53, 0x41, 0x42, 0x4c, 0x45, 0x5f, 0x49, 0x4e, 0x50, 0x55, 0x54,
	0x53, 0x00,
	/* PSWHST2: HST header FIFO status is not 0 */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x32, 0x3a, 0x20, 0x48, 0x53, 0x54,
	0x20, 0x68, 0x65, 0x61, 0x64, 0x65, 0x72, 0x20, 0x46, 0x49, 0x46, 0x4f,
	0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWHST2_REG_HEADER_FIFO_STATUS */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x48, 0x45, 0x41, 0x44, 0x45, 0x52, 0x5f, 0x46, 0x49, 0x46, 0x4f, 0x5f,
	0x53, 0x54, 0x41, 0x54, 0x55, 0x53, 0x00,
	/* PSWHST2: HST data FIFO status is not 0 */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x32, 0x3a, 0x20, 0x48, 0x53, 0x54,
	0x20, 0x64, 0x61, 0x74, 0x61, 0x20, 0x46, 0x49, 0x46, 0x4f, 0x20, 0x73,
	0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74,
	0x20, 0x30, 0x00,
	/* PSWHST2_REG_DATA_FIFO_STATUS */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x44, 0x41, 0x54, 0x41, 0x5f, 0x46, 0x49, 0x46, 0x4f, 0x5f, 0x53, 0x54,
	0x41, 0x54, 0x55, 0x53, 0x00,
	/* PSWHST2: Interrupt status is not 0 */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x32, 0x3a, 0x20, 0x49, 0x6e, 0x74,
	0x65, 0x72, 0x72, 0x75, 0x70, 0x74, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75,
	0x73, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWHST2_REG_INT_STS */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x49, 0x4e, 0x54, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWHST2_REG_INT_MASK */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x32, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x49, 0x4e, 0x54, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PSWHST: HST Close the gates: Discarding internal writes */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x48, 0x53, 0x54, 0x20,
	0x43, 0x6c, 0x6f, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x67, 0x61,
	0x74, 0x65, 0x73, 0x3a, 0x20, 0x44, 0x69, 0x73, 0x63, 0x61, 0x72, 0x64,
	0x69, 0x6e, 0x67, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x61, 0x6c,
	0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x73, 0x00,
	/* PSWHST_REG_DISCARD_INTERNAL_WRITES_STATUS */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44,
	0x49, 0x53, 0x43, 0x41, 0x52, 0x44, 0x5f, 0x49, 0x4e, 0x54, 0x45, 0x52,
	0x4e, 0x41, 0x4c, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x53, 0x5f, 0x53,
	0x54, 0x41, 0x54, 0x55, 0x53, 0x00,
	/* PSWHST: HST Close the gates: Discarding doorbells */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x48, 0x53, 0x54, 0x20,
	0x43, 0x6c, 0x6f, 0x73, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x67, 0x61,
	0x74, 0x65, 0x73, 0x3a, 0x20, 0x44, 0x69, 0x73, 0x63, 0x61, 0x72, 0x64,
	0x69, 0x6e, 0x67, 0x20, 0x64, 0x6f, 0x6f, 0x72, 0x62, 0x65, 0x6c, 0x6c,
	0x73, 0x00,
	/* PSWHST_REG_DISCARD_DOORBELLS_STATUS */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44,
	0x49, 0x53, 0x43, 0x41, 0x52, 0x44, 0x5f, 0x44, 0x4f, 0x4f, 0x52, 0x42,
	0x45, 0x4c, 0x4c, 0x53, 0x5f, 0x53, 0x54, 0x41, 0x54, 0x55, 0x53, 0x00,
	/* PSWHST: HST arbiter is not idle */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x48, 0x53, 0x54, 0x20,
	0x61, 0x72, 0x62, 0x69, 0x74, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x69, 0x64, 0x6c, 0x65, 0x00,
	/* PSWHST_REG_ARB_IS_IDLE */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x41,
	0x52, 0x42, 0x5f, 0x49, 0x53, 0x5f, 0x49, 0x44, 0x4c, 0x45, 0x00,
	/* PSWHST: Incorrect transaction took place */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x49, 0x6e, 0x63, 0x6f,
	0x72, 0x72, 0x65, 0x63, 0x74, 0x20, 0x74, 0x72, 0x61, 0x6e, 0x73, 0x61,
	0x63, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x74, 0x6f, 0x6f, 0x6b, 0x20, 0x70,
	0x6c, 0x61, 0x63, 0x65, 0x00,
	/* PSWHST_REG_INCORRECT_ACCESS_VALID */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x4e, 0x43, 0x4f, 0x52, 0x52, 0x45, 0x43, 0x54, 0x5f, 0x41, 0x43, 0x43,
	0x45, 0x53, 0x53, 0x5f, 0x56, 0x41, 0x4c, 0x49, 0x44, 0x00,
	/* PSWHST_REG_INCORRECT_ACCESS_DATA */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x4e, 0x43, 0x4f, 0x52, 0x52, 0x45, 0x43, 0x54, 0x5f, 0x41, 0x43, 0x43,
	0x45, 0x53, 0x53, 0x5f, 0x44, 0x41, 0x54, 0x41, 0x00,
	/* PSWHST_REG_INCORRECT_ACCESS_LENGTH */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x4e, 0x43, 0x4f, 0x52, 0x52, 0x45, 0x43, 0x54, 0x5f, 0x41, 0x43, 0x43,
	0x45, 0x53, 0x53, 0x5f, 0x4c, 0x45, 0x4e, 0x47, 0x54, 0x48, 0x00,
	/* PSWHST_REG_INCORRECT_ACCESS_ADDRESS */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x4e, 0x43, 0x4f, 0x52, 0x52, 0x45, 0x43, 0x54, 0x5f, 0x41, 0x43, 0x43,
	0x45, 0x53, 0x53, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x45, 0x53, 0x53, 0x00,
	/* PSWHST: Zone A permission violation occurred */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x5a, 0x6f, 0x6e, 0x65,
	0x20, 0x41, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x73, 0x73, 0x69, 0x6f,
	0x6e, 0x20, 0x76, 0x69, 0x6f, 0x6c, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
	0x6f, 0x63, 0x63, 0x75, 0x72, 0x72, 0x65, 0x64, 0x00,
	/* PSWHST_REG_PER_VIOLATION_VALID */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x45, 0x52, 0x5f, 0x56, 0x49, 0x4f, 0x4c, 0x41, 0x54, 0x49, 0x4f, 0x4e,
	0x5f, 0x56, 0x41, 0x4c, 0x49, 0x44, 0x00,
	/* PSWHST_REG_PER_VIOLATION_DATA */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x45, 0x52, 0x5f, 0x56, 0x49, 0x4f, 0x4c, 0x41, 0x54, 0x49, 0x4f, 0x4e,
	0x5f, 0x44, 0x41, 0x54, 0x41, 0x00,
	/* PSWHST: HST: not all credits available from internal write sources. (1:0) usdm; (3:2) xsdm; (5:4) msdm; (7:6) ysdm; (9:8) psdm; (11:10) tsdm; (13:12) pbf; (16:14) qm; (18:17) nig */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x48, 0x53, 0x54, 0x3a,
	0x20, 0x6e, 0x6f, 0x74, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x72, 0x65,
	0x64, 0x69, 0x74, 0x73, 0x20, 0x61, 0x76, 0x61, 0x69, 0x6c, 0x61, 0x62,
	0x6c, 0x65, 0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x69, 0x6e, 0x74, 0x65,
	0x72, 0x6e, 0x61, 0x6c, 0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x20, 0x73,
	0x6f, 0x75, 0x72, 0x63, 0x65, 0x73, 0x2e, 0x20, 0x28, 0x31, 0x3a, 0x30,
	0x29, 0x20, 0x75, 0x73, 0x64, 0x6d, 0x3b, 0x20, 0x28, 0x33, 0x3a, 0x32,
	0x29, 0x20, 0x78, 0x73, 0x64, 0x6d, 0x3b, 0x20, 0x28, 0x35, 0x3a, 0x34,
	0x29, 0x20, 0x6d, 0x73, 0x64, 0x6d, 0x3b, 0x20, 0x28, 0x37, 0x3a, 0x36,
	0x29, 0x20, 0x79, 0x73, 0x64, 0x6d, 0x3b, 0x20, 0x28, 0x39, 0x3a, 0x38,
	0x29, 0x20, 0x70, 0x73, 0x64, 0x6d, 0x3b, 0x20, 0x28, 0x31, 0x31, 0x3a,
	0x31, 0x30, 0x29, 0x20, 0x74, 0x73, 0x64, 0x6d, 0x3b, 0x20, 0x28, 0x31,
	0x33, 0x3a, 0x31, 0x32, 0x29, 0x20, 0x70, 0x62, 0x66, 0x3b, 0x20, 0x28,
	0x31, 0x36, 0x3a, 0x31, 0x34, 0x29, 0x20, 0x71, 0x6d, 0x3b, 0x20, 0x28,
	0x31, 0x38, 0x3a, 0x31, 0x37, 0x29, 0x20, 0x6e, 0x69, 0x67, 0x00,
	/* PSWHST_REG_SOURCE_CREDITS_AVAIL */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x4f, 0x55, 0x52, 0x43, 0x45, 0x5f, 0x43, 0x52, 0x45, 0x44, 0x49, 0x54,
	0x53, 0x5f, 0x41, 0x56, 0x41, 0x49, 0x4c, 0x00,
	/* PSWHST: HST: Internal write source client consumed more credits than it is allowed. 0 TSDM; 1 MSDM; 2 USDM; 3 XSDM; 4 YSDM; 5 PSDM; 6 TSDM; 7 PBF; 8 QM; 9 NIG. */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x48, 0x53, 0x54, 0x3a,
	0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x61, 0x6c, 0x20, 0x77, 0x72,
	0x69, 0x74, 0x65, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x63,
	0x6c, 0x69, 0x65, 0x6e, 0x74, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x75, 0x6d,
	0x65, 0x64, 0x20, 0x6d, 0x6f, 0x72, 0x65, 0x20, 0x63, 0x72, 0x65, 0x64,
	0x69, 0x74, 0x73, 0x20, 0x74, 0x68, 0x61, 0x6e, 0x20, 0x69, 0x74, 0x20,
	0x69, 0x73, 0x20, 0x61, 0x6c, 0x6c, 0x6f, 0x77, 0x65, 0x64, 0x2e, 0x20,
	0x30, 0x20, 0x54, 0x53, 0x44, 0x4d, 0x3b, 0x20, 0x31, 0x20, 0x4d, 0x53,
	0x44, 0x4d, 0x3b, 0x20, 0x32, 0x20, 0x55, 0x53, 0x44, 0x4d, 0x3b, 0x20,
	0x33, 0x20, 0x58, 0x53, 0x44, 0x4d, 0x3b, 0x20, 0x34, 0x20, 0x59, 0x53,
	0x44, 0x4d, 0x3b, 0x20, 0x35, 0x20, 0x50, 0x53, 0x44, 0x4d, 0x3b, 0x20,
	0x36, 0x20, 0x54, 0x53, 0x44, 0x4d, 0x3b, 0x20, 0x37, 0x20, 0x50, 0x42,
	0x46, 0x3b, 0x20, 0x38, 0x20, 0x51, 0x4d, 0x3b, 0x20, 0x39, 0x20, 0x4e,
	0x49, 0x47, 0x2e, 0x00,
	/* PSWHST_REG_SOURCE_CREDIT_VIOL_VALID */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x4f, 0x55, 0x52, 0x43, 0x45, 0x5f, 0x43, 0x52, 0x45, 0x44, 0x49, 0x54,
	0x5f, 0x56, 0x49, 0x4f, 0x4c, 0x5f, 0x56, 0x41, 0x4c, 0x49, 0x44, 0x00,
	/* PSWHST_REG_SOURCE_CREDIT_VIOL_DATA */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x53,
	0x4f, 0x55, 0x52, 0x43, 0x45, 0x5f, 0x43, 0x52, 0x45, 0x44, 0x49, 0x54,
	0x5f, 0x56, 0x49, 0x4f, 0x4c, 0x5f, 0x44, 0x41, 0x54, 0x41, 0x00,
	/* PSWHST: HST: not all credits available to internal write destinations. (1:0) usdm; (3:2) xsdm; (5:4) msdm; (7:6) ysdm; (9:8) psdm; (11:10) tsdm; (13:12) igu; (15:14) cau; (17:16) dorq */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x48, 0x53, 0x54, 0x3a,
	0x20, 0x6e, 0x6f, 0x74, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x72, 0x65,
	0x64, 0x69, 0x74, 0x73, 0x20, 0x61, 0x76, 0x61, 0x69, 0x6c, 0x61, 0x62,
	0x6c, 0x65, 0x20, 0x74, 0x6f, 0x20, 0x69, 0x6e, 0x74, 0x65, 0x72, 0x6e,
	0x61, 0x6c, 0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x20, 0x64, 0x65, 0x73,
	0x74, 0x69, 0x6e, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x2e, 0x20, 0x28,
	0x31, 0x3a, 0x30, 0x29, 0x20, 0x75, 0x73, 0x64, 0x6d, 0x3b, 0x20, 0x28,
	0x33, 0x3a, 0x32, 0x29, 0x20, 0x78, 0x73, 0x64, 0x6d, 0x3b, 0x20, 0x28,
	0x35, 0x3a, 0x34, 0x29, 0x20, 0x6d, 0x73, 0x64, 0x6d, 0x3b, 0x20, 0x28,
	0x37, 0x3a, 0x36, 0x29, 0x20, 0x79, 0x73, 0x64, 0x6d, 0x3b, 0x20, 0x28,
	0x39, 0x3a, 0x38, 0x29, 0x20, 0x70, 0x73, 0x64, 0x6d, 0x3b, 0x20, 0x28,
	0x31, 0x31, 0x3a, 0x31, 0x30, 0x29, 0x20, 0x74, 0x73, 0x64, 0x6d, 0x3b,
	0x20, 0x28, 0x31, 0x33, 0x3a, 0x31, 0x32, 0x29, 0x20, 0x69, 0x67, 0x75,
	0x3b, 0x20, 0x28, 0x31, 0x35, 0x3a, 0x31, 0x34, 0x29, 0x20, 0x63, 0x61,
	0x75, 0x3b, 0x20, 0x28, 0x31, 0x37, 0x3a, 0x31, 0x36, 0x29, 0x20, 0x64,
	0x6f, 0x72, 0x71, 0x00,
	/* PSWHST_REG_DEST_CREDITS_AVAIL */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x44,
	0x45, 0x53, 0x54, 0x5f, 0x43, 0x52, 0x45, 0x44, 0x49, 0x54, 0x53, 0x5f,
	0x41, 0x56, 0x41, 0x49, 0x4c, 0x00,
	/* PSWHST: HST: Is in drain mode (after target timeout). */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x48, 0x53, 0x54, 0x3a,
	0x20, 0x49, 0x73, 0x20, 0x69, 0x6e, 0x20, 0x64, 0x72, 0x61, 0x69, 0x6e,
	0x20, 0x6d, 0x6f, 0x64, 0x65, 0x20, 0x28, 0x61, 0x66, 0x74, 0x65, 0x72,
	0x20, 0x74, 0x61, 0x72, 0x67, 0x65, 0x74, 0x20, 0x74, 0x69, 0x6d, 0x65,
	0x6f, 0x75, 0x74, 0x29, 0x2e, 0x00,
	/* PSWHST_REG_IS_IN_DRAIN_MODE */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x53, 0x5f, 0x49, 0x4e, 0x5f, 0x44, 0x52, 0x41, 0x49, 0x4e, 0x5f, 0x4d,
	0x4f, 0x44, 0x45, 0x00,
	/* PSWHST: HST: Target Timeout details. (29:24) - length in DWs; (23:16) - byte enable; (15:13) - pfid; (12:6) - vfid; (5) - vf_valid; (4:1) - client (0 TSDM; 1 MSDM; 2 USDM; 3 XSDM; 4 YSDM; 5 PSDM; 6 HC; 7 GRC; 8 DQ; 9 ATC; 10 RESERVED SPACE); (0) - 0 - read 1 - write. */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x48, 0x53, 0x54, 0x3a,
	0x20, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74, 0x20, 0x54, 0x69, 0x6d, 0x65,
	0x6f, 0x75, 0x74, 0x20, 0x64, 0x65, 0x74, 0x61, 0x69, 0x6c, 0x73, 0x2e,
	0x20, 0x28, 0x32, 0x39, 0x3a, 0x32, 0x34, 0x29, 0x20, 0x2d, 0x20, 0x6c,
	0x65, 0x6e, 0x67, 0x74, 0x68, 0x20, 0x69, 0x6e, 0x20, 0x44, 0x57, 0x73,
	0x3b, 0x20, 0x28, 0x32, 0x33, 0x3a, 0x31, 0x36, 0x29, 0x20, 0x2d, 0x20,
	0x62, 0x79, 0x74, 0x65, 0x20, 0x65, 0x6e, 0x61, 0x62, 0x6c, 0x65, 0x3b,
	0x20, 0x28, 0x31, 0x35, 0x3a, 0x31, 0x33, 0x29, 0x20, 0x2d, 0x20, 0x70,
	0x66, 0x69, 0x64, 0x3b, 0x20, 0x28, 0x31, 0x32, 0x3a, 0x36, 0x29, 0x20,
	0x2d, 0x20, 0x76, 0x66, 0x69, 0x64, 0x3b, 0x20, 0x28, 0x35, 0x29, 0x20,
	0x2d, 0x20, 0x76, 0x66, 0x5f, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x3b, 0x20,
	0x28, 0x34, 0x3a, 0x31, 0x29, 0x20, 0x2d, 0x20, 0x63, 0x6c, 0x69, 0x65,
	0x6e, 0x74, 0x20, 0x28, 0x30, 0x20, 0x54, 0x53, 0x44, 0x4d, 0x3b, 0x20,
	0x31, 0x20, 0x4d, 0x53, 0x44, 0x4d, 0x3b, 0x20, 0x32, 0x20, 0x55, 0x53,
	0x44, 0x4d, 0x3b, 0x20, 0x33, 0x20, 0x58, 0x53, 0x44, 0x4d, 0x3b, 0x20,
	0x34, 0x20, 0x59, 0x53, 0x44, 0x4d, 0x3b, 0x20, 0x35, 0x20, 0x50, 0x53,
	0x44, 0x4d, 0x3b, 0x20, 0x36, 0x20, 0x48, 0x43, 0x3b, 0x20, 0x37, 0x20,
	0x47, 0x52, 0x43, 0x3b, 0x20, 0x38, 0x20, 0x44, 0x51, 0x3b, 0x20, 0x39,
	0x20, 0x41, 0x54, 0x43, 0x3b, 0x20, 0x31, 0x30, 0x20, 0x52, 0x45, 0x53,
	0x45, 0x52, 0x56, 0x45, 0x44, 0x20, 0x53, 0x50, 0x41, 0x43, 0x45, 0x29,
	0x3b, 0x20, 0x28, 0x30, 0x29, 0x20, 0x2d, 0x20, 0x30, 0x20, 0x2d, 0x20,
	0x72, 0x65, 0x61, 0x64, 0x20, 0x31, 0x20, 0x2d, 0x20, 0x77, 0x72, 0x69,
	0x74, 0x65, 0x2e, 0x00,
	/* PSWHST_REG_TIMEOUT_VALID */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54,
	0x49, 0x4d, 0x45, 0x4f, 0x55, 0x54, 0x5f, 0x56, 0x41, 0x4c, 0x49, 0x44,
	0x00,
	/* PSWHST_REG_TIMEOUT_DATA */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54,
	0x49, 0x4d, 0x45, 0x4f, 0x55, 0x54, 0x5f, 0x44, 0x41, 0x54, 0x41, 0x00,
	/* PSWHST: HST: Target Timeout address. */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x48, 0x53, 0x54, 0x3a,
	0x20, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74, 0x20, 0x54, 0x69, 0x6d, 0x65,
	0x6f, 0x75, 0x74, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x2e,
	0x00,
	/* PSWHST_REG_TIMEOUT_VALID */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54,
	0x49, 0x4d, 0x45, 0x4f, 0x55, 0x54, 0x5f, 0x56, 0x41, 0x4c, 0x49, 0x44,
	0x00,
	/* PSWHST_REG_TIMEOUT_ADDRESS */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x54,
	0x49, 0x4d, 0x45, 0x4f, 0x55, 0x54, 0x5f, 0x41, 0x44, 0x44, 0x52, 0x45,
	0x53, 0x53, 0x00,
	/* PSWHST: Each entry contains a bit mask for PSWHST source arbiter clients. '1' means this client is waiting for the arbiter. Each entry refers to a different source arbiter. Entry decoding: (0 TSDM; 1 MSDM; 2 USDM; 3 XSDM; 4 YSDM; 5 PSDM; 6 IGU; 7 CAU). Bit mask decoding: (0 TSDM; 1 MSDM; 2 USDM; 3 XSDM; 4 YSDM; 5 PSDM; 6 PBF; 7 QM; 8 NIG; 9 HOST WR; 10 HOST RD) */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x45, 0x61, 0x63, 0x68,
	0x20, 0x65, 0x6e, 0x74, 0x72, 0x79, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x61,
	0x69, 0x6e, 0x73, 0x20, 0x61, 0x20, 0x62, 0x69, 0x74, 0x20, 0x6d, 0x61,
	0x73, 0x6b, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x50, 0x53, 0x57, 0x48, 0x53,
	0x54, 0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x61, 0x72, 0x62,
	0x69, 0x74, 0x65, 0x72, 0x20, 0x63, 0x6c, 0x69, 0x65, 0x6e, 0x74, 0x73,
	0x2e, 0x20, 0x27, 0x31, 0x27, 0x20, 0x6d, 0x65, 0x61, 0x6e, 0x73, 0x20,
	0x74, 0x68, 0x69, 0x73, 0x20, 0x63, 0x6c, 0x69, 0x65, 0x6e, 0x74, 0x20,
	0x69, 0x73, 0x20, 0x77, 0x61, 0x69, 0x74, 0x69, 0x6e, 0x67, 0x20, 0x66,
	0x6f, 0x72, 0x20, 0x74, 0x68, 0x65, 0x20, 0x61, 0x72, 0x62, 0x69, 0x74,
	0x65, 0x72, 0x2e, 0x20, 0x45, 0x61, 0x63, 0x68, 0x20, 0x65, 0x6e, 0x74,
	0x72, 0x79, 0x20, 0x72, 0x65, 0x66, 0x65, 0x72, 0x73, 0x20, 0x74, 0x6f,
	0x20, 0x61, 0x20, 0x64, 0x69, 0x66, 0x66, 0x65, 0x72, 0x65, 0x6e, 0x74,
	0x20, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x20, 0x61, 0x72, 0x62, 0x69,
	0x74, 0x65, 0x72, 0x2e, 0x20, 0x45, 0x6e, 0x74, 0x72, 0x79, 0x20, 0x64,
	0x65, 0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67, 0x3a, 0x20, 0x28, 0x30, 0x20,
	0x54, 0x53, 0x44, 0x4d, 0x3b, 0x20, 0x31, 0x20, 0x4d, 0x53, 0x44, 0x4d,
	0x3b, 0x20, 0x32, 0x20, 0x55, 0x53, 0x44, 0x4d, 0x3b, 0x20, 0x33, 0x20,
	0x58, 0x53, 0x44, 0x4d, 0x3b, 0x20, 0x34, 0x20, 0x59, 0x53, 0x44, 0x4d,
	0x3b, 0x20, 0x35, 0x20, 0x50, 0x53, 0x44, 0x4d, 0x3b, 0x20, 0x36, 0x20,
	0x49, 0x47, 0x55, 0x3b, 0x20, 0x37, 0x20, 0x43, 0x41, 0x55, 0x29, 0x2e,
	0x20, 0x42, 0x69, 0x74, 0x20, 0x6d, 0x61, 0x73, 0x6b, 0x20, 0x64, 0x65,
	0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67, 0x3a, 0x20, 0x28, 0x30, 0x20, 0x54,
	0x53, 0x44, 0x4d, 0x3b, 0x20, 0x31, 0x20, 0x4d, 0x53, 0x44, 0x4d, 0x3b,
	0x20, 0x32, 0x20, 0x55, 0x53, 0x44, 0x4d, 0x3b, 0x20, 0x33, 0x20, 0x58,
	0x53, 0x44, 0x4d, 0x3b, 0x20, 0x34, 0x20, 0x59, 0x53, 0x44, 0x4d, 0x3b,
	0x20, 0x35, 0x20, 0x50, 0x53, 0x44, 0x4d, 0x3b, 0x20, 0x36, 0x20, 0x50,
	0x42, 0x46, 0x3b, 0x20, 0x37, 0x20, 0x51, 0x4d, 0x3b, 0x20, 0x38, 0x20,
	0x4e, 0x49, 0x47, 0x3b, 0x20, 0x39, 0x20, 0x48, 0x4f, 0x53, 0x54, 0x20,
	0x57, 0x52, 0x3b, 0x20, 0x31, 0x30, 0x20, 0x48, 0x4f, 0x53, 0x54, 0x20,
	0x52, 0x44, 0x29, 0x00,
	/* PSWHST_REG_CLIENTS_WAITING_TO_SOURCE_ARB */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x43,
	0x4c, 0x49, 0x45, 0x4e, 0x54, 0x53, 0x5f, 0x57, 0x41, 0x49, 0x54, 0x49,
	0x4e, 0x47, 0x5f, 0x54, 0x4f, 0x5f, 0x53, 0x4f, 0x55, 0x52, 0x43, 0x45,
	0x5f, 0x41, 0x52, 0x42, 0x00,
	/* PSWHST: Interrupt status is not 0 */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x49, 0x6e, 0x74, 0x65,
	0x72, 0x72, 0x75, 0x70, 0x74, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73,
	0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWHST_REG_INT_STS */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x4e, 0x54, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWHST_REG_INT_MASK */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x49,
	0x4e, 0x54, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PSWHST: parity error datapath registers */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x70, 0x61, 0x72, 0x69,
	0x74, 0x79, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x64, 0x61, 0x74,
	0x61, 0x70, 0x61, 0x74, 0x68, 0x20, 0x72, 0x65, 0x67, 0x69, 0x73, 0x74,
	0x65, 0x72, 0x73, 0x00,
	/* PSWHST_REG_PRTY_STS */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PSWHST_REG_PRTY_MASK */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PSWHST: Parity status is not 0 */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x3a, 0x20, 0x50, 0x61, 0x72, 0x69,
	0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69, 0x73,
	0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PSWHST_REG_PRTY_STS_H_0 */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30, 0x00,
	/* PSWHST_REG_PRTY_MASK_H_0 */
	0x50, 0x53, 0x57, 0x48, 0x53, 0x54, 0x5f, 0x52, 0x45, 0x47, 0x5f, 0x50,
	0x52, 0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f, 0x30,
	0x00,
	/* PGLUE: PTT and GTT initialization is not done */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x3a, 0x20, 0x50, 0x54, 0x54, 0x20, 0x61,
	0x6e, 0x64, 0x20, 0x47, 0x54, 0x54, 0x20, 0x69, 0x6e, 0x69, 0x74, 0x69,
	0x61, 0x6c, 0x69, 0x7a, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x69, 0x73,
	0x20, 0x6e, 0x6f, 0x74, 0x20, 0x64, 0x6f, 0x6e, 0x65, 0x00,
	/* PGLUE_B_REG_INIT_DONE_PTT_GTT */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x49, 0x4e, 0x49, 0x54, 0x5f, 0x44, 0x4f, 0x4e, 0x45, 0x5f, 0x50, 0x54,
	0x54, 0x5f, 0x47, 0x54, 0x54, 0x00,
	/* PGLUE_B: Parity status is not 0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x50, 0x61, 0x72,
	0x69, 0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PGLUE_B_REG_PRTY_STS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x50, 0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x00,
	/* PGLUE_B_REG_PRTY_MASK */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x50, 0x52, 0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x00,
	/* PGLUE_B: Parity status is not 0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x50, 0x61, 0x72,
	0x69, 0x74, 0x79, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PGLUE_B_REG_PRTY_STS_H_0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x50, 0x52, 0x54, 0x59, 0x5f, 0x53, 0x54, 0x53, 0x5f, 0x48, 0x5f, 0x30,
	0x00,
	/* PGLUE_B_REG_PRTY_MASK_H_0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x50, 0x52, 0x54, 0x59, 0x5f, 0x4d, 0x41, 0x53, 0x4b, 0x5f, 0x48, 0x5f,
	0x30, 0x00,
	/* PGLUE_B: pgl_write_blocked is not 0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x70, 0x67, 0x6c,
	0x5f, 0x77, 0x72, 0x69, 0x74, 0x65, 0x5f, 0x62, 0x6c, 0x6f, 0x63, 0x6b,
	0x65, 0x64, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PGLUE_B: PCI was closed for write requests while a request arrived (might happen in DrvSim due to Timers) */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x50, 0x43, 0x49,
	0x20, 0x77, 0x61, 0x73, 0x20, 0x63, 0x6c, 0x6f, 0x73, 0x65, 0x64, 0x20,
	0x66, 0x6f, 0x72, 0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x20, 0x72, 0x65,
	0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x20, 0x77, 0x68, 0x69, 0x6c, 0x65,
	0x20, 0x61, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x20, 0x61,
	0x72, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x28, 0x6d, 0x69, 0x67, 0x68,
	0x74, 0x20, 0x68, 0x61, 0x70, 0x70, 0x65, 0x6e, 0x20, 0x69, 0x6e, 0x20,
	0x44, 0x72, 0x76, 0x53, 0x69, 0x6d, 0x20, 0x64, 0x75, 0x65, 0x20, 0x74,
	0x6f, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x72, 0x73, 0x29, 0x00,
	/* PGLUE_B_REG_PGL_WRITE_BLOCKED */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x50, 0x47, 0x4c, 0x5f, 0x57, 0x52, 0x49, 0x54, 0x45, 0x5f, 0x42, 0x4c,
	0x4f, 0x43, 0x4b, 0x45, 0x44, 0x00,
	/* PGLUE_B: pgl_read_blocked is not 0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x70, 0x67, 0x6c,
	0x5f, 0x72, 0x65, 0x61, 0x64, 0x5f, 0x62, 0x6c, 0x6f, 0x63, 0x6b, 0x65,
	0x64, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PGLUE_B: PCI was closed for read requests while a request arrived (might happen in DrvSim due to Timers) */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x50, 0x43, 0x49,
	0x20, 0x77, 0x61, 0x73, 0x20, 0x63, 0x6c, 0x6f, 0x73, 0x65, 0x64, 0x20,
	0x66, 0x6f, 0x72, 0x20, 0x72, 0x65, 0x61, 0x64, 0x20, 0x72, 0x65, 0x71,
	0x75, 0x65, 0x73, 0x74, 0x73, 0x20, 0x77, 0x68, 0x69, 0x6c, 0x65, 0x20,
	0x61, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x20, 0x61, 0x72,
	0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x28, 0x6d, 0x69, 0x67, 0x68, 0x74,
	0x20, 0x68, 0x61, 0x70, 0x70, 0x65, 0x6e, 0x20, 0x69, 0x6e, 0x20, 0x44,
	0x72, 0x76, 0x53, 0x69, 0x6d, 0x20, 0x64, 0x75, 0x65, 0x20, 0x74, 0x6f,
	0x20, 0x54, 0x69, 0x6d, 0x65, 0x72, 0x73, 0x29, 0x00,
	/* PGLUE_B_REG_PGL_READ_BLOCKED */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x50, 0x47, 0x4c, 0x5f, 0x52, 0x45, 0x41, 0x44, 0x5f, 0x42, 0x4c, 0x4f,
	0x43, 0x4b, 0x45, 0x44, 0x00,
	/* PGL TX Read FIFO is not empty. */
	0x50, 0x47, 0x4c, 0x20, 0x54, 0x58, 0x20, 0x52, 0x65, 0x61, 0x64, 0x20,
	0x46, 0x49, 0x46, 0x4f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20,
	0x65, 0x6d, 0x70, 0x74, 0x79, 0x2e, 0x00,
	/* PGLUE_B_REG_READ_FIFO_OCCUPANCY_LEVEL */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x52, 0x45, 0x41, 0x44, 0x5f, 0x46, 0x49, 0x46, 0x4f, 0x5f, 0x4f, 0x43,
	0x43, 0x55, 0x50, 0x41, 0x4e, 0x43, 0x59, 0x5f, 0x4c, 0x45, 0x56, 0x45,
	0x4c, 0x00,
	/* Bit 0 - Target memory read with a correctable error. Bit 1 - Target memory read with an uncorrectable error. Bit 2 - Configuration RW with a correctable error. Bit 3 - Configuration RW with an uncorrectable error. Bit 4 - Target memory write or MSGD with a correctable error. Bit 5 - Target memory write or MSGD with an uncorrectable error. Bit 6 - dev->host completion with a correctable error. Bit 7 - dev->host completion with an uncorrectable error. */
	0x42, 0x69, 0x74, 0x20, 0x30, 0x20, 0x2d, 0x20, 0x54, 0x61, 0x72, 0x67,
	0x65, 0x74, 0x20, 0x6d, 0x65, 0x6d, 0x6f, 0x72, 0x79, 0x20, 0x72, 0x65,
	0x61, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x61, 0x20, 0x63, 0x6f,
	0x72, 0x72, 0x65, 0x63, 0x74, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x65, 0x72,
	0x72, 0x6f, 0x72, 0x2e, 0x20, 0x42, 0x69, 0x74, 0x20, 0x31, 0x20, 0x2d,
	0x20, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74, 0x20, 0x6d, 0x65, 0x6d, 0x6f,
	0x72, 0x79, 0x20, 0x72, 0x65, 0x61, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68,
	0x20, 0x61, 0x6e, 0x20, 0x75, 0x6e, 0x63, 0x6f, 0x72, 0x72, 0x65, 0x63,
	0x74, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x2e,
	0x20, 0x42, 0x69, 0x74, 0x20, 0x32, 0x20, 0x2d, 0x20, 0x43, 0x6f, 0x6e,
	0x66, 0x69, 0x67, 0x75, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x52,
	0x57, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x72,
	0x72, 0x65, 0x63, 0x74, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x65, 0x72, 0x72,
	0x6f, 0x72, 0x2e, 0x20, 0x42, 0x69, 0x74, 0x20, 0x33, 0x20, 0x2d, 0x20,
	0x43, 0x6f, 0x6e, 0x66, 0x69, 0x67, 0x75, 0x72, 0x61, 0x74, 0x69, 0x6f,
	0x6e, 0x20, 0x52, 0x57, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x61, 0x6e,
	0x20, 0x75, 0x6e, 0x63, 0x6f, 0x72, 0x72, 0x65, 0x63, 0x74, 0x61, 0x62,
	0x6c, 0x65, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x2e, 0x20, 0x42, 0x69,
	0x74, 0x20, 0x34, 0x20, 0x2d, 0x20, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74,
	0x20, 0x6d, 0x65, 0x6d, 0x6f, 0x72, 0x79, 0x20, 0x77, 0x72, 0x69, 0x74,
	0x65, 0x20, 0x6f, 0x72, 0x20, 0x4d, 0x53, 0x47, 0x44, 0x20, 0x77, 0x69,
	0x74, 0x68, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x72, 0x72, 0x65, 0x63, 0x74,
	0x61, 0x62, 0x6c, 0x65, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x2e, 0x20,
	0x42, 0x69, 0x74, 0x20, 0x35, 0x20, 0x2d, 0x20, 0x54, 0x61, 0x72, 0x67,
	0x65, 0x74, 0x20, 0x6d, 0x65, 0x6d, 0x6f, 0x72, 0x79, 0x20, 0x77, 0x72,
	0x69, 0x74, 0x65, 0x20, 0x6f, 0x72, 0x20, 0x4d, 0x53, 0x47, 0x44, 0x20,
	0x77, 0x69, 0x74, 0x68, 0x20, 0x61, 0x6e, 0x20, 0x75, 0x6e, 0x63, 0x6f,
	0x72, 0x72, 0x65, 0x63, 0x74, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x65, 0x72,
	0x72, 0x6f, 0x72, 0x2e, 0x20, 0x42, 0x69, 0x74, 0x20, 0x36, 0x20, 0x2d,
	0x20, 0x64, 0x65, 0x76, 0x2d, 0x3e, 0x68, 0x6f, 0x73, 0x74, 0x20, 0x63,
	0x6f, 0x6d, 0x70, 0x6c, 0x65, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x77, 0x69,
	0x74, 0x68, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x72, 0x72, 0x65, 0x63, 0x74,
	0x61, 0x62, 0x6c, 0x65, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x2e, 0x20,
	0x42, 0x69, 0x74, 0x20, 0x37, 0x20, 0x2d, 0x20, 0x64, 0x65, 0x76, 0x2d,
	0x3e, 0x68, 0x6f, 0x73, 0x74, 0x20, 0x63, 0x6f, 0x6d, 0x70, 0x6c, 0x65,
	0x74, 0x69, 0x6f, 0x6e, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x61, 0x6e,
	0x20, 0x75, 0x6e, 0x63, 0x6f, 0x72, 0x72, 0x65, 0x63, 0x74, 0x61, 0x62,
	0x6c, 0x65, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x2e, 0x00,
	/* PGLUE_B_REG_RX_LEGACY_ERRORS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x52, 0x58, 0x5f, 0x4c, 0x45, 0x47, 0x41, 0x43, 0x59, 0x5f, 0x45, 0x52,
	0x52, 0x4f, 0x52, 0x53, 0x00,
	/* PGLUE_B: There is data which is ready */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x54, 0x68, 0x65,
	0x72, 0x65, 0x20, 0x69, 0x73, 0x20, 0x64, 0x61, 0x74, 0x61, 0x20, 0x77,
	0x68, 0x69, 0x63, 0x68, 0x20, 0x69, 0x73, 0x20, 0x72, 0x65, 0x61, 0x64,
	0x79, 0x00,
	/* PGLUE_B_REG_PGL_TXW_CDTS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x50, 0x47, 0x4c, 0x5f, 0x54, 0x58, 0x57, 0x5f, 0x43, 0x44, 0x54, 0x53,
	0x00,
	/* PGL: Cfg-Space A request is set */
	0x50, 0x47, 0x4c, 0x3a, 0x20, 0x43, 0x66, 0x67, 0x2d, 0x53, 0x70, 0x61,
	0x63, 0x65, 0x20, 0x41, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74,
	0x20, 0x69, 0x73, 0x20, 0x73, 0x65, 0x74, 0x00,
	/* PGLUE_B_REG_CFG_SPACE_A_REQUEST */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x43, 0x46, 0x47, 0x5f, 0x53, 0x50, 0x41, 0x43, 0x45, 0x5f, 0x41, 0x5f,
	0x52, 0x45, 0x51, 0x55, 0x45, 0x53, 0x54, 0x00,
	/* PGL: Cfg-Space B request is set */
	0x50, 0x47, 0x4c, 0x3a, 0x20, 0x43, 0x66, 0x67, 0x2d, 0x53, 0x70, 0x61,
	0x63, 0x65, 0x20, 0x42, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74,
	0x20, 0x69, 0x73, 0x20, 0x73, 0x65, 0x74, 0x00,
	/* PGLUE_B_REG_CFG_SPACE_B_REQUEST */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x43, 0x46, 0x47, 0x5f, 0x53, 0x50, 0x41, 0x43, 0x45, 0x5f, 0x42, 0x5f,
	0x52, 0x45, 0x51, 0x55, 0x45, 0x53, 0x54, 0x00,
	/* PGL: FLR request is set for VF addresses 31-0 */
	0x50, 0x47, 0x4c, 0x3a, 0x20, 0x46, 0x4c, 0x52, 0x20, 0x72, 0x65, 0x71,
	0x75, 0x65, 0x73, 0x74, 0x20, 0x69, 0x73, 0x20, 0x73, 0x65, 0x74, 0x20,
	0x66, 0x6f, 0x72, 0x20, 0x56, 0x46, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65,
	0x73, 0x73, 0x65, 0x73, 0x20, 0x33, 0x31, 0x2d, 0x30, 0x00,
	/* PGLUE_B_REG_FLR_REQUEST_VF_31_0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x46, 0x4c, 0x52, 0x5f, 0x52, 0x45, 0x51, 0x55, 0x45, 0x53, 0x54, 0x5f,
	0x56, 0x46, 0x5f, 0x33, 0x31, 0x5f, 0x30, 0x00,
	/* PGL: FLR request is set for VF addresses 63-32 */
	0x50, 0x47, 0x4c, 0x3a, 0x20, 0x46, 0x4c, 0x52, 0x20, 0x72, 0x65, 0x71,
	0x75, 0x65, 0x73, 0x74, 0x20, 0x69, 0x73, 0x20, 0x73, 0x65, 0x74, 0x20,
	0x66, 0x6f, 0x72, 0x20, 0x56, 0x46, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65,
	0x73, 0x73, 0x65, 0x73, 0x20, 0x36, 0x33, 0x2d, 0x33, 0x32, 0x00,
	/* PGLUE_B_REG_FLR_REQUEST_VF_63_32 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x46, 0x4c, 0x52, 0x5f, 0x52, 0x45, 0x51, 0x55, 0x45, 0x53, 0x54, 0x5f,
	0x56, 0x46, 0x5f, 0x36, 0x33, 0x5f, 0x33, 0x32, 0x00,
	/* PGL: FLR request is set for VF addresses 95-64 */
	0x50, 0x47, 0x4c, 0x3a, 0x20, 0x46, 0x4c, 0x52, 0x20, 0x72, 0x65, 0x71,
	0x75, 0x65, 0x73, 0x74, 0x20, 0x69, 0x73, 0x20, 0x73, 0x65, 0x74, 0x20,
	0x66, 0x6f, 0x72, 0x20, 0x56, 0x46, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65,
	0x73, 0x73, 0x65, 0x73, 0x20, 0x39, 0x35, 0x2d, 0x36, 0x34, 0x00,
	/* PGLUE_B_REG_FLR_REQUEST_VF_95_64 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x46, 0x4c, 0x52, 0x5f, 0x52, 0x45, 0x51, 0x55, 0x45, 0x53, 0x54, 0x5f,
	0x56, 0x46, 0x5f, 0x39, 0x35, 0x5f, 0x36, 0x34, 0x00,
	/* PGL: FLR request is set for VF addresses 127-96 */
	0x50, 0x47, 0x4c, 0x3a, 0x20, 0x46, 0x4c, 0x52, 0x20, 0x72, 0x65, 0x71,
	0x75, 0x65, 0x73, 0x74, 0x20, 0x69, 0x73, 0x20, 0x73, 0x65, 0x74, 0x20,
	0x66, 0x6f, 0x72, 0x20, 0x56, 0x46, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65,
	0x73, 0x73, 0x65, 0x73, 0x20, 0x31, 0x32, 0x37, 0x2d, 0x39, 0x36, 0x00,
	/* PGLUE_B_REG_FLR_REQUEST_VF_127_96 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x46, 0x4c, 0x52, 0x5f, 0x52, 0x45, 0x51, 0x55, 0x45, 0x53, 0x54, 0x5f,
	0x56, 0x46, 0x5f, 0x31, 0x32, 0x37, 0x5f, 0x39, 0x36, 0x00,
	/* PGL: FLR request is set for VF addresses 159-128 */
	0x50, 0x47, 0x4c, 0x3a, 0x20, 0x46, 0x4c, 0x52, 0x20, 0x72, 0x65, 0x71,
	0x75, 0x65, 0x73, 0x74, 0x20, 0x69, 0x73, 0x20, 0x73, 0x65, 0x74, 0x20,
	0x66, 0x6f, 0x72, 0x20, 0x56, 0x46, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65,
	0x73, 0x73, 0x65, 0x73, 0x20, 0x31, 0x35, 0x39, 0x2d, 0x31, 0x32, 0x38,
	0x00,
	/* PGLUE_B_REG_FLR_REQUEST_VF_159_128 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x46, 0x4c, 0x52, 0x5f, 0x52, 0x45, 0x51, 0x55, 0x45, 0x53, 0x54, 0x5f,
	0x56, 0x46, 0x5f, 0x31, 0x35, 0x39, 0x5f, 0x31, 0x32, 0x38, 0x00,
	/* PGL: FLR request is set for VF addresses 191-160 */
	0x50, 0x47, 0x4c, 0x3a, 0x20, 0x46, 0x4c, 0x52, 0x20, 0x72, 0x65, 0x71,
	0x75, 0x65, 0x73, 0x74, 0x20, 0x69, 0x73, 0x20, 0x73, 0x65, 0x74, 0x20,
	0x66, 0x6f, 0x72, 0x20, 0x56, 0x46, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65,
	0x73, 0x73, 0x65, 0x73, 0x20, 0x31, 0x39, 0x31, 0x2d, 0x31, 0x36, 0x30,
	0x00,
	/* PGLUE_B_REG_FLR_REQUEST_VF_191_160 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x46, 0x4c, 0x52, 0x5f, 0x52, 0x45, 0x51, 0x55, 0x45, 0x53, 0x54, 0x5f,
	0x56, 0x46, 0x5f, 0x31, 0x39, 0x31, 0x5f, 0x31, 0x36, 0x30, 0x00,
	/* PGL: FLR request is set for PF addresses 7-0 */
	0x50, 0x47, 0x4c, 0x3a, 0x20, 0x46, 0x4c, 0x52, 0x20, 0x72, 0x65, 0x71,
	0x75, 0x65, 0x73, 0x74, 0x20, 0x69, 0x73, 0x20, 0x73, 0x65, 0x74, 0x20,
	0x66, 0x6f, 0x72, 0x20, 0x50, 0x46, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65,
	0x73, 0x73, 0x65, 0x73, 0x20, 0x37, 0x2d, 0x30, 0x00,
	/* PGLUE_B_REG_FLR_REQUEST_PF_31_0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x46, 0x4c, 0x52, 0x5f, 0x52, 0x45, 0x51, 0x55, 0x45, 0x53, 0x54, 0x5f,
	0x50, 0x46, 0x5f, 0x33, 0x31, 0x5f, 0x30, 0x00,
	/* PGL: SR-IOV disable request is set */
	0x50, 0x47, 0x4c, 0x3a, 0x20, 0x53, 0x52, 0x2d, 0x49, 0x4f, 0x56, 0x20,
	0x64, 0x69, 0x73, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x72, 0x65, 0x71, 0x75,
	0x65, 0x73, 0x74, 0x20, 0x69, 0x73, 0x20, 0x73, 0x65, 0x74, 0x00,
	/* PGLUE_B_REG_SR_IOV_DISABLED_REQUEST */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x53, 0x52, 0x5f, 0x49, 0x4f, 0x56, 0x5f, 0x44, 0x49, 0x53, 0x41, 0x42,
	0x4c, 0x45, 0x44, 0x5f, 0x52, 0x45, 0x51, 0x55, 0x45, 0x53, 0x54, 0x00,
	/* PGLUE_B: Packet received from PCIe not according to the rules. */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x50, 0x61, 0x63,
	0x6b, 0x65, 0x74, 0x20, 0x72, 0x65, 0x63, 0x65, 0x69, 0x76, 0x65, 0x64,
	0x20, 0x66, 0x72, 0x6f, 0x6d, 0x20, 0x50, 0x43, 0x49, 0x65, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x61, 0x63, 0x63, 0x6f, 0x72, 0x64, 0x69, 0x6e, 0x67,
	0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65, 0x20, 0x72, 0x75, 0x6c, 0x65,
	0x73, 0x2e, 0x00,
	/* PGLUE_B_REG_INCORRECT_RCV_DETAILS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x49, 0x4e, 0x43, 0x4f, 0x52, 0x52, 0x45, 0x43, 0x54, 0x5f, 0x52, 0x43,
	0x56, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x53, 0x00,
	/* PGLUE_B: was_error for VFs 0-31 is not 0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x77, 0x61, 0x73,
	0x5f, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x56,
	0x46, 0x73, 0x20, 0x30, 0x2d, 0x33, 0x31, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PGLUE_B_REG_WAS_ERROR_VF_31_0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x57, 0x41, 0x53, 0x5f, 0x45, 0x52, 0x52, 0x4f, 0x52, 0x5f, 0x56, 0x46,
	0x5f, 0x33, 0x31, 0x5f, 0x30, 0x00,
	/* PGLUE_B: was_error for VFs 32-63 is not 0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x77, 0x61, 0x73,
	0x5f, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x56,
	0x46, 0x73, 0x20, 0x33, 0x32, 0x2d, 0x36, 0x33, 0x20, 0x69, 0x73, 0x20,
	0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PGLUE_B_REG_WAS_ERROR_VF_63_32 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x57, 0x41, 0x53, 0x5f, 0x45, 0x52, 0x52, 0x4f, 0x52, 0x5f, 0x56, 0x46,
	0x5f, 0x36, 0x33, 0x5f, 0x33, 0x32, 0x00,
	/* PGLUE_B: was_error for VFs 64-95 is not 0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x77, 0x61, 0x73,
	0x5f, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x56,
	0x46, 0x73, 0x20, 0x36, 0x34, 0x2d, 0x39, 0x35, 0x20, 0x69, 0x73, 0x20,
	0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PGLUE_B_REG_WAS_ERROR_VF_95_64 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x57, 0x41, 0x53, 0x5f, 0x45, 0x52, 0x52, 0x4f, 0x52, 0x5f, 0x56, 0x46,
	0x5f, 0x39, 0x35, 0x5f, 0x36, 0x34, 0x00,
	/* PGLUE_B: was_error for VFs 96-127 is not 0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x77, 0x61, 0x73,
	0x5f, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x56,
	0x46, 0x73, 0x20, 0x39, 0x36, 0x2d, 0x31, 0x32, 0x37, 0x20, 0x69, 0x73,
	0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PGLUE_B_REG_WAS_ERROR_VF_127_96 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x57, 0x41, 0x53, 0x5f, 0x45, 0x52, 0x52, 0x4f, 0x52, 0x5f, 0x56, 0x46,
	0x5f, 0x31, 0x32, 0x37, 0x5f, 0x39, 0x36, 0x00,
	/* PGLUE_B: was_error for VFs 128-159 is not 0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x77, 0x61, 0x73,
	0x5f, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x56,
	0x46, 0x73, 0x20, 0x31, 0x32, 0x38, 0x2d, 0x31, 0x35, 0x39, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PGLUE_B_REG_WAS_ERROR_VF_159_128 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x57, 0x41, 0x53, 0x5f, 0x45, 0x52, 0x52, 0x4f, 0x52, 0x5f, 0x56, 0x46,
	0x5f, 0x31, 0x35, 0x39, 0x5f, 0x31, 0x32, 0x38, 0x00,
	/* PGLUE_B: was_error for VFs 160-191 is not 0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x77, 0x61, 0x73,
	0x5f, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x56,
	0x46, 0x73, 0x20, 0x31, 0x36, 0x30, 0x2d, 0x31, 0x39, 0x31, 0x20, 0x69,
	0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PGLUE_B_REG_WAS_ERROR_VF_191_160 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x57, 0x41, 0x53, 0x5f, 0x45, 0x52, 0x52, 0x4f, 0x52, 0x5f, 0x56, 0x46,
	0x5f, 0x31, 0x39, 0x31, 0x5f, 0x31, 0x36, 0x30, 0x00,
	/* PGLUE_B: was_error for PFs 0-15 is not 0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x77, 0x61, 0x73,
	0x5f, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x50,
	0x46, 0x73, 0x20, 0x30, 0x2d, 0x31, 0x35, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x00,
	/* PGLUE_B_REG_WAS_ERROR_PF_31_0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x57, 0x41, 0x53, 0x5f, 0x45, 0x52, 0x52, 0x4f, 0x52, 0x5f, 0x50, 0x46,
	0x5f, 0x33, 0x31, 0x5f, 0x30, 0x00,
	/* PGLUE_B: Completion received with error. (3:0) - PFID. (4) - VF_VALID. (12:5) - VFID. (14:13) - Error code : 0 - Completion Timeout ; 1 - Unsupported Request; 2 - Completer Abort; 3 - VF en or poison error is also detected.. (15) - valid bit. */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x43, 0x6f, 0x6d,
	0x70, 0x6c, 0x65, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x72, 0x65, 0x63, 0x65,
	0x69, 0x76, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x65, 0x72,
	0x72, 0x6f, 0x72, 0x2e, 0x20, 0x28, 0x33, 0x3a, 0x30, 0x29, 0x20, 0x2d,
	0x20, 0x50, 0x46, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x34, 0x29, 0x20, 0x2d,
	0x20, 0x56, 0x46, 0x5f, 0x56, 0x41, 0x4c, 0x49, 0x44, 0x2e, 0x20, 0x28,
	0x31, 0x32, 0x3a, 0x35, 0x29, 0x20, 0x2d, 0x20, 0x56, 0x46, 0x49, 0x44,
	0x2e, 0x20, 0x28, 0x31, 0x34, 0x3a, 0x31, 0x33, 0x29, 0x20, 0x2d, 0x20,
	0x45, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x20, 0x3a,
	0x20, 0x30, 0x20, 0x2d, 0x20, 0x43, 0x6f, 0x6d, 0x70, 0x6c, 0x65, 0x74,
	0x69, 0x6f, 0x6e, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x6f, 0x75, 0x74, 0x20,
	0x3b, 0x20, 0x31, 0x20, 0x2d, 0x20, 0x55, 0x6e, 0x73, 0x75, 0x70, 0x70,
	0x6f, 0x72, 0x74, 0x65, 0x64, 0x20, 0x52, 0x65, 0x71, 0x75, 0x65, 0x73,
	0x74, 0x3b, 0x20, 0x32, 0x20, 0x2d, 0x20, 0x43, 0x6f, 0x6d, 0x70, 0x6c,
	0x65, 0x74, 0x65, 0x72, 0x20, 0x41, 0x62, 0x6f, 0x72, 0x74, 0x3b, 0x20,
	0x33, 0x20, 0x2d, 0x20, 0x56, 0x46, 0x20, 0x65, 0x6e, 0x20, 0x6f, 0x72,
	0x20, 0x70, 0x6f, 0x69, 0x73, 0x6f, 0x6e, 0x20, 0x65, 0x72, 0x72, 0x6f,
	0x72, 0x20, 0x69, 0x73, 0x20, 0x61, 0x6c, 0x73, 0x6f, 0x20, 0x64, 0x65,
	0x74, 0x65, 0x63, 0x74, 0x65, 0x64, 0x2e, 0x2e, 0x20, 0x28, 0x31, 0x35,
	0x29, 0x20, 0x2d, 0x20, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20, 0x62, 0x69,
	0x74, 0x2e, 0x00,
	/* PGLUE_B_REG_RX_ERR_DETAILS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x52, 0x58, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49,
	0x4c, 0x53, 0x00,
	/* PGLUE_B: ATS TCPL received with error. (3:0) - PFID. (4) - VF_VALID. (12:5) - VFID. (14:13) - Error code : 0 - Completion Timeout ; 1 - Unsupported Request; 2 - Completer Abort. (19:15) - OTB Entry ID. (20) - valid bit. */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x41, 0x54, 0x53,
	0x20, 0x54, 0x43, 0x50, 0x4c, 0x20, 0x72, 0x65, 0x63, 0x65, 0x69, 0x76,
	0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x65, 0x72, 0x72, 0x6f,
	0x72, 0x2e, 0x20, 0x28, 0x33, 0x3a, 0x30, 0x29, 0x20, 0x2d, 0x20, 0x50,
	0x46, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x34, 0x29, 0x20, 0x2d, 0x20, 0x56,
	0x46, 0x5f, 0x56, 0x41, 0x4c, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x31, 0x32,
	0x3a, 0x35, 0x29, 0x20, 0x2d, 0x20, 0x56, 0x46, 0x49, 0x44, 0x2e, 0x20,
	0x28, 0x31, 0x34, 0x3a, 0x31, 0x33, 0x29, 0x20, 0x2d, 0x20, 0x45, 0x72,
	0x72, 0x6f, 0x72, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x20, 0x3a, 0x20, 0x30,
	0x20, 0x2d, 0x20, 0x43, 0x6f, 0x6d, 0x70, 0x6c, 0x65, 0x74, 0x69, 0x6f,
	0x6e, 0x20, 0x54, 0x69, 0x6d, 0x65, 0x6f, 0x75, 0x74, 0x20, 0x3b, 0x20,
	0x31, 0x20, 0x2d, 0x20, 0x55, 0x6e, 0x73, 0x75, 0x70, 0x70, 0x6f, 0x72,
	0x74, 0x65, 0x64, 0x20, 0x52, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x3b,
	0x20, 0x32, 0x20, 0x2d, 0x20, 0x43, 0x6f, 0x6d, 0x70, 0x6c, 0x65, 0x74,
	0x65, 0x72, 0x20, 0x41, 0x62, 0x6f, 0x72, 0x74, 0x2e, 0x20, 0x28, 0x31,
	0x39, 0x3a, 0x31, 0x35, 0x29, 0x20, 0x2d, 0x20, 0x4f, 0x54, 0x42, 0x20,
	0x45, 0x6e, 0x74, 0x72, 0x79, 0x20, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x32,
	0x30, 0x29, 0x20, 0x2d, 0x20, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20, 0x62,
	0x69, 0x74, 0x2e, 0x00,
	/* PGLUE_B_REG_RX_TCPL_ERR_DETAILS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x52, 0x58, 0x5f, 0x54, 0x43, 0x50, 0x4c, 0x5f, 0x45, 0x52, 0x52, 0x5f,
	0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x53, 0x00,
	/* PGLUE_B: Error in dev->host write. Error details register is not 0. (4:0) VQID. (17:5) - Length in bytes. (19) - VF_VALID. (23:20) - PFID. (31:24) - VFID. */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x45, 0x72, 0x72,
	0x6f, 0x72, 0x20, 0x69, 0x6e, 0x20, 0x64, 0x65, 0x76, 0x2d, 0x3e, 0x68,
	0x6f, 0x73, 0x74, 0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x2e, 0x20, 0x45,
	0x72, 0x72, 0x6f, 0x72, 0x20, 0x64, 0x65, 0x74, 0x61, 0x69, 0x6c, 0x73,
	0x20, 0x72, 0x65, 0x67, 0x69, 0x73, 0x74, 0x65, 0x72, 0x20, 0x69, 0x73,
	0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x2e, 0x20, 0x28, 0x34, 0x3a, 0x30,
	0x29, 0x20, 0x56, 0x51, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x31, 0x37, 0x3a,
	0x35, 0x29, 0x20, 0x2d, 0x20, 0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x20,
	0x69, 0x6e, 0x20, 0x62, 0x79, 0x74, 0x65, 0x73, 0x2e, 0x20, 0x28, 0x31,
	0x39, 0x29, 0x20, 0x2d, 0x20, 0x56, 0x46, 0x5f, 0x56, 0x41, 0x4c, 0x49,
	0x44, 0x2e, 0x20, 0x28, 0x32, 0x33, 0x3a, 0x32, 0x30, 0x29, 0x20, 0x2d,
	0x20, 0x50, 0x46, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x33, 0x31, 0x3a, 0x32,
	0x34, 0x29, 0x20, 0x2d, 0x20, 0x56, 0x46, 0x49, 0x44, 0x2e, 0x00,
	/* PGLUE_B_REG_TX_ERR_WR_DETAILS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x54, 0x58, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x57, 0x52, 0x5f, 0x44, 0x45,
	0x54, 0x41, 0x49, 0x4c, 0x53, 0x00,
	/* PGLUE_B_REG_TX_ERR_WR_ADD_63_32 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x54, 0x58, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x57, 0x52, 0x5f, 0x41, 0x44,
	0x44, 0x5f, 0x36, 0x33, 0x5f, 0x33, 0x32, 0x00,
	/* PGLUE_B_REG_TX_ERR_WR_ADD_31_0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x54, 0x58, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x57, 0x52, 0x5f, 0x41, 0x44,
	0x44, 0x5f, 0x33, 0x31, 0x5f, 0x30, 0x00,
	/* PGLUE_B: Error in dev->host write. Error details 2nd register is not 0. [15:0] Request ID. [20:16] client ID. [24:21] - Error type. [21] - Indicates was_error was set; [22] - Indicates BME was cleared. [23] - Indicates FID_enable was cleared; [24] - Indicates VF with parent PF FLR_request or IOV_disable_request dirty bit is set; [25] - Indicates AtomicOp Requester Enable was cleared for Atomic Operation; [26] - last SR [28:27] - Atomic - 0 - Regular request (not Atomic). 1 - CAS. 2 - FetchAdd. 3 - Swap. [29] - valid - indicates if there was a request not submitted due to error since the last time this register was cleared. [30] - dstate_0. ; [31] - first SR. */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x45, 0x72, 0x72,
	0x6f, 0x72, 0x20, 0x69, 0x6e, 0x20, 0x64, 0x65, 0x76, 0x2d, 0x3e, 0x68,
	0x6f, 0x73, 0x74, 0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x2e, 0x20, 0x45,
	0x72, 0x72, 0x6f, 0x72, 0x20, 0x64, 0x65, 0x74, 0x61, 0x69, 0x6c, 0x73,
	0x20, 0x32, 0x6e, 0x64, 0x20, 0x72, 0x65, 0x67, 0x69, 0x73, 0x74, 0x65,
	0x72, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x2e, 0x20,
	0x5b, 0x31, 0x35, 0x3a, 0x30, 0x5d, 0x20, 0x52, 0x65, 0x71, 0x75, 0x65,
	0x73, 0x74, 0x20, 0x49, 0x44, 0x2e, 0x20, 0x5b, 0x32, 0x30, 0x3a, 0x31,
	0x36, 0x5d, 0x20, 0x63, 0x6c, 0x69, 0x65, 0x6e, 0x74, 0x20, 0x49, 0x44,
	0x2e, 0x20, 0x5b, 0x32, 0x34, 0x3a, 0x32, 0x31, 0x5d, 0x20, 0x2d, 0x20,
	0x45, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x74, 0x79, 0x70, 0x65, 0x2e, 0x20,
	0x5b, 0x32, 0x31, 0x5d, 0x20, 0x2d, 0x20, 0x49, 0x6e, 0x64, 0x69, 0x63,
	0x61, 0x74, 0x65, 0x73, 0x20, 0x77, 0x61, 0x73, 0x5f, 0x65, 0x72, 0x72,
	0x6f, 0x72, 0x20, 0x77, 0x61, 0x73, 0x20, 0x73, 0x65, 0x74, 0x3b, 0x20,
	0x5b, 0x32, 0x32, 0x5d, 0x20, 0x2d, 0x20, 0x49, 0x6e, 0x64, 0x69, 0x63,
	0x61, 0x74, 0x65, 0x73, 0x20, 0x42, 0x4d, 0x45, 0x20, 0x77, 0x61, 0x73,
	0x20, 0x63, 0x6c, 0x65, 0x61, 0x72, 0x65, 0x64, 0x2e, 0x20, 0x5b, 0x32,
	0x33, 0x5d, 0x20, 0x2d, 0x20, 0x49, 0x6e, 0x64, 0x69, 0x63, 0x61, 0x74,
	0x65, 0x73, 0x20, 0x46, 0x49, 0x44, 0x5f, 0x65, 0x6e, 0x61, 0x62, 0x6c,
	0x65, 0x20, 0x77, 0x61, 0x73, 0x20, 0x63, 0x6c, 0x65, 0x61, 0x72, 0x65,
	0x64, 0x3b, 0x20, 0x5b, 0x32, 0x34, 0x5d, 0x20, 0x2d, 0x20, 0x49, 0x6e,
	0x64, 0x69, 0x63, 0x61, 0x74, 0x65, 0x73, 0x20, 0x56, 0x46, 0x20, 0x77,
	0x69, 0x74, 0x68, 0x20, 0x70, 0x61, 0x72, 0x65, 0x6e, 0x74, 0x20, 0x50,
	0x46, 0x20, 0x46, 0x4c, 0x52, 0x5f, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73,
	0x74, 0x20, 0x6f, 0x72, 0x20, 0x49, 0x4f, 0x56, 0x5f, 0x64, 0x69, 0x73,
	0x61, 0x62, 0x6c, 0x65, 0x5f, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74,
	0x20, 0x64, 0x69, 0x72, 0x74, 0x79, 0x20, 0x62, 0x69, 0x74, 0x20, 0x69,
	0x73, 0x20, 0x73, 0x65, 0x74, 0x3b, 0x20, 0x5b, 0x32, 0x35, 0x5d, 0x20,
	0x2d, 0x20, 0x49, 0x6e, 0x64, 0x69, 0x63, 0x61, 0x74, 0x65, 0x73, 0x20,
	0x41, 0x74, 0x6f, 0x6d, 0x69, 0x63, 0x4f, 0x70, 0x20, 0x52, 0x65, 0x71,
	0x75, 0x65, 0x73, 0x74, 0x65, 0x72, 0x20, 0x45, 0x6e, 0x61, 0x62, 0x6c,
	0x65, 0x20, 0x77, 0x61, 0x73, 0x20, 0x63, 0x6c, 0x65, 0x61, 0x72, 0x65,
	0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x41, 0x74, 0x6f, 0x6d, 0x69, 0x63,
	0x20, 0x4f, 0x70, 0x65, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x3b, 0x20,
	0x5b, 0x32, 0x36, 0x5d, 0x20, 0x2d, 0x20, 0x6c, 0x61, 0x73, 0x74, 0x20,
	0x53, 0x52, 0x20, 0x5b, 0x32, 0x38, 0x3a, 0x32, 0x37, 0x5d, 0x20, 0x2d,
	0x20, 0x41, 0x74, 0x6f, 0x6d, 0x69, 0x63, 0x20, 0x2d, 0x20, 0x30, 0x20,
	0x2d, 0x20, 0x52, 0x65, 0x67, 0x75, 0x6c, 0x61, 0x72, 0x20, 0x72, 0x65,
	0x71, 0x75, 0x65, 0x73, 0x74, 0x20, 0x28, 0x6e, 0x6f, 0x74, 0x20, 0x41,
	0x74, 0x6f, 0x6d, 0x69, 0x63, 0x29, 0x2e, 0x20, 0x31, 0x20, 0x2d, 0x20,
	0x43, 0x41, 0x53, 0x2e, 0x20, 0x32, 0x20, 0x2d, 0x20, 0x46, 0x65, 0x74,
	0x63, 0x68, 0x41, 0x64, 0x64, 0x2e, 0x20, 0x33, 0x20, 0x2d, 0x20, 0x53,
	0x77, 0x61, 0x70, 0x2e, 0x20, 0x5b, 0x32, 0x39, 0x5d, 0x20, 0x2d, 0x20,
	0x76, 0x61, 0x6c, 0x69, 0x64, 0x20, 0x2d, 0x20, 0x69, 0x6e, 0x64, 0x69,
	0x63, 0x61, 0x74, 0x65, 0x73, 0x20, 0x69, 0x66, 0x20, 0x74, 0x68, 0x65,
	0x72, 0x65, 0x20, 0x77, 0x61, 0x73, 0x20, 0x61, 0x20, 0x72, 0x65, 0x71,
	0x75, 0x65, 0x73, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x73, 0x75, 0x62,
	0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x64, 0x75, 0x65, 0x20, 0x74,
	0x6f, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x73, 0x69, 0x6e, 0x63,
	0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6c, 0x61, 0x73, 0x74, 0x20, 0x74,
	0x69, 0x6d, 0x65, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x72, 0x65, 0x67,
	0x69, 0x73, 0x74, 0x65, 0x72, 0x20, 0x77, 0x61, 0x73, 0x20, 0x63, 0x6c,
	0x65, 0x61, 0x72, 0x65, 0x64, 0x2e, 0x20, 0x5b, 0x33, 0x30, 0x5d, 0x20,
	0x2d, 0x20, 0x64, 0x73, 0x74, 0x61, 0x74, 0x65, 0x5f, 0x30, 0x2e, 0x20,
	0x3b, 0x20, 0x5b, 0x33, 0x31, 0x5d, 0x20, 0x2d, 0x20, 0x66, 0x69, 0x72,
	0x73, 0x74, 0x20, 0x53, 0x52, 0x2e, 0x00,
	/* PGLUE_B_REG_TX_ERR_WR_DETAILS2 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x54, 0x58, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x57, 0x52, 0x5f, 0x44, 0x45,
	0x54, 0x41, 0x49, 0x4c, 0x53, 0x32, 0x00,
	/* PGLUE_B: Error in dev->host read Error details register is not 0. (4:0) VQID. (5) Is TREQ. (18:6) - Length in bytes. (19) - VF_VALID. (23:20) - PFID. (31:24) - VFID. */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x45, 0x72, 0x72,
	0x6f, 0x72, 0x20, 0x69, 0x6e, 0x20, 0x64, 0x65, 0x76, 0x2d, 0x3e, 0x68,
	0x6f, 0x73, 0x74, 0x20, 0x72, 0x65, 0x61, 0x64, 0x20, 0x45, 0x72, 0x72,
	0x6f, 0x72, 0x20, 0x64, 0x65, 0x74, 0x61, 0x69, 0x6c, 0x73, 0x20, 0x72,
	0x65, 0x67, 0x69, 0x73, 0x74, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20, 0x6e,
	0x6f, 0x74, 0x20, 0x30, 0x2e, 0x20, 0x28, 0x34, 0x3a, 0x30, 0x29, 0x20,
	0x56, 0x51, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x35, 0x29, 0x20, 0x49, 0x73,
	0x20, 0x54, 0x52, 0x45, 0x51, 0x2e, 0x20, 0x28, 0x31, 0x38, 0x3a, 0x36,
	0x29, 0x20, 0x2d, 0x20, 0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x20, 0x69,
	0x6e, 0x20, 0x62, 0x79, 0x74, 0x65, 0x73, 0x2e, 0x20, 0x28, 0x31, 0x39,
	0x29, 0x20, 0x2d, 0x20, 0x56, 0x46, 0x5f, 0x56, 0x41, 0x4c, 0x49, 0x44,
	0x2e, 0x20, 0x28, 0x32, 0x33, 0x3a, 0x32, 0x30, 0x29, 0x20, 0x2d, 0x20,
	0x50, 0x46, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x33, 0x31, 0x3a, 0x32, 0x34,
	0x29, 0x20, 0x2d, 0x20, 0x56, 0x46, 0x49, 0x44, 0x2e, 0x00,
	/* PGLUE_B_REG_TX_ERR_RD_DETAILS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x54, 0x58, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x52, 0x44, 0x5f, 0x44, 0x45,
	0x54, 0x41, 0x49, 0x4c, 0x53, 0x00,
	/* PGLUE_B_REG_TX_ERR_RD_ADD_63_32 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x54, 0x58, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x52, 0x44, 0x5f, 0x41, 0x44,
	0x44, 0x5f, 0x36, 0x33, 0x5f, 0x33, 0x32, 0x00,
	/* PGLUE_B_REG_TX_ERR_RD_ADD_31_0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x54, 0x58, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x52, 0x44, 0x5f, 0x41, 0x44,
	0x44, 0x5f, 0x33, 0x31, 0x5f, 0x30, 0x00,
	/* PGLUE_B: Error in dev->host read Error details 2nd register is not 0. [15:0] Request ID. [20:16] client ID. [24:21] - Error type. [21] - Indicates was_error was set; [22] - Indicates BME was cleared; [23] - Indicates FID_enable was cleared; [24] - Indicates VF with parent PF FLR_request or IOV_disable_request dirty bit is set; [25] - last SR. [26] valid - indicates if there was a request not submitted due to error since the last time this register was cleared. [27] - dstate_0; [28] - first SR. */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x45, 0x72, 0x72,
	0x6f, 0x72, 0x20, 0x69, 0x6e, 0x20, 0x64, 0x65, 0x76, 0x2d, 0x3e, 0x68,
	0x6f, 0x73, 0x74, 0x20, 0x72, 0x65, 0x61, 0x64, 0x20, 0x45, 0x72, 0x72,
	0x6f, 0x72, 0x20, 0x64, 0x65, 0x74, 0x61, 0x69, 0x6c, 0x73, 0x20, 0x32,
	0x6e, 0x64, 0x20, 0x72, 0x65, 0x67, 0x69, 0x73, 0x74, 0x65, 0x72, 0x20,
	0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x30, 0x2e, 0x20, 0x5b, 0x31,
	0x35, 0x3a, 0x30, 0x5d, 0x20, 0x52, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74,
	0x20, 0x49, 0x44, 0x2e, 0x20, 0x5b, 0x32, 0x30, 0x3a, 0x31, 0x36, 0x5d,
	0x20, 0x63, 0x6c, 0x69, 0x65, 0x6e, 0x74, 0x20, 0x49, 0x44, 0x2e, 0x20,
	0x5b, 0x32, 0x34, 0x3a, 0x32, 0x31, 0x5d, 0x20, 0x2d, 0x20, 0x45, 0x72,
	0x72, 0x6f, 0x72, 0x20, 0x74, 0x79, 0x70, 0x65, 0x2e, 0x20, 0x5b, 0x32,
	0x31, 0x5d, 0x20, 0x2d, 0x20, 0x49, 0x6e, 0x64, 0x69, 0x63, 0x61, 0x74,
	0x65, 0x73, 0x20, 0x77, 0x61, 0x73, 0x5f, 0x65, 0x72, 0x72, 0x6f, 0x72,
	0x20, 0x77, 0x61, 0x73, 0x20, 0x73, 0x65, 0x74, 0x3b, 0x20, 0x5b, 0x32,
	0x32, 0x5d, 0x20, 0x2d, 0x20, 0x49, 0x6e, 0x64, 0x69, 0x63, 0x61, 0x74,
	0x65, 0x73, 0x20, 0x42, 0x4d, 0x45, 0x20, 0x77, 0x61, 0x73, 0x20, 0x63,
	0x6c, 0x65, 0x61, 0x72, 0x65, 0x64, 0x3b, 0x20, 0x5b, 0x32, 0x33, 0x5d,
	0x20, 0x2d, 0x20, 0x49, 0x6e, 0x64, 0x69, 0x63, 0x61, 0x74, 0x65, 0x73,
	0x20, 0x46, 0x49, 0x44, 0x5f, 0x65, 0x6e, 0x61, 0x62, 0x6c, 0x65, 0x20,
	0x77, 0x61, 0x73, 0x20, 0x63, 0x6c, 0x65, 0x61, 0x72, 0x65, 0x64, 0x3b,
	0x20, 0x5b, 0x32, 0x34, 0x5d, 0x20, 0x2d, 0x20, 0x49, 0x6e, 0x64, 0x69,
	0x63, 0x61, 0x74, 0x65, 0x73, 0x20, 0x56, 0x46, 0x20, 0x77, 0x69, 0x74,
	0x68, 0x20, 0x70, 0x61, 0x72, 0x65, 0x6e, 0x74, 0x20, 0x50, 0x46, 0x20,
	0x46, 0x4c, 0x52, 0x5f, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x20,
	0x6f, 0x72, 0x20, 0x49, 0x4f, 0x56, 0x5f, 0x64, 0x69, 0x73, 0x61, 0x62,
	0x6c, 0x65, 0x5f, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x20, 0x64,
	0x69, 0x72, 0x74, 0x79, 0x20, 0x62, 0x69, 0x74, 0x20, 0x69, 0x73, 0x20,
	0x73, 0x65, 0x74, 0x3b, 0x20, 0x5b, 0x32, 0x35, 0x5d, 0x20, 0x2d, 0x20,
	0x6c, 0x61, 0x73, 0x74, 0x20, 0x53, 0x52, 0x2e, 0x20, 0x5b, 0x32, 0x36,
	0x5d, 0x20, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20, 0x2d, 0x20, 0x69, 0x6e,
	0x64, 0x69, 0x63, 0x61, 0x74, 0x65, 0x73, 0x20, 0x69, 0x66, 0x20, 0x74,
	0x68, 0x65, 0x72, 0x65, 0x20, 0x77, 0x61, 0x73, 0x20, 0x61, 0x20, 0x72,
	0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x20, 0x6e, 0x6f, 0x74, 0x20, 0x73,
	0x75, 0x62, 0x6d, 0x69, 0x74, 0x74, 0x65, 0x64, 0x20, 0x64, 0x75, 0x65,
	0x20, 0x74, 0x6f, 0x20, 0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x73, 0x69,
	0x6e, 0x63, 0x65, 0x20, 0x74, 0x68, 0x65, 0x20, 0x6c, 0x61, 0x73, 0x74,
	0x20, 0x74, 0x69, 0x6d, 0x65, 0x20, 0x74, 0x68, 0x69, 0x73, 0x20, 0x72,
	0x65, 0x67, 0x69, 0x73, 0x74, 0x65, 0x72, 0x20, 0x77, 0x61, 0x73, 0x20,
	0x63, 0x6c, 0x65, 0x61, 0x72, 0x65, 0x64, 0x2e, 0x20, 0x5b, 0x32, 0x37,
	0x5d, 0x20, 0x2d, 0x20, 0x64, 0x73, 0x74, 0x61, 0x74, 0x65, 0x5f, 0x30,
	0x3b, 0x20, 0x5b, 0x32, 0x38, 0x5d, 0x20, 0x2d, 0x20, 0x66, 0x69, 0x72,
	0x73, 0x74, 0x20, 0x53, 0x52, 0x2e, 0x00,
	/* PGLUE_B_REG_TX_ERR_RD_DETAILS2 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x54, 0x58, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x52, 0x44, 0x5f, 0x44, 0x45,
	0x54, 0x41, 0x49, 0x4c, 0x53, 0x32, 0x00,
	/* PGLUE_B: Target VF length violation access. (12:0) Address in DWs (bits (14:2) of byte address). (14:13) BAR. (22:15) VFID. (26:23) - PFID. */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x54, 0x61, 0x72,
	0x67, 0x65, 0x74, 0x20, 0x56, 0x46, 0x20, 0x6c, 0x65, 0x6e, 0x67, 0x74,
	0x68, 0x20, 0x76, 0x69, 0x6f, 0x6c, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
	0x61, 0x63, 0x63, 0x65, 0x73, 0x73, 0x2e, 0x20, 0x28, 0x31, 0x32, 0x3a,
	0x30, 0x29, 0x20, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x20, 0x69,
	0x6e, 0x20, 0x44, 0x57, 0x73, 0x20, 0x28, 0x62, 0x69, 0x74, 0x73, 0x20,
	0x28, 0x31, 0x34, 0x3a, 0x32, 0x29, 0x20, 0x6f, 0x66, 0x20, 0x62, 0x79,
	0x74, 0x65, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x29, 0x2e,
	0x20, 0x28, 0x31, 0x34, 0x3a, 0x31, 0x33, 0x29, 0x20, 0x42, 0x41, 0x52,
	0x2e, 0x20, 0x28, 0x32, 0x32, 0x3a, 0x31, 0x35, 0x29, 0x20, 0x56, 0x46,
	0x49, 0x44, 0x2e, 0x20, 0x28, 0x32, 0x36, 0x3a, 0x32, 0x33, 0x29, 0x20,
	0x2d, 0x20, 0x50, 0x46, 0x49, 0x44, 0x2e, 0x00,
	/* PGLUE_B_REG_VF_LENGTH_VIOLATION_DETAILS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x56, 0x46, 0x5f, 0x4c, 0x45, 0x4e, 0x47, 0x54, 0x48, 0x5f, 0x56, 0x49,
	0x4f, 0x4c, 0x41, 0x54, 0x49, 0x4f, 0x4e, 0x5f, 0x44, 0x45, 0x54, 0x41,
	0x49, 0x4c, 0x53, 0x00,
	/* PGLUE_B: Target VF length violation access. (5:0) - Length in DWs. (6) valid bit. */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x54, 0x61, 0x72,
	0x67, 0x65, 0x74, 0x20, 0x56, 0x46, 0x20, 0x6c, 0x65, 0x6e, 0x67, 0x74,
	0x68, 0x20, 0x76, 0x69, 0x6f, 0x6c, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20,
	0x61, 0x63, 0x63, 0x65, 0x73, 0x73, 0x2e, 0x20, 0x28, 0x35, 0x3a, 0x30,
	0x29, 0x20, 0x2d, 0x20, 0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x20, 0x69,
	0x6e, 0x20, 0x44, 0x57, 0x73, 0x2e, 0x20, 0x28, 0x36, 0x29, 0x20, 0x76,
	0x61, 0x6c, 0x69, 0x64, 0x20, 0x62, 0x69, 0x74, 0x2e, 0x00,
	/* PGLUE_B_REG_VF_LENGTH_VIOLATION_DETAILS2 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x56, 0x46, 0x5f, 0x4c, 0x45, 0x4e, 0x47, 0x54, 0x48, 0x5f, 0x56, 0x49,
	0x4f, 0x4c, 0x41, 0x54, 0x49, 0x4f, 0x4e, 0x5f, 0x44, 0x45, 0x54, 0x41,
	0x49, 0x4c, 0x53, 0x32, 0x00,
	/* PGLUE_B: Target VF GRC space access failed permission check. (14:0) Address. (15) w_nr: 0 - Read; 1 - Write. (23:16) VFID. (27:24) - PFID. (28) - valid bit. */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x54, 0x61, 0x72,
	0x67, 0x65, 0x74, 0x20, 0x56, 0x46, 0x20, 0x47, 0x52, 0x43, 0x20, 0x73,
	0x70, 0x61, 0x63, 0x65, 0x20, 0x61, 0x63, 0x63, 0x65, 0x73, 0x73, 0x20,
	0x66, 0x61, 0x69, 0x6c, 0x65, 0x64, 0x20, 0x70, 0x65, 0x72, 0x6d, 0x69,
	0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20, 0x63, 0x68, 0x65, 0x63, 0x6b, 0x2e,
	0x20, 0x28, 0x31, 0x34, 0x3a, 0x30, 0x29, 0x20, 0x41, 0x64, 0x64, 0x72,
	0x65, 0x73, 0x73, 0x2e, 0x20, 0x28, 0x31, 0x35, 0x29, 0x20, 0x77, 0x5f,
	0x6e, 0x72, 0x3a, 0x20, 0x30, 0x20, 0x2d, 0x20, 0x52, 0x65, 0x61, 0x64,
	0x3b, 0x20, 0x31, 0x20, 0x2d, 0x20, 0x57, 0x72, 0x69, 0x74, 0x65, 0x2e,
	0x20, 0x28, 0x32, 0x33, 0x3a, 0x31, 0x36, 0x29, 0x20, 0x56, 0x46, 0x49,
	0x44, 0x2e, 0x20, 0x28, 0x32, 0x37, 0x3a, 0x32, 0x34, 0x29, 0x20, 0x2d,
	0x20, 0x50, 0x46, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x32, 0x38, 0x29, 0x20,
	0x2d, 0x20, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x20, 0x62, 0x69, 0x74, 0x2e,
	0x00,
	/* PGLUE_B_REG_VF_GRC_SPACE_VIOLATION_DETAILS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x56, 0x46, 0x5f, 0x47, 0x52, 0x43, 0x5f, 0x53, 0x50, 0x41, 0x43, 0x45,
	0x5f, 0x56, 0x49, 0x4f, 0x4c, 0x41, 0x54, 0x49, 0x4f, 0x4e, 0x5f, 0x44,
	0x45, 0x54, 0x41, 0x49, 0x4c, 0x53, 0x00,
	/* PGLUE: PCIE error status is set */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x3a, 0x20, 0x50, 0x43, 0x49, 0x45, 0x20,
	0x65, 0x72, 0x72, 0x6f, 0x72, 0x20, 0x73, 0x74, 0x61, 0x74, 0x75, 0x73,
	0x20, 0x69, 0x73, 0x20, 0x73, 0x65, 0x74, 0x00,
	/* PGLUE_B_REG_PCIE_ERR_STATUS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x50, 0x43, 0x49, 0x45, 0x5f, 0x45, 0x52, 0x52, 0x5f, 0x53, 0x54, 0x41,
	0x54, 0x55, 0x53, 0x00,
	/* Details of first read request with length = 0. (4:0) VQID. (5) Is TREQ. (9:6) - PFID. (10) - VF_VALID. (18:11) - VFID. (23:19) client ID. (24) - last SR. (25) valid. */
	0x44, 0x65, 0x74, 0x61, 0x69, 0x6c, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x66,
	0x69, 0x72, 0x73, 0x74, 0x20, 0x72, 0x65, 0x61, 0x64, 0x20, 0x72, 0x65,
	0x71, 0x75, 0x65, 0x73, 0x74, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x6c,
	0x65, 0x6e, 0x67, 0x74, 0x68, 0x20, 0x3d, 0x20, 0x30, 0x2e, 0x20, 0x28,
	0x34, 0x3a, 0x30, 0x29, 0x20, 0x56, 0x51, 0x49, 0x44, 0x2e, 0x20, 0x28,
	0x35, 0x29, 0x20, 0x49, 0x73, 0x20, 0x54, 0x52, 0x45, 0x51, 0x2e, 0x20,
	0x28, 0x39, 0x3a, 0x36, 0x29, 0x20, 0x2d, 0x20, 0x50, 0x46, 0x49, 0x44,
	0x2e, 0x20, 0x28, 0x31, 0x30, 0x29, 0x20, 0x2d, 0x20, 0x56, 0x46, 0x5f,
	0x56, 0x41, 0x4c, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x31, 0x38, 0x3a, 0x31,
	0x31, 0x29, 0x20, 0x2d, 0x20, 0x56, 0x46, 0x49, 0x44, 0x2e, 0x20, 0x28,
	0x32, 0x33, 0x3a, 0x31, 0x39, 0x29, 0x20, 0x63, 0x6c, 0x69, 0x65, 0x6e,
	0x74, 0x20, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x32, 0x34, 0x29, 0x20, 0x2d,
	0x20, 0x6c, 0x61, 0x73, 0x74, 0x20, 0x53, 0x52, 0x2e, 0x20, 0x28, 0x32,
	0x35, 0x29, 0x20, 0x76, 0x61, 0x6c, 0x69, 0x64, 0x2e, 0x00,
	/* PGLUE_B_REG_MASTER_ZLR_ERR_DETAILS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x4d, 0x41, 0x53, 0x54, 0x45, 0x52, 0x5f, 0x5a, 0x4c, 0x52, 0x5f, 0x45,
	0x52, 0x52, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x53, 0x00,
	/* PGLUE_B_REG_MASTER_ZLR_ERR_ADD_63_32 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x4d, 0x41, 0x53, 0x54, 0x45, 0x52, 0x5f, 0x5a, 0x4c, 0x52, 0x5f, 0x45,
	0x52, 0x52, 0x5f, 0x41, 0x44, 0x44, 0x5f, 0x36, 0x33, 0x5f, 0x33, 0x32,
	0x00,
	/* PGLUE_B_REG_MASTER_ZLR_ERR_ADD_31_0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x4d, 0x41, 0x53, 0x54, 0x45, 0x52, 0x5f, 0x5a, 0x4c, 0x52, 0x5f, 0x45,
	0x52, 0x52, 0x5f, 0x41, 0x44, 0x44, 0x5f, 0x33, 0x31, 0x5f, 0x30, 0x00,
	/* Details of first target Read Write access to the admin window that have a length bigger than 1DW or first byte enable != 0xf . (9:0) Address in DWs (bits (11:2) of byte address). (13:10) BE first. (17:14) BE last. (21:18) - PFID. (27:22) - Length in DWs. (28) valid bit. */
	0x44, 0x65, 0x74, 0x61, 0x69, 0x6c, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x66,
	0x69, 0x72, 0x73, 0x74, 0x20, 0x74, 0x61, 0x72, 0x67, 0x65, 0x74, 0x20,
	0x52, 0x65, 0x61, 0x64, 0x20, 0x57, 0x72, 0x69, 0x74, 0x65, 0x20, 0x61,
	0x63, 0x63, 0x65, 0x73, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x74, 0x68, 0x65,
	0x20, 0x61, 0x64, 0x6d, 0x69, 0x6e, 0x20, 0x77, 0x69, 0x6e, 0x64, 0x6f,
	0x77, 0x20, 0x74, 0x68, 0x61, 0x74, 0x20, 0x68, 0x61, 0x76, 0x65, 0x20,
	0x61, 0x20, 0x6c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x20, 0x62, 0x69, 0x67,
	0x67, 0x65, 0x72, 0x20, 0x74, 0x68, 0x61, 0x6e, 0x20, 0x31, 0x44, 0x57,
	0x20, 0x6f, 0x72, 0x20, 0x66, 0x69, 0x72, 0x73, 0x74, 0x20, 0x62, 0x79,
	0x74, 0x65, 0x20, 0x65, 0x6e, 0x61, 0x62, 0x6c, 0x65, 0x20, 0x21, 0x3d,
	0x20, 0x30, 0x78, 0x66, 0x20, 0x2e, 0x20, 0x28, 0x39, 0x3a, 0x30, 0x29,
	0x20, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x20, 0x69, 0x6e, 0x20,
	0x44, 0x57, 0x73, 0x20, 0x28, 0x62, 0x69, 0x74, 0x73, 0x20, 0x28, 0x31,
	0x31, 0x3a, 0x32, 0x29, 0x20, 0x6f, 0x66, 0x20, 0x62, 0x79, 0x74, 0x65,
	0x20, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x29, 0x2e, 0x20, 0x28,
	0x31, 0x33, 0x3a, 0x31, 0x30, 0x29, 0x20, 0x42, 0x45, 0x20, 0x66, 0x69,
	0x72, 0x73, 0x74, 0x2e, 0x20, 0x28, 0x31, 0x37, 0x3a, 0x31, 0x34, 0x29,
	0x20, 0x42, 0x45, 0x20, 0x6c, 0x61, 0x73, 0x74, 0x2e, 0x20, 0x28, 0x32,
	0x31, 0x3a, 0x31, 0x38, 0x29, 0x20, 0x2d, 0x20, 0x50, 0x46, 0x49, 0x44,
	0x2e, 0x20, 0x28, 0x32, 0x37, 0x3a, 0x32, 0x32, 0x29, 0x20, 0x2d, 0x20,
	0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x20, 0x69, 0x6e, 0x20, 0x44, 0x57,
	0x73, 0x2e, 0x20, 0x28, 0x32, 0x38, 0x29, 0x20, 0x76, 0x61, 0x6c, 0x69,
	0x64, 0x20, 0x62, 0x69, 0x74, 0x2e, 0x00,
	/* PGLUE_B_REG_ADMIN_WINDOW_VIOLATION_DETAILS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x41, 0x44, 0x4d, 0x49, 0x4e, 0x5f, 0x57, 0x49, 0x4e, 0x44, 0x4f, 0x57,
	0x5f, 0x56, 0x49, 0x4f, 0x4c, 0x41, 0x54, 0x49, 0x4f, 0x4e, 0x5f, 0x44,
	0x45, 0x54, 0x41, 0x49, 0x4c, 0x53, 0x00,
	/* Details of first target Read/Write access where pretend register contains an out of range function. (3:0) - original PFID. (7:4) Pretend PFID. (15:8) Pretend VFID. (16) Pretend vf_valid. (20:17) Pretend register: 0-11 - One of the PF windows pretend. 12 - global pretend register. (21) valid bit. */
	0x44, 0x65, 0x74, 0x61, 0x69, 0x6c, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x66,
	0x69, 0x72, 0x73, 0x74, 0x20, 0x74, 0x61, 0x72, 0x67, 0x65, 0x74, 0x20,
	0x52, 0x65, 0x61, 0x64, 0x2f, 0x57, 0x72, 0x69, 0x74, 0x65, 0x20, 0x61,
	0x63, 0x63, 0x65, 0x73, 0x73, 0x20, 0x77, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x70, 0x72, 0x65, 0x74, 0x65, 0x6e, 0x64, 0x20, 0x72, 0x65, 0x67, 0x69,
	0x73, 0x74, 0x65, 0x72, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x61, 0x69, 0x6e,
	0x73, 0x20, 0x61, 0x6e, 0x20, 0x6f, 0x75, 0x74, 0x20, 0x6f, 0x66, 0x20,
	0x72, 0x61, 0x6e, 0x67, 0x65, 0x20, 0x66, 0x75, 0x6e, 0x63, 0x74, 0x69,
	0x6f, 0x6e, 0x2e, 0x20, 0x28, 0x33, 0x3a, 0x30, 0x29, 0x20, 0x2d, 0x20,
	0x6f, 0x72, 0x69, 0x67, 0x69, 0x6e, 0x61, 0x6c, 0x20, 0x50, 0x46, 0x49,
	0x44, 0x2e, 0x20, 0x28, 0x37, 0x3a, 0x34, 0x29, 0x20, 0x50, 0x72, 0x65,
	0x74, 0x65, 0x6e, 0x64, 0x20, 0x50, 0x46, 0x49, 0x44, 0x2e, 0x20, 0x28,
	0x31, 0x35, 0x3a, 0x38, 0x29, 0x20, 0x50, 0x72, 0x65, 0x74, 0x65, 0x6e,
	0x64, 0x20, 0x56, 0x46, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x31, 0x36, 0x29,
	0x20, 0x50, 0x72, 0x65, 0x74, 0x65, 0x6e, 0x64, 0x20, 0x76, 0x66, 0x5f,
	0x76, 0x61, 0x6c, 0x69, 0x64, 0x2e, 0x20, 0x28, 0x32, 0x30, 0x3a, 0x31,
	0x37, 0x29, 0x20, 0x50, 0x72, 0x65, 0x74, 0x65, 0x6e, 0x64, 0x20, 0x72,
	0x65, 0x67, 0x69, 0x73, 0x74, 0x65, 0x72, 0x3a, 0x20, 0x30, 0x2d, 0x31,
	0x31, 0x20, 0x2d, 0x20, 0x4f, 0x6e, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x74,
	0x68, 0x65, 0x20, 0x50, 0x46, 0x20, 0x77, 0x69, 0x6e, 0x64, 0x6f, 0x77,
	0x73, 0x20, 0x70, 0x72, 0x65, 0x74, 0x65, 0x6e, 0x64, 0x2e, 0x20, 0x31,
	0x32, 0x20, 0x2d, 0x20, 0x67, 0x6c, 0x6f, 0x62, 0x61, 0x6c, 0x20, 0x70,
	0x72, 0x65, 0x74, 0x65, 0x6e, 0x64, 0x20, 0x72, 0x65, 0x67, 0x69, 0x73,
	0x74, 0x65, 0x72, 0x2e, 0x20, 0x28, 0x32, 0x31, 0x29, 0x20, 0x76, 0x61,
	0x6c, 0x69, 0x64, 0x20, 0x62, 0x69, 0x74, 0x2e, 0x00,
	/* PGLUE_B_REG_OUT_OF_RANGE_FUNCTION_IN_PRETEND_DETAILS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x4f, 0x55, 0x54, 0x5f, 0x4f, 0x46, 0x5f, 0x52, 0x41, 0x4e, 0x47, 0x45,
	0x5f, 0x46, 0x55, 0x4e, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x5f, 0x49, 0x4e,
	0x5f, 0x50, 0x52, 0x45, 0x54, 0x45, 0x4e, 0x44, 0x5f, 0x44, 0x45, 0x54,
	0x41, 0x49, 0x4c, 0x53, 0x00,
	/* Address of first target Read Write access where pretend register contains an out of range function. */
	0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x20, 0x6f, 0x66, 0x20, 0x66,
	0x69, 0x72, 0x73, 0x74, 0x20, 0x74, 0x61, 0x72, 0x67, 0x65, 0x74, 0x20,
	0x52, 0x65, 0x61, 0x64, 0x20, 0x57, 0x72, 0x69, 0x74, 0x65, 0x20, 0x61,
	0x63, 0x63, 0x65, 0x73, 0x73, 0x20, 0x77, 0x68, 0x65, 0x72, 0x65, 0x20,
	0x70, 0x72, 0x65, 0x74, 0x65, 0x6e, 0x64, 0x20, 0x72, 0x65, 0x67, 0x69,
	0x73, 0x74, 0x65, 0x72, 0x20, 0x63, 0x6f, 0x6e, 0x74, 0x61, 0x69, 0x6e,
	0x73, 0x20, 0x61, 0x6e, 0x20, 0x6f, 0x75, 0x74, 0x20, 0x6f, 0x66, 0x20,
	0x72, 0x61, 0x6e, 0x67, 0x65, 0x20, 0x66, 0x75, 0x6e, 0x63, 0x74, 0x69,
	0x6f, 0x6e, 0x2e, 0x00,
	/* PGLUE_B_REG_OUT_OF_RANGE_FUNCTION_IN_PRETEND_ADDRESS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x4f, 0x55, 0x54, 0x5f, 0x4f, 0x46, 0x5f, 0x52, 0x41, 0x4e, 0x47, 0x45,
	0x5f, 0x46, 0x55, 0x4e, 0x43, 0x54, 0x49, 0x4f, 0x4e, 0x5f, 0x49, 0x4e,
	0x5f, 0x50, 0x52, 0x45, 0x54, 0x45, 0x4e, 0x44, 0x5f, 0x41, 0x44, 0x44,
	0x52, 0x45, 0x53, 0x53, 0x00,
	/* PGL TX Write FIFO is not empty. */
	0x50, 0x47, 0x4c, 0x20, 0x54, 0x58, 0x20, 0x57, 0x72, 0x69, 0x74, 0x65,
	0x20, 0x46, 0x49, 0x46, 0x4f, 0x20, 0x69, 0x73, 0x20, 0x6e, 0x6f, 0x74,
	0x20, 0x65, 0x6d, 0x70, 0x74, 0x79, 0x2e, 0x00,
	/* PGLUE_B_REG_WRITE_FIFO_OCCUPANCY_LEVEL */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x57, 0x52, 0x49, 0x54, 0x45, 0x5f, 0x46, 0x49, 0x46, 0x4f, 0x5f, 0x4f,
	0x43, 0x43, 0x55, 0x50, 0x41, 0x4e, 0x43, 0x59, 0x5f, 0x4c, 0x45, 0x56,
	0x45, 0x4c, 0x00,
	/* PGLUE_B: Illegal address in dev->host flow. (4:0) VQID. (5) - first SR. (18:6) - Length in bytes. (19) - VF_VALID. (23:20) - PFID. (31:24) - VFID. */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x49, 0x6c, 0x6c,
	0x65, 0x67, 0x61, 0x6c, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73,
	0x20, 0x69, 0x6e, 0x20, 0x64, 0x65, 0x76, 0x2d, 0x3e, 0x68, 0x6f, 0x73,
	0x74, 0x20, 0x66, 0x6c, 0x6f, 0x77, 0x2e, 0x20, 0x28, 0x34, 0x3a, 0x30,
	0x29, 0x20, 0x56, 0x51, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x35, 0x29, 0x20,
	0x2d, 0x20, 0x66, 0x69, 0x72, 0x73, 0x74, 0x20, 0x53, 0x52, 0x2e, 0x20,
	0x28, 0x31, 0x38, 0x3a, 0x36, 0x29, 0x20, 0x2d, 0x20, 0x4c, 0x65, 0x6e,
	0x67, 0x74, 0x68, 0x20, 0x69, 0x6e, 0x20, 0x62, 0x79, 0x74, 0x65, 0x73,
	0x2e, 0x20, 0x28, 0x31, 0x39, 0x29, 0x20, 0x2d, 0x20, 0x56, 0x46, 0x5f,
	0x56, 0x41, 0x4c, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x32, 0x33, 0x3a, 0x32,
	0x30, 0x29, 0x20, 0x2d, 0x20, 0x50, 0x46, 0x49, 0x44, 0x2e, 0x20, 0x28,
	0x33, 0x31, 0x3a, 0x32, 0x34, 0x29, 0x20, 0x2d, 0x20, 0x56, 0x46, 0x49,
	0x44, 0x2e, 0x00,
	/* PGLUE_B_REG_ILLEGAL_ADDRESS_DETAILS */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x49, 0x4c, 0x4c, 0x45, 0x47, 0x41, 0x4c, 0x5f, 0x41, 0x44, 0x44, 0x52,
	0x45, 0x53, 0x53, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x53, 0x00,
	/* PGLUE_B_REG_ILLEGAL_ADDRESS_ADD_63_32 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x49, 0x4c, 0x4c, 0x45, 0x47, 0x41, 0x4c, 0x5f, 0x41, 0x44, 0x44, 0x52,
	0x45, 0x53, 0x53, 0x5f, 0x41, 0x44, 0x44, 0x5f, 0x36, 0x33, 0x5f, 0x33,
	0x32, 0x00,
	/* PGLUE_B_REG_ILLEGAL_ADDRESS_ADD_31_0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x49, 0x4c, 0x4c, 0x45, 0x47, 0x41, 0x4c, 0x5f, 0x41, 0x44, 0x44, 0x52,
	0x45, 0x53, 0x53, 0x5f, 0x41, 0x44, 0x44, 0x5f, 0x33, 0x31, 0x5f, 0x30,
	0x00,
	/* PGLUE_B: Illegal address in dev->host flow. (15:0) Request ID. (20:16) client ID. (21) Illegal address cause: 0 - address was smaller than minimal_address_log;  1 - address was bigger than maximal_address_log. (22) - write_n_read: 0 - read; 1 - write. (23) last SR. (24) valid bit. */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x49, 0x6c, 0x6c,
	0x65, 0x67, 0x61, 0x6c, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73,
	0x20, 0x69, 0x6e, 0x20, 0x64, 0x65, 0x76, 0x2d, 0x3e, 0x68, 0x6f, 0x73,
	0x74, 0x20, 0x66, 0x6c, 0x6f, 0x77, 0x2e, 0x20, 0x28, 0x31, 0x35, 0x3a,
	0x30, 0x29, 0x20, 0x52, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x20, 0x49,
	0x44, 0x2e, 0x20, 0x28, 0x32, 0x30, 0x3a, 0x31, 0x36, 0x29, 0x20, 0x63,
	0x6c, 0x69, 0x65, 0x6e, 0x74, 0x20, 0x49, 0x44, 0x2e, 0x20, 0x28, 0x32,
	0x31, 0x29, 0x20, 0x49, 0x6c, 0x6c, 0x65, 0x67, 0x61, 0x6c, 0x20, 0x61,
	0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x20, 0x63, 0x61, 0x75, 0x73, 0x65,
	0x3a, 0x20, 0x30, 0x20, 0x2d, 0x20, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73,
	0x73, 0x20, 0x77, 0x61, 0x73, 0x20, 0x73, 0x6d, 0x61, 0x6c, 0x6c, 0x65,
	0x72, 0x20, 0x74, 0x68, 0x61, 0x6e, 0x20, 0x6d, 0x69, 0x6e, 0x69, 0x6d,
	0x61, 0x6c, 0x5f, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x5f, 0x6c,
	0x6f, 0x67, 0x3b, 0x20, 0x20, 0x31, 0x20, 0x2d, 0x20, 0x61, 0x64, 0x64,
	0x72, 0x65, 0x73, 0x73, 0x20, 0x77, 0x61, 0x73, 0x20, 0x62, 0x69, 0x67,
	0x67, 0x65, 0x72, 0x20, 0x74, 0x68, 0x61, 0x6e, 0x20, 0x6d, 0x61, 0x78,
	0x69, 0x6d, 0x61, 0x6c, 0x5f, 0x61, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73,
	0x5f, 0x6c, 0x6f, 0x67, 0x2e, 0x20, 0x28, 0x32, 0x32, 0x29, 0x20, 0x2d,
	0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x5f, 0x6e, 0x5f, 0x72, 0x65, 0x61,
	0x64, 0x3a, 0x20, 0x30, 0x20, 0x2d, 0x20, 0x72, 0x65, 0x61, 0x64, 0x3b,
	0x20, 0x31, 0x20, 0x2d, 0x20, 0x77, 0x72, 0x69, 0x74, 0x65, 0x2e, 0x20,
	0x28, 0x32, 0x33, 0x29, 0x20, 0x6c, 0x61, 0x73, 0x74, 0x20, 0x53, 0x52,
	0x2e, 0x20, 0x28, 0x32, 0x34, 0x29, 0x20, 0x76, 0x61, 0x6c, 0x69, 0x64,
	0x20, 0x62, 0x69, 0x74, 0x2e, 0x00,
	/* PGLUE_B_REG_ILLEGAL_ADDRESS_DETAILS2 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x49, 0x4c, 0x4c, 0x45, 0x47, 0x41, 0x4c, 0x5f, 0x41, 0x44, 0x44, 0x52,
	0x45, 0x53, 0x53, 0x5f, 0x44, 0x45, 0x54, 0x41, 0x49, 0x4c, 0x53, 0x32,
	0x00,
	/* PGLUE_B: There are outstanding read requests for tags 0-31. Not all completions have arrived for read requests on tags that are marked with 0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x54, 0x68, 0x65,
	0x72, 0x65, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74,
	0x61, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x72, 0x65, 0x61, 0x64, 0x20,
	0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x20, 0x66, 0x6f, 0x72,
	0x20, 0x74, 0x61, 0x67, 0x73, 0x20, 0x30, 0x2d, 0x33, 0x31, 0x2e, 0x20,
	0x4e, 0x6f, 0x74, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x6d, 0x70,
	0x6c, 0x65, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x68, 0x61, 0x76, 0x65,
	0x20, 0x61, 0x72, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x6f, 0x72,
	0x20, 0x72, 0x65, 0x61, 0x64, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65, 0x73,
	0x74, 0x73, 0x20, 0x6f, 0x6e, 0x20, 0x74, 0x61, 0x67, 0x73, 0x20, 0x74,
	0x68, 0x61, 0x74, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6d, 0x61, 0x72, 0x6b,
	0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x30, 0x00,
	/* PGLUE_B_REG_TAGS_31_0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x5f, 0x52, 0x45, 0x47, 0x5f,
	0x54, 0x41, 0x47, 0x53, 0x5f, 0x33, 0x31, 0x5f, 0x30, 0x00,
	/* PGLUE_B: There are outstanding read requests for tags 32-63. Not all completions have arrived for read requests on tags that are marked with 0 */
	0x50, 0x47, 0x4c, 0x55, 0x45, 0x5f, 0x42, 0x3a, 0x20, 0x54, 0x68, 0x65,
	0x72, 0x65, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6f, 0x75, 0x74, 0x73, 0x74,
	0x61, 0x6e, 0x64, 0x69, 0x6e, 0x67, 0x20, 0x72, 0x65, 0x61, 0x64, 0x20,
	0x72, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x73, 0x20, 0x66, 0x6f, 0x72,
	0x20, 0x74, 0x61, 0x67, 0x73, 0x20, 0x33, 0x32, 0x2d, 0x36, 0x33, 0x2e,
	0x20, 0x4e, 0x6f, 0x74, 0x20, 0x61, 0x6c, 0x6c, 0x20, 0x63, 0x6f, 0x6d,
	0x70, 0x6c, 0x65, 0x74, 0x69, 0x6f, 0x6e, 0x73, 0x20, 0x68, 0x61, 0x76,
	0x65, 0x20, 0x61, 0x72, 0x72, 0x69, 0x76, 0x65, 0x64, 0x20, 0x66, 0x6f,
	0x72, 0x20, 0x72, 0x65, 0x61, 0x64, 0x20, 0x72, 0x65, 0x71, 0x75, 0x65,
	0x73, 0x74, 0x73, 0x20, 0x6f, 0x6e, 0x20, 0x74, 0x61, 0x67, 0x73, 0x20,
	0x74, 0x68, 0x61, 0x74, 0x20, 0x61, 0x72, 0x65, 0x20, 0x6d, 0x61, 0x72,
	0x6b, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 