/*****************************************************************************/
/* Software Testing Automation Framework (STAF)                              */
/* (C) Copyright IBM Corp. 2001, 2004, 2005                                  */
/*                                                                           */
/* This software is licensed under the Eclipse Public License (EPL) V1.0.    */
/*****************************************************************************/

package com.ibm.staf.service.event;

import com.ibm.staf.*;
import com.ibm.staf.service.*;
import java.util.*;
import java.text.SimpleDateFormat;
import java.io.*;

/**********************************************************************/
/* STAF Event Service                                                 */
/* By Doug McWhorter and Benjamin Vera-Tudela (C) IBM 2000            */
/* Version 0.90                                                       */
/**********************************************************************/
/* Supplied Methods                                                   */
/* ------------------                                                 */
/* public int init(String name, String parms)                         */
/*                                                                    */
/* public int term()                                                  */
/*                                                                    */
/* public STAFResult acceptRequest(String machine, String process,    */
/*                                     int handle, String request)    */
/*                                                                    */
/* synchronized private void serialize(String theFileName)            */
/*                                                                    */
/* synchronized private void deSerialize(String theFileName)          */
/*                                                                    */
/**********************************************************************/

//  Class EventService administrates client requests to the Event Service.
//  Requests include:
//
//  REGISTER a process to be notified of an event of TYPE t, SUBTYPE s.
//  GENERATE an event of TYPE t, SUBTYPE s.
//  ACKNOWLEDGE that an event of TYPE t, SUBTYPE s has notified.
//  UNREGISTER a process that is currently registered for event notification.
//  QUERY information about processes that have generated an event by
//      EVENTID #.
//  LIST information about TYPES and SUBTYPES of events that processes are
//      registered to be notified about or lists information about events
//      that have been generated by EVENTID #, or lists information about
//      proesses that are registered to be notified about events (all or
//      by type, or by type/subtype).
//  VERSION: display the installed version of Event Service.
//  HELP

public class EventService implements STAFServiceInterfaceLevel30
{
    static final long serialVersionUID = 0;
    transient static final boolean DEBUG = false;
    transient final String kVersion = "3.1.5";

    // Version of STAF (or later) required for this service
    // STAF Version 3.1.0 or later is required so that the privacy methods
    // in STAFUtil are available.
    transient final String kRequiredSTAFVersion = "3.1.0";

    public static String lineSep;
    private static String sHelpMsg;

    // global error constants
    transient static final int kNoAckPending = 4001;
    transient static final int kNoSuchID = 4002;
    transient static final int kNotRegisteredForType = 4003;
    transient static final int kNotRegisteredForSubtype = 4004;
    transient static final int kNoClientsForEvent = 4005;

    // Default service initialization parameters
    private int fMaxAttempts = 1;
    private int fAckTimeout = 60000;
    private int fPriority = 5;
    private int fPriorityDelta = 1;

    transient private STAFHandle fHandle = null;
    transient private String fServiceName = null;
    transient private String fLocalMachineName = "";

    private final static SimpleDateFormat sTimestampFormat = 
        new SimpleDateFormat("yyyyMMdd-HH:mm:ss");

    // Directory where the Event Service writes data to
    transient private String fDataDir = null;
    transient private String fRegManagerFileName = null;
    transient private String fGenManagerFileName = null;

    // stores & processes registration information
    transient RegistrationManager fRegManager = null;

    // stores & processes information about events
    transient GenerationManager fGenManager = null;

    // a STAFCommandParser for each request's options
    transient STAFCommandParser fListParser = null;
    transient STAFCommandParser fQueryParser = null;
    transient STAFCommandParser fRegisterParser = null;
    transient STAFCommandParser fGenerateParser = null;
    transient STAFCommandParser fAckParser = null;
    transient STAFCommandParser fResetParser = null;
    transient STAFCommandParser fUnregisterParser = null;
    transient STAFCommandParser fVersionParser = null;

    // parses any initialization arguments
    transient STAFCommandParser fParmsParser = null;

    // Map Class Definitions used to create marshalled results
    public static STAFMapClassDefinition fTypeMapClass;
    public static STAFMapClassDefinition fEventGeneratorMapClass;
    private static STAFMapClassDefinition fListEventIDsMapClass;
    public static STAFMapClassDefinition fEventIDLongMapClass;
    private static STAFMapClassDefinition fQueryEventIDMapClass;
    public static STAFMapClassDefinition fNotifieeMapClass;
    private static STAFMapClassDefinition fListRegMapClass;
    private static STAFMapClassDefinition fListRegLongMapClass;
    private static STAFMapClassDefinition fSettingsMapClass;

    // default constructor
    public EventService()
    { /* Do Nothing */ }

    /***************************************************************/
    /* init - Initializes Event Service with STAF.  Instantiates   */
    /* all parsers, fGenManager (deserializes fGenManager.EventID) */
    /* and fRegManager (Deserializes all registration data from    */
    /* file.                                                       */
    /*                                                             */
    /* accepts: STAFServiceInterfaceLevel30 initialization info    */
    /*                                                             */
    /* Returns: STAFResult.Ok or STAFResult.STAFRegistrationError  */
    /***************************************************************/
    public STAFResult init(STAFServiceInterfaceLevel30.InitInfo info)
    {
        try
        {
            // instantiate parsers as not case sensitive
            fRegisterParser = new STAFCommandParser(0, false);
            fGenerateParser = new STAFCommandParser(0, false);
            fAckParser = new STAFCommandParser(0, false);
            fResetParser = new STAFCommandParser(0, false);
            fUnregisterParser = new STAFCommandParser(0, false);
            fListParser = new STAFCommandParser(0, false);
            fQueryParser = new STAFCommandParser(0, false);
            fVersionParser = new STAFCommandParser(0, false);
            fParmsParser = new STAFCommandParser(0, false);

            //initialize all parsers to accept request options
            fParmsParser.addOption("MAXATTEMPTS", 1,
                                   STAFCommandParser.VALUEALLOWED);
            fParmsParser.addOption("ACKNOWLEDGETIMEOUT", 1,
                                   STAFCommandParser.VALUEALLOWED);
            fParmsParser.addOption("PRIORITY", 1,
                                   STAFCommandParser.VALUEALLOWED);
            fParmsParser.addOption("PRIORITYDELTA", 1,
                                   STAFCommandParser.VALUEALLOWED);

            fRegisterParser.addOption("REGISTER", 1,
                                      STAFCommandParser.VALUENOTALLOWED);
            fRegisterParser.addOption("TYPE", 1,
                                      STAFCommandParser.VALUEREQUIRED);
            fRegisterParser.addOption("SUBTYPE", 0,
                                      STAFCommandParser.VALUEREQUIRED);
            fRegisterParser.addOption("ACKNOWLEDGETIMEOUT", 1,
                                      STAFCommandParser.VALUEREQUIRED);
            fRegisterParser.addOption("MAXATTEMPTS", 1,
                                      STAFCommandParser.VALUEREQUIRED);
            fRegisterParser.addOption("PRIORITY", 1,
                                      STAFCommandParser.VALUEREQUIRED);
            fRegisterParser.addOption("PRIORITYDELTA", 1,
                                      STAFCommandParser.VALUEREQUIRED);
            fRegisterParser.addOption("BYHANDLE", 1,
                                      STAFCommandParser.VALUENOTALLOWED);
            fRegisterParser.addOption("BYNAME", 1,
                                      STAFCommandParser.VALUENOTALLOWED);

            fRegisterParser.addOptionGroup("BYHANDLE BYNAME", 0, 1);

            fRegisterParser.addOptionNeed("REGISTER", "TYPE");

            fGenerateParser.addOption("GENERATE", 1,
                                      STAFCommandParser.VALUENOTALLOWED);

            fGenerateParser.addOption("TYPE", 1,
                                      STAFCommandParser.VALUEREQUIRED);
            fGenerateParser.addOption("SUBTYPE", 1,
                                      STAFCommandParser.VALUEREQUIRED);
            fGenerateParser.addOption("PROPERTY", 0,
                                      STAFCommandParser.VALUEREQUIRED);
            fGenerateParser.addOption("NOTIFY", 1,
                                      STAFCommandParser.VALUENOTALLOWED);
            // ASYNC is an undocumented option, used only by the STAX service
            fGenerateParser.addOption("ASYNC", 1,
                                      STAFCommandParser.VALUENOTALLOWED);

            fGenerateParser.addOptionNeed("GENERATE", "TYPE, SUBTYPE");

            fGenerateParser.addOptionGroup("TYPE SUBTYPE", 2, 2);

            fAckParser.addOption("ACKNOWLEDGE", 1,
                                 STAFCommandParser.VALUENOTALLOWED);

            fAckParser.addOption("EVENTID", 1,
                                 STAFCommandParser.VALUEREQUIRED);
            fAckParser.addOption("FORCE", 1,
                                 STAFCommandParser.VALUENOTALLOWED);
            fAckParser.addOption("MACHINE", 1,
                                 STAFCommandParser.VALUEREQUIRED);
            fAckParser.addOption("NAME", 1,
                                 STAFCommandParser.VALUEREQUIRED);
            fAckParser.addOption("HANDLE", 1,
                                 STAFCommandParser.VALUEREQUIRED);
            fAckParser.addOptionGroup("EVENTID", 1, 1);

            fAckParser.addOptionNeed("MACHINE", "FORCE");
            fAckParser.addOptionNeed("HANDLE", "FORCE");
            fAckParser.addOptionNeed("NAME", "FORCE");
            fAckParser.addOptionNeed("ACKNOWLEDGE", "EVENTID");

            fResetParser.addOption("RESET", 1,
                                 STAFCommandParser.VALUENOTALLOWED);

            fResetParser.addOption("GEN", 1,
                                 STAFCommandParser.VALUENOTALLOWED);
            fResetParser.addOption("REG", 1,
                                 STAFCommandParser.VALUENOTALLOWED);
            fResetParser.addOption("FORCE", 1,
                                 STAFCommandParser.VALUENOTALLOWED);
            fResetParser.addOptionGroup("GEN REG", 1, 1);

            fUnregisterParser.addOption("UNREGISTER", 1,
                                        STAFCommandParser.VALUENOTALLOWED);
            fUnregisterParser.addOption("TYPE", 1,
                                        STAFCommandParser.VALUEREQUIRED);
            fUnregisterParser.addOption("SUBTYPE", 0,
                                        STAFCommandParser.VALUEREQUIRED);
            fUnregisterParser.addOption("BYNAME", 1,
                                        STAFCommandParser.VALUENOTALLOWED);
            fUnregisterParser.addOption("FORCE", 1,
                                        STAFCommandParser.VALUENOTALLOWED);
            fUnregisterParser.addOption("MACHINE", 1,
                                        STAFCommandParser.VALUEREQUIRED);
            fUnregisterParser.addOption("NAME", 1,
                                        STAFCommandParser.VALUEREQUIRED);
            fUnregisterParser.addOption("HANDLE", 1,
                                        STAFCommandParser.VALUEREQUIRED);

            fUnregisterParser.addOptionGroup("NAME HANDLE", 0, 1);
            fUnregisterParser.addOptionGroup("TYPE", 1, 1);
            fUnregisterParser.addOptionNeed("UNREGISTER", "TYPE");

            fUnregisterParser.addOptionNeed("SUBTYPE", "TYPE");
            fUnregisterParser.addOptionNeed("FORCE", "TYPE");
            fUnregisterParser.addOptionNeed("MACHINE", "FORCE");
            fUnregisterParser.addOptionNeed("HANDLE", "FORCE");
            fUnregisterParser.addOptionNeed("NAME", "FORCE");

            fUnregisterParser.addOptionGroup("FORCE", 0, 1);

            fListParser.addOption("LIST", 1,
                                  STAFCommandParser.VALUENOTALLOWED);
            fListParser.addOption("TYPES", 1,
                                  STAFCommandParser.VALUENOTALLOWED);
            fListParser.addOption("SUBTYPES", 1,
                                  STAFCommandParser.VALUENOTALLOWED);
            fListParser.addOption("TYPE", 1,
                                  STAFCommandParser.VALUEREQUIRED);
            fListParser.addOption("SUBTYPE", 1,
                                  STAFCommandParser.VALUEREQUIRED);
            fListParser.addOption("REGISTRATIONS", 1,
                                  STAFCommandParser.VALUENOTALLOWED);
            fListParser.addOption("EVENTIDS", 1,
                                  STAFCommandParser.VALUENOTALLOWED);
            fListParser.addOption("LONG", 1,
                                  STAFCommandParser.VALUENOTALLOWED);
            fListParser.addOption("SETTINGS", 1,
                                  STAFCommandParser.VALUENOTALLOWED);

            fListParser.addOptionGroup(
                "TYPES SUBTYPES REGISTRATIONS EVENTIDS SETTINGS", 1, 1);
            fListParser.addOptionNeed("LONG", "TYPES EVENTIDS REGISTRATIONS");
            fListParser.addOptionNeed("TYPE", "SUBTYPES REGISTRATIONS EVENTIDS");
            fListParser.addOptionNeed("SUBTYPE", "TYPE");
            fListParser.addOptionNeed("SUBTYPES", "TYPE");

            fQueryParser.addOption("QUERY", 1,
                                   STAFCommandParser.VALUENOTALLOWED);

            fQueryParser.addOption("EVENTID", 1,
                                   STAFCommandParser.VALUEREQUIRED);
            fQueryParser.addOption("LONG", 1,
                                   STAFCommandParser.VALUENOTALLOWED);

            fQueryParser.addOptionNeed("LONG", "EVENTID");
            fQueryParser.addOptionNeed("QUERY", "EVENTID");

            fVersionParser.addOption("VERSION", 1,
                                     STAFCommandParser.VALUENOTALLOWED);

            fHandle = new STAFHandle("STAF/SERVICE/" + info.name);
            fServiceName = info.name;

            // Construct map class for a LIST EVENTIDS request.
            // List each event ID that has not yet been acknowledged and
            // the number of pending notifications (notifiees that have been
            // notified about the event but have not yet acknowledged).

            fListEventIDsMapClass = new STAFMapClassDefinition(
                "STAF/Service/Event/EventID");
            fListEventIDsMapClass.addKey("eventID",      "Event ID");
            fListEventIDsMapClass.addKey("type",         "Type");
            fListEventIDsMapClass.addKey("subtype",      "Subtype");
            fListEventIDsMapClass.addKey("numNotifiees", "# Notifiees");

            // Construct map class for a LIST EVENTIDS LONG request and for a
            // QUERY EVENTID <Event ID> LONG request.
            // List all event IDs, showing the who generated the event as well
            // as a list of the pending notifications (notifiees that have been
            // notified about the event but have not yet acknowledged).

            fEventIDLongMapClass = new STAFMapClassDefinition(
                "STAF/Service/Event/EventIDLong");
            fEventIDLongMapClass.addKey("eventID",      "Event ID");
            fEventIDLongMapClass.addKey("type",         "Type");
            fEventIDLongMapClass.addKey("subtype",      "Subtype");
            fEventIDLongMapClass.addKey("propertyMap",  "Properties");
            fEventIDLongMapClass.addKey("generatedBy",  "Generated By");
            fEventIDLongMapClass.addKey("notificationList",
                                        "Pending Notifications");

            // Construct map class for a QUERY EVENTID <Event ID> request.
            // Query the event ID, showing the number of pending notifications
            // (notifiees that have been notified about the event but have not
            // yet acknowledged).

            fQueryEventIDMapClass = new STAFMapClassDefinition(
                "STAF/Service/Event/QueryEventID");
            fQueryEventIDMapClass.addKey("eventID",      "Event ID");
            fQueryEventIDMapClass.addKey("type",         "Type");
            fQueryEventIDMapClass.addKey("subtype",      "Subtype");
            fQueryEventIDMapClass.addKey("propertyMap",  "Properties");
            fQueryEventIDMapClass.addKey("generatedBy",  "Generated By");
            fQueryEventIDMapClass.addKey("numNotifiees", "# Notifiees");

            // Construct map class for the generator of an event ID (used by
            // a LIST EVENTIDS LONG request (for the generatedBy field).

            fEventGeneratorMapClass = new STAFMapClassDefinition(
                "STAF/Service/Event/Generator");
            fEventGeneratorMapClass.addKey("machine", "Machine");
            fEventGeneratorMapClass.addKey("handleName", "Handle Name");
            fEventGeneratorMapClass.addKey("handle", "Handle");

            // Construct map class for an event pending notification (used by
            // a LIST EVENTIDS LONG request (for the notificationList field).

            fNotifieeMapClass = new STAFMapClassDefinition(
                "STAF/Service/Event/PendingNotification");
            fNotifieeMapClass.addKey("machine",       "Machine");
            fNotifieeMapClass.addKey("notifyBy",      "Notify By");
            fNotifieeMapClass.addKey("notifiee",      "Notifiee");
            fNotifieeMapClass.addKey("attempts",      "Attempts Remaining");
            fNotifieeMapClass.addKey("timeout",       "Timeout");
            fNotifieeMapClass.addKey("priority",      "Priority");
            fNotifieeMapClass.addKey("priorityDelta", "Priority Delta");

            // Construct map class for a LIST REGISTRATIONS request.
            // List all the registrations along with the types and subtypes
            // for which they are registered (in a short format).

            fListRegMapClass = new STAFMapClassDefinition(
                "STAF/Service/Event/ListRegistrations");
            fListRegMapClass.addKey("type",     "Type");
            fListRegMapClass.addKey("subtype",  "Subtype");
            fListRegMapClass.addKey("machine",  "Machine");
            fListRegMapClass.addKey("notifyBy", "Notify By");
            fListRegMapClass.addKey("notifiee", "Notifiee");

            // Construct map class for a LIST REGISTRATIONS LONG request
            // List the registrations along with the types and subtypes for
            // which they are registered (in a long format).

            fListRegLongMapClass = new STAFMapClassDefinition(
                "STAF/Service/Event/ListRegistrationsLong");
            fListRegLongMapClass.addKey("type",    "Type");
            fListRegLongMapClass.addKey("subtype", "Subtype");
            fListRegLongMapClass.addKey("machine",       "Machine");
            fListRegLongMapClass.addKey("notifyBy",      "Notify By");
            fListRegLongMapClass.addKey("notifiee",      "Notifiee");
            fListRegLongMapClass.addKey("attempts",      "Max Attempts");
            fListRegLongMapClass.setKeyProperty(
                "attempts", "display-short-name", "Att"); 
            fListRegLongMapClass.addKey("timeout",       "Timeout");
            fListRegLongMapClass.addKey("priority",      "Priority");
            fListRegLongMapClass.setKeyProperty(
                "priority", "display-short-name", "P");
            fListRegLongMapClass.addKey("priorityDelta", "Priority Delta");
            fListRegLongMapClass.setKeyProperty(
                "priorityDelta", "display-short-name", "D");

            // Construct map class for listing types and their subtypes

            fTypeMapClass = new STAFMapClassDefinition(
                "STAF/Service/Event/Type");
            fTypeMapClass.addKey("type",        "Type");
            fTypeMapClass.addKey("subtypeList", "Subtypes");
            
            // Construct map-class for list settings information

            fSettingsMapClass = new STAFMapClassDefinition(
                "STAF/Service/Event/Settings");
            fSettingsMapClass.addKey("maxAttempts", "Maximum Attempts");
            fSettingsMapClass.addKey("ackTimeout", "Acknowledgement Timeout");
            fSettingsMapClass.addKey("priority", "Priority");
            fSettingsMapClass.addKey("priorityDelta", "Priority Delta");

            STAFResult res = new STAFResult();

            // Resolve the line separator variable for the local machine

            res = STAFUtil.resolveInitVar("{STAF/Config/Sep/Line}", fHandle);

            if (res.rc != STAFResult.Ok) return res;

            lineSep = res.result;
            
            // Resolve the file separator variable for the local machine

            res = STAFUtil.resolveInitVar("{STAF/Config/Sep/File}", fHandle);

            if (res.rc != STAFResult.Ok) return res;

            String fileSep = res.result;

            // Resolve the machine name variable for the local machine

            res = STAFUtil.resolveInitVar("{STAF/Config/Machine}", fHandle);

            if (res.rc != STAFResult.Ok) return res;

            fLocalMachineName = res.result;

            // Verify that the parameters specified for the service are valid

            res = handleParms(info);

            if (res.rc != STAFResult.Ok)
            {
                return new STAFResult(
                    STAFResult.ServiceConfigurationError,
                    "Error validating parameters: RC=" + res.rc +
                    ", Result=" + res.result);
            }

            // Verify that the required version of STAF is running on the
            // local service machine.  
            // Note:  Method compareSTAFVersion was added in STAF V3.1.0

            try
            {
                res = STAFUtil.compareSTAFVersion(
                    "local", fHandle, kRequiredSTAFVersion);

                if (res.rc != STAFResult.Ok)
                {
                    if (res.rc == STAFResult.InvalidSTAFVersion)
                    {
                        return new STAFResult(
                            STAFResult.ServiceConfigurationError,
                            "Minimum required STAF version for this service " +
                            "is not running." + lineSep + res.result);
                    }
                    else
                    {
                        return new STAFResult(
                            STAFResult.ServiceConfigurationError,
                            "Error verifying the STAF version. RC: " + res.rc +
                            ", Additional info: " + res.result);
                    }
                }
            }
            catch (Error err)
            {
                return new STAFResult(
                    STAFResult.ServiceConfigurationError,
                    "This service requires STAF Version " +
                    kRequiredSTAFVersion + " or later."); 
            }

            // Store data for the event service in directory:
            //   <STAF writeLocation>/service/<event service name (lower-case)>

            fDataDir = info.writeLocation;

            if (!fDataDir.endsWith(fileSep))
            {
                fDataDir += fileSep;
            }

            fDataDir = fDataDir + "service" + fileSep +
                fServiceName.toLowerCase();

            File dir = new File(fDataDir);

            if (!dir.exists())
            {
                dir.mkdirs();
            }

            fRegManagerFileName = fDataDir + fileSep + "RegManager.out";
            fGenManagerFileName = fDataDir + fileSep + "GenManager.out";

            // read registration data from the RegManager.out file
            deSerialize(fRegManagerFileName);

            fGenManager = new GenerationManager(fHandle, fServiceName,
                                                fRegManager,
                                                fGenManagerFileName);

            // read EventID from the GenManager.out file
            deSerialize("fGenManagerFileName");

            // Assign the help text string for the service

            sHelpMsg = "*** " + info.name + " Service Help ***" +
                lineSep + lineSep +
                "REGISTER          TYPE <Type> [SUBTYPE <Subtype>]..." +
                lineSep +
                "                  [ACKNOWLEDGETIMEOUT <Timeout>] [MAXATTEMPTS <Number>]" +
                lineSep +
                "                  [PRIORITY <Number>] [PRIORITYDELTA <Number>]" +
                lineSep +
                "                  [BYNAME | BYHANDLE]" +
                lineSep + lineSep +
                "UNREGISTER        TYPE <Type> [SUBTYPE <Subtype>]..." +
                lineSep +
                "                  [FORCE [MACHINE <Machine>] [HANDLE <Handle> | NAME <Name>]]" +
                lineSep + lineSep +
                "ACKNOWLEDGE       EVENTID <Event ID>" +
                lineSep +
                "                  [FORCE [MACHINE <Machine>] [HANDLE <Handle> | NAME <Name>]]" +
                lineSep + lineSep +
                "GENERATE          TYPE <Type> SUBTYPE <Subtype> [PROPERTY <Name>[=<Value>]]..." +
                lineSep +
                "                  [NOTIFY]" +
                lineSep + lineSep +
                "LIST              TYPES [LONG]" +
                lineSep +
                "LIST              SUBTYPES TYPE <Type>" +
                lineSep +
                "LIST              REGISTRATIONS [TYPE <Type> [SUBTYPE <Subtype>]] [LONG]" +
                lineSep +
                "LIST              EVENTIDS [LONG]" +
                lineSep +
                "LIST              SETTINGS" +
                lineSep + lineSep +
                "QUERY             EVENTID <Event ID> [LONG]" +
                lineSep + lineSep +
                "RESET             <REG | GEN> FORCE" +
                lineSep + lineSep +
                "VERSION" +
                lineSep + lineSep +
                "HELP";

            // Register Help Data

            registerHelpData(
                kNoAckPending,
                "No Acknowledgement Pending",
                "Already received an acknowledgement for this event from " +
                "the acknowledging process.");

            registerHelpData(
                kNoSuchID,
                "No such event ID",
                "An acknowledgement was received but no event with the " +
                "submitted Event ID has been generated.");

            registerHelpData(
                kNotRegisteredForType,
                "Not registered for Type",
                "A process tried to unregister for an event Type that it " +
                "was not currently registered for.");

            registerHelpData(
                kNotRegisteredForSubtype,
                "Not registered for Subtype",
                "A process tried to unregister for an event Subtype that " +
                "it was not currently registered for.");

            registerHelpData(
                kNoClientsForEvent,
                "No clients registered",
                "No clients were registered for the event.");

            res.rc = STAFResult.Ok;
            res.result = "";
            return res;
        }
        catch(STAFException e)
        {
            if (DEBUG) e.printStackTrace();
            STAFResult res = new STAFResult(STAFResult.Ok);

            res.rc = STAFResult.STAFRegistrationError;
            res.result = e.toString();

            return res;
        }
    }


    private STAFResult handleParms(STAFServiceInterfaceLevel30.InitInfo info)
        throws STAFException
    {
        STAFCommandParseResult parsedResult= fParmsParser.parse(info.parms);

        String errmsg = "ERROR:  Service Configuration Error for Service " +
                        fServiceName + lineSep +
                        "EventService::handleParms() - ";

        if (parsedResult.rc != STAFResult.Ok)
        {
            return new STAFResult(
                STAFResult.InvalidRequestString, parsedResult.errorBuffer);
        }

        STAFResult resolvedResult;

        if (parsedResult.optionTimes("MAXATTEMPTS") > 0)
        {
            resolvedResult = STAFUtil.resolveInitVarAndCheckInt(
                "MAXATTEMPTS", parsedResult.optionValue("MAXATTEMPTS"),
                fHandle);

            if (resolvedResult.rc != STAFResult.Ok) return resolvedResult;

            fMaxAttempts = Integer.parseInt(resolvedResult.result);
        }

        if (parsedResult.optionTimes("ACKNOWLEDGETIMEOUT") > 0)
        {
            resolvedResult = STAFUtil.resolveInitVarAndCheckInt(
                "ACKNOWLEDGETIMEOUT",
                parsedResult.optionValue("ACKNOWLEDGETIMEOUT"), fHandle);

            if (resolvedResult.rc != STAFResult.Ok) return resolvedResult;
            
            fAckTimeout = Integer.parseInt(resolvedResult.result);
        }

        // PRIORITY is an undocumented service parameter

        if (parsedResult.optionTimes("PRIORITY") > 0)
        {
            resolvedResult = STAFUtil.resolveInitVarAndCheckInt(
                "PRIORITY", parsedResult.optionValue("PRIORITY"), fHandle);

            if (resolvedResult.rc != STAFResult.Ok) return resolvedResult;

            fPriority = Integer.parseInt(resolvedResult.result);
        }

        // PRIORITYDELTA is an undocumented service parameter

        if (parsedResult.optionTimes("PRIORITYDELTA") > 0)
        {
            resolvedResult = STAFUtil.resolveInitVarAndCheckInt(
                "PRIORITYDELTA", parsedResult.optionValue("PRIORITYDELTA"),
                fHandle);

            if (resolvedResult.rc != STAFResult.Ok) return resolvedResult;

            fPriorityDelta = Integer.parseInt(resolvedResult.result);
        }

        return new STAFResult(STAFResult.Ok);
    }

    public STAFResult term()
    {
        fHandle.submit2(
            STAFHandle.ReqFireAndForget,
            "local", "QUEUE", "QUEUE TYPE " + "STAF/Service/Event/End" +
            " MESSAGE " + STAFUtil.wrapData(""));

        // Un-register Help Data

        unregisterHelpData(kNoAckPending);
        unregisterHelpData(kNoSuchID);
        unregisterHelpData(kNotRegisteredForType);
        unregisterHelpData(kNotRegisteredForSubtype);
        unregisterHelpData(kNoClientsForEvent);

        // Un-register the service handle

        try
        {
            fHandle.unRegister();
        }
        catch (STAFException ex)
        {
            return new STAFResult(STAFResult.STAFRegistrationError,
                                  "Error unregistering handle.");
        }

        return new STAFResult(STAFResult.Ok);
    }

    /***************************************************************************/
    /* acceptRequest- Calls appropriate methods to process a request from a    */
    /*                client process.                                          */
    /*                                                                         */
    /* accepts: STAFServiceInterfaceLevel30 request information                */
    /*                                                                         */
    /* returns: STAFResult.rc = STAFResult.Ok, if successful; STAFResult.      */
    /*          InvalidRequestString, if unsuccessful;                         */
    /*                                                                         */
    /*          STAFResult.result contains the result returned by the called   */
    /*          method, if successful;                                         */
    /*          STAFResult.result contains the command, if unsuccessful        */
    /***************************************************************************/
    public STAFResult acceptRequest(STAFServiceInterfaceLevel30.RequestInfo info)
    {
        // Try block is here to catch any unexpected errors/exceptions

        try
        {
            // Determine the command request (the first word in the request)

            String action;
            int spaceIndex = info.request.indexOf(" ");

            if (spaceIndex != -1)
                action = info.request.substring(0, spaceIndex);
            else
                action = info.request;

            String actionLC = action.toLowerCase();

            // Call the appropriate method to handle the command request

            if (actionLC.equals("register"))
                return handleRegister(info);
            else if (actionLC.equals("generate"))
                return handleGenerate(info);
            else if (actionLC.equals("acknowledge"))
                return handleAck(info);
            else if (actionLC.equals("reset"))
                return handleReset(info);
            else if (actionLC.equals("unregister"))
                return handleUnRegister(info);
            else if (actionLC.equals("query"))
                return handleQuery(info);
            else if (actionLC.equals("list"))
                return handleList(info);
            else if (actionLC.equals("help"))
                return handleHelp(info);
            else if (actionLC.equals("version"))
                return handleVersion(info);
            else
            {
                String errMsg = "'" + action +
                    "' is not a valid command request for the " +
                    fServiceName + " service" + lineSep + lineSep +
                    sHelpMsg;

                return new STAFResult(
                    STAFResult.InvalidRequestString, errMsg);
            }
        }
        catch (Throwable t)
        {
            // Write the Java stack trace to the JVM log for the service

            System.out.println(
                sTimestampFormat.format(Calendar.getInstance().getTime()) +
                " ERROR: Exception on " + fServiceName + " service request:" +
                lineSep + lineSep + info.request + lineSep);

            t.printStackTrace();
            
            // And also return the Java stack trace in the result

            StringWriter sr = new StringWriter();
            t.printStackTrace(new PrintWriter(sr));

            if (t.getMessage() != null)
            {
                return new STAFResult(
                    STAFResult.JavaError,
                    t.getMessage() + lineSep + sr.toString());
            }
            else
            {
                return new STAFResult(
                    STAFResult.JavaError, sr.toString());
            }
        }

    }

    /***********************************************************************/
    /* serialize- object serializer that writes structured data to a file  */
    /* using the ObjectOutputStream.writeObject method().                  */
    /*                                                                     */
    /* accepts: the file name.                                             */
    /* returns: void.                                                      */
    /***********************************************************************/

    synchronized private void serialize(String theFileName)
    {
        try
        {
            if (theFileName == null) theFileName = fRegManagerFileName;

            if (theFileName.equals(fRegManagerFileName))
            {
                ObjectOutputStream out = new ObjectOutputStream(
                                         new FileOutputStream(theFileName));
                out.writeObject(fRegManager);
                out.close();
            }
            else fGenManager.serialize();
        }
        catch(Exception e)
        {
           if (DEBUG) e.printStackTrace();
        }
    }

    /***********************************************************************/
    /* deSerialize- object deSerializer that reads fGenManager.fEventID    */
    /* from a file using the ObjectInputStream.readObject method().        */
    /*                                                                     */
    /* accepts: the file name.                                             */
    /* returns: void.                                                      */
    /***********************************************************************/

    synchronized private void deSerialize(String theFileName)
    {
        try
        {
            ObjectInputStream in = new ObjectInputStream(
                                   new FileInputStream(theFileName));

            if (theFileName == null) theFileName = fRegManagerFileName;

            if (theFileName.equals(fRegManagerFileName))
                fRegManager = (RegistrationManager)in.readObject();
            else
                fGenManager.deSerialize();
        }
        catch(Exception e)
        {
            if (theFileName.equals(fRegManagerFileName))
                fRegManager = new RegistrationManager();

            if (DEBUG) e.printStackTrace();
        }
    }


    // Register error codes for the Event Service with the HELP service

    private void registerHelpData(int errorNumber, String info,
                                 String description)
    {
        STAFResult res = fHandle.submit2("local", "HELP",
                         "REGISTER SERVICE " + fServiceName +
                         " ERROR " + errorNumber +
                         " INFO " + STAFUtil.wrapData(info) +
                         " DESCRIPTION " + STAFUtil.wrapData(description));
    }

    // Un-register error codes for the Event Service with the HELP service

    private void unregisterHelpData(int errorNumber)
    {
        STAFResult res = fHandle.submit2("local", "HELP",
                         "UNREGISTER SERVICE " + fServiceName +
                         " ERROR " + errorNumber);
    }
    
    /***************************************************************************/
    /* getRegClientMap - Returns a map containing the information for an entry */
    /*                   in a LIST REGISTRATIONS request                       */
    /*                   (Called by the handleList method)                     */
    /*                                                                         */
    /* accepts: the event type, event subtype, client, and whether long format */
    /*          was specified                                                  */
    /*                                                                         */
    /* returns: a Map to be added to the marshalled list of registrations      */
    /***************************************************************************/

    private Map getRegClientMap(String type, String subtype, Client client,
                                boolean longFormat)
    {
        if (!longFormat)
        {
            // Short format for LIST REGISTRATIONS request

            Map regClientMap = fListRegMapClass.createInstance();
            Map clientMap = client.getClientMap(fListRegLongMapClass);

            regClientMap.put("type", type);
            regClientMap.put("subtype", subtype);
            regClientMap.put("machine", clientMap.get("machine"));
            regClientMap.put("notifyBy", clientMap.get("notifyBy"));
            regClientMap.put("notifiee", clientMap.get("notifiee"));

            return regClientMap;
        }
        else
        {
            // Long format for LIST REGISTRATIONS request

            Map regClientMap = fListRegLongMapClass.createInstance();
            Map clientMap = client.getClientMap(fListRegLongMapClass);

            regClientMap.putAll(clientMap);
            regClientMap.put("type", type);
            regClientMap.put("subtype", subtype);

            return regClientMap;
        }
    }

    /***************************************************************************/
    /* handleRegister - calls fRegManager.registerClient() to register a       */
    /*                  process for future notification when events occur.     */
    /*                                                                         */
    /* accepts: STAFServiceInterfaceLevel30 request information                */
    /*          (including the submitting machine name, the submitting process,*/
    /*          the submitting process handle, a string containing any command */
    /*          options & option values).                                      */
    /*                                                                         */
    /* returns: if successful, STAFResult.rc = STAFResult.Ok (0);              */
    /*                         STAFResult.result is empty;                     */
    /*          if unsuccessful, STAFResult.rc   = error code > 0;             */
    /*                           STAFResult.result contains message            */
    /*                           correspond to error code in STAFResult.rc     */
    /***************************************************************************/

    private STAFResult handleRegister(
        STAFServiceInterfaceLevel30.RequestInfo info)
    {

        String type = "";
        String subtype = "";

        // Set to default values for the service
        int maxAttempts = fMaxAttempts;
        int timeout = fAckTimeout;
        int priority = fPriority;
        int priorityDelta = fPriorityDelta;

        STAFResult sResult = new STAFResult(STAFResult.Ok, "");
        STAFResult resolvedResult = null;
        STAFResult resolvedValue = null;

        // Verify the requesting machine/user has at least trust level 3

        STAFResult trustResult = STAFUtil.validateTrust(
            3, fServiceName, "REGISTER", fLocalMachineName, info);

        if (trustResult.rc != STAFResult.Ok) return trustResult;

        // Parse the request

        STAFCommandParseResult parsedRequest = fRegisterParser.parse(
            info.request);

        if (parsedRequest.rc != STAFResult.Ok)
        {
            return new STAFResult(STAFResult.InvalidRequestString,
                                  parsedRequest.errorBuffer);
        }

        resolvedValue = STAFUtil.resolveRequestVar(
            parsedRequest.optionValue("type"), fHandle, info.requestNumber);

        if (resolvedValue.rc != 0) return resolvedValue;

        type = resolvedValue.result;

        if (parsedRequest.optionTimes("maxAttempts") > 0)
        {
            resolvedResult = STAFUtil.resolveRequestVarAndCheckInt(
                "maxAttempts", parsedRequest.optionValue("maxAttempts"),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            maxAttempts = Integer.parseInt(resolvedResult.result);
        }

        if (parsedRequest.optionTimes("acknowledgeTimeout") > 0)
        {
            resolvedResult = STAFUtil.resolveRequestVarAndCheckInt(
                "acknowledgeTimeout",
                parsedRequest.optionValue("acknowledgeTimeout"),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            timeout = Integer.parseInt(resolvedResult.result);
        }

        if (parsedRequest.optionTimes("priority") > 0)
        {
            resolvedResult = STAFUtil.resolveRequestVarAndCheckInt(
                "priority", parsedRequest.optionValue("priority"),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            priority = Integer.parseInt(resolvedResult.result);
        }

        if (parsedRequest.optionTimes("priorityDelta") > 0)
        {
            resolvedResult = STAFUtil.resolveRequestVarAndCheckInt(
                "priorityDelta", parsedRequest.optionValue("priorityDelta"),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            priorityDelta = Integer.parseInt(resolvedResult.result);
        }

        int subTimes = parsedRequest.optionTimes("subtype");
        String[] subTypes = new String[subTimes];

        for (int count = 1; count <= subTimes; count++)
        {
            resolvedResult = STAFUtil.resolveRequestVar(
                parsedRequest.optionValue("subtype", count),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            subTypes[count - 1] = resolvedResult.result;
        }

        if ((parsedRequest.optionTimes("byHandle") == 0) &&
            (sResult.rc == 0))
        {
            fRegManager.registerClient(
                new Client(info.endpoint, info.handleName,
                           maxAttempts, timeout, priority,
                           priorityDelta), type, subTypes);
        }
        else if (sResult.rc == 0)
        {
            fRegManager.registerClient(
                new Client(info.endpoint, info.handle,
                           maxAttempts, timeout, priority,
                           priorityDelta), type, subTypes);
        }

        serialize(fRegManagerFileName);

        return sResult;
    }

    /***************************************************************************/
    /* handleQuery:                                                            */
    /*                                                                         */
    /* Handles QUERY EVENTID <EventID> [LONG] reqests.                         */
    /* Lists, by EVENTID, the above data about registered                      */
    /*              processes that have not yet acknowledged a generated event.*/
    /*              Also lists the following data about the process that       */
    /*              generated the event:                                       */
    /*                                                                         */
    /*              eventID#, generating machine name, generating process name,*/
    /*              generating process handle, generating, type of event,      */
    /*              subtype of event                                           */
    /*                                                                         */
    /* accepts: the submitting machine name, the submitting process, the sub-  */
    /*          mitting process handle, a string containing any command options*/
    /*          & option values.                                               */
    /*                                                                         */
    /* returns: if successful, STAFResult.rc = STAFResult.Ok (0);              */
    /*                         STAFResult.result contains the data described   */
    /*                         above.                                          */
    /*          if unsuccessful, STAFResult.rc   = error code > 0;             */
    /*                           STAFResult.result contains a message          */
    /*                           correspond to error code in STAFResult.rc     */
    /***************************************************************************/

    private STAFResult handleQuery(STAFServiceInterfaceLevel30.RequestInfo info)
    {

        String type = null;
        String subtype = null;
        String eid = null;
        int id = 0;
        STAFResult resolvedResult = null;
        STAFMarshallingContext mc = new STAFMarshallingContext();

        // Verify the requesting machine/user has at least trust level 2

        STAFResult trustResult = STAFUtil.validateTrust(
            2, fServiceName, "QUERY", fLocalMachineName, info);

        if (trustResult.rc != STAFResult.Ok) return trustResult;

        // Parse the reqeust

        STAFCommandParseResult parsedRequest = fQueryParser.parse(
            info.request);

        if (parsedRequest.rc != STAFResult.Ok)
        {
            return new STAFResult(STAFResult.InvalidRequestString,
                                  parsedRequest.errorBuffer);
        }

        resolvedResult = STAFUtil.resolveRequestVarAndCheckInt(
            "eventID", parsedRequest.optionValue("eventID"),
            fHandle, info.requestNumber);

        if (resolvedResult.rc != 0) return resolvedResult;

        id = Integer.parseInt(resolvedResult.result);

        Vector clientsForEvent = fGenManager.getClientsForEvent(id);

        boolean longFormat = false;

        if (parsedRequest.optionTimes("LONG") > 0)
        {
            longFormat = true;
            mc.setMapClassDefinition(fEventIDLongMapClass);
            mc.setMapClassDefinition(fNotifieeMapClass);
            mc.setMapClassDefinition(fEventGeneratorMapClass);
        }
        else
        {
            mc.setMapClassDefinition(fQueryEventIDMapClass);
            mc.setMapClassDefinition(fEventGeneratorMapClass);
        }

        if ((clientsForEvent != null) && (clientsForEvent.size() > 0))
        {
            for (Enumeration cfe = clientsForEvent.elements();
                 cfe.hasMoreElements();)
            {
                Map notificationMap =
                    ((GenerationManager.Notification)
                     cfe.nextElement()).getNotificationMap();

                if (longFormat)
                {
                    mc.setRootObject(notificationMap);
                    break;
                }
                else
                {
                    Map eventIDMap = fQueryEventIDMapClass.
                        createInstance();

                    eventIDMap.put(
                        "eventID", notificationMap.get("eventID"));
                    eventIDMap.put(
                        "type", notificationMap.get("type"));
                    eventIDMap.put(
                        "subtype", notificationMap.get("subtype"));

                    // Note that property values that contain private
                    // data will be masked.
                    eventIDMap.put(
                        "propertyMap",
                        notificationMap.get("propertyMap"));

                    eventIDMap.put(
                        "generatedBy",
                        notificationMap.get("generatedBy"));

                    int numNotifiees = ((List)notificationMap.get(
                        "notificationList")).size();

                    eventIDMap.put("numNotifiees", "" + numNotifiees);

                    mc.setRootObject(eventIDMap);
                    break;
                }
            }
        }
        else
        {
            return new STAFResult(kNoClientsForEvent,
                   "No processes are waiting to be notified about " +
                   "eventID " + id + ".");
        }

        return new STAFResult(STAFResult.Ok, mc.marshall());
    }

    /***************************************************************************/
    /* handleList:                                                             */
    /*                                                                         */
    /* For a LIST TYPES request, lists all event types for which at least one  */
    /* process is currently registered.                                        */
    /*                                                                         */
    /* For a LIST SUBTYPES TYPE <Type> request, lists all event subtypes for   */
    /* the specified TYPE for which at least one process is currently          */
    /* registered.                                                             */
    /*                                                                         */
    /* For a LIST REGISTRATIONS request, lists all the registrations, or if    */
    /* TYPE/SUBTYPE is specified, just the registrations for the specified     */
    /* types and subtypes.                                                     */
    /*                                                                         */
    /* For a LIST EVENTIDS request, lists, by EVENTID, data about registered   */
    /* processes that have not yet acknowledged a generated event.             */
    /*                                                                         */
    /* For a LIST SETTINGS request, lists the operational settings for the     */
    /* event service (maxAttempts, ackTimeout, priority, priorityDelta)        */
    /*                                                                         */
    /* accepts: STAFServiceInterfaceLevel30 request information                */
    /*          (including the submitting machine name, the submitting process,*/
    /*          the submitting process handle, a string containing any command */
    /*          options & option values).                                      */
    /*                                                                         */
    /* returns: if successful, STAFResult.rc = STAFResult.Ok (0);              */
    /*                         STAFResult.result contains the marshalled data  */
    /*                         described above.                                */
    /*          if unsuccessful, STAFResult.rc   = error code > 0;             */
    /*                           STAFResult.result contains a message          */
    /*                           correspond to error code in STAFResult.rc     */
    /***************************************************************************/

    private STAFResult handleList(STAFServiceInterfaceLevel30.RequestInfo info)
    {

        String type = null;
        String subtype = null;
        String eid = null;
        int id = 0;
        STAFResult resolvedResult = null;
        STAFMarshallingContext mc = new STAFMarshallingContext();

        // Verify that the requesting machine/user has at least trust level 2

        STAFResult trustResult = STAFUtil.validateTrust(
            2, fServiceName, "LIST", fLocalMachineName, info);

        if (trustResult.rc != STAFResult.Ok) return trustResult;

        // Parse the reqeust

        STAFCommandParseResult parsedRequest = fListParser.parse(
            info.request);

        if (parsedRequest.rc != STAFResult.Ok)
        {
            return new STAFResult(STAFResult.InvalidRequestString,
                                  parsedRequest.errorBuffer);
        }

        if (parsedRequest.optionTimes("REGISTRATIONS") > 0)
        {
            boolean longFormat = false;

            if (parsedRequest.optionTimes("LONG") > 0)
            {
                longFormat = true;
                mc.setMapClassDefinition(fListRegLongMapClass);
            }
            else
            {
                mc.setMapClassDefinition(fListRegMapClass);
            }

            List regClientList = new ArrayList();

            if (parsedRequest.optionTimes("TYPE") == 0)
            {
                // List all registrations

                for (Enumeration e = fRegManager.fRegistrationTable.keys();
                     e.hasMoreElements();)
                {
                    type = (String)e.nextElement();
                    RegistrationManager.Type typeObj =
                        (RegistrationManager.Type)
                        (fRegManager.fRegistrationTable.get(type));
                    Vector typeClients = fRegManager.
                        clientsRegisteredFor(type);

                    if (typeClients != null)
                    {
                        for (Enumeration vectorEnum = typeClients.
                             elements(); vectorEnum.hasMoreElements();)
                        {
                            Client client = (Client)vectorEnum.
                                nextElement();

                            regClientList.add(getRegClientMap(
                                type, null, client, longFormat));
                        }
                    }

                    if (typeObj != null)
                    {
                        for (Enumeration subEnum = typeObj.subtypeTable.
                             keys(); subEnum.hasMoreElements();)
                        {
                            subtype = (String)subEnum.nextElement();
                            Vector subtypeClients = fRegManager.
                                clientsRegisteredFor(type, subtype);

                            if (subtypeClients != null)
                            {
                                for (Enumeration vectorEnum =
                                     subtypeClients.elements();
                                     vectorEnum.hasMoreElements();)
                                {
                                    Client client = (Client)vectorEnum.
                                        nextElement();

                                    regClientList.add(getRegClientMap(
                                        type, subtype, client, longFormat));
                                }
                            }
                        }
                    }
                }
            }
            else if (parsedRequest.optionTimes("SUBTYPE") == 0)
            {
                // Option TYPE is specified, without SUBTYPE

                resolvedResult = STAFUtil.resolveRequestVar(
                    parsedRequest.optionValue("type"),
                    fHandle, info.requestNumber);

                if (resolvedResult.rc != 0) return resolvedResult;

                type = resolvedResult.result;

                // List all registrations for a specified type

                RegistrationManager.Type typeObj =
                    (RegistrationManager.Type)
                        (fRegManager.fRegistrationTable.get(
                            type.toLowerCase()));
                Vector typeClients = fRegManager.clientsRegisteredFor(
                    type.toLowerCase());

                if (typeClients != null)
                {
                    for (Enumeration vectorEnum = typeClients.
                         elements(); vectorEnum.hasMoreElements();)
                    {
                        Client client = (Client)vectorEnum.nextElement();

                        regClientList.add(getRegClientMap(
                            type, null, client, longFormat));
                    }
                }

                if (typeObj != null)
                {
                    for (Enumeration subEnum = typeObj.subtypeTable.keys();
                         subEnum.hasMoreElements();)
                    {
                        subtype = (String)subEnum.nextElement();
                        Vector subtypeClients = fRegManager.
                            clientsRegisteredFor(type, subtype);

                        if (subtypeClients != null)
                        {
                            for (Enumeration vectorEnum = subtypeClients.
                                 elements(); vectorEnum.hasMoreElements();)
                            {
                                Client client = (Client)vectorEnum.
                                    nextElement();

                                regClientList.add(getRegClientMap(
                                    type, subtype, client, longFormat));
                            }
                        }
                    }
                }

                if (regClientList.size() == 0)
                {
                    return new STAFResult(
                        kNotRegisteredForType,
                        "No processes are registered to be notified " +
                        "about events of type " + type + ".");
                }

                mc.setRootObject(regClientList);
            }
            else
            {
                // TYPE and SUBTYPE are specified

                resolvedResult = STAFUtil.resolveRequestVar(
                    parsedRequest.optionValue("type"),
                    fHandle, info.requestNumber);

                if (resolvedResult.rc != 0) return resolvedResult;

                type = resolvedResult.result;

                resolvedResult = STAFUtil.resolveRequestVar(
                    parsedRequest.optionValue("subtype"),
                    fHandle, info.requestNumber);

                if (resolvedResult.rc != 0) return resolvedResult;

                subtype = resolvedResult.result;

                Vector crf = new Vector();

                crf = fRegManager.clientsRegisteredFor(type, subtype);

                if ((crf == null) || (crf.size() == 0))
                {
                    return new STAFResult(
                        subtype == null ?
                        kNotRegisteredForType : kNotRegisteredForSubtype,
                        "No processes are registered to be notified about " +
                        "events of type " + type +
                        (subtype == null ? "." :
                         ", subtype " + subtype + "."));
                }

                if (crf != null)
                {
                    for (Enumeration e = crf.elements();
                         e.hasMoreElements();)
                    {
                        Client client = (Client)e.nextElement();

                        regClientList.add(getRegClientMap(
                            type, subtype, client, longFormat));
                    }
                }
            }

            mc.setRootObject(regClientList);
        }
        else if (parsedRequest.optionTimes("TYPES") > 0)
        {
            if (parsedRequest.optionTimes("LONG") > 0)
            {
                mc.setMapClassDefinition(fTypeMapClass);
                mc.setRootObject(fRegManager.getTypes(true));
            }
            else
            {
                mc.setRootObject(fRegManager.getTypes(false));
            }
        }
        else if (parsedRequest.optionTimes("SUBTYPES") > 0)
        {
            resolvedResult = STAFUtil.resolveRequestVar(
                parsedRequest.optionValue("type"),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            type = resolvedResult.result;
            List subtypesList = fRegManager.getSubtypes(type);

            if (subtypesList == null)
            {
                return new STAFResult(
                    kNotRegisteredForType,
                    "No processes are registered to be notified " +
                    "about events of type " + type + ".");
            }

            mc.setRootObject(
                fRegManager.getSubtypes(resolvedResult.result));
        }
        else if (parsedRequest.optionTimes("EVENTIDS") > 0)
        {
            if (!fGenManager.fEventTable.keys().hasMoreElements())
            {
                return new STAFResult(kNoClientsForEvent,
                       "No events are currently waiting to be " +
                       "acknowledged");
            }

            boolean longFormat = false;

            if (parsedRequest.optionTimes("LONG") > 0)
            {
                longFormat = true;

                mc.setMapClassDefinition(fEventIDLongMapClass);
                mc.setMapClassDefinition(fNotifieeMapClass);
                mc.setMapClassDefinition(fEventGeneratorMapClass);
            }
            else
            {
                mc.setMapClassDefinition(fListEventIDsMapClass);
            }

            List eventIDList = new ArrayList();

            for (Enumeration e = fGenManager.fEventTable.keys();
                 e.hasMoreElements();)
            {
                id = ((Integer)e.nextElement()).intValue();

                Vector clientsForEvent =
                       fGenManager.getClientsForEvent(id);

                if (clientsForEvent.size() > 0)
                {
                    for (Enumeration cfe = clientsForEvent.elements();
                         cfe.hasMoreElements();)
                    {
                        Map notifyMap =
                            ((GenerationManager.Notification)
                             cfe.nextElement()).getNotificationMap();

                        if (!longFormat)
                        {
                            Map resultMap =
                                fListEventIDsMapClass.createInstance();

                            resultMap.put("eventID", "" + id);
                            resultMap.put("type",
                                          notifyMap.get("type"));
                            resultMap.put("subtype",
                                          notifyMap.get("subtype"));
                            resultMap.put(
                                "numNotifiees", "" +
                                ((List)notifyMap.get("notificationList")).
                                size());

                            eventIDList.add(resultMap);
                        }
                        else
                        {
                            // Long format

                            eventIDList.add(notifyMap);
                        }
                    }
                }
            }

            if (eventIDList.size() == 0)
            {
                return new STAFResult(kNoClientsForEvent,
                       "No events are currently waiting to be " +
                       "acknowledged.");
            }

            mc.setRootObject(eventIDList);
        }
        else if (parsedRequest.optionTimes("SETTINGS") > 0)
        {
            mc.setMapClassDefinition(fSettingsMapClass);

            Map outputMap = fSettingsMapClass.createInstance();
            outputMap.put("maxAttempts", new Integer(fMaxAttempts));
            outputMap.put("ackTimeout", new Integer(fAckTimeout));
            outputMap.put("priority", new Integer(fPriority));
            outputMap.put("priorityDelta", new Integer(fPriorityDelta));

            mc.setRootObject(outputMap);
        }

        return new STAFResult(STAFResult.Ok, mc.marshall());
    }

    /***************************************************************************/
    /* handleGenerate - calls fGenManager.generateEvent() to notify all        */
    /*                  processes registered for events of the type and        */
    /*                  subtype contained in the "args" string. Includes a     */
    /*                  map of properties in the message sent to registered    */
    /*                  clients.                                               */
    /*                                                                         */
    /* accepts: STAFServiceInterfaceLevel30 request information                */
    /*          (including the submitting machine name, the submitting process,*/
    /*          the submitting process handle, a string containing any command */
    /*          options & option values).                                      */
    /*                                                                         */
    /* returns: if successful, STAFResult.rc = STAFResult.Ok (0);              */
    /*                         STAFResult.result contains the data described   */
    /*                         above.                                          */
    /*          if unsuccessful, STAFResult.rc   = error code > 0;             */
    /*                           STAFResult.result contains a message          */
    /*                           correspond to error code in STAFResult.rc     */
    /***************************************************************************/

    private STAFResult handleGenerate(
        STAFServiceInterfaceLevel30.RequestInfo info)
    {
        Vector v = new Vector();
        STAFResult resolvedResult = null;
        String type = null;     //parsedRequest.optionValue("type");
        String subtype = null;  //parsedRequest.optionValue("subtype");
        boolean notify = false; // Notify generator when everyone has ackd
        boolean async = false;
        int eventID;
        // Verify the requesting machine/user has at least trust level 4

        STAFResult trustResult = STAFUtil.validateTrust(
            4, fServiceName, "GENERATE", fLocalMachineName, info);

        if (trustResult.rc != STAFResult.Ok) return trustResult;

        // Parse the reqeust

        STAFCommandParseResult parsedRequest = fGenerateParser.parse(
            info.request);

        if (parsedRequest.rc != STAFResult.Ok)
        {
            return new STAFResult(STAFResult.InvalidRequestString,
                                  parsedRequest.errorBuffer);
        }

        int propertyTimes = parsedRequest.optionTimes("property");

        if (parsedRequest.optionTimes("type") > 0)
        {
            resolvedResult = STAFUtil.resolveRequestVar(
                parsedRequest.optionValue("type"),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            type = resolvedResult.result;

            if (parsedRequest.optionTimes("subtype") > 0)
            {
                resolvedResult = STAFUtil.resolveRequestVar(
                    parsedRequest.optionValue("subtype"),
                    fHandle, info.requestNumber);

                if (resolvedResult.rc != 0) return resolvedResult;

                subtype = resolvedResult.result;
            }
        }

        // Do NOT resolve PROPERTY option values

        for (int count = 1; count <= propertyTimes; count++)
        {
            String property = parsedRequest.optionValue("property", count);

            if (property.indexOf('=') == 0)
            {
                return new STAFResult(
                    STAFResult.InvalidValue,
                    "Invalid value for PROPERTY " + property +
                    "\nThe format for a PROPERTY option value must be:" +
                    " <Name>[=<Value>]");
            }

            v.addElement(property);
        }

        if (parsedRequest.optionTimes("notify") > 0)
        {
            notify = true;
        }

        if (parsedRequest.optionTimes("async") > 0)
        {
            async = true;
        }

        String[] properties = new String[v.size()];
        int index = 0;

        for (Enumeration e = v.elements(); e.hasMoreElements();)
            properties[index++] = (String)e.nextElement();

        eventID = fGenManager.generateEvent(
            info.endpoint, info.handleName, info.handle, type, subtype,
            properties, notify, async, fRegManager.clientsFor(type, subtype));

        if (eventID == kNoClientsForEvent)
        {
            return new STAFResult(
                kNoClientsForEvent,
                "No processes are registered to be notified about " +
                "events of type " + type + ", subtype " + subtype + ".");
        }

        serialize(fGenManagerFileName);

        return new STAFResult(STAFResult.Ok, "" + eventID);
    }

    /***************************************************************************/
    /* handleUnRegister - calls fRegManager.unRegisterClient() to unregister   */
    /*                    the submitting process. The submitting process may   */
    /*                    also FORCE another process (including a process on   */
    /*                    another machine: Trust = 4 required) to unregister.  */
    /*                                                                         */
    /* accepts: STAFServiceInterfaceLevel30 request information                */
    /*          (including the submitting machine name, the submitting process,*/
    /*          the submitting process handle, a string containing any command */
    /*          options & option values).                                      */
    /*                                                                         */
    /* returns: if successful, STAFResult.rc = STAFResult.Ok (0);              */
    /*                         STAFResult.result contains the data described   */
    /*                         above.                                          */
    /*          if unsuccessful, STAFResult.rc   = error code > 0;             */
    /*                           STAFResult.result contains a message          */
    /*                           correspond to error code in STAFResult.rc     */
    /***************************************************************************/

    private STAFResult handleUnRegister(
        STAFServiceInterfaceLevel30.RequestInfo info)
    {
        int subTimes = 0;
        String type = null;
        String subtype = null;
        String machine = info.endpoint;
        int handle = info.handle;
        String handleName = info.handleName;
        STAFResult resolvedResult = null;

        // Parse the request

        STAFCommandParseResult parsedRequest = fUnregisterParser.parse(
            info.request);

        if (parsedRequest.rc != STAFResult.Ok)
        {
            return new STAFResult(
                STAFResult.InvalidRequestString,
                parsedRequest.errorBuffer);
        }

        // Verify the requesting machine/user has at least trust level 3
        // if FORCE isn't specified, or 4 if FORCE is specified

        STAFResult trustResult;

        if (parsedRequest.optionTimes("FORCE") == 0)
        {
            trustResult = STAFUtil.validateTrust(
                3, fServiceName, "UNREGISTER", fLocalMachineName, info);
        }
        else
        {
            trustResult = STAFUtil.validateTrust(
                4, fServiceName, "UNREGISTER FORCE", fLocalMachineName,
                info);
        }

        if (trustResult.rc != STAFResult.Ok) return trustResult;

        // Get and resolve option values specified for the request

        if (parsedRequest.optionTimes("MACHINE") > 0)
        {
            resolvedResult = STAFUtil.resolveRequestVar(
                parsedRequest.optionValue("machine"),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            if (resolvedResult.result.toLowerCase().equals("local"))
                machine = "local://local";
            else
                machine = resolvedResult.result;
        }

        if (parsedRequest.optionTimes("NAME") > 0)
        {
            resolvedResult = STAFUtil.resolveRequestVar(
                parsedRequest.optionValue("name"),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            handleName = resolvedResult.result;
            handle = 0;  // Need to set so won't match by handle number
        }
        else if (parsedRequest.optionTimes("HANDLE") > 0)
        {
            resolvedResult = STAFUtil.resolveRequestVarAndCheckInt(
                "handle", parsedRequest.optionValue("handle"),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            handle = Integer.parseInt(resolvedResult.result);
            handleName = ""; // Need to set so won't match by handle name
        }

        if (parsedRequest.optionTimes("type") > 0)
        {
            resolvedResult = STAFUtil.resolveRequestVar(
                parsedRequest.optionValue("type"),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            type = resolvedResult.result;
        }

        subTimes = parsedRequest.optionTimes("subtype");

        String[] subTypes = new String[subTimes];

        for (int count = 1; count <= subTimes; count++)
        {
            resolvedResult = STAFUtil.resolveRequestVar(
                parsedRequest.optionValue("subtype", count),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            subTypes[count - 1] = resolvedResult.result;
        }

        STAFResult result = fRegManager.unRegisterClient(
            machine, handleName, handle, type, subTypes);

        if (result.rc != STAFResult.Ok)
            return result;

        serialize(fRegManagerFileName);

        return new STAFResult(STAFResult.Ok, "");
    }

    private STAFResult handleAck(STAFServiceInterfaceLevel30.RequestInfo info)
    {
        String machine = info.endpoint;
        String handleName = info.handleName;
        int handle = info.handle;
        int eventID = 0;
        int rc;
        STAFResult resolvedResult = new STAFResult(STAFResult.Ok, "");

        // Parse the request

        STAFCommandParseResult parsedRequest = fAckParser.parse(
            info.request);

        if (parsedRequest.rc != STAFResult.Ok)
        {
            return new STAFResult(STAFResult.InvalidRequestString,
                                  parsedRequest.errorBuffer);
        }

        // Verify the requesting machine/user has at least trust level 3
        // if FORCE isn't specified, or 4 if FORCE is specified

        STAFResult trustResult;

        if (parsedRequest.optionTimes("FORCE") == 0)
        {
            trustResult = STAFUtil.validateTrust(
                3, fServiceName, "ACKNOWLEDGE", fLocalMachineName, info);
        }
        else
        {
            trustResult = STAFUtil.validateTrust(
                4, fServiceName, "ACKNOWLEDGE FORCE", fLocalMachineName,
                info);
        }

        if (trustResult.rc != STAFResult.Ok) return trustResult;

        // Get and resolve option values specified for the request

        resolvedResult = STAFUtil.resolveRequestVarAndCheckInt(
            "eventID", parsedRequest.optionValue("eventID"),
            fHandle, info.requestNumber);

        if (resolvedResult.rc != 0) return resolvedResult;

        eventID = Integer.parseInt(resolvedResult.result);

        if (parsedRequest.optionTimes("MACHINE") > 0)
        {
            resolvedResult = STAFUtil.resolveRequestVar(
                parsedRequest.optionValue("machine"),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            if (resolvedResult.result.toLowerCase().equals("local"))
                machine = "local://local";
            else
                machine = resolvedResult.result;
        }

        if (parsedRequest.optionTimes("NAME") > 0)
        {
            resolvedResult = STAFUtil.resolveRequestVar(
                parsedRequest.optionValue("name"),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            handleName = resolvedResult.result;
            handle = 0;  // Need to set so won't match by handle number
        }
        else if (parsedRequest.optionTimes("HANDLE") > 0)
        {
            resolvedResult = STAFUtil.resolveRequestVarAndCheckInt(
                "handle", parsedRequest.optionValue("handle"),
                fHandle, info.requestNumber);

            if (resolvedResult.rc != 0) return resolvedResult;

            handle = Integer.parseInt(resolvedResult.result);
            handleName = "";  // Need to set so won't match by handle name
        }

        rc = fGenManager.ackEvent(eventID, machine, handleName, handle);

        if (rc == STAFResult.Ok)
            return new STAFResult (STAFResult.Ok, "" + eventID);
        else if (rc == kNoAckPending)
            return new STAFResult (kNoAckPending, "No ack pending");
        else if (rc == kNoSuchID)
            return new STAFResult (kNoSuchID, "No such event ID");
        else
            return new STAFResult(rc, "Error acknowledging event");
    }

    private STAFResult handleReset(STAFServiceInterfaceLevel30.RequestInfo info)
    {
        // Verify the requesting machine/user has at least trust level 4

        STAFResult trustResult = STAFUtil.validateTrust(
            4, fServiceName, "RESET", fLocalMachineName, info);

        if (trustResult.rc != STAFResult.Ok) return trustResult;

        // Parse the request

        STAFCommandParseResult parsedRequest = fResetParser.parse(
            info.request);

        if (parsedRequest.rc != STAFResult.Ok)
        {
            return new STAFResult(
                STAFResult.InvalidRequestString,
                parsedRequest.errorBuffer);
        }

        if (parsedRequest.optionTimes("FORCE") == 0)
        {
            return new STAFResult(
                STAFResult.InvalidRequestString, "FORCE required");
        }
        else if (parsedRequest.optionTimes("REG") > 0)
        {
            fRegManager.handleReset();
            serialize(fRegManagerFileName);

        }
        else if (parsedRequest.optionTimes("GEN") > 0)
        {
            fGenManager.handleReset();
            serialize(fGenManagerFileName);
        }

        return new STAFResult (STAFResult.Ok, "");
    }

    private STAFResult handleVersion(
        STAFServiceInterfaceLevel30.RequestInfo info)
    {
        // Verify the requesting machine/user has at least trust level 1

        STAFResult trustResult = STAFUtil.validateTrust(
            1, fServiceName, "VERSION", fLocalMachineName, info);

        if (trustResult.rc != STAFResult.Ok) return trustResult;

        // Parse the request

        STAFCommandParseResult parsedRequest = fVersionParser.parse(
            info.request);

        if (parsedRequest.rc != STAFResult.Ok)
        {
            return new STAFResult(STAFResult.InvalidRequestString,
                                  parsedRequest.errorBuffer);
        }
        
        // Return the service version

        return new STAFResult(STAFResult.Ok, kVersion);
    }
    
    private STAFResult handleHelp(STAFServiceInterfaceLevel30.RequestInfo info)
    {
        // Verify the requesting machine/user has at least trust level 1

        STAFResult trustResult = STAFUtil.validateTrust(
            1, fServiceName, "HELP", fLocalMachineName, info);

        if (trustResult.rc != STAFResult.Ok) return trustResult;
        
        // Return help text

        return new STAFResult(STAFResult.Ok, sHelpMsg);
    }
}
